﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/redshift/RedshiftRequest.h>
#include <aws/redshift/Redshift_EXPORTS.h>
#include <aws/redshift/model/PartnerIntegrationStatus.h>

#include <utility>

namespace Aws {
namespace Redshift {
namespace Model {

/**
 */
class UpdatePartnerStatusRequest : public RedshiftRequest {
 public:
  AWS_REDSHIFT_API UpdatePartnerStatusRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdatePartnerStatus"; }

  AWS_REDSHIFT_API Aws::String SerializePayload() const override;

 protected:
  AWS_REDSHIFT_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The Amazon Web Services account ID that owns the cluster.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  UpdatePartnerStatusRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The cluster identifier of the cluster whose partner integration status is
   * being updated.</p>
   */
  inline const Aws::String& GetClusterIdentifier() const { return m_clusterIdentifier; }
  inline bool ClusterIdentifierHasBeenSet() const { return m_clusterIdentifierHasBeenSet; }
  template <typename ClusterIdentifierT = Aws::String>
  void SetClusterIdentifier(ClusterIdentifierT&& value) {
    m_clusterIdentifierHasBeenSet = true;
    m_clusterIdentifier = std::forward<ClusterIdentifierT>(value);
  }
  template <typename ClusterIdentifierT = Aws::String>
  UpdatePartnerStatusRequest& WithClusterIdentifier(ClusterIdentifierT&& value) {
    SetClusterIdentifier(std::forward<ClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the database whose partner integration status is being
   * updated.</p>
   */
  inline const Aws::String& GetDatabaseName() const { return m_databaseName; }
  inline bool DatabaseNameHasBeenSet() const { return m_databaseNameHasBeenSet; }
  template <typename DatabaseNameT = Aws::String>
  void SetDatabaseName(DatabaseNameT&& value) {
    m_databaseNameHasBeenSet = true;
    m_databaseName = std::forward<DatabaseNameT>(value);
  }
  template <typename DatabaseNameT = Aws::String>
  UpdatePartnerStatusRequest& WithDatabaseName(DatabaseNameT&& value) {
    SetDatabaseName(std::forward<DatabaseNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the partner whose integration status is being updated.</p>
   */
  inline const Aws::String& GetPartnerName() const { return m_partnerName; }
  inline bool PartnerNameHasBeenSet() const { return m_partnerNameHasBeenSet; }
  template <typename PartnerNameT = Aws::String>
  void SetPartnerName(PartnerNameT&& value) {
    m_partnerNameHasBeenSet = true;
    m_partnerName = std::forward<PartnerNameT>(value);
  }
  template <typename PartnerNameT = Aws::String>
  UpdatePartnerStatusRequest& WithPartnerName(PartnerNameT&& value) {
    SetPartnerName(std::forward<PartnerNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value of the updated status.</p>
   */
  inline PartnerIntegrationStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(PartnerIntegrationStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline UpdatePartnerStatusRequest& WithStatus(PartnerIntegrationStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status message provided by the partner.</p>
   */
  inline const Aws::String& GetStatusMessage() const { return m_statusMessage; }
  inline bool StatusMessageHasBeenSet() const { return m_statusMessageHasBeenSet; }
  template <typename StatusMessageT = Aws::String>
  void SetStatusMessage(StatusMessageT&& value) {
    m_statusMessageHasBeenSet = true;
    m_statusMessage = std::forward<StatusMessageT>(value);
  }
  template <typename StatusMessageT = Aws::String>
  UpdatePartnerStatusRequest& WithStatusMessage(StatusMessageT&& value) {
    SetStatusMessage(std::forward<StatusMessageT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_accountId;

  Aws::String m_clusterIdentifier;

  Aws::String m_databaseName;

  Aws::String m_partnerName;

  PartnerIntegrationStatus m_status{PartnerIntegrationStatus::NOT_SET};

  Aws::String m_statusMessage;
  bool m_accountIdHasBeenSet = false;
  bool m_clusterIdentifierHasBeenSet = false;
  bool m_databaseNameHasBeenSet = false;
  bool m_partnerNameHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_statusMessageHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
