using ..DelaunayTriangulation
const DT = DelaunayTriangulation
using LinearAlgebra
using AdaptivePredicates
using CairoMakie
using BenchmarkTools
using StableRNGs

@testset "is_true" begin
    @test DT.is_true(true)
    @test DT.is_true(Val(true))
    @test !DT.is_true(false)
    @test !DT.is_true(Val(false))
end

@testset "number_type" begin
    @test DT.number_type([1, 2, 3]) == Int
    @test DT.number_type([1.0, 2.0, 3.0]) == Float64
    @test DT.number_type([1.0 2.0; 3.0 3.5; 10.0 17.3]) == Float64
    @test DT.number_type((1.0, 5.0)) == Float64
    @test DT.number_type([(1.0f0, 2.0f0), (1.7f0, 2.5f0)]) == Float32
    @test DT.number_type(2.4) == Float64
    @test DT.number_type(((1.0, 2.0), (2.0, 3.0))) == Float64
    @test DT.number_type(((1.0f0, 2.0f0), (2.0f0, 3.0f0))) == Float32
end

@testset "get_ghost_vertex" begin
    @test DT.get_ghost_vertex(1, 2, -3) == -3
    @test DT.get_ghost_vertex(1, 2, -1) == -1
    @test DT.get_ghost_vertex(1, -5, 2) == -5
    @test DT.get_ghost_vertex(-1, 2, 3) == -1
    @test DT.get_ghost_vertex(2, 5, 7) == 7
    @test DT.get_ghost_vertex(1, -2) == -2
    @test DT.get_ghost_vertex(-5, 1) == -5
    @test DT.get_ghost_vertex(2, 5) == 5
end

@testset "sort_triangle" begin
    @test DT.sort_triangle(1, 2, DT.𝒢) == (1, 2, DT.𝒢)
    @test DT.sort_triangle(DT.𝒢 - 2, 2, 3) == (2, 3, DT.𝒢 - 2)
    @test DT.sort_triangle(5, DT.𝒢 - 1, 3) == (3, 5, DT.𝒢 - 1)
    @test DT.sort_triangle((1, 5, DT.𝒢)) == (1, 5, DT.𝒢)
    @test DT.sort_triangle([1, DT.𝒢 - 2, 7]) == [7, 1, DT.𝒢 - 2]
    @test DT.sort_triangle((DT.𝒢 - 10, 5, 3)) == (5, 3, DT.𝒢 - 10)
end

const _POINTS = [
    ## Don't know why the points on 1.10 get slightly changed, e.g. with 357 replacing 363 and 363 replacing 357, so this is needed to make the next set of tests more robust. This is NOT some hack to force the test to pass when it shouldn't, it just makes sure the point order matches the hard-coded node numbers below.
    0.0 0.0
    0.6666666666666666 0.0
    1.3333333333333333 0.0
    2.0 0.0
    2.0 2.0
    2.0 4.0
    2.0 6.0
    1.3333333333333335 6.0
    0.6666666666666667 6.0
    0.0 6.0
    0.0 4.0
    0.0 2.0
    1.1 0.5999999999999999
    1.095895006911623 0.5360614191577466
    1.0836474315195146 0.47317270804524625
    1.063458378673011 0.41236649756031307
    1.0356593520616948 0.35464122399803105
    1.0007068109339783 0.3009447347543919
    0.9591746750488634 0.2521587246982565
    0.9117449009293667 0.20908425876598502
    0.8591962841552626 0.172428618497327
    0.8023916715611968 0.14279368849209373
    0.742263793315516 0.12066607348166958
    0.6797999475166896 0.10640910829277489
    0.6160257887858274 0.10025689189965603
    0.5519884870461588 0.10231044352540092
    0.4887395330218427 0.11253604390908817
    0.4273174727893459 0.13076578897511987
    0.36873085487958235 0.15670034681349998
    0.31394166993891537 0.18991387270152188
    0.26384955486934164 0.22986100146234217
    0.21927702081543265 0.2758858023461059
    0.18095594755407962 0.32723254939472574
    0.1495155660487904 0.38305813044122095
    0.12547212649466555 0.44244589098818987
    0.10922042150446726 0.5044206856493141
    0.10102730362483181 0.5679648900096435
    0.10102730362483181 0.6320351099903566
    0.10922042150446731 0.6955793143506862
    0.1254721264946656 0.7575541090118102
    0.14951556604879046 0.8169418695587791
    0.18095594755407968 0.8727674506052743
    0.21927702081543277 0.9241141976538942
    0.2638495548693417 0.9701389985376578
    0.3139416699389151 1.0100861272984778
    0.36873085487958224 1.0432996531865
    0.42731747278934623 1.0692342110248803
    0.4887395330218428 1.0874639560909118
    0.5519884870461592 1.0976895564745992
    0.6160257887858274 1.099743108100344
    0.6797999475166895 1.093590891707225
    0.7422637933155162 1.0793339265183302
    0.802391671561197 1.0572063115079062
    0.8591962841552626 1.027571381502673
    0.9117449009293666 0.990915741234015
    0.9591746750488637 0.9478412753017432
    1.0007068109339783 0.899055265245608
    1.0356593520616948 0.8453587760019688
    1.063458378673011 0.7876335024396869
    1.0836474315195146 0.7268272919547538
    1.095895006911623 0.6639385808422531
    1.7 0.49999999999999994
    1.6983580027646492 0.47442456766309865
    1.693458972607806 0.4492690832180985
    1.6853833514692045 0.42494659902412524
    1.6742637408246779 0.40185648959921244
    1.6602827243735914 0.38037789390175675
    1.6436698700195453 0.3608634898793026
    1.6246979603717466 0.343633703506394
    1.603678513662105 0.3289714473989308
    1.5809566686244787 0.3171174753968375
    1.5569055173262063 0.3082664293926678
    1.5319199790066758 0.30256364331710994
    1.506410315514331 0.3001027567598624
    1.4807953948184636 0.30092417741016037
    1.455495813208737 0.30501441756363523
    1.4309269891157383 0.31230631559004796
    1.407492341951833 0.3226801387254
    1.3855766679755661 0.3359655490806087
    1.3655398219477366 0.35194440058493687
    1.347710808326173 0.37035432093844234
    1.3323823790216318 0.3908930197578903
    1.3198062264195163 0.41322325217648836
    1.3101888505978663 0.43697835639527594
    1.303688168601787 0.4617682742597256
    1.3004109214499326 0.4871859560038574
    1.3004109214499326 0.5128140439961426
    1.303688168601787 0.5382317257402746
    1.3101888505978663 0.5630216436047241
    1.3198062264195163 0.5867767478235116
    1.3323823790216318 0.6091069802421097
    1.347710808326173 0.6296456790615577
    1.3655398219477366 0.6480555994150632
    1.3855766679755661 0.6640344509193912
    1.4074923419518328 0.6773198612746
    1.4309269891157386 0.6876936844099522
    1.4554958132087372 0.6949855824363648
    1.4807953948184638 0.6990758225898397
    1.506410315514331 0.6998972432401376
    1.5319199790066758 0.6974363566828901
    1.5569055173262065 0.6917335706073321
    1.5809566686244787 0.6828825246031625
    1.603678513662105 0.6710285526010692
    1.6246979603717466 0.656366296493606
    1.6436698700195456 0.6391365101206973
    1.6602827243735914 0.6196221060982432
    1.6742637408246779 0.5981435104007875
    1.6853833514692043 0.5750534009758748
    1.693458972607806 0.5507309167819016
    1.6983580027646492 0.5255754323369012
    1.0 2.0
    1.0 3.0
    1.0 4.000000000000001
    1.0 5.0
    1.1666666666666665 5.0
    1.3333333333333333 5.0
    1.5 5.0
    1.5 4.0
    1.5 3.0
    1.5 2.0
    1.3333333333333335 2.0
    1.1666666666666667 2.0
    0.2 2.0
    0.2 5.0
    0.75 4.0
    0.75 3.0
    0.5 2.0
    0.2 3.5
    2.0 5.0
    0.0 3.0
    2.0 3.0
    2.0 1.0
    0.0 5.0
    0.0 1.0
    0.2 4.5
    0.2 3.0
    0.2 4.0
    0.0 4.5
    0.0 0.5
    2.0 0.5
    0.0 3.5
    1.6666666666666665 0.0
    1.0 0.0
    0.3333333333333333 0.0
    0.0 0.25
    0.0 0.75
    0.7916666666666666 0.0
    0.5416666666666666 0.0
    0.0 0.625
    1.4740544522373191 1.309077721762752
    1.3499698939387508 1.0055012112239021
    1.6795817213270934 0.9879815502040521
    2.0 0.75
    2.0 0.25
    1.7828347010132077 0.1885675610394171
    1.19953813886343 0.20901891414515464
    1.7742377105455027 0.8019661970196654
    0.0 1.5
    0.9388315428915 1.5207147605911762
    1.2572693115310343 0.8043746560736462
    1.5416666666666665 0.0
    1.4877125131091204 0.8831690890011553
    1.8624474074396582 0.6203408303941387
    1.862447407439658 0.37965916960586316
    0.4487051545335375 1.535812115275885
    1.441199151969013 0.13629596364476348
    1.624263604487171 0.16233564360848637
    1.6185917257370095 0.8222520295432142
    1.1917579283891429 0.37520661583202014
    1.3293751090374275 0.21853639574589828
    1.2191572487719817 0.6581676907984143
    1.0529166570030704 1.2492900864648844
    0.0 1.25
    1.3724252107559454 0.7881936711216704
    1.7396173332189573 0.2960107901378373
    1.8110061512826319 0.5199672573998425
    1.761187193038501 0.6700140220581073
    0.6745044586363241 1.372316652669158
    0.28592779118073197 1.3094945905319997
    1.1953751211197607 0.5
    1.5289331661582095 0.2000774658667932
    1.6681251875542065 0.7410200991200366
    1.528217827526302 0.7925073008997648
    1.775506905040127 0.40852540520001573
    1.26831267650853 0.3270872565921958
    1.418160934512214 0.22425703203449873
    1.295455646032882 0.6980901602542381
    1.6315588938199306 0.24782623088951797
    1.137591811617686 1.057658581735622
    1.0833333333333333 5.527777777777778
    1.7661043175005355 0.5883527206490259
    0.48832789133187804 1.290731478047606
    0.8416520629881481 1.2566467201424116
    0.16692117116205538 1.1430636744441975
    1.2314189884870388 0.5704302352100531
    1.4563071262047091 0.7702558737066645
    1.2378485908218748 0.4312558050732817
    1.5935642184148526 0.7542442070846888
    1.3320629484886526 0.2894135981807518
    0.125 0.0
    0.0 2.5
    0.2 2.5
    0.0 5.5
    1.363898107854514 0.7245143067518455
    1.6763532155896466 0.3058155104275537
    1.7198413454244592 0.35689914227621217
    1.7603049822572803 0.48328786565231235
    1.6986162550807449 0.6690844831441323
    1.2732086411580512 0.6277265137313147
    0.1357338269089258 0.15038328438526705
    0.9681905883861459 1.127828895829673
    1.5569357954505727 0.2505479812612332
    1.4917990544438116 0.2442643181947033
    1.4166666666666665 5.362573099415204
    0.6610307417000465 1.2326474818074256
    0.3427260032157782 1.181186440179403
    1.4166666666666667 1.6571810441100427
    1.0238883307733782 0.13325183852345296
    1.2486746582268176 0.5
    1.0 3.5000000000000004
    0.75 3.5
    0.47500000000000003 4.5
    1.0 4.5
    0.625 2.5
    1.0 2.5
    1.2821205078623963 0.3772925560903774
    1.3990607282139222 0.271976426778746
    1.1173493764498315 0.9367571253255303
    0.0 0.375
    1.741444624549862 0.5470213364192468
    1.7216209675111749 0.6067270329489496
    1.640658487617645 0.7016445193064772
    1.523607866265877 0.7447202299680109
    0.0 0.875
    0.16036607072667938 1.0257617177590372
    1.731605030618596 0.4231018317802659
    1.4308793419171846 0.7328904331187506
    1.1070433482398665 0.2699513556188709
    0.875 0.0
    1.326733072952154 0.6677996614425488
    1.611490638036034 0.2862933200657552
    1.3269281024201038 0.33238921406445593
    1.0833333333333333 5.257309941520468
    0.5647973445143255 5.392543859649123
    0.5047539390232997 1.1891305650202075
    0.8061070342543368 1.1600593943533914
    1.0833333333333335 1.7458187278913946
    0.4583333333333333 0.0
    1.6810772213147687 0.34584671399258676
    0.0820730652316044 0.30830898459576883
    0.29187380922086326 0.09171361966873727
    1.2701959364187478 0.5602617219999282
    1.1611993909112273 0.7863310354349395
    0.9961530910081264 1.036208580108681
    1.462487223491946 0.2679702132241154
    1.5225695030046766 0.2660434914632048
    0.2962726981257435 1.1010299529782763
    1.2735053984345062 0.44060612113937503
    1.5808501167410969 0.7196958855831329
    0.09063815833613441 0.886867244835494
    0.9340092492886052 0.1211726241519101
    1.3795327553781789 0.3012766796536391
    0.0 3.75
    0.2 3.75
    0.2 2.25
    0.0 2.25
    0.0 4.25
    0.2 4.25
    0.2 4.75
    0.0 4.75
    0.0 2.75
    0.2 2.75
    0.0 5.25
    0.0 3.25
    0.2 3.25
    0.0 1.75
    1.7305994280094303 0.5148049744879046
    1.6936605339465862 0.6260590379916767
    1.7193022924897394 0.5728133777310767
    1.4925952162260299 0.7309084256300289
    1.6553197408864884 0.6710242963459636
    2.0 1.5
    1.7261014731069777 0.45596674205660387
    1.7075375557401247 0.4000551807062032
    1.406921135389704 0.7102667764910945
    1.0370994517765144 0.22789181149262006
    1.125 0.0
    1.3570113367409984 0.6793021124621966
    1.316206997826056 0.6371682824339902
    1.6311751370246665 0.31195057690225014
    1.5791854380321182 0.284827584238829
    1.316282242285531 0.36288787396558325
    1.1154407188558464 0.3517768322991175
    0.5816906605550696 1.1709526266542052
    0.864223205343458 1.1064664170034129
    0.4374647190632682 1.1476352949226916
    0.7271142209552935 1.1569237906798149
    0.1428028406433256 0.25878489202052457
    0.3690795881569608 0.07834676725460561
    0.2443121381562888 0.15398152865993447
    0.25 0.0
    1.2737384055431993 0.5291732020831812
    1.2885382384146338 0.5856113791101021
    1.113130134633655 0.847110448984858
    1.1590288414186054 0.7088708571143922
    1.033197322819307 0.9687862576934944
    1.4353052654987943 0.28202197206461327
    0.24588090864430434 1.044051295369198
    0.2688387528122659 5.375
    0.33333333333333337 6.0
    1.2894711087692583 0.4147662982391812
    1.2747365658359935 0.47095549643502305
    1.6050489090123727 0.701359091410823
    1.550537613507314 0.7214197520535767
    0.07387658485528542 0.8130037639664341
    0.14511290432603197 0.939491062646291
    0.7291666666666666 0.0
    0.8624443884933936 0.09694324186960815
    0.07431874516150205 0.3871752469020991
    1.3586733685981895 0.32278200956665326
    0.7743125163556892 1.7852573023241183
    1.3951319674273832 0.7467489457540966
    1.6723555903905802 0.6467284533501322
    1.4638740821559169 0.7234515755624243
    1.4927477124566209 0.2738469683956681
    1.6895818983651612 0.3765958595614554
    1.725742831711937 0.4855068293396377
    1.3828391994677471 0.6932689949811435
    1.5502891882510008 0.2796686701104153
    1.6519383611714682 0.33269897851362074
    1.4166666666666665 5.171709936804376
    2.0 5.5
    1.7470040926571286 5.26714151810979
    1.762711901635018 0.44520425603505154
    1.8366469302468644 0.45042453510946423
    1.9060886700063842 0.5155263895621286
    1.7432094576086072 0.3868933176340165
    1.7985506191032619 0.3391937098408574
    1.8878694231956714 0.26883597168004963
    1.7916666666666665 0.0
    2.0 0.125
    0.9230460198845826 1.0631107620790716
    1.1358084873427654 0.4220993253764972
    0.19447305594557093 0.20726905545353125
    1.3032549844320298 0.6108047284553858
    2.0 0.625
    1.2500000000000002 1.8319281123363502
    0.6541111351264756 1.160918521091478
    0.371895480593367 1.115292082681653
    1.494273560202694 0.7688633005731953
    1.4773323581048208 0.8252320844062707
    1.4030882061420913 0.8684322443508078
    0.7944598296275528 0.07158982321352167
    1.0 3.25
    0.75 3.25
    0.595479687443039 4.280946022830838
    1.0 4.25
    2.0 4.5
    1.5 4.5
    2.0 2.5
    1.5 2.5
    2.0 3.5
    1.5 3.5
    0.6893660937409167 2.757464374963667
    1.0 2.75
    1.0 3.750000000000001
    0.75 3.75
    0.20607015431451464 0.07547310822581126
    1.1137966762594949 0.1795335656241746
    1.2083333333333333 0.0
    1.1666666666666665 0.09271643952630988
    0.3994151065612831 5.677353293275649
    1.6783486674776207 0.70087084995281
    1.7392543857928202 0.7377559547642759
    1.8302710234553803 0.7298028546873176
    2.0 0.875
    1.8888519575683367 0.8331257280027056
    1.800336932191529 0.9323381323744324
    2.0 1.25
    1.8227502338782238 1.1398294920180019
    1.7994421968210412 1.375
    1.7395586704615544 1.6944745976100235
    1.7267593429098054 0.6440448038896638
    1.7678827403626947 0.5201876828851476
    1.2035918506774017 0.8877754751066075
    1.16093306626047 0.5639869022785741
    0.0 0.4375
    1.291551342829673 0.6627651416324978
    1.2538080175158937 0.6848149997324289
    1.2577128301399019 0.7445318850453346
    1.2090408575917337 0.7071751969044885
    1.3306373761553634 0.7031246800642893
    1.3160528628068418 0.7623584294631466
    1.6462859848898292 0.27997768754971447
    1.7020584160349093 0.234982296630739
    1.5967635469661356 0.2541418634055588
    1.6061662460929913 0.20693653438447443
    1.570825246339907 0.16662443239081526
    1.5905483961970126 0.07730943253130455
    1.51018175705595 0.13363920248553696
    1.4583333333333333 0.0
    1.6590520810618847 0.20242355254666858
    1.7125852936455277 0.12082842146382869
    1.431068012499545 0.9581022021491937
    1.477452094723081 1.130541403025688
    1.5531080005501499 0.9914800857679006
    1.2813569642026876 1.216110095730245
    1.2018317226108364 1.4715704518745842
    1.6198909607221894 1.1179240078762014
    1.7983548141425743 0.46657478567124433
    0.9770513066521348 0.18482460736504314
    0.7938123559004582 5.2089259691333805
    0.6407123629255758 4.851627018928146
    1.319484941905665 0.8292893422114065
    1.2761296818296555 0.5
    1.8099178972915033 0.26187472706351006
    1.7111729419512756 0.17826049305063374
    1.759634673818009 0.34999662375758167
    1.718142438071527 0.703889427730984
    1.473230604388981 0.071097758315114
    1.3863435254451124 0.06887822829586042
    1.2782854698688741 0.09720405683617961
    0.7375 4.501716097368583
    0.44121509031025474 0.06500095027943562
    1.305524838337339 0.3904736294480411
    1.6684438208715549 0.25037958358667844
    1.6996902177676332 0.2742036674155371
    1.3624484903113168 0.2687327852709218
    0.06974484917671708 0.9630027278134988
    0.0 1.125
    0.07352010305180143 1.054357451563032
    1.1563399977503028 0.6357182129391813
    0.5597884117036155 1.2501563862805738
    0.5715382250636158 1.354065450073582
    0.5980032696789671 1.504333348664305
    0.5656039401938803 1.7204555756412738
    0.3037866871454204 1.7568871511798612
    0.46161571048938355 1.4121561268225087
    0.3161306261019238 1.459468430239305
    0.149758655496108 1.4149454869965006
    0.16391023227107437 1.6142327799518283
    0.7683057766893729 1.5148465969437765
    0.7637530559523047 1.230480162353276
    0.7720830332001749 1.3347191336230664
    0.9050972636859582 1.3832137934898427
    1.0559347141051296 1.4190962037054105
    0.43329341865156223 1.2297056567872398
    0.3893995908276335 1.3246186881965123
    0.8840060412901801 1.1862259521425271
    0.9479694782050885 1.272643328918064
    1.4361522732329282 0.25034394087790574
    1.4603794219490087 0.21438773060391503
    1.4301726911242494 0.18040552911767954
    1.3664948543862412 0.14105443922716654
    1.3800869372880815 0.19025696838292824
    1.4771932167810962 0.16902551127432658
    1.40980282641639 0.2962430137342318
    0.0 0.6875
    2.0 0.375
    1.9132736730754663 0.4382760179317226
    0.055964514785385895 0.46875
    1.6666208006494743 5.608342938440153
    1.2445796724660527 0.5415967162439137
    1.2035138373071206 0.5402730217783609
    1.257035252439734 0.5890952409660676
    1.2196380370307442 0.6135093363907388
    1.383611268856942 0.23811038285103012
    1.6226023422523703 0.7328606305005508
    1.6335516540160544 0.7781375493960542
    1.698042873799217 0.824601373114044
    1.6559767278839401 0.902581204581636
    1.5575918148450751 0.7567851055371636
    1.5786042901358082 0.7983586872318487
    1.561956830704648 0.8716273239716548
    1.25 5.146519289061359
    1.4166666666666667 1.853986608235224
    1.6106377773288556 1.84006951687984
    1.75 2.0547125573551916
    1.5 2.25
    1.75 2.375
    1.75 2.6875
    1.7351882330544273 0.25666486669089317
    1.740696001877012 0.21209770142048218
    1.7716317714671403 0.27248473247545524
    1.780747250326824 0.23099526710030466
    1.8315421621777004 0.21222918129712579
    1.8571475226715097 0.09755754869299872
    1.9212830619181216 0.17925938128868346
    1.8307671503240839 0.15186156285696936
    1.782563159254053 0.09488484980088335
    1.7291666666666665 0.0
    1.6979166666666665 0.05726348727905952
    0.0635245517925628 0.7406804293007325
]
_NODES1 = [
    [1, 200, 301, 144, 248, 148, 2, 317, 147, 239, 143, 287, 370, 3, 401, 161, 142, 491, 340, 4],
    [4, 341, 154, 459, 140, 346, 153, 376, 132, 379, 282, 5, 360, 131, 362, 6, 358, 129, 332, 7],
    [7, 8, 9, 310, 10],
    [10 203 273 133 270 138 267 11 263 141 274 130 271 201 266 12 276 158 173 430 134 234 146 458 149 139 387 229 145 1][:],
]
_NODES2 = [
    [13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 57 58 59 60 61 13][:],
]
_NODES3 = [
    [62 63 64 65 66 67 68 69 70 71 72 73 74 75 76 77 78 79 80 81 82 83 84 85 86 87 88 89 90 91 92 93 94 95 96 97 98 99 100 101 102 103 104 105 106 107 108 109 110 62][:],
]
_NODES4 = [
    [111, 225, 365, 112, 354, 220, 366, 113, 357, 223, 114][:],
    [114, 115, 116, 117][:],
    [117, 359, 118, 363, 119, 361, 479, 120][:],
    [120, 121, 122, 111][:],
]
_NODES5 = [
    [123 265 202 272 136 275 128 264 137 268 135 269 124 222 356 125 367 221 355 126 364 224 127 123][:],
]
const __BOUNDARY_NODES = [
    _NODES1, _NODES2, _NODES3, _NODES4, _NODES5,
]

@testset "get_left/right_boundary_node" begin
    rng = StableRNG(1234555)
    #=
      _x, _y = complicated_geometry()
      x = _x
      y = _y
      boundary_nodes, points = convert_boundary_points_to_indices(x, y)
      tri = triangulate(points; boundary_nodes, delete_ghosts=false, rng)
      A = get_area(tri)
      refine!(tri; max_area=1e-1A, rng)
      =#
    points = tuple.(_POINTS[:, 1], _POINTS[:, 2])
    tri = triangulate(points; boundary_nodes=__BOUNDARY_NODES, rng, delete_ghosts=false)
    nodes = [1, 200, 301, 144, 248, 148, 2, 317, 147, 239, 143, 287, 370]
    right = [200, 301, 144, 248, 148, 2, 317, 147, 239, 143, 287, 370, 3]
    left = [145, 1, 200, 301, 144, 248, 148, 2, 317, 147, 239, 143, 287]
    @inferred DT.get_right_boundary_node(tri, 1, DT.𝒢)
    @inferred DT.get_left_boundary_node(tri, 1, DT.𝒢)

    for (i, r, ℓ) in zip(nodes, right, left)
        DT.get_right_boundary_node(tri, i, DT.𝒢) == r
        DT.get_left_boundary_node(tri, i, DT.𝒢) == ℓ
        @inferred DT.get_left_boundary_node(tri, i, DT.𝒢)
    end
    nodes = [13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 57 58 59 60 61]
    right = [nodes[2:end]..., nodes[1]]
    left = [nodes[end]..., nodes[1:(end-1)]...]
    for (i, r, ℓ) in zip(nodes, right, left)
        @test DT.get_right_boundary_node(tri, i, DT.𝒢 - 4) == r
        @test DT.get_left_boundary_node(tri, i, DT.𝒢 - 4) == ℓ
    end
    nodes = [62 63 64 65 66 67 68 69 70 71 72 73 74 75 76 77 78 79 80 81 82 83 84 85 86 87 88 89 90 91 92 93 94 95 96 97 98 99 100 101 102 103 104 105 106 107 108 109 110]
    right = [nodes[2:end]..., nodes[1]]
    left = [nodes[end]..., nodes[1:(end-1)]...]
    for (i, r, ℓ) in zip(nodes, right, left)
        @test DT.get_right_boundary_node(tri, i, DT.𝒢 - 5) == r
        @test DT.get_left_boundary_node(tri, i, DT.𝒢 - 5) == ℓ
    end
    nodes = [111 225 365 112 354 220 366 113 357 223 114 115 116 117 359 118 363 119 361 479 120 121 122]
    right = [225 365 112 354 220 366 113 357 223 114 115 116 117 359 118 363 119 361 479 120 121 122 111]
    left = [122 111 225 365 112 354 220 366 113 357 223 114 115 116 117 359 118 363 119 361 479 120 121]
    for (i, r, ℓ) in zip(nodes, right, left)
        @test DT.get_right_boundary_node(tri, i, DT.𝒢 - 7) == r
        @test DT.get_left_boundary_node(tri, i, DT.𝒢 - 8) == ℓ
    end
    nodes = [123, 265, 202, 272, 136, 275, 128, 264]
    right = [265, 202, 272, 136, 275, 128, 264, 137]
    left = [127, 123, 265, 202, 272, 136, 275, 128]
    for (i, r, ℓ) in zip(nodes, right, left)
        @test DT.get_right_boundary_node(tri, i, DT.𝒢 - 10) == r
        @test DT.get_left_boundary_node(tri, i, DT.𝒢 - 10) == ℓ
    end
end

@testset "find_edge" begin
    for PT in subtypes(DT.AbstractPredicateKernel)
        points = [(0.0, 0.0), (1.0, 0.0), (0.0, 1.0), (0.5, 0.0), (0.5, 0.5), (0.0, 0.5)]
        tri = triangulate(points, randomise=false)
        T = (1, 2, 3)
        ℓ = 4
        @test DT.find_edge(PT(), tri, T, ℓ) == (1, 2)
        T = (2, 3, 1)
        @test DT.find_edge(tri, T, ℓ) == (1, 2)
        T = (1, 2, 3)
        ℓ = 5
        @test DT.find_edge(PT(), tri, T, ℓ) == (2, 3)
        T = (2, 3, 1)
        @test DT.find_edge(tri, T, ℓ) == (2, 3)
        T = (1, 2, 3)
        ℓ = 6
        @test DT.find_edge(PT(), tri, T, ℓ) == (3, 1)
        T = (2, 3, 1)
        @test DT.find_edge(tri, T, ℓ) == (3, 1)
        p1 = [2.0, 3.5]
        p2 = [0.0, 0.0]
        p3 = [3.0, 0.0]
        p4 = [17.2, -2.5]
        p5 = [0.0, 3.0]
        points = [p1, p2, p3, p4, p5]
        tri = triangulate(points, randomise=false)
        T = (2, 3, 5)
        push!(points, [1.0, 0.0])
        @test DT.find_edge(PT(), tri, T, length(points)) == (2, 3)
        push!(points, [2.0, 0.0])
        @test DT.find_edge(PT(), tri, T, length(points)) == (2, 3)
        push!(points, [1.5, 0.0])
        @test DT.find_edge(tri, T, length(points)) == (2, 3)
        push!(points, [1.0, 0.0])
        @test DT.find_edge(tri, T, length(points)) == (2, 3)
        push!(points, [0.5, 0.0])
        @test DT.find_edge(tri, T, length(points)) == (2, 3)
        push!(points, [2.5, 0.5])
        @test DT.find_edge(tri, T, length(points)) == (3, 5)
        push!(points, [2.0, 1.0])
        @test DT.find_edge(tri, T, length(points)) == (3, 5)
        push!(points, [1.5, 1.5])
        @test DT.find_edge(tri, T, length(points)) == (3, 5)
        push!(points, [1.0, 2.0])
        @test DT.find_edge(PT(), tri, T, length(points)) == (3, 5)
        push!(points, [0.5, 2.5])
        @test DT.find_edge(PT(), tri, T, length(points)) == (3, 5)
        push!(points, [0.0, 2.5])
        @test DT.find_edge(PT(), tri, T, length(points)) == (5, 2)
        push!(points, [0.0, 2.2])
        @test DT.find_edge(tri, T, length(points)) == (5, 2)
        push!(points, [0.0, 2.0])
        @test DT.find_edge(tri, T, length(points)) == (5, 2)
        push!(points, [0.0, 1.5])
        @test DT.find_edge(tri, T, length(points)) == (5, 2)
        push!(points, [0.0, 0.8])
        @test DT.find_edge(tri, T, length(points)) == (5, 2)
        push!(points, [0.0, 0.2])
        @test DT.find_edge(PT(), tri, T, length(points)) == (5, 2)
    end
end

@testset "choose_uvw" begin
    i, j, k = rand(Int, 3)
    @test DT.choose_uvw(true, false, false, i, j, k) == (i, j, k)
    @test DT.choose_uvw(false, true, false, i, j, k) == (j, k, i)
    @test DT.choose_uvw(false, false, true, i, j, k) == (k, i, j)
end

@testset "is_circular" begin
    x = rand(10)
    @test !DT.is_circular(x)
    push!(x, x[begin])
    @test DT.is_circular(x)
    @test DT.is_circular(Float64[])
end

@testset "circular_equality" begin
    @test !DT.circular_equality([1, 2, 3, 4, 1], [3, 2, 4, 1, 3])
    @test !DT.circular_equality([1, 2, 3, 1], [3, 4, 5, 3])
    @test_throws AssertionError DT.circular_equality([1, 2, 3, 1], [3, 4, 5])
    @test_throws AssertionError DT.circular_equality([1, 2, 3], [5, 4, 3])
    @test !DT.circular_equality([1, 2, 3, 4, 1], [1, 2, 1])
    x = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1]
    for i in 1:10
        local y
        y = @views circshift(x[begin:(end-1)], i)
        push!(y, y[begin])
        @test DT.circular_equality(x, y) && DT.circular_equality(y, x)
    end
    @test DT.circular_equality([3, 2, 1, 13, 12, 11, 5, 4, 3], [1, 13, 12, 11, 5, 4, 3, 2, 1])
    @test DT.circular_equality(Float64[], Float64[])
end

@testset "get_surrounding_polygon" begin
    tri = example_triangulation()
    rng = StableRNG(999987897899)
    tri = triangulate(get_points(tri); delete_ghosts=false, rng)
    polys = Dict(
        1 => [6, 3, 7, 4, 6],
        2 => [8, 3, 6, DT.𝒢, 8],
        3 => [8, 7, 1, 6, 2, 8],
        4 => [6, 1, 7, 5, DT.𝒢, 6],
        5 => [4, 7, 8, DT.𝒢, 4],
        6 => [2, 3, 1, 4, DT.𝒢, 2],
        7 => [8, 5, 4, 1, 3, 8],
        8 => [5, 7, 3, 2, DT.𝒢, 5],
        DT.𝒢 => [5, 8, 2, 6, 4, 5],
    )
    fnc_polys = Dict{Int,Vector{Int}}()
    for i in keys(polys)
        fnc_polys[i] = DT.get_surrounding_polygon(tri, i)
        push!(fnc_polys[i], fnc_polys[i][begin])
    end
    for (poly_true, poly_f) in zip(values(polys), values(fnc_polys))
        @test DT.circular_equality(poly_true, poly_f)
    end
    fnc_polys = Dict{Int,Vector{Int}}()
    for i in keys(polys)
        fnc_polys[i] = DT.get_surrounding_polygon(tri, i; skip_ghost_vertices=true)
        push!(fnc_polys[i], fnc_polys[i][begin])
    end
    for (poly_true, poly_f) in zip(values(polys), values(fnc_polys))
        @test DT.circular_equality(filter(!DT.is_ghost_vertex, poly_true), poly_f)
    end
    tri, label_map, index_map = simple_geometry()
    DT.compute_representative_points!(tri)
    add_ghost_triangles!(tri)
    polys = Dict(
        1 => [[2, 20, 19, 8, DT.𝒢, 2]],
        2 => [[3, 16, 20, 1, DT.𝒢, 3]],
        3 => [[4, 17, 16, 2, DT.𝒢, 4]],
        4 => [[5, 18, 17, 3, DT.𝒢, 5]],
        5 => [[6, 22, 24, 18, 4, DT.𝒢, 6]],
        6 => [[7, 25, 9, 12, 23, 22, 5, DT.𝒢, 7]],
        7 => [[8, 26, 9, 25, 6, DT.𝒢, 8]],
        8 => [[1, 19, 10, 26, 7, DT.𝒢, 1]],
        9 => [[6, 25, 7, 26, 10, DT.𝒢 - 1, 12, 6]],
        10 => [[11, DT.𝒢 - 1, 9, 26, 8, 19, 20, 21, 11]],
        11 => [[12, DT.𝒢 - 1, 10, 21, 23, 12]],
        12 => [[23, 6, 9, DT.𝒢 - 1, 11, 23]],
        13 => [
            [18, 24, 22, 23, 21, 14, DT.𝒢 - 2, 18],
            [18, 24, 22, 23, 21, 14, DT.𝒢 - 3, 18],
        ],
        14 => [
            [13, 21, 15, DT.𝒢 - 2, 13],
            [13, 21, 15, DT.𝒢 - 3, 13],
        ],
        15 => [
            [14, 21, 20, 16, DT.𝒢 - 2, 14],
            [14, 21, 20, 16, DT.𝒢 - 3, 14],
        ],
        16 => [
            [15, 20, 2, 3, 17, DT.𝒢 - 2, 15],
            [15, 20, 2, 3, 17, DT.𝒢 - 3, 15],
        ],
        17 => [
            [16, 3, 4, 18, DT.𝒢 - 2, 16],
            [16, 3, 4, 18, DT.𝒢 - 3, 16],
        ],
        18 => [
            [17, 4, 5, 24, 13, DT.𝒢 - 2, 17],
            [17, 4, 5, 24, 13, DT.𝒢 - 3, 17],
        ],
        19 => [[1, 20, 10, 8, 1]],
        20 => [[16, 15, 21, 10, 19, 1, 2, 16]],
        21 => [[15, 14, 13, 23, 11, 10, 20, 15]],
        22 => [[24, 5, 6, 23, 13, 24]],
        23 => [[13, 22, 6, 12, 11, 21, 13]],
        24 => [[18, 5, 22, 13, 18]],
        25 => [[9, 6, 7, 9]],
        26 => [[10, 9, 7, 8, 10]],
        DT.𝒢 => [[5, 4, 3, 2, 1, 8, 7, 6, 5]],
        DT.𝒢 - 1 => [[9, 10, 11, 12, 9]],
        DT.𝒢 - 2 => [[14, 15, 16, 17, 18, 13, 14]],
        DT.𝒢 - 3 => [[14, 15, 16, 17, 18, 13, 14]],
    )
    fnc_polys = Dict{Int,Vector{Int}}()
    for i in keys(polys)
        fnc_polys[i] = DT.get_surrounding_polygon(tri, i)
        push!(fnc_polys[i], fnc_polys[i][begin])
    end
    for (poly_true, poly_f) in zip(values(polys), values(fnc_polys))
        @test any(DT.circular_equality(S, poly_f) for S in poly_true)
    end
    fnc_polys = Dict{Int,Vector{Int}}()
    for i in keys(polys)
        fnc_polys[i] = DT.get_surrounding_polygon(tri, i; skip_ghost_vertices=true)
        push!(fnc_polys[i], fnc_polys[i][begin])
    end
    for (poly_true, poly_f) in zip(values(polys), values(fnc_polys))
        @test any(DT.circular_equality(filter(!DT.is_ghost_vertex, S), poly_f) for S in poly_true)
    end

    tri2 = example_with_special_corners()
    polys = Dict(
        1 => [DT.𝒢, 5, 4, 16, 3, 2, DT.𝒢],
        2 => [DT.𝒢, 1, 3, DT.𝒢],
        3 => [2, 1, 16, 6, 7, DT.𝒢, 2],
        4 => [5, 16, 1, 5],
        5 => [13, 18, 16, 4, 1, DT.𝒢, 13],
        6 => [3, 16, 17, 7, 3],
        7 => [3, 6, 17, 15, 8, DT.𝒢, 3],
        8 => [7, 15, 9, DT.𝒢, 7],
        9 => [8, 15, 11, 10, DT.𝒢, 8],
        10 => [9, 11, 12, 13, DT.𝒢, 9],
        11 => [9, 15, 12, 10, 9],
        12 => [18, 13, 10, 11, 15, 14, 18],
        13 => [10, 12, 18, 5, DT.𝒢, 10],
        14 => [15, 17, 18, 12, 15],
        15 => [8, 7, 17, 14, 12, 11, 9, 8],
        16 => [4, 5, 18, 17, 6, 3, 1, 4],
        17 => [7, 6, 16, 18, 14, 15, 7],
        18 => [5, 13, 12, 14, 17, 16, 5],
    )
    for mmm in 1:1000
        local tri, fnc_polys
        tri = triangulate(get_points(tri2); delete_ghosts=false)
        fnc_polys = Dict{Int,Vector{Int}}()
        for i in keys(polys)
            fnc_polys[i] = DT.get_surrounding_polygon(tri, i)
            push!(fnc_polys[i], fnc_polys[i][begin])
        end
        for (k, S) in polys
            @test DT.circular_equality(S, fnc_polys[k])
        end
        fnc_polys = Dict{Int,Vector{Int}}()
        for i in keys(polys)
            fnc_polys[i] = DT.get_surrounding_polygon(tri, i; skip_ghost_vertices=true)
            push!(fnc_polys[i], fnc_polys[i][begin])
        end
        for (k, S) in polys
            _S = filter(!DT.is_ghost_vertex, S)
            DT.is_ghost_vertex(S[begin]) && push!(_S, _S[begin]) # might have removed a boundary index in the first entry, so we'd no longer have a circular array
            @test DT.circular_equality(_S, fnc_polys[k])
        end
    end
end

@testset "sort_edge_by_degree" begin
    tri = triangulate(rand(2, 500); delete_ghosts=false)
    graph = get_graph(tri)
    for e in DT.get_edges(graph)
        new_e = DT.sort_edge_by_degree(tri, e)
        d1 = DT.num_neighbours(graph, e[1])
        d2 = DT.num_neighbours(graph, e[2])
        if d1 ≤ d2
            @test new_e == e
        else
            @test new_e == (e[2], e[1])
        end
    end
end

@testset "split_segment!" begin
    constrained_edges = Set{NTuple{2,Int}}(((2, 7),))
    DT.split_segment!(constrained_edges, (2, 7), [])
    @test constrained_edges == Set{NTuple{2,Int}}(((2, 7),))
    DT.split_segment!(constrained_edges, (2, 7), [(2, 3), (3, 5), (10, 12)])
    @test constrained_edges == Set{NTuple{2,Int}}(((2, 3), (3, 5), (10, 12)))
    DT.split_segment!(constrained_edges, (2, 7), [])
    @test constrained_edges == Set{NTuple{2,Int}}(((2, 3), (3, 5), (10, 12)))
    DT.split_segment!(constrained_edges, (3, 5), [(2, 10), (11, 15), (2, 3)])
    @test constrained_edges == Set{NTuple{2,Int}}(((2, 3), (2, 10), (11, 15), (10, 12)))
    DT.split_segment!(constrained_edges, (3, 2), [])
    @test constrained_edges == Set{NTuple{2,Int}}(((2, 3), (2, 10), (11, 15), (10, 12)))
    DT.split_segment!(constrained_edges, (3, 2), [(10, 2), (23, 10)])
    @test constrained_edges == Set{NTuple{2,Int}}(((11, 15), (10, 12), (23, 10), (2, 10)))
end

@testset "connect_segments!" begin
    C = [(7, 12), (12, 17), (17, 22), (32, 37), (37, 42), (42, 47)]
    DT.connect_segments!(C)
    @test C == [(7, 12), (12, 17), (17, 22), (22, 32), (32, 37), (37, 42), (42, 47)]
    C = [(4, 9), (19, 24), (24, 29), (34, 39), (39, 44), (44, 49)]
    DT.connect_segments!(C)
    @test C == [(4, 9), (9, 19), (19, 24), (24, 29), (29, 34), (34, 39), (39, 44), (44, 49)]
    C = [(4, 9), (9, 5)]
    DT.connect_segments!(C)
    @test C == [(4, 9), (9, 5)]
    C = [(49, 44), (44, 39), (39, 34), (29, 24), (24, 19), (9, 4)]
    DT.connect_segments!(C)
    @test C == [(49, 44), (44, 39), (39, 34), (34, 29), (29, 24), (24, 19), (19, 9), (9, 4)]
end

@testset "extend_segments!" begin
    segments = [(7, 12), (12, 17), (17, 22), (22, 27)]
    constrained_edge = (7, 27)
    DT.extend_segments!(segments, constrained_edge)
    @test segments == [(7, 12), (12, 17), (17, 22), (22, 27)]
    constrained_edge = (2, 32)
    DT.extend_segments!(segments, constrained_edge)
    @test segments == [(2, 7), (7, 12), (12, 17), (17, 22), (22, 27), (27, 32)]
    segments = [(33, 29)]
    constrained_edge = (37, 29)
    DT.extend_segments!(segments, constrained_edge)
    @test segments == [(37, 33), (33, 29)]
    segments = [(29, 33)]
    constrained_edge = (29, 37)
    DT.extend_segments!(segments, constrained_edge)
    @test segments == [(29, 33), (33, 37)]
    segments = [(3, 25), (25, 1)]
    DT.extend_segments!(segments, (3, 1))
    @test segments == [(3, 25), (25, 1)]
end

@testset "convert_boundary_points_to_indices" begin
    x = [[1.0, 2.0, 3.0, 4.0, 5.0], [5.0, 6.0, 7.0, 8.0], [8.0, 13.0, 15.0, 1.0]]
    y = [[0.0, 2.5, 3.0, 9.0, 7.0], [7.0, 9.0, 2.0, 1.0], [1.0, 23.0, 25.0, 0.0]]
    nodes, _pts = convert_boundary_points_to_indices(x, y)
    @test nodes == [[1, 2, 3, 4, 5], [5, 6, 7, 8], [8, 9, 10, 1]]
    @test _pts == [
        (1.0, 0.0), (2.0, 2.5), (3.0, 3.0), (4.0, 9.0), (5.0, 7.0), (6.0, 9.0),
        (7.0, 2.0), (8.0, 1.0), (13.0, 23.0), (15.0, 25.0),
    ]
    existing_points = [(1.0, 3.0), (15.0, 17.3), (9.3, 2.5), (11.0, 29.0), (35.0, -5.0)]
    nodes, _pts = convert_boundary_points_to_indices(x, y; existing_points)
    @test nodes == [[1, 2, 3, 4, 5] .+ 5, [5, 6, 7, 8] .+ 5, [8, 9, 10, 1] .+ 5]
    @test _pts == existing_points == append!(
              [(1.0, 3.0), (15.0, 17.3), (9.3, 2.5), (11.0, 29.0), (35.0, -5.0)],
              [
                  (1.0, 0.0), (2.0, 2.5), (3.0, 3.0), (4.0, 9.0), (5.0, 7.0), (6.0, 9.0),
                  (7.0, 2.0), (8.0, 1.0), (13.0, 23.0), (15.0, 25.0),
              ],
          )
    nodes, _pts = convert_boundary_points_to_indices(
        [
        [(1.0, 0.0), (2.0, 2.5), (3.0, 3.0), (4.0, 9.0), (5.0, 7.0)],
        [(5.0, 7.0), (6.0, 9.0), (7.0, 2.0), (8.0, 1.0)], [(8.0, 1.0), (13.0, 23.0), (15.0, 25.0), (1.0, 0.0)],
    ],
    )
    @test nodes == [[1, 2, 3, 4, 5], [5, 6, 7, 8], [8, 9, 10, 1]]
    @test _pts == [
        (1.0, 0.0), (2.0, 2.5), (3.0, 3.0), (4.0, 9.0), (5.0, 7.0), (6.0, 9.0),
        (7.0, 2.0), (8.0, 1.0), (13.0, 23.0), (15.0, 25.0),
    ]
    existing_points = [(1.0, 3.0), (15.0, 17.3), (9.3, 2.5), (11.0, 29.0), (35.0, -5.0)]
    nodes, _pts = convert_boundary_points_to_indices(x, y; existing_points)
    @test nodes == [[1, 2, 3, 4, 5] .+ 5, [5, 6, 7, 8] .+ 5, [8, 9, 10, 1] .+ 5]
    @test _pts == existing_points == append!(
              [(1.0, 3.0), (15.0, 17.3), (9.3, 2.5), (11.0, 29.0), (35.0, -5.0)],
              [
                  (1.0, 0.0), (2.0, 2.5), (3.0, 3.0), (4.0, 9.0), (5.0, 7.0), (6.0, 9.0),
                  (7.0, 2.0), (8.0, 1.0), (13.0, 23.0), (15.0, 25.0),
              ],
          )

    x1 = [[1.0, 2.0, 3.0], [3.0, 4.0, 5.5, 6.7], [6.7, 2.0, 1.0]]
    y1 = [[2.0, 2.5, 3.5], [3.5, 4.5, 7.7, 9.9], [9.9, 1.1, 2.0]]
    x2 = [[1.0, 1.2, 1.3, 1.4, 1.5, 1.0]]
    y2 = [[2.5, 2.7, 9.9, 2.0, 3.5, 2.5]]
    x3 = [[9.5, 13.7, 3.3], [3.3, 5.5, 9.5]]
    y3 = [[2.5, 11.7, 3.9], [3.9, 1.0, 2.5]]
    x = [x1, x2, x3]
    y = [y1, y2, y3]
    nodes, _pts = convert_boundary_points_to_indices(x, y)
    node1 = [[1, 2, 3], [3, 4, 5, 6], [6, 7, 1]]
    node2 = [[8, 9, 10, 11, 12, 8]]
    node3 = [[13, 14, 15], [15, 16, 13]]
    @test nodes == [node1, node2, node3]
    @test _pts == [
        (1.0, 2.0), (2.0, 2.5), (3.0, 3.5), (4.0, 4.5), (5.5, 7.7),
        (6.7, 9.9), (2.0, 1.1), (1.0, 2.5), (1.2, 2.7), (1.3, 9.9), (1.4, 2.0), (1.5, 3.5),
        (9.5, 2.5), (13.7, 11.7), (3.3, 3.9), (5.5, 1.0),
    ]
    existing_points = [(1.0, 3.0), (3.5, 5.5), (13.7, 25.0), (19.0, 37.3), (100.0, 100.0), (10.3, 5.5)]
    nodes, _pts = convert_boundary_points_to_indices(x, y; existing_points)
    node1 = [[1, 2, 3] .+ 6, [3, 4, 5, 6] .+ 6, [6, 7, 1] .+ 6]
    node2 = [[8, 9, 10, 11, 12, 8] .+ 6]
    node3 = [[13, 14, 15] .+ 6, [15, 16, 13] .+ 6]
    @test nodes == [node1, node2, node3]
    @test _pts == append!(
        existing_points,
        [
            (1.0, 2.0), (2.0, 2.5), (3.0, 3.5), (4.0, 4.5), (5.5, 7.7),
            (6.7, 9.9), (2.0, 1.1), (1.0, 2.5), (1.2, 2.7), (1.3, 9.9), (1.4, 2.0), (1.5, 3.5),
            (9.5, 2.5), (13.7, 11.7), (3.3, 3.9), (5.5, 1.0),
        ],
    )
    xy1 = [[(1.0, 2.0), (2.0, 2.5), (3.0, 3.5)], [(3.0, 3.5), (4.0, 4.5), (5.5, 7.7), (6.7, 9.9)], [(6.7, 9.9), (2.0, 1.1), (1.0, 2.0)]]
    xy2 = [[(1.0, 2.5), (1.2, 2.7), (1.3, 9.9), (1.4, 2.0), (1.5, 3.5), (1.0, 2.5)]]
    xy3 = [[(9.5, 2.5), (13.7, 11.7), (3.3, 3.9)], [(3.3, 3.9), (5.5, 1.0), (9.5, 2.5)]]
    xy = [xy1, xy2, xy3]
    nodes, _pts = convert_boundary_points_to_indices(xy)
    node1 = [[1, 2, 3], [3, 4, 5, 6], [6, 7, 1]]
    node2 = [[8, 9, 10, 11, 12, 8]]
    node3 = [[13, 14, 15], [15, 16, 13]]
    @test nodes == [node1, node2, node3]
    @test _pts == [
        (1.0, 2.0), (2.0, 2.5), (3.0, 3.5), (4.0, 4.5), (5.5, 7.7),
        (6.7, 9.9), (2.0, 1.1), (1.0, 2.5), (1.2, 2.7), (1.3, 9.9), (1.4, 2.0), (1.5, 3.5),
        (9.5, 2.5), (13.7, 11.7), (3.3, 3.9), (5.5, 1.0),
    ]
    existing_points = [(1.0, 3.0), (3.5, 5.5), (13.7, 25.0), (19.0, 37.3), (100.0, 100.0), (10.3, 5.5)]
    nodes, _pts = convert_boundary_points_to_indices(xy; existing_points)
    node1 = [[1, 2, 3] .+ 6, [3, 4, 5, 6] .+ 6, [6, 7, 1] .+ 6]
    node2 = [[8, 9, 10, 11, 12, 8] .+ 6]
    node3 = [[13, 14, 15] .+ 6, [15, 16, 13] .+ 6]
    @test nodes == [node1, node2, node3]
    @test _pts == append!(
        existing_points,
        [
            (1.0, 2.0), (2.0, 2.5), (3.0, 3.5), (4.0, 4.5), (5.5, 7.7),
            (6.7, 9.9), (2.0, 1.1), (1.0, 2.5), (1.2, 2.7), (1.3, 9.9), (1.4, 2.0), (1.5, 3.5),
            (9.5, 2.5), (13.7, 11.7), (3.3, 3.9), (5.5, 1.0),
        ],
    )

    boundary_points = [[0.0, 0.0], [1.0, 0.0], [1.0, 1.0], [0.0, 1.0], [0.0, 0.0]]
    nodes, _pts = convert_boundary_points_to_indices(boundary_points)
    @test nodes == [1, 2, 3, 4, 1]
    @test _pts == [(0.0, 0.0), (1.0, 0.0), (1.0, 1.0), (0.0, 1.0)]

    points = rand(Makie.Point2{Float64}, 10)
    _pt = copy(points)
    _boundary_points = decompose(Point2f, Rect2{Float64}(0, 0, 1, 1))
    boundary_points = push!(copy(_boundary_points), [0.0, 0.0])
    boundary_nodes, pts = convert_boundary_points_to_indices(boundary_points; existing_points=points)
    @test boundary_nodes == [11, 12, 13, 14, 11]
    @test pts == append!(_pt, _boundary_points)
end

@testset "get_ordinal_suffix" begin
    @test DT.get_ordinal_suffix.(0:115) == [
        "th"
        "st"
        "nd"
        "rd"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "st"
        "nd"
        "rd"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "st"
        "nd"
        "rd"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "st"
        "nd"
        "rd"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "st"
        "nd"
        "rd"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "st"
        "nd"
        "rd"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "st"
        "nd"
        "rd"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "st"
        "nd"
        "rd"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "st"
        "nd"
        "rd"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "st"
        "nd"
        "rd"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
        "th"
    ]
end

@testset "fix_segment!" begin
    c = [(2, 15), (2, 28), (2, 41), (2, 54)]
    bad_indices = [1, 2, 3, 4]
    DT.fix_segments!(c, bad_indices)
    @test c == [(2, 15), (15, 28), (28, 41), (41, 54)]
    c = [(2, 15), (15, 28), (28, 41), (2, 54)]
    bad_indices = [1, 4]
    DT.fix_segments!(c, bad_indices)
    @test c == [(2, 15), (15, 28), (28, 41), (41, 54)]
    c = [(2, 15), (15, 28), (2, 41), (41, 54)]
    bad_indices = [1, 3]
    DT.fix_segments!(c, bad_indices)
    @test c == [(2, 15), (15, 28), (28, 41), (41, 54)]
    c = [(2, 15), (15, 28), (2, 41), (41, 54)]
    bad_indices = [3]
    DT.fix_segments!(c, bad_indices)
    @test c == [(2, 15), (15, 28), (28, 41), (41, 54)]
    c = [(2, 7), (2, 12), (12, 17), (2, 22), (2, 27), (2, 32), (32, 37), (2, 42), (42, 47)]
    bad_indices = [2, 4, 5, 6, 8]
    DT.fix_segments!(c, bad_indices)
    @test c == [(2, 7), (7, 12), (12, 17), (17, 22), (22, 27), (27, 32), (32, 37), (37, 42), (42, 47)]
end

@testset "next/previndex_circular" begin
    @test DT.nextindex_circular([1, 2, 3, 4, 5, 6], 1) == 2
    @test DT.nextindex_circular([1, 2, 3, 4, 5, 6], 2) == 3
    @test DT.nextindex_circular([1, 2, 3, 4, 5, 6], 3) == 4
    @test DT.nextindex_circular([1, 2, 3, 4, 5, 6], 4) == 5
    @test DT.nextindex_circular([1, 2, 3, 4, 5, 6], 5) == 6
    @test DT.nextindex_circular([1, 2, 3, 4, 5, 6], 6) == 1

    @test DT.previndex_circular([1, 2, 3, 4, 5, 6], 1) == 5
    @test DT.previndex_circular([1, 2, 3, 4, 5, 6], 2) == 1
    @test DT.previndex_circular([1, 2, 3, 4, 5, 6], 3) == 2
    @test DT.previndex_circular([1, 2, 3, 4, 5, 6], 4) == 3
    @test DT.previndex_circular([1, 2, 3, 4, 5, 6], 5) == 4
    @test DT.previndex_circular([1, 2, 3, 4, 5, 6], 6) == 5
end

@testset "first/last_ghost_vertex" begin
    @test DT.is_first_ghost_vertex([1, 2, 3, -1, -5], 4)
    @test !DT.is_first_ghost_vertex([1, 2, 3, -1, -5], 5)
    @test DT.is_first_ghost_vertex([-1, -2, 5, 4], 1)
    @test DT.is_last_ghost_vertex([-1, 5, 4, 6, -2, -1], 1)
    @test DT.is_first_ghost_vertex([-1, 2, 3, 4, 5, -6, -1], 6)
end

@testset "get_neighbouring_boundary_edges" begin
    tri = fixed_shewchuk_example_constrained()
    left_e, right_e = DT.get_neighbouring_boundary_edges(tri, (10, 11))
    @test left_e == (11, 7) && right_e == (3, 10)
    left_e, right_e = DT.get_neighbouring_boundary_edges(tri, (11, 7))
    @test left_e == (7, 6) && right_e == (10, 11)
    left_e, right_e = DT.get_neighbouring_boundary_edges(tri, (7, 6))
    @test left_e == (6, 5) && right_e == (11, 7)
    left_e, right_e = DT.get_neighbouring_boundary_edges(tri, (6, 5))
    @test left_e == (5, 4) && right_e == (7, 6)
    left_e, right_e = DT.get_neighbouring_boundary_edges(tri, (5, 4))
    @test left_e == (4, 1) && right_e == (6, 5)
    left_e, right_e = DT.get_neighbouring_boundary_edges(tri, (4, 1))
    @test left_e == (1, 2) && right_e == (5, 4)
    left_e, right_e = DT.get_neighbouring_boundary_edges(tri, (1, 2))
    @test left_e == (2, 3) && right_e == (4, 1)
    left_e, right_e = DT.get_neighbouring_boundary_edges(tri, (2, 3))
    @test left_e == (3, 10) && right_e == (1, 2)
    @test DT.get_neighbouring_boundary_edges(tri, (10, 11)) == DT.get_neighbouring_boundary_edges(tri, (11, 10))
    @test DT.get_neighbouring_boundary_edges(tri, (11, 7)) == DT.get_neighbouring_boundary_edges(tri, (7, 11))
    @test DT.get_neighbouring_boundary_edges(tri, (7, 6)) == DT.get_neighbouring_boundary_edges(tri, (6, 7))
    @test DT.get_neighbouring_boundary_edges(tri, (6, 5)) == DT.get_neighbouring_boundary_edges(tri, (5, 6))
    @test DT.get_neighbouring_boundary_edges(tri, (5, 4)) == DT.get_neighbouring_boundary_edges(tri, (4, 5))
    @test DT.get_neighbouring_boundary_edges(tri, (4, 1)) == DT.get_neighbouring_boundary_edges(tri, (1, 4))
    @test DT.get_neighbouring_boundary_edges(tri, (1, 2)) == DT.get_neighbouring_boundary_edges(tri, (2, 1))
    @test DT.get_neighbouring_boundary_edges(tri, (2, 3)) == DT.get_neighbouring_boundary_edges(tri, (3, 2))

    tri, _, _ = simple_geometry()
    add_ghost_triangles!(tri)
    left_e, right_e = DT.get_neighbouring_boundary_edges(tri, (18, 13))
    @test left_e == (13, 14) && right_e == (17, 18)
    left_e, right_e = DT.get_neighbouring_boundary_edges(tri, (13, 14))
    @test left_e == (14, 15) && right_e == (18, 13)
    left_e, right_e = DT.get_neighbouring_boundary_edges(tri, (12, 9))
    @test left_e == (9, 10) && right_e == (11, 12)
    left_e, right_e = DT.get_neighbouring_boundary_edges(tri, (5, 4))
    @test left_e == (4, 3) && right_e == (6, 5)
end

@testset "convert_to_edge_adjoining_ghost_vertex" begin
    a = (3.0, 3.0)
    b = (0.0, 3.0)
    c = (0.0, 0.0)
    d = (4.0, 0.0)
    e = (1.0, 1.5)
    pts = [a, b, c, d, e]
    tri = triangulate(pts, delete_ghosts=false, randomise=false)
    @test DT.convert_to_edge_adjoining_ghost_vertex(tri, (1, 2)) == (2, 1)
    @test DT.convert_to_edge_adjoining_ghost_vertex(tri, (2, 1)) == (2, 1)
end

@testset "get_shared_vertex" begin
    e = (1, 3)
    f = (7, 5)
    @test DT.get_shared_vertex(e, f) == DT.∅
    @test DT.get_shared_vertex(f, e) == DT.∅
    e = (7, 3)
    f = (3, 8)
    @test DT.get_shared_vertex(e, f) == 3
    @test DT.get_shared_vertex(f, e) == 3
    f = (9, 7)
    @test DT.get_shared_vertex(e, f) == 7
    @test DT.get_shared_vertex(f, e) == 7
end

@testset "replace_(boundary/ghost)_triangle_with_(ghost/boundary)_triangle" begin
    for _ in 1:10
        tri = triangulate(rand(2, 5000), delete_ghosts=false)
        _tri = triangulate_rectangle(0.0, 1.0, 0.0, 1.0, 25, 25, delete_ghosts=false)
        for tri in (tri, _tri)
            for T in each_ghost_triangle(tri)
                T = DT.sort_triangle(T)
                i, j, k = triangle_vertices(T)
                w = get_adjacent(tri, j, i)
                V = (j, i, w)
                kr = count(e -> DT.is_boundary_edge(tri, e...), DT.triangle_edges(V))
                kr == 2 && continue # two associated ghosts, test is not clear
                @test DT.replace_boundary_triangle_with_ghost_triangle(tri, V) == T
                @test DT.replace_ghost_triangle_with_boundary_triangle(tri, T) == V
                V = (i, w, j)
                @test DT.replace_boundary_triangle_with_ghost_triangle(tri, V) == T
                @test DT.replace_ghost_triangle_with_boundary_triangle(tri, T) == (j, i, w)
                V = (w, j, i)
                @test DT.replace_boundary_triangle_with_ghost_triangle(tri, V) == T
                @test DT.replace_ghost_triangle_with_boundary_triangle(tri, T) == (j, i, w)
            end
        end
    end
end

@testset "iterated_neighbourhood" begin
    points = NTuple{2,Float64}[]
    for i in 1:5
        t = [0.0, π / 2, π, 3π / 2]
        push!(points, [(i^2 * cos(t), i^2 * sin(t)) for t in t]...)
    end
    push!(points, (0.0, 0.0))
    n = DT.num_points(points)
    tri = triangulate(points, delete_ghosts=false)

    neighbours = DT.iterated_neighbourhood(tri, n, 1)
    @test neighbours == filter(!DT.is_ghost_vertex, get_neighbours(tri, n))
    neighbours = DT.iterated_neighbourhood(tri, n, 2)
    S1 = get_neighbours(tri, n)
    S2 = [get_neighbours(tri, i) for i in S1]
    [union!(S1, S) for S in S2]
    filter!(s -> !(s == n || DT.is_ghost_vertex(s)), S1)
    @test S1 == neighbours
    neighbours = DT.iterated_neighbourhood(tri, n, 3)
    S1 = get_neighbours(tri, n)
    S2 = [get_neighbours(tri, i) for i in S1]
    S3 = [get_neighbours(tri, i) for i in reduce(union, S2)]
    [union!(S1, S) for S in S2]
    [union!(S1, S) for S in S3]
    filter!(s -> !(s == n || DT.is_ghost_vertex(s)), S1)
    @test S1 == neighbours
    neighbours = DT.iterated_neighbourhood(tri, n, 4)
    S1 = get_neighbours(tri, n)
    S2 = [get_neighbours(tri, i) for i in S1]
    S3 = [get_neighbours(tri, i) for i in reduce(union, S2)]
    S4 = [get_neighbours(tri, i) for i in reduce(union, S3)]
    [union!(S1, S) for S in S2]
    [union!(S1, S) for S in S3]
    [union!(S1, S) for S in S4]
    filter!(s -> !(s == n || DT.is_ghost_vertex(s)), S1)
    @test S1 == neighbours

    tri = triangulate_rectangle(0, 1, 0, 1, 10, 10)
    neighbours = DT.iterated_neighbourhood(tri, 1, 3)
    @test neighbours == Set(
        (
        2, 11,
        21, 12, 3,
        31, 22, 13, 4,
    ),
    )
    neighbours = DT.iterated_neighbourhood!(neighbours, tri, 1, 2)
    @test neighbours == Set(
        (
        2, 11,
        21, 12, 3,
    ),
    )
    neighbours = DT.iterated_neighbourhood!(neighbours, tri, 1, 6)
    @test neighbours == Set(
        (
        2, 11,
        21, 12, 3,
        31, 22, 13, 4,
        41, 32, 23, 14, 5,
        51, 42, 33, 24, 15, 6,
        61, 52, 43, 34, 25, 16, 7,
    ),
    )
    add_ghost_triangles!(tri)
    neighbours = DT.iterated_neighbourhood(tri, 1, 3)
    @test neighbours == Set(
        (
        2, 11,
        21, 12, 3,
        31, 22, 13, 4,
    ),
    )
    neighbours = DT.iterated_neighbourhood(tri, 1, 2)
    @test neighbours == Set(
        (
        2, 11,
        21, 12, 3,
    ),
    )
    neighbours = DT.iterated_neighbourhood(tri, 1, 6)
    @test neighbours == Set(
        (
        2, 11,
        21, 12, 3,
        31, 22, 13, 4,
        41, 32, 23, 14, 5,
        51, 42, 33, 24, 15, 6,
        61, 52, 43, 34, 25, 16, 7,
    ),
    )
end

@testset "getxy" begin
    @test DT.getxy((0.3, 0.5)) == (0.3, 0.5)
    @test DT.getxy((0.3f0, 0.7f0)) == (Float64(0.3f0), Float64(0.7f0))
end

@testset "norm" begin
    # test that we avoid underflow  
    @test DT.norm((1.0e-300, 1.0e-300)) ≈ norm([1.0e-300, 1.0e-300]) ≈ 1.414213562373095e-300
    @test DT.norm((1.0e-300, 0.0)) ≈ norm([1.0e-300, 0.0]) ≈ 1.0e-300
    @test DT.norm((0.0, 1.0e-300)) ≈ norm([0.0, 1.0e-300]) ≈ 1.0e-300

    # that we avoid overflow 
    @test DT.norm((1.0e300, 1.0e300)) ≈ norm([1.0e300, 1.0e300]) ≈ 1.414213562373095e300
    @test DT.norm((1.0e300, 0.0)) ≈ norm([1.0e300, 0.0]) ≈ 1.0e300
    @test DT.norm((0.0, 1.0e300)) ≈ norm([0.0, 1.0e300]) ≈ 1.0e300

    # some other random tests 
    for _ in 1:10000
        x, y = rand(2)
        @test DT.norm((x, y)) ≈ norm((x, y)) ≈ DT.norm((y, x))
        @test DT.norm((x, y))^2 ≈ DT.norm_sqr((x, y))
    end
    @inferred DT.norm((1.0, 1.0))
    @inferred DT.norm((1.0f0, 1.0f0))

    # specific case 
    @test DT.dist_sqr((3.0, -1.0), (2.0, 0.0)) == 2.0
end

@testset "edge_length" begin
    tri = triangulate(rand(2, 50))
    for e in each_solid_edge(tri)
        @test DT.edge_length(tri, e) ≈
              DT.edge_length(tri, DT.reverse_edge(e)) ≈
              DT.edge_length(tri, e...) ≈
              norm(get_point(tri, e[1]) .- get_point(tri, e[2]))
        @test DT.edge_length_sqr(tri, e) ≈
              DT.edge_length_sqr(tri, DT.reverse_edge(e)) ≈
              DT.edge_length_sqr(tri, e...) ≈
              LinearAlgebra.norm_sqr(get_point(tri, e[1]) .- get_point(tri, e[2]))
    end
end

@testset "midpoint" begin
    tri = triangulate(rand(2, 50))
    for e in each_solid_edge(tri)
        @test collect(DT.midpoint(tri, e)) ≈
              collect(DT.midpoint(tri, DT.reverse_edge(e))) ≈
              collect(DT.midpoint(tri, e...)) ≈
              0.5 .* collect((get_point(tri, e[1]) .+ get_point(tri, e[2])))
    end
end

@testset "check_precision" begin
    @test DT.check_precision(sqrt(eps(Float64)) - 1.0e-32)
    @test !DT.check_precision(1.0)
    @test DT.check_precision(sqrt(eps(Float64)))

    @test DT.check_absolute_precision(1.0e-32, 0.0)
    @test DT.check_absolute_precision(sqrt(eps(Float64)), 0)
    @test !DT.check_absolute_precision(0.5, 2.0)

    @test !DT.check_relative_precision(1.0e-32, 0.0)
    @test !DT.check_relative_precision(sqrt(eps(Float64)), 0)
    @test !DT.check_relative_precision(0, 1.0e-32)
    @test !DT.check_relative_precision(0, sqrt(eps(Float64)))

    @test DT.check_ratio_precision(1.0, 1.0)
    @test !DT.check_ratio_precision(5, 2)
    @test !DT.check_ratio_precision(0.0, 1.0)
    @test !DT.check_ratio_precision(1.0, 0.0)
end

@testset "get_boundary_chain" begin
    rng = StableRNG(123)
    points = randn(rng, 2, 250)
    tri = triangulate(points; rng)
    hull = get_convex_hull_vertices(tri)
    chain = DT.get_boundary_chain(tri, 65, 147, -1)
    @test DT.circular_equality([chain..., chain[1]], hull)
    chain = DT.get_boundary_chain(tri, 199, 96, -1)
    @test chain == [199, 151, 96]
    chain = DT.get_boundary_chain(tri, 151, 96, -1)
    @test chain == [151, 96]
end

@testset "dist" begin
    for i in 1:10
        tri = triangulate(rand(StableRNG(i), 2, 500); rng=StableRNG(i + 1))
        for j in 1:10
            p = randn(StableRNG(i * j), 2)
            δ = DT.distance_to_polygon(p, get_points(tri), get_convex_hull_vertices(tri))
            V = find_triangle(tri, p)
            if DT.is_ghost_triangle(V)
                @test δ < 0
            else
                @test δ ≥ 0.0
            end
            @test δ ≈ DT.dist(tri, p)
        end
    end
    points = [(0.0, 0.0), (1.0, 0.0), (1.0, 1.0), (0.5, 0.9), (0.0, 1.0), (0.2, 0.2), (0.8, 0.2), (0.8, 0.8), (0.2, 0.8)]
    tri = triangulate(points; boundary_nodes=[[[1, 2, 3, 4, 5, 1]], [[6, 9, 8, 7, 6]]])
    for i in 1:100
        p = randn(StableRNG(i^2), 2)
        δ = DT.distance_to_polygon(p, get_points(tri), get_boundary_nodes(tri))
        V = find_triangle(tri, p; rng=StableRNG(i^3), concavity_protection=true)
        if DT.is_ghost_triangle(V)
            @test δ < 0
        else
            @test δ ≥ 0.0
        end
        @test δ ≈ DT.dist(tri, p)
    end
end

@testset "adjust_θ" begin
    θ₁ = deg2rad(77.4711922908485)
    θ₂ = deg2rad(175.2363583092738)
    θ₁′, θ₂′ = DT.adjust_θ(θ₁, θ₂, true)
    @test (θ₁, θ₂) == (θ₁′, θ₂′)
    θ₁′, θ₂′ = DT.adjust_θ(θ₂, θ₁, true)
    @test (θ₁′, θ₂′) ⪧ (θ₂, θ₁ + 2π) && θ₂′ - θ₁′ ≈ deg2rad(262.2348339815747)
    θ₁′, θ₂′ = DT.adjust_θ(θ₁, θ₂, false)
    @test (θ₁′, θ₂′) ⪧ (θ₁, θ₂ - 2π) && θ₂′ - θ₁′ ≈ -deg2rad(360 - 97.7651660184254)
    θ₁′, θ₂′ = DT.adjust_θ(θ₂, θ₁, false)
    @test (θ₁′, θ₂′) ⪧ (θ₂, θ₁) && θ₂′ - θ₁′ ≈ -deg2rad(97.7651660184254)
    for _ in 1:100000
        θ₁, θ₂ = 2π .* rand(2)
        θ₁′, θ₂′ = DT.adjust_θ(θ₁, θ₂, rand() < 1 / 2)
        @test abs(θ₂′ - θ₁′) ≤ 2π
    end
end

@testset "uniquetol" begin
    A = [0.0, 0.1, 0.2, 0.3, 0.4, 0.4 + 1.0e-16]
    B = DT.uniquetol(A)
    @test B == [0.0, 0.1, 0.2, 0.3, 0.4]
    A = [-5, -4, -3, -3 - 1.0e-16, 0, 5 - 1.0e-16, 5 - 1.0e-14, 5 + 1.0e-16, 5 + 1.0e-14, 10, 15 + 1.0e-16]
    sort!(A)
    B = DT.uniquetol(A)
    @test B == [-5, -4, -3, 0, 5 - 1.0e-14, 10.0, 15.0]
end

@testset "Evaluating functions over heterogeneous tuples" begin
    @testset "eval_fnc_at_het_tuple_element" begin
        global basic_def
        f = x -> x isa Number
        tup = (1, 2.0, "string", [1 2 3], [5, 7, 9], 0x00, 'A')
        basic_def(f, tup, idx) = f(tup[idx])
        DT.eval_fnc_at_het_tuple_element(f, tup, 1)
        DT.eval_fnc_at_het_tuple_element(f, tup, 4)
        DT.eval_fnc_at_het_tuple_element(f, tup, 7)
        basic_def(f, tup, 1)
        basic_def(f, tup, 4)
        basic_def(f, tup, 7)
        a1 = @allocated DT.eval_fnc_at_het_tuple_element(f, tup, 1)
        a2 = @allocated DT.eval_fnc_at_het_tuple_element(f, tup, 2)
        a3 = @allocated DT.eval_fnc_at_het_tuple_element(f, tup, 3)
        a4 = @allocated DT.eval_fnc_at_het_tuple_element(f, tup, 4)
        a5 = @allocated DT.eval_fnc_at_het_tuple_element(f, tup, 5)
        a6 = @allocated DT.eval_fnc_at_het_tuple_element(f, tup, 6)
        a7 = @allocated DT.eval_fnc_at_het_tuple_element(f, tup, 7)
        b1 = @allocated basic_def(f, tup, 1)
        b2 = @allocated basic_def(f, tup, 2)
        b3 = @allocated basic_def(f, tup, 3)
        b4 = @allocated basic_def(f, tup, 4)
        b5 = @allocated basic_def(f, tup, 5)
        b6 = @allocated basic_def(f, tup, 6)
        b7 = @allocated basic_def(f, tup, 7)
        a = (a1, a2, a3, a4, a5, a6, a7)
        b = (b1, b2, b3, b4, b5, b6, b7)
        @test all(iszero, a) || all(iszero, a .- 16)
        for i in 1:7
            global basic_def
            @test DT.eval_fnc_at_het_tuple_element(f, tup, i) == basic_def(f, tup, i)
            @inferred DT.eval_fnc_at_het_tuple_element(f, tup, i)
        end

        ## large tuples
        tup = ntuple(_ -> rand((1, 2.0, "string", [1 2 3], [5, 7, 9], 0x00, 'A')), 32)
        @test DT.eval_fnc_at_het_tuple_element(f, tup, 14) == basic_def(f, tup, 14)
        b = @allocated DT.eval_fnc_at_het_tuple_element(f, tup, 14)
        b = @allocated DT.eval_fnc_at_het_tuple_element(f, tup, 14)
        b2 = @allocated basic_def(f, tup, 14)
        b2 = @allocated basic_def(f, tup, 14)
        @inferred DT.eval_fnc_at_het_tuple_element(f, tup, 14)
        @test iszero(b) || iszero(b .- 16)
        @test b < b2
        @test !(tup isa DT.ANY32)

        tup = ntuple(_ -> rand((1, 2.0, "string", [1 2 3], [5, 7, 9], 0x00, 'A')), 33)
        @test DT.eval_fnc_at_het_tuple_element(f, tup, 33) == basic_def(f, tup, 33)
        b = @allocated DT.eval_fnc_at_het_tuple_element(f, tup, 33)
        b = @allocated DT.eval_fnc_at_het_tuple_element(f, tup, 33)
        b2 = @allocated basic_def(f, tup, 33)
        b2 = @allocated basic_def(f, tup, 33)
        @test b == b2
    end

    @testset "eval_fnc_in_het_tuple" begin
        global basic_def2
        gg1(x) = x
        gg2(x) = x^2
        gg3(x) = x^3
        gg4(x) = x^4
        gg5(x) = x^5
        gg6(x) = x^6
        gg7(x) = x^7
        tup = (gg1, gg2, gg3, gg4, gg5, gg6, gg7)
        arg = rand()
        basic_def2(tup, arg, idx) = tup[idx](arg)
        DT.eval_fnc_in_het_tuple(tup, arg, 1)
        DT.eval_fnc_in_het_tuple(tup, arg, 4)
        DT.eval_fnc_in_het_tuple(tup, arg, 7)
        basic_def2(tup, arg, 1)
        basic_def2(tup, arg, 4)
        basic_def2(tup, arg, 7)
        @allocated DT.eval_fnc_in_het_tuple(tup, arg, 1)
        a1 = @allocated DT.eval_fnc_in_het_tuple(tup, arg, 1)
        a2 = @allocated DT.eval_fnc_in_het_tuple(tup, arg, 2)
        a3 = @allocated DT.eval_fnc_in_het_tuple(tup, arg, 3)
        a4 = @allocated DT.eval_fnc_in_het_tuple(tup, arg, 4)
        a5 = @allocated DT.eval_fnc_in_het_tuple(tup, arg, 5)
        a6 = @allocated DT.eval_fnc_in_het_tuple(tup, arg, 6)
        a7 = @allocated DT.eval_fnc_in_het_tuple(tup, arg, 7)
        b1 = @allocated basic_def2(tup, arg, 1)
        b2 = @allocated basic_def2(tup, arg, 2)
        b3 = @allocated basic_def2(tup, arg, 3)
        b4 = @allocated basic_def2(tup, arg, 4)
        b5 = @allocated basic_def2(tup, arg, 5)
        b6 = @allocated basic_def2(tup, arg, 6)
        b7 = @allocated basic_def2(tup, arg, 7)
        a = (a1, a2, a3, a4, a5, a6, a7)
        @test all(iszero, a) || all(iszero, a .- 16) || all(iszero, a .- 32)
        for i in 1:7
            global basic_def2
            @test DT.eval_fnc_in_het_tuple(tup, arg, i) == basic_def2(tup, arg, i)
            @inferred DT.eval_fnc_in_het_tuple(tup, arg, i)
        end

        ## large tuples 
        tup = ntuple(_ -> rand((gg1, gg2, gg3, gg4, gg5, gg6, gg7)), 32)
        @test DT.eval_fnc_in_het_tuple(tup, arg, 14) == basic_def2(tup, arg, 14)
        b = @allocated DT.eval_fnc_in_het_tuple(tup, arg, 14)
        b = @allocated DT.eval_fnc_in_het_tuple(tup, arg, 14)
        b2 = @allocated basic_def2(tup, arg, 14)
        b2 = @allocated basic_def2(tup, arg, 14)
        @inferred DT.eval_fnc_in_het_tuple(tup, arg, 14)
        @test iszero(b) || iszero(b .- 16) || iszero(b .- 32)

        tup = ntuple(_ -> rand((gg1, gg2, gg3, gg4, gg5, gg6, gg7)), 33)
        @test DT.eval_fnc_in_het_tuple(tup, arg, 33) == basic_def2(tup, arg, 33)
    end

    @testset "eval_fnc_at_het_tuple_two_elements" begin
        global fft
        global basic_defft
        fft(x, y) = objectid(x) + objectid(y)
        tup = (1, 2.0, "string", [1 2 3], [5, 7, 9], 0x00, 'A')
        basic_defft(f, tup, idx, idx2) = f(tup[idx], tup[idx2])
        DT.eval_fnc_at_het_tuple_two_elements(fft, tup, 1, 4)
        DT.eval_fnc_at_het_tuple_two_elements(fft, tup, 4, 3)
        DT.eval_fnc_at_het_tuple_two_elements(fft, tup, 2, 5)
        basic_defft(fft, tup, 1, 4)
        basic_defft(fft, tup, 4, 3)
        basic_defft(fft, tup, 2, 5)
        a = zeros(length(tup), length(tup))
        b = similar(a)
        for i in eachindex(tup)
            for j in eachindex(tup)
                global fft
                global basic_defft
                a[i, j] = @allocated DT.eval_fnc_at_het_tuple_two_elements(fft, tup, i, j)
                b[i, j] = @allocated basic_defft(fft, tup, i, j)
                @test DT.eval_fnc_at_het_tuple_two_elements(fft, tup, i, j) == basic_defft(fft, tup, i, j)
                @inferred DT.eval_fnc_at_het_tuple_two_elements(fft, tup, i, j)
            end
        end
        @test all(iszero, a .- 16) || all(iszero, a)

        ## large tuples
        tup = ntuple(_ -> rand((1, 2.0, "string", [1 2 3], [5, 7, 9], 0x00, 'A')), 32)
        @test DT.eval_fnc_at_het_tuple_two_elements(fft, tup, 14, 15) == basic_defft(fft, tup, 14, 15)
        b = @allocated DT.eval_fnc_at_het_tuple_two_elements(fft, tup, 14, 15)
        b = @allocated DT.eval_fnc_at_het_tuple_two_elements(fft, tup, 14, 15)
        b2 = @allocated basic_defft(fft, tup, 14, 15)
        b2 = @allocated basic_defft(fft, tup, 14, 15)
        @inferred DT.eval_fnc_at_het_tuple_two_elements(fft, tup, 14, 15)
        @test iszero(b) || iszero(b .- 16)

        tup = ntuple(_ -> rand((1, 2.0, "string", [1 2 3], [5, 7, 9], 0x00, 'A')), 33)
        @test DT.eval_fnc_at_het_tuple_two_elements(fft, tup, 33, 32) == basic_defft(fft, tup, 33, 32)
    end

    @testset "eval_fnc_at_het_tuple_element_with_arg" begin
        global fft
        global basic_defft
        fft(x, y, z, w) = objectid(x) + objectid(y) + objectid(z) + objectid(w)
        tup = (1, 2.0, "string", [1 2 3], [5, 7, 9], 0x00, 'A')
        basic_defft(f, tup, arg, idx) = f(tup[idx], arg...)
        DT.eval_fnc_at_het_tuple_element_with_arg(fft, tup, ((2.0, 3.0), -1.0, "5"), 1)
        DT.eval_fnc_at_het_tuple_element_with_arg(fft, tup, ((2.0, 3.0), -1.0, "5"), 2)
        DT.eval_fnc_at_het_tuple_element_with_arg(fft, tup, ((2.0, 3.0), -1.0, "5"), 3)
        DT.eval_fnc_at_het_tuple_element_with_arg(fft, tup, ((2.0, 3.0), -1.0, "5"), 4)
        basic_defft(fft, tup, ((2.0, 3.0), -1.0, "5"), 1)
        basic_defft(fft, tup, ((2.0, 3.0), -1.0, "5"), 2)
        basic_defft(fft, tup, ((2.0, 3.0), -1.0, "5"), 3)
        basic_defft(fft, tup, ((2.0, 3.0), -1.0, "5"), 4)
        a = zeros(length(tup))
        b = similar(a)
        arg = ((2.0, 3.0), -1.0, "5")
        for i in eachindex(tup)
            global fft
            global basic_defft
            a[i] = @allocated DT.eval_fnc_at_het_tuple_element_with_arg(fft, tup, arg, i)
            b[i] = @allocated basic_defft(fft, tup, arg, i)
            @test DT.eval_fnc_at_het_tuple_element_with_arg(fft, tup, arg, i) == basic_defft(fft, tup, arg, i)
            @inferred DT.eval_fnc_at_het_tuple_element_with_arg(fft, tup, arg, i)
        end
        @test all(iszero, a .- 16) || all(iszero, a)

        ## large tuples
        tup = ntuple(_ -> rand((1, 2.0, "string", [1 2 3], [5, 7, 9], 0x00, 'A')), 32)
        @test DT.eval_fnc_at_het_tuple_element_with_arg(fft, tup, ((2.0, 3.0), -1.0, "5"), 14) == basic_defft(fft, tup, ((2.0, 3.0), -1.0, "5"), 14)
        arg = ((2.0, 3.0), -1.0, "5")
        b = @allocated DT.eval_fnc_at_het_tuple_element_with_arg(fft, tup, arg, 14)
        b = @allocated DT.eval_fnc_at_het_tuple_element_with_arg(fft, tup, arg, 14)
        b2 = @allocated basic_defft(fft, tup, ((2.0, 3.0), -1.0, "5"), 14)
        b2 = @allocated basic_defft(fft, tup, ((2.0, 3.0), -1.0, "5"), 14)
        @inferred DT.eval_fnc_at_het_tuple_element_with_arg(fft, tup, ((2.0, 3.0), -1.0, "5"), 14)
        @test iszero(b) || iszero(b .- 16)

        tup = ntuple(_ -> rand((1, 2.0, "string", [1 2 3], [5, 7, 9], 0x00, 'A')), 33)
        @test DT.eval_fnc_at_het_tuple_element_with_arg(fft, tup, ((2.0, 3.0), -1.0, "5"), 33) == basic_defft(fft, tup, ((2.0, 3.0), -1.0, "5"), 33)
    end

    @testset "eval_fnc_at_het_tuple_element_with_arg_and_prearg" begin
        global fft
        global basic_defft
        fft(x, y, z, w) = objectid(x) + objectid(y) + objectid(z) + objectid(w)
        tup = (1, 2.0, "string", [1 2 3], [5, 7, 9], 0x00, 'A')
        basic_defft(f, tup, prearg, arg, idx) = f(prearg, tup[idx], arg...)
        DT.eval_fnc_at_het_tuple_element_with_arg_and_prearg(fft, tup, -3.0, ((2.0, 3.0), "5"), 1)
        DT.eval_fnc_at_het_tuple_element_with_arg_and_prearg(fft, tup, -3.0, ((2.0, 3.0), "5"), 2)
        DT.eval_fnc_at_het_tuple_element_with_arg_and_prearg(fft, tup, -3.0, ((2.0, 3.0), "5"), 3)
        DT.eval_fnc_at_het_tuple_element_with_arg_and_prearg(fft, tup, -3.0, ((2.0, 3.0), "5"), 4)
        basic_defft(fft, tup, -3.0, ((2.0, 3.0), "5"), 1)
        basic_defft(fft, tup, -3.0, ((2.0, 3.0), "5"), 2)
        basic_defft(fft, tup, -3.0, ((2.0, 3.0), "5"), 3)
        basic_defft(fft, tup, -3.0, ((2.0, 3.0), "5"), 4)
        a = zeros(length(tup))
        b = similar(a)
        arg = ((2.0, 3.0), "5")
        prearg = -3.0
        for i in eachindex(tup)
            global fft
            global basic_defft
            a[i] = @allocated DT.eval_fnc_at_het_tuple_element_with_arg_and_prearg(fft, tup, prearg, arg, i)
            b[i] = @allocated basic_defft(fft, tup, prearg, arg, i)
            @test DT.eval_fnc_at_het_tuple_element_with_arg_and_prearg(fft, tup, prearg, arg, i) == basic_defft(fft, tup, prearg, arg, i)
            @inferred DT.eval_fnc_at_het_tuple_element_with_arg_and_prearg(fft, tup, prearg, arg, i)
        end
        @test all(iszero, a .- 16) || all(iszero, a) || all(iszero, a .- 32)

        ## large tuples
        tup = ntuple(_ -> rand((1, 2.0, "string", [1 2 3], [5, 7, 9], 0x00, 'A')), 32)
        @test DT.eval_fnc_at_het_tuple_element_with_arg_and_prearg(fft, tup, -3.0, ((2.0, 3.0), "5"), 14) == basic_defft(fft, tup, -3.0, ((2.0, 3.0), "5"), 14)
        arg = ((2.0, 3.0), "5")
        b = @allocated DT.eval_fnc_at_het_tuple_element_with_arg_and_prearg(fft, tup, -3.0, arg, 14)
        b = @allocated DT.eval_fnc_at_het_tuple_element_with_arg_and_prearg(fft, tup, -3.0, arg, 14)
        b2 = @allocated basic_defft(fft, tup, -3.0, ((2.0, 3.0), "5"), 14)
        b2 = @allocated basic_defft(fft, tup, -3.0, ((2.0, 3.0), "5"), 14)
        @inferred DT.eval_fnc_at_het_tuple_element_with_arg_and_prearg(fft, tup, -3.0, ((2.0, 3.0), "5"), 14)
        @test iszero(b) || iszero(b .- 16)

        tup = ntuple(_ -> rand((1, 2.0, "string", [1 2 3], [5, 7, 9], 0x00, 'A')), 33)
        @test DT.eval_fnc_at_het_tuple_element_with_arg_and_prearg(fft, tup, -3.0, ((2.0, 3.0), "5"), 33) == basic_defft(fft, tup, -3.0, ((2.0, 3.0), "5"), 33)
    end
end

@testset "_to_val" begin
    @test DT._to_val(2) == Val(2)
    @test DT._to_val(Val(2)) == Val(2)
end

@testset "ε" begin
    @test DT.ε(Float64) == DT.ε(1.0) == sqrt(eps(Float64))
    @test DT.ε(Float32) == DT.ε(1.0f0) == sqrt(eps(Float32))
end

@testset "Fixing caches" begin
    tri = triangulate(rand(2, 50))
    orient3_cache = AdaptivePredicates.orient3adapt_cache(Float64)
    incircle_cache = AdaptivePredicates.incircleadapt_cache(Float64)
    @test DT.validate_incircle_cache(tri, incircle_cache)
    @test DT.validate_incircle_cache(tri, nothing)
    @test !DT.validate_incircle_cache(tri, orient3_cache)
    @test DT.validate_orient3_cache(tri, orient3_cache)
    @test DT.validate_orient3_cache(tri, nothing)
    @test !DT.validate_orient3_cache(tri, incircle_cache)
    tri32 = triangulate(rand(Float32, 2, 50))
    orient3_cache32 = AdaptivePredicates.orient3adapt_cache(Float32)
    incircle_cache32 = AdaptivePredicates.incircleadapt_cache(Float32)
    @test DT.validate_incircle_cache(tri32, incircle_cache32)
    @test DT.validate_incircle_cache(tri32, nothing)
    @test !DT.validate_incircle_cache(tri32, orient3_cache32)
    @test DT.validate_orient3_cache(tri32, orient3_cache32)
    @test DT.validate_orient3_cache(tri32, nothing)
    @test !DT.validate_orient3_cache(tri32, incircle_cache32)

    @test DT.fix_incircle_cache(tri, incircle_cache) === incircle_cache
    @test DT.fix_incircle_cache(tri, nothing) === nothing
    @test DT.fix_incircle_cache(tri, orient3_cache) === nothing
    @test DT.fix_orient3_cache(tri, orient3_cache) === orient3_cache
    @test DT.fix_orient3_cache(tri, nothing) === nothing
    @test DT.fix_orient3_cache(tri, incircle_cache) === nothing
    @test DT.fix_incircle_cache(tri32, incircle_cache32) === incircle_cache32
    @test DT.fix_incircle_cache(tri32, nothing) === nothing
    @test DT.fix_incircle_cache(tri32, orient3_cache32) === nothing
    @test DT.fix_orient3_cache(tri32, orient3_cache32) === orient3_cache32
    @test DT.fix_orient3_cache(tri32, nothing) === nothing
    @test DT.fix_orient3_cache(tri32, incircle_cache32) === nothing
end