from .backports import Backports

def bp(name, config):
  """Checking if backport is used."""
  return name in config.backports()

def bpv(name, default_mins, config):
  """Checking if backport, or any versioned variant thereof, are used and returns the minimum
  versions, and returns default versions otherwise."""
  for (name_, _, mins) in Backports.expand_versions(name):
    if bp(name_, config):
      return mins
  return default_mins

# Module requirements: name -> min version per major or None if N.A.
def MOD_REQS(config):
  return {
    "BaseHTTPServer": ((2, 0), None),
    "ConfigParser": ((2, 0), None),
    "DocXMLRPCServer": ((2, 3), None),
    "HTMLParser": ((2, 2), None),
    "Queue": ((2, 0), None),
    "SimpleXMLRPCServer": ((2, 2), None),
    "SocketServer": ((2, 0), None),
    "Tkinter": ((2, 0), None),
    "__builtin__": ((2, 0), None),
    "__future__": ((2, 1), (3, 0)),
    "_ast": ((2, 5), (3, 0)),
    "_dummy_thread": (None, (3, 0)),
    "_markupbase": (None, (3, 0)),
    "_thread": (None, (3, 0)),
    "_winreg": ((2, 0), None),
    "abc": ((2, 6), (3, 0)),
    "annotationlib": (None, (3, 14)),
    "argparse": bpv("argparse", ((2, 7), (3, 2)), config),
    "ast": ((2, 6), (3, 0)),
    "asyncio": bpv("asyncio", (None, (3, 4)), config),
    "asyncio.graph": (None, (3, 14)),
    "asyncore": ((2, 6), (3, 0)),
    "atexit": ((2, 0), (3, 0)),
    "audioop": ((2, 5), (3, 0)),
    "bisect": ((2, 1), (3, 0)),
    "builtins": (None, (3, 0)),
    "bz2": ((2, 3), (3, 0)),
    "cProfile": ((2, 5), (3, 0)),
    "calendar": ((2, 0), (3, 0)),
    "cgitb": ((2, 2), (3, 0)),
    "collections": ((2, 4), (3, 0)),
    "collections.abc": (None, (3, 3)),
    "compileall": ((2, 6), (3, 0)),
    "compression": (None, (3, 14)),
    "compression.bz2": (None, (3, 14)),
    "compression.gzip": (None, (3, 14)),
    "compression.lzma": (None, (3, 14)),
    "compression.zlib": (None, (3, 14)),
    "compression.zstd": (None, (3, 14)),
    "concurrent.futures": (None, (3, 2)),
    "concurrent.interpreters": (None, (3, 14)),
    "configparser": bpv("configparser", (None, (3, 0)), config),
    "contextlib": ((2, 5), (3, 0)),
    "contextvars": bpv("contextvars", (None, (3, 7)), config),
    "cookielib": ((2, 4), None),
    "copy_reg": ((2, 0), None),
    "copyreg": (None, (3, 0)),
    "csv": ((2, 3), (3, 0)),
    "ctypes": ((2, 5), (3, 0)),
    "dataclasses": bpv("dataclasses", (None, (3, 7)), config),
    "datetime": ((2, 3), (3, 0)),
    "dbm.dumb": (None, (3, 0)),
    "dbm.gnu": (None, (3, 0)),
    "dbm.io": (None, (3, 0)),
    "dbm.ndbm": (None, (3, 0)),
    "dbm.os": (None, (3, 0)),
    "dbm.sqlite3": (None, (3, 13)),
    "dbm.struct": (None, (3, 0)),
    "dbm.sys": (None, (3, 0)),
    "dbm.whichdb": (None, (3, 0)),
    "decimal": ((2, 4), (3, 0)),
    "difflib": ((2, 1), (3, 0)),
    "dummy_thread": ((2, 3), None),
    "dummy_threading": ((2, 3), None),
    "email": ((2, 2), (3, 0)),
    "email.charset": ((2, 2), (3, 0)),
    "email.contentmanager": (None, (3, 4)),
    "email.errors": ((2, 4), (3, 0)),
    "email.header": ((2, 2), (3, 0)),
    "email.headerregistry": (None, (3, 3)),
    "email.message": ((2, 2), (3, 0)),
    "email.parser": ((2, 2), (3, 0)),
    "email.policy": (None, (3, 3)),
    "email.utils": ((2, 2), (3, 0)),
    "encodings.idna": ((2, 3), (3, 0)),
    "encodings.utf_8_sig": ((2, 5), (3, 0)),
    "ensurepip": ((2, 7), (3, 4)),
    "enum": bpv("enum", (None, (3, 4)), config),
    "faulthandler": bpv("faulthandler", (None, (3, 3)), config),
    "fractions": ((2, 6), (3, 0)),
    "functools": ((2, 5), (3, 0)),
    "future_builtins": ((2, 6), None),
    "graphlib": (None, (3, 9)),
    "hashlib": ((2, 5), (3, 0)),
    "heapq": ((2, 3), (3, 0)),
    "hmac": ((2, 2), (3, 0)),
    "hotshot": ((2, 2), None),
    "html": (None, (3, 0)),
    "htmlentitydefs": ((2, 0), None),
    "http": (None, (3, 0)),
    "http.cookiejar": (None, (3, 0)),
    "http.server": (None, (3, 0)),
    "imghdr": ((2, 5), (3, 0)),
    "imp": ((2, 0), (3, 0)),
    "importlib": bpv("importlib", ((2, 7), (3, 1)), config),
    "importlib.metadata": (None, (3, 8)),
    "importlib.resources": (None, (3, 7)),
    "importlib.resources.abc": (None, (3, 11)),
    "inspect": ((2, 1), (3, 0)),
    "io": ((2, 6), (3, 0)),
    "ipaddress": bpv("ipaddress", (None, (3, 3)), config),
    "itertools": ((2, 3), (3, 0)),
    "json": ((2, 6), (3, 0)),
    "locale": ((2, 0), (3, 0)),
    "logging": ((2, 3), (3, 0)),
    "logging.config": ((2, 3), (3, 0)),
    "logging.handlers": ((2, 3), (3, 0)),
    "lzma": (None, (3, 3)),
    "markupbase": ((2, 0), None),
    "md5": ((2, 0), None),
    "modulefinder": ((2, 3), (3, 0)),
    "msilib": ((2, 5), (3, 0)),
    "multiprocessing": ((2, 6), (3, 0)),
    "multiprocessing.pool": (None, (3, 0)),
    "multiprocessing.shared_memory": (None, (3, 8)),
    "new": ((2, 0), None),
    "numbers": ((2, 6), (3, 0)),
    "optparse": ((2, 3), (3, 0)),
    "ossaudiodev": ((2, 3), (3, 0)),
    "pathlib": (None, (3, 4)),
    "pathlib.types": (None, (3, 14)),
    "pdb": ((2, 0), (3, 0)),
    "pickletools": ((2, 3), (3, 0)),
    "pkgutil": ((2, 3), (3, 0)),
    "platform": ((2, 3), (3, 0)),
    "popen2": ((2, 0), None),
    "pyclbr": ((2, 0), (3, 0)),
    "pydoc": ((2, 1), (3, 0)),
    "queue": (None, (3, 0)),
    "random": ((2, 0), (3, 0)),
    "repr": ((2, 0), None),
    "reprlib": (None, (3, 0)),
    "runpy": ((2, 5), (3, 0)),
    "secrets": (None, (3, 6)),
    "selectors": (None, (3, 4)),
    "sets": ((2, 3), None),
    "socket": ((2, 0), (3, 0)),
    "socketserver": (None, (3, 0)),
    "spwd": ((2, 5), (3, 0)),
    "sqlite3": ((2, 5), (3, 0)),
    "ssl": ((2, 6), (3, 0)),
    "statistics": bpv("statistics", (None, (3, 4)), config),
    "string.templatelib": (None, (3, 14)),
    "stringprep": ((2, 3), (3, 0)),
    "subprocess": ((2, 4), (3, 0)),
    "sysconfig": ((2, 7), (3, 2)),
    "tarfile": ((2, 3), (3, 0)),
    "test.support.bytecode_helper": (None, (3, 9)),
    "test.support.import_helper": (None, (3, 10)),
    "test.support.os_helper": (None, (3, 10)),
    "test.support.socket_helper": (None, (3, 9)),
    "test.support.threading_helper": (None, (3, 10)),
    "test.support.warnings_helper": (None, (3, 10)),
    "textwrap": ((2, 3), (3, 0)),
    "timeit": ((2, 3), (3, 0)),
    "tkinter": (None, (3, 0)),
    "tomllib": (None, (3, 11)),
    "tracemalloc": (None, (3, 4)),
    "typing": bpv("typing", (None, (3, 5)), config),
    "typing_extensions": bpv("typing_extensions", ((0, 0), (0, 0)), config),
    "unittest": ((2, 1), (3, 0)),
    "unittest.mock": (None, (3, 3)),
    "urllib.parse": (None, (3, 0)),
    "urllib2": ((2, 0), None),
    "urlparse": ((2, 0), None),
    "uuid": ((2, 5), (3, 0)),
    "venv": (None, (3, 3)),
    "warnings": ((2, 1), (3, 0)),
    "weakref": ((2, 1), (3, 0)),
    "webbrowser": ((2, 0), (3, 0)),
    "winreg": (None, (3, 0)),
    "wsgiref": ((2, 5), (3, 0)),
    "wsgiref.types": (None, (3, 11)),
    "xml.dom": ((2, 0), (3, 0)),
    "xml.dom.minidom": ((2, 0), (3, 0)),
    "xml.dom.pulldom": ((2, 0), (3, 0)),
    "xml.etree.ElementInclude": ((2, 5), (3, 0)),
    "xml.etree.ElementTree": ((2, 5), (3, 0)),
    "xml.parsers.expat": ((2, 0), (3, 0)),
    "xml.sax": ((2, 0), (3, 0)),
    "xml.sax.handler": ((2, 0), (3, 0)),
    "xml.sax.saxutils": ((2, 0), (3, 0)),
    "xml.sax.xmlreader": ((2, 0), (3, 0)),
    "xmlrpc": (None, (3, 0)),
    "xmlrpc.client": (None, (3, 0)),
    "xmlrpc.server": (None, (3, 0)),
    "xmlrpclib": ((2, 2), None),
    "zipapp": (None, (3, 5)),
    "zipimport": ((2, 3), (3, 0)),
    "zoneinfo": bpv("zoneinfo", (None, (3, 9)), config),
   }

# Module member requirements: member -> (module, requirements)
def MOD_MEM_REQS(config):
  return {
    # Builtin classes
    "bool": ((2, 2), (3, 0)),
    "bytearray": ((2, 6), (3, 0)),
    "frozenset": ((2, 4), (3, 0)),
    "memoryview": ((2, 7), (3, 0)),
    "object": ((2, 2), (3, 0)),
    "set": ((2, 4), (3, 0)),
    "dict": ((2, 2), (3, 0)),
    "long": ((2, 0), None),

    # Classes
    "ConfigParser.ConfigParser": ((2, 3), None),
    "ConfigParser.RawConfigParser": ((2, 3), None),
    "ConfigParser.SafeConfigParser": ((2, 3), None),
    "Queue.LifoQueue": ((2, 6), None),
    "Queue.PriorityQueue": ((2, 6), None),
    "SimpleXMLRPCServer.CGIXMLRPCRequestHandler": ((2, 3), None),
    "abc.ABC": (None, (3, 4)),
    "argparse.BooleanOptionalAction": (None, (3, 9)),
    "asyncio.Barrier": (None, (3, 11)),
    "asyncio.BufferedProtocol": (None, (3, 7)),
    "asyncio.MultiLoopChildWatcher": (None, (3, 8)),
    "asyncio.PidfdChildWatcher": (None, (3, 9)),
    "asyncio.Runner": (None, (3, 11)),
    "asyncio.TaskGroup": (None, (3, 11)),
    "asyncio.ThreadedChildWatcher": (None, (3, 8)),
    "calendar.Calendar": ((2, 5), (3, 0)),
    "calendar.Day": (None, (3, 12)),
    "calendar.HTMLCalendar": ((2, 5), (3, 0)),
    "calendar.LocaleHTMLCalendar": ((2, 5), (3, 0)),
    "calendar.LocaleTextCalendar": ((2, 5), (3, 0)),
    "calendar.Month": (None, (3, 12)),
    "calendar.TextCalendar": ((2, 5), (3, 0)),
    "codecs.IncrementalEncoder": ((2, 5), (3, 0)),
    "collections.ChainMap": (None, (3, 3)),
    "collections.Counter": ((2, 7), (3, 1)),
    "collections.OrderedDict": ((2, 7), (3, 1)),
    "collections.UserDict": (None, (3, 0)),
    "collections.UserList": (None, (3, 0)),
    "collections.UserString": (None, (3, 0)),
    "collections.abc.AsyncGenerator": (None, (3, 6)),
    "collections.abc.AsyncIterable": (None, (3, 5)),
    "collections.abc.AsyncIterator": (None, (3, 5)),
    "collections.abc.Awaitable": (None, (3, 5)),
    "collections.abc.Buffer": (None, (3, 12)),
    "collections.abc.Collection": (None, (3, 6)),
    "collections.abc.Coroutine": (None, (3, 5)),
    "collections.abc.Generator": (None, (3, 5)),
    "collections.abc.Reversible": (None, (3, 6)),
    "collections.defaultdict": ((2, 5), (3, 0)),
    "collections.deque": ((2, 4), (3, 0)),
    "collections.namedtuple": ((2, 6), (3, 0)),
    "concurrent.futures.InterpreterPoolExecutor": (None, (3, 14)),
    "contextlib.AbstractAsyncContextManager": (None, (3, 7)),
    "contextlib.AbstractContextManager": (None, (3, 6)),
    "contextlib.AsyncContextDecorator": (None, (3, 10)),
    "contextlib.AsyncExitStack": (None, (3, 7)),
    "contextlib.ContextDecorator": (None, (3, 2)),
    "contextlib.ExitStack": (None, (3, 3)),
    "csv.unix_dialect": (None, (3, 2)),
    "ctypes.BigEndianUnion": (None, (3, 11)),
    "ctypes.LittleEndianUnion": (None, (3, 11)),
    "ctypes.c_bool": ((2, 6), (3, 0)),
    "ctypes.c_double_complex": (None, (3, 14)),
    "ctypes.c_float_complex": (None, (3, 14)),
    "ctypes.c_longdouble": ((2, 6), (3, 0)),
    "ctypes.c_longdouble_complex": (None, (3, 14)),
    "ctypes.c_ssize_t": ((2, 7), (3, 2)),
    "datetime.timezone": (None, (3, 2)),
    "difflib.HtmlDiff": ((2, 4), (3, 0)),
    "dis.Bytecode": (None, (3, 4)),
    "dis.Instruction": (None, (3, 4)),
    "dis.Positions": (None, (3, 11)),
    "doctest.DocTest": ((2, 4), (3, 0)),
    "doctest.DocTestFinder": ((2, 4), (3, 0)),
    "doctest.DocTestParser": ((2, 4), (3, 0)),
    "doctest.DocTestRunner": ((2, 4), (3, 0)),
    "doctest.Example": ((2, 4), (3, 0)),
    "doctest.OutputChecker": ((2, 4), (3, 0)),
    "email.generator.BytesGenerator": (None, (3, 2)),
    "email.generator.DecodedGenerator": ((2, 2), (3, 0)),
    "email.mime.application.MIMEApplication": ((2, 5), (3, 0)),
    "email.mime.multipart.MIMEMultipart": ((2, 2), (3, 0)),
    "email.mime.nonmultipart.MIMENonMultipart": ((2, 2), (3, 0)),
    "email.parser.BytesFeedParser": (None, (3, 2)),
    "email.parser.BytesHeaderParser": (None, (3, 3)),
    "email.parser.BytesParser": (None, (3, 2)),
    "email.parser.FeedParser": ((2, 4), (3, 0)),
    "email.policy.EmailPolicy": (None, (3, 3)),
    "enum.EnumCheck": (None, (3, 11)),
    "enum.EnumDict": (None, (3, 13)),
    "enum.Flag": (None, (3, 6)),
    "enum.FlagBoundary": (None, (3, 11)),
    "enum.IntFlag": (None, (3, 6)),
    "enum.StrEnum": (None, (3, 11)),
    "enum.auto": (None, (3, 6)),
    "ftplib.FTP_TLS": ((2, 7), (3, 2)),
    "functools.partialmethod": (None, (3, 4)),
    "http.HTTPMethod": (None, (3, 11)),
    "http.HTTPStatus": (None, (3, 5)),
    "http.server.HTTPSServer": (None, (3, 14)),
    "http.server.ThreadingHTTPSServer": (None, (3, 14)),
    "http.server.ThreadingHTTPServer": (None, (3, 7)),
    "httplib.HTTPConnection": ((2, 0), None),
    "httplib.HTTPResponse": ((2, 0), None),
    "httplib.HTTPSConnection": ((2, 0), None),
    "imaplib.IMAP4_stream": ((2, 3), (3, 0)),
    "imp.NullImporter": ((2, 5), (3, 0)),
    "importlib.abc.FileLoader": (None, (3, 3)),
    "importlib.abc.MetaPathFinder": (None, (3, 3)),
    "importlib.abc.PathEntryFinder": (None, (3, 3)),
    "importlib.abc.ResourceReader": (None, (3, 7)),
    "importlib.abc.Traversable": (None, (3, 9)),
    "importlib.abc.TraversableReader": (None, (3, 9)),
    "importlib.abc.TraversableResources": (None, (3, 9)),
    "importlib.machinery.ExtensionFileLoader": (None, (3, 3)),
    "importlib.machinery.FileFinder": (None, (3, 3)),
    "importlib.machinery.ModuleSpec": (None, (3, 4)),
    "importlib.machinery.NamespaceLoader": (None, (3, 11)),
    "importlib.machinery.SourceFileLoader": (None, (3, 3)),
    "importlib.machinery.SourcelessFileLoader": (None, (3, 3)),
    "importlib.machinery.WindowsRegistryFinder": (None, (3, 3)),
    "importlib.metadata.EntryPoints": (None, (3, 10)),
    "importlib.util.LazyLoader": (None, (3, 5)),
    "inspect.BoundArguments": (None, (3, 3)),
    "inspect.BufferFlags": (None, (3, 12)),
    "inspect.Parameter": (None, (3, 3)),
    "inspect.Signature": (None, (3, 3)),
    "io.Reader": (None, (3, 14)),
    "io.Writer": (None, (3, 14)),
    "logging.LoggerAdapter": ((2, 6), (3, 0)),
    "logging.NullHandler": ((2, 7), (3, 1)),
    "logging.StreamHandler": ((2, 6), (3, 0)),
    "logging.handlers.QueueHandler": (None, (3, 2)),
    "logging.handlers.QueueListener": (None, (3, 2)),
    "logging.handlers.WatchedFileHandler": ((2, 6), (3, 0)),
    "multiprocessing.Barrier": (None, (3, 3)),
    "nntplib.NNTP_SSL": (None, (3, 2)),
    "os.DirEntry": (None, (3, 5)),
    "os.PathLike": (None, (3, 6)),
    "os.sched_param": (None, (3, 3)),
    "os.terminal_size": (None, (3, 3)),
    "pickle.PickleBuffer": (None, (3, 8)),
    "pkgutil.ModuleInfo": (None, (3, 6)),
    "plistlib.UID": (None, (3, 8)),
    "popen2.Popen4": ((2, 0), None),
    "poplib.POP3_SSL": ((2, 4), (3, 0)),
    "pstats.FunctionProfile": (None, (3, 9)),
    "pstats.Stats": ((2, 3), (3, 0)),
    "pstats.StatsProfile": (None, (3, 9)),
    "py_compile.PycInvalidationMode": (None, (3, 7)),
    "queue.SimpleQueue": (None, (3, 7)),
    "random.SystemRandom": ((2, 4), (3, 0)),
    "selectors.DevpollSelector": (None, (3, 5)),
    "signals.Handlers": (None, (3, 5)),
    "signals.Sigmasks": (None, (3, 5)),
    "signals.Signals": (None, (3, 5)),
    "smtplib.LMTP": ((2, 6), (3, 0)),
    "smtplib.SMTP_SSL": ((2, 6), (3, 0)),
    "sqlite3.Blob": (None, (3, 11)),
    "ssl.AlertDescription": (None, (3, 6)),
    "ssl.MemoryBIO": (None, (3, 5)),
    "ssl.Options": (None, (3, 6)),
    "ssl.SSLContext": ((2, 7), (3, 2)),
    "ssl.SSLErrorNumber": (None, (3, 6)),
    "ssl.SSLObject": (None, (3, 5)),
    "ssl.SSLSession": (None, (3, 6)),
    "ssl.TLSVersion": (None, (3, 7)),
    "ssl.VerifyFlags": (None, (3, 6)),
    "ssl.VerifyMode": (None, (3, 6)),
    "statistics.NormalDist": (None, (3, 8)),
    "string.Formatter": ((2, 6), (3, 0)),
    "string.Template": ((2, 4), (3, 0)),
    "struct.Struct": ((2, 5), (3, 0)),
    "subprocess.CompletedProcess": (None, (3, 5)),
    "test.support.EnvironmentVarGuard": ((2, 6), None),
    "test.support.TransientResource": ((2, 6), None),
    "test.support.WarningsRecorder": ((2, 6), None),
    "test.support.os_helper.EnvironmentVarGuard": (None, (3, 10)),
    "test.support.socket_helper.TransientResource": (None, (3, 9)),
    "test.support.warnings_helper.WarningsRecorder": (None, (3, 10)),
    "threading.Barrier": (None, (3, 2)),
    "threading.local": ((2, 4), (3, 0)),
    "time.struct_time": ((2, 2), (3, 0)),
    "traceback.FrameSummary": (None, (3, 5)),
    "traceback.StackSummary": (None, (3, 5)),
    "traceback.TracebackException": (None, (3, 5)),
    "tracemalloc.DomainFilter": (None, (3, 6)),
    "types.CapsuleType": (None, (3, 13)),
    "types.GenericAlias": (None, (3, 9)),
    "types.MappingProxyType": (None, (3, 3)),
    "types.SimpleNamespace": (None, (3, 3)),
    "typing.AsyncContextManager": (None, (3, 5)),
    "typing.AsyncGenerator": (None, (3, 6)),
    "typing.AsyncIterable": (None, (3, 5)),
    "typing.AsyncIterator": (None, (3, 5)),
    "typing.Awaitable": (None, (3, 5)),
    # Notes: `typing.ChainMap` relies on `collections.ChainMap`, which is 3.3+.
    "typing.ChainMap": (None, (3, 5) if not bp("typing", config) else (3, 3)),
    "typing.Collection": (None, (3, 6)),
    # Notes:
    #  `typing.ContextManager` was added in `typing` backport 3.6.2 (which requires 2.7 or 3.3+).
    "typing.ContextManager": (None, (3, 5)) if not bp("typing", config) else ((2, 7), (3, 3)),
    "typing.Coroutine": (None, (3, 5)),
    # Notes: `typing.Counter` was added in `typing` backport 3.6.1 (which requires 2.7 or 3.3+).
    "typing.Counter": (None, (3, 5)) if not bp("typing", config) else ((2, 7), (3, 3)),
    # Notes: `typing.Deque` was added in `typing` backport 3.6.1 (which requires 2.7 or 3.3+).
    "typing.Deque": (None, (3, 5)) if not bp("typing", config) else ((2, 7), (3, 3)),
    "typing.ForwardRef": (None, (3, 7)),
    "typing.NamedTuple": (None, (3, 5)),
    "typing.NoDefault": (None, (3, 13)),
    "typing.OrderedDict": (None, (3, 7)),
    # Notes:
    #  `typing.Protocol` is backported to Python 2.7. For Python 3 it goes to `typing_extensions`.
    #  The same applies for some other rules below.
    "typing.Protocol": (None, (3, 8)) if not bp("typing", config) else ((2, 7), (3, 8)),
    "typing.ParamSpec": (None, (3, 10)),
    "typing.ReadOnly": (None, (3, 13)),
    # Notes: `object.__bytes__` is 3.2+
    "typing.SupportsBytes": (None, (3, 5)) if not bp("typing", config) else (None, (3, 2)),
    # Notes:
    #  `typing.SupportsIndex` was added in `typing` backport 3.7.4 (which requires 2.7 or 3.4+).
    "typing.SupportsIndex": (None, (3, 8)) if not bp("typing", config) else ((2, 7), (3, 4)),
    # Notes: `object.__round__` is 3.0+
    "typing.SupportsRound": (None, (3, 5)) if not bp("typing", config) else (None, (3, 2)),
    "typing.TypedDict": (None, (3, 8)) if not bp("typing", config) else ((2, 7), (3, 8)),
    "typing.TypeAliasType": (None, (3, 12)),
    "typing.TypeIs": (None, (3, 13)),
    "unittest.IsolatedAsyncioTestCase": (None, (3, 8)),
    "unittest.TextTestResult": ((2, 7), (3, 2)),
    "unittest.mock.AsyncMock": bpv("mock", (None, (3, 8)), config),
    "urllib.parse.DefragResult": (None, (3, 2)),
    "urllib.parse.DefragResultBytes": (None, (3, 2)),
    "urllib.parse.ParseResultBytes": (None, (3, 2)),
    "urllib.parse.SplitResultBytes": (None, (3, 2)),
    "urllib.request.DataHandler": (None, (3, 4)),
    "urllib.request.HTTPPasswordMgrWithPriorAuth": (None, (3, 5)),
    "urllib2.HTTPCookieProcessor": ((2, 4), None),
    "urllib2.HTTPErrorProcessor": ((2, 4), None),
    "uuid.SafeUUID": (None, (3, 7)),
    "warnings.catch_warnings": ((2, 6), (3, 0)),
    "weakref.WeakMethod": (None, (3, 4)),
    "weakref.WeakSet": ((2, 7), (3, 0)),
    "weakref.finalize": (None, (3, 4)),
    "wsgiref.handlers.IISCGIHandler": (None, (3, 2)),
    "xml.etree.ElementTree.C14NWriterTarget": (None, (3, 8)),
    "xml.etree.ElementTree.XMLPullParser": (None, (3, 4)),
    "xml.sax.handler.LexicalHandler": (None, (3, 10)),
    "xmlrpclib.MultiCall": ((2, 4), None),
    "zipfile.Path": (None, (3, 8)),
    "zipimport.zipimporter": ((2, 7), (3, 0)),

    # Builtin exceptions
    "BaseException": ((2, 5), (3, 0)),
    "exceptions.BaseException": ((2, 5), None),
    "BaseExceptionGroup": (None, (3, 11)),
    "BlockingIOError": (None, (3, 3)),
    "BrokenPipeError": (None, (3, 3)),
    "BytesWarning": ((2, 6), (3, 0)),
    "exceptions.BytesWarning": ((2, 6), None),
    "ChildProcessError": (None, (3, 3)),
    "ConnectionAbortedError": (None, (3, 3)),
    "ConnectionError": (None, (3, 3)),
    "ConnectionRefusedError": (None, (3, 3)),
    "ConnectionResetError": (None, (3, 3)),
    "EncodingWarning": (None, (3, 10)),
    "ExceptionGroup": (None, (3, 11)),
    "FileExistsError": (None, (3, 3)),
    "FileNotFoundError": (None, (3, 3)),
    "GeneratorExit": ((2, 5), (3, 0)),
    "exceptions.GeneratorExit": ((2, 5), None),
    "ImportWarning": ((2, 5), (3, 0)),
    "exceptions.ImportWarning": ((2, 5), None),
    "InterruptedError": (None, (3, 3)),
    "IsADirectoryError": (None, (3, 3)),
    "ModuleNotFoundError": (None, (3, 6)),
    "NotADirectoryError": (None, (3, 3)),
    "PermissionError": (None, (3, 3)),
    "ProcessLookupError": (None, (3, 3)),
    "PythonFinalizationError": (None, (3, 13)),
    "RecursionError": (None, (3, 5)),
    "ReferenceError": ((2, 2), (3, 0)),
    "exceptions.ReferenceError": ((2, 2), None),
    "ResourceWarning": (None, (3, 2)),
    "StopAsyncIteration": (None, (3, 5)),
    "StopIteration": ((2, 2), (3, 0)),
    "exceptions.StopIteration": ((2, 2), None),
    "TimeoutError": (None, (3, 3)),
    "UnicodeDecodeError": ((2, 3), (3, 0)),
    "exceptions.UnicodeDecodeError": ((2, 3), None),
    "UnicodeEncodeError": ((2, 3), (3, 0)),
    "exceptions.UnicodeEncodeError": ((2, 3), None),
    "UnicodeTranslateError": ((2, 3), (3, 0)),
    "exceptions.UnicodeTranslateError": ((2, 3), None),
    "UnicodeWarning": ((2, 5), (3, 0)),
    "exceptions.UnicodeWarning": ((2, 5), None),
    "UnboundLocalError": ((2, 0), (3, 0)),
    "exceptions.UnboundLocalError": ((2, 0), None),
    "UnicodeError": ((2, 0), (3, 0)),
    "exceptions.UnicodeError": ((2, 0), None),
    "WindowsError": ((2, 0), (3, 0)),
    "exceptions.WindowsError": ((2, 0), None),

    # Exceptions
    "ConfigParser.InterpolationMissingOptionError": ((2, 3), None),
    "ConfigParser.InterpolationSyntaxError": ((2, 3), None),
    "asyncio.QueueShutDown": (None, (3, 13)),
    "concurrent.futures.BrokenExecutor": (None, (3, 7)),
    "concurrent.futures.InvalidStateError": (None, (3, 8)),
    "concurrent.futures.interpreter.BrokenInterpreterPool": (None, (3, 14)),
    "concurrent.futures.interpreter.ExecutionFailed": (None, (3, 14)),
    "concurrent.futures.process.BrokenProcessPool": (None, (3, 3)),
    "concurrent.futures.thread.BrokenThreadPool": (None, (3, 7)),
    "configparser.InvalidWriteError": (None, (3, 14)),
    "configparser.MultilineContinuationError": (None, (3, 13)),
    "configparser.UnnamedSectionDisabledError": (None, (3, 14)),
    "ctypes.COMError": (None, (3, 14)),
    "email.errors.CloseBoundaryNotFoundDefect": (None, (3, 3)),
    "email.errors.FirstHeaderLineIsContinuationDefect": ((2, 4), (3, 0)),
    "email.errors.MalformedHeaderDefect": ((2, 4), (3, 0)),
    "email.errors.MisplacedEnvelopeHeaderDefect": ((2, 4), (3, 0)),
    "email.errors.MissingHeaderBodySeparatorDefect": (None, (3, 3)),
    "email.errors.MultipartInvariantViolationDefect": ((2, 4), (3, 0)),
    "email.errors.NoBoundaryInMultipartDefect": ((2, 4), (3, 0)),
    "email.errors.StartBoundaryNotFoundDefect": ((2, 4), (3, 0)),
    "gzip.BadGzipFile": (None, (3, 8)),
    "htmllib.HTMLParseError": ((2, 4), None),
    "httplib.BadStatusLine": ((2, 0), None),
    "httplib.CannotSendHeader": ((2, 0), None),
    "httplib.CannotSendRequest": ((2, 0), None),
    "httplib.HTTPException": ((2, 0), None),
    "httplib.ImproperConnectionState": ((2, 0), None),
    "httplib.IncompleteRead": ((2, 0), None),
    "httplib.InvalidURL": ((2, 3), None),
    "httplib.NotConnected": ((2, 0), None),
    "httplib.ResponseNotReady": ((2, 0), None),
    "httplib.UnimplementedFileMode": ((2, 0), None),
    "httplib.UnknownProtocol": ((2, 0), None),
    "httplib.UnknownTransferEncoding": ((2, 0), None),
    "json.JSONDecodeError": (None, (3, 5)),
    "pathlib.UnsupportedOperation": (None, (3, 13)),
    "queue.ShutDown": (None, (3, 13)),
    "re.PatternError": (None, (3, 13)),
    "sgmllib.SGMLParseError": ((2, 1), None),
    "shutil.Error": ((2, 3), (3, 0)),
    "shutil.SameFileError": (None, (3, 4)),
    "smtplib.SMTPNotSupportedError": (None, (3, 5)),
    "socket.timeout": ((2, 3), (3, 0)),
    "ssl.SSLCertVerificationError": (None, (3, 7)),
    "ssl.SSLEOFError": ((2, 7), (3, 3)),
    "ssl.SSLSyscallError": ((2, 7), (3, 3)),
    "ssl.SSLWantReadError": ((2, 7), (3, 3)),
    "ssl.SSLWantWriteError": ((2, 7), (3, 3)),
    "ssl.SSLZeroReturnError": ((2, 7), (3, 3)),
    "subprocess.SubprocessError": (None, (3, 3)),
    "subprocess.TimeoutExpired": (None, (3, 3)),
    "tarfile.HeaderError": ((2, 6), (3, 0)),
    "tarfile.LinkFallbackError": (None, (3, 9)),  # Backported 3.14->3.13->3.12->3.11->3.10->3.9
    "threading.BrokenBarrierError": (None, (3, 2)),
    "unittest.SkipTest": ((2, 7), (3, 1)),
    "urllib.ContentTooShortError": ((2, 5), None),
    "xml.dom.DOMException": ((2, 1), (3, 0)),
    "xml.dom.DomstringSizeErr": ((2, 1), (3, 0)),
    "xml.dom.HierarchyRequestErr": ((2, 1), (3, 0)),
    "xml.dom.IndexSizeErr": ((2, 1), (3, 0)),
    "xml.dom.InuseAttributeErr": ((2, 1), (3, 0)),
    "xml.dom.InvalidAccessErr": ((2, 1), (3, 0)),
    "xml.dom.InvalidCharacterErr": ((2, 1), (3, 0)),
    "xml.dom.InvalidModificationErr": ((2, 1), (3, 0)),
    "xml.dom.InvalidStateErr": ((2, 1), (3, 0)),
    "xml.dom.NamespaceErr": ((2, 1), (3, 0)),
    "xml.dom.NoDataAllowedErr": ((2, 1), (3, 0)),
    "xml.dom.NoModificationAllowedErr": ((2, 1), (3, 0)),
    "xml.dom.NotFoundErr": ((2, 1), (3, 0)),
    "xml.dom.NotSupportedErr": ((2, 1), (3, 0)),
    "xml.dom.SyntaxErr": ((2, 1), (3, 0)),
    "xml.dom.WrongDocumentErr": ((2, 1), (3, 0)),
    "zipfile.BadZipFile": (None, (3, 2)),

    # Builtin functions
    # Notes:
    #  file() is kept at 2.0 instead of 2.2, where the function was introduced, because file objects
    #  have existed since 2.0 and we don't do type deduction. Therefore analysis on file objects
    #  wouldn't be detected, like `open(..).xreadlines()`, but using `file(..).xreadlines()` would
    #  trigger rule "file.xreadlines" 2.1. So if file() was kept at 2.2 it would introduce conflict.
    "aiter": (None, (3, 10)),
    "all": ((2, 5), (3, 0)),
    "anext": (None, (3, 10)),
    "any": ((2, 5), (3, 0)),
    "basestring": ((2, 3), None),
    "bin": ((2, 6), (3, 0)),
    "breakpoint": (None, (3, 7)),
    "bytearray.hex": (None, (3, 5)),
    "bytearray.isascii": (None, (3, 7)),
    "bytearray.maketrans": (None, (3, 1)),
    "bytearray.removeprefix": (None, (3, 9)),
    "bytearray.removesuffix": (None, (3, 9)),
    "bytes.hex": (None, (3, 5)),
    "bytes.isascii": (None, (3, 7)),
    "bytes.maketrans": (None, (3, 1)),
    "bytes.removeprefix": (None, (3, 9)),
    "bytes.removesuffix": (None, (3, 9)),
    "callable": ((2, 0), (3, 2)),
    "classmethod": ((2, 2), (3, 0)),
    "dict.fromkeys": ((2, 3), (3, 0)),
    "dict.has_key": ((2, 2), None),
    "dict.iteritems": ((2, 2), None),
    "dict.iterkeys": ((2, 2), None),
    "dict.itervalues": ((2, 2), None),
    "dict.pop": ((2, 3), (3, 0)),
    "dict.viewitems": ((2, 7), None),
    "dict.viewkeys": ((2, 7), None),
    "dict.viewvalues": ((2, 7), None),
    "enumerate": ((2, 3), (3, 0)),
    "file": ((2, 0), None),  # Kept at 2.0 on purpose (see note above).
    "file.next": ((2, 3), None),
    "file.xreadlines": ((2, 1), None),
    "float.as_integer_ratio": ((2, 6), (3, 0)),
    "float.fromhex": ((2, 6), (3, 0)),
    "float.hex": ((2, 6), (3, 0)),
    "float.is_integer": ((2, 6), (3, 0)),
    "format": ((2, 6), (3, 0)),
    "frozenset.isdisjoint": ((2, 6), (3, 0)),
    "help": ((2, 2), (3, 0)),
    "int.as_integer_ratio": (None, (3, 8)),
    "int.bit_count": (None, (3, 10)),
    "int.bit_length": ((2, 7), (3, 1)),
    "int.from_bytes": (None, (3, 2)),
    "int.is_integer": (None, (3, 12)),
    "int.to_bytes": (None, (3, 2)),
    "iter": ((2, 2), (3, 0)),
    "list.clear": (None, (3, 0)),
    "list.copy": (None, (3, 0)),
    "long.bit_length": ((2, 7), None),
    "memoryview.cast": (None, (3, 3)),
    "memoryview.hex": (None, (3, 5)),
    "memoryview.release": (None, (3, 2)),
    "memoryview.toreadonly": (None, (3, 8)),
    "next": ((2, 6), (3, 0)),
    "property": ((2, 2), (3, 0)),
    "property.deleter": ((2, 6), (3, 0)),
    "property.getter": ((2, 6), (3, 0)),
    "property.setter": ((2, 6), (3, 0)),
    "reversed": ((2, 4), (3, 0)),
    "set.isdisjoint": ((2, 6), (3, 0)),
    "slice.indices": ((2, 3), (3, 0)),
    "sorted": ((2, 4), (3, 0)),
    "staticmethod": ((2, 2), (3, 0)),
    "str.casefold": (None, (3, 3)),
    "str.decode": ((2, 2), None),
    "str.encode": ((2, 0), (3, 0)),
    "str.format": ((2, 6), (3, 0)),
    "str.format_map": (None, (3, 2)),
    "str.isascii": (None, (3, 7)),
    "str.isdecimal": (None, (3, 0)),
    "str.isidentifier": (None, (3, 0)),
    "str.isnumeric": (None, (3, 0)),
    "str.isprintable": (None, (3, 0)),
    "str.maketrans": (None, (3, 0)),
    "str.partition": ((2, 5), (3, 0)),
    "str.removeprefix": (None, (3, 9)),
    "str.removesuffix": (None, (3, 9)),
    "str.rpartition": ((2, 5), (3, 0)),
    "str.rsplit": ((2, 4), (3, 0)),
    "str.zfill": ((2, 2), (3, 0)),
    "sum": ((2, 3), (3, 0)),
    "super": ((2, 2), (3, 0)),
    "type": ((2, 0), (3, 0)),
    "unichr": ((2, 0), None),
    "unicode": ((2, 0), None),
    "unicode.isdecimal": ((2, 0), None),
    "unicode.isnumeric": ((2, 0), None),
    "zip": ((2, 0), (3, 0)),

    # Builtin constants
    "memoryview.c_contiguous": (None, (3, 3)),
    "memoryview.contiguous": (None, (3, 3)),
    "memoryview.f_contiguous": (None, (3, 3)),
    "memoryview.nbytes": (None, (3, 3)),
    "memoryview.obj": (None, (3, 3)),

    # Dict view constants (via `dict.keys()`, `dict.values()`, or `dict.items()`)
    "dict.items.mapping": (None, (3, 10)),
    "dict.keys.mapping": (None, (3, 10)),
    "dict.values.mapping": (None, (3, 10)),

    # Functions
    "BaseException.__notes__": (None, (3, 11)),
    "BaseException.add_note": (None, (3, 11)),
    "ConfigParser.SafeConfigParser.set": ((2, 4), None),
    "EasyDialogs.GetArgv": ((2, 0), None),
    "Queue.Queue.join": ((2, 5), None),
    "Queue.Queue.task_done": ((2, 5), None),
    "SimpleHTTPServer.SimpleHTTPRequestHandler.do_GET": ((2, 5), None),
    "SimpleXMLRPCServer.SimpleXMLRPCServer.register_introspection_functions": ((2, 3), None),
    "SocketServer.BaseServer.server_close": ((2, 6), None),
    "SocketServer.BaseServer.shutdown": ((2, 6), None),
    "Tkinter.Tcl": ((2, 4), None),
    "_thread.get_native_id": (None, (3, 8)),
    "_winreg.CreateKeyEx": ((2, 7), None),
    "_winreg.DeleteKeyEx": ((2, 7), None),
    "_winreg.ExpandEnvironmentStrings": ((2, 6), None),
    "abc.get_cache_token": (None, (3, 4)),
    "abc.update_abstractmethods": (None, (3, 10)),
    "argparse.ArgumentParser.parse_intermixed_args": (None, (3, 7)),
    "argparse.ArgumentParser.parse_known_intermixed_args": (None, (3, 7)),
    "array.array.clear": (None, (3, 13)),
    "array.array.frombytes": (None, (3, 2)),
    "array.array.tobytes": (None, (3, 2)),
    "ast.compare": (None, (3, 14)),
    "ast.get_source_segment": (None, (3, 8)),
    "ast.unparse": (None, (3, 9)),
    "asyncio.AbstractChildWatcher.is_active": (None, (3, 8)),
    "asyncio.Future.get_loop": (None, (3, 7)),
    "asyncio.Handle.cancelled": (None, (3, 7)),
    "asyncio.Queue.shutdown": (None, (3, 13)),
    "asyncio.ReadTransport.is_reading": (None, (3, 7)),
    "asyncio.Server.abort_clients": (None, (3, 13)),
    "asyncio.Server.close_clients": (None, (3, 13)),
    "asyncio.Server.get_loop": (None, (3, 7)),
    "asyncio.Server.is_serving": (None, (3, 7)),
    "asyncio.Server.serve_forever": (None, (3, 7)),
    "asyncio.Server.start_serving": (None, (3, 7)),
    "asyncio.StreamReader.readuntil": (None, (3, 5)),
    "asyncio.StreamWriter.is_closing": (None, (3, 7)),
    "asyncio.StreamWriter.start_tls": (None, (3, 11)),
    "asyncio.StreamWriter.wait_closed": (None, (3, 7)),
    "asyncio.Task._check_future": (None, (3, 11)),
    "asyncio.Task.cancelling": (None, (3, 11)),
    "asyncio.Task.get_context": (None, (3, 12)),
    "asyncio.Task.get_coro": (None, (3, 8)),
    "asyncio.Task.get_name": (None, (3, 8)),
    "asyncio.Task.set_name": (None, (3, 8)),
    "asyncio.Task.uncancel": (None, (3, 11)),
    "asyncio.TimerHandle.when": (None, (3, 7)),
    "asyncio.WriteTransport.get_write_buffer_limits": (None, (3, 4)),
    "asyncio.all_tasks": (None, (3, 7)),
    "asyncio.capture_call_graph": (None, (3, 14)),
    "asyncio.create_eager_task_factory": (None, (3, 12)),
    "asyncio.create_task": (None, (3, 7)),
    "asyncio.current_task": (None, (3, 7)),
    "asyncio.eager_task_factory": (None, (3, 12)),
    "asyncio.get_running_loop": (None, (3, 7)),
    "asyncio.isfuture": (None, (3, 5)),
    "asyncio.loop.connect_accepted_socket": (None, (3, 5)),
    "asyncio.loop.create_future": (None, (3, 5)),
    "asyncio.loop.get_exception_handler": (None, (3, 5)),
    "asyncio.loop.sendfile": (None, (3, 7)),
    "asyncio.loop.shutdown_asyncgens": (None, (3, 6)),
    "asyncio.loop.shutdown_default_executor": (None, (3, 9)),
    "asyncio.loop.sock_recv_into": (None, (3, 7)),
    "asyncio.loop.sock_recvfrom": (None, (3, 11)),
    "asyncio.loop.sock_recvfrom_into": (None, (3, 11)),
    "asyncio.loop.sock_sendfile": (None, (3, 7)),
    "asyncio.loop.sock_sendto": (None, (3, 11)),
    "asyncio.loop.start_tls": (None, (3, 7)),
    "asyncio.print_call_graph": (None, (3, 14)),
    "asyncio.run": (None, (3, 7)),
    "asyncio.run_coroutine_threadsafe": (None, (3, 5)),
    "asyncio.timeout": (None, (3, 11)),
    "asyncio.timeout_at": (None, (3, 11)),
    "asyncio.to_thread": (None, (3, 9)),
    "asyncore.dispatcher.handle_accepted": (None, (3, 2)),
    "audioop.alaw2lin": ((2, 5), (3, 0)),
    "audioop.byteswap": (None, (3, 4)),
    "audioop.lin2alaw": ((2, 5), (3, 0)),
    "base64.a85decode": (None, (3, 4)),
    "base64.a85encode": (None, (3, 4)),
    "base64.b32hexdecode": (None, (3, 10)),
    "base64.b32hexencode": (None, (3, 10)),
    "base64.b85decode": (None, (3, 4)),
    "base64.b85encode": (None, (3, 4)),
    "base64.decodebytes": (None, (3, 1)),
    "base64.encodebytes": (None, (3, 1)),
    "base64.z85decode": (None, (3, 13)),
    "base64.z85encode": (None, (3, 13)),
    "bdb.Bdb.get_bpbynumber": (None, (3, 2)),
    "bdb.Breakpoint.bpformat": (None, (3, 2)),
    "bdb.Breakpoint.clearBreakpoints": (None, (3, 10)),
    "bz2.BZ2File.fileno": (None, (3, 3)),
    "bz2.BZ2File.peek": (None, (3, 3)),
    "bz2.BZ2File.read1": (None, (3, 3)),
    "bz2.BZ2File.readable": (None, (3, 3)),
    "bz2.BZ2File.readinto": (None, (3, 3)),
    "bz2.BZ2File.seekable": (None, (3, 3)),
    "bz2.BZ2File.writable": (None, (3, 3)),
    "bz2.open": (None, (3, 3)),
    "calendar.Calendar.itermonthdays3": (None, (3, 7)),
    "calendar.Calendar.itermonthdays4": (None, (3, 7)),
    "cgi.FieldStorage.getfirst": ((2, 2), (3, 0)),
    "cgi.FieldStorage.getlist": ((2, 2), (3, 0)),
    "cmath.isclose": (None, (3, 5)),
    "cmath.isfinite": (None, (3, 2)),
    "cmath.isinf": ((2, 6), (3, 0)),
    "cmath.isnan": ((2, 6), (3, 0)),
    "cmath.phase": ((2, 6), (3, 0)),
    "cmath.polar": ((2, 6), (3, 0)),
    "cmath.rect": ((2, 6), (3, 0)),
    "codecs.decode": ((2, 4), (3, 0)),
    "codecs.encode": ((2, 4), (3, 0)),
    "codecs.getincrementaldecoder": ((2, 5), (3, 0)),
    "codecs.getincrementalencoder": ((2, 5), (3, 0)),
    "codecs.iterdecode": ((2, 5), (3, 0)),
    "codecs.iterencode": ((2, 5), (3, 0)),
    "codecs.namereplace_errors": (None, (3, 5)),
    "codecs.unregister": (None, (3, 10)),
    "collections.Counter.subtract": (None, (3, 2)),
    "collections.Counter.total": (None, (3, 10)),
    "collections.OrderedDict.move_to_end": (None, (3, 2)),
    "collections.UserString.__getnewargs__": (None, (3, 5)),
    "collections.UserString.__rmod__": (None, (3, 5)),
    "collections.UserString.casefold": (None, (3, 5)),
    "collections.UserString.format_map": (None, (3, 5)),
    "collections.UserString.isprintable": (None, (3, 5)),
    "collections.UserString.maketrans": (None, (3, 5)),
    "collections.abc.MutableSequence.clear": (None, (3, 3)),
    "collections.abc.MutableSequence.copy": (None, (3, 3)),
    "collections.deque.copy": (None, (3, 5)),
    "collections.deque.count": ((2, 7), (3, 2)),
    "collections.deque.index": (None, (3, 5)),
    "collections.deque.insert": (None, (3, 5)),
    "collections.deque.remove": ((2, 5), (3, 0)),
    "collections.deque.reverse": ((2, 7), (3, 2)),
    "compileall.compile_file": ((2, 7), (3, 2)),
    "concurrent.futures.ProcessPoolExecutor.kill_workers": (None, (3, 14)),
    "concurrent.futures.ProcessPoolExecutor.terminate_workers": (None, (3, 14)),
    "configparser.ConfigParser.read_dict": bpv("configparser", (None, (3, 2)), config),
    "configparser.ConfigParser.read_file": bpv("configparser", (None, (3, 2)), config),
    "configparser.ConfigParser.read_string": bpv("configparser", (None, (3, 2)), config),
    "configparser.RawConfigParser.read_dict": bpv("configparser", (None, (3, 2)), config),
    "configparser.RawConfigParser.read_file": bpv("configparser", (None, (3, 2)), config),
    "configparser.RawConfigParser.read_string": bpv("configparser", (None, (3, 2)), config),
    "contextlib.aclosing": (None, (3, 10)),
    "contextlib.chdir": (None, (3, 11)),
    "contextlib.nullcontext": (None, (3, 7)),
    "contextlib.redirect_stderr": (None, (3, 5)),
    "contextlib.redirect_stdout": (None, (3, 4)),
    "contextlib.suppress": (None, (3, 4)),
    "copy.replace": (None, (3, 13)),
    "crypt.mksalt": (None, (3, 3)),
    "csv.DictWriter.writeheader": ((2, 7), (3, 2)),
    "csv.field_size_limit": ((2, 5), (3, 0)),
    "ctypes.CopyComPointer": (None, (3, 14)),
    "ctypes._CData.from_buffer": ((2, 6), (3, 0)),
    "ctypes._CData.from_buffer_copy": ((2, 6), (3, 0)),
    "ctypes.get_errno": ((2, 6), (3, 0)),
    "ctypes.get_last_error": ((2, 6), (3, 0)),
    "ctypes.set_errno": ((2, 6), (3, 0)),
    "ctypes.set_last_error": ((2, 6), (3, 0)),
    "ctypes.util.dllist": (None, (3, 14)),
    "ctypes.util.find_msvcrt": ((2, 6), (3, 0)),
    "curses.assume_default_colors": (None, (3, 14)),
    "curses.get_escdelay": (None, (3, 9)),
    "curses.get_tabsize": (None, (3, 9)),
    "curses.has_extended_color_support": (None, (3, 10)),
    "curses.memoryview_at": (None, (3, 14)),
    "curses.set_escdelay": (None, (3, 9)),
    "curses.set_tabsize": (None, (3, 9)),
    "curses.unget_wch": (None, (3, 3)),
    "curses.update_lines_cols": (None, (3, 5)),
    "curses.window.get_wch": (None, (3, 3)),
    "dbm.gdbm.clear": (None, (3, 13)),
    "dbm.ndbm.clear": (None, (3, 13)),
    "datetime.date.fromisocalendar": (None, (3, 8)),
    "datetime.date.fromisoformat": (None, (3, 7)),
    "datetime.date.strptime": (None, (3, 14)),
    "datetime.datetime.fromisocalendar": (None, (3, 8)),
    "datetime.datetime.fromisoformat": (None, (3, 7)),
    "datetime.datetime.strptime": ((2, 5), (3, 0)),
    "datetime.datetime.timestamp": (None, (3, 3)),
    "datetime.time.fromisoformat": (None, (3, 7)),
    "datetime.time.strptime": (None, (3, 14)),
    "datetime.timedelta.total_seconds": ((2, 7), (3, 2)),
    "decimal.Context.clear_traps": (None, (3, 3)),
    "decimal.Context.create_decimal_from_float": ((2, 7), (3, 1)),
    "decimal.Decimal.as_integer_ratio": (None, (3, 6)),
    "decimal.Decimal.canonical": ((2, 6), (3, 0)),
    "decimal.Decimal.compare_signal": ((2, 6), (3, 0)),
    "decimal.Decimal.compare_total": ((2, 6), (3, 0)),
    "decimal.Decimal.compare_total_mag": ((2, 6), (3, 0)),
    "decimal.Decimal.conjugate": ((2, 6), (3, 0)),
    "decimal.Decimal.copy_abs": ((2, 6), (3, 0)),
    "decimal.Decimal.copy_negate": ((2, 6), (3, 0)),
    "decimal.Decimal.copy_sign": ((2, 6), (3, 0)),
    "decimal.Decimal.exp": ((2, 6), (3, 0)),
    "decimal.Decimal.fma": ((2, 6), (3, 0)),
    "decimal.Decimal.from_float": ((2, 7), (3, 1)),
    "decimal.Decimal.from_number": (None, (3, 14)),
    "decimal.Decimal.is_canonical": ((2, 6), (3, 0)),
    "decimal.Decimal.is_finite": ((2, 6), (3, 0)),
    "decimal.Decimal.is_infinite": ((2, 6), (3, 0)),
    "decimal.Decimal.is_nan": ((2, 6), (3, 0)),
    "decimal.Decimal.is_normal": ((2, 6), (3, 0)),
    "decimal.Decimal.is_qnan": ((2, 6), (3, 0)),
    "decimal.Decimal.is_signed": ((2, 6), (3, 0)),
    "decimal.Decimal.is_snan": ((2, 6), (3, 0)),
    "decimal.Decimal.is_subnormal": ((2, 6), (3, 0)),
    "decimal.Decimal.is_zero": ((2, 6), (3, 0)),
    "decimal.Decimal.ln": ((2, 6), (3, 0)),
    "decimal.Decimal.log10": ((2, 6), (3, 0)),
    "decimal.Decimal.logb": ((2, 6), (3, 0)),
    "decimal.Decimal.logical_and": ((2, 6), (3, 0)),
    "decimal.Decimal.logical_invert": ((2, 6), (3, 0)),
    "decimal.Decimal.logical_or": ((2, 6), (3, 0)),
    "decimal.Decimal.logical_xor": ((2, 6), (3, 0)),
    "decimal.Decimal.max_mag": ((2, 6), (3, 0)),
    "decimal.Decimal.min_mag": ((2, 6), (3, 0)),
    "decimal.Decimal.next_minus": ((2, 6), (3, 0)),
    "decimal.Decimal.next_plus": ((2, 6), (3, 0)),
    "decimal.Decimal.next_toward": ((2, 6), (3, 0)),
    "decimal.Decimal.number_class": ((2, 6), (3, 0)),
    "decimal.Decimal.radix": ((2, 6), (3, 0)),
    "decimal.Decimal.rotate": ((2, 6), (3, 0)),
    "decimal.Decimal.scaleb": ((2, 6), (3, 0)),
    "decimal.Decimal.shift": ((2, 6), (3, 0)),
    "decimal.Decimal.to_integral_exact": ((2, 6), (3, 0)),
    "decimal.Decimal.to_integral_value": ((2, 6), (3, 0)),
    "decimal.IEEEContext": (None, (3, 14)),
    "decimal.localcontext": ((2, 5), (3, 0)),
    "difflib.SequenceMatcher.get_grouped_opcodes": ((2, 3), (3, 0)),
    "difflib.context_diff": ((2, 3), (3, 0)),
    "difflib.diff_bytes": (None, (3, 5)),
    "difflib.unified_diff": ((2, 3), (3, 0)),
    "dis.code_info": (None, (3, 2)),
    "dis.get_instructions": (None, (3, 4)),
    "dis.show_code": (None, (3, 2)),
    "dis.stack_effect": (None, (3, 4)),
    "doctest.DocFileSuite": ((2, 4), (3, 0)),
    "doctest.DocTestSuite": ((2, 3), (3, 0)),
    "doctest.debug": ((2, 3), (3, 0)),
    "doctest.debug_src": ((2, 4), (3, 0)),
    "doctest.register_optionflag": ((2, 4), (3, 0)),
    "doctest.script_from_examples": ((2, 4), (3, 0)),
    "doctest.set_unittest_reportflags": ((2, 4), (3, 0)),
    "doctest.testfile": ((2, 4), (3, 0)),
    "doctest.testsource": ((2, 3), (3, 0)),
    "email.generator.Generator.clone": ((2, 2), (3, 0)),
    "email.generator.Generator.flatten": ((2, 2), (3, 0)),
    "email.message.EmailMessage.get_content_disposition": (None, (3, 5)),
    "email.message.Message.__bytes__": (None, (3, 4)),
    "email.message.Message.as_bytes": (None, (3, 4)),
    "email.message.Message.del_param": ((2, 2), (3, 0)),
    "email.message.Message.get_charset": ((2, 2), (3, 0)),
    "email.message.Message.get_content_charset": ((2, 2), (3, 0)),
    "email.message.Message.get_content_disposition": (None, (3, 5)),
    "email.message.Message.get_content_maintype": ((2, 2), (3, 0)),
    "email.message.Message.get_content_subtype": ((2, 2), (3, 0)),
    "email.message.Message.get_content_type": ((2, 2), (3, 0)),
    "email.message.Message.get_default_type": ((2, 2), (3, 0)),
    "email.message.Message.replace_header": ((2, 2), (3, 0)),
    "email.message.Message.set_charset": ((2, 2), (3, 0)),
    "email.message.Message.set_default_type": ((2, 2), (3, 0)),
    "email.message.Message.set_param": ((2, 2), (3, 0)),
    "email.message.Message.set_type": ((2, 2), (3, 0)),
    "email.message_from_binary_file": (None, (3, 2)),
    "email.message_from_bytes": (None, (3, 2)),
    "email.utils.format_datetime": (None, (3, 3)),
    "email.utils.formatdate": ((2, 4), (3, 0)),
    "email.utils.localtime": (None, (3, 3)),
    "email.utils.parsedate_to_datetime": (None, (3, 3)),
    "enum.Enum._add_alias_": (None, (3, 13)),
    "enum.Enum._add_value_alias_": (None, (3, 13)),
    "enum.show_flag_values": (None, (3, 11)),
    "faulthandler.dump_c_stack": (None, (3, 14)),
    "filecmp.clear_cache": (None, (3, 4)),
    "fileinput.fileno": ((2, 5), (3, 0)),
    "fileinput.hook_compressed": ((2, 5), (3, 0)),
    "fileinput.hook_encoded": ((2, 5), (3, 0)),
    "fnmatch.filter": ((2, 2), (3, 0)),
    "fnmatch.filterfalse": (None, (3, 14)),
    "fractions.Fraction.as_integer_ratio": (None, (3, 8)),
    "fractions.Fraction.from_number": (None, (3, 14)),
    "fractions.Fraction.is_integer": (None, (3, 12)),
    "ftplib.FTP.mlsd": (None, (3, 3)),
    "ftplib.FTP_TLS.ccc": (None, (3, 3)),
    "functools.cmp_to_key": ((2, 7), (3, 2)),
    "functools.reduce": ((2, 6), (3, 0)),
    "gc.freeze": (None, (3, 7)),
    "gc.get_count": ((2, 5), (3, 0)),
    "gc.get_freeze_count": (None, (3, 7)),
    "gc.get_objects": ((2, 2), (3, 0)),
    "gc.get_referents": ((2, 3), (3, 0)),
    "gc.get_referrers": ((2, 2), (3, 0)),
    "gc.get_stats": (None, (3, 4)),
    "gc.is_finalized": (None, (3, 9)),
    "gc.is_tracked": ((2, 7), (3, 1)),
    "gc.unfreeze": (None, (3, 7)),
    "getopt.gnu_getopt": ((2, 3), (3, 0)),
    "gettext.GNUTranslations.lgettext": ((2, 4), (3, 0)),
    "gettext.GNUTranslations.lngettext": ((2, 4), (3, 0)),
    "gettext.GNUTranslations.ngettext": ((2, 3), (3, 0)),
    "gettext.GNUTranslations.npgettext": (None, (3, 8)),
    "gettext.GNUTranslations.pgettext": (None, (3, 8)),
    "gettext.GNUTranslations.ungettext": ((2, 3), None),
    "gettext.NullTranslations.lgettext": ((2, 4), (3, 0)),
    "gettext.NullTranslations.lngettext": ((2, 4), (3, 0)),
    "gettext.NullTranslations.ngettext": ((2, 3), (3, 0)),
    "gettext.NullTranslations.npgettext": (None, (3, 8)),
    "gettext.NullTranslations.output_charset": ((2, 4), (3, 0)),
    "gettext.NullTranslations.pgettext": (None, (3, 8)),
    "gettext.NullTranslations.set_output_charset": ((2, 4), (3, 0)),
    "gettext.NullTranslations.ungettext": ((2, 3), None),
    "gettext.bind_textdomain_codeset": ((2, 4), (3, 0)),
    "gettext.dngettext": ((2, 3), (3, 0)),
    "gettext.dnpgettext": (None, (3, 8)),
    "gettext.dpgettext": (None, (3, 8)),
    "gettext.ldgettext": ((2, 4), (3, 0)),
    "gettext.ldngettext": ((2, 4), (3, 0)),
    "gettext.lgettext": ((2, 4), (3, 0)),
    "gettext.lngettext": ((2, 4), (3, 0)),
    "gettext.ngettext": ((2, 3), (3, 0)),
    "gettext.npgettext": (None, (3, 8)),
    "gettext.pgettext": (None, (3, 8)),
    "glob.escape": (None, (3, 4)),
    "glob.iglob": ((2, 5), (3, 0)),
    "glob.translate": (None, (3, 13)),
    "gzip.GzipFile.peek": (None, (3, 2)),
    "gzip.compress": (None, (3, 2)),
    "gzip.decompress": (None, (3, 2)),
    "hashlib.blake2b": (None, (3, 6)),
    "hashlib.blake2s": (None, (3, 6)),
    "hashlib.file_digest": (None, (3, 11)),
    "hashlib.pbkdf2_hmac": ((2, 7), (3, 4)),
    "hashlib.scrypt": (None, (3, 6)),
    "hashlib.sha3_224": (None, (3, 6)),
    "hashlib.sha3_256": (None, (3, 6)),
    "hashlib.sha3_384": (None, (3, 6)),
    "hashlib.sha3_512": (None, (3, 6)),
    "hashlib.shake_128": (None, (3, 6)),
    "hashlib.shake_256": (None, (3, 6)),
    "heapq.heapify_max": (None, (3, 14)),
    "heapq.heappop_max": (None, (3, 14)),
    "heapq.heappush_max": (None, (3, 14)),
    "heapq.heappushpop": ((2, 6), (3, 0)),
    "heapq.heappushpop_max": (None, (3, 14)),
    "heapq.heapreplace_max": (None, (3, 14)),
    "heapq.merge": ((2, 6), (3, 0)),
    "heapq.nlargest": ((2, 4), (3, 0)),
    "heapq.nsmallest": ((2, 4), (3, 0)),
    "hmac.compare_digest": ((2, 7), (3, 3)),
    "hmac.digest": (None, (3, 7)),
    "html.escape": (None, (3, 2)),
    "html.unescape": (None, (3, 4)),
    "http.client.HTTPConnection.set_debuglevel": (None, (3, 1)),
    "http.client.HTTPConnection.set_tunnel": (None, (3, 2)),
    "http.client.HTTPResponse.readinto": (None, (3, 3)),
    "http.server.BaseHTTPRequestHandler.flush_headers": (None, (3, 3)),
    "http.server.BaseHTTPRequestHandler.handle_expect_100": (None, (3, 2)),
    "http.server.BaseHTTPRequestHandler.send_response_only": (None, (3, 2)),
    "httplib.HTTPConnection.set_tunnel": ((2, 7), None),
    "httplib.HTTPResponse.getheaders": ((2, 4), None),
    "imaplib.IMAP4.Idler.burst": (None, (3, 14)),
    "imaplib.IMAP4.deleteacl": ((2, 4), (3, 0)),
    "imaplib.IMAP4.enable": (None, (3, 5)),
    "imaplib.IMAP4.getannotation": ((2, 5), (3, 0)),
    "imaplib.IMAP4.getquota": ((2, 3), (3, 0)),
    "imaplib.IMAP4.getquotaroot": ((2, 3), (3, 0)),
    "imaplib.IMAP4.idle": (None, (3, 14)),
    "imaplib.IMAP4.login_cram_md5": ((2, 3), (3, 0)),
    "imaplib.IMAP4.myrights": ((2, 4), (3, 0)),
    "imaplib.IMAP4.namespace": ((2, 3), (3, 0)),
    "imaplib.IMAP4.proxyauth": ((2, 3), (3, 0)),
    "imaplib.IMAP4.setannotation": ((2, 5), (3, 0)),
    "imaplib.IMAP4.setquota": ((2, 3), (3, 0)),
    "imaplib.IMAP4.starttls": (None, (3, 2)),
    "imaplib.IMAP4.thread": ((2, 4), (3, 0)),
    "imaplib.IMAP4.unselect": (None, (3, 9)),
    "imp.acquire_lock": ((2, 3), (3, 0)),
    "imp.cache_from_source": (None, (3, 2)),
    "imp.get_tag": (None, (3, 2)),
    "imp.release_lock": ((2, 3), (3, 0)),
    "imp.source_from_cache": (None, (3, 2)),
    "importlib.abc.InspectLoader.exec_module": (None, (3, 4)),
    "importlib.abc.InspectLoader.source_to_code": (None, (3, 4)),
    "importlib.abc.Loader.create_module": (None, (3, 4)),
    "importlib.abc.Loader.exec_module": (None, (3, 4)),
    "importlib.abc.Loader.module_repr": (None, (3, 3)),
    "importlib.abc.MetaPathFinder.find_spec": (None, (3, 4)),
    "importlib.abc.PathEntryFinder.find_spec": (None, (3, 4)),
    "importlib.abc.SourceLoader.exec_module": (None, (3, 4)),
    "importlib.abc.SourceLoader.path_stats": (None, (3, 3)),
    "importlib.find_loader": (None, (3, 3)),
    "importlib.invalidate_caches": (None, (3, 3)),
    "importlib.machinery.BuiltinImporter.create_module": (None, (3, 5)),
    "importlib.machinery.BuiltinImporter.exec_module": (None, (3, 5)),
    "importlib.machinery.ExtensionFileLoader.create_module": (None, (3, 5)),
    "importlib.machinery.ExtensionFileLoader.exec_module": (None, (3, 5)),
    "importlib.machinery.ExtensionFileLoader.get_filename": (None, (3, 4)),
    "importlib.machinery.FileFinder.find_spec": (None, (3, 4)),
    "importlib.machinery.FrozenImporter.create_module": (None, (3, 4)),
    "importlib.machinery.FrozenImporter.exec_module": (None, (3, 4)),
    "importlib.machinery.PathFinder.find_spec": (None, (3, 4)),
    "importlib.machinery.all_suffixes": (None, (3, 3)),
    "importlib.metadata.packages_distributions": (None, (3, 10)),
    "importlib.reload": (None, (3, 4)),
    "importlib.resources.as_file": (None, (3, 9)),
    "importlib.resources.files": (None, (3, 9)),
    "importlib.util.cache_from_source": (None, (3, 4)),
    "importlib.util.decode_source": (None, (3, 4)),
    "importlib.util.find_spec": (None, (3, 4)),
    "importlib.util.module_from_spec": (None, (3, 5)),
    "importlib.util.resolve_name": (None, (3, 3)),
    "importlib.util.source_from_cache": (None, (3, 4)),
    "importlib.util.source_hash": (None, (3, 7)),
    "importlib.util.spec_from_file_location": (None, (3, 4)),
    "importlib.util.spec_from_loader": (None, (3, 4)),
    "inspect.BoundArguments.apply_defaults": (None, (3, 5)),
    "inspect.Signature.from_callable": (None, (3, 5)),
    "inspect.cleandoc": ((2, 6), (3, 0)),
    "inspect.get_annotations": (None, (3, 10)),
    "inspect.getasyncgenlocals": (None, (3, 12)),
    "inspect.getasyncgenstate": (None, (3, 12)),
    "inspect.getattr_static": (None, (3, 2)),
    "inspect.getcallargs": ((2, 7), (3, 2)),
    "inspect.getclosurevars": (None, (3, 3)),
    "inspect.getcoroutinelocals": (None, (3, 5)),
    "inspect.getcoroutinestate": (None, (3, 5)),
    "inspect.getgeneratorlocals": (None, (3, 3)),
    "inspect.getgeneratorstate": (None, (3, 2)),
    "inspect.getmembers_static": (None, (3, 11)),
    "inspect.isabstract": ((2, 6), (3, 0)),
    "inspect.isasyncgen": (None, (3, 6)),
    "inspect.isasyncgenfunction": (None, (3, 6)),
    "inspect.isawaitable": (None, (3, 5)),
    "inspect.iscoroutine": (None, (3, 5)),
    "inspect.iscoroutinefunction": (None, (3, 5)),
    "inspect.isdatadescriptor": ((2, 3), (3, 0)),
    "inspect.isgenerator": ((2, 6), (3, 0)),
    "inspect.isgeneratorfunction": ((2, 6), (3, 0)),
    "inspect.isgetsetdescriptor": ((2, 5), (3, 0)),
    "inspect.ismemberdescriptor": ((2, 5), (3, 0)),
    "inspect.ismethodwrapper": (None, (3, 11)),
    "inspect.ispackage": (None, (3, 14)),
    "inspect.markcoroutinefunction": (None, (3, 12)),
    "inspect.signature": (None, (3, 3)),
    "inspect.unwrap": (None, (3, 4)),
    "io.BufferedIOBase.detach": ((2, 7), (3, 1)),
    "io.BufferedIOBase.readinto1": (None, (3, 5)),
    "io.BytesIO.getbuffer": (None, (3, 2)),
    "io.BytesIO.readinto1": (None, (3, 5)),
    "io.TextIOBase.detach": ((2, 7), (3, 1)),
    "io.TextIOWrapper.reconfigure": (None, (3, 7)),
    "io.open_code": (None, (3, 8)),
    "io.text_encoding": (None, (3, 10)),
    "ipaddress.IPv4Address.__format__": (None, (3, 9)),
    "ipaddress.IPv4Network.subnet_of": bpv("ipaddress", (None, (3, 7)), config),
    "ipaddress.IPv4Network.supernet_of": bpv("ipaddress", (None, (3, 7)), config),
    "ipaddress.IPv6Address.__format__": (None, (3, 9)),
    "itertools.accumulate": (None, (3, 2)),
    "itertools.batched": (None, (3, 12)),
    "itertools.chain.from_iterable": ((2, 6), (3, 0)),
    "itertools.combinations": ((2, 6), (3, 0)),
    "itertools.combinations_with_replacement": ((2, 7), (3, 1)),
    "itertools.compress": ((2, 7), (3, 1)),
    "itertools.groupby": ((2, 4), (3, 0)),
    "itertools.izip_longest": ((2, 6), None),
    "itertools.pairwise": (None, (3, 10)),
    "itertools.permutations": ((2, 6), (3, 0)),
    "itertools.product": ((2, 6), (3, 0)),
    "itertools.tee": ((2, 4), (3, 0)),
    "keyword.issoftkeyword": (None, (3, 9)),
    "linecache.lazycache": (None, (3, 5)),
    "locale.currency": ((2, 5), (3, 0)),
    "locale.delocalize": (None, (3, 5)),
    "locale.format_string": ((2, 5), (3, 0)),
    "locale.getdefaultlocale": ((2, 0), (3, 0)),
    "locale.getencoding": (None, (3, 11)),
    "locale.getlocale": ((2, 0), (3, 0)),
    "locale.getpreferredencoding": ((2, 3), (3, 0)),
    "locale.localize": (None, (3, 10)),
    "locale.normalize": ((2, 0), (3, 0)),
    "locale.resetlocale": ((2, 0), (3, 0)),
    "logging.Logger.getChild": ((2, 7), (3, 2)),
    "logging.Logger.hasHandlers": (None, (3, 2)),
    "logging.LoggerAdapter._log": (None, (3, 6)),
    "logging.LoggerAdapter.getEffectiveLevel": (None, (3, 2)),
    "logging.LoggerAdapter.hasHandlers": (None, (3, 2)),
    "logging.LoggerAdapter.isEnabledFor": ((2, 7), (3, 2)),
    "logging.LoggerAdapter.setLevel": (None, (3, 2)),
    "logging.StreamHandler.setStream": (None, (3, 7)),
    "logging.config.dictConfig": ((2, 7), (3, 2)),
    "logging.getLevelNamesMapping": (None, (3, 11)),
    "logging.getLogRecordFactory": (None, (3, 2)),
    "logging.handlers.BaseRotatingHandler.rotate": (None, (3, 3)),
    "logging.handlers.BaseRotatingHandler.rotation_filename": (None, (3, 3)),
    "logging.handlers.QueueListener.enqueue_sentinel": (None, (3, 3)),
    "logging.handlers.WatchedFileHandler.reopenIfNeeded": (None, (3, 6)),
    "logging.setLogRecordFactory": (None, (3, 2)),
    "mailbox.Mailbox.get_bytes": (None, (3, 2)),
    "math.acosh": ((2, 6), (3, 0)),
    "math.asinh": ((2, 6), (3, 0)),
    "math.atanh": ((2, 6), (3, 0)),
    "math.cbrt": (None, (3, 11)),
    "math.comb": (None, (3, 8)),
    "math.copysign": ((2, 6), (3, 0)),
    "math.dist": (None, (3, 8)),
    "math.erf": ((2, 7), (3, 2)),
    "math.erfc": ((2, 7), (3, 2)),
    "math.exp2": (None, (3, 11)),
    "math.expm1": ((2, 7), (3, 2)),
    "math.factorial": ((2, 6), (3, 0)),
    "math.fma": (None, (3, 13)),
    "math.fsum": ((2, 6), (3, 0)),
    "math.gamma": ((2, 7), (3, 2)),
    "math.gcd": (None, (3, 5)),
    "math.isclose": (None, (3, 5)),
    "math.isfinite": (None, (3, 2)),
    "math.isinf": ((2, 6), (3, 0)),
    "math.isnan": ((2, 6), (3, 0)),
    "math.isqrt": (None, (3, 8)),
    "math.lcm": (None, (3, 9)),
    "math.lgamma": ((2, 7), (3, 2)),
    "math.log1p": ((2, 6), (3, 0)),
    "math.log2": (None, (3, 3)),
    "math.nextafter": (None, (3, 9)),
    "math.perm": (None, (3, 8)),
    "math.prod": (None, (3, 8)),
    "math.remainder": (None, (3, 7)),
    "math.sumprod": (None, (3, 12)),
    "math.trunc": ((2, 6), (3, 0)),
    "math.ulp": (None, (3, 9)),
    "mimetypes.MimeTypes.read_windows_registry": ((2, 7), (3, 2)),
    "mimetypes.MimeTypes.guess_file_type": (None, (3, 13)),
    "mimetypes.guess_file_type": (None, (3, 13)),
    "mmap.mmap.madvise": (None, (3, 8)),
    "mmap.mmap.seekable": (None, (3, 13)),
    "msilib.Database.Close": (None, (3, 7)),
    "msvcrt.getwch": ((2, 6), (3, 0)),
    "msvcrt.getwche": ((2, 6), (3, 0)),
    "msvcrt.putwch": ((2, 6), (3, 0)),
    "msvcrt.ungetwch": ((2, 6), (3, 0)),
    "multiprocessing.Condition.wait_for": (None, (3, 3)),
    "multiprocessing.Lock.locked": (None, (3, 14)),
    "multiprocessing.Pool.starmap": (None, (3, 3)),
    "multiprocessing.Pool.starmap_async": (None, (3, 3)),
    "multiprocessing.Process.close": (None, (3, 7)),
    "multiprocessing.Process.interrupt": (None, (3, 14)),
    "multiprocessing.Process.kill": (None, (3, 7)),
    "multiprocessing.RLock.locked": (None, (3, 14)),
    "multiprocessing.SimpleQueue.close": (None, (3, 9)),
    "multiprocessing.connection.wait": (None, (3, 3)),
    "multiprocessing.get_all_start_methods": (None, (3, 4)),
    "multiprocessing.get_context": (None, (3, 4)),
    "multiprocessing.get_start_method": (None, (3, 4)),
    "multiprocessing.managers.SyncManager.Barrier": (None, (3, 3)),
    "multiprocessing.managers.SyncManager.Condition.wait_for": (None, (3, 3)),
    "multiprocessing.parent_process": (None, (3, 8)),
    "multiprocessing.pool.Pool.starmap": (None, (3, 3)),
    "multiprocessing.pool.Pool.starmap_async": (None, (3, 3)),
    "multiprocessing.set_start_method": (None, (3, 4)),
    "nis.get_default_domain": ((2, 5), (3, 0)),
    "nntplib.NNTP.description": ((2, 4), (3, 0)),
    "nntplib.NNTP.descriptions": ((2, 4), (3, 0)),
    "nntplib.NNTP.getcapabilities": (None, (3, 2)),
    "nntplib.NNTP.login": (None, (3, 2)),
    "nntplib.NNTP.over": (None, (3, 2)),
    "nntplib.NNTP.starttls": (None, (3, 2)),
    "operator.__call__": (None, (3, 11)),
    "operator.__eq__": ((2, 2), (3, 0)),
    "operator.__floordiv__": ((2, 2), (3, 0)),
    "operator.__ge__": ((2, 2), (3, 0)),
    "operator.__gt__": ((2, 2), (3, 0)),
    "operator.__iadd__": ((2, 5), (3, 0)),
    "operator.__iand__": ((2, 5), (3, 0)),
    "operator.__iconcat__": ((2, 5), (3, 0)),
    "operator.__idiv__": ((2, 5), None),
    "operator.__ifloordiv__": ((2, 5), (3, 0)),
    "operator.__ilshift__": ((2, 5), (3, 0)),
    "operator.__imatmul__": (None, (3, 5)),
    "operator.__imod__": ((2, 5), (3, 0)),
    "operator.__imul__": ((2, 5), (3, 0)),
    "operator.__index__": ((2, 5), (3, 0)),
    "operator.__ior__": ((2, 5), (3, 0)),
    "operator.__ipow__": ((2, 5), (3, 0)),
    "operator.__irepeat__": ((2, 5), None),
    "operator.__irshift__": ((2, 5), (3, 0)),
    "operator.__isub__": ((2, 5), (3, 0)),
    "operator.__itruediv__": ((2, 5), (3, 0)),
    "operator.__ixor__": ((2, 5), (3, 0)),
    "operator.__le__": ((2, 2), (3, 0)),
    "operator.__lt__": ((2, 2), (3, 0)),
    "operator.__matmul__": (None, (3, 5)),
    "operator.__ne__": ((2, 2), (3, 0)),
    "operator.__pow__": ((2, 3), (3, 0)),
    "operator.__repeat__": ((2, 0), None),
    "operator.__truediv__": ((2, 2), (3, 0)),
    "operator.attrgetter": ((2, 4), (3, 0)),
    "operator.call": (None, (3, 11)),
    "operator.eq": ((2, 2), (3, 0)),
    "operator.floordiv": ((2, 2), (3, 0)),
    "operator.ge": ((2, 2), (3, 0)),
    "operator.gt": ((2, 2), (3, 0)),
    "operator.iadd": ((2, 5), (3, 0)),
    "operator.iand": ((2, 5), (3, 0)),
    "operator.iconcat": ((2, 5), (3, 0)),
    "operator.idiv": ((2, 5), None),
    "operator.ifloordiv": ((2, 5), (3, 0)),
    "operator.ilshift": ((2, 5), (3, 0)),
    "operator.imatmul": (None, (3, 5)),
    "operator.imod": ((2, 5), (3, 0)),
    "operator.imul": ((2, 5), (3, 0)),
    "operator.index": ((2, 5), (3, 0)),
    "operator.ior": ((2, 5), (3, 0)),
    "operator.ipow": ((2, 5), (3, 0)),
    "operator.irepeat": ((2, 5), None),
    "operator.irshift": ((2, 5), (3, 0)),
    "operator.is_": ((2, 3), (3, 0)),
    "operator.is_none": (None, (3, 14)),
    "operator.is_not": ((2, 3), (3, 0)),
    "operator.is_not_none": (None, (3, 14)),
    "operator.isub": ((2, 5), (3, 0)),
    "operator.itemgetter": ((2, 4), (3, 0)),
    "operator.itruediv": ((2, 5), (3, 0)),
    "operator.ixor": ((2, 5), (3, 0)),
    "operator.le": ((2, 2), (3, 0)),
    "operator.length_hint": (None, (3, 4)),
    "operator.lt": ((2, 2), (3, 0)),
    "operator.matmul": (None, (3, 5)),
    "operator.methodcaller": ((2, 6), (3, 0)),
    "operator.ne": ((2, 2), (3, 0)),
    "operator.pow": ((2, 3), (3, 0)),
    "operator.repeat": ((2, 0), None),
    "operator.truediv": ((2, 2), (3, 0)),
    "os.DirEntry.is_junction": (None, (3, 12)),
    "os.WCOREDUMP": ((2, 3), (3, 0)),
    "os.WIFCONTINUED": ((2, 3), (3, 0)),
    "os.add_dll_directory": (None, (3, 8)),
    "os.chflags": ((2, 6), (3, 0)),
    "os.chroot": ((2, 2), (3, 0)),
    "os.closerange": ((2, 6), (3, 0)),
    "os.copy_file_range": (None, (3, 8)),
    "os.cpu_count": (None, (3, 4)),
    "os.eventfd": (None, (3, 10)),
    "os.eventfd_read": (None, (3, 10)),
    "os.eventfd_write": (None, (3, 10)),
    "os.fchdir": ((2, 3), (3, 0)),
    "os.fchmod": ((2, 6), (3, 0)),
    "os.fchown": ((2, 6), (3, 0)),
    "os.fsdecode": (None, (3, 2)),
    "os.fsencode": (None, (3, 2)),
    "os.fspath": (None, (3, 6)),
    "os.fwalk": (None, (3, 3)),
    "os.get_blocking": (None, (3, 5)),
    "os.get_exec_path": (None, (3, 2)),
    "os.get_handle_inheritable": (None, (3, 4)),
    "os.get_inheritable": (None, (3, 4)),
    "os.get_terminal_size": (None, (3, 3)),
    "os.getcwdu": ((2, 3), None),
    "os.getenvb": (None, (3, 2)),
    "os.getgrouplist": (None, (3, 3)),
    "os.getloadavg": ((2, 3), (3, 0)),
    "os.getpgid": ((2, 3), (3, 0)),
    "os.getpriority": (None, (3, 3)),
    "os.getrandom": (None, (3, 6)),
    "os.getresgid": ((2, 7), (3, 2)),
    "os.getresuid": ((2, 7), (3, 2)),
    "os.getsid": ((2, 4), (3, 0)),
    "os.getxattr": (None, (3, 3)),
    "os.grantpt": (None, (3, 13)),
    "os.initgroups": ((2, 7), (3, 2)),
    "os.lchflags": ((2, 6), (3, 0)),
    "os.lchmod": ((2, 6), (3, 0)),
    "os.lchown": ((2, 3), (3, 0)),
    "os.listdrives": (None, (3, 12)),
    "os.listmounts": (None, (3, 12)),
    "os.listvolumes": (None, (3, 12)),
    "os.listxattr": (None, (3, 3)),
    "os.lockf": (None, (3, 3)),
    "os.login_tty": (None, (3, 11)),
    "os.major": ((2, 3), (3, 0)),
    "os.makedev": ((2, 3), (3, 0)),
    "os.memfd_create": (None, (3, 8)),
    "os.minor": ((2, 3), (3, 0)),
    "os.mknod": ((2, 3), (3, 0)),
    "os.path.commonpath": (None, (3, 5)),
    "os.path.getctime": ((2, 3), (3, 0)),
    "os.path.isdevdrive": (None, (3, 12)),
    "os.path.isjunction": (None, (3, 12)),
    "os.path.ismount": (None, (3, 4)),
    "os.path.isreserved": (None, (3, 13)),
    "os.path.lexists": ((2, 4), (3, 0)),
    "os.path.realpath": ((2, 2), (3, 0)),
    "os.path.relpath": ((2, 6), (3, 0)),
    "os.path.splitroot": (None, (3, 12)),
    "os.pidfd_open": (None, (3, 9)),
    "os.pipe2": (None, (3, 3)),
    "os.popen2": ((2, 0), None),
    "os.popen3": ((2, 0), None),
    "os.popen4": ((2, 0), None),
    "os.posix_fadvise": (None, (3, 3)),
    "os.posix_fallocate": (None, (3, 3)),
    "os.posix_openpt": (None, (3, 13)),
    "os.posix_spawn": (None, (3, 8)),
    "os.posix_spawnp": (None, (3, 8)),
    "os.pread": (None, (3, 3)),
    "os.preadv": (None, (3, 7)),
    "os.process_cpu_count": (None, (3, 13)),
    "os.ptsname": (None, (3, 13)),
    "os.pwrite": (None, (3, 3)),
    "os.pwritev": (None, (3, 7)),
    "os.readinto": (None, (3, 14)),
    "os.readv": (None, (3, 3)),
    "os.register_at_fork": (None, (3, 7)),
    "os.reload_environ": (None, (3, 14)),
    "os.removexattr": (None, (3, 3)),
    "os.replace": (None, (3, 3)),
    "os.scandir": (None, (3, 5)),
    "os.scandir.close": (None, (3, 6)),
    "os.sched_get_priority_max": (None, (3, 3)),
    "os.sched_get_priority_min": (None, (3, 3)),
    "os.sched_getaffinity": (None, (3, 3)),
    "os.sched_getparam": (None, (3, 3)),
    "os.sched_getscheduler": (None, (3, 3)),
    "os.sched_rr_get_interval": (None, (3, 3)),
    "os.sched_setaffinity": (None, (3, 3)),
    "os.sched_setparam": (None, (3, 3)),
    "os.sched_setscheduler": (None, (3, 3)),
    "os.sched_yield": (None, (3, 3)),
    "os.sendfile": (None, (3, 3)),
    "os.set_blocking": (None, (3, 5)),
    "os.set_handle_inheritable": (None, (3, 4)),
    "os.set_inheritable": (None, (3, 4)),
    "os.setgroups": ((2, 2), (3, 0)),
    "os.setns": (None, (3, 12)),
    "os.setpriority": (None, (3, 3)),
    "os.setresgid": ((2, 7), (3, 2)),
    "os.setresuid": ((2, 7), (3, 2)),
    "os.setxattr": (None, (3, 3)),
    "os.splice": (None, (3, 10)),
    "os.startfile": ((2, 0), (3, 0)),
    "os.sync": (None, (3, 3)),
    "os.timerfd_create": (None, (3, 13)),
    "os.timerfd_gettime": (None, (3, 13)),
    "os.timerfd_gettime_ns": (None, (3, 13)),
    "os.timerfd_settime": (None, (3, 13)),
    "os.timerfd_settime_ns": (None, (3, 13)),
    "os.truncate": (None, (3, 3)),
    "os.unlockpt": (None, (3, 13)),
    "os.unshare": (None, (3, 12)),
    "os.urandom": ((2, 4), (3, 0)),
    "os.wait3": ((2, 5), (3, 0)),
    "os.wait4": ((2, 5), (3, 0)),
    "os.waitid": (None, (3, 3)),
    "os.waitstatus_to_exitcode": (None, (3, 9)),
    "os.walk": ((2, 3), (3, 0)),
    "os.writev": (None, (3, 3)),
    "pathlib.Path.copy": (None, (3, 14)),
    "pathlib.Path.copy_into": (None, (3, 14)),
    "pathlib.Path.expanduser": (None, (3, 5)),
    "pathlib.Path.from_uri": (None, (3, 13)),
    "pathlib.Path.hardlink_to": (None, (3, 10)),
    "pathlib.Path.home": (None, (3, 5)),
    "pathlib.Path.is_junction": (None, (3, 12)),
    "pathlib.Path.is_mount": (None, (3, 7)),
    "pathlib.Path.link_to": (None, (3, 8)),
    "pathlib.Path.move": (None, (3, 14)),
    "pathlib.Path.move_into": (None, (3, 14)),
    "pathlib.Path.read_bytes": (None, (3, 5)),
    "pathlib.Path.read_text": (None, (3, 5)),
    "pathlib.Path.readlink": (None, (3, 9)),
    "pathlib.Path.samefile": (None, (3, 5)),
    "pathlib.Path.walk": (None, (3, 12)),
    "pathlib.Path.write_bytes": (None, (3, 5)),
    "pathlib.Path.write_text": (None, (3, 5)),
    "pathlib.PurePath.full_match": (None, (3, 13)),
    "pathlib.PurePath.is_relative_to": (None, (3, 9)),
    "pathlib.PurePath.parser": (None, (3, 13)),
    "pathlib.PurePath.with_segments": (None, (3, 12)),
    "pathlib.PurePath.with_stem": (None, (3, 9)),
    "pdb.set_trace_async": (None, (3, 14)),
    "pdb.set_default_backend": (None, (3, 14)),
    "pdb.get_default_backend": (None, (3, 14)),
    "pickle.Pickler.reducer_override": (None, (3, 8)),
    "pickletools.optimize": ((2, 6), (3, 0)),
    "pkgutil.get_data": ((2, 6), (3, 0)),
    "pkgutil.resolve_name": (None, (3, 9)),
    "platform.freedesktop_os_release": (None, (3, 10)),
    "platform.invalidate_caches": (None, (3, 14)),
    "platform.linux_distribution": ((2, 6), (3, 0)),
    "platform.python_branch": ((2, 6), (3, 0)),
    "platform.python_implementation": ((2, 6), (3, 0)),
    "platform.python_revision": ((2, 6), (3, 0)),
    "platform.win32_edition": (None, (3, 8)),
    "platform.win32_is_iot": (None, (3, 8)),
    "plistlib.dump": (None, (3, 4)),
    "plistlib.dumps": (None, (3, 4)),
    "plistlib.load": (None, (3, 4)),
    "plistlib.loads": (None, (3, 4)),
    "poplib.POP3.capa": (None, (3, 4)),
    "poplib.POP3.stls": (None, (3, 4)),
    "poplib.POP3.utf8": (None, (3, 5)),
    "pprint.PrettyPrinter.format": ((2, 3), (3, 0)),
    "pprint.pp": (None, (3, 8)),
    "pstats.Stats.get_stats_profile": (None, (3, 9)),
    "queue.Queue.shutdown": (None, (3, 13)),
    "random.binomialvariate": (None, (3, 12)),
    "random.choices": (None, (3, 6)),
    "random.getrandbits": ((2, 4), (3, 0)),
    "random.getstate": ((2, 1), (3, 0)),
    "random.jumpahead": ((2, 1), None),
    "random.randbytes": (None, (3, 9)),
    "random.sample": ((2, 3), (3, 0)),
    "random.setstate": ((2, 1), (3, 0)),
    "random.triangular": ((2, 6), (3, 0)),
    "re.Pattern.fullmatch": (None, (3, 4)),
    "re.finditer": ((2, 2), (3, 0)),
    "re.fullmatch": (None, (3, 4)),
    "readline.append_history_file": (None, (3, 5)),
    "readline.clear_history": ((2, 4), (3, 0)),
    "readline.get_completer": ((2, 3), (3, 0)),
    "readline.get_completion_type": ((2, 6), (3, 0)),
    "readline.get_current_history_length": ((2, 3), (3, 0)),
    "readline.get_history_item": ((2, 3), (3, 0)),
    "readline.remove_history_item": ((2, 4), (3, 0)),
    "readline.replace_history_item": ((2, 4), (3, 0)),
    "readline.set_auto_history": (None, (3, 6)),
    "readline.set_completion_display_matches_hook": ((2, 6), (3, 0)),
    "readline.set_pre_input_hook": ((2, 3), (3, 0)),
    "readline.set_startup_hook": ((2, 3), (3, 0)),
    "resource.prlimit": (None, (3, 4)),
    "runpy.run_path": ((2, 7), (3, 2)),
    "select.devpoll": (None, (3, 3)),
    "select.devpoll.close": (None, (3, 4)),
    "select.devpoll.fileno": (None, (3, 4)),
    "select.epoll": ((2, 6), (3, 0)),
    "select.kevent": ((2, 6), (3, 0)),
    "select.kqueue": ((2, 6), (3, 0)),
    "select.poll.modify": ((2, 6), (3, 0)),
    "sgmllib.SGMLParser.convert_charref": ((2, 5), None),
    "sgmllib.SGMLParser.convert_codepoint": ((2, 5), None),
    "sgmllib.SGMLParser.convert_entityref": ((2, 5), None),
    "shlex.join": (None, (3, 8)),
    "shlex.quote": (None, (3, 3)),
    "shlex.shlex.pop_source": ((2, 1), (3, 0)),
    "shlex.shlex.push_source": ((2, 1), (3, 0)),
    "shlex.split": ((2, 3), (3, 0)),
    "shutil.chown": (None, (3, 3)),
    "shutil.disk_usage": (None, (3, 3)),
    "shutil.get_archive_formats": ((2, 7), (3, 2)),
    "shutil.get_terminal_size": (None, (3, 3)),
    "shutil.get_unpack_formats": (None, (3, 2)),
    "shutil.ignore_patterns": ((2, 6), (3, 0)),
    "shutil.make_archive": ((2, 7), (3, 2)),
    "shutil.move": ((2, 3), (3, 0)),
    "shutil.register_archive_format": ((2, 7), (3, 2)),
    "shutil.register_unpack_format": (None, (3, 2)),
    "shutil.unpack_archive": (None, (3, 2)),
    "shutil.unregister_archive_format": ((2, 7), (3, 2)),
    "shutil.unregister_unpack_format": (None, (3, 2)),
    "shutil.which": (None, (3, 3)),
    "signal.getitimer": ((2, 6), (3, 0)),
    "signal.pidfd_send_signal": (None, (3, 9)),
    "signal.pthread_kill": (None, (3, 3)),
    "signal.pthread_sigmask": (None, (3, 3)),
    "signal.raise_signal": (None, (3, 8)),
    "signal.set_wakeup_fd": ((2, 6), (3, 0)),
    "signal.setitimer": ((2, 6), (3, 0)),
    "signal.siginterrupt": ((2, 6), (3, 0)),
    "signal.sigpending": (None, (3, 3)),
    "signal.sigtimedwait": (None, (3, 3)),
    "signal.sigwait": (None, (3, 3)),
    "signal.sigwaitinfo": (None, (3, 3)),
    "signal.strsignal": (None, (3, 8)),
    "signal.valid_signals": (None, (3, 8)),
    "site.getsitepackages": ((2, 7), (3, 2)),
    "site.getuserbase": ((2, 7), (3, 2)),
    "site.getusersitepackages": ((2, 7), (3, 2)),
    "smtplib.SMTP.auth": (None, (3, 5)),
    "smtplib.SMTP.ehlo_or_helo_if_needed": ((2, 6), (3, 0)),
    "smtplib.SMTP.send_message": (None, (3, 2)),
    "smtplib.SMTP.starttls": ((2, 6), (3, 0)),
    "socket.CMSG_LEN": (None, (3, 3)),
    "socket.CMSG_SPACE": (None, (3, 3)),
    "socket.close": (None, (3, 7)),
    "socket.create_connection": ((2, 6), (3, 0)),
    "socket.create_server": (None, (3, 8)),
    "socket.fromshare": (None, (3, 3)),
    "socket.getaddrinfo": ((2, 2), (3, 0)),
    "socket.getdefaulttimeout": ((2, 3), (3, 0)),
    "socket.getnameinfo": ((2, 2), (3, 0)),
    "socket.has_dualstack_ipv6": (None, (3, 8)),
    "socket.if_indextoname": (None, (3, 3)),
    "socket.if_nameindex": (None, (3, 3)),
    "socket.if_nametoindex": (None, (3, 3)),
    "socket.inet_ntop": ((2, 3), (3, 0)),
    "socket.inet_pton": ((2, 3), (3, 0)),
    "socket.recv_fds": (None, (3, 9)),
    "socket.send_fds": (None, (3, 9)),
    "socket.setdefaulttimeout": ((2, 3), (3, 0)),
    "socket.sethostname": (None, (3, 3)),
    "socket.socket.detach": (None, (3, 2)),
    "socket.socket.get_inheritable": (None, (3, 4)),
    "socket.socket.getblocking": (None, (3, 7)),
    "socket.socket.gettimeout": ((2, 3), (3, 0)),
    "socket.socket.ioctl": ((2, 6), (3, 0)),
    "socket.socket.recv_into": ((2, 5), (3, 0)),
    "socket.socket.recvfrom_into": ((2, 5), (3, 0)),
    "socket.socket.recvmsg": (None, (3, 3)),
    "socket.socket.recvmsg_into": (None, (3, 3)),
    "socket.socket.sendfile": (None, (3, 5)),
    "socket.socket.sendmsg": (None, (3, 3)),
    "socket.socket.sendmsg_afalg": (None, (3, 6)),
    "socket.socket.set_inheritable": (None, (3, 4)),
    "socket.socket.settimeout": ((2, 3), (3, 0)),
    "socket.socket.share": (None, (3, 3)),
    "socket.socketpair": ((2, 4), (3, 0)),
    "socketserver.BaseServer.service_actions": (None, (3, 3)),
    "sysconfig._get_preferred_schemes": (None, (3, 10)),
    "sysconfig.get_default_scheme": (None, (3, 10)),
    "sysconfig.get_preferred_scheme": (None, (3, 10)),
    "sqlite3.Connection.backup": (None, (3, 7)),
    "sqlite3.Connection.blobopen": (None, (3, 11)),
    "sqlite3.Connection.create_window_function": (None, (3, 11)),
    "sqlite3.Connection.deserialize": (None, (3, 11)),
    "sqlite3.Connection.enable_load_extension": ((2, 7), (3, 2)),
    "sqlite3.Connection.getconfig": (None, (3, 12)),
    "sqlite3.Connection.getlimit": (None, (3, 11)),
    "sqlite3.Connection.in_transaction": (None, (3, 2)),
    "sqlite3.Connection.iter_dump": ((2, 6), (3, 0)),
    "sqlite3.Connection.iterdump": ((2, 6), (3, 0)),
    "sqlite3.Connection.load_extension": ((2, 7), (3, 2)),
    "sqlite3.Connection.serialize": (None, (3, 11)),
    "sqlite3.Connection.set_progress_handler": ((2, 6), (3, 0)),
    "sqlite3.Connection.set_trace_callback": (None, (3, 3)),
    "sqlite3.Connection.setconfig": (None, (3, 12)),
    "sqlite3.Connection.setlimit": (None, (3, 11)),
    "sqlite3.Row.keys": ((2, 6), (3, 0)),
    "ssl.RAND_bytes": (None, (3, 3)),
    "ssl.RAND_pseudo_bytes": (None, (3, 3)),
    "ssl.SSLContext.cert_store_stats": ((2, 7), (3, 4)),
    "ssl.SSLContext.get_ca_certs": ((2, 7), (3, 4)),
    "ssl.SSLContext.get_ciphers": (None, (3, 6)),
    "ssl.SSLContext.load_default_certs": ((2, 7), (3, 4)),
    "ssl.SSLContext.load_dh_params": ((2, 7), (3, 3)),
    "ssl.SSLContext.set_alpn_protocols": ((2, 7), (3, 5)),
    "ssl.SSLContext.set_ecdh_curve": ((2, 7), (3, 3)),
    "ssl.SSLContext.set_npn_protocols": ((2, 7), (3, 3)),
    "ssl.SSLContext.set_psk_client_callback": (None, (3, 13)),
    "ssl.SSLContext.set_psk_server_callback": (None, (3, 13)),
    "ssl.SSLContext.set_servername_callback": ((2, 7), (3, 4)),
    "ssl.SSLSocket.compression": ((2, 7), (3, 3)),
    "ssl.SSLSocket.get_channel_binding": ((2, 7), (3, 3)),
    "ssl.SSLSocket.get_unverified_chain": (None, (3, 13)),
    "ssl.SSLSocket.get_verified_chain": (None, (3, 13)),
    "ssl.SSLSocket.selected_alpn_protocol": ((2, 7), (3, 5)),
    "ssl.SSLSocket.selected_npn_protocol": ((2, 7), (3, 3)),
    "ssl.SSLSocket.sendfile": (None, (3, 5)),
    "ssl.SSLSocket.shared_ciphers": (None, (3, 5)),
    "ssl.SSLSocket.verify_client_post_handshake": (None, (3, 8)),
    "ssl.SSLSocket.version": ((2, 7), (3, 5)),
    "ssl._https_verify_certificates": ((2, 7), None),
    "ssl.create_default_context": ((2, 7), (3, 4)),
    "ssl.enum_certificates": ((2, 7), (3, 4)),
    "ssl.enum_crls": ((2, 7), (3, 4)),
    "ssl.get_default_verify_paths": ((2, 7), (3, 4)),
    "ssl.match_hostname": ((2, 7), (3, 2)),
    "stat.S_ISDOOR": (None, (3, 4)),
    "stat.S_ISPORT": (None, (3, 4)),
    "stat.S_ISWHT": (None, (3, 4)),
    "stat.filemode": (None, (3, 3)),
    "statistics.NormalDist.zscore": (None, (3, 9)),
    "statistics.correlation": (None, (3, 10)),
    "statistics.covariance": (None, (3, 10)),
    "statistics.fmean": (None, (3, 8)),
    "statistics.geometric_mean": (None, (3, 8)),
    "statistics.harmonic_mean": (None, (3, 6)),
    "statistics.kde": (None, (3, 13)),
    "statistics.kde_random": (None, (3, 13)),
    "statistics.linear_regression": (None, (3, 10)),
    "statistics.multimode": (None, (3, 8)),
    "statistics.quantiles": (None, (3, 8)),
    "string.Template.get_identifiers": (None, (3, 11)),
    "string.Template.is_valid": (None, (3, 11)),
    "string.rsplit": ((2, 4), None),
    "struct.Struct.iter_unpack": (None, (3, 4)),
    "struct.iter_unpack": (None, (3, 4)),
    "struct.pack_into": ((2, 5), (3, 0)),
    "struct.unpack_from": ((2, 5), (3, 0)),
    "subprocess.Popen.kill": ((2, 6), (3, 0)),
    "subprocess.Popen.send_signal": ((2, 6), (3, 0)),
    "subprocess.Popen.terminate": ((2, 6), (3, 0)),
    "subprocess.check_call": ((2, 5), (3, 0)),
    "subprocess.check_output": ((2, 7), (3, 1)),
    "subprocess.run": (None, (3, 5)),
    "symtable.Symbol.is_annotated": (None, (3, 6)),
    "symtable.Symbol.is_comp_cell": (None, (3, 14)),
    "symtable.Symbol.is_comp_iter": (None, (3, 14)),
    "symtable.Symbol.is_free_class": (None, (3, 14)),
    "symtable.Symbol.is_type_parameter": (None, (3, 14)),
    "sys._clear_internal_caches": (None, (3, 13)),
    "sys._clear_type_cache": ((2, 6), (3, 0)),
    "sys._current_frames": ((2, 5), (3, 0)),
    "sys._debugmallocstats": (None, (3, 3)),
    "sys._enablelegacywindowsfsencoding": (None, (3, 6)),
    "sys._is_gil_enabled": (None, (3, 13)),
    "sys._is_immortal": (None, (3, 14)),
    "sys._is_interned": (None, (3, 13)),
    "sys.addaudithook": (None, (3, 8)),
    "sys.audit": (None, (3, 8)),
    "sys.breakpointhook": (None, (3, 7)),
    "sys.exc_clear": ((2, 3), None),
    "sys.exception": (None, (3, 11)),
    "sys.get_asyncgen_hooks": (None, (3, 6)),
    "sys.get_coroutine_origin_tracking_depth": (None, (3, 7)),
    "sys.get_int_max_str_digits": (None, (3, 11)),
    "sys.getallocatedblocks": (None, (3, 4)),
    "sys.getandroidapilevel": (None, (3, 7)),
    "sys.getcheckinterval": ((2, 3), (3, 0)),
    "sys.getdlopenflags": ((2, 2), (3, 0)),
    "sys.getfilesystemencodeerrors": (None, (3, 6)),
    "sys.getfilesystemencoding": ((2, 3), (3, 0)),
    "sys.getprofile": ((2, 6), (3, 0)),
    "sys.getsizeof": ((2, 6), (3, 0)),
    "sys.getswitchinterval": (None, (3, 2)),
    "sys.gettrace": ((2, 6), (3, 0)),
    "sys.getwindowsversion": ((2, 3), (3, 0)),
    "sys.is_finalizing": (None, (3, 5)),
    "sys.remote_exec": (None, (3, 14)),
    "sys.set_asyncgen_hooks": (None, (3, 6)),
    "sys.set_coroutine_origin_tracking_depth": (None, (3, 7)),
    "sys.set_int_max_str_digits": (None, (3, 11)),
    "sys.setdefaultencoding": ((2, 0), None),
    "sys.setdlopenflags": ((2, 2), (3, 0)),
    "sys.setswitchinterval": (None, (3, 2)),
    "sys.settscdump": ((2, 4), None),
    "sys.unraisablehook": (None, (3, 8)),
    "tarfile.TarFile.extractall": ((2, 5), (3, 0)),
    "tarfile.TarInfo.fromtarfile": ((2, 6), (3, 0)),
    "tarfile.TarInfo.replace": (None, (3, 12)),
    "telnetlib.Telnet.read_sb_data": ((2, 3), (3, 0)),
    "tempfile.NamedTemporaryFile": ((2, 3), (3, 0)),
    "tempfile.SpooledTemporaryFile": ((2, 6), (3, 0)),
    "tempfile.TemporaryDirectory": (None, (3, 2)),
    "tempfile.gettempdir": ((2, 3), (3, 0)),
    "tempfile.gettempdirb": (None, (3, 5)),
    "tempfile.gettempprefixb": (None, (3, 5)),
    "tempfile.mkdtemp": ((2, 3), (3, 0)),
    "tempfile.mkstemp": ((2, 3), (3, 0)),
    "termios.tcgetwinsize": (None, (3, 11)),
    "termios.tcsetwinsize": (None, (3, 11)),
    "test.support.adjust_int_max_str_digits": (None, (3, 11)),
    "test.support.captured_stdout": ((2, 6), (3, 0)),
    "test.support.catch_threading_exception": (None, (3, 8)),
    "test.support.catch_unraisable_exception": (None, (3, 8)),
    "test.support.check__all__": (None, (3, 6)),
    "test.support.check_disallow_instantiation": (None, (3, 10)),
    "test.support.check_py3k_warnings": ((2, 7), None),
    "test.support.check_syntax_warning": (None, (3, 8)),
    "test.support.check_warnings": ((2, 6), (3, 0)),
    "test.support.detect_api_mismatch": (None, (3, 5)),
    "test.support.flush_std_streams": (None, (3, 11)),
    "test.support.import_fresh_module": ((2, 7), (3, 1)),
    "test.support.import_module": ((2, 7), (3, 1)),
    "test.support.print_warning": (None, (3, 9)),
    "test.support.skip_if_broken_multiprocessing_synchronize": (None, (3, 10)),
    "test.support.wait_process": (None, (3, 9)),
    "textwrap.indent": (None, (3, 3)),
    "textwrap.shorten": (None, (3, 4)),
    "thread.interrupt_main": ((2, 3), None),
    "thread.stack_size": ((2, 5), None),
    "threading.Condition.locked": (None, (3, 14)),
    "threading.Condition.notify_all": ((2, 6), (3, 0)),
    "threading.Condition.wait_for": (None, (3, 2)),
    "threading.Event.is_set": ((2, 6), (3, 0)),
    "threading.RLock.locked": (None, (3, 14)),
    "threading.Thread.is_alive": ((2, 6), (3, 0)),
    "threading.active_count": ((2, 6), (3, 0)),
    "threading.current_thread": ((2, 6), (3, 0)),
    "threading.excepthook": (None, (3, 8)),
    "threading.get_ident": (None, (3, 3)),
    "threading.get_native_id": (None, (3, 8)),
    "threading.getprofile": (None, (3, 10)),
    "threading.gettrace": (None, (3, 10)),
    "threading.main_thread": (None, (3, 4)),
    "threading.setprofile": ((2, 3), (3, 0)),
    "threading.setprofile_all_threads": (None, (3, 12)),
    "threading.settrace": ((2, 3), (3, 0)),
    "threading.settrace_all_threads": (None, (3, 12)),
    "threading.stack_size": ((2, 5), (3, 0)),
    "time.clock_getres": (None, (3, 3)),
    "time.clock_gettime": (None, (3, 3)),
    "time.clock_gettime_ns": (None, (3, 7)),
    "time.clock_settime": (None, (3, 3)),
    "time.clock_settime_ns": (None, (3, 7)),
    "time.get_clock_info": (None, (3, 3)),
    "time.monotonic": (None, (3, 3)),
    "time.monotonic_ns": (None, (3, 7)),
    "time.perf_counter": (None, (3, 3)),
    "time.perf_counter_ns": (None, (3, 7)),
    "time.process_time": (None, (3, 3)),
    "time.process_time_ns": (None, (3, 7)),
    "time.pthread_getcpuclockid": (None, (3, 7)),
    "time.thread_time": (None, (3, 7)),
    "time.thread_time_ns": (None, (3, 7)),
    "time.time_ns": (None, (3, 7)),
    "time.tzset": ((2, 3), (3, 0)),
    "timeit.Timer.autorange": (None, (3, 6)),
    "timeit.repeat": ((2, 6), (3, 0)),
    "timeit.timeit": ((2, 6), (3, 0)),
    "tkinter.info_patchlevel": (None, (3, 11)),
    "tokenize.generate_tokens": ((2, 2), (3, 0)),
    "tokenize.open": (None, (3, 2)),
    "tokenize.untokenize": ((2, 5), (3, 0)),
    "traceback.StackSummary.format_frame_summary": (None, (3, 11)),
    "traceback.TracebackException.print": (None, (3, 11)),
    "traceback.clear_frames": (None, (3, 4)),
    "traceback.format_exc": ((2, 4), (3, 0)),
    "traceback.walk_stack": (None, (3, 5)),
    "traceback.walk_tb": (None, (3, 5)),
    "tracemalloc.reset_peak": (None, (3, 9)),
    "turtle.fill": (None, (3, 14)),
    "turtle.no_animation": (None, (3, 14)),
    "turtle.poly": (None, (3, 14)),
    "turtle.save": (None, (3, 14)),
    "types.CodeType.replace": (None, (3, 8)),
    "types.DynamicClassAttribute": (None, (3, 4)),
    "types.FrameType.clear": (None, (3, 4)),
    "types.coroutine": (None, (3, 5)),
    "types.get_original_bases": (None, (3, 12)),
    "types.new_class": (None, (3, 3)),
    "types.prepare_class": (None, (3, 3)),
    "types.resolve_bases": (None, (3, 7)),
    "typing.ParamSpec.evaluate_default": (None, (3, 14)),
    "typing.ParamSpec.has_default": (None, (3, 13)),
    "typing.TypeAliasType.evaluate_value": (None, (3, 14)),
    "typing.TypeVar.evaluate_bound": (None, (3, 14)),
    "typing.TypeVar.evaluate_constraints": (None, (3, 14)),
    "typing.TypeVar.evaluate_default": (None, (3, 14)),
    "typing.TypeVar.has_default": (None, (3, 13)),
    "typing.TypeVarTuple.evaluate_default": (None, (3, 14)),
    "typing.TypeVarTuple.has_default": (None, (3, 13)),
    "typing.assert_never": (None, (3, 11)),
    "typing.assert_type": (None, (3, 11)),
    "typing.clear_overloads": (None, (3, 11)),
    "typing.evaluate_forward_ref": (None, (3, 14)),
    "typing.get_args": (None, (3, 8)),
    "typing.get_origin": (None, (3, 8)),
    "typing.get_overloads": (None, (3, 11)),
    "typing.get_protocol_members": (None, (3, 13)),
    "typing.is_protocol": (None, (3, 13)),
    "typing.is_typeddict": (None, (3, 10)),
    "typing.reveal_type": (None, (3, 11)),
    "unicodedata.east_asian_width": ((2, 4), (3, 0)),
    "unicodedata.is_normalized": (None, (3, 8)),
    "unicodedata.normalize": ((2, 3), (3, 0)),
    "unittest.IsolatedAsyncioTestCase.enterAsyncContext": (None, (3, 11)),
    "unittest.TestCase.addClassCleanup": (None, (3, 8)),
    "unittest.TestCase.addCleanup": ((2, 7), (3, 1)),
    "unittest.TestCase.addTypeEqualityFunc": ((2, 7), (3, 1)),
    "unittest.TestCase.assertCountEqual": (None, (3, 2)),
    "unittest.TestCase.assertDictContainsSubset": ((2, 7), (3, 1)),
    "unittest.TestCase.assertDictEqual": ((2, 7), (3, 1)),
    "unittest.TestCase.assertEndsWith": (None, (3, 14)),
    "unittest.TestCase.assertGreater": ((2, 7), (3, 1)),
    "unittest.TestCase.assertGreaterEqual": ((2, 7), (3, 1)),
    "unittest.TestCase.assertHasAttr": (None, (3, 14)),
    "unittest.TestCase.assertIn": ((2, 7), (3, 1)),
    "unittest.TestCase.assertIs": ((2, 7), (3, 1)),
    "unittest.TestCase.assertIsInstance": ((2, 7), (3, 2)),
    "unittest.TestCase.assertIsNone": ((2, 7), (3, 1)),
    "unittest.TestCase.assertIsNot": ((2, 7), (3, 1)),
    "unittest.TestCase.assertIsNotNone": ((2, 7), (3, 1)),
    "unittest.TestCase.assertIsSubclass": (None, (3, 14)),
    "unittest.TestCase.assertItemsEqual": ((2, 7), None),
    "unittest.TestCase.assertLess": ((2, 7), (3, 1)),
    "unittest.TestCase.assertLessEqual": ((2, 7), (3, 1)),
    "unittest.TestCase.assertListEqual": ((2, 7), (3, 1)),
    "unittest.TestCase.assertLogs": (None, (3, 4)),
    "unittest.TestCase.assertMultiLineEqual": ((2, 7), (3, 1)),
    "unittest.TestCase.assertNoLogs": (None, (3, 10)),
    "unittest.TestCase.assertNotEndsWith": (None, (3, 14)),
    "unittest.TestCase.assertNotHasAttr": (None, (3, 14)),
    "unittest.TestCase.assertNotIn": ((2, 7), (3, 1)),
    "unittest.TestCase.assertNotIsInstance": ((2, 7), (3, 2)),
    "unittest.TestCase.assertNotIsSubclass": (None, (3, 14)),
    "unittest.TestCase.assertNotRegex": (None, (3, 2)),
    "unittest.TestCase.assertNotRegexpMatches": ((2, 7), (3, 5)),
    "unittest.TestCase.assertNotStartsWith": (None, (3, 14)),
    "unittest.TestCase.assertRaisesRegex": (None, (3, 2)),
    "unittest.TestCase.assertRaisesRegexp": ((2, 7), (3, 1)),
    "unittest.TestCase.assertRegex": (None, (3, 2)),
    "unittest.TestCase.assertRegexpMatches": ((2, 7), (3, 1)),
    "unittest.TestCase.assertSequenceEqual": ((2, 7), (3, 1)),
    "unittest.TestCase.assertSetEqual": ((2, 7), (3, 1)),
    "unittest.TestCase.assertStartsWith": (None, (3, 14)),
    "unittest.TestCase.assertTupleEqual": ((2, 7), (3, 1)),
    "unittest.TestCase.assertWarns": (None, (3, 2)),
    "unittest.TestCase.assertWarnsRegex": (None, (3, 2)),
    "unittest.TestCase.doClassCleanups": (None, (3, 8)),
    "unittest.TestCase.doCleanups": ((2, 7), (3, 1)),
    "unittest.TestCase.enterClassContext": (None, (3, 11)),
    "unittest.TestCase.enterContext": (None, (3, 11)),
    "unittest.TestCase.setUpClass": ((2, 7), (3, 2)),
    "unittest.TestCase.skipTest": ((2, 7), (3, 1)),
    "unittest.TestCase.subTest": (None, (3, 4)),
    "unittest.TestCase.tearDownClass": ((2, 7), (3, 2)),
    "unittest.TestLoader.discover": ((2, 7), (3, 2)),
    "unittest.TestResult.addSubTest": (None, (3, 4)),
    "unittest.TestResult.startTestRun": ((2, 7), (3, 1)),
    "unittest.TestResult.stopTestRun": ((2, 7), (3, 1)),
    "unittest.addModuleCleanup": (None, (3, 8)),
    "unittest.doModuleCleanups": (None, (3, 8)),
    "unittest.enterModuleContext": (None, (3, 11)),
    "unittest.installHandler": ((2, 7), (3, 2)),
    "unittest.mock.Mock.assert_called": (None, (3, 6)),
    "unittest.mock.Mock.assert_called_once": (None, (3, 6)),
    "unittest.mock.Mock.assert_not_called": (None, (3, 5)),
    "unittest.mock.seal": bpv("mock", (None, (3, 7)), config),
    "unittest.registerResult": ((2, 7), (3, 2)),
    "unittest.removeHandler": ((2, 7), (3, 2)),
    "unittest.removeResult": ((2, 7), (3, 2)),
    "urllib.request.HTTPRedirectHandler.http_error_308": (None, (3, 11)),
    "urllib.request.Request.remove_header": (None, (3, 4)),
    "urllib.robotparser.RobotFileParser.crawl_delay": (None, (3, 6)),
    "urllib.robotparser.RobotFileParser.request_rate": (None, (3, 6)),
    "urllib.robotparser.RobotFileParser.site_maps": (None, (3, 8)),
    "urllib.urlopen.getcode": ((2, 6), None),
    "urllib2.Request.add_unredirected_header": ((2, 4), None),
    "urllib2.Request.has_header": ((2, 4), None),
    "urlparse.ParseResult.geturl": ((2, 5), None),
    "urlparse.parse_qs": ((2, 6), None),
    "urlparse.parse_qsl": ((2, 6), None),
    "urlparse.urlsplit": ((2, 2), None),
    "urlparse.urlunsplit": ((2, 2), None),
    "uuid.uuid6": (None, (3, 14)),
    "uuid.uuid7": (None, (3, 14)),
    "uuid.uuid8": (None, (3, 14)),
    "venv.EnvBuilder.create_git_ignore_file": (None, (3, 13)),
    "venv.EnvBuilder.upgrade_dependencies": (None, (3, 9)),
    "venv.create": (None, (3, 3)),
    "warnings.warnpy3k": ((2, 6), None),
    "weakref.WeakKeyDictionary.iterkeyrefs": ((2, 5), None),
    "weakref.WeakKeyDictionary.keyrefs": ((2, 5), (3, 0)),
    "weakref.WeakValueDictionary.itervaluerefs": ((2, 5), None),
    "weakref.WeakValueDictionary.valuerefs": ((2, 5), (3, 0)),
    "webbrowser.controller.open_new_tab": ((2, 5), (3, 0)),
    "webbrowser.open_new_tab": ((2, 5), (3, 0)),
    "winreg.CreateKeyEx": (None, (3, 2)),
    "winreg.DeleteKeyEx": (None, (3, 2)),
    "winsound.MessageBeep": ((2, 3), (3, 0)),
    "wsgiref.handlers.read_environ": (None, (3, 2)),
    "xml.dom.Node.normalize": ((2, 1), (3, 0)),
    "xml.dom.minidom.Node.toprettyxml": ((2, 1), (3, 0)),
    "xml.etree.ElementTree.Element.extend": ((2, 7), (3, 2)),
    "xml.etree.ElementTree.Element.iter": ((2, 7), (3, 2)),
    "xml.etree.ElementTree.Element.iterfind": ((2, 7), (3, 2)),
    "xml.etree.ElementTree.Element.itertext": ((2, 7), (3, 2)),
    "xml.etree.ElementTree.ElementTree.iterfind": ((2, 7), (3, 2)),
    "xml.etree.ElementTree.TreeBuilder.comment": (None, (3, 8)),
    "xml.etree.ElementTree.TreeBuilder.doctype": ((2, 7), (3, 2)),
    "xml.etree.ElementTree.TreeBuilder.end_ns": (None, (3, 8)),
    "xml.etree.ElementTree.TreeBuilder.pi": (None, (3, 8)),
    "xml.etree.ElementTree.TreeBuilder.start_ns": (None, (3, 8)),
    "xml.etree.ElementTree.XMLParser.flush": (None, (3, 13)),
    "xml.etree.ElementTree.XMLPullParser.flush": (None, (3, 13)),
    "xml.etree.ElementTree.canonicalize": (None, (3, 8)),
    "xml.etree.ElementTree.fromstringlist": ((2, 7), (3, 2)),
    "xml.etree.ElementTree.indent": (None, (3, 9)),
    "xml.etree.ElementTree.register_namespace": ((2, 7), (3, 2)),
    "xml.etree.ElementTree.tostringlist": ((2, 7), (3, 2)),
    "xml.parsers.expat.xmlparser.EntityDeclHandler": ((2, 1), (3, 0)),
    "xml.parsers.expat.xmlparser.GetInputContext": ((2, 1), (3, 0)),
    "xml.parsers.expat.xmlparser.UseForeignDTD": ((2, 3), (3, 0)),
    "xml.parsers.expat.xmlparser.XmlDeclHandler": ((2, 1), (3, 0)),
    "xml.parsers.expat.xmlparser.buffer_size": ((2, 3), (3, 0)),
    "xml.parsers.expat.xmlparser.buffer_text": ((2, 3), (3, 0)),
    "xml.parsers.expat.xmlparser.buffer_used": ((2, 3), (3, 0)),
    "xml.parsers.expat.xmlparser.ordered_attributes": ((2, 1), (3, 0)),
    "xml.parsers.expat.xmlparser.specified_attributes": ((2, 1), (3, 0)),
    "xml.sax.saxutils.quoteattr": ((2, 2), (3, 0)),
    "xml.sax.saxutils.unescape": ((2, 3), (3, 0)),
    "xmlrpc.server.SimpleXMLRPCServer.register_introspection_functions": (None, (3, 0)),
    "zipfile.ZipFile.extract": ((2, 6), (3, 0)),
    "zipfile.ZipFile.extractall": ((2, 6), (3, 0)),
    "zipfile.ZipFile.mkdir": (None, (3, 11)),
    "zipfile.ZipFile.open": ((2, 6), (3, 0)),
    "zipfile.ZipFile.setpassword": ((2, 6), (3, 0)),
    "zipfile.ZipInfo._for_archive": (None, (3, 14)),
    "zipfile.ZipInfo.from_file": (None, (3, 6)),
    "zipfile.ZipInfo.is_dir": (None, (3, 6)),
    "zipimport.zipimporter.create_module": (None, (3, 10)),
    "zipimport.zipimporter.exec_module": (None, (3, 10)),
    "zipimport.zipimporter.find_spec": (None, (3, 10)),
    "zipimport.zipimporter.get_filename": ((2, 7), (3, 1)),
    "zipimport.zipimporter.invalidate_caches": (None, (3, 10)),
    "zlib.Compress.copy": ((2, 5), (3, 0)),
    "zlib.Decompress.copy": ((2, 5), (3, 0)),

    # Variables and Constants
    "AttributeError.name": (None, (3, 10)),
    "AttributeError.obj": (None, (3, 10)),
    "BaseException.__suppress_context__": (None, (3, 3)),
    "BaseHTTPServer.BaseHTTPRequestHandler.error_content_type": ((2, 6), None),
    "Cookie.Morsel.httponly": ((2, 6), None),
    "ImportError.name": (None, (3, 3)),
    "ImportError.path": (None, (3, 3)),
    "NameError.name": (None, (3, 10)),
    "OSError.filename2": (None, (3, 4)),
    "SimpleXMLRPCServer.SimpleXMLRPCRequestHandler.encode_threshold": ((2, 7), None),
    "SimpleXMLRPCServer.SimpleXMLRPCRequestHandler.rpc_paths": ((2, 5), None),
    "StopIteration.value": (None, (3, 3)),
    "SyntaxError.end_lineno": (None, (3, 10)),
    "SyntaxError.end_offset": (None, (3, 10)),
    "__future__.absolute_import": ((2, 5), (3, 0)),
    "__future__.annotations": (None, (3, 7)),
    "__future__.division": ((2, 2), (3, 0)),
    "__future__.generator_stop": (None, (3, 5)),
    "__future__.generators": ((2, 2), (3, 0)),
    "__future__.nested_scopes": ((2, 1), (3, 0)),
    "__future__.print_function": ((2, 6), (3, 0)),
    "__future__.unicode_literals": ((2, 6), (3, 0)),
    "__future__.with_statement": ((2, 5), (3, 0)),
    "_thread.TIMEOUT_MAX": (None, (3, 2)),
    "ast.AST._field_types": (None, (3, 13)),
    "ast.PyCF_ALLOW_TOP_LEVEL_AWAIT": (None, (3, 8)),
    "ast.PyCF_OPTIMIZED_AST": (None, (3, 13)),
    "ast.PyCF_TYPE_COMMENTS": (None, (3, 8)),
    "bz2.BZ2Decompressor.eof": (None, (3, 3)),
    "bz2.BZ2Decompressor.needs_input": (None, (3, 5)),
    "bz2.BZ2File.mode": (None, (3, 13)),
    "bz2.BZ2File.name": (None, (3, 13)),
    "calendar.HTMLCalendar.cssclass_month": (None, (3, 7)),
    "calendar.HTMLCalendar.cssclass_month_head": (None, (3, 7)),
    "calendar.HTMLCalendar.cssclass_noday": (None, (3, 7)),
    "calendar.HTMLCalendar.cssclass_year": (None, (3, 7)),
    "calendar.HTMLCalendar.cssclass_year_head": (None, (3, 7)),
    "calendar.HTMLCalendar.cssclasses_weekday_head": (None, (3, 7)),
    "calendar.Day.FRIDAY": (None, (3, 12)),
    "calendar.Day.MONDAY": (None, (3, 12)),
    "calendar.Day.SATURDAY": (None, (3, 12)),
    "calendar.Day.SUNDAY": (None, (3, 12)),
    "calendar.Day.THURSDAY": (None, (3, 12)),
    "calendar.Day.TUESDAY": (None, (3, 12)),
    "calendar.Day.WEDNESDAY": (None, (3, 12)),
    "calendar.FRIDAY": (None, (3, 10)),
    "calendar.MONDAY": (None, (3, 10)),
    "calendar.SATURDAY": (None, (3, 10)),
    "calendar.THURSDAY": (None, (3, 10)),
    "calendar.TUESDAY": (None, (3, 10)),
    "calendar.WEDNESDAY": (None, (3, 10)),
    "calendar.SUNDAY": (None, (3, 10)),
    "calendar.Month.JANUARY": (None, (3, 12)),
    "calendar.Month.FEBRUARY": (None, (3, 12)),
    "calendar.Month.MARCH": (None, (3, 12)),
    "calendar.Month.APRIL": (None, (3, 12)),
    "calendar.Month.MAY": (None, (3, 12)),
    "calendar.Month.JUNE": (None, (3, 12)),
    "calendar.Month.JULY": (None, (3, 12)),
    "calendar.Month.AUGUST": (None, (3, 12)),
    "calendar.Month.SEPTEMBER": (None, (3, 12)),
    "calendar.Month.OCTOBER": (None, (3, 12)),
    "calendar.Month.NOVEMBER": (None, (3, 12)),
    "calendar.Month.DECEMBER": (None, (3, 12)),
    "calendar.JANUARY": (None, (3, 12)),
    "calendar.FEBRUARY": (None, (3, 12)),
    "calendar.MARCH": (None, (3, 12)),
    "calendar.APRIL": (None, (3, 12)),
    "calendar.MAY": (None, (3, 12)),
    "calendar.JUNE": (None, (3, 12)),
    "calendar.JULY": (None, (3, 12)),
    "calendar.AUGUST": (None, (3, 12)),
    "calendar.SEPTEMBER": (None, (3, 12)),
    "calendar.OCTOBER": (None, (3, 12)),
    "calendar.NOVEMBER": (None, (3, 12)),
    "calendar.DECEMBER": (None, (3, 12)),
    "classmethod.__wrapped__": (None, (3, 10)),
    "cmath.inf": (None, (3, 6)),
    "cmath.infj": (None, (3, 6)),
    "cmath.nan": (None, (3, 6)),
    "cmath.nanj": (None, (3, 6)),
    "cmath.tau": (None, (3, 6)),
    "collections.deque.maxlen": ((2, 7), (3, 1)),
    "collections.namedtuple._field_defaults": (None, (3, 7)),
    "configparser.DuplicateSectionError.lineno": bpv("configparser", (None, (3, 2)), config),
    "configparser.DuplicateSectionError.source": bpv("configparser", (None, (3, 2)), config),
    "configparser.ParsingError.source": bpv("configparser", (None, (3, 2)), config),
    "cookielib.Cookie.rfc2109": ((2, 5), None),
    "cookielib.DefaultCookiePolicy.rfc2109_as_netscape": ((2, 5), None),
    "crypt.METHOD_BLOWFISH": (None, (3, 7)),
    "crypt.METHOD_CRYPT": (None, (3, 3)),
    "crypt.METHOD_MD5": (None, (3, 3)),
    "crypt.METHOD_SHA256": (None, (3, 3)),
    "crypt.METHOD_SHA512": (None, (3, 3)),
    "crypt.methods": (None, (3, 3)),
    "csv.QUOTE_NOTNULL": (None, (3, 12)),
    "csv.QUOTE_STRINGS": (None, (3, 12)),
    "csv.csvreader.fieldnames": ((2, 6), (3, 0)),
    "csv.csvreader.line_num": ((2, 5), (3, 0)),
    "ctypes.Structure._align_": (None, (3, 13)),
    "curses.A_ITALIC": (None, (3, 7)),
    "curses.ncurses_version": (None, (3, 8)),
    "curses.window.encoding": (None, (3, 3)),
    "dataclasses.KW_ONLY": (None, (3, 10)),
    "datetime.UTC": (None, (3, 11)),
    "datetime.datetime.fold": (None, (3, 6)),
    "datetime.time.fold": (None, (3, 6)),
    "decimal.HAVE_CONTEXTVAR": (None, (3, 7)),  # Backported from 3.9 -> 3.8 -> 3.7
    "difflib.SequenceMatcher.bjunk": (None, (3, 2)),
    "difflib.SequenceMatcher.bpopular": (None, (3, 2)),
    "dis.Instruction.positions": (None, (3, 11)),
    "dis.Instruction.start_offset": (None, (3, 13)),
    "dis.Instruction.cache_offset": (None, (3, 13)),
    "dis.Instruction.end_offset": (None, (3, 13)),
    "dis.Instruction.baseopname": (None, (3, 13)),
    "dis.Instruction.baseopcode": (None, (3, 13)),
    "dis.Instruction.jump_target": (None, (3, 13)),
    "dis.Instruction.oparg": (None, (3, 13)),
    "dis.Instruction.line_number": (None, (3, 13)),
    "dis.Instruction.cache_info": (None, (3, 13)),
    "doctest.COMPARISON_FLAGS": ((2, 4), (3, 0)),
    "doctest.DONT_ACCEPT_BLANKLINE": ((2, 4), (3, 0)),
    "doctest.ELLIPSIS": ((2, 4), (3, 0)),
    "doctest.FAIL_FAST": (None, (3, 4)),
    "doctest.IGNORE_EXCEPTION_DETAIL": ((2, 4), (3, 0)),
    "doctest.NORMALIZE_WHITESPACE": ((2, 4), (3, 0)),
    "doctest.REPORTING_FLAGS": ((2, 4), (3, 0)),
    "doctest.REPORT_CDIFF": ((2, 4), (3, 0)),
    "doctest.REPORT_NDIFF": ((2, 4), (3, 0)),
    "doctest.REPORT_ONLY_FIRST_FAILURE": ((2, 4), (3, 0)),
    "doctest.REPORT_UDIFF": ((2, 4), (3, 0)),
    "doctest.SKIP": ((2, 5), (3, 0)),
    "doctest.DocTestRunner.skips": (None, (3, 13)),
    "doctest.DocTestRunner.skipped": (None, (3, 13)),
    "email.message.Message.defects": ((2, 4), (3, 0)),
    "email.policy.EmailPolicy.content_manager": (None, (3, 4)),
    "email.policy.Policy.message_factory": (None, (3, 6)),
    "email.policy.Policy.verify_generated_headers": (None, (3, 13)),
    "errno.EHWPOISON": (None, (3, 14)),
    "errno.EQFULL": (None, (3, 11)),
    "fcntl.F_ADD_SEALS": (None, (3, 8)),
    "fcntl.F_DUP2FD": (None, (3, 11)),
    "fcntl.F_DUP2FD_CLOEXEC": (None, (3, 11)),
    "fcntl.F_GETPATH": (None, (3, 9)),
    "fcntl.F_GETPIPE_SZ": (None, (3, 10)),
    "fcntl.F_GET_SEALS": (None, (3, 8)),
    "fcntl.F_OFD_GETLK": (None, (3, 9)),
    "fcntl.F_OFD_SETLK": (None, (3, 9)),
    "fcntl.F_OFD_SETLKW": (None, (3, 9)),
    "fcntl.F_SEAL_GROW": (None, (3, 8)),
    "fcntl.F_SEAL_SEAL": (None, (3, 8)),
    "fcntl.F_SEAL_SHRINK": (None, (3, 8)),
    "fcntl.F_SEAL_WRITE": (None, (3, 8)),
    "fcntl.F_SETPIPE_SZ": (None, (3, 10)),
    "file.encoding": ((2, 3), None),
    "file.errors": ((2, 6), None),
    "filecmp.DEFAULT_IGNORES": (None, (3, 4)),
    "functools.Placeholder": (None, (3, 14)),
    "gc.callbacks": (None, (3, 3)),
    "gzip.GzipFile.mtime": (None, (3, 1)),
    "hashlib.algorithms": ((2, 7), None),
    "hashlib.algorithms_available": ((2, 7), (3, 2)),
    "hashlib.algorithms_guaranteed": ((2, 7), (3, 2)),
    "hashlib.hash.name": (None, (3, 4)),
    "hmac.HMAC.block_size": (None, (3, 4)),
    "hmac.HMAC.name": (None, (3, 4)),
    "html.entities.html5": (None, (3, 3)),
    "htmlentitydefs.codepoint2name": ((2, 3), None),
    "htmlentitydefs.name2codepoint": ((2, 3), None),
    "http.HTTPStatus.EARLY_HINTS": (None, (3, 9)),
    "http.HTTPStatus.IM_A_TEAPOT": (None, (3, 9)),
    "http.HTTPStatus.MISDIRECTED_REQUEST": (None, (3, 7)),
    "http.HTTPStatus.TOO_EARLY": (None, (3, 9)),
    "http.HTTPStatus.UNAVAILABLE_FOR_LEGAL_REASONS": (None, (3, 8)),
    "http.client.HTTPConnection.blocksize": (None, (3, 7)),
    "httplib.responses": ((2, 5), None),
    "imaplib.IMAP4.utf8_enabled": (None, (3, 5)),
    "importlib.machinery.BYTECODE_SUFFIXES": (None, (3, 3)),
    "importlib.machinery.DEBUG_BYTECODE_SUFFIXES": (None, (3, 3)),
    "importlib.machinery.EXTENSION_SUFFIXES": (None, (3, 3)),
    "importlib.machinery.OPTIMIZED_BYTECODE_SUFFIXES": (None, (3, 3)),
    "importlib.machinery.SOURCE_SUFFIXES": (None, (3, 3)),
    "importlib.metadata.metadata.json": (None, (3, 10)),
    "importlib.util.MAGIC_NUMBER": (None, (3, 4)),
    "inspect.CO_ASYNC_GENERATOR": (None, (3, 6)),
    "inspect.CO_COROUTINE": (None, (3, 5)),
    "inspect.CO_HAS_DOCSTRING": (None, (3, 14)),
    "inspect.CO_ITERABLE_COROUTINE": (None, (3, 5)),
    "inspect.CO_METHOD": (None, (3, 14)),
    "inspect.Parameter.kind.description": (None, (3, 8)),
    "io.SEEK_CUR": ((2, 7), (3, 1)),
    "io.SEEK_END": ((2, 7), (3, 1)),
    "io.SEEK_SET": ((2, 7), (3, 1)),
    "io.TextIOWrapper.write_through": (None, (3, 7)),
    "ipaddress.IPv4Address.ipv6_mapped": bpv("ipaddress", (None, (3, 13)), config),
    "ipaddress.IPv4Address.is_global": bpv("ipaddress", (None, (3, 4)), config),
    "ipaddress.IPv4Address.reverse_pointer": bpv("ipaddress", (None, (3, 5)), config),
    "ipaddress.IPv6Address.is_global": bpv("ipaddress", (None, (3, 4)), config),
    "keyword.softkwlist": (None, (3, 9)),
    "logging.Formatter.default_msec_format": (None, (3, 3)),
    "logging.Formatter.default_time_format": (None, (3, 3)),
    "logging.LogRecord.funcName": ((2, 5), (3, 0)),
    "logging.LogRecord.processName": ((2, 6), (3, 0)),
    "logging.LoggerAdapter.manager": (None, (3, 6)),
    "logging.StreamHandler.terminator": (None, (3, 2)),
    "logging.handlers.BaseRotatingHandler.namer": (None, (3, 3)),
    "logging.handlers.BaseRotatingHandler.rotator": (None, (3, 3)),
    "logging.lastResort": (None, (3, 2)),
    "lzma.LZMADecompressor.needs_input": (None, (3, 5)),
    "lzma.LZMAFile.mode": (None, (3, 13)),
    "lzma.LZMAFile.name": (None, (3, 13)),
    "marshal.version": ((2, 4), (3, 0)),
    "math.inf": (None, (3, 5)),
    "math.nan": (None, (3, 5)),
    "math.tau": (None, (3, 6)),
    "mmap.ACCESS_DEFAULT": (None, (3, 7)),
    "mmap.MADV_AUTOSYNC": (None, (3, 8)),
    "mmap.MADV_CORE": (None, (3, 8)),
    "mmap.MADV_DODUMP": (None, (3, 8)),
    "mmap.MADV_DOFORK": (None, (3, 8)),
    "mmap.MADV_DONTDUMP": (None, (3, 8)),
    "mmap.MADV_DONTFORK": (None, (3, 8)),
    "mmap.MADV_DONTNEED": (None, (3, 8)),
    "mmap.MADV_FREE": (None, (3, 8)),
    "mmap.MADV_FREE_REUSABLE": (None, (3, 8)),
    "mmap.MADV_FREE_REUSE": (None, (3, 8)),
    "mmap.MADV_HUGEPAGE": (None, (3, 8)),
    "mmap.MADV_HWPOISON": (None, (3, 8)),
    "mmap.MADV_MERGEABLE": (None, (3, 8)),
    "mmap.MADV_NOCORE": (None, (3, 8)),
    "mmap.MADV_NOHUGEPAGE": (None, (3, 8)),
    "mmap.MADV_NORMAL": (None, (3, 8)),
    "mmap.MADV_NOSYNC": (None, (3, 8)),
    "mmap.MADV_PROTECT": (None, (3, 8)),
    "mmap.MADV_RANDOM": (None, (3, 8)),
    "mmap.MADV_REMOVE": (None, (3, 8)),
    "mmap.MADV_SEQUENTIAL": (None, (3, 8)),
    "mmap.MADV_SOFT_OFFLINE": (None, (3, 8)),
    "mmap.MADV_UNMERGEABLE": (None, (3, 8)),
    "mmap.MADV_WILLNEED": (None, (3, 8)),
    "mmap.MAP_32BIT": (None, (3, 13)),
    "mmap.MAP_HASSEMAPHORE": (None, (3, 13)),
    "mmap.MAP_JIT": (None, (3, 13)),
    "mmap.MAP_NOCACHE": (None, (3, 13)),
    "mmap.MAP_NOEXTEND": (None, (3, 13)),
    "mmap.MAP_NORESERVE": (None, (3, 13)),
    "mmap.MAP_POPULATE": (None, (3, 10)),
    "mmap.MAP_RESILIENT_CODESIGN": (None, (3, 13)),
    "mmap.MAP_RESILIENT_MEDIA": (None, (3, 13)),
    "mmap.MAP_STACK": (None, (3, 11)),
    "mmap.MAP_TPRO": (None, (3, 13)),
    "mmap.MAP_TRANSLATED_ALLOW_EXECUTE": (None, (3, 13)),
    "mmap.MAP_UNIX03": (None, (3, 13)),
    "mmap.mmap.closed": (None, (3, 2)),
    "multiprocessing.Process.sentinel": (None, (3, 3)),
    "nntplib.NNTP.nntp_implementation": (None, (3, 2)),
    "nntplib.NNTP.nntp_version": (None, (3, 2)),
    "object.__match_args__": (None, (3, 10)),
    "os.CLD_CONTINUED": (None, (3, 3)),
    "os.CLD_DUMPED": (None, (3, 3)),
    "os.CLD_EXITED": (None, (3, 3)),
    "os.CLD_KILLED": (None, (3, 9)),
    "os.CLD_STOPPED": (None, (3, 9)),
    "os.CLD_TRAPPED": (None, (3, 3)),
    "os.EFD_CLOEXEC": (None, (3, 10)),
    "os.EFD_NONBLOCK": (None, (3, 10)),
    "os.EFD_SEMAPHORE": (None, (3, 10)),
    "os.EX_CANTCREAT": ((2, 3), (3, 0)),
    "os.EX_CONFIG": ((2, 3), (3, 0)),
    "os.EX_DATAERR": ((2, 3), (3, 0)),
    "os.EX_IOERR": ((2, 3), (3, 0)),
    "os.EX_NOHOST": ((2, 3), (3, 0)),
    "os.EX_NOINPUT": ((2, 3), (3, 0)),
    "os.EX_NOPERM": ((2, 3), (3, 0)),
    "os.EX_NOTFOUND": ((2, 3), (3, 0)),
    "os.EX_NOUSER": ((2, 3), (3, 0)),
    "os.EX_OK": ((2, 3), (3, 0)),
    "os.EX_OSERR": ((2, 3), (3, 0)),
    "os.EX_OSFILE": ((2, 3), (3, 0)),
    "os.EX_PROTOCOL": ((2, 3), (3, 0)),
    "os.EX_SOFTWARE": ((2, 3), (3, 0)),
    "os.EX_TEMPFAIL": ((2, 3), (3, 0)),
    "os.EX_UNAVAILABLE": ((2, 3), (3, 0)),
    "os.EX_USAGE": ((2, 3), (3, 0)),
    "os.F_LOCK": (None, (3, 3)),
    "os.F_TEST": (None, (3, 3)),
    "os.F_TLOCK": (None, (3, 3)),
    "os.F_ULOCK": (None, (3, 3)),
    "os.GRND_NONBLOCK": (None, (3, 6)),
    "os.GRND_RANDOM": (None, (3, 6)),
    "os.MFD_ALLOW_SEALING": (None, (3, 8)),
    "os.MFD_CLOEXEC": (None, (3, 8)),
    "os.MFD_HUGETLB": (None, (3, 8)),
    "os.MFD_HUGE_16GB": (None, (3, 8)),
    "os.MFD_HUGE_16MB": (None, (3, 8)),
    "os.MFD_HUGE_1GB": (None, (3, 8)),
    "os.MFD_HUGE_1MB": (None, (3, 8)),
    "os.MFD_HUGE_256MB": (None, (3, 8)),
    "os.MFD_HUGE_2GB": (None, (3, 8)),
    "os.MFD_HUGE_2MB": (None, (3, 8)),
    "os.MFD_HUGE_32MB": (None, (3, 8)),
    "os.MFD_HUGE_512KB": (None, (3, 8)),
    "os.MFD_HUGE_512MB": (None, (3, 8)),
    "os.MFD_HUGE_64KB": (None, (3, 8)),
    "os.MFD_HUGE_8MB": (None, (3, 8)),
    "os.MFD_HUGE_MASK": (None, (3, 8)),
    "os.MFD_HUGE_SHIFT": (None, (3, 8)),
    "os.O_CLOEXEC": (None, (3, 3)),
    "os.O_EVTONLY": (None, (3, 10)),
    "os.O_FSYNC": (None, (3, 10)),
    "os.O_NOFOLLOW_ANY": (None, (3, 10)),
    "os.O_PATH": (None, (3, 4)),
    "os.O_SYMLINK": (None, (3, 10)),
    "os.O_TMPFILE": (None, (3, 4)),
    "os.PIDFD_NONBLOCK": (None, (3, 12)),
    "os.POSIX_FADV_DONTNEED": (None, (3, 3)),
    "os.POSIX_FADV_NOREUSE": (None, (3, 3)),
    "os.POSIX_FADV_NORMAL": (None, (3, 3)),
    "os.POSIX_FADV_RANDOM": (None, (3, 3)),
    "os.POSIX_FADV_SEQUENTIAL": (None, (3, 3)),
    "os.POSIX_FADV_WILLNEED": (None, (3, 3)),
    "os.POSIX_SPAWN_CLOSE": (None, (3, 8)),
    "os.POSIX_SPAWN_DUP2": (None, (3, 8)),
    "os.POSIX_SPAWN_OPEN": (None, (3, 8)),
    "os.PRIO_DARWIN_BG": (None, (3, 12)),
    "os.PRIO_DARWIN_NONUI": (None, (3, 12)),
    "os.PRIO_DARWIN_PROCESS": (None, (3, 12)),
    "os.PRIO_DARWIN_THREAD": (None, (3, 12)),
    "os.PRIO_PGRP": (None, (3, 3)),
    "os.PRIO_PROCESS": (None, (3, 3)),
    "os.PRIO_USER": (None, (3, 3)),
    "os.P_ALL": (None, (3, 3)),
    "os.P_PGID": (None, (3, 3)),
    "os.P_PID": (None, (3, 3)),
    "os.P_PIDFD": (None, (3, 9)),
    "os.RTLD_DEEPBIND": (None, (3, 3)),
    "os.RTLD_GLOBAL": (None, (3, 3)),
    "os.RTLD_LAZY": (None, (3, 3)),
    "os.RTLD_LOCAL": (None, (3, 3)),
    "os.RTLD_NODELETE": (None, (3, 3)),
    "os.RTLD_NOLOAD": (None, (3, 3)),
    "os.RTLD_NOW": (None, (3, 3)),
    "os.RWF_APPEND": (None, (3, 10)),
    "os.RWF_DSYNC": (None, (3, 7)),
    "os.RWF_HIPRI": (None, (3, 7)),
    "os.RWF_NOWAIT": (None, (3, 7)),
    "os.RWF_SYNC": (None, (3, 7)),
    "os.SCHED_BATCH": (None, (3, 3)),
    "os.SCHED_DEADLINE": (None, (3, 14)),
    "os.SCHED_FIFO": (None, (3, 3)),
    "os.SCHED_IDLE": (None, (3, 3)),
    "os.SCHED_NORMAL": (None, (3, 14)),
    "os.SCHED_OTHER": (None, (3, 3)),
    "os.SCHED_RESET_ON_FORK": (None, (3, 3)),
    "os.SCHED_RR": (None, (3, 3)),
    "os.SCHED_SPORADIC": (None, (3, 3)),
    "os.SEEK_CUR": ((2, 5), (3, 0)),
    "os.SEEK_DATA": (None, (3, 3)),
    "os.SEEK_END": ((2, 5), (3, 0)),
    "os.SEEK_HOLE": (None, (3, 3)),
    "os.SEEK_SET": ((2, 5), (3, 0)),
    "os.SF_MNOWAIT": (None, (3, 3)),
    "os.SF_NOCACHE": (None, (3, 11)),
    "os.SF_NODISKIO": (None, (3, 3)),
    "os.SF_SYNC": (None, (3, 3)),
    "os.SPLICE_F_MORE": (None, (3, 10)),
    "os.SPLICE_F_MOVE": (None, (3, 10)),
    "os.SPLICE_F_NONBLOCK": (None, (3, 10)),
    "os.ST_APPEND": (None, (3, 4)),
    "os.ST_IMMUTABLE": (None, (3, 4)),
    "os.ST_MANDLOCK": (None, (3, 4)),
    "os.ST_NOATIME": (None, (3, 4)),
    "os.ST_NODEV": (None, (3, 4)),
    "os.ST_NODIRATIME": (None, (3, 4)),
    "os.ST_NOEXEC": (None, (3, 4)),
    "os.ST_NOSUID": (None, (3, 2)),
    "os.ST_RDONLY": (None, (3, 2)),
    "os.ST_RELATIME": (None, (3, 4)),
    "os.ST_SYNCHRONOUS": (None, (3, 4)),
    "os.ST_WRITE": (None, (3, 4)),
    "os.TFD_CLOEXEC": (None, (3, 13)),
    "os.TFD_NONBLOCK": (None, (3, 13)),
    "os.TFD_TIMER_ABSTIME": (None, (3, 13)),
    "os.TFD_TIMER_CANCEL_ON_SET": (None, (3, 13)),
    "os.WCONTINUED": ((2, 3), (3, 0)),
    "os.WEXITED": (None, (3, 3)),
    "os.WNOWAIT": (None, (3, 3)),
    "os.WSTOPPED": (None, (3, 3)),
    "os.WUNTRACED": ((2, 3), (3, 0)),
    "os.XATTR_CREATE": (None, (3, 3)),
    "os.XATTR_REPLACE": (None, (3, 3)),
    "os.XATTR_SIZE_MAX": (None, (3, 3)),
    "os._COPYFILE_ACL": (None, (3, 11)),
    "os._COPYFILE_STAT": (None, (3, 11)),
    "os._COPYFILE_XATTR": (None, (3, 11)),
    "os.devnull": ((2, 4), (3, 0)),
    "os.environb": (None, (3, 2)),
    "os.extsep": ((2, 2), (3, 0)),
    "os.killpg": ((2, 3), (3, 0)),
    "os.path.ALLOW_MISSING": (None, (3, 14)),
    "os.path.supports_unicode_filenames": ((2, 3), (3, 0)),
    "os.stat.st_atime": ((2, 2), (3, 0)),
    "os.stat.st_atime_ns": (None, (3, 3)),
    "os.stat.st_attrs": ((2, 2), (3, 0)),
    "os.stat.st_birthtime": ((2, 5), (3, 0)),
    "os.stat.st_birthtime_ns": (None, (3, 12)),
    "os.stat.st_blksize": ((2, 2), (3, 0)),
    "os.stat.st_blocks": ((2, 2), (3, 0)),
    "os.stat.st_ctime": ((2, 2), (3, 0)),
    "os.stat.st_ctime_ns": (None, (3, 3)),
    "os.stat.st_dev": ((2, 2), (3, 0)),
    "os.stat.st_file_attributes": (None, (3, 5)),
    "os.stat.st_flags": ((2, 2), (3, 0)),
    "os.stat.st_ftype": ((2, 2), (3, 0)),
    "os.stat.st_gen": ((2, 5), (3, 0)),
    "os.stat.st_gid": ((2, 2), (3, 0)),
    "os.stat.st_ino": ((2, 2), (3, 0)),
    "os.stat.st_mode": ((2, 2), (3, 0)),
    "os.stat.st_mtime": ((2, 2), (3, 0)),
    "os.stat.st_mtime_ns": (None, (3, 3)),
    "os.stat.st_nlink": ((2, 2), (3, 0)),
    "os.stat.st_obtype": ((2, 2), (3, 0)),
    "os.stat.st_rdev": ((2, 2), (3, 0)),
    "os.stat.st_reparse_tag": (None, (3, 8)),
    "os.stat.st_size": ((2, 2), (3, 0)),
    "os.stat.st_uid": ((2, 2), (3, 0)),
    "os.statvfs.f_bavail": ((2, 2), (3, 0)),
    "os.statvfs.f_bfree": ((2, 2), (3, 0)),
    "os.statvfs.f_blocks": ((2, 2), (3, 0)),
    "os.statvfs.f_bsize": ((2, 2), (3, 0)),
    "os.statvfs.f_favail": ((2, 2), (3, 0)),
    "os.statvfs.f_ffree": ((2, 2), (3, 0)),
    "os.statvfs.f_files": ((2, 2), (3, 0)),
    "os.statvfs.f_flag": ((2, 2), (3, 0)),
    "os.statvfs.f_frsize": ((2, 2), (3, 0)),
    "os.statvfs.f_fsid": (None, (3, 7)),
    "os.statvfs.f_namemax": ((2, 2), (3, 0)),
    "os.supports_bytes_environ": (None, (3, 2)),
    "os.supports_dir_fd": (None, (3, 3)),
    "os.supports_effective_ids": (None, (3, 3)),
    "os.supports_fd": (None, (3, 3)),
    "os.supports_follow_symlinks": (None, (3, 3)),
    "pathlib.info": (None, (3, 14)),
    "pickle.HIGHEST_PROTOCOL": ((2, 3), (3, 0)),
    "pickle.Pickler.dispatch_table": (None, (3, 3)),
    "plistlib.FMT_BINARY": (None, (3, 4)),
    "plistlib.FMT_XML": (None, (3, 4)),
    "pstats.SortKey": (None, (3, 7)),
    "pyclbr.Class.children": (None, (3, 7)),
    "pyclbr.Class.end_lineno": (None, (3, 10)),
    "pyclbr.Class.parent": (None, (3, 7)),
    "pyclbr.Function.children": (None, (3, 7)),
    "pyclbr.Function.end_lineno": (None, (3, 10)),
    "pyclbr.Function.is_async": (None, (3, 10)),
    "pyclbr.Function.parent": (None, (3, 7)),
    "range.start": (None, (3, 3)),
    "range.step": (None, (3, 3)),
    "range.stop": (None, (3, 3)),
    "re.NOFLAG": (None, (3, 11)),
    "re.U": ((2, 0), (3, 0)),
    "re.UNICODE": ((2, 0), (3, 0)),
    "re.error.colno": (None, (3, 5)),
    "re.error.lineno": (None, (3, 5)),
    "re.error.msg": (None, (3, 5)),
    "re.error.pattern": (None, (3, 5)),
    "re.error.pos": (None, (3, 5)),
    "repr.Repr.maxfrozenset": ((2, 4), None),
    "repr.Repr.maxset": ((2, 4), None),
    "reprlib.Repr.fillvalue": (None, (3, 11)),
    "resource.RLIMIT_KQUEUES": (None, (3, 10)),
    "resource.RLIMIT_MSGQUEUE": (None, (3, 4)),
    "resource.RLIMIT_NICE": (None, (3, 4)),
    "resource.RLIMIT_NPTS": (None, (3, 4)),
    "resource.RLIMIT_RTPRIO": (None, (3, 4)),
    "resource.RLIMIT_RTTIME": (None, (3, 4)),
    "resource.RLIMIT_SBSIZE": (None, (3, 4)),
    "resource.RLIMIT_SIGPENDING": (None, (3, 4)),
    "resource.RLIMIT_SWAP": (None, (3, 4)),
    "resource.RUSAGE_THREAD": (None, (3, 2)),
    "resource.getrusage.ru_idrss": ((2, 3), (3, 0)),
    "resource.getrusage.ru_inblock": ((2, 3), (3, 0)),
    "resource.getrusage.ru_isrss": ((2, 3), (3, 0)),
    "resource.getrusage.ru_ixrss": ((2, 3), (3, 0)),
    "resource.getrusage.ru_majflt": ((2, 3), (3, 0)),
    "resource.getrusage.ru_maxrss": ((2, 3), (3, 0)),
    "resource.getrusage.ru_minflt": ((2, 3), (3, 0)),
    "resource.getrusage.ru_msgrcv": ((2, 3), (3, 0)),
    "resource.getrusage.ru_msgsnd": ((2, 3), (3, 0)),
    "resource.getrusage.ru_nivcsw": ((2, 3), (3, 0)),
    "resource.getrusage.ru_nsignals": ((2, 3), (3, 0)),
    "resource.getrusage.ru_nswap": ((2, 3), (3, 0)),
    "resource.getrusage.ru_nvcsw": ((2, 3), (3, 0)),
    "resource.getrusage.ru_oublock": ((2, 3), (3, 0)),
    "resource.getrusage.ru_stime": ((2, 3), (3, 0)),
    "resource.getrusage.ru_utime": ((2, 3), (3, 0)),
    "sched.scheduler.queue": ((2, 6), (3, 0)),
    "select.EPOLLEXCLUSIVE": (None, (3, 6)),
    "select.PIPE_BUF": ((2, 7), (3, 2)),
    "select.devpoll.closed": (None, (3, 4)),
    "shlex.shlex.eof": ((2, 3), (3, 0)),
    "shlex.shlex.escape": ((2, 3), (3, 0)),
    "shlex.shlex.escapedquotes": ((2, 3), (3, 0)),
    "shlex.shlex.punctuation_chars": (None, (3, 6)),
    "shlex.shlex.whitespace_split": ((2, 3), (3, 0)),
    "shutil.rmtree.avoids_symlink_attacks": (None, (3, 3)),
    "signal.CTRL_BREAK_EVENT": ((2, 7), (3, 2)),
    "signal.CTRL_C_EVENT": ((2, 7), (3, 2)),
    "signal.SIGSTKFLT": (None, (3, 11)),
    "signal.SIG_BLOCK": (None, (3, 3)),
    "signal.SIG_SETMASK": (None, (3, 3)),
    "signal.SIG_UNBLOCK": (None, (3, 3)),
    "site.ENABLE_USER_SITE": ((2, 6), (3, 0)),
    "site.PREFIXES": ((2, 6), (3, 0)),
    "site.USER_BASE": ((2, 6), (3, 0)),
    "site.USER_SITE": ((2, 6), (3, 0)),
    "socket.AF_ALG": (None, (3, 6)),
    "socket.AF_CAN": (None, (3, 3)),
    "socket.AF_LINK": (None, (3, 4)),
    "socket.AF_QIPCRTR": (None, (3, 8)),
    "socket.AF_RDS": (None, (3, 3)),
    "socket.AF_TIPC": ((2, 6), (3, 0)),
    "socket.AF_VSOCK": (None, (3, 7)),
    "socket.ALG_OP_DECRYPT": (None, (3, 6)),
    "socket.ALG_OP_ENCRYPT": (None, (3, 6)),
    "socket.ALG_OP_SIGN": (None, (3, 6)),
    "socket.ALG_OP_VERIFY": (None, (3, 6)),
    "socket.ALG_SET_AEAD_ASSOCLEN": (None, (3, 6)),
    "socket.ALG_SET_AEAD_AUTHSIZE": (None, (3, 6)),
    "socket.ALG_SET_IV": (None, (3, 6)),
    "socket.ALG_SET_KEY": (None, (3, 6)),
    "socket.ALG_SET_OP": (None, (3, 6)),
    "socket.ALG_SET_PUBKEY": (None, (3, 6)),
    "socket.BDADDR_BREDR": (None, (3, 14)),
    "socket.BDADDR_LE_PUBLIC": (None, (3, 14)),
    "socket.BDADDR_LE_RANDOM": (None, (3, 14)),
    "socket.BT_CHANNEL_POLICY": (None, (3, 14)),
    "socket.BT_CHANNEL_POLICY_BREDR_ONLY": (None, (3, 14)),
    "socket.BT_CHANNEL_POLICY_BREDR_PREFERRED": (None, (3, 14)),
    "socket.BT_CODEC": (None, (3, 14)),
    "socket.BT_DEFER_SETUP": (None, (3, 14)),
    "socket.BT_FLUSHABLE": (None, (3, 14)),
    "socket.BT_FLUSHABLE_OFF": (None, (3, 14)),
    "socket.BT_FLUSHABLE_ON": (None, (3, 14)),
    "socket.BT_ISO_QOS": (None, (3, 14)),
    "socket.BT_MODE": (None, (3, 14)),
    "socket.BT_MODE_BASIC": (None, (3, 14)),
    "socket.BT_MODE_ERTM": (None, (3, 14)),
    "socket.BT_MODE_EXT_FLOWCTL": (None, (3, 14)),
    "socket.BT_MODE_LE_FLOWCTL": (None, (3, 14)),
    "socket.BT_MODE_STREAMING": (None, (3, 14)),
    "socket.BT_PHY": (None, (3, 14)),
    "socket.BT_PHY_BR_1M_1SLOT": (None, (3, 14)),
    "socket.BT_PHY_BR_1M_3SLOT": (None, (3, 14)),
    "socket.BT_PHY_BR_1M_5SLOT": (None, (3, 14)),
    "socket.BT_PHY_EDR_2M_1SLOT": (None, (3, 14)),
    "socket.BT_PHY_EDR_2M_3SLOT": (None, (3, 14)),
    "socket.BT_PHY_EDR_2M_5SLOT": (None, (3, 14)),
    "socket.BT_PHY_EDR_3M_1SLOT": (None, (3, 14)),
    "socket.BT_PHY_EDR_3M_3SLOT": (None, (3, 14)),
    "socket.BT_PHY_EDR_3M_5SLOT": (None, (3, 14)),
    "socket.BT_PHY_LE_1M_RX": (None, (3, 14)),
    "socket.BT_PHY_LE_1M_TX": (None, (3, 14)),
    "socket.BT_PHY_LE_2M_RX": (None, (3, 14)),
    "socket.BT_PHY_LE_2M_TX": (None, (3, 14)),
    "socket.BT_PHY_LE_CODED_RX": (None, (3, 14)),
    "socket.BT_PHY_LE_CODED_TX": (None, (3, 14)),
    "socket.BT_PKT_STATUS": (None, (3, 14)),
    "socket.BT_POWER": (None, (3, 14)),
    "socket.BT_POWER_FORCE_ACTIVE_OFF": (None, (3, 14)),
    "socket.BT_POWER_FORCE_ACTIVE_ON": (None, (3, 14)),
    "socket.BT_RCVMTU": (None, (3, 14)),
    "socket.BT_SECURITY": (None, (3, 14)),
    "socket.BT_SECURITY_FIPS": (None, (3, 14)),
    "socket.BT_SECURITY_HIGH": (None, (3, 14)),
    "socket.BT_SECURITY_LOW": (None, (3, 14)),
    "socket.BT_SECURITY_MEDIUM": (None, (3, 14)),
    "socket.BT_SECURITY_SDP": (None, (3, 14)),
    "socket.BT_SNDMTU": (None, (3, 14)),
    "socket.BT_VOICE": (None, (3, 14)),
    "socket.BT_VOICE_CVSD_16BIT": (None, (3, 14)),
    "socket.BT_VOICE_TRANSPARENT": (None, (3, 14)),
    "socket.CAN_BCM": (None, (3, 4)),
    "socket.CAN_BCM_CAN_FD_FRAME": (None, (3, 4)),
    "socket.CAN_BCM_RX_ANNOUNCE_RESUME": (None, (3, 4)),
    "socket.CAN_BCM_RX_CHANGED": (None, (3, 4)),
    "socket.CAN_BCM_RX_CHECK_DLC": (None, (3, 4)),
    "socket.CAN_BCM_RX_DELETE": (None, (3, 4)),
    "socket.CAN_BCM_RX_FILTER_ID": (None, (3, 4)),
    "socket.CAN_BCM_RX_NO_AUTOTIMER": (None, (3, 4)),
    "socket.CAN_BCM_RX_READ": (None, (3, 4)),
    "socket.CAN_BCM_RX_RTR_FRAME": (None, (3, 4)),
    "socket.CAN_BCM_RX_SETUP": (None, (3, 4)),
    "socket.CAN_BCM_RX_STATUS": (None, (3, 4)),
    "socket.CAN_BCM_RX_TIMEOUT": (None, (3, 4)),
    "socket.CAN_BCM_SETTIMER": (None, (3, 4)),
    "socket.CAN_BCM_STARTTIMER": (None, (3, 4)),
    "socket.CAN_BCM_TX_ANNOUNCE": (None, (3, 4)),
    "socket.CAN_BCM_TX_COUNTEVT": (None, (3, 4)),
    "socket.CAN_BCM_TX_CP_CAN_ID": (None, (3, 4)),
    "socket.CAN_BCM_TX_DELETE": (None, (3, 4)),
    "socket.CAN_BCM_TX_EXPIRED": (None, (3, 4)),
    "socket.CAN_BCM_TX_READ": (None, (3, 4)),
    "socket.CAN_BCM_TX_RESET_MULTI_IDX": (None, (3, 4)),
    "socket.CAN_BCM_TX_SEND": (None, (3, 4)),
    "socket.CAN_BCM_TX_SETUP": (None, (3, 4)),
    "socket.CAN_BCM_TX_STATUS": (None, (3, 4)),
    "socket.CAN_EFF_FLAG": (None, (3, 3)),
    "socket.CAN_EFF_MASK": (None, (3, 3)),
    "socket.CAN_ERR_FLAG": (None, (3, 3)),
    "socket.CAN_ERR_MASK": (None, (3, 3)),
    "socket.CAN_ISOTP": (None, (3, 7)),
    "socket.CAN_J1939": (None, (3, 9)),
    "socket.CAN_RAW": (None, (3, 3)),
    "socket.CAN_RAW_ERR_FILTER": (None, (3, 3)),
    "socket.CAN_RAW_FD_FRAMES": (None, (3, 5)),
    "socket.CAN_RAW_FILTER": (None, (3, 3)),
    "socket.CAN_RAW_JOIN_FILTERS": (None, (3, 9)),
    "socket.CAN_RAW_LOOPBACK": (None, (3, 3)),
    "socket.CAN_RAW_RECV_OWN_MSGS": (None, (3, 3)),
    "socket.CAN_RTR_FLAG": (None, (3, 3)),
    "socket.CAN_SFF_MASK": (None, (3, 3)),
    "socket.HCI_CHANNEL_CONTROL": (None, (3, 14)),
    "socket.HCI_CHANNEL_LOGGING": (None, (3, 14)),
    "socket.HCI_CHANNEL_MONITOR": (None, (3, 14)),
    "socket.HCI_CHANNEL_RAW": (None, (3, 14)),
    "socket.HCI_CHANNEL_USER": (None, (3, 14)),
    "socket.HCI_DEV_NONE": (None, (3, 14)),
    "socket.IOCTL_VM_SOCKETS_GET_LOCAL_CID": (None, (3, 7)),
    "socket.IPPROTO_MPTCP": (None, (3, 10)),
    "socket.IPPROTO_UDPLITE": (None, (3, 9)),
    "socket.IP_RECVTOS": (None, (3, 10)),
    "socket.L2CAP_LM": (None, (3, 14)),
    "socket.L2CAP_LM_AUTH": (None, (3, 14)),
    "socket.L2CAP_LM_ENCRYPT": (None, (3, 14)),
    "socket.L2CAP_LM_MASTER": (None, (3, 14)),
    "socket.L2CAP_LM_RELIABLE": (None, (3, 14)),
    "socket.L2CAP_LM_SECURE": (None, (3, 14)),
    "socket.L2CAP_LM_TRUSTED": (None, (3, 14)),
    "socket.LOCAL_CREDS": (None, (3, 11)),
    "socket.LOCAL_CREDS_PERSISTENT": (None, (3, 11)),
    "socket.PF_CAN": (None, (3, 3)),
    "socket.PF_RDS": (None, (3, 3)),
    "socket.RCVALL_IPLEVEL": ((2, 6), (3, 0)),
    "socket.RCVALL_MAX": ((2, 6), (3, 0)),
    "socket.RCVALL_OFF": ((2, 6), (3, 0)),
    "socket.RCVALL_ON": ((2, 6), (3, 0)),
    "socket.RCVALL_SOCKETLEVELONLY": ((2, 6), (3, 0)),
    "socket.RDS_CANCEL_SENT_TO": (None, (3, 3)),
    "socket.RDS_CMSG_RDMA_ARGS": (None, (3, 3)),
    "socket.RDS_CMSG_RDMA_DEST": (None, (3, 3)),
    "socket.RDS_CMSG_RDMA_MAP": (None, (3, 3)),
    "socket.RDS_CMSG_RDMA_STATUS": (None, (3, 3)),
    "socket.RDS_CMSG_RDMA_UPDATE": (None, (3, 3)),
    "socket.RDS_CONG_MONITOR": (None, (3, 3)),
    "socket.RDS_FREE_MR": (None, (3, 3)),
    "socket.RDS_GET_MR": (None, (3, 3)),
    "socket.RDS_GET_MR_FOR_DEST": (None, (3, 3)),
    "socket.RDS_RDMA_DONTWAIT": (None, (3, 3)),
    "socket.RDS_RDMA_FENCE": (None, (3, 3)),
    "socket.RDS_RDMA_INVALIDATE": (None, (3, 3)),
    "socket.RDS_RDMA_NOTIFY_ME": (None, (3, 3)),
    "socket.RDS_RDMA_READWRITE": (None, (3, 3)),
    "socket.RDS_RDMA_SILENT": (None, (3, 3)),
    "socket.RDS_RDMA_USE_ONCE": (None, (3, 3)),
    "socket.RDS_RECVERR": (None, (3, 3)),
    "socket.SCM_CREDS2": (None, (3, 11)),
    "socket.SIO_KEEPALIVE_VALS": ((2, 6), (3, 0)),
    "socket.SIO_LOOPBACK_FAST_PATH": (None, (3, 6)),
    "socket.SIO_RCVALL": ((2, 6), (3, 0)),
    "socket.SOCK_CLOEXEC": (None, (3, 2)),
    "socket.SOCK_NONBLOCK": (None, (3, 2)),
    "socket.SOL_ALG": (None, (3, 6)),
    "socket.SOL_BLUETOOTH": (None, (3, 14)),
    "socket.SOL_CAN_BASE": (None, (3, 3)),
    "socket.SOL_CAN_RAW": (None, (3, 3)),
    "socket.SOL_HCI": (None, (3, 14)),
    "socket.SOL_L2CAP": (None, (3, 14)),
    "socket.SOL_RDS": (None, (3, 3)),
    "socket.SOL_RFCOMM": (None, (3, 14)),
    "socket.SOL_SCO": (None, (3, 14)),
    "socket.SOL_TIPC": ((2, 6), (3, 0)),
    "socket.SO_DOMAIN": (None, (3, 6)),
    "socket.SO_HCI_EVT_FILTER": (None, (3, 14)),
    "socket.SO_HCI_PKT_FILTER": (None, (3, 14)),
    "socket.SO_INCOMING_CPU": (None, (3, 11)),
    "socket.SO_PASSSEC": (None, (3, 6)),
    "socket.SO_PEERSEC": (None, (3, 6)),
    "socket.SO_PROTOCOL": (None, (3, 6)),
    "socket.SO_REUSEPORT_LB": (None, (3, 14)),
    "socket.SO_VM_SOCKETS_BUFFER_MAX_SIZE": (None, (3, 7)),
    "socket.SO_VM_SOCKETS_BUFFER_MIN_SIZE": (None, (3, 7)),
    "socket.SO_VM_SOCKETS_BUFFER_SIZE": (None, (3, 7)),
    "socket.TCP_CONGESTION": (None, (3, 6)),
    "socket.TCP_CONNECTION_INFO": (None, (3, 11)),
    "socket.TCP_KEEPALIVE": (None, (3, 10)),
    "socket.TCP_NOTSENT_LOWAT": (None, (3, 7)),
    "socket.TCP_USER_TIMEOUT": (None, (3, 6)),
    "socket.TIPC_ADDR_ID": ((2, 6), (3, 0)),
    "socket.TIPC_ADDR_NAME": ((2, 6), (3, 0)),
    "socket.TIPC_ADDR_NAMESEQ": ((2, 6), (3, 0)),
    "socket.TIPC_CFG_SRV": ((2, 6), (3, 0)),
    "socket.TIPC_CLUSTER_SCOPE": ((2, 6), (3, 0)),
    "socket.TIPC_CONN_TIMEOUT": ((2, 6), (3, 0)),
    "socket.TIPC_CRITICAL_IMPORTANCE": ((2, 6), (3, 0)),
    "socket.TIPC_DEST_DROPPABLE": ((2, 6), (3, 0)),
    "socket.TIPC_HIGH_IMPORTANCE": ((2, 6), (3, 0)),
    "socket.TIPC_IMPORTANCE": ((2, 6), (3, 0)),
    "socket.TIPC_LOW_IMPORTANCE": ((2, 6), (3, 0)),
    "socket.TIPC_MEDIUM_IMPORTANCE": ((2, 6), (3, 0)),
    "socket.TIPC_NODE_SCOPE": ((2, 6), (3, 0)),
    "socket.TIPC_PUBLISHED": ((2, 6), (3, 0)),
    "socket.TIPC_SRC_DROPPABLE": ((2, 6), (3, 0)),
    "socket.TIPC_SUBSCR_TIMEOUT": ((2, 6), (3, 0)),
    "socket.TIPC_SUB_CANCEL": ((2, 6), (3, 0)),
    "socket.TIPC_SUB_PORTS": ((2, 6), (3, 0)),
    "socket.TIPC_SUB_SERVICE": ((2, 6), (3, 0)),
    "socket.TIPC_TOP_SRV": ((2, 6), (3, 0)),
    "socket.TIPC_WAIT_FOREVER": ((2, 6), (3, 0)),
    "socket.TIPC_WITHDRAWN": ((2, 6), (3, 0)),
    "socket.TIPC_ZONE_SCOPE": ((2, 6), (3, 0)),
    "socket.VMADDR_CID_ANY": (None, (3, 7)),
    "socket.VMADDR_CID_HOST": (None, (3, 7)),
    "socket.VMADDR_PORT_ANY": (None, (3, 7)),
    "socket.has_ipv6": ((2, 3), (3, 0)),
    "socket.socket.family": ((2, 5), (3, 0)),
    "socket.socket.proto": ((2, 5), (3, 0)),
    "socket.socket.type": ((2, 5), (3, 0)),
    "socketserver.ForkingMixIn.block_on_close": (None, (3, 7)),
    "socketserver.ThreadingMixIn.block_on_close": (None, (3, 7)),
    "sqlite3.Connection.autocommit": (None, (3, 12)),
    "sqlite3.Error.sqlite_errorcode": (None, (3, 11)),
    "sqlite3.Error.sqlite_errorname": (None, (3, 11)),
    "sqlite3.SQLITE_DBCONFIG_DEFENSIVE": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_DQS_DDL": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_DQS_DML": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_ENABLE_FKEY": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_ENABLE_FTS3_TOKENIZER": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_ENABLE_LOAD_EXTENSION": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_ENABLE_QPSG": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_ENABLE_TRIGGER": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_ENABLE_VIEW": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_LEGACY_ALTER_TABLE": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_LEGACY_FILE_FORMAT": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_NO_CKPT_ON_CLOSE": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_RESET_DATABASE": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_TRIGGER_EQP": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_TRUSTED_SCHEMA": (None, (3, 12)),
    "sqlite3.SQLITE_DBCONFIG_WRITABLE_SCHEMA": (None, (3, 12)),
    "ssl.ALERT_DESCRIPTION_ACCESS_DENIED": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_BAD_CERTIFICATE": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_BAD_CERTIFICATE_HASH_VALUE": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_BAD_CERTIFICATE_STATUS_RESPONSE": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_BAD_RECORD_MAC": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_CERTIFICATE_EXPIRED": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_CERTIFICATE_REVOKED": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_CERTIFICATE_UNKNOWN": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_CERTIFICATE_UNOBTAINABLE": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_CLOSE_NOTIFY": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_DECODE_ERROR": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_DECOMPRESSION_FAILURE": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_DECRYPT_ERROR": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_HANDSHAKE_FAILURE": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_ILLEGAL_PARAMETER": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_INSUFFICIENT_SECURITY": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_INTERNAL_ERROR": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_NO_RENEGOTIATION": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_PROTOCOL_VERSION": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_RECORD_OVERFLOW": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_UNEXPECTED_MESSAGE": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_UNKNOWN_CA": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_UNKNOWN_PSK_IDENTITY": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_UNRECOGNIZED_NAME": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_UNSUPPORTED_CERTIFICATE": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_UNSUPPORTED_EXTENSION": ((2, 7), (3, 4)),
    "ssl.ALERT_DESCRIPTION_USER_CANCELLED": ((2, 7), (3, 4)),
    "ssl.CHANNEL_BINDING_TYPES": ((2, 7), (3, 3)),
    "ssl.HAS_ALPN": ((2, 7), (3, 5)),
    "ssl.HAS_ECDH": ((2, 7), (3, 3)),
    "ssl.HAS_NEVER_CHECK_COMMON_NAME": (None, (3, 7)),
    "ssl.HAS_NPN": ((2, 7), (3, 3)),
    "ssl.HAS_PHA": (None, (3, 14)),
    "ssl.HAS_PSK": (None, (3, 13)),
    "ssl.HAS_SNI": ((2, 7), (3, 2)),
    "ssl.HAS_SSLv2": (None, (3, 7)),
    "ssl.HAS_SSLv3": (None, (3, 7)),
    "ssl.HAS_TLSv1": (None, (3, 7)),
    "ssl.HAS_TLSv1_1": (None, (3, 7)),
    "ssl.HAS_TLSv1_2": (None, (3, 7)),
    "ssl.HAS_TLSv1_3": ((2, 7), (3, 7)),
    "ssl.OPENSSL_VERSION": ((2, 7), (3, 2)),
    "ssl.OPENSSL_VERSION_INFO": ((2, 7), (3, 2)),
    "ssl.OPENSSL_VERSION_NUMBER": ((2, 7), (3, 2)),
    "ssl.OP_ALL": ((2, 7), (3, 2)),
    "ssl.OP_CIPHER_SERVER_PREFERENCE": ((2, 7), (3, 3)),
    "ssl.OP_ENABLE_MIDDLEBOX_COMPAT": ((2, 7), (3, 8)),
    "ssl.OP_IGNORE_UNEXPECTED_EOF": (None, (3, 10)),
    "ssl.OP_NO_COMPRESSION": ((2, 7), (3, 3)),
    "ssl.OP_NO_RENEGOTIATION": (None, (3, 7)),
    "ssl.OP_NO_SSLv2": ((2, 7), (3, 2)),
    "ssl.OP_NO_SSLv3": ((2, 7), (3, 2)),
    "ssl.OP_NO_TICKET": (None, (3, 6)),
    "ssl.OP_NO_TLSv1": ((2, 7), (3, 2)),
    "ssl.OP_NO_TLSv1_1": ((2, 7), (3, 4)),
    "ssl.OP_NO_TLSv1_2": ((2, 7), (3, 4)),
    "ssl.OP_NO_TLSv1_3": ((2, 7), (3, 7)),
    "ssl.OP_SINGLE_DH_USE": ((2, 7), (3, 3)),
    "ssl.OP_SINGLE_ECDH_USE": ((2, 7), (3, 3)),
    "ssl.PROTOCOL_TLS": ((2, 7), (3, 6)),
    "ssl.PROTOCOL_TLS_CLIENT": (None, (3, 6)),
    "ssl.PROTOCOL_TLS_SERVER": (None, (3, 6)),
    "ssl.PROTOCOL_TLSv1_1": ((2, 7), (3, 4)),
    "ssl.PROTOCOL_TLSv1_2": ((2, 7), (3, 4)),
    "ssl.Purpose.CLIENT_AUTH": ((2, 7), (3, 4)),
    "ssl.Purpose.SERVER_AUTH": ((2, 7), (3, 4)),
    "ssl.SSLContext.check_hostname": (None, (3, 4)),
    "ssl.SSLContext.hostname_checks_common_name": (None, (3, 7)),
    "ssl.SSLContext.keylog_filename": (None, (3, 8)),
    "ssl.SSLContext.maximum_version": (None, (3, 7)),
    "ssl.SSLContext.minimum_version": (None, (3, 7)),
    "ssl.SSLContext.num_tickets": (None, (3, 8)),
    "ssl.SSLContext.post_handshake_auth": (None, (3, 8)),
    "ssl.SSLContext.security_level": (None, (3, 10)),
    "ssl.SSLContext.sni_callback": (None, (3, 7)),
    "ssl.SSLContext.sslobject_class": (None, (3, 7)),
    "ssl.SSLContext.sslsocket_class": (None, (3, 7)),
    "ssl.SSLContext.verify_flags": (None, (3, 4)),
    "ssl.SSLError.library": ((2, 7), (3, 3)),
    "ssl.SSLError.reason": ((2, 7), (3, 3)),
    "ssl.SSLSocket.context": ((2, 7), (3, 2)),
    "ssl.SSLSocket.server_hostname": (None, (3, 2)),
    "ssl.SSLSocket.server_side": (None, (3, 2)),
    "ssl.SSLSocket.session": (None, (3, 6)),
    "ssl.SSLSocket.session_reused": (None, (3, 6)),
    "ssl.VERIFY_ALLOW_PROXY_CERTS": (None, (3, 10)),
    "ssl.VERIFY_CRL_CHECK_CHAIN": ((2, 7), (3, 4)),
    "ssl.VERIFY_CRL_CHECK_LEAF": ((2, 7), (3, 4)),
    "ssl.VERIFY_DEFAULT": ((2, 7), (3, 4)),
    "ssl.VERIFY_X509_PARTIAL_CHAIN": (None, (3, 10)),
    "ssl.VERIFY_X509_STRICT": ((2, 7), (3, 4)),
    "ssl.VERIFY_X509_TRUSTED_FIRST": ((2, 7), (3, 4)),
    "stat.FILE_ATTRIBUTE_ARCHIVE": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_COMPRESSED": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_DEVICE": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_DIRECTORY": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_ENCRYPTED": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_HIDDEN": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_INTEGRITY_STREAM": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_NORMAL": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_NOT_CONTENT_INDEXED": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_NO_SCRUB_DATA": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_OFFLINE": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_READONLY": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_REPARSE_POINT": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_SPARSE_FILE": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_SYSTEM": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_TEMPORARY": (None, (3, 5)),
    "stat.FILE_ATTRIBUTE_VIRTUAL": (None, (3, 5)),
    "stat.IO_REPARSE_TAG_APPEXECLINK": (None, (3, 8)),
    "stat.IO_REPARSE_TAG_MOUNT_POINT": (None, (3, 8)),
    "stat.IO_REPARSE_TAG_SYMLINK": (None, (3, 8)),
    "stat.S_IFDOOR": (None, (3, 4)),
    "stat.S_IFPORT": (None, (3, 4)),
    "stat.S_IFWHT": (None, (3, 4)),
    "staticmethod.__wrapped__": (None, (3, 10)),
    "string.Template.braceidpattern": (None, (3, 7)),
    "string.Template.flags": (None, (3, 2)),
    "string.letters": ((2, 0), None),
    "string.lowercase": ((2, 0), None),
    "string.uppercase": ((2, 0), None),
    "subprocess.ABOVE_NORMAL_PRIORITY_CLASS": (None, (3, 7)),
    "subprocess.BELOW_NORMAL_PRIORITY_CLASS": (None, (3, 7)),
    "subprocess.CREATE_BREAKAWAY_FROM_JOB": (None, (3, 7)),
    "subprocess.CREATE_DEFAULT_ERROR_MODE": (None, (3, 7)),
    "subprocess.CREATE_NO_WINDOW": (None, (3, 7)),
    "subprocess.CalledProcessError.stderr": (None, (3, 5)),
    "subprocess.CalledProcessError.stdout": (None, (3, 5)),
    "subprocess.DETACHED_PROCESS": (None, (3, 7)),
    "subprocess.DEVNULL": (None, (3, 3)),
    "subprocess.HIGH_PRIORITY_CLASS": (None, (3, 7)),
    "subprocess.IDLE_PRIORITY_CLASS": (None, (3, 7)),
    "subprocess.NORMAL_PRIORITY_CLASS": (None, (3, 7)),
    "subprocess.Popen.args": (None, (3, 3)),
    "subprocess.REALTIME_PRIORITY_CLASS": (None, (3, 7)),
    "subprocess.STARTF_FORCEOFFFEEDBACK": (None, (3, 13)),
    "subprocess.STARTF_FORCEONFEEDBACK": (None, (3, 13)),
    "subprocess.STARTUPINFO.lpAttributeList": (None, (3, 7)),
    "subprocess.TimeoutExpired.stderr": (None, (3, 5)),
    "subprocess.TimeoutExpired.stdout": (None, (3, 5)),
    "sys.__breakpointhook__": (None, (3, 7)),
    "sys.__interactivehook__": (None, (3, 4)),
    "sys.__unraisablehook__": (None, (3, 8)),
    "sys._emscripten_info": (None, (3, 11)),
    "sys._jit": (None, (3, 14)),
    "sys._xoptions": (None, (3, 2)),
    "sys.abiflags": (None, (3, 2)),
    "sys.activate_stack_trampoline": (None, (3, 12)),
    "sys.api_version": ((2, 3), (3, 0)),
    "sys.base_exec_prefix": (None, (3, 3)),
    "sys.base_prefix": (None, (3, 3)),
    "sys.deactivate_stack_trampoline": (None, (3, 12)),
    "sys.dont_write_bytecode": ((2, 6), (3, 0)),
    "sys.flags": ((2, 6), (3, 0)),
    "sys.flags.context_aware_warnings": (None, (3, 14)),
    "sys.flags.dev_mode": (None, (3, 7)),
    "sys.flags.gil": (None, (3, 13)),
    "sys.flags.hash_randomization": ((2, 7), (3, 2)),
    "sys.flags.int_max_str_digits": (None, (3, 11)),
    "sys.flags.isolated": (None, (3, 4)),
    "sys.flags.quiet": (None, (3, 2)),
    "sys.flags.safe_path": (None, (3, 11)),
    "sys.flags.thread_inherit_context": (None, (3, 14)),
    "sys.flags.utf8_mode": (None, (3, 7)),
    "sys.flags.warn_default_encoding": (None, (3, 10)),
    "sys.float_info": ((2, 6), (3, 0)),
    "sys.float_repr_style": ((2, 7), (3, 1)),
    "sys.getunicodeinternedsize": (None, (3, 12)),
    "sys.getwindowsversion.build": ((2, 7), (3, 0)),
    "sys.getwindowsversion.major": ((2, 7), (3, 0)),
    "sys.getwindowsversion.minor": ((2, 7), (3, 0)),
    "sys.getwindowsversion.platform": ((2, 7), (3, 0)),
    "sys.getwindowsversion.product_type": ((2, 7), (3, 0)),
    "sys.getwindowsversion.service_pack": ((2, 7), (3, 0)),
    "sys.getwindowsversion.service_pack_major": ((2, 7), (3, 0)),
    "sys.getwindowsversion.service_pack_minor": ((2, 7), (3, 0)),
    "sys.getwindowsversion.suite_mask": ((2, 7), (3, 0)),
    "sys.hash_info": (None, (3, 2)),
    "sys.hash_info.algorithm": (None, (3, 4)),
    "sys.hash_info.hash_bits": (None, (3, 4)),
    "sys.hash_info.seed_bits": (None, (3, 4)),
    "sys.implementation": (None, (3, 3)),
    "sys.int_info": (None, (3, 1)),
    "sys.is_stack_trampoline_active": (None, (3, 12)),
    "sys.last_exc": (None, (3, 12)),
    "sys.long_info": ((2, 7), None),
    "sys.orig_argv": (None, (3, 10)),
    "sys.platlibdir": (None, (3, 9)),
    "sys.py3kwarning": ((2, 6), None),
    "sys.pycache_prefix": (None, (3, 8)),
    "sys.stdlib_module_names": (None, (3, 10)),
    "sys.subversion": ((2, 5), (3, 0)),
    "sys.thread_info": (None, (3, 3)),
    "sys.version_info": ((2, 0), (3, 0)),
    "sys.version_info.major": ((2, 7), (3, 0)),
    "sys.version_info.micro": ((2, 7), (3, 0)),
    "sys.version_info.minor": ((2, 7), (3, 0)),
    "sys.version_info.releaselevel": ((2, 7), (3, 0)),
    "tarfile.DEFAULT_FORMAT": ((2, 6), (3, 0)),
    "tarfile.GNU_FORMAT": ((2, 6), (3, 0)),
    "tarfile.PAX_FORMAT": ((2, 6), (3, 0)),
    "tarfile.TarFile.extraction_filter": (None, (3, 12)),
    "tarfile.TarFile.pax_headers": ((2, 6), (3, 0)),
    "tarfile.TarInfo.pax_headers": ((2, 6), (3, 0)),
    "tarfile.USTAR_FORMAT": ((2, 6), (3, 0)),
    "textwrap.TextWrapper.break_on_hyphens": ((2, 6), (3, 0)),
    "textwrap.TextWrapper.drop_whitespace": ((2, 6), (3, 0)),
    "textwrap.TextWrapper.max_lines": (None, (3, 4)),
    "textwrap.TextWrapper.placeholder": (None, (3, 4)),
    "textwrap.TextWrapper.tabsize": (None, (3, 3)),
    "threading.TIMEOUT_MAX": (None, (3, 2)),
    "threading.Thread.daemon": ((2, 6), (3, 0)),
    "threading.Thread.ident": ((2, 6), (3, 0)),
    "threading.Thread.name": ((2, 6), (3, 0)),
    "threading.Thread.native_id": (None, (3, 8)),
    "threading.__excepthook__": (None, (3, 10)),
    "time.CLOCK_BOOTTIME": (None, (3, 7)),
    "time.CLOCK_HIGHRES": (None, (3, 3)),
    "time.CLOCK_MONOTONIC": (None, (3, 3)),
    "time.CLOCK_MONOTONIC_RAW": (None, (3, 3)),
    "time.CLOCK_MONOTONIC_RAW_APPROX": (None, (3, 13)),
    "time.CLOCK_PROCESS_CPUTIME_ID": (None, (3, 3)),
    "time.CLOCK_PROF": (None, (3, 7)),
    "time.CLOCK_REALTIME": (None, (3, 3)),
    "time.CLOCK_TAI": (None, (3, 9)),
    "time.CLOCK_THREAD_CPUTIME_ID": (None, (3, 3)),
    "time.CLOCK_UPTIME": (None, (3, 7)),
    "time.CLOCK_UPTIME_RAW": (None, (3, 8)),
    "time.CLOCK_UPTIME_RAW_APPROX": (None, (3, 13)),
    "time.struct_time.tm_gmtoff": (None, (3, 3)),
    "time.struct_time.tm_zone": (None, (3, 3)),
    "token.ASYNC": (None, (3, 5)),
    "token.AWAIT": (None, (3, 5)),
    "token.COLONEQUAL": (None, (3, 8)),
    "token.COMMENT": (None, (3, 7)),
    "token.ENCODING": (None, (3, 7)),
    "token.NL": (None, (3, 7)),
    "token.TYPE_COMMENT": (None, (3, 8)),
    "token.TYPE_IGNORE": (None, (3, 8)),
    "traceback.TracebackException.__notes__": (None, (3, 11)),
    "traceback.TracebackException.exc_type_str": (None, (3, 13)),
    "tracemalloc.Filter.domain": (None, (3, 6)),
    "tracemalloc.Trace.domain": (None, (3, 6)),
    "tracemalloc.Traceback.total_nframe": (None, (3, 9)),
    "types.AsyncGeneratorType": (None, (3, 6)),
    "types.BooleanType": ((2, 3), None),
    "types.CellType": (None, (3, 8)),
    "types.ClassMethodDescriptorType": (None, (3, 7)),
    "types.CoroutineType": (None, (3, 5)),
    "types.EllipsisType": ((2, 0), (3, 10)),
    "types.GeneratorType": ((2, 2), (3, 0)),
    "types.GetSetDescriptorType": ((2, 5), (3, 0)),
    "types.MemberDescriptorType": ((2, 5), (3, 0)),
    "types.MethodDescriptorType": (None, (3, 7)),
    "types.MethodWrapperType": (None, (3, 7)),
    "types.ModuleType.__spec__": (None, (3, 4)),
    "types.NoneType": ((2, 0), (3, 10)),
    "types.NotImplementedType": ((2, 5), (3, 10)),
    "types.StringTypes": ((2, 2), None),
    "types.UnionType": (None, (3, 10)),
    "types.WrapperDescriptorType": (None, (3, 7)),
    "typing.Annotated": (None, (3, 9)),
    "typing.Concatenate": (None, (3, 10)),
    # Notes: `typing.NoReturn` was added in `typing` backport 3.6.2 (which requires 2.7 or 3.3+).
    "typing.NoReturn": (None, (3, 5)) if not bp("typing", config) else ((2, 7), (3, 3)),
    "typing.Final": (None, (3, 8)) if not bp("typing", config) else ((2, 7), (3, 8)),
    "typing.Literal": (None, (3, 8)) if not bp("typing", config) else ((2, 7), (3, 8)),
    "typing.ParamSpecArgs": (None, (3, 10)),
    "typing.ParamSpecKwargs": (None, (3, 10)),
    "typing.LiteralString": (None, (3, 11)),
    "typing.Never": (None, (3, 11)),
    "typing.NotRequired": (None, (3, 11)),
    "typing.Required": (None, (3, 11)),
    "typing.Self": (None, (3, 11)),
    "typing.TypeAlias": (None, (3, 10)),
    "typing.TypeGuard": (None, (3, 10)),
    "typing.TypeVarTuple": (None, (3, 11)),
    "typing.Unpack": (None, (3, 11)),
    "unicodedata.ucd_3_2_0": ((2, 5), (3, 0)),
    "unicodedata.unidata_version": ((2, 3), (3, 0)),
    "unittest.TestCase.longMessage": ((2, 7), (3, 1)),
    "unittest.TestCase.maxDiff": ((2, 7), (3, 2)),
    "unittest.TestLoader.errors": (None, (3, 5)),
    "unittest.TestLoader.testNamePatterns": (None, (3, 7)),
    "unittest.TestResult.buffer": ((2, 7), (3, 2)),
    "unittest.TestResult.failfast": ((2, 7), (3, 2)),
    "unittest.TestResult.skipped": ((2, 7), (3, 1)),
    "unittest.TestResult.tb_locals": (None, (3, 5)),
    "unittest.mock.Mock.call_args.args": bpv("mock", (None, (3, 8)), config),
    "unittest.mock.Mock.call_args.kwargs": bpv("mock", (None, (3, 8)), config),
    "urllib.error.HTTPError.headers": (None, (3, 4)),
    "urllib.parse._UNSAFE_URL_BYTES_TO_REMOVE": (None, (3, 10)),
    "urllib.request.Request.method": (None, (3, 3)),
    "urllib.response.addinfourl.status": (None, (3, 9)),
    "urlparse.urlparse.fragment": ((2, 5), None),
    "urlparse.urlparse.hostname": ((2, 5), None),
    "urlparse.urlparse.netloc": ((2, 5), None),
    "urlparse.urlparse.params": ((2, 5), None),
    "urlparse.urlparse.password": ((2, 5), None),
    "urlparse.urlparse.path": ((2, 5), None),
    "urlparse.urlparse.port": ((2, 5), None),
    "urlparse.urlparse.query": ((2, 5), None),
    "urlparse.urlparse.scheme": ((2, 5), None),
    "urlparse.urlparse.username": ((2, 5), None),
    "urlparse.urlsplit.fragment": ((2, 5), None),
    "urlparse.urlsplit.hostname": ((2, 5), None),
    "urlparse.urlsplit.netloc": ((2, 5), None),
    "urlparse.urlsplit.password": ((2, 5), None),
    "urlparse.urlsplit.path": ((2, 5), None),
    "urlparse.urlsplit.port": ((2, 5), None),
    "urlparse.urlsplit.query": ((2, 5), None),
    "urlparse.urlsplit.scheme": ((2, 5), None),
    "urlparse.urlsplit.username": ((2, 5), None),
    "uuid.MAX": (None, (3, 14)),
    "uuid.NIL": (None, (3, 14)),
    "uuid.UUID.is_safe": (None, (3, 7)),
    "weakref.ref.__callback__": (None, (3, 4)),
    "winreg.REG_QWORD": (None, (3, 6)),
    "winreg.REG_QWORD_LITTLE_ENDIAN": (None, (3, 6)),
    "xml.dom.DOMSTRING_SIZE_ERR": ((2, 1), (3, 0)),
    "xml.dom.EMPTY_NAMESPACE": ((2, 2), (3, 0)),
    "xml.dom.HIERARCHY_REQUEST_ERR": ((2, 1), (3, 0)),
    "xml.dom.INDEX_SIZE_ERR": ((2, 1), (3, 0)),
    "xml.dom.INUSE_ATTRIBUTE_ERR": ((2, 1), (3, 0)),
    "xml.dom.INVALID_ACCESS_ERR": ((2, 1), (3, 0)),
    "xml.dom.INVALID_CHARACTER_ERR": ((2, 1), (3, 0)),
    "xml.dom.INVALID_MODIFICATION_ERR": ((2, 1), (3, 0)),
    "xml.dom.INVALID_STATE_ERR": ((2, 1), (3, 0)),
    "xml.dom.NAMESPACE_ERR": ((2, 1), (3, 0)),
    "xml.dom.NOT_FOUND_ERR": ((2, 1), (3, 0)),
    "xml.dom.NOT_SUPPORTED_ERR": ((2, 1), (3, 0)),
    "xml.dom.NO_DATA_ALLOWED_ERR": ((2, 1), (3, 0)),
    "xml.dom.NO_MODIFICATION_ALLOWED_ERR": ((2, 1), (3, 0)),
    "xml.dom.SYNTAX_ERR": ((2, 1), (3, 0)),
    "xml.dom.WRONG_DOCUMENT_ERR": ((2, 1), (3, 0)),
    "xml.dom.XHTML_NAMESPACE": ((2, 2), (3, 0)),
    "xml.dom.XMLNS_NAMESPACE": ((2, 2), (3, 0)),
    "xml.dom.XML_NAMESPACE": ((2, 2), (3, 0)),
    "xml.parsers.expat.ExpatError.code": ((2, 1), (3, 0)),
    "xml.parsers.expat.ExpatError.lineno": ((2, 1), (3, 0)),
    "xml.parsers.expat.ExpatError.offset": ((2, 1), (3, 0)),
    "xml.parsers.expat.XMLParserType.CurrentByteIndex": ((2, 4), (3, 0)),
    "xml.parsers.expat.XMLParserType.CurrentColumnNumber": ((2, 4), (3, 0)),
    "xml.parsers.expat.XMLParserType.CurrentLineNumber": ((2, 4), (3, 0)),
    "xml.parsers.expat.XMLParserType.buffer_size": ((2, 3), (3, 0)),
    "xml.parsers.expat.XMLParserType.buffer_text": ((2, 3), (3, 0)),
    "xml.parsers.expat.XMLParserType.buffer_used": ((2, 3), (3, 0)),
    "xml.parsers.expat.XMLParserType.ordered_attributes": ((2, 1), (3, 0)),
    "xml.parsers.expat.XMLParserType.returns_unicode": ((2, 0), None),
    "xml.parsers.expat.XMLParserType.specified_attributes": ((2, 1), (3, 0)),
    "xml.parsers.expat.errors.codes": (None, (3, 2)),
    "xml.parsers.expat.errors.messages": (None, (3, 2)),
    "xmlrpc.server.SimpleXMLRPCRequestHandler.rpc_paths": (None, (3, 0)),
    "zipfile.Path.stem": (None, (3, 11)),
    "zipfile.Path.suffix": (None, (3, 11)),
    "zipfile.Path.suffixes": (None, (3, 11)),
    "zipfile.ZIP_BZIP2": (None, (3, 3)),
    "zipfile.ZIP_LZMA": (None, (3, 3)),
    "zipfile.ZIP_ZSTANDARD": (None, (3, 14)),
    "zlib.Decompress.eof": (None, (3, 3)),
    "zlib.ZLIB_RUNTIME_VERSION": (None, (3, 3)),

    # Decorators
    "abc.abstractclassmethod": (None, (3, 2)),
    "abc.abstractstaticmethod": (None, (3, 2)),
    "contextlib.asynccontextmanager": (None, (3, 7)),
    "enum.global_enum": (None, (3, 11)),
    "enum.member": (None, (3, 11)),
    "enum.nonmember": (None, (3, 11)),
    "enum.property": (None, (3, 11)),
    "enum.verify": (None, (3, 11)),
    "functools.cache": (None, (3, 9)),
    "functools.cached_property": (None, (3, 8)),
    "functools.lru_cache": (None, (3, 2)),
    "functools.partial_method": (None, (3, 4)),
    "functools.singledispatch": (None, (3, 4)),
    "functools.singledispatchmethod": (None, (3, 8)),
    "functools.total_ordering": ((2, 7), (3, 2)),
    "reprlib.recursive_repr": (None, (3, 2)),
    "typing.dataclass_transform": (None, (3, 11)),
    "typing.final": (None, (3, 8)) if not bp("typing", config) else ((2, 7), (3, 8)),
    "typing.override": (None, (3, 12)),
    "typing.runtime_checkable": (None, (3, 8)) if not bp("typing", config) else ((2, 7), (3, 8)),
    "unittest.expectedFailure": ((2, 7), (3, 1)),
    "unittest.skip": ((2, 7), (3, 1)),
    "unittest.skipIf": ((2, 7), (3, 1)),
    "unittest.skipUnless": ((2, 7), (3, 1)),
    "warnings.deprecated": (None, (3, 13)),
   }

# Keyword arguments requirements: (function, keyword argument) -> (requirements)
# Notes:
#  `bytes.decode(encoding, errors)` and `bytearray.decode(encoding, errors)`
#  are documented to be in 3.1 but they do not actually work, they work on 3.2+
def KWARGS_REQS(config):
  return {
    ("BaseHTTPServer.BaseHTTPRequestHandler.date_time_string", "timestamp"): ((2, 5), None),
    ("ConfigParser.ConfigParser", "allow_no_value"): ((2, 6), None),
    ("ConfigParser.ConfigParser", "dict_type"): ((2, 6), None),
    ("ConfigParser.RawConfigParser", "allow_no_value"): ((2, 6), None),
    ("ConfigParser.RawConfigParser", "dict_type"): ((2, 6), None),
    ("ConfigParser.SafeConfigParser", "allow_no_value"): ((2, 6), None),
    ("ConfigParser.SafeConfigParser", "dict_type"): ((2, 6), None),
    ("ImportError", "name"): (None, (3, 3)),
    ("ImportError", "path"): (None, (3, 3)),
    ("OSError", "filename2"): (None, (3, 4)),
    ("Queue.Queue.get", "timeout"): ((2, 3), None),
    ("Queue.Queue.put", "timeout"): ((2, 3), None),
    ("SimpleXMLRPCServer.CGIXMLRPCRequestHandler", "allow_none"): ((2, 5), None),
    ("SimpleXMLRPCServer.CGIXMLRPCRequestHandler", "encoding"): ((2, 5), None),
    ("SimpleXMLRPCServer.SimpleXMLRPCServer", "allow_none"): ((2, 5), None),
    ("SimpleXMLRPCServer.SimpleXMLRPCServer", "bind_and_activate"): ((2, 6), None),
    ("SimpleXMLRPCServer.SimpleXMLRPCServer", "encoding"): ((2, 5), None),
    ("SimpleXMLRPCServer.SimpleXMLRPCServer.register_instance", "allow_dotted_names"):
    ((2, 3), None),
    ("Tkinter.Tk", "useTk"): ((2, 4), None),
    ("__import__", "level"): ((2, 5), (3, 0)),
    ("_thread.lock.acquire", "timeout"): (None, (3, 2)),
    ("_thread.interrupt_main", "signum"): (None, (3, 10)),
    ("argparse.ArgumentParser", "allow_abbrev"): (None, (3, 5)),
    ("argparse.ArgumentParser", "color"): (None, (3, 14)),
    ("argparse.ArgumentParser", "exit_on_error"): (None, (3, 9)),
    ("argparse.ArgumentParser", "suggest_on_error"): (None, (3, 14)),
    ("argparse.ArgumentParser.add_argument", "deprecated"): (None, (3, 13)),
    ("argparse.ArgumentParser.add_parser", "deprecated"): (None, (3, 13)),
    ("argparse.ArgumentParser.add_subparsers", "required"): (None, (3, 7)),
    ("argparse.FileType", "encodings"): (None, (3, 4)),
    ("argparse.FileType", "errors"): (None, (3, 4)),
    ("array.array.index", "start"): (None, (3, 10)),
    ("array.array.index", "stop"): (None, (3, 10)),
    ("ast.dump", "indent"): (None, (3, 9)),
    ("ast.dump", "show_empty"): (None, (3, 13)),
    ("ast.parse", "feature_version"): (None, (3, 8)),
    ("ast.parse", "type_comments"): (None, (3, 8)),
    ("ast.parse", "optimize"): (None, (3, 13)),
    ("asyncio.Future.add_done_callback", "context"): (None, (3, 7)),
    ("asyncio.Future.cancel", "msg"): (None, (3, 9)),
    ("asyncio.Task", "context"): (None, (3, 11)),
    ("asyncio.Task", "eager_start"): (None, (3, 12)),
    ("asyncio.Task", "name"): (None, (3, 8)),
    ("asyncio.Task.cancel", "msg"): (None, (3, 9)),
    ("asyncio.create_task", "context"): (None, (3, 11)),
    ("asyncio.create_task", "eager_start"): (None, (3, 14)),
    ("asyncio.create_task", "name"): (None, (3, 8)),
    ("asyncio.loop.call_at", "context"): (None, (3, 7)),
    ("asyncio.loop.call_later", "context"): (None, (3, 7)),
    ("asyncio.loop.call_soon", "context"): (None, (3, 7)),
    ("asyncio.loop.call_soon_threadsafe", "context"): (None, (3, 7)),
    ("asyncio.loop.connect_accepted_socket", "ssl_handshake_timeout"): (None, (3, 7)),
    ("asyncio.loop.connect_accepted_socket", "ssl_shutdown_timeout"): (None, (3, 11)),
    ("asyncio.loop.create_connection", "happy_eyeballs_delay"): (None, (3, 8)),
    ("asyncio.loop.create_connection", "interleave"): (None, (3, 8)),
    ("asyncio.loop.create_connection", "ssl_handshake_timeout"): (None, (3, 7)),
    ("asyncio.loop.create_connection", "ssl_shutdown_timeout"): (None, (3, 11)),
    ("asyncio.loop.create_datagram_endpoint", "allow_broadcast"): (None, (3, 4)),
    ("asyncio.loop.create_datagram_endpoint", "family"): (None, (3, 4)),
    ("asyncio.loop.create_datagram_endpoint", "flags"): (None, (3, 4)),
    ("asyncio.loop.create_datagram_endpoint", "proto"): (None, (3, 4)),
    ("asyncio.loop.create_datagram_endpoint", "reuse_address"): (None, (3, 4)),
    ("asyncio.loop.create_datagram_endpoint", "reuse_port"): (None, (3, 4)),
    ("asyncio.loop.create_datagram_endpoint", "sock"): (None, (3, 4)),
    ("asyncio.loop.create_server", "ssl_handshake_timeout"): (None, (3, 6)),
    ("asyncio.loop.create_server", "ssl_shutdown_timeout"): (None, (3, 11)),
    ("asyncio.loop.create_server", "start_serving"): (None, (3, 6)),
    ("asyncio.loop.create_task", "context"): (None, (3, 11)),
    ("asyncio.loop.create_task", "name"): (None, (3, 8)),
    ("asyncio.loop.create_unix_connection", "path"): (None, (3, 7)),
    ("asyncio.loop.create_unix_connection", "ssl_handshake_timeout"): (None, (3, 7)),
    ("asyncio.loop.create_unix_connection", "ssl_shutdown_timeout"): (None, (3, 11)),
    ("asyncio.loop.create_unix_server", "path"): (None, (3, 7)),
    ("asyncio.loop.create_unix_server", "ssl_handshake_timeout"): (None, (3, 7)),
    ("asyncio.loop.create_unix_server", "ssl_shutdown_timeout"): (None, (3, 11)),
    ("asyncio.loop.create_unix_server", "start_serving"): (None, (3, 7)),
    ("asyncio.loop.start_tls", "ssl_shutdown_timeout"): (None, (3, 11)),
    ("asyncio.open_connection", "ssl_handshake_timeout"): (None, (3, 7)),
    ("asyncio.open_unix_connection", "path"): (None, (3, 7)),
    ("asyncio.open_unix_connection", "ssl_handshake_timeout"): (None, (3, 7)),
    ("asyncio.run", "loop_factory"): (None, (3, 12)),
    ("asyncio.start_server", "ssl_handshake_timeout"): (None, (3, 7)),
    ("asyncio.start_server", "start_serving"): (None, (3, 7)),
    ("asyncio.start_unix_server", "path"): (None, (3, 7)),
    ("asyncio.start_unix_server", "ssl_handshake_timeout"): (None, (3, 7)),
    ("asyncio.start_unix_server", "start_serving"): (None, (3, 7)),
    ("bdb.Bdb", "skip"): ((2, 7), (3, 1)),
    ("binascii.a2b_base64", "strict_mode"): (None, (3, 11)),
    ("binascii.b2a_base64", "newline"): (None, (3, 6)),
    ("binascii.b2a_hex", "bytes_per_sep"): (None, (3, 8)),
    ("binascii.b2a_hex", "sep"): (None, (3, 8)),
    ("binascii.b2a_uu", "backtick"): (None, (3, 7)),
    ("binascii.hexlify", "bytes_per_sep"): (None, (3, 8)),
    ("binascii.hexlify", "sep"): (None, (3, 8)),
    ("bisect.bisect", "key"): (None, (3, 10)),
    ("bisect.bisect_left", "key"): (None, (3, 10)),
    ("bisect.bisect_right", "key"): (None, (3, 10)),
    ("bisect.insort", "key"): (None, (3, 10)),
    ("bisect.insort_left", "key"): (None, (3, 10)),
    ("bisect.insort_right", "key"): (None, (3, 10)),
    ("bytearray.decode", "encoding"): (None, (3, 2)),
    ("bytearray.decode", "errors"): (None, (3, 2)),
    ("bytearray.hex", "bytes_per_sep"): (None, (3, 8)),
    ("bytearray.hex", "sep"): (None, (3, 8)),
    ("bytearray.translate", "delete"): (None, (3, 6)),
    ("bytes.decode", "encoding"): (None, (3, 2)),
    ("bytes.decode", "errors"): (None, (3, 2)),
    ("bytes.hex", "bytes_per_sep"): (None, (3, 8)),
    ("bytes.hex", "sep"): (None, (3, 8)),
    ("bytes.translate", "delete"): (None, (3, 6)),
    ("bz2.BZ2Decompressor.decompress", "max_length"): (None, (3, 5)),
    # Security fix backported to 3.6.13+, 3.7.10+, 3.8.8+, 3.9.2+, 3.10+
    ("cgi.parse", "separator"): (None, (3, 6)),
    ("cgi.parse_multipart", "encoding"): (None, (3, 7)),
    ("cgi.parse_multipart", "errors"): (None, (3, 7)),
    # Security fix backported to 3.6.13+, 3.7.10+, 3.8.8+, 3.9.2+, 3.10+
    ("cgi.parse_multipart", "separator"): (None, (3, 6)),
    ("cgi.parse_qs", "max_num_fields"): ((2, 7), None),
    ("cgi.parse_qsl", "max_num_fields"): ((2, 7), None),
    ("cmath.log", "base"): ((2, 4), (3, 0)),
    ("cmd.Cmd", "stdin"): ((2, 3), (3, 0)),
    ("cmd.Cmd", "stdout"): ((2, 3), (3, 0)),
    ("code.interact", "exitmsg"): (None, (3, 6)),
    ("codecs.StreamReader.read", "chars"): ((2, 4), (3, 0)),
    ("codecs.StreamReader.read", "firstline"): ((2, 4), (3, 0)),
    ("codecs.StreamReader.readline", "keepends"): ((2, 4), (3, 0)),
    ("collections.ChainMap.new_child", "m"): (None, (3, 4)),
    ("collections.abc.ByteString.index", "start"): (None, (3, 5)),
    ("collections.abc.ByteString.index", "stop"): (None, (3, 5)),
    ("collections.abc.MutableSequence.index", "start"): (None, (3, 5)),
    ("collections.abc.MutableSequence.index", "stop"): (None, (3, 5)),
    ("collections.abc.Sequence.index", "start"): (None, (3, 5)),
    ("collections.abc.Sequence.index", "stop"): (None, (3, 5)),
    ("collections.deque", "maxlen"): ((2, 6), (3, 0)),
    ("collections.namedtuple", "defaults"): (None, (3, 7)),
    ("collections.namedtuple", "module"): (None, (3, 6)),
    ("collections.namedtuple", "rename"): ((2, 7), (3, 1)),
    ("compile", "dont_inherit"): ((2, 3), (3, 0)),
    ("compile", "flags"): ((2, 3), (3, 0)),
    ("compile", "optimize"): (None, (3, 2)),
    ("compileall.compile_dir", "hardlink_dupes"): (None, (3, 9)),
    ("compileall.compile_dir", "invalidation_mode"): (None, (3, 7)),
    ("compileall.compile_dir", "legacy"): (None, (3, 2)),
    ("compileall.compile_dir", "limit_sl_dest"): (None, (3, 9)),
    ("compileall.compile_dir", "optimize"): (None, (3, 2)),
    ("compileall.compile_dir", "prependdir"): (None, (3, 9)),
    ("compileall.compile_dir", "stripdir"): (None, (3, 9)),
    ("compileall.compile_dir", "workers"): (None, (3, 5)),
    ("compileall.compile_file", "hardlink_dupes"): (None, (3, 9)),
    ("compileall.compile_file", "invalidation_mode"): (None, (3, 7)),
    ("compileall.compile_file", "limit_sl_dest"): (None, (3, 9)),
    ("compileall.compile_file", "prependdir"): (None, (3, 9)),
    ("compileall.compile_file", "stripdir"): (None, (3, 9)),
    ("compileall.compile_path", "invalidation_mode"): (None, (3, 7)),
    ("compileall.compile_path", "legacy"): (None, (3, 2)),
    ("compileall.compile_path", "optimize"): (None, (3, 2)),
    ("concurrent.futures.Executor.map", "buffersize"): (None, (3, 14)),
    ("concurrent.futures.Executor.map", "chunksize"): (None, (3, 5)),
    ("concurrent.futures.Executor.shutdown", "cancel_futures"): (None, (3, 9)),
    ("concurrent.futures.ProcessPoolExecutor", "initargs"): (None, (3, 7)),
    ("concurrent.futures.ProcessPoolExecutor", "initializer"): (None, (3, 7)),
    ("concurrent.futures.ProcessPoolExecutor", "max_tasks_per_child"): (None, (3, 11)),
    ("concurrent.futures.ProcessPoolExecutor", "mp_context"): (None, (3, 7)),
    ("concurrent.futures.ThreadPoolExecutor", "initargs"): (None, (3, 7)),
    ("concurrent.futures.ThreadPoolExecutor", "initializer"): (None, (3, 7)),
    ("concurrent.futures.ThreadPoolExecutor", "thread_name_prefix"): (None, (3, 6)),
    ("configparser.ConfigParser", "allow_no_value"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.ConfigParser", "allow_unnamed_section"):
    bpv("configparser", (None, (3, 13)), config),
    ("configparser.ConfigParser", "comment_prefixes"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.ConfigParser", "converters"): bpv("configparser", (None, (3, 5)), config),
    ("configparser.ConfigParser", "default_section"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.ConfigParser", "delimiters"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.ConfigParser", "empty_lines_in_values"): bpv("configparser", (None, (3, 2)),
                                                                config),
    ("configparser.ConfigParser", "interpolation"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.ConfigParser", "strict"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.ConfigParser.read", "encoding"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.RawConfigParser", "allow_no_value"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.RawConfigParser", "allow_unnamed_section"):
    bpv("configparser", (None, (3, 13)), config),
    ("configparser.RawConfigParser", "comment_prefixes"): bpv("configparser", (None, (3, 2)),
                                                              config),
    ("configparser.RawConfigParser", "converters"): bpv("configparser", (None, (3, 5)), config),
    ("configparser.RawConfigParser", "default_section"): bpv("configparser", (None, (3, 2)),
                                                             config),
    ("configparser.RawConfigParser", "delimiters"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.RawConfigParser", "empty_lines_in_values"): bpv("configparser", (None, (3, 2)),
                                                                   config),
    ("configparser.RawConfigParser", "interpolation"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.RawConfigParser", "strict"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.RawConfigParser.read", "encoding"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.DuplicateSectionError", "lineno"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.DuplicateSectionError", "source"): bpv("configparser", (None, (3, 2)), config),
    ("configparser.ParsingError", "source"): bpv("configparser", (None, (3, 2)), config),
    ("crypt.mksalt", "rounds"): (None, (3, 7)),
    ("ctypes.CDLL", "use_errno"): ((2, 6), (3, 0)),
    ("ctypes.CDLL", "use_last_error"): ((2, 6), (3, 0)),
    ("ctypes.CDLL", "winmode"): (None, (3, 8)),
    ("ctypes.CFUNCTYPE", "use_errno"): ((2, 6), (3, 0)),
    ("ctypes.CFUNCTYPE", "use_last_error"): ((2, 6), (3, 0)),
    ("ctypes.OleDLL", "use_errno"): ((2, 6), (3, 0)),
    ("ctypes.OleDLL", "use_last_error"): ((2, 6), (3, 0)),
    ("ctypes.OleDLL", "winmode"): (None, (3, 8)),
    ("ctypes.WinDLL", "use_errno"): ((2, 6), (3, 0)),
    ("ctypes.WinDLL", "use_last_error"): ((2, 6), (3, 0)),
    ("ctypes.WinDLL", "winmode"): (None, (3, 8)),
    ("ctypes.byref", "offset"): ((2, 6), (3, 0)),
    ("dataclasses.dataclass", "kw_only"): (None, (3, 10)),
    ("dataclasses.dataclass", "match_args"): (None, (3, 10)),
    ("dataclasses.dataclass", "slots"): (None, (3, 10)),
    ("dataclasses.field", "kw_only"): (None, (3, 10)),
    ("datetime.datetime", "fold"): (None, (3, 6)),
    ("datetime.datetime.combine", "tzinfo"): (None, (3, 6)),
    ("datetime.datetime.isoformat", "timespec"): (None, (3, 6)),
    ("datetime.datetime.replace", "fold"): (None, (3, 6)),
    ("datetime.time.isoformat", "timespec"): (None, (3, 6)),
    ("datetime.time.replace", "fold"): (None, (3, 6)),
    ("difflib.HtmlDiff.make_file", "charset"): (None, (3, 5)),
    ("difflib.SequenceMatcher", "autojunk"): ((2, 7), (3, 2)),
    ("dis.Bytecode", "show_caches"): (None, (3, 11)),
    ("dis.Bytecode", "show_offsets"): (None, (3, 13)),
    ("dis.Bytecode", "show_positions"): (None, (3, 14)),
    ("dis.dis", "depth"): (None, (3, 7)),
    ("dis.dis", "file"): (None, (3, 4)),
    ("dis.dis", "show_caches"): (None, (3, 11)),
    ("dis.dis", "show_positions"): (None, (3, 14)),
    ("dis.disassemble", "file"): (None, (3, 4)),
    ("dis.disassemble", "show_caches"): (None, (3, 11)),
    ("dis.disassemble", "show_offsets"): (None, (3, 13)),
    ("dis.disco", "file"): (None, (3, 4)),
    ("dis.disco", "show_caches"): (None, (3, 11)),
    ("dis.disco", "show_positions"): (None, (3, 14)),
    ("dis.distb", "file"): (None, (3, 4)),
    ("dis.distb", "show_caches"): (None, (3, 11)),
    ("dis.distb", "show_offsets"): (None, (3, 13)),
    ("dis.distb", "show_positions"): (None, (3, 14)),
    ("dis.get_instructions", "show_caches"): (None, (3, 11)),
    ("dis.show_code", "file"): (None, (3, 4)),
    ("dis.stack_effect", "jump"): (None, (3, 8)),
    ("doctest.DocFileSuite", "encoding"): ((2, 5), (3, 0)),
    ("doctest.DocTestSuite", "extraglobs"): ((2, 4), (3, 0)),
    ("doctest.DocTestSuite", "globs"): ((2, 4), (3, 0)),
    ("doctest.DocTestSuite", "optionflags"): ((2, 4), (3, 0)),
    ("doctest.DocTestSuite", "setUp"): ((2, 4), (3, 0)),
    ("doctest.DocTestSuite", "tearDown"): ((2, 4), (3, 0)),
    ("doctest.DocTestSuite", "test_finder"): ((2, 4), (3, 0)),
    ("doctest.debug", "pm"): ((2, 4), (3, 0)),
    ("doctest.testfile", "encoding"): ((2, 5), (3, 0)),
    ("doctest.testmod", "exclude_empty"): ((2, 4), (3, 0)),
    ("doctest.testmod", "extraglobs"): ((2, 4), (3, 0)),
    ("doctest.testmod", "optionflags"): ((2, 3), (3, 0)),
    ("doctest.testmod", "raise_on_error"): ((2, 4), (3, 0)),
    ("email.generator.BytesGenerator", "policy"): (None, (3, 3)),
    ("email.generator.Generator", "policy"): (None, (3, 3)),
    ("email.generator.Generator.flatten", "linesep"): (None, (3, 2)),
    ("email.header.Header.encode", "linesep"): (None, (3, 2)),
    ("email.message.EmailMessage.set_param", "replace"): (None, (3, 4)),
    ("email.message.Message", "policy"): (None, (3, 3)),
    ("email.message.Message.as_string", "policy"): (None, (3, 4)),
    ("email.message.Message.get_param", "unquote"): ((2, 2), (3, 0)),
    ("email.message.Message.get_params", "unquote"): ((2, 2), (3, 0)),
    ("email.message.Message.set_param", "replace"): (None, (3, 4)),
    ("email.message.Message.set_payload", "charset"): ((2, 2), (3, 0)),
    ("email.message_from_binary_file", "policy"): (None, (3, 3)),
    ("email.message_from_bytes", "policy"): (None, (3, 3)),
    ("email.message_from_file", "policy"): (None, (3, 3)),
    ("email.message_from_file", "strict"): ((2, 2), (3, 0)),
    ("email.message_from_string", "policy"): (None, (3, 3)),
    ("email.message_from_string", "strict"): ((2, 2), (3, 0)),
    ("email.mime.application.MIMEApplication", "policy"): (None, (3, 6)),
    ("email.mime.audio.MIMEAudio", "policy"): (None, (3, 6)),
    ("email.mime.base.MIMEBase", "policy"): (None, (3, 6)),
    ("email.mime.image.MIMEImage", "policy"): (None, (3, 6)),
    ("email.mime.message.MIMEMessage", "policy"): (None, (3, 6)),
    ("email.mime.multipart.MIMEMultipart", "policy"): (None, (3, 6)),
    ("email.mime.text.MIMEText", "policy"): (None, (3, 6)),
    ("email.parser.BytesFeedParser", "policy"): (None, (3, 3)),
    ("email.parser.BytesParser", "policy"): (None, (3, 3)),
    ("email.parser.FeedParser", "policy"): (None, (3, 3)),
    ("email.parser.Parser", "policy"): (None, (3, 3)),
    ("email.parser.Parser", "strict"): ((2, 2), (3, 0)),
    ("email.parser.Parser.parse", "headersonly"): ((2, 2), (3, 0)),
    ("email.parser.Parser.parsestr", "headersonly"): ((2, 2), (3, 0)),
    ("email.policy.Policy", "mangle_from_"): (None, (3, 5)),
    ("email.utils.getaddresses", "strict"): (None, (3, 13)),
    ("email.utils.formataddr", "charset"): (None, (3, 3)),
    ("email.utils.formatdate", "charset"): (None, (3, 3)),
    ("email.utils.make_msgid", "domain"): (None, (3, 2)),
    ("email.utils.parseaddr", "strict"): (None, (3, 13)),
    ("enum.Enum", "start"): bpv("enum", (None, (3, 5)), config),
    ("enumerate", "start"): ((2, 6), (3, 0)),
    ("exec", "closure"): (None, (3, 11)),
    ("exec", "globals"): (None, (3, 13)),
    ("exec", "locals"): (None, (3, 13)),
    ("eval", "globals"): (None, (3, 13)),
    ("eval", "locals"): (None, (3, 13)),
    ("fileinput.FileInput", "encoding"): (None, (3, 10)),
    ("fileinput.FileInput", "errors"): (None, (3, 10)),
    ("fileinput.FileInput", "mode"): ((2, 5), (3, 0)),
    ("fileinput.FileInput", "openhook"): ((2, 5), (3, 0)),
    ("fileinput.hook_compressed", "encoding"): (None, (3, 10)),
    ("fileinput.hook_compressed", "errors"): (None, (3, 10)),
    ("fileinput.hook_encoded", "errors"): (None, (3, 6)),
    ("fileinput.input", "encoding"): (None, (3, 10)),
    ("fileinput.input", "errors"): (None, (3, 10)),
    ("fileinput.input", "mode"): ((2, 5), (3, 0)),
    ("fileinput.input", "openhook"): ((2, 5), (3, 0)),
    ("ftplib.FTP", "encoding"): (None, (3, 9)),
    ("ftplib.FTP", "source_address"): (None, (3, 3)),
    ("ftplib.FTP", "timeout"): ((2, 6), (3, 0)),
    ("ftplib.FTP.connect", "source_address"): (None, (3, 3)),
    ("ftplib.FTP.connect", "timeout"): ((2, 6), (3, 0)),
    ("ftplib.FTP.storbinary", "callback"): ((2, 6), (3, 0)),
    ("ftplib.FTP.storbinary", "rest"): ((2, 7), (3, 2)),
    ("ftplib.FTP.storlines", "callback"): ((2, 6), (3, 0)),
    ("ftplib.FTP_TLS", "context"): ((2, 7), (3, 2)),
    ("ftplib.FTP_TLS", "encoding"): (None, (3, 9)),
    ("ftplib.FTP_TLS", "source_address"): (None, (3, 3)),
    ("functools.reduce", "initial"): (None, (3, 14)),
    ("functools.lru_cache", "typed"): (None, (3, 3)),
    # ("functools.lru_cache", "user_function"): (None, (3, 8)), # See DECORATOR_USER_FUNCTIONS
    ("gc.collect", "generation"): ((2, 5), (3, 0)),
    ("gc.get_objects", "generation"): (None, (3, 8)),
    ("getpass.getpass", "echo_char"): (None, (3, 14)),
    ("getpass.getpass", "stream"): ((2, 5), (3, 0)),
    ("gettext.NullTranslations.install", "names"): ((2, 5), (3, 0)),
    ("gettext.install", "codeset"): ((2, 4), (3, 0)),
    ("gettext.install", "names"): ((2, 5), (3, 0)),
    ("gettext.translation", "codeset"): ((2, 4), (3, 0)),
    ("glob.glob", "dir_fd"): (None, (3, 10)),
    ("glob.glob", "include_hidden"): (None, (3, 11)),
    ("glob.glob", "recursive"): (None, (3, 5)),
    ("glob.glob", "root_dir"): (None, (3, 10)),
    ("glob.iglob", "dir_fd"): (None, (3, 10)),
    ("glob.iglob", "include_hidden"): (None, (3, 11)),
    ("glob.iglob", "recursive"): (None, (3, 5)),
    ("glob.iglob", "root_dir"): (None, (3, 10)),
    ("gzip.GzipFile", "mtime"): ((2, 7), (3, 1)),
    ("gzip.compress", "mtime"): (None, (3, 8)),
    ("gzip.open", "encoding"): (None, (3, 3)),
    ("gzip.open", "errors"): (None, (3, 3)),
    ("gzip.open", "newline"): (None, (3, 3)),
    ("hashlib.blake2b", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.blake2s", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.md5", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.new", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.sha1", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.sha224", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.sha256", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.sha384", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.sha3_224", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.sha3_256", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.sha3_384", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.sha3_512", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.sha512", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.shake_128", "usedforsecurity"): (None, (3, 9)),
    ("hashlib.shake_256", "usedforsecurity"): (None, (3, 9)),
    ("heapq.merge", "key"): (None, (3, 5)),
    ("heapq.merge", "reverse"): (None, (3, 5)),
    ("heapq.nlargest", "key"): ((2, 5), (3, 0)),
    ("heapq.nsmallest", "key"): ((2, 5), (3, 0)),
    ("html.parser.HTMLParser", "convert_charrefs"): (None, (3, 4)),
    ("http.client.HTTPConnection", "blocksize"): (None, (3, 7)),
    ("http.client.HTTPConnection", "source_address"): (None, (3, 2)),
    ("http.client.HTTPConnection.endheaders", "encode_chunked"): (None, (3, 6)),
    ("http.client.HTTPConnection.request", "encode_chunked"): (None, (3, 6)),
    ("http.client.HTTPSConnection", "check_hostname"): (None, (3, 2)),
    ("http.client.HTTPSConnection", "context"): (None, (3, 2)),
    ("http.client.HTTPSConnection", "source_address"): (None, (3, 2)),
    ("http.server.BaseHTTPRequestHandler.send_error", "explain"): (None, (3, 4)),
    ("http.server.SimpleHTTPRequestHandler", "directory"): (None, (3, 7)),
    ("httplib.HTTPConnection", "source_address"): ((2, 7), None),
    ("httplib.HTTPConnection", "timeout"): ((2, 6), None),
    ("httplib.HTTPConnection.endheaders", "message_body"): ((2, 7), None),
    ("httplib.HTTPConnection.putrequest", "skip_accept_encoding"): ((2, 4), None),
    ("httplib.HTTPSConnection", "context"): ((2, 7), None),
    ("httplib.HTTPSConnection", "source_address"): ((2, 7), None),
    ("httplib.HTTPSConnection", "timeout"): ((2, 6), None),
    ("imaplib.IMAP4", "timeout"): (None, (3, 9)),
    ("imaplib.IMAP4.open", "timeout"): (None, (3, 9)),
    ("imaplib.IMAP4_SSL", "ssl_context"): (None, (3, 3)),
    ("imaplib.IMAP4_SSL", "timeout"): (None, (3, 9)),
    ("importlib.util.cache_from_source", "optimization"): (None, (3, 5)),
    ("inspect.Signature.format", "unquote_annotations"): (None, (3, 14)),
    ("inspect.Signature.from_callable", "globals"): (None, (3, 10)),
    ("inspect.Signature.from_callable", "locals"): (None, (3, 10)),
    ("inspect.signature", "annotation_format"): (None, (3, 14)),
    ("inspect.signature", "eval_str"): (None, (3, 10)),
    ("inspect.signature", "follow_wrapped"): (None, (3, 5)),
    ("inspect.signature", "globals"): (None, (3, 10)),
    ("inspect.signature", "locals"): (None, (3, 10)),
    ("io.FileIO", "opener"): (None, (3, 3)),
    ("io.TextIOWrapper", "write_through"): (None, (3, 3)),
    ("io.TextIOWrapper.reconfigure", "encoding"): (None, (3, 11)),
    ("io.open", "opener"): (None, (3, 3)),
    ("itertools.accumulate", "func"): (None, (3, 3)),
    ("itertools.accumulate", "initial"): (None, (3, 8)),
    ("itertools.batched", "strict"): (None, (3, 13)),
    ("itertools.count", "step"): ((2, 7), (3, 1)),
    ("json.JSONDecoder", "object_pairs_hook"): ((2, 7), (3, 1)),
    ("json.load", "object_pairs_hook"): ((2, 7), (3, 1)),
    ("linecache.getline", "module_globals"): ((2, 5), (3, 0)),
    ("locale.format", "monetary"): ((2, 5), (3, 0)),
    ("locale.format_string", "monetary"): (None, (3, 7)),
    ("logging.FileHandler", "delay"): ((2, 6), (3, 0)),
    ("logging.FileHandler", "errors"): (None, (3, 9)),
    ("logging.Formatter", "defaults"): (None, (3, 10)),
    ("logging.Formatter", "style"): (None, (3, 2)),
    ("logging.Formatter", "validate"): (None, (3, 8)),
    ("logging.LogRecord", "func"): ((2, 5), (3, 0)),
    ("logging.Logger.critical", "stack_info"): (None, (3, 2)),
    ("logging.Logger.critical", "stacklevel"): (None, (3, 8)),
    ("logging.Logger.debug", "stack_info"): (None, (3, 2)),
    ("logging.Logger.debug", "stacklevel"): (None, (3, 8)),
    ("logging.Logger.error", "stack_info"): (None, (3, 2)),
    ("logging.Logger.error", "stacklevel"): (None, (3, 8)),
    ("logging.Logger.exception", "stack_info"): (None, (3, 2)),
    ("logging.Logger.exception", "stacklevel"): (None, (3, 8)),
    ("logging.Logger.info", "stack_info"): (None, (3, 2)),
    ("logging.Logger.info", "stacklevel"): (None, (3, 8)),
    ("logging.Logger.log", "stack_info"): (None, (3, 2)),
    ("logging.Logger.log", "stacklevel"): (None, (3, 8)),
    ("logging.Logger.makeRecord", "extra"): ((2, 5), (3, 0)),
    ("logging.Logger.makeRecord", "func"): ((2, 5), (3, 0)),
    ("logging.Logger.warn", "stack_info"): (None, (3, 2)),
    ("logging.Logger.warn", "stacklevel"): (None, (3, 8)),
    ("logging.Logger.warning", "stack_info"): (None, (3, 2)),
    ("logging.Logger.warning", "stacklevel"): (None, (3, 8)),
    ("logging.basicConfig", "datefmt"): ((2, 4), (3, 0)),
    ("logging.basicConfig", "encoding"): (None, (3, 9)),
    ("logging.basicConfig", "errors"): (None, (3, 9)),
    ("logging.basicConfig", "filemode"): ((2, 4), (3, 0)),
    ("logging.basicConfig", "filename"): ((2, 4), (3, 0)),
    ("logging.basicConfig", "force"): (None, (3, 8)),
    ("logging.basicConfig", "format"): ((2, 4), (3, 0)),
    ("logging.basicConfig", "handlers"): (None, (3, 3)),
    ("logging.basicConfig", "level"): ((2, 4), (3, 0)),
    ("logging.basicConfig", "stream"): ((2, 4), (3, 0)),
    ("logging.basicConfig", "style"): (None, (3, 2)),
    ("logging.config.fileConfig", "disable_existing_loggers"): ((2, 6), (3, 0)),
    ("logging.config.fileConfig", "encoding"): (None, (3, 10)),
    ("logging.config.listen", "verify"): (None, (3, 4)),
    ("logging.critical", "extra"): ((2, 5), (3, 0)),
    ("logging.critical", "stack_info"): (None, (3, 2)),
    ("logging.critical", "stacklevel"): (None, (3, 8)),
    ("logging.debug", "extra"): ((2, 5), (3, 0)),
    ("logging.debug", "stack_info"): (None, (3, 2)),
    ("logging.debug", "stacklevel"): (None, (3, 8)),
    ("logging.error", "extra"): ((2, 5), (3, 0)),
    ("logging.error", "stack_info"): (None, (3, 2)),
    ("logging.error", "stacklevel"): (None, (3, 8)),
    ("logging.exception", "extra"): ((2, 5), (3, 0)),
    ("logging.exception", "stack_info"): (None, (3, 2)),
    ("logging.exception", "stacklevel"): (None, (3, 8)),
    ("logging.handlers.HTTPHandler", "context"): (None, (3, 5)),
    ("logging.handlers.MemoryHandler", "flushOnClose"): (None, (3, 6)),
    ("logging.handlers.QueueListener", "respect_handler_level"): (None, (3, 5)),
    ("logging.handlers.RotatingFileHandler", "delay"): ((2, 6), (3, 0)),
    ("logging.handlers.RotatingFileHandler", "errors"): (None, (3, 9)),
    ("logging.handlers.SMTPHandler", "credentials"): ((2, 6), (3, 0)),
    ("logging.handlers.SMTPHandler", "secure"): ((2, 7), (3, 0)),
    ("logging.handlers.SMTPHandler", "timeout"): (None, (3, 3)),
    ("logging.handlers.SysLogHandler", "socktype"): ((2, 7), (3, 2)),
    ("logging.handlers.SysLogHandler", "timeout"): (None, (3, 14)),
    ("logging.handlers.TimedRotatingFileHandler", "atTime"): (None, (3, 4)),
    ("logging.handlers.TimedRotatingFileHandler", "delay"): ((2, 6), (3, 0)),
    ("logging.handlers.TimedRotatingFileHandler", "errors"): (None, (3, 9)),
    ("logging.handlers.TimedRotatingFileHandler", "utc"): ((2, 6), (3, 0)),
    ("logging.handlers.WatchedFileHandler", "errors"): (None, (3, 9)),
    ("logging.info", "extra"): ((2, 5), (3, 0)),
    ("logging.info", "stack_info"): (None, (3, 2)),
    ("logging.info", "stacklevel"): (None, (3, 8)),
    ("logging.log", "extra"): ((2, 5), (3, 0)),
    ("logging.log", "stack_info"): (None, (3, 2)),
    ("logging.log", "stacklevel"): (None, (3, 8)),
    ("logging.warn", "stack_info"): (None, (3, 2)),
    ("logging.warn", "stacklevel"): (None, (3, 8)),
    ("logging.warning", "extra"): ((2, 5), (3, 0)),
    ("logging.warning", "stack_info"): (None, (3, 2)),
    ("logging.warning", "stacklevel"): (None, (3, 8)),
    ("lzma.LZMADecompressor.decompress", "max_length"): (None, (3, 5)),
    ("marshal.dump", "allow_code"): (None, (3, 13)),
    ("marshal.dump", "version"): ((2, 4), (3, 0)),
    ("marshal.dumps", "allow_code"): (None, (3, 13)),
    ("marshal.dumps", "version"): ((2, 4), (3, 0)),
    ("marshal.load", "allow_code"): (None, (3, 13)),
    ("marshal.loads", "allow_code"): (None, (3, 13)),
    ("math.log", "base"): ((2, 3), (3, 0)),
    ("math.nextafter", "steps"): (None, (3, 12)),
    ("max", "default"): (None, (3, 4)),
    ("max", "key"): ((2, 5), (3, 0)),
    ("memoryview.hex", "bytes_per_sep"): (None, (3, 8)),
    ("memoryview.hex", "sep"): (None, (3, 8)),
    ("min", "default"): (None, (3, 4)),
    ("min", "key"): ((2, 5), (3, 0)),
    ("mmap.mmap", "trackfd"): (None, (3, 13)),
    ("multiprocessing.Pool", "maxtasksperchild"): ((2, 7), (3, 2)),
    ("multiprocessing.Process", "daemon"): (None, (3, 3)),
    ("multiprocessing.managers.BaseManager", "shutdown_timeout"): (None, (3, 11)),
    ("multiprocessing.pool.Pool", "context"): (None, (3, 4)),
    ("multiprocessing.pool.Pool", "maxtasksperchild"): (None, (3, 2)),
    ("nis.cat", "domain"): ((2, 5), (3, 0)),
    ("nis.maps", "domain"): ((2, 5), (3, 0)),
    ("nis.match", "domain"): ((2, 5), (3, 0)),
    ("nntplib.NNTP", "usenetrc"): ((2, 4), (3, 0)),
    ("nntplib.NNTP.list", "group_pattern"): (None, (3, 2)),
    ("open", "closefd"): (None, (3, 0)),
    ("open", "encoding"): (None, (3, 0)),
    ("open", "errors"): (None, (3, 0)),
    ("open", "newline"): (None, (3, 0)),
    ("open", "opener"): (None, (3, 3)),
    ("os.access", "dir_fd"): (None, (3, 3)),
    ("os.access", "effective_ids"): (None, (3, 3)),
    ("os.access", "follow_symlinks"): (None, (3, 3)),
    ("os.chflags", "follow_symlinks"): (None, (3, 3)),
    ("os.chmod", "dir_fd"): (None, (3, 3)),
    ("os.chmod", "follow_symlinks"): (None, (3, 3)),
    ("os.chown", "dir_fd"): (None, (3, 3)),
    ("os.chown", "follow_symlinks"): (None, (3, 3)),
    ("os.dup2", "inheritable"): (None, (3, 4)),
    ("os.link", "dst_dir_fd"): (None, (3, 3)),
    ("os.link", "follow_symlinks"): (None, (3, 3)),
    ("os.link", "src_dir_fd"): (None, (3, 3)),
    ("os.lstat", "dir_fd"): (None, (3, 3)),
    ("os.makedirs", "exist_ok"): (None, (3, 2)),
    ("os.mkdir", "dir_fd"): (None, (3, 3)),
    ("os.mkfifo", "dir_fd"): (None, (3, 3)),
    ("os.mknod", "dir_fd"): (None, (3, 3)),
    ("os.open", "dir_fd"): (None, (3, 3)),
    ("os.path.realpath", "strict"): (None, (3, 10)),
    ("os.readlink", "dir_fd"): (None, (3, 3)),
    ("os.remove", "dir_fd"): (None, (3, 3)),
    ("os.rename", "dst_dir_fd"): (None, (3, 3)),
    ("os.rename", "src_dir_fd"): (None, (3, 3)),
    ("os.rmdir", "dir_fd"): (None, (3, 3)),
    ("os.sendfile", "in_fd"): (None, (3, 9)),
    ("os.sendfile", "out_fd"): (None, (3, 9)),
    ("os.startfile", "arguments"): (None, (3, 10)),
    ("os.startfile", "cwd"): (None, (3, 10)),
    ("os.startfile", "operation"): ((2, 5), (3, 0)),
    ("os.startfile", "show_cmd"): (None, (3, 10)),
    ("os.stat", "dir_fd"): (None, (3, 3)),
    ("os.stat", "follow_symlinks"): (None, (3, 3)),
    ("os.symlink", "dir_fd"): (None, (3, 3)),
    ("os.unlink", "dir_fd"): (None, (3, 3)),
    ("os.utime", "dir_fd"): (None, (3, 3)),
    ("os.utime", "follow_symlinks"): (None, (3, 3)),
    ("os.utime", "ns"): (None, (3, 3)),
    ("os.walk", "followlinks"): ((2, 6), (3, 0)),
    ("pathlib.Path.chmod", "follow_symlinks"): (None, (3, 10)),
    ("pathlib.Path.exists", "follow_symlinks"): (None, (3, 12)),
    ("pathlib.Path.glob", "case_sensitive"): (None, (3, 12)),
    ("pathlib.Path.glob", "recurse_symlinks"): (None, (3, 13)),
    ("pathlib.Path.group", "follow_symlinks"): (None, (3, 13)),
    ("pathlib.Path.is_dir", "follow_symlinks"): (None, (3, 13)),
    ("pathlib.Path.is_file", "follow_symlinks"): (None, (3, 13)),
    ("pathlib.Path.mkdir", "exist_ok"): (None, (3, 5)),
    ("pathlib.Path.owner", "follow_symlinks"): (None, (3, 13)),
    ("pathlib.Path.read_text", "newline"): (None, (3, 13)),
    ("pathlib.Path.resolve", "strict"): (None, (3, 6)),
    ("pathlib.Path.rglob", "case_sensitive"): (None, (3, 12)),
    ("pathlib.Path.rglob", "recurse_symlinks"): (None, (3, 13)),
    ("pathlib.Path.stat", "follow_symlinks"): (None, (3, 10)),
    ("pathlib.Path.unlink", "missing_ok"): (None, (3, 8)),
    ("pathlib.Path.write_text", "newline"): (None, (3, 10)),
    ("pathlib.PurePath.match", "case_sensitive"): (None, (3, 12)),
    ("pathlib.PurePath.relative_to", "walk_up"): (None, (3, 12)),
    ("pdb.Pdb", "backend"): (None, (3, 14)),
    ("pdb.Pdb", "colorize"): (None, (3, 14)),
    ("pdb.Pdb", "mode"): (None, (3, 14)),
    ("pdb.Pdb", "nosigint"): (None, (3, 2)),
    ("pdb.Pdb", "readrc"): (None, (3, 6)),
    ("pdb.Pdb", "skip"): ((2, 7), (3, 1)),
    ("pdb.set_trace", "commands"): (None, (3, 14)),
    ("pdb.set_trace", "header"): (None, (3, 7)),
    ("pickle.Pickler", "buffer_callback"): (None, (3, 8)),
    ("pickle.Pickler", "protocol"): ((2, 3), (3, 0)),
    ("pickle.Unpickler", "buffers"): (None, (3, 8)),
    ("pickle.Unpickler", "encoding"): (None, (3, 0)),
    ("pickle.Unpickler", "errors"): (None, (3, 0)),
    ("pickle.dump", "buffer_callback"): (None, (3, 8)),
    ("pickle.dump", "protocol"): ((2, 3), (3, 0)),
    ("pickle.dumps", "buffer_callback"): (None, (3, 8)),
    ("pickle.dumps", "protocol"): ((2, 3), (3, 0)),
    ("pickle.load", "buffers"): (None, (3, 8)),
    ("pickle.load", "encoding"): (None, (3, 0)),
    ("pickle.load", "errors"): (None, (3, 0)),
    ("pickle.loads", "buffers"): (None, (3, 8)),
    ("pickle.loads", "encoding"): (None, (3, 0)),
    ("pickle.loads", "errors"): (None, (3, 0)),
    ("pickletools.dis", "annotate"): (None, (3, 2)),
    ("poplib.POP3", "timeout"): ((2, 6), (3, 0)),
    ("poplib.POP3_SSL", "context"): (None, (3, 2)),
    ("pow", "base"): (None, (3, 8)),
    ("pow", "exp"): (None, (3, 8)),
    ("pow", "mod"): (None, (3, 8)),
    ("pprint.PrettyPrinter", "compact"): (None, (3, 4)),
    ("pprint.PrettyPrinter", "sort_dicts"): (None, (3, 8)),
    ("pprint.PrettyPrinter", "underscore_numbers"): (None, (3, 10)),
    ("pprint.pformat", "compact"): (None, (3, 4)),
    ("pprint.pformat", "depth"): ((2, 4), (3, 0)),
    ("pprint.pformat", "indent"): ((2, 4), (3, 0)),
    ("pprint.pformat", "sort_dicts"): (None, (3, 8)),
    ("pprint.pformat", "underscore_numbers"): (None, (3, 10)),
    ("pprint.pformat", "width"): ((2, 4), (3, 0)),
    ("pprint.pprint", "compact"): (None, (3, 4)),
    ("pprint.pprint", "depth"): ((2, 4), (3, 0)),
    ("pprint.pprint", "indent"): ((2, 4), (3, 0)),
    ("pprint.pprint", "sort_dicts"): (None, (3, 8)),
    ("pprint.pprint", "underscore_numbers"): (None, (3, 10)),
    ("pprint.pprint", "width"): ((2, 4), (3, 0)),
    ("print", "flush"): (None, (3, 3)),
    ("py_compile.compile", "invalidation_mode"): (None, (3, 7)),
    ("py_compile.compile", "optimize"): (None, (3, 2)),
    ("py_compile.compile", "quiet"): (None, (3, 8)),
    ("pyclbr.Class", "end_lineno"): (None, (3, 10)),
    ("pyclbr.Class", "parent"): (None, (3, 7)),
    ("pyclbr.Function", "end_lineno"): (None, (3, 10)),
    ("pyclbr.Function", "is_async"): (None, (3, 10)),
    ("pyclbr.Function", "parent"): (None, (3, 7)),
    ("random.sample", "counts"): (None, (3, 9)),
    ("re.findall", "flags"): ((2, 4), (3, 0)),
    ("re.finditer", "flags"): ((2, 4), (3, 0)),
    ("re.split", "flags"): ((2, 7), (3, 1)),
    ("re.sub", "flags"): ((2, 7), (3, 1)),
    ("re.subn", "flags"): ((2, 7), (3, 1)),
    ("sched.scheduler.enter", "kwargs"): (None, (3, 3)),
    ("sched.scheduler.enterabs", "kwargs"): (None, (3, 3)),
    ("sched.scheduler.run", "blocking"): (None, (3, 3)),
    ("select.epoll", "flags"): (None, (3, 3)),
    ("select.kqueue.control", "timeout"): (None, (3, 0)),
    ("shelve.Shelf", "keyencoding"): (None, (3, 2)),
    ("shelve.Shelf", "protocol"): ((2, 3), (3, 0)),
    ("shelve.open", "protocol"): ((2, 3), (3, 0)),
    ("shlex.shlex", "punctuation_chars"): (None, (3, 6)),
    ("shlex.split", "posix"): ((2, 6), (3, 0)),
    ("shutil.chown", "dir_fd"): (None, (3, 13)),
    ("shutil.chown", "follow_symlinks"): (None, (3, 13)),
    ("shutil.copy", "follow_symlinks"): (None, (3, 3)),
    ("shutil.copy2", "follow_symlinks"): (None, (3, 3)),
    ("shutil.copyfile", "follow_symlinks"): (None, (3, 3)),
    ("shutil.copymode", "follow_symlinks"): (None, (3, 3)),
    ("shutil.copystat", "follow_symlinks"): (None, (3, 3)),
    ("shutil.copytree", "copy_function"): (None, (3, 2)),
    ("shutil.copytree", "dirs_exist_ok"): (None, (3, 8)),
    ("shutil.copytree", "ignore"): ((2, 6), (3, 0)),
    ("shutil.copytree", "ignore_dangling_symlinks"): (None, (3, 2)),
    ("shutil.move", "copy_function"): (None, (3, 5)),
    ("shutil.rmtree", "dir_fd"): (None, (3, 11)),
    ("shutil.rmtree", "onexc"): (None, (3, 12)),
    ("shutil.unpack_archive", "filter"): (None, (3, 12)),
    ("signal.set_wakeup_fd", "warn_on_full_buffer"): (None, (3, 7)),
    ("smtpd.SMTPChannel", "decode_data"): (None, (3, 5)),
    ("smtpd.SMTPChannel", "enable_SMTPUTF8"): (None, (3, 5)),
    ("smtpd.SMTPServer", "decode_data"): (None, (3, 5)),
    ("smtpd.SMTPServer", "enable_SMTPUTF8"): (None, (3, 5)),
    ("smtpd.SMTPServer", "map"): (None, (3, 4)),
    ("smtplib.LMTP", "timeout"): (None, (3, 9)),
    ("smtplib.SMTP", "source_address"): (None, (3, 3)),
    ("smtplib.SMTP", "timeout"): ((2, 6), (3, 0)),
    ("smtplib.SMTP.login", "initial_response_ok"): (None, (3, 5)),
    ("smtplib.SMTP.starttls", "context"): (None, (3, 3)),
    ("smtplib.SMTP_SSL", "context"): (None, (3, 3)),
    ("smtplib.SMTP_SSL", "source_address"): (None, (3, 3)),
    ("socket.create_connection", "all_errors"): (None, (3, 11)),
    ("socket.create_connection", "source_address"): ((2, 7), (3, 2)),
    ("socket.setsockopt", "optlen"): (None, (3, 6)),
    ("sorted", "key"): ((2, 4), (3, 0)),
    ("sorted", "reverse"): ((2, 4), (3, 0)),
    ("sqlite3.Connection.create_function", "deterministic"): (None, (3, 8)),
    ("sqlite3.Connection.iterdump", "filter"): (None, (3, 13)),
    ("sqlite3.Connection.load_extension", "entrypoint"): (None, (3, 12)),
    ("sqlite3.connect", "autocommit"): (None, (3, 12)),
    ("sqlite3.connect", "uri"): (None, (3, 4)),
    ("ssl.SSLContext.load_cert_chain", "password"): (None, (3, 3)),
    ("ssl.SSLContext.load_verify_locations", "cadata"): (None, (3, 4)),
    ("ssl.SSLContext.wrap_bio", "session"): (None, (3, 6)),
    ("ssl.SSLContext.wrap_socket", "session"): (None, (3, 6)),
    ("ssl.get_server_certificate", "timeout"): (None, (3, 10)),
    ("ssl.wrap_socket", "ciphers"): ((2, 7), (3, 2)),
    ("statistics.fmean", "weights"): (None, (3, 11)),
    ("statistics.harmonic_mean", "weights"): (None, (3, 10)),
    ("statistics.linear_regression", "proportional"): (None, (3, 11)),
    ("str.decode", "encoding"): ((2, 7), None),
    ("str.decode", "errors"): ((2, 7), None),
    ("str.encode", "encoding"): ((2, 7), (3, 1)),
    ("str.encode", "errors"): ((2, 7), (3, 1)),
    ("str.ljust", "fillchar"): ((2, 4), (3, 0)),
    ("str.lstrip", "chars"): ((2, 2), (3, 0)),
    ("str.replace", "count"): (None, (3, 13)),
    ("str.rjust", "fillchar"): ((2, 4), (3, 0)),
    ("str.rstrip", "chars"): ((2, 2), (3, 0)),
    ("str.strip", "chars"): ((2, 2), (3, 0)),
    ("string.lstrip", "chars"): ((2, 3), None),
    ("string.rstrip", "chars"): ((2, 3), None),
    ("string.strip", "chars"): ((2, 3), None),
    # Notes:
    #  Creating keyword argument rules for the `subprocess` module is tricky. Newly added arguments
    #  may only be documented in `subprocess.Popen`. But the new arguments are actually also
    #  supported by `subprocess.run`, `subprocess.call`, `subprocess.check_call` and
    #  `subprocess.check_output` (we can see `**other_popen_kwargs` in the documentation, the kwargs
    #  are passed through). Therefore, whenever a new argument is added, we need to add rules for
    #  all the 5 functions mentioned above.
    ("subprocess.Popen", "encoding"): (None, (3, 6)),
    ("subprocess.Popen", "errors"): (None, (3, 6)),
    ("subprocess.Popen", "extra_groups"): (None, (3, 9)),
    ("subprocess.Popen", "group"): (None, (3, 9)),
    ("subprocess.Popen", "pass_fds"): (None, (3, 2)),
    ("subprocess.Popen", "pipesize"): (None, (3, 10)),
    ("subprocess.Popen", "process_group"): (None, (3, 11)),
    ("subprocess.Popen", "restore_signals"): (None, (3, 2)),
    ("subprocess.Popen", "start_new_session"): (None, (3, 2)),
    ("subprocess.Popen", "text"): (None, (3, 7)),
    ("subprocess.Popen", "umask"): (None, (3, 9)),
    ("subprocess.Popen", "user"): (None, (3, 9)),
    ("subprocess.Popen.communicate", "timeout"): (None, (3, 3)),
    ("subprocess.Popen.wait", "timeout"): (None, (3, 3)),
    ("subprocess.CalledProcessError", "stderr"): (None, (3, 5)),
    ("subprocess.CalledProcessError", "stdout"): (None, (3, 5)),
    ("subprocess.STARTUPINFO", "lpAttributeList"): (None, (3, 7)),
    ("subprocess.TimeoutExpired", "stderr"): (None, (3, 5)),
    ("subprocess.TimeoutExpired", "stdout"): (None, (3, 5)),
    ("subprocess.call", "encoding"): (None, (3, 6)),
    ("subprocess.call", "errors"): (None, (3, 6)),
    ("subprocess.call", "extra_groups"): (None, (3, 9)),
    ("subprocess.call", "group"): (None, (3, 9)),
    ("subprocess.call", "pass_fds"): (None, (3, 2)),
    ("subprocess.call", "pipesize"): (None, (3, 10)),
    ("subprocess.call", "process_group"): (None, (3, 11)),
    ("subprocess.call", "restore_signals"): (None, (3, 2)),
    ("subprocess.call", "start_new_session"): (None, (3, 2)),
    ("subprocess.call", "text"): (None, (3, 7)),
    ("subprocess.call", "timeout"): (None, (3, 3)),
    ("subprocess.call", "umask"): (None, (3, 9)),
    ("subprocess.call", "user"): (None, (3, 9)),
    ("subprocess.check_call", "encoding"): (None, (3, 6)),
    ("subprocess.check_call", "errors"): (None, (3, 6)),
    ("subprocess.check_call", "extra_groups"): (None, (3, 9)),
    ("subprocess.check_call", "group"): (None, (3, 9)),
    ("subprocess.check_call", "pass_fds"): (None, (3, 2)),
    ("subprocess.check_call", "pipesize"): (None, (3, 10)),
    ("subprocess.check_call", "process_group"): (None, (3, 11)),
    ("subprocess.check_call", "restore_signals"): (None, (3, 2)),
    ("subprocess.check_call", "start_new_session"): (None, (3, 2)),
    ("subprocess.check_call", "text"): (None, (3, 7)),
    ("subprocess.check_call", "timeout"): (None, (3, 3)),
    ("subprocess.check_call", "umask"): (None, (3, 9)),
    ("subprocess.check_call", "user"): (None, (3, 9)),
    ("subprocess.check_output", "encoding"): (None, (3, 6)),
    ("subprocess.check_output", "errors"): (None, (3, 6)),
    ("subprocess.check_output", "extra_groups"): (None, (3, 9)),
    ("subprocess.check_output", "group"): (None, (3, 9)),
    ("subprocess.check_output", "input"): (None, (3, 4)),
    ("subprocess.check_output", "pass_fds"): (None, (3, 2)),
    ("subprocess.check_output", "pipesize"): (None, (3, 10)),
    ("subprocess.check_output", "process_group"): (None, (3, 11)),
    ("subprocess.check_output", "restore_signals"): (None, (3, 2)),
    ("subprocess.check_output", "start_new_session"): (None, (3, 2)),
    ("subprocess.check_output", "text"): (None, (3, 7)),
    ("subprocess.check_output", "timeout"): (None, (3, 3)),
    ("subprocess.check_output", "umask"): (None, (3, 9)),
    ("subprocess.check_output", "user"): (None, (3, 9)),
    ("subprocess.getoutput", "encoding"): (None, (3, 11)),
    ("subprocess.getoutput", "errors"): (None, (3, 11)),
    ("subprocess.getstatusoutput", "encoding"): (None, (3, 11)),
    ("subprocess.getstatusoutput", "errors"): (None, (3, 11)),
    ("subprocess.run", "capture_output"): (None, (3, 7)),
    ("subprocess.run", "encoding"): (None, (3, 6)),
    ("subprocess.run", "errors"): (None, (3, 6)),
    ("subprocess.run", "extra_groups"): (None, (3, 9)),
    ("subprocess.run", "group"): (None, (3, 9)),
    ("subprocess.run", "pipesize"): (None, (3, 10)),
    ("subprocess.run", "text"): (None, (3, 7)),
    ("subprocess.run", "umask"): (None, (3, 9)),
    ("subprocess.run", "user"): (None, (3, 9)),
    ("sum", "start"): (None, (3, 8)),
    ("sys.getwindowsversion", "platform_version"): (None, (3, 6)),
    ("tarfile.TarFile", "encoding"): ((2, 6), (3, 0)),
    ("tarfile.TarFile", "errors"): ((2, 6), (3, 0)),
    ("tarfile.TarFile", "format"): ((2, 6), (3, 0)),
    ("tarfile.TarFile", "pax_headers"): ((2, 6), (3, 0)),
    ("tarfile.TarFile", "stream"): (None, (3, 13)),
    ("tarfile.TarFile", "tarinfo"): ((2, 6), (3, 0)),
    ("tarfile.TarFile.add", "exclude"): ((2, 6), (3, 0)),
    ("tarfile.TarFile.add", "filter"): ((2, 7), (3, 2)),
    ("tarfile.TarFile.extract", "filter"): (None, (3, 12)),
    ("tarfile.TarFile.extract", "numeric_owner"): (None, (3, 5)),
    ("tarfile.TarFile.extract", "set_attrs"): (None, (3, 2)),
    ("tarfile.TarFile.extractall", "filter"): (None, (3, 12)),
    ("tarfile.TarFile.extractall", "numeric_owner"): (None, (3, 5)),
    ("tarfile.TarFile.list", "members"): (None, (3, 5)),
    ("tarfile.TarInfo.tobuf", "encoding"): ((2, 6), (3, 0)),
    ("tarfile.TarInfo.tobuf", "errors"): ((2, 6), (3, 0)),
    ("tarfile.TarInfo.tobuf", "format"): ((2, 6), (3, 0)),
    ("telnetlib.Telnet", "timeout"): ((2, 6), (3, 0)),
    ("telnetlib.Telnet.open", "timeout"): ((2, 6), (3, 0)),
    ("tempfile.NamedTemporaryFile", "delete"): ((2, 6), (3, 0)),
    ("tempfile.NamedTemporaryFile", "errors"): (None, (3, 8)),
    ("tempfile.NamedTemporaryFile", "delete_on_close"): (None, (3, 12)),
    ("tempfile.SpooledTemporaryFile", "errors"): (None, (3, 8)),
    ("tempfile.SpooledTemporaryFile.truncate", "size"): (None, (3, 3)),
    ("tempfile.TemporaryDirectory", "delete"): (None, (3, 12)),
    ("tempfile.TemporaryDirectory", "ignore_cleanup_errors"): (None, (3, 10)),
    ("tempfile.TemporaryFile", "errors"): (None, (3, 8)),
    ("test.support.check_warnings", "quiet"): ((2, 7), (3, 2)),
    ("threading.Lock.acquire", "blocking"): (None, (3, 0)),
    ("threading.Lock.acquire", "timeout"): (None, (3, 2)),
    ("threading.RLock.acquire", "blocking"): (None, (3, 0)),
    ("threading.RLock.acquire", "timeout"): (None, (3, 2)),
    ("threading.Semaphore.acquire", "timeout"): (None, (3, 2)),
    ("threading.Semaphore.release", "n"): (None, (3, 9)),
    ("threading.Thread", "context"): (None, (3, 14)),
    ("threading.Thread", "daemon"): (None, (3, 3)),
    ("timeit.Timer", "globals"): (None, (3, 5)),
    ("timeit.repeat", "globals"): (None, (3, 5)),
    ("timeit.timeit", "globals"): (None, (3, 5)),
    ("tkinter.font.nametofont", "root"): (None, (3, 10)),
    ("traceback.TracebackException", "compact"): (None, (3, 10)),
    ("traceback.TracebackException.format_exception_only", "show_group"): (None, (3, 13)),
    ("tracemalloc.Filter", "domain"): (None, (3, 6)),
    ("typing.get_type_hints", "include_extras"): (None, (3, 9)),
    ("unittest.TestCase.assertAlmostEqual", "delta"): ((2, 7), (3, 2)),
    ("unittest.TestCase.assertNotAlmostEqual", "delta"): ((2, 7), (3, 2)),
    ("unittest.TestCase.assertRaises", "msg"): (None, (3, 3)),
    ("unittest.TestCase.assertRaisesRegex", "msg"): (None, (3, 3)),
    ("unittest.TestCase.assertWarns", "msg"): (None, (3, 3)),
    ("unittest.TestCase.assertWarnsRegex", "msg"): (None, (3, 3)),
    ("unittest.TestLoader.loadTestsFromModule", "pattern"): (None, (3, 5)),
    ("unittest.TextTestRunner", "tb_locals"): (None, (3, 5)),
    ("unittest.TextTestRunner", "warnings"): (None, (3, 2)),
    ("unittest.main", "buffer"): ((2, 7), (3, 2)),
    ("unittest.main", "catchbreak"): ((2, 7), (3, 2)),
    ("unittest.main", "exit"): ((2, 7), (3, 1)),
    ("unittest.main", "failfast"): ((2, 7), (3, 2)),
    ("unittest.main", "verbosity"): ((2, 7), (3, 2)),
    ("unittest.main", "warnings"): (None, (3, 2)),
    ("unittest.mock.Mock", "unsafe"): (None, (3, 5)),
    ("unittest.mock.Mock.reset_mock", "return_value"): (None, (3, 6)),
    ("unittest.mock.Mock.reset_mock", "side_effect"): (None, (3, 6)),
    ("urllib.URLopener", "context"): ((2, 7), None),
    ("urllib.parse.parse_qs", "encoding"): (None, (3, 2)),
    ("urllib.parse.parse_qs", "errors"): (None, (3, 2)),
    # Security fix backported to 3.6.8+, 3.7.2+, 3.8+
    ("urllib.parse.parse_qs", "max_num_fields"): (None, (3, 6)),
    # Security fix backported to 3.6.13+, 3.7.10+, 3.8.8+, 3.9.2+, 3.10+
    ("urllib.parse.parse_qs", "separator"): (None, (3, 6)),
    ("urllib.parse.parse_qsl", "encoding"): (None, (3, 2)),
    ("urllib.parse.parse_qsl", "errors"): (None, (3, 2)),
    # Security fix backported to 3.6.8+, 3.7.2+, 3.8+
    ("urllib.parse.parse_qsl", "max_num_fields"): (None, (3, 6)),
    # Security fix backported to 3.6.13+, 3.7.10+, 3.8.8+, 3.9.2+, 3.10+
    ("urllib.parse.parse_qsl", "separator"): (None, (3, 6)),
    ("urllib.parse.urlencode", "quote_via"): (None, (3, 5)),
    ("urllib.request.HTTPSHandler", "check_hostname"): (None, (3, 2)),
    ("urllib.request.HTTPSHandler", "context"): (None, (3, 2)),
    ("urllib.request.Request", "method"): (None, (3, 3)),
    ("urllib.request.urlopen", "cadefault"): (None, (3, 3)),
    ("urllib.request.urlopen", "cafile"): (None, (3, 2)),
    ("urllib.request.urlopen", "capath"): (None, (3, 2)),
    ("urllib.request.urlopen", "context"): (None, (3, 4)),
    ("urllib.request.pathname2url", "add_scheme"): (None, (3, 14)),
    ("urllib.request.url2pathname", "require_scheme"): (None, (3, 14)),
    ("urllib.request.url2pathname", "resolve_host"): (None, (3, 14)),
    ("urllib.urlopen", "context"): ((2, 7), None),
    ("urllib.urlopen", "proxies"): ((2, 3), None),
    ("urllib.urlretrieve", "context"): ((2, 7), None),
    ("urllib2.HTTPSHandler", "context"): ((2, 7), None),
    ("urllib2.OpenerDirector.open", "timeout"): ((2, 6), None),
    ("urllib2.urlopen", "cadefault"): ((2, 7), None),
    ("urllib2.urlopen", "cafile"): ((2, 7), None),
    ("urllib2.urlopen", "capth"): ((2, 7), None),
    ("urllib2.urlopen", "context"): ((2, 7), None),
    ("urllib2.urlopen", "timeout"): ((2, 6), None),
    ("urlparse.parse_qs", "max_num_fields"): ((2, 7), None),
    ("urlparse.parse_qsl", "max_num_fields"): ((2, 7), None),
    ("uu.encode", "backtick"): (None, (3, 7)),
    ("venv.EnvBuilder", "prompt"): (None, (3, 6)),
    ("venv.EnvBuilder", "scm_ignore_files"): (None, (3, 13)),
    ("venv.EnvBuilder", "upgrade_deps"): (None, (3, 9)),
    ("venv.EnvBuilder", "with_pip"): (None, (3, 4)),
    ("venv.create", "prompt"): (None, (3, 6)),
    ("venv.create", "scm_ignore_files"): (None, (3, 13)),
    ("venv.create", "upgrade_deps"): (None, (3, 9)),
    ("venv.create", "with_pip"): (None, (3, 4)),
    ("warnings.catch_warnings", "action"): (None, (3, 11)),
    ("warnings.catch_warnings", "append"): (None, (3, 11)),
    ("warnings.catch_warnings", "category"): (None, (3, 11)),
    ("warnings.catch_warnings", "lineno"): (None, (3, 11)),
    ("warnings.formatwarning", "line"): ((2, 6), (3, 0)),
    ("warnings.showwarning", "line"): ((2, 7), (3, 0)),
    ("warnings.warn", "source"): (None, (3, 6)),
    ("warnings.warn_explicit", "module_globals"): ((2, 5), (3, 0)),
    ("warnings.warn_explicit", "source"): (None, (3, 6)),
    ("webbrowser.register", "preferred"): (None, (3, 7)),
    ("winreg.OpenKey", "access"): (None, (3, 2)),
    ("winreg.OpenKey", "key"): (None, (3, 2)),
    ("winreg.OpenKey", "reserved"): (None, (3, 2)),
    ("winreg.OpenKey", "sub_key"): (None, (3, 2)),
    ("winreg.OpenKeyEx", "access"): (None, (3, 2)),
    ("winreg.OpenKeyEx", "key"): (None, (3, 2)),
    ("winreg.OpenKeyEx", "reserved"): (None, (3, 2)),
    ("winreg.OpenKeyEx", "sub_key"): (None, (3, 2)),
    ("xml.dom.minidom.Document.writexml", "encoding"): ((2, 3), (3, 0)),
    ("xml.dom.minidom.Node.toprettyxml", "encoding"): ((2, 3), (3, 0)),
    ("xml.dom.minidom.Node.toprettyxml", "standalone"): (None, (3, 9)),
    ("xml.dom.minidom.Node.toxml", "encoding"): ((2, 3), (3, 0)),
    ("xml.dom.minidom.Node.toxml", "standalone"): (None, (3, 9)),
    ("xml.dom.minidom.Node.writexml", "addindent"): ((2, 1), (3, 0)),
    ("xml.dom.minidom.Node.writexml", "encoding"): (None, (3, 0)),
    ("xml.dom.minidom.Node.writexml", "indent"): ((2, 1), (3, 0)),
    ("xml.dom.minidom.Node.writexml", "newl"): ((2, 1), (3, 0)),
    ("xml.dom.minidom.Node.writexml", "standalone"): (None, (3, 9)),
    ("xml.etree.ElementInclude.include", "base_url"): (None, (3, 9)),
    ("xml.etree.ElementInclude.include", "max_depth"): (None, (3, 9)),
    ("xml.etree.ElementTree.ElementTree.write", "short_empty_elements"): (None, (3, 4)),
    ("xml.etree.ElementTree.tostring", "default_namespace"): (None, (3, 8)),
    ("xml.etree.ElementTree.tostring", "short_empty_elements"): (None, (3, 4)),
    ("xml.etree.ElementTree.tostring", "xml_declaration"): (None, (3, 8)),
    ("xml.etree.ElementTree.tostringlist", "default_namespace"): (None, (3, 8)),
    ("xml.etree.ElementTree.tostringlist", "short_empty_elements"): (None, (3, 4)),
    ("xml.etree.ElementTree.tostringlist", "xml_declaration"): (None, (3, 8)),
    ("xml.etree.ElementTree.write", "short_empty_elements"): (None, (3, 4)),
    ("xml.sax.saxutils.XMLGenerator", "short_empty_elements"): (None, (3, 2)),
    ("xmlrpc.client.ServerProxy", "context"): (None, (3, 4)),
    ("xmlrpc.client.ServerProxy", "headers"): (None, (3, 8)),
    ("xmlrpc.client.ServerProxy", "use_builtin_types"): (None, (3, 3)),
    ("xmlrpc.client.loads", "use_builtin_types"): (None, (3, 3)),
    ("xmlrpc.server.CGIXMLRPCRequestHandler", "use_builtin_types"): (None, (3, 3)),
    ("xmlrpc.server.DocXMLRPCServer", "use_builtin_types"): (None, (3, 3)),
    ("xmlrpc.server.SimpleXMLRPCServer", "use_builtin_types"): (None, (3, 3)),
    ("xmlrpclib.ServerProxy", "context"): ((2, 7), None),
    ("xmlrpclib.ServerProxy", "use_datetime"): ((2, 5), None),
    ("xmlrpclib.loads", "use_datetime"): ((2, 5), None),
    ("zip", "strict"): (None, (3, 10)),
    ("zipapp.create_archive", "compressed"): (None, (3, 7)),
    ("zipapp.create_archive", "filter"): (None, (3, 7)),
    ("zipfile.PyZipFile", "optimize"): (None, (3, 2)),
    ("zipfile.PyZipFile.writepy", "filterfunc"): (None, (3, 4)),
    ("zipfile.ZipFile", "compresslevel"): (None, (3, 7)),
    ("zipfile.ZipFile", "metadata_encoding"): (None, (3, 11)),
    ("zipfile.ZipFile", "strict_timestamps"): (None, (3, 8)),
    ("zipfile.ZipFile.read", "pwd"): ((2, 6), (3, 0)),
    ("zipfile.ZipFile.writestr", "compress_type"): ((2, 7), (3, 2)),
    ("zipfile.ZipInfo.from_file", "strict_timestamps"): (None, (3, 8)),
    ("zlib.Decompress.decompress", "max_length"): (None, (3, 6)),
    ("zlib.compress", "level"): (None, (3, 6)),
    ("zlib.compress", "wbits"): (None, (3, 11)),
    ("zlib.compressobj", "zdict"): (None, (3, 3)),
    ("zlib.decompress", "bufsize"): (None, (3, 6)),
    ("zlib.decompress", "wbits"): (None, (3, 6)),
    ("zlib.decompressobj", "zdict"): (None, (3, 3)),
   }

# datetime+time strftime/strptime requirements: directive -> requirements
STRFTIME_REQS = {
  "G": (None, (3, 6)),
  "V": (None, (3, 6)),
  "f": ((2, 6), (3, 0)),
  "u": (None, (3, 6)),
}

# bytes/bytearray (and str for 2 compatibility) requirements: directive -> requirements
BYTES_REQS = {
  "a": (None, (3, 5)),
  "b": (None, (3, 5)),
  "r": ((2, 7), (3, 5)),
}

# array.array typecode requirements: typecode -> requirements
ARRAY_TYPECODE_REQS = {
  "q": (None, (3, 3)),
  "Q": (None, (3, 3)),
  "w": (None, (3, 13)),
}

# codecs error handler requirements: error name -> requirements
CODECS_ERROR_HANDLERS = {
  "surrogateescape": (None, (3, 1)),
  "surrogatepass": (None, (3, 1)),
  "namereplace": (None, (3, 5)),
}

# Correlate the value of kwarg "errors" of the following functions/constructors with
# CODECS_ERROR_HANDLERS: function -> argument index
CODECS_ERRORS_INDICES = {
  "codecs.encode": 2,
  "codecs.decode": 2,
  "codecs.open": 3,
  "codecs.EncodedFile": 3,
  "codecs.iterencode": 2,
  "codecs.iterdecode": 2,
  "codecs.IncrementalEncoder": 0,
  "codecs.IncrementalDecoder": 0,
  "codecs.StreamWriter": 1,
  "codecs.StreamReader": 1,
  "codecs.StreamReaderWriter": 3,
  "codecs.StreamRecorder": 5,
}

# codecs encodings and aliases -> requirements
# Checks case-insensitively.
CODECS_ENCODINGS = {
  # German
  ("cp273", "273", "ibm273", "csibm273"): (None, (3, 4)),

  # Ukrainian
  ("cp1125", "1125", "ibm1125", "cp866u", "ruscii"): (None, (3, 4)),

  # Windows UTF-8 (CP_UTF8)
  ("cp65001",): (None, (3, 3)),

  # Tajik
  ("koi8_t",): (None, (3, 5)),

  # Kazakh
  ("kz1048", "kz_1048", "strk1048_2002", "kz-1048", "strk1048-2002", "rk1048"): (None, (3, 5)),

  # Windows OEM
  ("oem",): (None, (3, 6)),

  # == Binary transforms ==

  ("base64_codec", "base64-codec"): (None, (3, 2)),
  ("base64", "base_64", "base-64"): (None, (3, 4)),

  ("bz2_codec", "bz2-codec"): (None, (3, 2)),
  ("bz2"): (None, (3, 4)),

  ("hex_codec", "hex-codec"): (None, (3, 2)),
  ("hex"): (None, (3, 4)),

  ("quopri_codec", "quopri-codec"): (None, (3, 2)),
  ("quopri", "quotedprintable", "quoted_printable", "quoted-printable"): (None, (3, 4)),

  ("uu_codec", "uu-codec"): (None, (3, 2)),
  ("uu"): (None, (3, 4)),

  ("zip_codec", "zip-codec"): (None, (3, 2)),
  ("zip", "zlib"): (None, (3, 4)),

  # == Text transforms ==

  # Caesar cipher
  ("rot_13", "rot-13"): (None, (3, 2)),
  ("rot13"): (None, (3, 4)),
}

# Correlate the value of kwarg "encoding", "file_encoding", "data_encoding" of the following
# functions with CODECS_ENCODINGS: function -> set of argument indices
CODECS_ENCODINGS_INDICES = {
  "codecs.encode": (1,),
  "codecs.decode": (1,),
  "codecs.lookup": (0,),
  "codecs.getencoder": (0,),
  "codecs.getdecoder": (0,),
  "codecs.getincrementalencoder": (0,),
  "codecs.getincrementaldecoder": (0,),
  "codecs.getreader": (0,),
  "codecs.getwriter": (0,),
  "codecs.open": (2,),
  "codecs.EncodedFile": (1, 2),
  "codecs.iterencode": (2,),
  "codecs.iterdecode": (2,),
}

# Builtin types that work as annotation types from 3.9+
# The list is taken from https://github.com/python/cpython/blob/master/Lib/test/test_genericalias.py
BUILTIN_GENERIC_ANNOTATION_TYPES = (
  "collections.ChainMap",
  "collections.Counter",
  "collections.OrderedDict",
  "collections.UserDict",
  "collections.UserList",
  "collections.abc.AsyncGenerator",
  "collections.abc.AsyncIterable",
  "collections.abc.AsyncIterator",
  "collections.abc.Awaitable",
  "collections.abc.ByteString",
  "collections.abc.Callable",
  "collections.abc.Collection",
  "collections.abc.Container",
  "collections.abc.Coroutine",
  "collections.abc.Generator",
  "collections.abc.ItemsView",
  "collections.abc.Iterable",
  "collections.abc.Iterator",
  "collections.abc.KeysView",
  "collections.abc.Mapping",
  "collections.abc.MappingView",
  "collections.abc.MutableMapping",
  "collections.abc.MutableSequence",
  "collections.abc.MutableSet",
  "collections.abc.Reversible",
  "collections.abc.Sequence",
  "collections.abc.Set",
  "collections.abc.ValuesView",
  "collections.defaultdict",
  "collections.deque",
  "concurrent.futures.Future",
  "concurrent.futures.thread._WorkItem",
  "contextlib.AbstractAsyncContextManager",
  "contextlib.AbstractContextManager",
  "contextvars.ContextVar",
  "contextvars.Token",
  "ctypes.Array",
  "ctypes.LibraryLoader",
  "dataclasses.Field",
  "dict",
  "difflib.SequenceMatcher",
  "enumerate",
  "filecmp.dircmp",
  "fileinput.FileInput",
  "frozenset",
  "functools.cached_property",
  "functools.partial",
  "functools.partialmethod",
  "graphlib.TopologicalSorter",
  "http.cookies.Morsel",
  "itertools.chain",
  "list",
  "logging.LoggerAdapter",
  "logging.StreamHandler",
  "mailbox.Mailbox",
  "mailbox._PartialFile",
  "multiprocessing.managers.ValueProxy",
  "multiprocessing.pool.ApplyResult",
  "multiprocessing.queues.SimpleQueue",
  "multiprocessing.shared_memory.ShareableList",
  "os.DirEntry",
  "queue.Queue",
  "queue.SimpleQueue",
  "re.Match",
  "re.Pattern",
  "set",
  "tempfile.SpooledTemporaryFile",
  "tempfile.TemporaryDirectory",
  "tuple",
  "type",
  "types.AsyncGeneratorType",
  "types.MappingProxyType",
  "unittest.case._AssertRaisesContext",
  "urllib.parse.ParseResult",
  "urllib.parse.SplitResult",
  "weakref.ReferenceType",
  "weakref.WeakSet",
  "weakref.ref",
)

# Types that support dictionary union operator | (3.9).
DICT_UNION_SUPPORTED_TYPES = (
  "collections.ChainMap",
  "collections.OrderedDict",
  "collections.UserDict",
  "collections.defaultdict",
  "os.environ",
  "os.environb",
  "types.MappingProxyType",
  "weakref.WeakKeyDictionary",
  "weakref.WeakValueDictionary",
)

# Types that support dictionary union merge operator |= (3.9).
DICT_UNION_MERGE_SUPPORTED_TYPES = (
  "collections.ChainMap",
  "collections.OrderedDict",
  "collections.UserDict",
  "collections.defaultdict",
  "os.environ",
  "os.environb",
  "weakref.WeakKeyDictionary",
  "weakref.WeakValueDictionary",
)

# Certain decorators allow being used not as function calls, where they instead apply directly to
# the "user function".
# Ref: https://docs.python.org/3/library/functools.html#functools.lru_cache
DECORATOR_USER_FUNCTIONS = {
  "functools.lru_cache": (None, (3, 8)),
  "functools.cache": (None, (3, 9)),
}
