"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardSetMembershipRevision = exports.SetMembershipChangeType = exports.SetMembershipKind = void 0;
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
const MembershipChange_1 = require("./MembershipChange");
const immutable_1 = require("immutable");
var SetMembershipKind;
(function (SetMembershipKind) {
    // incorporates knock, join, invite
    SetMembershipKind["Present"] = "present";
    // incorporates leave, ban, and never present
    SetMembershipKind["Absent"] = "absent";
})(SetMembershipKind || (exports.SetMembershipKind = SetMembershipKind = {}));
var SetMembershipChangeType;
(function (SetMembershipChangeType) {
    SetMembershipChangeType["BecamePresent"] = "became_present";
    SetMembershipChangeType["BecameAbsent"] = "became_absent";
    SetMembershipChangeType["NoOverallChange"] = "no_overall_change";
})(SetMembershipChangeType || (exports.SetMembershipChangeType = SetMembershipChangeType = {}));
class StandardSetMembershipRevision {
    constructor(memberships, internedRooms) {
        this.memberships = memberships;
        this.internedRooms = internedRooms;
        // nothing to do.
    }
    getMembershipCount(userID) {
        return this.memberships.get(userID, 0);
    }
    changesFromMembershipChanges(membershipChanges) {
        if (!membershipChanges.every((change) => this.internedRooms.has(change.roomID))) {
            throw new TypeError('Cannot revise from changes that do not all belong to the same room set.');
        }
        const changes = new Map();
        for (const membershipChange of membershipChanges) {
            const userID = membershipChange.userID;
            const changeType = membershipChange.membershipChangeType;
            const existingEntry = changes.get(userID);
            const change = existingEntry === undefined
                ? ((template) => (changes.set(userID, template), template))({
                    userID,
                    changeType: SetMembershipChangeType.NoOverallChange,
                    roomsJoined: 0,
                    roomsLeft: 0,
                })
                : existingEntry;
            switch (changeType) {
                case MembershipChange_1.MembershipChangeType.Joined:
                case MembershipChange_1.MembershipChangeType.Rejoined:
                case MembershipChange_1.MembershipChangeType.Invited:
                case MembershipChange_1.MembershipChangeType.Knocked:
                case MembershipChange_1.MembershipChangeType.Reknocked:
                    change.roomsJoined += 1;
                    break;
                case MembershipChange_1.MembershipChangeType.Left:
                case MembershipChange_1.MembershipChangeType.Kicked:
                case MembershipChange_1.MembershipChangeType.Banned:
                    change.roomsLeft += 1;
                    break;
            }
            const oldCount = this.getMembershipCount(userID);
            const newCount = oldCount + change.roomsJoined - change.roomsLeft;
            if (oldCount > 0) {
                if (newCount === 0) {
                    change.changeType = SetMembershipChangeType.BecameAbsent;
                }
            }
            else {
                if (newCount > 0) {
                    change.changeType = SetMembershipChangeType.BecamePresent;
                }
            }
        }
        return {
            addedRoom: undefined,
            removedRoom: undefined,
            changes: Array.from(changes.values()),
        };
    }
    changesFromAddedRoom(roomMembershipRevision) {
        if (this.internedRooms.has(roomMembershipRevision.room.toRoomIDOrAlias())) {
            throw new TypeError('Cannot revise from a room that is already in the room set.');
        }
        const changes = [];
        for (const member of roomMembershipRevision.members()) {
            const existingCount = this.getMembershipCount(member.userID);
            switch (member.membership) {
                case MembershipChange_1.Membership.Join:
                case MembershipChange_1.Membership.Invite:
                case MembershipChange_1.Membership.Knock:
                    changes.push({
                        userID: member.userID,
                        changeType: existingCount === 0
                            ? SetMembershipChangeType.BecamePresent
                            : SetMembershipChangeType.NoOverallChange,
                        roomsJoined: 1,
                        roomsLeft: 0,
                    });
                    break;
            }
        }
        return {
            addedRoom: roomMembershipRevision.room.toRoomIDOrAlias(),
            removedRoom: undefined,
            changes,
        };
    }
    changesFromRemovedRoom(roomMembershipRevision) {
        const changes = [];
        for (const member of roomMembershipRevision.members()) {
            const existingCount = this.getMembershipCount(member.userID);
            switch (member.membership) {
                case MembershipChange_1.Membership.Join:
                case MembershipChange_1.Membership.Invite:
                case MembershipChange_1.Membership.Knock: {
                    changes.push({
                        userID: member.userID,
                        changeType: existingCount === 1
                            ? SetMembershipChangeType.BecameAbsent
                            : SetMembershipChangeType.NoOverallChange,
                        roomsJoined: 0,
                        roomsLeft: 1,
                    });
                    break;
                }
            }
        }
        return {
            removedRoom: roomMembershipRevision.room.toRoomIDOrAlias(),
            addedRoom: undefined,
            changes,
        };
    }
    reviseFromChanges(delta) {
        let internedRooms = this.internedRooms;
        if (delta.addedRoom !== undefined) {
            if (internedRooms.has(delta.addedRoom)) {
                throw new TypeError('Cannot revise from a room that is already in the room set.');
            }
            internedRooms = internedRooms.add(delta.addedRoom);
        }
        if (delta.removedRoom !== undefined) {
            if (!internedRooms.has(delta.removedRoom)) {
                throw new TypeError('Cannot revise from a room that is not in the room set.');
            }
            internedRooms = internedRooms.remove(delta.removedRoom);
        }
        let memberships = this.memberships;
        for (const change of delta.changes) {
            const oldCount = memberships.get(change.userID, 0);
            const newCount = oldCount + change.roomsJoined - change.roomsLeft;
            if (newCount === 0) {
                memberships = memberships.delete(change.userID);
            }
            else {
                memberships = memberships.set(change.userID, newCount);
            }
        }
        return new StandardSetMembershipRevision(memberships, internedRooms);
    }
    membershipForUser(userID) {
        return this.getMembershipCount(userID) > 0
            ? SetMembershipKind.Present
            : SetMembershipKind.Absent;
    }
    presentMembers() {
        return this.memberships.keys();
    }
    uniqueMemberCount() {
        return this.memberships.size;
    }
    static blankRevision() {
        return new StandardSetMembershipRevision((0, immutable_1.Map)(), (0, immutable_1.Set)());
    }
}
exports.StandardSetMembershipRevision = StandardSetMembershipRevision;
//# sourceMappingURL=SetMembershipRevision.js.map