"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardRoomMembershipRevision = void 0;
const MembershipChange_1 = require("./MembershipChange");
const immutable_1 = require("immutable");
const Logger_1 = require("../Logging/Logger");
const SafeMembershipEvent_1 = require("../SafeMatrixEvents/SafeMembershipEvent");
const log = new Logger_1.Logger('StandardRoomMembershipRevision');
class StandardRoomMembershipRevision {
    constructor(room, membershipByUserID, membershipByEventID, membershipByMembership) {
        this.room = room;
        this.membershipByUserID = membershipByUserID;
        this.membershipByEventID = membershipByEventID;
        this.membershipByMembership = membershipByMembership;
        // nothing to do.
    }
    static blankRevision(room) {
        return new StandardRoomMembershipRevision(room, (0, immutable_1.Map)(), (0, immutable_1.Map)(), (0, immutable_1.Map)());
    }
    members() {
        return this.membershipByEventID.values();
    }
    membersOfMembership(membership) {
        return this.membershipByMembership
            .get(membership, (0, immutable_1.Set)())
            .values();
    }
    hasEvent(eventID) {
        return this.membershipByEventID.has(eventID);
    }
    membershipForUser(userID) {
        return this.membershipByUserID.get(userID);
    }
    changesFromMembership(membershipEvents) {
        var _a;
        const changes = [];
        for (const event of membershipEvents) {
            if (this.hasEvent(event.event_id)) {
                continue;
            }
            // There is a distinguishment between our previous event
            // and the server's claim for prev_content.
            const localPreviousEvent = this.membershipForUser(event.state_key);
            // interestingly, if the parser for MembershipEvent eagerly parsed
            // previous_content and there was an error in the previous_content,
            // but not the top level. Then there would be a very bad situation.
            // So we need SafeMembershipEventMirror that can parse unsigned for us
            // in the same way. Perhaps there needs to be a generic SafeMatrixEvent
            // utility to use as a base though.
            const citedPreviousMembership = ((_a = event.unsigned) === null || _a === void 0 ? void 0 : _a.prev_content) === undefined
                ? undefined
                : event.unsigned.prev_content === null
                    ? undefined
                    : SafeMembershipEvent_1.SafeMembershipEventMirror.parse(event.unsigned.prev_content).match((ok) => ok, (error) => {
                        var _a;
                        log.error(`Unable to decode previous membership for ${event.state_key} within ${this.room.toPermalink()}. This is a serious error and the developers should be notified.`, JSON.stringify((_a = event.unsigned) === null || _a === void 0 ? void 0 : _a.prev_content), error);
                        return undefined;
                    });
            const membershipChange = (0, MembershipChange_1.membershipChangeType)(event, localPreviousEvent !== null && localPreviousEvent !== void 0 ? localPreviousEvent : citedPreviousMembership);
            const profileChange = (0, MembershipChange_1.profileChangeType)(event, localPreviousEvent !== null && localPreviousEvent !== void 0 ? localPreviousEvent : citedPreviousMembership);
            changes.push(new MembershipChange_1.MembershipChange(event.state_key, event.sender, event.room_id, event.event_id, event.content.membership, membershipChange, profileChange, event.content));
        }
        return changes;
    }
    reviseFromChanges(changes) {
        let nextMembershipByUserID = this.membershipByUserID;
        let nextMembershipByEventID = this.membershipByEventID;
        let nextMembershipByMembership = this.membershipByMembership;
        for (const change of changes) {
            nextMembershipByUserID = nextMembershipByUserID.set(change.userID, change);
            const existingMembership = this.membershipForUser(change.userID);
            if (existingMembership !== undefined) {
                nextMembershipByEventID = nextMembershipByEventID.delete(existingMembership.eventID);
            }
            nextMembershipByEventID = nextMembershipByEventID.set(change.eventID, change);
            if (existingMembership) {
                nextMembershipByMembership = nextMembershipByMembership.set(existingMembership.membership, nextMembershipByMembership
                    .get(existingMembership.membership, (0, immutable_1.Set)())
                    .delete(existingMembership));
            }
            nextMembershipByMembership = nextMembershipByMembership.set(change.membership, nextMembershipByMembership
                .get(change.membership, (0, immutable_1.Set)())
                .add(change));
        }
        return new StandardRoomMembershipRevision(this.room, nextMembershipByUserID, nextMembershipByEventID, nextMembershipByMembership);
    }
    reviseFromMembership(membershipEvents) {
        const changes = this.changesFromMembership(membershipEvents);
        return this.reviseFromChanges(changes);
    }
}
exports.StandardRoomMembershipRevision = StandardRoomMembershipRevision;
//# sourceMappingURL=StandardRoomMembershipRevision.js.map