"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.BlockInvitationsOnServerProtection = exports.SynapseHTTPUserMayInvite = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const crypto_1 = require("crypto");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const typescript_result_1 = require("@gnuxie/typescript-result");
const log = new matrix_protection_suite_1.Logger("SynapseHTTPUserMayInvite");
class SynapseHTTPUserMayInvite {
    constructor(watchedPolicyRooms, automaticallyRedactForReasons, synapseHTTPAntispam) {
        this.watchedPolicyRooms = watchedPolicyRooms;
        this.synapseHTTPAntispam = synapseHTTPAntispam;
        this.synapseHTTPCallback = function ({ inviter: sender, room_id }) {
            const userHash = (0, crypto_1.createHash)("sha256")
                .update(sender, "utf8")
                .digest("base64");
            const serverHash = (0, crypto_1.createHash)("sha256")
                .update((0, matrix_basic_types_1.userServerName)(sender), "utf8")
                .digest("base64");
            // We only want to block the invitation with user policies with recommendation Ban
            // when they match the automaticRedactReasons. This is so they can still appeal
            // being banned in COC use cases.
            // Server policies are fine to be used to derive the block.
            const matchingUserPolicy = [
                ...this.watchedPolicyRooms.currentRevision.allRulesMatchingEntity(sender, {}),
                ...this.watchedPolicyRooms.currentRevision.findRulesMatchingHash(userHash, "sha256", { type: matrix_protection_suite_1.PolicyRuleType.User }),
            ].find((policy) => policy.recommendation === matrix_protection_suite_1.Recommendation.Takedown ||
                (policy.recommendation === matrix_protection_suite_1.Recommendation.Ban &&
                    this.automaticRedactionReasons.some((reason) => reason.test(policy.reason ?? "<no reason supplied>"))));
            const matchingServerOrRoomPolicy = [
                ...this.watchedPolicyRooms.currentRevision.allRulesMatchingEntity((0, matrix_basic_types_1.userServerName)(sender), {}),
                ...this.watchedPolicyRooms.currentRevision.findRulesMatchingHash(serverHash, "sha256", { type: matrix_protection_suite_1.PolicyRuleType.Server }),
                ...this.watchedPolicyRooms.currentRevision.allRulesMatchingEntity(room_id, {}),
            ].find((policy) => policy.recommendation === matrix_protection_suite_1.Recommendation.Takedown ||
                policy.recommendation === matrix_protection_suite_1.Recommendation.Ban);
            if (matchingUserPolicy !== undefined ||
                matchingServerOrRoomPolicy !== undefined) {
                log.debug(`Blocking an invitation from ${sender}`, matchingServerOrRoomPolicy);
                return Promise.resolve({
                    errcode: "M_FORBIDDEN",
                    error: "You are not allowed to send invitations to this homeserver",
                });
            }
            else {
                return Promise.resolve("NOT_SPAM");
            }
        }.bind(this);
        this.automaticRedactionReasons = [];
        for (const reason of automaticallyRedactForReasons) {
            this.automaticRedactionReasons.push(new matrix_basic_types_1.MatrixGlob(reason.toLowerCase()));
        }
        synapseHTTPAntispam.userMayInviteHandles.registerBlockingHandle(this.synapseHTTPCallback);
    }
    unregisterListeners() {
        this.synapseHTTPAntispam.userMayInviteHandles.unregisterHandle(this.synapseHTTPCallback);
    }
}
exports.SynapseHTTPUserMayInvite = SynapseHTTPUserMayInvite;
class BlockInvitationsOnServerProtection extends matrix_protection_suite_1.AbstractProtection {
    constructor(description, lifetime, capabilities, protectedRoomsSet, automaticallyRedactForReasons, synapseHTTPAntispam) {
        super(description, lifetime, capabilities, protectedRoomsSet, {});
        this.userMayInvite = new SynapseHTTPUserMayInvite(protectedRoomsSet.watchedPolicyRooms, automaticallyRedactForReasons, synapseHTTPAntispam);
    }
    handleProtectionDisable() {
        this.userMayInvite.unregisterListeners();
    }
}
exports.BlockInvitationsOnServerProtection = BlockInvitationsOnServerProtection;
(0, matrix_protection_suite_1.describeProtection)({
    name: BlockInvitationsOnServerProtection.name,
    description: "Blocks invitations from users marked as takedown or have bans matching the the configured automaticallyRedactForReasons",
    capabilityInterfaces: {},
    defaultCapabilities: {},
    async factory(description, lifetime, protectedRoomsSet, draupnir, capabilities, _settings) {
        if (draupnir.synapseHTTPAntispam === undefined) {
            return typescript_result_1.ResultError.Result("This protection requires synapse-http-antispam to be enabled");
        }
        return (0, matrix_protection_suite_1.allocateProtection)(lifetime, new BlockInvitationsOnServerProtection(description, lifetime, capabilities, protectedRoomsSet, draupnir.config.automaticallyRedactForReasons, draupnir.synapseHTTPAntispam));
    },
});
//# sourceMappingURL=BlockInvitationsOnServerProtection.js.map