using MAT, Test
using Dates
using SparseArrays, LinearAlgebra

function check(filename, result)
    matfile = matopen(filename)
    for (k, v) in result
        @test haskey(matfile, k)
        got = read(matfile, k)
        if !isequal(got, v) || (typeof(got) != typeof(v) && (!isa(got, String) || !(isa(v, String))))
            close(matfile)
            error("""
                Data mismatch reading $k from $filename ($format)

                Got $(typeof(got)):

                $(repr(got))

                Expected $(typeof(v)):

                $(repr(v))
                """)
        end
    end
    @test union!(Set(), keys(matfile)) == union!(Set(), keys(result))
    close(matfile)

    mat = matread(filename)
    if !isequal(mat, result)
        error("""
            Data mismatch reading $filename ($format)

            Got:

            $(repr(mat))

            Expected:

            $(repr(result))
            """)
        close(matfile)
        return false
    end

    return true
end

global format
for _format in ["v6", "v7", "v7.3"]
    global format = _format
    cd(joinpath(dirname(@__FILE__), format))

    result = Dict(
        "int8" => Int8(1),
        "uint8" => UInt8(1),
        "int16" => Int16(1),
        "uint16" => UInt16(1),
        "int32" => Int32(1),
        "uint32" => UInt32(1),
        "int64" => Int64(1),
        "uint64" => UInt64(1),
        "single" => Float32(1),
        "double" => Float64(1),
        "logical" => true
    )
    check("simple.mat", result)
    matfile = matopen("simple.mat")
    mat = read(matfile)
    close(matfile)
    for (k, v) in result
        if(typeof(mat[k]) != typeof(v))
            error("Types for $k didn't match (expected $(typeof(v)), got $(typeof(mat[k])))")
        end
    end

    result = Dict(
        "imaginary" => ComplexF64[1 -1 1+im 1-im -1+im -1-im im]
    )
    check("complex.mat", result)

    result = Dict(
        "simple_string" => "the quick brown fox",
        "accented_string" => "thé qüîck browñ fòx",
        "concatenated_strings" => String["this is a string", "this is another string"],
        "cell_strings" => Any["this is a string" "this is another string"],
        "empty_string" => ""
    )
    check("string.mat", result)

    result = Dict(
        "a1x2" => [1.0 2.0],
        "a2x1" => zeros(2, 1)+[1.0, 2.0],
        "a2x2" => [1.0 3.0; 4.0 2.0],
        "a2x2x2" => cat([1.0 3.0; 4.0 2.0], [1.0 2.0; 3.0 4.0]; dims=3),
        "empty" => zeros(0, 0),
        "string" => "string"
    )
    check("array.mat", result)

    result = Dict(
        "cell" => Any[v for _ in 1:1,
                      v in (1.0, 2.01, "string", Any["string1" "string2"])]
    )
    check("cell.mat", result)

    result = Dict{String,Any}(
        "s" => Dict{String,Any}(
            "a" => 1.0,
            "b" => [1.0 2.0],
            "c" => [1.0 2.0 3.0]
        ),
        "s2" => MAT.MatlabStructArray(["a"], [Any[1.0 2.0]])
    )
    check("struct.mat", result)

    result = Dict(
        "s00" => MAT.MatlabStructArray(["a", "b", "c"], (0,0)),
        "s01" => MAT.MatlabStructArray(["a", "b", "c"], (0,1)),
        "s10" => MAT.MatlabStructArray(["a", "b", "c"], (1,0))
    )
    check("empty_struct_arrays.mat", result)

    result = Dict{String,Any}(
        "s" => Dict{String, Any}("c"=>Matrix{Any}(undef, 0, 0), "b"=>Matrix{Any}(undef, 0, 0), "a"=>Matrix{Any}(undef, 0, 0)),
    )
    check("empty_cell_struct.mat", result)

    result = Dict(
        "logical" => false,
        "logical_mat" => [
            true false false
            false true false
            true false false
           ]
    )
    check("logical.mat", result)

    result = Dict(
        "empty_cells" => Any[v for _ in 1:1, v in (zeros(0, 0), "test", zeros(0, 0))]
    )
    check("empty_cells.mat", result)

    result = Dict(
        "sparse_empty" => sparse(Matrix{Float64}(undef, 0, 0)),
        "sparse_eye" => sparse(1.0I, 20, 20),
        "sparse_logical" => SparseMatrixCSC{Bool,Int}(5, 5, [1:6;], [1:5;], fill(true, 5)),
        "sparse_random" => sparse([0 6. 0; 8. 0 1.; 0 0 9.]),
        "sparse_complex" => sparse([0 6. 0; 8. 0 1.; 0 0 9.]*(1. + 1.0im)),
        "sparse_zeros" => SparseMatrixCSC(20, 20, ones(Int, 21), Int[], Float64[])
    )
    check("sparse.mat", result)

    matfile = matopen("partial.mat")
    var1 = read(matfile, "var1")
    @assert var1[28, 33] == 5
    var2 = read(matfile, "var2")
    @assert var2[27, 90] == 10
    close(matfile)

end

result = Dict(
    "index" => [8.8604784000000000e+04   9.8707212000000000e+04   1.0394035200000000e+05   1.1429712000000000e+05   1.5474923999999999e+05   1.5475435200000001e+05   1.5501074400000001e+05   1.5505315200000000e+05   1.5505718400000001e+05   1.5506168400000001e+05   1.5506524799999999e+05   5.4945741599999997e+05   5.6345896799999999e+05   5.9956981200000003e+05   7.0691623199999996e+05   7.9063657200000004e+05   8.4311938800000004e+05   9.2225131200000003e+05   1.1248994160000000e+06   1.2508148520000000e+06   1.4164141320000000e+06   1.4275988280000000e+06   1.4744331000000001e+06   1.4982212879999999e+06   1.5549058440000000e+06   1.5870300840000000e+06   1.6192005120000001e+06   1.6766071560000000e+06   1.9386816839999999e+06   1.9969427879999999e+06   2.0021861880000001e+06   2.3272494120000000e+06   2.5309351080000000e+06   2.6743788720000000e+06],
    "spikes" => [
      -3.9146236245031032e+00  -6.7657651330021364e+00  -1.0780027188484372e+01  -1.4345619557780790e+01  -1.5488781013877338e+01  -1.3241531877846004e+01  -8.6339302778751907e+00  -4.1571900578409995e+00  -1.4845719040296610e+00   2.3147400250828232e-01   2.8917910181412778e+00   6.4067867244186800e+00   8.3368575385567603e+00   7.0732985406218223e+00   4.4095174940268036e+00   3.8495932342509342e+00   7.0605464919276546e+00   1.2892731012948772e+01   1.8593404980539656e+01   2.1332908128411184e+01   2.0142332517120792e+01   1.6740473413471157e+01   1.3650330377340575e+01   1.1913871749214691e+01   1.0804794411826084e+01   8.8366401987297127e+00   5.1092331749990514e+00   5.1218216653980408e-01  -2.9327647633922682e+00  -4.4870896208146753e+00  -5.0598199463728655e+00  -4.8330524336350118e+00  -2.8556000012645000e+00   2.9794817723619027e-01   1.8265416505730325e+00  -8.6155940979615875e-02  -3.9623352473810947e+00  -6.9070013227561047e+00  -7.3941131196997647e+00  -5.7411207637544166e+00  -3.2366812420300106e+00  -1.1460492068000723e+00   1.2381260731009580e-01   1.0930145325605314e+00   2.1927876983540933e+00   2.6570284430776856e+00   1.3381366125210661e+00  -1.2539624260623763e+00  -3.3642620416729994e+00  -4.1849749207505456e+00  -3.8760400918509301e+00  -2.6869552030388291e+00  -1.6718246062697015e+00  -2.3709942853677934e+00  -4.6623835517993664e+00  -6.6575320887201714e+00  -6.9891263747717174e+00  -5.7017039068420186e+00  -3.4759011423153079e+00  -1.7092931352045238e+00  -2.3854494206243695e+00  -5.8068462168496913e+00  -9.1001745572212531e+00  -8.8479323560036516e+00
      -5.0745139212223540e+00  -9.6338046242625506e+00  -1.3472006614220170e+01  -1.5245910823426385e+01  -1.5914718584705716e+01  -1.6397777086548157e+01  -1.5688652912166024e+01  -1.2863641020969972e+01  -9.5044110719151487e+00  -8.3463890698794305e+00  -1.0177024989874276e+01  -1.3445743563344164e+01  -1.6692449999817043e+01  -1.9680052373752346e+01  -2.0121942799899024e+01  -8.3700572911264981e+00   3.0032951243048100e+01   9.4582543108269547e+01   1.5377989075611117e+02   1.7130866838277802e+02   1.4748486763605970e+02   1.0947277486760775e+02   7.2684596239613015e+01   3.8193102601464858e+01   1.0540246012312117e+01  -3.8091287155762972e+00  -5.9643997244651556e+00  -3.0829122045048947e+00  -9.3969579584981044e-01  -4.1011214330806744e-01   1.1485354724913588e-01   8.3209939369519947e-01   1.7132389429717065e-03  -3.6057351605809611e+00  -8.8203434292989460e+00  -1.2692989286632969e+01  -1.3074091545751195e+01  -1.0580269175201330e+01  -7.7262110976707312e+00  -6.1406117115548504e+00  -5.0818589866084025e+00  -3.3150901769832091e+00  -1.7731765969545847e+00  -2.5627661695922046e+00  -5.6404917366725886e+00  -8.3842617579066747e+00  -8.5769172149145199e+00  -6.8248629640086911e+00  -5.7747794329330571e+00  -7.1063437611930578e+00  -9.7162880960954556e+00  -1.1170428842658552e+01  -1.0378119546974839e+01  -8.3419883744685741e+00  -6.9209209230224635e+00  -7.4145555946251820e+00  -9.6713314325255784e+00  -1.2062946758371979e+01  -1.2965017154593347e+01  -1.2408773058757985e+01  -1.1624355346583785e+01  -1.1127849526328186e+01  -1.0048811168388408e+01  -7.6327226475166228e+00
      -3.6172800577681850e+00  -3.0496107525626650e+00  -4.0450961689646379e+00  -4.9841853952779340e+00  -4.3137243411079638e+00  -2.4211699563520748e+00  -9.3469136675988018e-01  -9.3834475312754062e-01  -2.3043613016166664e+00  -3.9905808684096256e+00  -4.6842071022425413e+00  -3.8310453992051836e+00  -2.8818230979207713e+00  -5.2038837297130556e+00  -1.1725055393804819e+01  -1.3208091586845086e+01   9.2482123494256854e+00   6.3181823866266974e+01   1.2177486062412244e+02   1.4070171740815530e+02   1.0609880427064739e+02   4.8491974791356185e+01   5.8467389218168400e+00  -9.6802170255950291e+00  -8.6605173391900614e+00  -5.1555616199626577e+00  -5.4237382663488169e+00  -8.2575373220146044e+00  -1.0156241701964554e+01  -9.1574620977201118e+00  -5.6665700132115431e+00  -1.5504659563007614e+00   1.1334383183944798e+00   1.2640431009400714e+00  -6.6794751018025034e-01  -2.9543901520038176e+00  -4.1963329340429310e+00  -4.3642482086638417e+00  -4.0042244694146101e+00  -3.1394295409389494e+00  -1.8179756080601912e+00  -9.3024527470805374e-01  -1.4736656790371958e+00  -3.4645773489091209e+00  -5.8588062538620047e+00  -7.0734948204842896e+00  -6.0481893429370261e+00  -3.3344016972431088e+00  -4.8650896484328832e-01   1.6052150099271567e+00   2.9838901666777322e+00   3.7836445807114574e+00   3.9345735120816623e+00   3.1644840267225094e+00   1.0070686684300534e+00  -2.4713922778659478e+00  -5.9984216012494951e+00  -7.8885829262850233e+00  -7.5154376078825695e+00  -5.8326333185237758e+00  -4.5003931071444478e+00  -4.2328547992043450e+00  -4.1140676583157889e+00  -3.1460951038002301e+00
      -3.6781701228839154e+01  -3.7303084834807457e+01  -3.8884510028175633e+01  -4.1148035399962872e+01  -4.3316242853459642e+01  -4.4384384580854572e+01  -4.4790855343692897e+01  -4.7038755365884079e+01  -5.2501365276546458e+01  -5.6812084874163638e+01  -4.9570438054352273e+01  -2.3981805280442412e+01   1.1015352613446286e+01   3.7025927055247365e+01   4.6196840812364755e+01   4.5435239856395953e+01   4.4108776047185387e+01   4.5957015883875258e+01   4.9338962205718175e+01   5.0907715343642117e+01   4.9487314118838199e+01   4.6376644839165742e+01   4.2907993314061748e+01   3.9175485818078528e+01   3.4646364450142663e+01   2.9438295855526434e+01   2.5290001942034166e+01   2.4585557992414284e+01   2.7869881512937130e+01   3.2856199330415350e+01   3.6123557386279117e+01   3.5905020129260436e+01   3.3291843240497656e+01   3.0657151735220140e+01   2.9399978478943620e+01   2.9238216866099457e+01   2.8701738303043562e+01   2.6077460100630137e+01   2.0689205683518288e+01   1.3779281580459998e+01   8.1712538443203133e+00   6.6925061080176569e+00   1.0147005424138726e+01   1.6509827576474752e+01   2.2276297357800129e+01   2.4765444980789738e+01   2.3565808601878054e+01   2.0365852060539318e+01   1.7482427576467405e+01   1.5922922611993604e+01   1.4534460949429915e+01   1.1655908219543699e+01   7.5216074902940893e+00   3.9334852970087364e+00   1.7985868168036778e+00   3.9751100517608062e-01  -9.6689135481901367e-01  -1.8291316756437888e+00  -1.2108732556384698e+00   1.1422814255806824e+00   4.0827052431935931e+00   5.6696392298883893e+00   4.7889899662105178e+00   1.9874261259261794e+00
      -1.1686614851500112e+00   9.8124871505894706e-01   2.4682927864975457e+00   2.9222372863110442e+00   1.6794130795786746e+00  -6.8820557107116076e-01  -2.4491510090825823e+00  -2.5573541785944220e+00  -1.5657335653505560e+00  -6.2326635393824792e-01   7.5542615186640072e-03   1.2160520506156904e+00   3.7959025732449647e+00   7.1695274203178112e+00   9.8085862745466752e+00   1.1024667458560756e+01   1.1849030121520164e+01   1.3645183550697372e+01   1.6153045543872672e+01   1.7653460112285362e+01   1.6917192047304358e+01   1.4287778990065465e+01   1.1181854556421472e+01   9.2813981144484536e+00   9.6980182244576554e+00   1.1975656236851366e+01   1.4218362205978007e+01   1.4823189770395057e+01   1.3782281414778801e+01   1.2224714159620135e+01   1.1308817315730568e+01   1.1413207645545857e+01   1.1802812942398777e+01   1.1304006120772028e+01   9.4673823725009267e+00   6.7674104407290443e+00   3.9034391463718467e+00   1.0696510964445158e+00  -2.2265622627917740e+00  -6.1619053880194183e+00  -9.4895708918063448e+00  -1.0511848180773416e+01  -9.2016379963305752e+00  -7.3655072801953869e+00  -6.6664600174160604e+00  -7.2291846775841488e+00  -8.2107481715781105e+00  -9.1021811818757321e+00  -1.0311278259980540e+01  -1.2741511638678721e+01  -1.6437776102727412e+01  -1.9514979330779195e+01  -1.9610190285680936e+01  -1.6644297020966704e+01  -1.2800100439249650e+01  -9.9046719354109900e+00  -8.2923627062655303e+00  -7.8966764067275435e+00  -8.8257663696547084e+00  -1.0735616318501446e+01  -1.2599202496900540e+01  -1.3220431400502429e+01  -1.1920594647006027e+01  -9.3084512934042376e+00
       1.1099942135772691e+01   1.0097197127580243e+01   1.1065906114342178e+01   1.3374123078938322e+01   1.6449533911253752e+01   2.0017355902687680e+01   2.3638247141687746e+01   2.6292880854006093e+01   2.6783333166190332e+01   2.5183008648660202e+01   2.3296988595385344e+01   2.2810464758877668e+01   2.3624918756652875e+01   2.4607374587222779e+01   2.5393398749862719e+01   2.6769469521678889e+01   2.9281995275221178e+01   3.2213436799025317e+01   3.4285343581139983e+01   3.4785296465751173e+01   3.3883822298572078e+01   3.2386332385252516e+01   3.1231943637716320e+01   3.0932840342758819e+01   3.1246878043151604e+01   3.1160302181567783e+01   2.9550070933834700e+01   2.6540966708675843e+01   2.3785330200047358e+01   2.2611886801417548e+01   2.2531515536471172e+01   2.2340044415310590e+01   2.1822535590221452e+01   2.1399364443416083e+01   2.0755992097134378e+01   1.8948315967557182e+01   1.5769917402888659e+01   1.2205733819672014e+01   9.4099541981525228e+00   7.7046512181258402e+00   6.5719664417242711e+00   5.2343013227680553e+00   3.3389338216778657e+00   1.2573655827507051e+00  -5.0199990120897509e-01  -1.8987019795142805e+00  -2.9172592054133535e+00  -3.2152176994632509e+00  -2.9689210401896537e+00  -3.2720957493420673e+00  -4.8582690799070640e+00  -7.1594797906170751e+00  -9.1803890906391956e+00  -1.0495134027231670e+01  -1.1056911353417380e+01  -1.0726534970818260e+01  -9.3753223680626547e+00  -7.2907274381158134e+00  -5.1694262729300959e+00  -3.7312476583791785e+00  -3.6141234953544066e+00  -5.1662008322483448e+00  -7.7776079749577836e+00  -9.8010587078412321e+00
      -1.4685530727984641e+01  -1.0292328330297664e+01  -6.2040173116488546e+00  -2.3375015307800595e+00   1.3923917658511158e+00   5.0564769748901313e+00   8.6976966359851104e+00   1.2323245348083425e+01   1.5903913125353689e+01   1.9380777962659089e+01   2.2677898949762564e+01   2.5718454097742246e+01   2.8441073995486732e+01   3.0813082951470030e+01   3.2837981366611231e+01   3.4555661605993883e+01   3.6035311542317494e+01   3.7362431137561401e+01   3.8622571934083787e+01   3.9885068827368393e+01   4.1190035284914742e+01   4.2541236867236606e+01   4.3906275503545629e+01   4.5224046329587310e+01   4.6417965861852949e+01   4.7412309197387181e+01   4.8148368395523121e+01   4.8597180088665191e+01   4.8766260494684808e+01   4.8698991553936125e+01   4.8466780378761598e+01   4.8155568154983840e+01   4.7849401563031734e+01   4.7614370946271123e+01   4.7486145888939269e+01   4.7463615344086584e+01   4.7509911501785595e+01   4.7560611227134473e+01   4.7537464942025593e+01   4.7364890774312840e+01   4.6985915393397903e+01   4.6374354157541411e+01   4.5540779674504876e+01   4.4531077464674446e+01   4.3417878743397615e+01   4.2286593389902350e+01   4.1218852636871070e+01   4.0276692303289153e+01   3.9490658602725233e+01   3.8854229591844380e+01   3.8325674875439489e+01   3.7836979823323738e+01   3.7308039311521902e+01   3.6663265807716996e+01   3.5847270787923392e+01   3.4836464766174224e+01   3.3644242194420237e+01   3.2318708010392434e+01   3.0933403069436942e+01   2.9572894236760884e+01   2.8316128161485977e+01   2.7220897815058663e+01   2.6312547181929837e+01   2.5579187011419549e+01
       1.5763956152315169e+01   1.6523243082892250e+01   1.7337427631570872e+01   1.8120936638484590e+01   1.8792445157811912e+01   1.9291006994381139e+01   1.9588077541817789e+01   1.9693009891213325e+01   1.9650918197573489e+01   1.9533337877188348e+01   1.9423561429962543e+01   1.9399599967723205e+01   1.9518198338118580e+01   1.9803119911795868e+01   2.0240058237539227e+01   2.0779199633821964e+01   2.1344921568275133e+01   2.1850676042659646e+01   2.2216063986224665e+01   2.2382665762479274e+01   2.2325443040524132e+01   2.2057418172501748e+01   2.1626689639781716e+01   2.1106384004760407e+01   2.0579565930487874e+01   2.0122142297473431e+01   1.9787200170220586e+01   1.9593930129327656e+01   1.9523364223698717e+01   1.9521786868823959e+01   1.9511132956806740e+01   1.9404280892123200e+01   1.9122163091585080e+01   1.8609252789360340e+01   1.7844318015586811e+01   1.6844290518050197e+01   1.5660474424296631e+01   1.4367827117720619e+01   1.3049394428891683e+01   1.1778975433290963e+01   1.0605636123380648e+01   9.5436591203415890e+00   8.5705174399266060e+00   7.6329046830511684e+00   6.6569045386579333e+00   5.5557424579780292e+00   4.2330232952205966e+00   2.5948387638090815e+00   6.0010139846456756e-01  -1.6403240489988979e+00  -3.7983903715983196e+00  -5.5143675763583140e+00  -7.0197817000212250e+00  -9.2991809829904906e+00  -1.2710108289977208e+01  -1.5773321560596763e+01  -1.6492611312664831e+01  -1.5291065260668367e+01  -1.5553348127486492e+01  -1.9596399167644933e+01  -2.5711586491024381e+01  -3.1294712703031326e+01  -3.6140274982880157e+01  -4.0736526948594026e+01
      -7.5455089032603496e+01  -8.8091850484650791e+01  -7.6894847494814442e+01  -5.1931618057453484e+01  -2.8833831644552824e+01  -6.8895988088626297e+00   2.1168348264522379e+01   4.6693994244353583e+01   5.3777387223823958e+01   4.4603130335345831e+01   3.3530654869558120e+01   2.3587826739136968e+01   8.4238787409792160e+00  -9.2318222698507917e+00  -1.4713136414898660e+01   1.9717165915666754e+00   3.1968641163889899e+01   5.7269930354848498e+01   7.0152688867705734e+01   7.3409072049238773e+01   7.0166235593951257e+01   6.1829204326045371e+01   4.6696134605544373e+01   1.8622054473547621e+01  -2.2829591964997910e+01  -6.2715671170700787e+01  -8.2653742502672856e+01  -8.1088328703707077e+01  -7.2961806812740548e+01  -7.3632394853923415e+01  -8.9841480609702060e+01  -1.1962228131535032e+02  -1.5066572610100792e+02  -1.6038121312642346e+02  -1.2981471355911390e+02  -6.3653163261418783e+01   9.3811209125033130e+00   5.8695741367623484e+01   7.0437047869761585e+01   4.9630793312944938e+01   1.6599415269056415e+01  -2.5402904227425278e+00   4.0559101821793178e+00   2.4429205550104435e+01   3.9021358713588448e+01   3.8520783430674051e+01   1.9952697815867261e+01  -2.1877257859243720e+01  -8.3907527060881065e+01  -1.4250639755111564e+02  -1.6619398657243511e+02  -1.3819218483975862e+02  -6.6785267226843359e+01   7.8497452778842174e+00   2.7663034892382331e+01  -1.9252281399357905e+01  -5.9646303432603780e+01  -1.8852103558667594e+01   6.7639555235119275e+01   9.6932460076734856e+01   4.1262889176984515e+01  -3.3610639804651925e+01  -7.8423083296187542e+01  -1.0839727633754131e+02
       4.7736577112711117e+01   4.5700337234538978e+01   4.5322631494156710e+01   4.9171444949072850e+01   5.3757473474671947e+01   5.4141589236927501e+01   4.8790940853843097e+01   4.1857477273482019e+01   4.0754533608850998e+01   4.8608027316167508e+01   6.0100216178867996e+01   6.8180296544356494e+01   7.1421072557515430e+01   7.0807306032014466e+01   6.6514973934144194e+01   6.2945074970931898e+01   6.8196514720480167e+01   8.5012727237677268e+01   1.0720726826115141e+02   1.1957256746876999e+02   9.9138254538272335e+01   3.8069276174166163e+01  -2.7233524821672358e+01  -3.9334632563706222e+01   1.3886634524961096e+01   7.3584276961424322e+01   6.3628518198466388e+01  -2.9054572516934122e+01  -1.4209888641029568e+02  -2.0910522987863314e+02  -2.1415046392049794e+02  -1.7586701353513934e+02  -1.1844102599759782e+02  -6.1431963885049129e+01  -1.4806030025628601e+01   1.4027944301627443e+01   1.0652586257860358e+01  -3.0054086554043813e+01  -8.5072361204847368e+01  -1.2046643138190960e+02  -1.2106054250546094e+02  -9.5841527604366775e+01  -6.6055322458778861e+01  -4.4458770387691487e+01  -2.4336370991880798e+01   4.5840797349366902e-01   2.1040288319943066e+01   2.3611095638912182e+01   1.3051021255932143e+00  -2.8851154140515209e+01  -3.7619486391456242e+01  -2.9915658771811749e+01  -3.0708973922309450e+01  -3.1924567737368655e+01  -1.0202022943830535e+01   2.9706495601613049e+01   5.9067239546813376e+01   5.4702088800913174e+01   2.8343487643024787e+01   1.5951389347675942e+01   2.2837902575396686e+01   1.2908948645547047e+01  -3.3255358730535995e+01  -8.7746125812653432e+01
      -8.2377418542911354e+00  -3.4972931824407372e+00  -7.7569626244497876e+00  -1.5166206734460895e+01  -1.5445771078633909e+01  -5.2413228186791176e+00   8.4387175092257500e+00   1.5431757026108208e+01   1.2399978335869392e+01   7.4095099729504046e+00   1.0679579816928053e+01   1.8701188652942292e+01   1.7701213689196198e+01   4.8431711642004913e+00  -4.7779662604481938e+00   3.1723005655630221e+00   2.4531952072999687e+01   4.5698045473740862e+01   5.9257083049327122e+01   6.2627426268528140e+01   5.7411981420461260e+01   5.2570470576359099e+01   5.2813113343401156e+01   5.0453507704104304e+01   3.9691119012785791e+01   2.6528767918502940e+01   1.8741299694240368e+01   1.7724441294399568e+01   2.0821175647900695e+01   2.4417659444952996e+01   2.5870167533339281e+01   2.4983092349575369e+01   2.2998436116876093e+01   2.0558711472598521e+01   1.7502822279045098e+01   1.3739438787271743e+01   9.9370928813924273e+00   7.9696531198819107e+00   9.3237074864808100e+00   1.2496048137913172e+01   1.4202138161523894e+01   1.3776871923186704e+01   1.4057294551750237e+01   1.6919973545359703e+01   2.0190481439159552e+01   2.0371692873037446e+01   1.6567959095411027e+01   1.0520230640011906e+01   4.5794135735479564e+00   8.7710329733688541e-01   8.9182399282027713e-01   4.3463285075996909e+00   8.7315858121978973e+00   1.0903936631346470e+01   1.0288756426506817e+01   1.0049725699443105e+01   1.3054576528907717e+01   1.6854021523448839e+01   1.6041035800270304e+01   1.0367866836597134e+01   6.4265096090680434e+00   9.4949675891987368e+00   1.6836520945082036e+01   2.1325105933919993e+01
      -4.1679158148256432e+00  -5.2810404688221357e+00  -5.9038977834292563e+00  -6.2634044553555412e+00  -5.9749855839469195e+00  -4.3218694766614334e+00  -1.1788772851158797e+00   2.4374543533437842e+00   4.8050829080472326e+00   4.8908091608309068e+00   3.3699448624872437e+00   1.8764385857350361e+00   1.0658937699825455e+00  -2.7882253437288551e-01  -3.1356733669314134e+00  -4.1284337659893531e+00   4.9241012162347957e+00   2.8218865342091348e+01   5.5056325524296241e+01   6.5099843979883218e+01   4.9609447627592324e+01   2.1423693795580775e+01   2.5132863903596458e-01  -5.8604600478907667e+00  -2.3380828579082129e+00   1.9600991106694594e+00   2.5915747007003080e+00   2.0190047306038350e-01  -2.8577160575959448e+00  -5.2869354732703258e+00  -6.9099491413979388e+00  -7.7559496509908179e+00  -8.0339926236897785e+00  -8.1760789963289717e+00  -7.9859360445805283e+00  -6.3703713820186918e+00  -3.0321001946270005e+00  -2.8201117047924162e-02  -7.8351721876626979e-02  -3.2229219827610613e+00  -6.6022607253341592e+00  -7.6428337012652356e+00  -6.3520480710327867e+00  -4.5468067925800151e+00  -3.6792489626697655e+00  -3.7265752054412307e+00  -3.8104694479464527e+00  -3.1502606288908388e+00  -1.5302095841423975e+00   2.6517264914271876e-01   7.6568764912213050e-01  -5.6131323023144386e-01  -2.6412705089170849e+00  -4.2923899829303380e+00  -5.1769894068210993e+00  -4.9614565332821146e+00  -3.2859163281387294e+00  -1.1703534382204590e+00  -7.6910847698287388e-01  -2.6122681986842187e+00  -4.2742848318737003e+00  -3.0739009488926641e+00   6.8120601334287323e-01   3.9336266256502612e+00
      -4.6484775115089541e+00  -8.3581561181698660e+00  -1.0902142569692860e+01  -1.1511797760460409e+01  -9.8276834132977005e+00  -6.2404161549022161e+00  -1.8432175810231679e+00   1.9347492334298952e+00   4.2938226787082971e+00   5.6566425031547753e+00   6.9370739014200060e+00   8.7323889596425897e+00   1.1151007876295226e+01   1.3469254774199312e+01   1.4378076971481226e+01   1.3628707243729355e+01   1.2810890536991595e+01   1.3603571552449427e+01   1.5792969414511386e+01   1.7425203016009618e+01   1.6493276434801135e+01   1.2586445678344095e+01   7.3401463249477024e+00   3.2087842017129540e+00   1.5030100447830739e+00   1.3898465906994972e+00   8.8198090743093560e-01  -1.0493063115005812e+00  -3.7608545130075726e+00  -6.3949540211998350e+00  -9.0066585358720026e+00  -1.1745588278397154e+01  -1.3842632605375675e+01  -1.4123365855042982e+01  -1.2335802612795424e+01  -9.4739532170207532e+00  -6.3782197090918142e+00  -2.8832164731098446e+00   4.8447063694123349e-01   1.6128314119847449e+00  -4.1774621162461167e-01  -3.3629973208103769e+00  -4.5025431650456200e+00  -3.9290643942165806e+00  -3.7022642550448306e+00  -4.6991587651781321e+00  -5.6909794907882132e+00  -5.2906700399814426e+00  -3.7661436808309654e+00  -2.5302504318642383e+00  -2.0969878128246422e+00  -1.3265870686252779e+00   7.0961290345922357e-01   3.0385814819877943e+00   3.9548508946305954e+00   2.9954312845408939e+00   9.9988805713822959e-01  -9.2502312206375614e-01  -2.0384940259267319e+00  -2.2297243620823197e+00  -2.0769146009998161e+00  -2.3170471037592764e+00  -2.8506427004880215e+00  -2.6777747842245225e+00
      -7.7992155184377432e-01   8.4208602530413512e-01   2.4887283142722940e+00   2.4338293029048339e+00   8.3831805409641036e-01  -5.6099977991794492e-01  -7.1266715703072170e-01  -2.3092906084045711e-01   1.6955881483329160e-01   8.8504105274391720e-01   2.2423969436547071e+00   3.3863034262368776e+00   3.6616215908416012e+00   4.2752627336864073e+00   7.0581057720790223e+00   1.1758833658720096e+01   1.6047422019307188e+01   1.8405075233788665e+01   1.9427396781993370e+01   1.9879444229356857e+01   1.9348574654086402e+01   1.7629540788742673e+01   1.5792875449818970e+01   1.4872230427632566e+01   1.4471792816861928e+01   1.3426898347317799e+01   1.1319456367524939e+01   8.8575138135993203e+00   7.0846315948436089e+00   6.5127944804059057e+00   6.6781793515550252e+00   6.2758575263602134e+00   4.2213211609500574e+00   1.0187928511802555e+00  -1.3365602410322484e+00  -1.2247622425200726e+00   1.1481723220280773e+00   4.3321910406078903e+00   6.9648334152330520e+00   8.0154771868425438e+00   6.5719188712588483e+00   2.4474218319276826e+00  -3.0491523253799624e+00  -7.4154828913997903e+00  -8.5953677824666350e+00  -6.7813390775209372e+00  -4.2068257455106703e+00  -2.7309337921106196e+00  -2.0348235578019351e+00  -6.2767747718278888e-01   1.7379756336568186e+00   3.5393101620046572e+00   3.5444884355037840e+00   2.3746522750550279e+00   1.3500592448824880e+00   6.9976097822180139e-01  -1.5618921435797500e-01  -1.3162226599662721e+00  -2.4845798158016903e+00  -3.7797798352858396e+00  -5.2368125011457316e+00  -5.9486138818233414e+00  -5.1354371684134463e+00  -3.7528582973371494e+00
      -3.0447537175403006e+00  -1.7667805638958940e+00  -1.9778229750617011e+00  -2.2515604403315699e+00  -1.3449261643785633e+00  -1.0662626484421048e-01  -5.7657486030100791e-02  -7.4315041221388778e-01  -3.2337810363445102e-01   1.8577438758015754e+00   4.8425765941254983e+00   7.8123978114524020e+00   1.1105854493054094e+01   1.4841903991300153e+01   1.7468871344145363e+01   1.7574512951317725e+01   1.6451820522767886e+01   1.6628323973953727e+01   1.8301434258000356e+01   1.8931382166939795e+01   1.6217394592510296e+01   1.0559530299242045e+01   4.4576293834236687e+00  -1.1822470014800246e-02  -2.5727563770885062e+00  -3.7547505719161225e+00  -3.5081536657220189e+00  -1.4835065248806825e+00   1.8247576781703743e+00   4.8275703093820033e+00   5.7228574725963268e+00   3.8558888609578723e+00   4.0990146070366057e-01  -2.4574645653496310e+00  -3.1102243476353593e+00  -1.4474225309731716e+00   7.4127875680854416e-01   1.0383069562131020e+00  -1.0150232301722759e+00  -3.4198210424706135e+00  -4.3088303898183238e+00  -3.9165170232756070e+00  -3.6605958183355347e+00  -4.2469618414480417e+00  -5.2237494007048566e+00  -5.9420620236257076e+00  -6.1698135259889568e+00  -5.7089733374935268e+00  -4.0492707065696969e+00  -8.0849384208890562e-01   3.3216805308527921e+00   6.2990262694136261e+00   6.1812329879916819e+00   3.1266169114028388e+00  -4.8505368345761951e-01  -2.2625885070785667e+00  -2.0371523965480165e+00  -1.4033783831712108e+00  -1.2651726669536290e+00  -8.7131680693218405e-01   5.0106877171388586e-01   1.8406747295741790e+00   1.3085085083594870e+00  -1.5293730324086736e+00
       2.3513445326289961e+00   1.7384880477187443e+00   2.2184897775735659e+00   5.2051223974837342e+00   1.0452816121087638e+01   1.5377084182445486e+01   1.7198975360259148e+01   1.6104713320124159e+01   1.4833420837246866e+01   1.5100670860303905e+01   1.6073099350257220e+01   1.6269109488250638e+01   1.5637757524959511e+01   1.5476499170789715e+01   1.6604024256326596e+01   1.8317676582762779e+01   1.9829075755286041e+01   2.1816151151864332e+01   2.5055347925633036e+01   2.7333907737059032e+01   2.2427805378565086e+01   3.7141556286936721e+00  -2.8648138173116212e+01  -6.3843957225821207e+01  -8.7089860614432553e+01  -9.1923311472937954e+01  -8.3938126428996938e+01  -7.2885847777427841e+01  -6.4297677965088411e+01  -5.7914366258401401e+01  -5.0470379597131561e+01  -3.9499120538780375e+01  -2.5751695307781368e+01  -1.1778187088014317e+01   1.0189820693630494e+00   1.1960524761686607e+01   1.8733303083800365e+01   1.8356414318508428e+01   1.0668647686854321e+01  -1.0075011878329772e+00  -1.2633619248121462e+01  -2.1404683668367543e+01  -2.5602377289037388e+01  -2.4560926814716566e+01  -1.8847733387633440e+01  -9.8404336731293878e+00   3.5966958558048301e-01   9.0662935627188972e+00   1.4335829915263089e+01   1.6036694595613060e+01   1.5184337823870287e+01   1.2469246394512929e+01   7.6677145828817306e+00   8.5165822752424969e-01  -5.8806346340252293e+00  -9.1084583818696743e+00  -7.2999534121027425e+00  -2.4813921667292815e+00   1.7215656085634032e+00   3.0197896060174911e+00   1.4579843578258131e+00  -1.6420102258369735e+00  -4.8732983588159042e+00  -6.9044980772313806e+00
      -7.1748224092297468e+00  -3.8936102456845245e+00   1.0743831294909865e+00   4.8816301487736782e+00   5.4189196440465226e+00   2.8421806913201926e+00  -7.8193972997551731e-01  -3.3953855854612938e+00  -3.9985877326731809e+00  -2.4284915455410818e+00   3.3095855413398589e-01   2.1860065858780535e+00   1.8280988855495470e+00   3.4048799490454618e-01   3.8790118108287441e-01   3.8556647578240639e+00   9.9718322097927121e+00   1.5912297294502212e+01   1.9511397698602284e+01   2.0651366980982488e+01   2.0134282655984631e+01   1.8560445836154724e+01   1.6287447146155721e+01   1.3561600277652611e+01   1.0795264078502715e+01   8.6030785401577266e+00   7.0022103875078896e+00   5.1888911547538754e+00   2.7145704050730384e+00   1.4143845469113936e-01  -1.9107456843838713e+00  -3.4559243419646326e+00  -4.2223452047916581e+00  -3.1130434517080205e+00   1.7183260785086762e-01   3.7869557639525033e+00   5.3380153379310666e+00   4.2352058165010487e+00   1.6573488592368961e+00  -8.5887188564406203e-01  -2.1681503131724709e+00  -2.1279908375294667e+00  -2.0792749208808359e+00  -3.3997834567720187e+00  -5.3580052019440032e+00  -5.9792374734377622e+00  -4.8403202446726707e+00  -3.3487890303481347e+00  -2.6600981462061033e+00  -2.7228934357253367e+00  -3.1308053901450172e+00  -3.8963449071942557e+00  -5.2729284759196471e+00  -7.2764195946828067e+00  -9.3283947520693822e+00  -1.0227824562298673e+01  -9.0739885142293417e+00  -6.5681575524140818e+00  -4.8038541028704778e+00  -5.3349275900632449e+00  -7.6025577687023391e+00  -9.3130362450711317e+00  -8.3152888687476345e+00  -4.2484255376825795e+00
      -1.2094820634316039e+00  -1.2706817164948352e+00  -1.2775018026388452e+00  -2.3357176552894954e-01   1.8928499829725349e+00   4.4677863928736414e+00   6.9513585826945681e+00   9.0204123867514259e+00   1.0482452392061710e+01   1.1455309752506835e+01   1.2237810752650413e+01   1.2688637793112969e+01   1.2235313891126804e+01   1.1100297213735251e+01   1.0773414547739115e+01   1.2217993905654721e+01   1.4701466325393794e+01   1.7319490559451381e+01   1.9808972202344144e+01   2.1257689253929254e+01   2.0368628657184544e+01   1.7524563363309646e+01   1.4843567969750833e+01   1.3863574192388871e+01   1.4031045614008303e+01   1.3693669004482535e+01   1.2122436338523961e+01   9.9467781708780620e+00   7.8498090003437868e+00   5.7967683127528629e+00   3.7206536395890204e+00   2.3267777177674382e+00   2.6303789001713365e+00   4.7445695138569235e+00   7.6683738268993942e+00   9.9717476628589878e+00   1.0271279571889663e+01   8.1958760665984958e+00   5.3998213275104048e+00   4.2899793853628481e+00   5.5462911375305124e+00   7.6595253723593695e+00   8.4981448223169700e+00   7.2294579833209385e+00   5.1143564806110922e+00   3.8260968045635870e+00   3.3065755059745818e+00   2.5815464286120089e+00   1.7488042411202225e+00   1.5184709880337026e+00   1.6370045591241176e+00   1.1531897091681369e+00  -1.5863929796862442e-01  -1.5759013113892080e+00  -2.3911490323759002e+00  -2.4147368159491429e+00  -1.8749121657426793e+00  -1.4086065112641286e+00  -1.6573036686621481e+00  -2.4019595691049194e+00  -2.7197582423545761e+00  -2.3228923158464436e+00  -1.8987417992229161e+00  -1.9762436025825927e+00
      -2.1788608089091870e+00  -2.3721017929273520e+00  -2.2248744539174794e+00  -1.3562719882732486e+00   1.9303974836178317e-01   1.3110471548593794e+00   1.2797866409946126e+00   1.1446512790093115e+00   2.4218364442454297e+00   5.3568199699566073e+00   9.1928444896243846e+00   1.2911855868338753e+01   1.5131303558516208e+01   1.4796675429288374e+01   1.2747818159896775e+01   1.1384776546745211e+01   1.2304499348204516e+01   1.5012766622841385e+01   1.7706586949808607e+01   1.8615946254339946e+01   1.7088985921306737e+01   1.4118105664819083e+01   1.1547667913836323e+01   1.0268505344072013e+01   9.5426651513383796e+00   8.3936752159092531e+00   6.7898835952875585e+00   4.9190137058105190e+00   2.3035160443938620e+00  -1.4454604882529436e+00  -5.5578141560543060e+00  -8.6285337833508056e+00  -9.8768639485094845e+00  -9.3497459396540972e+00  -7.4531759065479113e+00  -4.8343191161214989e+00  -2.2707557414183874e+00  -2.1273932165940168e-01   1.4763518504563440e+00   2.9929246806770484e+00   3.6484933755646152e+00   1.9940019916984792e+00  -2.3622776610960083e+00  -7.2558689942915287e+00  -9.3484617741468909e+00  -7.4954744905729234e+00  -3.9544592514927057e+00  -1.7075406868672076e+00  -1.4165546781975049e+00  -1.6221832630448796e+00  -1.0260840386708110e+00   1.6403282116014173e-01   6.0805828587662303e-01  -7.8558049118458273e-01  -3.6478240721938118e+00  -6.4046749238136291e+00  -7.7290809125845756e+00  -7.3840184127203816e+00  -5.7754694634749093e+00  -3.6798105136804447e+00  -2.5013142267663810e+00  -3.3639333122560284e+00  -5.7371424805126106e+00  -8.0051890648261459e+00
      -1.5740146351038696e+01  -1.5127188568935047e+01  -1.6157259030112712e+01  -1.7330034352768784e+01  -1.7263465597740716e+01  -1.6336957355980775e+01  -1.5946556228923413e+01  -1.6451747846316870e+01  -1.6476134055902989e+01  -1.4472580313892262e+01  -1.0935457247727644e+01  -7.9876368021814939e+00  -6.0265186692348891e+00  -2.0146808151690361e+00   8.0163390797194936e+00   2.5505537996776390e+01   4.7787314568418353e+01   6.8639540131510500e+01   8.1758985020608336e+01   8.5294009078541549e+01   8.2283381938562812e+01   7.6550409062053944e+01   6.9344678720567757e+01   6.0023044834365038e+01   4.8849934461625729e+01   3.8040291720134526e+01   2.9945383165919644e+01   2.5344624124046817e+01   2.3679402477458130e+01   2.3454441703713346e+01   2.2281261219059516e+01   1.7989994660541914e+01   1.0436193819072196e+01   2.1102305275848425e+00  -3.6577185659374600e+00  -5.5145847757485971e+00  -4.5324408328651318e+00  -1.8554008253732013e+00   2.7089378387782199e+00   9.2187555916040793e+00   1.5945566856263504e+01   2.0472163602212458e+01   2.2294516323769198e+01   2.2881411820375533e+01   2.2911237311157720e+01   2.1364859320014958e+01   1.7602082925628398e+01   1.2398805245993104e+01   6.8726800470759422e+00   1.6538594259187884e+00  -2.9437120155973195e+00  -6.5367981004048668e+00  -8.6437256178931250e+00  -9.1231101796868952e+00  -8.6648085339554566e+00  -8.4735847323575335e+00  -8.9963274994615468e+00  -9.3198443907122499e+00  -8.2504681130406006e+00  -5.5321797455350028e+00  -1.9690195798275383e+00   9.0573194795780632e-01   1.6386927935969133e+00   3.9409445444748448e-01
      -1.2097853394633344e+00  -2.3372661136012782e+00  -4.5868126352335468e+00  -7.7696106024369840e+00  -9.8847492594673341e+00  -8.4351485269365991e+00  -3.4815504396923176e+00   2.0318425367749775e+00   5.4688409099903446e+00   6.8155210844128202e+00   7.5289034993747936e+00   8.2470533843717533e+00   8.5005207895890198e+00   8.1730893089208134e+00   8.1217939375154362e+00   9.2472099093868128e+00   1.1682275429968531e+01   1.4843432519914167e+01   1.7625325319637877e+01   1.8827596749646286e+01   1.8262944365019820e+01   1.7257770010849043e+01   1.7139858491940043e+01   1.7319082568288064e+01   1.5832240015066093e+01   1.1889468934637401e+01   7.0260166885118247e+00   3.4084227701886709e+00   1.5903858359584313e+00   6.4628466394257855e-01   2.9208281149091886e-01   1.3680522046018677e+00   3.7148191371537109e+00   5.3471095100773640e+00   4.5999975157196396e+00   1.8618036691603201e+00  -1.1675467460557312e+00  -3.2014095470489683e+00  -4.1751621005792883e+00  -4.4441528031142692e+00  -3.7587484122597536e+00  -2.0282422842477690e+00  -4.8339177039881087e-01  -2.6711860399748077e-01  -3.8790549777032224e-01   8.5572500658487782e-01   3.0975713952792994e+00   4.3374149891908393e+00   3.2920941855788466e+00   3.9295487513328070e-01  -2.4783937608256150e+00  -3.2055601186298843e+00  -1.7160184560799154e+00  -6.3702741131976026e-01  -2.2383884758776662e+00  -5.5570116369254112e+00  -8.2084983887639105e+00  -9.5917739048230395e+00  -1.0331039070410487e+01  -1.0417944274770905e+01  -9.6648501662164410e+00  -8.3293893029282344e+00  -6.5549338022834966e+00  -4.3174230553074713e+00
      -1.9521686822118689e+01  -2.0828890952194818e+01  -2.1026487257169343e+01  -2.0644194340000958e+01  -2.0356661177644074e+01  -2.0486033910028034e+01  -2.0592315338305198e+01  -1.9336484142295909e+01  -1.5938473018577051e+01  -1.1921626681007268e+01  -9.9587671774048481e+00  -1.0450212320468967e+01  -1.0318268307692627e+01  -5.6239283444522234e+00   5.5321151308175374e+00   2.2417493712423962e+01   4.2078909484239603e+01   5.9973160754273493e+01   7.1702694488382562e+01   7.5072713982579657e+01   7.0924560394362075e+01   6.2663339872630232e+01   5.4356911122207862e+01   4.7989283986803173e+01   4.2765855626087045e+01   3.7486699590135927e+01   3.2734101027576685e+01   2.9993606241210717e+01   2.9069222310304095e+01   2.7645584279325117e+01   2.3767330134444784e+01   1.7501891196670105e+01   1.0097030989509490e+01   3.3105933448286935e+00  -4.4358692407392453e-01   4.2309373559202790e-01   4.6272362980750925e+00   9.2452802627883770e+00   1.2881829396441868e+01   1.6139200491488932e+01   1.9696338650752981e+01   2.2945254570047432e+01   2.4052713164028159e+01   2.1836262493236529e+01   1.7755371912217502e+01   1.4544324719156007e+01   1.3076166058655325e+01   1.2023365019266210e+01   9.9330468323536198e+00   6.5642773658552329e+00   2.6211689892425389e+00  -1.2463040312411424e+00  -4.6931152521918520e+00  -7.0610915480926959e+00  -7.8896098442295468e+00  -8.1377876309837323e+00  -9.2879431013150882e+00  -1.1118321847684978e+01  -1.1512230767538767e+01  -8.6570795004612968e+00  -3.0565564637606264e+00   2.6614202613936517e+00   5.7492933678780842e+00   5.1870636768943843e+00
      -1.2768528401242900e+01  -1.7338863074172309e+01  -2.0237569092156448e+01  -2.0576492513490894e+01  -1.9431123286614632e+01  -1.8641772316619317e+01  -1.8943170420368556e+01  -1.9167043655793719e+01  -1.7528873877656416e+01  -1.3999909814005159e+01  -1.0805366953545310e+01  -9.6142597372311247e+00  -8.9149462386606366e+00  -5.4471587606889269e+00   2.2906924892770721e+00   1.2948039993449356e+01   2.3895472387004183e+01   3.2785154492570712e+01   3.8224124421928316e+01   3.9996932598102731e+01   3.8917325110890907e+01   3.6404969952831500e+01   3.3603549356134771e+01   3.0482131529017348e+01   2.6418551514199979e+01   2.1753968035179085e+01   1.7865996396778279e+01   1.5605745867809237e+01   1.4445507979298107e+01   1.3294236667868901e+01   1.1519440091973062e+01   9.2364945825762419e+00   7.2532314388618229e+00   6.2602214003637382e+00   5.4987881290564387e+00   3.2320466512581985e+00  -9.0506248164303926e-01  -4.9456160363826012e+00  -6.3608335770462485e+00  -4.2734547891318062e+00   4.6309862256935785e-01   6.7021945441538548e+00   1.3111777285636606e+01   1.7514480785911530e+01   1.8248810290360364e+01   1.5766174142459235e+01   1.1757377659322158e+01   7.9733313460406592e+00   6.0274343310982506e+00   6.1770944745629279e+00   6.2936748428494491e+00   4.0368463533344734e+00  -4.2685170771143977e-02  -3.1865409383399417e+00  -3.8901205566515773e+00  -2.9576974863775090e+00  -1.4738094422502874e+00   4.0118819835881747e-01   2.6525695143418675e+00   4.6014650691595245e+00   5.1565376735843209e+00   4.0636905672697523e+00   2.6453975037372648e+00   2.3284476158012066e+00
      -1.3007839852110759e+01  -1.1025538159980380e+01  -1.0686371074924564e+01  -1.2744092667984233e+01  -1.5642265829843563e+01  -1.7140214084438682e+01  -1.6617034625505930e+01  -1.5300909110451405e+01  -1.4577809836556332e+01  -1.4955079002970214e+01  -1.5654258710135920e+01  -1.4407648124599390e+01  -9.0954740513659118e+00  -6.2982400317265430e-02   1.0332127909943164e+01   2.0193720207165487e+01   2.8972473785793127e+01   3.6317935037933722e+01   4.1210257687869564e+01   4.2631382536124164e+01   4.0972150948544574e+01   3.8156297398596216e+01   3.5717588235143509e+01   3.3207917004761363e+01   2.9304640994990272e+01   2.4112801189469973e+01   1.9353557646954098e+01   1.6308754098888908e+01   1.4516954476871140e+01   1.2392183920492100e+01   8.3965380716467521e+00   2.2326430594848903e+00  -4.3996278411324354e+00  -8.8062298459598374e+00  -9.2226405800100828e+00  -5.6733723455263254e+00   3.7975785593939992e-02   5.0496912837583618e+00   7.5061362215685907e+00   8.1297047935707027e+00   8.9880765155299258e+00   1.0960262021995606e+01   1.3066019970002715e+01   1.3738034632375317e+01   1.1896051602413694e+01   7.5099861905659733e+00   2.1853151873444245e+00  -1.4107524690116966e+00  -1.7374936024760426e+00   4.7474258642688838e-02   1.2777176662851293e+00   5.9151627918367655e-01  -9.4200465373474518e-01  -1.3835697125582160e+00  -1.1439767142326973e-02   2.0431818226269547e+00   2.8931387751496276e+00   1.7217225031925740e+00  -3.6057387583206468e-01  -1.4188595163897433e+00  -5.3401986141825453e-01   1.7331384545428845e+00   4.1028021413802733e+00   5.1530807234519207e+00
      -5.9423907462970762e+00  -6.6393370296983978e+00  -7.7903948031970467e+00  -9.0887616470108981e+00  -9.0730958856624682e+00  -7.3051960638397269e+00  -5.1092125692791504e+00  -4.1499019811620803e+00  -5.1825095304735012e+00  -7.4650822660744725e+00  -8.7858341576439987e+00  -6.9165862119378128e+00  -1.1152780425762092e+00   7.4456360751231196e+00   1.5504355240213883e+01   1.9891995742575336e+01   2.1880531164099967e+01   2.5705009499507383e+01   3.1274011045582068e+01   3.4010306181075244e+01   3.2770325333513270e+01   3.0948804502035770e+01   2.9539194246932787e+01   2.6197352223405726e+01   2.0544443634266784e+01   1.4590436095771173e+01   9.4132762585289544e+00   5.0726802461989733e+00   1.8845604581530462e+00  -2.0830751870538844e-01  -2.0248221543626945e+00  -3.9513021238798549e+00  -5.3937857883630116e+00  -5.7622214722998990e+00  -5.0964393510051575e+00  -3.6685468807689410e+00  -1.9925824037801600e+00  -1.3116773229001528e+00  -2.6369224209795075e+00  -4.8818299037998507e+00  -5.7575190263162872e+00  -4.9151597170028811e+00  -4.2580797535905948e+00  -5.3162000432783687e+00  -8.0260447395650854e+00  -1.1124603514599908e+01  -1.2372738496954803e+01  -9.9327640511974646e+00  -4.8837906274471230e+00  -8.7396451836104794e-01  -1.7270469647056230e-01  -1.5776571699924549e+00  -2.7982572746004415e+00  -3.1062409337303580e+00  -3.0845221509902201e+00  -3.0432684148630136e+00  -2.5011943998532136e+00  -1.1481969001182208e+00   3.2261647132979526e-01   7.8134574976169557e-01  -1.2641763242828769e-01  -1.4193228153058111e+00  -1.7047633076684265e+00  -8.4627935364228413e-01
      -3.3511583497758619e+00  -5.1854891048629845e+00  -8.2045548190539002e+00  -1.1321927344458468e+01  -1.3232156516018623e+01  -1.2891455905593750e+01  -1.0308983110741357e+01  -7.2770589395120400e+00  -6.3230641518260899e+00  -7.7668924711145673e+00  -8.5533116261484974e+00  -5.1754472720840532e+00   2.5464912688278725e+00   1.1150146477409665e+01   1.6749765390457195e+01   1.8459480819125350e+01   1.8853443625394785e+01   2.0968383227964456e+01   2.4745356436846293e+01   2.6900729147509413e+01   2.4586028405507879e+01   1.8423893823872788e+01   1.1687464461034605e+01   7.4694246602111161e+00   6.5063740610776568e+00   6.6399254866014452e+00   4.8727847407542608e+00   6.2948434274342469e-01  -3.8180925201354374e+00  -5.9745650071051903e+00  -5.0248037653094384e+00  -1.7240345099049890e+00   2.4707226972860665e+00   6.1789893211512830e+00   8.4606300098802265e+00   9.1837056122684562e+00   9.4175247215558571e+00   1.0361111194478536e+01   1.1355825197106681e+01   1.0443694556801294e+01   7.3275805488596522e+00   4.0729400267880624e+00   2.7849489919865249e+00   4.0449011920910980e+00   6.8903062423979726e+00   8.8579270913084418e+00   7.2409153007186635e+00   1.9815538314749910e+00  -3.4963983686766915e+00  -5.7025552070066450e+00  -4.6316183670653945e+00  -2.8570944089319230e+00  -2.1945685959883381e+00  -2.7295179820568904e+00  -4.1440094951781976e+00  -6.1739578627827907e+00  -7.9441831061071788e+00  -8.3008782660488869e+00  -7.1697557563545278e+00  -5.6642787357607363e+00  -4.5049090776747240e+00  -3.1919541675350729e+00  -9.9748944534117390e-01   1.8822644573222371e+00
       1.6022978144703672e+00  -1.4311786252489094e+00  -4.9990745496272506e+00  -7.1229734861916816e+00  -6.0298488474363783e+00  -1.8843154305695582e+00   2.8057312397445893e+00   5.3045500670547181e+00   5.2217607939007324e+00   4.6015143428315151e+00   5.3985890315998715e+00   7.6627280386075034e+00   1.0581969356003244e+01   1.3680704869130523e+01   1.6133038629367640e+01   1.6836134617835810e+01   1.5985757935422852e+01   1.5071121283266066e+01   1.5015214850097710e+01   1.5311877473734373e+01   1.4952636894507775e+01   1.3509895518873545e+01   1.1287529167684998e+01   8.8944719667341001e+00   6.8056307961237392e+00   5.1599082109694479e+00   3.9096368310365088e+00   3.0884796567205659e+00   2.8027781380901557e+00   2.7427378963657629e+00   1.8080112317158328e+00  -7.2093739358622477e-01  -3.2256888875692162e+00  -2.5485135905025946e+00   2.1938957914977939e+00   7.8096722519713868e+00   1.0108587362109473e+01   8.0119639048454889e+00   3.7875674256494771e+00  -1.7909623605085379e-01  -3.6408942410859053e+00  -7.2043954289116119e+00  -1.0036996806105844e+01  -1.0480191286127010e+01  -8.6104715538045582e+00  -6.5737607267403781e+00  -6.4789888225111065e+00  -8.9832745466307422e+00  -1.3098938079216277e+01  -1.6488189562785699e+01  -1.7072074337181476e+01  -1.4675862164708811e+01  -1.0393673234405423e+01  -5.7706460525680994e+00  -3.2783034075657795e+00  -4.8699351422351356e+00  -9.0537498769047904e+00  -1.1681066334269010e+01  -1.0244235531230570e+01  -5.9719523689580347e+00  -1.9644630163354910e+00  -6.8837039873644690e-01  -2.7600482002062674e+00  -6.6765535392302047e+00
      -2.6793902875688205e+00   4.3700049391893547e-01   5.2207839165959946e-01  -2.5289383815408999e+00  -6.4357928708025778e+00  -8.6801895727348093e+00  -8.6519420551714941e+00  -7.5157946794804973e+00  -6.9229481363763350e+00  -7.7994153444953156e+00  -9.3675453247907754e+00  -1.0006058503692298e+01  -9.5066143246051311e+00  -9.1193248923592503e+00  -9.0487355120456403e+00  -6.9972012375818347e+00  -1.6948530860574240e-01   1.0752688787226244e+01   2.0728963668504598e+01   2.4824092328193505e+01   2.2702194691381209e+01   1.8103111981612606e+01   1.5186220180215653e+01   1.5435411462559298e+01   1.6851176785915300e+01   1.6256911281706046e+01   1.2853830842714240e+01   8.7992906000849906e+00   6.3848237032232804e+00   6.1798137367968202e+00   7.8154545356284144e+00   1.0467840599577846e+01   1.2480989360982822e+01   1.2582337684046703e+01   1.1590145096205708e+01   1.1337803770903093e+01   1.2054473739609797e+01   1.2250402949861966e+01   1.0983877637654345e+01   8.6551288071765491e+00   5.8174031977868585e+00   2.9188587271482644e+00   8.3930412496566609e-01   2.3622649323786860e-01   7.1444423031664739e-01   1.2655853456754595e+00   1.3229336834759118e+00   1.3578720434847311e+00   2.1076821170226498e+00   3.0728270343869779e+00   2.7843524321077324e+00   9.0852714062149698e-01  -1.0017936404451253e+00  -1.3974751167516584e+00  -6.4988794902308711e-01  -5.0807738982896855e-01  -2.0685431131427148e+00  -4.6510643108399936e+00  -6.4865750445095642e+00  -6.4858500824649585e+00  -5.1351447719989922e+00  -3.4051840546128562e+00  -1.7184951490663052e+00  -5.0407691071619776e-01
      -3.6981497476642828e+00  -3.0390710048817726e+00  -1.4989922553639727e+00   1.0284003810800462e+00   3.6622719317496890e+00   4.7174258831055269e+00   3.5121017080473034e+00   2.1663525068616294e+00   3.8933497703894613e+00   8.9756839635313099e+00   1.4369540972251285e+01   1.7517537513748934e+01   1.8367043951422612e+01   1.7487527428352760e+01   1.4910602011410518e+01   1.1788546361447585e+01   1.0695666486764603e+01   1.2768873027185609e+01   1.6000928788239836e+01   1.7360605935279679e+01   1.5930200170467559e+01   1.3260280784655849e+01   1.0849461771852765e+01   8.4797732256321510e+00   5.3784098602866708e+00   1.7165101724011995e+00  -1.4089618767998704e+00  -2.7952639837436752e+00  -2.2276522503149407e+00  -9.5316276326730420e-01  -5.3299313777074875e-01  -1.5019739358173170e+00  -3.5515201972803432e+00  -6.3187612986855708e+00  -9.3137337609793054e+00  -1.1318424950944941e+01  -1.0816559293228659e+01  -7.7508782461822854e+00  -4.2600195254335116e+00  -2.8712335676010809e+00  -4.2724290569942225e+00  -6.8899881431600578e+00  -8.1742447353742449e+00  -6.8082430068650392e+00  -3.9299615464428408e+00  -1.4896688083453651e+00  -4.1155002553450060e-02   6.2487647569787685e-01  -2.9020287195242966e-01  -3.8853316012394776e+00  -9.2125322820605025e+00  -1.3050926297558480e+01  -1.2737108773295276e+01  -8.7818443279864002e+00  -4.3127261410662650e+00  -2.3302722768023965e+00  -3.5325752167188584e+00  -6.2805892811261916e+00  -8.5016614679139195e+00  -9.4759820546177966e+00  -9.7204496390574704e+00  -9.7546714698047463e+00  -9.5752380341344967e+00  -9.3036539221877597e+00
      -1.7414985024478632e+01  -1.7710392269365567e+01  -1.8830345987045778e+01  -2.0905808395741552e+01  -2.3089781413981267e+01  -2.3899680760146289e+01  -2.2409641713666709e+01  -1.8930485800816182e+01  -1.4601034805586801e+01  -1.0520204067883871e+01  -7.1082264501151942e+00  -4.1771499408845667e+00  -1.2985932549142798e+00   2.5888198678848857e+00   9.5645053119594277e+00   2.1215332248500019e+01   3.6152478620496041e+01   5.0021402817940384e+01   5.8774208038896376e+01   6.1292360127806575e+01   5.8550762166546200e+01   5.1882091757242364e+01   4.2968986888033839e+01   3.3918424522864342e+01   2.6125731850668569e+01   1.9638134139516286e+01   1.3840450777647449e+01   8.3996525004490294e+00   3.8021795817073860e+00   1.1190628236490561e+00   7.5962516407305214e-01   1.4058807055741134e+00   8.2163118862213358e-01  -2.0188943881722592e+00  -6.0499550558624495e+00  -9.4755634193659208e+00  -1.1248638187248371e+01  -1.0876954767354995e+01  -7.8849814096440030e+00  -2.2277330312359691e+00   5.4138464272246747e+00   1.3796881454212489e+01   2.0984396320719696e+01   2.4893133374375935e+01   2.4942654855570076e+01   2.2326104353812248e+01   1.8202892776374178e+01   1.2706268587571685e+01   5.8859262076676249e+00  -1.1947559660524161e+00  -6.5689254319385437e+00  -8.7772879515580264e+00  -8.1899879887551386e+00  -6.5832072140625550e+00  -5.4688753261733805e+00  -4.9175403512819100e+00  -3.8699212831075047e+00  -1.6574438279084263e+00   8.5455765389835547e-01   2.0911729100665450e+00   1.8202474430112003e+00   1.6089416985684442e+00   2.9743808071641795e+00   5.8948563383564005e+00
      -3.3338514474187440e+00  -4.1130389374748058e+00  -5.1547704706642552e+00  -5.4193898987702704e+00  -4.0201652371671948e+00  -1.8076212696539353e+00  -8.9324938220319039e-01  -1.9868709786288514e+00  -3.3105995821020633e+00  -2.6515298234057667e+00   5.1903435078819071e-01   4.9231535819829615e+00   8.5469168923530585e+00   1.0071331008104004e+01   9.9283440319809362e+00   9.9281856288240444e+00   1.1689028944723646e+01   1.5180332000597723e+01   1.8495089144282854e+01   1.9468749633612084e+01   1.7774035163293483e+01   1.4795935112020807e+01   1.1632060118366278e+01   8.3107388002809923e+00   4.7969367455220766e+00   1.5228719476515731e+00  -1.2245839851149136e+00  -3.6724142837818849e+00  -6.1258400070447641e+00  -8.3043113945111795e+00  -9.2382296924864278e+00  -8.2818369343941640e+00  -6.3024540208842872e+00  -5.1153884354297716e+00  -5.4012598664097871e+00  -5.9086039260782988e+00  -5.2469498739198066e+00  -3.8701397078727497e+00  -3.4699836914403539e+00  -4.6999263838482115e+00  -6.4704904874754252e+00  -7.1253409419073517e+00  -5.1911313857217234e+00  -5.8248621615308416e-02   6.3735411041921815e+00   9.9842526040059774e+00   8.0901302020184040e+00   2.0765046689514737e+00  -4.0814624682782448e+00  -6.9335386550902367e+00  -5.3396830806898752e+00  -6.1569207664081205e-01   4.2371402928715165e+00   6.1453027828107594e+00   4.0729416305299182e+00  -5.6892224917549727e-01  -5.7266782214867291e+00  -9.4817569691134302e+00  -9.4466837493039364e+00  -4.4716815460577326e+00   2.5480231258059796e+00   6.3862725120871939e+00   4.7626827206150235e+00   2.7250354238558483e-01
      -6.6519531318265823e+00  -5.1307688760840637e+00  -1.5990450796308475e+00   3.2882196698155921e+00   7.5434887899979923e+00   9.8601744151731392e+00   1.0431338476778441e+01   1.0132207096708374e+01   9.8633193310657052e+00   1.0117849589142009e+01   1.0556320935364319e+01   1.0529237565608828e+01   1.0064399849178292e+01   9.8902096970315743e+00   1.0802538876610429e+01   1.3065613777540369e+01   1.5904751019353327e+01   1.7996878960824926e+01   1.8887262226773334e+01   1.9137392198836288e+01   1.8914272666242024e+01   1.7510508054314936e+01   1.4350155185280446e+01   9.5073777371400769e+00   3.7350791719348706e+00  -9.6044902572999469e-01  -2.1523757668680674e+00   1.2726446256922297e-01   2.7975425794223376e+00   3.1310049165812357e+00   1.4400885075259127e+00  -9.4699453877260531e-03   5.2252907539724913e-01   2.9730000242572210e+00   5.8714424054021102e+00   7.9092364851498598e+00   8.7344188791480644e+00   7.9030314193993458e+00   4.5624392185691374e+00  -5.8583551307959691e-01  -4.3527827486539321e+00  -3.8542689679266942e+00  -3.3407572539104535e-02   3.0190256438646816e+00   2.5671043830395353e+00  -2.7379787916712028e-01  -2.8159393124422332e+00  -3.9496120934673513e+00  -4.3869096765280995e+00  -4.8326892938827388e+00  -4.8327167767960031e+00  -3.6048592531173460e+00  -1.4049856452797012e+00   6.1244691414035868e-01   1.4045344432842721e+00   4.0378165896259843e-01  -2.2594911404427971e+00  -5.2758916592334257e+00  -6.6711436023807975e+00  -5.4988956630405870e+00  -2.7595297745975564e+00  -4.3083978002099643e-01   1.2310669289941012e-02  -1.8074724020788719e+00
      -2.5417468582312877e+00  -1.9298735695592950e-01   1.4524591906329190e+00   1.4111623962585536e+00  -3.3194621049731687e-01  -2.5702905387021753e+00  -3.3145376224302021e+00  -1.6531002526600589e+00   1.5085470936791965e+00   4.9384825610490894e+00   8.1158180601966112e+00   1.0592317046651424e+01   1.1720352844683582e+01   1.1731547138555802e+01   1.1880590058094510e+01   1.2779413370052000e+01   1.4075646883094780e+01   1.5918248275430194e+01   1.8326489365386571e+01   1.9316206123129387e+01   1.6690629155310674e+01   1.1661681239753394e+01   8.0053968635282153e+00   7.5533291045585287e+00   8.4805002524471416e+00   7.9431009917907147e+00   4.7893096294793169e+00  -2.9205326061448478e-01  -5.9299905729305458e+00  -1.0889025474342041e+01  -1.4253802911511571e+01  -1.5424425634403272e+01  -1.4219613607718578e+01  -1.0988129270893715e+01  -6.7271231240914489e+00  -2.8803629271713325e+00  -5.3221218228894118e-01  -1.4819897860306691e-01  -2.2309622192950278e+00  -6.7555441266626337e+00  -1.1288796495269558e+01  -1.1879369455774969e+01  -7.6695223775198915e+00  -2.7345250210875287e+00  -1.2464522929563033e+00  -2.5982520939275595e+00  -3.2907753012277650e+00  -2.0946248103194671e+00  -1.2670396502483379e+00  -2.9734875330694628e+00  -6.1944576306095183e+00  -8.0259110770747917e+00  -7.0406747672630017e+00  -4.3818830819913117e+00  -2.2337026899490038e+00  -1.7757039249321303e+00  -2.1690903031406541e+00  -1.7112483834128271e+00  -2.2443726961269805e-01   7.8312271984712389e-01   7.2340304513709708e-01   9.8143716696324113e-01   2.5456848227724063e+00   4.1446278797649949e+00
       5.9509298374097952e+00   5.0678810528340978e+00   4.6985100256137695e+00   5.6806474798139019e+00   7.1270807068985986e+00   7.3896907513926973e+00   6.1956015245970972e+00   5.5852197604153861e+00   7.9143806050507850e+00   1.2903336221013145e+01   1.7478615210727796e+01   1.8733526717119613e+01   1.6557524998442712e+01   1.3237833414525580e+01   1.1194587213855545e+01   1.1742137197718762e+01   1.5292487123411739e+01   2.1089872768798958e+01   2.6615090233197662e+01   2.9112067948790987e+01   2.8381112807191901e+01   2.6745309556602031e+01   2.6230622139582557e+01   2.6873626067912664e+01   2.7379954661291766e+01   2.6724918542001461e+01   2.5352283105591539e+01   2.4558243711441179e+01   2.4278527981980282e+01   2.2429473534878777e+01   1.7501264178786681e+01   1.0987541340332779e+01   5.6451288588220212e+00   1.5426875751835269e+00  -4.2209755312065607e+00  -1.3196792287091672e+01  -2.2788705236686901e+01  -2.9082273423343587e+01  -3.1121693514905303e+01  -3.0849786691440926e+01  -2.9901975701231901e+01  -2.8187902353798506e+01  -2.4897002861955549e+01  -1.9629404309588349e+01  -1.3381778061564034e+01  -8.8077830839680740e+00  -8.0332756560395637e+00  -1.0120926859300432e+01  -1.2004163217689651e+01  -1.1842485990658060e+01  -1.0575962516702482e+01  -1.0149295488895902e+01  -1.0971618377841839e+01  -1.1645676947992518e+01  -1.0824020132311130e+01  -8.7473035146007145e+00  -6.7816366706045397e+00  -5.8695256679524599e+00  -6.1984801392743734e+00  -7.8537905857434129e+00  -1.0416960930281348e+01  -1.2429989647191839e+01  -1.2650740393840184e+01  -1.1521198276477271e+01
    ]
)
check(joinpath(dirname(@__FILE__), "big_endian.mat"), result)

# test reading of mxOBJECT_CLASS (#57)
let objtestfile = "obj.mat"
    vars = matread(joinpath(dirname(@__FILE__), objtestfile))
    # check if all variables were read
    for key in ("A", "tm", "signal")
        @test key in keys(vars)
    end
    # check if class name was read correctly
    @test vars["A"].class == "Assoc"
end

# test reading of a Matlab figure
let objtestfile = "figure.fig"
    vars = matread(joinpath(dirname(@__FILE__), objtestfile))
    @test "hgS_070000" in keys(vars)
    @test vars["hgS_070000"]["handle"] == 1.0
    @test vars["hgS_070000"]["type"] == "figure"
end

# test reading file containing Matlab function handle, table, and datetime objects
let objtestfile = "function_handles.mat"
    vars = matread(joinpath(dirname(@__FILE__), "v7.3", objtestfile))
    @test "sin" in keys(vars)
    @test typeof(vars["sin"]) == Dict{String, Any}
    @test Set(keys(vars["sin"])) == Set(["function_handle", "sentinel", "separator", "matlabroot"])
    @test "anonymous" in keys(vars)
    @test typeof(vars["anonymous"]) == Dict{String, Any}
    @test Set(keys(vars["anonymous"])) == Set(["function_handle", "sentinel", "separator", "matlabroot"])
end

for format in ["v7", "v7.3"]
    @testset "struct_table_datetime $format" begin
    let objtestfile = "struct_table_datetime.mat"
        filepath = joinpath(dirname(@__FILE__), format, objtestfile)

        # make sure read(matopen(filepath), ::String) works
        fid = matopen(filepath)
        @test haskey(fid, "s")
        var_s = read(fid, "s")
        @test haskey(var_s, "testTable")
        @test haskey(var_s, "testDatetime")
        close(fid)

        # matread interface
        vars = matread(filepath)["s"]
        @test haskey(vars, "testTable")
        t = vars["testTable"]
        @test t isa MatlabTable
        @test t.names == [:FlightNum, :Customer, :Date, :Rating, :Comment]
        @test t[:Date] isa Vector{DateTime}
        @test t[:Rating] isa AbstractVector{String}
        @test all(x->length(x)==3, t.columns)

        # using Nothing will keep the MatlabOpaque
        vars = matread(filepath; table=Nothing)["s"]
        t = vars["testTable"]
        @test Set(keys(t)) == Set(["props", "varnames", "nrows", "data", "rownames", "ndims", "nvars"])
        @test t.class == "table"
        @test t["ndims"] === 2.0
        @test t["nvars"] === 5.0
        @test t["nrows"] === 3.0
        @test t["data"][1] == reshape([1261.0, 547.0, 3489.0], 3, 1)
        @test t["data"][2] isa Matrix{String}
        @test t["data"][3] isa Matrix{DateTime}
        @test t["data"][4] isa AbstractMatrix{String}
        @test t["data"][5] isa Matrix{String}
        @test all(x->size(x)==(3,1), t["data"])

        @test "testDatetime" in keys(vars)
        dt = vars["testDatetime"]
        @test dt isa DateTime
        @test dt - DateTime(2019, 12, 2, 16, 42, 49) < Second(1)

        # test no conversion at all
        vars = matread(filepath; convert_opaque=false)["s"]
        t = vars["testTable"]
        @test t isa MatlabOpaque
        @test t["data"][3] isa MatlabOpaque
    end
    end

    @testset "user defined classdef $format" begin
    let objtestfile = "user_defined_classdefs.mat"
        filepath = joinpath(dirname(@__FILE__), format, objtestfile)

        vars = matread(filepath)
        @test haskey(vars, "obj_no_vals")
        obj_no_vals = vars["obj_no_vals"]
        @test obj_no_vals isa MatlabOpaque
        @test obj_no_vals.class == "TestClasses.BasicClass"
        @test obj_no_vals["a"] isa Matrix{Float64}

        @test haskey(vars, "obj_with_vals")
        obj_with_vals = vars["obj_with_vals"]
        @test obj_with_vals isa MatlabOpaque
        @test obj_with_vals.class == "TestClasses.BasicClass"
        @test obj_with_vals["a"] == 10.0

        @test haskey(vars, "obj_with_default_val")
        obj_with_default_val = vars["obj_with_default_val"]
        @test obj_with_default_val isa MatlabOpaque
        @test obj_with_default_val.class == "TestClasses.DefaultClass"
        @test obj_with_default_val["a"] == "Default String"
        @test obj_with_default_val["b"] == 10.0

        @test haskey(vars, "obj_array")
        obj_array = vars["obj_array"]
        @test obj_array isa Array{MatlabOpaque}
        @test size(obj_array) == (2, 2)
        @test obj_array[1, 1] isa MatlabOpaque
        @test obj_array[1, 1]["a"] == 1.0
        @test obj_array[1, 2]["a"] == 2.0

        @test haskey(vars, "obj_with_nested_props")
        obj_with_nested_props = vars["obj_with_nested_props"]
        @test obj_with_nested_props isa MatlabOpaque
        @test obj_with_nested_props.class == "TestClasses.BasicClass"
        @test obj_with_nested_props["a"] isa MatlabOpaque
        @test obj_with_nested_props["a"]["a"] == 1.0

        @test obj_with_nested_props["b"] isa Matrix{Any}
        @test obj_with_nested_props["b"][1] isa MatlabOpaque
        @test obj_with_nested_props["b"][1]["b"] == "Obj1"

        @test obj_with_nested_props["c"] isa Dict{String, Any}
        @test obj_with_nested_props["c"]["InnerProp"] isa MatlabOpaque
        @test obj_with_nested_props["c"]["InnerProp"]["a"] == 2.0

        @test haskey(vars, "obj_handle_1")
        @test haskey(vars, "obj_handle_2")
        obj_handle_1 = vars["obj_handle_1"]
        obj_handle_2 = vars["obj_handle_2"]
        @test obj_handle_1 === obj_handle_2
        @test obj_handle_1 isa MatlabOpaque

    end
    end

    @testset "dynamic property" begin
    let objtestfile = "dynamicprops.mat"
        filepath = joinpath(dirname(@__FILE__), format, objtestfile)

        vars = matread(filepath)
        @test haskey(vars, "obj")
        obj = vars["obj"]
        @test obj isa MatlabOpaque
        @test obj.class == "TestClasses.BasicDynamic"
        @test haskey(obj, "__dynamic_property_1__")
        @test obj["__dynamic_property_1__"]["Name"] == "DynamicData"
        @test obj["__dynamic_property_1__"]["DynamicValue_"] == 42.0
    end
    end

end

# test reading of old-style Matlab object in v7.3 format
let objtestfile = "old_class.mat"
    vars = matread(joinpath(dirname(@__FILE__), "v7.3", objtestfile))
    @test "tc_old" in keys(vars)
    @test "foo" in keys(vars["tc_old"])
    @test vars["tc_old"].class == "TestClassOld"
end

let objtestfile = "old_class_array.mat"
    vars = matread(joinpath(dirname(@__FILE__), "v7.3", objtestfile))
    c_arr = vars["class_arr"]
    @test c_arr isa MatlabStructArray
    @test c_arr.class == "TestClassOld"
    @test c_arr["foo"] == Any[5.0 "test"]

    vars = matread(joinpath(dirname(@__FILE__), "v7", objtestfile))
    c_arr = vars["class_arr"]
    @test c_arr isa MatlabStructArray
    @test c_arr.class == "TestClassOld"
    @test c_arr["foo"] == Any[5.0 "test"]
end


