#include "glx.h"

#include <xcb/xcb.h>
#include <xcb/xcbext.h>

#include "base/logging.h"
#include "base/posix/eintr_wrapper.h"
#include "ui/gfx/x/xproto_internal.h"

namespace x11 {

Glx::Glx(x11::Connection* connection,
    const x11::QueryExtensionReply& info)
    : connection_(connection), info_(info) {}

template <> COMPONENT_EXPORT(X11)
void ReadEvent<Glx::PbufferClobberEvent>(
    Glx::PbufferClobberEvent* event_, ReadBuffer* buffer) {
  auto& buf = *buffer;

  auto& sequence = (*event_).sequence;
  auto& event_type = (*event_).event_type;
  auto& draw_type = (*event_).draw_type;
  auto& drawable = (*event_).drawable;
  auto& b_mask = (*event_).b_mask;
  auto& aux_buffer = (*event_).aux_buffer;
  auto& x = (*event_).x;
  auto& y = (*event_).y;
  auto& width = (*event_).width;
  auto& height = (*event_).height;
  auto& count = (*event_).count;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // event_type
  Read(&event_type, &buf);

  // draw_type
  Read(&draw_type, &buf);

  // drawable
  Read(&drawable, &buf);

  // b_mask
  Read(&b_mask, &buf);

  // aux_buffer
  Read(&aux_buffer, &buf);

  // x
  Read(&x, &buf);

  // y
  Read(&y, &buf);

  // width
  Read(&width, &buf);

  // height
  Read(&height, &buf);

  // count
  Read(&count, &buf);

  // pad1
  Pad(&buf, 4);

  DCHECK_LE(buf.offset, 32ul);
}

template <> COMPONENT_EXPORT(X11)
void ReadEvent<Glx::BufferSwapCompleteEvent>(
    Glx::BufferSwapCompleteEvent* event_, ReadBuffer* buffer) {
  auto& buf = *buffer;

  auto& sequence = (*event_).sequence;
  auto& event_type = (*event_).event_type;
  auto& drawable = (*event_).drawable;
  auto& ust_hi = (*event_).ust_hi;
  auto& ust_lo = (*event_).ust_lo;
  auto& msc_hi = (*event_).msc_hi;
  auto& msc_lo = (*event_).msc_lo;
  auto& sbc = (*event_).sbc;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // event_type
  Read(&event_type, &buf);

  // pad1
  Pad(&buf, 2);

  // drawable
  Read(&drawable, &buf);

  // ust_hi
  Read(&ust_hi, &buf);

  // ust_lo
  Read(&ust_lo, &buf);

  // msc_hi
  Read(&msc_hi, &buf);

  // msc_lo
  Read(&msc_lo, &buf);

  // sbc
  Read(&sbc, &buf);

  DCHECK_LE(buf.offset, 32ul);
}

Future<void>
Glx::Render(
    const Glx::RenderRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& data = request.data;
  size_t data_len = data.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 1;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // data
  DCHECK_EQ(static_cast<size_t>(data_len), data.size());
  for (auto& data_elem : data) {
    // data_elem
    buf.Write(&data_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::RenderLarge(
    const Glx::RenderLargeRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& request_num = request.request_num;
  auto& request_total = request.request_total;
  uint32_t data_len{};
  auto& data = request.data;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 2;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // request_num
  buf.Write(&request_num);

  // request_total
  buf.Write(&request_total);

  // data_len
  data_len = data.size();
  buf.Write(&data_len);

  // data
  DCHECK_EQ(static_cast<size_t>(data_len), data.size());
  for (auto& data_elem : data) {
    // data_elem
    buf.Write(&data_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::CreateContext(
    const Glx::CreateContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context = request.context;
  auto& visual = request.visual;
  auto& screen = request.screen;
  auto& share_list = request.share_list;
  auto& is_direct = request.is_direct;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 3;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context
  buf.Write(&context);

  // visual
  buf.Write(&visual);

  // screen
  buf.Write(&screen);

  // share_list
  buf.Write(&share_list);

  // is_direct
  buf.Write(&is_direct);

  // pad0
  Pad(&buf, 3);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::DestroyContext(
    const Glx::DestroyContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context = request.context;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 4;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context
  buf.Write(&context);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<Glx::MakeCurrentReply>
Glx::MakeCurrent(
    const Glx::MakeCurrentRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& drawable = request.drawable;
  auto& context = request.context;
  auto& old_context_tag = request.old_context_tag;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 5;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // drawable
  buf.Write(&drawable);

  // context
  buf.Write(&context);

  // old_context_tag
  buf.Write(&old_context_tag);

  Align(&buf, 4);

  return x11::SendRequest<Glx::MakeCurrentReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::MakeCurrentReply>
ReadReply<Glx::MakeCurrentReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::MakeCurrentReply>();

  auto& sequence = (*reply).sequence;
  auto& context_tag = (*reply).context_tag;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_tag
  Read(&context_tag, &buf);

  // pad1
  Pad(&buf, 20);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::IsDirectReply>
Glx::IsDirect(
    const Glx::IsDirectRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context = request.context;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 6;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context
  buf.Write(&context);

  Align(&buf, 4);

  return x11::SendRequest<Glx::IsDirectReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::IsDirectReply>
ReadReply<Glx::IsDirectReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::IsDirectReply>();

  auto& sequence = (*reply).sequence;
  auto& is_direct = (*reply).is_direct;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // is_direct
  Read(&is_direct, &buf);

  // pad1
  Pad(&buf, 23);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::QueryVersionReply>
Glx::QueryVersion(
    const Glx::QueryVersionRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& major_version = request.major_version;
  auto& minor_version = request.minor_version;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 7;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // major_version
  buf.Write(&major_version);

  // minor_version
  buf.Write(&minor_version);

  Align(&buf, 4);

  return x11::SendRequest<Glx::QueryVersionReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::QueryVersionReply>
ReadReply<Glx::QueryVersionReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::QueryVersionReply>();

  auto& sequence = (*reply).sequence;
  auto& major_version = (*reply).major_version;
  auto& minor_version = (*reply).minor_version;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // major_version
  Read(&major_version, &buf);

  // minor_version
  Read(&minor_version, &buf);

  // pad1
  Pad(&buf, 16);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
Glx::WaitGL(
    const Glx::WaitGLRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 8;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::WaitX(
    const Glx::WaitXRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 9;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::CopyContext(
    const Glx::CopyContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& src = request.src;
  auto& dest = request.dest;
  auto& mask = request.mask;
  auto& src_context_tag = request.src_context_tag;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 10;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // src
  buf.Write(&src);

  // dest
  buf.Write(&dest);

  // mask
  buf.Write(&mask);

  // src_context_tag
  buf.Write(&src_context_tag);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::SwapBuffers(
    const Glx::SwapBuffersRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& drawable = request.drawable;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 11;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // drawable
  buf.Write(&drawable);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::UseXFont(
    const Glx::UseXFontRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& font = request.font;
  auto& first = request.first;
  auto& count = request.count;
  auto& list_base = request.list_base;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 12;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // font
  buf.Write(&font);

  // first
  buf.Write(&first);

  // count
  buf.Write(&count);

  // list_base
  buf.Write(&list_base);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::CreateGLXPixmap(
    const Glx::CreateGLXPixmapRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& screen = request.screen;
  auto& visual = request.visual;
  auto& pixmap = request.pixmap;
  auto& glx_pixmap = request.glx_pixmap;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 13;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // screen
  buf.Write(&screen);

  // visual
  buf.Write(&visual);

  // pixmap
  buf.Write(&pixmap);

  // glx_pixmap
  buf.Write(&glx_pixmap);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<Glx::GetVisualConfigsReply>
Glx::GetVisualConfigs(
    const Glx::GetVisualConfigsRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& screen = request.screen;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 14;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // screen
  buf.Write(&screen);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetVisualConfigsReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetVisualConfigsReply>
ReadReply<Glx::GetVisualConfigsReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetVisualConfigsReply>();

  auto& sequence = (*reply).sequence;
  auto& num_visuals = (*reply).num_visuals;
  auto& num_properties = (*reply).num_properties;
  auto& property_list = (*reply).property_list;
  size_t property_list_len = property_list.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // num_visuals
  Read(&num_visuals, &buf);

  // num_properties
  Read(&num_properties, &buf);

  // pad1
  Pad(&buf, 16);

  // property_list
  property_list.resize(length);
  for (auto& property_list_elem : property_list) {
    // property_list_elem
    Read(&property_list_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
Glx::DestroyGLXPixmap(
    const Glx::DestroyGLXPixmapRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& glx_pixmap = request.glx_pixmap;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 15;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // glx_pixmap
  buf.Write(&glx_pixmap);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::VendorPrivate(
    const Glx::VendorPrivateRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& vendor_code = request.vendor_code;
  auto& context_tag = request.context_tag;
  auto& data = request.data;
  size_t data_len = data.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 16;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // vendor_code
  buf.Write(&vendor_code);

  // context_tag
  buf.Write(&context_tag);

  // data
  DCHECK_EQ(static_cast<size_t>(data_len), data.size());
  for (auto& data_elem : data) {
    // data_elem
    buf.Write(&data_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<Glx::VendorPrivateWithReplyReply>
Glx::VendorPrivateWithReply(
    const Glx::VendorPrivateWithReplyRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& vendor_code = request.vendor_code;
  auto& context_tag = request.context_tag;
  auto& data = request.data;
  size_t data_len = data.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 17;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // vendor_code
  buf.Write(&vendor_code);

  // context_tag
  buf.Write(&context_tag);

  // data
  DCHECK_EQ(static_cast<size_t>(data_len), data.size());
  for (auto& data_elem : data) {
    // data_elem
    buf.Write(&data_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<Glx::VendorPrivateWithReplyReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::VendorPrivateWithReplyReply>
ReadReply<Glx::VendorPrivateWithReplyReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::VendorPrivateWithReplyReply>();

  auto& sequence = (*reply).sequence;
  auto& retval = (*reply).retval;
  auto& data1 = (*reply).data1;
  size_t data1_len = data1.size();
  auto& data2 = (*reply).data2;
  size_t data2_len = data2.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // retval
  Read(&retval, &buf);

  // data1
  for (auto& data1_elem : data1) {
    // data1_elem
    Read(&data1_elem, &buf);

  }

  // data2
  data2.resize((length) * (4));
  for (auto& data2_elem : data2) {
    // data2_elem
    Read(&data2_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::QueryExtensionsStringReply>
Glx::QueryExtensionsString(
    const Glx::QueryExtensionsStringRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& screen = request.screen;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 18;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // screen
  buf.Write(&screen);

  Align(&buf, 4);

  return x11::SendRequest<Glx::QueryExtensionsStringReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::QueryExtensionsStringReply>
ReadReply<Glx::QueryExtensionsStringReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::QueryExtensionsStringReply>();

  auto& sequence = (*reply).sequence;
  auto& n = (*reply).n;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // pad2
  Pad(&buf, 16);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::QueryServerStringReply>
Glx::QueryServerString(
    const Glx::QueryServerStringRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& screen = request.screen;
  auto& name = request.name;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 19;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // screen
  buf.Write(&screen);

  // name
  buf.Write(&name);

  Align(&buf, 4);

  return x11::SendRequest<Glx::QueryServerStringReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::QueryServerStringReply>
ReadReply<Glx::QueryServerStringReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::QueryServerStringReply>();

  auto& sequence = (*reply).sequence;
  uint32_t str_len{};
  auto& string = (*reply).string;
  size_t string_len = string.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // str_len
  Read(&str_len, &buf);

  // pad2
  Pad(&buf, 16);

  // string
  string.resize(str_len);
  for (auto& string_elem : string) {
    // string_elem
    Read(&string_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
Glx::ClientInfo(
    const Glx::ClientInfoRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& major_version = request.major_version;
  auto& minor_version = request.minor_version;
  uint32_t str_len{};
  auto& string = request.string;
  size_t string_len = string.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 20;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // major_version
  buf.Write(&major_version);

  // minor_version
  buf.Write(&minor_version);

  // str_len
  str_len = string.size();
  buf.Write(&str_len);

  // string
  DCHECK_EQ(static_cast<size_t>(str_len), string.size());
  for (auto& string_elem : string) {
    // string_elem
    buf.Write(&string_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<Glx::GetFBConfigsReply>
Glx::GetFBConfigs(
    const Glx::GetFBConfigsRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& screen = request.screen;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 21;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // screen
  buf.Write(&screen);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetFBConfigsReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetFBConfigsReply>
ReadReply<Glx::GetFBConfigsReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetFBConfigsReply>();

  auto& sequence = (*reply).sequence;
  auto& num_FB_configs = (*reply).num_FB_configs;
  auto& num_properties = (*reply).num_properties;
  auto& property_list = (*reply).property_list;
  size_t property_list_len = property_list.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // num_FB_configs
  Read(&num_FB_configs, &buf);

  // num_properties
  Read(&num_properties, &buf);

  // pad1
  Pad(&buf, 16);

  // property_list
  property_list.resize(length);
  for (auto& property_list_elem : property_list) {
    // property_list_elem
    Read(&property_list_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
Glx::CreatePixmap(
    const Glx::CreatePixmapRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& screen = request.screen;
  auto& fbconfig = request.fbconfig;
  auto& pixmap = request.pixmap;
  auto& glx_pixmap = request.glx_pixmap;
  auto& num_attribs = request.num_attribs;
  auto& attribs = request.attribs;
  size_t attribs_len = attribs.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 22;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // screen
  buf.Write(&screen);

  // fbconfig
  buf.Write(&fbconfig);

  // pixmap
  buf.Write(&pixmap);

  // glx_pixmap
  buf.Write(&glx_pixmap);

  // num_attribs
  buf.Write(&num_attribs);

  // attribs
  DCHECK_EQ(static_cast<size_t>((num_attribs) * (2)), attribs.size());
  for (auto& attribs_elem : attribs) {
    // attribs_elem
    buf.Write(&attribs_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::DestroyPixmap(
    const Glx::DestroyPixmapRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& glx_pixmap = request.glx_pixmap;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 23;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // glx_pixmap
  buf.Write(&glx_pixmap);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::CreateNewContext(
    const Glx::CreateNewContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context = request.context;
  auto& fbconfig = request.fbconfig;
  auto& screen = request.screen;
  auto& render_type = request.render_type;
  auto& share_list = request.share_list;
  auto& is_direct = request.is_direct;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 24;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context
  buf.Write(&context);

  // fbconfig
  buf.Write(&fbconfig);

  // screen
  buf.Write(&screen);

  // render_type
  buf.Write(&render_type);

  // share_list
  buf.Write(&share_list);

  // is_direct
  buf.Write(&is_direct);

  // pad0
  Pad(&buf, 3);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<Glx::QueryContextReply>
Glx::QueryContext(
    const Glx::QueryContextRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context = request.context;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 25;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context
  buf.Write(&context);

  Align(&buf, 4);

  return x11::SendRequest<Glx::QueryContextReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::QueryContextReply>
ReadReply<Glx::QueryContextReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::QueryContextReply>();

  auto& sequence = (*reply).sequence;
  auto& num_attribs = (*reply).num_attribs;
  auto& attribs = (*reply).attribs;
  size_t attribs_len = attribs.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // num_attribs
  Read(&num_attribs, &buf);

  // pad1
  Pad(&buf, 20);

  // attribs
  attribs.resize((num_attribs) * (2));
  for (auto& attribs_elem : attribs) {
    // attribs_elem
    Read(&attribs_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::MakeContextCurrentReply>
Glx::MakeContextCurrent(
    const Glx::MakeContextCurrentRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& old_context_tag = request.old_context_tag;
  auto& drawable = request.drawable;
  auto& read_drawable = request.read_drawable;
  auto& context = request.context;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 26;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // old_context_tag
  buf.Write(&old_context_tag);

  // drawable
  buf.Write(&drawable);

  // read_drawable
  buf.Write(&read_drawable);

  // context
  buf.Write(&context);

  Align(&buf, 4);

  return x11::SendRequest<Glx::MakeContextCurrentReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::MakeContextCurrentReply>
ReadReply<Glx::MakeContextCurrentReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::MakeContextCurrentReply>();

  auto& sequence = (*reply).sequence;
  auto& context_tag = (*reply).context_tag;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // context_tag
  Read(&context_tag, &buf);

  // pad1
  Pad(&buf, 20);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
Glx::CreatePbuffer(
    const Glx::CreatePbufferRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& screen = request.screen;
  auto& fbconfig = request.fbconfig;
  auto& pbuffer = request.pbuffer;
  auto& num_attribs = request.num_attribs;
  auto& attribs = request.attribs;
  size_t attribs_len = attribs.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 27;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // screen
  buf.Write(&screen);

  // fbconfig
  buf.Write(&fbconfig);

  // pbuffer
  buf.Write(&pbuffer);

  // num_attribs
  buf.Write(&num_attribs);

  // attribs
  DCHECK_EQ(static_cast<size_t>((num_attribs) * (2)), attribs.size());
  for (auto& attribs_elem : attribs) {
    // attribs_elem
    buf.Write(&attribs_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::DestroyPbuffer(
    const Glx::DestroyPbufferRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& pbuffer = request.pbuffer;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 28;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // pbuffer
  buf.Write(&pbuffer);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<Glx::GetDrawableAttributesReply>
Glx::GetDrawableAttributes(
    const Glx::GetDrawableAttributesRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& drawable = request.drawable;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 29;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // drawable
  buf.Write(&drawable);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetDrawableAttributesReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetDrawableAttributesReply>
ReadReply<Glx::GetDrawableAttributesReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetDrawableAttributesReply>();

  auto& sequence = (*reply).sequence;
  auto& num_attribs = (*reply).num_attribs;
  auto& attribs = (*reply).attribs;
  size_t attribs_len = attribs.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // num_attribs
  Read(&num_attribs, &buf);

  // pad1
  Pad(&buf, 20);

  // attribs
  attribs.resize((num_attribs) * (2));
  for (auto& attribs_elem : attribs) {
    // attribs_elem
    Read(&attribs_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
Glx::ChangeDrawableAttributes(
    const Glx::ChangeDrawableAttributesRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& drawable = request.drawable;
  auto& num_attribs = request.num_attribs;
  auto& attribs = request.attribs;
  size_t attribs_len = attribs.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 30;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // drawable
  buf.Write(&drawable);

  // num_attribs
  buf.Write(&num_attribs);

  // attribs
  DCHECK_EQ(static_cast<size_t>((num_attribs) * (2)), attribs.size());
  for (auto& attribs_elem : attribs) {
    // attribs_elem
    buf.Write(&attribs_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::CreateWindow(
    const Glx::CreateWindowRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& screen = request.screen;
  auto& fbconfig = request.fbconfig;
  auto& window = request.window;
  auto& glx_window = request.glx_window;
  auto& num_attribs = request.num_attribs;
  auto& attribs = request.attribs;
  size_t attribs_len = attribs.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 31;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // screen
  buf.Write(&screen);

  // fbconfig
  buf.Write(&fbconfig);

  // window
  buf.Write(&window);

  // glx_window
  buf.Write(&glx_window);

  // num_attribs
  buf.Write(&num_attribs);

  // attribs
  DCHECK_EQ(static_cast<size_t>((num_attribs) * (2)), attribs.size());
  for (auto& attribs_elem : attribs) {
    // attribs_elem
    buf.Write(&attribs_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::DeleteWindow(
    const Glx::DeleteWindowRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& glxwindow = request.glxwindow;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 32;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // glxwindow
  buf.Write(&glxwindow);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::SetClientInfoARB(
    const Glx::SetClientInfoARBRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& major_version = request.major_version;
  auto& minor_version = request.minor_version;
  auto& num_versions = request.num_versions;
  uint32_t gl_str_len{};
  uint32_t glx_str_len{};
  auto& gl_versions = request.gl_versions;
  size_t gl_versions_len = gl_versions.size();
  auto& gl_extension_string = request.gl_extension_string;
  size_t gl_extension_string_len = gl_extension_string.size();
  auto& glx_extension_string = request.glx_extension_string;
  size_t glx_extension_string_len = glx_extension_string.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 33;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // major_version
  buf.Write(&major_version);

  // minor_version
  buf.Write(&minor_version);

  // num_versions
  buf.Write(&num_versions);

  // gl_str_len
  gl_str_len = gl_extension_string.size();
  buf.Write(&gl_str_len);

  // glx_str_len
  glx_str_len = glx_extension_string.size();
  buf.Write(&glx_str_len);

  // gl_versions
  DCHECK_EQ(static_cast<size_t>((num_versions) * (2)), gl_versions.size());
  for (auto& gl_versions_elem : gl_versions) {
    // gl_versions_elem
    buf.Write(&gl_versions_elem);

  }

  // gl_extension_string
  DCHECK_EQ(static_cast<size_t>(gl_str_len), gl_extension_string.size());
  for (auto& gl_extension_string_elem : gl_extension_string) {
    // gl_extension_string_elem
    buf.Write(&gl_extension_string_elem);

  }

  // glx_extension_string
  DCHECK_EQ(static_cast<size_t>(glx_str_len), glx_extension_string.size());
  for (auto& glx_extension_string_elem : glx_extension_string) {
    // glx_extension_string_elem
    buf.Write(&glx_extension_string_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::CreateContextAttribsARB(
    const Glx::CreateContextAttribsARBRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context = request.context;
  auto& fbconfig = request.fbconfig;
  auto& screen = request.screen;
  auto& share_list = request.share_list;
  auto& is_direct = request.is_direct;
  auto& num_attribs = request.num_attribs;
  auto& attribs = request.attribs;
  size_t attribs_len = attribs.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 34;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context
  buf.Write(&context);

  // fbconfig
  buf.Write(&fbconfig);

  // screen
  buf.Write(&screen);

  // share_list
  buf.Write(&share_list);

  // is_direct
  buf.Write(&is_direct);

  // pad0
  Pad(&buf, 3);

  // num_attribs
  buf.Write(&num_attribs);

  // attribs
  DCHECK_EQ(static_cast<size_t>((num_attribs) * (2)), attribs.size());
  for (auto& attribs_elem : attribs) {
    // attribs_elem
    buf.Write(&attribs_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::SetClientInfo2ARB(
    const Glx::SetClientInfo2ARBRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& major_version = request.major_version;
  auto& minor_version = request.minor_version;
  auto& num_versions = request.num_versions;
  uint32_t gl_str_len{};
  uint32_t glx_str_len{};
  auto& gl_versions = request.gl_versions;
  size_t gl_versions_len = gl_versions.size();
  auto& gl_extension_string = request.gl_extension_string;
  size_t gl_extension_string_len = gl_extension_string.size();
  auto& glx_extension_string = request.glx_extension_string;
  size_t glx_extension_string_len = glx_extension_string.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 35;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // major_version
  buf.Write(&major_version);

  // minor_version
  buf.Write(&minor_version);

  // num_versions
  buf.Write(&num_versions);

  // gl_str_len
  gl_str_len = gl_extension_string.size();
  buf.Write(&gl_str_len);

  // glx_str_len
  glx_str_len = glx_extension_string.size();
  buf.Write(&glx_str_len);

  // gl_versions
  DCHECK_EQ(static_cast<size_t>((num_versions) * (3)), gl_versions.size());
  for (auto& gl_versions_elem : gl_versions) {
    // gl_versions_elem
    buf.Write(&gl_versions_elem);

  }

  // gl_extension_string
  DCHECK_EQ(static_cast<size_t>(gl_str_len), gl_extension_string.size());
  for (auto& gl_extension_string_elem : gl_extension_string) {
    // gl_extension_string_elem
    buf.Write(&gl_extension_string_elem);

  }

  // glx_extension_string
  DCHECK_EQ(static_cast<size_t>(glx_str_len), glx_extension_string.size());
  for (auto& glx_extension_string_elem : glx_extension_string) {
    // glx_extension_string_elem
    buf.Write(&glx_extension_string_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::NewList(
    const Glx::NewListRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& list = request.list;
  auto& mode = request.mode;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 101;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // list
  buf.Write(&list);

  // mode
  buf.Write(&mode);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::EndList(
    const Glx::EndListRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 102;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::DeleteLists(
    const Glx::DeleteListsRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& list = request.list;
  auto& range = request.range;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 103;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // list
  buf.Write(&list);

  // range
  buf.Write(&range);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<Glx::GenListsReply>
Glx::GenLists(
    const Glx::GenListsRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& range = request.range;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 104;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // range
  buf.Write(&range);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GenListsReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GenListsReply>
ReadReply<Glx::GenListsReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GenListsReply>();

  auto& sequence = (*reply).sequence;
  auto& ret_val = (*reply).ret_val;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // ret_val
  Read(&ret_val, &buf);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
Glx::FeedbackBuffer(
    const Glx::FeedbackBufferRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& size = request.size;
  auto& type = request.type;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 105;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // size
  buf.Write(&size);

  // type
  buf.Write(&type);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::SelectBuffer(
    const Glx::SelectBufferRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& size = request.size;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 106;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // size
  buf.Write(&size);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<Glx::RenderModeReply>
Glx::RenderMode(
    const Glx::RenderModeRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& mode = request.mode;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 107;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // mode
  buf.Write(&mode);

  Align(&buf, 4);

  return x11::SendRequest<Glx::RenderModeReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::RenderModeReply>
ReadReply<Glx::RenderModeReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::RenderModeReply>();

  auto& sequence = (*reply).sequence;
  auto& ret_val = (*reply).ret_val;
  uint32_t n{};
  auto& new_mode = (*reply).new_mode;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // ret_val
  Read(&ret_val, &buf);

  // n
  Read(&n, &buf);

  // new_mode
  Read(&new_mode, &buf);

  // pad1
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::FinishReply>
Glx::Finish(
    const Glx::FinishRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 108;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  Align(&buf, 4);

  return x11::SendRequest<Glx::FinishReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::FinishReply>
ReadReply<Glx::FinishReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::FinishReply>();

  auto& sequence = (*reply).sequence;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
Glx::PixelStoref(
    const Glx::PixelStorefRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& pname = request.pname;
  auto& datum = request.datum;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 109;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // pname
  buf.Write(&pname);

  // datum
  buf.Write(&datum);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<void>
Glx::PixelStorei(
    const Glx::PixelStoreiRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& pname = request.pname;
  auto& datum = request.datum;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 110;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // pname
  buf.Write(&pname);

  // datum
  buf.Write(&datum);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<Glx::ReadPixelsReply>
Glx::ReadPixels(
    const Glx::ReadPixelsRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& x = request.x;
  auto& y = request.y;
  auto& width = request.width;
  auto& height = request.height;
  auto& format = request.format;
  auto& type = request.type;
  auto& swap_bytes = request.swap_bytes;
  auto& lsb_first = request.lsb_first;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 111;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // x
  buf.Write(&x);

  // y
  buf.Write(&y);

  // width
  buf.Write(&width);

  // height
  buf.Write(&height);

  // format
  buf.Write(&format);

  // type
  buf.Write(&type);

  // swap_bytes
  buf.Write(&swap_bytes);

  // lsb_first
  buf.Write(&lsb_first);

  Align(&buf, 4);

  return x11::SendRequest<Glx::ReadPixelsReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::ReadPixelsReply>
ReadReply<Glx::ReadPixelsReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::ReadPixelsReply>();

  auto& sequence = (*reply).sequence;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 24);

  // data
  data.resize((length) * (4));
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetBooleanvReply>
Glx::GetBooleanv(
    const Glx::GetBooleanvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 112;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetBooleanvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetBooleanvReply>
ReadReply<Glx::GetBooleanvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetBooleanvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 15);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetClipPlaneReply>
Glx::GetClipPlane(
    const Glx::GetClipPlaneRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& plane = request.plane;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 113;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // plane
  buf.Write(&plane);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetClipPlaneReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetClipPlaneReply>
ReadReply<Glx::GetClipPlaneReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetClipPlaneReply>();

  auto& sequence = (*reply).sequence;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 24);

  // data
  data.resize((length) / (2));
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetDoublevReply>
Glx::GetDoublev(
    const Glx::GetDoublevRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 114;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetDoublevReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetDoublevReply>
ReadReply<Glx::GetDoublevReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetDoublevReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 8);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetErrorReply>
Glx::GetError(
    const Glx::GetErrorRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 115;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetErrorReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetErrorReply>
ReadReply<Glx::GetErrorReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetErrorReply>();

  auto& sequence = (*reply).sequence;
  auto& error = (*reply).error;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // error
  Read(&error, &buf);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetFloatvReply>
Glx::GetFloatv(
    const Glx::GetFloatvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 116;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetFloatvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetFloatvReply>
ReadReply<Glx::GetFloatvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetFloatvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetIntegervReply>
Glx::GetIntegerv(
    const Glx::GetIntegervRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 117;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetIntegervReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetIntegervReply>
ReadReply<Glx::GetIntegervReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetIntegervReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetLightfvReply>
Glx::GetLightfv(
    const Glx::GetLightfvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& light = request.light;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 118;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // light
  buf.Write(&light);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetLightfvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetLightfvReply>
ReadReply<Glx::GetLightfvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetLightfvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetLightivReply>
Glx::GetLightiv(
    const Glx::GetLightivRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& light = request.light;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 119;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // light
  buf.Write(&light);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetLightivReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetLightivReply>
ReadReply<Glx::GetLightivReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetLightivReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetMapdvReply>
Glx::GetMapdv(
    const Glx::GetMapdvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& query = request.query;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 120;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // query
  buf.Write(&query);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetMapdvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetMapdvReply>
ReadReply<Glx::GetMapdvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetMapdvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 8);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetMapfvReply>
Glx::GetMapfv(
    const Glx::GetMapfvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& query = request.query;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 121;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // query
  buf.Write(&query);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetMapfvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetMapfvReply>
ReadReply<Glx::GetMapfvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetMapfvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetMapivReply>
Glx::GetMapiv(
    const Glx::GetMapivRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& query = request.query;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 122;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // query
  buf.Write(&query);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetMapivReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetMapivReply>
ReadReply<Glx::GetMapivReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetMapivReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetMaterialfvReply>
Glx::GetMaterialfv(
    const Glx::GetMaterialfvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& face = request.face;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 123;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // face
  buf.Write(&face);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetMaterialfvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetMaterialfvReply>
ReadReply<Glx::GetMaterialfvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetMaterialfvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetMaterialivReply>
Glx::GetMaterialiv(
    const Glx::GetMaterialivRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& face = request.face;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 124;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // face
  buf.Write(&face);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetMaterialivReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetMaterialivReply>
ReadReply<Glx::GetMaterialivReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetMaterialivReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetPixelMapfvReply>
Glx::GetPixelMapfv(
    const Glx::GetPixelMapfvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& map = request.map;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 125;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // map
  buf.Write(&map);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetPixelMapfvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetPixelMapfvReply>
ReadReply<Glx::GetPixelMapfvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetPixelMapfvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetPixelMapuivReply>
Glx::GetPixelMapuiv(
    const Glx::GetPixelMapuivRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& map = request.map;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 126;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // map
  buf.Write(&map);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetPixelMapuivReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetPixelMapuivReply>
ReadReply<Glx::GetPixelMapuivReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetPixelMapuivReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetPixelMapusvReply>
Glx::GetPixelMapusv(
    const Glx::GetPixelMapusvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& map = request.map;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 127;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // map
  buf.Write(&map);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetPixelMapusvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetPixelMapusvReply>
ReadReply<Glx::GetPixelMapusvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetPixelMapusvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 16);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetPolygonStippleReply>
Glx::GetPolygonStipple(
    const Glx::GetPolygonStippleRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& lsb_first = request.lsb_first;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 128;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // lsb_first
  buf.Write(&lsb_first);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetPolygonStippleReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetPolygonStippleReply>
ReadReply<Glx::GetPolygonStippleReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetPolygonStippleReply>();

  auto& sequence = (*reply).sequence;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 24);

  // data
  data.resize((length) * (4));
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetStringReply>
Glx::GetString(
    const Glx::GetStringRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& name = request.name;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 129;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // name
  buf.Write(&name);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetStringReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetStringReply>
ReadReply<Glx::GetStringReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetStringReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& string = (*reply).string;
  size_t string_len = string.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // pad2
  Pad(&buf, 16);

  // string
  string.resize(n);
  for (auto& string_elem : string) {
    // string_elem
    Read(&string_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetTexEnvfvReply>
Glx::GetTexEnvfv(
    const Glx::GetTexEnvfvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 130;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetTexEnvfvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetTexEnvfvReply>
ReadReply<Glx::GetTexEnvfvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetTexEnvfvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetTexEnvivReply>
Glx::GetTexEnviv(
    const Glx::GetTexEnvivRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 131;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetTexEnvivReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetTexEnvivReply>
ReadReply<Glx::GetTexEnvivReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetTexEnvivReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetTexGendvReply>
Glx::GetTexGendv(
    const Glx::GetTexGendvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& coord = request.coord;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 132;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // coord
  buf.Write(&coord);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetTexGendvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetTexGendvReply>
ReadReply<Glx::GetTexGendvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetTexGendvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 8);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetTexGenfvReply>
Glx::GetTexGenfv(
    const Glx::GetTexGenfvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& coord = request.coord;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 133;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // coord
  buf.Write(&coord);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetTexGenfvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetTexGenfvReply>
ReadReply<Glx::GetTexGenfvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetTexGenfvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetTexGenivReply>
Glx::GetTexGeniv(
    const Glx::GetTexGenivRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& coord = request.coord;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 134;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // coord
  buf.Write(&coord);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetTexGenivReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetTexGenivReply>
ReadReply<Glx::GetTexGenivReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetTexGenivReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetTexImageReply>
Glx::GetTexImage(
    const Glx::GetTexImageRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& level = request.level;
  auto& format = request.format;
  auto& type = request.type;
  auto& swap_bytes = request.swap_bytes;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 135;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // level
  buf.Write(&level);

  // format
  buf.Write(&format);

  // type
  buf.Write(&type);

  // swap_bytes
  buf.Write(&swap_bytes);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetTexImageReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetTexImageReply>
ReadReply<Glx::GetTexImageReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetTexImageReply>();

  auto& sequence = (*reply).sequence;
  auto& width = (*reply).width;
  auto& height = (*reply).height;
  auto& depth = (*reply).depth;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 8);

  // width
  Read(&width, &buf);

  // height
  Read(&height, &buf);

  // depth
  Read(&depth, &buf);

  // pad2
  Pad(&buf, 4);

  // data
  data.resize((length) * (4));
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetTexParameterfvReply>
Glx::GetTexParameterfv(
    const Glx::GetTexParameterfvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 136;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetTexParameterfvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetTexParameterfvReply>
ReadReply<Glx::GetTexParameterfvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetTexParameterfvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetTexParameterivReply>
Glx::GetTexParameteriv(
    const Glx::GetTexParameterivRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 137;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetTexParameterivReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetTexParameterivReply>
ReadReply<Glx::GetTexParameterivReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetTexParameterivReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetTexLevelParameterfvReply>
Glx::GetTexLevelParameterfv(
    const Glx::GetTexLevelParameterfvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& level = request.level;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 138;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // level
  buf.Write(&level);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetTexLevelParameterfvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetTexLevelParameterfvReply>
ReadReply<Glx::GetTexLevelParameterfvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetTexLevelParameterfvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetTexLevelParameterivReply>
Glx::GetTexLevelParameteriv(
    const Glx::GetTexLevelParameterivRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& level = request.level;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 139;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // level
  buf.Write(&level);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetTexLevelParameterivReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetTexLevelParameterivReply>
ReadReply<Glx::GetTexLevelParameterivReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetTexLevelParameterivReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::IsEnabledReply>
Glx::IsEnabled(
    const Glx::IsEnabledRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& capability = request.capability;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 140;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // capability
  buf.Write(&capability);

  Align(&buf, 4);

  return x11::SendRequest<Glx::IsEnabledReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::IsEnabledReply>
ReadReply<Glx::IsEnabledReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::IsEnabledReply>();

  auto& sequence = (*reply).sequence;
  auto& ret_val = (*reply).ret_val;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // ret_val
  Read(&ret_val, &buf);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::IsListReply>
Glx::IsList(
    const Glx::IsListRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& list = request.list;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 141;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // list
  buf.Write(&list);

  Align(&buf, 4);

  return x11::SendRequest<Glx::IsListReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::IsListReply>
ReadReply<Glx::IsListReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::IsListReply>();

  auto& sequence = (*reply).sequence;
  auto& ret_val = (*reply).ret_val;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // ret_val
  Read(&ret_val, &buf);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
Glx::Flush(
    const Glx::FlushRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 142;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<Glx::AreTexturesResidentReply>
Glx::AreTexturesResident(
    const Glx::AreTexturesResidentRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  int32_t n{};
  auto& textures = request.textures;
  size_t textures_len = textures.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 143;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // n
  n = textures.size();
  buf.Write(&n);

  // textures
  DCHECK_EQ(static_cast<size_t>(n), textures.size());
  for (auto& textures_elem : textures) {
    // textures_elem
    buf.Write(&textures_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<Glx::AreTexturesResidentReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::AreTexturesResidentReply>
ReadReply<Glx::AreTexturesResidentReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::AreTexturesResidentReply>();

  auto& sequence = (*reply).sequence;
  auto& ret_val = (*reply).ret_val;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // ret_val
  Read(&ret_val, &buf);

  // pad1
  Pad(&buf, 20);

  // data
  data.resize((length) * (4));
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
Glx::DeleteTextures(
    const Glx::DeleteTexturesRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  int32_t n{};
  auto& textures = request.textures;
  size_t textures_len = textures.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 144;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // n
  n = textures.size();
  buf.Write(&n);

  // textures
  DCHECK_EQ(static_cast<size_t>(n), textures.size());
  for (auto& textures_elem : textures) {
    // textures_elem
    buf.Write(&textures_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<Glx::GenTexturesReply>
Glx::GenTextures(
    const Glx::GenTexturesRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& n = request.n;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 145;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // n
  buf.Write(&n);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GenTexturesReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GenTexturesReply>
ReadReply<Glx::GenTexturesReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GenTexturesReply>();

  auto& sequence = (*reply).sequence;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 24);

  // data
  data.resize(length);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::IsTextureReply>
Glx::IsTexture(
    const Glx::IsTextureRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& texture = request.texture;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 146;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // texture
  buf.Write(&texture);

  Align(&buf, 4);

  return x11::SendRequest<Glx::IsTextureReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::IsTextureReply>
ReadReply<Glx::IsTextureReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::IsTextureReply>();

  auto& sequence = (*reply).sequence;
  auto& ret_val = (*reply).ret_val;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // ret_val
  Read(&ret_val, &buf);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetColorTableReply>
Glx::GetColorTable(
    const Glx::GetColorTableRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& format = request.format;
  auto& type = request.type;
  auto& swap_bytes = request.swap_bytes;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 147;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // format
  buf.Write(&format);

  // type
  buf.Write(&type);

  // swap_bytes
  buf.Write(&swap_bytes);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetColorTableReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetColorTableReply>
ReadReply<Glx::GetColorTableReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetColorTableReply>();

  auto& sequence = (*reply).sequence;
  auto& width = (*reply).width;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 8);

  // width
  Read(&width, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize((length) * (4));
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetColorTableParameterfvReply>
Glx::GetColorTableParameterfv(
    const Glx::GetColorTableParameterfvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 148;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetColorTableParameterfvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetColorTableParameterfvReply>
ReadReply<Glx::GetColorTableParameterfvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetColorTableParameterfvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetColorTableParameterivReply>
Glx::GetColorTableParameteriv(
    const Glx::GetColorTableParameterivRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 149;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetColorTableParameterivReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetColorTableParameterivReply>
ReadReply<Glx::GetColorTableParameterivReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetColorTableParameterivReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetConvolutionFilterReply>
Glx::GetConvolutionFilter(
    const Glx::GetConvolutionFilterRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& format = request.format;
  auto& type = request.type;
  auto& swap_bytes = request.swap_bytes;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 150;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // format
  buf.Write(&format);

  // type
  buf.Write(&type);

  // swap_bytes
  buf.Write(&swap_bytes);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetConvolutionFilterReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetConvolutionFilterReply>
ReadReply<Glx::GetConvolutionFilterReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetConvolutionFilterReply>();

  auto& sequence = (*reply).sequence;
  auto& width = (*reply).width;
  auto& height = (*reply).height;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 8);

  // width
  Read(&width, &buf);

  // height
  Read(&height, &buf);

  // pad2
  Pad(&buf, 8);

  // data
  data.resize((length) * (4));
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetConvolutionParameterfvReply>
Glx::GetConvolutionParameterfv(
    const Glx::GetConvolutionParameterfvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 151;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetConvolutionParameterfvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetConvolutionParameterfvReply>
ReadReply<Glx::GetConvolutionParameterfvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetConvolutionParameterfvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetConvolutionParameterivReply>
Glx::GetConvolutionParameteriv(
    const Glx::GetConvolutionParameterivRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 152;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetConvolutionParameterivReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetConvolutionParameterivReply>
ReadReply<Glx::GetConvolutionParameterivReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetConvolutionParameterivReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetSeparableFilterReply>
Glx::GetSeparableFilter(
    const Glx::GetSeparableFilterRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& format = request.format;
  auto& type = request.type;
  auto& swap_bytes = request.swap_bytes;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 153;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // format
  buf.Write(&format);

  // type
  buf.Write(&type);

  // swap_bytes
  buf.Write(&swap_bytes);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetSeparableFilterReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetSeparableFilterReply>
ReadReply<Glx::GetSeparableFilterReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetSeparableFilterReply>();

  auto& sequence = (*reply).sequence;
  auto& row_w = (*reply).row_w;
  auto& col_h = (*reply).col_h;
  auto& rows_and_cols = (*reply).rows_and_cols;
  size_t rows_and_cols_len = rows_and_cols.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 8);

  // row_w
  Read(&row_w, &buf);

  // col_h
  Read(&col_h, &buf);

  // pad2
  Pad(&buf, 8);

  // rows_and_cols
  rows_and_cols.resize((length) * (4));
  for (auto& rows_and_cols_elem : rows_and_cols) {
    // rows_and_cols_elem
    Read(&rows_and_cols_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetHistogramReply>
Glx::GetHistogram(
    const Glx::GetHistogramRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& format = request.format;
  auto& type = request.type;
  auto& swap_bytes = request.swap_bytes;
  auto& reset = request.reset;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 154;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // format
  buf.Write(&format);

  // type
  buf.Write(&type);

  // swap_bytes
  buf.Write(&swap_bytes);

  // reset
  buf.Write(&reset);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetHistogramReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetHistogramReply>
ReadReply<Glx::GetHistogramReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetHistogramReply>();

  auto& sequence = (*reply).sequence;
  auto& width = (*reply).width;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 8);

  // width
  Read(&width, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize((length) * (4));
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetHistogramParameterfvReply>
Glx::GetHistogramParameterfv(
    const Glx::GetHistogramParameterfvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 155;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetHistogramParameterfvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetHistogramParameterfvReply>
ReadReply<Glx::GetHistogramParameterfvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetHistogramParameterfvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetHistogramParameterivReply>
Glx::GetHistogramParameteriv(
    const Glx::GetHistogramParameterivRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 156;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetHistogramParameterivReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetHistogramParameterivReply>
ReadReply<Glx::GetHistogramParameterivReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetHistogramParameterivReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetMinmaxReply>
Glx::GetMinmax(
    const Glx::GetMinmaxRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& format = request.format;
  auto& type = request.type;
  auto& swap_bytes = request.swap_bytes;
  auto& reset = request.reset;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 157;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // format
  buf.Write(&format);

  // type
  buf.Write(&type);

  // swap_bytes
  buf.Write(&swap_bytes);

  // reset
  buf.Write(&reset);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetMinmaxReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetMinmaxReply>
ReadReply<Glx::GetMinmaxReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetMinmaxReply>();

  auto& sequence = (*reply).sequence;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 24);

  // data
  data.resize((length) * (4));
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetMinmaxParameterfvReply>
Glx::GetMinmaxParameterfv(
    const Glx::GetMinmaxParameterfvRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 158;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetMinmaxParameterfvReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetMinmaxParameterfvReply>
ReadReply<Glx::GetMinmaxParameterfvReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetMinmaxParameterfvReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetMinmaxParameterivReply>
Glx::GetMinmaxParameteriv(
    const Glx::GetMinmaxParameterivRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 159;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetMinmaxParameterivReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetMinmaxParameterivReply>
ReadReply<Glx::GetMinmaxParameterivReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetMinmaxParameterivReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetCompressedTexImageARBReply>
Glx::GetCompressedTexImageARB(
    const Glx::GetCompressedTexImageARBRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& level = request.level;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 160;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // level
  buf.Write(&level);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetCompressedTexImageARBReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetCompressedTexImageARBReply>
ReadReply<Glx::GetCompressedTexImageARBReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetCompressedTexImageARBReply>();

  auto& sequence = (*reply).sequence;
  auto& size = (*reply).size;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 8);

  // size
  Read(&size, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize((length) * (4));
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<void>
Glx::DeleteQueriesARB(
    const Glx::DeleteQueriesARBRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  int32_t n{};
  auto& ids = request.ids;
  size_t ids_len = ids.size();

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 161;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // n
  n = ids.size();
  buf.Write(&n);

  // ids
  DCHECK_EQ(static_cast<size_t>(n), ids.size());
  for (auto& ids_elem : ids) {
    // ids_elem
    buf.Write(&ids_elem);

  }

  Align(&buf, 4);

  return x11::SendRequest<void>(connection_, &buf, false);
}

Future<Glx::GenQueriesARBReply>
Glx::GenQueriesARB(
    const Glx::GenQueriesARBRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& n = request.n;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 162;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // n
  buf.Write(&n);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GenQueriesARBReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GenQueriesARBReply>
ReadReply<Glx::GenQueriesARBReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GenQueriesARBReply>();

  auto& sequence = (*reply).sequence;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 24);

  // data
  data.resize(length);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::IsQueryARBReply>
Glx::IsQueryARB(
    const Glx::IsQueryARBRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& id = request.id;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 163;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // id
  buf.Write(&id);

  Align(&buf, 4);

  return x11::SendRequest<Glx::IsQueryARBReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::IsQueryARBReply>
ReadReply<Glx::IsQueryARBReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::IsQueryARBReply>();

  auto& sequence = (*reply).sequence;
  auto& ret_val = (*reply).ret_val;

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // ret_val
  Read(&ret_val, &buf);

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetQueryivARBReply>
Glx::GetQueryivARB(
    const Glx::GetQueryivARBRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& target = request.target;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 164;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // target
  buf.Write(&target);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetQueryivARBReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetQueryivARBReply>
ReadReply<Glx::GetQueryivARBReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetQueryivARBReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetQueryObjectivARBReply>
Glx::GetQueryObjectivARB(
    const Glx::GetQueryObjectivARBRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& id = request.id;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 165;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // id
  buf.Write(&id);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetQueryObjectivARBReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetQueryObjectivARBReply>
ReadReply<Glx::GetQueryObjectivARBReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetQueryObjectivARBReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

Future<Glx::GetQueryObjectuivARBReply>
Glx::GetQueryObjectuivARB(
    const Glx::GetQueryObjectuivARBRequest& request) {
  if (!connection_->Ready() || !present())
    return {};

  WriteBuffer buf;

  auto& context_tag = request.context_tag;
  auto& id = request.id;
  auto& pname = request.pname;

  // major_opcode
  uint8_t major_opcode = info_.major_opcode;
  buf.Write(&major_opcode);

  // minor_opcode
  uint8_t minor_opcode = 166;
  buf.Write(&minor_opcode);

  // length
  // Caller fills in length for writes.
  Pad(&buf, sizeof(uint16_t));

  // context_tag
  buf.Write(&context_tag);

  // id
  buf.Write(&id);

  // pname
  buf.Write(&pname);

  Align(&buf, 4);

  return x11::SendRequest<Glx::GetQueryObjectuivARBReply>(connection_, &buf, false);
}

namespace detail {
template<> COMPONENT_EXPORT(X11)
std::unique_ptr<Glx::GetQueryObjectuivARBReply>
ReadReply<Glx::GetQueryObjectuivARBReply>(ReadBuffer* buffer) {
  auto& buf = *buffer;
  auto reply = std::make_unique<Glx::GetQueryObjectuivARBReply>();

  auto& sequence = (*reply).sequence;
  uint32_t n{};
  auto& datum = (*reply).datum;
  auto& data = (*reply).data;
  size_t data_len = data.size();

  // response_type
  uint8_t response_type;
  Read(&response_type, &buf);

  // pad0
  Pad(&buf, 1);

  // sequence
  Read(&sequence, &buf);

  // length
  uint32_t length;
  Read(&length, &buf);

  // pad1
  Pad(&buf, 4);

  // n
  Read(&n, &buf);

  // datum
  Read(&datum, &buf);

  // pad2
  Pad(&buf, 12);

  // data
  data.resize(n);
  for (auto& data_elem : data) {
    // data_elem
    Read(&data_elem, &buf);

  }

  Align(&buf, 4);
  DCHECK_EQ(buf.offset < 32 ? 0 : buf.offset - 32, 4 * length);

  return reply;
}
}  // namespace detail

}  // namespace x11
