/**
 * Numeric log level. A higher number is higher severity.
 */
export declare enum Level {
    Trace = 0,
    Debug = 1,
    Info = 2,
    Warn = 3,
    Error = 4
}
/**
 * A message field.
 */
export declare class Field<T> {
    readonly identifier: string;
    readonly value: T;
    constructor(identifier: string, value: T);
    toJSON(): {
        identifier: string;
        value: T;
    };
}
/**
 * A time field.
 */
export declare class Time {
    readonly expected: number;
    readonly ms: number;
    constructor(expected: number, ms: number);
}
/**
 * A generic argument to log alongside a message.
 */
export declare type Argument = any;
/**
 * `undefined` is allowed to make it easier to conditionally display a field.
 * For example: `error && field("error", error)`
 */
export declare type FieldArray = Array<Field<Argument> | undefined>;
/**
 * Functions can be used to remove the need to perform operations when the
 * logging level won't output the result anyway.
 */
export declare type LogCallback = () => [string, ...FieldArray];
/**
 * Log how long something took. Call this before doing the thing then pass it
 * into the logger after finished to log the time it took.
 */
export declare const time: (expected: number) => Time;
/**
 * A field to show with the message.
 */
export declare const field: <T>(name: string, value: T) => Field<T>;
/**
 * Represents a log message, its fields, level, and color.
 */
export declare type Message = {
    message: string | LogCallback;
    fields?: FieldArray;
    level: Level;
    tagColor: string;
};
/**
 * An extra function to call with a message.
 */
export declare type Extender = (msg: Message & {
    section?: string;
}) => void;
/**
 * Represents a message formatted for use with something like `console.log`.
 */
export declare type MessageFormat = {
    /**
     * For example, something like `[%s] %s`.
     */
    format: "";
    /**
     * The arguments that get applied to the format string.
     */
    args: string[];
    /**
     * The fields to add under the message.
     */
    fields: Field<Argument>[];
};
export declare function doLog(level: Level, ...args: Argument): void;
/**
 * Format and build a *single* log entry at a time.
 */
export declare abstract class Formatter {
    private readonly formatType;
    private readonly colors;
    private message;
    private readonly minimumTagWidth;
    /**
     * formatType is used for the strings returned from style() and reset().
     */
    constructor(formatType?: string, colors?: boolean);
    /**
     * Return a new/empty message.
     */
    private newMessage;
    /**
     * Add a tag.
     */
    tag(name: string, color: string): void;
    /**
     * Add a field or an argument. Arguments will display inline in the order they
     * were pushed. Fields will display differently based on the formatter. Fields
     * cannot have custom colors.
     */
    push(fields: Field<Argument>[]): void;
    push(arg: Argument, color?: string, weight?: string): void;
    /**
     * Write everything out and reset state.
     */
    write(level: Level): void;
    /**
     * Return current values and reset state.
     */
    protected flush(): MessageFormat;
    /**
     * Return a string that applies the specified color and weight.
     */
    protected abstract style(color?: string, weight?: string): string;
    /**
     * Return a string that resets all styles.
     */
    protected abstract reset(): string;
    /**
     * Write everything out.
     */
    protected abstract doWrite(level: Level, message: MessageFormat): void;
    /**
     * Get the format string for the value type.
     */
    private getType;
}
/**
 * Display logs in the browser using CSS in the output. Fields are displayed on
 * individual lines within a group.
 */
export declare class BrowserFormatter extends Formatter {
    constructor();
    protected style(color?: string, weight?: string): string;
    protected reset(): string;
    doWrite(level: Level, message: MessageFormat): void;
}
/**
 * Display logs on the command line using ANSI color codes. Fields are displayed
 * in a single stringified object inline.
 */
export declare class ServerFormatter extends Formatter {
    constructor();
    protected style(color?: string, weight?: string): string;
    protected reset(): string;
    private hex;
    private hexToRgb;
    protected doWrite(level: Level, message: MessageFormat): void;
}
export declare class Logger {
    private _formatter;
    private readonly name?;
    private readonly defaultFields?;
    private readonly extenders;
    level: Level;
    private readonly nameColor?;
    private muted;
    constructor(_formatter: Formatter, name?: string | undefined, defaultFields?: FieldArray | undefined, extenders?: Extender[]);
    set formatter(formatter: Formatter);
    /**
     * Supresses all output
     */
    mute(): void;
    extend(extender: Extender): void;
    info(fn: LogCallback): void;
    info(message: string, ...fields: FieldArray): void;
    warn(fn: LogCallback): void;
    warn(message: string, ...fields: FieldArray): void;
    trace(fn: LogCallback): void;
    trace(message: string, ...fields: FieldArray): void;
    debug(fn: LogCallback): void;
    debug(message: string, ...fields: FieldArray): void;
    error(fn: LogCallback): void;
    error(message: string, ...fields: FieldArray): void;
    /**
     * Returns a sub-logger with a name.
     * Each name is deterministically generated a color.
     */
    named(name: string, ...fields: FieldArray): Logger;
    private handle;
    /**
     * Hashes a string.
     */
    private djb2;
    private rgbToHex;
    /**
     * Generates a deterministic color from a string using hashing.
     */
    private hashStringToColor;
}
export declare const logger: Logger;
