"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mobx_1 = require("mobx");
const lodash_1 = __importDefault(require("lodash"));
const Options_1 = __importDefault(require("./Options"));
const Bindings_1 = __importDefault(require("./Bindings"));
const utils_1 = require("./utils");
const StateInterface_1 = require("./models/StateInterface");
const OptionsModel_1 = require("./models/OptionsModel");
class State {
    constructor({ form, initial, options, bindings }) {
        Object.defineProperty(this, "mode", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: StateInterface_1.RuntimeMode.mixed
        });
        Object.defineProperty(this, "strict", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: false
        });
        Object.defineProperty(this, "form", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "options", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "bindings", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "$extra", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "$struct", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(this, "disposers", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {
                interceptor: {},
                observer: {},
            }
        });
        Object.defineProperty(this, "initial", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {
                props: {},
                fields: {},
            }
        });
        Object.defineProperty(this, "current", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: {
                props: {},
                fields: {},
            }
        });
        this.set("form", form);
        this.initProps(initial);
        this.options = new Options_1.default();
        this.options.set(options);
        this.bindings = new Bindings_1.default();
        this.bindings.register(bindings);
        this.observeOptions();
    }
    initProps(initial = {}) {
        const initialProps = lodash_1.default.pick(initial, [
            ...utils_1.props.editable,
            ...utils_1.props.separated,
            ...utils_1.props.validation,
            ...utils_1.props.functions,
            ...utils_1.props.handlers,
        ]);
        this.set("initial", "props", initialProps);
        const $unified = (0, utils_1.hasUnifiedProps)(initial);
        const $separated = (0, utils_1.hasSeparatedProps)(initial);
        if (($separated || (0, utils_1.isArrayOfStrings)(initial.fields)) && !$unified) {
            const struct = (0, utils_1.$try)(initial.struct, initial.fields);
            this.struct(struct);
            this.strict = true;
            this.mode = StateInterface_1.RuntimeMode.separated;
            return;
        }
        this.struct(initial.struct);
        this.mode = StateInterface_1.RuntimeMode.unified;
    }
    /**
      Get/Set Fields Structure
    */
    struct(data = null) {
        if (data)
            this.$struct = data;
        return this.$struct;
    }
    /**
      Get Props/Fields
    */
    get(type, subtype) {
        return this[type][subtype];
    }
    /**
      Set Props/Fields
    */
    set(type, subtype, state = null) {
        if (type === "form") {
            // subtype is the form here
            this.form = subtype;
        }
        if (type === "initial") {
            Object.assign(this.initial[subtype], state);
            Object.assign(this.current[subtype], state);
        }
        if (type === "current") {
            Object.assign(this.current[subtype], state);
        }
    }
    extra(data = null) {
        if (lodash_1.default.isString(data))
            return lodash_1.default.get(this.$extra, data);
        if (data === null)
            return this.$extra;
        this.$extra = data;
        return null;
    }
    observeOptions() {
        // Fix Issue #201
        (0, mobx_1.observe)(this.options.options, (0, utils_1.checkObserve)([
            {
                // start observing fields validateOnChange
                type: "update",
                key: OptionsModel_1.OptionsEnum.validateOnChange,
                to: true,
                exec: () => this.form.each((field) => field.observeValidationOnChange()),
            },
            {
                // stop observing fields validateOnChange
                type: "update",
                key: OptionsModel_1.OptionsEnum.validateOnChange,
                to: false,
                exec: () => this.form.each((field) => field.disposeValidationOnChange()),
            },
            {
                // start observing fields validateOnBlur
                type: "update",
                key: OptionsModel_1.OptionsEnum.validateOnBlur,
                to: true,
                exec: () => this.form.each((field) => field.observeValidationOnBlur()),
            },
            {
                // stop observing fields validateOnBlur
                type: "update",
                key: OptionsModel_1.OptionsEnum.validateOnBlur,
                to: false,
                exec: () => this.form.each((field) => field.disposeValidationOnBlur()),
            },
        ]));
    }
}
exports.default = State;
//# sourceMappingURL=State.js.map