# HTML Backend Examples

```@meta
CurrentModule = PrettyTables
```

```@setup html_examples
using PrettyTables

function pt_to_html(args...; kwargs...)
    str = pretty_table(String, args...; kwargs...)
    return Base.HTML(str)
end
```

Here, we can see some examples of text tables generated by PrettyTables.jl. The `A` object,
when referenced, is defined as:

```julia-repl
julia> A = Any[
    1    false      1.0     0x01
    2     true      2.0     0x02
    3    false      3.0     0x03
    4     true      4.0     0x04
    5    false      5.0     0x05
    6     true      6.0     0x06
]
```

```@setup html_examples
A = Any[
    1    false      1.0     0x01
    2     true      2.0     0x02
    3    false      3.0     0x03
    4     true      4.0     0x04
    5    false      5.0     0x05
    6     true      6.0     0x06
]
```

---

```julia-repl
julia> pretty_table(A; backend = :html)
```

```@example html_examples
pt_to_html(         # hide
    A;              # hide
    backend = :html # hide
)                   # hide
```

---

```julia-repl
julia> pretty_table(
    A;
    backend = :html,
    style = HtmlTableStyle(;
        first_line_column_label = ["color" => "BurlyWood", "font-weight" => "bold"]
    )
)
```

```@example html_examples
pt_to_html(                                                                         # hide
    A;                                                                              # hide
    backend = :html,                                                                # hide
    style = HtmlTableStyle(;                                                        # hide
        first_line_column_label = ["color" => "BurlyWood", "font-weight" => "bold"] # hide
    )                                                                               # hide
)                                                                                   # hide
```

---

```julia-repl
julia> data = [
    10.0 6.5
     3.0 3.0
     0.1 1.0
]

julia> row_labels = [
    "Atmospheric drag"
    "Gravity gradient"
    "Solar radiation pressure"
]

julia> column_labels = [
    [MultiColumn(2, "Value", :c)],
    [
        "Torque [10⁻⁶ Nm]",
        "Angular Momentum [10⁻³ Nms]"
    ]
]

julia> pretty_table(
    data;
    backend = :html,
    column_labels,
    merge_column_label_cells = :auto,
    row_labels,
    stubhead_label = "Effect",
    style = HtmlTableStyle(;
        first_line_merged_column_label = ["color" => "BurlyWood", "font-weight" => "bold"],
        column_label                   = ["color" => "DarkGrey"],
        stubhead_label                 = ["color" => "BurlyWood", "font-weight" => "bold"],
        summary_row_label              = ["color" => "red",       "font-weight" => "bold"]
    ),
    summary_row_labels = ["Total"],
    summary_rows = [(data, i) -> sum(data[:, i])],
)
```

```@example html_examples
data = [                                                                                    # hide
    10.0 6.5                                                                                # hide
     3.0 3.0                                                                                # hide
     0.1 1.0                                                                                # hide
]                                                                                           # hide
                                                                                            # hide
row_labels = [                                                                              # hide
    "Atmospheric drag"                                                                      # hide
    "Gravity gradient"                                                                      # hide
    "Solar radiation pressure"                                                              # hide
]                                                                                           # hide
                                                                                            # hide
column_labels = [                                                                           # hide
    [MultiColumn(2, "Value", :c)],                                                          # hide
    [                                                                                       # hide
        "Torque [10⁻⁶ Nm]",                                                                 # hide
        "Angular Momentum [10⁻³ Nms]"                                                       # hide
    ]                                                                                       # hide
]                                                                                           # hide
                                                                                            # hide
pt_to_html(                                                                                 # hide
    data;                                                                                   # hide
    backend = :html,                                                                        # hide
    column_labels,                                                                          # hide
    merge_column_label_cells = :auto,                                                       # hide
    row_labels,                                                                             # hide
    stubhead_label = "Effect",                                                              # hide
    style = HtmlTableStyle(;                                                                # hide
        first_line_merged_column_label = ["color" => "BurlyWood", "font-weight" => "bold"], # hide
        column_label                   = ["color" => "DarkGrey"],                           # hide
        stubhead_label                 = ["color" => "BurlyWood", "font-weight" => "bold"], # hide
        summary_row_label              = ["color" => "red",       "font-weight" => "bold"]  # hide
    ),                                                                                      # hide
    summary_row_labels = ["Total"],                                                         # hide
    summary_rows = [(data, i) -> sum(data[:, i])],                                          # hide
)                                                                                           # hide
```

---

```julia-repl
julia> t = 0:1:20

julia> data = hcat(t, ones(length(t) ), t, 0.5.*t.^2);

julia> column_labels = [
    ["Time", "Acceleration", "Velocity", "Distance"],
    [ "[s]",     "[m / s²]",  "[m / s]",      "[m]"]
]

julia> hl_p = HtmlHighlighter(
    (data, i, j) -> (j == 4) && (data[i, j] > 9),
    ["color" => "blue"]
)

julia> hl_v = TextHighlighter(
    (data, i, j) -> (j == 3) && (data[i, j] > 9),
    ["color" => "red"]
)

julia> pretty_table(
    data;
    backend = :html,
    column_labels = column_labels,
    highlighters  = [hl_p, hl_v],
    style = HtmlTableStyle(;
        first_line_column_label = ["color" => "BurlyWood", "font-weight" => "bold"],
        column_label            = ["color" => "DarkGrey"],
    )
)
```

```@example html_examples
t = 0:1:20                                                                           # hide
                                                                                     # hide
data = hcat(t, ones(length(t) ), t, 0.5.*t.^2);                                      # hide
                                                                                     # hide
column_labels = [                                                                    # hide
    ["Time", "Acceleration", "Velocity", "Distance"],                                # hide
    [ "[s]",     "[m / s²]",  "[m / s]",      "[m]"]                                 # hide
]                                                                                    # hide
                                                                                     # hide
hl_p = HtmlHighlighter(                                                              # hide
    (data, i, j) -> (j == 4) && (data[i, j] > 9),                                    # hide
    ["color" => "blue"]                                                              # hide
)                                                                                    # hide
                                                                                     # hide
hl_v = HtmlHighlighter(                                                              # hide
    (data, i, j) -> (j == 3) && (data[i, j] > 9),                                    # hide
    ["color" => "red"]                                                               # hide
)                                                                                    # hide
                                                                                     # hide
pt_to_html(                                                                          # hide
    data;                                                                            # hide
    backend = :html,                                                                 # hide
    column_labels = column_labels,                                                   # hide
    highlighters  = [hl_p, hl_v],                                                    # hide
    style = HtmlTableStyle(;                                                         # hide
        first_line_column_label = ["color" => "BurlyWood", "font-weight" => "bold"], # hide
        column_label            = ["color" => "DarkGrey"],                           # hide
    )                                                                                # hide
)                                                                                    # hide
```
