# Markdown Backend Examples

```@meta
CurrentModule = PrettyTables
```

```@setup markdown_examples
using PrettyTables, Markdown
```

This page contains examples of markdown tables generated by PrettyTables.jl. 

The `A` object, when referenced, is defined as:

```julia-repl
julia> A = Any[
    1    false      1.0     0x01
    2     true      2.0     0x02
    3    false      3.0     0x03
    4     true      4.0     0x04
    5    false      5.0     0x05
    6     true      6.0     0x06
]
```

```@setup markdown_examples
A = Any[
    1    false      1.0     0x01
    2     true      2.0     0x02
    3    false      3.0     0x03
    4     true      4.0     0x04
    5    false      5.0     0x05
    6     true      6.0     0x06
]
```

## Basic Table

First, we create a basic markdown table using the [`pretty_table`](@ref) function with 
the `backend` keyword set to `:markdown`.

```@example markdown_examples
table = pretty_table(String, A; backend = :markdown)
Markdown.parse(table) # hide
```

We can add custom column labels by passing a vector of strings to the `column_labels` 
keyword.

```@example markdown_examples
table = pretty_table(
    String,
    A;
    backend = :markdown,
    column_labels = ["ID", "Flag", "Value", "Hex"]
)
Markdown.parse(table) # hide
```

## Column Label Style

The style of the column labels can be customized by passing a [`MarkdownTableStyle`](@ref)  
to the `style` keyword. In the first example, we set the `first_line_column_label` field 
of the style to a [`MarkdownStyle`](@ref) with the `bold` field set to `true` to obtain 
bold column labels.

!!! details "**bold** Column Labels"
    ```@example markdown_examples
    table = pretty_table(
        String,
        A;
        backend = :markdown,
        style = MarkdownTableStyle(first_line_column_label = MarkdownStyle(bold = true))
    )

    print(table) # hide
    ```

```@example markdown_examples
Markdown.parse(table) # hide
```

In the second example we use italic column labels:

!!! details "*italic* Column Labels"
    ```@example markdown_examples
    table = pretty_table(
        String,
        A;
        backend = :markdown,
        style = MarkdownTableStyle(first_line_column_label = MarkdownStyle(italic = true))
    )

    print(table) # hide
    ```

```@example markdown_examples
Markdown.parse(table) # hide
```

## Custom Alignment

The alignment of each column can be specified by passing a vector of symbols to the 
`alignment` keyword. The supported alignment values are:

- `:l`: Left alignment
- `:c`: Center alignment
- `:r`: Right alignment

The default alignment is left alignment (`:l`).

!!! details "Example code"
    ```@example markdown_examples
    table = pretty_table(
        String,
        A;
        backend = :markdown,
        column_labels = ["Left", "Center", "Right", "Default"],
        alignment = [:l, :c, :r, :l]
    )

    print(table) # hide
    ```

```@example markdown_examples
Markdown.parse(table) # hide
```

## Row and Column Labels

Additional row and column labels can be added to the table by passing the `row_labels` 
and `column_labels` keywords to the `pretty_table` function. The `stubhead_label` keyword 
can be used to specify the label of the stubhead (corner cell).

!!! details "Example code"
    ```@example markdown_examples
    data = [
        10.0 6.5
        3.0 3.0
        0.1 1.0
    ]

    row_labels = [
        "Atmospheric drag"
        "Gravity gradient"
        "Solar radiation pressure"
    ]

    column_labels = [
        "Torque [10⁻⁶ Nm]",
        "Angular Momentum [10⁻³ Nms]"
    ]

    table = pretty_table(
        String,
        data;
        backend = :markdown,
        column_labels,
        row_labels,
        stubhead_label = "Effect",
        summary_row_labels = ["Total"],
        summary_rows = [(data, i) -> sum(data[:, i])],
    )

    print(table) # hide
    ```

```@example markdown_examples
Markdown.parse(table) # hide
```

## Markdown Highlighters

A markdown highlighter is defined using an object of type [`MarkdownHighlighter`](@ref).
Two types of highlighters are supported: *italic* and **bold**.

!!! details "Example code"
    ```@example markdown_examples
    t = 0:1:20

    data = hcat(t, ones(length(t)), t, 0.5.*t.^2)

    column_labels = [
        ["Time", "Acceleration", "Velocity", "Distance"],
        ["[s]", "[m / s²]", "[m / s]", "[m]"]
    ]

    hl_p = MarkdownHighlighter(
        (data, i, j) -> (j == 4) && (data[i, j] > 9),
        MarkdownStyle(italic = true)
    )

    hl_v = MarkdownHighlighter(
        (data, i, j) -> (j == 3) && (data[i, j] > 9),
        MarkdownStyle(bold = true)
    )

    table = pretty_table(
        String,
        data;
        backend = :markdown,
        column_labels = column_labels,
        highlighters = [hl_p, hl_v],
    )
    
    print(table) # hide
    ```

```@example markdown_examples
Markdown.parse(table) # hide
```