"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cosmiconfig = require("cosmiconfig");
const minimatch = require("minimatch");
const resolve_from_1 = require("resolve-from");
exports.DEFAULT_CONFIGS = {
    ".js, .jsx, .es6, .es, .mjs, .ts, .tsx": {
        parser: "babylon",
        style: "eslint",
    },
};
function getConfig(extension, directory, defaultConfigs = exports.DEFAULT_CONFIGS) {
    const defaultConfig = getConfigForExtension(defaultConfigs, extension);
    let packageConfig;
    if (directory) {
        packageConfig = getConfigFromDirectory(directory, extension);
    }
    const actualConfig = mergeConfigs([defaultConfig, packageConfig]);
    if (!actualConfig) {
        return undefined;
    }
    const resolvedConfig = resolveConfig(actualConfig, directory);
    return resolvedConfig;
}
exports.getConfig = getConfig;
function getConfigFromDirectory(directory, extension) {
    const packageConfigs = getAllConfigsFromDirectory(directory);
    if (!packageConfigs) {
        return undefined;
    }
    return getConfigForExtension(packageConfigs, extension);
}
function getConfigForExtension(configs, extension) {
    const foundConfigs = Object.keys(configs).map(joinedGlobs => {
        const globs = joinedGlobs.split(",").map(rawGlob => rawGlob.trim());
        const config = configs[joinedGlobs];
        if (globs.some(glob => minimatch(extension, glob))) {
            return config;
        }
        return undefined;
    });
    return mergeConfigs(foundConfigs);
}
function getAllConfigsFromDirectory(directory) {
    const configsLoader = cosmiconfig("importsort", {
        sync: true,
        packageProp: "importSort",
        rcExtensions: true,
    });
    try {
        const configsResult = configsLoader.searchSync(directory);
        if (!configsResult) {
            return undefined;
        }
        return configsResult.config;
    }
    catch (e) {
        // Do nothing…
    }
    return undefined;
}
function mergeConfigs(rawConfigs) {
    const configs = rawConfigs.filter(rawConfig => !!rawConfig);
    if (configs.length === 0) {
        return undefined;
    }
    return configs.reduce((previousConfig, currentConfig) => {
        if (!currentConfig) {
            return previousConfig;
        }
        const config = Object.assign({}, previousConfig);
        if (currentConfig.parser) {
            config.parser = currentConfig.parser;
        }
        if (currentConfig.style) {
            config.style = currentConfig.style;
        }
        if (currentConfig.options) {
            config.options = currentConfig.options;
        }
        return config;
    });
}
function resolveConfig(config, directory) {
    const resolvedConfig = {
        config,
    };
    if (config.parser) {
        resolvedConfig.parser = resolveParser(config.parser, directory);
    }
    if (config.style) {
        resolvedConfig.style = resolveStyle(config.style, directory);
    }
    return resolvedConfig;
}
function resolveParser(module, directory) {
    return (resolveModule(`import-sort-parser-${module}`, directory) ||
        resolveModule(module, directory));
}
function resolveStyle(module, directory) {
    return (resolveModule(`import-sort-style-${module}`, directory) ||
        resolveModule(module, directory));
}
function resolveModule(module, directory) {
    if (directory) {
        const path = resolve_from_1.silent(directory, module);
        if (path) {
            return path;
        }
    }
    const defaultPath = resolve_from_1.silent(__dirname, module);
    if (defaultPath) {
        return defaultPath;
    }
    return undefined;
}
