from collections.abc import Iterable
import weakref as _weakref
import sys

from vulkan._vulkancache import ffi


_weakkey_dict = _weakref.WeakKeyDictionary()
PY3 = sys.version_info >= (3, 0)


class ProcedureNotFoundError(Exception):
    pass


class ExtensionNotSupportedError(Exception):
    pass


def _cstr(x):
    if not isinstance(x, ffi.CData):
        return x

    t = ffi.typeof(x)
    if 'item' not in dir(t) or t.item.cname != 'char':
        return x

    if PY3:
        try:
            return ffi.string(x).decode('ascii')
        except UnicodeDecodeError:
            return ffi.string(x).decode('utf-8')
    else:
        return ffi.string(x)


class StrWrap(object):
    """Wrap a FFI Cdata object

    This class is a proxy class which auto-convert FFI string to Python
    string. It must be used only on object containing string data.
    Original CFFI string can always be accessed by prefixing the property with
    an underscore.
    """
    def __init__(self, obj):
        self.obj = obj

    def __setattr__(self, key, value):
        if key == 'obj':
            return super(StrWrap, self).__setattr__(key, value)

        setattr(self.obj, key, value)

    def __getattr__(self, key):
        try:
            attr = getattr(self.obj, key)
        except AttributeError as origin_exc:
            # Remove the first underscore if exists
            if key.startswith('_'):
                try:
                    return getattr(self.obj, key[1:])
                except AttributeError:
                    raise origin_exc
            raise origin_exc

        return _cstr(attr)


def _cast_ptr2(x, _type):
    if isinstance(x, ffi.CData):
        if (_type.item == ffi.typeof(x) or
            (_type.item.cname == 'void' and ffi.typeof(x).kind in
             ['struct', 'union'])):
            return ffi.addressof(x), x
        return x, x

    if isinstance(x, Iterable):
        if _type.item.kind == 'pointer':
            ptrs = [_cast_ptr(i, _type.item) for i in x]
            ret = ffi.new(_type.item.cname+'[]', [i for i, _ in ptrs])
            _weakkey_dict[ret] = tuple(i for _, i in ptrs if i != ffi.NULL)
        else:
            ret = ffi.new(_type.item.cname+'[]', x)

        return ret, ret

    return ffi.cast(_type, x), x


def _cast_ptr3(x, _type):
    if isinstance(x, str):
        try:
            x = x.encode('ascii')
        except UnicodeEncodeError:
            x = x.encode('utf-8')
    return _cast_ptr2(x, _type)


_cast_ptr = _cast_ptr3 if PY3 else _cast_ptr2


# Load SDK
_lib_names = ('libvulkan.so.1', 'vulkan-1.dll', 'libvulkan.dylib')
for name in _lib_names:
    try:
        lib = ffi.dlopen(name)
        break
    except OSError:
        pass
else:
    raise OSError('Cannot find Vulkan SDK version. Please ensure that it is '
                  'installed and that the <sdk_root>/<version>/lib/ folder is '
                  'in the library path')


VK_ATTACHMENT_LOAD_OP_LOAD = 0
VK_ATTACHMENT_LOAD_OP_CLEAR = 1
VK_ATTACHMENT_LOAD_OP_DONT_CARE = 2
VK_ATTACHMENT_LOAD_OP__BEGIN_RANGE = 0
VK_ATTACHMENT_LOAD_OP__END_RANGE = 2
VK_ATTACHMENT_LOAD_OP__RANGE_SIZE = 3
VK_ATTACHMENT_LOAD_OP__MAX_ENUM = 2147483647
VK_ATTACHMENT_STORE_OP_STORE = 0
VK_ATTACHMENT_STORE_OP_DONT_CARE = 1
VK_ATTACHMENT_STORE_OP__BEGIN_RANGE = 0
VK_ATTACHMENT_STORE_OP__END_RANGE = 1
VK_ATTACHMENT_STORE_OP__RANGE_SIZE = 2
VK_ATTACHMENT_STORE_OP__MAX_ENUM = 2147483647
VK_ATTACHMENT_STORE_OP_NONE = 1000301000
VK_BLEND_FACTOR_ZERO = 0
VK_BLEND_FACTOR_ONE = 1
VK_BLEND_FACTOR_SRC_COLOR = 2
VK_BLEND_FACTOR_ONE_MINUS_SRC_COLOR = 3
VK_BLEND_FACTOR_DST_COLOR = 4
VK_BLEND_FACTOR_ONE_MINUS_DST_COLOR = 5
VK_BLEND_FACTOR_SRC_ALPHA = 6
VK_BLEND_FACTOR_ONE_MINUS_SRC_ALPHA = 7
VK_BLEND_FACTOR_DST_ALPHA = 8
VK_BLEND_FACTOR_ONE_MINUS_DST_ALPHA = 9
VK_BLEND_FACTOR_CONSTANT_COLOR = 10
VK_BLEND_FACTOR_ONE_MINUS_CONSTANT_COLOR = 11
VK_BLEND_FACTOR_CONSTANT_ALPHA = 12
VK_BLEND_FACTOR_ONE_MINUS_CONSTANT_ALPHA = 13
VK_BLEND_FACTOR_SRC_ALPHA_SATURATE = 14
VK_BLEND_FACTOR_SRC1_COLOR = 15
VK_BLEND_FACTOR_ONE_MINUS_SRC1_COLOR = 16
VK_BLEND_FACTOR_SRC1_ALPHA = 17
VK_BLEND_FACTOR_ONE_MINUS_SRC1_ALPHA = 18
VK_BLEND_FACTOR__BEGIN_RANGE = 0
VK_BLEND_FACTOR__END_RANGE = 18
VK_BLEND_FACTOR__RANGE_SIZE = 19
VK_BLEND_FACTOR__MAX_ENUM = 2147483647
VK_BLEND_OP_ADD = 0
VK_BLEND_OP_SUBTRACT = 1
VK_BLEND_OP_REVERSE_SUBTRACT = 2
VK_BLEND_OP_MIN = 3
VK_BLEND_OP_MAX = 4
VK_BLEND_OP__BEGIN_RANGE = 0
VK_BLEND_OP__END_RANGE = 4
VK_BLEND_OP__RANGE_SIZE = 5
VK_BLEND_OP__MAX_ENUM = 2147483647
VK_BORDER_COLOR_FLOAT_TRANSPARENT_BLACK = 0
VK_BORDER_COLOR_INT_TRANSPARENT_BLACK = 1
VK_BORDER_COLOR_FLOAT_OPAQUE_BLACK = 2
VK_BORDER_COLOR_INT_OPAQUE_BLACK = 3
VK_BORDER_COLOR_FLOAT_OPAQUE_WHITE = 4
VK_BORDER_COLOR_INT_OPAQUE_WHITE = 5
VK_BORDER_COLOR__BEGIN_RANGE = 0
VK_BORDER_COLOR__END_RANGE = 5
VK_BORDER_COLOR__RANGE_SIZE = 6
VK_BORDER_COLOR__MAX_ENUM = 2147483647
VK_PIPELINE_CACHE_HEADER_VERSION_ONE = 1
VK_PIPELINE_CACHE_HEADER_VERSION__BEGIN_RANGE = 1
VK_PIPELINE_CACHE_HEADER_VERSION__END_RANGE = 1
VK_PIPELINE_CACHE_HEADER_VERSION__RANGE_SIZE = 1
VK_PIPELINE_CACHE_HEADER_VERSION__MAX_ENUM = 2147483647
VK_PIPELINE_CACHE_HEADER_VERSION_SAFETY_CRITICAL_ONE = 1000298001
VK_BUFFER_CREATE_SPARSE_BINDING_BIT = 0x00000001
VK_BUFFER_CREATE_SPARSE_RESIDENCY_BIT = 0x00000002
VK_BUFFER_CREATE_SPARSE_ALIASED_BIT = 0x00000004
VK_BUFFER_CREATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_BUFFER_USAGE_TRANSFER_SRC_BIT = 0x00000001
VK_BUFFER_USAGE_TRANSFER_DST_BIT = 0x00000002
VK_BUFFER_USAGE_UNIFORM_TEXEL_BUFFER_BIT = 0x00000004
VK_BUFFER_USAGE_STORAGE_TEXEL_BUFFER_BIT = 0x00000008
VK_BUFFER_USAGE_UNIFORM_BUFFER_BIT = 0x00000010
VK_BUFFER_USAGE_STORAGE_BUFFER_BIT = 0x00000020
VK_BUFFER_USAGE_INDEX_BUFFER_BIT = 0x00000040
VK_BUFFER_USAGE_VERTEX_BUFFER_BIT = 0x00000080
VK_BUFFER_USAGE_INDIRECT_BUFFER_BIT = 0x00000100
VK_BUFFER_USAGE_FLAG_BITS__MAX_ENUM = 2147483647
VK_COLOR_COMPONENT_R_BIT = 0x00000001
VK_COLOR_COMPONENT_G_BIT = 0x00000002
VK_COLOR_COMPONENT_B_BIT = 0x00000004
VK_COLOR_COMPONENT_A_BIT = 0x00000008
VK_COLOR_COMPONENT_FLAG_BITS__MAX_ENUM = 2147483647
VK_COMPONENT_SWIZZLE_IDENTITY = 0
VK_COMPONENT_SWIZZLE_ZERO = 1
VK_COMPONENT_SWIZZLE_ONE = 2
VK_COMPONENT_SWIZZLE_R = 3
VK_COMPONENT_SWIZZLE_G = 4
VK_COMPONENT_SWIZZLE_B = 5
VK_COMPONENT_SWIZZLE_A = 6
VK_COMPONENT_SWIZZLE__BEGIN_RANGE = 0
VK_COMPONENT_SWIZZLE__END_RANGE = 6
VK_COMPONENT_SWIZZLE__RANGE_SIZE = 7
VK_COMPONENT_SWIZZLE__MAX_ENUM = 2147483647
VK_COMMAND_POOL_CREATE_TRANSIENT_BIT = 0x00000001
VK_COMMAND_POOL_CREATE_RESET_COMMAND_BUFFER_BIT = 0x00000002
VK_COMMAND_POOL_CREATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_COMMAND_POOL_RESET_RELEASE_RESOURCES_BIT = 0x00000001
VK_COMMAND_POOL_RESET_FLAG_BITS__MAX_ENUM = 2147483647
VK_COMMAND_BUFFER_RESET_RELEASE_RESOURCES_BIT = 0x00000001
VK_COMMAND_BUFFER_RESET_FLAG_BITS__MAX_ENUM = 2147483647
VK_COMMAND_BUFFER_LEVEL_PRIMARY = 0
VK_COMMAND_BUFFER_LEVEL_SECONDARY = 1
VK_COMMAND_BUFFER_LEVEL__BEGIN_RANGE = 0
VK_COMMAND_BUFFER_LEVEL__END_RANGE = 1
VK_COMMAND_BUFFER_LEVEL__RANGE_SIZE = 2
VK_COMMAND_BUFFER_LEVEL__MAX_ENUM = 2147483647
VK_COMMAND_BUFFER_USAGE_ONE_TIME_SUBMIT_BIT = 0x00000001
VK_COMMAND_BUFFER_USAGE_RENDER_PASS_CONTINUE_BIT = 0x00000002
VK_COMMAND_BUFFER_USAGE_SIMULTANEOUS_USE_BIT = 0x00000004
VK_COMMAND_BUFFER_USAGE_FLAG_BITS__MAX_ENUM = 2147483647
VK_COMPARE_OP_NEVER = 0
VK_COMPARE_OP_LESS = 1
VK_COMPARE_OP_EQUAL = 2
VK_COMPARE_OP_LESS_OR_EQUAL = 3
VK_COMPARE_OP_GREATER = 4
VK_COMPARE_OP_NOT_EQUAL = 5
VK_COMPARE_OP_GREATER_OR_EQUAL = 6
VK_COMPARE_OP_ALWAYS = 7
VK_COMPARE_OP__BEGIN_RANGE = 0
VK_COMPARE_OP__END_RANGE = 7
VK_COMPARE_OP__RANGE_SIZE = 8
VK_COMPARE_OP__MAX_ENUM = 2147483647
VK_CULL_MODE_NONE = 0
VK_CULL_MODE_FRONT_BIT = 0x00000001
VK_CULL_MODE_BACK_BIT = 0x00000002
VK_CULL_MODE_FRONT_AND_BACK = 0x00000003
VK_CULL_MODE_FLAG_BITS__MAX_ENUM = 2147483647
VK_DESCRIPTOR_TYPE_SAMPLER = 0
VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER = 1
VK_DESCRIPTOR_TYPE_SAMPLED_IMAGE = 2
VK_DESCRIPTOR_TYPE_STORAGE_IMAGE = 3
VK_DESCRIPTOR_TYPE_UNIFORM_TEXEL_BUFFER = 4
VK_DESCRIPTOR_TYPE_STORAGE_TEXEL_BUFFER = 5
VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER = 6
VK_DESCRIPTOR_TYPE_STORAGE_BUFFER = 7
VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER_DYNAMIC = 8
VK_DESCRIPTOR_TYPE_STORAGE_BUFFER_DYNAMIC = 9
VK_DESCRIPTOR_TYPE_INPUT_ATTACHMENT = 10
VK_DESCRIPTOR_TYPE__BEGIN_RANGE = 0
VK_DESCRIPTOR_TYPE__END_RANGE = 10
VK_DESCRIPTOR_TYPE__RANGE_SIZE = 11
VK_DESCRIPTOR_TYPE__MAX_ENUM = 2147483647
VK_DESCRIPTOR_TYPE_INLINE_UNIFORM_BLOCK = 1000138000
VK_DYNAMIC_STATE_VIEWPORT = 0
VK_DYNAMIC_STATE_SCISSOR = 1
VK_DYNAMIC_STATE_LINE_WIDTH = 2
VK_DYNAMIC_STATE_DEPTH_BIAS = 3
VK_DYNAMIC_STATE_BLEND_CONSTANTS = 4
VK_DYNAMIC_STATE_DEPTH_BOUNDS = 5
VK_DYNAMIC_STATE_STENCIL_COMPARE_MASK = 6
VK_DYNAMIC_STATE_STENCIL_WRITE_MASK = 7
VK_DYNAMIC_STATE_STENCIL_REFERENCE = 8
VK_DYNAMIC_STATE__BEGIN_RANGE = 0
VK_DYNAMIC_STATE__END_RANGE = 8
VK_DYNAMIC_STATE__RANGE_SIZE = 9
VK_DYNAMIC_STATE__MAX_ENUM = 2147483647
VK_DYNAMIC_STATE_CULL_MODE = 1000267000
VK_DYNAMIC_STATE_FRONT_FACE = 1000267001
VK_DYNAMIC_STATE_PRIMITIVE_TOPOLOGY = 1000267002
VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT = 1000267003
VK_DYNAMIC_STATE_SCISSOR_WITH_COUNT = 1000267004
VK_DYNAMIC_STATE_VERTEX_INPUT_BINDING_STRIDE = 1000267005
VK_DYNAMIC_STATE_DEPTH_TEST_ENABLE = 1000267006
VK_DYNAMIC_STATE_DEPTH_WRITE_ENABLE = 1000267007
VK_DYNAMIC_STATE_DEPTH_COMPARE_OP = 1000267008
VK_DYNAMIC_STATE_DEPTH_BOUNDS_TEST_ENABLE = 1000267009
VK_DYNAMIC_STATE_STENCIL_TEST_ENABLE = 1000267010
VK_DYNAMIC_STATE_STENCIL_OP = 1000267011
VK_DYNAMIC_STATE_RASTERIZER_DISCARD_ENABLE = 1000377001
VK_DYNAMIC_STATE_DEPTH_BIAS_ENABLE = 1000377002
VK_DYNAMIC_STATE_PRIMITIVE_RESTART_ENABLE = 1000377004
VK_FENCE_CREATE_SIGNALED_BIT = 0x00000001
VK_FENCE_CREATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_POLYGON_MODE_FILL = 0
VK_POLYGON_MODE_LINE = 1
VK_POLYGON_MODE_POINT = 2
VK_POLYGON_MODE__BEGIN_RANGE = 0
VK_POLYGON_MODE__END_RANGE = 2
VK_POLYGON_MODE__RANGE_SIZE = 3
VK_POLYGON_MODE__MAX_ENUM = 2147483647
VK_FORMAT_UNDEFINED = 0
VK_FORMAT_R4G4_UNORM_PACK8 = 1
VK_FORMAT_R4G4B4A4_UNORM_PACK16 = 2
VK_FORMAT_B4G4R4A4_UNORM_PACK16 = 3
VK_FORMAT_R5G6B5_UNORM_PACK16 = 4
VK_FORMAT_B5G6R5_UNORM_PACK16 = 5
VK_FORMAT_R5G5B5A1_UNORM_PACK16 = 6
VK_FORMAT_B5G5R5A1_UNORM_PACK16 = 7
VK_FORMAT_A1R5G5B5_UNORM_PACK16 = 8
VK_FORMAT_R8_UNORM = 9
VK_FORMAT_R8_SNORM = 10
VK_FORMAT_R8_USCALED = 11
VK_FORMAT_R8_SSCALED = 12
VK_FORMAT_R8_UINT = 13
VK_FORMAT_R8_SINT = 14
VK_FORMAT_R8_SRGB = 15
VK_FORMAT_R8G8_UNORM = 16
VK_FORMAT_R8G8_SNORM = 17
VK_FORMAT_R8G8_USCALED = 18
VK_FORMAT_R8G8_SSCALED = 19
VK_FORMAT_R8G8_UINT = 20
VK_FORMAT_R8G8_SINT = 21
VK_FORMAT_R8G8_SRGB = 22
VK_FORMAT_R8G8B8_UNORM = 23
VK_FORMAT_R8G8B8_SNORM = 24
VK_FORMAT_R8G8B8_USCALED = 25
VK_FORMAT_R8G8B8_SSCALED = 26
VK_FORMAT_R8G8B8_UINT = 27
VK_FORMAT_R8G8B8_SINT = 28
VK_FORMAT_R8G8B8_SRGB = 29
VK_FORMAT_B8G8R8_UNORM = 30
VK_FORMAT_B8G8R8_SNORM = 31
VK_FORMAT_B8G8R8_USCALED = 32
VK_FORMAT_B8G8R8_SSCALED = 33
VK_FORMAT_B8G8R8_UINT = 34
VK_FORMAT_B8G8R8_SINT = 35
VK_FORMAT_B8G8R8_SRGB = 36
VK_FORMAT_R8G8B8A8_UNORM = 37
VK_FORMAT_R8G8B8A8_SNORM = 38
VK_FORMAT_R8G8B8A8_USCALED = 39
VK_FORMAT_R8G8B8A8_SSCALED = 40
VK_FORMAT_R8G8B8A8_UINT = 41
VK_FORMAT_R8G8B8A8_SINT = 42
VK_FORMAT_R8G8B8A8_SRGB = 43
VK_FORMAT_B8G8R8A8_UNORM = 44
VK_FORMAT_B8G8R8A8_SNORM = 45
VK_FORMAT_B8G8R8A8_USCALED = 46
VK_FORMAT_B8G8R8A8_SSCALED = 47
VK_FORMAT_B8G8R8A8_UINT = 48
VK_FORMAT_B8G8R8A8_SINT = 49
VK_FORMAT_B8G8R8A8_SRGB = 50
VK_FORMAT_A8B8G8R8_UNORM_PACK32 = 51
VK_FORMAT_A8B8G8R8_SNORM_PACK32 = 52
VK_FORMAT_A8B8G8R8_USCALED_PACK32 = 53
VK_FORMAT_A8B8G8R8_SSCALED_PACK32 = 54
VK_FORMAT_A8B8G8R8_UINT_PACK32 = 55
VK_FORMAT_A8B8G8R8_SINT_PACK32 = 56
VK_FORMAT_A8B8G8R8_SRGB_PACK32 = 57
VK_FORMAT_A2R10G10B10_UNORM_PACK32 = 58
VK_FORMAT_A2R10G10B10_SNORM_PACK32 = 59
VK_FORMAT_A2R10G10B10_USCALED_PACK32 = 60
VK_FORMAT_A2R10G10B10_SSCALED_PACK32 = 61
VK_FORMAT_A2R10G10B10_UINT_PACK32 = 62
VK_FORMAT_A2R10G10B10_SINT_PACK32 = 63
VK_FORMAT_A2B10G10R10_UNORM_PACK32 = 64
VK_FORMAT_A2B10G10R10_SNORM_PACK32 = 65
VK_FORMAT_A2B10G10R10_USCALED_PACK32 = 66
VK_FORMAT_A2B10G10R10_SSCALED_PACK32 = 67
VK_FORMAT_A2B10G10R10_UINT_PACK32 = 68
VK_FORMAT_A2B10G10R10_SINT_PACK32 = 69
VK_FORMAT_R16_UNORM = 70
VK_FORMAT_R16_SNORM = 71
VK_FORMAT_R16_USCALED = 72
VK_FORMAT_R16_SSCALED = 73
VK_FORMAT_R16_UINT = 74
VK_FORMAT_R16_SINT = 75
VK_FORMAT_R16_SFLOAT = 76
VK_FORMAT_R16G16_UNORM = 77
VK_FORMAT_R16G16_SNORM = 78
VK_FORMAT_R16G16_USCALED = 79
VK_FORMAT_R16G16_SSCALED = 80
VK_FORMAT_R16G16_UINT = 81
VK_FORMAT_R16G16_SINT = 82
VK_FORMAT_R16G16_SFLOAT = 83
VK_FORMAT_R16G16B16_UNORM = 84
VK_FORMAT_R16G16B16_SNORM = 85
VK_FORMAT_R16G16B16_USCALED = 86
VK_FORMAT_R16G16B16_SSCALED = 87
VK_FORMAT_R16G16B16_UINT = 88
VK_FORMAT_R16G16B16_SINT = 89
VK_FORMAT_R16G16B16_SFLOAT = 90
VK_FORMAT_R16G16B16A16_UNORM = 91
VK_FORMAT_R16G16B16A16_SNORM = 92
VK_FORMAT_R16G16B16A16_USCALED = 93
VK_FORMAT_R16G16B16A16_SSCALED = 94
VK_FORMAT_R16G16B16A16_UINT = 95
VK_FORMAT_R16G16B16A16_SINT = 96
VK_FORMAT_R16G16B16A16_SFLOAT = 97
VK_FORMAT_R32_UINT = 98
VK_FORMAT_R32_SINT = 99
VK_FORMAT_R32_SFLOAT = 100
VK_FORMAT_R32G32_UINT = 101
VK_FORMAT_R32G32_SINT = 102
VK_FORMAT_R32G32_SFLOAT = 103
VK_FORMAT_R32G32B32_UINT = 104
VK_FORMAT_R32G32B32_SINT = 105
VK_FORMAT_R32G32B32_SFLOAT = 106
VK_FORMAT_R32G32B32A32_UINT = 107
VK_FORMAT_R32G32B32A32_SINT = 108
VK_FORMAT_R32G32B32A32_SFLOAT = 109
VK_FORMAT_R64_UINT = 110
VK_FORMAT_R64_SINT = 111
VK_FORMAT_R64_SFLOAT = 112
VK_FORMAT_R64G64_UINT = 113
VK_FORMAT_R64G64_SINT = 114
VK_FORMAT_R64G64_SFLOAT = 115
VK_FORMAT_R64G64B64_UINT = 116
VK_FORMAT_R64G64B64_SINT = 117
VK_FORMAT_R64G64B64_SFLOAT = 118
VK_FORMAT_R64G64B64A64_UINT = 119
VK_FORMAT_R64G64B64A64_SINT = 120
VK_FORMAT_R64G64B64A64_SFLOAT = 121
VK_FORMAT_B10G11R11_UFLOAT_PACK32 = 122
VK_FORMAT_E5B9G9R9_UFLOAT_PACK32 = 123
VK_FORMAT_D16_UNORM = 124
VK_FORMAT_X8_D24_UNORM_PACK32 = 125
VK_FORMAT_D32_SFLOAT = 126
VK_FORMAT_S8_UINT = 127
VK_FORMAT_D16_UNORM_S8_UINT = 128
VK_FORMAT_D24_UNORM_S8_UINT = 129
VK_FORMAT_D32_SFLOAT_S8_UINT = 130
VK_FORMAT_BC1_RGB_UNORM_BLOCK = 131
VK_FORMAT_BC1_RGB_SRGB_BLOCK = 132
VK_FORMAT_BC1_RGBA_UNORM_BLOCK = 133
VK_FORMAT_BC1_RGBA_SRGB_BLOCK = 134
VK_FORMAT_BC2_UNORM_BLOCK = 135
VK_FORMAT_BC2_SRGB_BLOCK = 136
VK_FORMAT_BC3_UNORM_BLOCK = 137
VK_FORMAT_BC3_SRGB_BLOCK = 138
VK_FORMAT_BC4_UNORM_BLOCK = 139
VK_FORMAT_BC4_SNORM_BLOCK = 140
VK_FORMAT_BC5_UNORM_BLOCK = 141
VK_FORMAT_BC5_SNORM_BLOCK = 142
VK_FORMAT_BC6H_UFLOAT_BLOCK = 143
VK_FORMAT_BC6H_SFLOAT_BLOCK = 144
VK_FORMAT_BC7_UNORM_BLOCK = 145
VK_FORMAT_BC7_SRGB_BLOCK = 146
VK_FORMAT_ETC2_R8G8B8_UNORM_BLOCK = 147
VK_FORMAT_ETC2_R8G8B8_SRGB_BLOCK = 148
VK_FORMAT_ETC2_R8G8B8A1_UNORM_BLOCK = 149
VK_FORMAT_ETC2_R8G8B8A1_SRGB_BLOCK = 150
VK_FORMAT_ETC2_R8G8B8A8_UNORM_BLOCK = 151
VK_FORMAT_ETC2_R8G8B8A8_SRGB_BLOCK = 152
VK_FORMAT_EAC_R11_UNORM_BLOCK = 153
VK_FORMAT_EAC_R11_SNORM_BLOCK = 154
VK_FORMAT_EAC_R11G11_UNORM_BLOCK = 155
VK_FORMAT_EAC_R11G11_SNORM_BLOCK = 156
VK_FORMAT_ASTC_4x4_UNORM_BLOCK = 157
VK_FORMAT_ASTC_4x4_SRGB_BLOCK = 158
VK_FORMAT_ASTC_5x4_UNORM_BLOCK = 159
VK_FORMAT_ASTC_5x4_SRGB_BLOCK = 160
VK_FORMAT_ASTC_5x5_UNORM_BLOCK = 161
VK_FORMAT_ASTC_5x5_SRGB_BLOCK = 162
VK_FORMAT_ASTC_6x5_UNORM_BLOCK = 163
VK_FORMAT_ASTC_6x5_SRGB_BLOCK = 164
VK_FORMAT_ASTC_6x6_UNORM_BLOCK = 165
VK_FORMAT_ASTC_6x6_SRGB_BLOCK = 166
VK_FORMAT_ASTC_8x5_UNORM_BLOCK = 167
VK_FORMAT_ASTC_8x5_SRGB_BLOCK = 168
VK_FORMAT_ASTC_8x6_UNORM_BLOCK = 169
VK_FORMAT_ASTC_8x6_SRGB_BLOCK = 170
VK_FORMAT_ASTC_8x8_UNORM_BLOCK = 171
VK_FORMAT_ASTC_8x8_SRGB_BLOCK = 172
VK_FORMAT_ASTC_10x5_UNORM_BLOCK = 173
VK_FORMAT_ASTC_10x5_SRGB_BLOCK = 174
VK_FORMAT_ASTC_10x6_UNORM_BLOCK = 175
VK_FORMAT_ASTC_10x6_SRGB_BLOCK = 176
VK_FORMAT_ASTC_10x8_UNORM_BLOCK = 177
VK_FORMAT_ASTC_10x8_SRGB_BLOCK = 178
VK_FORMAT_ASTC_10x10_UNORM_BLOCK = 179
VK_FORMAT_ASTC_10x10_SRGB_BLOCK = 180
VK_FORMAT_ASTC_12x10_UNORM_BLOCK = 181
VK_FORMAT_ASTC_12x10_SRGB_BLOCK = 182
VK_FORMAT_ASTC_12x12_UNORM_BLOCK = 183
VK_FORMAT_ASTC_12x12_SRGB_BLOCK = 184
VK_FORMAT__BEGIN_RANGE = 0
VK_FORMAT__END_RANGE = 184
VK_FORMAT__RANGE_SIZE = 185
VK_FORMAT__MAX_ENUM = 2147483647
VK_FORMAT_G8B8G8R8_422_UNORM = 1000156000
VK_FORMAT_B8G8R8G8_422_UNORM = 1000156001
VK_FORMAT_G8_B8_R8_3PLANE_420_UNORM = 1000156002
VK_FORMAT_G8_B8R8_2PLANE_420_UNORM = 1000156003
VK_FORMAT_G8_B8_R8_3PLANE_422_UNORM = 1000156004
VK_FORMAT_G8_B8R8_2PLANE_422_UNORM = 1000156005
VK_FORMAT_G8_B8_R8_3PLANE_444_UNORM = 1000156006
VK_FORMAT_R10X6_UNORM_PACK16 = 1000156007
VK_FORMAT_R10X6G10X6_UNORM_2PACK16 = 1000156008
VK_FORMAT_R10X6G10X6B10X6A10X6_UNORM_4PACK16 = 1000156009
VK_FORMAT_G10X6B10X6G10X6R10X6_422_UNORM_4PACK16 = 1000156010
VK_FORMAT_B10X6G10X6R10X6G10X6_422_UNORM_4PACK16 = 1000156011
VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_420_UNORM_3PACK16 = 1000156012
VK_FORMAT_G10X6_B10X6R10X6_2PLANE_420_UNORM_3PACK16 = 1000156013
VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_422_UNORM_3PACK16 = 1000156014
VK_FORMAT_G10X6_B10X6R10X6_2PLANE_422_UNORM_3PACK16 = 1000156015
VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_444_UNORM_3PACK16 = 1000156016
VK_FORMAT_R12X4_UNORM_PACK16 = 1000156017
VK_FORMAT_R12X4G12X4_UNORM_2PACK16 = 1000156018
VK_FORMAT_R12X4G12X4B12X4A12X4_UNORM_4PACK16 = 1000156019
VK_FORMAT_G12X4B12X4G12X4R12X4_422_UNORM_4PACK16 = 1000156020
VK_FORMAT_B12X4G12X4R12X4G12X4_422_UNORM_4PACK16 = 1000156021
VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_420_UNORM_3PACK16 = 1000156022
VK_FORMAT_G12X4_B12X4R12X4_2PLANE_420_UNORM_3PACK16 = 1000156023
VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_422_UNORM_3PACK16 = 1000156024
VK_FORMAT_G12X4_B12X4R12X4_2PLANE_422_UNORM_3PACK16 = 1000156025
VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_444_UNORM_3PACK16 = 1000156026
VK_FORMAT_G16B16G16R16_422_UNORM = 1000156027
VK_FORMAT_B16G16R16G16_422_UNORM = 1000156028
VK_FORMAT_G16_B16_R16_3PLANE_420_UNORM = 1000156029
VK_FORMAT_G16_B16R16_2PLANE_420_UNORM = 1000156030
VK_FORMAT_G16_B16_R16_3PLANE_422_UNORM = 1000156031
VK_FORMAT_G16_B16R16_2PLANE_422_UNORM = 1000156032
VK_FORMAT_G16_B16_R16_3PLANE_444_UNORM = 1000156033
VK_FORMAT_G8_B8R8_2PLANE_444_UNORM = 1000330000
VK_FORMAT_G10X6_B10X6R10X6_2PLANE_444_UNORM_3PACK16 = 1000330001
VK_FORMAT_G12X4_B12X4R12X4_2PLANE_444_UNORM_3PACK16 = 1000330002
VK_FORMAT_G16_B16R16_2PLANE_444_UNORM = 1000330003
VK_FORMAT_A4R4G4B4_UNORM_PACK16 = 1000340000
VK_FORMAT_A4B4G4R4_UNORM_PACK16 = 1000340001
VK_FORMAT_ASTC_4x4_SFLOAT_BLOCK = 1000066000
VK_FORMAT_ASTC_5x4_SFLOAT_BLOCK = 1000066001
VK_FORMAT_ASTC_5x5_SFLOAT_BLOCK = 1000066002
VK_FORMAT_ASTC_6x5_SFLOAT_BLOCK = 1000066003
VK_FORMAT_ASTC_6x6_SFLOAT_BLOCK = 1000066004
VK_FORMAT_ASTC_8x5_SFLOAT_BLOCK = 1000066005
VK_FORMAT_ASTC_8x6_SFLOAT_BLOCK = 1000066006
VK_FORMAT_ASTC_8x8_SFLOAT_BLOCK = 1000066007
VK_FORMAT_ASTC_10x5_SFLOAT_BLOCK = 1000066008
VK_FORMAT_ASTC_10x6_SFLOAT_BLOCK = 1000066009
VK_FORMAT_ASTC_10x8_SFLOAT_BLOCK = 1000066010
VK_FORMAT_ASTC_10x10_SFLOAT_BLOCK = 1000066011
VK_FORMAT_ASTC_12x10_SFLOAT_BLOCK = 1000066012
VK_FORMAT_ASTC_12x12_SFLOAT_BLOCK = 1000066013
VK_FORMAT_FEATURE_SAMPLED_IMAGE_BIT = 0x00000001
VK_FORMAT_FEATURE_STORAGE_IMAGE_BIT = 0x00000002
VK_FORMAT_FEATURE_STORAGE_IMAGE_ATOMIC_BIT = 0x00000004
VK_FORMAT_FEATURE_UNIFORM_TEXEL_BUFFER_BIT = 0x00000008
VK_FORMAT_FEATURE_STORAGE_TEXEL_BUFFER_BIT = 0x00000010
VK_FORMAT_FEATURE_STORAGE_TEXEL_BUFFER_ATOMIC_BIT = 0x00000020
VK_FORMAT_FEATURE_VERTEX_BUFFER_BIT = 0x00000040
VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BIT = 0x00000080
VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BLEND_BIT = 0x00000100
VK_FORMAT_FEATURE_DEPTH_STENCIL_ATTACHMENT_BIT = 0x00000200
VK_FORMAT_FEATURE_BLIT_SRC_BIT = 0x00000400
VK_FORMAT_FEATURE_BLIT_DST_BIT = 0x00000800
VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_LINEAR_BIT = 0x00001000
VK_FORMAT_FEATURE_FLAG_BITS__MAX_ENUM = 2147483647
VK_FRONT_FACE_COUNTER_CLOCKWISE = 0
VK_FRONT_FACE_CLOCKWISE = 1
VK_FRONT_FACE__BEGIN_RANGE = 0
VK_FRONT_FACE__END_RANGE = 1
VK_FRONT_FACE__RANGE_SIZE = 2
VK_FRONT_FACE__MAX_ENUM = 2147483647
VK_IMAGE_ASPECT_COLOR_BIT = 0x00000001
VK_IMAGE_ASPECT_DEPTH_BIT = 0x00000002
VK_IMAGE_ASPECT_STENCIL_BIT = 0x00000004
VK_IMAGE_ASPECT_METADATA_BIT = 0x00000008
VK_IMAGE_ASPECT_FLAG_BITS__MAX_ENUM = 2147483647
VK_IMAGE_ASPECT_NONE = 0
VK_IMAGE_CREATE_SPARSE_BINDING_BIT = 0x00000001
VK_IMAGE_CREATE_SPARSE_RESIDENCY_BIT = 0x00000002
VK_IMAGE_CREATE_SPARSE_ALIASED_BIT = 0x00000004
VK_IMAGE_CREATE_MUTABLE_FORMAT_BIT = 0x00000008
VK_IMAGE_CREATE_CUBE_COMPATIBLE_BIT = 0x00000010
VK_IMAGE_CREATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_IMAGE_LAYOUT_UNDEFINED = 0
VK_IMAGE_LAYOUT_GENERAL = 1
VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL = 2
VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL = 3
VK_IMAGE_LAYOUT_DEPTH_STENCIL_READ_ONLY_OPTIMAL = 4
VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL = 5
VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL = 6
VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL = 7
VK_IMAGE_LAYOUT_PREINITIALIZED = 8
VK_IMAGE_LAYOUT__BEGIN_RANGE = 0
VK_IMAGE_LAYOUT__END_RANGE = 8
VK_IMAGE_LAYOUT__RANGE_SIZE = 9
VK_IMAGE_LAYOUT__MAX_ENUM = 2147483647
VK_IMAGE_LAYOUT_DEPTH_READ_ONLY_STENCIL_ATTACHMENT_OPTIMAL = 1000117000
VK_IMAGE_LAYOUT_DEPTH_ATTACHMENT_STENCIL_READ_ONLY_OPTIMAL = 1000117001
VK_IMAGE_LAYOUT_DEPTH_ATTACHMENT_OPTIMAL = 1000241000
VK_IMAGE_LAYOUT_DEPTH_READ_ONLY_OPTIMAL = 1000241001
VK_IMAGE_LAYOUT_STENCIL_ATTACHMENT_OPTIMAL = 1000241002
VK_IMAGE_LAYOUT_STENCIL_READ_ONLY_OPTIMAL = 1000241003
VK_IMAGE_LAYOUT_READ_ONLY_OPTIMAL = 1000314000
VK_IMAGE_LAYOUT_ATTACHMENT_OPTIMAL = 1000314001
VK_IMAGE_TILING_OPTIMAL = 0
VK_IMAGE_TILING_LINEAR = 1
VK_IMAGE_TILING__BEGIN_RANGE = 0
VK_IMAGE_TILING__END_RANGE = 1
VK_IMAGE_TILING__RANGE_SIZE = 2
VK_IMAGE_TILING__MAX_ENUM = 2147483647
VK_IMAGE_TYPE_1D = 0
VK_IMAGE_TYPE_2D = 1
VK_IMAGE_TYPE_3D = 2
VK_IMAGE_TYPE__BEGIN_RANGE = 0
VK_IMAGE_TYPE__END_RANGE = 2
VK_IMAGE_TYPE__RANGE_SIZE = 3
VK_IMAGE_TYPE__MAX_ENUM = 2147483647
VK_IMAGE_USAGE_TRANSFER_SRC_BIT = 0x00000001
VK_IMAGE_USAGE_TRANSFER_DST_BIT = 0x00000002
VK_IMAGE_USAGE_SAMPLED_BIT = 0x00000004
VK_IMAGE_USAGE_STORAGE_BIT = 0x00000008
VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT = 0x00000010
VK_IMAGE_USAGE_DEPTH_STENCIL_ATTACHMENT_BIT = 0x00000020
VK_IMAGE_USAGE_TRANSIENT_ATTACHMENT_BIT = 0x00000040
VK_IMAGE_USAGE_INPUT_ATTACHMENT_BIT = 0x00000080
VK_IMAGE_USAGE_FLAG_BITS__MAX_ENUM = 2147483647
VK_IMAGE_VIEW_TYPE_1D = 0
VK_IMAGE_VIEW_TYPE_2D = 1
VK_IMAGE_VIEW_TYPE_3D = 2
VK_IMAGE_VIEW_TYPE_CUBE = 3
VK_IMAGE_VIEW_TYPE_1D_ARRAY = 4
VK_IMAGE_VIEW_TYPE_2D_ARRAY = 5
VK_IMAGE_VIEW_TYPE_CUBE_ARRAY = 6
VK_IMAGE_VIEW_TYPE__BEGIN_RANGE = 0
VK_IMAGE_VIEW_TYPE__END_RANGE = 6
VK_IMAGE_VIEW_TYPE__RANGE_SIZE = 7
VK_IMAGE_VIEW_TYPE__MAX_ENUM = 2147483647
VK_SHARING_MODE_EXCLUSIVE = 0
VK_SHARING_MODE_CONCURRENT = 1
VK_SHARING_MODE__BEGIN_RANGE = 0
VK_SHARING_MODE__END_RANGE = 1
VK_SHARING_MODE__RANGE_SIZE = 2
VK_SHARING_MODE__MAX_ENUM = 2147483647
VK_INDEX_TYPE_UINT16 = 0
VK_INDEX_TYPE_UINT32 = 1
VK_INDEX_TYPE__BEGIN_RANGE = 0
VK_INDEX_TYPE__END_RANGE = 1
VK_INDEX_TYPE__RANGE_SIZE = 2
VK_INDEX_TYPE__MAX_ENUM = 2147483647
VK_LOGIC_OP_CLEAR = 0
VK_LOGIC_OP_AND = 1
VK_LOGIC_OP_AND_REVERSE = 2
VK_LOGIC_OP_COPY = 3
VK_LOGIC_OP_AND_INVERTED = 4
VK_LOGIC_OP_NO_OP = 5
VK_LOGIC_OP_XOR = 6
VK_LOGIC_OP_OR = 7
VK_LOGIC_OP_NOR = 8
VK_LOGIC_OP_EQUIVALENT = 9
VK_LOGIC_OP_INVERT = 10
VK_LOGIC_OP_OR_REVERSE = 11
VK_LOGIC_OP_COPY_INVERTED = 12
VK_LOGIC_OP_OR_INVERTED = 13
VK_LOGIC_OP_NAND = 14
VK_LOGIC_OP_SET = 15
VK_LOGIC_OP__BEGIN_RANGE = 0
VK_LOGIC_OP__END_RANGE = 15
VK_LOGIC_OP__RANGE_SIZE = 16
VK_LOGIC_OP__MAX_ENUM = 2147483647
VK_MEMORY_HEAP_DEVICE_LOCAL_BIT = 0x00000001
VK_MEMORY_HEAP_FLAG_BITS__MAX_ENUM = 2147483647
VK_ACCESS_INDIRECT_COMMAND_READ_BIT = 0x00000001
VK_ACCESS_INDEX_READ_BIT = 0x00000002
VK_ACCESS_VERTEX_ATTRIBUTE_READ_BIT = 0x00000004
VK_ACCESS_UNIFORM_READ_BIT = 0x00000008
VK_ACCESS_INPUT_ATTACHMENT_READ_BIT = 0x00000010
VK_ACCESS_SHADER_READ_BIT = 0x00000020
VK_ACCESS_SHADER_WRITE_BIT = 0x00000040
VK_ACCESS_COLOR_ATTACHMENT_READ_BIT = 0x00000080
VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT = 0x00000100
VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_READ_BIT = 0x00000200
VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT = 0x00000400
VK_ACCESS_TRANSFER_READ_BIT = 0x00000800
VK_ACCESS_TRANSFER_WRITE_BIT = 0x00001000
VK_ACCESS_HOST_READ_BIT = 0x00002000
VK_ACCESS_HOST_WRITE_BIT = 0x00004000
VK_ACCESS_MEMORY_READ_BIT = 0x00008000
VK_ACCESS_MEMORY_WRITE_BIT = 0x00010000
VK_ACCESS_FLAG_BITS__MAX_ENUM = 2147483647
VK_ACCESS_NONE = 0
VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT = 0x00000001
VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT = 0x00000002
VK_MEMORY_PROPERTY_HOST_COHERENT_BIT = 0x00000004
VK_MEMORY_PROPERTY_HOST_CACHED_BIT = 0x00000008
VK_MEMORY_PROPERTY_LAZILY_ALLOCATED_BIT = 0x00000010
VK_MEMORY_PROPERTY_FLAG_BITS__MAX_ENUM = 2147483647
VK_PHYSICAL_DEVICE_TYPE_OTHER = 0
VK_PHYSICAL_DEVICE_TYPE_INTEGRATED_GPU = 1
VK_PHYSICAL_DEVICE_TYPE_DISCRETE_GPU = 2
VK_PHYSICAL_DEVICE_TYPE_VIRTUAL_GPU = 3
VK_PHYSICAL_DEVICE_TYPE_CPU = 4
VK_PHYSICAL_DEVICE_TYPE__BEGIN_RANGE = 0
VK_PHYSICAL_DEVICE_TYPE__END_RANGE = 4
VK_PHYSICAL_DEVICE_TYPE__RANGE_SIZE = 5
VK_PHYSICAL_DEVICE_TYPE__MAX_ENUM = 2147483647
VK_PIPELINE_BIND_POINT_GRAPHICS = 0
VK_PIPELINE_BIND_POINT_COMPUTE = 1
VK_PIPELINE_BIND_POINT__BEGIN_RANGE = 0
VK_PIPELINE_BIND_POINT__END_RANGE = 1
VK_PIPELINE_BIND_POINT__RANGE_SIZE = 2
VK_PIPELINE_BIND_POINT__MAX_ENUM = 2147483647
VK_PIPELINE_CREATE_DISABLE_OPTIMIZATION_BIT = 0x00000001
VK_PIPELINE_CREATE_ALLOW_DERIVATIVES_BIT = 0x00000002
VK_PIPELINE_CREATE_DERIVATIVE_BIT = 0x00000004
VK_PIPELINE_CREATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_PRIMITIVE_TOPOLOGY_POINT_LIST = 0
VK_PRIMITIVE_TOPOLOGY_LINE_LIST = 1
VK_PRIMITIVE_TOPOLOGY_LINE_STRIP = 2
VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST = 3
VK_PRIMITIVE_TOPOLOGY_TRIANGLE_STRIP = 4
VK_PRIMITIVE_TOPOLOGY_TRIANGLE_FAN = 5
VK_PRIMITIVE_TOPOLOGY_LINE_LIST_WITH_ADJACENCY = 6
VK_PRIMITIVE_TOPOLOGY_LINE_STRIP_WITH_ADJACENCY = 7
VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST_WITH_ADJACENCY = 8
VK_PRIMITIVE_TOPOLOGY_TRIANGLE_STRIP_WITH_ADJACENCY = 9
VK_PRIMITIVE_TOPOLOGY_PATCH_LIST = 10
VK_PRIMITIVE_TOPOLOGY__BEGIN_RANGE = 0
VK_PRIMITIVE_TOPOLOGY__END_RANGE = 10
VK_PRIMITIVE_TOPOLOGY__RANGE_SIZE = 11
VK_PRIMITIVE_TOPOLOGY__MAX_ENUM = 2147483647
VK_QUERY_CONTROL_PRECISE_BIT = 0x00000001
VK_QUERY_CONTROL_FLAG_BITS__MAX_ENUM = 2147483647
VK_QUERY_PIPELINE_STATISTIC_INPUT_ASSEMBLY_VERTICES_BIT = 0x00000001
VK_QUERY_PIPELINE_STATISTIC_INPUT_ASSEMBLY_PRIMITIVES_BIT = 0x00000002
VK_QUERY_PIPELINE_STATISTIC_VERTEX_SHADER_INVOCATIONS_BIT = 0x00000004
VK_QUERY_PIPELINE_STATISTIC_GEOMETRY_SHADER_INVOCATIONS_BIT = 0x00000008
VK_QUERY_PIPELINE_STATISTIC_GEOMETRY_SHADER_PRIMITIVES_BIT = 0x00000010
VK_QUERY_PIPELINE_STATISTIC_CLIPPING_INVOCATIONS_BIT = 0x00000020
VK_QUERY_PIPELINE_STATISTIC_CLIPPING_PRIMITIVES_BIT = 0x00000040
VK_QUERY_PIPELINE_STATISTIC_FRAGMENT_SHADER_INVOCATIONS_BIT = 0x00000080
VK_QUERY_PIPELINE_STATISTIC_TESSELLATION_CONTROL_SHADER_PATCHES_BIT = 0x00000100
VK_QUERY_PIPELINE_STATISTIC_TESSELLATION_EVALUATION_SHADER_INVOCATIONS_BIT = 0x00000200
VK_QUERY_PIPELINE_STATISTIC_COMPUTE_SHADER_INVOCATIONS_BIT = 0x00000400
VK_QUERY_PIPELINE_STATISTIC_FLAG_BITS__MAX_ENUM = 2147483647
VK_QUERY_RESULT_64_BIT = 0x00000001
VK_QUERY_RESULT_WAIT_BIT = 0x00000002
VK_QUERY_RESULT_WITH_AVAILABILITY_BIT = 0x00000004
VK_QUERY_RESULT_PARTIAL_BIT = 0x00000008
VK_QUERY_RESULT_FLAG_BITS__MAX_ENUM = 2147483647
VK_QUERY_TYPE_OCCLUSION = 0
VK_QUERY_TYPE_PIPELINE_STATISTICS = 1
VK_QUERY_TYPE_TIMESTAMP = 2
VK_QUERY_TYPE__BEGIN_RANGE = 0
VK_QUERY_TYPE__END_RANGE = 2
VK_QUERY_TYPE__RANGE_SIZE = 3
VK_QUERY_TYPE__MAX_ENUM = 2147483647
VK_QUEUE_GRAPHICS_BIT = 0x00000001
VK_QUEUE_COMPUTE_BIT = 0x00000002
VK_QUEUE_TRANSFER_BIT = 0x00000004
VK_QUEUE_SPARSE_BINDING_BIT = 0x00000008
VK_QUEUE_FLAG_BITS__MAX_ENUM = 2147483647
VK_SUBPASS_CONTENTS_INLINE = 0
VK_SUBPASS_CONTENTS_SECONDARY_COMMAND_BUFFERS = 1
VK_SUBPASS_CONTENTS__BEGIN_RANGE = 0
VK_SUBPASS_CONTENTS__END_RANGE = 1
VK_SUBPASS_CONTENTS__RANGE_SIZE = 2
VK_SUBPASS_CONTENTS__MAX_ENUM = 2147483647
VK_SUCCESS = 0
VK_NOT_READY = 1
VK_TIMEOUT = 2
VK_EVENT_SET = 3
VK_EVENT_RESET = 4
VK_INCOMPLETE = 5
VK_ERROR_OUT_OF_HOST_MEMORY = -1
VK_ERROR_OUT_OF_DEVICE_MEMORY = -2
VK_ERROR_INITIALIZATION_FAILED = -3
VK_ERROR_DEVICE_LOST = -4
VK_ERROR_MEMORY_MAP_FAILED = -5
VK_ERROR_LAYER_NOT_PRESENT = -6
VK_ERROR_EXTENSION_NOT_PRESENT = -7
VK_ERROR_FEATURE_NOT_PRESENT = -8
VK_ERROR_INCOMPATIBLE_DRIVER = -9
VK_ERROR_TOO_MANY_OBJECTS = -10
VK_ERROR_FORMAT_NOT_SUPPORTED = -11
VK_ERROR_FRAGMENTED_POOL = -12
VK_ERROR_UNKNOWN = -13
VK_RESULT__BEGIN_RANGE = -13
VK_RESULT__END_RANGE = 5
VK_RESULT__RANGE_SIZE = 19
VK_RESULT__MAX_ENUM = 2147483647
VK_ERROR_OUT_OF_POOL_MEMORY = 1000069000
VK_ERROR_INVALID_EXTERNAL_HANDLE = 1000072003
VK_ERROR_FRAGMENTATION = 1000161000
VK_ERROR_INVALID_OPAQUE_CAPTURE_ADDRESS = 1000257000
VK_PIPELINE_COMPILE_REQUIRED = 1000297000
VK_ERROR_VALIDATION_FAILED = 1000011001
VK_ERROR_INVALID_PIPELINE_CACHE_DATA = 1000298000
VK_ERROR_NO_PIPELINE_MATCH = 1000298001
VK_ERROR_SURFACE_LOST_KHR = -1000000000
VK_ERROR_NATIVE_WINDOW_IN_USE_KHR = -1000000001
VK_SUBOPTIMAL_KHR = 1000001003
VK_ERROR_OUT_OF_DATE_KHR = -1000001004
VK_ERROR_INCOMPATIBLE_DISPLAY_KHR = -1000003001
VK_ERROR_VALIDATION_FAILED_EXT = None
VK_ERROR_INVALID_SHADER_NV = -1000012000
VK_ERROR_IMAGE_USAGE_NOT_SUPPORTED_KHR = -1000023000
VK_ERROR_VIDEO_PICTURE_LAYOUT_NOT_SUPPORTED_KHR = -1000023001
VK_ERROR_VIDEO_PROFILE_OPERATION_NOT_SUPPORTED_KHR = -1000023002
VK_ERROR_VIDEO_PROFILE_FORMAT_NOT_SUPPORTED_KHR = -1000023003
VK_ERROR_VIDEO_PROFILE_CODEC_NOT_SUPPORTED_KHR = -1000023004
VK_ERROR_VIDEO_STD_VERSION_NOT_SUPPORTED_KHR = -1000023005
VK_ERROR_OUT_OF_POOL_MEMORY_KHR = None
VK_ERROR_INVALID_EXTERNAL_HANDLE_KHR = None
VK_ERROR_INVALID_DRM_FORMAT_MODIFIER_PLANE_LAYOUT_EXT = -1000158000
VK_ERROR_FRAGMENTATION_EXT = None
VK_ERROR_NOT_PERMITTED_EXT = None
VK_ERROR_NOT_PERMITTED_KHR = -1000188001
VK_ERROR_INVALID_DEVICE_ADDRESS_EXT = None
VK_ERROR_FULL_SCREEN_EXCLUSIVE_MODE_LOST_EXT = -1000255000
VK_ERROR_INVALID_OPAQUE_CAPTURE_ADDRESS_KHR = None
VK_THREAD_IDLE_KHR = 1000268000
VK_THREAD_DONE_KHR = 1000268001
VK_OPERATION_DEFERRED_KHR = 1000268002
VK_OPERATION_NOT_DEFERRED_KHR = 1000268003
VK_PIPELINE_COMPILE_REQUIRED_EXT = None
VK_ERROR_PIPELINE_COMPILE_REQUIRED_EXT = None
VK_ERROR_INVALID_VIDEO_STD_PARAMETERS_KHR = -1000299000
VK_ERROR_COMPRESSION_EXHAUSTED_EXT = -1000338000
VK_ERROR_INCOMPATIBLE_SHADER_BINARY_EXT = 1000482000
VK_SHADER_STAGE_VERTEX_BIT = 0x00000001
VK_SHADER_STAGE_TESSELLATION_CONTROL_BIT = 0x00000002
VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT = 0x00000004
VK_SHADER_STAGE_GEOMETRY_BIT = 0x00000008
VK_SHADER_STAGE_FRAGMENT_BIT = 0x00000010
VK_SHADER_STAGE_COMPUTE_BIT = 0x00000020
VK_SHADER_STAGE_ALL_GRAPHICS = 0x0000001F
VK_SHADER_STAGE_ALL = 0x7FFFFFFF
VK_SHADER_STAGE_FLAG_BITS__MAX_ENUM = 2147483647
VK_SPARSE_MEMORY_BIND_METADATA_BIT = 0x00000001
VK_SPARSE_MEMORY_BIND_FLAG_BITS__MAX_ENUM = 2147483647
VK_STENCIL_FACE_FRONT_BIT = 0x00000001
VK_STENCIL_FACE_BACK_BIT = 0x00000002
VK_STENCIL_FACE_FRONT_AND_BACK = 0x00000003
VK_STENCIL_FRONT_AND_BACK = 0x00000003
VK_STENCIL_FACE_FLAG_BITS__MAX_ENUM = 2147483647
VK_STENCIL_OP_KEEP = 0
VK_STENCIL_OP_ZERO = 1
VK_STENCIL_OP_REPLACE = 2
VK_STENCIL_OP_INCREMENT_AND_CLAMP = 3
VK_STENCIL_OP_DECREMENT_AND_CLAMP = 4
VK_STENCIL_OP_INVERT = 5
VK_STENCIL_OP_INCREMENT_AND_WRAP = 6
VK_STENCIL_OP_DECREMENT_AND_WRAP = 7
VK_STENCIL_OP__BEGIN_RANGE = 0
VK_STENCIL_OP__END_RANGE = 7
VK_STENCIL_OP__RANGE_SIZE = 8
VK_STENCIL_OP__MAX_ENUM = 2147483647
VK_STRUCTURE_TYPE_APPLICATION_INFO = 0
VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO = 1
VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO = 2
VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO = 3
VK_STRUCTURE_TYPE_SUBMIT_INFO = 4
VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO = 5
VK_STRUCTURE_TYPE_MAPPED_MEMORY_RANGE = 6
VK_STRUCTURE_TYPE_BIND_SPARSE_INFO = 7
VK_STRUCTURE_TYPE_FENCE_CREATE_INFO = 8
VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO = 9
VK_STRUCTURE_TYPE_EVENT_CREATE_INFO = 10
VK_STRUCTURE_TYPE_QUERY_POOL_CREATE_INFO = 11
VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO = 12
VK_STRUCTURE_TYPE_BUFFER_VIEW_CREATE_INFO = 13
VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO = 14
VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO = 15
VK_STRUCTURE_TYPE_SHADER_MODULE_CREATE_INFO = 16
VK_STRUCTURE_TYPE_PIPELINE_CACHE_CREATE_INFO = 17
VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_CREATE_INFO = 18
VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_STATE_CREATE_INFO = 19
VK_STRUCTURE_TYPE_PIPELINE_INPUT_ASSEMBLY_STATE_CREATE_INFO = 20
VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_STATE_CREATE_INFO = 21
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_STATE_CREATE_INFO = 22
VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_CREATE_INFO = 23
VK_STRUCTURE_TYPE_PIPELINE_MULTISAMPLE_STATE_CREATE_INFO = 24
VK_STRUCTURE_TYPE_PIPELINE_DEPTH_STENCIL_STATE_CREATE_INFO = 25
VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_STATE_CREATE_INFO = 26
VK_STRUCTURE_TYPE_PIPELINE_DYNAMIC_STATE_CREATE_INFO = 27
VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_CREATE_INFO = 28
VK_STRUCTURE_TYPE_COMPUTE_PIPELINE_CREATE_INFO = 29
VK_STRUCTURE_TYPE_PIPELINE_LAYOUT_CREATE_INFO = 30
VK_STRUCTURE_TYPE_SAMPLER_CREATE_INFO = 31
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_CREATE_INFO = 32
VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_CREATE_INFO = 33
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_ALLOCATE_INFO = 34
VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET = 35
VK_STRUCTURE_TYPE_COPY_DESCRIPTOR_SET = 36
VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO = 37
VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO = 38
VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO = 39
VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO = 40
VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_INFO = 41
VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO = 42
VK_STRUCTURE_TYPE_RENDER_PASS_BEGIN_INFO = 43
VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER = 44
VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER = 45
VK_STRUCTURE_TYPE_MEMORY_BARRIER = 46
VK_STRUCTURE_TYPE_LOADER_INSTANCE_CREATE_INFO = 47
VK_STRUCTURE_TYPE_LOADER_DEVICE_CREATE_INFO = 48
VK_STRUCTURE_TYPE__BEGIN_RANGE = 0
VK_STRUCTURE_TYPE__END_RANGE = 48
VK_STRUCTURE_TYPE__RANGE_SIZE = 49
VK_STRUCTURE_TYPE__MAX_ENUM = 2147483647
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_PROPERTIES = 1000094000
VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_INFO = 1000157000
VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_INFO = 1000157001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_16BIT_STORAGE_FEATURES = 1000083000
VK_STRUCTURE_TYPE_MEMORY_DEDICATED_REQUIREMENTS = 1000127000
VK_STRUCTURE_TYPE_MEMORY_DEDICATED_ALLOCATE_INFO = 1000127001
VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_FLAGS_INFO = 1000060000
VK_STRUCTURE_TYPE_DEVICE_GROUP_RENDER_PASS_BEGIN_INFO = 1000060003
VK_STRUCTURE_TYPE_DEVICE_GROUP_COMMAND_BUFFER_BEGIN_INFO = 1000060004
VK_STRUCTURE_TYPE_DEVICE_GROUP_SUBMIT_INFO = 1000060005
VK_STRUCTURE_TYPE_DEVICE_GROUP_BIND_SPARSE_INFO = 1000060006
VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_DEVICE_GROUP_INFO = 1000060013
VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_DEVICE_GROUP_INFO = 1000060014
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GROUP_PROPERTIES = 1000070000
VK_STRUCTURE_TYPE_DEVICE_GROUP_DEVICE_CREATE_INFO = 1000070001
VK_STRUCTURE_TYPE_BUFFER_MEMORY_REQUIREMENTS_INFO_2 = 1000146000
VK_STRUCTURE_TYPE_IMAGE_MEMORY_REQUIREMENTS_INFO_2 = 1000146001
VK_STRUCTURE_TYPE_IMAGE_SPARSE_MEMORY_REQUIREMENTS_INFO_2 = 1000146002
VK_STRUCTURE_TYPE_MEMORY_REQUIREMENTS_2 = 1000146003
VK_STRUCTURE_TYPE_SPARSE_IMAGE_MEMORY_REQUIREMENTS_2 = 1000146004
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FEATURES_2 = 1000059000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROPERTIES_2 = 1000059001
VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_2 = 1000059002
VK_STRUCTURE_TYPE_IMAGE_FORMAT_PROPERTIES_2 = 1000059003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_FORMAT_INFO_2 = 1000059004
VK_STRUCTURE_TYPE_QUEUE_FAMILY_PROPERTIES_2 = 1000059005
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PROPERTIES_2 = 1000059006
VK_STRUCTURE_TYPE_SPARSE_IMAGE_FORMAT_PROPERTIES_2 = 1000059007
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SPARSE_IMAGE_FORMAT_INFO_2 = 1000059008
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_POINT_CLIPPING_PROPERTIES = 1000117000
VK_STRUCTURE_TYPE_RENDER_PASS_INPUT_ATTACHMENT_ASPECT_CREATE_INFO = 1000117001
VK_STRUCTURE_TYPE_IMAGE_VIEW_USAGE_CREATE_INFO = 1000117002
VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_DOMAIN_ORIGIN_STATE_CREATE_INFO = 1000117003
VK_STRUCTURE_TYPE_RENDER_PASS_MULTIVIEW_CREATE_INFO = 1000053000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_FEATURES = 1000053001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PROPERTIES = 1000053002
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VARIABLE_POINTERS_FEATURES = 1000120000
VK_STRUCTURE_TYPE_PROTECTED_SUBMIT_INFO = 1000145000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_FEATURES = 1000145001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_PROPERTIES = 1000145002
VK_STRUCTURE_TYPE_DEVICE_QUEUE_INFO_2 = 1000145003
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_CREATE_INFO = 1000156000
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_INFO = 1000156001
VK_STRUCTURE_TYPE_BIND_IMAGE_PLANE_MEMORY_INFO = 1000156002
VK_STRUCTURE_TYPE_IMAGE_PLANE_MEMORY_REQUIREMENTS_INFO = 1000156003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_YCBCR_CONVERSION_FEATURES = 1000156004
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_IMAGE_FORMAT_PROPERTIES = 1000156005
VK_STRUCTURE_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_CREATE_INFO = 1000085000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_IMAGE_FORMAT_INFO = 1000071000
VK_STRUCTURE_TYPE_EXTERNAL_IMAGE_FORMAT_PROPERTIES = 1000071001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_BUFFER_INFO = 1000071002
VK_STRUCTURE_TYPE_EXTERNAL_BUFFER_PROPERTIES = 1000071003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ID_PROPERTIES = 1000071004
VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_BUFFER_CREATE_INFO = 1000072000
VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO = 1000072001
VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO = 1000072002
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FENCE_INFO = 1000112000
VK_STRUCTURE_TYPE_EXTERNAL_FENCE_PROPERTIES = 1000112001
VK_STRUCTURE_TYPE_EXPORT_FENCE_CREATE_INFO = 1000113000
VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_CREATE_INFO = 1000077000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SEMAPHORE_INFO = 1000076000
VK_STRUCTURE_TYPE_EXTERNAL_SEMAPHORE_PROPERTIES = 1000076001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_3_PROPERTIES = 1000168000
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_SUPPORT = 1000168001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_DRAW_PARAMETERS_FEATURES = 1000063000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_1_FEATURES = 49
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_1_PROPERTIES = 50
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_2_FEATURES = 51
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_2_PROPERTIES = 52
VK_STRUCTURE_TYPE_IMAGE_FORMAT_LIST_CREATE_INFO = 1000147000
VK_STRUCTURE_TYPE_ATTACHMENT_DESCRIPTION_2 = 1000109000
VK_STRUCTURE_TYPE_ATTACHMENT_REFERENCE_2 = 1000109001
VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_2 = 1000109002
VK_STRUCTURE_TYPE_SUBPASS_DEPENDENCY_2 = 1000109003
VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO_2 = 1000109004
VK_STRUCTURE_TYPE_SUBPASS_BEGIN_INFO = 1000109005
VK_STRUCTURE_TYPE_SUBPASS_END_INFO = 1000109006
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_8BIT_STORAGE_FEATURES = 1000177000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DRIVER_PROPERTIES = 1000196000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_INT64_FEATURES = 1000180000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_FLOAT16_INT8_FEATURES = 1000082000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT_CONTROLS_PROPERTIES = 1000197000
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_BINDING_FLAGS_CREATE_INFO = 1000161000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_FEATURES = 1000161001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_PROPERTIES = 1000161002
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_ALLOCATE_INFO = 1000161003
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_LAYOUT_SUPPORT = 1000161004
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_STENCIL_RESOLVE_PROPERTIES = 1000199000
VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_DEPTH_STENCIL_RESOLVE = 1000199001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SCALAR_BLOCK_LAYOUT_FEATURES = 1000221000
VK_STRUCTURE_TYPE_IMAGE_STENCIL_USAGE_CREATE_INFO = 1000246000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_FILTER_MINMAX_PROPERTIES = 1000130000
VK_STRUCTURE_TYPE_SAMPLER_REDUCTION_MODE_CREATE_INFO = 1000130001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_MEMORY_MODEL_FEATURES = 1000211000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGELESS_FRAMEBUFFER_FEATURES = 1000108000
VK_STRUCTURE_TYPE_FRAMEBUFFER_ATTACHMENTS_CREATE_INFO = 1000108001
VK_STRUCTURE_TYPE_FRAMEBUFFER_ATTACHMENT_IMAGE_INFO = 1000108002
VK_STRUCTURE_TYPE_RENDER_PASS_ATTACHMENT_BEGIN_INFO = 1000108003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_UNIFORM_BUFFER_STANDARD_LAYOUT_FEATURES = 1000253000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_SUBGROUP_EXTENDED_TYPES_FEATURES = 1000175000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SEPARATE_DEPTH_STENCIL_LAYOUTS_FEATURES = 1000241000
VK_STRUCTURE_TYPE_ATTACHMENT_REFERENCE_STENCIL_LAYOUT = 1000241001
VK_STRUCTURE_TYPE_ATTACHMENT_DESCRIPTION_STENCIL_LAYOUT = 1000241002
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_HOST_QUERY_RESET_FEATURES = 1000261000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TIMELINE_SEMAPHORE_FEATURES = 1000207000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TIMELINE_SEMAPHORE_PROPERTIES = 1000207001
VK_STRUCTURE_TYPE_SEMAPHORE_TYPE_CREATE_INFO = 1000207002
VK_STRUCTURE_TYPE_TIMELINE_SEMAPHORE_SUBMIT_INFO = 1000207003
VK_STRUCTURE_TYPE_SEMAPHORE_WAIT_INFO = 1000207004
VK_STRUCTURE_TYPE_SEMAPHORE_SIGNAL_INFO = 1000207005
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BUFFER_DEVICE_ADDRESS_FEATURES = 1000257000
VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_INFO = 1000244001
VK_STRUCTURE_TYPE_BUFFER_OPAQUE_CAPTURE_ADDRESS_CREATE_INFO = 1000257002
VK_STRUCTURE_TYPE_MEMORY_OPAQUE_CAPTURE_ADDRESS_ALLOCATE_INFO = 1000257003
VK_STRUCTURE_TYPE_DEVICE_MEMORY_OPAQUE_CAPTURE_ADDRESS_INFO = 1000257004
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_3_FEATURES = 53
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_3_PROPERTIES = 54
VK_STRUCTURE_TYPE_PIPELINE_CREATION_FEEDBACK_CREATE_INFO = 1000192000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_TERMINATE_INVOCATION_FEATURES = 1000215000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TOOL_PROPERTIES = 1000245000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_DEMOTE_TO_HELPER_INVOCATION_FEATURES = 1000276000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRIVATE_DATA_FEATURES = 1000295000
VK_STRUCTURE_TYPE_DEVICE_PRIVATE_DATA_CREATE_INFO = 1000295001
VK_STRUCTURE_TYPE_PRIVATE_DATA_SLOT_CREATE_INFO = 1000295002
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_CREATION_CACHE_CONTROL_FEATURES = 1000297000
VK_STRUCTURE_TYPE_MEMORY_BARRIER_2 = 1000314000
VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER_2 = 1000314001
VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER_2 = 1000314002
VK_STRUCTURE_TYPE_DEPENDENCY_INFO = 1000314003
VK_STRUCTURE_TYPE_SUBMIT_INFO_2 = 1000314004
VK_STRUCTURE_TYPE_SEMAPHORE_SUBMIT_INFO = 1000314005
VK_STRUCTURE_TYPE_COMMAND_BUFFER_SUBMIT_INFO = 1000314006
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SYNCHRONIZATION_2_FEATURES = 1000314007
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ZERO_INITIALIZE_WORKGROUP_MEMORY_FEATURES = 1000325000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_ROBUSTNESS_FEATURES = 1000335000
VK_STRUCTURE_TYPE_COPY_BUFFER_INFO_2 = 1000337000
VK_STRUCTURE_TYPE_COPY_IMAGE_INFO_2 = 1000337001
VK_STRUCTURE_TYPE_COPY_BUFFER_TO_IMAGE_INFO_2 = 1000337002
VK_STRUCTURE_TYPE_COPY_IMAGE_TO_BUFFER_INFO_2 = 1000337003
VK_STRUCTURE_TYPE_BLIT_IMAGE_INFO_2 = 1000337004
VK_STRUCTURE_TYPE_RESOLVE_IMAGE_INFO_2 = 1000337005
VK_STRUCTURE_TYPE_BUFFER_COPY_2 = 1000337006
VK_STRUCTURE_TYPE_IMAGE_COPY_2 = 1000337007
VK_STRUCTURE_TYPE_IMAGE_BLIT_2 = 1000337008
VK_STRUCTURE_TYPE_BUFFER_IMAGE_COPY_2 = 1000337009
VK_STRUCTURE_TYPE_IMAGE_RESOLVE_2 = 1000337010
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_SIZE_CONTROL_PROPERTIES = 1000225000
VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_REQUIRED_SUBGROUP_SIZE_CREATE_INFO = 1000225001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_SIZE_CONTROL_FEATURES = 1000225002
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_FEATURES = 1000138000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_PROPERTIES = 1000138001
VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_INLINE_UNIFORM_BLOCK = 1000138002
VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_INLINE_UNIFORM_BLOCK_CREATE_INFO = 1000138003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXTURE_COMPRESSION_ASTC_HDR_FEATURES = 1000066000
VK_STRUCTURE_TYPE_RENDERING_INFO = 1000044000
VK_STRUCTURE_TYPE_RENDERING_ATTACHMENT_INFO = 1000044001
VK_STRUCTURE_TYPE_PIPELINE_RENDERING_CREATE_INFO = 1000044002
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DYNAMIC_RENDERING_FEATURES = 1000044003
VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_RENDERING_INFO = 1000044004
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_INTEGER_DOT_PRODUCT_FEATURES = 1000280000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_INTEGER_DOT_PRODUCT_PROPERTIES = 1000280001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXEL_BUFFER_ALIGNMENT_PROPERTIES = 1000281001
VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_3 = 1000360000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_4_FEATURES = 1000413000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_4_PROPERTIES = 1000413001
VK_STRUCTURE_TYPE_DEVICE_BUFFER_MEMORY_REQUIREMENTS = 1000413002
VK_STRUCTURE_TYPE_DEVICE_IMAGE_MEMORY_REQUIREMENTS = 1000413003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_SC_1_0_FEATURES = 1000298000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_SC_1_0_PROPERTIES = 1000298001
VK_STRUCTURE_TYPE_DEVICE_OBJECT_RESERVATION_CREATE_INFO = 1000298002
VK_STRUCTURE_TYPE_COMMAND_POOL_MEMORY_RESERVATION_CREATE_INFO = 1000298003
VK_STRUCTURE_TYPE_COMMAND_POOL_MEMORY_CONSUMPTION = 1000298004
VK_STRUCTURE_TYPE_PIPELINE_POOL_SIZE = 1000298005
VK_STRUCTURE_TYPE_FAULT_DATA = 1000298007
VK_STRUCTURE_TYPE_FAULT_CALLBACK_INFO = 1000298008
VK_STRUCTURE_TYPE_PIPELINE_OFFLINE_CREATE_INFO = 1000298010
VK_SYSTEM_ALLOCATION_SCOPE_COMMAND = 0
VK_SYSTEM_ALLOCATION_SCOPE_OBJECT = 1
VK_SYSTEM_ALLOCATION_SCOPE_CACHE = 2
VK_SYSTEM_ALLOCATION_SCOPE_DEVICE = 3
VK_SYSTEM_ALLOCATION_SCOPE_INSTANCE = 4
VK_SYSTEM_ALLOCATION_SCOPE__BEGIN_RANGE = 0
VK_SYSTEM_ALLOCATION_SCOPE__END_RANGE = 4
VK_SYSTEM_ALLOCATION_SCOPE__RANGE_SIZE = 5
VK_SYSTEM_ALLOCATION_SCOPE__MAX_ENUM = 2147483647
VK_INTERNAL_ALLOCATION_TYPE_EXECUTABLE = 0
VK_INTERNAL_ALLOCATION_TYPE__BEGIN_RANGE = 0
VK_INTERNAL_ALLOCATION_TYPE__END_RANGE = 0
VK_INTERNAL_ALLOCATION_TYPE__RANGE_SIZE = 1
VK_INTERNAL_ALLOCATION_TYPE__MAX_ENUM = 2147483647
VK_SAMPLER_ADDRESS_MODE_REPEAT = 0
VK_SAMPLER_ADDRESS_MODE_MIRRORED_REPEAT = 1
VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE = 2
VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_BORDER = 3
VK_SAMPLER_ADDRESS_MODE__BEGIN_RANGE = 0
VK_SAMPLER_ADDRESS_MODE__END_RANGE = 3
VK_SAMPLER_ADDRESS_MODE__RANGE_SIZE = 4
VK_SAMPLER_ADDRESS_MODE__MAX_ENUM = 2147483647
VK_SAMPLER_ADDRESS_MODE_MIRROR_CLAMP_TO_EDGE = 4
VK_FILTER_NEAREST = 0
VK_FILTER_LINEAR = 1
VK_FILTER__BEGIN_RANGE = 0
VK_FILTER__END_RANGE = 1
VK_FILTER__RANGE_SIZE = 2
VK_FILTER__MAX_ENUM = 2147483647
VK_SAMPLER_MIPMAP_MODE_NEAREST = 0
VK_SAMPLER_MIPMAP_MODE_LINEAR = 1
VK_SAMPLER_MIPMAP_MODE__BEGIN_RANGE = 0
VK_SAMPLER_MIPMAP_MODE__END_RANGE = 1
VK_SAMPLER_MIPMAP_MODE__RANGE_SIZE = 2
VK_SAMPLER_MIPMAP_MODE__MAX_ENUM = 2147483647
VK_VERTEX_INPUT_RATE_VERTEX = 0
VK_VERTEX_INPUT_RATE_INSTANCE = 1
VK_VERTEX_INPUT_RATE__BEGIN_RANGE = 0
VK_VERTEX_INPUT_RATE__END_RANGE = 1
VK_VERTEX_INPUT_RATE__RANGE_SIZE = 2
VK_VERTEX_INPUT_RATE__MAX_ENUM = 2147483647
VK_PIPELINE_STAGE_TOP_OF_PIPE_BIT = 0x00000001
VK_PIPELINE_STAGE_DRAW_INDIRECT_BIT = 0x00000002
VK_PIPELINE_STAGE_VERTEX_INPUT_BIT = 0x00000004
VK_PIPELINE_STAGE_VERTEX_SHADER_BIT = 0x00000008
VK_PIPELINE_STAGE_TESSELLATION_CONTROL_SHADER_BIT = 0x00000010
VK_PIPELINE_STAGE_TESSELLATION_EVALUATION_SHADER_BIT = 0x00000020
VK_PIPELINE_STAGE_GEOMETRY_SHADER_BIT = 0x00000040
VK_PIPELINE_STAGE_FRAGMENT_SHADER_BIT = 0x00000080
VK_PIPELINE_STAGE_EARLY_FRAGMENT_TESTS_BIT = 0x00000100
VK_PIPELINE_STAGE_LATE_FRAGMENT_TESTS_BIT = 0x00000200
VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT = 0x00000400
VK_PIPELINE_STAGE_COMPUTE_SHADER_BIT = 0x00000800
VK_PIPELINE_STAGE_TRANSFER_BIT = 0x00001000
VK_PIPELINE_STAGE_BOTTOM_OF_PIPE_BIT = 0x00002000
VK_PIPELINE_STAGE_HOST_BIT = 0x00004000
VK_PIPELINE_STAGE_ALL_GRAPHICS_BIT = 0x00008000
VK_PIPELINE_STAGE_ALL_COMMANDS_BIT = 0x00010000
VK_PIPELINE_STAGE_FLAG_BITS__MAX_ENUM = 2147483647
VK_PIPELINE_STAGE_NONE = 0
VK_SPARSE_IMAGE_FORMAT_SINGLE_MIPTAIL_BIT = 0x00000001
VK_SPARSE_IMAGE_FORMAT_ALIGNED_MIP_SIZE_BIT = 0x00000002
VK_SPARSE_IMAGE_FORMAT_NONSTANDARD_BLOCK_SIZE_BIT = 0x00000004
VK_SPARSE_IMAGE_FORMAT_FLAG_BITS__MAX_ENUM = 2147483647
VK_SAMPLE_COUNT_1_BIT = 0x00000001
VK_SAMPLE_COUNT_2_BIT = 0x00000002
VK_SAMPLE_COUNT_4_BIT = 0x00000004
VK_SAMPLE_COUNT_8_BIT = 0x00000008
VK_SAMPLE_COUNT_16_BIT = 0x00000010
VK_SAMPLE_COUNT_32_BIT = 0x00000020
VK_SAMPLE_COUNT_64_BIT = 0x00000040
VK_SAMPLE_COUNT_FLAG_BITS__MAX_ENUM = 2147483647
VK_ATTACHMENT_DESCRIPTION_MAY_ALIAS_BIT = 0x00000001
VK_ATTACHMENT_DESCRIPTION_FLAG_BITS__MAX_ENUM = 2147483647
VK_DESCRIPTOR_POOL_CREATE_FREE_DESCRIPTOR_SET_BIT = 0x00000001
VK_DESCRIPTOR_POOL_CREATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_DEPENDENCY_BY_REGION_BIT = 0x00000001
VK_DEPENDENCY_FLAG_BITS__MAX_ENUM = 2147483647
VK_OBJECT_TYPE_UNKNOWN = 0
VK_OBJECT_TYPE_INSTANCE = 1
VK_OBJECT_TYPE_PHYSICAL_DEVICE = 2
VK_OBJECT_TYPE_DEVICE = 3
VK_OBJECT_TYPE_QUEUE = 4
VK_OBJECT_TYPE_SEMAPHORE = 5
VK_OBJECT_TYPE_COMMAND_BUFFER = 6
VK_OBJECT_TYPE_FENCE = 7
VK_OBJECT_TYPE_DEVICE_MEMORY = 8
VK_OBJECT_TYPE_BUFFER = 9
VK_OBJECT_TYPE_IMAGE = 10
VK_OBJECT_TYPE_EVENT = 11
VK_OBJECT_TYPE_QUERY_POOL = 12
VK_OBJECT_TYPE_BUFFER_VIEW = 13
VK_OBJECT_TYPE_IMAGE_VIEW = 14
VK_OBJECT_TYPE_SHADER_MODULE = 15
VK_OBJECT_TYPE_PIPELINE_CACHE = 16
VK_OBJECT_TYPE_PIPELINE_LAYOUT = 17
VK_OBJECT_TYPE_RENDER_PASS = 18
VK_OBJECT_TYPE_PIPELINE = 19
VK_OBJECT_TYPE_DESCRIPTOR_SET_LAYOUT = 20
VK_OBJECT_TYPE_SAMPLER = 21
VK_OBJECT_TYPE_DESCRIPTOR_POOL = 22
VK_OBJECT_TYPE_DESCRIPTOR_SET = 23
VK_OBJECT_TYPE_FRAMEBUFFER = 24
VK_OBJECT_TYPE_COMMAND_POOL = 25
VK_OBJECT_TYPE__BEGIN_RANGE = 0
VK_OBJECT_TYPE__END_RANGE = 25
VK_OBJECT_TYPE__RANGE_SIZE = 26
VK_OBJECT_TYPE__MAX_ENUM = 2147483647
VK_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION = 1000156000
VK_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE = 1000085000
VK_OBJECT_TYPE_PRIVATE_DATA_SLOT = 1000295000
VK_RAY_TRACING_INVOCATION_REORDER_MODE_NONE_NV = 0
VK_RAY_TRACING_INVOCATION_REORDER_MODE_REORDER_NV = 1
VK_RAY_TRACING_INVOCATION_REORDER_MODE_NV__BEGIN_RANGE = 0
VK_RAY_TRACING_INVOCATION_REORDER_MODE_NV__END_RANGE = 1
VK_RAY_TRACING_INVOCATION_REORDER_MODE_NV__RANGE_SIZE = 2
VK_RAY_TRACING_INVOCATION_REORDER_MODE_NV__MAX_ENUM = 2147483647
VK_INDIRECT_COMMANDS_LAYOUT_USAGE_EXPLICIT_PREPROCESS_BIT_NV = 0x00000001
VK_INDIRECT_COMMANDS_LAYOUT_USAGE_INDEXED_SEQUENCES_BIT_NV = 0x00000002
VK_INDIRECT_COMMANDS_LAYOUT_USAGE_UNORDERED_SEQUENCES_BIT_NV = 0x00000004
VK_INDIRECT_COMMANDS_LAYOUT_USAGE_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_INDIRECT_COMMANDS_TOKEN_TYPE_SHADER_GROUP_NV = 0
VK_INDIRECT_COMMANDS_TOKEN_TYPE_STATE_FLAGS_NV = 1
VK_INDIRECT_COMMANDS_TOKEN_TYPE_INDEX_BUFFER_NV = 2
VK_INDIRECT_COMMANDS_TOKEN_TYPE_VERTEX_BUFFER_NV = 3
VK_INDIRECT_COMMANDS_TOKEN_TYPE_PUSH_CONSTANT_NV = 4
VK_INDIRECT_COMMANDS_TOKEN_TYPE_DRAW_INDEXED_NV = 5
VK_INDIRECT_COMMANDS_TOKEN_TYPE_DRAW_NV = 6
VK_INDIRECT_COMMANDS_TOKEN_TYPE_DRAW_TASKS_NV = 7
VK_INDIRECT_COMMANDS_TOKEN_TYPE_NV__BEGIN_RANGE = 0
VK_INDIRECT_COMMANDS_TOKEN_TYPE_NV__END_RANGE = 7
VK_INDIRECT_COMMANDS_TOKEN_TYPE_NV__RANGE_SIZE = 8
VK_INDIRECT_COMMANDS_TOKEN_TYPE_NV__MAX_ENUM = 2147483647
VK_INDIRECT_STATE_FLAG_FRONTFACE_BIT_NV = 0x00000001
VK_INDIRECT_STATE_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE_DESCRIPTOR_SET = 0
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE__BEGIN_RANGE = 0
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE__END_RANGE = 0
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE__RANGE_SIZE = 1
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE__MAX_ENUM = 2147483647
VK_VIEWPORT_COORDINATE_SWIZZLE_POSITIVE_X_NV = 0
VK_VIEWPORT_COORDINATE_SWIZZLE_NEGATIVE_X_NV = 1
VK_VIEWPORT_COORDINATE_SWIZZLE_POSITIVE_Y_NV = 2
VK_VIEWPORT_COORDINATE_SWIZZLE_NEGATIVE_Y_NV = 3
VK_VIEWPORT_COORDINATE_SWIZZLE_POSITIVE_Z_NV = 4
VK_VIEWPORT_COORDINATE_SWIZZLE_NEGATIVE_Z_NV = 5
VK_VIEWPORT_COORDINATE_SWIZZLE_POSITIVE_W_NV = 6
VK_VIEWPORT_COORDINATE_SWIZZLE_NEGATIVE_W_NV = 7
VK_VIEWPORT_COORDINATE_SWIZZLE_NV__BEGIN_RANGE = 0
VK_VIEWPORT_COORDINATE_SWIZZLE_NV__END_RANGE = 7
VK_VIEWPORT_COORDINATE_SWIZZLE_NV__RANGE_SIZE = 8
VK_VIEWPORT_COORDINATE_SWIZZLE_NV__MAX_ENUM = 2147483647
VK_DISCARD_RECTANGLE_MODE_INCLUSIVE_EXT = 0
VK_DISCARD_RECTANGLE_MODE_EXCLUSIVE_EXT = 1
VK_DISCARD_RECTANGLE_MODE_EXT__BEGIN_RANGE = 0
VK_DISCARD_RECTANGLE_MODE_EXT__END_RANGE = 1
VK_DISCARD_RECTANGLE_MODE_EXT__RANGE_SIZE = 2
VK_DISCARD_RECTANGLE_MODE_EXT__MAX_ENUM = 2147483647
VK_POINT_CLIPPING_BEHAVIOR_ALL_CLIP_PLANES = 0
VK_POINT_CLIPPING_BEHAVIOR_USER_CLIP_PLANES_ONLY = 1
VK_POINT_CLIPPING_BEHAVIOR__BEGIN_RANGE = 0
VK_POINT_CLIPPING_BEHAVIOR__END_RANGE = 1
VK_POINT_CLIPPING_BEHAVIOR__RANGE_SIZE = 2
VK_POINT_CLIPPING_BEHAVIOR__MAX_ENUM = 2147483647
VK_COVERAGE_MODULATION_MODE_NONE_NV = 0
VK_COVERAGE_MODULATION_MODE_RGB_NV = 1
VK_COVERAGE_MODULATION_MODE_ALPHA_NV = 2
VK_COVERAGE_MODULATION_MODE_RGBA_NV = 3
VK_COVERAGE_MODULATION_MODE_NV__BEGIN_RANGE = 0
VK_COVERAGE_MODULATION_MODE_NV__END_RANGE = 3
VK_COVERAGE_MODULATION_MODE_NV__RANGE_SIZE = 4
VK_COVERAGE_MODULATION_MODE_NV__MAX_ENUM = 2147483647
VK_COVERAGE_REDUCTION_MODE_MERGE_NV = 0
VK_COVERAGE_REDUCTION_MODE_TRUNCATE_NV = 1
VK_COVERAGE_REDUCTION_MODE_NV__BEGIN_RANGE = 0
VK_COVERAGE_REDUCTION_MODE_NV__END_RANGE = 1
VK_COVERAGE_REDUCTION_MODE_NV__RANGE_SIZE = 2
VK_COVERAGE_REDUCTION_MODE_NV__MAX_ENUM = 2147483647
VK_VALIDATION_CACHE_HEADER_VERSION_ONE_EXT = 1
VK_VALIDATION_CACHE_HEADER_VERSION_EXT__BEGIN_RANGE = 1
VK_VALIDATION_CACHE_HEADER_VERSION_EXT__END_RANGE = 1
VK_VALIDATION_CACHE_HEADER_VERSION_EXT__RANGE_SIZE = 1
VK_VALIDATION_CACHE_HEADER_VERSION_EXT__MAX_ENUM = 2147483647
VK_SHADER_INFO_TYPE_STATISTICS_AMD = 0
VK_SHADER_INFO_TYPE_BINARY_AMD = 1
VK_SHADER_INFO_TYPE_DISASSEMBLY_AMD = 2
VK_SHADER_INFO_TYPE_AMD__BEGIN_RANGE = 0
VK_SHADER_INFO_TYPE_AMD__END_RANGE = 2
VK_SHADER_INFO_TYPE_AMD__RANGE_SIZE = 3
VK_SHADER_INFO_TYPE_AMD__MAX_ENUM = 2147483647
VK_QUEUE_GLOBAL_PRIORITY_LOW_KHR = 128
VK_QUEUE_GLOBAL_PRIORITY_MEDIUM_KHR = 256
VK_QUEUE_GLOBAL_PRIORITY_HIGH_KHR = 512
VK_QUEUE_GLOBAL_PRIORITY_REALTIME_KHR = 1024
VK_QUEUE_GLOBAL_PRIORITY_LOW_EXT = 1024
VK_QUEUE_GLOBAL_PRIORITY_MEDIUM_EXT = 1024
VK_QUEUE_GLOBAL_PRIORITY_HIGH_EXT = 1024
VK_QUEUE_GLOBAL_PRIORITY_REALTIME_EXT = 1024
VK_QUEUE_GLOBAL_PRIORITY_KHR__BEGIN_RANGE = 128
VK_QUEUE_GLOBAL_PRIORITY_KHR__END_RANGE = 1024
VK_QUEUE_GLOBAL_PRIORITY_KHR__RANGE_SIZE = 897
VK_QUEUE_GLOBAL_PRIORITY_KHR__MAX_ENUM = 2147483647
VK_TIME_DOMAIN_DEVICE_KHR = 0
VK_TIME_DOMAIN_CLOCK_MONOTONIC_KHR = 1
VK_TIME_DOMAIN_CLOCK_MONOTONIC_RAW_KHR = 2
VK_TIME_DOMAIN_QUERY_PERFORMANCE_COUNTER_KHR = 3
VK_TIME_DOMAIN_KHR__BEGIN_RANGE = 0
VK_TIME_DOMAIN_KHR__END_RANGE = 3
VK_TIME_DOMAIN_KHR__RANGE_SIZE = 4
VK_TIME_DOMAIN_KHR__MAX_ENUM = 2147483647
VK_CONSERVATIVE_RASTERIZATION_MODE_DISABLED_EXT = 0
VK_CONSERVATIVE_RASTERIZATION_MODE_OVERESTIMATE_EXT = 1
VK_CONSERVATIVE_RASTERIZATION_MODE_UNDERESTIMATE_EXT = 2
VK_CONSERVATIVE_RASTERIZATION_MODE_EXT__BEGIN_RANGE = 0
VK_CONSERVATIVE_RASTERIZATION_MODE_EXT__END_RANGE = 2
VK_CONSERVATIVE_RASTERIZATION_MODE_EXT__RANGE_SIZE = 3
VK_CONSERVATIVE_RASTERIZATION_MODE_EXT__MAX_ENUM = 2147483647
VK_RESOLVE_MODE_NONE = 0
VK_RESOLVE_MODE_SAMPLE_ZERO_BIT = 0x00000001
VK_RESOLVE_MODE_AVERAGE_BIT = 0x00000002
VK_RESOLVE_MODE_MIN_BIT = 0x00000004
VK_RESOLVE_MODE_MAX_BIT = 0x00000008
VK_RESOLVE_MODE_FLAG_BITS__MAX_ENUM = 2147483647
VK_DESCRIPTOR_BINDING_UPDATE_AFTER_BIND_BIT = 0x00000001
VK_DESCRIPTOR_BINDING_UPDATE_UNUSED_WHILE_PENDING_BIT = 0x00000002
VK_DESCRIPTOR_BINDING_PARTIALLY_BOUND_BIT = 0x00000004
VK_DESCRIPTOR_BINDING_VARIABLE_DESCRIPTOR_COUNT_BIT = 0x00000008
VK_DESCRIPTOR_BINDING_FLAG_BITS__MAX_ENUM = 2147483647
VK_CONDITIONAL_RENDERING_INVERTED_BIT_EXT = 0x00000001
VK_CONDITIONAL_RENDERING_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_SEMAPHORE_TYPE_BINARY = 0
VK_SEMAPHORE_TYPE_TIMELINE = 1
VK_SEMAPHORE_TYPE__BEGIN_RANGE = 0
VK_SEMAPHORE_TYPE__END_RANGE = 1
VK_SEMAPHORE_TYPE__RANGE_SIZE = 2
VK_SEMAPHORE_TYPE__MAX_ENUM = 2147483647
VK_GEOMETRY_OPAQUE_BIT_KHR = 0x00000001
VK_GEOMETRY_NO_DUPLICATE_ANY_HIT_INVOCATION_BIT_KHR = 0x00000002
VK_GEOMETRY_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_GEOMETRY_INSTANCE_TRIANGLE_FACING_CULL_DISABLE_BIT_KHR = 0x00000001
VK_GEOMETRY_INSTANCE_TRIANGLE_FLIP_FACING_BIT_KHR = 0x00000002
VK_GEOMETRY_INSTANCE_FORCE_OPAQUE_BIT_KHR = 0x00000004
VK_GEOMETRY_INSTANCE_FORCE_NO_OPAQUE_BIT_KHR = 0x00000008
VK_GEOMETRY_INSTANCE_TRIANGLE_FRONT_COUNTERCLOCKWISE_BIT_KHR = 0x00000008
VK_GEOMETRY_INSTANCE_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_UPDATE_BIT_KHR = 0x00000001
VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_COMPACTION_BIT_KHR = 0x00000002
VK_BUILD_ACCELERATION_STRUCTURE_PREFER_FAST_TRACE_BIT_KHR = 0x00000004
VK_BUILD_ACCELERATION_STRUCTURE_PREFER_FAST_BUILD_BIT_KHR = 0x00000008
VK_BUILD_ACCELERATION_STRUCTURE_LOW_MEMORY_BIT_KHR = 0x00000010
VK_BUILD_ACCELERATION_STRUCTURE_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_ACCELERATION_STRUCTURE_CREATE_DEVICE_ADDRESS_CAPTURE_REPLAY_BIT_KHR = 0x00000001
VK_ACCELERATION_STRUCTURE_CREATE_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_BUILD_ACCELERATION_STRUCTURE_MODE_BUILD_KHR = 0
VK_BUILD_ACCELERATION_STRUCTURE_MODE_UPDATE_KHR = 1
VK_BUILD_ACCELERATION_STRUCTURE_MODE_KHR__BEGIN_RANGE = 0
VK_BUILD_ACCELERATION_STRUCTURE_MODE_KHR__END_RANGE = 1
VK_BUILD_ACCELERATION_STRUCTURE_MODE_KHR__RANGE_SIZE = 2
VK_BUILD_ACCELERATION_STRUCTURE_MODE_KHR__MAX_ENUM = 2147483647
VK_COPY_ACCELERATION_STRUCTURE_MODE_CLONE_KHR = 0
VK_COPY_ACCELERATION_STRUCTURE_MODE_COMPACT_KHR = 1
VK_COPY_ACCELERATION_STRUCTURE_MODE_SERIALIZE_KHR = 2
VK_COPY_ACCELERATION_STRUCTURE_MODE_DESERIALIZE_KHR = 3
VK_COPY_ACCELERATION_STRUCTURE_MODE_KHR__BEGIN_RANGE = 0
VK_COPY_ACCELERATION_STRUCTURE_MODE_KHR__END_RANGE = 3
VK_COPY_ACCELERATION_STRUCTURE_MODE_KHR__RANGE_SIZE = 4
VK_COPY_ACCELERATION_STRUCTURE_MODE_KHR__MAX_ENUM = 2147483647
VK_ACCELERATION_STRUCTURE_TYPE_TOP_LEVEL_KHR = 0
VK_ACCELERATION_STRUCTURE_TYPE_BOTTOM_LEVEL_KHR = 1
VK_ACCELERATION_STRUCTURE_TYPE_GENERIC_KHR = 2
VK_ACCELERATION_STRUCTURE_TYPE_KHR__BEGIN_RANGE = 0
VK_ACCELERATION_STRUCTURE_TYPE_KHR__END_RANGE = 2
VK_ACCELERATION_STRUCTURE_TYPE_KHR__RANGE_SIZE = 3
VK_ACCELERATION_STRUCTURE_TYPE_KHR__MAX_ENUM = 2147483647
VK_GEOMETRY_TYPE_TRIANGLES_KHR = 0
VK_GEOMETRY_TYPE_AABBS_KHR = 1
VK_GEOMETRY_TYPE_INSTANCES_KHR = 2
VK_GEOMETRY_TYPE_KHR__BEGIN_RANGE = 0
VK_GEOMETRY_TYPE_KHR__END_RANGE = 2
VK_GEOMETRY_TYPE_KHR__RANGE_SIZE = 3
VK_GEOMETRY_TYPE_KHR__MAX_ENUM = 2147483647
VK_RAY_TRACING_SHADER_GROUP_TYPE_GENERAL_KHR = 0
VK_RAY_TRACING_SHADER_GROUP_TYPE_TRIANGLES_HIT_GROUP_KHR = 1
VK_RAY_TRACING_SHADER_GROUP_TYPE_PROCEDURAL_HIT_GROUP_KHR = 2
VK_RAY_TRACING_SHADER_GROUP_TYPE_KHR__BEGIN_RANGE = 0
VK_RAY_TRACING_SHADER_GROUP_TYPE_KHR__END_RANGE = 2
VK_RAY_TRACING_SHADER_GROUP_TYPE_KHR__RANGE_SIZE = 3
VK_RAY_TRACING_SHADER_GROUP_TYPE_KHR__MAX_ENUM = 2147483647
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_OBJECT_NV = 0
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_BUILD_SCRATCH_NV = 1
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_UPDATE_SCRATCH_NV = 2
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_NV__BEGIN_RANGE = 0
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_NV__END_RANGE = 2
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_NV__RANGE_SIZE = 3
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_NV__MAX_ENUM = 2147483647
VK_ACCELERATION_STRUCTURE_BUILD_TYPE_HOST_KHR = 0
VK_ACCELERATION_STRUCTURE_BUILD_TYPE_DEVICE_KHR = 1
VK_ACCELERATION_STRUCTURE_BUILD_TYPE_HOST_OR_DEVICE_KHR = 2
VK_ACCELERATION_STRUCTURE_BUILD_TYPE_KHR__BEGIN_RANGE = 0
VK_ACCELERATION_STRUCTURE_BUILD_TYPE_KHR__END_RANGE = 2
VK_ACCELERATION_STRUCTURE_BUILD_TYPE_KHR__RANGE_SIZE = 3
VK_ACCELERATION_STRUCTURE_BUILD_TYPE_KHR__MAX_ENUM = 2147483647
VK_ACCELERATION_STRUCTURE_COMPATIBILITY_COMPATIBLE_KHR = 0
VK_ACCELERATION_STRUCTURE_COMPATIBILITY_INCOMPATIBLE_KHR = 1
VK_ACCELERATION_STRUCTURE_COMPATIBILITY_KHR__BEGIN_RANGE = 0
VK_ACCELERATION_STRUCTURE_COMPATIBILITY_KHR__END_RANGE = 1
VK_ACCELERATION_STRUCTURE_COMPATIBILITY_KHR__RANGE_SIZE = 2
VK_ACCELERATION_STRUCTURE_COMPATIBILITY_KHR__MAX_ENUM = 2147483647
VK_SHADER_GROUP_SHADER_GENERAL_KHR = 0
VK_SHADER_GROUP_SHADER_CLOSEST_HIT_KHR = 1
VK_SHADER_GROUP_SHADER_ANY_HIT_KHR = 2
VK_SHADER_GROUP_SHADER_INTERSECTION_KHR = 3
VK_SHADER_GROUP_SHADER_KHR__BEGIN_RANGE = 0
VK_SHADER_GROUP_SHADER_KHR__END_RANGE = 3
VK_SHADER_GROUP_SHADER_KHR__RANGE_SIZE = 4
VK_SHADER_GROUP_SHADER_KHR__MAX_ENUM = 2147483647
VK_MEMORY_OVERALLOCATION_BEHAVIOR_DEFAULT_AMD = 0
VK_MEMORY_OVERALLOCATION_BEHAVIOR_ALLOWED_AMD = 1
VK_MEMORY_OVERALLOCATION_BEHAVIOR_DISALLOWED_AMD = 2
VK_MEMORY_OVERALLOCATION_BEHAVIOR_AMD__BEGIN_RANGE = 0
VK_MEMORY_OVERALLOCATION_BEHAVIOR_AMD__END_RANGE = 2
VK_MEMORY_OVERALLOCATION_BEHAVIOR_AMD__RANGE_SIZE = 3
VK_MEMORY_OVERALLOCATION_BEHAVIOR_AMD__MAX_ENUM = 2147483647
VK_DEVICE_DIAGNOSTICS_CONFIG_ENABLE_SHADER_DEBUG_INFO_BIT_NV = 0x00000001
VK_DEVICE_DIAGNOSTICS_CONFIG_ENABLE_RESOURCE_TRACKING_BIT_NV = 0x00000002
VK_DEVICE_DIAGNOSTICS_CONFIG_ENABLE_AUTOMATIC_CHECKPOINTS_BIT_NV = 0x00000004
VK_DEVICE_DIAGNOSTICS_CONFIG_ENABLE_SHADER_ERROR_REPORTING_BIT_NV = 0x00000008
VK_DEVICE_DIAGNOSTICS_CONFIG_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_PIPELINE_CREATION_FEEDBACK_VALID_BIT = 0x00000001
VK_PIPELINE_CREATION_FEEDBACK_VALID_BIT_EXT = 0x00000001
VK_PIPELINE_CREATION_FEEDBACK_APPLICATION_PIPELINE_CACHE_HIT_BIT = 0x00000002
VK_PIPELINE_CREATION_FEEDBACK_APPLICATION_PIPELINE_CACHE_HIT_BIT_EXT = 0x00000002
VK_PIPELINE_CREATION_FEEDBACK_BASE_PIPELINE_ACCELERATION_BIT = 0x00000004
VK_PIPELINE_CREATION_FEEDBACK_BASE_PIPELINE_ACCELERATION_BIT_EXT = 0x00000004
VK_PIPELINE_CREATION_FEEDBACK_FLAG_BITS__MAX_ENUM = 2147483647
VK_PERFORMANCE_COUNTER_SCOPE_COMMAND_BUFFER_KHR = 0
VK_PERFORMANCE_COUNTER_SCOPE_RENDER_PASS_KHR = 1
VK_PERFORMANCE_COUNTER_SCOPE_COMMAND_KHR = 2
VK_QUERY_SCOPE_COMMAND_BUFFER_KHR = 2
VK_QUERY_SCOPE_RENDER_PASS_KHR = 2
VK_QUERY_SCOPE_COMMAND_KHR = 2
VK_PERFORMANCE_COUNTER_SCOPE_KHR__BEGIN_RANGE = 0
VK_PERFORMANCE_COUNTER_SCOPE_KHR__END_RANGE = 2
VK_PERFORMANCE_COUNTER_SCOPE_KHR__RANGE_SIZE = 3
VK_PERFORMANCE_COUNTER_SCOPE_KHR__MAX_ENUM = 2147483647
VK_PERFORMANCE_COUNTER_UNIT_GENERIC_KHR = 0
VK_PERFORMANCE_COUNTER_UNIT_PERCENTAGE_KHR = 1
VK_PERFORMANCE_COUNTER_UNIT_NANOSECONDS_KHR = 2
VK_PERFORMANCE_COUNTER_UNIT_BYTES_KHR = 3
VK_PERFORMANCE_COUNTER_UNIT_BYTES_PER_SECOND_KHR = 4
VK_PERFORMANCE_COUNTER_UNIT_KELVIN_KHR = 5
VK_PERFORMANCE_COUNTER_UNIT_WATTS_KHR = 6
VK_PERFORMANCE_COUNTER_UNIT_VOLTS_KHR = 7
VK_PERFORMANCE_COUNTER_UNIT_AMPS_KHR = 8
VK_PERFORMANCE_COUNTER_UNIT_HERTZ_KHR = 9
VK_PERFORMANCE_COUNTER_UNIT_CYCLES_KHR = 10
VK_PERFORMANCE_COUNTER_UNIT_KHR__BEGIN_RANGE = 0
VK_PERFORMANCE_COUNTER_UNIT_KHR__END_RANGE = 10
VK_PERFORMANCE_COUNTER_UNIT_KHR__RANGE_SIZE = 11
VK_PERFORMANCE_COUNTER_UNIT_KHR__MAX_ENUM = 2147483647
VK_PERFORMANCE_COUNTER_STORAGE_INT32_KHR = 0
VK_PERFORMANCE_COUNTER_STORAGE_INT64_KHR = 1
VK_PERFORMANCE_COUNTER_STORAGE_UINT32_KHR = 2
VK_PERFORMANCE_COUNTER_STORAGE_UINT64_KHR = 3
VK_PERFORMANCE_COUNTER_STORAGE_FLOAT32_KHR = 4
VK_PERFORMANCE_COUNTER_STORAGE_FLOAT64_KHR = 5
VK_PERFORMANCE_COUNTER_STORAGE_KHR__BEGIN_RANGE = 0
VK_PERFORMANCE_COUNTER_STORAGE_KHR__END_RANGE = 5
VK_PERFORMANCE_COUNTER_STORAGE_KHR__RANGE_SIZE = 6
VK_PERFORMANCE_COUNTER_STORAGE_KHR__MAX_ENUM = 2147483647
VK_PERFORMANCE_COUNTER_DESCRIPTION_PERFORMANCE_IMPACTING_BIT_KHR = 0x00000001
VK_PERFORMANCE_COUNTER_DESCRIPTION_PERFORMANCE_IMPACTING_KHR = 0x00000001
VK_PERFORMANCE_COUNTER_DESCRIPTION_CONCURRENTLY_IMPACTED_BIT_KHR = 0x00000002
VK_PERFORMANCE_COUNTER_DESCRIPTION_CONCURRENTLY_IMPACTED_KHR = 0x00000002
VK_PERFORMANCE_COUNTER_DESCRIPTION_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_SEMAPHORE_WAIT_ANY_BIT = 0x00000001
VK_SEMAPHORE_WAIT_FLAG_BITS__MAX_ENUM = 2147483647
VK_PERFORMANCE_CONFIGURATION_TYPE_COMMAND_QUEUE_METRICS_DISCOVERY_ACTIVATED_INTEL = 0
VK_PERFORMANCE_CONFIGURATION_TYPE_INTEL__BEGIN_RANGE = 0
VK_PERFORMANCE_CONFIGURATION_TYPE_INTEL__END_RANGE = 0
VK_PERFORMANCE_CONFIGURATION_TYPE_INTEL__RANGE_SIZE = 1
VK_PERFORMANCE_CONFIGURATION_TYPE_INTEL__MAX_ENUM = 2147483647
VK_QUERY_POOL_SAMPLING_MODE_MANUAL_INTEL = 0
VK_QUERY_POOL_SAMPLING_MODE_INTEL__BEGIN_RANGE = 0
VK_QUERY_POOL_SAMPLING_MODE_INTEL__END_RANGE = 0
VK_QUERY_POOL_SAMPLING_MODE_INTEL__RANGE_SIZE = 1
VK_QUERY_POOL_SAMPLING_MODE_INTEL__MAX_ENUM = 2147483647
VK_PERFORMANCE_OVERRIDE_TYPE_NULL_HARDWARE_INTEL = 0
VK_PERFORMANCE_OVERRIDE_TYPE_FLUSH_GPU_CACHES_INTEL = 1
VK_PERFORMANCE_OVERRIDE_TYPE_INTEL__BEGIN_RANGE = 0
VK_PERFORMANCE_OVERRIDE_TYPE_INTEL__END_RANGE = 1
VK_PERFORMANCE_OVERRIDE_TYPE_INTEL__RANGE_SIZE = 2
VK_PERFORMANCE_OVERRIDE_TYPE_INTEL__MAX_ENUM = 2147483647
VK_PERFORMANCE_PARAMETER_TYPE_HW_COUNTERS_SUPPORTED_INTEL = 0
VK_PERFORMANCE_PARAMETER_TYPE_STREAM_MARKER_VALID_BITS_INTEL = 1
VK_PERFORMANCE_PARAMETER_TYPE_INTEL__BEGIN_RANGE = 0
VK_PERFORMANCE_PARAMETER_TYPE_INTEL__END_RANGE = 1
VK_PERFORMANCE_PARAMETER_TYPE_INTEL__RANGE_SIZE = 2
VK_PERFORMANCE_PARAMETER_TYPE_INTEL__MAX_ENUM = 2147483647
VK_PERFORMANCE_VALUE_TYPE_UINT32_INTEL = 0
VK_PERFORMANCE_VALUE_TYPE_UINT64_INTEL = 1
VK_PERFORMANCE_VALUE_TYPE_FLOAT_INTEL = 2
VK_PERFORMANCE_VALUE_TYPE_BOOL_INTEL = 3
VK_PERFORMANCE_VALUE_TYPE_STRING_INTEL = 4
VK_PERFORMANCE_VALUE_TYPE_INTEL__BEGIN_RANGE = 0
VK_PERFORMANCE_VALUE_TYPE_INTEL__END_RANGE = 4
VK_PERFORMANCE_VALUE_TYPE_INTEL__RANGE_SIZE = 5
VK_PERFORMANCE_VALUE_TYPE_INTEL__MAX_ENUM = 2147483647
VK_LINE_RASTERIZATION_MODE_DEFAULT_EXT = 0
VK_LINE_RASTERIZATION_MODE_RECTANGULAR_EXT = 1
VK_LINE_RASTERIZATION_MODE_BRESENHAM_EXT = 2
VK_LINE_RASTERIZATION_MODE_RECTANGULAR_SMOOTH_EXT = 3
VK_LINE_RASTERIZATION_MODE_EXT__BEGIN_RANGE = 0
VK_LINE_RASTERIZATION_MODE_EXT__END_RANGE = 3
VK_LINE_RASTERIZATION_MODE_EXT__RANGE_SIZE = 4
VK_LINE_RASTERIZATION_MODE_EXT__MAX_ENUM = 2147483647
VK_FAULT_LEVEL_UNASSIGNED = 0
VK_FAULT_LEVEL_CRITICAL = 1
VK_FAULT_LEVEL_RECOVERABLE = 2
VK_FAULT_LEVEL_WARNING = 3
VK_FAULT_LEVEL__BEGIN_RANGE = 0
VK_FAULT_LEVEL__END_RANGE = 3
VK_FAULT_LEVEL__RANGE_SIZE = 4
VK_FAULT_LEVEL__MAX_ENUM = 2147483647
VK_FAULT_TYPE_INVALID = 0
VK_FAULT_TYPE_UNASSIGNED = 1
VK_FAULT_TYPE_IMPLEMENTATION = 2
VK_FAULT_TYPE_SYSTEM = 3
VK_FAULT_TYPE_PHYSICAL_DEVICE = 4
VK_FAULT_TYPE_COMMAND_BUFFER_FULL = 5
VK_FAULT_TYPE_INVALID_API_USAGE = 6
VK_FAULT_TYPE__BEGIN_RANGE = 0
VK_FAULT_TYPE__END_RANGE = 6
VK_FAULT_TYPE__RANGE_SIZE = 7
VK_FAULT_TYPE__MAX_ENUM = 2147483647
VK_FAULT_QUERY_BEHAVIOR_GET_AND_CLEAR_ALL_FAULTS = 0
VK_FAULT_QUERY_BEHAVIOR__BEGIN_RANGE = 0
VK_FAULT_QUERY_BEHAVIOR__END_RANGE = 0
VK_FAULT_QUERY_BEHAVIOR__RANGE_SIZE = 1
VK_FAULT_QUERY_BEHAVIOR__MAX_ENUM = 2147483647
VK_PIPELINE_MATCH_CONTROL_APPLICATION_UUID_EXACT_MATCH = 0
VK_PIPELINE_MATCH_CONTROL__BEGIN_RANGE = 0
VK_PIPELINE_MATCH_CONTROL__END_RANGE = 0
VK_PIPELINE_MATCH_CONTROL__RANGE_SIZE = 1
VK_PIPELINE_MATCH_CONTROL__MAX_ENUM = 2147483647
VK_SCI_SYNC_CLIENT_TYPE_SIGNALER_NV = 0
VK_SCI_SYNC_CLIENT_TYPE_WAITER_NV = 1
VK_SCI_SYNC_CLIENT_TYPE_SIGNALER_WAITER_NV = 2
VK_SCI_SYNC_CLIENT_TYPE_NV__BEGIN_RANGE = 0
VK_SCI_SYNC_CLIENT_TYPE_NV__END_RANGE = 2
VK_SCI_SYNC_CLIENT_TYPE_NV__RANGE_SIZE = 3
VK_SCI_SYNC_CLIENT_TYPE_NV__MAX_ENUM = 2147483647
VK_SCI_SYNC_PRIMITIVE_TYPE_FENCE_NV = 0
VK_SCI_SYNC_PRIMITIVE_TYPE_SEMAPHORE_NV = 1
VK_SCI_SYNC_PRIMITIVE_TYPE_NV__BEGIN_RANGE = 0
VK_SCI_SYNC_PRIMITIVE_TYPE_NV__END_RANGE = 1
VK_SCI_SYNC_PRIMITIVE_TYPE_NV__RANGE_SIZE = 2
VK_SCI_SYNC_PRIMITIVE_TYPE_NV__MAX_ENUM = 2147483647
VK_TOOL_PURPOSE_VALIDATION_BIT = 0x00000001
VK_TOOL_PURPOSE_VALIDATION_BIT_EXT = 0x00000001
VK_TOOL_PURPOSE_PROFILING_BIT = 0x00000002
VK_TOOL_PURPOSE_PROFILING_BIT_EXT = 0x00000002
VK_TOOL_PURPOSE_TRACING_BIT = 0x00000004
VK_TOOL_PURPOSE_TRACING_BIT_EXT = 0x00000004
VK_TOOL_PURPOSE_ADDITIONAL_FEATURES_BIT = 0x00000008
VK_TOOL_PURPOSE_ADDITIONAL_FEATURES_BIT_EXT = 0x00000008
VK_TOOL_PURPOSE_MODIFYING_FEATURES_BIT = 0x00000010
VK_TOOL_PURPOSE_MODIFYING_FEATURES_BIT_EXT = 0x00000010
VK_TOOL_PURPOSE_FLAG_BITS__MAX_ENUM = 2147483647
VK_FRAGMENT_SHADING_RATE_1_INVOCATION_PER_PIXEL_NV = 0
VK_FRAGMENT_SHADING_RATE_1_INVOCATION_PER_1X2_PIXELS_NV = 1
VK_FRAGMENT_SHADING_RATE_1_INVOCATION_PER_2X1_PIXELS_NV = 4
VK_FRAGMENT_SHADING_RATE_1_INVOCATION_PER_2X2_PIXELS_NV = 5
VK_FRAGMENT_SHADING_RATE_1_INVOCATION_PER_2X4_PIXELS_NV = 6
VK_FRAGMENT_SHADING_RATE_1_INVOCATION_PER_4X2_PIXELS_NV = 9
VK_FRAGMENT_SHADING_RATE_1_INVOCATION_PER_4X4_PIXELS_NV = 10
VK_FRAGMENT_SHADING_RATE_2_INVOCATIONS_PER_PIXEL_NV = 11
VK_FRAGMENT_SHADING_RATE_4_INVOCATIONS_PER_PIXEL_NV = 12
VK_FRAGMENT_SHADING_RATE_8_INVOCATIONS_PER_PIXEL_NV = 13
VK_FRAGMENT_SHADING_RATE_16_INVOCATIONS_PER_PIXEL_NV = 14
VK_FRAGMENT_SHADING_RATE_NO_INVOCATIONS_NV = 15
VK_FRAGMENT_SHADING_RATE_NV__BEGIN_RANGE = 0
VK_FRAGMENT_SHADING_RATE_NV__END_RANGE = 15
VK_FRAGMENT_SHADING_RATE_NV__RANGE_SIZE = 16
VK_FRAGMENT_SHADING_RATE_NV__MAX_ENUM = 2147483647
VK_FRAGMENT_SHADING_RATE_TYPE_FRAGMENT_SIZE_NV = 0
VK_FRAGMENT_SHADING_RATE_TYPE_ENUMS_NV = 1
VK_FRAGMENT_SHADING_RATE_TYPE_NV__BEGIN_RANGE = 0
VK_FRAGMENT_SHADING_RATE_TYPE_NV__END_RANGE = 1
VK_FRAGMENT_SHADING_RATE_TYPE_NV__RANGE_SIZE = 2
VK_FRAGMENT_SHADING_RATE_TYPE_NV__MAX_ENUM = 2147483647
VK_SUBPASS_MERGE_STATUS_MERGED_EXT = 0
VK_SUBPASS_MERGE_STATUS_DISALLOWED_EXT = 1
VK_SUBPASS_MERGE_STATUS_NOT_MERGED_SIDE_EFFECTS_EXT = 2
VK_SUBPASS_MERGE_STATUS_NOT_MERGED_SAMPLES_MISMATCH_EXT = 3
VK_SUBPASS_MERGE_STATUS_NOT_MERGED_VIEWS_MISMATCH_EXT = 4
VK_SUBPASS_MERGE_STATUS_NOT_MERGED_ALIASING_EXT = 5
VK_SUBPASS_MERGE_STATUS_NOT_MERGED_DEPENDENCIES_EXT = 6
VK_SUBPASS_MERGE_STATUS_NOT_MERGED_INCOMPATIBLE_INPUT_ATTACHMENT_EXT = 7
VK_SUBPASS_MERGE_STATUS_NOT_MERGED_TOO_MANY_ATTACHMENTS_EXT = 8
VK_SUBPASS_MERGE_STATUS_NOT_MERGED_INSUFFICIENT_STORAGE_EXT = 9
VK_SUBPASS_MERGE_STATUS_NOT_MERGED_DEPTH_STENCIL_COUNT_EXT = 10
VK_SUBPASS_MERGE_STATUS_NOT_MERGED_RESOLVE_ATTACHMENT_REUSE_EXT = 11
VK_SUBPASS_MERGE_STATUS_NOT_MERGED_SINGLE_SUBPASS_EXT = 12
VK_SUBPASS_MERGE_STATUS_NOT_MERGED_UNSPECIFIED_EXT = 13
VK_SUBPASS_MERGE_STATUS_EXT__BEGIN_RANGE = 0
VK_SUBPASS_MERGE_STATUS_EXT__END_RANGE = 13
VK_SUBPASS_MERGE_STATUS_EXT__RANGE_SIZE = 14
VK_SUBPASS_MERGE_STATUS_EXT__MAX_ENUM = 2147483647
VK_ACCESS_2_NONE = 0
VK_ACCESS_2_NONE_KHR = 0
VK_ACCESS_2_INDIRECT_COMMAND_READ_BIT = 0x00000001
VK_ACCESS_2_INDIRECT_COMMAND_READ_BIT_KHR = 0x00000001
VK_ACCESS_2_INDEX_READ_BIT = 0x00000002
VK_ACCESS_2_INDEX_READ_BIT_KHR = 0x00000002
VK_ACCESS_2_VERTEX_ATTRIBUTE_READ_BIT = 0x00000004
VK_ACCESS_2_VERTEX_ATTRIBUTE_READ_BIT_KHR = 0x00000004
VK_ACCESS_2_UNIFORM_READ_BIT = 0x00000008
VK_ACCESS_2_UNIFORM_READ_BIT_KHR = 0x00000008
VK_ACCESS_2_INPUT_ATTACHMENT_READ_BIT = 0x00000010
VK_ACCESS_2_INPUT_ATTACHMENT_READ_BIT_KHR = 0x00000010
VK_ACCESS_2_SHADER_READ_BIT = 0x00000020
VK_ACCESS_2_SHADER_READ_BIT_KHR = 0x00000020
VK_ACCESS_2_SHADER_WRITE_BIT = 0x00000040
VK_ACCESS_2_SHADER_WRITE_BIT_KHR = 0x00000040
VK_ACCESS_2_COLOR_ATTACHMENT_READ_BIT = 0x00000080
VK_ACCESS_2_COLOR_ATTACHMENT_READ_BIT_KHR = 0x00000080
VK_ACCESS_2_COLOR_ATTACHMENT_WRITE_BIT = 0x00000100
VK_ACCESS_2_COLOR_ATTACHMENT_WRITE_BIT_KHR = 0x00000100
VK_ACCESS_2_DEPTH_STENCIL_ATTACHMENT_READ_BIT = 0x00000200
VK_ACCESS_2_DEPTH_STENCIL_ATTACHMENT_READ_BIT_KHR = 0x00000200
VK_ACCESS_2_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT = 0x00000400
VK_ACCESS_2_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT_KHR = 0x00000400
VK_ACCESS_2_TRANSFER_READ_BIT = 0x00000800
VK_ACCESS_2_TRANSFER_READ_BIT_KHR = 0x00000800
VK_ACCESS_2_TRANSFER_WRITE_BIT = 0x00001000
VK_ACCESS_2_TRANSFER_WRITE_BIT_KHR = 0x00001000
VK_ACCESS_2_HOST_READ_BIT = 0x00002000
VK_ACCESS_2_HOST_READ_BIT_KHR = 0x00002000
VK_ACCESS_2_HOST_WRITE_BIT = 0x00004000
VK_ACCESS_2_HOST_WRITE_BIT_KHR = 0x00004000
VK_ACCESS_2_MEMORY_READ_BIT = 0x00008000
VK_ACCESS_2_MEMORY_READ_BIT_KHR = 0x00008000
VK_ACCESS_2_MEMORY_WRITE_BIT = 0x00010000
VK_ACCESS_2_MEMORY_WRITE_BIT_KHR = 0x00010000
VK_ACCESS_2_SHADER_SAMPLED_READ_BIT = 0x100000000
VK_ACCESS_2_SHADER_SAMPLED_READ_BIT_KHR = 0x100000000
VK_ACCESS_2_SHADER_STORAGE_READ_BIT = 0x200000000
VK_ACCESS_2_SHADER_STORAGE_READ_BIT_KHR = 0x200000000
VK_ACCESS_2_SHADER_STORAGE_WRITE_BIT = 0x400000000
VK_ACCESS_2_SHADER_STORAGE_WRITE_BIT_KHR = 0x400000000
VK_ACCESS_FLAG_BITS2__MAX_ENUM = 2147483647
VK_PIPELINE_STAGE_2_NONE = 0
VK_PIPELINE_STAGE_2_NONE_KHR = 0
VK_PIPELINE_STAGE_2_TOP_OF_PIPE_BIT = 0x00000001
VK_PIPELINE_STAGE_2_TOP_OF_PIPE_BIT_KHR = 0x00000001
VK_PIPELINE_STAGE_2_DRAW_INDIRECT_BIT = 0x00000002
VK_PIPELINE_STAGE_2_DRAW_INDIRECT_BIT_KHR = 0x00000002
VK_PIPELINE_STAGE_2_VERTEX_INPUT_BIT = 0x00000004
VK_PIPELINE_STAGE_2_VERTEX_INPUT_BIT_KHR = 0x00000004
VK_PIPELINE_STAGE_2_VERTEX_SHADER_BIT = 0x00000008
VK_PIPELINE_STAGE_2_VERTEX_SHADER_BIT_KHR = 0x00000008
VK_PIPELINE_STAGE_2_TESSELLATION_CONTROL_SHADER_BIT = 0x00000010
VK_PIPELINE_STAGE_2_TESSELLATION_CONTROL_SHADER_BIT_KHR = 0x00000010
VK_PIPELINE_STAGE_2_TESSELLATION_EVALUATION_SHADER_BIT = 0x00000020
VK_PIPELINE_STAGE_2_TESSELLATION_EVALUATION_SHADER_BIT_KHR = 0x00000020
VK_PIPELINE_STAGE_2_GEOMETRY_SHADER_BIT = 0x00000040
VK_PIPELINE_STAGE_2_GEOMETRY_SHADER_BIT_KHR = 0x00000040
VK_PIPELINE_STAGE_2_FRAGMENT_SHADER_BIT = 0x00000080
VK_PIPELINE_STAGE_2_FRAGMENT_SHADER_BIT_KHR = 0x00000080
VK_PIPELINE_STAGE_2_EARLY_FRAGMENT_TESTS_BIT = 0x00000100
VK_PIPELINE_STAGE_2_EARLY_FRAGMENT_TESTS_BIT_KHR = 0x00000100
VK_PIPELINE_STAGE_2_LATE_FRAGMENT_TESTS_BIT = 0x00000200
VK_PIPELINE_STAGE_2_LATE_FRAGMENT_TESTS_BIT_KHR = 0x00000200
VK_PIPELINE_STAGE_2_COLOR_ATTACHMENT_OUTPUT_BIT = 0x00000400
VK_PIPELINE_STAGE_2_COLOR_ATTACHMENT_OUTPUT_BIT_KHR = 0x00000400
VK_PIPELINE_STAGE_2_COMPUTE_SHADER_BIT = 0x00000800
VK_PIPELINE_STAGE_2_COMPUTE_SHADER_BIT_KHR = 0x00000800
VK_PIPELINE_STAGE_2_ALL_TRANSFER_BIT = 0x00001000
VK_PIPELINE_STAGE_2_ALL_TRANSFER_BIT_KHR = 0x00001000
VK_PIPELINE_STAGE_2_TRANSFER_BIT = 0x00001000
VK_PIPELINE_STAGE_2_TRANSFER_BIT_KHR = 0x00001000
VK_PIPELINE_STAGE_2_BOTTOM_OF_PIPE_BIT = 0x00002000
VK_PIPELINE_STAGE_2_BOTTOM_OF_PIPE_BIT_KHR = 0x00002000
VK_PIPELINE_STAGE_2_HOST_BIT = 0x00004000
VK_PIPELINE_STAGE_2_HOST_BIT_KHR = 0x00004000
VK_PIPELINE_STAGE_2_ALL_GRAPHICS_BIT = 0x00008000
VK_PIPELINE_STAGE_2_ALL_GRAPHICS_BIT_KHR = 0x00008000
VK_PIPELINE_STAGE_2_ALL_COMMANDS_BIT = 0x00010000
VK_PIPELINE_STAGE_2_ALL_COMMANDS_BIT_KHR = 0x00010000
VK_PIPELINE_STAGE_2_COPY_BIT = 0x100000000
VK_PIPELINE_STAGE_2_COPY_BIT_KHR = 0x100000000
VK_PIPELINE_STAGE_2_RESOLVE_BIT = 0x200000000
VK_PIPELINE_STAGE_2_RESOLVE_BIT_KHR = 0x200000000
VK_PIPELINE_STAGE_2_BLIT_BIT = 0x400000000
VK_PIPELINE_STAGE_2_BLIT_BIT_KHR = 0x400000000
VK_PIPELINE_STAGE_2_CLEAR_BIT = 0x800000000
VK_PIPELINE_STAGE_2_CLEAR_BIT_KHR = 0x800000000
VK_PIPELINE_STAGE_2_INDEX_INPUT_BIT = 0x1000000000
VK_PIPELINE_STAGE_2_INDEX_INPUT_BIT_KHR = 0x1000000000
VK_PIPELINE_STAGE_2_VERTEX_ATTRIBUTE_INPUT_BIT = 0x2000000000
VK_PIPELINE_STAGE_2_VERTEX_ATTRIBUTE_INPUT_BIT_KHR = 0x2000000000
VK_PIPELINE_STAGE_2_PRE_RASTERIZATION_SHADERS_BIT = 0x4000000000
VK_PIPELINE_STAGE_2_PRE_RASTERIZATION_SHADERS_BIT_KHR = 0x4000000000
VK_PIPELINE_STAGE_FLAG_BITS2__MAX_ENUM = 2147483647
VK_PROVOKING_VERTEX_MODE_FIRST_VERTEX_EXT = 0
VK_PROVOKING_VERTEX_MODE_LAST_VERTEX_EXT = 1
VK_PROVOKING_VERTEX_MODE_EXT__BEGIN_RANGE = 0
VK_PROVOKING_VERTEX_MODE_EXT__END_RANGE = 1
VK_PROVOKING_VERTEX_MODE_EXT__RANGE_SIZE = 2
VK_PROVOKING_VERTEX_MODE_EXT__MAX_ENUM = 2147483647
VK_PIPELINE_CACHE_VALIDATION_VERSION_SAFETY_CRITICAL_ONE = 1
VK_PIPELINE_CACHE_VALIDATION_VERSION__BEGIN_RANGE = 1
VK_PIPELINE_CACHE_VALIDATION_VERSION__END_RANGE = 1
VK_PIPELINE_CACHE_VALIDATION_VERSION__RANGE_SIZE = 1
VK_PIPELINE_CACHE_VALIDATION_VERSION__MAX_ENUM = 2147483647
VK_HOST_IMAGE_COPY_MEMCPY_EXT = 0x00000001
VK_HOST_IMAGE_COPY_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_IMAGE_CONSTRAINTS_INFO_CPU_READ_RARELY_FUCHSIA = 0x00000001
VK_IMAGE_CONSTRAINTS_INFO_CPU_READ_OFTEN_FUCHSIA = 0x00000002
VK_IMAGE_CONSTRAINTS_INFO_CPU_WRITE_RARELY_FUCHSIA = 0x00000004
VK_IMAGE_CONSTRAINTS_INFO_CPU_WRITE_OFTEN_FUCHSIA = 0x00000008
VK_IMAGE_CONSTRAINTS_INFO_PROTECTED_OPTIONAL_FUCHSIA = 0x00000010
VK_IMAGE_CONSTRAINTS_INFO_FLAG_BITS_FUCHSIA__MAX_ENUM = 2147483647
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_BIT = 0x00000001
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_BIT_KHR = 0x00000001
VK_FORMAT_FEATURE_2_STORAGE_IMAGE_BIT = 0x00000002
VK_FORMAT_FEATURE_2_STORAGE_IMAGE_BIT_KHR = 0x00000002
VK_FORMAT_FEATURE_2_STORAGE_IMAGE_ATOMIC_BIT = 0x00000004
VK_FORMAT_FEATURE_2_STORAGE_IMAGE_ATOMIC_BIT_KHR = 0x00000004
VK_FORMAT_FEATURE_2_UNIFORM_TEXEL_BUFFER_BIT = 0x00000008
VK_FORMAT_FEATURE_2_UNIFORM_TEXEL_BUFFER_BIT_KHR = 0x00000008
VK_FORMAT_FEATURE_2_STORAGE_TEXEL_BUFFER_BIT = 0x00000010
VK_FORMAT_FEATURE_2_STORAGE_TEXEL_BUFFER_BIT_KHR = 0x00000010
VK_FORMAT_FEATURE_2_STORAGE_TEXEL_BUFFER_ATOMIC_BIT = 0x00000020
VK_FORMAT_FEATURE_2_STORAGE_TEXEL_BUFFER_ATOMIC_BIT_KHR = 0x00000020
VK_FORMAT_FEATURE_2_VERTEX_BUFFER_BIT = 0x00000040
VK_FORMAT_FEATURE_2_VERTEX_BUFFER_BIT_KHR = 0x00000040
VK_FORMAT_FEATURE_2_COLOR_ATTACHMENT_BIT = 0x00000080
VK_FORMAT_FEATURE_2_COLOR_ATTACHMENT_BIT_KHR = 0x00000080
VK_FORMAT_FEATURE_2_COLOR_ATTACHMENT_BLEND_BIT = 0x00000100
VK_FORMAT_FEATURE_2_COLOR_ATTACHMENT_BLEND_BIT_KHR = 0x00000100
VK_FORMAT_FEATURE_2_DEPTH_STENCIL_ATTACHMENT_BIT = 0x00000200
VK_FORMAT_FEATURE_2_DEPTH_STENCIL_ATTACHMENT_BIT_KHR = 0x00000200
VK_FORMAT_FEATURE_2_BLIT_SRC_BIT = 0x00000400
VK_FORMAT_FEATURE_2_BLIT_SRC_BIT_KHR = 0x00000400
VK_FORMAT_FEATURE_2_BLIT_DST_BIT = 0x00000800
VK_FORMAT_FEATURE_2_BLIT_DST_BIT_KHR = 0x00000800
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_FILTER_LINEAR_BIT = 0x00001000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_FILTER_LINEAR_BIT_KHR = 0x00001000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_FILTER_CUBIC_BIT = 0x00002000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_FILTER_CUBIC_BIT_EXT = 0x00002000
VK_FORMAT_FEATURE_2_TRANSFER_SRC_BIT = 0x00004000
VK_FORMAT_FEATURE_2_TRANSFER_SRC_BIT_KHR = 0x00004000
VK_FORMAT_FEATURE_2_TRANSFER_DST_BIT = 0x00008000
VK_FORMAT_FEATURE_2_TRANSFER_DST_BIT_KHR = 0x00008000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_FILTER_MINMAX_BIT = 0x00010000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_FILTER_MINMAX_BIT_KHR = 0x00010000
VK_FORMAT_FEATURE_2_MIDPOINT_CHROMA_SAMPLES_BIT = 0x00020000
VK_FORMAT_FEATURE_2_MIDPOINT_CHROMA_SAMPLES_BIT_KHR = 0x00020000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_YCBCR_CONVERSION_LINEAR_FILTER_BIT = 0x00040000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_YCBCR_CONVERSION_LINEAR_FILTER_BIT_KHR = 0x00040000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_YCBCR_CONVERSION_SEPARATE_RECONSTRUCTION_FILTER_BIT = 0x00080000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_YCBCR_CONVERSION_SEPARATE_RECONSTRUCTION_FILTER_BIT_KHR = 0x00080000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_BIT = 0x00100000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_BIT_KHR = 0x00100000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_FORCEABLE_BIT = 0x00200000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_FORCEABLE_BIT_KHR = 0x00200000
VK_FORMAT_FEATURE_2_DISJOINT_BIT = 0x00400000
VK_FORMAT_FEATURE_2_DISJOINT_BIT_KHR = 0x00400000
VK_FORMAT_FEATURE_2_COSITED_CHROMA_SAMPLES_BIT = 0x00800000
VK_FORMAT_FEATURE_2_COSITED_CHROMA_SAMPLES_BIT_KHR = 0x00800000
VK_FORMAT_FEATURE_2_STORAGE_READ_WITHOUT_FORMAT_BIT = 0x80000000
VK_FORMAT_FEATURE_2_STORAGE_READ_WITHOUT_FORMAT_BIT_KHR = 0x80000000
VK_FORMAT_FEATURE_2_STORAGE_WRITE_WITHOUT_FORMAT_BIT = 0x100000000
VK_FORMAT_FEATURE_2_STORAGE_WRITE_WITHOUT_FORMAT_BIT_KHR = 0x100000000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_DEPTH_COMPARISON_BIT = 0x200000000
VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_DEPTH_COMPARISON_BIT_KHR = 0x200000000
VK_FORMAT_FEATURE_FLAG_BITS2__MAX_ENUM = 2147483647
VK_RENDERING_CONTENTS_SECONDARY_COMMAND_BUFFERS_BIT = 0x00000001
VK_RENDERING_CONTENTS_SECONDARY_COMMAND_BUFFERS_BIT_KHR = 0x00000001
VK_RENDERING_SUSPENDING_BIT = 0x00000002
VK_RENDERING_SUSPENDING_BIT_KHR = 0x00000002
VK_RENDERING_RESUMING_BIT = 0x00000004
VK_RENDERING_RESUMING_BIT_KHR = 0x00000004
VK_RENDERING_FLAG_BITS__MAX_ENUM = 2147483647
VK_IMAGE_COMPRESSION_DEFAULT_EXT = 0
VK_IMAGE_COMPRESSION_FIXED_RATE_DEFAULT_EXT = 0x00000001
VK_IMAGE_COMPRESSION_FIXED_RATE_EXPLICIT_EXT = 0x00000002
VK_IMAGE_COMPRESSION_DISABLED_EXT = 0x00000004
VK_IMAGE_COMPRESSION_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_IMAGE_COMPRESSION_FIXED_RATE_NONE_EXT = 0
VK_IMAGE_COMPRESSION_FIXED_RATE_1BPC_BIT_EXT = 0x00000001
VK_IMAGE_COMPRESSION_FIXED_RATE_2BPC_BIT_EXT = 0x00000002
VK_IMAGE_COMPRESSION_FIXED_RATE_3BPC_BIT_EXT = 0x00000004
VK_IMAGE_COMPRESSION_FIXED_RATE_4BPC_BIT_EXT = 0x00000008
VK_IMAGE_COMPRESSION_FIXED_RATE_5BPC_BIT_EXT = 0x00000010
VK_IMAGE_COMPRESSION_FIXED_RATE_6BPC_BIT_EXT = 0x00000020
VK_IMAGE_COMPRESSION_FIXED_RATE_7BPC_BIT_EXT = 0x00000040
VK_IMAGE_COMPRESSION_FIXED_RATE_8BPC_BIT_EXT = 0x00000080
VK_IMAGE_COMPRESSION_FIXED_RATE_9BPC_BIT_EXT = 0x00000100
VK_IMAGE_COMPRESSION_FIXED_RATE_10BPC_BIT_EXT = 0x00000200
VK_IMAGE_COMPRESSION_FIXED_RATE_11BPC_BIT_EXT = 0x00000400
VK_IMAGE_COMPRESSION_FIXED_RATE_12BPC_BIT_EXT = 0x00000800
VK_IMAGE_COMPRESSION_FIXED_RATE_13BPC_BIT_EXT = 0x00001000
VK_IMAGE_COMPRESSION_FIXED_RATE_14BPC_BIT_EXT = 0x00002000
VK_IMAGE_COMPRESSION_FIXED_RATE_15BPC_BIT_EXT = 0x00004000
VK_IMAGE_COMPRESSION_FIXED_RATE_16BPC_BIT_EXT = 0x00008000
VK_IMAGE_COMPRESSION_FIXED_RATE_17BPC_BIT_EXT = 0x00010000
VK_IMAGE_COMPRESSION_FIXED_RATE_18BPC_BIT_EXT = 0x00020000
VK_IMAGE_COMPRESSION_FIXED_RATE_19BPC_BIT_EXT = 0x00040000
VK_IMAGE_COMPRESSION_FIXED_RATE_20BPC_BIT_EXT = 0x00080000
VK_IMAGE_COMPRESSION_FIXED_RATE_21BPC_BIT_EXT = 0x00100000
VK_IMAGE_COMPRESSION_FIXED_RATE_22BPC_BIT_EXT = 0x00200000
VK_IMAGE_COMPRESSION_FIXED_RATE_23BPC_BIT_EXT = 0x00400000
VK_IMAGE_COMPRESSION_FIXED_RATE_24BPC_BIT_EXT = 0x00800000
VK_IMAGE_COMPRESSION_FIXED_RATE_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_EXPORT_METAL_OBJECT_TYPE_METAL_DEVICE_BIT_EXT = 0x00000001
VK_EXPORT_METAL_OBJECT_TYPE_METAL_COMMAND_QUEUE_BIT_EXT = 0x00000002
VK_EXPORT_METAL_OBJECT_TYPE_METAL_BUFFER_BIT_EXT = 0x00000004
VK_EXPORT_METAL_OBJECT_TYPE_METAL_TEXTURE_BIT_EXT = 0x00000008
VK_EXPORT_METAL_OBJECT_TYPE_METAL_IOSURFACE_BIT_EXT = 0x00000010
VK_EXPORT_METAL_OBJECT_TYPE_METAL_SHARED_EVENT_BIT_EXT = 0x00000020
VK_EXPORT_METAL_OBJECT_TYPE_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_DEVICE_DEFAULT_EXT = 0
VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_DISABLED_EXT = 1
VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_ROBUST_BUFFER_ACCESS_EXT = 2
VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_ROBUST_BUFFER_ACCESS_2_EXT = 3
VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_EXT__BEGIN_RANGE = 0
VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_EXT__END_RANGE = 3
VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_EXT__RANGE_SIZE = 4
VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_EXT__MAX_ENUM = 2147483647
VK_PIPELINE_ROBUSTNESS_IMAGE_BEHAVIOR_DEVICE_DEFAULT_EXT = 0
VK_PIPELINE_ROBUSTNESS_IMAGE_BEHAVIOR_DISABLED_EXT = 1
VK_PIPELINE_ROBUSTNESS_IMAGE_BEHAVIOR_ROBUST_IMAGE_ACCESS_EXT = 2
VK_PIPELINE_ROBUSTNESS_IMAGE_BEHAVIOR_ROBUST_IMAGE_ACCESS_2_EXT = 3
VK_PIPELINE_ROBUSTNESS_IMAGE_BEHAVIOR_EXT__BEGIN_RANGE = 0
VK_PIPELINE_ROBUSTNESS_IMAGE_BEHAVIOR_EXT__END_RANGE = 3
VK_PIPELINE_ROBUSTNESS_IMAGE_BEHAVIOR_EXT__RANGE_SIZE = 4
VK_PIPELINE_ROBUSTNESS_IMAGE_BEHAVIOR_EXT__MAX_ENUM = 2147483647
VK_DEVICE_ADDRESS_BINDING_INTERNAL_OBJECT_BIT_EXT = 0x00000001
VK_DEVICE_ADDRESS_BINDING_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_DEVICE_ADDRESS_BINDING_TYPE_BIND_EXT = 0
VK_DEVICE_ADDRESS_BINDING_TYPE_UNBIND_EXT = 1
VK_DEVICE_ADDRESS_BINDING_TYPE_EXT__BEGIN_RANGE = 0
VK_DEVICE_ADDRESS_BINDING_TYPE_EXT__END_RANGE = 1
VK_DEVICE_ADDRESS_BINDING_TYPE_EXT__RANGE_SIZE = 2
VK_DEVICE_ADDRESS_BINDING_TYPE_EXT__MAX_ENUM = 2147483647
VK_MICROMAP_TYPE_OPACITY_MICROMAP_EXT = 0
VK_MICROMAP_TYPE_EXT__BEGIN_RANGE = 0
VK_MICROMAP_TYPE_EXT__END_RANGE = 0
VK_MICROMAP_TYPE_EXT__RANGE_SIZE = 1
VK_MICROMAP_TYPE_EXT__MAX_ENUM = 2147483647
VK_BUILD_MICROMAP_MODE_BUILD_EXT = 0
VK_BUILD_MICROMAP_MODE_EXT__BEGIN_RANGE = 0
VK_BUILD_MICROMAP_MODE_EXT__END_RANGE = 0
VK_BUILD_MICROMAP_MODE_EXT__RANGE_SIZE = 1
VK_BUILD_MICROMAP_MODE_EXT__MAX_ENUM = 2147483647
VK_COPY_MICROMAP_MODE_CLONE_EXT = 0
VK_COPY_MICROMAP_MODE_SERIALIZE_EXT = 1
VK_COPY_MICROMAP_MODE_DESERIALIZE_EXT = 2
VK_COPY_MICROMAP_MODE_COMPACT_EXT = 3
VK_COPY_MICROMAP_MODE_EXT__BEGIN_RANGE = 0
VK_COPY_MICROMAP_MODE_EXT__END_RANGE = 3
VK_COPY_MICROMAP_MODE_EXT__RANGE_SIZE = 4
VK_COPY_MICROMAP_MODE_EXT__MAX_ENUM = 2147483647
VK_BUILD_MICROMAP_PREFER_FAST_TRACE_BIT_EXT = 0x00000001
VK_BUILD_MICROMAP_PREFER_FAST_BUILD_BIT_EXT = 0x00000002
VK_BUILD_MICROMAP_ALLOW_COMPACTION_BIT_EXT = 0x00000004
VK_BUILD_MICROMAP_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_MICROMAP_CREATE_DEVICE_ADDRESS_CAPTURE_REPLAY_BIT_EXT = 0x00000001
VK_MICROMAP_CREATE_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_OPACITY_MICROMAP_FORMAT_2_STATE_EXT = 1
VK_OPACITY_MICROMAP_FORMAT_4_STATE_EXT = 2
VK_OPACITY_MICROMAP_FORMAT_EXT__BEGIN_RANGE = 1
VK_OPACITY_MICROMAP_FORMAT_EXT__END_RANGE = 2
VK_OPACITY_MICROMAP_FORMAT_EXT__RANGE_SIZE = 2
VK_OPACITY_MICROMAP_FORMAT_EXT__MAX_ENUM = 2147483647
VK_OPACITY_MICROMAP_SPECIAL_INDEX_FULLY_TRANSPARENT_EXT = -1
VK_OPACITY_MICROMAP_SPECIAL_INDEX_FULLY_OPAQUE_EXT = -2
VK_OPACITY_MICROMAP_SPECIAL_INDEX_FULLY_UNKNOWN_TRANSPARENT_EXT = -3
VK_OPACITY_MICROMAP_SPECIAL_INDEX_FULLY_UNKNOWN_OPAQUE_EXT = -4
VK_OPACITY_MICROMAP_SPECIAL_INDEX_EXT__BEGIN_RANGE = -4
VK_OPACITY_MICROMAP_SPECIAL_INDEX_EXT__END_RANGE = -1
VK_OPACITY_MICROMAP_SPECIAL_INDEX_EXT__RANGE_SIZE = 4
VK_OPACITY_MICROMAP_SPECIAL_INDEX_EXT__MAX_ENUM = 2147483647
VK_DEVICE_FAULT_VENDOR_BINARY_HEADER_VERSION_ONE_EXT = 1
VK_DEVICE_FAULT_VENDOR_BINARY_HEADER_VERSION_EXT__BEGIN_RANGE = 1
VK_DEVICE_FAULT_VENDOR_BINARY_HEADER_VERSION_EXT__END_RANGE = 1
VK_DEVICE_FAULT_VENDOR_BINARY_HEADER_VERSION_EXT__RANGE_SIZE = 1
VK_DEVICE_FAULT_VENDOR_BINARY_HEADER_VERSION_EXT__MAX_ENUM = 2147483647
VK_FRAME_BOUNDARY_FRAME_END_BIT_EXT = 0x00000001
VK_FRAME_BOUNDARY_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_MEMORY_DECOMPRESSION_METHOD_GDEFLATE_1_0_BIT_NV = 0x00000001
VK_MEMORY_DECOMPRESSION_METHOD_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_DEPTH_BIAS_REPRESENTATION_LEAST_REPRESENTABLE_VALUE_FORMAT_EXT = 0
VK_DEPTH_BIAS_REPRESENTATION_LEAST_REPRESENTABLE_VALUE_FORCE_UNORM_EXT = 1
VK_DEPTH_BIAS_REPRESENTATION_FLOAT_EXT = 2
VK_DEPTH_BIAS_REPRESENTATION_EXT__BEGIN_RANGE = 0
VK_DEPTH_BIAS_REPRESENTATION_EXT__END_RANGE = 2
VK_DEPTH_BIAS_REPRESENTATION_EXT__RANGE_SIZE = 3
VK_DEPTH_BIAS_REPRESENTATION_EXT__MAX_ENUM = 2147483647
VK_DIRECT_DRIVER_LOADING_MODE_EXCLUSIVE_LUNARG = 0
VK_DIRECT_DRIVER_LOADING_MODE_INCLUSIVE_LUNARG = 1
VK_DIRECT_DRIVER_LOADING_MODE_LUNARG__BEGIN_RANGE = 0
VK_DIRECT_DRIVER_LOADING_MODE_LUNARG__END_RANGE = 1
VK_DIRECT_DRIVER_LOADING_MODE_LUNARG__RANGE_SIZE = 2
VK_DIRECT_DRIVER_LOADING_MODE_LUNARG__MAX_ENUM = 2147483647
VK_PIPELINE_CREATE_2_DISABLE_OPTIMIZATION_BIT_KHR = 0x00000001
VK_PIPELINE_CREATE_2_ALLOW_DERIVATIVES_BIT_KHR = 0x00000002
VK_PIPELINE_CREATE_2_DERIVATIVE_BIT_KHR = 0x00000004
VK_PIPELINE_CREATE_FLAG_BITS2_KHR__MAX_ENUM = 2147483647
VK_BUFFER_USAGE_2_TRANSFER_SRC_BIT_KHR = 0x00000001
VK_BUFFER_USAGE_2_TRANSFER_DST_BIT_KHR = 0x00000002
VK_BUFFER_USAGE_2_UNIFORM_TEXEL_BUFFER_BIT_KHR = 0x00000004
VK_BUFFER_USAGE_2_STORAGE_TEXEL_BUFFER_BIT_KHR = 0x00000008
VK_BUFFER_USAGE_2_UNIFORM_BUFFER_BIT_KHR = 0x00000010
VK_BUFFER_USAGE_2_STORAGE_BUFFER_BIT_KHR = 0x00000020
VK_BUFFER_USAGE_2_INDEX_BUFFER_BIT_KHR = 0x00000040
VK_BUFFER_USAGE_2_VERTEX_BUFFER_BIT_KHR = 0x00000080
VK_BUFFER_USAGE_2_INDIRECT_BUFFER_BIT_KHR = 0x00000100
VK_BUFFER_USAGE_FLAG_BITS2_KHR__MAX_ENUM = 2147483647
VK_DISPLACEMENT_MICROMAP_FORMAT_64_TRIANGLES_64_BYTES_NV = 1
VK_DISPLACEMENT_MICROMAP_FORMAT_256_TRIANGLES_128_BYTES_NV = 2
VK_DISPLACEMENT_MICROMAP_FORMAT_1024_TRIANGLES_128_BYTES_NV = 3
VK_DISPLACEMENT_MICROMAP_FORMAT_NV__BEGIN_RANGE = 1
VK_DISPLACEMENT_MICROMAP_FORMAT_NV__END_RANGE = 3
VK_DISPLACEMENT_MICROMAP_FORMAT_NV__RANGE_SIZE = 3
VK_DISPLACEMENT_MICROMAP_FORMAT_NV__MAX_ENUM = 2147483647
VK_SHADER_CREATE_LINK_STAGE_BIT_EXT = 0x00000001
VK_SHADER_CREATE_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_SHADER_CODE_TYPE_BINARY_EXT = 0
VK_SHADER_CODE_TYPE_SPIRV_EXT = 1
VK_SHADER_CODE_TYPE_EXT__BEGIN_RANGE = 0
VK_SHADER_CODE_TYPE_EXT__END_RANGE = 1
VK_SHADER_CODE_TYPE_EXT__RANGE_SIZE = 2
VK_SHADER_CODE_TYPE_EXT__MAX_ENUM = 2147483647
VK_SCOPE_DEVICE_KHR = 1
VK_SCOPE_WORKGROUP_KHR = 2
VK_SCOPE_SUBGROUP_KHR = 3
VK_SCOPE_QUEUE_FAMILY_KHR = 5
VK_SCOPE_KHR__BEGIN_RANGE = 1
VK_SCOPE_KHR__END_RANGE = 5
VK_SCOPE_KHR__RANGE_SIZE = 5
VK_SCOPE_KHR__MAX_ENUM = 2147483647
VK_COMPONENT_TYPE_FLOAT16_KHR = 0
VK_COMPONENT_TYPE_FLOAT32_KHR = 1
VK_COMPONENT_TYPE_FLOAT64_KHR = 2
VK_COMPONENT_TYPE_SINT8_KHR = 3
VK_COMPONENT_TYPE_SINT16_KHR = 4
VK_COMPONENT_TYPE_SINT32_KHR = 5
VK_COMPONENT_TYPE_SINT64_KHR = 6
VK_COMPONENT_TYPE_UINT8_KHR = 7
VK_COMPONENT_TYPE_UINT16_KHR = 8
VK_COMPONENT_TYPE_UINT32_KHR = 9
VK_COMPONENT_TYPE_UINT64_KHR = 10
VK_COMPONENT_TYPE_KHR__BEGIN_RANGE = 0
VK_COMPONENT_TYPE_KHR__END_RANGE = 10
VK_COMPONENT_TYPE_KHR__RANGE_SIZE = 11
VK_COMPONENT_TYPE_KHR__MAX_ENUM = 2147483647
VK_CUBIC_FILTER_WEIGHTS_CATMULL_ROM_QCOM = 0
VK_CUBIC_FILTER_WEIGHTS_ZERO_TANGENT_CARDINAL_QCOM = 1
VK_CUBIC_FILTER_WEIGHTS_B_SPLINE_QCOM = 2
VK_CUBIC_FILTER_WEIGHTS_MITCHELL_NETRAVALI_QCOM = 3
VK_CUBIC_FILTER_WEIGHTS_QCOM__BEGIN_RANGE = 0
VK_CUBIC_FILTER_WEIGHTS_QCOM__END_RANGE = 3
VK_CUBIC_FILTER_WEIGHTS_QCOM__RANGE_SIZE = 4
VK_CUBIC_FILTER_WEIGHTS_QCOM__MAX_ENUM = 2147483647
VK_BLOCK_MATCH_WINDOW_COMPARE_MODE_MIN_QCOM = 0
VK_BLOCK_MATCH_WINDOW_COMPARE_MODE_MAX_QCOM = 1
VK_BLOCK_MATCH_WINDOW_COMPARE_MODE_QCOM__BEGIN_RANGE = 0
VK_BLOCK_MATCH_WINDOW_COMPARE_MODE_QCOM__END_RANGE = 1
VK_BLOCK_MATCH_WINDOW_COMPARE_MODE_QCOM__RANGE_SIZE = 2
VK_BLOCK_MATCH_WINDOW_COMPARE_MODE_QCOM__MAX_ENUM = 2147483647
VK_LAYERED_DRIVER_UNDERLYING_API_NONE_MSFT = 0
VK_LAYERED_DRIVER_UNDERLYING_API_D3D12_MSFT = 1
VK_LAYERED_DRIVER_UNDERLYING_API_MSFT__BEGIN_RANGE = 0
VK_LAYERED_DRIVER_UNDERLYING_API_MSFT__END_RANGE = 1
VK_LAYERED_DRIVER_UNDERLYING_API_MSFT__RANGE_SIZE = 2
VK_LAYERED_DRIVER_UNDERLYING_API_MSFT__MAX_ENUM = 2147483647
VK_COLOR_SPACE_SRGB_NONLINEAR_KHR = 0
VK_COLORSPACE_SRGB_NONLINEAR_KHR = 0
VK_COLOR_SPACE_KHR__BEGIN_RANGE = 0
VK_COLOR_SPACE_KHR__END_RANGE = 0
VK_COLOR_SPACE_KHR__RANGE_SIZE = 1
VK_COLOR_SPACE_KHR__MAX_ENUM = 2147483647
VK_COMPOSITE_ALPHA_OPAQUE_BIT_KHR = 0x00000001
VK_COMPOSITE_ALPHA_PRE_MULTIPLIED_BIT_KHR = 0x00000002
VK_COMPOSITE_ALPHA_POST_MULTIPLIED_BIT_KHR = 0x00000004
VK_COMPOSITE_ALPHA_INHERIT_BIT_KHR = 0x00000008
VK_COMPOSITE_ALPHA_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_DISPLAY_PLANE_ALPHA_OPAQUE_BIT_KHR = 0x00000001
VK_DISPLAY_PLANE_ALPHA_GLOBAL_BIT_KHR = 0x00000002
VK_DISPLAY_PLANE_ALPHA_PER_PIXEL_BIT_KHR = 0x00000004
VK_DISPLAY_PLANE_ALPHA_PER_PIXEL_PREMULTIPLIED_BIT_KHR = 0x00000008
VK_DISPLAY_PLANE_ALPHA_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_PRESENT_MODE_IMMEDIATE_KHR = 0
VK_PRESENT_MODE_MAILBOX_KHR = 1
VK_PRESENT_MODE_FIFO_KHR = 2
VK_PRESENT_MODE_FIFO_RELAXED_KHR = 3
VK_PRESENT_MODE_KHR__BEGIN_RANGE = 0
VK_PRESENT_MODE_KHR__END_RANGE = 3
VK_PRESENT_MODE_KHR__RANGE_SIZE = 4
VK_PRESENT_MODE_KHR__MAX_ENUM = 2147483647
VK_SURFACE_TRANSFORM_IDENTITY_BIT_KHR = 0x00000001
VK_SURFACE_TRANSFORM_ROTATE_90_BIT_KHR = 0x00000002
VK_SURFACE_TRANSFORM_ROTATE_180_BIT_KHR = 0x00000004
VK_SURFACE_TRANSFORM_ROTATE_270_BIT_KHR = 0x00000008
VK_SURFACE_TRANSFORM_HORIZONTAL_MIRROR_BIT_KHR = 0x00000010
VK_SURFACE_TRANSFORM_HORIZONTAL_MIRROR_ROTATE_90_BIT_KHR = 0x00000020
VK_SURFACE_TRANSFORM_HORIZONTAL_MIRROR_ROTATE_180_BIT_KHR = 0x00000040
VK_SURFACE_TRANSFORM_HORIZONTAL_MIRROR_ROTATE_270_BIT_KHR = 0x00000080
VK_SURFACE_TRANSFORM_INHERIT_BIT_KHR = 0x00000100
VK_SURFACE_TRANSFORM_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_DEBUG_REPORT_INFORMATION_BIT_EXT = 0x00000001
VK_DEBUG_REPORT_WARNING_BIT_EXT = 0x00000002
VK_DEBUG_REPORT_PERFORMANCE_WARNING_BIT_EXT = 0x00000004
VK_DEBUG_REPORT_ERROR_BIT_EXT = 0x00000008
VK_DEBUG_REPORT_DEBUG_BIT_EXT = 0x00000010
VK_DEBUG_REPORT_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_DEBUG_REPORT_OBJECT_TYPE_UNKNOWN_EXT = 0
VK_DEBUG_REPORT_OBJECT_TYPE_INSTANCE_EXT = 1
VK_DEBUG_REPORT_OBJECT_TYPE_PHYSICAL_DEVICE_EXT = 2
VK_DEBUG_REPORT_OBJECT_TYPE_DEVICE_EXT = 3
VK_DEBUG_REPORT_OBJECT_TYPE_QUEUE_EXT = 4
VK_DEBUG_REPORT_OBJECT_TYPE_SEMAPHORE_EXT = 5
VK_DEBUG_REPORT_OBJECT_TYPE_COMMAND_BUFFER_EXT = 6
VK_DEBUG_REPORT_OBJECT_TYPE_FENCE_EXT = 7
VK_DEBUG_REPORT_OBJECT_TYPE_DEVICE_MEMORY_EXT = 8
VK_DEBUG_REPORT_OBJECT_TYPE_BUFFER_EXT = 9
VK_DEBUG_REPORT_OBJECT_TYPE_IMAGE_EXT = 10
VK_DEBUG_REPORT_OBJECT_TYPE_EVENT_EXT = 11
VK_DEBUG_REPORT_OBJECT_TYPE_QUERY_POOL_EXT = 12
VK_DEBUG_REPORT_OBJECT_TYPE_BUFFER_VIEW_EXT = 13
VK_DEBUG_REPORT_OBJECT_TYPE_IMAGE_VIEW_EXT = 14
VK_DEBUG_REPORT_OBJECT_TYPE_SHADER_MODULE_EXT = 15
VK_DEBUG_REPORT_OBJECT_TYPE_PIPELINE_CACHE_EXT = 16
VK_DEBUG_REPORT_OBJECT_TYPE_PIPELINE_LAYOUT_EXT = 17
VK_DEBUG_REPORT_OBJECT_TYPE_RENDER_PASS_EXT = 18
VK_DEBUG_REPORT_OBJECT_TYPE_PIPELINE_EXT = 19
VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_SET_LAYOUT_EXT = 20
VK_DEBUG_REPORT_OBJECT_TYPE_SAMPLER_EXT = 21
VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_POOL_EXT = 22
VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_SET_EXT = 23
VK_DEBUG_REPORT_OBJECT_TYPE_FRAMEBUFFER_EXT = 24
VK_DEBUG_REPORT_OBJECT_TYPE_COMMAND_POOL_EXT = 25
VK_DEBUG_REPORT_OBJECT_TYPE_SURFACE_KHR_EXT = 26
VK_DEBUG_REPORT_OBJECT_TYPE_SWAPCHAIN_KHR_EXT = 27
VK_DEBUG_REPORT_OBJECT_TYPE_DEBUG_REPORT_CALLBACK_EXT_EXT = 28
VK_DEBUG_REPORT_OBJECT_TYPE_DEBUG_REPORT_EXT = 28
VK_DEBUG_REPORT_OBJECT_TYPE_DISPLAY_KHR_EXT = 29
VK_DEBUG_REPORT_OBJECT_TYPE_DISPLAY_MODE_KHR_EXT = 30
VK_DEBUG_REPORT_OBJECT_TYPE_VALIDATION_CACHE_EXT_EXT = 33
VK_DEBUG_REPORT_OBJECT_TYPE_VALIDATION_CACHE_EXT = 33
VK_DEBUG_REPORT_OBJECT_TYPE_EXT__BEGIN_RANGE = 0
VK_DEBUG_REPORT_OBJECT_TYPE_EXT__END_RANGE = 33
VK_DEBUG_REPORT_OBJECT_TYPE_EXT__RANGE_SIZE = 34
VK_DEBUG_REPORT_OBJECT_TYPE_EXT__MAX_ENUM = 2147483647
VK_DEVICE_MEMORY_REPORT_EVENT_TYPE_ALLOCATE_EXT = 0
VK_DEVICE_MEMORY_REPORT_EVENT_TYPE_FREE_EXT = 1
VK_DEVICE_MEMORY_REPORT_EVENT_TYPE_IMPORT_EXT = 2
VK_DEVICE_MEMORY_REPORT_EVENT_TYPE_UNIMPORT_EXT = 3
VK_DEVICE_MEMORY_REPORT_EVENT_TYPE_ALLOCATION_FAILED_EXT = 4
VK_DEVICE_MEMORY_REPORT_EVENT_TYPE_EXT__BEGIN_RANGE = 0
VK_DEVICE_MEMORY_REPORT_EVENT_TYPE_EXT__END_RANGE = 4
VK_DEVICE_MEMORY_REPORT_EVENT_TYPE_EXT__RANGE_SIZE = 5
VK_DEVICE_MEMORY_REPORT_EVENT_TYPE_EXT__MAX_ENUM = 2147483647
VK_RASTERIZATION_ORDER_STRICT_AMD = 0
VK_RASTERIZATION_ORDER_RELAXED_AMD = 1
VK_RASTERIZATION_ORDER_AMD__BEGIN_RANGE = 0
VK_RASTERIZATION_ORDER_AMD__END_RANGE = 1
VK_RASTERIZATION_ORDER_AMD__RANGE_SIZE = 2
VK_RASTERIZATION_ORDER_AMD__MAX_ENUM = 2147483647
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_BIT_NV = 0x00000001
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT_NV = 0x00000002
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_IMAGE_BIT_NV = 0x00000004
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_IMAGE_KMT_BIT_NV = 0x00000008
VK_EXTERNAL_MEMORY_HANDLE_TYPE_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_EXTERNAL_MEMORY_FEATURE_DEDICATED_ONLY_BIT_NV = 0x00000001
VK_EXTERNAL_MEMORY_FEATURE_EXPORTABLE_BIT_NV = 0x00000002
VK_EXTERNAL_MEMORY_FEATURE_IMPORTABLE_BIT_NV = 0x00000004
VK_EXTERNAL_MEMORY_FEATURE_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_VALIDATION_CHECK_ALL_EXT = 0
VK_VALIDATION_CHECK_SHADERS_EXT = 1
VK_VALIDATION_CHECK_EXT__BEGIN_RANGE = 0
VK_VALIDATION_CHECK_EXT__END_RANGE = 1
VK_VALIDATION_CHECK_EXT__RANGE_SIZE = 2
VK_VALIDATION_CHECK_EXT__MAX_ENUM = 2147483647
VK_VALIDATION_FEATURE_ENABLE_GPU_ASSISTED_EXT = 0
VK_VALIDATION_FEATURE_ENABLE_GPU_ASSISTED_RESERVE_BINDING_SLOT_EXT = 1
VK_VALIDATION_FEATURE_ENABLE_BEST_PRACTICES_EXT = 2
VK_VALIDATION_FEATURE_ENABLE_DEBUG_PRINTF_EXT = 3
VK_VALIDATION_FEATURE_ENABLE_SYNCHRONIZATION_VALIDATION_EXT = 4
VK_VALIDATION_FEATURE_ENABLE_EXT__BEGIN_RANGE = 0
VK_VALIDATION_FEATURE_ENABLE_EXT__END_RANGE = 4
VK_VALIDATION_FEATURE_ENABLE_EXT__RANGE_SIZE = 5
VK_VALIDATION_FEATURE_ENABLE_EXT__MAX_ENUM = 2147483647
VK_VALIDATION_FEATURE_DISABLE_ALL_EXT = 0
VK_VALIDATION_FEATURE_DISABLE_SHADERS_EXT = 1
VK_VALIDATION_FEATURE_DISABLE_THREAD_SAFETY_EXT = 2
VK_VALIDATION_FEATURE_DISABLE_API_PARAMETERS_EXT = 3
VK_VALIDATION_FEATURE_DISABLE_OBJECT_LIFETIMES_EXT = 4
VK_VALIDATION_FEATURE_DISABLE_CORE_CHECKS_EXT = 5
VK_VALIDATION_FEATURE_DISABLE_UNIQUE_HANDLES_EXT = 6
VK_VALIDATION_FEATURE_DISABLE_SHADER_VALIDATION_CACHE_EXT = 7
VK_VALIDATION_FEATURE_DISABLE_EXT__BEGIN_RANGE = 0
VK_VALIDATION_FEATURE_DISABLE_EXT__END_RANGE = 7
VK_VALIDATION_FEATURE_DISABLE_EXT__RANGE_SIZE = 8
VK_VALIDATION_FEATURE_DISABLE_EXT__MAX_ENUM = 2147483647
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_FD_BIT = 0x00000001
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_BIT = 0x00000002
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT = 0x00000004
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_TEXTURE_BIT = 0x00000008
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_TEXTURE_KMT_BIT = 0x00000010
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_HEAP_BIT = 0x00000020
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_RESOURCE_BIT = 0x00000040
VK_EXTERNAL_MEMORY_HANDLE_TYPE_FLAG_BITS__MAX_ENUM = 2147483647
VK_EXTERNAL_MEMORY_FEATURE_DEDICATED_ONLY_BIT = 0x00000001
VK_EXTERNAL_MEMORY_FEATURE_EXPORTABLE_BIT = 0x00000002
VK_EXTERNAL_MEMORY_FEATURE_IMPORTABLE_BIT = 0x00000004
VK_EXTERNAL_MEMORY_FEATURE_FLAG_BITS__MAX_ENUM = 2147483647
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_FD_BIT = 0x00000001
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_BIT = 0x00000002
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT = 0x00000004
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D12_FENCE_BIT = 0x00000008
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_FENCE_BIT = 0x00000008
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_SYNC_FD_BIT = 0x00000010
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_FLAG_BITS__MAX_ENUM = 2147483647
VK_EXTERNAL_SEMAPHORE_FEATURE_EXPORTABLE_BIT = 0x00000001
VK_EXTERNAL_SEMAPHORE_FEATURE_IMPORTABLE_BIT = 0x00000002
VK_EXTERNAL_SEMAPHORE_FEATURE_FLAG_BITS__MAX_ENUM = 2147483647
VK_SEMAPHORE_IMPORT_TEMPORARY_BIT = 0x00000001
VK_SEMAPHORE_IMPORT_FLAG_BITS__MAX_ENUM = 2147483647
VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_FD_BIT = 0x00000001
VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_WIN32_BIT = 0x00000002
VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT = 0x00000004
VK_EXTERNAL_FENCE_HANDLE_TYPE_SYNC_FD_BIT = 0x00000008
VK_EXTERNAL_FENCE_HANDLE_TYPE_FLAG_BITS__MAX_ENUM = 2147483647
VK_EXTERNAL_FENCE_FEATURE_EXPORTABLE_BIT = 0x00000001
VK_EXTERNAL_FENCE_FEATURE_IMPORTABLE_BIT = 0x00000002
VK_EXTERNAL_FENCE_FEATURE_FLAG_BITS__MAX_ENUM = 2147483647
VK_FENCE_IMPORT_TEMPORARY_BIT = 0x00000001
VK_FENCE_IMPORT_FLAG_BITS__MAX_ENUM = 2147483647
VK_SURFACE_COUNTER_VBLANK_BIT_EXT = 0x00000001
VK_SURFACE_COUNTER_VBLANK_EXT = 0x00000001
VK_SURFACE_COUNTER_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_DISPLAY_POWER_STATE_OFF_EXT = 0
VK_DISPLAY_POWER_STATE_SUSPEND_EXT = 1
VK_DISPLAY_POWER_STATE_ON_EXT = 2
VK_DISPLAY_POWER_STATE_EXT__BEGIN_RANGE = 0
VK_DISPLAY_POWER_STATE_EXT__END_RANGE = 2
VK_DISPLAY_POWER_STATE_EXT__RANGE_SIZE = 3
VK_DISPLAY_POWER_STATE_EXT__MAX_ENUM = 2147483647
VK_DEVICE_EVENT_TYPE_DISPLAY_HOTPLUG_EXT = 0
VK_DEVICE_EVENT_TYPE_EXT__BEGIN_RANGE = 0
VK_DEVICE_EVENT_TYPE_EXT__END_RANGE = 0
VK_DEVICE_EVENT_TYPE_EXT__RANGE_SIZE = 1
VK_DEVICE_EVENT_TYPE_EXT__MAX_ENUM = 2147483647
VK_DISPLAY_EVENT_TYPE_FIRST_PIXEL_OUT_EXT = 0
VK_DISPLAY_EVENT_TYPE_EXT__BEGIN_RANGE = 0
VK_DISPLAY_EVENT_TYPE_EXT__END_RANGE = 0
VK_DISPLAY_EVENT_TYPE_EXT__RANGE_SIZE = 1
VK_DISPLAY_EVENT_TYPE_EXT__MAX_ENUM = 2147483647
VK_PEER_MEMORY_FEATURE_COPY_SRC_BIT = 0x00000001
VK_PEER_MEMORY_FEATURE_COPY_DST_BIT = 0x00000002
VK_PEER_MEMORY_FEATURE_GENERIC_SRC_BIT = 0x00000004
VK_PEER_MEMORY_FEATURE_GENERIC_DST_BIT = 0x00000008
VK_PEER_MEMORY_FEATURE_FLAG_BITS__MAX_ENUM = 2147483647
VK_MEMORY_ALLOCATE_DEVICE_MASK_BIT = 0x00000001
VK_MEMORY_ALLOCATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_DEVICE_GROUP_PRESENT_MODE_LOCAL_BIT_KHR = 0x00000001
VK_DEVICE_GROUP_PRESENT_MODE_REMOTE_BIT_KHR = 0x00000002
VK_DEVICE_GROUP_PRESENT_MODE_SUM_BIT_KHR = 0x00000004
VK_DEVICE_GROUP_PRESENT_MODE_LOCAL_MULTI_DEVICE_BIT_KHR = 0x00000008
VK_DEVICE_GROUP_PRESENT_MODE_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_SUBGROUP_FEATURE_BASIC_BIT = 0x00000001
VK_SUBGROUP_FEATURE_VOTE_BIT = 0x00000002
VK_SUBGROUP_FEATURE_ARITHMETIC_BIT = 0x00000004
VK_SUBGROUP_FEATURE_BALLOT_BIT = 0x00000008
VK_SUBGROUP_FEATURE_SHUFFLE_BIT = 0x00000010
VK_SUBGROUP_FEATURE_SHUFFLE_RELATIVE_BIT = 0x00000020
VK_SUBGROUP_FEATURE_CLUSTERED_BIT = 0x00000040
VK_SUBGROUP_FEATURE_QUAD_BIT = 0x00000080
VK_SUBGROUP_FEATURE_FLAG_BITS__MAX_ENUM = 2147483647
VK_TESSELLATION_DOMAIN_ORIGIN_UPPER_LEFT = 0
VK_TESSELLATION_DOMAIN_ORIGIN_LOWER_LEFT = 1
VK_TESSELLATION_DOMAIN_ORIGIN__BEGIN_RANGE = 0
VK_TESSELLATION_DOMAIN_ORIGIN__END_RANGE = 1
VK_TESSELLATION_DOMAIN_ORIGIN__RANGE_SIZE = 2
VK_TESSELLATION_DOMAIN_ORIGIN__MAX_ENUM = 2147483647
VK_SAMPLER_YCBCR_MODEL_CONVERSION_RGB_IDENTITY = 0
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_IDENTITY = 1
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_709 = 2
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_601 = 3
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_2020 = 4
VK_SAMPLER_YCBCR_MODEL_CONVERSION__BEGIN_RANGE = 0
VK_SAMPLER_YCBCR_MODEL_CONVERSION__END_RANGE = 4
VK_SAMPLER_YCBCR_MODEL_CONVERSION__RANGE_SIZE = 5
VK_SAMPLER_YCBCR_MODEL_CONVERSION__MAX_ENUM = 2147483647
VK_SAMPLER_YCBCR_RANGE_ITU_FULL = 0
VK_SAMPLER_YCBCR_RANGE_ITU_NARROW = 1
VK_SAMPLER_YCBCR_RANGE__BEGIN_RANGE = 0
VK_SAMPLER_YCBCR_RANGE__END_RANGE = 1
VK_SAMPLER_YCBCR_RANGE__RANGE_SIZE = 2
VK_SAMPLER_YCBCR_RANGE__MAX_ENUM = 2147483647
VK_CHROMA_LOCATION_COSITED_EVEN = 0
VK_CHROMA_LOCATION_MIDPOINT = 1
VK_CHROMA_LOCATION__BEGIN_RANGE = 0
VK_CHROMA_LOCATION__END_RANGE = 1
VK_CHROMA_LOCATION__RANGE_SIZE = 2
VK_CHROMA_LOCATION__MAX_ENUM = 2147483647
VK_SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE = 0
VK_SAMPLER_REDUCTION_MODE_MIN = 1
VK_SAMPLER_REDUCTION_MODE_MAX = 2
VK_SAMPLER_REDUCTION_MODE__BEGIN_RANGE = 0
VK_SAMPLER_REDUCTION_MODE__END_RANGE = 2
VK_SAMPLER_REDUCTION_MODE__RANGE_SIZE = 3
VK_SAMPLER_REDUCTION_MODE__MAX_ENUM = 2147483647
VK_BLEND_OVERLAP_UNCORRELATED_EXT = 0
VK_BLEND_OVERLAP_DISJOINT_EXT = 1
VK_BLEND_OVERLAP_CONJOINT_EXT = 2
VK_BLEND_OVERLAP_EXT__BEGIN_RANGE = 0
VK_BLEND_OVERLAP_EXT__END_RANGE = 2
VK_BLEND_OVERLAP_EXT__RANGE_SIZE = 3
VK_BLEND_OVERLAP_EXT__MAX_ENUM = 2147483647
VK_DEBUG_UTILS_MESSAGE_SEVERITY_VERBOSE_BIT_EXT = 0x00000001
VK_DEBUG_UTILS_MESSAGE_SEVERITY_INFO_BIT_EXT = 0x00000010
VK_DEBUG_UTILS_MESSAGE_SEVERITY_WARNING_BIT_EXT = 0x00000100
VK_DEBUG_UTILS_MESSAGE_SEVERITY_ERROR_BIT_EXT = 0x00001000
VK_DEBUG_UTILS_MESSAGE_SEVERITY_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_DEBUG_UTILS_MESSAGE_TYPE_GENERAL_BIT_EXT = 0x00000001
VK_DEBUG_UTILS_MESSAGE_TYPE_VALIDATION_BIT_EXT = 0x00000002
VK_DEBUG_UTILS_MESSAGE_TYPE_PERFORMANCE_BIT_EXT = 0x00000004
VK_DEBUG_UTILS_MESSAGE_TYPE_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_FULL_SCREEN_EXCLUSIVE_DEFAULT_EXT = 0
VK_FULL_SCREEN_EXCLUSIVE_ALLOWED_EXT = 1
VK_FULL_SCREEN_EXCLUSIVE_DISALLOWED_EXT = 2
VK_FULL_SCREEN_EXCLUSIVE_APPLICATION_CONTROLLED_EXT = 3
VK_FULL_SCREEN_EXCLUSIVE_EXT__BEGIN_RANGE = 0
VK_FULL_SCREEN_EXCLUSIVE_EXT__END_RANGE = 3
VK_FULL_SCREEN_EXCLUSIVE_EXT__RANGE_SIZE = 4
VK_FULL_SCREEN_EXCLUSIVE_EXT__MAX_ENUM = 2147483647
VK_SHADER_FLOAT_CONTROLS_INDEPENDENCE_32_BIT_ONLY = 0
VK_SHADER_FLOAT_CONTROLS_INDEPENDENCE_ALL = 1
VK_SHADER_FLOAT_CONTROLS_INDEPENDENCE_NONE = 2
VK_SHADER_FLOAT_CONTROLS_INDEPENDENCE__BEGIN_RANGE = 0
VK_SHADER_FLOAT_CONTROLS_INDEPENDENCE__END_RANGE = 2
VK_SHADER_FLOAT_CONTROLS_INDEPENDENCE__RANGE_SIZE = 3
VK_SHADER_FLOAT_CONTROLS_INDEPENDENCE__MAX_ENUM = 2147483647
VK_SWAPCHAIN_IMAGE_USAGE_SHARED_BIT_ANDROID = 0x00000001
VK_SWAPCHAIN_IMAGE_USAGE_FLAG_BITS_ANDROID__MAX_ENUM = 2147483647
VK_FRAGMENT_SHADING_RATE_COMBINER_OP_KEEP_KHR = 0
VK_FRAGMENT_SHADING_RATE_COMBINER_OP_REPLACE_KHR = 1
VK_FRAGMENT_SHADING_RATE_COMBINER_OP_MIN_KHR = 2
VK_FRAGMENT_SHADING_RATE_COMBINER_OP_MAX_KHR = 3
VK_FRAGMENT_SHADING_RATE_COMBINER_OP_MUL_KHR = 4
VK_FRAGMENT_SHADING_RATE_COMBINER_OP_KHR__BEGIN_RANGE = 0
VK_FRAGMENT_SHADING_RATE_COMBINER_OP_KHR__END_RANGE = 4
VK_FRAGMENT_SHADING_RATE_COMBINER_OP_KHR__RANGE_SIZE = 5
VK_FRAGMENT_SHADING_RATE_COMBINER_OP_KHR__MAX_ENUM = 2147483647
VK_SUBMIT_PROTECTED_BIT = 0x00000001
VK_SUBMIT_PROTECTED_BIT_KHR = 0x00000001
VK_SUBMIT_FLAG_BITS__MAX_ENUM = 2147483647
VK_GRAPHICS_PIPELINE_LIBRARY_VERTEX_INPUT_INTERFACE_BIT_EXT = 0x00000001
VK_GRAPHICS_PIPELINE_LIBRARY_PRE_RASTERIZATION_SHADERS_BIT_EXT = 0x00000002
VK_GRAPHICS_PIPELINE_LIBRARY_FRAGMENT_SHADER_BIT_EXT = 0x00000004
VK_GRAPHICS_PIPELINE_LIBRARY_FRAGMENT_OUTPUT_INTERFACE_BIT_EXT = 0x00000008
VK_GRAPHICS_PIPELINE_LIBRARY_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_OPTICAL_FLOW_GRID_SIZE_UNKNOWN_NV = 0
VK_OPTICAL_FLOW_GRID_SIZE_1X1_BIT_NV = 0x00000001
VK_OPTICAL_FLOW_GRID_SIZE_2X2_BIT_NV = 0x00000002
VK_OPTICAL_FLOW_GRID_SIZE_4X4_BIT_NV = 0x00000004
VK_OPTICAL_FLOW_GRID_SIZE_8X8_BIT_NV = 0x00000008
VK_OPTICAL_FLOW_GRID_SIZE_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_OPTICAL_FLOW_USAGE_UNKNOWN_NV = 0
VK_OPTICAL_FLOW_USAGE_INPUT_BIT_NV = 0x00000001
VK_OPTICAL_FLOW_USAGE_OUTPUT_BIT_NV = 0x00000002
VK_OPTICAL_FLOW_USAGE_HINT_BIT_NV = 0x00000004
VK_OPTICAL_FLOW_USAGE_COST_BIT_NV = 0x00000008
VK_OPTICAL_FLOW_USAGE_GLOBAL_FLOW_BIT_NV = 0x00000010
VK_OPTICAL_FLOW_USAGE_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_OPTICAL_FLOW_PERFORMANCE_LEVEL_UNKNOWN_NV = 0
VK_OPTICAL_FLOW_PERFORMANCE_LEVEL_SLOW_NV = 1
VK_OPTICAL_FLOW_PERFORMANCE_LEVEL_MEDIUM_NV = 2
VK_OPTICAL_FLOW_PERFORMANCE_LEVEL_FAST_NV = 3
VK_OPTICAL_FLOW_PERFORMANCE_LEVEL_NV__BEGIN_RANGE = 0
VK_OPTICAL_FLOW_PERFORMANCE_LEVEL_NV__END_RANGE = 3
VK_OPTICAL_FLOW_PERFORMANCE_LEVEL_NV__RANGE_SIZE = 4
VK_OPTICAL_FLOW_PERFORMANCE_LEVEL_NV__MAX_ENUM = 2147483647
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_UNKNOWN_NV = 0
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_INPUT_NV = 1
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_REFERENCE_NV = 2
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_HINT_NV = 3
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_FLOW_VECTOR_NV = 4
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_BACKWARD_FLOW_VECTOR_NV = 5
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_COST_NV = 6
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_BACKWARD_COST_NV = 7
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_GLOBAL_FLOW_NV = 8
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_NV__BEGIN_RANGE = 0
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_NV__END_RANGE = 8
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_NV__RANGE_SIZE = 9
VK_OPTICAL_FLOW_SESSION_BINDING_POINT_NV__MAX_ENUM = 2147483647
VK_OPTICAL_FLOW_SESSION_CREATE_ENABLE_HINT_BIT_NV = 0x00000001
VK_OPTICAL_FLOW_SESSION_CREATE_ENABLE_COST_BIT_NV = 0x00000002
VK_OPTICAL_FLOW_SESSION_CREATE_ENABLE_GLOBAL_FLOW_BIT_NV = 0x00000004
VK_OPTICAL_FLOW_SESSION_CREATE_ALLOW_REGIONS_BIT_NV = 0x00000008
VK_OPTICAL_FLOW_SESSION_CREATE_BOTH_DIRECTIONS_BIT_NV = 0x00000010
VK_OPTICAL_FLOW_SESSION_CREATE_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_OPTICAL_FLOW_EXECUTE_DISABLE_TEMPORAL_HINTS_BIT_NV = 0x00000001
VK_OPTICAL_FLOW_EXECUTE_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_DEVICE_FAULT_ADDRESS_TYPE_NONE_EXT = 0
VK_DEVICE_FAULT_ADDRESS_TYPE_READ_INVALID_EXT = 1
VK_DEVICE_FAULT_ADDRESS_TYPE_WRITE_INVALID_EXT = 2
VK_DEVICE_FAULT_ADDRESS_TYPE_EXECUTE_INVALID_EXT = 3
VK_DEVICE_FAULT_ADDRESS_TYPE_INSTRUCTION_POINTER_UNKNOWN_EXT = 4
VK_DEVICE_FAULT_ADDRESS_TYPE_INSTRUCTION_POINTER_INVALID_EXT = 5
VK_DEVICE_FAULT_ADDRESS_TYPE_INSTRUCTION_POINTER_FAULT_EXT = 6
VK_DEVICE_FAULT_ADDRESS_TYPE_EXT__BEGIN_RANGE = 0
VK_DEVICE_FAULT_ADDRESS_TYPE_EXT__END_RANGE = 6
VK_DEVICE_FAULT_ADDRESS_TYPE_EXT__RANGE_SIZE = 7
VK_DEVICE_FAULT_ADDRESS_TYPE_EXT__MAX_ENUM = 2147483647
VK_PRESENT_SCALING_ONE_TO_ONE_BIT_EXT = 0x00000001
VK_PRESENT_SCALING_ASPECT_RATIO_STRETCH_BIT_EXT = 0x00000002
VK_PRESENT_SCALING_STRETCH_BIT_EXT = 0x00000004
VK_PRESENT_SCALING_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_PRESENT_GRAVITY_MIN_BIT_EXT = 0x00000001
VK_PRESENT_GRAVITY_MAX_BIT_EXT = 0x00000002
VK_PRESENT_GRAVITY_CENTERED_BIT_EXT = 0x00000004
VK_PRESENT_GRAVITY_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_LAYER_SETTING_TYPE_BOOL32_EXT = 0
VK_LAYER_SETTING_TYPE_INT32_EXT = 1
VK_LAYER_SETTING_TYPE_INT64_EXT = 2
VK_LAYER_SETTING_TYPE_UINT32_EXT = 3
VK_LAYER_SETTING_TYPE_UINT64_EXT = 4
VK_LAYER_SETTING_TYPE_FLOAT32_EXT = 5
VK_LAYER_SETTING_TYPE_FLOAT64_EXT = 6
VK_LAYER_SETTING_TYPE_STRING_EXT = 7
VK_LAYER_SETTING_TYPE_EXT__BEGIN_RANGE = 0
VK_LAYER_SETTING_TYPE_EXT__END_RANGE = 7
VK_LAYER_SETTING_TYPE_EXT__RANGE_SIZE = 8
VK_LAYER_SETTING_TYPE_EXT__MAX_ENUM = 2147483647
VK_LATENCY_MARKER_SIMULATION_START_NV = 0
VK_LATENCY_MARKER_SIMULATION_END_NV = 1
VK_LATENCY_MARKER_RENDERSUBMIT_START_NV = 2
VK_LATENCY_MARKER_RENDERSUBMIT_END_NV = 3
VK_LATENCY_MARKER_PRESENT_START_NV = 4
VK_LATENCY_MARKER_PRESENT_END_NV = 5
VK_LATENCY_MARKER_INPUT_SAMPLE_NV = 6
VK_LATENCY_MARKER_TRIGGER_FLASH_NV = 7
VK_LATENCY_MARKER_OUT_OF_BAND_RENDERSUBMIT_START_NV = 8
VK_LATENCY_MARKER_OUT_OF_BAND_RENDERSUBMIT_END_NV = 9
VK_LATENCY_MARKER_OUT_OF_BAND_PRESENT_START_NV = 10
VK_LATENCY_MARKER_OUT_OF_BAND_PRESENT_END_NV = 11
VK_LATENCY_MARKER_NV__BEGIN_RANGE = 0
VK_LATENCY_MARKER_NV__END_RANGE = 11
VK_LATENCY_MARKER_NV__RANGE_SIZE = 12
VK_LATENCY_MARKER_NV__MAX_ENUM = 2147483647
VK_OUT_OF_BAND_QUEUE_TYPE_RENDER_NV = 0
VK_OUT_OF_BAND_QUEUE_TYPE_PRESENT_NV = 1
VK_OUT_OF_BAND_QUEUE_TYPE_NV__BEGIN_RANGE = 0
VK_OUT_OF_BAND_QUEUE_TYPE_NV__END_RANGE = 1
VK_OUT_OF_BAND_QUEUE_TYPE_NV__RANGE_SIZE = 2
VK_OUT_OF_BAND_QUEUE_TYPE_NV__MAX_ENUM = 2147483647
VK_PHYSICAL_DEVICE_SCHEDULING_CONTROLS_SHADER_CORE_COUNT_ARM = 0x00000001
VK_PHYSICAL_DEVICE_SCHEDULING_CONTROLS_FLAG_BITS_ARM__MAX_ENUM = 2147483647
VK_VENDOR_ID_VIV = 0x10001
VK_VENDOR_ID_VSI = 0x10002
VK_VENDOR_ID_KAZAN = 0x10003
VK_VENDOR_ID_CODEPLAY = 0x10004
VK_VENDOR_ID_MESA = 0x10005
VK_VENDOR_ID_POCL = 0x10006
VK_VENDOR_ID_MOBILEYE = 0x10007
VK_VENDOR_ID__BEGIN_RANGE = 65537
VK_VENDOR_ID__END_RANGE = 65543
VK_VENDOR_ID__RANGE_SIZE = 7
VK_VENDOR_ID__MAX_ENUM = 2147483647
VK_DRIVER_ID_AMD_PROPRIETARY = 1
VK_DRIVER_ID_AMD_OPEN_SOURCE = 2
VK_DRIVER_ID_MESA_RADV = 3
VK_DRIVER_ID_NVIDIA_PROPRIETARY = 4
VK_DRIVER_ID_INTEL_PROPRIETARY_WINDOWS = 5
VK_DRIVER_ID_INTEL_OPEN_SOURCE_MESA = 6
VK_DRIVER_ID_IMAGINATION_PROPRIETARY = 7
VK_DRIVER_ID_QUALCOMM_PROPRIETARY = 8
VK_DRIVER_ID_ARM_PROPRIETARY = 9
VK_DRIVER_ID_GOOGLE_SWIFTSHADER = 10
VK_DRIVER_ID_GGP_PROPRIETARY = 11
VK_DRIVER_ID_BROADCOM_PROPRIETARY = 12
VK_DRIVER_ID_MESA_LLVMPIPE = 13
VK_DRIVER_ID_MOLTENVK = 14
VK_DRIVER_ID_COREAVI_PROPRIETARY = 15
VK_DRIVER_ID_JUICE_PROPRIETARY = 16
VK_DRIVER_ID_VERISILICON_PROPRIETARY = 17
VK_DRIVER_ID_MESA_TURNIP = 18
VK_DRIVER_ID_MESA_V3DV = 19
VK_DRIVER_ID_MESA_PANVK = 20
VK_DRIVER_ID_SAMSUNG_PROPRIETARY = 21
VK_DRIVER_ID_MESA_VENUS = 22
VK_DRIVER_ID_MESA_DOZEN = 23
VK_DRIVER_ID_MESA_NVK = 24
VK_DRIVER_ID_IMAGINATION_OPEN_SOURCE_MESA = 25
VK_DRIVER_ID_MESA_AGXV = 26
VK_DRIVER_ID__BEGIN_RANGE = 1
VK_DRIVER_ID__END_RANGE = 26
VK_DRIVER_ID__RANGE_SIZE = 26
VK_DRIVER_ID__MAX_ENUM = 2147483647
VK_SHADING_RATE_PALETTE_ENTRY_NO_INVOCATIONS_NV = 0
VK_SHADING_RATE_PALETTE_ENTRY_16_INVOCATIONS_PER_PIXEL_NV = 1
VK_SHADING_RATE_PALETTE_ENTRY_8_INVOCATIONS_PER_PIXEL_NV = 2
VK_SHADING_RATE_PALETTE_ENTRY_4_INVOCATIONS_PER_PIXEL_NV = 3
VK_SHADING_RATE_PALETTE_ENTRY_2_INVOCATIONS_PER_PIXEL_NV = 4
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_PIXEL_NV = 5
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_2X1_PIXELS_NV = 6
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_1X2_PIXELS_NV = 7
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_2X2_PIXELS_NV = 8
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_4X2_PIXELS_NV = 9
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_2X4_PIXELS_NV = 10
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_4X4_PIXELS_NV = 11
VK_SHADING_RATE_PALETTE_ENTRY_NV__BEGIN_RANGE = 0
VK_SHADING_RATE_PALETTE_ENTRY_NV__END_RANGE = 11
VK_SHADING_RATE_PALETTE_ENTRY_NV__RANGE_SIZE = 12
VK_SHADING_RATE_PALETTE_ENTRY_NV__MAX_ENUM = 2147483647
VK_COARSE_SAMPLE_ORDER_TYPE_DEFAULT_NV = 0
VK_COARSE_SAMPLE_ORDER_TYPE_CUSTOM_NV = 1
VK_COARSE_SAMPLE_ORDER_TYPE_PIXEL_MAJOR_NV = 2
VK_COARSE_SAMPLE_ORDER_TYPE_SAMPLE_MAJOR_NV = 3
VK_COARSE_SAMPLE_ORDER_TYPE_NV__BEGIN_RANGE = 0
VK_COARSE_SAMPLE_ORDER_TYPE_NV__END_RANGE = 3
VK_COARSE_SAMPLE_ORDER_TYPE_NV__RANGE_SIZE = 4
VK_COARSE_SAMPLE_ORDER_TYPE_NV__MAX_ENUM = 2147483647
VK_PIPELINE_EXECUTABLE_STATISTIC_FORMAT_BOOL32_KHR = 0
VK_PIPELINE_EXECUTABLE_STATISTIC_FORMAT_INT64_KHR = 1
VK_PIPELINE_EXECUTABLE_STATISTIC_FORMAT_UINT64_KHR = 2
VK_PIPELINE_EXECUTABLE_STATISTIC_FORMAT_FLOAT64_KHR = 3
VK_PIPELINE_EXECUTABLE_STATISTIC_FORMAT_KHR__BEGIN_RANGE = 0
VK_PIPELINE_EXECUTABLE_STATISTIC_FORMAT_KHR__END_RANGE = 3
VK_PIPELINE_EXECUTABLE_STATISTIC_FORMAT_KHR__RANGE_SIZE = 4
VK_PIPELINE_EXECUTABLE_STATISTIC_FORMAT_KHR__MAX_ENUM = 2147483647
VK_VIDEO_CODEC_OPERATION_NONE_KHR = 0
VK_VIDEO_CODEC_OPERATION_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_CHROMA_SUBSAMPLING_INVALID_KHR = 0
VK_VIDEO_CHROMA_SUBSAMPLING_MONOCHROME_BIT_KHR = 0x00000001
VK_VIDEO_CHROMA_SUBSAMPLING_420_BIT_KHR = 0x00000002
VK_VIDEO_CHROMA_SUBSAMPLING_422_BIT_KHR = 0x00000004
VK_VIDEO_CHROMA_SUBSAMPLING_444_BIT_KHR = 0x00000008
VK_VIDEO_CHROMA_SUBSAMPLING_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_COMPONENT_BIT_DEPTH_INVALID_KHR = 0
VK_VIDEO_COMPONENT_BIT_DEPTH_8_BIT_KHR = 0x00000001
VK_VIDEO_COMPONENT_BIT_DEPTH_10_BIT_KHR = 0x00000004
VK_VIDEO_COMPONENT_BIT_DEPTH_12_BIT_KHR = 0x00000010
VK_VIDEO_COMPONENT_BIT_DEPTH_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_CAPABILITY_PROTECTED_CONTENT_BIT_KHR = 0x00000001
VK_VIDEO_CAPABILITY_SEPARATE_REFERENCE_IMAGES_BIT_KHR = 0x00000002
VK_VIDEO_CAPABILITY_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_SESSION_CREATE_PROTECTED_CONTENT_BIT_KHR = 0x00000001
VK_VIDEO_SESSION_CREATE_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_CODING_CONTROL_RESET_BIT_KHR = 0x00000001
VK_VIDEO_CODING_CONTROL_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_QUERY_RESULT_STATUS_ERROR_KHR = -1
VK_QUERY_RESULT_STATUS_NOT_READY_KHR = 0
VK_QUERY_RESULT_STATUS_COMPLETE_KHR = 1
VK_QUERY_RESULT_STATUS_KHR__BEGIN_RANGE = -1
VK_QUERY_RESULT_STATUS_KHR__END_RANGE = 1
VK_QUERY_RESULT_STATUS_KHR__RANGE_SIZE = 3
VK_QUERY_RESULT_STATUS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_DECODE_USAGE_DEFAULT_KHR = 0
VK_VIDEO_DECODE_USAGE_TRANSCODING_BIT_KHR = 0x00000001
VK_VIDEO_DECODE_USAGE_OFFLINE_BIT_KHR = 0x00000002
VK_VIDEO_DECODE_USAGE_STREAMING_BIT_KHR = 0x00000004
VK_VIDEO_DECODE_USAGE_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_DECODE_CAPABILITY_DPB_AND_OUTPUT_COINCIDE_BIT_KHR = 0x00000001
VK_VIDEO_DECODE_CAPABILITY_DPB_AND_OUTPUT_DISTINCT_BIT_KHR = 0x00000002
VK_VIDEO_DECODE_CAPABILITY_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_DECODE_H264_PICTURE_LAYOUT_PROGRESSIVE_KHR = 0
VK_VIDEO_DECODE_H264_PICTURE_LAYOUT_INTERLACED_INTERLEAVED_LINES_BIT_KHR = 0x00000001
VK_VIDEO_DECODE_H264_PICTURE_LAYOUT_INTERLACED_SEPARATE_PLANES_BIT_KHR = 0x00000002
VK_VIDEO_DECODE_H264_PICTURE_LAYOUT_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_USAGE_DEFAULT_KHR = 0
VK_VIDEO_ENCODE_USAGE_TRANSCODING_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_USAGE_STREAMING_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_USAGE_RECORDING_BIT_KHR = 0x00000004
VK_VIDEO_ENCODE_USAGE_CONFERENCING_BIT_KHR = 0x00000008
VK_VIDEO_ENCODE_USAGE_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_CONTENT_DEFAULT_KHR = 0
VK_VIDEO_ENCODE_CONTENT_CAMERA_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_CONTENT_DESKTOP_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_CONTENT_RENDERED_BIT_KHR = 0x00000004
VK_VIDEO_ENCODE_CONTENT_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_TUNING_MODE_DEFAULT_KHR = 0
VK_VIDEO_ENCODE_TUNING_MODE_HIGH_QUALITY_KHR = 1
VK_VIDEO_ENCODE_TUNING_MODE_LOW_LATENCY_KHR = 2
VK_VIDEO_ENCODE_TUNING_MODE_ULTRA_LOW_LATENCY_KHR = 3
VK_VIDEO_ENCODE_TUNING_MODE_LOSSLESS_KHR = 4
VK_VIDEO_ENCODE_TUNING_MODE_KHR__BEGIN_RANGE = 0
VK_VIDEO_ENCODE_TUNING_MODE_KHR__END_RANGE = 4
VK_VIDEO_ENCODE_TUNING_MODE_KHR__RANGE_SIZE = 5
VK_VIDEO_ENCODE_TUNING_MODE_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_CAPABILITY_PRECEDING_EXTERNALLY_ENCODED_BYTES_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_CAPABILITY_INSUFFICIENT_BITSTREAM_BUFFER_RANGE_DETECTION_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_CAPABILITY_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_FEEDBACK_BITSTREAM_BUFFER_OFFSET_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_FEEDBACK_BITSTREAM_BYTES_WRITTEN_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_FEEDBACK_BITSTREAM_HAS_OVERRIDES_BIT_KHR = 0x00000004
VK_VIDEO_ENCODE_FEEDBACK_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_RATE_CONTROL_MODE_DEFAULT_KHR = 0
VK_VIDEO_ENCODE_RATE_CONTROL_MODE_DISABLED_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_RATE_CONTROL_MODE_CBR_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_RATE_CONTROL_MODE_VBR_BIT_KHR = 0x00000004
VK_VIDEO_ENCODE_RATE_CONTROL_MODE_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_H264_CAPABILITY_HRD_COMPLIANCE_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_H264_CAPABILITY_PREDICTION_WEIGHT_TABLE_GENERATED_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_H264_CAPABILITY_ROW_UNALIGNED_SLICE_BIT_KHR = 0x00000004
VK_VIDEO_ENCODE_H264_CAPABILITY_DIFFERENT_SLICE_TYPE_BIT_KHR = 0x00000008
VK_VIDEO_ENCODE_H264_CAPABILITY_B_FRAME_IN_L0_LIST_BIT_KHR = 0x00000010
VK_VIDEO_ENCODE_H264_CAPABILITY_B_FRAME_IN_L1_LIST_BIT_KHR = 0x00000020
VK_VIDEO_ENCODE_H264_CAPABILITY_PER_PICTURE_TYPE_MIN_MAX_QP_BIT_KHR = 0x00000040
VK_VIDEO_ENCODE_H264_CAPABILITY_PER_SLICE_CONSTANT_QP_BIT_KHR = 0x00000080
VK_VIDEO_ENCODE_H264_CAPABILITY_GENERATE_PREFIX_NALU_BIT_KHR = 0x00000100
VK_VIDEO_ENCODE_H264_CAPABILITY_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_H264_STD_SEPARATE_COLOR_PLANE_FLAG_SET_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_H264_STD_QPPRIME_Y_ZERO_TRANSFORM_BYPASS_FLAG_SET_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_H264_STD_SCALING_MATRIX_PRESENT_FLAG_SET_BIT_KHR = 0x00000004
VK_VIDEO_ENCODE_H264_STD_CHROMA_QP_INDEX_OFFSET_BIT_KHR = 0x00000008
VK_VIDEO_ENCODE_H264_STD_SECOND_CHROMA_QP_INDEX_OFFSET_BIT_KHR = 0x00000010
VK_VIDEO_ENCODE_H264_STD_PIC_INIT_QP_MINUS26_BIT_KHR = 0x00000020
VK_VIDEO_ENCODE_H264_STD_WEIGHTED_PRED_FLAG_SET_BIT_KHR = 0x00000040
VK_VIDEO_ENCODE_H264_STD_WEIGHTED_BIPRED_IDC_EXPLICIT_BIT_KHR = 0x00000080
VK_VIDEO_ENCODE_H264_STD_WEIGHTED_BIPRED_IDC_IMPLICIT_BIT_KHR = 0x00000100
VK_VIDEO_ENCODE_H264_STD_TRANSFORM_8X8_MODE_FLAG_SET_BIT_KHR = 0x00000200
VK_VIDEO_ENCODE_H264_STD_DIRECT_SPATIAL_MV_PRED_FLAG_UNSET_BIT_KHR = 0x00000400
VK_VIDEO_ENCODE_H264_STD_ENTROPY_CODING_MODE_FLAG_UNSET_BIT_KHR = 0x00000800
VK_VIDEO_ENCODE_H264_STD_ENTROPY_CODING_MODE_FLAG_SET_BIT_KHR = 0x00001000
VK_VIDEO_ENCODE_H264_STD_DIRECT_8X8_INFERENCE_FLAG_UNSET_BIT_KHR = 0x00002000
VK_VIDEO_ENCODE_H264_STD_CONSTRAINED_INTRA_PRED_FLAG_SET_BIT_KHR = 0x00004000
VK_VIDEO_ENCODE_H264_STD_DEBLOCKING_FILTER_DISABLED_BIT_KHR = 0x00008000
VK_VIDEO_ENCODE_H264_STD_DEBLOCKING_FILTER_ENABLED_BIT_KHR = 0x00010000
VK_VIDEO_ENCODE_H264_STD_DEBLOCKING_FILTER_PARTIAL_BIT_KHR = 0x00020000
VK_VIDEO_ENCODE_H264_STD_SLICE_QP_DELTA_BIT_KHR = 0x00080000
VK_VIDEO_ENCODE_H264_STD_DIFFERENT_SLICE_QP_DELTA_BIT_KHR = 0x00100000
VK_VIDEO_ENCODE_H264_STD_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_H264_RATE_CONTROL_ATTEMPT_HRD_COMPLIANCE_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_H264_RATE_CONTROL_REGULAR_GOP_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_H264_RATE_CONTROL_REFERENCE_PATTERN_FLAT_BIT_KHR = 0x00000004
VK_VIDEO_ENCODE_H264_RATE_CONTROL_REFERENCE_PATTERN_DYADIC_BIT_KHR = 0x00000008
VK_VIDEO_ENCODE_H264_RATE_CONTROL_TEMPORAL_LAYER_PATTERN_DYADIC_BIT_KHR = 0x00000010
VK_VIDEO_ENCODE_H264_RATE_CONTROL_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_H265_CAPABILITY_HRD_COMPLIANCE_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_H265_CAPABILITY_PREDICTION_WEIGHT_TABLE_GENERATED_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_H265_CAPABILITY_ROW_UNALIGNED_SLICE_SEGMENT_BIT_KHR = 0x00000004
VK_VIDEO_ENCODE_H265_CAPABILITY_DIFFERENT_SLICE_SEGMENT_TYPE_BIT_KHR = 0x00000008
VK_VIDEO_ENCODE_H265_CAPABILITY_B_FRAME_IN_L0_LIST_BIT_KHR = 0x00000010
VK_VIDEO_ENCODE_H265_CAPABILITY_B_FRAME_IN_L1_LIST_BIT_KHR = 0x00000020
VK_VIDEO_ENCODE_H265_CAPABILITY_PER_PICTURE_TYPE_MIN_MAX_QP_BIT_KHR = 0x00000040
VK_VIDEO_ENCODE_H265_CAPABILITY_PER_SLICE_SEGMENT_CONSTANT_QP_BIT_KHR = 0x00000080
VK_VIDEO_ENCODE_H265_CAPABILITY_MULTIPLE_TILES_PER_SLICE_SEGMENT_BIT_KHR = 0x00000100
VK_VIDEO_ENCODE_H265_CAPABILITY_MULTIPLE_SLICE_SEGMENTS_PER_TILE_BIT_KHR = 0x00000200
VK_VIDEO_ENCODE_H265_CAPABILITY_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_H265_STD_SEPARATE_COLOR_PLANE_FLAG_SET_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_H265_STD_SAMPLE_ADAPTIVE_OFFSET_ENABLED_FLAG_SET_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_H265_STD_SCALING_LIST_DATA_PRESENT_FLAG_SET_BIT_KHR = 0x00000004
VK_VIDEO_ENCODE_H265_STD_PCM_ENABLED_FLAG_SET_BIT_KHR = 0x00000008
VK_VIDEO_ENCODE_H265_STD_SPS_TEMPORAL_MVP_ENABLED_FLAG_SET_BIT_KHR = 0x00000010
VK_VIDEO_ENCODE_H265_STD_INIT_QP_MINUS26_BIT_KHR = 0x00000020
VK_VIDEO_ENCODE_H265_STD_WEIGHTED_PRED_FLAG_SET_BIT_KHR = 0x00000040
VK_VIDEO_ENCODE_H265_STD_WEIGHTED_BIPRED_FLAG_SET_BIT_KHR = 0x00000080
VK_VIDEO_ENCODE_H265_STD_LOG2_PARALLEL_MERGE_LEVEL_MINUS2_BIT_KHR = 0x00000100
VK_VIDEO_ENCODE_H265_STD_SIGN_DATA_HIDING_ENABLED_FLAG_SET_BIT_KHR = 0x00000200
VK_VIDEO_ENCODE_H265_STD_TRANSFORM_SKIP_ENABLED_FLAG_SET_BIT_KHR = 0x00000400
VK_VIDEO_ENCODE_H265_STD_TRANSFORM_SKIP_ENABLED_FLAG_UNSET_BIT_KHR = 0x00000800
VK_VIDEO_ENCODE_H265_STD_PPS_SLICE_CHROMA_QP_OFFSETS_PRESENT_FLAG_SET_BIT_KHR = 0x00001000
VK_VIDEO_ENCODE_H265_STD_TRANSQUANT_BYPASS_ENABLED_FLAG_SET_BIT_KHR = 0x00002000
VK_VIDEO_ENCODE_H265_STD_CONSTRAINED_INTRA_PRED_FLAG_SET_BIT_KHR = 0x00004000
VK_VIDEO_ENCODE_H265_STD_ENTROPY_CODING_SYNC_ENABLED_FLAG_SET_BIT_KHR = 0x00008000
VK_VIDEO_ENCODE_H265_STD_DEBLOCKING_FILTER_OVERRIDE_ENABLED_FLAG_SET_BIT_KHR = 0x00010000
VK_VIDEO_ENCODE_H265_STD_DEPENDENT_SLICE_SEGMENTS_ENABLED_FLAG_SET_BIT_KHR = 0x00020000
VK_VIDEO_ENCODE_H265_STD_DEPENDENT_SLICE_SEGMENT_FLAG_SET_BIT_KHR = 0x00040000
VK_VIDEO_ENCODE_H265_STD_SLICE_QP_DELTA_BIT_KHR = 0x00080000
VK_VIDEO_ENCODE_H265_STD_DIFFERENT_SLICE_QP_DELTA_BIT_KHR = 0x00100000
VK_VIDEO_ENCODE_H265_STD_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_H265_RATE_CONTROL_ATTEMPT_HRD_COMPLIANCE_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_H265_RATE_CONTROL_REGULAR_GOP_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_H265_RATE_CONTROL_REFERENCE_PATTERN_FLAT_BIT_KHR = 0x00000004
VK_VIDEO_ENCODE_H265_RATE_CONTROL_REFERENCE_PATTERN_DYADIC_BIT_KHR = 0x00000008
VK_VIDEO_ENCODE_H265_RATE_CONTROL_TEMPORAL_SUB_LAYER_PATTERN_DYADIC_BIT_KHR = 0x00000010
VK_VIDEO_ENCODE_H265_RATE_CONTROL_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_H265_CTB_SIZE_16_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_H265_CTB_SIZE_32_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_H265_CTB_SIZE_64_BIT_KHR = 0x00000004
VK_VIDEO_ENCODE_H265_CTB_SIZE_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_VIDEO_ENCODE_H265_TRANSFORM_BLOCK_SIZE_4_BIT_KHR = 0x00000001
VK_VIDEO_ENCODE_H265_TRANSFORM_BLOCK_SIZE_8_BIT_KHR = 0x00000002
VK_VIDEO_ENCODE_H265_TRANSFORM_BLOCK_SIZE_16_BIT_KHR = 0x00000004
VK_VIDEO_ENCODE_H265_TRANSFORM_BLOCK_SIZE_32_BIT_KHR = 0x00000008
VK_VIDEO_ENCODE_H265_TRANSFORM_BLOCK_SIZE_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_ACCELERATION_STRUCTURE_MOTION_INSTANCE_TYPE_STATIC_NV = 0
VK_ACCELERATION_STRUCTURE_MOTION_INSTANCE_TYPE_MATRIX_MOTION_NV = 1
VK_ACCELERATION_STRUCTURE_MOTION_INSTANCE_TYPE_SRT_MOTION_NV = 2
VK_ACCELERATION_STRUCTURE_MOTION_INSTANCE_TYPE_NV__BEGIN_RANGE = 0
VK_ACCELERATION_STRUCTURE_MOTION_INSTANCE_TYPE_NV__END_RANGE = 2
VK_ACCELERATION_STRUCTURE_MOTION_INSTANCE_TYPE_NV__RANGE_SIZE = 3
VK_ACCELERATION_STRUCTURE_MOTION_INSTANCE_TYPE_NV__MAX_ENUM = 2147483647

def VK_MAKE_VERSION(major, minor, patch):
    return (((major) << 22) | ((minor) << 12) | (patch))


def VK_VERSION_MAJOR(version):
    return version >> 22


def VK_VERSION_MINOR(version):
    return (version >> 12) & 0x3ff


def VK_VERSION_PATCH(version):
    return version & 0xfff


VK_API_VERSION = VK_MAKE_VERSION(1, 0, 0)
VK_API_VERSION_1_0 = VK_MAKE_VERSION(1, 0, 0)
VK_NULL_HANDLE = 0
_UINT64_MAX = ffi.new('unsigned long long int*', 18446744073709551615)
UINT64_MAX = _UINT64_MAX[0]


VK_MAX_PHYSICAL_DEVICE_NAME_SIZE = 256
VK_UUID_SIZE = 16
VK_LUID_SIZE = 8
VK_MAX_EXTENSION_NAME_SIZE = 256
VK_MAX_DESCRIPTION_SIZE = 256
VK_MAX_MEMORY_TYPES = 32
VK_MAX_MEMORY_HEAPS = 16
VK_LOD_CLAMP_NONE = 1000.0
VK_REMAINING_MIP_LEVELS = 4294967295
VK_REMAINING_ARRAY_LAYERS = 4294967295
VK_REMAINING_3D_SLICES_EXT = 4294967295
VK_WHOLE_SIZE = -1
VK_ATTACHMENT_UNUSED = 4294967295
VK_TRUE = 1
VK_FALSE = 0
VK_QUEUE_FAMILY_IGNORED = 4294967295
VK_QUEUE_FAMILY_EXTERNAL = 4294967294
VK_QUEUE_FAMILY_FOREIGN_EXT = 4294967293
VK_SUBPASS_EXTERNAL = 4294967295
VK_MAX_DEVICE_GROUP_SIZE = 32
VK_MAX_DRIVER_NAME_SIZE = 256
VK_MAX_DRIVER_INFO_SIZE = 256
VK_SHADER_UNUSED_KHR = 4294967295
VK_MAX_GLOBAL_PRIORITY_SIZE_KHR = 16
VK_MAX_SHADER_MODULE_IDENTIFIER_SIZE_EXT = 32
VK_SHADER_INDEX_UNUSED_AMDX = 4294967295
VK_KHR_surface = 1
VK_KHR_SURFACE_SPEC_VERSION = 25
VK_KHR_SURFACE_EXTENSION_NAME = "VK_KHR_surface"
VK_OBJECT_TYPE_SURFACE_KHR = 1000000000
VK_KHR_swapchain = 1
VK_KHR_SWAPCHAIN_SPEC_VERSION = 70
VK_KHR_SWAPCHAIN_EXTENSION_NAME = "VK_KHR_swapchain"
VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR = 1000001000
VK_STRUCTURE_TYPE_PRESENT_INFO_KHR = 1000001001
VK_IMAGE_LAYOUT_PRESENT_SRC_KHR = 1000001002
VK_OBJECT_TYPE_SWAPCHAIN_KHR = 1000001000
VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_CAPABILITIES_KHR = 1000001007
VK_STRUCTURE_TYPE_IMAGE_SWAPCHAIN_CREATE_INFO_KHR = 1000001008
VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_SWAPCHAIN_INFO_KHR = 1000001009
VK_STRUCTURE_TYPE_ACQUIRE_NEXT_IMAGE_INFO_KHR = 1000001010
VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_INFO_KHR = 1000001011
VK_STRUCTURE_TYPE_DEVICE_GROUP_SWAPCHAIN_CREATE_INFO_KHR = 1000001012
VK_SWAPCHAIN_CREATE_SPLIT_INSTANCE_BIND_REGIONS_BIT_KHR = 0x00000001
VK_SWAPCHAIN_CREATE_PROTECTED_BIT_KHR = 0x00000002
VK_KHR_display = 1
VK_KHR_DISPLAY_SPEC_VERSION = 23
VK_KHR_DISPLAY_EXTENSION_NAME = "VK_KHR_display"
VK_STRUCTURE_TYPE_DISPLAY_MODE_CREATE_INFO_KHR = 1000002000
VK_STRUCTURE_TYPE_DISPLAY_SURFACE_CREATE_INFO_KHR = 1000002001
VK_OBJECT_TYPE_DISPLAY_KHR = 1000002000
VK_OBJECT_TYPE_DISPLAY_MODE_KHR = 1000002001
VK_KHR_display_swapchain = 1
VK_KHR_DISPLAY_SWAPCHAIN_SPEC_VERSION = 10
VK_KHR_DISPLAY_SWAPCHAIN_EXTENSION_NAME = "VK_KHR_display_swapchain"
VK_STRUCTURE_TYPE_DISPLAY_PRESENT_INFO_KHR = 1000003000
VK_KHR_xlib_surface = 1
VK_KHR_XLIB_SURFACE_SPEC_VERSION = 6
VK_KHR_XLIB_SURFACE_EXTENSION_NAME = "VK_KHR_xlib_surface"
VK_STRUCTURE_TYPE_XLIB_SURFACE_CREATE_INFO_KHR = 1000004000
VK_KHR_xcb_surface = 1
VK_KHR_XCB_SURFACE_SPEC_VERSION = 6
VK_KHR_XCB_SURFACE_EXTENSION_NAME = "VK_KHR_xcb_surface"
VK_STRUCTURE_TYPE_XCB_SURFACE_CREATE_INFO_KHR = 1000005000
VK_KHR_wayland_surface = 1
VK_KHR_WAYLAND_SURFACE_SPEC_VERSION = 6
VK_KHR_WAYLAND_SURFACE_EXTENSION_NAME = "VK_KHR_wayland_surface"
VK_STRUCTURE_TYPE_WAYLAND_SURFACE_CREATE_INFO_KHR = 1000006000
VK_KHR_mir_surface = 1
VK_KHR_MIR_SURFACE_SPEC_VERSION = 4
VK_KHR_MIR_SURFACE_EXTENSION_NAME = "VK_KHR_mir_surface"
VK_KHR_android_surface = 1
VK_KHR_ANDROID_SURFACE_SPEC_VERSION = 6
VK_KHR_ANDROID_SURFACE_EXTENSION_NAME = "VK_KHR_android_surface"
VK_STRUCTURE_TYPE_ANDROID_SURFACE_CREATE_INFO_KHR = 1000008000
VK_KHR_win32_surface = 1
VK_KHR_WIN32_SURFACE_SPEC_VERSION = 6
VK_KHR_WIN32_SURFACE_EXTENSION_NAME = "VK_KHR_win32_surface"
VK_STRUCTURE_TYPE_WIN32_SURFACE_CREATE_INFO_KHR = 1000009000
VK_ANDROID_native_buffer = 1
VK_ANDROID_NATIVE_BUFFER_SPEC_VERSION = 8
VK_ANDROID_NATIVE_BUFFER_NUMBER = 11
VK_ANDROID_NATIVE_BUFFER_EXTENSION_NAME = "VK_ANDROID_native_buffer"
VK_ANDROID_NATIVE_BUFFER_NAME = None
VK_STRUCTURE_TYPE_NATIVE_BUFFER_ANDROID = 1000010000
VK_STRUCTURE_TYPE_SWAPCHAIN_IMAGE_CREATE_INFO_ANDROID = 1000010001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRESENTATION_PROPERTIES_ANDROID = 1000010002
VK_EXT_debug_report = 1
VK_EXT_DEBUG_REPORT_SPEC_VERSION = 10
VK_EXT_DEBUG_REPORT_EXTENSION_NAME = "VK_EXT_debug_report"
VK_STRUCTURE_TYPE_DEBUG_REPORT_CALLBACK_CREATE_INFO_EXT = 1000011000
VK_STRUCTURE_TYPE_DEBUG_REPORT_CREATE_INFO_EXT = None
VK_OBJECT_TYPE_DEBUG_REPORT_CALLBACK_EXT = 1000011000
VK_DEBUG_REPORT_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION_EXT = 1000011000
VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_EXT = 1000011000
VK_NV_glsl_shader = 1
VK_NV_GLSL_SHADER_SPEC_VERSION = 1
VK_NV_GLSL_SHADER_EXTENSION_NAME = "VK_NV_glsl_shader"
VK_EXT_depth_range_unrestricted = 1
VK_EXT_DEPTH_RANGE_UNRESTRICTED_SPEC_VERSION = 1
VK_EXT_DEPTH_RANGE_UNRESTRICTED_EXTENSION_NAME = "VK_EXT_depth_range_unrestricted"
VK_KHR_sampler_mirror_clamp_to_edge = 1
VK_KHR_SAMPLER_MIRROR_CLAMP_TO_EDGE_SPEC_VERSION = 3
VK_KHR_SAMPLER_MIRROR_CLAMP_TO_EDGE_EXTENSION_NAME = "VK_KHR_sampler_mirror_clamp_to_edge"
VK_SAMPLER_ADDRESS_MODE_MIRROR_CLAMP_TO_EDGE = 4
VK_SAMPLER_ADDRESS_MODE_MIRROR_CLAMP_TO_EDGE_KHR = None
VK_IMG_filter_cubic = 1
VK_IMG_FILTER_CUBIC_SPEC_VERSION = 1
VK_IMG_FILTER_CUBIC_EXTENSION_NAME = "VK_IMG_filter_cubic"
VK_FILTER_CUBIC_IMG = None
VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_CUBIC_BIT_IMG = None
VK_AMD_extension_17 = 1
VK_AMD_EXTENSION_17_SPEC_VERSION = 0
VK_AMD_EXTENSION_17_EXTENSION_NAME = "VK_AMD_extension_17"
VK_AMD_extension_18 = 1
VK_AMD_EXTENSION_18_SPEC_VERSION = 0
VK_AMD_EXTENSION_18_EXTENSION_NAME = "VK_AMD_extension_18"
VK_AMD_rasterization_order = 1
VK_AMD_RASTERIZATION_ORDER_SPEC_VERSION = 1
VK_AMD_RASTERIZATION_ORDER_EXTENSION_NAME = "VK_AMD_rasterization_order"
VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_RASTERIZATION_ORDER_AMD = 1000018000
VK_AMD_extension_20 = 1
VK_AMD_EXTENSION_20_SPEC_VERSION = 0
VK_AMD_EXTENSION_20_EXTENSION_NAME = "VK_AMD_extension_20"
VK_AMD_shader_trinary_minmax = 1
VK_AMD_SHADER_TRINARY_MINMAX_SPEC_VERSION = 1
VK_AMD_SHADER_TRINARY_MINMAX_EXTENSION_NAME = "VK_AMD_shader_trinary_minmax"
VK_AMD_shader_explicit_vertex_parameter = 1
VK_AMD_SHADER_EXPLICIT_VERTEX_PARAMETER_SPEC_VERSION = 1
VK_AMD_SHADER_EXPLICIT_VERTEX_PARAMETER_EXTENSION_NAME = "VK_AMD_shader_explicit_vertex_parameter"
VK_EXT_debug_marker = 1
VK_EXT_DEBUG_MARKER_SPEC_VERSION = 4
VK_EXT_DEBUG_MARKER_EXTENSION_NAME = "VK_EXT_debug_marker"
VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_NAME_INFO_EXT = 1000022000
VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_TAG_INFO_EXT = 1000022001
VK_STRUCTURE_TYPE_DEBUG_MARKER_MARKER_INFO_EXT = 1000022002
VK_KHR_video_queue = 1
VK_KHR_VIDEO_QUEUE_SPEC_VERSION = 8
VK_KHR_VIDEO_QUEUE_EXTENSION_NAME = "VK_KHR_video_queue"
VK_STRUCTURE_TYPE_VIDEO_PROFILE_INFO_KHR = 1000023000
VK_STRUCTURE_TYPE_VIDEO_CAPABILITIES_KHR = 1000023001
VK_STRUCTURE_TYPE_VIDEO_PICTURE_RESOURCE_INFO_KHR = 1000023002
VK_STRUCTURE_TYPE_VIDEO_SESSION_MEMORY_REQUIREMENTS_KHR = 1000023003
VK_STRUCTURE_TYPE_BIND_VIDEO_SESSION_MEMORY_INFO_KHR = 1000023004
VK_STRUCTURE_TYPE_VIDEO_SESSION_CREATE_INFO_KHR = 1000023005
VK_STRUCTURE_TYPE_VIDEO_SESSION_PARAMETERS_CREATE_INFO_KHR = 1000023006
VK_STRUCTURE_TYPE_VIDEO_SESSION_PARAMETERS_UPDATE_INFO_KHR = 1000023007
VK_STRUCTURE_TYPE_VIDEO_BEGIN_CODING_INFO_KHR = 1000023008
VK_STRUCTURE_TYPE_VIDEO_END_CODING_INFO_KHR = 1000023009
VK_STRUCTURE_TYPE_VIDEO_CODING_CONTROL_INFO_KHR = 1000023010
VK_STRUCTURE_TYPE_VIDEO_REFERENCE_SLOT_INFO_KHR = 1000023011
VK_STRUCTURE_TYPE_QUEUE_FAMILY_VIDEO_PROPERTIES_KHR = 1000023012
VK_STRUCTURE_TYPE_VIDEO_PROFILE_LIST_INFO_KHR = 1000023013
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VIDEO_FORMAT_INFO_KHR = 1000023014
VK_STRUCTURE_TYPE_VIDEO_FORMAT_PROPERTIES_KHR = 1000023015
VK_STRUCTURE_TYPE_QUEUE_FAMILY_QUERY_RESULT_STATUS_PROPERTIES_KHR = 1000023016
VK_OBJECT_TYPE_VIDEO_SESSION_KHR = 1000023000
VK_OBJECT_TYPE_VIDEO_SESSION_PARAMETERS_KHR = 1000023001
VK_QUERY_TYPE_RESULT_STATUS_ONLY_KHR = 1000023000
VK_QUERY_RESULT_WITH_STATUS_BIT_KHR = 0x00000010
VK_KHR_video_decode_queue = 1
VK_KHR_VIDEO_DECODE_QUEUE_SPEC_VERSION = 8
VK_KHR_VIDEO_DECODE_QUEUE_EXTENSION_NAME = "VK_KHR_video_decode_queue"
VK_STRUCTURE_TYPE_VIDEO_DECODE_INFO_KHR = 1000024000
VK_STRUCTURE_TYPE_VIDEO_DECODE_CAPABILITIES_KHR = 1000024001
VK_STRUCTURE_TYPE_VIDEO_DECODE_USAGE_INFO_KHR = 1000024002
VK_QUEUE_VIDEO_DECODE_BIT_KHR = 0x00000020
VK_PIPELINE_STAGE_2_VIDEO_DECODE_BIT_KHR = 0x04000000
VK_ACCESS_2_VIDEO_DECODE_READ_BIT_KHR = 0x800000000
VK_ACCESS_2_VIDEO_DECODE_WRITE_BIT_KHR = 0x1000000000
VK_BUFFER_USAGE_VIDEO_DECODE_SRC_BIT_KHR = 0x00002000
VK_BUFFER_USAGE_VIDEO_DECODE_DST_BIT_KHR = 0x00004000
VK_IMAGE_USAGE_VIDEO_DECODE_DST_BIT_KHR = 0x00000400
VK_IMAGE_USAGE_VIDEO_DECODE_SRC_BIT_KHR = 0x00000800
VK_IMAGE_USAGE_VIDEO_DECODE_DPB_BIT_KHR = 0x00001000
VK_FORMAT_FEATURE_VIDEO_DECODE_OUTPUT_BIT_KHR = 0x02000000
VK_FORMAT_FEATURE_VIDEO_DECODE_DPB_BIT_KHR = 0x04000000
VK_IMAGE_LAYOUT_VIDEO_DECODE_DST_KHR = 1000024000
VK_IMAGE_LAYOUT_VIDEO_DECODE_SRC_KHR = 1000024001
VK_IMAGE_LAYOUT_VIDEO_DECODE_DPB_KHR = 1000024002
VK_FORMAT_FEATURE_2_VIDEO_DECODE_OUTPUT_BIT_KHR = 0x02000000
VK_FORMAT_FEATURE_2_VIDEO_DECODE_DPB_BIT_KHR = 0x04000000
VK_AMD_gcn_shader = 1
VK_AMD_GCN_SHADER_SPEC_VERSION = 1
VK_AMD_GCN_SHADER_EXTENSION_NAME = "VK_AMD_gcn_shader"
VK_NV_dedicated_allocation = 1
VK_NV_DEDICATED_ALLOCATION_SPEC_VERSION = 1
VK_NV_DEDICATED_ALLOCATION_EXTENSION_NAME = "VK_NV_dedicated_allocation"
VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_IMAGE_CREATE_INFO_NV = 1000026000
VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_BUFFER_CREATE_INFO_NV = 1000026001
VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_MEMORY_ALLOCATE_INFO_NV = 1000026002
VK_EXT_extension_28 = 1
VK_EXT_EXTENSION_28_SPEC_VERSION = 0
VK_EXT_EXTENSION_28_EXTENSION_NAME = "VK_EXT_extension_28"
VK_EXT_transform_feedback = 1
VK_EXT_TRANSFORM_FEEDBACK_SPEC_VERSION = 1
VK_EXT_TRANSFORM_FEEDBACK_EXTENSION_NAME = "VK_EXT_transform_feedback"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_FEATURES_EXT = 1000028000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_PROPERTIES_EXT = 1000028001
VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_STREAM_CREATE_INFO_EXT = 1000028002
VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT = 1000028004
VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_BUFFER_BIT_EXT = 0x00000800
VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_COUNTER_BUFFER_BIT_EXT = 0x00001000
VK_ACCESS_TRANSFORM_FEEDBACK_WRITE_BIT_EXT = 0x02000000
VK_ACCESS_TRANSFORM_FEEDBACK_COUNTER_READ_BIT_EXT = 0x04000000
VK_ACCESS_TRANSFORM_FEEDBACK_COUNTER_WRITE_BIT_EXT = 0x08000000
VK_PIPELINE_STAGE_TRANSFORM_FEEDBACK_BIT_EXT = 0x01000000
VK_NVX_binary_import = 1
VK_NVX_BINARY_IMPORT_SPEC_VERSION = 1
VK_NVX_BINARY_IMPORT_EXTENSION_NAME = "VK_NVX_binary_import"
VK_STRUCTURE_TYPE_CU_MODULE_CREATE_INFO_NVX = 1000029000
VK_STRUCTURE_TYPE_CU_FUNCTION_CREATE_INFO_NVX = 1000029001
VK_STRUCTURE_TYPE_CU_LAUNCH_INFO_NVX = 1000029002
VK_OBJECT_TYPE_CU_MODULE_NVX = 1000029000
VK_OBJECT_TYPE_CU_FUNCTION_NVX = 1000029001
VK_DEBUG_REPORT_OBJECT_TYPE_CU_MODULE_NVX_EXT = 1000029000
VK_DEBUG_REPORT_OBJECT_TYPE_CU_FUNCTION_NVX_EXT = 1000029001
VK_NVX_image_view_handle = 1
VK_NVX_IMAGE_VIEW_HANDLE_SPEC_VERSION = 2
VK_NVX_IMAGE_VIEW_HANDLE_EXTENSION_NAME = "VK_NVX_image_view_handle"
VK_STRUCTURE_TYPE_IMAGE_VIEW_HANDLE_INFO_NVX = 1000030000
VK_STRUCTURE_TYPE_IMAGE_VIEW_ADDRESS_PROPERTIES_NVX = 1000030001
VK_AMD_extension_32 = 1
VK_AMD_EXTENSION_32_SPEC_VERSION = 0
VK_AMD_EXTENSION_32_EXTENSION_NAME = "VK_AMD_extension_32"
VK_AMD_extension_33 = 1
VK_AMD_EXTENSION_33_SPEC_VERSION = 0
VK_AMD_EXTENSION_33_EXTENSION_NAME = "VK_AMD_extension_33"
VK_AMD_draw_indirect_count = 1
VK_AMD_DRAW_INDIRECT_COUNT_SPEC_VERSION = 2
VK_AMD_DRAW_INDIRECT_COUNT_EXTENSION_NAME = "VK_AMD_draw_indirect_count"
VK_AMD_extension_35 = 1
VK_AMD_EXTENSION_35_SPEC_VERSION = 0
VK_AMD_EXTENSION_35_EXTENSION_NAME = "VK_AMD_extension_35"
VK_AMD_negative_viewport_height = 1
VK_AMD_NEGATIVE_VIEWPORT_HEIGHT_SPEC_VERSION = 1
VK_AMD_NEGATIVE_VIEWPORT_HEIGHT_EXTENSION_NAME = "VK_AMD_negative_viewport_height"
VK_AMD_gpu_shader_half_float = 1
VK_AMD_GPU_SHADER_HALF_FLOAT_SPEC_VERSION = 2
VK_AMD_GPU_SHADER_HALF_FLOAT_EXTENSION_NAME = "VK_AMD_gpu_shader_half_float"
VK_AMD_shader_ballot = 1
VK_AMD_SHADER_BALLOT_SPEC_VERSION = 1
VK_AMD_SHADER_BALLOT_EXTENSION_NAME = "VK_AMD_shader_ballot"
VK_KHR_video_encode_h264 = 1
VK_KHR_VIDEO_ENCODE_H264_SPEC_VERSION = 14
VK_KHR_VIDEO_ENCODE_H264_EXTENSION_NAME = "VK_KHR_video_encode_h264"
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_CAPABILITIES_KHR = 1000038000
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_SESSION_PARAMETERS_CREATE_INFO_KHR = 1000038001
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_SESSION_PARAMETERS_ADD_INFO_KHR = 1000038002
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_PICTURE_INFO_KHR = 1000038003
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_DPB_SLOT_INFO_KHR = 1000038004
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_NALU_SLICE_INFO_KHR = 1000038005
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_GOP_REMAINING_FRAME_INFO_KHR = 1000038006
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_PROFILE_INFO_KHR = 1000038007
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_RATE_CONTROL_INFO_KHR = 1000038008
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_RATE_CONTROL_LAYER_INFO_KHR = 1000038009
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_SESSION_CREATE_INFO_KHR = 1000038010
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_QUALITY_LEVEL_PROPERTIES_KHR = 1000038011
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_SESSION_PARAMETERS_GET_INFO_KHR = 1000038012
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_SESSION_PARAMETERS_FEEDBACK_INFO_KHR = 1000038013
VK_VIDEO_CODEC_OPERATION_ENCODE_H264_BIT_KHR = 0x00010000
VK_KHR_video_encode_h265 = 1
VK_KHR_VIDEO_ENCODE_H265_SPEC_VERSION = 14
VK_KHR_VIDEO_ENCODE_H265_EXTENSION_NAME = "VK_KHR_video_encode_h265"
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_CAPABILITIES_KHR = 1000039000
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_SESSION_PARAMETERS_CREATE_INFO_KHR = 1000039001
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_SESSION_PARAMETERS_ADD_INFO_KHR = 1000039002
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_PICTURE_INFO_KHR = 1000039003
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_DPB_SLOT_INFO_KHR = 1000039004
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_NALU_SLICE_SEGMENT_INFO_KHR = 1000039005
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_GOP_REMAINING_FRAME_INFO_KHR = 1000039006
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_PROFILE_INFO_KHR = 1000039007
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_RATE_CONTROL_INFO_KHR = 1000039009
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_RATE_CONTROL_LAYER_INFO_KHR = 1000039010
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_SESSION_CREATE_INFO_KHR = 1000039011
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_QUALITY_LEVEL_PROPERTIES_KHR = 1000039012
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_SESSION_PARAMETERS_GET_INFO_KHR = 1000039013
VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_SESSION_PARAMETERS_FEEDBACK_INFO_KHR = 1000039014
VK_VIDEO_CODEC_OPERATION_ENCODE_H265_BIT_KHR = 0x00020000
VK_KHR_video_decode_h264 = 1
VK_KHR_VIDEO_DECODE_H264_SPEC_VERSION = 9
VK_KHR_VIDEO_DECODE_H264_EXTENSION_NAME = "VK_KHR_video_decode_h264"
VK_STRUCTURE_TYPE_VIDEO_DECODE_H264_CAPABILITIES_KHR = 1000040000
VK_STRUCTURE_TYPE_VIDEO_DECODE_H264_PICTURE_INFO_KHR = 1000040001
VK_STRUCTURE_TYPE_VIDEO_DECODE_H264_PROFILE_INFO_KHR = 1000040003
VK_STRUCTURE_TYPE_VIDEO_DECODE_H264_SESSION_PARAMETERS_CREATE_INFO_KHR = 1000040004
VK_STRUCTURE_TYPE_VIDEO_DECODE_H264_SESSION_PARAMETERS_ADD_INFO_KHR = 1000040005
VK_STRUCTURE_TYPE_VIDEO_DECODE_H264_DPB_SLOT_INFO_KHR = 1000040006
VK_VIDEO_CODEC_OPERATION_DECODE_H264_BIT_KHR = 0x00000001
VK_AMD_texture_gather_bias_lod = 1
VK_AMD_TEXTURE_GATHER_BIAS_LOD_SPEC_VERSION = 1
VK_AMD_TEXTURE_GATHER_BIAS_LOD_EXTENSION_NAME = "VK_AMD_texture_gather_bias_lod"
VK_STRUCTURE_TYPE_TEXTURE_LOD_GATHER_FORMAT_PROPERTIES_AMD = 1000041000
VK_AMD_shader_info = 1
VK_AMD_SHADER_INFO_SPEC_VERSION = 1
VK_AMD_SHADER_INFO_EXTENSION_NAME = "VK_AMD_shader_info"
VK_AMD_extension_44 = 1
VK_AMD_EXTENSION_44_SPEC_VERSION = 0
VK_AMD_EXTENSION_44_EXTENSION_NAME = "VK_AMD_extension_44"
VK_KHR_dynamic_rendering = 1
VK_KHR_DYNAMIC_RENDERING_SPEC_VERSION = 1
VK_KHR_DYNAMIC_RENDERING_EXTENSION_NAME = "VK_KHR_dynamic_rendering"
VK_STRUCTURE_TYPE_RENDERING_INFO_KHR = None
VK_STRUCTURE_TYPE_RENDERING_ATTACHMENT_INFO_KHR = None
VK_STRUCTURE_TYPE_PIPELINE_RENDERING_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DYNAMIC_RENDERING_FEATURES_KHR = None
VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_RENDERING_INFO_KHR = None
VK_ATTACHMENT_STORE_OP_NONE_KHR = None
VK_PIPELINE_CREATE_RENDERING_FRAGMENT_SHADING_RATE_ATTACHMENT_BIT_KHR = 0x00200000
VK_PIPELINE_RASTERIZATION_STATE_CREATE_FRAGMENT_SHADING_RATE_ATTACHMENT_BIT_KHR = None
VK_STRUCTURE_TYPE_RENDERING_FRAGMENT_SHADING_RATE_ATTACHMENT_INFO_KHR = 1000044006
VK_PIPELINE_CREATE_RENDERING_FRAGMENT_DENSITY_MAP_ATTACHMENT_BIT_EXT = 0x00400000
VK_PIPELINE_RASTERIZATION_STATE_CREATE_FRAGMENT_DENSITY_MAP_ATTACHMENT_BIT_EXT = None
VK_STRUCTURE_TYPE_RENDERING_FRAGMENT_DENSITY_MAP_ATTACHMENT_INFO_EXT = 1000044007
VK_STRUCTURE_TYPE_ATTACHMENT_SAMPLE_COUNT_INFO_AMD = 1000044008
VK_STRUCTURE_TYPE_ATTACHMENT_SAMPLE_COUNT_INFO_NV = None
VK_STRUCTURE_TYPE_MULTIVIEW_PER_VIEW_ATTRIBUTES_INFO_NVX = 1000044009
VK_AMD_extension_46 = 1
VK_AMD_EXTENSION_46_SPEC_VERSION = 0
VK_AMD_EXTENSION_46_EXTENSION_NAME = "VK_AMD_extension_46"
VK_AMD_shader_image_load_store_lod = 1
VK_AMD_SHADER_IMAGE_LOAD_STORE_LOD_SPEC_VERSION = 1
VK_AMD_SHADER_IMAGE_LOAD_STORE_LOD_EXTENSION_NAME = "VK_AMD_shader_image_load_store_lod"
VK_NVX_extension_48 = 1
VK_NVX_EXTENSION_48_SPEC_VERSION = 0
VK_NVX_EXTENSION_48_EXTENSION_NAME = "VK_NVX_extension_48"
VK_GOOGLE_extension_49 = 1
VK_GOOGLE_EXTENSION_49_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_49_EXTENSION_NAME = "VK_GOOGLE_extension_49"
VK_GGP_stream_descriptor_surface = 1
VK_GGP_STREAM_DESCRIPTOR_SURFACE_SPEC_VERSION = 1
VK_GGP_STREAM_DESCRIPTOR_SURFACE_EXTENSION_NAME = "VK_GGP_stream_descriptor_surface"
VK_STRUCTURE_TYPE_STREAM_DESCRIPTOR_SURFACE_CREATE_INFO_GGP = 1000049000
VK_NV_corner_sampled_image = 1
VK_NV_CORNER_SAMPLED_IMAGE_SPEC_VERSION = 2
VK_NV_CORNER_SAMPLED_IMAGE_EXTENSION_NAME = "VK_NV_corner_sampled_image"
VK_IMAGE_CREATE_CORNER_SAMPLED_BIT_NV = 0x00002000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CORNER_SAMPLED_IMAGE_FEATURES_NV = 1000050000
VK_NV_private_vendor_info = 1
VK_NV_PRIVATE_VENDOR_INFO_SPEC_VERSION = 2
VK_NV_PRIVATE_VENDOR_INFO_EXTENSION_NAME = "VK_NV_private_vendor_info"
VK_STRUCTURE_TYPE_PRIVATE_VENDOR_INFO_PLACEHOLDER_OFFSET_0_NV = 1000051000
VK_NV_extension_53 = 1
VK_NV_EXTENSION_53_SPEC_VERSION = 0
VK_NV_EXTENSION_53_EXTENSION_NAME = "VK_NV_extension_53"
VK_KHR_multiview = 1
VK_KHR_MULTIVIEW_SPEC_VERSION = 1
VK_KHR_MULTIVIEW_EXTENSION_NAME = "VK_KHR_multiview"
VK_STRUCTURE_TYPE_RENDER_PASS_MULTIVIEW_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_FEATURES_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PROPERTIES_KHR = None
VK_DEPENDENCY_VIEW_LOCAL_BIT_KHR = None
VK_IMG_format_pvrtc = 1
VK_IMG_FORMAT_PVRTC_SPEC_VERSION = 1
VK_IMG_FORMAT_PVRTC_EXTENSION_NAME = "VK_IMG_format_pvrtc"
VK_FORMAT_PVRTC1_2BPP_UNORM_BLOCK_IMG = 1000054000
VK_FORMAT_PVRTC1_4BPP_UNORM_BLOCK_IMG = 1000054001
VK_FORMAT_PVRTC2_2BPP_UNORM_BLOCK_IMG = 1000054002
VK_FORMAT_PVRTC2_4BPP_UNORM_BLOCK_IMG = 1000054003
VK_FORMAT_PVRTC1_2BPP_SRGB_BLOCK_IMG = 1000054004
VK_FORMAT_PVRTC1_4BPP_SRGB_BLOCK_IMG = 1000054005
VK_FORMAT_PVRTC2_2BPP_SRGB_BLOCK_IMG = 1000054006
VK_FORMAT_PVRTC2_4BPP_SRGB_BLOCK_IMG = 1000054007
VK_NV_external_memory_capabilities = 1
VK_NV_EXTERNAL_MEMORY_CAPABILITIES_SPEC_VERSION = 1
VK_NV_EXTERNAL_MEMORY_CAPABILITIES_EXTENSION_NAME = "VK_NV_external_memory_capabilities"
VK_NV_external_memory = 1
VK_NV_EXTERNAL_MEMORY_SPEC_VERSION = 1
VK_NV_EXTERNAL_MEMORY_EXTENSION_NAME = "VK_NV_external_memory"
VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO_NV = 1000056000
VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO_NV = 1000056001
VK_NV_external_memory_win32 = 1
VK_NV_EXTERNAL_MEMORY_WIN32_SPEC_VERSION = 1
VK_NV_EXTERNAL_MEMORY_WIN32_EXTENSION_NAME = "VK_NV_external_memory_win32"
VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_NV = 1000057000
VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_NV = 1000057001
VK_NV_win32_keyed_mutex = 1
VK_NV_WIN32_KEYED_MUTEX_SPEC_VERSION = 2
VK_NV_WIN32_KEYED_MUTEX_EXTENSION_NAME = "VK_NV_win32_keyed_mutex"
VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_NV = 1000058000
VK_KHR_get_physical_device_properties2 = 1
VK_KHR_GET_PHYSICAL_DEVICE_PROPERTIES_2_SPEC_VERSION = 2
VK_KHR_GET_PHYSICAL_DEVICE_PROPERTIES_2_EXTENSION_NAME = "VK_KHR_get_physical_device_properties2"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FEATURES_2_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROPERTIES_2_KHR = None
VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_2_KHR = None
VK_STRUCTURE_TYPE_IMAGE_FORMAT_PROPERTIES_2_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_FORMAT_INFO_2_KHR = None
VK_STRUCTURE_TYPE_QUEUE_FAMILY_PROPERTIES_2_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PROPERTIES_2_KHR = None
VK_STRUCTURE_TYPE_SPARSE_IMAGE_FORMAT_PROPERTIES_2_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SPARSE_IMAGE_FORMAT_INFO_2_KHR = None
VK_KHR_device_group = 1
VK_KHR_DEVICE_GROUP_SPEC_VERSION = 4
VK_KHR_DEVICE_GROUP_EXTENSION_NAME = "VK_KHR_device_group"
VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_FLAGS_INFO_KHR = None
VK_STRUCTURE_TYPE_DEVICE_GROUP_RENDER_PASS_BEGIN_INFO_KHR = None
VK_STRUCTURE_TYPE_DEVICE_GROUP_COMMAND_BUFFER_BEGIN_INFO_KHR = None
VK_STRUCTURE_TYPE_DEVICE_GROUP_SUBMIT_INFO_KHR = None
VK_STRUCTURE_TYPE_DEVICE_GROUP_BIND_SPARSE_INFO_KHR = None
VK_PEER_MEMORY_FEATURE_COPY_SRC_BIT_KHR = None
VK_PEER_MEMORY_FEATURE_COPY_DST_BIT_KHR = None
VK_PEER_MEMORY_FEATURE_GENERIC_SRC_BIT_KHR = None
VK_PEER_MEMORY_FEATURE_GENERIC_DST_BIT_KHR = None
VK_MEMORY_ALLOCATE_DEVICE_MASK_BIT_KHR = None
VK_PIPELINE_CREATE_VIEW_INDEX_FROM_DEVICE_INDEX_BIT_KHR = None
VK_PIPELINE_CREATE_DISPATCH_BASE_KHR = None
VK_DEPENDENCY_DEVICE_GROUP_BIT_KHR = None
VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_DEVICE_GROUP_INFO_KHR = None
VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_DEVICE_GROUP_INFO_KHR = None
VK_IMAGE_CREATE_SPLIT_INSTANCE_BIND_REGIONS_BIT_KHR = None
VK_EXT_validation_flags = 1
VK_EXT_VALIDATION_FLAGS_SPEC_VERSION = 3
VK_EXT_VALIDATION_FLAGS_EXTENSION_NAME = "VK_EXT_validation_flags"
VK_STRUCTURE_TYPE_VALIDATION_FLAGS_EXT = 1000061000
VK_NN_vi_surface = 1
VK_NN_VI_SURFACE_SPEC_VERSION = 1
VK_NN_VI_SURFACE_EXTENSION_NAME = "VK_NN_vi_surface"
VK_STRUCTURE_TYPE_VI_SURFACE_CREATE_INFO_NN = 1000062000
VK_KHR_shader_draw_parameters = 1
VK_KHR_SHADER_DRAW_PARAMETERS_SPEC_VERSION = 1
VK_KHR_SHADER_DRAW_PARAMETERS_EXTENSION_NAME = "VK_KHR_shader_draw_parameters"
VK_EXT_shader_subgroup_ballot = 1
VK_EXT_SHADER_SUBGROUP_BALLOT_SPEC_VERSION = 1
VK_EXT_SHADER_SUBGROUP_BALLOT_EXTENSION_NAME = "VK_EXT_shader_subgroup_ballot"
VK_EXT_shader_subgroup_vote = 1
VK_EXT_SHADER_SUBGROUP_VOTE_SPEC_VERSION = 1
VK_EXT_SHADER_SUBGROUP_VOTE_EXTENSION_NAME = "VK_EXT_shader_subgroup_vote"
VK_EXT_texture_compression_astc_hdr = 1
VK_EXT_TEXTURE_COMPRESSION_ASTC_HDR_SPEC_VERSION = 1
VK_EXT_TEXTURE_COMPRESSION_ASTC_HDR_EXTENSION_NAME = "VK_EXT_texture_compression_astc_hdr"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXTURE_COMPRESSION_ASTC_HDR_FEATURES_EXT = None
VK_FORMAT_ASTC_4x4_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_5x4_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_5x5_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_6x5_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_6x6_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_8x5_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_8x6_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_8x8_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_10x5_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_10x6_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_10x8_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_10x10_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_12x10_SFLOAT_BLOCK_EXT = None
VK_FORMAT_ASTC_12x12_SFLOAT_BLOCK_EXT = None
VK_EXT_astc_decode_mode = 1
VK_EXT_ASTC_DECODE_MODE_SPEC_VERSION = 1
VK_EXT_ASTC_DECODE_MODE_EXTENSION_NAME = "VK_EXT_astc_decode_mode"
VK_STRUCTURE_TYPE_IMAGE_VIEW_ASTC_DECODE_MODE_EXT = 1000067000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ASTC_DECODE_FEATURES_EXT = 1000067001
VK_EXT_pipeline_robustness = 1
VK_EXT_PIPELINE_ROBUSTNESS_SPEC_VERSION = 1
VK_EXT_PIPELINE_ROBUSTNESS_EXTENSION_NAME = "VK_EXT_pipeline_robustness"
VK_STRUCTURE_TYPE_PIPELINE_ROBUSTNESS_CREATE_INFO_EXT = 1000068000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_ROBUSTNESS_FEATURES_EXT = 1000068001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_ROBUSTNESS_PROPERTIES_EXT = 1000068002
VK_KHR_maintenance1 = 1
VK_KHR_MAINTENANCE_1_SPEC_VERSION = 2
VK_KHR_MAINTENANCE_1_EXTENSION_NAME = "VK_KHR_maintenance1"
VK_KHR_MAINTENANCE1_SPEC_VERSION = None
VK_KHR_MAINTENANCE1_EXTENSION_NAME = None
VK_FORMAT_FEATURE_TRANSFER_SRC_BIT_KHR = None
VK_FORMAT_FEATURE_TRANSFER_DST_BIT_KHR = None
VK_IMAGE_CREATE_2D_ARRAY_COMPATIBLE_BIT_KHR = None
VK_KHR_device_group_creation = 1
VK_KHR_DEVICE_GROUP_CREATION_SPEC_VERSION = 1
VK_KHR_DEVICE_GROUP_CREATION_EXTENSION_NAME = "VK_KHR_device_group_creation"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GROUP_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_DEVICE_GROUP_DEVICE_CREATE_INFO_KHR = None
VK_MAX_DEVICE_GROUP_SIZE_KHR = None
VK_MEMORY_HEAP_MULTI_INSTANCE_BIT_KHR = None
VK_KHR_external_memory_capabilities = 1
VK_KHR_EXTERNAL_MEMORY_CAPABILITIES_SPEC_VERSION = 1
VK_KHR_EXTERNAL_MEMORY_CAPABILITIES_EXTENSION_NAME = "VK_KHR_external_memory_capabilities"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_IMAGE_FORMAT_INFO_KHR = None
VK_STRUCTURE_TYPE_EXTERNAL_IMAGE_FORMAT_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_BUFFER_INFO_KHR = None
VK_STRUCTURE_TYPE_EXTERNAL_BUFFER_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ID_PROPERTIES_KHR = None
VK_LUID_SIZE_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_FD_BIT_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_BIT_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_TEXTURE_BIT_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_TEXTURE_KMT_BIT_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_HEAP_BIT_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_RESOURCE_BIT_KHR = None
VK_EXTERNAL_MEMORY_FEATURE_DEDICATED_ONLY_BIT_KHR = None
VK_EXTERNAL_MEMORY_FEATURE_EXPORTABLE_BIT_KHR = None
VK_EXTERNAL_MEMORY_FEATURE_IMPORTABLE_BIT_KHR = None
VK_KHR_external_memory = 1
VK_KHR_EXTERNAL_MEMORY_SPEC_VERSION = 1
VK_KHR_EXTERNAL_MEMORY_EXTENSION_NAME = "VK_KHR_external_memory"
VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_BUFFER_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO_KHR = None
VK_QUEUE_FAMILY_EXTERNAL_KHR = None
VK_KHR_external_memory_win32 = 1
VK_KHR_EXTERNAL_MEMORY_WIN32_SPEC_VERSION = 1
VK_KHR_EXTERNAL_MEMORY_WIN32_EXTENSION_NAME = "VK_KHR_external_memory_win32"
VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_KHR = 1000073000
VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_KHR = 1000073001
VK_STRUCTURE_TYPE_MEMORY_WIN32_HANDLE_PROPERTIES_KHR = 1000073002
VK_STRUCTURE_TYPE_MEMORY_GET_WIN32_HANDLE_INFO_KHR = 1000073003
VK_KHR_external_memory_fd = 1
VK_KHR_EXTERNAL_MEMORY_FD_SPEC_VERSION = 1
VK_KHR_EXTERNAL_MEMORY_FD_EXTENSION_NAME = "VK_KHR_external_memory_fd"
VK_STRUCTURE_TYPE_IMPORT_MEMORY_FD_INFO_KHR = 1000074000
VK_STRUCTURE_TYPE_MEMORY_FD_PROPERTIES_KHR = 1000074001
VK_STRUCTURE_TYPE_MEMORY_GET_FD_INFO_KHR = 1000074002
VK_KHR_win32_keyed_mutex = 1
VK_KHR_WIN32_KEYED_MUTEX_SPEC_VERSION = 1
VK_KHR_WIN32_KEYED_MUTEX_EXTENSION_NAME = "VK_KHR_win32_keyed_mutex"
VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_KHR = 1000075000
VK_KHR_external_semaphore_capabilities = 1
VK_KHR_EXTERNAL_SEMAPHORE_CAPABILITIES_SPEC_VERSION = 1
VK_KHR_EXTERNAL_SEMAPHORE_CAPABILITIES_EXTENSION_NAME = "VK_KHR_external_semaphore_capabilities"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SEMAPHORE_INFO_KHR = None
VK_STRUCTURE_TYPE_EXTERNAL_SEMAPHORE_PROPERTIES_KHR = None
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_FD_BIT_KHR = None
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_BIT_KHR = None
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT_KHR = None
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D12_FENCE_BIT_KHR = None
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_SYNC_FD_BIT_KHR = None
VK_EXTERNAL_SEMAPHORE_FEATURE_EXPORTABLE_BIT_KHR = None
VK_EXTERNAL_SEMAPHORE_FEATURE_IMPORTABLE_BIT_KHR = None
VK_KHR_external_semaphore = 1
VK_KHR_EXTERNAL_SEMAPHORE_SPEC_VERSION = 1
VK_KHR_EXTERNAL_SEMAPHORE_EXTENSION_NAME = "VK_KHR_external_semaphore"
VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_CREATE_INFO_KHR = None
VK_SEMAPHORE_IMPORT_TEMPORARY_BIT_KHR = None
VK_KHR_external_semaphore_win32 = 1
VK_KHR_EXTERNAL_SEMAPHORE_WIN32_SPEC_VERSION = 1
VK_KHR_EXTERNAL_SEMAPHORE_WIN32_EXTENSION_NAME = "VK_KHR_external_semaphore_win32"
VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR = 1000078000
VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR = 1000078001
VK_STRUCTURE_TYPE_D3D12_FENCE_SUBMIT_INFO_KHR = 1000078002
VK_STRUCTURE_TYPE_SEMAPHORE_GET_WIN32_HANDLE_INFO_KHR = 1000078003
VK_KHR_external_semaphore_fd = 1
VK_KHR_EXTERNAL_SEMAPHORE_FD_SPEC_VERSION = 1
VK_KHR_EXTERNAL_SEMAPHORE_FD_EXTENSION_NAME = "VK_KHR_external_semaphore_fd"
VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_FD_INFO_KHR = 1000079000
VK_STRUCTURE_TYPE_SEMAPHORE_GET_FD_INFO_KHR = 1000079001
VK_KHR_push_descriptor = 1
VK_KHR_PUSH_DESCRIPTOR_SPEC_VERSION = 2
VK_KHR_PUSH_DESCRIPTOR_EXTENSION_NAME = "VK_KHR_push_descriptor"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PUSH_DESCRIPTOR_PROPERTIES_KHR = 1000080000
VK_DESCRIPTOR_SET_LAYOUT_CREATE_PUSH_DESCRIPTOR_BIT_KHR = 0x00000001
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE_PUSH_DESCRIPTORS_KHR = 1
VK_EXT_conditional_rendering = 1
VK_EXT_CONDITIONAL_RENDERING_SPEC_VERSION = 2
VK_EXT_CONDITIONAL_RENDERING_EXTENSION_NAME = "VK_EXT_conditional_rendering"
VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_CONDITIONAL_RENDERING_INFO_EXT = 1000081000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONDITIONAL_RENDERING_FEATURES_EXT = 1000081001
VK_STRUCTURE_TYPE_CONDITIONAL_RENDERING_BEGIN_INFO_EXT = 1000081002
VK_ACCESS_CONDITIONAL_RENDERING_READ_BIT_EXT = 0x00100000
VK_BUFFER_USAGE_CONDITIONAL_RENDERING_BIT_EXT = 0x00000200
VK_PIPELINE_STAGE_CONDITIONAL_RENDERING_BIT_EXT = 0x00040000
VK_KHR_shader_float16_int8 = 1
VK_KHR_SHADER_FLOAT16_INT8_SPEC_VERSION = 1
VK_KHR_SHADER_FLOAT16_INT8_EXTENSION_NAME = "VK_KHR_shader_float16_int8"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_FLOAT16_INT8_FEATURES_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT16_INT8_FEATURES_KHR = None
VK_KHR_16bit_storage = 1
VK_KHR_16BIT_STORAGE_SPEC_VERSION = 1
VK_KHR_16BIT_STORAGE_EXTENSION_NAME = "VK_KHR_16bit_storage"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_16BIT_STORAGE_FEATURES_KHR = None
VK_KHR_incremental_present = 1
VK_KHR_INCREMENTAL_PRESENT_SPEC_VERSION = 2
VK_KHR_INCREMENTAL_PRESENT_EXTENSION_NAME = "VK_KHR_incremental_present"
VK_STRUCTURE_TYPE_PRESENT_REGIONS_KHR = 1000084000
VK_KHR_descriptor_update_template = 1
VK_KHR_DESCRIPTOR_UPDATE_TEMPLATE_SPEC_VERSION = 1
VK_KHR_DESCRIPTOR_UPDATE_TEMPLATE_EXTENSION_NAME = "VK_KHR_descriptor_update_template"
VK_STRUCTURE_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_CREATE_INFO_KHR = None
VK_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_KHR = None
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE_DESCRIPTOR_SET_KHR = None
VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_KHR_EXT = None
VK_NVX_device_generated_commands = 1
VK_NVX_DEVICE_GENERATED_COMMANDS_SPEC_VERSION = 3
VK_NVX_DEVICE_GENERATED_COMMANDS_EXTENSION_NAME = "VK_NVX_device_generated_commands"
VK_NV_clip_space_w_scaling = 1
VK_NV_CLIP_SPACE_W_SCALING_SPEC_VERSION = 1
VK_NV_CLIP_SPACE_W_SCALING_EXTENSION_NAME = "VK_NV_clip_space_w_scaling"
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_W_SCALING_STATE_CREATE_INFO_NV = 1000087000
VK_DYNAMIC_STATE_VIEWPORT_W_SCALING_NV = 1000087000
VK_EXT_direct_mode_display = 1
VK_EXT_DIRECT_MODE_DISPLAY_SPEC_VERSION = 1
VK_EXT_DIRECT_MODE_DISPLAY_EXTENSION_NAME = "VK_EXT_direct_mode_display"
VK_EXT_acquire_xlib_display = 1
VK_EXT_ACQUIRE_XLIB_DISPLAY_SPEC_VERSION = 1
VK_EXT_ACQUIRE_XLIB_DISPLAY_EXTENSION_NAME = "VK_EXT_acquire_xlib_display"
VK_EXT_display_surface_counter = 1
VK_EXT_DISPLAY_SURFACE_COUNTER_SPEC_VERSION = 1
VK_EXT_DISPLAY_SURFACE_COUNTER_EXTENSION_NAME = "VK_EXT_display_surface_counter"
VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_EXT = 1000090000
VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES2_EXT = None
VK_EXT_display_control = 1
VK_EXT_DISPLAY_CONTROL_SPEC_VERSION = 1
VK_EXT_DISPLAY_CONTROL_EXTENSION_NAME = "VK_EXT_display_control"
VK_STRUCTURE_TYPE_DISPLAY_POWER_INFO_EXT = 1000091000
VK_STRUCTURE_TYPE_DEVICE_EVENT_INFO_EXT = 1000091001
VK_STRUCTURE_TYPE_DISPLAY_EVENT_INFO_EXT = 1000091002
VK_STRUCTURE_TYPE_SWAPCHAIN_COUNTER_CREATE_INFO_EXT = 1000091003
VK_GOOGLE_display_timing = 1
VK_GOOGLE_DISPLAY_TIMING_SPEC_VERSION = 1
VK_GOOGLE_DISPLAY_TIMING_EXTENSION_NAME = "VK_GOOGLE_display_timing"
VK_STRUCTURE_TYPE_PRESENT_TIMES_INFO_GOOGLE = 1000092000
VK_RESERVED_do_not_use_94 = 1
VK_RESERVED_DO_NOT_USE_94_SPEC_VERSION = 1
VK_RESERVED_DO_NOT_USE_94_EXTENSION_NAME = "VK_RESERVED_do_not_use_94"
VK_NV_sample_mask_override_coverage = 1
VK_NV_SAMPLE_MASK_OVERRIDE_COVERAGE_SPEC_VERSION = 1
VK_NV_SAMPLE_MASK_OVERRIDE_COVERAGE_EXTENSION_NAME = "VK_NV_sample_mask_override_coverage"
VK_NV_geometry_shader_passthrough = 1
VK_NV_GEOMETRY_SHADER_PASSTHROUGH_SPEC_VERSION = 1
VK_NV_GEOMETRY_SHADER_PASSTHROUGH_EXTENSION_NAME = "VK_NV_geometry_shader_passthrough"
VK_NV_viewport_array2 = 1
VK_NV_VIEWPORT_ARRAY_2_SPEC_VERSION = 1
VK_NV_VIEWPORT_ARRAY_2_EXTENSION_NAME = "VK_NV_viewport_array2"
VK_NV_VIEWPORT_ARRAY2_SPEC_VERSION = None
VK_NV_VIEWPORT_ARRAY2_EXTENSION_NAME = None
VK_NVX_multiview_per_view_attributes = 1
VK_NVX_MULTIVIEW_PER_VIEW_ATTRIBUTES_SPEC_VERSION = 1
VK_NVX_MULTIVIEW_PER_VIEW_ATTRIBUTES_EXTENSION_NAME = "VK_NVX_multiview_per_view_attributes"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PER_VIEW_ATTRIBUTES_PROPERTIES_NVX = 1000097000
VK_SUBPASS_DESCRIPTION_PER_VIEW_ATTRIBUTES_BIT_NVX = 0x00000001
VK_SUBPASS_DESCRIPTION_PER_VIEW_POSITION_X_ONLY_BIT_NVX = 0x00000002
VK_NV_viewport_swizzle = 1
VK_NV_VIEWPORT_SWIZZLE_SPEC_VERSION = 1
VK_NV_VIEWPORT_SWIZZLE_EXTENSION_NAME = "VK_NV_viewport_swizzle"
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SWIZZLE_STATE_CREATE_INFO_NV = 1000098000
VK_EXT_discard_rectangles = 1
VK_EXT_DISCARD_RECTANGLES_SPEC_VERSION = 2
VK_EXT_DISCARD_RECTANGLES_EXTENSION_NAME = "VK_EXT_discard_rectangles"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DISCARD_RECTANGLE_PROPERTIES_EXT = 1000099000
VK_STRUCTURE_TYPE_PIPELINE_DISCARD_RECTANGLE_STATE_CREATE_INFO_EXT = 1000099001
VK_DYNAMIC_STATE_DISCARD_RECTANGLE_EXT = 1000099000
VK_DYNAMIC_STATE_DISCARD_RECTANGLE_ENABLE_EXT = 1000099001
VK_DYNAMIC_STATE_DISCARD_RECTANGLE_MODE_EXT = 1000099002
VK_NV_extension_101 = 1
VK_NV_EXTENSION_101_SPEC_VERSION = 0
VK_NV_EXTENSION_101_EXTENSION_NAME = "VK_NV_extension_101"
VK_EXT_conservative_rasterization = 1
VK_EXT_CONSERVATIVE_RASTERIZATION_SPEC_VERSION = 1
VK_EXT_CONSERVATIVE_RASTERIZATION_EXTENSION_NAME = "VK_EXT_conservative_rasterization"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONSERVATIVE_RASTERIZATION_PROPERTIES_EXT = 1000101000
VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_CONSERVATIVE_STATE_CREATE_INFO_EXT = 1000101001
VK_EXT_depth_clip_enable = 1
VK_EXT_DEPTH_CLIP_ENABLE_SPEC_VERSION = 1
VK_EXT_DEPTH_CLIP_ENABLE_EXTENSION_NAME = "VK_EXT_depth_clip_enable"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_CLIP_ENABLE_FEATURES_EXT = 1000102000
VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_DEPTH_CLIP_STATE_CREATE_INFO_EXT = 1000102001
VK_NV_extension_104 = 1
VK_NV_EXTENSION_104_SPEC_VERSION = 0
VK_NV_EXTENSION_104_EXTENSION_NAME = "VK_NV_extension_104"
VK_PRIVATE_DATA_SLOT_CREATE_RESERVED_0_BIT_NV = 0x00000001
VK_EXT_swapchain_colorspace = 1
VK_EXT_SWAPCHAIN_COLOR_SPACE_SPEC_VERSION = 4
VK_EXT_SWAPCHAIN_COLOR_SPACE_EXTENSION_NAME = "VK_EXT_swapchain_colorspace"
VK_COLOR_SPACE_DISPLAY_P3_NONLINEAR_EXT = 1000104001
VK_COLOR_SPACE_EXTENDED_SRGB_LINEAR_EXT = 1000104002
VK_COLOR_SPACE_DISPLAY_P3_LINEAR_EXT = 1000104003
VK_COLOR_SPACE_DCI_P3_NONLINEAR_EXT = 1000104004
VK_COLOR_SPACE_BT709_LINEAR_EXT = 1000104005
VK_COLOR_SPACE_BT709_NONLINEAR_EXT = 1000104006
VK_COLOR_SPACE_BT2020_LINEAR_EXT = 1000104007
VK_COLOR_SPACE_HDR10_ST2084_EXT = 1000104008
VK_COLOR_SPACE_DOLBYVISION_EXT = 1000104009
VK_COLOR_SPACE_HDR10_HLG_EXT = 1000104010
VK_COLOR_SPACE_ADOBERGB_LINEAR_EXT = 1000104011
VK_COLOR_SPACE_ADOBERGB_NONLINEAR_EXT = 1000104012
VK_COLOR_SPACE_PASS_THROUGH_EXT = 1000104013
VK_COLOR_SPACE_EXTENDED_SRGB_NONLINEAR_EXT = 1000104014
VK_COLOR_SPACE_DCI_P3_LINEAR_EXT = None
VK_EXT_hdr_metadata = 1
VK_EXT_HDR_METADATA_SPEC_VERSION = 2
VK_EXT_HDR_METADATA_EXTENSION_NAME = "VK_EXT_hdr_metadata"
VK_STRUCTURE_TYPE_HDR_METADATA_EXT = 1000105000
VK_IMG_extension_107 = 1
VK_IMG_EXTENSION_107_SPEC_VERSION = 0
VK_IMG_EXTENSION_107_EXTENSION_NAME = "VK_IMG_extension_107"
VK_IMG_extension_108 = 1
VK_IMG_EXTENSION_108_SPEC_VERSION = 0
VK_IMG_EXTENSION_108_EXTENSION_NAME = "VK_IMG_extension_108"
VK_KHR_imageless_framebuffer = 1
VK_KHR_IMAGELESS_FRAMEBUFFER_SPEC_VERSION = 1
VK_KHR_IMAGELESS_FRAMEBUFFER_EXTENSION_NAME = "VK_KHR_imageless_framebuffer"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGELESS_FRAMEBUFFER_FEATURES_KHR = None
VK_STRUCTURE_TYPE_FRAMEBUFFER_ATTACHMENTS_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_FRAMEBUFFER_ATTACHMENT_IMAGE_INFO_KHR = None
VK_STRUCTURE_TYPE_RENDER_PASS_ATTACHMENT_BEGIN_INFO_KHR = None
VK_FRAMEBUFFER_CREATE_IMAGELESS_BIT_KHR = None
VK_KHR_create_renderpass2 = 1
VK_KHR_CREATE_RENDERPASS_2_SPEC_VERSION = 1
VK_KHR_CREATE_RENDERPASS_2_EXTENSION_NAME = "VK_KHR_create_renderpass2"
VK_STRUCTURE_TYPE_ATTACHMENT_DESCRIPTION_2_KHR = None
VK_STRUCTURE_TYPE_ATTACHMENT_REFERENCE_2_KHR = None
VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_2_KHR = None
VK_STRUCTURE_TYPE_SUBPASS_DEPENDENCY_2_KHR = None
VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO_2_KHR = None
VK_STRUCTURE_TYPE_SUBPASS_BEGIN_INFO_KHR = None
VK_STRUCTURE_TYPE_SUBPASS_END_INFO_KHR = None
VK_IMG_relaxed_line_rasterization = 1
VK_IMG_RELAXED_LINE_RASTERIZATION_SPEC_VERSION = 1
VK_IMG_RELAXED_LINE_RASTERIZATION_EXTENSION_NAME = "VK_IMG_relaxed_line_rasterization"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RELAXED_LINE_RASTERIZATION_FEATURES_IMG = 1000110000
VK_KHR_shared_presentable_image = 1
VK_KHR_SHARED_PRESENTABLE_IMAGE_SPEC_VERSION = 1
VK_KHR_SHARED_PRESENTABLE_IMAGE_EXTENSION_NAME = "VK_KHR_shared_presentable_image"
VK_STRUCTURE_TYPE_SHARED_PRESENT_SURFACE_CAPABILITIES_KHR = 1000111000
VK_PRESENT_MODE_SHARED_DEMAND_REFRESH_KHR = 1000111000
VK_PRESENT_MODE_SHARED_CONTINUOUS_REFRESH_KHR = 1000111001
VK_IMAGE_LAYOUT_SHARED_PRESENT_KHR = 1000111000
VK_KHR_external_fence_capabilities = 1
VK_KHR_EXTERNAL_FENCE_CAPABILITIES_SPEC_VERSION = 1
VK_KHR_EXTERNAL_FENCE_CAPABILITIES_EXTENSION_NAME = "VK_KHR_external_fence_capabilities"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FENCE_INFO_KHR = None
VK_STRUCTURE_TYPE_EXTERNAL_FENCE_PROPERTIES_KHR = None
VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_FD_BIT_KHR = None
VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_WIN32_BIT_KHR = None
VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT_KHR = None
VK_EXTERNAL_FENCE_HANDLE_TYPE_SYNC_FD_BIT_KHR = None
VK_EXTERNAL_FENCE_FEATURE_EXPORTABLE_BIT_KHR = None
VK_EXTERNAL_FENCE_FEATURE_IMPORTABLE_BIT_KHR = None
VK_KHR_external_fence = 1
VK_KHR_EXTERNAL_FENCE_SPEC_VERSION = 1
VK_KHR_EXTERNAL_FENCE_EXTENSION_NAME = "VK_KHR_external_fence"
VK_STRUCTURE_TYPE_EXPORT_FENCE_CREATE_INFO_KHR = None
VK_FENCE_IMPORT_TEMPORARY_BIT_KHR = None
VK_KHR_external_fence_win32 = 1
VK_KHR_EXTERNAL_FENCE_WIN32_SPEC_VERSION = 1
VK_KHR_EXTERNAL_FENCE_WIN32_EXTENSION_NAME = "VK_KHR_external_fence_win32"
VK_STRUCTURE_TYPE_IMPORT_FENCE_WIN32_HANDLE_INFO_KHR = 1000114000
VK_STRUCTURE_TYPE_EXPORT_FENCE_WIN32_HANDLE_INFO_KHR = 1000114001
VK_STRUCTURE_TYPE_FENCE_GET_WIN32_HANDLE_INFO_KHR = 1000114002
VK_KHR_external_fence_fd = 1
VK_KHR_EXTERNAL_FENCE_FD_SPEC_VERSION = 1
VK_KHR_EXTERNAL_FENCE_FD_EXTENSION_NAME = "VK_KHR_external_fence_fd"
VK_STRUCTURE_TYPE_IMPORT_FENCE_FD_INFO_KHR = 1000115000
VK_STRUCTURE_TYPE_FENCE_GET_FD_INFO_KHR = 1000115001
VK_KHR_performance_query = 1
VK_KHR_PERFORMANCE_QUERY_SPEC_VERSION = 1
VK_KHR_PERFORMANCE_QUERY_EXTENSION_NAME = "VK_KHR_performance_query"
VK_QUERY_TYPE_PERFORMANCE_QUERY_KHR = 1000116000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PERFORMANCE_QUERY_FEATURES_KHR = 1000116000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PERFORMANCE_QUERY_PROPERTIES_KHR = 1000116001
VK_STRUCTURE_TYPE_QUERY_POOL_PERFORMANCE_CREATE_INFO_KHR = 1000116002
VK_STRUCTURE_TYPE_PERFORMANCE_QUERY_SUBMIT_INFO_KHR = 1000116003
VK_STRUCTURE_TYPE_ACQUIRE_PROFILING_LOCK_INFO_KHR = 1000116004
VK_STRUCTURE_TYPE_PERFORMANCE_COUNTER_KHR = 1000116005
VK_STRUCTURE_TYPE_PERFORMANCE_COUNTER_DESCRIPTION_KHR = 1000116006
VK_STRUCTURE_TYPE_PERFORMANCE_QUERY_RESERVATION_INFO_KHR = 1000116007
VK_KHR_maintenance2 = 1
VK_KHR_MAINTENANCE_2_SPEC_VERSION = 1
VK_KHR_MAINTENANCE_2_EXTENSION_NAME = "VK_KHR_maintenance2"
VK_KHR_MAINTENANCE2_SPEC_VERSION = None
VK_KHR_MAINTENANCE2_EXTENSION_NAME = None
VK_IMAGE_CREATE_BLOCK_TEXEL_VIEW_COMPATIBLE_BIT_KHR = None
VK_IMAGE_CREATE_EXTENDED_USAGE_BIT_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_POINT_CLIPPING_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_RENDER_PASS_INPUT_ATTACHMENT_ASPECT_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_IMAGE_VIEW_USAGE_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_DOMAIN_ORIGIN_STATE_CREATE_INFO_KHR = None
VK_IMAGE_LAYOUT_DEPTH_READ_ONLY_STENCIL_ATTACHMENT_OPTIMAL_KHR = None
VK_IMAGE_LAYOUT_DEPTH_ATTACHMENT_STENCIL_READ_ONLY_OPTIMAL_KHR = None
VK_POINT_CLIPPING_BEHAVIOR_ALL_CLIP_PLANES_KHR = None
VK_POINT_CLIPPING_BEHAVIOR_USER_CLIP_PLANES_ONLY_KHR = None
VK_TESSELLATION_DOMAIN_ORIGIN_UPPER_LEFT_KHR = None
VK_TESSELLATION_DOMAIN_ORIGIN_LOWER_LEFT_KHR = None
VK_KHR_extension_119 = 1
VK_KHR_EXTENSION_119_SPEC_VERSION = 0
VK_KHR_EXTENSION_119_EXTENSION_NAME = "VK_KHR_extension_119"
VK_KHR_get_surface_capabilities2 = 1
VK_KHR_GET_SURFACE_CAPABILITIES_2_SPEC_VERSION = 1
VK_KHR_GET_SURFACE_CAPABILITIES_2_EXTENSION_NAME = "VK_KHR_get_surface_capabilities2"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SURFACE_INFO_2_KHR = 1000119000
VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_KHR = 1000119001
VK_STRUCTURE_TYPE_SURFACE_FORMAT_2_KHR = 1000119002
VK_KHR_variable_pointers = 1
VK_KHR_VARIABLE_POINTERS_SPEC_VERSION = 1
VK_KHR_VARIABLE_POINTERS_EXTENSION_NAME = "VK_KHR_variable_pointers"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VARIABLE_POINTERS_FEATURES_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VARIABLE_POINTER_FEATURES_KHR = None
VK_KHR_get_display_properties2 = 1
VK_KHR_GET_DISPLAY_PROPERTIES_2_SPEC_VERSION = 1
VK_KHR_GET_DISPLAY_PROPERTIES_2_EXTENSION_NAME = "VK_KHR_get_display_properties2"
VK_STRUCTURE_TYPE_DISPLAY_PROPERTIES_2_KHR = 1000121000
VK_STRUCTURE_TYPE_DISPLAY_PLANE_PROPERTIES_2_KHR = 1000121001
VK_STRUCTURE_TYPE_DISPLAY_MODE_PROPERTIES_2_KHR = 1000121002
VK_STRUCTURE_TYPE_DISPLAY_PLANE_INFO_2_KHR = 1000121003
VK_STRUCTURE_TYPE_DISPLAY_PLANE_CAPABILITIES_2_KHR = 1000121004
VK_MVK_ios_surface = 1
VK_MVK_IOS_SURFACE_SPEC_VERSION = 3
VK_MVK_IOS_SURFACE_EXTENSION_NAME = "VK_MVK_ios_surface"
VK_STRUCTURE_TYPE_IOS_SURFACE_CREATE_INFO_MVK = 1000122000
VK_MVK_macos_surface = 1
VK_MVK_MACOS_SURFACE_SPEC_VERSION = 3
VK_MVK_MACOS_SURFACE_EXTENSION_NAME = "VK_MVK_macos_surface"
VK_STRUCTURE_TYPE_MACOS_SURFACE_CREATE_INFO_MVK = 1000123000
VK_MVK_moltenvk = 1
VK_MVK_MOLTENVK_SPEC_VERSION = 0
VK_MVK_MOLTENVK_EXTENSION_NAME = "VK_MVK_moltenvk"
VK_EXT_external_memory_dma_buf = 1
VK_EXT_EXTERNAL_MEMORY_DMA_BUF_SPEC_VERSION = 1
VK_EXT_EXTERNAL_MEMORY_DMA_BUF_EXTENSION_NAME = "VK_EXT_external_memory_dma_buf"
VK_EXTERNAL_MEMORY_HANDLE_TYPE_DMA_BUF_BIT_EXT = 0x00000200
VK_EXT_queue_family_foreign = 1
VK_EXT_QUEUE_FAMILY_FOREIGN_SPEC_VERSION = 1
VK_EXT_QUEUE_FAMILY_FOREIGN_EXTENSION_NAME = "VK_EXT_queue_family_foreign"
VK_KHR_dedicated_allocation = 1
VK_KHR_DEDICATED_ALLOCATION_SPEC_VERSION = 3
VK_KHR_DEDICATED_ALLOCATION_EXTENSION_NAME = "VK_KHR_dedicated_allocation"
VK_STRUCTURE_TYPE_MEMORY_DEDICATED_REQUIREMENTS_KHR = None
VK_STRUCTURE_TYPE_MEMORY_DEDICATED_ALLOCATE_INFO_KHR = None
VK_EXT_debug_utils = 1
VK_EXT_DEBUG_UTILS_SPEC_VERSION = 2
VK_EXT_DEBUG_UTILS_EXTENSION_NAME = "VK_EXT_debug_utils"
VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_NAME_INFO_EXT = 1000128000
VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_TAG_INFO_EXT = 1000128001
VK_STRUCTURE_TYPE_DEBUG_UTILS_LABEL_EXT = 1000128002
VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CALLBACK_DATA_EXT = 1000128003
VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CREATE_INFO_EXT = 1000128004
VK_OBJECT_TYPE_DEBUG_UTILS_MESSENGER_EXT = 1000128000
VK_ANDROID_external_memory_android_hardware_buffer = 1
VK_ANDROID_EXTERNAL_MEMORY_ANDROID_HARDWARE_BUFFER_SPEC_VERSION = 5
VK_ANDROID_EXTERNAL_MEMORY_ANDROID_HARDWARE_BUFFER_EXTENSION_NAME = "VK_ANDROID_external_memory_android_hardware_buffer"
VK_EXTERNAL_MEMORY_HANDLE_TYPE_ANDROID_HARDWARE_BUFFER_BIT_ANDROID = 0x00000400
VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_USAGE_ANDROID = 1000129000
VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_PROPERTIES_ANDROID = 1000129001
VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_FORMAT_PROPERTIES_ANDROID = 1000129002
VK_STRUCTURE_TYPE_IMPORT_ANDROID_HARDWARE_BUFFER_INFO_ANDROID = 1000129003
VK_STRUCTURE_TYPE_MEMORY_GET_ANDROID_HARDWARE_BUFFER_INFO_ANDROID = 1000129004
VK_STRUCTURE_TYPE_EXTERNAL_FORMAT_ANDROID = 1000129005
VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_FORMAT_PROPERTIES_2_ANDROID = 1000129006
VK_EXT_sampler_filter_minmax = 1
VK_EXT_SAMPLER_FILTER_MINMAX_SPEC_VERSION = 2
VK_EXT_SAMPLER_FILTER_MINMAX_EXTENSION_NAME = "VK_EXT_sampler_filter_minmax"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_FILTER_MINMAX_PROPERTIES_EXT = None
VK_STRUCTURE_TYPE_SAMPLER_REDUCTION_MODE_CREATE_INFO_EXT = None
VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_MINMAX_BIT_EXT = None
VK_SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE_EXT = None
VK_SAMPLER_REDUCTION_MODE_MIN_EXT = None
VK_SAMPLER_REDUCTION_MODE_MAX_EXT = None
VK_KHR_storage_buffer_storage_class = 1
VK_KHR_STORAGE_BUFFER_STORAGE_CLASS_SPEC_VERSION = 1
VK_KHR_STORAGE_BUFFER_STORAGE_CLASS_EXTENSION_NAME = "VK_KHR_storage_buffer_storage_class"
VK_AMD_gpu_shader_int16 = 1
VK_AMD_GPU_SHADER_INT16_SPEC_VERSION = 2
VK_AMD_GPU_SHADER_INT16_EXTENSION_NAME = "VK_AMD_gpu_shader_int16"
VK_AMD_extension_134 = 1
VK_AMD_EXTENSION_134_SPEC_VERSION = 0
VK_AMD_EXTENSION_134_EXTENSION_NAME = "VK_AMD_extension_134"
VK_AMDX_shader_enqueue = 1
VK_AMDX_SHADER_ENQUEUE_SPEC_VERSION = 1
VK_AMDX_SHADER_ENQUEUE_EXTENSION_NAME = "VK_AMDX_shader_enqueue"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ENQUEUE_FEATURES_AMDX = 1000134000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ENQUEUE_PROPERTIES_AMDX = 1000134001
VK_STRUCTURE_TYPE_EXECUTION_GRAPH_PIPELINE_SCRATCH_SIZE_AMDX = 1000134002
VK_STRUCTURE_TYPE_EXECUTION_GRAPH_PIPELINE_CREATE_INFO_AMDX = 1000134003
VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_NODE_CREATE_INFO_AMDX = 1000134004
VK_BUFFER_USAGE_EXECUTION_GRAPH_SCRATCH_BIT_AMDX = 0x02000000
VK_PIPELINE_BIND_POINT_EXECUTION_GRAPH_AMDX = 1000134000
VK_BUFFER_USAGE_2_EXECUTION_GRAPH_SCRATCH_BIT_AMDX = 0x02000000
VK_KHR_extension_136 = 1
VK_KHR_EXTENSION_136_SPEC_VERSION = 0
VK_KHR_EXTENSION_136_EXTENSION_NAME = "VK_KHR_extension_136"
VK_BUFFER_USAGE_RESERVED_28_BIT_KHR = 0x10000000
VK_BUFFER_USAGE_RESERVED_29_BIT_KHR = 0x20000000
VK_BUFFER_USAGE_RESERVED_30_BIT_KHR = 0x40000000
VK_BUFFER_USAGE_2_RESERVED_28_BIT_KHR = 0x10000000
VK_BUFFER_USAGE_2_RESERVED_29_BIT_KHR = 0x20000000
VK_BUFFER_USAGE_2_RESERVED_30_BIT_KHR = 0x40000000
VK_AMD_mixed_attachment_samples = 1
VK_AMD_MIXED_ATTACHMENT_SAMPLES_SPEC_VERSION = 1
VK_AMD_MIXED_ATTACHMENT_SAMPLES_EXTENSION_NAME = "VK_AMD_mixed_attachment_samples"
VK_AMD_shader_fragment_mask = 1
VK_AMD_SHADER_FRAGMENT_MASK_SPEC_VERSION = 1
VK_AMD_SHADER_FRAGMENT_MASK_EXTENSION_NAME = "VK_AMD_shader_fragment_mask"
VK_EXT_inline_uniform_block = 1
VK_EXT_INLINE_UNIFORM_BLOCK_SPEC_VERSION = 1
VK_EXT_INLINE_UNIFORM_BLOCK_EXTENSION_NAME = "VK_EXT_inline_uniform_block"
VK_DESCRIPTOR_TYPE_INLINE_UNIFORM_BLOCK_EXT = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_FEATURES_EXT = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_PROPERTIES_EXT = None
VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_INLINE_UNIFORM_BLOCK_EXT = None
VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_INLINE_UNIFORM_BLOCK_CREATE_INFO_EXT = None
VK_AMD_extension_140 = 1
VK_AMD_EXTENSION_140_SPEC_VERSION = 0
VK_AMD_EXTENSION_140_EXTENSION_NAME = "VK_AMD_extension_140"
VK_EXT_shader_stencil_export = 1
VK_EXT_SHADER_STENCIL_EXPORT_SPEC_VERSION = 1
VK_EXT_SHADER_STENCIL_EXPORT_EXTENSION_NAME = "VK_EXT_shader_stencil_export"
VK_AMD_extension_142 = 1
VK_AMD_EXTENSION_142_SPEC_VERSION = 0
VK_AMD_EXTENSION_142_EXTENSION_NAME = "VK_AMD_extension_142"
VK_AMD_extension_143 = 1
VK_AMD_EXTENSION_143_SPEC_VERSION = 0
VK_AMD_EXTENSION_143_EXTENSION_NAME = "VK_AMD_extension_143"
VK_EXT_sample_locations = 1
VK_EXT_SAMPLE_LOCATIONS_SPEC_VERSION = 1
VK_EXT_SAMPLE_LOCATIONS_EXTENSION_NAME = "VK_EXT_sample_locations"
VK_IMAGE_CREATE_SAMPLE_LOCATIONS_COMPATIBLE_DEPTH_BIT_EXT = 0x00001000
VK_STRUCTURE_TYPE_SAMPLE_LOCATIONS_INFO_EXT = 1000143000
VK_STRUCTURE_TYPE_RENDER_PASS_SAMPLE_LOCATIONS_BEGIN_INFO_EXT = 1000143001
VK_STRUCTURE_TYPE_PIPELINE_SAMPLE_LOCATIONS_STATE_CREATE_INFO_EXT = 1000143002
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLE_LOCATIONS_PROPERTIES_EXT = 1000143003
VK_STRUCTURE_TYPE_MULTISAMPLE_PROPERTIES_EXT = 1000143004
VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT = 1000143000
VK_KHR_relaxed_block_layout = 1
VK_KHR_RELAXED_BLOCK_LAYOUT_SPEC_VERSION = 1
VK_KHR_RELAXED_BLOCK_LAYOUT_EXTENSION_NAME = "VK_KHR_relaxed_block_layout"
VK_RESERVED_do_not_use_146 = 1
VK_RESERVED_DO_NOT_USE_146_SPEC_VERSION = 1
VK_RESERVED_DO_NOT_USE_146_EXTENSION_NAME = "VK_RESERVED_do_not_use_146"
VK_KHR_get_memory_requirements2 = 1
VK_KHR_GET_MEMORY_REQUIREMENTS_2_SPEC_VERSION = 1
VK_KHR_GET_MEMORY_REQUIREMENTS_2_EXTENSION_NAME = "VK_KHR_get_memory_requirements2"
VK_STRUCTURE_TYPE_BUFFER_MEMORY_REQUIREMENTS_INFO_2_KHR = None
VK_STRUCTURE_TYPE_IMAGE_MEMORY_REQUIREMENTS_INFO_2_KHR = None
VK_STRUCTURE_TYPE_IMAGE_SPARSE_MEMORY_REQUIREMENTS_INFO_2_KHR = None
VK_STRUCTURE_TYPE_MEMORY_REQUIREMENTS_2_KHR = None
VK_STRUCTURE_TYPE_SPARSE_IMAGE_MEMORY_REQUIREMENTS_2_KHR = None
VK_KHR_image_format_list = 1
VK_KHR_IMAGE_FORMAT_LIST_SPEC_VERSION = 1
VK_KHR_IMAGE_FORMAT_LIST_EXTENSION_NAME = "VK_KHR_image_format_list"
VK_STRUCTURE_TYPE_IMAGE_FORMAT_LIST_CREATE_INFO_KHR = None
VK_EXT_blend_operation_advanced = 1
VK_EXT_BLEND_OPERATION_ADVANCED_SPEC_VERSION = 2
VK_EXT_BLEND_OPERATION_ADVANCED_EXTENSION_NAME = "VK_EXT_blend_operation_advanced"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_FEATURES_EXT = 1000148000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_PROPERTIES_EXT = 1000148001
VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_ADVANCED_STATE_CREATE_INFO_EXT = 1000148002
VK_BLEND_OP_ZERO_EXT = 1000148000
VK_BLEND_OP_SRC_EXT = 1000148001
VK_BLEND_OP_DST_EXT = 1000148002
VK_BLEND_OP_SRC_OVER_EXT = 1000148003
VK_BLEND_OP_DST_OVER_EXT = 1000148004
VK_BLEND_OP_SRC_IN_EXT = 1000148005
VK_BLEND_OP_DST_IN_EXT = 1000148006
VK_BLEND_OP_SRC_OUT_EXT = 1000148007
VK_BLEND_OP_DST_OUT_EXT = 1000148008
VK_BLEND_OP_SRC_ATOP_EXT = 1000148009
VK_BLEND_OP_DST_ATOP_EXT = 1000148010
VK_BLEND_OP_XOR_EXT = 1000148011
VK_BLEND_OP_MULTIPLY_EXT = 1000148012
VK_BLEND_OP_SCREEN_EXT = 1000148013
VK_BLEND_OP_OVERLAY_EXT = 1000148014
VK_BLEND_OP_DARKEN_EXT = 1000148015
VK_BLEND_OP_LIGHTEN_EXT = 1000148016
VK_BLEND_OP_COLORDODGE_EXT = 1000148017
VK_BLEND_OP_COLORBURN_EXT = 1000148018
VK_BLEND_OP_HARDLIGHT_EXT = 1000148019
VK_BLEND_OP_SOFTLIGHT_EXT = 1000148020
VK_BLEND_OP_DIFFERENCE_EXT = 1000148021
VK_BLEND_OP_EXCLUSION_EXT = 1000148022
VK_BLEND_OP_INVERT_EXT = 1000148023
VK_BLEND_OP_INVERT_RGB_EXT = 1000148024
VK_BLEND_OP_LINEARDODGE_EXT = 1000148025
VK_BLEND_OP_LINEARBURN_EXT = 1000148026
VK_BLEND_OP_VIVIDLIGHT_EXT = 1000148027
VK_BLEND_OP_LINEARLIGHT_EXT = 1000148028
VK_BLEND_OP_PINLIGHT_EXT = 1000148029
VK_BLEND_OP_HARDMIX_EXT = 1000148030
VK_BLEND_OP_HSL_HUE_EXT = 1000148031
VK_BLEND_OP_HSL_SATURATION_EXT = 1000148032
VK_BLEND_OP_HSL_COLOR_EXT = 1000148033
VK_BLEND_OP_HSL_LUMINOSITY_EXT = 1000148034
VK_BLEND_OP_PLUS_EXT = 1000148035
VK_BLEND_OP_PLUS_CLAMPED_EXT = 1000148036
VK_BLEND_OP_PLUS_CLAMPED_ALPHA_EXT = 1000148037
VK_BLEND_OP_PLUS_DARKER_EXT = 1000148038
VK_BLEND_OP_MINUS_EXT = 1000148039
VK_BLEND_OP_MINUS_CLAMPED_EXT = 1000148040
VK_BLEND_OP_CONTRAST_EXT = 1000148041
VK_BLEND_OP_INVERT_OVG_EXT = 1000148042
VK_BLEND_OP_RED_EXT = 1000148043
VK_BLEND_OP_GREEN_EXT = 1000148044
VK_BLEND_OP_BLUE_EXT = 1000148045
VK_ACCESS_COLOR_ATTACHMENT_READ_NONCOHERENT_BIT_EXT = 0x00080000
VK_NV_fragment_coverage_to_color = 1
VK_NV_FRAGMENT_COVERAGE_TO_COLOR_SPEC_VERSION = 1
VK_NV_FRAGMENT_COVERAGE_TO_COLOR_EXTENSION_NAME = "VK_NV_fragment_coverage_to_color"
VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_TO_COLOR_STATE_CREATE_INFO_NV = 1000149000
VK_KHR_acceleration_structure = 1
VK_KHR_ACCELERATION_STRUCTURE_SPEC_VERSION = 13
VK_KHR_ACCELERATION_STRUCTURE_EXTENSION_NAME = "VK_KHR_acceleration_structure"
VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_ACCELERATION_STRUCTURE_KHR = 1000150007
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_BUILD_GEOMETRY_INFO_KHR = 1000150000
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_DEVICE_ADDRESS_INFO_KHR = 1000150002
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_AABBS_DATA_KHR = 1000150003
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_INSTANCES_DATA_KHR = 1000150004
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_TRIANGLES_DATA_KHR = 1000150005
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_KHR = 1000150006
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_VERSION_INFO_KHR = 1000150009
VK_STRUCTURE_TYPE_COPY_ACCELERATION_STRUCTURE_INFO_KHR = 1000150010
VK_STRUCTURE_TYPE_COPY_ACCELERATION_STRUCTURE_TO_MEMORY_INFO_KHR = 1000150011
VK_STRUCTURE_TYPE_COPY_MEMORY_TO_ACCELERATION_STRUCTURE_INFO_KHR = 1000150012
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ACCELERATION_STRUCTURE_FEATURES_KHR = 1000150013
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ACCELERATION_STRUCTURE_PROPERTIES_KHR = 1000150014
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_CREATE_INFO_KHR = 1000150017
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_BUILD_SIZES_INFO_KHR = 1000150020
VK_PIPELINE_STAGE_ACCELERATION_STRUCTURE_BUILD_BIT_KHR = 0x02000000
VK_DESCRIPTOR_TYPE_ACCELERATION_STRUCTURE_KHR = 1000150000
VK_ACCESS_ACCELERATION_STRUCTURE_READ_BIT_KHR = 0x00200000
VK_ACCESS_ACCELERATION_STRUCTURE_WRITE_BIT_KHR = 0x00400000
VK_QUERY_TYPE_ACCELERATION_STRUCTURE_COMPACTED_SIZE_KHR = 1000150000
VK_QUERY_TYPE_ACCELERATION_STRUCTURE_SERIALIZATION_SIZE_KHR = 1000150001
VK_OBJECT_TYPE_ACCELERATION_STRUCTURE_KHR = 1000150000
VK_INDEX_TYPE_NONE_KHR = 1000150000
VK_FORMAT_FEATURE_ACCELERATION_STRUCTURE_VERTEX_BUFFER_BIT_KHR = 0x20000000
VK_BUFFER_USAGE_ACCELERATION_STRUCTURE_BUILD_INPUT_READ_ONLY_BIT_KHR = 0x00080000
VK_BUFFER_USAGE_ACCELERATION_STRUCTURE_STORAGE_BIT_KHR = 0x00100000
VK_FORMAT_FEATURE_2_ACCELERATION_STRUCTURE_VERTEX_BUFFER_BIT_KHR = 0x20000000
VK_DEBUG_REPORT_OBJECT_TYPE_ACCELERATION_STRUCTURE_KHR_EXT = 1000150000
VK_KHR_ray_tracing_pipeline = 1
VK_KHR_RAY_TRACING_PIPELINE_SPEC_VERSION = 1
VK_KHR_RAY_TRACING_PIPELINE_EXTENSION_NAME = "VK_KHR_ray_tracing_pipeline"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PIPELINE_FEATURES_KHR = 1000347000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PIPELINE_PROPERTIES_KHR = 1000347001
VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_CREATE_INFO_KHR = 1000347015
VK_STRUCTURE_TYPE_RAY_TRACING_SHADER_GROUP_CREATE_INFO_KHR = 1000347016
VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_INTERFACE_CREATE_INFO_KHR = 1000347018
VK_SHADER_STAGE_RAYGEN_BIT_KHR = 0x00000100
VK_SHADER_STAGE_ANY_HIT_BIT_KHR = 0x00000200
VK_SHADER_STAGE_CLOSEST_HIT_BIT_KHR = 0x00000400
VK_SHADER_STAGE_MISS_BIT_KHR = 0x00000800
VK_SHADER_STAGE_INTERSECTION_BIT_KHR = 0x00001000
VK_SHADER_STAGE_CALLABLE_BIT_KHR = 0x00002000
VK_PIPELINE_STAGE_RAY_TRACING_SHADER_BIT_KHR = 0x00200000
VK_BUFFER_USAGE_SHADER_BINDING_TABLE_BIT_KHR = 0x00000400
VK_PIPELINE_BIND_POINT_RAY_TRACING_KHR = 1000347000
VK_PIPELINE_CREATE_RAY_TRACING_NO_NULL_ANY_HIT_SHADERS_BIT_KHR = 0x00004000
VK_PIPELINE_CREATE_RAY_TRACING_NO_NULL_CLOSEST_HIT_SHADERS_BIT_KHR = 0x00008000
VK_PIPELINE_CREATE_RAY_TRACING_NO_NULL_MISS_SHADERS_BIT_KHR = 0x00010000
VK_PIPELINE_CREATE_RAY_TRACING_NO_NULL_INTERSECTION_SHADERS_BIT_KHR = 0x00020000
VK_PIPELINE_CREATE_RAY_TRACING_SKIP_TRIANGLES_BIT_KHR = 0x00001000
VK_PIPELINE_CREATE_RAY_TRACING_SKIP_AABBS_BIT_KHR = 0x00002000
VK_PIPELINE_CREATE_RAY_TRACING_SHADER_GROUP_HANDLE_CAPTURE_REPLAY_BIT_KHR = 0x00080000
VK_DYNAMIC_STATE_RAY_TRACING_PIPELINE_STACK_SIZE_KHR = 1000347000
VK_KHR_ray_query = 1
VK_KHR_RAY_QUERY_SPEC_VERSION = 1
VK_KHR_RAY_QUERY_EXTENSION_NAME = "VK_KHR_ray_query"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_QUERY_FEATURES_KHR = 1000348013
VK_NV_extension_152 = 1
VK_NV_EXTENSION_152_SPEC_VERSION = 0
VK_NV_EXTENSION_152_EXTENSION_NAME = "VK_NV_extension_152"
VK_NV_framebuffer_mixed_samples = 1
VK_NV_FRAMEBUFFER_MIXED_SAMPLES_SPEC_VERSION = 1
VK_NV_FRAMEBUFFER_MIXED_SAMPLES_EXTENSION_NAME = "VK_NV_framebuffer_mixed_samples"
VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_MODULATION_STATE_CREATE_INFO_NV = 1000152000
VK_NV_fill_rectangle = 1
VK_NV_FILL_RECTANGLE_SPEC_VERSION = 1
VK_NV_FILL_RECTANGLE_EXTENSION_NAME = "VK_NV_fill_rectangle"
VK_POLYGON_MODE_FILL_RECTANGLE_NV = 1000153000
VK_NV_shader_sm_builtins = 1
VK_NV_SHADER_SM_BUILTINS_SPEC_VERSION = 1
VK_NV_SHADER_SM_BUILTINS_EXTENSION_NAME = "VK_NV_shader_sm_builtins"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_SM_BUILTINS_FEATURES_NV = 1000154000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_SM_BUILTINS_PROPERTIES_NV = 1000154001
VK_EXT_post_depth_coverage = 1
VK_EXT_POST_DEPTH_COVERAGE_SPEC_VERSION = 1
VK_EXT_POST_DEPTH_COVERAGE_EXTENSION_NAME = "VK_EXT_post_depth_coverage"
VK_KHR_sampler_ycbcr_conversion = 1
VK_KHR_SAMPLER_YCBCR_CONVERSION_SPEC_VERSION = 14
VK_KHR_SAMPLER_YCBCR_CONVERSION_EXTENSION_NAME = "VK_KHR_sampler_ycbcr_conversion"
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_INFO_KHR = None
VK_STRUCTURE_TYPE_BIND_IMAGE_PLANE_MEMORY_INFO_KHR = None
VK_STRUCTURE_TYPE_IMAGE_PLANE_MEMORY_REQUIREMENTS_INFO_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_YCBCR_CONVERSION_FEATURES_KHR = None
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_IMAGE_FORMAT_PROPERTIES_KHR = None
VK_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION_KHR = None
VK_FORMAT_G8B8G8R8_422_UNORM_KHR = None
VK_FORMAT_B8G8R8G8_422_UNORM_KHR = None
VK_FORMAT_G8_B8_R8_3PLANE_420_UNORM_KHR = None
VK_FORMAT_G8_B8R8_2PLANE_420_UNORM_KHR = None
VK_FORMAT_G8_B8_R8_3PLANE_422_UNORM_KHR = None
VK_FORMAT_G8_B8R8_2PLANE_422_UNORM_KHR = None
VK_FORMAT_G8_B8_R8_3PLANE_444_UNORM_KHR = None
VK_FORMAT_R10X6_UNORM_PACK16_KHR = None
VK_FORMAT_R10X6G10X6_UNORM_2PACK16_KHR = None
VK_FORMAT_R10X6G10X6B10X6A10X6_UNORM_4PACK16_KHR = None
VK_FORMAT_G10X6B10X6G10X6R10X6_422_UNORM_4PACK16_KHR = None
VK_FORMAT_B10X6G10X6R10X6G10X6_422_UNORM_4PACK16_KHR = None
VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_420_UNORM_3PACK16_KHR = None
VK_FORMAT_G10X6_B10X6R10X6_2PLANE_420_UNORM_3PACK16_KHR = None
VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_422_UNORM_3PACK16_KHR = None
VK_FORMAT_G10X6_B10X6R10X6_2PLANE_422_UNORM_3PACK16_KHR = None
VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_444_UNORM_3PACK16_KHR = None
VK_FORMAT_R12X4_UNORM_PACK16_KHR = None
VK_FORMAT_R12X4G12X4_UNORM_2PACK16_KHR = None
VK_FORMAT_R12X4G12X4B12X4A12X4_UNORM_4PACK16_KHR = None
VK_FORMAT_G12X4B12X4G12X4R12X4_422_UNORM_4PACK16_KHR = None
VK_FORMAT_B12X4G12X4R12X4G12X4_422_UNORM_4PACK16_KHR = None
VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_420_UNORM_3PACK16_KHR = None
VK_FORMAT_G12X4_B12X4R12X4_2PLANE_420_UNORM_3PACK16_KHR = None
VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_422_UNORM_3PACK16_KHR = None
VK_FORMAT_G12X4_B12X4R12X4_2PLANE_422_UNORM_3PACK16_KHR = None
VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_444_UNORM_3PACK16_KHR = None
VK_FORMAT_G16B16G16R16_422_UNORM_KHR = None
VK_FORMAT_B16G16R16G16_422_UNORM_KHR = None
VK_FORMAT_G16_B16_R16_3PLANE_420_UNORM_KHR = None
VK_FORMAT_G16_B16R16_2PLANE_420_UNORM_KHR = None
VK_FORMAT_G16_B16_R16_3PLANE_422_UNORM_KHR = None
VK_FORMAT_G16_B16R16_2PLANE_422_UNORM_KHR = None
VK_FORMAT_G16_B16_R16_3PLANE_444_UNORM_KHR = None
VK_IMAGE_ASPECT_PLANE_0_BIT_KHR = None
VK_IMAGE_ASPECT_PLANE_1_BIT_KHR = None
VK_IMAGE_ASPECT_PLANE_2_BIT_KHR = None
VK_IMAGE_CREATE_DISJOINT_BIT_KHR = None
VK_FORMAT_FEATURE_MIDPOINT_CHROMA_SAMPLES_BIT_KHR = None
VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_LINEAR_FILTER_BIT_KHR = None
VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_SEPARATE_RECONSTRUCTION_FILTER_BIT_KHR = None
VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_BIT_KHR = None
VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_FORCEABLE_BIT_KHR = None
VK_FORMAT_FEATURE_DISJOINT_BIT_KHR = None
VK_FORMAT_FEATURE_COSITED_CHROMA_SAMPLES_BIT_KHR = None
VK_SAMPLER_YCBCR_MODEL_CONVERSION_RGB_IDENTITY_KHR = None
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_IDENTITY_KHR = None
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_709_KHR = None
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_601_KHR = None
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_2020_KHR = None
VK_SAMPLER_YCBCR_RANGE_ITU_FULL_KHR = None
VK_SAMPLER_YCBCR_RANGE_ITU_NARROW_KHR = None
VK_CHROMA_LOCATION_COSITED_EVEN_KHR = None
VK_CHROMA_LOCATION_MIDPOINT_KHR = None
VK_DEBUG_REPORT_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION_KHR_EXT = None
VK_KHR_bind_memory2 = 1
VK_KHR_BIND_MEMORY_2_SPEC_VERSION = 1
VK_KHR_BIND_MEMORY_2_EXTENSION_NAME = "VK_KHR_bind_memory2"
VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_INFO_KHR = None
VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_INFO_KHR = None
VK_IMAGE_CREATE_ALIAS_BIT_KHR = None
VK_EXT_image_drm_format_modifier = 1
VK_EXT_IMAGE_DRM_FORMAT_MODIFIER_SPEC_VERSION = 2
VK_EXT_IMAGE_DRM_FORMAT_MODIFIER_EXTENSION_NAME = "VK_EXT_image_drm_format_modifier"
VK_STRUCTURE_TYPE_DRM_FORMAT_MODIFIER_PROPERTIES_LIST_EXT = 1000158000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_DRM_FORMAT_MODIFIER_INFO_EXT = 1000158002
VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_LIST_CREATE_INFO_EXT = 1000158003
VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_EXPLICIT_CREATE_INFO_EXT = 1000158004
VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_PROPERTIES_EXT = 1000158005
VK_IMAGE_TILING_DRM_FORMAT_MODIFIER_EXT = 1000158000
VK_IMAGE_ASPECT_MEMORY_PLANE_0_BIT_EXT = 0x00000080
VK_IMAGE_ASPECT_MEMORY_PLANE_1_BIT_EXT = 0x00000100
VK_IMAGE_ASPECT_MEMORY_PLANE_2_BIT_EXT = 0x00000200
VK_IMAGE_ASPECT_MEMORY_PLANE_3_BIT_EXT = 0x00000400
VK_STRUCTURE_TYPE_DRM_FORMAT_MODIFIER_PROPERTIES_LIST_2_EXT = 1000158006
VK_EXT_extension_160 = 1
VK_EXT_EXTENSION_160_SPEC_VERSION = 0
VK_EXT_EXTENSION_160_EXTENSION_NAME = "VK_EXT_extension_160"
VK_EXT_validation_cache = 1
VK_EXT_VALIDATION_CACHE_SPEC_VERSION = 1
VK_EXT_VALIDATION_CACHE_EXTENSION_NAME = "VK_EXT_validation_cache"
VK_STRUCTURE_TYPE_VALIDATION_CACHE_CREATE_INFO_EXT = 1000160000
VK_STRUCTURE_TYPE_SHADER_MODULE_VALIDATION_CACHE_CREATE_INFO_EXT = 1000160001
VK_OBJECT_TYPE_VALIDATION_CACHE_EXT = 1000160000
VK_EXT_descriptor_indexing = 1
VK_EXT_DESCRIPTOR_INDEXING_SPEC_VERSION = 2
VK_EXT_DESCRIPTOR_INDEXING_EXTENSION_NAME = "VK_EXT_descriptor_indexing"
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_BINDING_FLAGS_CREATE_INFO_EXT = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_FEATURES_EXT = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_PROPERTIES_EXT = None
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_ALLOCATE_INFO_EXT = None
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_LAYOUT_SUPPORT_EXT = None
VK_DESCRIPTOR_BINDING_UPDATE_AFTER_BIND_BIT_EXT = None
VK_DESCRIPTOR_BINDING_UPDATE_UNUSED_WHILE_PENDING_BIT_EXT = None
VK_DESCRIPTOR_BINDING_PARTIALLY_BOUND_BIT_EXT = None
VK_DESCRIPTOR_BINDING_VARIABLE_DESCRIPTOR_COUNT_BIT_EXT = None
VK_DESCRIPTOR_POOL_CREATE_UPDATE_AFTER_BIND_BIT_EXT = None
VK_DESCRIPTOR_SET_LAYOUT_CREATE_UPDATE_AFTER_BIND_POOL_BIT_EXT = None
VK_EXT_shader_viewport_index_layer = 1
VK_EXT_SHADER_VIEWPORT_INDEX_LAYER_SPEC_VERSION = 1
VK_EXT_SHADER_VIEWPORT_INDEX_LAYER_EXTENSION_NAME = "VK_EXT_shader_viewport_index_layer"
VK_KHR_portability_subset = 1
VK_KHR_PORTABILITY_SUBSET_SPEC_VERSION = 1
VK_KHR_PORTABILITY_SUBSET_EXTENSION_NAME = "VK_KHR_portability_subset"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PORTABILITY_SUBSET_FEATURES_KHR = 1000163000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PORTABILITY_SUBSET_PROPERTIES_KHR = 1000163001
VK_NV_shading_rate_image = 1
VK_NV_SHADING_RATE_IMAGE_SPEC_VERSION = 3
VK_NV_SHADING_RATE_IMAGE_EXTENSION_NAME = "VK_NV_shading_rate_image"
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SHADING_RATE_IMAGE_STATE_CREATE_INFO_NV = 1000164000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_FEATURES_NV = 1000164001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_PROPERTIES_NV = 1000164002
VK_IMAGE_LAYOUT_SHADING_RATE_OPTIMAL_NV = None
VK_DYNAMIC_STATE_VIEWPORT_SHADING_RATE_PALETTE_NV = 1000164004
VK_ACCESS_SHADING_RATE_IMAGE_READ_BIT_NV = None
VK_IMAGE_USAGE_SHADING_RATE_IMAGE_BIT_NV = None
VK_PIPELINE_STAGE_SHADING_RATE_IMAGE_BIT_NV = None
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_COARSE_SAMPLE_ORDER_STATE_CREATE_INFO_NV = 1000164005
VK_DYNAMIC_STATE_VIEWPORT_COARSE_SAMPLE_ORDER_NV = 1000164006
VK_NV_ray_tracing = 1
VK_NV_RAY_TRACING_SPEC_VERSION = 3
VK_NV_RAY_TRACING_EXTENSION_NAME = "VK_NV_ray_tracing"
VK_SHADER_UNUSED_NV = None
VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_CREATE_INFO_NV = 1000165000
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_CREATE_INFO_NV = 1000165001
VK_STRUCTURE_TYPE_GEOMETRY_NV = 1000165003
VK_STRUCTURE_TYPE_GEOMETRY_TRIANGLES_NV = 1000165004
VK_STRUCTURE_TYPE_GEOMETRY_AABB_NV = 1000165005
VK_STRUCTURE_TYPE_BIND_ACCELERATION_STRUCTURE_MEMORY_INFO_NV = 1000165006
VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_ACCELERATION_STRUCTURE_NV = 1000165007
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_INFO_NV = 1000165008
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PROPERTIES_NV = 1000165009
VK_STRUCTURE_TYPE_RAY_TRACING_SHADER_GROUP_CREATE_INFO_NV = 1000165011
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_INFO_NV = 1000165012
VK_SHADER_STAGE_RAYGEN_BIT_NV = None
VK_SHADER_STAGE_ANY_HIT_BIT_NV = None
VK_SHADER_STAGE_CLOSEST_HIT_BIT_NV = None
VK_SHADER_STAGE_MISS_BIT_NV = None
VK_SHADER_STAGE_INTERSECTION_BIT_NV = None
VK_SHADER_STAGE_CALLABLE_BIT_NV = None
VK_PIPELINE_STAGE_RAY_TRACING_SHADER_BIT_NV = None
VK_PIPELINE_STAGE_ACCELERATION_STRUCTURE_BUILD_BIT_NV = None
VK_BUFFER_USAGE_RAY_TRACING_BIT_NV = None
VK_PIPELINE_BIND_POINT_RAY_TRACING_NV = None
VK_DESCRIPTOR_TYPE_ACCELERATION_STRUCTURE_NV = 1000165000
VK_ACCESS_ACCELERATION_STRUCTURE_READ_BIT_NV = None
VK_ACCESS_ACCELERATION_STRUCTURE_WRITE_BIT_NV = None
VK_QUERY_TYPE_ACCELERATION_STRUCTURE_COMPACTED_SIZE_NV = 1000165000
VK_PIPELINE_CREATE_DEFER_COMPILE_BIT_NV = 0x00000020
VK_OBJECT_TYPE_ACCELERATION_STRUCTURE_NV = 1000165000
VK_INDEX_TYPE_NONE_NV = None
VK_RAY_TRACING_SHADER_GROUP_TYPE_GENERAL_NV = None
VK_RAY_TRACING_SHADER_GROUP_TYPE_TRIANGLES_HIT_GROUP_NV = None
VK_RAY_TRACING_SHADER_GROUP_TYPE_PROCEDURAL_HIT_GROUP_NV = None
VK_GEOMETRY_TYPE_TRIANGLES_NV = None
VK_GEOMETRY_TYPE_AABBS_NV = None
VK_ACCELERATION_STRUCTURE_TYPE_TOP_LEVEL_NV = None
VK_ACCELERATION_STRUCTURE_TYPE_BOTTOM_LEVEL_NV = None
VK_GEOMETRY_OPAQUE_BIT_NV = None
VK_GEOMETRY_NO_DUPLICATE_ANY_HIT_INVOCATION_BIT_NV = None
VK_GEOMETRY_INSTANCE_TRIANGLE_CULL_DISABLE_BIT_NV = None
VK_GEOMETRY_INSTANCE_TRIANGLE_FRONT_COUNTERCLOCKWISE_BIT_NV = None
VK_GEOMETRY_INSTANCE_FORCE_OPAQUE_BIT_NV = None
VK_GEOMETRY_INSTANCE_FORCE_NO_OPAQUE_BIT_NV = None
VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_UPDATE_BIT_NV = None
VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_COMPACTION_BIT_NV = None
VK_BUILD_ACCELERATION_STRUCTURE_PREFER_FAST_TRACE_BIT_NV = None
VK_BUILD_ACCELERATION_STRUCTURE_PREFER_FAST_BUILD_BIT_NV = None
VK_BUILD_ACCELERATION_STRUCTURE_LOW_MEMORY_BIT_NV = None
VK_COPY_ACCELERATION_STRUCTURE_MODE_CLONE_NV = None
VK_COPY_ACCELERATION_STRUCTURE_MODE_COMPACT_NV = None
VK_DEBUG_REPORT_OBJECT_TYPE_ACCELERATION_STRUCTURE_NV_EXT = 1000165000
VK_NV_representative_fragment_test = 1
VK_NV_REPRESENTATIVE_FRAGMENT_TEST_SPEC_VERSION = 2
VK_NV_REPRESENTATIVE_FRAGMENT_TEST_EXTENSION_NAME = "VK_NV_representative_fragment_test"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_REPRESENTATIVE_FRAGMENT_TEST_FEATURES_NV = 1000166000
VK_STRUCTURE_TYPE_PIPELINE_REPRESENTATIVE_FRAGMENT_TEST_STATE_CREATE_INFO_NV = 1000166001
VK_NV_extension_168 = 1
VK_NV_EXTENSION_168_SPEC_VERSION = 0
VK_NV_EXTENSION_168_EXTENSION_NAME = "VK_NV_extension_168"
VK_KHR_maintenance3 = 1
VK_KHR_MAINTENANCE_3_SPEC_VERSION = 1
VK_KHR_MAINTENANCE_3_EXTENSION_NAME = "VK_KHR_maintenance3"
VK_KHR_MAINTENANCE3_SPEC_VERSION = None
VK_KHR_MAINTENANCE3_EXTENSION_NAME = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_3_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_SUPPORT_KHR = None
VK_KHR_draw_indirect_count = 1
VK_KHR_DRAW_INDIRECT_COUNT_SPEC_VERSION = 1
VK_KHR_DRAW_INDIRECT_COUNT_EXTENSION_NAME = "VK_KHR_draw_indirect_count"
VK_EXT_filter_cubic = 1
VK_EXT_FILTER_CUBIC_SPEC_VERSION = 3
VK_EXT_FILTER_CUBIC_EXTENSION_NAME = "VK_EXT_filter_cubic"
VK_FILTER_CUBIC_EXT = 1000170000
VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_CUBIC_BIT_EXT = 0x00002000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_VIEW_IMAGE_FORMAT_INFO_EXT = 1000170000
VK_STRUCTURE_TYPE_FILTER_CUBIC_IMAGE_VIEW_IMAGE_FORMAT_PROPERTIES_EXT = 1000170001
VK_QCOM_render_pass_shader_resolve = 1
VK_QCOM_RENDER_PASS_SHADER_RESOLVE_SPEC_VERSION = 4
VK_QCOM_RENDER_PASS_SHADER_RESOLVE_EXTENSION_NAME = "VK_QCOM_render_pass_shader_resolve"
VK_SUBPASS_DESCRIPTION_FRAGMENT_REGION_BIT_QCOM = 0x00000004
VK_SUBPASS_DESCRIPTION_SHADER_RESOLVE_BIT_QCOM = 0x00000008
VK_QCOM_extension_173 = 1
VK_QCOM_EXTENSION_173_SPEC_VERSION = 0
VK_QCOM_EXTENSION_173_EXTENSION_NAME = "VK_QCOM_extension_173"
VK_QCOM_extension_174 = 1
VK_QCOM_EXTENSION_174_SPEC_VERSION = 0
VK_QCOM_EXTENSION_174_EXTENSION_NAME = "VK_QCOM_extension_174"
VK_EXT_global_priority = 1
VK_EXT_GLOBAL_PRIORITY_SPEC_VERSION = 2
VK_EXT_GLOBAL_PRIORITY_EXTENSION_NAME = "VK_EXT_global_priority"
VK_STRUCTURE_TYPE_DEVICE_QUEUE_GLOBAL_PRIORITY_CREATE_INFO_EXT = None
VK_KHR_shader_subgroup_extended_types = 1
VK_KHR_SHADER_SUBGROUP_EXTENDED_TYPES_SPEC_VERSION = 1
VK_KHR_SHADER_SUBGROUP_EXTENDED_TYPES_EXTENSION_NAME = "VK_KHR_shader_subgroup_extended_types"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_SUBGROUP_EXTENDED_TYPES_FEATURES_KHR = None
VK_EXT_extension_177 = 1
VK_EXT_EXTENSION_177_SPEC_VERSION = 0
VK_EXT_EXTENSION_177_EXTENSION_NAME = "VK_EXT_extension_177"
VK_KHR_8bit_storage = 1
VK_KHR_8BIT_STORAGE_SPEC_VERSION = 1
VK_KHR_8BIT_STORAGE_EXTENSION_NAME = "VK_KHR_8bit_storage"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_8BIT_STORAGE_FEATURES_KHR = None
VK_EXT_external_memory_host = 1
VK_EXT_EXTERNAL_MEMORY_HOST_SPEC_VERSION = 1
VK_EXT_EXTERNAL_MEMORY_HOST_EXTENSION_NAME = "VK_EXT_external_memory_host"
VK_STRUCTURE_TYPE_IMPORT_MEMORY_HOST_POINTER_INFO_EXT = 1000178000
VK_STRUCTURE_TYPE_MEMORY_HOST_POINTER_PROPERTIES_EXT = 1000178001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_HOST_PROPERTIES_EXT = 1000178002
VK_EXTERNAL_MEMORY_HANDLE_TYPE_HOST_ALLOCATION_BIT_EXT = 0x00000080
VK_EXTERNAL_MEMORY_HANDLE_TYPE_HOST_MAPPED_FOREIGN_MEMORY_BIT_EXT = 0x00000100
VK_AMD_buffer_marker = 1
VK_AMD_BUFFER_MARKER_SPEC_VERSION = 1
VK_AMD_BUFFER_MARKER_EXTENSION_NAME = "VK_AMD_buffer_marker"
VK_KHR_shader_atomic_int64 = 1
VK_KHR_SHADER_ATOMIC_INT64_SPEC_VERSION = 1
VK_KHR_SHADER_ATOMIC_INT64_EXTENSION_NAME = "VK_KHR_shader_atomic_int64"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_INT64_FEATURES_KHR = None
VK_KHR_shader_clock = 1
VK_KHR_SHADER_CLOCK_SPEC_VERSION = 1
VK_KHR_SHADER_CLOCK_EXTENSION_NAME = "VK_KHR_shader_clock"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CLOCK_FEATURES_KHR = 1000181000
VK_AMD_extension_183 = 1
VK_AMD_EXTENSION_183_SPEC_VERSION = 0
VK_AMD_EXTENSION_183_EXTENSION_NAME = "VK_AMD_extension_183"
VK_AMD_pipeline_compiler_control = 1
VK_AMD_PIPELINE_COMPILER_CONTROL_SPEC_VERSION = 1
VK_AMD_PIPELINE_COMPILER_CONTROL_EXTENSION_NAME = "VK_AMD_pipeline_compiler_control"
VK_STRUCTURE_TYPE_PIPELINE_COMPILER_CONTROL_CREATE_INFO_AMD = 1000183000
VK_EXT_calibrated_timestamps = 1
VK_EXT_CALIBRATED_TIMESTAMPS_SPEC_VERSION = 2
VK_EXT_CALIBRATED_TIMESTAMPS_EXTENSION_NAME = "VK_EXT_calibrated_timestamps"
VK_STRUCTURE_TYPE_CALIBRATED_TIMESTAMP_INFO_EXT = None
VK_TIME_DOMAIN_DEVICE_EXT = None
VK_TIME_DOMAIN_CLOCK_MONOTONIC_EXT = None
VK_TIME_DOMAIN_CLOCK_MONOTONIC_RAW_EXT = None
VK_TIME_DOMAIN_QUERY_PERFORMANCE_COUNTER_EXT = None
VK_AMD_shader_core_properties = 1
VK_AMD_SHADER_CORE_PROPERTIES_SPEC_VERSION = 2
VK_AMD_SHADER_CORE_PROPERTIES_EXTENSION_NAME = "VK_AMD_shader_core_properties"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_PROPERTIES_AMD = 1000185000
VK_AMD_extension_187 = 1
VK_AMD_EXTENSION_187_SPEC_VERSION = 0
VK_AMD_EXTENSION_187_EXTENSION_NAME = "VK_AMD_extension_187"
VK_KHR_video_decode_h265 = 1
VK_KHR_VIDEO_DECODE_H265_SPEC_VERSION = 8
VK_KHR_VIDEO_DECODE_H265_EXTENSION_NAME = "VK_KHR_video_decode_h265"
VK_STRUCTURE_TYPE_VIDEO_DECODE_H265_CAPABILITIES_KHR = 1000187000
VK_STRUCTURE_TYPE_VIDEO_DECODE_H265_SESSION_PARAMETERS_CREATE_INFO_KHR = 1000187001
VK_STRUCTURE_TYPE_VIDEO_DECODE_H265_SESSION_PARAMETERS_ADD_INFO_KHR = 1000187002
VK_STRUCTURE_TYPE_VIDEO_DECODE_H265_PROFILE_INFO_KHR = 1000187003
VK_STRUCTURE_TYPE_VIDEO_DECODE_H265_PICTURE_INFO_KHR = 1000187004
VK_STRUCTURE_TYPE_VIDEO_DECODE_H265_DPB_SLOT_INFO_KHR = 1000187005
VK_VIDEO_CODEC_OPERATION_DECODE_H265_BIT_KHR = 0x00000002
VK_KHR_global_priority = 1
VK_KHR_GLOBAL_PRIORITY_SPEC_VERSION = 1
VK_KHR_GLOBAL_PRIORITY_EXTENSION_NAME = "VK_KHR_global_priority"
VK_STRUCTURE_TYPE_DEVICE_QUEUE_GLOBAL_PRIORITY_CREATE_INFO_KHR = 1000188000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GLOBAL_PRIORITY_QUERY_FEATURES_KHR = 1000188000
VK_STRUCTURE_TYPE_QUEUE_FAMILY_GLOBAL_PRIORITY_PROPERTIES_KHR = 1000188001
VK_AMD_memory_overallocation_behavior = 1
VK_AMD_MEMORY_OVERALLOCATION_BEHAVIOR_SPEC_VERSION = 1
VK_AMD_MEMORY_OVERALLOCATION_BEHAVIOR_EXTENSION_NAME = "VK_AMD_memory_overallocation_behavior"
VK_STRUCTURE_TYPE_DEVICE_MEMORY_OVERALLOCATION_CREATE_INFO_AMD = 1000189000
VK_EXT_vertex_attribute_divisor = 1
VK_EXT_VERTEX_ATTRIBUTE_DIVISOR_SPEC_VERSION = 3
VK_EXT_VERTEX_ATTRIBUTE_DIVISOR_EXTENSION_NAME = "VK_EXT_vertex_attribute_divisor"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_PROPERTIES_EXT = 1000190000
VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_DIVISOR_STATE_CREATE_INFO_EXT = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_FEATURES_EXT = None
VK_GGP_frame_token = 1
VK_GGP_FRAME_TOKEN_SPEC_VERSION = 1
VK_GGP_FRAME_TOKEN_EXTENSION_NAME = "VK_GGP_frame_token"
VK_STRUCTURE_TYPE_PRESENT_FRAME_TOKEN_GGP = 1000191000
VK_EXT_pipeline_creation_feedback = 1
VK_EXT_PIPELINE_CREATION_FEEDBACK_SPEC_VERSION = 1
VK_EXT_PIPELINE_CREATION_FEEDBACK_EXTENSION_NAME = "VK_EXT_pipeline_creation_feedback"
VK_STRUCTURE_TYPE_PIPELINE_CREATION_FEEDBACK_CREATE_INFO_EXT = None
VK_GOOGLE_extension_194 = 1
VK_GOOGLE_EXTENSION_194_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_194_EXTENSION_NAME = "VK_GOOGLE_extension_194"
VK_GOOGLE_extension_195 = 1
VK_GOOGLE_EXTENSION_195_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_195_EXTENSION_NAME = "VK_GOOGLE_extension_195"
VK_GOOGLE_extension_196 = 1
VK_GOOGLE_EXTENSION_196_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_196_EXTENSION_NAME = "VK_GOOGLE_extension_196"
VK_KHR_driver_properties = 1
VK_KHR_DRIVER_PROPERTIES_SPEC_VERSION = 1
VK_KHR_DRIVER_PROPERTIES_EXTENSION_NAME = "VK_KHR_driver_properties"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DRIVER_PROPERTIES_KHR = None
VK_MAX_DRIVER_NAME_SIZE_KHR = None
VK_MAX_DRIVER_INFO_SIZE_KHR = None
VK_DRIVER_ID_AMD_PROPRIETARY_KHR = None
VK_DRIVER_ID_AMD_OPEN_SOURCE_KHR = None
VK_DRIVER_ID_MESA_RADV_KHR = None
VK_DRIVER_ID_NVIDIA_PROPRIETARY_KHR = None
VK_DRIVER_ID_INTEL_PROPRIETARY_WINDOWS_KHR = None
VK_DRIVER_ID_INTEL_OPEN_SOURCE_MESA_KHR = None
VK_DRIVER_ID_IMAGINATION_PROPRIETARY_KHR = None
VK_DRIVER_ID_QUALCOMM_PROPRIETARY_KHR = None
VK_DRIVER_ID_ARM_PROPRIETARY_KHR = None
VK_DRIVER_ID_GOOGLE_SWIFTSHADER_KHR = None
VK_DRIVER_ID_GGP_PROPRIETARY_KHR = None
VK_DRIVER_ID_BROADCOM_PROPRIETARY_KHR = None
VK_KHR_shader_float_controls = 1
VK_KHR_SHADER_FLOAT_CONTROLS_SPEC_VERSION = 4
VK_KHR_SHADER_FLOAT_CONTROLS_EXTENSION_NAME = "VK_KHR_shader_float_controls"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT_CONTROLS_PROPERTIES_KHR = None
VK_SHADER_FLOAT_CONTROLS_INDEPENDENCE_32_BIT_ONLY_KHR = None
VK_SHADER_FLOAT_CONTROLS_INDEPENDENCE_ALL_KHR = None
VK_SHADER_FLOAT_CONTROLS_INDEPENDENCE_NONE_KHR = None
VK_NV_shader_subgroup_partitioned = 1
VK_NV_SHADER_SUBGROUP_PARTITIONED_SPEC_VERSION = 1
VK_NV_SHADER_SUBGROUP_PARTITIONED_EXTENSION_NAME = "VK_NV_shader_subgroup_partitioned"
VK_SUBGROUP_FEATURE_PARTITIONED_BIT_NV = 0x00000100
VK_KHR_depth_stencil_resolve = 1
VK_KHR_DEPTH_STENCIL_RESOLVE_SPEC_VERSION = 1
VK_KHR_DEPTH_STENCIL_RESOLVE_EXTENSION_NAME = "VK_KHR_depth_stencil_resolve"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_STENCIL_RESOLVE_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_DEPTH_STENCIL_RESOLVE_KHR = None
VK_RESOLVE_MODE_NONE_KHR = None
VK_RESOLVE_MODE_SAMPLE_ZERO_BIT_KHR = None
VK_RESOLVE_MODE_AVERAGE_BIT_KHR = None
VK_RESOLVE_MODE_MIN_BIT_KHR = None
VK_RESOLVE_MODE_MAX_BIT_KHR = None
VK_KHR_swapchain_mutable_format = 1
VK_KHR_SWAPCHAIN_MUTABLE_FORMAT_SPEC_VERSION = 1
VK_KHR_SWAPCHAIN_MUTABLE_FORMAT_EXTENSION_NAME = "VK_KHR_swapchain_mutable_format"
VK_SWAPCHAIN_CREATE_MUTABLE_FORMAT_BIT_KHR = 0x00000004
VK_NV_compute_shader_derivatives = 1
VK_NV_COMPUTE_SHADER_DERIVATIVES_SPEC_VERSION = 1
VK_NV_COMPUTE_SHADER_DERIVATIVES_EXTENSION_NAME = "VK_NV_compute_shader_derivatives"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COMPUTE_SHADER_DERIVATIVES_FEATURES_NV = 1000201000
VK_NV_mesh_shader = 1
VK_NV_MESH_SHADER_SPEC_VERSION = 1
VK_NV_MESH_SHADER_EXTENSION_NAME = "VK_NV_mesh_shader"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_FEATURES_NV = 1000202000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_PROPERTIES_NV = 1000202001
VK_SHADER_STAGE_TASK_BIT_NV = None
VK_SHADER_STAGE_MESH_BIT_NV = None
VK_PIPELINE_STAGE_TASK_SHADER_BIT_NV = None
VK_PIPELINE_STAGE_MESH_SHADER_BIT_NV = None
VK_NV_fragment_shader_barycentric = 1
VK_NV_FRAGMENT_SHADER_BARYCENTRIC_SPEC_VERSION = 1
VK_NV_FRAGMENT_SHADER_BARYCENTRIC_EXTENSION_NAME = "VK_NV_fragment_shader_barycentric"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_BARYCENTRIC_FEATURES_NV = None
VK_NV_shader_image_footprint = 1
VK_NV_SHADER_IMAGE_FOOTPRINT_SPEC_VERSION = 2
VK_NV_SHADER_IMAGE_FOOTPRINT_EXTENSION_NAME = "VK_NV_shader_image_footprint"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_IMAGE_FOOTPRINT_FEATURES_NV = 1000204000
VK_NV_scissor_exclusive = 1
VK_NV_SCISSOR_EXCLUSIVE_SPEC_VERSION = 2
VK_NV_SCISSOR_EXCLUSIVE_EXTENSION_NAME = "VK_NV_scissor_exclusive"
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_EXCLUSIVE_SCISSOR_STATE_CREATE_INFO_NV = 1000205000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXCLUSIVE_SCISSOR_FEATURES_NV = 1000205002
VK_DYNAMIC_STATE_EXCLUSIVE_SCISSOR_ENABLE_NV = 1000205000
VK_DYNAMIC_STATE_EXCLUSIVE_SCISSOR_NV = 1000205001
VK_NV_device_diagnostic_checkpoints = 1
VK_NV_DEVICE_DIAGNOSTIC_CHECKPOINTS_SPEC_VERSION = 2
VK_NV_DEVICE_DIAGNOSTIC_CHECKPOINTS_EXTENSION_NAME = "VK_NV_device_diagnostic_checkpoints"
VK_STRUCTURE_TYPE_CHECKPOINT_DATA_NV = 1000206000
VK_STRUCTURE_TYPE_QUEUE_FAMILY_CHECKPOINT_PROPERTIES_NV = 1000206001
VK_KHR_timeline_semaphore = 1
VK_KHR_TIMELINE_SEMAPHORE_SPEC_VERSION = 2
VK_KHR_TIMELINE_SEMAPHORE_EXTENSION_NAME = "VK_KHR_timeline_semaphore"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TIMELINE_SEMAPHORE_FEATURES_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TIMELINE_SEMAPHORE_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_SEMAPHORE_TYPE_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_TIMELINE_SEMAPHORE_SUBMIT_INFO_KHR = None
VK_STRUCTURE_TYPE_SEMAPHORE_WAIT_INFO_KHR = None
VK_STRUCTURE_TYPE_SEMAPHORE_SIGNAL_INFO_KHR = None
VK_SEMAPHORE_TYPE_BINARY_KHR = None
VK_SEMAPHORE_TYPE_TIMELINE_KHR = None
VK_SEMAPHORE_WAIT_ANY_BIT_KHR = None
VK_KHR_extension_209 = 1
VK_KHR_EXTENSION_209_SPEC_VERSION = 0
VK_KHR_EXTENSION_209_EXTENSION_NAME = "VK_KHR_extension_209"
VK_INTEL_shader_integer_functions2 = 1
VK_INTEL_SHADER_INTEGER_FUNCTIONS_2_SPEC_VERSION = 1
VK_INTEL_SHADER_INTEGER_FUNCTIONS_2_EXTENSION_NAME = "VK_INTEL_shader_integer_functions2"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_INTEGER_FUNCTIONS_2_FEATURES_INTEL = 1000209000
VK_INTEL_performance_query = 1
VK_INTEL_PERFORMANCE_QUERY_SPEC_VERSION = 2
VK_INTEL_PERFORMANCE_QUERY_EXTENSION_NAME = "VK_INTEL_performance_query"
VK_STRUCTURE_TYPE_QUERY_POOL_PERFORMANCE_QUERY_CREATE_INFO_INTEL = 1000210000
VK_STRUCTURE_TYPE_QUERY_POOL_CREATE_INFO_INTEL = None
VK_STRUCTURE_TYPE_INITIALIZE_PERFORMANCE_API_INFO_INTEL = 1000210001
VK_STRUCTURE_TYPE_PERFORMANCE_MARKER_INFO_INTEL = 1000210002
VK_STRUCTURE_TYPE_PERFORMANCE_STREAM_MARKER_INFO_INTEL = 1000210003
VK_STRUCTURE_TYPE_PERFORMANCE_OVERRIDE_INFO_INTEL = 1000210004
VK_STRUCTURE_TYPE_PERFORMANCE_CONFIGURATION_ACQUIRE_INFO_INTEL = 1000210005
VK_QUERY_TYPE_PERFORMANCE_QUERY_INTEL = 1000210000
VK_OBJECT_TYPE_PERFORMANCE_CONFIGURATION_INTEL = 1000210000
VK_KHR_vulkan_memory_model = 1
VK_KHR_VULKAN_MEMORY_MODEL_SPEC_VERSION = 3
VK_KHR_VULKAN_MEMORY_MODEL_EXTENSION_NAME = "VK_KHR_vulkan_memory_model"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_MEMORY_MODEL_FEATURES_KHR = None
VK_EXT_pci_bus_info = 1
VK_EXT_PCI_BUS_INFO_SPEC_VERSION = 2
VK_EXT_PCI_BUS_INFO_EXTENSION_NAME = "VK_EXT_pci_bus_info"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PCI_BUS_INFO_PROPERTIES_EXT = 1000212000
VK_AMD_display_native_hdr = 1
VK_AMD_DISPLAY_NATIVE_HDR_SPEC_VERSION = 1
VK_AMD_DISPLAY_NATIVE_HDR_EXTENSION_NAME = "VK_AMD_display_native_hdr"
VK_STRUCTURE_TYPE_DISPLAY_NATIVE_HDR_SURFACE_CAPABILITIES_AMD = 1000213000
VK_STRUCTURE_TYPE_SWAPCHAIN_DISPLAY_NATIVE_HDR_CREATE_INFO_AMD = 1000213001
VK_COLOR_SPACE_DISPLAY_NATIVE_AMD = 1000213000
VK_FUCHSIA_imagepipe_surface = 1
VK_FUCHSIA_IMAGEPIPE_SURFACE_SPEC_VERSION = 1
VK_FUCHSIA_IMAGEPIPE_SURFACE_EXTENSION_NAME = "VK_FUCHSIA_imagepipe_surface"
VK_STRUCTURE_TYPE_IMAGEPIPE_SURFACE_CREATE_INFO_FUCHSIA = 1000214000
VK_KHR_shader_terminate_invocation = 1
VK_KHR_SHADER_TERMINATE_INVOCATION_SPEC_VERSION = 1
VK_KHR_SHADER_TERMINATE_INVOCATION_EXTENSION_NAME = "VK_KHR_shader_terminate_invocation"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_TERMINATE_INVOCATION_FEATURES_KHR = None
VK_GOOGLE_extension_217 = 1
VK_GOOGLE_EXTENSION_217_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_217_EXTENSION_NAME = "VK_GOOGLE_extension_217"
VK_EXT_metal_surface = 1
VK_EXT_METAL_SURFACE_SPEC_VERSION = 1
VK_EXT_METAL_SURFACE_EXTENSION_NAME = "VK_EXT_metal_surface"
VK_STRUCTURE_TYPE_METAL_SURFACE_CREATE_INFO_EXT = 1000217000
VK_EXT_fragment_density_map = 1
VK_EXT_FRAGMENT_DENSITY_MAP_SPEC_VERSION = 2
VK_EXT_FRAGMENT_DENSITY_MAP_EXTENSION_NAME = "VK_EXT_fragment_density_map"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_FEATURES_EXT = 1000218000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_PROPERTIES_EXT = 1000218001
VK_STRUCTURE_TYPE_RENDER_PASS_FRAGMENT_DENSITY_MAP_CREATE_INFO_EXT = 1000218002
VK_IMAGE_CREATE_SUBSAMPLED_BIT_EXT = 0x00004000
VK_IMAGE_LAYOUT_FRAGMENT_DENSITY_MAP_OPTIMAL_EXT = 1000218000
VK_ACCESS_FRAGMENT_DENSITY_MAP_READ_BIT_EXT = 0x01000000
VK_FORMAT_FEATURE_FRAGMENT_DENSITY_MAP_BIT_EXT = 0x01000000
VK_IMAGE_USAGE_FRAGMENT_DENSITY_MAP_BIT_EXT = 0x00000200
VK_IMAGE_VIEW_CREATE_FRAGMENT_DENSITY_MAP_DYNAMIC_BIT_EXT = 0x00000001
VK_PIPELINE_STAGE_FRAGMENT_DENSITY_PROCESS_BIT_EXT = 0x00800000
VK_SAMPLER_CREATE_SUBSAMPLED_BIT_EXT = 0x00000001
VK_SAMPLER_CREATE_SUBSAMPLED_COARSE_RECONSTRUCTION_BIT_EXT = 0x00000002
VK_FORMAT_FEATURE_2_FRAGMENT_DENSITY_MAP_BIT_EXT = 0x01000000
VK_EXT_extension_220 = 1
VK_EXT_EXTENSION_220_SPEC_VERSION = 0
VK_EXT_EXTENSION_220_EXTENSION_NAME = "VK_EXT_extension_220"
VK_KHR_extension_221 = 1
VK_KHR_EXTENSION_221_SPEC_VERSION = 0
VK_KHR_EXTENSION_221_EXTENSION_NAME = "VK_KHR_extension_221"
VK_RENDER_PASS_CREATE_RESERVED_0_BIT_KHR = 0x00000001
VK_EXT_scalar_block_layout = 1
VK_EXT_SCALAR_BLOCK_LAYOUT_SPEC_VERSION = 1
VK_EXT_SCALAR_BLOCK_LAYOUT_EXTENSION_NAME = "VK_EXT_scalar_block_layout"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SCALAR_BLOCK_LAYOUT_FEATURES_EXT = None
VK_EXT_extension_223 = 1
VK_EXT_EXTENSION_223_SPEC_VERSION = 0
VK_EXT_EXTENSION_223_EXTENSION_NAME = "VK_EXT_extension_223"
VK_GOOGLE_hlsl_functionality1 = 1
VK_GOOGLE_HLSL_FUNCTIONALITY_1_SPEC_VERSION = 1
VK_GOOGLE_HLSL_FUNCTIONALITY_1_EXTENSION_NAME = "VK_GOOGLE_hlsl_functionality1"
VK_GOOGLE_HLSL_FUNCTIONALITY1_SPEC_VERSION = None
VK_GOOGLE_HLSL_FUNCTIONALITY1_EXTENSION_NAME = None
VK_GOOGLE_decorate_string = 1
VK_GOOGLE_DECORATE_STRING_SPEC_VERSION = 1
VK_GOOGLE_DECORATE_STRING_EXTENSION_NAME = "VK_GOOGLE_decorate_string"
VK_EXT_subgroup_size_control = 1
VK_EXT_SUBGROUP_SIZE_CONTROL_SPEC_VERSION = 2
VK_EXT_SUBGROUP_SIZE_CONTROL_EXTENSION_NAME = "VK_EXT_subgroup_size_control"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_SIZE_CONTROL_PROPERTIES_EXT = None
VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_REQUIRED_SUBGROUP_SIZE_CREATE_INFO_EXT = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_SIZE_CONTROL_FEATURES_EXT = None
VK_PIPELINE_SHADER_STAGE_CREATE_ALLOW_VARYING_SUBGROUP_SIZE_BIT_EXT = None
VK_PIPELINE_SHADER_STAGE_CREATE_REQUIRE_FULL_SUBGROUPS_BIT_EXT = None
VK_KHR_fragment_shading_rate = 1
VK_KHR_FRAGMENT_SHADING_RATE_SPEC_VERSION = 2
VK_KHR_FRAGMENT_SHADING_RATE_EXTENSION_NAME = "VK_KHR_fragment_shading_rate"
VK_IMAGE_LAYOUT_FRAGMENT_SHADING_RATE_ATTACHMENT_OPTIMAL_KHR = 1000226003
VK_DYNAMIC_STATE_FRAGMENT_SHADING_RATE_KHR = 1000226000
VK_STRUCTURE_TYPE_FRAGMENT_SHADING_RATE_ATTACHMENT_INFO_KHR = 1000226000
VK_STRUCTURE_TYPE_PIPELINE_FRAGMENT_SHADING_RATE_STATE_CREATE_INFO_KHR = 1000226001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_PROPERTIES_KHR = 1000226002
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_FEATURES_KHR = 1000226003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_KHR = 1000226004
VK_ACCESS_FRAGMENT_SHADING_RATE_ATTACHMENT_READ_BIT_KHR = 0x00800000
VK_IMAGE_USAGE_FRAGMENT_SHADING_RATE_ATTACHMENT_BIT_KHR = 0x00000100
VK_PIPELINE_STAGE_FRAGMENT_SHADING_RATE_ATTACHMENT_BIT_KHR = 0x00400000
VK_FORMAT_FEATURE_FRAGMENT_SHADING_RATE_ATTACHMENT_BIT_KHR = 0x40000000
VK_FORMAT_FEATURE_2_FRAGMENT_SHADING_RATE_ATTACHMENT_BIT_KHR = 0x40000000
VK_AMD_shader_core_properties2 = 1
VK_AMD_SHADER_CORE_PROPERTIES_2_SPEC_VERSION = 1
VK_AMD_SHADER_CORE_PROPERTIES_2_EXTENSION_NAME = "VK_AMD_shader_core_properties2"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_PROPERTIES_2_AMD = 1000227000
VK_AMD_extension_229 = 1
VK_AMD_EXTENSION_229_SPEC_VERSION = 0
VK_AMD_EXTENSION_229_EXTENSION_NAME = "VK_AMD_extension_229"
VK_AMD_device_coherent_memory = 1
VK_AMD_DEVICE_COHERENT_MEMORY_SPEC_VERSION = 1
VK_AMD_DEVICE_COHERENT_MEMORY_EXTENSION_NAME = "VK_AMD_device_coherent_memory"
VK_MEMORY_PROPERTY_DEVICE_COHERENT_BIT_AMD = 0x00000040
VK_MEMORY_PROPERTY_DEVICE_UNCACHED_BIT_AMD = 0x00000080
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COHERENT_MEMORY_FEATURES_AMD = 1000229000
VK_AMD_extension_231 = 1
VK_AMD_EXTENSION_231_SPEC_VERSION = 0
VK_AMD_EXTENSION_231_EXTENSION_NAME = "VK_AMD_extension_231"
VK_AMD_extension_232 = 1
VK_AMD_EXTENSION_232_SPEC_VERSION = 0
VK_AMD_EXTENSION_232_EXTENSION_NAME = "VK_AMD_extension_232"
VK_AMD_extension_233 = 1
VK_AMD_EXTENSION_233_SPEC_VERSION = 0
VK_AMD_EXTENSION_233_EXTENSION_NAME = "VK_AMD_extension_233"
VK_AMD_extension_234 = 1
VK_AMD_EXTENSION_234_SPEC_VERSION = 0
VK_AMD_EXTENSION_234_EXTENSION_NAME = "VK_AMD_extension_234"
VK_EXT_shader_image_atomic_int64 = 1
VK_EXT_SHADER_IMAGE_ATOMIC_INT64_SPEC_VERSION = 1
VK_EXT_SHADER_IMAGE_ATOMIC_INT64_EXTENSION_NAME = "VK_EXT_shader_image_atomic_int64"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_IMAGE_ATOMIC_INT64_FEATURES_EXT = 1000234000
VK_AMD_extension_236 = 1
VK_AMD_EXTENSION_236_SPEC_VERSION = 0
VK_AMD_EXTENSION_236_EXTENSION_NAME = "VK_AMD_extension_236"
VK_KHR_spirv_1_4 = 1
VK_KHR_SPIRV_1_4_SPEC_VERSION = 1
VK_KHR_SPIRV_1_4_EXTENSION_NAME = "VK_KHR_spirv_1_4"
VK_EXT_memory_budget = 1
VK_EXT_MEMORY_BUDGET_SPEC_VERSION = 1
VK_EXT_MEMORY_BUDGET_EXTENSION_NAME = "VK_EXT_memory_budget"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_BUDGET_PROPERTIES_EXT = 1000237000
VK_EXT_memory_priority = 1
VK_EXT_MEMORY_PRIORITY_SPEC_VERSION = 1
VK_EXT_MEMORY_PRIORITY_EXTENSION_NAME = "VK_EXT_memory_priority"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PRIORITY_FEATURES_EXT = 1000238000
VK_STRUCTURE_TYPE_MEMORY_PRIORITY_ALLOCATE_INFO_EXT = 1000238001
VK_KHR_surface_protected_capabilities = 1
VK_KHR_SURFACE_PROTECTED_CAPABILITIES_SPEC_VERSION = 1
VK_KHR_SURFACE_PROTECTED_CAPABILITIES_EXTENSION_NAME = "VK_KHR_surface_protected_capabilities"
VK_STRUCTURE_TYPE_SURFACE_PROTECTED_CAPABILITIES_KHR = 1000239000
VK_NV_dedicated_allocation_image_aliasing = 1
VK_NV_DEDICATED_ALLOCATION_IMAGE_ALIASING_SPEC_VERSION = 1
VK_NV_DEDICATED_ALLOCATION_IMAGE_ALIASING_EXTENSION_NAME = "VK_NV_dedicated_allocation_image_aliasing"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEDICATED_ALLOCATION_IMAGE_ALIASING_FEATURES_NV = 1000240000
VK_KHR_separate_depth_stencil_layouts = 1
VK_KHR_SEPARATE_DEPTH_STENCIL_LAYOUTS_SPEC_VERSION = 1
VK_KHR_SEPARATE_DEPTH_STENCIL_LAYOUTS_EXTENSION_NAME = "VK_KHR_separate_depth_stencil_layouts"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SEPARATE_DEPTH_STENCIL_LAYOUTS_FEATURES_KHR = None
VK_STRUCTURE_TYPE_ATTACHMENT_REFERENCE_STENCIL_LAYOUT_KHR = None
VK_STRUCTURE_TYPE_ATTACHMENT_DESCRIPTION_STENCIL_LAYOUT_KHR = None
VK_IMAGE_LAYOUT_DEPTH_ATTACHMENT_OPTIMAL_KHR = None
VK_IMAGE_LAYOUT_DEPTH_READ_ONLY_OPTIMAL_KHR = None
VK_IMAGE_LAYOUT_STENCIL_ATTACHMENT_OPTIMAL_KHR = None
VK_IMAGE_LAYOUT_STENCIL_READ_ONLY_OPTIMAL_KHR = None
VK_INTEL_extension_243 = 1
VK_INTEL_EXTENSION_243_SPEC_VERSION = 0
VK_INTEL_EXTENSION_243_EXTENSION_NAME = "VK_INTEL_extension_243"
VK_ACCESS_2_RESERVED_46_BIT_INTEL = 0x400000000000
VK_MESA_extension_244 = 1
VK_MESA_EXTENSION_244_SPEC_VERSION = 0
VK_MESA_EXTENSION_244_EXTENSION_NAME = "VK_MESA_extension_244"
VK_EXT_buffer_device_address = 1
VK_EXT_BUFFER_DEVICE_ADDRESS_SPEC_VERSION = 2
VK_EXT_BUFFER_DEVICE_ADDRESS_EXTENSION_NAME = "VK_EXT_buffer_device_address"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BUFFER_DEVICE_ADDRESS_FEATURES_EXT = 1000244000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BUFFER_ADDRESS_FEATURES_EXT = None
VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_INFO_EXT = None
VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_CREATE_INFO_EXT = 1000244002
VK_BUFFER_USAGE_SHADER_DEVICE_ADDRESS_BIT_EXT = None
VK_BUFFER_CREATE_DEVICE_ADDRESS_CAPTURE_REPLAY_BIT_EXT = None
VK_EXT_tooling_info = 1
VK_EXT_TOOLING_INFO_SPEC_VERSION = 1
VK_EXT_TOOLING_INFO_EXTENSION_NAME = "VK_EXT_tooling_info"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TOOL_PROPERTIES_EXT = None
VK_TOOL_PURPOSE_DEBUG_REPORTING_BIT_EXT = 0x00000020
VK_TOOL_PURPOSE_DEBUG_MARKERS_BIT_EXT = 0x00000040
VK_EXT_separate_stencil_usage = 1
VK_EXT_SEPARATE_STENCIL_USAGE_SPEC_VERSION = 1
VK_EXT_SEPARATE_STENCIL_USAGE_EXTENSION_NAME = "VK_EXT_separate_stencil_usage"
VK_STRUCTURE_TYPE_IMAGE_STENCIL_USAGE_CREATE_INFO_EXT = None
VK_EXT_validation_features = 1
VK_EXT_VALIDATION_FEATURES_SPEC_VERSION = 6
VK_EXT_VALIDATION_FEATURES_EXTENSION_NAME = "VK_EXT_validation_features"
VK_STRUCTURE_TYPE_VALIDATION_FEATURES_EXT = 1000247000
VK_KHR_present_wait = 1
VK_KHR_PRESENT_WAIT_SPEC_VERSION = 1
VK_KHR_PRESENT_WAIT_EXTENSION_NAME = "VK_KHR_present_wait"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRESENT_WAIT_FEATURES_KHR = 1000248000
VK_NV_cooperative_matrix = 1
VK_NV_COOPERATIVE_MATRIX_SPEC_VERSION = 1
VK_NV_COOPERATIVE_MATRIX_EXTENSION_NAME = "VK_NV_cooperative_matrix"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COOPERATIVE_MATRIX_FEATURES_NV = 1000249000
VK_STRUCTURE_TYPE_COOPERATIVE_MATRIX_PROPERTIES_NV = 1000249001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COOPERATIVE_MATRIX_PROPERTIES_NV = 1000249002
VK_SCOPE_DEVICE_NV = None
VK_SCOPE_WORKGROUP_NV = None
VK_SCOPE_SUBGROUP_NV = None
VK_SCOPE_QUEUE_FAMILY_NV = None
VK_COMPONENT_TYPE_FLOAT16_NV = None
VK_COMPONENT_TYPE_FLOAT32_NV = None
VK_COMPONENT_TYPE_FLOAT64_NV = None
VK_COMPONENT_TYPE_SINT8_NV = None
VK_COMPONENT_TYPE_SINT16_NV = None
VK_COMPONENT_TYPE_SINT32_NV = None
VK_COMPONENT_TYPE_SINT64_NV = None
VK_COMPONENT_TYPE_UINT8_NV = None
VK_COMPONENT_TYPE_UINT16_NV = None
VK_COMPONENT_TYPE_UINT32_NV = None
VK_COMPONENT_TYPE_UINT64_NV = None
VK_NV_coverage_reduction_mode = 1
VK_NV_COVERAGE_REDUCTION_MODE_SPEC_VERSION = 1
VK_NV_COVERAGE_REDUCTION_MODE_EXTENSION_NAME = "VK_NV_coverage_reduction_mode"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COVERAGE_REDUCTION_MODE_FEATURES_NV = 1000250000
VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_REDUCTION_STATE_CREATE_INFO_NV = 1000250001
VK_STRUCTURE_TYPE_FRAMEBUFFER_MIXED_SAMPLES_COMBINATION_NV = 1000250002
VK_EXT_fragment_shader_interlock = 1
VK_EXT_FRAGMENT_SHADER_INTERLOCK_SPEC_VERSION = 1
VK_EXT_FRAGMENT_SHADER_INTERLOCK_EXTENSION_NAME = "VK_EXT_fragment_shader_interlock"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_INTERLOCK_FEATURES_EXT = 1000251000
VK_EXT_ycbcr_image_arrays = 1
VK_EXT_YCBCR_IMAGE_ARRAYS_SPEC_VERSION = 1
VK_EXT_YCBCR_IMAGE_ARRAYS_EXTENSION_NAME = "VK_EXT_ycbcr_image_arrays"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_YCBCR_IMAGE_ARRAYS_FEATURES_EXT = 1000252000
VK_KHR_uniform_buffer_standard_layout = 1
VK_KHR_UNIFORM_BUFFER_STANDARD_LAYOUT_SPEC_VERSION = 1
VK_KHR_UNIFORM_BUFFER_STANDARD_LAYOUT_EXTENSION_NAME = "VK_KHR_uniform_buffer_standard_layout"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_UNIFORM_BUFFER_STANDARD_LAYOUT_FEATURES_KHR = None
VK_EXT_provoking_vertex = 1
VK_EXT_PROVOKING_VERTEX_SPEC_VERSION = 1
VK_EXT_PROVOKING_VERTEX_EXTENSION_NAME = "VK_EXT_provoking_vertex"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROVOKING_VERTEX_FEATURES_EXT = 1000254000
VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_PROVOKING_VERTEX_STATE_CREATE_INFO_EXT = 1000254001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROVOKING_VERTEX_PROPERTIES_EXT = 1000254002
VK_EXT_full_screen_exclusive = 1
VK_EXT_FULL_SCREEN_EXCLUSIVE_SPEC_VERSION = 4
VK_EXT_FULL_SCREEN_EXCLUSIVE_EXTENSION_NAME = "VK_EXT_full_screen_exclusive"
VK_STRUCTURE_TYPE_SURFACE_FULL_SCREEN_EXCLUSIVE_INFO_EXT = 1000255000
VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_FULL_SCREEN_EXCLUSIVE_EXT = 1000255002
VK_STRUCTURE_TYPE_SURFACE_FULL_SCREEN_EXCLUSIVE_WIN32_INFO_EXT = 1000255001
VK_EXT_headless_surface = 1
VK_EXT_HEADLESS_SURFACE_SPEC_VERSION = 1
VK_EXT_HEADLESS_SURFACE_EXTENSION_NAME = "VK_EXT_headless_surface"
VK_STRUCTURE_TYPE_HEADLESS_SURFACE_CREATE_INFO_EXT = 1000256000
VK_KHR_buffer_device_address = 1
VK_KHR_BUFFER_DEVICE_ADDRESS_SPEC_VERSION = 1
VK_KHR_BUFFER_DEVICE_ADDRESS_EXTENSION_NAME = "VK_KHR_buffer_device_address"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BUFFER_DEVICE_ADDRESS_FEATURES_KHR = None
VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_INFO_KHR = None
VK_STRUCTURE_TYPE_BUFFER_OPAQUE_CAPTURE_ADDRESS_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_MEMORY_OPAQUE_CAPTURE_ADDRESS_ALLOCATE_INFO_KHR = None
VK_STRUCTURE_TYPE_DEVICE_MEMORY_OPAQUE_CAPTURE_ADDRESS_INFO_KHR = None
VK_BUFFER_USAGE_SHADER_DEVICE_ADDRESS_BIT_KHR = None
VK_BUFFER_CREATE_DEVICE_ADDRESS_CAPTURE_REPLAY_BIT_KHR = None
VK_MEMORY_ALLOCATE_DEVICE_ADDRESS_BIT_KHR = None
VK_MEMORY_ALLOCATE_DEVICE_ADDRESS_CAPTURE_REPLAY_BIT_KHR = None
VK_EXT_extension_259 = 1
VK_EXT_EXTENSION_259_SPEC_VERSION = 0
VK_EXT_EXTENSION_259_EXTENSION_NAME = "VK_EXT_extension_259"
VK_QUEUE_RESERVED_9_BIT_EXT = 0x00000200
VK_FORMAT_FEATURE_2_RESERVED_44_BIT_EXT = 0x100000000000
VK_FORMAT_FEATURE_2_RESERVED_45_BIT_EXT = 0x200000000000
VK_IMAGE_CREATE_RESERVED_19_BIT_EXT = 0x00080000
VK_EXT_line_rasterization = 1
VK_EXT_LINE_RASTERIZATION_SPEC_VERSION = 1
VK_EXT_LINE_RASTERIZATION_EXTENSION_NAME = "VK_EXT_line_rasterization"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LINE_RASTERIZATION_FEATURES_EXT = 1000259000
VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_LINE_STATE_CREATE_INFO_EXT = 1000259001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LINE_RASTERIZATION_PROPERTIES_EXT = 1000259002
VK_DYNAMIC_STATE_LINE_STIPPLE_EXT = 1000259000
VK_EXT_shader_atomic_float = 1
VK_EXT_SHADER_ATOMIC_FLOAT_SPEC_VERSION = 1
VK_EXT_SHADER_ATOMIC_FLOAT_EXTENSION_NAME = "VK_EXT_shader_atomic_float"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_FLOAT_FEATURES_EXT = 1000260000
VK_EXT_host_query_reset = 1
VK_EXT_HOST_QUERY_RESET_SPEC_VERSION = 1
VK_EXT_HOST_QUERY_RESET_EXTENSION_NAME = "VK_EXT_host_query_reset"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_HOST_QUERY_RESET_FEATURES_EXT = None
VK_GGP_extension_263 = 1
VK_GGP_EXTENSION_263_SPEC_VERSION = 0
VK_GGP_EXTENSION_263_EXTENSION_NAME = "VK_GGP_extension_263"
VK_BRCM_extension_264 = 1
VK_BRCM_EXTENSION_264_SPEC_VERSION = 0
VK_BRCM_EXTENSION_264_EXTENSION_NAME = "VK_BRCM_extension_264"
VK_BRCM_extension_265 = 1
VK_BRCM_EXTENSION_265_SPEC_VERSION = 0
VK_BRCM_EXTENSION_265_EXTENSION_NAME = "VK_BRCM_extension_265"
VK_EXT_index_type_uint8 = 1
VK_EXT_INDEX_TYPE_UINT8_SPEC_VERSION = 1
VK_EXT_INDEX_TYPE_UINT8_EXTENSION_NAME = "VK_EXT_index_type_uint8"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INDEX_TYPE_UINT8_FEATURES_EXT = 1000265000
VK_INDEX_TYPE_UINT8_EXT = 1000265000
VK_EXT_extension_267 = 1
VK_EXT_EXTENSION_267_SPEC_VERSION = 0
VK_EXT_EXTENSION_267_EXTENSION_NAME = "VK_EXT_extension_267"
VK_EXT_extended_dynamic_state = 1
VK_EXT_EXTENDED_DYNAMIC_STATE_SPEC_VERSION = 1
VK_EXT_EXTENDED_DYNAMIC_STATE_EXTENSION_NAME = "VK_EXT_extended_dynamic_state"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_DYNAMIC_STATE_FEATURES_EXT = 1000267000
VK_DYNAMIC_STATE_CULL_MODE_EXT = None
VK_DYNAMIC_STATE_FRONT_FACE_EXT = None
VK_DYNAMIC_STATE_PRIMITIVE_TOPOLOGY_EXT = None
VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT_EXT = None
VK_DYNAMIC_STATE_SCISSOR_WITH_COUNT_EXT = None
VK_DYNAMIC_STATE_VERTEX_INPUT_BINDING_STRIDE_EXT = None
VK_DYNAMIC_STATE_DEPTH_TEST_ENABLE_EXT = None
VK_DYNAMIC_STATE_DEPTH_WRITE_ENABLE_EXT = None
VK_DYNAMIC_STATE_DEPTH_COMPARE_OP_EXT = None
VK_DYNAMIC_STATE_DEPTH_BOUNDS_TEST_ENABLE_EXT = None
VK_DYNAMIC_STATE_STENCIL_TEST_ENABLE_EXT = None
VK_DYNAMIC_STATE_STENCIL_OP_EXT = None
VK_KHR_deferred_host_operations = 1
VK_KHR_DEFERRED_HOST_OPERATIONS_SPEC_VERSION = 4
VK_KHR_DEFERRED_HOST_OPERATIONS_EXTENSION_NAME = "VK_KHR_deferred_host_operations"
VK_OBJECT_TYPE_DEFERRED_OPERATION_KHR = 1000268000
VK_KHR_pipeline_executable_properties = 1
VK_KHR_PIPELINE_EXECUTABLE_PROPERTIES_SPEC_VERSION = 1
VK_KHR_PIPELINE_EXECUTABLE_PROPERTIES_EXTENSION_NAME = "VK_KHR_pipeline_executable_properties"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_EXECUTABLE_PROPERTIES_FEATURES_KHR = 1000269000
VK_STRUCTURE_TYPE_PIPELINE_INFO_KHR = 1000269001
VK_STRUCTURE_TYPE_PIPELINE_EXECUTABLE_PROPERTIES_KHR = 1000269002
VK_STRUCTURE_TYPE_PIPELINE_EXECUTABLE_INFO_KHR = 1000269003
VK_STRUCTURE_TYPE_PIPELINE_EXECUTABLE_STATISTIC_KHR = 1000269004
VK_STRUCTURE_TYPE_PIPELINE_EXECUTABLE_INTERNAL_REPRESENTATION_KHR = 1000269005
VK_PIPELINE_CREATE_CAPTURE_STATISTICS_BIT_KHR = 0x00000040
VK_PIPELINE_CREATE_CAPTURE_INTERNAL_REPRESENTATIONS_BIT_KHR = 0x00000080
VK_EXT_host_image_copy = 1
VK_EXT_HOST_IMAGE_COPY_SPEC_VERSION = 1
VK_EXT_HOST_IMAGE_COPY_EXTENSION_NAME = "VK_EXT_host_image_copy"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_HOST_IMAGE_COPY_FEATURES_EXT = 1000270000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_HOST_IMAGE_COPY_PROPERTIES_EXT = 1000270001
VK_STRUCTURE_TYPE_MEMORY_TO_IMAGE_COPY_EXT = 1000270002
VK_STRUCTURE_TYPE_IMAGE_TO_MEMORY_COPY_EXT = 1000270003
VK_STRUCTURE_TYPE_COPY_IMAGE_TO_MEMORY_INFO_EXT = 1000270004
VK_STRUCTURE_TYPE_COPY_MEMORY_TO_IMAGE_INFO_EXT = 1000270005
VK_STRUCTURE_TYPE_HOST_IMAGE_LAYOUT_TRANSITION_INFO_EXT = 1000270006
VK_STRUCTURE_TYPE_COPY_IMAGE_TO_IMAGE_INFO_EXT = 1000270007
VK_STRUCTURE_TYPE_SUBRESOURCE_HOST_MEMCPY_SIZE_EXT = 1000270008
VK_STRUCTURE_TYPE_HOST_IMAGE_COPY_DEVICE_PERFORMANCE_QUERY_EXT = 1000270009
VK_IMAGE_USAGE_HOST_TRANSFER_BIT_EXT = 0x00400000
VK_FORMAT_FEATURE_2_HOST_IMAGE_TRANSFER_BIT_EXT = 0x400000000000
VK_KHR_map_memory2 = 1
VK_KHR_MAP_MEMORY_2_SPEC_VERSION = 1
VK_KHR_MAP_MEMORY_2_EXTENSION_NAME = "VK_KHR_map_memory2"
VK_STRUCTURE_TYPE_MEMORY_MAP_INFO_KHR = 1000271000
VK_STRUCTURE_TYPE_MEMORY_UNMAP_INFO_KHR = 1000271001
VK_INTEL_extension_273 = 1
VK_INTEL_EXTENSION_273_SPEC_VERSION = 0
VK_INTEL_EXTENSION_273_EXTENSION_NAME = "VK_INTEL_extension_273"
VK_EXT_shader_atomic_float2 = 1
VK_EXT_SHADER_ATOMIC_FLOAT_2_SPEC_VERSION = 1
VK_EXT_SHADER_ATOMIC_FLOAT_2_EXTENSION_NAME = "VK_EXT_shader_atomic_float2"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_FLOAT_2_FEATURES_EXT = 1000273000
VK_EXT_surface_maintenance1 = 1
VK_EXT_SURFACE_MAINTENANCE_1_SPEC_VERSION = 1
VK_EXT_SURFACE_MAINTENANCE_1_EXTENSION_NAME = "VK_EXT_surface_maintenance1"
VK_STRUCTURE_TYPE_SURFACE_PRESENT_MODE_EXT = 1000274000
VK_STRUCTURE_TYPE_SURFACE_PRESENT_SCALING_CAPABILITIES_EXT = 1000274001
VK_STRUCTURE_TYPE_SURFACE_PRESENT_MODE_COMPATIBILITY_EXT = 1000274002
VK_EXT_swapchain_maintenance1 = 1
VK_EXT_SWAPCHAIN_MAINTENANCE_1_SPEC_VERSION = 1
VK_EXT_SWAPCHAIN_MAINTENANCE_1_EXTENSION_NAME = "VK_EXT_swapchain_maintenance1"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SWAPCHAIN_MAINTENANCE_1_FEATURES_EXT = 1000275000
VK_STRUCTURE_TYPE_SWAPCHAIN_PRESENT_FENCE_INFO_EXT = 1000275001
VK_STRUCTURE_TYPE_SWAPCHAIN_PRESENT_MODES_CREATE_INFO_EXT = 1000275002
VK_STRUCTURE_TYPE_SWAPCHAIN_PRESENT_MODE_INFO_EXT = 1000275003
VK_STRUCTURE_TYPE_SWAPCHAIN_PRESENT_SCALING_CREATE_INFO_EXT = 1000275004
VK_STRUCTURE_TYPE_RELEASE_SWAPCHAIN_IMAGES_INFO_EXT = 1000275005
VK_SWAPCHAIN_CREATE_DEFERRED_MEMORY_ALLOCATION_BIT_EXT = 0x00000008
VK_EXT_shader_demote_to_helper_invocation = 1
VK_EXT_SHADER_DEMOTE_TO_HELPER_INVOCATION_SPEC_VERSION = 1
VK_EXT_SHADER_DEMOTE_TO_HELPER_INVOCATION_EXTENSION_NAME = "VK_EXT_shader_demote_to_helper_invocation"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_DEMOTE_TO_HELPER_INVOCATION_FEATURES_EXT = None
VK_NV_device_generated_commands = 1
VK_NV_DEVICE_GENERATED_COMMANDS_SPEC_VERSION = 3
VK_NV_DEVICE_GENERATED_COMMANDS_EXTENSION_NAME = "VK_NV_device_generated_commands"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEVICE_GENERATED_COMMANDS_PROPERTIES_NV = 1000277000
VK_STRUCTURE_TYPE_GRAPHICS_SHADER_GROUP_CREATE_INFO_NV = 1000277001
VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_SHADER_GROUPS_CREATE_INFO_NV = 1000277002
VK_STRUCTURE_TYPE_INDIRECT_COMMANDS_LAYOUT_TOKEN_NV = 1000277003
VK_STRUCTURE_TYPE_INDIRECT_COMMANDS_LAYOUT_CREATE_INFO_NV = 1000277004
VK_STRUCTURE_TYPE_GENERATED_COMMANDS_INFO_NV = 1000277005
VK_STRUCTURE_TYPE_GENERATED_COMMANDS_MEMORY_REQUIREMENTS_INFO_NV = 1000277006
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEVICE_GENERATED_COMMANDS_FEATURES_NV = 1000277007
VK_PIPELINE_CREATE_INDIRECT_BINDABLE_BIT_NV = 0x00040000
VK_PIPELINE_STAGE_COMMAND_PREPROCESS_BIT_NV = 0x00020000
VK_ACCESS_COMMAND_PREPROCESS_READ_BIT_NV = 0x00020000
VK_ACCESS_COMMAND_PREPROCESS_WRITE_BIT_NV = 0x00040000
VK_OBJECT_TYPE_INDIRECT_COMMANDS_LAYOUT_NV = 1000277000
VK_NV_inherited_viewport_scissor = 1
VK_NV_INHERITED_VIEWPORT_SCISSOR_SPEC_VERSION = 1
VK_NV_INHERITED_VIEWPORT_SCISSOR_EXTENSION_NAME = "VK_NV_inherited_viewport_scissor"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INHERITED_VIEWPORT_SCISSOR_FEATURES_NV = 1000278000
VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_VIEWPORT_SCISSOR_INFO_NV = 1000278001
VK_KHR_extension_280 = 1
VK_KHR_EXTENSION_280_SPEC_VERSION = 0
VK_KHR_EXTENSION_280_EXTENSION_NAME = "VK_KHR_extension_280"
VK_KHR_shader_integer_dot_product = 1
VK_KHR_SHADER_INTEGER_DOT_PRODUCT_SPEC_VERSION = 1
VK_KHR_SHADER_INTEGER_DOT_PRODUCT_EXTENSION_NAME = "VK_KHR_shader_integer_dot_product"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_INTEGER_DOT_PRODUCT_FEATURES_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_INTEGER_DOT_PRODUCT_PROPERTIES_KHR = None
VK_EXT_texel_buffer_alignment = 1
VK_EXT_TEXEL_BUFFER_ALIGNMENT_SPEC_VERSION = 1
VK_EXT_TEXEL_BUFFER_ALIGNMENT_EXTENSION_NAME = "VK_EXT_texel_buffer_alignment"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXEL_BUFFER_ALIGNMENT_FEATURES_EXT = 1000281000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXEL_BUFFER_ALIGNMENT_PROPERTIES_EXT = None
VK_QCOM_render_pass_transform = 1
VK_QCOM_RENDER_PASS_TRANSFORM_SPEC_VERSION = 4
VK_QCOM_RENDER_PASS_TRANSFORM_EXTENSION_NAME = "VK_QCOM_render_pass_transform"
VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_RENDER_PASS_TRANSFORM_INFO_QCOM = 1000282000
VK_STRUCTURE_TYPE_RENDER_PASS_TRANSFORM_BEGIN_INFO_QCOM = 1000282001
VK_RENDER_PASS_CREATE_TRANSFORM_BIT_QCOM = 0x00000002
VK_EXT_depth_bias_control = 1
VK_EXT_DEPTH_BIAS_CONTROL_SPEC_VERSION = 1
VK_EXT_DEPTH_BIAS_CONTROL_EXTENSION_NAME = "VK_EXT_depth_bias_control"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_BIAS_CONTROL_FEATURES_EXT = 1000283000
VK_STRUCTURE_TYPE_DEPTH_BIAS_INFO_EXT = 1000283001
VK_STRUCTURE_TYPE_DEPTH_BIAS_REPRESENTATION_INFO_EXT = 1000283002
VK_EXT_device_memory_report = 1
VK_EXT_DEVICE_MEMORY_REPORT_SPEC_VERSION = 2
VK_EXT_DEVICE_MEMORY_REPORT_EXTENSION_NAME = "VK_EXT_device_memory_report"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEVICE_MEMORY_REPORT_FEATURES_EXT = 1000284000
VK_STRUCTURE_TYPE_DEVICE_DEVICE_MEMORY_REPORT_CREATE_INFO_EXT = 1000284001
VK_STRUCTURE_TYPE_DEVICE_MEMORY_REPORT_CALLBACK_DATA_EXT = 1000284002
VK_EXT_acquire_drm_display = 1
VK_EXT_ACQUIRE_DRM_DISPLAY_SPEC_VERSION = 1
VK_EXT_ACQUIRE_DRM_DISPLAY_EXTENSION_NAME = "VK_EXT_acquire_drm_display"
VK_EXT_robustness2 = 1
VK_EXT_ROBUSTNESS_2_SPEC_VERSION = 1
VK_EXT_ROBUSTNESS_2_EXTENSION_NAME = "VK_EXT_robustness2"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ROBUSTNESS_2_FEATURES_EXT = 1000286000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ROBUSTNESS_2_PROPERTIES_EXT = 1000286001
VK_EXT_custom_border_color = 1
VK_EXT_CUSTOM_BORDER_COLOR_SPEC_VERSION = 12
VK_EXT_CUSTOM_BORDER_COLOR_EXTENSION_NAME = "VK_EXT_custom_border_color"
VK_STRUCTURE_TYPE_SAMPLER_CUSTOM_BORDER_COLOR_CREATE_INFO_EXT = 1000287000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUSTOM_BORDER_COLOR_PROPERTIES_EXT = 1000287001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUSTOM_BORDER_COLOR_FEATURES_EXT = 1000287002
VK_BORDER_COLOR_FLOAT_CUSTOM_EXT = 1000287003
VK_BORDER_COLOR_INT_CUSTOM_EXT = 1000287004
VK_EXT_extension_289 = 1
VK_EXT_EXTENSION_289_SPEC_VERSION = 0
VK_EXT_EXTENSION_289_EXTENSION_NAME = "VK_EXT_extension_289"
VK_FORMAT_ASTC_3x3x3_UNORM_BLOCK_EXT = 1000288000
VK_FORMAT_ASTC_3x3x3_SRGB_BLOCK_EXT = 1000288001
VK_FORMAT_ASTC_3x3x3_SFLOAT_BLOCK_EXT = 1000288002
VK_FORMAT_ASTC_4x3x3_UNORM_BLOCK_EXT = 1000288003
VK_FORMAT_ASTC_4x3x3_SRGB_BLOCK_EXT = 1000288004
VK_FORMAT_ASTC_4x3x3_SFLOAT_BLOCK_EXT = 1000288005
VK_FORMAT_ASTC_4x4x3_UNORM_BLOCK_EXT = 1000288006
VK_FORMAT_ASTC_4x4x3_SRGB_BLOCK_EXT = 1000288007
VK_FORMAT_ASTC_4x4x3_SFLOAT_BLOCK_EXT = 1000288008
VK_FORMAT_ASTC_4x4x4_UNORM_BLOCK_EXT = 1000288009
VK_FORMAT_ASTC_4x4x4_SRGB_BLOCK_EXT = 1000288010
VK_FORMAT_ASTC_4x4x4_SFLOAT_BLOCK_EXT = 1000288011
VK_FORMAT_ASTC_5x4x4_UNORM_BLOCK_EXT = 1000288012
VK_FORMAT_ASTC_5x4x4_SRGB_BLOCK_EXT = 1000288013
VK_FORMAT_ASTC_5x4x4_SFLOAT_BLOCK_EXT = 1000288014
VK_FORMAT_ASTC_5x5x4_UNORM_BLOCK_EXT = 1000288015
VK_FORMAT_ASTC_5x5x4_SRGB_BLOCK_EXT = 1000288016
VK_FORMAT_ASTC_5x5x4_SFLOAT_BLOCK_EXT = 1000288017
VK_FORMAT_ASTC_5x5x5_UNORM_BLOCK_EXT = 1000288018
VK_FORMAT_ASTC_5x5x5_SRGB_BLOCK_EXT = 1000288019
VK_FORMAT_ASTC_5x5x5_SFLOAT_BLOCK_EXT = 1000288020
VK_FORMAT_ASTC_6x5x5_UNORM_BLOCK_EXT = 1000288021
VK_FORMAT_ASTC_6x5x5_SRGB_BLOCK_EXT = 1000288022
VK_FORMAT_ASTC_6x5x5_SFLOAT_BLOCK_EXT = 1000288023
VK_FORMAT_ASTC_6x6x5_UNORM_BLOCK_EXT = 1000288024
VK_FORMAT_ASTC_6x6x5_SRGB_BLOCK_EXT = 1000288025
VK_FORMAT_ASTC_6x6x5_SFLOAT_BLOCK_EXT = 1000288026
VK_FORMAT_ASTC_6x6x6_UNORM_BLOCK_EXT = 1000288027
VK_FORMAT_ASTC_6x6x6_SRGB_BLOCK_EXT = 1000288028
VK_FORMAT_ASTC_6x6x6_SFLOAT_BLOCK_EXT = 1000288029
VK_GOOGLE_user_type = 1
VK_GOOGLE_USER_TYPE_SPEC_VERSION = 1
VK_GOOGLE_USER_TYPE_EXTENSION_NAME = "VK_GOOGLE_user_type"
VK_KHR_pipeline_library = 1
VK_KHR_PIPELINE_LIBRARY_SPEC_VERSION = 1
VK_KHR_PIPELINE_LIBRARY_EXTENSION_NAME = "VK_KHR_pipeline_library"
VK_PIPELINE_CREATE_LIBRARY_BIT_KHR = 0x00000800
VK_STRUCTURE_TYPE_PIPELINE_LIBRARY_CREATE_INFO_KHR = 1000290000
VK_NV_extension_292 = 1
VK_NV_EXTENSION_292_SPEC_VERSION = 0
VK_NV_EXTENSION_292_EXTENSION_NAME = "VK_NV_extension_292"
VK_NV_present_barrier = 1
VK_NV_PRESENT_BARRIER_SPEC_VERSION = 1
VK_NV_PRESENT_BARRIER_EXTENSION_NAME = "VK_NV_present_barrier"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRESENT_BARRIER_FEATURES_NV = 1000292000
VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_PRESENT_BARRIER_NV = 1000292001
VK_STRUCTURE_TYPE_SWAPCHAIN_PRESENT_BARRIER_CREATE_INFO_NV = 1000292002
VK_KHR_shader_non_semantic_info = 1
VK_KHR_SHADER_NON_SEMANTIC_INFO_SPEC_VERSION = 1
VK_KHR_SHADER_NON_SEMANTIC_INFO_EXTENSION_NAME = "VK_KHR_shader_non_semantic_info"
VK_KHR_present_id = 1
VK_KHR_PRESENT_ID_SPEC_VERSION = 1
VK_KHR_PRESENT_ID_EXTENSION_NAME = "VK_KHR_present_id"
VK_STRUCTURE_TYPE_PRESENT_ID_KHR = 1000294000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRESENT_ID_FEATURES_KHR = 1000294001
VK_EXT_private_data = 1
VK_EXT_PRIVATE_DATA_SPEC_VERSION = 1
VK_EXT_PRIVATE_DATA_EXTENSION_NAME = "VK_EXT_private_data"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRIVATE_DATA_FEATURES_EXT = None
VK_STRUCTURE_TYPE_DEVICE_PRIVATE_DATA_CREATE_INFO_EXT = None
VK_STRUCTURE_TYPE_PRIVATE_DATA_SLOT_CREATE_INFO_EXT = None
VK_OBJECT_TYPE_PRIVATE_DATA_SLOT_EXT = None
VK_KHR_extension_297 = 1
VK_KHR_EXTENSION_297_SPEC_VERSION = 0
VK_KHR_EXTENSION_297_EXTENSION_NAME = "VK_KHR_extension_297"
VK_PIPELINE_SHADER_STAGE_CREATE_RESERVED_3_BIT_KHR = 0x00000008
VK_EXT_pipeline_creation_cache_control = 1
VK_EXT_PIPELINE_CREATION_CACHE_CONTROL_SPEC_VERSION = 3
VK_EXT_PIPELINE_CREATION_CACHE_CONTROL_EXTENSION_NAME = "VK_EXT_pipeline_creation_cache_control"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_CREATION_CACHE_CONTROL_FEATURES_EXT = None
VK_PIPELINE_CREATE_FAIL_ON_PIPELINE_COMPILE_REQUIRED_BIT_EXT = None
VK_PIPELINE_CREATE_EARLY_RETURN_ON_FAILURE_BIT_EXT = None
VK_PIPELINE_CACHE_CREATE_EXTERNALLY_SYNCHRONIZED_BIT_EXT = None
VK_KHR_extension_299 = 1
VK_KHR_EXTENSION_299_SPEC_VERSION = 0
VK_KHR_EXTENSION_299_EXTENSION_NAME = "VK_KHR_extension_299"
VK_KHR_video_encode_queue = 1
VK_KHR_VIDEO_ENCODE_QUEUE_SPEC_VERSION = 12
VK_KHR_VIDEO_ENCODE_QUEUE_EXTENSION_NAME = "VK_KHR_video_encode_queue"
VK_PIPELINE_STAGE_2_VIDEO_ENCODE_BIT_KHR = 0x08000000
VK_ACCESS_2_VIDEO_ENCODE_READ_BIT_KHR = 0x2000000000
VK_ACCESS_2_VIDEO_ENCODE_WRITE_BIT_KHR = 0x4000000000
VK_STRUCTURE_TYPE_VIDEO_ENCODE_INFO_KHR = 1000299000
VK_STRUCTURE_TYPE_VIDEO_ENCODE_RATE_CONTROL_INFO_KHR = 1000299001
VK_STRUCTURE_TYPE_VIDEO_ENCODE_RATE_CONTROL_LAYER_INFO_KHR = 1000299002
VK_STRUCTURE_TYPE_VIDEO_ENCODE_CAPABILITIES_KHR = 1000299003
VK_STRUCTURE_TYPE_VIDEO_ENCODE_USAGE_INFO_KHR = 1000299004
VK_STRUCTURE_TYPE_QUERY_POOL_VIDEO_ENCODE_FEEDBACK_CREATE_INFO_KHR = 1000299005
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VIDEO_ENCODE_QUALITY_LEVEL_INFO_KHR = 1000299006
VK_STRUCTURE_TYPE_VIDEO_ENCODE_QUALITY_LEVEL_PROPERTIES_KHR = 1000299007
VK_STRUCTURE_TYPE_VIDEO_ENCODE_QUALITY_LEVEL_INFO_KHR = 1000299008
VK_STRUCTURE_TYPE_VIDEO_ENCODE_SESSION_PARAMETERS_GET_INFO_KHR = 1000299009
VK_STRUCTURE_TYPE_VIDEO_ENCODE_SESSION_PARAMETERS_FEEDBACK_INFO_KHR = 1000299010
VK_QUEUE_VIDEO_ENCODE_BIT_KHR = 0x00000040
VK_VIDEO_CODING_CONTROL_ENCODE_RATE_CONTROL_BIT_KHR = 0x00000002
VK_VIDEO_CODING_CONTROL_ENCODE_QUALITY_LEVEL_BIT_KHR = 0x00000004
VK_BUFFER_USAGE_VIDEO_ENCODE_DST_BIT_KHR = 0x00008000
VK_BUFFER_USAGE_VIDEO_ENCODE_SRC_BIT_KHR = 0x00010000
VK_IMAGE_USAGE_VIDEO_ENCODE_DST_BIT_KHR = 0x00002000
VK_IMAGE_USAGE_VIDEO_ENCODE_SRC_BIT_KHR = 0x00004000
VK_IMAGE_USAGE_VIDEO_ENCODE_DPB_BIT_KHR = 0x00008000
VK_FORMAT_FEATURE_VIDEO_ENCODE_INPUT_BIT_KHR = 0x08000000
VK_FORMAT_FEATURE_VIDEO_ENCODE_DPB_BIT_KHR = 0x10000000
VK_VIDEO_SESSION_CREATE_ALLOW_ENCODE_PARAMETER_OPTIMIZATIONS_BIT_KHR = 0x00000002
VK_IMAGE_LAYOUT_VIDEO_ENCODE_DST_KHR = 1000299000
VK_IMAGE_LAYOUT_VIDEO_ENCODE_SRC_KHR = 1000299001
VK_IMAGE_LAYOUT_VIDEO_ENCODE_DPB_KHR = 1000299002
VK_QUERY_TYPE_VIDEO_ENCODE_FEEDBACK_KHR = 1000299000
VK_QUERY_RESULT_STATUS_INSUFFICIENT_BITSTREAM_BUFFER_RANGE_KHR = -1000299000
VK_FORMAT_FEATURE_2_VIDEO_ENCODE_INPUT_BIT_KHR = 0x08000000
VK_FORMAT_FEATURE_2_VIDEO_ENCODE_DPB_BIT_KHR = 0x10000000
VK_NV_device_diagnostics_config = 1
VK_NV_DEVICE_DIAGNOSTICS_CONFIG_SPEC_VERSION = 2
VK_NV_DEVICE_DIAGNOSTICS_CONFIG_EXTENSION_NAME = "VK_NV_device_diagnostics_config"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DIAGNOSTICS_CONFIG_FEATURES_NV = 1000300000
VK_STRUCTURE_TYPE_DEVICE_DIAGNOSTICS_CONFIG_CREATE_INFO_NV = 1000300001
VK_QCOM_render_pass_store_ops = 1
VK_QCOM_RENDER_PASS_STORE_OPS_SPEC_VERSION = 2
VK_QCOM_RENDER_PASS_STORE_OPS_EXTENSION_NAME = "VK_QCOM_render_pass_store_ops"
VK_ATTACHMENT_STORE_OP_NONE_QCOM = None
VK_QCOM_extension_303 = 1
VK_QCOM_EXTENSION_303_SPEC_VERSION = 0
VK_QCOM_EXTENSION_303_EXTENSION_NAME = "VK_QCOM_extension_303"
VK_QCOM_extension_304 = 1
VK_QCOM_EXTENSION_304_SPEC_VERSION = 0
VK_QCOM_EXTENSION_304_EXTENSION_NAME = "VK_QCOM_extension_304"
VK_QCOM_extension_305 = 1
VK_QCOM_EXTENSION_305_SPEC_VERSION = 0
VK_QCOM_EXTENSION_305_EXTENSION_NAME = "VK_QCOM_extension_305"
VK_QCOM_extension_306 = 1
VK_QCOM_EXTENSION_306_SPEC_VERSION = 0
VK_QCOM_EXTENSION_306_EXTENSION_NAME = "VK_QCOM_extension_306"
VK_QCOM_extension_307 = 1
VK_QCOM_EXTENSION_307_SPEC_VERSION = 0
VK_QCOM_EXTENSION_307_EXTENSION_NAME = "VK_QCOM_extension_307"
VK_NV_cuda_kernel_launch = 1
VK_NV_CUDA_KERNEL_LAUNCH_SPEC_VERSION = 2
VK_NV_CUDA_KERNEL_LAUNCH_EXTENSION_NAME = "VK_NV_cuda_kernel_launch"
VK_STRUCTURE_TYPE_CUDA_MODULE_CREATE_INFO_NV = 1000307000
VK_STRUCTURE_TYPE_CUDA_FUNCTION_CREATE_INFO_NV = 1000307001
VK_STRUCTURE_TYPE_CUDA_LAUNCH_INFO_NV = 1000307002
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUDA_KERNEL_LAUNCH_FEATURES_NV = 1000307003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUDA_KERNEL_LAUNCH_PROPERTIES_NV = 1000307004
VK_OBJECT_TYPE_CUDA_MODULE_NV = 1000307000
VK_OBJECT_TYPE_CUDA_FUNCTION_NV = 1000307001
VK_DEBUG_REPORT_OBJECT_TYPE_CUDA_MODULE_NV_EXT = 1000307000
VK_DEBUG_REPORT_OBJECT_TYPE_CUDA_FUNCTION_NV_EXT = 1000307001
VK_KHR_object_refresh = 1
VK_KHR_OBJECT_REFRESH_SPEC_VERSION = 1
VK_KHR_OBJECT_REFRESH_EXTENSION_NAME = "VK_KHR_object_refresh"
VK_STRUCTURE_TYPE_REFRESH_OBJECT_LIST_KHR = 1000308000
VK_QCOM_extension_310 = 1
VK_QCOM_EXTENSION_310_SPEC_VERSION = 0
VK_QCOM_EXTENSION_310_EXTENSION_NAME = "VK_QCOM_extension_310"
VK_BUFFER_USAGE_RESERVED_27_BIT_QCOM = 0x08000000
VK_BUFFER_USAGE_2_RESERVED_27_BIT_QCOM = 0x08000000
VK_ACCESS_2_RESERVED_51_BIT_QCOM = 0x8000000000000
VK_ACCESS_2_RESERVED_52_BIT_QCOM = 0x10000000000000
VK_ACCESS_2_RESERVED_53_BIT_QCOM = 0x20000000000000
VK_ACCESS_2_RESERVED_54_BIT_QCOM = 0x40000000000000
VK_NV_low_latency = 1
VK_NV_LOW_LATENCY_SPEC_VERSION = 1
VK_NV_LOW_LATENCY_EXTENSION_NAME = "VK_NV_low_latency"
VK_STRUCTURE_TYPE_QUERY_LOW_LATENCY_SUPPORT_NV = 1000310000
VK_EXT_metal_objects = 1
VK_EXT_METAL_OBJECTS_SPEC_VERSION = 1
VK_EXT_METAL_OBJECTS_EXTENSION_NAME = "VK_EXT_metal_objects"
VK_STRUCTURE_TYPE_EXPORT_METAL_OBJECT_CREATE_INFO_EXT = 1000311000
VK_STRUCTURE_TYPE_EXPORT_METAL_OBJECTS_INFO_EXT = 1000311001
VK_STRUCTURE_TYPE_EXPORT_METAL_DEVICE_INFO_EXT = 1000311002
VK_STRUCTURE_TYPE_EXPORT_METAL_COMMAND_QUEUE_INFO_EXT = 1000311003
VK_STRUCTURE_TYPE_EXPORT_METAL_BUFFER_INFO_EXT = 1000311004
VK_STRUCTURE_TYPE_IMPORT_METAL_BUFFER_INFO_EXT = 1000311005
VK_STRUCTURE_TYPE_EXPORT_METAL_TEXTURE_INFO_EXT = 1000311006
VK_STRUCTURE_TYPE_IMPORT_METAL_TEXTURE_INFO_EXT = 1000311007
VK_STRUCTURE_TYPE_EXPORT_METAL_IO_SURFACE_INFO_EXT = 1000311008
VK_STRUCTURE_TYPE_IMPORT_METAL_IO_SURFACE_INFO_EXT = 1000311009
VK_STRUCTURE_TYPE_EXPORT_METAL_SHARED_EVENT_INFO_EXT = 1000311010
VK_STRUCTURE_TYPE_IMPORT_METAL_SHARED_EVENT_INFO_EXT = 1000311011
VK_EXT_extension_313 = 1
VK_EXT_EXTENSION_313_SPEC_VERSION = 0
VK_EXT_EXTENSION_313_EXTENSION_NAME = "VK_EXT_extension_313"
VK_AMD_extension_314 = 1
VK_AMD_EXTENSION_314_SPEC_VERSION = 0
VK_AMD_EXTENSION_314_EXTENSION_NAME = "VK_AMD_extension_314"
VK_KHR_synchronization2 = 1
VK_KHR_SYNCHRONIZATION_2_SPEC_VERSION = 1
VK_KHR_SYNCHRONIZATION_2_EXTENSION_NAME = "VK_KHR_synchronization2"
VK_STRUCTURE_TYPE_MEMORY_BARRIER_2_KHR = None
VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER_2_KHR = None
VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER_2_KHR = None
VK_STRUCTURE_TYPE_DEPENDENCY_INFO_KHR = None
VK_STRUCTURE_TYPE_SUBMIT_INFO_2_KHR = None
VK_STRUCTURE_TYPE_SEMAPHORE_SUBMIT_INFO_KHR = None
VK_STRUCTURE_TYPE_COMMAND_BUFFER_SUBMIT_INFO_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SYNCHRONIZATION_2_FEATURES_KHR = None
VK_EVENT_CREATE_DEVICE_ONLY_BIT_KHR = None
VK_IMAGE_LAYOUT_READ_ONLY_OPTIMAL_KHR = None
VK_IMAGE_LAYOUT_ATTACHMENT_OPTIMAL_KHR = None
VK_PIPELINE_STAGE_NONE_KHR = None
VK_ACCESS_NONE_KHR = None
VK_PIPELINE_STAGE_2_TRANSFORM_FEEDBACK_BIT_EXT = 0x01000000
VK_ACCESS_2_TRANSFORM_FEEDBACK_WRITE_BIT_EXT = 0x02000000
VK_ACCESS_2_TRANSFORM_FEEDBACK_COUNTER_READ_BIT_EXT = 0x04000000
VK_ACCESS_2_TRANSFORM_FEEDBACK_COUNTER_WRITE_BIT_EXT = 0x08000000
VK_PIPELINE_STAGE_2_CONDITIONAL_RENDERING_BIT_EXT = 0x00040000
VK_ACCESS_2_CONDITIONAL_RENDERING_READ_BIT_EXT = 0x00100000
VK_PIPELINE_STAGE_2_COMMAND_PREPROCESS_BIT_NV = 0x00020000
VK_ACCESS_2_COMMAND_PREPROCESS_READ_BIT_NV = 0x00020000
VK_ACCESS_2_COMMAND_PREPROCESS_WRITE_BIT_NV = 0x00040000
VK_PIPELINE_STAGE_2_FRAGMENT_SHADING_RATE_ATTACHMENT_BIT_KHR = 0x00400000
VK_ACCESS_2_FRAGMENT_SHADING_RATE_ATTACHMENT_READ_BIT_KHR = 0x00800000
VK_PIPELINE_STAGE_2_SHADING_RATE_IMAGE_BIT_NV = None
VK_ACCESS_2_SHADING_RATE_IMAGE_READ_BIT_NV = None
VK_PIPELINE_STAGE_2_ACCELERATION_STRUCTURE_BUILD_BIT_KHR = 0x02000000
VK_ACCESS_2_ACCELERATION_STRUCTURE_READ_BIT_KHR = 0x00200000
VK_ACCESS_2_ACCELERATION_STRUCTURE_WRITE_BIT_KHR = 0x00400000
VK_PIPELINE_STAGE_2_RAY_TRACING_SHADER_BIT_KHR = 0x00200000
VK_PIPELINE_STAGE_2_RAY_TRACING_SHADER_BIT_NV = None
VK_PIPELINE_STAGE_2_ACCELERATION_STRUCTURE_BUILD_BIT_NV = None
VK_ACCESS_2_ACCELERATION_STRUCTURE_READ_BIT_NV = None
VK_ACCESS_2_ACCELERATION_STRUCTURE_WRITE_BIT_NV = None
VK_PIPELINE_STAGE_2_FRAGMENT_DENSITY_PROCESS_BIT_EXT = 0x00800000
VK_ACCESS_2_FRAGMENT_DENSITY_MAP_READ_BIT_EXT = 0x01000000
VK_ACCESS_2_COLOR_ATTACHMENT_READ_NONCOHERENT_BIT_EXT = 0x00080000
VK_PIPELINE_STAGE_2_TASK_SHADER_BIT_NV = None
VK_PIPELINE_STAGE_2_MESH_SHADER_BIT_NV = None
VK_STRUCTURE_TYPE_QUEUE_FAMILY_CHECKPOINT_PROPERTIES_2_NV = 1000314008
VK_STRUCTURE_TYPE_CHECKPOINT_DATA_2_NV = 1000314009
VK_PIPELINE_STAGE_2_TASK_SHADER_BIT_EXT = 0x00080000
VK_PIPELINE_STAGE_2_MESH_SHADER_BIT_EXT = 0x00100000
VK_AMD_extension_316 = 1
VK_AMD_EXTENSION_316_SPEC_VERSION = 0
VK_AMD_EXTENSION_316_EXTENSION_NAME = "VK_AMD_extension_316"
VK_EXT_descriptor_buffer = 1
VK_EXT_DESCRIPTOR_BUFFER_SPEC_VERSION = 1
VK_EXT_DESCRIPTOR_BUFFER_EXTENSION_NAME = "VK_EXT_descriptor_buffer"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_BUFFER_PROPERTIES_EXT = 1000316000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_BUFFER_DENSITY_MAP_PROPERTIES_EXT = 1000316001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_BUFFER_FEATURES_EXT = 1000316002
VK_STRUCTURE_TYPE_DESCRIPTOR_ADDRESS_INFO_EXT = 1000316003
VK_STRUCTURE_TYPE_DESCRIPTOR_GET_INFO_EXT = 1000316004
VK_STRUCTURE_TYPE_BUFFER_CAPTURE_DESCRIPTOR_DATA_INFO_EXT = 1000316005
VK_STRUCTURE_TYPE_IMAGE_CAPTURE_DESCRIPTOR_DATA_INFO_EXT = 1000316006
VK_STRUCTURE_TYPE_IMAGE_VIEW_CAPTURE_DESCRIPTOR_DATA_INFO_EXT = 1000316007
VK_STRUCTURE_TYPE_SAMPLER_CAPTURE_DESCRIPTOR_DATA_INFO_EXT = 1000316008
VK_STRUCTURE_TYPE_OPAQUE_CAPTURE_DESCRIPTOR_DATA_CREATE_INFO_EXT = 1000316010
VK_STRUCTURE_TYPE_DESCRIPTOR_BUFFER_BINDING_INFO_EXT = 1000316011
VK_STRUCTURE_TYPE_DESCRIPTOR_BUFFER_BINDING_PUSH_DESCRIPTOR_BUFFER_HANDLE_EXT = 1000316012
VK_DESCRIPTOR_SET_LAYOUT_CREATE_DESCRIPTOR_BUFFER_BIT_EXT = 0x00000010
VK_DESCRIPTOR_SET_LAYOUT_CREATE_EMBEDDED_IMMUTABLE_SAMPLERS_BIT_EXT = 0x00000020
VK_BUFFER_USAGE_SAMPLER_DESCRIPTOR_BUFFER_BIT_EXT = 0x00200000
VK_BUFFER_USAGE_RESOURCE_DESCRIPTOR_BUFFER_BIT_EXT = 0x00400000
VK_BUFFER_USAGE_PUSH_DESCRIPTORS_DESCRIPTOR_BUFFER_BIT_EXT = 0x04000000
VK_BUFFER_CREATE_DESCRIPTOR_BUFFER_CAPTURE_REPLAY_BIT_EXT = 0x00000020
VK_IMAGE_CREATE_DESCRIPTOR_BUFFER_CAPTURE_REPLAY_BIT_EXT = 0x00010000
VK_IMAGE_VIEW_CREATE_DESCRIPTOR_BUFFER_CAPTURE_REPLAY_BIT_EXT = 0x00000004
VK_SAMPLER_CREATE_DESCRIPTOR_BUFFER_CAPTURE_REPLAY_BIT_EXT = 0x00000008
VK_ACCELERATION_STRUCTURE_CREATE_DESCRIPTOR_BUFFER_CAPTURE_REPLAY_BIT_EXT = 0x00000008
VK_ACCESS_2_DESCRIPTOR_BUFFER_READ_BIT_EXT = 0x20000000000
VK_PIPELINE_CREATE_DESCRIPTOR_BUFFER_BIT_EXT = 0x20000000
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_CAPTURE_DESCRIPTOR_DATA_INFO_EXT = 1000316009
VK_AMD_extension_318 = 1
VK_AMD_EXTENSION_318_SPEC_VERSION = 0
VK_AMD_EXTENSION_318_EXTENSION_NAME = "VK_AMD_extension_318"
VK_AMD_extension_319 = 1
VK_AMD_EXTENSION_319_SPEC_VERSION = 0
VK_AMD_EXTENSION_319_EXTENSION_NAME = "VK_AMD_extension_319"
VK_DESCRIPTOR_SET_LAYOUT_CREATE_RESERVED_3_BIT_AMD = 0x00000008
VK_PIPELINE_LAYOUT_CREATE_RESERVED_0_BIT_AMD = 0x00000001
VK_AMD_extension_320 = 1
VK_AMD_EXTENSION_320_SPEC_VERSION = 0
VK_AMD_EXTENSION_320_EXTENSION_NAME = "VK_AMD_extension_320"
VK_EXT_graphics_pipeline_library = 1
VK_EXT_GRAPHICS_PIPELINE_LIBRARY_SPEC_VERSION = 1
VK_EXT_GRAPHICS_PIPELINE_LIBRARY_EXTENSION_NAME = "VK_EXT_graphics_pipeline_library"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GRAPHICS_PIPELINE_LIBRARY_FEATURES_EXT = 1000320000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GRAPHICS_PIPELINE_LIBRARY_PROPERTIES_EXT = 1000320001
VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_LIBRARY_CREATE_INFO_EXT = 1000320002
VK_PIPELINE_CREATE_RETAIN_LINK_TIME_OPTIMIZATION_INFO_BIT_EXT = 0x00800000
VK_PIPELINE_CREATE_LINK_TIME_OPTIMIZATION_BIT_EXT = 0x00000400
VK_PIPELINE_LAYOUT_CREATE_INDEPENDENT_SETS_BIT_EXT = 0x00000002
VK_AMD_shader_early_and_late_fragment_tests = 1
VK_AMD_SHADER_EARLY_AND_LATE_FRAGMENT_TESTS_SPEC_VERSION = 1
VK_AMD_SHADER_EARLY_AND_LATE_FRAGMENT_TESTS_EXTENSION_NAME = "VK_AMD_shader_early_and_late_fragment_tests"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_EARLY_AND_LATE_FRAGMENT_TESTS_FEATURES_AMD = 1000321000
VK_KHR_fragment_shader_barycentric = 1
VK_KHR_FRAGMENT_SHADER_BARYCENTRIC_SPEC_VERSION = 1
VK_KHR_FRAGMENT_SHADER_BARYCENTRIC_EXTENSION_NAME = "VK_KHR_fragment_shader_barycentric"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_BARYCENTRIC_FEATURES_KHR = 1000322000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_BARYCENTRIC_PROPERTIES_KHR = 1000322000
VK_KHR_shader_subgroup_uniform_control_flow = 1
VK_KHR_SHADER_SUBGROUP_UNIFORM_CONTROL_FLOW_SPEC_VERSION = 1
VK_KHR_SHADER_SUBGROUP_UNIFORM_CONTROL_FLOW_EXTENSION_NAME = "VK_KHR_shader_subgroup_uniform_control_flow"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_SUBGROUP_UNIFORM_CONTROL_FLOW_FEATURES_KHR = 1000323000
VK_KHR_extension_325 = 1
VK_KHR_EXTENSION_325_SPEC_VERSION = 0
VK_KHR_EXTENSION_325_EXTENSION_NAME = "VK_KHR_extension_325"
VK_KHR_zero_initialize_workgroup_memory = 1
VK_KHR_ZERO_INITIALIZE_WORKGROUP_MEMORY_SPEC_VERSION = 1
VK_KHR_ZERO_INITIALIZE_WORKGROUP_MEMORY_EXTENSION_NAME = "VK_KHR_zero_initialize_workgroup_memory"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ZERO_INITIALIZE_WORKGROUP_MEMORY_FEATURES_KHR = None
VK_NV_fragment_shading_rate_enums = 1
VK_NV_FRAGMENT_SHADING_RATE_ENUMS_SPEC_VERSION = 1
VK_NV_FRAGMENT_SHADING_RATE_ENUMS_EXTENSION_NAME = "VK_NV_fragment_shading_rate_enums"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_ENUMS_PROPERTIES_NV = 1000326000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_ENUMS_FEATURES_NV = 1000326001
VK_STRUCTURE_TYPE_PIPELINE_FRAGMENT_SHADING_RATE_ENUM_STATE_CREATE_INFO_NV = 1000326002
VK_NV_ray_tracing_motion_blur = 1
VK_NV_RAY_TRACING_MOTION_BLUR_SPEC_VERSION = 1
VK_NV_RAY_TRACING_MOTION_BLUR_EXTENSION_NAME = "VK_NV_ray_tracing_motion_blur"
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_MOTION_TRIANGLES_DATA_NV = 1000327000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_MOTION_BLUR_FEATURES_NV = 1000327001
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_MOTION_INFO_NV = 1000327002
VK_BUILD_ACCELERATION_STRUCTURE_MOTION_BIT_NV = 0x00000020
VK_ACCELERATION_STRUCTURE_CREATE_MOTION_BIT_NV = 0x00000004
VK_PIPELINE_CREATE_RAY_TRACING_ALLOW_MOTION_BIT_NV = 0x00100000
VK_EXT_mesh_shader = 1
VK_EXT_MESH_SHADER_SPEC_VERSION = 1
VK_EXT_MESH_SHADER_EXTENSION_NAME = "VK_EXT_mesh_shader"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_FEATURES_EXT = 1000328000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_PROPERTIES_EXT = 1000328001
VK_SHADER_STAGE_TASK_BIT_EXT = 0x00000040
VK_SHADER_STAGE_MESH_BIT_EXT = 0x00000080
VK_PIPELINE_STAGE_TASK_SHADER_BIT_EXT = 0x00080000
VK_PIPELINE_STAGE_MESH_SHADER_BIT_EXT = 0x00100000
VK_QUERY_TYPE_MESH_PRIMITIVES_GENERATED_EXT = 1000328000
VK_QUERY_PIPELINE_STATISTIC_TASK_SHADER_INVOCATIONS_BIT_EXT = 0x00000800
VK_QUERY_PIPELINE_STATISTIC_MESH_SHADER_INVOCATIONS_BIT_EXT = 0x00001000
VK_INDIRECT_COMMANDS_TOKEN_TYPE_DRAW_MESH_TASKS_NV = 1000328000
VK_NV_extension_330 = 1
VK_NV_EXTENSION_330_SPEC_VERSION = 0
VK_NV_EXTENSION_330_EXTENSION_NAME = "VK_NV_extension_330"
VK_EXT_ycbcr_2plane_444_formats = 1
VK_EXT_YCBCR_2PLANE_444_FORMATS_SPEC_VERSION = 1
VK_EXT_YCBCR_2PLANE_444_FORMATS_EXTENSION_NAME = "VK_EXT_ycbcr_2plane_444_formats"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_YCBCR_2_PLANE_444_FORMATS_FEATURES_EXT = 1000330000
VK_FORMAT_G8_B8R8_2PLANE_444_UNORM_EXT = None
VK_FORMAT_G10X6_B10X6R10X6_2PLANE_444_UNORM_3PACK16_EXT = None
VK_FORMAT_G12X4_B12X4R12X4_2PLANE_444_UNORM_3PACK16_EXT = None
VK_FORMAT_G16_B16R16_2PLANE_444_UNORM_EXT = None
VK_NV_extension_332 = 1
VK_NV_EXTENSION_332_SPEC_VERSION = 0
VK_NV_EXTENSION_332_EXTENSION_NAME = "VK_NV_extension_332"
VK_EXT_fragment_density_map2 = 1
VK_EXT_FRAGMENT_DENSITY_MAP_2_SPEC_VERSION = 1
VK_EXT_FRAGMENT_DENSITY_MAP_2_EXTENSION_NAME = "VK_EXT_fragment_density_map2"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_2_FEATURES_EXT = 1000332000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_2_PROPERTIES_EXT = 1000332001
VK_IMAGE_VIEW_CREATE_FRAGMENT_DENSITY_MAP_DEFERRED_BIT_EXT = 0x00000002
VK_QCOM_rotated_copy_commands = 1
VK_QCOM_ROTATED_COPY_COMMANDS_SPEC_VERSION = 2
VK_QCOM_ROTATED_COPY_COMMANDS_EXTENSION_NAME = "VK_QCOM_rotated_copy_commands"
VK_STRUCTURE_TYPE_COPY_COMMAND_TRANSFORM_INFO_QCOM = 1000333000
VK_KHR_extension_335 = 1
VK_KHR_EXTENSION_335_SPEC_VERSION = 0
VK_KHR_EXTENSION_335_EXTENSION_NAME = "VK_KHR_extension_335"
VK_EXT_image_robustness = 1
VK_EXT_IMAGE_ROBUSTNESS_SPEC_VERSION = 1
VK_EXT_IMAGE_ROBUSTNESS_EXTENSION_NAME = "VK_EXT_image_robustness"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_ROBUSTNESS_FEATURES_EXT = None
VK_KHR_workgroup_memory_explicit_layout = 1
VK_KHR_WORKGROUP_MEMORY_EXPLICIT_LAYOUT_SPEC_VERSION = 1
VK_KHR_WORKGROUP_MEMORY_EXPLICIT_LAYOUT_EXTENSION_NAME = "VK_KHR_workgroup_memory_explicit_layout"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_WORKGROUP_MEMORY_EXPLICIT_LAYOUT_FEATURES_KHR = 1000336000
VK_KHR_copy_commands2 = 1
VK_KHR_COPY_COMMANDS_2_SPEC_VERSION = 1
VK_KHR_COPY_COMMANDS_2_EXTENSION_NAME = "VK_KHR_copy_commands2"
VK_STRUCTURE_TYPE_COPY_BUFFER_INFO_2_KHR = None
VK_STRUCTURE_TYPE_COPY_IMAGE_INFO_2_KHR = None
VK_STRUCTURE_TYPE_COPY_BUFFER_TO_IMAGE_INFO_2_KHR = None
VK_STRUCTURE_TYPE_COPY_IMAGE_TO_BUFFER_INFO_2_KHR = None
VK_STRUCTURE_TYPE_BLIT_IMAGE_INFO_2_KHR = None
VK_STRUCTURE_TYPE_RESOLVE_IMAGE_INFO_2_KHR = None
VK_STRUCTURE_TYPE_BUFFER_COPY_2_KHR = None
VK_STRUCTURE_TYPE_IMAGE_COPY_2_KHR = None
VK_STRUCTURE_TYPE_IMAGE_BLIT_2_KHR = None
VK_STRUCTURE_TYPE_BUFFER_IMAGE_COPY_2_KHR = None
VK_STRUCTURE_TYPE_IMAGE_RESOLVE_2_KHR = None
VK_EXT_image_compression_control = 1
VK_EXT_IMAGE_COMPRESSION_CONTROL_SPEC_VERSION = 1
VK_EXT_IMAGE_COMPRESSION_CONTROL_EXTENSION_NAME = "VK_EXT_image_compression_control"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_COMPRESSION_CONTROL_FEATURES_EXT = 1000338000
VK_STRUCTURE_TYPE_IMAGE_COMPRESSION_CONTROL_EXT = 1000338001
VK_STRUCTURE_TYPE_SUBRESOURCE_LAYOUT_2_EXT = None
VK_STRUCTURE_TYPE_IMAGE_SUBRESOURCE_2_EXT = None
VK_STRUCTURE_TYPE_IMAGE_COMPRESSION_PROPERTIES_EXT = 1000338004
VK_EXT_attachment_feedback_loop_layout = 1
VK_EXT_ATTACHMENT_FEEDBACK_LOOP_LAYOUT_SPEC_VERSION = 2
VK_EXT_ATTACHMENT_FEEDBACK_LOOP_LAYOUT_EXTENSION_NAME = "VK_EXT_attachment_feedback_loop_layout"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ATTACHMENT_FEEDBACK_LOOP_LAYOUT_FEATURES_EXT = 1000339000
VK_IMAGE_LAYOUT_ATTACHMENT_FEEDBACK_LOOP_OPTIMAL_EXT = 1000339000
VK_IMAGE_USAGE_ATTACHMENT_FEEDBACK_LOOP_BIT_EXT = 0x00080000
VK_PIPELINE_CREATE_COLOR_ATTACHMENT_FEEDBACK_LOOP_BIT_EXT = 0x02000000
VK_PIPELINE_CREATE_DEPTH_STENCIL_ATTACHMENT_FEEDBACK_LOOP_BIT_EXT = 0x04000000
VK_DEPENDENCY_FEEDBACK_LOOP_BIT_EXT = 0x00000008
VK_EXT_4444_formats = 1
VK_EXT_4444_FORMATS_SPEC_VERSION = 1
VK_EXT_4444_FORMATS_EXTENSION_NAME = "VK_EXT_4444_formats"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_4444_FORMATS_FEATURES_EXT = 1000340000
VK_FORMAT_A4R4G4B4_UNORM_PACK16_EXT = None
VK_FORMAT_A4B4G4R4_UNORM_PACK16_EXT = None
VK_EXT_device_fault = 1
VK_EXT_DEVICE_FAULT_SPEC_VERSION = 2
VK_EXT_DEVICE_FAULT_EXTENSION_NAME = "VK_EXT_device_fault"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FAULT_FEATURES_EXT = 1000341000
VK_STRUCTURE_TYPE_DEVICE_FAULT_COUNTS_EXT = 1000341001
VK_STRUCTURE_TYPE_DEVICE_FAULT_INFO_EXT = 1000341002
VK_ARM_rasterization_order_attachment_access = 1
VK_ARM_RASTERIZATION_ORDER_ATTACHMENT_ACCESS_SPEC_VERSION = 1
VK_ARM_RASTERIZATION_ORDER_ATTACHMENT_ACCESS_EXTENSION_NAME = "VK_ARM_rasterization_order_attachment_access"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RASTERIZATION_ORDER_ATTACHMENT_ACCESS_FEATURES_ARM = None
VK_PIPELINE_COLOR_BLEND_STATE_CREATE_RASTERIZATION_ORDER_ATTACHMENT_ACCESS_BIT_ARM = None
VK_PIPELINE_DEPTH_STENCIL_STATE_CREATE_RASTERIZATION_ORDER_ATTACHMENT_DEPTH_ACCESS_BIT_ARM = None
VK_PIPELINE_DEPTH_STENCIL_STATE_CREATE_RASTERIZATION_ORDER_ATTACHMENT_STENCIL_ACCESS_BIT_ARM = None
VK_SUBPASS_DESCRIPTION_RASTERIZATION_ORDER_ATTACHMENT_COLOR_ACCESS_BIT_ARM = None
VK_SUBPASS_DESCRIPTION_RASTERIZATION_ORDER_ATTACHMENT_DEPTH_ACCESS_BIT_ARM = None
VK_SUBPASS_DESCRIPTION_RASTERIZATION_ORDER_ATTACHMENT_STENCIL_ACCESS_BIT_ARM = None
VK_ARM_extension_344 = 1
VK_ARM_EXTENSION_344_SPEC_VERSION = 0
VK_ARM_EXTENSION_344_EXTENSION_NAME = "VK_ARM_extension_344"
VK_EXT_rgba10x6_formats = 1
VK_EXT_RGBA10X6_FORMATS_SPEC_VERSION = 1
VK_EXT_RGBA10X6_FORMATS_EXTENSION_NAME = "VK_EXT_rgba10x6_formats"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RGBA10X6_FORMATS_FEATURES_EXT = 1000344000
VK_NV_acquire_winrt_display = 1
VK_NV_ACQUIRE_WINRT_DISPLAY_SPEC_VERSION = 1
VK_NV_ACQUIRE_WINRT_DISPLAY_EXTENSION_NAME = "VK_NV_acquire_winrt_display"
VK_EXT_directfb_surface = 1
VK_EXT_DIRECTFB_SURFACE_SPEC_VERSION = 1
VK_EXT_DIRECTFB_SURFACE_EXTENSION_NAME = "VK_EXT_directfb_surface"
VK_STRUCTURE_TYPE_DIRECTFB_SURFACE_CREATE_INFO_EXT = 1000346000
VK_KHR_extension_350 = 1
VK_KHR_EXTENSION_350_SPEC_VERSION = 0
VK_KHR_EXTENSION_350_EXTENSION_NAME = "VK_KHR_extension_350"
VK_NV_extension_351 = 1
VK_NV_EXTENSION_351_SPEC_VERSION = 0
VK_NV_EXTENSION_351_EXTENSION_NAME = "VK_NV_extension_351"
VK_VALVE_mutable_descriptor_type = 1
VK_VALVE_MUTABLE_DESCRIPTOR_TYPE_SPEC_VERSION = 1
VK_VALVE_MUTABLE_DESCRIPTOR_TYPE_EXTENSION_NAME = "VK_VALVE_mutable_descriptor_type"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MUTABLE_DESCRIPTOR_TYPE_FEATURES_VALVE = None
VK_STRUCTURE_TYPE_MUTABLE_DESCRIPTOR_TYPE_CREATE_INFO_VALVE = None
VK_DESCRIPTOR_TYPE_MUTABLE_VALVE = None
VK_DESCRIPTOR_POOL_CREATE_HOST_ONLY_BIT_VALVE = None
VK_DESCRIPTOR_SET_LAYOUT_CREATE_HOST_ONLY_POOL_BIT_VALVE = None
VK_EXT_vertex_input_dynamic_state = 1
VK_EXT_VERTEX_INPUT_DYNAMIC_STATE_SPEC_VERSION = 2
VK_EXT_VERTEX_INPUT_DYNAMIC_STATE_EXTENSION_NAME = "VK_EXT_vertex_input_dynamic_state"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_INPUT_DYNAMIC_STATE_FEATURES_EXT = 1000352000
VK_STRUCTURE_TYPE_VERTEX_INPUT_BINDING_DESCRIPTION_2_EXT = 1000352001
VK_STRUCTURE_TYPE_VERTEX_INPUT_ATTRIBUTE_DESCRIPTION_2_EXT = 1000352002
VK_DYNAMIC_STATE_VERTEX_INPUT_EXT = 1000352000
VK_EXT_physical_device_drm = 1
VK_EXT_PHYSICAL_DEVICE_DRM_SPEC_VERSION = 1
VK_EXT_PHYSICAL_DEVICE_DRM_EXTENSION_NAME = "VK_EXT_physical_device_drm"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DRM_PROPERTIES_EXT = 1000353000
VK_EXT_device_address_binding_report = 1
VK_EXT_DEVICE_ADDRESS_BINDING_REPORT_SPEC_VERSION = 1
VK_EXT_DEVICE_ADDRESS_BINDING_REPORT_EXTENSION_NAME = "VK_EXT_device_address_binding_report"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ADDRESS_BINDING_REPORT_FEATURES_EXT = 1000354000
VK_STRUCTURE_TYPE_DEVICE_ADDRESS_BINDING_CALLBACK_DATA_EXT = 1000354001
VK_DEBUG_UTILS_MESSAGE_TYPE_DEVICE_ADDRESS_BINDING_BIT_EXT = 0x00000008
VK_EXT_depth_clip_control = 1
VK_EXT_DEPTH_CLIP_CONTROL_SPEC_VERSION = 1
VK_EXT_DEPTH_CLIP_CONTROL_EXTENSION_NAME = "VK_EXT_depth_clip_control"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_CLIP_CONTROL_FEATURES_EXT = 1000355000
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_DEPTH_CLIP_CONTROL_CREATE_INFO_EXT = 1000355001
VK_EXT_primitive_topology_list_restart = 1
VK_EXT_PRIMITIVE_TOPOLOGY_LIST_RESTART_SPEC_VERSION = 1
VK_EXT_PRIMITIVE_TOPOLOGY_LIST_RESTART_EXTENSION_NAME = "VK_EXT_primitive_topology_list_restart"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRIMITIVE_TOPOLOGY_LIST_RESTART_FEATURES_EXT = 1000356000
VK_KHR_extension_358 = 1
VK_KHR_EXTENSION_358_SPEC_VERSION = 0
VK_KHR_EXTENSION_358_EXTENSION_NAME = "VK_KHR_extension_358"
VK_EXT_extension_359 = 1
VK_EXT_EXTENSION_359_SPEC_VERSION = 0
VK_EXT_EXTENSION_359_EXTENSION_NAME = "VK_EXT_extension_359"
VK_EXT_extension_360 = 1
VK_EXT_EXTENSION_360_SPEC_VERSION = 0
VK_EXT_EXTENSION_360_EXTENSION_NAME = "VK_EXT_extension_360"
VK_KHR_format_feature_flags2 = 1
VK_KHR_FORMAT_FEATURE_FLAGS_2_SPEC_VERSION = 2
VK_KHR_FORMAT_FEATURE_FLAGS_2_EXTENSION_NAME = "VK_KHR_format_feature_flags2"
VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_3_KHR = None
VK_EXT_extension_362 = 1
VK_EXT_EXTENSION_362_SPEC_VERSION = 0
VK_EXT_EXTENSION_362_EXTENSION_NAME = "VK_EXT_extension_362"
VK_EXT_extension_363 = 1
VK_EXT_EXTENSION_363_SPEC_VERSION = 0
VK_EXT_EXTENSION_363_EXTENSION_NAME = "VK_EXT_extension_363"
VK_FUCHSIA_extension_364 = 1
VK_FUCHSIA_EXTENSION_364_SPEC_VERSION = 0
VK_FUCHSIA_EXTENSION_364_EXTENSION_NAME = "VK_FUCHSIA_extension_364"
VK_FUCHSIA_external_memory = 1
VK_FUCHSIA_EXTERNAL_MEMORY_SPEC_VERSION = 1
VK_FUCHSIA_EXTERNAL_MEMORY_EXTENSION_NAME = "VK_FUCHSIA_external_memory"
VK_STRUCTURE_TYPE_IMPORT_MEMORY_ZIRCON_HANDLE_INFO_FUCHSIA = 1000364000
VK_STRUCTURE_TYPE_MEMORY_ZIRCON_HANDLE_PROPERTIES_FUCHSIA = 1000364001
VK_STRUCTURE_TYPE_MEMORY_GET_ZIRCON_HANDLE_INFO_FUCHSIA = 1000364002
VK_EXTERNAL_MEMORY_HANDLE_TYPE_ZIRCON_VMO_BIT_FUCHSIA = 0x00000800
VK_FUCHSIA_external_semaphore = 1
VK_FUCHSIA_EXTERNAL_SEMAPHORE_SPEC_VERSION = 1
VK_FUCHSIA_EXTERNAL_SEMAPHORE_EXTENSION_NAME = "VK_FUCHSIA_external_semaphore"
VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_ZIRCON_HANDLE_INFO_FUCHSIA = 1000365000
VK_STRUCTURE_TYPE_SEMAPHORE_GET_ZIRCON_HANDLE_INFO_FUCHSIA = 1000365001
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_ZIRCON_EVENT_BIT_FUCHSIA = 0x00000080
VK_FUCHSIA_buffer_collection = 1
VK_FUCHSIA_BUFFER_COLLECTION_SPEC_VERSION = 2
VK_FUCHSIA_BUFFER_COLLECTION_EXTENSION_NAME = "VK_FUCHSIA_buffer_collection"
VK_STRUCTURE_TYPE_BUFFER_COLLECTION_CREATE_INFO_FUCHSIA = 1000366000
VK_OBJECT_TYPE_BUFFER_COLLECTION_FUCHSIA = 1000366000
VK_STRUCTURE_TYPE_IMPORT_MEMORY_BUFFER_COLLECTION_FUCHSIA = 1000366001
VK_STRUCTURE_TYPE_BUFFER_COLLECTION_IMAGE_CREATE_INFO_FUCHSIA = 1000366002
VK_STRUCTURE_TYPE_BUFFER_COLLECTION_PROPERTIES_FUCHSIA = 1000366003
VK_STRUCTURE_TYPE_BUFFER_CONSTRAINTS_INFO_FUCHSIA = 1000366004
VK_STRUCTURE_TYPE_BUFFER_COLLECTION_BUFFER_CREATE_INFO_FUCHSIA = 1000366005
VK_STRUCTURE_TYPE_IMAGE_CONSTRAINTS_INFO_FUCHSIA = 1000366006
VK_STRUCTURE_TYPE_IMAGE_FORMAT_CONSTRAINTS_INFO_FUCHSIA = 1000366007
VK_STRUCTURE_TYPE_SYSMEM_COLOR_SPACE_FUCHSIA = 1000366008
VK_STRUCTURE_TYPE_BUFFER_COLLECTION_CONSTRAINTS_INFO_FUCHSIA = 1000366009
VK_DEBUG_REPORT_OBJECT_TYPE_BUFFER_COLLECTION_FUCHSIA_EXT = 1000366000
VK_FUCHSIA_extension_368 = 1
VK_FUCHSIA_EXTENSION_368_SPEC_VERSION = 0
VK_FUCHSIA_EXTENSION_368_EXTENSION_NAME = "VK_FUCHSIA_extension_368"
VK_QCOM_extension_369 = 1
VK_QCOM_EXTENSION_369_SPEC_VERSION = 0
VK_QCOM_EXTENSION_369_EXTENSION_NAME = "VK_QCOM_extension_369"
VK_DESCRIPTOR_BINDING_RESERVED_4_BIT_QCOM = 0x00000010
VK_HUAWEI_subpass_shading = 1
VK_HUAWEI_SUBPASS_SHADING_SPEC_VERSION = 3
VK_HUAWEI_SUBPASS_SHADING_EXTENSION_NAME = "VK_HUAWEI_subpass_shading"
VK_STRUCTURE_TYPE_SUBPASS_SHADING_PIPELINE_CREATE_INFO_HUAWEI = 1000369000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBPASS_SHADING_FEATURES_HUAWEI = 1000369001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBPASS_SHADING_PROPERTIES_HUAWEI = 1000369002
VK_PIPELINE_BIND_POINT_SUBPASS_SHADING_HUAWEI = 1000369003
VK_PIPELINE_STAGE_2_SUBPASS_SHADER_BIT_HUAWEI = 0x8000000000
VK_PIPELINE_STAGE_2_SUBPASS_SHADING_BIT_HUAWEI = None
VK_SHADER_STAGE_SUBPASS_SHADING_BIT_HUAWEI = 0x00004000
VK_HUAWEI_invocation_mask = 1
VK_HUAWEI_INVOCATION_MASK_SPEC_VERSION = 1
VK_HUAWEI_INVOCATION_MASK_EXTENSION_NAME = "VK_HUAWEI_invocation_mask"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INVOCATION_MASK_FEATURES_HUAWEI = 1000370000
VK_ACCESS_2_INVOCATION_MASK_READ_BIT_HUAWEI = 0x8000000000
VK_IMAGE_USAGE_INVOCATION_MASK_BIT_HUAWEI = 0x00040000
VK_PIPELINE_STAGE_2_INVOCATION_MASK_BIT_HUAWEI = 0x10000000000
VK_NV_external_memory_rdma = 1
VK_NV_EXTERNAL_MEMORY_RDMA_SPEC_VERSION = 1
VK_NV_EXTERNAL_MEMORY_RDMA_EXTENSION_NAME = "VK_NV_external_memory_rdma"
VK_STRUCTURE_TYPE_MEMORY_GET_REMOTE_ADDRESS_INFO_NV = 1000371000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_RDMA_FEATURES_NV = 1000371001
VK_MEMORY_PROPERTY_RDMA_CAPABLE_BIT_NV = 0x00000100
VK_EXTERNAL_MEMORY_HANDLE_TYPE_RDMA_ADDRESS_BIT_NV = 0x00001000
VK_EXT_pipeline_properties = 1
VK_EXT_PIPELINE_PROPERTIES_SPEC_VERSION = 1
VK_EXT_PIPELINE_PROPERTIES_EXTENSION_NAME = "VK_EXT_pipeline_properties"
VK_STRUCTURE_TYPE_PIPELINE_PROPERTIES_IDENTIFIER_EXT = 1000372000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_PROPERTIES_FEATURES_EXT = 1000372001
VK_STRUCTURE_TYPE_PIPELINE_INFO_EXT = None
VK_NV_external_sci_sync = 1
VK_NV_EXTERNAL_SCI_SYNC_SPEC_VERSION = 2
VK_NV_EXTERNAL_SCI_SYNC_EXTENSION_NAME = "VK_NV_external_sci_sync"
VK_STRUCTURE_TYPE_IMPORT_FENCE_SCI_SYNC_INFO_NV = 1000373000
VK_STRUCTURE_TYPE_EXPORT_FENCE_SCI_SYNC_INFO_NV = 1000373001
VK_STRUCTURE_TYPE_FENCE_GET_SCI_SYNC_INFO_NV = 1000373002
VK_STRUCTURE_TYPE_SCI_SYNC_ATTRIBUTES_INFO_NV = 1000373003
VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_SCI_SYNC_INFO_NV = 1000373004
VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_SCI_SYNC_INFO_NV = 1000373005
VK_STRUCTURE_TYPE_SEMAPHORE_GET_SCI_SYNC_INFO_NV = 1000373006
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SCI_SYNC_FEATURES_NV = 1000373007
VK_EXTERNAL_FENCE_HANDLE_TYPE_SCI_SYNC_OBJ_BIT_NV = 0x00000010
VK_EXTERNAL_FENCE_HANDLE_TYPE_SCI_SYNC_FENCE_BIT_NV = 0x00000020
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_SCI_SYNC_OBJ_BIT_NV = 0x00000020
VK_NV_external_memory_sci_buf = 1
VK_NV_EXTERNAL_MEMORY_SCI_BUF_SPEC_VERSION = 2
VK_NV_EXTERNAL_MEMORY_SCI_BUF_EXTENSION_NAME = "VK_NV_external_memory_sci_buf"
VK_STRUCTURE_TYPE_IMPORT_MEMORY_SCI_BUF_INFO_NV = 1000374000
VK_STRUCTURE_TYPE_EXPORT_MEMORY_SCI_BUF_INFO_NV = 1000374001
VK_STRUCTURE_TYPE_MEMORY_GET_SCI_BUF_INFO_NV = 1000374002
VK_STRUCTURE_TYPE_MEMORY_SCI_BUF_PROPERTIES_NV = 1000374003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_SCI_BUF_FEATURES_NV = 1000374004
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SCI_BUF_FEATURES_NV = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_SCI_BUF_BIT_NV = 0x00002000
VK_EXT_frame_boundary = 1
VK_EXT_FRAME_BOUNDARY_SPEC_VERSION = 1
VK_EXT_FRAME_BOUNDARY_EXTENSION_NAME = "VK_EXT_frame_boundary"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAME_BOUNDARY_FEATURES_EXT = 1000375000
VK_STRUCTURE_TYPE_FRAME_BOUNDARY_EXT = 1000375001
VK_EXT_multisampled_render_to_single_sampled = 1
VK_EXT_MULTISAMPLED_RENDER_TO_SINGLE_SAMPLED_SPEC_VERSION = 1
VK_EXT_MULTISAMPLED_RENDER_TO_SINGLE_SAMPLED_EXTENSION_NAME = "VK_EXT_multisampled_render_to_single_sampled"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTISAMPLED_RENDER_TO_SINGLE_SAMPLED_FEATURES_EXT = 1000376000
VK_STRUCTURE_TYPE_SUBPASS_RESOLVE_PERFORMANCE_QUERY_EXT = 1000376001
VK_STRUCTURE_TYPE_MULTISAMPLED_RENDER_TO_SINGLE_SAMPLED_INFO_EXT = 1000376002
VK_IMAGE_CREATE_MULTISAMPLED_RENDER_TO_SINGLE_SAMPLED_BIT_EXT = 0x00040000
VK_EXT_extended_dynamic_state2 = 1
VK_EXT_EXTENDED_DYNAMIC_STATE_2_SPEC_VERSION = 1
VK_EXT_EXTENDED_DYNAMIC_STATE_2_EXTENSION_NAME = "VK_EXT_extended_dynamic_state2"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_DYNAMIC_STATE_2_FEATURES_EXT = 1000377000
VK_DYNAMIC_STATE_PATCH_CONTROL_POINTS_EXT = 1000377000
VK_DYNAMIC_STATE_RASTERIZER_DISCARD_ENABLE_EXT = None
VK_DYNAMIC_STATE_DEPTH_BIAS_ENABLE_EXT = None
VK_DYNAMIC_STATE_LOGIC_OP_EXT = 1000377003
VK_DYNAMIC_STATE_PRIMITIVE_RESTART_ENABLE_EXT = None
VK_QNX_screen_surface = 1
VK_QNX_SCREEN_SURFACE_SPEC_VERSION = 1
VK_QNX_SCREEN_SURFACE_EXTENSION_NAME = "VK_QNX_screen_surface"
VK_STRUCTURE_TYPE_SCREEN_SURFACE_CREATE_INFO_QNX = 1000378000
VK_KHR_extension_380 = 1
VK_KHR_EXTENSION_380_SPEC_VERSION = 0
VK_KHR_EXTENSION_380_EXTENSION_NAME = "VK_KHR_extension_380"
VK_KHR_extension_381 = 1
VK_KHR_EXTENSION_381_SPEC_VERSION = 0
VK_KHR_EXTENSION_381_EXTENSION_NAME = "VK_KHR_extension_381"
VK_EXT_color_write_enable = 1
VK_EXT_COLOR_WRITE_ENABLE_SPEC_VERSION = 1
VK_EXT_COLOR_WRITE_ENABLE_EXTENSION_NAME = "VK_EXT_color_write_enable"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COLOR_WRITE_ENABLE_FEATURES_EXT = 1000381000
VK_STRUCTURE_TYPE_PIPELINE_COLOR_WRITE_CREATE_INFO_EXT = 1000381001
VK_DYNAMIC_STATE_COLOR_WRITE_ENABLE_EXT = 1000381000
VK_EXT_primitives_generated_query = 1
VK_EXT_PRIMITIVES_GENERATED_QUERY_SPEC_VERSION = 1
VK_EXT_PRIMITIVES_GENERATED_QUERY_EXTENSION_NAME = "VK_EXT_primitives_generated_query"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRIMITIVES_GENERATED_QUERY_FEATURES_EXT = 1000382000
VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT = 1000382000
VK_EXT_extension_384 = 1
VK_EXT_EXTENSION_384_SPEC_VERSION = 0
VK_EXT_EXTENSION_384_EXTENSION_NAME = "VK_EXT_extension_384"
VK_MESA_extension_385 = 1
VK_MESA_EXTENSION_385_SPEC_VERSION = 0
VK_MESA_EXTENSION_385_EXTENSION_NAME = "VK_MESA_extension_385"
VK_GOOGLE_extension_386 = 1
VK_GOOGLE_EXTENSION_386_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_386_EXTENSION_NAME = "VK_GOOGLE_extension_386"
VK_KHR_ray_tracing_maintenance1 = 1
VK_KHR_RAY_TRACING_MAINTENANCE_1_SPEC_VERSION = 1
VK_KHR_RAY_TRACING_MAINTENANCE_1_EXTENSION_NAME = "VK_KHR_ray_tracing_maintenance1"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_MAINTENANCE_1_FEATURES_KHR = 1000386000
VK_QUERY_TYPE_ACCELERATION_STRUCTURE_SERIALIZATION_BOTTOM_LEVEL_POINTERS_KHR = 1000386000
VK_QUERY_TYPE_ACCELERATION_STRUCTURE_SIZE_KHR = 1000386001
VK_PIPELINE_STAGE_2_ACCELERATION_STRUCTURE_COPY_BIT_KHR = 0x10000000
VK_ACCESS_2_SHADER_BINDING_TABLE_READ_BIT_KHR = 0x10000000000
VK_EXT_extension_388 = 1
VK_EXT_EXTENSION_388_SPEC_VERSION = 0
VK_EXT_EXTENSION_388_EXTENSION_NAME = "VK_EXT_extension_388"
VK_EXT_global_priority_query = 1
VK_EXT_GLOBAL_PRIORITY_QUERY_SPEC_VERSION = 1
VK_EXT_GLOBAL_PRIORITY_QUERY_EXTENSION_NAME = "VK_EXT_global_priority_query"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GLOBAL_PRIORITY_QUERY_FEATURES_EXT = None
VK_STRUCTURE_TYPE_QUEUE_FAMILY_GLOBAL_PRIORITY_PROPERTIES_EXT = None
VK_MAX_GLOBAL_PRIORITY_SIZE_EXT = None
VK_EXT_extension_390 = 1
VK_EXT_EXTENSION_390_SPEC_VERSION = 0
VK_EXT_EXTENSION_390_EXTENSION_NAME = "VK_EXT_extension_390"
VK_EXT_extension_391 = 1
VK_EXT_EXTENSION_391_SPEC_VERSION = 0
VK_EXT_EXTENSION_391_EXTENSION_NAME = "VK_EXT_extension_391"
VK_EXT_image_view_min_lod = 1
VK_EXT_IMAGE_VIEW_MIN_LOD_SPEC_VERSION = 1
VK_EXT_IMAGE_VIEW_MIN_LOD_EXTENSION_NAME = "VK_EXT_image_view_min_lod"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_VIEW_MIN_LOD_FEATURES_EXT = 1000391000
VK_STRUCTURE_TYPE_IMAGE_VIEW_MIN_LOD_CREATE_INFO_EXT = 1000391001
VK_EXT_multi_draw = 1
VK_EXT_MULTI_DRAW_SPEC_VERSION = 1
VK_EXT_MULTI_DRAW_EXTENSION_NAME = "VK_EXT_multi_draw"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTI_DRAW_FEATURES_EXT = 1000392000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTI_DRAW_PROPERTIES_EXT = 1000392001
VK_EXT_image_2d_view_of_3d = 1
VK_EXT_IMAGE_2D_VIEW_OF_3D_SPEC_VERSION = 1
VK_EXT_IMAGE_2D_VIEW_OF_3D_EXTENSION_NAME = "VK_EXT_image_2d_view_of_3d"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_2D_VIEW_OF_3D_FEATURES_EXT = 1000393000
VK_IMAGE_CREATE_2D_VIEW_COMPATIBLE_BIT_EXT = 0x00020000
VK_KHR_portability_enumeration = 1
VK_KHR_PORTABILITY_ENUMERATION_SPEC_VERSION = 1
VK_KHR_PORTABILITY_ENUMERATION_EXTENSION_NAME = "VK_KHR_portability_enumeration"
VK_INSTANCE_CREATE_ENUMERATE_PORTABILITY_BIT_KHR = 0x00000001
VK_EXT_shader_tile_image = 1
VK_EXT_SHADER_TILE_IMAGE_SPEC_VERSION = 1
VK_EXT_SHADER_TILE_IMAGE_EXTENSION_NAME = "VK_EXT_shader_tile_image"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_TILE_IMAGE_FEATURES_EXT = 1000395000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_TILE_IMAGE_PROPERTIES_EXT = 1000395001
VK_EXT_opacity_micromap = 1
VK_EXT_OPACITY_MICROMAP_SPEC_VERSION = 2
VK_EXT_OPACITY_MICROMAP_EXTENSION_NAME = "VK_EXT_opacity_micromap"
VK_STRUCTURE_TYPE_MICROMAP_BUILD_INFO_EXT = 1000396000
VK_STRUCTURE_TYPE_MICROMAP_VERSION_INFO_EXT = 1000396001
VK_STRUCTURE_TYPE_COPY_MICROMAP_INFO_EXT = 1000396002
VK_STRUCTURE_TYPE_COPY_MICROMAP_TO_MEMORY_INFO_EXT = 1000396003
VK_STRUCTURE_TYPE_COPY_MEMORY_TO_MICROMAP_INFO_EXT = 1000396004
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_OPACITY_MICROMAP_FEATURES_EXT = 1000396005
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_OPACITY_MICROMAP_PROPERTIES_EXT = 1000396006
VK_STRUCTURE_TYPE_MICROMAP_CREATE_INFO_EXT = 1000396007
VK_STRUCTURE_TYPE_MICROMAP_BUILD_SIZES_INFO_EXT = 1000396008
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_TRIANGLES_OPACITY_MICROMAP_EXT = 1000396009
VK_PIPELINE_STAGE_2_MICROMAP_BUILD_BIT_EXT = 0x40000000
VK_ACCESS_2_MICROMAP_READ_BIT_EXT = 0x100000000000
VK_ACCESS_2_MICROMAP_WRITE_BIT_EXT = 0x200000000000
VK_QUERY_TYPE_MICROMAP_SERIALIZATION_SIZE_EXT = 1000396000
VK_QUERY_TYPE_MICROMAP_COMPACTED_SIZE_EXT = 1000396001
VK_OBJECT_TYPE_MICROMAP_EXT = 1000396000
VK_BUFFER_USAGE_MICROMAP_BUILD_INPUT_READ_ONLY_BIT_EXT = 0x00800000
VK_BUFFER_USAGE_MICROMAP_STORAGE_BIT_EXT = 0x01000000
VK_PIPELINE_CREATE_RAY_TRACING_OPACITY_MICROMAP_BIT_EXT = 0x01000000
VK_GEOMETRY_INSTANCE_FORCE_OPACITY_MICROMAP_2_STATE_EXT = 0x00000010
VK_GEOMETRY_INSTANCE_DISABLE_OPACITY_MICROMAPS_EXT = 0x00000020
VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_OPACITY_MICROMAP_UPDATE_EXT = 0x00000040
VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_DISABLE_OPACITY_MICROMAPS_EXT = 0x00000080
VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_OPACITY_MICROMAP_DATA_UPDATE_EXT = 0x00000100
VK_NV_displacement_micromap = 1
VK_NV_DISPLACEMENT_MICROMAP_SPEC_VERSION = 2
VK_NV_DISPLACEMENT_MICROMAP_EXTENSION_NAME = "VK_NV_displacement_micromap"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DISPLACEMENT_MICROMAP_FEATURES_NV = 1000397000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DISPLACEMENT_MICROMAP_PROPERTIES_NV = 1000397001
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_TRIANGLES_DISPLACEMENT_MICROMAP_NV = 1000397002
VK_PIPELINE_CREATE_RAY_TRACING_DISPLACEMENT_MICROMAP_BIT_NV = 0x10000000
VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_DISPLACEMENT_MICROMAP_UPDATE_NV = 0x00000200
VK_MICROMAP_TYPE_DISPLACEMENT_MICROMAP_NV = 1000397000
VK_JUICE_extension_399 = 1
VK_JUICE_EXTENSION_399_SPEC_VERSION = 0
VK_JUICE_EXTENSION_399_EXTENSION_NAME = "VK_JUICE_extension_399"
VK_JUICE_extension_400 = 1
VK_JUICE_EXTENSION_400_SPEC_VERSION = 0
VK_JUICE_EXTENSION_400_EXTENSION_NAME = "VK_JUICE_extension_400"
VK_EXT_load_store_op_none = 1
VK_EXT_LOAD_STORE_OP_NONE_SPEC_VERSION = 1
VK_EXT_LOAD_STORE_OP_NONE_EXTENSION_NAME = "VK_EXT_load_store_op_none"
VK_ATTACHMENT_LOAD_OP_NONE_EXT = 1000400000
VK_ATTACHMENT_STORE_OP_NONE_EXT = None
VK_FB_extension_402 = 1
VK_FB_EXTENSION_402_SPEC_VERSION = 0
VK_FB_EXTENSION_402_EXTENSION_NAME = "VK_FB_extension_402"
VK_FB_extension_403 = 1
VK_FB_EXTENSION_403_SPEC_VERSION = 0
VK_FB_EXTENSION_403_EXTENSION_NAME = "VK_FB_extension_403"
VK_FB_extension_404 = 1
VK_FB_EXTENSION_404_SPEC_VERSION = 0
VK_FB_EXTENSION_404_EXTENSION_NAME = "VK_FB_extension_404"
VK_HUAWEI_cluster_culling_shader = 1
VK_HUAWEI_CLUSTER_CULLING_SHADER_SPEC_VERSION = 3
VK_HUAWEI_CLUSTER_CULLING_SHADER_EXTENSION_NAME = "VK_HUAWEI_cluster_culling_shader"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CLUSTER_CULLING_SHADER_FEATURES_HUAWEI = 1000404000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CLUSTER_CULLING_SHADER_PROPERTIES_HUAWEI = 1000404001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CLUSTER_CULLING_SHADER_VRS_FEATURES_HUAWEI = 1000404002
VK_PIPELINE_STAGE_2_CLUSTER_CULLING_SHADER_BIT_HUAWEI = 0x20000000000
VK_SHADER_STAGE_CLUSTER_CULLING_BIT_HUAWEI = 0x00080000
VK_QUERY_PIPELINE_STATISTIC_CLUSTER_CULLING_SHADER_INVOCATIONS_BIT_HUAWEI = 0x00002000
VK_HUAWEI_extension_406 = 1
VK_HUAWEI_EXTENSION_406_SPEC_VERSION = 0
VK_HUAWEI_EXTENSION_406_EXTENSION_NAME = "VK_HUAWEI_extension_406"
VK_GGP_extension_407 = 1
VK_GGP_EXTENSION_407_SPEC_VERSION = 0
VK_GGP_EXTENSION_407_EXTENSION_NAME = "VK_GGP_extension_407"
VK_GGP_extension_408 = 1
VK_GGP_EXTENSION_408_SPEC_VERSION = 0
VK_GGP_EXTENSION_408_EXTENSION_NAME = "VK_GGP_extension_408"
VK_GGP_extension_409 = 1
VK_GGP_EXTENSION_409_SPEC_VERSION = 0
VK_GGP_EXTENSION_409_EXTENSION_NAME = "VK_GGP_extension_409"
VK_GGP_extension_410 = 1
VK_GGP_EXTENSION_410_SPEC_VERSION = 0
VK_GGP_EXTENSION_410_EXTENSION_NAME = "VK_GGP_extension_410"
VK_GGP_extension_411 = 1
VK_GGP_EXTENSION_411_SPEC_VERSION = 0
VK_GGP_EXTENSION_411_EXTENSION_NAME = "VK_GGP_extension_411"
VK_EXT_border_color_swizzle = 1
VK_EXT_BORDER_COLOR_SWIZZLE_SPEC_VERSION = 1
VK_EXT_BORDER_COLOR_SWIZZLE_EXTENSION_NAME = "VK_EXT_border_color_swizzle"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BORDER_COLOR_SWIZZLE_FEATURES_EXT = 1000411000
VK_STRUCTURE_TYPE_SAMPLER_BORDER_COLOR_COMPONENT_MAPPING_CREATE_INFO_EXT = 1000411001
VK_EXT_pageable_device_local_memory = 1
VK_EXT_PAGEABLE_DEVICE_LOCAL_MEMORY_SPEC_VERSION = 1
VK_EXT_PAGEABLE_DEVICE_LOCAL_MEMORY_EXTENSION_NAME = "VK_EXT_pageable_device_local_memory"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PAGEABLE_DEVICE_LOCAL_MEMORY_FEATURES_EXT = 1000412000
VK_KHR_maintenance4 = 1
VK_KHR_MAINTENANCE_4_SPEC_VERSION = 2
VK_KHR_MAINTENANCE_4_EXTENSION_NAME = "VK_KHR_maintenance4"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_4_FEATURES_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_4_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_DEVICE_BUFFER_MEMORY_REQUIREMENTS_KHR = None
VK_STRUCTURE_TYPE_DEVICE_IMAGE_MEMORY_REQUIREMENTS_KHR = None
VK_IMAGE_ASPECT_NONE_KHR = None
VK_HUAWEI_extension_415 = 1
VK_HUAWEI_EXTENSION_415_SPEC_VERSION = 0
VK_HUAWEI_EXTENSION_415_EXTENSION_NAME = "VK_HUAWEI_extension_415"
VK_ARM_shader_core_properties = 1
VK_ARM_SHADER_CORE_PROPERTIES_SPEC_VERSION = 1
VK_ARM_SHADER_CORE_PROPERTIES_EXTENSION_NAME = "VK_ARM_shader_core_properties"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_PROPERTIES_ARM = 1000415000
VK_KHR_extension_417 = 1
VK_KHR_EXTENSION_417_SPEC_VERSION = 0
VK_KHR_EXTENSION_417_EXTENSION_NAME = "VK_KHR_extension_417"
VK_ARM_scheduling_controls = 1
VK_ARM_SCHEDULING_CONTROLS_SPEC_VERSION = 1
VK_ARM_SCHEDULING_CONTROLS_EXTENSION_NAME = "VK_ARM_scheduling_controls"
VK_STRUCTURE_TYPE_DEVICE_QUEUE_SHADER_CORE_CONTROL_CREATE_INFO_ARM = 1000417000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SCHEDULING_CONTROLS_FEATURES_ARM = 1000417001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SCHEDULING_CONTROLS_PROPERTIES_ARM = 1000417002
VK_EXT_image_sliced_view_of_3d = 1
VK_EXT_IMAGE_SLICED_VIEW_OF_3D_SPEC_VERSION = 1
VK_EXT_IMAGE_SLICED_VIEW_OF_3D_EXTENSION_NAME = "VK_EXT_image_sliced_view_of_3d"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_SLICED_VIEW_OF_3D_FEATURES_EXT = 1000418000
VK_STRUCTURE_TYPE_IMAGE_VIEW_SLICED_CREATE_INFO_EXT = 1000418001
VK_EXT_extension_420 = 1
VK_EXT_EXTENSION_420_SPEC_VERSION = 0
VK_EXT_EXTENSION_420_EXTENSION_NAME = "VK_EXT_extension_420"
VK_SWAPCHAIN_CREATE_RESERVED_4_BIT_EXT = 0x00000010
VK_VALVE_descriptor_set_host_mapping = 1
VK_VALVE_DESCRIPTOR_SET_HOST_MAPPING_SPEC_VERSION = 1
VK_VALVE_DESCRIPTOR_SET_HOST_MAPPING_EXTENSION_NAME = "VK_VALVE_descriptor_set_host_mapping"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_SET_HOST_MAPPING_FEATURES_VALVE = 1000420000
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_BINDING_REFERENCE_VALVE = 1000420001
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_HOST_MAPPING_INFO_VALVE = 1000420002
VK_EXT_depth_clamp_zero_one = 1
VK_EXT_DEPTH_CLAMP_ZERO_ONE_SPEC_VERSION = 1
VK_EXT_DEPTH_CLAMP_ZERO_ONE_EXTENSION_NAME = "VK_EXT_depth_clamp_zero_one"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_CLAMP_ZERO_ONE_FEATURES_EXT = 1000421000
VK_EXT_non_seamless_cube_map = 1
VK_EXT_NON_SEAMLESS_CUBE_MAP_SPEC_VERSION = 1
VK_EXT_NON_SEAMLESS_CUBE_MAP_EXTENSION_NAME = "VK_EXT_non_seamless_cube_map"
VK_SAMPLER_CREATE_NON_SEAMLESS_CUBE_MAP_BIT_EXT = 0x00000004
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_NON_SEAMLESS_CUBE_MAP_FEATURES_EXT = 1000422000
VK_ARM_extension_424 = 1
VK_ARM_EXTENSION_424_SPEC_VERSION = 0
VK_ARM_EXTENSION_424_EXTENSION_NAME = "VK_ARM_extension_424"
VK_ARM_render_pass_striped = 1
VK_ARM_RENDER_PASS_STRIPED_SPEC_VERSION = 1
VK_ARM_RENDER_PASS_STRIPED_EXTENSION_NAME = "VK_ARM_render_pass_striped"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RENDER_PASS_STRIPED_FEATURES_ARM = 1000424000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RENDER_PASS_STRIPED_PROPERTIES_ARM = 1000424001
VK_STRUCTURE_TYPE_RENDER_PASS_STRIPE_BEGIN_INFO_ARM = 1000424002
VK_STRUCTURE_TYPE_RENDER_PASS_STRIPE_INFO_ARM = 1000424003
VK_STRUCTURE_TYPE_RENDER_PASS_STRIPE_SUBMIT_INFO_ARM = 1000424004
VK_QCOM_fragment_density_map_offset = 1
VK_QCOM_FRAGMENT_DENSITY_MAP_OFFSET_SPEC_VERSION = 1
VK_QCOM_FRAGMENT_DENSITY_MAP_OFFSET_EXTENSION_NAME = "VK_QCOM_fragment_density_map_offset"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_OFFSET_FEATURES_QCOM = 1000425000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_OFFSET_PROPERTIES_QCOM = 1000425001
VK_STRUCTURE_TYPE_SUBPASS_FRAGMENT_DENSITY_MAP_OFFSET_END_INFO_QCOM = 1000425002
VK_IMAGE_CREATE_FRAGMENT_DENSITY_MAP_OFFSET_BIT_QCOM = 0x00008000
VK_NV_copy_memory_indirect = 1
VK_NV_COPY_MEMORY_INDIRECT_SPEC_VERSION = 1
VK_NV_COPY_MEMORY_INDIRECT_EXTENSION_NAME = "VK_NV_copy_memory_indirect"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COPY_MEMORY_INDIRECT_FEATURES_NV = 1000426000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COPY_MEMORY_INDIRECT_PROPERTIES_NV = 1000426001
VK_NV_memory_decompression = 1
VK_NV_MEMORY_DECOMPRESSION_SPEC_VERSION = 1
VK_NV_MEMORY_DECOMPRESSION_EXTENSION_NAME = "VK_NV_memory_decompression"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_DECOMPRESSION_FEATURES_NV = 1000427000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_DECOMPRESSION_PROPERTIES_NV = 1000427001
VK_NV_device_generated_commands_compute = 1
VK_NV_DEVICE_GENERATED_COMMANDS_COMPUTE_SPEC_VERSION = 2
VK_NV_DEVICE_GENERATED_COMMANDS_COMPUTE_EXTENSION_NAME = "VK_NV_device_generated_commands_compute"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEVICE_GENERATED_COMMANDS_COMPUTE_FEATURES_NV = 1000428000
VK_STRUCTURE_TYPE_COMPUTE_PIPELINE_INDIRECT_BUFFER_INFO_NV = 1000428001
VK_STRUCTURE_TYPE_PIPELINE_INDIRECT_DEVICE_ADDRESS_INFO_NV = 1000428002
VK_INDIRECT_COMMANDS_TOKEN_TYPE_PIPELINE_NV = 1000428003
VK_INDIRECT_COMMANDS_TOKEN_TYPE_DISPATCH_NV = 1000428004
VK_DESCRIPTOR_SET_LAYOUT_CREATE_INDIRECT_BINDABLE_BIT_NV = 0x00000080
VK_NV_extension_430 = 1
VK_NV_EXTENSION_430_SPEC_VERSION = 0
VK_NV_EXTENSION_430_EXTENSION_NAME = "VK_NV_extension_430"
VK_NV_linear_color_attachment = 1
VK_NV_LINEAR_COLOR_ATTACHMENT_SPEC_VERSION = 1
VK_NV_LINEAR_COLOR_ATTACHMENT_EXTENSION_NAME = "VK_NV_linear_color_attachment"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LINEAR_COLOR_ATTACHMENT_FEATURES_NV = 1000430000
VK_FORMAT_FEATURE_2_LINEAR_COLOR_ATTACHMENT_BIT_NV = 0x4000000000
VK_NV_extension_432 = 1
VK_NV_EXTENSION_432_SPEC_VERSION = 0
VK_NV_EXTENSION_432_EXTENSION_NAME = "VK_NV_extension_432"
VK_NV_extension_433 = 1
VK_NV_EXTENSION_433_SPEC_VERSION = 0
VK_NV_EXTENSION_433_EXTENSION_NAME = "VK_NV_extension_433"
VK_GOOGLE_surfaceless_query = 1
VK_GOOGLE_SURFACELESS_QUERY_SPEC_VERSION = 2
VK_GOOGLE_SURFACELESS_QUERY_EXTENSION_NAME = "VK_GOOGLE_surfaceless_query"
VK_KHR_extension_435 = 1
VK_KHR_EXTENSION_435_SPEC_VERSION = 0
VK_KHR_EXTENSION_435_EXTENSION_NAME = "VK_KHR_extension_435"
VK_EXT_application_parameters = 1
VK_EXT_APPLICATION_PARAMETERS_SPEC_VERSION = 1
VK_EXT_APPLICATION_PARAMETERS_EXTENSION_NAME = "VK_EXT_application_parameters"
VK_STRUCTURE_TYPE_APPLICATION_PARAMETERS_EXT = 1000435000
VK_EXT_extension_437 = 1
VK_EXT_EXTENSION_437_SPEC_VERSION = 0
VK_EXT_EXTENSION_437_EXTENSION_NAME = "VK_EXT_extension_437"
VK_EXT_image_compression_control_swapchain = 1
VK_EXT_IMAGE_COMPRESSION_CONTROL_SWAPCHAIN_SPEC_VERSION = 1
VK_EXT_IMAGE_COMPRESSION_CONTROL_SWAPCHAIN_EXTENSION_NAME = "VK_EXT_image_compression_control_swapchain"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_COMPRESSION_CONTROL_SWAPCHAIN_FEATURES_EXT = 1000437000
VK_SEC_extension_439 = 1
VK_SEC_EXTENSION_439_SPEC_VERSION = 0
VK_SEC_EXTENSION_439_EXTENSION_NAME = "VK_SEC_extension_439"
VK_QCOM_extension_440 = 1
VK_QCOM_EXTENSION_440_SPEC_VERSION = 0
VK_QCOM_EXTENSION_440_EXTENSION_NAME = "VK_QCOM_extension_440"
VK_QUEUE_RESERVED_7_BIT_QCOM = 0x00000080
VK_DEVICE_QUEUE_CREATE_RESERVED_1_BIT_QCOM = 0x00000002
VK_QCOM_image_processing = 1
VK_QCOM_IMAGE_PROCESSING_SPEC_VERSION = 1
VK_QCOM_IMAGE_PROCESSING_EXTENSION_NAME = "VK_QCOM_image_processing"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_PROCESSING_FEATURES_QCOM = 1000440000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_PROCESSING_PROPERTIES_QCOM = 1000440001
VK_STRUCTURE_TYPE_IMAGE_VIEW_SAMPLE_WEIGHT_CREATE_INFO_QCOM = 1000440002
VK_SAMPLER_CREATE_IMAGE_PROCESSING_BIT_QCOM = 0x00000010
VK_IMAGE_USAGE_SAMPLE_WEIGHT_BIT_QCOM = 0x00100000
VK_IMAGE_USAGE_SAMPLE_BLOCK_MATCH_BIT_QCOM = 0x00200000
VK_DESCRIPTOR_TYPE_SAMPLE_WEIGHT_IMAGE_QCOM = 1000440000
VK_DESCRIPTOR_TYPE_BLOCK_MATCH_IMAGE_QCOM = 1000440001
VK_FORMAT_FEATURE_2_WEIGHT_IMAGE_BIT_QCOM = 0x400000000
VK_FORMAT_FEATURE_2_WEIGHT_SAMPLED_IMAGE_BIT_QCOM = 0x800000000
VK_FORMAT_FEATURE_2_BLOCK_MATCHING_BIT_QCOM = 0x1000000000
VK_FORMAT_FEATURE_2_BOX_FILTER_SAMPLED_BIT_QCOM = 0x2000000000
VK_COREAVI_extension_442 = 1
VK_COREAVI_EXTENSION_442_SPEC_VERSION = 0
VK_COREAVI_EXTENSION_442_EXTENSION_NAME = "VK_COREAVI_extension_442"
VK_COREAVI_extension_443 = 1
VK_COREAVI_EXTENSION_443_SPEC_VERSION = 0
VK_COREAVI_EXTENSION_443_EXTENSION_NAME = "VK_COREAVI_extension_443"
VK_COREAVI_extension_444 = 1
VK_COREAVI_EXTENSION_444_SPEC_VERSION = 0
VK_COREAVI_EXTENSION_444_EXTENSION_NAME = "VK_COREAVI_extension_444"
VK_COMMAND_POOL_RESET_RESERVED_1_BIT_COREAVI = 0x00000002
VK_COREAVI_extension_445 = 1
VK_COREAVI_EXTENSION_445_SPEC_VERSION = 0
VK_COREAVI_EXTENSION_445_EXTENSION_NAME = "VK_COREAVI_extension_445"
VK_COREAVI_extension_446 = 1
VK_COREAVI_EXTENSION_446_SPEC_VERSION = 0
VK_COREAVI_EXTENSION_446_EXTENSION_NAME = "VK_COREAVI_extension_446"
VK_IMAGE_USAGE_RESERVED_24_BIT_COREAVI = 0x01000000
VK_COREAVI_extension_447 = 1
VK_COREAVI_EXTENSION_447_SPEC_VERSION = 0
VK_COREAVI_EXTENSION_447_EXTENSION_NAME = "VK_COREAVI_extension_447"
VK_SEC_extension_448 = 1
VK_SEC_EXTENSION_448_SPEC_VERSION = 0
VK_SEC_EXTENSION_448_EXTENSION_NAME = "VK_SEC_extension_448"
VK_SEC_extension_449 = 1
VK_SEC_EXTENSION_449_SPEC_VERSION = 0
VK_SEC_EXTENSION_449_EXTENSION_NAME = "VK_SEC_extension_449"
VK_SEC_extension_450 = 1
VK_SEC_EXTENSION_450_SPEC_VERSION = 0
VK_SEC_EXTENSION_450_EXTENSION_NAME = "VK_SEC_extension_450"
VK_SEC_extension_451 = 1
VK_SEC_EXTENSION_451_SPEC_VERSION = 0
VK_SEC_EXTENSION_451_EXTENSION_NAME = "VK_SEC_extension_451"
VK_EXT_nested_command_buffer = 1
VK_EXT_NESTED_COMMAND_BUFFER_SPEC_VERSION = 1
VK_EXT_NESTED_COMMAND_BUFFER_EXTENSION_NAME = "VK_EXT_nested_command_buffer"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_NESTED_COMMAND_BUFFER_FEATURES_EXT = 1000451000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_NESTED_COMMAND_BUFFER_PROPERTIES_EXT = 1000451001
VK_SUBPASS_CONTENTS_INLINE_AND_SECONDARY_COMMAND_BUFFERS_EXT = 1000451000
VK_RENDERING_CONTENTS_INLINE_BIT_EXT = 0x00000010
VK_ARM_extension_453 = 1
VK_ARM_EXTENSION_453_SPEC_VERSION = 0
VK_ARM_EXTENSION_453_EXTENSION_NAME = "VK_ARM_extension_453"
VK_QUEUE_RESERVED_11_BIT_ARM = 0x00000800
VK_PIPELINE_STAGE_2_RESERVED_43_BIT_ARM = 0x80000000000
VK_ACCESS_2_RESERVED_49_BIT_ARM = 0x2000000000000
VK_ACCESS_2_RESERVED_50_BIT_ARM = 0x4000000000000
VK_FORMAT_FEATURE_2_RESERVED_47_BIT_ARM = 0x800000000000
VK_EXT_external_memory_acquire_unmodified = 1
VK_EXT_EXTERNAL_MEMORY_ACQUIRE_UNMODIFIED_SPEC_VERSION = 1
VK_EXT_EXTERNAL_MEMORY_ACQUIRE_UNMODIFIED_EXTENSION_NAME = "VK_EXT_external_memory_acquire_unmodified"
VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_ACQUIRE_UNMODIFIED_EXT = 1000453000
VK_GOOGLE_extension_455 = 1
VK_GOOGLE_EXTENSION_455_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_455_EXTENSION_NAME = "VK_GOOGLE_extension_455"
VK_EXT_extended_dynamic_state3 = 1
VK_EXT_EXTENDED_DYNAMIC_STATE_3_SPEC_VERSION = 2
VK_EXT_EXTENDED_DYNAMIC_STATE_3_EXTENSION_NAME = "VK_EXT_extended_dynamic_state3"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_DYNAMIC_STATE_3_FEATURES_EXT = 1000455000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_DYNAMIC_STATE_3_PROPERTIES_EXT = 1000455001
VK_DYNAMIC_STATE_TESSELLATION_DOMAIN_ORIGIN_EXT = 1000455002
VK_DYNAMIC_STATE_DEPTH_CLAMP_ENABLE_EXT = 1000455003
VK_DYNAMIC_STATE_POLYGON_MODE_EXT = 1000455004
VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT = 1000455005
VK_DYNAMIC_STATE_SAMPLE_MASK_EXT = 1000455006
VK_DYNAMIC_STATE_ALPHA_TO_COVERAGE_ENABLE_EXT = 1000455007
VK_DYNAMIC_STATE_ALPHA_TO_ONE_ENABLE_EXT = 1000455008
VK_DYNAMIC_STATE_LOGIC_OP_ENABLE_EXT = 1000455009
VK_DYNAMIC_STATE_COLOR_BLEND_ENABLE_EXT = 1000455010
VK_DYNAMIC_STATE_COLOR_BLEND_EQUATION_EXT = 1000455011
VK_DYNAMIC_STATE_COLOR_WRITE_MASK_EXT = 1000455012
VK_DYNAMIC_STATE_RASTERIZATION_STREAM_EXT = 1000455013
VK_DYNAMIC_STATE_CONSERVATIVE_RASTERIZATION_MODE_EXT = 1000455014
VK_DYNAMIC_STATE_EXTRA_PRIMITIVE_OVERESTIMATION_SIZE_EXT = 1000455015
VK_DYNAMIC_STATE_DEPTH_CLIP_ENABLE_EXT = 1000455016
VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_ENABLE_EXT = 1000455017
VK_DYNAMIC_STATE_COLOR_BLEND_ADVANCED_EXT = 1000455018
VK_DYNAMIC_STATE_PROVOKING_VERTEX_MODE_EXT = 1000455019
VK_DYNAMIC_STATE_LINE_RASTERIZATION_MODE_EXT = 1000455020
VK_DYNAMIC_STATE_LINE_STIPPLE_ENABLE_EXT = 1000455021
VK_DYNAMIC_STATE_DEPTH_CLIP_NEGATIVE_ONE_TO_ONE_EXT = 1000455022
VK_DYNAMIC_STATE_VIEWPORT_W_SCALING_ENABLE_NV = 1000455023
VK_DYNAMIC_STATE_VIEWPORT_SWIZZLE_NV = 1000455024
VK_DYNAMIC_STATE_COVERAGE_TO_COLOR_ENABLE_NV = 1000455025
VK_DYNAMIC_STATE_COVERAGE_TO_COLOR_LOCATION_NV = 1000455026
VK_DYNAMIC_STATE_COVERAGE_MODULATION_MODE_NV = 1000455027
VK_DYNAMIC_STATE_COVERAGE_MODULATION_TABLE_ENABLE_NV = 1000455028
VK_DYNAMIC_STATE_COVERAGE_MODULATION_TABLE_NV = 1000455029
VK_DYNAMIC_STATE_SHADING_RATE_IMAGE_ENABLE_NV = 1000455030
VK_DYNAMIC_STATE_REPRESENTATIVE_FRAGMENT_TEST_ENABLE_NV = 1000455031
VK_DYNAMIC_STATE_COVERAGE_REDUCTION_MODE_NV = 1000455032
VK_EXT_extension_457 = 1
VK_EXT_EXTENSION_457_SPEC_VERSION = 0
VK_EXT_EXTENSION_457_EXTENSION_NAME = "VK_EXT_extension_457"
VK_EXT_extension_458 = 1
VK_EXT_EXTENSION_458_SPEC_VERSION = 0
VK_EXT_EXTENSION_458_EXTENSION_NAME = "VK_EXT_extension_458"
VK_EXT_subpass_merge_feedback = 1
VK_EXT_SUBPASS_MERGE_FEEDBACK_SPEC_VERSION = 2
VK_EXT_SUBPASS_MERGE_FEEDBACK_EXTENSION_NAME = "VK_EXT_subpass_merge_feedback"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBPASS_MERGE_FEEDBACK_FEATURES_EXT = 1000458000
VK_STRUCTURE_TYPE_RENDER_PASS_CREATION_CONTROL_EXT = 1000458001
VK_STRUCTURE_TYPE_RENDER_PASS_CREATION_FEEDBACK_CREATE_INFO_EXT = 1000458002
VK_STRUCTURE_TYPE_RENDER_PASS_SUBPASS_FEEDBACK_CREATE_INFO_EXT = 1000458003
VK_LUNARG_direct_driver_loading = 1
VK_LUNARG_DIRECT_DRIVER_LOADING_SPEC_VERSION = 1
VK_LUNARG_DIRECT_DRIVER_LOADING_EXTENSION_NAME = "VK_LUNARG_direct_driver_loading"
VK_STRUCTURE_TYPE_DIRECT_DRIVER_LOADING_INFO_LUNARG = 1000459000
VK_STRUCTURE_TYPE_DIRECT_DRIVER_LOADING_LIST_LUNARG = 1000459001
VK_EXT_extension_461 = 1
VK_EXT_EXTENSION_461_SPEC_VERSION = 0
VK_EXT_EXTENSION_461_EXTENSION_NAME = "VK_EXT_extension_461"
VK_FORMAT_FEATURE_2_RESERVED_39_BIT_EXT = 0x8000000000
VK_IMAGE_USAGE_RESERVED_23_BIT_EXT = 0x00800000
VK_EXT_extension_462 = 1
VK_EXT_EXTENSION_462_SPEC_VERSION = 0
VK_EXT_EXTENSION_462_EXTENSION_NAME = "VK_EXT_extension_462"
VK_EXT_shader_module_identifier = 1
VK_EXT_SHADER_MODULE_IDENTIFIER_SPEC_VERSION = 1
VK_EXT_SHADER_MODULE_IDENTIFIER_EXTENSION_NAME = "VK_EXT_shader_module_identifier"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_MODULE_IDENTIFIER_FEATURES_EXT = 1000462000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_MODULE_IDENTIFIER_PROPERTIES_EXT = 1000462001
VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_MODULE_IDENTIFIER_CREATE_INFO_EXT = 1000462002
VK_STRUCTURE_TYPE_SHADER_MODULE_IDENTIFIER_EXT = 1000462003
VK_EXT_rasterization_order_attachment_access = 1
VK_EXT_RASTERIZATION_ORDER_ATTACHMENT_ACCESS_SPEC_VERSION = 1
VK_EXT_RASTERIZATION_ORDER_ATTACHMENT_ACCESS_EXTENSION_NAME = "VK_EXT_rasterization_order_attachment_access"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RASTERIZATION_ORDER_ATTACHMENT_ACCESS_FEATURES_EXT = 1000463000
VK_PIPELINE_COLOR_BLEND_STATE_CREATE_RASTERIZATION_ORDER_ATTACHMENT_ACCESS_BIT_EXT = 0x00000001
VK_PIPELINE_DEPTH_STENCIL_STATE_CREATE_RASTERIZATION_ORDER_ATTACHMENT_DEPTH_ACCESS_BIT_EXT = 0x00000001
VK_PIPELINE_DEPTH_STENCIL_STATE_CREATE_RASTERIZATION_ORDER_ATTACHMENT_STENCIL_ACCESS_BIT_EXT = 0x00000002
VK_SUBPASS_DESCRIPTION_RASTERIZATION_ORDER_ATTACHMENT_COLOR_ACCESS_BIT_EXT = 0x00000010
VK_SUBPASS_DESCRIPTION_RASTERIZATION_ORDER_ATTACHMENT_DEPTH_ACCESS_BIT_EXT = 0x00000020
VK_SUBPASS_DESCRIPTION_RASTERIZATION_ORDER_ATTACHMENT_STENCIL_ACCESS_BIT_EXT = 0x00000040
VK_NV_optical_flow = 1
VK_NV_OPTICAL_FLOW_SPEC_VERSION = 1
VK_NV_OPTICAL_FLOW_EXTENSION_NAME = "VK_NV_optical_flow"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_OPTICAL_FLOW_FEATURES_NV = 1000464000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_OPTICAL_FLOW_PROPERTIES_NV = 1000464001
VK_STRUCTURE_TYPE_OPTICAL_FLOW_IMAGE_FORMAT_INFO_NV = 1000464002
VK_STRUCTURE_TYPE_OPTICAL_FLOW_IMAGE_FORMAT_PROPERTIES_NV = 1000464003
VK_STRUCTURE_TYPE_OPTICAL_FLOW_SESSION_CREATE_INFO_NV = 1000464004
VK_STRUCTURE_TYPE_OPTICAL_FLOW_EXECUTE_INFO_NV = 1000464005
VK_STRUCTURE_TYPE_OPTICAL_FLOW_SESSION_CREATE_PRIVATE_DATA_INFO_NV = 1000464010
VK_FORMAT_R16G16_S10_5_NV = 1000464000
VK_OBJECT_TYPE_OPTICAL_FLOW_SESSION_NV = 1000464000
VK_QUEUE_OPTICAL_FLOW_BIT_NV = 0x00000100
VK_PIPELINE_STAGE_2_OPTICAL_FLOW_BIT_NV = 0x20000000
VK_ACCESS_2_OPTICAL_FLOW_READ_BIT_NV = 0x40000000000
VK_ACCESS_2_OPTICAL_FLOW_WRITE_BIT_NV = 0x80000000000
VK_FORMAT_FEATURE_2_OPTICAL_FLOW_IMAGE_BIT_NV = 0x10000000000
VK_FORMAT_FEATURE_2_OPTICAL_FLOW_VECTOR_BIT_NV = 0x20000000000
VK_FORMAT_FEATURE_2_OPTICAL_FLOW_COST_BIT_NV = 0x40000000000
VK_EXT_legacy_dithering = 1
VK_EXT_LEGACY_DITHERING_SPEC_VERSION = 1
VK_EXT_LEGACY_DITHERING_EXTENSION_NAME = "VK_EXT_legacy_dithering"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LEGACY_DITHERING_FEATURES_EXT = 1000465000
VK_SUBPASS_DESCRIPTION_ENABLE_LEGACY_DITHERING_BIT_EXT = 0x00000080
VK_RENDERING_ENABLE_LEGACY_DITHERING_BIT_EXT = 0x00000008
VK_EXT_pipeline_protected_access = 1
VK_EXT_PIPELINE_PROTECTED_ACCESS_SPEC_VERSION = 1
VK_EXT_PIPELINE_PROTECTED_ACCESS_EXTENSION_NAME = "VK_EXT_pipeline_protected_access"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_PROTECTED_ACCESS_FEATURES_EXT = 1000466000
VK_PIPELINE_CREATE_NO_PROTECTED_ACCESS_BIT_EXT = 0x08000000
VK_PIPELINE_CREATE_PROTECTED_ACCESS_ONLY_BIT_EXT = 0x40000000
VK_EXT_extension_468 = 1
VK_EXT_EXTENSION_468_SPEC_VERSION = 0
VK_EXT_EXTENSION_468_EXTENSION_NAME = "VK_EXT_extension_468"
VK_ANDROID_external_format_resolve = 1
VK_ANDROID_EXTERNAL_FORMAT_RESOLVE_SPEC_VERSION = 1
VK_ANDROID_EXTERNAL_FORMAT_RESOLVE_EXTENSION_NAME = "VK_ANDROID_external_format_resolve"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FORMAT_RESOLVE_FEATURES_ANDROID = 1000468000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FORMAT_RESOLVE_PROPERTIES_ANDROID = 1000468001
VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_FORMAT_RESOLVE_PROPERTIES_ANDROID = 1000468002
VK_RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID = 0x00000010
VK_AMD_extension_470 = 1
VK_AMD_EXTENSION_470_SPEC_VERSION = 0
VK_AMD_EXTENSION_470_EXTENSION_NAME = "VK_AMD_extension_470"
VK_KHR_maintenance5 = 1
VK_KHR_MAINTENANCE_5_SPEC_VERSION = 1
VK_KHR_MAINTENANCE_5_EXTENSION_NAME = "VK_KHR_maintenance5"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_5_FEATURES_KHR = 1000470000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_5_PROPERTIES_KHR = 1000470001
VK_STRUCTURE_TYPE_RENDERING_AREA_INFO_KHR = 1000470003
VK_STRUCTURE_TYPE_DEVICE_IMAGE_SUBRESOURCE_INFO_KHR = 1000470004
VK_FORMAT_A1B5G5R5_UNORM_PACK16_KHR = 1000470000
VK_FORMAT_A8_UNORM_KHR = 1000470001
VK_STRUCTURE_TYPE_SUBRESOURCE_LAYOUT_2_KHR = 1000470002
VK_STRUCTURE_TYPE_IMAGE_SUBRESOURCE_2_KHR = 1000470003
VK_STRUCTURE_TYPE_PIPELINE_CREATE_FLAGS_2_CREATE_INFO_KHR = 1000470005
VK_STRUCTURE_TYPE_BUFFER_USAGE_FLAGS_2_CREATE_INFO_KHR = 1000470006
VK_PIPELINE_CREATE_2_VIEW_INDEX_FROM_DEVICE_INDEX_BIT_KHR = 0x00000008
VK_PIPELINE_CREATE_2_DISPATCH_BASE_BIT_KHR = 0x00000010
VK_PIPELINE_CREATE_2_DEFER_COMPILE_BIT_NV = 0x00000020
VK_PIPELINE_CREATE_2_CAPTURE_STATISTICS_BIT_KHR = 0x00000040
VK_PIPELINE_CREATE_2_CAPTURE_INTERNAL_REPRESENTATIONS_BIT_KHR = 0x00000080
VK_PIPELINE_CREATE_2_FAIL_ON_PIPELINE_COMPILE_REQUIRED_BIT_KHR = 0x00000100
VK_PIPELINE_CREATE_2_EARLY_RETURN_ON_FAILURE_BIT_KHR = 0x00000200
VK_PIPELINE_CREATE_2_LINK_TIME_OPTIMIZATION_BIT_EXT = 0x00000400
VK_PIPELINE_CREATE_2_RETAIN_LINK_TIME_OPTIMIZATION_INFO_BIT_EXT = 0x00800000
VK_PIPELINE_CREATE_2_LIBRARY_BIT_KHR = 0x00000800
VK_PIPELINE_CREATE_2_RAY_TRACING_SKIP_TRIANGLES_BIT_KHR = 0x00001000
VK_PIPELINE_CREATE_2_RAY_TRACING_SKIP_AABBS_BIT_KHR = 0x00002000
VK_PIPELINE_CREATE_2_RAY_TRACING_NO_NULL_ANY_HIT_SHADERS_BIT_KHR = 0x00004000
VK_PIPELINE_CREATE_2_RAY_TRACING_NO_NULL_CLOSEST_HIT_SHADERS_BIT_KHR = 0x00008000
VK_PIPELINE_CREATE_2_RAY_TRACING_NO_NULL_MISS_SHADERS_BIT_KHR = 0x00010000
VK_PIPELINE_CREATE_2_RAY_TRACING_NO_NULL_INTERSECTION_SHADERS_BIT_KHR = 0x00020000
VK_PIPELINE_CREATE_2_RAY_TRACING_SHADER_GROUP_HANDLE_CAPTURE_REPLAY_BIT_KHR = 0x00080000
VK_PIPELINE_CREATE_2_INDIRECT_BINDABLE_BIT_NV = 0x00040000
VK_PIPELINE_CREATE_2_RAY_TRACING_ALLOW_MOTION_BIT_NV = 0x00100000
VK_PIPELINE_CREATE_2_RENDERING_FRAGMENT_SHADING_RATE_ATTACHMENT_BIT_KHR = 0x00200000
VK_PIPELINE_CREATE_2_RENDERING_FRAGMENT_DENSITY_MAP_ATTACHMENT_BIT_EXT = 0x00400000
VK_PIPELINE_CREATE_2_RAY_TRACING_OPACITY_MICROMAP_BIT_EXT = 0x01000000
VK_PIPELINE_CREATE_2_COLOR_ATTACHMENT_FEEDBACK_LOOP_BIT_EXT = 0x02000000
VK_PIPELINE_CREATE_2_DEPTH_STENCIL_ATTACHMENT_FEEDBACK_LOOP_BIT_EXT = 0x04000000
VK_PIPELINE_CREATE_2_NO_PROTECTED_ACCESS_BIT_EXT = 0x08000000
VK_PIPELINE_CREATE_2_PROTECTED_ACCESS_ONLY_BIT_EXT = 0x40000000
VK_PIPELINE_CREATE_2_RAY_TRACING_DISPLACEMENT_MICROMAP_BIT_NV = 0x10000000
VK_PIPELINE_CREATE_2_DESCRIPTOR_BUFFER_BIT_EXT = 0x20000000
VK_BUFFER_USAGE_2_CONDITIONAL_RENDERING_BIT_EXT = 0x00000200
VK_BUFFER_USAGE_2_SHADER_BINDING_TABLE_BIT_KHR = 0x00000400
VK_BUFFER_USAGE_2_RAY_TRACING_BIT_NV = None
VK_BUFFER_USAGE_2_TRANSFORM_FEEDBACK_BUFFER_BIT_EXT = 0x00000800
VK_BUFFER_USAGE_2_TRANSFORM_FEEDBACK_COUNTER_BUFFER_BIT_EXT = 0x00001000
VK_BUFFER_USAGE_2_VIDEO_DECODE_SRC_BIT_KHR = 0x00002000
VK_BUFFER_USAGE_2_VIDEO_DECODE_DST_BIT_KHR = 0x00004000
VK_BUFFER_USAGE_2_VIDEO_ENCODE_DST_BIT_KHR = 0x00008000
VK_BUFFER_USAGE_2_VIDEO_ENCODE_SRC_BIT_KHR = 0x00010000
VK_BUFFER_USAGE_2_SHADER_DEVICE_ADDRESS_BIT_KHR = 0x00020000
VK_BUFFER_USAGE_2_ACCELERATION_STRUCTURE_BUILD_INPUT_READ_ONLY_BIT_KHR = 0x00080000
VK_BUFFER_USAGE_2_ACCELERATION_STRUCTURE_STORAGE_BIT_KHR = 0x00100000
VK_BUFFER_USAGE_2_SAMPLER_DESCRIPTOR_BUFFER_BIT_EXT = 0x00200000
VK_BUFFER_USAGE_2_RESOURCE_DESCRIPTOR_BUFFER_BIT_EXT = 0x00400000
VK_BUFFER_USAGE_2_PUSH_DESCRIPTORS_DESCRIPTOR_BUFFER_BIT_EXT = 0x04000000
VK_BUFFER_USAGE_2_MICROMAP_BUILD_INPUT_READ_ONLY_BIT_EXT = 0x00800000
VK_BUFFER_USAGE_2_MICROMAP_STORAGE_BIT_EXT = 0x01000000
VK_AMD_extension_472 = 1
VK_AMD_EXTENSION_472_SPEC_VERSION = 0
VK_AMD_EXTENSION_472_EXTENSION_NAME = "VK_AMD_extension_472"
VK_AMD_extension_473 = 1
VK_AMD_EXTENSION_473_SPEC_VERSION = 0
VK_AMD_EXTENSION_473_EXTENSION_NAME = "VK_AMD_extension_473"
VK_AMD_extension_474 = 1
VK_AMD_EXTENSION_474_SPEC_VERSION = 0
VK_AMD_EXTENSION_474_EXTENSION_NAME = "VK_AMD_extension_474"
VK_AMD_extension_475 = 1
VK_AMD_EXTENSION_475_SPEC_VERSION = 0
VK_AMD_EXTENSION_475_EXTENSION_NAME = "VK_AMD_extension_475"
VK_AMD_extension_476 = 1
VK_AMD_EXTENSION_476_SPEC_VERSION = 0
VK_AMD_EXTENSION_476_EXTENSION_NAME = "VK_AMD_extension_476"
VK_AMD_extension_477 = 1
VK_AMD_EXTENSION_477_SPEC_VERSION = 0
VK_AMD_EXTENSION_477_EXTENSION_NAME = "VK_AMD_extension_477"
VK_AMD_extension_478 = 1
VK_AMD_EXTENSION_478_SPEC_VERSION = 0
VK_AMD_EXTENSION_478_EXTENSION_NAME = "VK_AMD_extension_478"
VK_AMD_extension_479 = 1
VK_AMD_EXTENSION_479_SPEC_VERSION = 0
VK_AMD_EXTENSION_479_EXTENSION_NAME = "VK_AMD_extension_479"
VK_PIPELINE_CREATE_2_RESERVED_32_BIT_KHR = 0x100000000
VK_EXT_extension_480 = 1
VK_EXT_EXTENSION_480_SPEC_VERSION = 0
VK_EXT_EXTENSION_480_EXTENSION_NAME = "VK_EXT_extension_480"
VK_EXT_extension_481 = 1
VK_EXT_EXTENSION_481_SPEC_VERSION = 0
VK_EXT_EXTENSION_481_EXTENSION_NAME = "VK_EXT_extension_481"
VK_KHR_ray_tracing_position_fetch = 1
VK_KHR_RAY_TRACING_POSITION_FETCH_SPEC_VERSION = 1
VK_KHR_RAY_TRACING_POSITION_FETCH_EXTENSION_NAME = "VK_KHR_ray_tracing_position_fetch"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_POSITION_FETCH_FEATURES_KHR = 1000481000
VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_DATA_ACCESS_KHR = 0x00000800
VK_EXT_shader_object = 1
VK_EXT_SHADER_OBJECT_SPEC_VERSION = 1
VK_EXT_SHADER_OBJECT_EXTENSION_NAME = "VK_EXT_shader_object"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_OBJECT_FEATURES_EXT = 1000482000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_OBJECT_PROPERTIES_EXT = 1000482001
VK_STRUCTURE_TYPE_SHADER_CREATE_INFO_EXT = 1000482002
VK_STRUCTURE_TYPE_SHADER_REQUIRED_SUBGROUP_SIZE_CREATE_INFO_EXT = None
VK_OBJECT_TYPE_SHADER_EXT = 1000482000
VK_SHADER_CREATE_ALLOW_VARYING_SUBGROUP_SIZE_BIT_EXT = 0x00000002
VK_SHADER_CREATE_REQUIRE_FULL_SUBGROUPS_BIT_EXT = 0x00000004
VK_SHADER_CREATE_NO_TASK_SHADER_BIT_EXT = 0x00000008
VK_SHADER_CREATE_DISPATCH_BASE_BIT_EXT = 0x00000010
VK_SHADER_CREATE_FRAGMENT_SHADING_RATE_ATTACHMENT_BIT_EXT = 0x00000020
VK_SHADER_CREATE_FRAGMENT_DENSITY_MAP_ATTACHMENT_BIT_EXT = 0x00000040
VK_EXT_extension_484 = 1
VK_EXT_EXTENSION_484_SPEC_VERSION = 0
VK_EXT_EXTENSION_484_EXTENSION_NAME = "VK_EXT_extension_484"
VK_PIPELINE_CREATE_2_RESERVED_31_BIT_KHR = 0x80000000
VK_QCOM_tile_properties = 1
VK_QCOM_TILE_PROPERTIES_SPEC_VERSION = 1
VK_QCOM_TILE_PROPERTIES_EXTENSION_NAME = "VK_QCOM_tile_properties"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TILE_PROPERTIES_FEATURES_QCOM = 1000484000
VK_STRUCTURE_TYPE_TILE_PROPERTIES_QCOM = 1000484001
VK_SEC_amigo_profiling = 1
VK_SEC_AMIGO_PROFILING_SPEC_VERSION = 1
VK_SEC_AMIGO_PROFILING_EXTENSION_NAME = "VK_SEC_amigo_profiling"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_AMIGO_PROFILING_FEATURES_SEC = 1000485000
VK_STRUCTURE_TYPE_AMIGO_PROFILING_SUBMIT_INFO_SEC = 1000485001
VK_EXT_extension_487 = 1
VK_EXT_EXTENSION_487_SPEC_VERSION = 0
VK_EXT_EXTENSION_487_EXTENSION_NAME = "VK_EXT_extension_487"
VK_EXT_extension_488 = 1
VK_EXT_EXTENSION_488_SPEC_VERSION = 0
VK_EXT_EXTENSION_488_EXTENSION_NAME = "VK_EXT_extension_488"
VK_QCOM_multiview_per_view_viewports = 1
VK_QCOM_MULTIVIEW_PER_VIEW_VIEWPORTS_SPEC_VERSION = 1
VK_QCOM_MULTIVIEW_PER_VIEW_VIEWPORTS_EXTENSION_NAME = "VK_QCOM_multiview_per_view_viewports"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PER_VIEW_VIEWPORTS_FEATURES_QCOM = 1000488000
VK_NV_external_sci_sync2 = 1
VK_NV_EXTERNAL_SCI_SYNC_2_SPEC_VERSION = 1
VK_NV_EXTERNAL_SCI_SYNC_2_EXTENSION_NAME = "VK_NV_external_sci_sync2"
VK_OBJECT_TYPE_SEMAPHORE_SCI_SYNC_POOL_NV = 1000489000
VK_STRUCTURE_TYPE_SEMAPHORE_SCI_SYNC_POOL_CREATE_INFO_NV = 1000489000
VK_STRUCTURE_TYPE_SEMAPHORE_SCI_SYNC_CREATE_INFO_NV = 1000489001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SCI_SYNC_2_FEATURES_NV = 1000489002
VK_STRUCTURE_TYPE_DEVICE_SEMAPHORE_SCI_SYNC_POOL_RESERVATION_CREATE_INFO_NV = 1000489003
VK_NV_ray_tracing_invocation_reorder = 1
VK_NV_RAY_TRACING_INVOCATION_REORDER_SPEC_VERSION = 1
VK_NV_RAY_TRACING_INVOCATION_REORDER_EXTENSION_NAME = "VK_NV_ray_tracing_invocation_reorder"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_INVOCATION_REORDER_FEATURES_NV = 1000490000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_INVOCATION_REORDER_PROPERTIES_NV = 1000490001
VK_NV_extension_492 = 1
VK_NV_EXTENSION_492_SPEC_VERSION = 0
VK_NV_EXTENSION_492_EXTENSION_NAME = "VK_NV_extension_492"
VK_NV_extended_sparse_address_space = 1
VK_NV_EXTENDED_SPARSE_ADDRESS_SPACE_SPEC_VERSION = 1
VK_NV_EXTENDED_SPARSE_ADDRESS_SPACE_EXTENSION_NAME = "VK_NV_extended_sparse_address_space"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_SPARSE_ADDRESS_SPACE_FEATURES_NV = 1000492000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_SPARSE_ADDRESS_SPACE_PROPERTIES_NV = 1000492001
VK_NV_extension_494 = 1
VK_NV_EXTENSION_494_SPEC_VERSION = 0
VK_NV_EXTENSION_494_EXTENSION_NAME = "VK_NV_extension_494"
VK_EXT_mutable_descriptor_type = 1
VK_EXT_MUTABLE_DESCRIPTOR_TYPE_SPEC_VERSION = 1
VK_EXT_MUTABLE_DESCRIPTOR_TYPE_EXTENSION_NAME = "VK_EXT_mutable_descriptor_type"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MUTABLE_DESCRIPTOR_TYPE_FEATURES_EXT = 1000494000
VK_STRUCTURE_TYPE_MUTABLE_DESCRIPTOR_TYPE_CREATE_INFO_EXT = 1000494002
VK_DESCRIPTOR_TYPE_MUTABLE_EXT = 1000494000
VK_DESCRIPTOR_POOL_CREATE_HOST_ONLY_BIT_EXT = 0x00000004
VK_DESCRIPTOR_SET_LAYOUT_CREATE_HOST_ONLY_POOL_BIT_EXT = 0x00000004
VK_EXT_extension_496 = 1
VK_EXT_EXTENSION_496_SPEC_VERSION = 0
VK_EXT_EXTENSION_496_EXTENSION_NAME = "VK_EXT_extension_496"
VK_EXT_layer_settings = 1
VK_EXT_LAYER_SETTINGS_SPEC_VERSION = 2
VK_EXT_LAYER_SETTINGS_EXTENSION_NAME = "VK_EXT_layer_settings"
VK_STRUCTURE_TYPE_LAYER_SETTINGS_CREATE_INFO_EXT = 1000496000
VK_ARM_shader_core_builtins = 1
VK_ARM_SHADER_CORE_BUILTINS_SPEC_VERSION = 2
VK_ARM_SHADER_CORE_BUILTINS_EXTENSION_NAME = "VK_ARM_shader_core_builtins"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_BUILTINS_FEATURES_ARM = 1000497000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_BUILTINS_PROPERTIES_ARM = 1000497001
VK_EXT_pipeline_library_group_handles = 1
VK_EXT_PIPELINE_LIBRARY_GROUP_HANDLES_SPEC_VERSION = 1
VK_EXT_PIPELINE_LIBRARY_GROUP_HANDLES_EXTENSION_NAME = "VK_EXT_pipeline_library_group_handles"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_LIBRARY_GROUP_HANDLES_FEATURES_EXT = 1000498000
VK_EXT_dynamic_rendering_unused_attachments = 1
VK_EXT_DYNAMIC_RENDERING_UNUSED_ATTACHMENTS_SPEC_VERSION = 1
VK_EXT_DYNAMIC_RENDERING_UNUSED_ATTACHMENTS_EXTENSION_NAME = "VK_EXT_dynamic_rendering_unused_attachments"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DYNAMIC_RENDERING_UNUSED_ATTACHMENTS_FEATURES_EXT = 1000499000
VK_EXT_extension_501 = 1
VK_EXT_EXTENSION_501_SPEC_VERSION = 0
VK_EXT_EXTENSION_501_EXTENSION_NAME = "VK_EXT_extension_501"
VK_EXT_extension_502 = 1
VK_EXT_EXTENSION_502_SPEC_VERSION = 0
VK_EXT_EXTENSION_502_EXTENSION_NAME = "VK_EXT_extension_502"
VK_EXT_extension_503 = 1
VK_EXT_EXTENSION_503_SPEC_VERSION = 0
VK_EXT_EXTENSION_503_EXTENSION_NAME = "VK_EXT_extension_503"
VK_NV_extension_504 = 1
VK_NV_EXTENSION_504_SPEC_VERSION = 0
VK_NV_EXTENSION_504_EXTENSION_NAME = "VK_NV_extension_504"
VK_EXT_extension_505 = 1
VK_EXT_EXTENSION_505_SPEC_VERSION = 0
VK_EXT_EXTENSION_505_EXTENSION_NAME = "VK_EXT_extension_505"
VK_NV_low_latency2 = 1
VK_NV_LOW_LATENCY_2_SPEC_VERSION = 2
VK_NV_LOW_LATENCY_2_EXTENSION_NAME = "VK_NV_low_latency2"
VK_STRUCTURE_TYPE_LATENCY_SLEEP_MODE_INFO_NV = 1000505000
VK_STRUCTURE_TYPE_LATENCY_SLEEP_INFO_NV = 1000505001
VK_STRUCTURE_TYPE_SET_LATENCY_MARKER_INFO_NV = 1000505002
VK_STRUCTURE_TYPE_GET_LATENCY_MARKER_INFO_NV = 1000505003
VK_STRUCTURE_TYPE_LATENCY_TIMINGS_FRAME_REPORT_NV = 1000505004
VK_STRUCTURE_TYPE_LATENCY_SUBMISSION_PRESENT_ID_NV = 1000505005
VK_STRUCTURE_TYPE_OUT_OF_BAND_QUEUE_TYPE_INFO_NV = 1000505006
VK_STRUCTURE_TYPE_SWAPCHAIN_LATENCY_CREATE_INFO_NV = 1000505007
VK_STRUCTURE_TYPE_LATENCY_SURFACE_CAPABILITIES_NV = 1000505008
VK_KHR_cooperative_matrix = 1
VK_KHR_COOPERATIVE_MATRIX_SPEC_VERSION = 2
VK_KHR_COOPERATIVE_MATRIX_EXTENSION_NAME = "VK_KHR_cooperative_matrix"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COOPERATIVE_MATRIX_FEATURES_KHR = 1000506000
VK_STRUCTURE_TYPE_COOPERATIVE_MATRIX_PROPERTIES_KHR = 1000506001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COOPERATIVE_MATRIX_PROPERTIES_KHR = 1000506002
VK_EXT_extension_508 = 1
VK_EXT_EXTENSION_508_SPEC_VERSION = 0
VK_EXT_EXTENSION_508_EXTENSION_NAME = "VK_EXT_extension_508"
VK_QUEUE_RESERVED_10_BIT_EXT = 0x00000400
VK_PIPELINE_STAGE_2_RESERVED_42_BIT_EXT = 0x40000000000
VK_ACCESS_2_RESERVED_47_BIT_EXT = 0x800000000000
VK_ACCESS_2_RESERVED_48_BIT_EXT = 0x1000000000000
VK_FORMAT_FEATURE_2_RESERVED_48_BIT_EXT = 0x1000000000000
VK_EXT_extension_509 = 1
VK_EXT_EXTENSION_509_SPEC_VERSION = 0
VK_EXT_EXTENSION_509_EXTENSION_NAME = "VK_EXT_extension_509"
VK_MESA_extension_510 = 1
VK_MESA_EXTENSION_510_SPEC_VERSION = 0
VK_MESA_EXTENSION_510_EXTENSION_NAME = "VK_MESA_extension_510"
VK_QCOM_multiview_per_view_render_areas = 1
VK_QCOM_MULTIVIEW_PER_VIEW_RENDER_AREAS_SPEC_VERSION = 1
VK_QCOM_MULTIVIEW_PER_VIEW_RENDER_AREAS_EXTENSION_NAME = "VK_QCOM_multiview_per_view_render_areas"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PER_VIEW_RENDER_AREAS_FEATURES_QCOM = 1000510000
VK_STRUCTURE_TYPE_MULTIVIEW_PER_VIEW_RENDER_AREAS_RENDER_PASS_BEGIN_INFO_QCOM = 1000510001
VK_EXT_extension_512 = 1
VK_EXT_EXTENSION_512_SPEC_VERSION = 0
VK_EXT_EXTENSION_512_EXTENSION_NAME = "VK_EXT_extension_512"
VK_KHR_extension_513 = 1
VK_KHR_EXTENSION_513_SPEC_VERSION = 0
VK_KHR_EXTENSION_513_EXTENSION_NAME = "VK_KHR_extension_513"
VK_KHR_extension_514 = 1
VK_KHR_EXTENSION_514_SPEC_VERSION = 0
VK_KHR_EXTENSION_514_EXTENSION_NAME = "VK_KHR_extension_514"
VK_KHR_extension_515 = 1
VK_KHR_EXTENSION_515_SPEC_VERSION = 0
VK_KHR_EXTENSION_515_EXTENSION_NAME = "VK_KHR_extension_515"
VK_KHR_video_maintenance1 = 1
VK_KHR_VIDEO_MAINTENANCE_1_SPEC_VERSION = 1
VK_KHR_VIDEO_MAINTENANCE_1_EXTENSION_NAME = "VK_KHR_video_maintenance1"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VIDEO_MAINTENANCE_1_FEATURES_KHR = 1000515000
VK_STRUCTURE_TYPE_VIDEO_INLINE_QUERY_INFO_KHR = 1000515001
VK_IMAGE_CREATE_VIDEO_PROFILE_INDEPENDENT_BIT_KHR = 0x00100000
VK_BUFFER_CREATE_VIDEO_PROFILE_INDEPENDENT_BIT_KHR = 0x00000040
VK_VIDEO_SESSION_CREATE_INLINE_QUERIES_BIT_KHR = 0x00000004
VK_NV_per_stage_descriptor_set = 1
VK_NV_PER_STAGE_DESCRIPTOR_SET_SPEC_VERSION = 1
VK_NV_PER_STAGE_DESCRIPTOR_SET_EXTENSION_NAME = "VK_NV_per_stage_descriptor_set"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PER_STAGE_DESCRIPTOR_SET_FEATURES_NV = 1000516000
VK_DESCRIPTOR_SET_LAYOUT_CREATE_PER_STAGE_BIT_NV = 0x00000040
VK_MESA_extension_518 = 1
VK_MESA_EXTENSION_518_SPEC_VERSION = 0
VK_MESA_EXTENSION_518_EXTENSION_NAME = "VK_MESA_extension_518"
VK_QCOM_image_processing2 = 1
VK_QCOM_IMAGE_PROCESSING_2_SPEC_VERSION = 1
VK_QCOM_IMAGE_PROCESSING_2_EXTENSION_NAME = "VK_QCOM_image_processing2"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_PROCESSING_2_FEATURES_QCOM = 1000518000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_PROCESSING_2_PROPERTIES_QCOM = 1000518001
VK_STRUCTURE_TYPE_SAMPLER_BLOCK_MATCH_WINDOW_CREATE_INFO_QCOM = 1000518002
VK_QCOM_filter_cubic_weights = 1
VK_QCOM_FILTER_CUBIC_WEIGHTS_SPEC_VERSION = 1
VK_QCOM_FILTER_CUBIC_WEIGHTS_EXTENSION_NAME = "VK_QCOM_filter_cubic_weights"
VK_STRUCTURE_TYPE_SAMPLER_CUBIC_WEIGHTS_CREATE_INFO_QCOM = 1000519000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUBIC_WEIGHTS_FEATURES_QCOM = 1000519001
VK_STRUCTURE_TYPE_BLIT_IMAGE_CUBIC_WEIGHTS_INFO_QCOM = 1000519002
VK_QCOM_ycbcr_degamma = 1
VK_QCOM_YCBCR_DEGAMMA_SPEC_VERSION = 1
VK_QCOM_YCBCR_DEGAMMA_EXTENSION_NAME = "VK_QCOM_ycbcr_degamma"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_YCBCR_DEGAMMA_FEATURES_QCOM = 1000520000
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_YCBCR_DEGAMMA_CREATE_INFO_QCOM = 1000520001
VK_QCOM_filter_cubic_clamp = 1
VK_QCOM_FILTER_CUBIC_CLAMP_SPEC_VERSION = 1
VK_QCOM_FILTER_CUBIC_CLAMP_EXTENSION_NAME = "VK_QCOM_filter_cubic_clamp"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUBIC_CLAMP_FEATURES_QCOM = 1000521000
VK_SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE_RANGECLAMP_QCOM = 1000521000
VK_EXT_extension_523 = 1
VK_EXT_EXTENSION_523_SPEC_VERSION = 0
VK_EXT_EXTENSION_523_EXTENSION_NAME = "VK_EXT_extension_523"
VK_EXT_extension_524 = 1
VK_EXT_EXTENSION_524_SPEC_VERSION = 0
VK_EXT_EXTENSION_524_EXTENSION_NAME = "VK_EXT_extension_524"
VK_EXT_attachment_feedback_loop_dynamic_state = 1
VK_EXT_ATTACHMENT_FEEDBACK_LOOP_DYNAMIC_STATE_SPEC_VERSION = 1
VK_EXT_ATTACHMENT_FEEDBACK_LOOP_DYNAMIC_STATE_EXTENSION_NAME = "VK_EXT_attachment_feedback_loop_dynamic_state"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ATTACHMENT_FEEDBACK_LOOP_DYNAMIC_STATE_FEATURES_EXT = 1000524000
VK_DYNAMIC_STATE_ATTACHMENT_FEEDBACK_LOOP_ENABLE_EXT = 1000524000
VK_KHR_vertex_attribute_divisor = 1
VK_KHR_VERTEX_ATTRIBUTE_DIVISOR_SPEC_VERSION = 1
VK_KHR_VERTEX_ATTRIBUTE_DIVISOR_EXTENSION_NAME = "VK_KHR_vertex_attribute_divisor"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_PROPERTIES_KHR = 1000525000
VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_DIVISOR_STATE_CREATE_INFO_KHR = 1000525001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_FEATURES_KHR = 1000525002
VK_EXT_extension_527 = 1
VK_EXT_EXTENSION_527_SPEC_VERSION = 0
VK_EXT_EXTENSION_527_EXTENSION_NAME = "VK_EXT_extension_527"
VK_EXT_extension_528 = 1
VK_EXT_EXTENSION_528_SPEC_VERSION = 0
VK_EXT_EXTENSION_528_EXTENSION_NAME = "VK_EXT_extension_528"
VK_KHR_extension_529 = 1
VK_KHR_EXTENSION_529_SPEC_VERSION = 0
VK_KHR_EXTENSION_529_EXTENSION_NAME = "VK_KHR_extension_529"
VK_QNX_external_memory_screen_buffer = 1
VK_QNX_EXTERNAL_MEMORY_SCREEN_BUFFER_SPEC_VERSION = 1
VK_QNX_EXTERNAL_MEMORY_SCREEN_BUFFER_EXTENSION_NAME = "VK_QNX_external_memory_screen_buffer"
VK_EXTERNAL_MEMORY_HANDLE_TYPE_SCREEN_BUFFER_BIT_QNX = 0x00004000
VK_STRUCTURE_TYPE_SCREEN_BUFFER_PROPERTIES_QNX = 1000529000
VK_STRUCTURE_TYPE_SCREEN_BUFFER_FORMAT_PROPERTIES_QNX = 1000529001
VK_STRUCTURE_TYPE_IMPORT_SCREEN_BUFFER_INFO_QNX = 1000529002
VK_STRUCTURE_TYPE_EXTERNAL_FORMAT_QNX = 1000529003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_SCREEN_BUFFER_FEATURES_QNX = 1000529004
VK_MSFT_layered_driver = 1
VK_MSFT_LAYERED_DRIVER_SPEC_VERSION = 1
VK_MSFT_LAYERED_DRIVER_EXTENSION_NAME = "VK_MSFT_layered_driver"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LAYERED_DRIVER_PROPERTIES_MSFT = 1000530000
VK_KHR_extension_532 = 1
VK_KHR_EXTENSION_532_SPEC_VERSION = 0
VK_KHR_EXTENSION_532_EXTENSION_NAME = "VK_KHR_extension_532"
VK_EXT_extension_533 = 1
VK_EXT_EXTENSION_533_SPEC_VERSION = 0
VK_EXT_EXTENSION_533_EXTENSION_NAME = "VK_EXT_extension_533"
VK_KHR_extension_534 = 1
VK_KHR_EXTENSION_534_SPEC_VERSION = 0
VK_KHR_EXTENSION_534_EXTENSION_NAME = "VK_KHR_extension_534"
VK_KHR_extension_535 = 1
VK_KHR_EXTENSION_535_SPEC_VERSION = 0
VK_KHR_EXTENSION_535_EXTENSION_NAME = "VK_KHR_extension_535"
VK_QCOM_extension_536 = 1
VK_QCOM_EXTENSION_536_SPEC_VERSION = 0
VK_QCOM_EXTENSION_536_EXTENSION_NAME = "VK_QCOM_extension_536"
VK_EXT_extension_537 = 1
VK_EXT_EXTENSION_537_SPEC_VERSION = 0
VK_EXT_EXTENSION_537_EXTENSION_NAME = "VK_EXT_extension_537"
VK_EXT_extension_538 = 1
VK_EXT_EXTENSION_538_SPEC_VERSION = 0
VK_EXT_EXTENSION_538_EXTENSION_NAME = "VK_EXT_extension_538"
VK_EXT_extension_539 = 1
VK_EXT_EXTENSION_539_SPEC_VERSION = 0
VK_EXT_EXTENSION_539_EXTENSION_NAME = "VK_EXT_extension_539"
VK_EXT_extension_540 = 1
VK_EXT_EXTENSION_540_SPEC_VERSION = 0
VK_EXT_EXTENSION_540_EXTENSION_NAME = "VK_EXT_extension_540"
VK_EXT_extension_541 = 1
VK_EXT_EXTENSION_541_SPEC_VERSION = 0
VK_EXT_EXTENSION_541_EXTENSION_NAME = "VK_EXT_extension_541"
VK_EXT_extension_542 = 1
VK_EXT_EXTENSION_542_SPEC_VERSION = 0
VK_EXT_EXTENSION_542_EXTENSION_NAME = "VK_EXT_extension_542"
VK_EXT_extension_543 = 1
VK_EXT_EXTENSION_543_SPEC_VERSION = 0
VK_EXT_EXTENSION_543_EXTENSION_NAME = "VK_EXT_extension_543"
VK_KHR_calibrated_timestamps = 1
VK_KHR_CALIBRATED_TIMESTAMPS_SPEC_VERSION = 1
VK_KHR_CALIBRATED_TIMESTAMPS_EXTENSION_NAME = "VK_KHR_calibrated_timestamps"
VK_STRUCTURE_TYPE_CALIBRATED_TIMESTAMP_INFO_KHR = 1000543000
VK_KHR_extension_545 = 1
VK_KHR_EXTENSION_545_SPEC_VERSION = 0
VK_KHR_EXTENSION_545_EXTENSION_NAME = "VK_KHR_extension_545"
VK_KHR_maintenance6 = 1
VK_KHR_MAINTENANCE_6_SPEC_VERSION = 1
VK_KHR_MAINTENANCE_6_EXTENSION_NAME = "VK_KHR_maintenance6"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_6_FEATURES_KHR = 1000545000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_6_PROPERTIES_KHR = 1000545001
VK_STRUCTURE_TYPE_BIND_MEMORY_STATUS_KHR = 1000545002
VK_STRUCTURE_TYPE_BIND_DESCRIPTOR_SETS_INFO_KHR = 1000545003
VK_STRUCTURE_TYPE_PUSH_CONSTANTS_INFO_KHR = 1000545004
VK_STRUCTURE_TYPE_PUSH_DESCRIPTOR_SET_INFO_KHR = 1000545005
VK_STRUCTURE_TYPE_PUSH_DESCRIPTOR_SET_WITH_TEMPLATE_INFO_KHR = 1000545006
VK_STRUCTURE_TYPE_SET_DESCRIPTOR_BUFFER_OFFSETS_INFO_EXT = 1000545007
VK_STRUCTURE_TYPE_BIND_DESCRIPTOR_BUFFER_EMBEDDED_SAMPLERS_INFO_EXT = 1000545008
VK_NV_descriptor_pool_overallocation = 1
VK_NV_DESCRIPTOR_POOL_OVERALLOCATION_SPEC_VERSION = 1
VK_NV_DESCRIPTOR_POOL_OVERALLOCATION_EXTENSION_NAME = "VK_NV_descriptor_pool_overallocation"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_POOL_OVERALLOCATION_FEATURES_NV = 1000546000
VK_DESCRIPTOR_POOL_CREATE_ALLOW_OVERALLOCATION_SETS_BIT_NV = 0x00000008
VK_DESCRIPTOR_POOL_CREATE_ALLOW_OVERALLOCATION_POOLS_BIT_NV = 0x00000010
VK_QCOM_extension_548 = 1
VK_QCOM_EXTENSION_548_SPEC_VERSION = 0
VK_QCOM_EXTENSION_548_EXTENSION_NAME = "VK_QCOM_extension_548"
VK_NV_extension_549 = 1
VK_NV_EXTENSION_549_SPEC_VERSION = 0
VK_NV_EXTENSION_549_EXTENSION_NAME = "VK_NV_extension_549"
VK_NV_extension_550 = 1
VK_NV_EXTENSION_550_SPEC_VERSION = 0
VK_NV_EXTENSION_550_EXTENSION_NAME = "VK_NV_extension_550"
VK_NV_extension_551 = 1
VK_NV_EXTENSION_551_SPEC_VERSION = 0
VK_NV_EXTENSION_551_EXTENSION_NAME = "VK_NV_extension_551"
VK_NV_extension_552 = 1
VK_NV_EXTENSION_552_SPEC_VERSION = 0
VK_NV_EXTENSION_552_EXTENSION_NAME = "VK_NV_extension_552"
VK_KHR_extension_553 = 1
VK_KHR_EXTENSION_553_SPEC_VERSION = 0
VK_KHR_EXTENSION_553_EXTENSION_NAME = "VK_KHR_extension_553"
VK_KHR_extension_554 = 1
VK_KHR_EXTENSION_554_SPEC_VERSION = 0
VK_KHR_EXTENSION_554_EXTENSION_NAME = "VK_KHR_extension_554"
VK_IMG_extension_555 = 1
VK_IMG_EXTENSION_555_SPEC_VERSION = 0
VK_IMG_EXTENSION_555_EXTENSION_NAME = "VK_IMG_extension_555"
VK_NV_extension_556 = 1
VK_NV_EXTENSION_556_SPEC_VERSION = 1
VK_NV_EXTENSION_556_EXTENSION_NAME = "VK_NV_extension_556"




class VkException(Exception):
    pass


class VkError(Exception):
    pass


class VkNotReady(VkException):
    pass
class VkTimeout(VkException):
    pass
class VkEventSet(VkException):
    pass
class VkEventReset(VkException):
    pass
class VkIncomplete(VkException):
    pass
class VkSuboptimalKhr(VkException):
    pass
class VkThreadIdleKhr(VkException):
    pass
class VkThreadDoneKhr(VkException):
    pass
class VkOperationDeferredKhr(VkException):
    pass
class VkOperationNotDeferredKhr(VkException):
    pass
class VkPipelineCompileRequiredExt(VkException):
    pass

class VkErrorOutOfHostMemory(VkError):
    pass
class VkErrorOutOfDeviceMemory(VkError):
    pass
class VkErrorInitializationFailed(VkError):
    pass
class VkErrorDeviceLost(VkError):
    pass
class VkErrorMemoryMapFailed(VkError):
    pass
class VkErrorLayerNotPresent(VkError):
    pass
class VkErrorExtensionNotPresent(VkError):
    pass
class VkErrorFeatureNotPresent(VkError):
    pass
class VkErrorIncompatibleDriver(VkError):
    pass
class VkErrorTooManyObjects(VkError):
    pass
class VkErrorFormatNotSupported(VkError):
    pass
class VkErrorFragmentedPool(VkError):
    pass
class VkErrorUnknown(VkError):
    pass
class VkErrorOutOfPoolMemory(VkError):
    pass
class VkErrorInvalidExternalHandle(VkError):
    pass
class VkErrorInvalidOpaqueCaptureAddress(VkError):
    pass
class VkErrorInvalidPipelineCacheData(VkError):
    pass
class VkErrorNoPipelineMatch(VkError):
    pass
class VkErrorSurfaceLostKhr(VkError):
    pass
class VkErrorNativeWindowInUseKhr(VkError):
    pass
class VkErrorOutOfDateKhr(VkError):
    pass
class VkErrorIncompatibleDisplayKhr(VkError):
    pass
class VkErrorInvalidShaderNv(VkError):
    pass
class VkErrorImageUsageNotSupportedKhr(VkError):
    pass
class VkErrorVideoPictureLayoutNotSupportedKhr(VkError):
    pass
class VkErrorVideoProfileOperationNotSupportedKhr(VkError):
    pass
class VkErrorVideoProfileFormatNotSupportedKhr(VkError):
    pass
class VkErrorVideoProfileCodecNotSupportedKhr(VkError):
    pass
class VkErrorVideoStdVersionNotSupportedKhr(VkError):
    pass
class VkErrorInvalidOpaqueCaptureAddressKhr(VkError):
    pass
class VkErrorFullScreenExclusiveModeLostExt(VkError):
    pass
class VkErrorInvalidVideoStdParametersKhr(VkError):
    pass
class VkErrorCompressionExhaustedExt(VkError):
    pass
class VkErrorIncompatibleShaderBinaryExt(VkError):
    pass

exception_codes = {
    1:VkNotReady,
    2:VkTimeout,
    3:VkEventSet,
    4:VkEventReset,
    5:VkIncomplete,
    1000001003:VkSuboptimalKhr,
    1000268000:VkThreadIdleKhr,
    1000268001:VkThreadDoneKhr,
    1000268002:VkOperationDeferredKhr,
    1000268003:VkOperationNotDeferredKhr,
    None:VkPipelineCompileRequiredExt,

    -1:VkErrorOutOfHostMemory,
    -2:VkErrorOutOfDeviceMemory,
    -3:VkErrorInitializationFailed,
    -4:VkErrorDeviceLost,
    -5:VkErrorMemoryMapFailed,
    -6:VkErrorLayerNotPresent,
    -7:VkErrorExtensionNotPresent,
    -8:VkErrorFeatureNotPresent,
    -9:VkErrorIncompatibleDriver,
    -10:VkErrorTooManyObjects,
    -11:VkErrorFormatNotSupported,
    -12:VkErrorFragmentedPool,
    -13:VkErrorUnknown,
    1000069000:VkErrorOutOfPoolMemory,
    1000072003:VkErrorInvalidExternalHandle,
    1000257000:VkErrorInvalidOpaqueCaptureAddress,
    1000298000:VkErrorInvalidPipelineCacheData,
    1000298001:VkErrorNoPipelineMatch,
    -1000000000:VkErrorSurfaceLostKhr,
    -1000000001:VkErrorNativeWindowInUseKhr,
    -1000001004:VkErrorOutOfDateKhr,
    -1000003001:VkErrorIncompatibleDisplayKhr,
    -1000012000:VkErrorInvalidShaderNv,
    -1000023000:VkErrorImageUsageNotSupportedKhr,
    -1000023001:VkErrorVideoPictureLayoutNotSupportedKhr,
    -1000023002:VkErrorVideoProfileOperationNotSupportedKhr,
    -1000023003:VkErrorVideoProfileFormatNotSupportedKhr,
    -1000023004:VkErrorVideoProfileCodecNotSupportedKhr,
    -1000023005:VkErrorVideoStdVersionNotSupportedKhr,
    None:VkErrorInvalidOpaqueCaptureAddressKhr,
    -1000255000:VkErrorFullScreenExclusiveModeLostExt,
    -1000299000:VkErrorInvalidVideoStdParametersKhr,
    -1000338000:VkErrorCompressionExhaustedExt,
    1000482000:VkErrorIncompatibleShaderBinaryExt,
}

_internal_vkInternalAllocationNotification = None

@ffi.callback('PFN_vkInternalAllocationNotification')
def _external_vkInternalAllocationNotification(*args):
    return _internal_vkInternalAllocationNotification(*[_cstr(x) for x in args])
_internal_vkInternalFreeNotification = None

@ffi.callback('PFN_vkInternalFreeNotification')
def _external_vkInternalFreeNotification(*args):
    return _internal_vkInternalFreeNotification(*[_cstr(x) for x in args])
_internal_vkReallocationFunction = None

@ffi.callback('PFN_vkReallocationFunction')
def _external_vkReallocationFunction(*args):
    return _internal_vkReallocationFunction(*[_cstr(x) for x in args])
_internal_vkAllocationFunction = None

@ffi.callback('PFN_vkAllocationFunction')
def _external_vkAllocationFunction(*args):
    return _internal_vkAllocationFunction(*[_cstr(x) for x in args])
_internal_vkFreeFunction = None

@ffi.callback('PFN_vkFreeFunction')
def _external_vkFreeFunction(*args):
    return _internal_vkFreeFunction(*[_cstr(x) for x in args])
_internal_vkVoidFunction = None

@ffi.callback('PFN_vkVoidFunction')
def _external_vkVoidFunction(*args):
    return _internal_vkVoidFunction(*[_cstr(x) for x in args])
_internal_vkDebugReportCallbackEXT = None

@ffi.callback('PFN_vkDebugReportCallbackEXT')
def _external_vkDebugReportCallbackEXT(*args):
    return _internal_vkDebugReportCallbackEXT(*[_cstr(x) for x in args])
_internal_vkDebugUtilsMessengerCallbackEXT = None

@ffi.callback('PFN_vkDebugUtilsMessengerCallbackEXT')
def _external_vkDebugUtilsMessengerCallbackEXT(*args):
    return _internal_vkDebugUtilsMessengerCallbackEXT(*[_cstr(x) for x in args])
_internal_vkDeviceMemoryReportCallbackEXT = None

@ffi.callback('PFN_vkDeviceMemoryReportCallbackEXT')
def _external_vkDeviceMemoryReportCallbackEXT(*args):
    return _internal_vkDeviceMemoryReportCallbackEXT(*[_cstr(x) for x in args])
_internal_vkGetInstanceProcAddrLUNARG = None

@ffi.callback('PFN_vkGetInstanceProcAddrLUNARG')
def _external_vkGetInstanceProcAddrLUNARG(*args):
    return _internal_vkGetInstanceProcAddrLUNARG(*[_cstr(x) for x in args])


def _get_pfn_name(struct_name):
    if struct_name == 'VkAllocationCallbacks':
        return 'vkInternalAllocationNotification'
    if struct_name == 'VkAllocationCallbacks':
        return 'vkInternalFreeNotification'
    if struct_name == 'VkAllocationCallbacks':
        return 'vkReallocationFunction'
    if struct_name == 'VkAllocationCallbacks':
        return 'vkAllocationFunction'
    if struct_name == 'VkAllocationCallbacks':
        return 'vkFreeFunction'
    if struct_name == 'VkAllocationCallbacks':
        return 'vkVoidFunction'
    if struct_name == 'VkDebugReportCallbackCreateInfoEXT':
        return 'vkDebugReportCallbackEXT'
    if struct_name == 'VkDebugUtilsMessengerCreateInfoEXT':
        return 'vkDebugUtilsMessengerCallbackEXT'
    if struct_name == 'VkDeviceDeviceMemoryReportCreateInfoEXT':
        return 'vkDeviceMemoryReportCallbackEXT'
    if struct_name == 'VkDirectDriverLoadingInfoLUNARG':
        return 'vkGetInstanceProcAddrLUNARG'


def _new(ctype, **kwargs):
    _type = ffi.typeof(ctype)

    # keep only valued kwargs
    kwargs = {k: kwargs[k] for k in kwargs if kwargs[k]}

    # cast pointer
    ptrs = {}
    pfns = {}
    pcs = {}
    for k, v in kwargs.items():
        # convert tuple pair to dict
        ktype = dict(_type.fields)[k].type

        if k == 'pCode':
            pcs[k] = ffi.cast('uint32_t*', ffi.from_buffer(v))
        elif k.startswith('pfn'):
            pfn_name = _get_pfn_name(ctype)
            mod = sys.modules[__name__]
            setattr(mod, '_internal_' + pfn_name, v)
            pfns[k] = getattr(mod, '_external_' + pfn_name)
        elif ktype.kind == 'pointer':
            ptrs[k] = _cast_ptr(v, ktype)

    # init object
    init = dict(kwargs,  **{k: v for k, (v, _) in ptrs.items()})
    init.update(pfns)
    init.update(pcs)

    ret = ffi.new(_type.cname + '*', init)[0]

    # reference created pointer in the object
    _weakkey_dict[ret] = [v for _, v in ptrs.values() if v != ffi.NULL]
    if pcs:
        _weakkey_dict[ret].extend([x for x in pcs.values()])

    return ret



def VkBaseOutStructure(sType=None,pNext=None,):

    return _new('VkBaseOutStructure', sType=sType,pNext=pNext)

def VkBaseInStructure(sType=None,pNext=None,):

    return _new('VkBaseInStructure', sType=sType,pNext=pNext)

def VkOffset2D(x=None,y=None,):

    return _new('VkOffset2D', x=x,y=y)

def VkOffset3D(x=None,y=None,z=None,):

    return _new('VkOffset3D', x=x,y=y,z=z)

def VkExtent2D(width=None,height=None,):

    return _new('VkExtent2D', width=width,height=height)

def VkExtent3D(width=None,height=None,depth=None,):

    return _new('VkExtent3D', width=width,height=height,depth=depth)

def VkViewport(x=None,y=None,width=None,height=None,minDepth=None,maxDepth=None,):

    return _new('VkViewport', x=x,y=y,width=width,height=height,minDepth=minDepth,maxDepth=maxDepth)

def VkRect2D(offset=None,extent=None,):

    return _new('VkRect2D', offset=offset,extent=extent)

def VkClearRect(rect=None,baseArrayLayer=None,layerCount=None,):

    return _new('VkClearRect', rect=rect,baseArrayLayer=baseArrayLayer,layerCount=layerCount)

def VkComponentMapping(r=None,g=None,b=None,a=None,):

    return _new('VkComponentMapping', r=r,g=g,b=b,a=a)

def VkPhysicalDeviceProperties(apiVersion=None,driverVersion=None,vendorID=None,deviceID=None,deviceType=None,deviceName=None,pipelineCacheUUID=None,limits=None,sparseProperties=None,):

    return _new('VkPhysicalDeviceProperties', apiVersion=apiVersion,driverVersion=driverVersion,vendorID=vendorID,deviceID=deviceID,deviceType=deviceType,deviceName=deviceName,pipelineCacheUUID=pipelineCacheUUID,limits=limits,sparseProperties=sparseProperties)

def VkExtensionProperties(extensionName=None,specVersion=None,):

    return _new('VkExtensionProperties', extensionName=extensionName,specVersion=specVersion)

def VkLayerProperties(layerName=None,specVersion=None,implementationVersion=None,description=None,):

    return _new('VkLayerProperties', layerName=layerName,specVersion=specVersion,implementationVersion=implementationVersion,description=description)

def VkApplicationInfo(sType=VK_STRUCTURE_TYPE_APPLICATION_INFO,pNext=None,pApplicationName=None,applicationVersion=None,pEngineName=None,engineVersion=None,apiVersion=None,):

    return _new('VkApplicationInfo', sType=sType,pNext=pNext,pApplicationName=pApplicationName,applicationVersion=applicationVersion,pEngineName=pEngineName,engineVersion=engineVersion,apiVersion=apiVersion)

def VkAllocationCallbacks(pUserData=None,pfnAllocation=None,pfnReallocation=None,pfnFree=None,pfnInternalAllocation=None,pfnInternalFree=None,):

    return _new('VkAllocationCallbacks', pUserData=pUserData,pfnAllocation=pfnAllocation,pfnReallocation=pfnReallocation,pfnFree=pfnFree,pfnInternalAllocation=pfnInternalAllocation,pfnInternalFree=pfnInternalFree)

def VkDeviceQueueCreateInfo(sType=VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO,pNext=None,flags=None,queueFamilyIndex=None,queueCount=None,pQueuePriorities=None,):
    if queueCount is None and pQueuePriorities is not None:
        queueCount = len(pQueuePriorities)

    return _new('VkDeviceQueueCreateInfo', sType=sType,pNext=pNext,flags=flags,queueFamilyIndex=queueFamilyIndex,queueCount=queueCount,pQueuePriorities=pQueuePriorities)

def VkDeviceCreateInfo(sType=VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO,pNext=None,flags=None,queueCreateInfoCount=None,pQueueCreateInfos=None,enabledLayerCount=None,ppEnabledLayerNames=None,enabledExtensionCount=None,ppEnabledExtensionNames=None,pEnabledFeatures=None,):
    if queueCreateInfoCount is None and pQueueCreateInfos is not None:
        queueCreateInfoCount = len(pQueueCreateInfos)
    if enabledLayerCount is None and ppEnabledLayerNames is not None:
        enabledLayerCount = len(ppEnabledLayerNames)
    if enabledExtensionCount is None and ppEnabledExtensionNames is not None:
        enabledExtensionCount = len(ppEnabledExtensionNames)

    return _new('VkDeviceCreateInfo', sType=sType,pNext=pNext,flags=flags,queueCreateInfoCount=queueCreateInfoCount,pQueueCreateInfos=pQueueCreateInfos,enabledLayerCount=enabledLayerCount,ppEnabledLayerNames=ppEnabledLayerNames,enabledExtensionCount=enabledExtensionCount,ppEnabledExtensionNames=ppEnabledExtensionNames,pEnabledFeatures=pEnabledFeatures)

def VkInstanceCreateInfo(sType=VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO,pNext=None,flags=None,pApplicationInfo=None,enabledLayerCount=None,ppEnabledLayerNames=None,enabledExtensionCount=None,ppEnabledExtensionNames=None,):
    if enabledLayerCount is None and ppEnabledLayerNames is not None:
        enabledLayerCount = len(ppEnabledLayerNames)
    if enabledExtensionCount is None and ppEnabledExtensionNames is not None:
        enabledExtensionCount = len(ppEnabledExtensionNames)

    return _new('VkInstanceCreateInfo', sType=sType,pNext=pNext,flags=flags,pApplicationInfo=pApplicationInfo,enabledLayerCount=enabledLayerCount,ppEnabledLayerNames=ppEnabledLayerNames,enabledExtensionCount=enabledExtensionCount,ppEnabledExtensionNames=ppEnabledExtensionNames)

def VkQueueFamilyProperties(queueFlags=None,queueCount=None,timestampValidBits=None,minImageTransferGranularity=None,):

    return _new('VkQueueFamilyProperties', queueFlags=queueFlags,queueCount=queueCount,timestampValidBits=timestampValidBits,minImageTransferGranularity=minImageTransferGranularity)

def VkPhysicalDeviceMemoryProperties(memoryTypeCount=None,memoryTypes=None,memoryHeapCount=None,memoryHeaps=None,):

    return _new('VkPhysicalDeviceMemoryProperties', memoryTypeCount=memoryTypeCount,memoryTypes=memoryTypes,memoryHeapCount=memoryHeapCount,memoryHeaps=memoryHeaps)

def VkMemoryAllocateInfo(sType=VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO,pNext=None,allocationSize=None,memoryTypeIndex=None,):

    return _new('VkMemoryAllocateInfo', sType=sType,pNext=pNext,allocationSize=allocationSize,memoryTypeIndex=memoryTypeIndex)

def VkMemoryRequirements(size=None,alignment=None,memoryTypeBits=None,):

    return _new('VkMemoryRequirements', size=size,alignment=alignment,memoryTypeBits=memoryTypeBits)

def VkSparseImageFormatProperties(aspectMask=None,imageGranularity=None,flags=None,):

    return _new('VkSparseImageFormatProperties', aspectMask=aspectMask,imageGranularity=imageGranularity,flags=flags)

def VkSparseImageMemoryRequirements(formatProperties=None,imageMipTailFirstLod=None,imageMipTailSize=None,imageMipTailOffset=None,imageMipTailStride=None,):

    return _new('VkSparseImageMemoryRequirements', formatProperties=formatProperties,imageMipTailFirstLod=imageMipTailFirstLod,imageMipTailSize=imageMipTailSize,imageMipTailOffset=imageMipTailOffset,imageMipTailStride=imageMipTailStride)

def VkMemoryType(propertyFlags=None,heapIndex=None,):

    return _new('VkMemoryType', propertyFlags=propertyFlags,heapIndex=heapIndex)

def VkMemoryHeap(size=None,flags=None,):

    return _new('VkMemoryHeap', size=size,flags=flags)

def VkMappedMemoryRange(sType=VK_STRUCTURE_TYPE_MAPPED_MEMORY_RANGE,pNext=None,memory=None,offset=None,size=None,):

    return _new('VkMappedMemoryRange', sType=sType,pNext=pNext,memory=memory,offset=offset,size=size)

def VkFormatProperties(linearTilingFeatures=None,optimalTilingFeatures=None,bufferFeatures=None,):

    return _new('VkFormatProperties', linearTilingFeatures=linearTilingFeatures,optimalTilingFeatures=optimalTilingFeatures,bufferFeatures=bufferFeatures)

def VkImageFormatProperties(maxExtent=None,maxMipLevels=None,maxArrayLayers=None,sampleCounts=None,maxResourceSize=None,):

    return _new('VkImageFormatProperties', maxExtent=maxExtent,maxMipLevels=maxMipLevels,maxArrayLayers=maxArrayLayers,sampleCounts=sampleCounts,maxResourceSize=maxResourceSize)

def VkDescriptorBufferInfo(buffer=None,offset=None,range=None,):

    return _new('VkDescriptorBufferInfo', buffer=buffer,offset=offset,range=range)

def VkDescriptorImageInfo(sampler=None,imageView=None,imageLayout=None,):

    return _new('VkDescriptorImageInfo', sampler=sampler,imageView=imageView,imageLayout=imageLayout)

def VkWriteDescriptorSet(sType=VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET,pNext=None,dstSet=None,dstBinding=None,dstArrayElement=None,descriptorCount=None,descriptorType=None,pImageInfo=None,pBufferInfo=None,pTexelBufferView=None,):
    if descriptorCount is None and pImageInfo is not None:
        descriptorCount = len(pImageInfo)
    if descriptorCount is None and pBufferInfo is not None:
        descriptorCount = len(pBufferInfo)
    if descriptorCount is None and pTexelBufferView is not None:
        descriptorCount = len(pTexelBufferView)

    return _new('VkWriteDescriptorSet', sType=sType,pNext=pNext,dstSet=dstSet,dstBinding=dstBinding,dstArrayElement=dstArrayElement,descriptorCount=descriptorCount,descriptorType=descriptorType,pImageInfo=pImageInfo,pBufferInfo=pBufferInfo,pTexelBufferView=pTexelBufferView)

def VkCopyDescriptorSet(sType=VK_STRUCTURE_TYPE_COPY_DESCRIPTOR_SET,pNext=None,srcSet=None,srcBinding=None,srcArrayElement=None,dstSet=None,dstBinding=None,dstArrayElement=None,descriptorCount=None,):

    return _new('VkCopyDescriptorSet', sType=sType,pNext=pNext,srcSet=srcSet,srcBinding=srcBinding,srcArrayElement=srcArrayElement,dstSet=dstSet,dstBinding=dstBinding,dstArrayElement=dstArrayElement,descriptorCount=descriptorCount)

def VkBufferUsageFlags2CreateInfoKHR(sType=VK_STRUCTURE_TYPE_BUFFER_USAGE_FLAGS_2_CREATE_INFO_KHR,pNext=None,usage=None,):

    return _new('VkBufferUsageFlags2CreateInfoKHR', sType=sType,pNext=pNext,usage=usage)

def VkBufferCreateInfo(sType=VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO,pNext=None,flags=None,size=None,usage=None,sharingMode=None,queueFamilyIndexCount=None,pQueueFamilyIndices=None,):
    if queueFamilyIndexCount is None and pQueueFamilyIndices is not None:
        queueFamilyIndexCount = len(pQueueFamilyIndices)

    return _new('VkBufferCreateInfo', sType=sType,pNext=pNext,flags=flags,size=size,usage=usage,sharingMode=sharingMode,queueFamilyIndexCount=queueFamilyIndexCount,pQueueFamilyIndices=pQueueFamilyIndices)

def VkBufferViewCreateInfo(sType=VK_STRUCTURE_TYPE_BUFFER_VIEW_CREATE_INFO,pNext=None,flags=None,buffer=None,format=None,offset=None,range=None,):

    return _new('VkBufferViewCreateInfo', sType=sType,pNext=pNext,flags=flags,buffer=buffer,format=format,offset=offset,range=range)

def VkImageSubresource(aspectMask=None,mipLevel=None,arrayLayer=None,):

    return _new('VkImageSubresource', aspectMask=aspectMask,mipLevel=mipLevel,arrayLayer=arrayLayer)

def VkImageSubresourceLayers(aspectMask=None,mipLevel=None,baseArrayLayer=None,layerCount=None,):

    return _new('VkImageSubresourceLayers', aspectMask=aspectMask,mipLevel=mipLevel,baseArrayLayer=baseArrayLayer,layerCount=layerCount)

def VkImageSubresourceRange(aspectMask=None,baseMipLevel=None,levelCount=None,baseArrayLayer=None,layerCount=None,):

    return _new('VkImageSubresourceRange', aspectMask=aspectMask,baseMipLevel=baseMipLevel,levelCount=levelCount,baseArrayLayer=baseArrayLayer,layerCount=layerCount)

def VkMemoryBarrier(sType=VK_STRUCTURE_TYPE_MEMORY_BARRIER,pNext=None,srcAccessMask=None,dstAccessMask=None,):

    return _new('VkMemoryBarrier', sType=sType,pNext=pNext,srcAccessMask=srcAccessMask,dstAccessMask=dstAccessMask)

def VkBufferMemoryBarrier(sType=VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER,pNext=None,srcAccessMask=None,dstAccessMask=None,srcQueueFamilyIndex=None,dstQueueFamilyIndex=None,buffer=None,offset=None,size=None,):

    return _new('VkBufferMemoryBarrier', sType=sType,pNext=pNext,srcAccessMask=srcAccessMask,dstAccessMask=dstAccessMask,srcQueueFamilyIndex=srcQueueFamilyIndex,dstQueueFamilyIndex=dstQueueFamilyIndex,buffer=buffer,offset=offset,size=size)

def VkImageMemoryBarrier(sType=VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER,pNext=None,srcAccessMask=None,dstAccessMask=None,oldLayout=None,newLayout=None,srcQueueFamilyIndex=None,dstQueueFamilyIndex=None,image=None,subresourceRange=None,):

    return _new('VkImageMemoryBarrier', sType=sType,pNext=pNext,srcAccessMask=srcAccessMask,dstAccessMask=dstAccessMask,oldLayout=oldLayout,newLayout=newLayout,srcQueueFamilyIndex=srcQueueFamilyIndex,dstQueueFamilyIndex=dstQueueFamilyIndex,image=image,subresourceRange=subresourceRange)

def VkImageCreateInfo(sType=VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO,pNext=None,flags=None,imageType=None,format=None,extent=None,mipLevels=None,arrayLayers=None,samples=None,tiling=None,usage=None,sharingMode=None,queueFamilyIndexCount=None,pQueueFamilyIndices=None,initialLayout=None,):
    if queueFamilyIndexCount is None and pQueueFamilyIndices is not None:
        queueFamilyIndexCount = len(pQueueFamilyIndices)

    return _new('VkImageCreateInfo', sType=sType,pNext=pNext,flags=flags,imageType=imageType,format=format,extent=extent,mipLevels=mipLevels,arrayLayers=arrayLayers,samples=samples,tiling=tiling,usage=usage,sharingMode=sharingMode,queueFamilyIndexCount=queueFamilyIndexCount,pQueueFamilyIndices=pQueueFamilyIndices,initialLayout=initialLayout)

def VkSubresourceLayout(offset=None,size=None,rowPitch=None,arrayPitch=None,depthPitch=None,):

    return _new('VkSubresourceLayout', offset=offset,size=size,rowPitch=rowPitch,arrayPitch=arrayPitch,depthPitch=depthPitch)

def VkImageViewCreateInfo(sType=VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO,pNext=None,flags=None,image=None,viewType=None,format=None,components=None,subresourceRange=None,):

    return _new('VkImageViewCreateInfo', sType=sType,pNext=pNext,flags=flags,image=image,viewType=viewType,format=format,components=components,subresourceRange=subresourceRange)

def VkBufferCopy(srcOffset=None,dstOffset=None,size=None,):

    return _new('VkBufferCopy', srcOffset=srcOffset,dstOffset=dstOffset,size=size)

def VkSparseMemoryBind(resourceOffset=None,size=None,memory=None,memoryOffset=None,flags=None,):

    return _new('VkSparseMemoryBind', resourceOffset=resourceOffset,size=size,memory=memory,memoryOffset=memoryOffset,flags=flags)

def VkSparseImageMemoryBind(subresource=None,offset=None,extent=None,memory=None,memoryOffset=None,flags=None,):

    return _new('VkSparseImageMemoryBind', subresource=subresource,offset=offset,extent=extent,memory=memory,memoryOffset=memoryOffset,flags=flags)

def VkSparseBufferMemoryBindInfo(buffer=None,bindCount=None,pBinds=None,):
    if bindCount is None and pBinds is not None:
        bindCount = len(pBinds)

    return _new('VkSparseBufferMemoryBindInfo', buffer=buffer,bindCount=bindCount,pBinds=pBinds)

def VkSparseImageOpaqueMemoryBindInfo(image=None,bindCount=None,pBinds=None,):
    if bindCount is None and pBinds is not None:
        bindCount = len(pBinds)

    return _new('VkSparseImageOpaqueMemoryBindInfo', image=image,bindCount=bindCount,pBinds=pBinds)

def VkSparseImageMemoryBindInfo(image=None,bindCount=None,pBinds=None,):
    if bindCount is None and pBinds is not None:
        bindCount = len(pBinds)

    return _new('VkSparseImageMemoryBindInfo', image=image,bindCount=bindCount,pBinds=pBinds)

def VkBindSparseInfo(sType=VK_STRUCTURE_TYPE_BIND_SPARSE_INFO,pNext=None,waitSemaphoreCount=None,pWaitSemaphores=None,bufferBindCount=None,pBufferBinds=None,imageOpaqueBindCount=None,pImageOpaqueBinds=None,imageBindCount=None,pImageBinds=None,signalSemaphoreCount=None,pSignalSemaphores=None,):
    if waitSemaphoreCount is None and pWaitSemaphores is not None:
        waitSemaphoreCount = len(pWaitSemaphores)
    if bufferBindCount is None and pBufferBinds is not None:
        bufferBindCount = len(pBufferBinds)
    if imageOpaqueBindCount is None and pImageOpaqueBinds is not None:
        imageOpaqueBindCount = len(pImageOpaqueBinds)
    if imageBindCount is None and pImageBinds is not None:
        imageBindCount = len(pImageBinds)
    if signalSemaphoreCount is None and pSignalSemaphores is not None:
        signalSemaphoreCount = len(pSignalSemaphores)

    return _new('VkBindSparseInfo', sType=sType,pNext=pNext,waitSemaphoreCount=waitSemaphoreCount,pWaitSemaphores=pWaitSemaphores,bufferBindCount=bufferBindCount,pBufferBinds=pBufferBinds,imageOpaqueBindCount=imageOpaqueBindCount,pImageOpaqueBinds=pImageOpaqueBinds,imageBindCount=imageBindCount,pImageBinds=pImageBinds,signalSemaphoreCount=signalSemaphoreCount,pSignalSemaphores=pSignalSemaphores)

def VkImageCopy(srcSubresource=None,srcOffset=None,dstSubresource=None,dstOffset=None,extent=None,):

    return _new('VkImageCopy', srcSubresource=srcSubresource,srcOffset=srcOffset,dstSubresource=dstSubresource,dstOffset=dstOffset,extent=extent)

def VkImageBlit(srcSubresource=None,srcOffsets=None,dstSubresource=None,dstOffsets=None,):

    return _new('VkImageBlit', srcSubresource=srcSubresource,srcOffsets=srcOffsets,dstSubresource=dstSubresource,dstOffsets=dstOffsets)

def VkBufferImageCopy(bufferOffset=None,bufferRowLength=None,bufferImageHeight=None,imageSubresource=None,imageOffset=None,imageExtent=None,):

    return _new('VkBufferImageCopy', bufferOffset=bufferOffset,bufferRowLength=bufferRowLength,bufferImageHeight=bufferImageHeight,imageSubresource=imageSubresource,imageOffset=imageOffset,imageExtent=imageExtent)

def VkCopyMemoryIndirectCommandNV(srcAddress=None,dstAddress=None,size=None,):

    return _new('VkCopyMemoryIndirectCommandNV', srcAddress=srcAddress,dstAddress=dstAddress,size=size)

def VkCopyMemoryToImageIndirectCommandNV(srcAddress=None,bufferRowLength=None,bufferImageHeight=None,imageSubresource=None,imageOffset=None,imageExtent=None,):

    return _new('VkCopyMemoryToImageIndirectCommandNV', srcAddress=srcAddress,bufferRowLength=bufferRowLength,bufferImageHeight=bufferImageHeight,imageSubresource=imageSubresource,imageOffset=imageOffset,imageExtent=imageExtent)

def VkImageResolve(srcSubresource=None,srcOffset=None,dstSubresource=None,dstOffset=None,extent=None,):

    return _new('VkImageResolve', srcSubresource=srcSubresource,srcOffset=srcOffset,dstSubresource=dstSubresource,dstOffset=dstOffset,extent=extent)

def VkShaderModuleCreateInfo(sType=VK_STRUCTURE_TYPE_SHADER_MODULE_CREATE_INFO,pNext=None,flags=None,codeSize=None,pCode=None,):

    return _new('VkShaderModuleCreateInfo', sType=sType,pNext=pNext,flags=flags,codeSize=codeSize,pCode=pCode)

def VkDescriptorSetLayoutBinding(binding=None,descriptorType=None,descriptorCount=None,stageFlags=None,pImmutableSamplers=None,):
    if descriptorCount is None and pImmutableSamplers is not None:
        descriptorCount = len(pImmutableSamplers)

    return _new('VkDescriptorSetLayoutBinding', binding=binding,descriptorType=descriptorType,descriptorCount=descriptorCount,stageFlags=stageFlags,pImmutableSamplers=pImmutableSamplers)

def VkDescriptorSetLayoutCreateInfo(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_CREATE_INFO,pNext=None,flags=None,bindingCount=None,pBindings=None,):
    if bindingCount is None and pBindings is not None:
        bindingCount = len(pBindings)

    return _new('VkDescriptorSetLayoutCreateInfo', sType=sType,pNext=pNext,flags=flags,bindingCount=bindingCount,pBindings=pBindings)

def VkDescriptorPoolSize(type=None,descriptorCount=None,):

    return _new('VkDescriptorPoolSize', type=type,descriptorCount=descriptorCount)

def VkDescriptorPoolCreateInfo(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_CREATE_INFO,pNext=None,flags=None,maxSets=None,poolSizeCount=None,pPoolSizes=None,):
    if poolSizeCount is None and pPoolSizes is not None:
        poolSizeCount = len(pPoolSizes)

    return _new('VkDescriptorPoolCreateInfo', sType=sType,pNext=pNext,flags=flags,maxSets=maxSets,poolSizeCount=poolSizeCount,pPoolSizes=pPoolSizes)

def VkDescriptorSetAllocateInfo(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_ALLOCATE_INFO,pNext=None,descriptorPool=None,descriptorSetCount=None,pSetLayouts=None,):
    if descriptorSetCount is None and pSetLayouts is not None:
        descriptorSetCount = len(pSetLayouts)

    return _new('VkDescriptorSetAllocateInfo', sType=sType,pNext=pNext,descriptorPool=descriptorPool,descriptorSetCount=descriptorSetCount,pSetLayouts=pSetLayouts)

def VkSpecializationMapEntry(constantID=None,offset=None,size=None,):

    return _new('VkSpecializationMapEntry', constantID=constantID,offset=offset,size=size)

def VkSpecializationInfo(mapEntryCount=None,pMapEntries=None,dataSize=None,pData=None,):
    if mapEntryCount is None and pMapEntries is not None:
        mapEntryCount = len(pMapEntries)
    if dataSize is None and pData is not None:
        dataSize = len(pData)

    return _new('VkSpecializationInfo', mapEntryCount=mapEntryCount,pMapEntries=pMapEntries,dataSize=dataSize,pData=pData)

def VkPipelineShaderStageCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_CREATE_INFO,pNext=None,flags=None,stage=None,module=None,pName=None,pSpecializationInfo=None,):

    return _new('VkPipelineShaderStageCreateInfo', sType=sType,pNext=pNext,flags=flags,stage=stage,module=module,pName=pName,pSpecializationInfo=pSpecializationInfo)

def VkComputePipelineCreateInfo(sType=VK_STRUCTURE_TYPE_COMPUTE_PIPELINE_CREATE_INFO,pNext=None,flags=None,stage=None,layout=None,basePipelineHandle=None,basePipelineIndex=None,):

    return _new('VkComputePipelineCreateInfo', sType=sType,pNext=pNext,flags=flags,stage=stage,layout=layout,basePipelineHandle=basePipelineHandle,basePipelineIndex=basePipelineIndex)

def VkComputePipelineIndirectBufferInfoNV(sType=VK_STRUCTURE_TYPE_COMPUTE_PIPELINE_INDIRECT_BUFFER_INFO_NV,pNext=None,deviceAddress=None,size=None,pipelineDeviceAddressCaptureReplay=None,):

    return _new('VkComputePipelineIndirectBufferInfoNV', sType=sType,pNext=pNext,deviceAddress=deviceAddress,size=size,pipelineDeviceAddressCaptureReplay=pipelineDeviceAddressCaptureReplay)

def VkPipelineCreateFlags2CreateInfoKHR(sType=VK_STRUCTURE_TYPE_PIPELINE_CREATE_FLAGS_2_CREATE_INFO_KHR,pNext=None,flags=None,):

    return _new('VkPipelineCreateFlags2CreateInfoKHR', sType=sType,pNext=pNext,flags=flags)

def VkVertexInputBindingDescription(binding=None,stride=None,inputRate=None,):

    return _new('VkVertexInputBindingDescription', binding=binding,stride=stride,inputRate=inputRate)

def VkVertexInputAttributeDescription(location=None,binding=None,format=None,offset=None,):

    return _new('VkVertexInputAttributeDescription', location=location,binding=binding,format=format,offset=offset)

def VkPipelineVertexInputStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_STATE_CREATE_INFO,pNext=None,flags=None,vertexBindingDescriptionCount=None,pVertexBindingDescriptions=None,vertexAttributeDescriptionCount=None,pVertexAttributeDescriptions=None,):
    if vertexBindingDescriptionCount is None and pVertexBindingDescriptions is not None:
        vertexBindingDescriptionCount = len(pVertexBindingDescriptions)
    if vertexAttributeDescriptionCount is None and pVertexAttributeDescriptions is not None:
        vertexAttributeDescriptionCount = len(pVertexAttributeDescriptions)

    return _new('VkPipelineVertexInputStateCreateInfo', sType=sType,pNext=pNext,flags=flags,vertexBindingDescriptionCount=vertexBindingDescriptionCount,pVertexBindingDescriptions=pVertexBindingDescriptions,vertexAttributeDescriptionCount=vertexAttributeDescriptionCount,pVertexAttributeDescriptions=pVertexAttributeDescriptions)

def VkPipelineInputAssemblyStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_INPUT_ASSEMBLY_STATE_CREATE_INFO,pNext=None,flags=None,topology=None,primitiveRestartEnable=None,):

    return _new('VkPipelineInputAssemblyStateCreateInfo', sType=sType,pNext=pNext,flags=flags,topology=topology,primitiveRestartEnable=primitiveRestartEnable)

def VkPipelineTessellationStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_STATE_CREATE_INFO,pNext=None,flags=None,patchControlPoints=None,):

    return _new('VkPipelineTessellationStateCreateInfo', sType=sType,pNext=pNext,flags=flags,patchControlPoints=patchControlPoints)

def VkPipelineViewportStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_STATE_CREATE_INFO,pNext=None,flags=None,viewportCount=None,pViewports=None,scissorCount=None,pScissors=None,):
    if viewportCount is None and pViewports is not None:
        viewportCount = len(pViewports)
    if scissorCount is None and pScissors is not None:
        scissorCount = len(pScissors)

    return _new('VkPipelineViewportStateCreateInfo', sType=sType,pNext=pNext,flags=flags,viewportCount=viewportCount,pViewports=pViewports,scissorCount=scissorCount,pScissors=pScissors)

def VkPipelineRasterizationStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_CREATE_INFO,pNext=None,flags=None,depthClampEnable=None,rasterizerDiscardEnable=None,polygonMode=None,cullMode=None,frontFace=None,depthBiasEnable=None,depthBiasConstantFactor=None,depthBiasClamp=None,depthBiasSlopeFactor=None,lineWidth=None,):

    return _new('VkPipelineRasterizationStateCreateInfo', sType=sType,pNext=pNext,flags=flags,depthClampEnable=depthClampEnable,rasterizerDiscardEnable=rasterizerDiscardEnable,polygonMode=polygonMode,cullMode=cullMode,frontFace=frontFace,depthBiasEnable=depthBiasEnable,depthBiasConstantFactor=depthBiasConstantFactor,depthBiasClamp=depthBiasClamp,depthBiasSlopeFactor=depthBiasSlopeFactor,lineWidth=lineWidth)

def VkPipelineMultisampleStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_MULTISAMPLE_STATE_CREATE_INFO,pNext=None,flags=None,rasterizationSamples=None,sampleShadingEnable=None,minSampleShading=None,pSampleMask=None,alphaToCoverageEnable=None,alphaToOneEnable=None,):

    return _new('VkPipelineMultisampleStateCreateInfo', sType=sType,pNext=pNext,flags=flags,rasterizationSamples=rasterizationSamples,sampleShadingEnable=sampleShadingEnable,minSampleShading=minSampleShading,pSampleMask=pSampleMask,alphaToCoverageEnable=alphaToCoverageEnable,alphaToOneEnable=alphaToOneEnable)

def VkPipelineColorBlendAttachmentState(blendEnable=None,srcColorBlendFactor=None,dstColorBlendFactor=None,colorBlendOp=None,srcAlphaBlendFactor=None,dstAlphaBlendFactor=None,alphaBlendOp=None,colorWriteMask=None,):

    return _new('VkPipelineColorBlendAttachmentState', blendEnable=blendEnable,srcColorBlendFactor=srcColorBlendFactor,dstColorBlendFactor=dstColorBlendFactor,colorBlendOp=colorBlendOp,srcAlphaBlendFactor=srcAlphaBlendFactor,dstAlphaBlendFactor=dstAlphaBlendFactor,alphaBlendOp=alphaBlendOp,colorWriteMask=colorWriteMask)

def VkPipelineColorBlendStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_STATE_CREATE_INFO,pNext=None,flags=None,logicOpEnable=None,logicOp=None,attachmentCount=None,pAttachments=None,blendConstants=None,):
    if attachmentCount is None and pAttachments is not None:
        attachmentCount = len(pAttachments)

    return _new('VkPipelineColorBlendStateCreateInfo', sType=sType,pNext=pNext,flags=flags,logicOpEnable=logicOpEnable,logicOp=logicOp,attachmentCount=attachmentCount,pAttachments=pAttachments,blendConstants=blendConstants)

def VkPipelineDynamicStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_DYNAMIC_STATE_CREATE_INFO,pNext=None,flags=None,dynamicStateCount=None,pDynamicStates=None,):
    if dynamicStateCount is None and pDynamicStates is not None:
        dynamicStateCount = len(pDynamicStates)

    return _new('VkPipelineDynamicStateCreateInfo', sType=sType,pNext=pNext,flags=flags,dynamicStateCount=dynamicStateCount,pDynamicStates=pDynamicStates)

def VkStencilOpState(failOp=None,passOp=None,depthFailOp=None,compareOp=None,compareMask=None,writeMask=None,reference=None,):

    return _new('VkStencilOpState', failOp=failOp,passOp=passOp,depthFailOp=depthFailOp,compareOp=compareOp,compareMask=compareMask,writeMask=writeMask,reference=reference)

def VkPipelineDepthStencilStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_DEPTH_STENCIL_STATE_CREATE_INFO,pNext=None,flags=None,depthTestEnable=None,depthWriteEnable=None,depthCompareOp=None,depthBoundsTestEnable=None,stencilTestEnable=None,front=None,back=None,minDepthBounds=None,maxDepthBounds=None,):

    return _new('VkPipelineDepthStencilStateCreateInfo', sType=sType,pNext=pNext,flags=flags,depthTestEnable=depthTestEnable,depthWriteEnable=depthWriteEnable,depthCompareOp=depthCompareOp,depthBoundsTestEnable=depthBoundsTestEnable,stencilTestEnable=stencilTestEnable,front=front,back=back,minDepthBounds=minDepthBounds,maxDepthBounds=maxDepthBounds)

def VkGraphicsPipelineCreateInfo(sType=VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_CREATE_INFO,pNext=None,flags=None,stageCount=None,pStages=None,pVertexInputState=None,pInputAssemblyState=None,pTessellationState=None,pViewportState=None,pRasterizationState=None,pMultisampleState=None,pDepthStencilState=None,pColorBlendState=None,pDynamicState=None,layout=None,renderPass=None,subpass=None,basePipelineHandle=None,basePipelineIndex=None,):
    if stageCount is None and pStages is not None:
        stageCount = len(pStages)

    return _new('VkGraphicsPipelineCreateInfo', sType=sType,pNext=pNext,flags=flags,stageCount=stageCount,pStages=pStages,pVertexInputState=pVertexInputState,pInputAssemblyState=pInputAssemblyState,pTessellationState=pTessellationState,pViewportState=pViewportState,pRasterizationState=pRasterizationState,pMultisampleState=pMultisampleState,pDepthStencilState=pDepthStencilState,pColorBlendState=pColorBlendState,pDynamicState=pDynamicState,layout=layout,renderPass=renderPass,subpass=subpass,basePipelineHandle=basePipelineHandle,basePipelineIndex=basePipelineIndex)

def VkPipelineCacheCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_CACHE_CREATE_INFO,pNext=None,flags=None,initialDataSize=None,pInitialData=None,):
    if initialDataSize is None and pInitialData is not None:
        initialDataSize = len(pInitialData)

    return _new('VkPipelineCacheCreateInfo', sType=sType,pNext=pNext,flags=flags,initialDataSize=initialDataSize,pInitialData=pInitialData)

def VkPipelineCacheHeaderVersionOne(headerSize=None,headerVersion=None,vendorID=None,deviceID=None,pipelineCacheUUID=None,):

    return _new('VkPipelineCacheHeaderVersionOne', headerSize=headerSize,headerVersion=headerVersion,vendorID=vendorID,deviceID=deviceID,pipelineCacheUUID=pipelineCacheUUID)

def VkPipelineCacheStageValidationIndexEntry(codeSize=None,codeOffset=None,):

    return _new('VkPipelineCacheStageValidationIndexEntry', codeSize=codeSize,codeOffset=codeOffset)

def VkPipelineCacheSafetyCriticalIndexEntry(pipelineIdentifier=None,pipelineMemorySize=None,jsonSize=None,jsonOffset=None,stageIndexCount=None,stageIndexStride=None,stageIndexOffset=None,):

    return _new('VkPipelineCacheSafetyCriticalIndexEntry', pipelineIdentifier=pipelineIdentifier,pipelineMemorySize=pipelineMemorySize,jsonSize=jsonSize,jsonOffset=jsonOffset,stageIndexCount=stageIndexCount,stageIndexStride=stageIndexStride,stageIndexOffset=stageIndexOffset)

def VkPipelineCacheHeaderVersionSafetyCriticalOne(headerVersionOne=None,validationVersion=None,implementationData=None,pipelineIndexCount=None,pipelineIndexStride=None,pipelineIndexOffset=None,):

    return _new('VkPipelineCacheHeaderVersionSafetyCriticalOne', headerVersionOne=headerVersionOne,validationVersion=validationVersion,implementationData=implementationData,pipelineIndexCount=pipelineIndexCount,pipelineIndexStride=pipelineIndexStride,pipelineIndexOffset=pipelineIndexOffset)

def VkPushConstantRange(stageFlags=None,offset=None,size=None,):

    return _new('VkPushConstantRange', stageFlags=stageFlags,offset=offset,size=size)

def VkPipelineLayoutCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_LAYOUT_CREATE_INFO,pNext=None,flags=None,setLayoutCount=None,pSetLayouts=None,pushConstantRangeCount=None,pPushConstantRanges=None,):
    if setLayoutCount is None and pSetLayouts is not None:
        setLayoutCount = len(pSetLayouts)
    if pushConstantRangeCount is None and pPushConstantRanges is not None:
        pushConstantRangeCount = len(pPushConstantRanges)

    return _new('VkPipelineLayoutCreateInfo', sType=sType,pNext=pNext,flags=flags,setLayoutCount=setLayoutCount,pSetLayouts=pSetLayouts,pushConstantRangeCount=pushConstantRangeCount,pPushConstantRanges=pPushConstantRanges)

def VkSamplerCreateInfo(sType=VK_STRUCTURE_TYPE_SAMPLER_CREATE_INFO,pNext=None,flags=None,magFilter=None,minFilter=None,mipmapMode=None,addressModeU=None,addressModeV=None,addressModeW=None,mipLodBias=None,anisotropyEnable=None,maxAnisotropy=None,compareEnable=None,compareOp=None,minLod=None,maxLod=None,borderColor=None,unnormalizedCoordinates=None,):

    return _new('VkSamplerCreateInfo', sType=sType,pNext=pNext,flags=flags,magFilter=magFilter,minFilter=minFilter,mipmapMode=mipmapMode,addressModeU=addressModeU,addressModeV=addressModeV,addressModeW=addressModeW,mipLodBias=mipLodBias,anisotropyEnable=anisotropyEnable,maxAnisotropy=maxAnisotropy,compareEnable=compareEnable,compareOp=compareOp,minLod=minLod,maxLod=maxLod,borderColor=borderColor,unnormalizedCoordinates=unnormalizedCoordinates)

def VkCommandPoolCreateInfo(sType=VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO,pNext=None,flags=None,queueFamilyIndex=None,):

    return _new('VkCommandPoolCreateInfo', sType=sType,pNext=pNext,flags=flags,queueFamilyIndex=queueFamilyIndex)

def VkCommandBufferAllocateInfo(sType=VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO,pNext=None,commandPool=None,level=None,commandBufferCount=None,):

    return _new('VkCommandBufferAllocateInfo', sType=sType,pNext=pNext,commandPool=commandPool,level=level,commandBufferCount=commandBufferCount)

def VkCommandBufferInheritanceInfo(sType=VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_INFO,pNext=None,renderPass=None,subpass=None,framebuffer=None,occlusionQueryEnable=None,queryFlags=None,pipelineStatistics=None,):

    return _new('VkCommandBufferInheritanceInfo', sType=sType,pNext=pNext,renderPass=renderPass,subpass=subpass,framebuffer=framebuffer,occlusionQueryEnable=occlusionQueryEnable,queryFlags=queryFlags,pipelineStatistics=pipelineStatistics)

def VkCommandBufferBeginInfo(sType=VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO,pNext=None,flags=None,pInheritanceInfo=None,):

    return _new('VkCommandBufferBeginInfo', sType=sType,pNext=pNext,flags=flags,pInheritanceInfo=pInheritanceInfo)

def VkRenderPassBeginInfo(sType=VK_STRUCTURE_TYPE_RENDER_PASS_BEGIN_INFO,pNext=None,renderPass=None,framebuffer=None,renderArea=None,clearValueCount=None,pClearValues=None,):
    if clearValueCount is None and pClearValues is not None:
        clearValueCount = len(pClearValues)

    return _new('VkRenderPassBeginInfo', sType=sType,pNext=pNext,renderPass=renderPass,framebuffer=framebuffer,renderArea=renderArea,clearValueCount=clearValueCount,pClearValues=pClearValues)

def VkClearColorValue(float32=None,int32=None,uint32=None,):

    return _new('VkClearColorValue', float32=float32,int32=int32,uint32=uint32)

def VkClearDepthStencilValue(depth=None,stencil=None,):

    return _new('VkClearDepthStencilValue', depth=depth,stencil=stencil)

def VkClearValue(color=None,depthStencil=None,):

    return _new('VkClearValue', color=color,depthStencil=depthStencil)

def VkClearAttachment(aspectMask=None,colorAttachment=None,clearValue=None,):

    return _new('VkClearAttachment', aspectMask=aspectMask,colorAttachment=colorAttachment,clearValue=clearValue)

def VkAttachmentDescription(flags=None,format=None,samples=None,loadOp=None,storeOp=None,stencilLoadOp=None,stencilStoreOp=None,initialLayout=None,finalLayout=None,):

    return _new('VkAttachmentDescription', flags=flags,format=format,samples=samples,loadOp=loadOp,storeOp=storeOp,stencilLoadOp=stencilLoadOp,stencilStoreOp=stencilStoreOp,initialLayout=initialLayout,finalLayout=finalLayout)

def VkAttachmentReference(attachment=None,layout=None,):

    return _new('VkAttachmentReference', attachment=attachment,layout=layout)

def VkSubpassDescription(flags=None,pipelineBindPoint=None,inputAttachmentCount=None,pInputAttachments=None,colorAttachmentCount=None,pColorAttachments=None,pResolveAttachments=None,pDepthStencilAttachment=None,preserveAttachmentCount=None,pPreserveAttachments=None,):
    if inputAttachmentCount is None and pInputAttachments is not None:
        inputAttachmentCount = len(pInputAttachments)
    if colorAttachmentCount is None and pColorAttachments is not None:
        colorAttachmentCount = len(pColorAttachments)
    if colorAttachmentCount is None and pResolveAttachments is not None:
        colorAttachmentCount = len(pResolveAttachments)
    if preserveAttachmentCount is None and pPreserveAttachments is not None:
        preserveAttachmentCount = len(pPreserveAttachments)

    return _new('VkSubpassDescription', flags=flags,pipelineBindPoint=pipelineBindPoint,inputAttachmentCount=inputAttachmentCount,pInputAttachments=pInputAttachments,colorAttachmentCount=colorAttachmentCount,pColorAttachments=pColorAttachments,pResolveAttachments=pResolveAttachments,pDepthStencilAttachment=pDepthStencilAttachment,preserveAttachmentCount=preserveAttachmentCount,pPreserveAttachments=pPreserveAttachments)

def VkSubpassDependency(srcSubpass=None,dstSubpass=None,srcStageMask=None,dstStageMask=None,srcAccessMask=None,dstAccessMask=None,dependencyFlags=None,):

    return _new('VkSubpassDependency', srcSubpass=srcSubpass,dstSubpass=dstSubpass,srcStageMask=srcStageMask,dstStageMask=dstStageMask,srcAccessMask=srcAccessMask,dstAccessMask=dstAccessMask,dependencyFlags=dependencyFlags)

def VkRenderPassCreateInfo(sType=VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO,pNext=None,flags=None,attachmentCount=None,pAttachments=None,subpassCount=None,pSubpasses=None,dependencyCount=None,pDependencies=None,):
    if attachmentCount is None and pAttachments is not None:
        attachmentCount = len(pAttachments)
    if subpassCount is None and pSubpasses is not None:
        subpassCount = len(pSubpasses)
    if dependencyCount is None and pDependencies is not None:
        dependencyCount = len(pDependencies)

    return _new('VkRenderPassCreateInfo', sType=sType,pNext=pNext,flags=flags,attachmentCount=attachmentCount,pAttachments=pAttachments,subpassCount=subpassCount,pSubpasses=pSubpasses,dependencyCount=dependencyCount,pDependencies=pDependencies)

def VkEventCreateInfo(sType=VK_STRUCTURE_TYPE_EVENT_CREATE_INFO,pNext=None,flags=None,):

    return _new('VkEventCreateInfo', sType=sType,pNext=pNext,flags=flags)

def VkFenceCreateInfo(sType=VK_STRUCTURE_TYPE_FENCE_CREATE_INFO,pNext=None,flags=None,):

    return _new('VkFenceCreateInfo', sType=sType,pNext=pNext,flags=flags)

def VkPhysicalDeviceFeatures(robustBufferAccess=None,fullDrawIndexUint32=None,imageCubeArray=None,independentBlend=None,geometryShader=None,tessellationShader=None,sampleRateShading=None,dualSrcBlend=None,logicOp=None,multiDrawIndirect=None,drawIndirectFirstInstance=None,depthClamp=None,depthBiasClamp=None,fillModeNonSolid=None,depthBounds=None,wideLines=None,largePoints=None,alphaToOne=None,multiViewport=None,samplerAnisotropy=None,textureCompressionETC2=None,textureCompressionASTC_LDR=None,textureCompressionBC=None,occlusionQueryPrecise=None,pipelineStatisticsQuery=None,vertexPipelineStoresAndAtomics=None,fragmentStoresAndAtomics=None,shaderTessellationAndGeometryPointSize=None,shaderImageGatherExtended=None,shaderStorageImageExtendedFormats=None,shaderStorageImageMultisample=None,shaderStorageImageReadWithoutFormat=None,shaderStorageImageWriteWithoutFormat=None,shaderUniformBufferArrayDynamicIndexing=None,shaderSampledImageArrayDynamicIndexing=None,shaderStorageBufferArrayDynamicIndexing=None,shaderStorageImageArrayDynamicIndexing=None,shaderClipDistance=None,shaderCullDistance=None,shaderFloat64=None,shaderInt64=None,shaderInt16=None,shaderResourceResidency=None,shaderResourceMinLod=None,sparseBinding=None,sparseResidencyBuffer=None,sparseResidencyImage2D=None,sparseResidencyImage3D=None,sparseResidency2Samples=None,sparseResidency4Samples=None,sparseResidency8Samples=None,sparseResidency16Samples=None,sparseResidencyAliased=None,variableMultisampleRate=None,inheritedQueries=None,):

    return _new('VkPhysicalDeviceFeatures', robustBufferAccess=robustBufferAccess,fullDrawIndexUint32=fullDrawIndexUint32,imageCubeArray=imageCubeArray,independentBlend=independentBlend,geometryShader=geometryShader,tessellationShader=tessellationShader,sampleRateShading=sampleRateShading,dualSrcBlend=dualSrcBlend,logicOp=logicOp,multiDrawIndirect=multiDrawIndirect,drawIndirectFirstInstance=drawIndirectFirstInstance,depthClamp=depthClamp,depthBiasClamp=depthBiasClamp,fillModeNonSolid=fillModeNonSolid,depthBounds=depthBounds,wideLines=wideLines,largePoints=largePoints,alphaToOne=alphaToOne,multiViewport=multiViewport,samplerAnisotropy=samplerAnisotropy,textureCompressionETC2=textureCompressionETC2,textureCompressionASTC_LDR=textureCompressionASTC_LDR,textureCompressionBC=textureCompressionBC,occlusionQueryPrecise=occlusionQueryPrecise,pipelineStatisticsQuery=pipelineStatisticsQuery,vertexPipelineStoresAndAtomics=vertexPipelineStoresAndAtomics,fragmentStoresAndAtomics=fragmentStoresAndAtomics,shaderTessellationAndGeometryPointSize=shaderTessellationAndGeometryPointSize,shaderImageGatherExtended=shaderImageGatherExtended,shaderStorageImageExtendedFormats=shaderStorageImageExtendedFormats,shaderStorageImageMultisample=shaderStorageImageMultisample,shaderStorageImageReadWithoutFormat=shaderStorageImageReadWithoutFormat,shaderStorageImageWriteWithoutFormat=shaderStorageImageWriteWithoutFormat,shaderUniformBufferArrayDynamicIndexing=shaderUniformBufferArrayDynamicIndexing,shaderSampledImageArrayDynamicIndexing=shaderSampledImageArrayDynamicIndexing,shaderStorageBufferArrayDynamicIndexing=shaderStorageBufferArrayDynamicIndexing,shaderStorageImageArrayDynamicIndexing=shaderStorageImageArrayDynamicIndexing,shaderClipDistance=shaderClipDistance,shaderCullDistance=shaderCullDistance,shaderFloat64=shaderFloat64,shaderInt64=shaderInt64,shaderInt16=shaderInt16,shaderResourceResidency=shaderResourceResidency,shaderResourceMinLod=shaderResourceMinLod,sparseBinding=sparseBinding,sparseResidencyBuffer=sparseResidencyBuffer,sparseResidencyImage2D=sparseResidencyImage2D,sparseResidencyImage3D=sparseResidencyImage3D,sparseResidency2Samples=sparseResidency2Samples,sparseResidency4Samples=sparseResidency4Samples,sparseResidency8Samples=sparseResidency8Samples,sparseResidency16Samples=sparseResidency16Samples,sparseResidencyAliased=sparseResidencyAliased,variableMultisampleRate=variableMultisampleRate,inheritedQueries=inheritedQueries)

def VkPhysicalDeviceSparseProperties(residencyStandard2DBlockShape=None,residencyStandard2DMultisampleBlockShape=None,residencyStandard3DBlockShape=None,residencyAlignedMipSize=None,residencyNonResidentStrict=None,):

    return _new('VkPhysicalDeviceSparseProperties', residencyStandard2DBlockShape=residencyStandard2DBlockShape,residencyStandard2DMultisampleBlockShape=residencyStandard2DMultisampleBlockShape,residencyStandard3DBlockShape=residencyStandard3DBlockShape,residencyAlignedMipSize=residencyAlignedMipSize,residencyNonResidentStrict=residencyNonResidentStrict)

def VkPhysicalDeviceLimits(maxImageDimension1D=None,maxImageDimension2D=None,maxImageDimension3D=None,maxImageDimensionCube=None,maxImageArrayLayers=None,maxTexelBufferElements=None,maxUniformBufferRange=None,maxStorageBufferRange=None,maxPushConstantsSize=None,maxMemoryAllocationCount=None,maxSamplerAllocationCount=None,bufferImageGranularity=None,sparseAddressSpaceSize=None,maxBoundDescriptorSets=None,maxPerStageDescriptorSamplers=None,maxPerStageDescriptorUniformBuffers=None,maxPerStageDescriptorStorageBuffers=None,maxPerStageDescriptorSampledImages=None,maxPerStageDescriptorStorageImages=None,maxPerStageDescriptorInputAttachments=None,maxPerStageResources=None,maxDescriptorSetSamplers=None,maxDescriptorSetUniformBuffers=None,maxDescriptorSetUniformBuffersDynamic=None,maxDescriptorSetStorageBuffers=None,maxDescriptorSetStorageBuffersDynamic=None,maxDescriptorSetSampledImages=None,maxDescriptorSetStorageImages=None,maxDescriptorSetInputAttachments=None,maxVertexInputAttributes=None,maxVertexInputBindings=None,maxVertexInputAttributeOffset=None,maxVertexInputBindingStride=None,maxVertexOutputComponents=None,maxTessellationGenerationLevel=None,maxTessellationPatchSize=None,maxTessellationControlPerVertexInputComponents=None,maxTessellationControlPerVertexOutputComponents=None,maxTessellationControlPerPatchOutputComponents=None,maxTessellationControlTotalOutputComponents=None,maxTessellationEvaluationInputComponents=None,maxTessellationEvaluationOutputComponents=None,maxGeometryShaderInvocations=None,maxGeometryInputComponents=None,maxGeometryOutputComponents=None,maxGeometryOutputVertices=None,maxGeometryTotalOutputComponents=None,maxFragmentInputComponents=None,maxFragmentOutputAttachments=None,maxFragmentDualSrcAttachments=None,maxFragmentCombinedOutputResources=None,maxComputeSharedMemorySize=None,maxComputeWorkGroupCount=None,maxComputeWorkGroupInvocations=None,maxComputeWorkGroupSize=None,subPixelPrecisionBits=None,subTexelPrecisionBits=None,mipmapPrecisionBits=None,maxDrawIndexedIndexValue=None,maxDrawIndirectCount=None,maxSamplerLodBias=None,maxSamplerAnisotropy=None,maxViewports=None,maxViewportDimensions=None,viewportBoundsRange=None,viewportSubPixelBits=None,minMemoryMapAlignment=None,minTexelBufferOffsetAlignment=None,minUniformBufferOffsetAlignment=None,minStorageBufferOffsetAlignment=None,minTexelOffset=None,maxTexelOffset=None,minTexelGatherOffset=None,maxTexelGatherOffset=None,minInterpolationOffset=None,maxInterpolationOffset=None,subPixelInterpolationOffsetBits=None,maxFramebufferWidth=None,maxFramebufferHeight=None,maxFramebufferLayers=None,framebufferColorSampleCounts=None,framebufferDepthSampleCounts=None,framebufferStencilSampleCounts=None,framebufferNoAttachmentsSampleCounts=None,maxColorAttachments=None,sampledImageColorSampleCounts=None,sampledImageIntegerSampleCounts=None,sampledImageDepthSampleCounts=None,sampledImageStencilSampleCounts=None,storageImageSampleCounts=None,maxSampleMaskWords=None,timestampComputeAndGraphics=None,timestampPeriod=None,maxClipDistances=None,maxCullDistances=None,maxCombinedClipAndCullDistances=None,discreteQueuePriorities=None,pointSizeRange=None,lineWidthRange=None,pointSizeGranularity=None,lineWidthGranularity=None,strictLines=None,standardSampleLocations=None,optimalBufferCopyOffsetAlignment=None,optimalBufferCopyRowPitchAlignment=None,nonCoherentAtomSize=None,):

    return _new('VkPhysicalDeviceLimits', maxImageDimension1D=maxImageDimension1D,maxImageDimension2D=maxImageDimension2D,maxImageDimension3D=maxImageDimension3D,maxImageDimensionCube=maxImageDimensionCube,maxImageArrayLayers=maxImageArrayLayers,maxTexelBufferElements=maxTexelBufferElements,maxUniformBufferRange=maxUniformBufferRange,maxStorageBufferRange=maxStorageBufferRange,maxPushConstantsSize=maxPushConstantsSize,maxMemoryAllocationCount=maxMemoryAllocationCount,maxSamplerAllocationCount=maxSamplerAllocationCount,bufferImageGranularity=bufferImageGranularity,sparseAddressSpaceSize=sparseAddressSpaceSize,maxBoundDescriptorSets=maxBoundDescriptorSets,maxPerStageDescriptorSamplers=maxPerStageDescriptorSamplers,maxPerStageDescriptorUniformBuffers=maxPerStageDescriptorUniformBuffers,maxPerStageDescriptorStorageBuffers=maxPerStageDescriptorStorageBuffers,maxPerStageDescriptorSampledImages=maxPerStageDescriptorSampledImages,maxPerStageDescriptorStorageImages=maxPerStageDescriptorStorageImages,maxPerStageDescriptorInputAttachments=maxPerStageDescriptorInputAttachments,maxPerStageResources=maxPerStageResources,maxDescriptorSetSamplers=maxDescriptorSetSamplers,maxDescriptorSetUniformBuffers=maxDescriptorSetUniformBuffers,maxDescriptorSetUniformBuffersDynamic=maxDescriptorSetUniformBuffersDynamic,maxDescriptorSetStorageBuffers=maxDescriptorSetStorageBuffers,maxDescriptorSetStorageBuffersDynamic=maxDescriptorSetStorageBuffersDynamic,maxDescriptorSetSampledImages=maxDescriptorSetSampledImages,maxDescriptorSetStorageImages=maxDescriptorSetStorageImages,maxDescriptorSetInputAttachments=maxDescriptorSetInputAttachments,maxVertexInputAttributes=maxVertexInputAttributes,maxVertexInputBindings=maxVertexInputBindings,maxVertexInputAttributeOffset=maxVertexInputAttributeOffset,maxVertexInputBindingStride=maxVertexInputBindingStride,maxVertexOutputComponents=maxVertexOutputComponents,maxTessellationGenerationLevel=maxTessellationGenerationLevel,maxTessellationPatchSize=maxTessellationPatchSize,maxTessellationControlPerVertexInputComponents=maxTessellationControlPerVertexInputComponents,maxTessellationControlPerVertexOutputComponents=maxTessellationControlPerVertexOutputComponents,maxTessellationControlPerPatchOutputComponents=maxTessellationControlPerPatchOutputComponents,maxTessellationControlTotalOutputComponents=maxTessellationControlTotalOutputComponents,maxTessellationEvaluationInputComponents=maxTessellationEvaluationInputComponents,maxTessellationEvaluationOutputComponents=maxTessellationEvaluationOutputComponents,maxGeometryShaderInvocations=maxGeometryShaderInvocations,maxGeometryInputComponents=maxGeometryInputComponents,maxGeometryOutputComponents=maxGeometryOutputComponents,maxGeometryOutputVertices=maxGeometryOutputVertices,maxGeometryTotalOutputComponents=maxGeometryTotalOutputComponents,maxFragmentInputComponents=maxFragmentInputComponents,maxFragmentOutputAttachments=maxFragmentOutputAttachments,maxFragmentDualSrcAttachments=maxFragmentDualSrcAttachments,maxFragmentCombinedOutputResources=maxFragmentCombinedOutputResources,maxComputeSharedMemorySize=maxComputeSharedMemorySize,maxComputeWorkGroupCount=maxComputeWorkGroupCount,maxComputeWorkGroupInvocations=maxComputeWorkGroupInvocations,maxComputeWorkGroupSize=maxComputeWorkGroupSize,subPixelPrecisionBits=subPixelPrecisionBits,subTexelPrecisionBits=subTexelPrecisionBits,mipmapPrecisionBits=mipmapPrecisionBits,maxDrawIndexedIndexValue=maxDrawIndexedIndexValue,maxDrawIndirectCount=maxDrawIndirectCount,maxSamplerLodBias=maxSamplerLodBias,maxSamplerAnisotropy=maxSamplerAnisotropy,maxViewports=maxViewports,maxViewportDimensions=maxViewportDimensions,viewportBoundsRange=viewportBoundsRange,viewportSubPixelBits=viewportSubPixelBits,minMemoryMapAlignment=minMemoryMapAlignment,minTexelBufferOffsetAlignment=minTexelBufferOffsetAlignment,minUniformBufferOffsetAlignment=minUniformBufferOffsetAlignment,minStorageBufferOffsetAlignment=minStorageBufferOffsetAlignment,minTexelOffset=minTexelOffset,maxTexelOffset=maxTexelOffset,minTexelGatherOffset=minTexelGatherOffset,maxTexelGatherOffset=maxTexelGatherOffset,minInterpolationOffset=minInterpolationOffset,maxInterpolationOffset=maxInterpolationOffset,subPixelInterpolationOffsetBits=subPixelInterpolationOffsetBits,maxFramebufferWidth=maxFramebufferWidth,maxFramebufferHeight=maxFramebufferHeight,maxFramebufferLayers=maxFramebufferLayers,framebufferColorSampleCounts=framebufferColorSampleCounts,framebufferDepthSampleCounts=framebufferDepthSampleCounts,framebufferStencilSampleCounts=framebufferStencilSampleCounts,framebufferNoAttachmentsSampleCounts=framebufferNoAttachmentsSampleCounts,maxColorAttachments=maxColorAttachments,sampledImageColorSampleCounts=sampledImageColorSampleCounts,sampledImageIntegerSampleCounts=sampledImageIntegerSampleCounts,sampledImageDepthSampleCounts=sampledImageDepthSampleCounts,sampledImageStencilSampleCounts=sampledImageStencilSampleCounts,storageImageSampleCounts=storageImageSampleCounts,maxSampleMaskWords=maxSampleMaskWords,timestampComputeAndGraphics=timestampComputeAndGraphics,timestampPeriod=timestampPeriod,maxClipDistances=maxClipDistances,maxCullDistances=maxCullDistances,maxCombinedClipAndCullDistances=maxCombinedClipAndCullDistances,discreteQueuePriorities=discreteQueuePriorities,pointSizeRange=pointSizeRange,lineWidthRange=lineWidthRange,pointSizeGranularity=pointSizeGranularity,lineWidthGranularity=lineWidthGranularity,strictLines=strictLines,standardSampleLocations=standardSampleLocations,optimalBufferCopyOffsetAlignment=optimalBufferCopyOffsetAlignment,optimalBufferCopyRowPitchAlignment=optimalBufferCopyRowPitchAlignment,nonCoherentAtomSize=nonCoherentAtomSize)

def VkSemaphoreCreateInfo(sType=VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO,pNext=None,flags=None,):

    return _new('VkSemaphoreCreateInfo', sType=sType,pNext=pNext,flags=flags)

def VkQueryPoolCreateInfo(sType=VK_STRUCTURE_TYPE_QUERY_POOL_CREATE_INFO,pNext=None,flags=None,queryType=None,queryCount=None,pipelineStatistics=None,):

    return _new('VkQueryPoolCreateInfo', sType=sType,pNext=pNext,flags=flags,queryType=queryType,queryCount=queryCount,pipelineStatistics=pipelineStatistics)

def VkFramebufferCreateInfo(sType=VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO,pNext=None,flags=None,renderPass=None,attachmentCount=None,pAttachments=None,width=None,height=None,layers=None,):
    if attachmentCount is None and pAttachments is not None:
        attachmentCount = len(pAttachments)

    return _new('VkFramebufferCreateInfo', sType=sType,pNext=pNext,flags=flags,renderPass=renderPass,attachmentCount=attachmentCount,pAttachments=pAttachments,width=width,height=height,layers=layers)

def VkDrawIndirectCommand(vertexCount=None,instanceCount=None,firstVertex=None,firstInstance=None,):

    return _new('VkDrawIndirectCommand', vertexCount=vertexCount,instanceCount=instanceCount,firstVertex=firstVertex,firstInstance=firstInstance)

def VkDrawIndexedIndirectCommand(indexCount=None,instanceCount=None,firstIndex=None,vertexOffset=None,firstInstance=None,):

    return _new('VkDrawIndexedIndirectCommand', indexCount=indexCount,instanceCount=instanceCount,firstIndex=firstIndex,vertexOffset=vertexOffset,firstInstance=firstInstance)

def VkDispatchIndirectCommand(x=None,y=None,z=None,):

    return _new('VkDispatchIndirectCommand', x=x,y=y,z=z)

def VkMultiDrawInfoEXT(firstVertex=None,vertexCount=None,):

    return _new('VkMultiDrawInfoEXT', firstVertex=firstVertex,vertexCount=vertexCount)

def VkMultiDrawIndexedInfoEXT(firstIndex=None,indexCount=None,vertexOffset=None,):

    return _new('VkMultiDrawIndexedInfoEXT', firstIndex=firstIndex,indexCount=indexCount,vertexOffset=vertexOffset)

def VkSubmitInfo(sType=VK_STRUCTURE_TYPE_SUBMIT_INFO,pNext=None,waitSemaphoreCount=None,pWaitSemaphores=None,pWaitDstStageMask=None,commandBufferCount=None,pCommandBuffers=None,signalSemaphoreCount=None,pSignalSemaphores=None,):
    if waitSemaphoreCount is None and pWaitSemaphores is not None:
        waitSemaphoreCount = len(pWaitSemaphores)
    if waitSemaphoreCount is None and pWaitDstStageMask is not None:
        waitSemaphoreCount = len(pWaitDstStageMask)
    if commandBufferCount is None and pCommandBuffers is not None:
        commandBufferCount = len(pCommandBuffers)
    if signalSemaphoreCount is None and pSignalSemaphores is not None:
        signalSemaphoreCount = len(pSignalSemaphores)

    return _new('VkSubmitInfo', sType=sType,pNext=pNext,waitSemaphoreCount=waitSemaphoreCount,pWaitSemaphores=pWaitSemaphores,pWaitDstStageMask=pWaitDstStageMask,commandBufferCount=commandBufferCount,pCommandBuffers=pCommandBuffers,signalSemaphoreCount=signalSemaphoreCount,pSignalSemaphores=pSignalSemaphores)

def VkDisplayPropertiesKHR(display=None,displayName=None,physicalDimensions=None,physicalResolution=None,supportedTransforms=None,planeReorderPossible=None,persistentContent=None,):

    return _new('VkDisplayPropertiesKHR', display=display,displayName=displayName,physicalDimensions=physicalDimensions,physicalResolution=physicalResolution,supportedTransforms=supportedTransforms,planeReorderPossible=planeReorderPossible,persistentContent=persistentContent)

def VkDisplayPlanePropertiesKHR(currentDisplay=None,currentStackIndex=None,):

    return _new('VkDisplayPlanePropertiesKHR', currentDisplay=currentDisplay,currentStackIndex=currentStackIndex)

def VkDisplayModeParametersKHR(visibleRegion=None,refreshRate=None,):

    return _new('VkDisplayModeParametersKHR', visibleRegion=visibleRegion,refreshRate=refreshRate)

def VkDisplayModePropertiesKHR(displayMode=None,parameters=None,):

    return _new('VkDisplayModePropertiesKHR', displayMode=displayMode,parameters=parameters)

def VkDisplayModeCreateInfoKHR(sType=VK_STRUCTURE_TYPE_DISPLAY_MODE_CREATE_INFO_KHR,pNext=None,flags=None,parameters=None,):

    return _new('VkDisplayModeCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,parameters=parameters)

def VkDisplayPlaneCapabilitiesKHR(supportedAlpha=None,minSrcPosition=None,maxSrcPosition=None,minSrcExtent=None,maxSrcExtent=None,minDstPosition=None,maxDstPosition=None,minDstExtent=None,maxDstExtent=None,):

    return _new('VkDisplayPlaneCapabilitiesKHR', supportedAlpha=supportedAlpha,minSrcPosition=minSrcPosition,maxSrcPosition=maxSrcPosition,minSrcExtent=minSrcExtent,maxSrcExtent=maxSrcExtent,minDstPosition=minDstPosition,maxDstPosition=maxDstPosition,minDstExtent=minDstExtent,maxDstExtent=maxDstExtent)

def VkDisplaySurfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_DISPLAY_SURFACE_CREATE_INFO_KHR,pNext=None,flags=None,displayMode=None,planeIndex=None,planeStackIndex=None,transform=None,globalAlpha=None,alphaMode=None,imageExtent=None,):

    return _new('VkDisplaySurfaceCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,displayMode=displayMode,planeIndex=planeIndex,planeStackIndex=planeStackIndex,transform=transform,globalAlpha=globalAlpha,alphaMode=alphaMode,imageExtent=imageExtent)

def VkDisplayPresentInfoKHR(sType=VK_STRUCTURE_TYPE_DISPLAY_PRESENT_INFO_KHR,pNext=None,srcRect=None,dstRect=None,persistent=None,):

    return _new('VkDisplayPresentInfoKHR', sType=sType,pNext=pNext,srcRect=srcRect,dstRect=dstRect,persistent=persistent)

def VkSurfaceCapabilitiesKHR(minImageCount=None,maxImageCount=None,currentExtent=None,minImageExtent=None,maxImageExtent=None,maxImageArrayLayers=None,supportedTransforms=None,currentTransform=None,supportedCompositeAlpha=None,supportedUsageFlags=None,):

    return _new('VkSurfaceCapabilitiesKHR', minImageCount=minImageCount,maxImageCount=maxImageCount,currentExtent=currentExtent,minImageExtent=minImageExtent,maxImageExtent=maxImageExtent,maxImageArrayLayers=maxImageArrayLayers,supportedTransforms=supportedTransforms,currentTransform=currentTransform,supportedCompositeAlpha=supportedCompositeAlpha,supportedUsageFlags=supportedUsageFlags)

def VkAndroidSurfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_ANDROID_SURFACE_CREATE_INFO_KHR,pNext=None,flags=None,window=None,):

    return _new('VkAndroidSurfaceCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,window=window)

def VkViSurfaceCreateInfoNN(sType=VK_STRUCTURE_TYPE_VI_SURFACE_CREATE_INFO_NN,pNext=None,flags=None,window=None,):

    return _new('VkViSurfaceCreateInfoNN', sType=sType,pNext=pNext,flags=flags,window=window)

def VkWaylandSurfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_WAYLAND_SURFACE_CREATE_INFO_KHR,pNext=None,flags=None,display=None,surface=None,):

    return _new('VkWaylandSurfaceCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,display=display,surface=surface)

def VkWin32SurfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_WIN32_SURFACE_CREATE_INFO_KHR,pNext=None,flags=None,hinstance=None,hwnd=None,):

    return _new('VkWin32SurfaceCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,hinstance=hinstance,hwnd=hwnd)

def VkXlibSurfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_XLIB_SURFACE_CREATE_INFO_KHR,pNext=None,flags=None,dpy=None,window=None,):

    return _new('VkXlibSurfaceCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,dpy=dpy,window=window)

def VkXcbSurfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_XCB_SURFACE_CREATE_INFO_KHR,pNext=None,flags=None,connection=None,window=None,):

    return _new('VkXcbSurfaceCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,connection=connection,window=window)

def VkDirectFBSurfaceCreateInfoEXT(sType=VK_STRUCTURE_TYPE_DIRECTFB_SURFACE_CREATE_INFO_EXT,pNext=None,flags=None,dfb=None,surface=None,):

    return _new('VkDirectFBSurfaceCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,dfb=dfb,surface=surface)

def VkImagePipeSurfaceCreateInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_IMAGEPIPE_SURFACE_CREATE_INFO_FUCHSIA,pNext=None,flags=None,imagePipeHandle=None,):

    return _new('VkImagePipeSurfaceCreateInfoFUCHSIA', sType=sType,pNext=pNext,flags=flags,imagePipeHandle=imagePipeHandle)

def VkStreamDescriptorSurfaceCreateInfoGGP(sType=VK_STRUCTURE_TYPE_STREAM_DESCRIPTOR_SURFACE_CREATE_INFO_GGP,pNext=None,flags=None,streamDescriptor=None,):

    return _new('VkStreamDescriptorSurfaceCreateInfoGGP', sType=sType,pNext=pNext,flags=flags,streamDescriptor=streamDescriptor)

def VkScreenSurfaceCreateInfoQNX(sType=VK_STRUCTURE_TYPE_SCREEN_SURFACE_CREATE_INFO_QNX,pNext=None,flags=None,context=None,window=None,):

    return _new('VkScreenSurfaceCreateInfoQNX', sType=sType,pNext=pNext,flags=flags,context=context,window=window)

def VkSurfaceFormatKHR(format=None,colorSpace=None,):

    return _new('VkSurfaceFormatKHR', format=format,colorSpace=colorSpace)

def VkSwapchainCreateInfoKHR(sType=VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR,pNext=None,flags=None,surface=None,minImageCount=None,imageFormat=None,imageColorSpace=None,imageExtent=None,imageArrayLayers=None,imageUsage=None,imageSharingMode=None,queueFamilyIndexCount=None,pQueueFamilyIndices=None,preTransform=None,compositeAlpha=None,presentMode=None,clipped=None,oldSwapchain=None,):
    if queueFamilyIndexCount is None and pQueueFamilyIndices is not None:
        queueFamilyIndexCount = len(pQueueFamilyIndices)

    return _new('VkSwapchainCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,surface=surface,minImageCount=minImageCount,imageFormat=imageFormat,imageColorSpace=imageColorSpace,imageExtent=imageExtent,imageArrayLayers=imageArrayLayers,imageUsage=imageUsage,imageSharingMode=imageSharingMode,queueFamilyIndexCount=queueFamilyIndexCount,pQueueFamilyIndices=pQueueFamilyIndices,preTransform=preTransform,compositeAlpha=compositeAlpha,presentMode=presentMode,clipped=clipped,oldSwapchain=oldSwapchain)

def VkPresentInfoKHR(sType=VK_STRUCTURE_TYPE_PRESENT_INFO_KHR,pNext=None,waitSemaphoreCount=None,pWaitSemaphores=None,swapchainCount=None,pSwapchains=None,pImageIndices=None,pResults=None,):
    if waitSemaphoreCount is None and pWaitSemaphores is not None:
        waitSemaphoreCount = len(pWaitSemaphores)
    if swapchainCount is None and pSwapchains is not None:
        swapchainCount = len(pSwapchains)
    if swapchainCount is None and pImageIndices is not None:
        swapchainCount = len(pImageIndices)
    if swapchainCount is None and pResults is not None:
        swapchainCount = len(pResults)

    return _new('VkPresentInfoKHR', sType=sType,pNext=pNext,waitSemaphoreCount=waitSemaphoreCount,pWaitSemaphores=pWaitSemaphores,swapchainCount=swapchainCount,pSwapchains=pSwapchains,pImageIndices=pImageIndices,pResults=pResults)

def VkDebugReportCallbackCreateInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_REPORT_CALLBACK_CREATE_INFO_EXT,pNext=None,flags=None,pfnCallback=None,pUserData=None,):

    return _new('VkDebugReportCallbackCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,pfnCallback=pfnCallback,pUserData=pUserData)

def VkValidationFlagsEXT(sType=VK_STRUCTURE_TYPE_VALIDATION_FLAGS_EXT,pNext=None,disabledValidationCheckCount=None,pDisabledValidationChecks=None,):
    if disabledValidationCheckCount is None and pDisabledValidationChecks is not None:
        disabledValidationCheckCount = len(pDisabledValidationChecks)

    return _new('VkValidationFlagsEXT', sType=sType,pNext=pNext,disabledValidationCheckCount=disabledValidationCheckCount,pDisabledValidationChecks=pDisabledValidationChecks)

def VkValidationFeaturesEXT(sType=VK_STRUCTURE_TYPE_VALIDATION_FEATURES_EXT,pNext=None,enabledValidationFeatureCount=None,pEnabledValidationFeatures=None,disabledValidationFeatureCount=None,pDisabledValidationFeatures=None,):
    if enabledValidationFeatureCount is None and pEnabledValidationFeatures is not None:
        enabledValidationFeatureCount = len(pEnabledValidationFeatures)
    if disabledValidationFeatureCount is None and pDisabledValidationFeatures is not None:
        disabledValidationFeatureCount = len(pDisabledValidationFeatures)

    return _new('VkValidationFeaturesEXT', sType=sType,pNext=pNext,enabledValidationFeatureCount=enabledValidationFeatureCount,pEnabledValidationFeatures=pEnabledValidationFeatures,disabledValidationFeatureCount=disabledValidationFeatureCount,pDisabledValidationFeatures=pDisabledValidationFeatures)

def VkLayerSettingsCreateInfoEXT(sType=VK_STRUCTURE_TYPE_LAYER_SETTINGS_CREATE_INFO_EXT,pNext=None,settingCount=None,pSettings=None,):
    if settingCount is None and pSettings is not None:
        settingCount = len(pSettings)

    return _new('VkLayerSettingsCreateInfoEXT', sType=sType,pNext=pNext,settingCount=settingCount,pSettings=pSettings)

def VkLayerSettingEXT(pLayerName=None,pSettingName=None,type=None,valueCount=None,pValues=None,):
    if valueCount is None and pValues is not None:
        valueCount = len(pValues)

    return _new('VkLayerSettingEXT', pLayerName=pLayerName,pSettingName=pSettingName,type=type,valueCount=valueCount,pValues=pValues)

def VkApplicationParametersEXT(sType=VK_STRUCTURE_TYPE_APPLICATION_PARAMETERS_EXT,pNext=None,vendorID=None,deviceID=None,key=None,value=None,):

    return _new('VkApplicationParametersEXT', sType=sType,pNext=pNext,vendorID=vendorID,deviceID=deviceID,key=key,value=value)

def VkPipelineRasterizationStateRasterizationOrderAMD(sType=VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_RASTERIZATION_ORDER_AMD,pNext=None,rasterizationOrder=None,):

    return _new('VkPipelineRasterizationStateRasterizationOrderAMD', sType=sType,pNext=pNext,rasterizationOrder=rasterizationOrder)

def VkDebugMarkerObjectNameInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_NAME_INFO_EXT,pNext=None,objectType=None,object=None,pObjectName=None,):

    return _new('VkDebugMarkerObjectNameInfoEXT', sType=sType,pNext=pNext,objectType=objectType,object=object,pObjectName=pObjectName)

def VkDebugMarkerObjectTagInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_TAG_INFO_EXT,pNext=None,objectType=None,object=None,tagName=None,tagSize=None,pTag=None,):
    if tagSize is None and pTag is not None:
        tagSize = len(pTag)

    return _new('VkDebugMarkerObjectTagInfoEXT', sType=sType,pNext=pNext,objectType=objectType,object=object,tagName=tagName,tagSize=tagSize,pTag=pTag)

def VkDebugMarkerMarkerInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_MARKER_MARKER_INFO_EXT,pNext=None,pMarkerName=None,color=None,):

    return _new('VkDebugMarkerMarkerInfoEXT', sType=sType,pNext=pNext,pMarkerName=pMarkerName,color=color)

def VkDedicatedAllocationImageCreateInfoNV(sType=VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_IMAGE_CREATE_INFO_NV,pNext=None,dedicatedAllocation=None,):

    return _new('VkDedicatedAllocationImageCreateInfoNV', sType=sType,pNext=pNext,dedicatedAllocation=dedicatedAllocation)

def VkDedicatedAllocationBufferCreateInfoNV(sType=VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_BUFFER_CREATE_INFO_NV,pNext=None,dedicatedAllocation=None,):

    return _new('VkDedicatedAllocationBufferCreateInfoNV', sType=sType,pNext=pNext,dedicatedAllocation=dedicatedAllocation)

def VkDedicatedAllocationMemoryAllocateInfoNV(sType=VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_MEMORY_ALLOCATE_INFO_NV,pNext=None,image=None,buffer=None,):

    return _new('VkDedicatedAllocationMemoryAllocateInfoNV', sType=sType,pNext=pNext,image=image,buffer=buffer)

def VkExternalImageFormatPropertiesNV(imageFormatProperties=None,externalMemoryFeatures=None,exportFromImportedHandleTypes=None,compatibleHandleTypes=None,):

    return _new('VkExternalImageFormatPropertiesNV', imageFormatProperties=imageFormatProperties,externalMemoryFeatures=externalMemoryFeatures,exportFromImportedHandleTypes=exportFromImportedHandleTypes,compatibleHandleTypes=compatibleHandleTypes)

def VkExternalMemoryImageCreateInfoNV(sType=VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO_NV,pNext=None,handleTypes=None,):

    return _new('VkExternalMemoryImageCreateInfoNV', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkExportMemoryAllocateInfoNV(sType=VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO_NV,pNext=None,handleTypes=None,):

    return _new('VkExportMemoryAllocateInfoNV', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkImportMemoryWin32HandleInfoNV(sType=VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_NV,pNext=None,handleType=None,handle=None,):

    return _new('VkImportMemoryWin32HandleInfoNV', sType=sType,pNext=pNext,handleType=handleType,handle=handle)

def VkExportMemoryWin32HandleInfoNV(sType=VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_NV,pNext=None,pAttributes=None,dwAccess=None,):

    return _new('VkExportMemoryWin32HandleInfoNV', sType=sType,pNext=pNext,pAttributes=pAttributes,dwAccess=dwAccess)

def VkExportMemorySciBufInfoNV(sType=VK_STRUCTURE_TYPE_EXPORT_MEMORY_SCI_BUF_INFO_NV,pNext=None,pAttributes=None,):

    return _new('VkExportMemorySciBufInfoNV', sType=sType,pNext=pNext,pAttributes=pAttributes)

def VkImportMemorySciBufInfoNV(sType=VK_STRUCTURE_TYPE_IMPORT_MEMORY_SCI_BUF_INFO_NV,pNext=None,handleType=None,handle=None,):

    return _new('VkImportMemorySciBufInfoNV', sType=sType,pNext=pNext,handleType=handleType,handle=handle)

def VkMemoryGetSciBufInfoNV(sType=VK_STRUCTURE_TYPE_MEMORY_GET_SCI_BUF_INFO_NV,pNext=None,memory=None,handleType=None,):

    return _new('VkMemoryGetSciBufInfoNV', sType=sType,pNext=pNext,memory=memory,handleType=handleType)

def VkMemorySciBufPropertiesNV(sType=VK_STRUCTURE_TYPE_MEMORY_SCI_BUF_PROPERTIES_NV,pNext=None,memoryTypeBits=None,):

    return _new('VkMemorySciBufPropertiesNV', sType=sType,pNext=pNext,memoryTypeBits=memoryTypeBits)

def VkPhysicalDeviceExternalMemorySciBufFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_SCI_BUF_FEATURES_NV,pNext=None,sciBufImport=None,sciBufExport=None,):

    return _new('VkPhysicalDeviceExternalMemorySciBufFeaturesNV', sType=sType,pNext=pNext,sciBufImport=sciBufImport,sciBufExport=sciBufExport)

def VkWin32KeyedMutexAcquireReleaseInfoNV(sType=VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_NV,pNext=None,acquireCount=None,pAcquireSyncs=None,pAcquireKeys=None,pAcquireTimeoutMilliseconds=None,releaseCount=None,pReleaseSyncs=None,pReleaseKeys=None,):
    if acquireCount is None and pAcquireSyncs is not None:
        acquireCount = len(pAcquireSyncs)
    if acquireCount is None and pAcquireKeys is not None:
        acquireCount = len(pAcquireKeys)
    if acquireCount is None and pAcquireTimeoutMilliseconds is not None:
        acquireCount = len(pAcquireTimeoutMilliseconds)
    if releaseCount is None and pReleaseSyncs is not None:
        releaseCount = len(pReleaseSyncs)
    if releaseCount is None and pReleaseKeys is not None:
        releaseCount = len(pReleaseKeys)

    return _new('VkWin32KeyedMutexAcquireReleaseInfoNV', sType=sType,pNext=pNext,acquireCount=acquireCount,pAcquireSyncs=pAcquireSyncs,pAcquireKeys=pAcquireKeys,pAcquireTimeoutMilliseconds=pAcquireTimeoutMilliseconds,releaseCount=releaseCount,pReleaseSyncs=pReleaseSyncs,pReleaseKeys=pReleaseKeys)

def VkPhysicalDeviceDeviceGeneratedCommandsFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEVICE_GENERATED_COMMANDS_FEATURES_NV,pNext=None,deviceGeneratedCommands=None,):

    return _new('VkPhysicalDeviceDeviceGeneratedCommandsFeaturesNV', sType=sType,pNext=pNext,deviceGeneratedCommands=deviceGeneratedCommands)

def VkPhysicalDeviceDeviceGeneratedCommandsComputeFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEVICE_GENERATED_COMMANDS_COMPUTE_FEATURES_NV,pNext=None,deviceGeneratedCompute=None,deviceGeneratedComputePipelines=None,deviceGeneratedComputeCaptureReplay=None,):

    return _new('VkPhysicalDeviceDeviceGeneratedCommandsComputeFeaturesNV', sType=sType,pNext=pNext,deviceGeneratedCompute=deviceGeneratedCompute,deviceGeneratedComputePipelines=deviceGeneratedComputePipelines,deviceGeneratedComputeCaptureReplay=deviceGeneratedComputeCaptureReplay)

def VkDevicePrivateDataCreateInfo(sType=VK_STRUCTURE_TYPE_DEVICE_PRIVATE_DATA_CREATE_INFO,pNext=None,privateDataSlotRequestCount=None,):

    return _new('VkDevicePrivateDataCreateInfo', sType=sType,pNext=pNext,privateDataSlotRequestCount=privateDataSlotRequestCount)

def VkPrivateDataSlotCreateInfo(sType=VK_STRUCTURE_TYPE_PRIVATE_DATA_SLOT_CREATE_INFO,pNext=None,flags=None,):

    return _new('VkPrivateDataSlotCreateInfo', sType=sType,pNext=pNext,flags=flags)

def VkPhysicalDevicePrivateDataFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRIVATE_DATA_FEATURES,pNext=None,privateData=None,):

    return _new('VkPhysicalDevicePrivateDataFeatures', sType=sType,pNext=pNext,privateData=privateData)

def VkPhysicalDeviceDeviceGeneratedCommandsPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEVICE_GENERATED_COMMANDS_PROPERTIES_NV,pNext=None,maxGraphicsShaderGroupCount=None,maxIndirectSequenceCount=None,maxIndirectCommandsTokenCount=None,maxIndirectCommandsStreamCount=None,maxIndirectCommandsTokenOffset=None,maxIndirectCommandsStreamStride=None,minSequencesCountBufferOffsetAlignment=None,minSequencesIndexBufferOffsetAlignment=None,minIndirectCommandsBufferOffsetAlignment=None,):

    return _new('VkPhysicalDeviceDeviceGeneratedCommandsPropertiesNV', sType=sType,pNext=pNext,maxGraphicsShaderGroupCount=maxGraphicsShaderGroupCount,maxIndirectSequenceCount=maxIndirectSequenceCount,maxIndirectCommandsTokenCount=maxIndirectCommandsTokenCount,maxIndirectCommandsStreamCount=maxIndirectCommandsStreamCount,maxIndirectCommandsTokenOffset=maxIndirectCommandsTokenOffset,maxIndirectCommandsStreamStride=maxIndirectCommandsStreamStride,minSequencesCountBufferOffsetAlignment=minSequencesCountBufferOffsetAlignment,minSequencesIndexBufferOffsetAlignment=minSequencesIndexBufferOffsetAlignment,minIndirectCommandsBufferOffsetAlignment=minIndirectCommandsBufferOffsetAlignment)

def VkPhysicalDeviceMultiDrawPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTI_DRAW_PROPERTIES_EXT,pNext=None,maxMultiDrawCount=None,):

    return _new('VkPhysicalDeviceMultiDrawPropertiesEXT', sType=sType,pNext=pNext,maxMultiDrawCount=maxMultiDrawCount)

def VkGraphicsShaderGroupCreateInfoNV(sType=VK_STRUCTURE_TYPE_GRAPHICS_SHADER_GROUP_CREATE_INFO_NV,pNext=None,stageCount=None,pStages=None,pVertexInputState=None,pTessellationState=None,):
    if stageCount is None and pStages is not None:
        stageCount = len(pStages)

    return _new('VkGraphicsShaderGroupCreateInfoNV', sType=sType,pNext=pNext,stageCount=stageCount,pStages=pStages,pVertexInputState=pVertexInputState,pTessellationState=pTessellationState)

def VkGraphicsPipelineShaderGroupsCreateInfoNV(sType=VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_SHADER_GROUPS_CREATE_INFO_NV,pNext=None,groupCount=None,pGroups=None,pipelineCount=None,pPipelines=None,):
    if groupCount is None and pGroups is not None:
        groupCount = len(pGroups)
    if pipelineCount is None and pPipelines is not None:
        pipelineCount = len(pPipelines)

    return _new('VkGraphicsPipelineShaderGroupsCreateInfoNV', sType=sType,pNext=pNext,groupCount=groupCount,pGroups=pGroups,pipelineCount=pipelineCount,pPipelines=pPipelines)

def VkBindShaderGroupIndirectCommandNV(groupIndex=None,):

    return _new('VkBindShaderGroupIndirectCommandNV', groupIndex=groupIndex)

def VkBindIndexBufferIndirectCommandNV(bufferAddress=None,size=None,indexType=None,):

    return _new('VkBindIndexBufferIndirectCommandNV', bufferAddress=bufferAddress,size=size,indexType=indexType)

def VkBindVertexBufferIndirectCommandNV(bufferAddress=None,size=None,stride=None,):

    return _new('VkBindVertexBufferIndirectCommandNV', bufferAddress=bufferAddress,size=size,stride=stride)

def VkSetStateFlagsIndirectCommandNV(data=None,):

    return _new('VkSetStateFlagsIndirectCommandNV', data=data)

def VkIndirectCommandsStreamNV(buffer=None,offset=None,):

    return _new('VkIndirectCommandsStreamNV', buffer=buffer,offset=offset)

def VkIndirectCommandsLayoutTokenNV(sType=VK_STRUCTURE_TYPE_INDIRECT_COMMANDS_LAYOUT_TOKEN_NV,pNext=None,tokenType=None,stream=None,offset=None,vertexBindingUnit=None,vertexDynamicStride=None,pushconstantPipelineLayout=None,pushconstantShaderStageFlags=None,pushconstantOffset=None,pushconstantSize=None,indirectStateFlags=None,indexTypeCount=None,pIndexTypes=None,pIndexTypeValues=None,):
    if indexTypeCount is None and pIndexTypes is not None:
        indexTypeCount = len(pIndexTypes)
    if indexTypeCount is None and pIndexTypeValues is not None:
        indexTypeCount = len(pIndexTypeValues)

    return _new('VkIndirectCommandsLayoutTokenNV', sType=sType,pNext=pNext,tokenType=tokenType,stream=stream,offset=offset,vertexBindingUnit=vertexBindingUnit,vertexDynamicStride=vertexDynamicStride,pushconstantPipelineLayout=pushconstantPipelineLayout,pushconstantShaderStageFlags=pushconstantShaderStageFlags,pushconstantOffset=pushconstantOffset,pushconstantSize=pushconstantSize,indirectStateFlags=indirectStateFlags,indexTypeCount=indexTypeCount,pIndexTypes=pIndexTypes,pIndexTypeValues=pIndexTypeValues)

def VkIndirectCommandsLayoutCreateInfoNV(sType=VK_STRUCTURE_TYPE_INDIRECT_COMMANDS_LAYOUT_CREATE_INFO_NV,pNext=None,flags=None,pipelineBindPoint=None,tokenCount=None,pTokens=None,streamCount=None,pStreamStrides=None,):
    if tokenCount is None and pTokens is not None:
        tokenCount = len(pTokens)
    if streamCount is None and pStreamStrides is not None:
        streamCount = len(pStreamStrides)

    return _new('VkIndirectCommandsLayoutCreateInfoNV', sType=sType,pNext=pNext,flags=flags,pipelineBindPoint=pipelineBindPoint,tokenCount=tokenCount,pTokens=pTokens,streamCount=streamCount,pStreamStrides=pStreamStrides)

def VkGeneratedCommandsInfoNV(sType=VK_STRUCTURE_TYPE_GENERATED_COMMANDS_INFO_NV,pNext=None,pipelineBindPoint=None,pipeline=None,indirectCommandsLayout=None,streamCount=None,pStreams=None,sequencesCount=None,preprocessBuffer=None,preprocessOffset=None,preprocessSize=None,sequencesCountBuffer=None,sequencesCountOffset=None,sequencesIndexBuffer=None,sequencesIndexOffset=None,):
    if streamCount is None and pStreams is not None:
        streamCount = len(pStreams)

    return _new('VkGeneratedCommandsInfoNV', sType=sType,pNext=pNext,pipelineBindPoint=pipelineBindPoint,pipeline=pipeline,indirectCommandsLayout=indirectCommandsLayout,streamCount=streamCount,pStreams=pStreams,sequencesCount=sequencesCount,preprocessBuffer=preprocessBuffer,preprocessOffset=preprocessOffset,preprocessSize=preprocessSize,sequencesCountBuffer=sequencesCountBuffer,sequencesCountOffset=sequencesCountOffset,sequencesIndexBuffer=sequencesIndexBuffer,sequencesIndexOffset=sequencesIndexOffset)

def VkGeneratedCommandsMemoryRequirementsInfoNV(sType=VK_STRUCTURE_TYPE_GENERATED_COMMANDS_MEMORY_REQUIREMENTS_INFO_NV,pNext=None,pipelineBindPoint=None,pipeline=None,indirectCommandsLayout=None,maxSequencesCount=None,):

    return _new('VkGeneratedCommandsMemoryRequirementsInfoNV', sType=sType,pNext=pNext,pipelineBindPoint=pipelineBindPoint,pipeline=pipeline,indirectCommandsLayout=indirectCommandsLayout,maxSequencesCount=maxSequencesCount)

def VkPipelineIndirectDeviceAddressInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_INDIRECT_DEVICE_ADDRESS_INFO_NV,pNext=None,pipelineBindPoint=None,pipeline=None,):

    return _new('VkPipelineIndirectDeviceAddressInfoNV', sType=sType,pNext=pNext,pipelineBindPoint=pipelineBindPoint,pipeline=pipeline)

def VkBindPipelineIndirectCommandNV(pipelineAddress=None,):

    return _new('VkBindPipelineIndirectCommandNV', pipelineAddress=pipelineAddress)

def VkPhysicalDeviceFeatures2(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FEATURES_2,pNext=None,features=None,):

    return _new('VkPhysicalDeviceFeatures2', sType=sType,pNext=pNext,features=features)

def VkPhysicalDeviceProperties2(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROPERTIES_2,pNext=None,properties=None,):

    return _new('VkPhysicalDeviceProperties2', sType=sType,pNext=pNext,properties=properties)

def VkFormatProperties2(sType=VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_2,pNext=None,formatProperties=None,):

    return _new('VkFormatProperties2', sType=sType,pNext=pNext,formatProperties=formatProperties)

def VkImageFormatProperties2(sType=VK_STRUCTURE_TYPE_IMAGE_FORMAT_PROPERTIES_2,pNext=None,imageFormatProperties=None,):

    return _new('VkImageFormatProperties2', sType=sType,pNext=pNext,imageFormatProperties=imageFormatProperties)

def VkPhysicalDeviceImageFormatInfo2(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_FORMAT_INFO_2,pNext=None,format=None,type=None,tiling=None,usage=None,flags=None,):

    return _new('VkPhysicalDeviceImageFormatInfo2', sType=sType,pNext=pNext,format=format,type=type,tiling=tiling,usage=usage,flags=flags)

def VkQueueFamilyProperties2(sType=VK_STRUCTURE_TYPE_QUEUE_FAMILY_PROPERTIES_2,pNext=None,queueFamilyProperties=None,):

    return _new('VkQueueFamilyProperties2', sType=sType,pNext=pNext,queueFamilyProperties=queueFamilyProperties)

def VkPhysicalDeviceMemoryProperties2(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PROPERTIES_2,pNext=None,memoryProperties=None,):

    return _new('VkPhysicalDeviceMemoryProperties2', sType=sType,pNext=pNext,memoryProperties=memoryProperties)

def VkSparseImageFormatProperties2(sType=VK_STRUCTURE_TYPE_SPARSE_IMAGE_FORMAT_PROPERTIES_2,pNext=None,properties=None,):

    return _new('VkSparseImageFormatProperties2', sType=sType,pNext=pNext,properties=properties)

def VkPhysicalDeviceSparseImageFormatInfo2(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SPARSE_IMAGE_FORMAT_INFO_2,pNext=None,format=None,type=None,samples=None,usage=None,tiling=None,):

    return _new('VkPhysicalDeviceSparseImageFormatInfo2', sType=sType,pNext=pNext,format=format,type=type,samples=samples,usage=usage,tiling=tiling)

def VkPhysicalDevicePushDescriptorPropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PUSH_DESCRIPTOR_PROPERTIES_KHR,pNext=None,maxPushDescriptors=None,):

    return _new('VkPhysicalDevicePushDescriptorPropertiesKHR', sType=sType,pNext=pNext,maxPushDescriptors=maxPushDescriptors)

def VkConformanceVersion(major=None,minor=None,subminor=None,patch=None,):

    return _new('VkConformanceVersion', major=major,minor=minor,subminor=subminor,patch=patch)

def VkPhysicalDeviceDriverProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DRIVER_PROPERTIES,pNext=None,driverID=None,driverName=None,driverInfo=None,conformanceVersion=None,):

    return _new('VkPhysicalDeviceDriverProperties', sType=sType,pNext=pNext,driverID=driverID,driverName=driverName,driverInfo=driverInfo,conformanceVersion=conformanceVersion)

def VkPresentRegionsKHR(sType=VK_STRUCTURE_TYPE_PRESENT_REGIONS_KHR,pNext=None,swapchainCount=None,pRegions=None,):
    if swapchainCount is None and pRegions is not None:
        swapchainCount = len(pRegions)

    return _new('VkPresentRegionsKHR', sType=sType,pNext=pNext,swapchainCount=swapchainCount,pRegions=pRegions)

def VkPresentRegionKHR(rectangleCount=None,pRectangles=None,):
    if rectangleCount is None and pRectangles is not None:
        rectangleCount = len(pRectangles)

    return _new('VkPresentRegionKHR', rectangleCount=rectangleCount,pRectangles=pRectangles)

def VkRectLayerKHR(offset=None,extent=None,layer=None,):

    return _new('VkRectLayerKHR', offset=offset,extent=extent,layer=layer)

def VkPhysicalDeviceVariablePointersFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VARIABLE_POINTERS_FEATURES,pNext=None,variablePointersStorageBuffer=None,variablePointers=None,):

    return _new('VkPhysicalDeviceVariablePointersFeatures', sType=sType,pNext=pNext,variablePointersStorageBuffer=variablePointersStorageBuffer,variablePointers=variablePointers)

def VkExternalMemoryProperties(externalMemoryFeatures=None,exportFromImportedHandleTypes=None,compatibleHandleTypes=None,):

    return _new('VkExternalMemoryProperties', externalMemoryFeatures=externalMemoryFeatures,exportFromImportedHandleTypes=exportFromImportedHandleTypes,compatibleHandleTypes=compatibleHandleTypes)

def VkPhysicalDeviceExternalImageFormatInfo(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_IMAGE_FORMAT_INFO,pNext=None,handleType=None,):

    return _new('VkPhysicalDeviceExternalImageFormatInfo', sType=sType,pNext=pNext,handleType=handleType)

def VkExternalImageFormatProperties(sType=VK_STRUCTURE_TYPE_EXTERNAL_IMAGE_FORMAT_PROPERTIES,pNext=None,externalMemoryProperties=None,):

    return _new('VkExternalImageFormatProperties', sType=sType,pNext=pNext,externalMemoryProperties=externalMemoryProperties)

def VkPhysicalDeviceExternalBufferInfo(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_BUFFER_INFO,pNext=None,flags=None,usage=None,handleType=None,):

    return _new('VkPhysicalDeviceExternalBufferInfo', sType=sType,pNext=pNext,flags=flags,usage=usage,handleType=handleType)

def VkExternalBufferProperties(sType=VK_STRUCTURE_TYPE_EXTERNAL_BUFFER_PROPERTIES,pNext=None,externalMemoryProperties=None,):

    return _new('VkExternalBufferProperties', sType=sType,pNext=pNext,externalMemoryProperties=externalMemoryProperties)

def VkPhysicalDeviceIDProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ID_PROPERTIES,pNext=None,deviceUUID=None,driverUUID=None,deviceLUID=None,deviceNodeMask=None,deviceLUIDValid=None,):

    return _new('VkPhysicalDeviceIDProperties', sType=sType,pNext=pNext,deviceUUID=deviceUUID,driverUUID=driverUUID,deviceLUID=deviceLUID,deviceNodeMask=deviceNodeMask,deviceLUIDValid=deviceLUIDValid)

def VkExternalMemoryImageCreateInfo(sType=VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO,pNext=None,handleTypes=None,):

    return _new('VkExternalMemoryImageCreateInfo', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkExternalMemoryBufferCreateInfo(sType=VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_BUFFER_CREATE_INFO,pNext=None,handleTypes=None,):

    return _new('VkExternalMemoryBufferCreateInfo', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkExportMemoryAllocateInfo(sType=VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO,pNext=None,handleTypes=None,):

    return _new('VkExportMemoryAllocateInfo', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkImportMemoryWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_KHR,pNext=None,handleType=None,handle=None,name=None,):

    return _new('VkImportMemoryWin32HandleInfoKHR', sType=sType,pNext=pNext,handleType=handleType,handle=handle,name=name)

def VkExportMemoryWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_KHR,pNext=None,pAttributes=None,dwAccess=None,name=None,):

    return _new('VkExportMemoryWin32HandleInfoKHR', sType=sType,pNext=pNext,pAttributes=pAttributes,dwAccess=dwAccess,name=name)

def VkImportMemoryZirconHandleInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_IMPORT_MEMORY_ZIRCON_HANDLE_INFO_FUCHSIA,pNext=None,handleType=None,handle=None,):

    return _new('VkImportMemoryZirconHandleInfoFUCHSIA', sType=sType,pNext=pNext,handleType=handleType,handle=handle)

def VkMemoryZirconHandlePropertiesFUCHSIA(sType=VK_STRUCTURE_TYPE_MEMORY_ZIRCON_HANDLE_PROPERTIES_FUCHSIA,pNext=None,memoryTypeBits=None,):

    return _new('VkMemoryZirconHandlePropertiesFUCHSIA', sType=sType,pNext=pNext,memoryTypeBits=memoryTypeBits)

def VkMemoryGetZirconHandleInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_MEMORY_GET_ZIRCON_HANDLE_INFO_FUCHSIA,pNext=None,memory=None,handleType=None,):

    return _new('VkMemoryGetZirconHandleInfoFUCHSIA', sType=sType,pNext=pNext,memory=memory,handleType=handleType)

def VkMemoryWin32HandlePropertiesKHR(sType=VK_STRUCTURE_TYPE_MEMORY_WIN32_HANDLE_PROPERTIES_KHR,pNext=None,memoryTypeBits=None,):

    return _new('VkMemoryWin32HandlePropertiesKHR', sType=sType,pNext=pNext,memoryTypeBits=memoryTypeBits)

def VkMemoryGetWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_MEMORY_GET_WIN32_HANDLE_INFO_KHR,pNext=None,memory=None,handleType=None,):

    return _new('VkMemoryGetWin32HandleInfoKHR', sType=sType,pNext=pNext,memory=memory,handleType=handleType)

def VkImportMemoryFdInfoKHR(sType=VK_STRUCTURE_TYPE_IMPORT_MEMORY_FD_INFO_KHR,pNext=None,handleType=None,fd=None,):

    return _new('VkImportMemoryFdInfoKHR', sType=sType,pNext=pNext,handleType=handleType,fd=fd)

def VkMemoryFdPropertiesKHR(sType=VK_STRUCTURE_TYPE_MEMORY_FD_PROPERTIES_KHR,pNext=None,memoryTypeBits=None,):

    return _new('VkMemoryFdPropertiesKHR', sType=sType,pNext=pNext,memoryTypeBits=memoryTypeBits)

def VkMemoryGetFdInfoKHR(sType=VK_STRUCTURE_TYPE_MEMORY_GET_FD_INFO_KHR,pNext=None,memory=None,handleType=None,):

    return _new('VkMemoryGetFdInfoKHR', sType=sType,pNext=pNext,memory=memory,handleType=handleType)

def VkWin32KeyedMutexAcquireReleaseInfoKHR(sType=VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_KHR,pNext=None,acquireCount=None,pAcquireSyncs=None,pAcquireKeys=None,pAcquireTimeouts=None,releaseCount=None,pReleaseSyncs=None,pReleaseKeys=None,):
    if acquireCount is None and pAcquireSyncs is not None:
        acquireCount = len(pAcquireSyncs)
    if acquireCount is None and pAcquireKeys is not None:
        acquireCount = len(pAcquireKeys)
    if acquireCount is None and pAcquireTimeouts is not None:
        acquireCount = len(pAcquireTimeouts)
    if releaseCount is None and pReleaseSyncs is not None:
        releaseCount = len(pReleaseSyncs)
    if releaseCount is None and pReleaseKeys is not None:
        releaseCount = len(pReleaseKeys)

    return _new('VkWin32KeyedMutexAcquireReleaseInfoKHR', sType=sType,pNext=pNext,acquireCount=acquireCount,pAcquireSyncs=pAcquireSyncs,pAcquireKeys=pAcquireKeys,pAcquireTimeouts=pAcquireTimeouts,releaseCount=releaseCount,pReleaseSyncs=pReleaseSyncs,pReleaseKeys=pReleaseKeys)

def VkPhysicalDeviceExternalSemaphoreInfo(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SEMAPHORE_INFO,pNext=None,handleType=None,):

    return _new('VkPhysicalDeviceExternalSemaphoreInfo', sType=sType,pNext=pNext,handleType=handleType)

def VkExternalSemaphoreProperties(sType=VK_STRUCTURE_TYPE_EXTERNAL_SEMAPHORE_PROPERTIES,pNext=None,exportFromImportedHandleTypes=None,compatibleHandleTypes=None,externalSemaphoreFeatures=None,):

    return _new('VkExternalSemaphoreProperties', sType=sType,pNext=pNext,exportFromImportedHandleTypes=exportFromImportedHandleTypes,compatibleHandleTypes=compatibleHandleTypes,externalSemaphoreFeatures=externalSemaphoreFeatures)

def VkExportSemaphoreCreateInfo(sType=VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_CREATE_INFO,pNext=None,handleTypes=None,):

    return _new('VkExportSemaphoreCreateInfo', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkImportSemaphoreWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR,pNext=None,semaphore=None,flags=None,handleType=None,handle=None,name=None,):

    return _new('VkImportSemaphoreWin32HandleInfoKHR', sType=sType,pNext=pNext,semaphore=semaphore,flags=flags,handleType=handleType,handle=handle,name=name)

def VkExportSemaphoreWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR,pNext=None,pAttributes=None,dwAccess=None,name=None,):

    return _new('VkExportSemaphoreWin32HandleInfoKHR', sType=sType,pNext=pNext,pAttributes=pAttributes,dwAccess=dwAccess,name=name)

def VkD3D12FenceSubmitInfoKHR(sType=VK_STRUCTURE_TYPE_D3D12_FENCE_SUBMIT_INFO_KHR,pNext=None,waitSemaphoreValuesCount=None,pWaitSemaphoreValues=None,signalSemaphoreValuesCount=None,pSignalSemaphoreValues=None,):
    if waitSemaphoreValuesCount is None and pWaitSemaphoreValues is not None:
        waitSemaphoreValuesCount = len(pWaitSemaphoreValues)
    if signalSemaphoreValuesCount is None and pSignalSemaphoreValues is not None:
        signalSemaphoreValuesCount = len(pSignalSemaphoreValues)

    return _new('VkD3D12FenceSubmitInfoKHR', sType=sType,pNext=pNext,waitSemaphoreValuesCount=waitSemaphoreValuesCount,pWaitSemaphoreValues=pWaitSemaphoreValues,signalSemaphoreValuesCount=signalSemaphoreValuesCount,pSignalSemaphoreValues=pSignalSemaphoreValues)

def VkSemaphoreGetWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_SEMAPHORE_GET_WIN32_HANDLE_INFO_KHR,pNext=None,semaphore=None,handleType=None,):

    return _new('VkSemaphoreGetWin32HandleInfoKHR', sType=sType,pNext=pNext,semaphore=semaphore,handleType=handleType)

def VkImportSemaphoreFdInfoKHR(sType=VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_FD_INFO_KHR,pNext=None,semaphore=None,flags=None,handleType=None,fd=None,):

    return _new('VkImportSemaphoreFdInfoKHR', sType=sType,pNext=pNext,semaphore=semaphore,flags=flags,handleType=handleType,fd=fd)

def VkSemaphoreGetFdInfoKHR(sType=VK_STRUCTURE_TYPE_SEMAPHORE_GET_FD_INFO_KHR,pNext=None,semaphore=None,handleType=None,):

    return _new('VkSemaphoreGetFdInfoKHR', sType=sType,pNext=pNext,semaphore=semaphore,handleType=handleType)

def VkImportSemaphoreZirconHandleInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_ZIRCON_HANDLE_INFO_FUCHSIA,pNext=None,semaphore=None,flags=None,handleType=None,zirconHandle=None,):

    return _new('VkImportSemaphoreZirconHandleInfoFUCHSIA', sType=sType,pNext=pNext,semaphore=semaphore,flags=flags,handleType=handleType,zirconHandle=zirconHandle)

def VkSemaphoreGetZirconHandleInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_SEMAPHORE_GET_ZIRCON_HANDLE_INFO_FUCHSIA,pNext=None,semaphore=None,handleType=None,):

    return _new('VkSemaphoreGetZirconHandleInfoFUCHSIA', sType=sType,pNext=pNext,semaphore=semaphore,handleType=handleType)

def VkPhysicalDeviceExternalFenceInfo(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FENCE_INFO,pNext=None,handleType=None,):

    return _new('VkPhysicalDeviceExternalFenceInfo', sType=sType,pNext=pNext,handleType=handleType)

def VkExternalFenceProperties(sType=VK_STRUCTURE_TYPE_EXTERNAL_FENCE_PROPERTIES,pNext=None,exportFromImportedHandleTypes=None,compatibleHandleTypes=None,externalFenceFeatures=None,):

    return _new('VkExternalFenceProperties', sType=sType,pNext=pNext,exportFromImportedHandleTypes=exportFromImportedHandleTypes,compatibleHandleTypes=compatibleHandleTypes,externalFenceFeatures=externalFenceFeatures)

def VkExportFenceCreateInfo(sType=VK_STRUCTURE_TYPE_EXPORT_FENCE_CREATE_INFO,pNext=None,handleTypes=None,):

    return _new('VkExportFenceCreateInfo', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkImportFenceWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_IMPORT_FENCE_WIN32_HANDLE_INFO_KHR,pNext=None,fence=None,flags=None,handleType=None,handle=None,name=None,):

    return _new('VkImportFenceWin32HandleInfoKHR', sType=sType,pNext=pNext,fence=fence,flags=flags,handleType=handleType,handle=handle,name=name)

def VkExportFenceWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_EXPORT_FENCE_WIN32_HANDLE_INFO_KHR,pNext=None,pAttributes=None,dwAccess=None,name=None,):

    return _new('VkExportFenceWin32HandleInfoKHR', sType=sType,pNext=pNext,pAttributes=pAttributes,dwAccess=dwAccess,name=name)

def VkFenceGetWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_FENCE_GET_WIN32_HANDLE_INFO_KHR,pNext=None,fence=None,handleType=None,):

    return _new('VkFenceGetWin32HandleInfoKHR', sType=sType,pNext=pNext,fence=fence,handleType=handleType)

def VkImportFenceFdInfoKHR(sType=VK_STRUCTURE_TYPE_IMPORT_FENCE_FD_INFO_KHR,pNext=None,fence=None,flags=None,handleType=None,fd=None,):

    return _new('VkImportFenceFdInfoKHR', sType=sType,pNext=pNext,fence=fence,flags=flags,handleType=handleType,fd=fd)

def VkFenceGetFdInfoKHR(sType=VK_STRUCTURE_TYPE_FENCE_GET_FD_INFO_KHR,pNext=None,fence=None,handleType=None,):

    return _new('VkFenceGetFdInfoKHR', sType=sType,pNext=pNext,fence=fence,handleType=handleType)

def VkExportFenceSciSyncInfoNV(sType=VK_STRUCTURE_TYPE_EXPORT_FENCE_SCI_SYNC_INFO_NV,pNext=None,pAttributes=None,):

    return _new('VkExportFenceSciSyncInfoNV', sType=sType,pNext=pNext,pAttributes=pAttributes)

def VkImportFenceSciSyncInfoNV(sType=VK_STRUCTURE_TYPE_IMPORT_FENCE_SCI_SYNC_INFO_NV,pNext=None,fence=None,handleType=None,handle=None,):

    return _new('VkImportFenceSciSyncInfoNV', sType=sType,pNext=pNext,fence=fence,handleType=handleType,handle=handle)

def VkFenceGetSciSyncInfoNV(sType=VK_STRUCTURE_TYPE_FENCE_GET_SCI_SYNC_INFO_NV,pNext=None,fence=None,handleType=None,):

    return _new('VkFenceGetSciSyncInfoNV', sType=sType,pNext=pNext,fence=fence,handleType=handleType)

def VkExportSemaphoreSciSyncInfoNV(sType=VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_SCI_SYNC_INFO_NV,pNext=None,pAttributes=None,):

    return _new('VkExportSemaphoreSciSyncInfoNV', sType=sType,pNext=pNext,pAttributes=pAttributes)

def VkImportSemaphoreSciSyncInfoNV(sType=VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_SCI_SYNC_INFO_NV,pNext=None,semaphore=None,handleType=None,handle=None,):

    return _new('VkImportSemaphoreSciSyncInfoNV', sType=sType,pNext=pNext,semaphore=semaphore,handleType=handleType,handle=handle)

def VkSemaphoreGetSciSyncInfoNV(sType=VK_STRUCTURE_TYPE_SEMAPHORE_GET_SCI_SYNC_INFO_NV,pNext=None,semaphore=None,handleType=None,):

    return _new('VkSemaphoreGetSciSyncInfoNV', sType=sType,pNext=pNext,semaphore=semaphore,handleType=handleType)

def VkSciSyncAttributesInfoNV(sType=VK_STRUCTURE_TYPE_SCI_SYNC_ATTRIBUTES_INFO_NV,pNext=None,clientType=None,primitiveType=None,):

    return _new('VkSciSyncAttributesInfoNV', sType=sType,pNext=pNext,clientType=clientType,primitiveType=primitiveType)

def VkPhysicalDeviceExternalSciSyncFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SCI_SYNC_FEATURES_NV,pNext=None,sciSyncFence=None,sciSyncSemaphore=None,sciSyncImport=None,sciSyncExport=None,):

    return _new('VkPhysicalDeviceExternalSciSyncFeaturesNV', sType=sType,pNext=pNext,sciSyncFence=sciSyncFence,sciSyncSemaphore=sciSyncSemaphore,sciSyncImport=sciSyncImport,sciSyncExport=sciSyncExport)

def VkPhysicalDeviceExternalSciSync2FeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SCI_SYNC_2_FEATURES_NV,pNext=None,sciSyncFence=None,sciSyncSemaphore2=None,sciSyncImport=None,sciSyncExport=None,):

    return _new('VkPhysicalDeviceExternalSciSync2FeaturesNV', sType=sType,pNext=pNext,sciSyncFence=sciSyncFence,sciSyncSemaphore2=sciSyncSemaphore2,sciSyncImport=sciSyncImport,sciSyncExport=sciSyncExport)

def VkSemaphoreSciSyncPoolCreateInfoNV(sType=VK_STRUCTURE_TYPE_SEMAPHORE_SCI_SYNC_POOL_CREATE_INFO_NV,pNext=None,handle=None,):

    return _new('VkSemaphoreSciSyncPoolCreateInfoNV', sType=sType,pNext=pNext,handle=handle)

def VkSemaphoreSciSyncCreateInfoNV(sType=VK_STRUCTURE_TYPE_SEMAPHORE_SCI_SYNC_CREATE_INFO_NV,pNext=None,semaphorePool=None,pFence=None,):

    return _new('VkSemaphoreSciSyncCreateInfoNV', sType=sType,pNext=pNext,semaphorePool=semaphorePool,pFence=pFence)

def VkDeviceSemaphoreSciSyncPoolReservationCreateInfoNV(sType=VK_STRUCTURE_TYPE_DEVICE_SEMAPHORE_SCI_SYNC_POOL_RESERVATION_CREATE_INFO_NV,pNext=None,semaphoreSciSyncPoolRequestCount=None,):

    return _new('VkDeviceSemaphoreSciSyncPoolReservationCreateInfoNV', sType=sType,pNext=pNext,semaphoreSciSyncPoolRequestCount=semaphoreSciSyncPoolRequestCount)

def VkPhysicalDeviceMultiviewFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_FEATURES,pNext=None,multiview=None,multiviewGeometryShader=None,multiviewTessellationShader=None,):

    return _new('VkPhysicalDeviceMultiviewFeatures', sType=sType,pNext=pNext,multiview=multiview,multiviewGeometryShader=multiviewGeometryShader,multiviewTessellationShader=multiviewTessellationShader)

def VkPhysicalDeviceMultiviewProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PROPERTIES,pNext=None,maxMultiviewViewCount=None,maxMultiviewInstanceIndex=None,):

    return _new('VkPhysicalDeviceMultiviewProperties', sType=sType,pNext=pNext,maxMultiviewViewCount=maxMultiviewViewCount,maxMultiviewInstanceIndex=maxMultiviewInstanceIndex)

def VkRenderPassMultiviewCreateInfo(sType=VK_STRUCTURE_TYPE_RENDER_PASS_MULTIVIEW_CREATE_INFO,pNext=None,subpassCount=None,pViewMasks=None,dependencyCount=None,pViewOffsets=None,correlationMaskCount=None,pCorrelationMasks=None,):
    if subpassCount is None and pViewMasks is not None:
        subpassCount = len(pViewMasks)
    if dependencyCount is None and pViewOffsets is not None:
        dependencyCount = len(pViewOffsets)
    if correlationMaskCount is None and pCorrelationMasks is not None:
        correlationMaskCount = len(pCorrelationMasks)

    return _new('VkRenderPassMultiviewCreateInfo', sType=sType,pNext=pNext,subpassCount=subpassCount,pViewMasks=pViewMasks,dependencyCount=dependencyCount,pViewOffsets=pViewOffsets,correlationMaskCount=correlationMaskCount,pCorrelationMasks=pCorrelationMasks)

def VkSurfaceCapabilities2EXT(sType=VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_EXT,pNext=None,minImageCount=None,maxImageCount=None,currentExtent=None,minImageExtent=None,maxImageExtent=None,maxImageArrayLayers=None,supportedTransforms=None,currentTransform=None,supportedCompositeAlpha=None,supportedUsageFlags=None,supportedSurfaceCounters=None,):

    return _new('VkSurfaceCapabilities2EXT', sType=sType,pNext=pNext,minImageCount=minImageCount,maxImageCount=maxImageCount,currentExtent=currentExtent,minImageExtent=minImageExtent,maxImageExtent=maxImageExtent,maxImageArrayLayers=maxImageArrayLayers,supportedTransforms=supportedTransforms,currentTransform=currentTransform,supportedCompositeAlpha=supportedCompositeAlpha,supportedUsageFlags=supportedUsageFlags,supportedSurfaceCounters=supportedSurfaceCounters)

def VkDisplayPowerInfoEXT(sType=VK_STRUCTURE_TYPE_DISPLAY_POWER_INFO_EXT,pNext=None,powerState=None,):

    return _new('VkDisplayPowerInfoEXT', sType=sType,pNext=pNext,powerState=powerState)

def VkDeviceEventInfoEXT(sType=VK_STRUCTURE_TYPE_DEVICE_EVENT_INFO_EXT,pNext=None,deviceEvent=None,):

    return _new('VkDeviceEventInfoEXT', sType=sType,pNext=pNext,deviceEvent=deviceEvent)

def VkDisplayEventInfoEXT(sType=VK_STRUCTURE_TYPE_DISPLAY_EVENT_INFO_EXT,pNext=None,displayEvent=None,):

    return _new('VkDisplayEventInfoEXT', sType=sType,pNext=pNext,displayEvent=displayEvent)

def VkSwapchainCounterCreateInfoEXT(sType=VK_STRUCTURE_TYPE_SWAPCHAIN_COUNTER_CREATE_INFO_EXT,pNext=None,surfaceCounters=None,):

    return _new('VkSwapchainCounterCreateInfoEXT', sType=sType,pNext=pNext,surfaceCounters=surfaceCounters)

def VkPhysicalDeviceGroupProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GROUP_PROPERTIES,pNext=None,physicalDeviceCount=None,physicalDevices=None,subsetAllocation=None,):

    return _new('VkPhysicalDeviceGroupProperties', sType=sType,pNext=pNext,physicalDeviceCount=physicalDeviceCount,physicalDevices=physicalDevices,subsetAllocation=subsetAllocation)

def VkMemoryAllocateFlagsInfo(sType=VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_FLAGS_INFO,pNext=None,flags=None,deviceMask=None,):

    return _new('VkMemoryAllocateFlagsInfo', sType=sType,pNext=pNext,flags=flags,deviceMask=deviceMask)

def VkBindBufferMemoryInfo(sType=VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_INFO,pNext=None,buffer=None,memory=None,memoryOffset=None,):

    return _new('VkBindBufferMemoryInfo', sType=sType,pNext=pNext,buffer=buffer,memory=memory,memoryOffset=memoryOffset)

def VkBindBufferMemoryDeviceGroupInfo(sType=VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_DEVICE_GROUP_INFO,pNext=None,deviceIndexCount=None,pDeviceIndices=None,):
    if deviceIndexCount is None and pDeviceIndices is not None:
        deviceIndexCount = len(pDeviceIndices)

    return _new('VkBindBufferMemoryDeviceGroupInfo', sType=sType,pNext=pNext,deviceIndexCount=deviceIndexCount,pDeviceIndices=pDeviceIndices)

def VkBindImageMemoryInfo(sType=VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_INFO,pNext=None,image=None,memory=None,memoryOffset=None,):

    return _new('VkBindImageMemoryInfo', sType=sType,pNext=pNext,image=image,memory=memory,memoryOffset=memoryOffset)

def VkBindImageMemoryDeviceGroupInfo(sType=VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_DEVICE_GROUP_INFO,pNext=None,deviceIndexCount=None,pDeviceIndices=None,splitInstanceBindRegionCount=None,pSplitInstanceBindRegions=None,):
    if deviceIndexCount is None and pDeviceIndices is not None:
        deviceIndexCount = len(pDeviceIndices)
    if splitInstanceBindRegionCount is None and pSplitInstanceBindRegions is not None:
        splitInstanceBindRegionCount = len(pSplitInstanceBindRegions)

    return _new('VkBindImageMemoryDeviceGroupInfo', sType=sType,pNext=pNext,deviceIndexCount=deviceIndexCount,pDeviceIndices=pDeviceIndices,splitInstanceBindRegionCount=splitInstanceBindRegionCount,pSplitInstanceBindRegions=pSplitInstanceBindRegions)

def VkDeviceGroupRenderPassBeginInfo(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_RENDER_PASS_BEGIN_INFO,pNext=None,deviceMask=None,deviceRenderAreaCount=None,pDeviceRenderAreas=None,):
    if deviceRenderAreaCount is None and pDeviceRenderAreas is not None:
        deviceRenderAreaCount = len(pDeviceRenderAreas)

    return _new('VkDeviceGroupRenderPassBeginInfo', sType=sType,pNext=pNext,deviceMask=deviceMask,deviceRenderAreaCount=deviceRenderAreaCount,pDeviceRenderAreas=pDeviceRenderAreas)

def VkDeviceGroupCommandBufferBeginInfo(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_COMMAND_BUFFER_BEGIN_INFO,pNext=None,deviceMask=None,):

    return _new('VkDeviceGroupCommandBufferBeginInfo', sType=sType,pNext=pNext,deviceMask=deviceMask)

def VkDeviceGroupSubmitInfo(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_SUBMIT_INFO,pNext=None,waitSemaphoreCount=None,pWaitSemaphoreDeviceIndices=None,commandBufferCount=None,pCommandBufferDeviceMasks=None,signalSemaphoreCount=None,pSignalSemaphoreDeviceIndices=None,):
    if waitSemaphoreCount is None and pWaitSemaphoreDeviceIndices is not None:
        waitSemaphoreCount = len(pWaitSemaphoreDeviceIndices)
    if commandBufferCount is None and pCommandBufferDeviceMasks is not None:
        commandBufferCount = len(pCommandBufferDeviceMasks)
    if signalSemaphoreCount is None and pSignalSemaphoreDeviceIndices is not None:
        signalSemaphoreCount = len(pSignalSemaphoreDeviceIndices)

    return _new('VkDeviceGroupSubmitInfo', sType=sType,pNext=pNext,waitSemaphoreCount=waitSemaphoreCount,pWaitSemaphoreDeviceIndices=pWaitSemaphoreDeviceIndices,commandBufferCount=commandBufferCount,pCommandBufferDeviceMasks=pCommandBufferDeviceMasks,signalSemaphoreCount=signalSemaphoreCount,pSignalSemaphoreDeviceIndices=pSignalSemaphoreDeviceIndices)

def VkDeviceGroupBindSparseInfo(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_BIND_SPARSE_INFO,pNext=None,resourceDeviceIndex=None,memoryDeviceIndex=None,):

    return _new('VkDeviceGroupBindSparseInfo', sType=sType,pNext=pNext,resourceDeviceIndex=resourceDeviceIndex,memoryDeviceIndex=memoryDeviceIndex)

def VkDeviceGroupPresentCapabilitiesKHR(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_CAPABILITIES_KHR,pNext=None,presentMask=None,modes=None,):

    return _new('VkDeviceGroupPresentCapabilitiesKHR', sType=sType,pNext=pNext,presentMask=presentMask,modes=modes)

def VkImageSwapchainCreateInfoKHR(sType=VK_STRUCTURE_TYPE_IMAGE_SWAPCHAIN_CREATE_INFO_KHR,pNext=None,swapchain=None,):

    return _new('VkImageSwapchainCreateInfoKHR', sType=sType,pNext=pNext,swapchain=swapchain)

def VkBindImageMemorySwapchainInfoKHR(sType=VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_SWAPCHAIN_INFO_KHR,pNext=None,swapchain=None,imageIndex=None,):

    return _new('VkBindImageMemorySwapchainInfoKHR', sType=sType,pNext=pNext,swapchain=swapchain,imageIndex=imageIndex)

def VkAcquireNextImageInfoKHR(sType=VK_STRUCTURE_TYPE_ACQUIRE_NEXT_IMAGE_INFO_KHR,pNext=None,swapchain=None,timeout=None,semaphore=None,fence=None,deviceMask=None,):

    return _new('VkAcquireNextImageInfoKHR', sType=sType,pNext=pNext,swapchain=swapchain,timeout=timeout,semaphore=semaphore,fence=fence,deviceMask=deviceMask)

def VkDeviceGroupPresentInfoKHR(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_INFO_KHR,pNext=None,swapchainCount=None,pDeviceMasks=None,mode=None,):
    if swapchainCount is None and pDeviceMasks is not None:
        swapchainCount = len(pDeviceMasks)

    return _new('VkDeviceGroupPresentInfoKHR', sType=sType,pNext=pNext,swapchainCount=swapchainCount,pDeviceMasks=pDeviceMasks,mode=mode)

def VkDeviceGroupDeviceCreateInfo(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_DEVICE_CREATE_INFO,pNext=None,physicalDeviceCount=None,pPhysicalDevices=None,):
    if physicalDeviceCount is None and pPhysicalDevices is not None:
        physicalDeviceCount = len(pPhysicalDevices)

    return _new('VkDeviceGroupDeviceCreateInfo', sType=sType,pNext=pNext,physicalDeviceCount=physicalDeviceCount,pPhysicalDevices=pPhysicalDevices)

def VkDeviceGroupSwapchainCreateInfoKHR(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_SWAPCHAIN_CREATE_INFO_KHR,pNext=None,modes=None,):

    return _new('VkDeviceGroupSwapchainCreateInfoKHR', sType=sType,pNext=pNext,modes=modes)

def VkDescriptorUpdateTemplateEntry(dstBinding=None,dstArrayElement=None,descriptorCount=None,descriptorType=None,offset=None,stride=None,):

    return _new('VkDescriptorUpdateTemplateEntry', dstBinding=dstBinding,dstArrayElement=dstArrayElement,descriptorCount=descriptorCount,descriptorType=descriptorType,offset=offset,stride=stride)

def VkDescriptorUpdateTemplateCreateInfo(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_CREATE_INFO,pNext=None,flags=None,descriptorUpdateEntryCount=None,pDescriptorUpdateEntries=None,templateType=None,descriptorSetLayout=None,pipelineBindPoint=None,pipelineLayout=None,set=None,):
    if descriptorUpdateEntryCount is None and pDescriptorUpdateEntries is not None:
        descriptorUpdateEntryCount = len(pDescriptorUpdateEntries)

    return _new('VkDescriptorUpdateTemplateCreateInfo', sType=sType,pNext=pNext,flags=flags,descriptorUpdateEntryCount=descriptorUpdateEntryCount,pDescriptorUpdateEntries=pDescriptorUpdateEntries,templateType=templateType,descriptorSetLayout=descriptorSetLayout,pipelineBindPoint=pipelineBindPoint,pipelineLayout=pipelineLayout,set=set)

def VkXYColorEXT(x=None,y=None,):

    return _new('VkXYColorEXT', x=x,y=y)

def VkPhysicalDevicePresentIdFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRESENT_ID_FEATURES_KHR,pNext=None,presentId=None,):

    return _new('VkPhysicalDevicePresentIdFeaturesKHR', sType=sType,pNext=pNext,presentId=presentId)

def VkPresentIdKHR(sType=VK_STRUCTURE_TYPE_PRESENT_ID_KHR,pNext=None,swapchainCount=None,pPresentIds=None,):
    if swapchainCount is None and pPresentIds is not None:
        swapchainCount = len(pPresentIds)

    return _new('VkPresentIdKHR', sType=sType,pNext=pNext,swapchainCount=swapchainCount,pPresentIds=pPresentIds)

def VkPhysicalDevicePresentWaitFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRESENT_WAIT_FEATURES_KHR,pNext=None,presentWait=None,):

    return _new('VkPhysicalDevicePresentWaitFeaturesKHR', sType=sType,pNext=pNext,presentWait=presentWait)

def VkHdrMetadataEXT(sType=VK_STRUCTURE_TYPE_HDR_METADATA_EXT,pNext=None,displayPrimaryRed=None,displayPrimaryGreen=None,displayPrimaryBlue=None,whitePoint=None,maxLuminance=None,minLuminance=None,maxContentLightLevel=None,maxFrameAverageLightLevel=None,):

    return _new('VkHdrMetadataEXT', sType=sType,pNext=pNext,displayPrimaryRed=displayPrimaryRed,displayPrimaryGreen=displayPrimaryGreen,displayPrimaryBlue=displayPrimaryBlue,whitePoint=whitePoint,maxLuminance=maxLuminance,minLuminance=minLuminance,maxContentLightLevel=maxContentLightLevel,maxFrameAverageLightLevel=maxFrameAverageLightLevel)

def VkDisplayNativeHdrSurfaceCapabilitiesAMD(sType=VK_STRUCTURE_TYPE_DISPLAY_NATIVE_HDR_SURFACE_CAPABILITIES_AMD,pNext=None,localDimmingSupport=None,):

    return _new('VkDisplayNativeHdrSurfaceCapabilitiesAMD', sType=sType,pNext=pNext,localDimmingSupport=localDimmingSupport)

def VkSwapchainDisplayNativeHdrCreateInfoAMD(sType=VK_STRUCTURE_TYPE_SWAPCHAIN_DISPLAY_NATIVE_HDR_CREATE_INFO_AMD,pNext=None,localDimmingEnable=None,):

    return _new('VkSwapchainDisplayNativeHdrCreateInfoAMD', sType=sType,pNext=pNext,localDimmingEnable=localDimmingEnable)

def VkRefreshCycleDurationGOOGLE(refreshDuration=None,):

    return _new('VkRefreshCycleDurationGOOGLE', refreshDuration=refreshDuration)

def VkPastPresentationTimingGOOGLE(presentID=None,desiredPresentTime=None,actualPresentTime=None,earliestPresentTime=None,presentMargin=None,):

    return _new('VkPastPresentationTimingGOOGLE', presentID=presentID,desiredPresentTime=desiredPresentTime,actualPresentTime=actualPresentTime,earliestPresentTime=earliestPresentTime,presentMargin=presentMargin)

def VkPresentTimesInfoGOOGLE(sType=VK_STRUCTURE_TYPE_PRESENT_TIMES_INFO_GOOGLE,pNext=None,swapchainCount=None,pTimes=None,):
    if swapchainCount is None and pTimes is not None:
        swapchainCount = len(pTimes)

    return _new('VkPresentTimesInfoGOOGLE', sType=sType,pNext=pNext,swapchainCount=swapchainCount,pTimes=pTimes)

def VkPresentTimeGOOGLE(presentID=None,desiredPresentTime=None,):

    return _new('VkPresentTimeGOOGLE', presentID=presentID,desiredPresentTime=desiredPresentTime)

def VkIOSSurfaceCreateInfoMVK(sType=VK_STRUCTURE_TYPE_IOS_SURFACE_CREATE_INFO_MVK,pNext=None,flags=None,pView=None,):

    return _new('VkIOSSurfaceCreateInfoMVK', sType=sType,pNext=pNext,flags=flags,pView=pView)

def VkMacOSSurfaceCreateInfoMVK(sType=VK_STRUCTURE_TYPE_MACOS_SURFACE_CREATE_INFO_MVK,pNext=None,flags=None,pView=None,):

    return _new('VkMacOSSurfaceCreateInfoMVK', sType=sType,pNext=pNext,flags=flags,pView=pView)

def VkMetalSurfaceCreateInfoEXT(sType=VK_STRUCTURE_TYPE_METAL_SURFACE_CREATE_INFO_EXT,pNext=None,flags=None,pLayer=None,):

    return _new('VkMetalSurfaceCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,pLayer=pLayer)

def VkViewportWScalingNV(xcoeff=None,ycoeff=None,):

    return _new('VkViewportWScalingNV', xcoeff=xcoeff,ycoeff=ycoeff)

def VkPipelineViewportWScalingStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_W_SCALING_STATE_CREATE_INFO_NV,pNext=None,viewportWScalingEnable=None,viewportCount=None,pViewportWScalings=None,):
    if viewportCount is None and pViewportWScalings is not None:
        viewportCount = len(pViewportWScalings)

    return _new('VkPipelineViewportWScalingStateCreateInfoNV', sType=sType,pNext=pNext,viewportWScalingEnable=viewportWScalingEnable,viewportCount=viewportCount,pViewportWScalings=pViewportWScalings)

def VkViewportSwizzleNV(x=None,y=None,z=None,w=None,):

    return _new('VkViewportSwizzleNV', x=x,y=y,z=z,w=w)

def VkPipelineViewportSwizzleStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SWIZZLE_STATE_CREATE_INFO_NV,pNext=None,flags=None,viewportCount=None,pViewportSwizzles=None,):
    if viewportCount is None and pViewportSwizzles is not None:
        viewportCount = len(pViewportSwizzles)

    return _new('VkPipelineViewportSwizzleStateCreateInfoNV', sType=sType,pNext=pNext,flags=flags,viewportCount=viewportCount,pViewportSwizzles=pViewportSwizzles)

def VkPhysicalDeviceDiscardRectanglePropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DISCARD_RECTANGLE_PROPERTIES_EXT,pNext=None,maxDiscardRectangles=None,):

    return _new('VkPhysicalDeviceDiscardRectanglePropertiesEXT', sType=sType,pNext=pNext,maxDiscardRectangles=maxDiscardRectangles)

def VkPipelineDiscardRectangleStateCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_DISCARD_RECTANGLE_STATE_CREATE_INFO_EXT,pNext=None,flags=None,discardRectangleMode=None,discardRectangleCount=None,pDiscardRectangles=None,):
    if discardRectangleCount is None and pDiscardRectangles is not None:
        discardRectangleCount = len(pDiscardRectangles)

    return _new('VkPipelineDiscardRectangleStateCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,discardRectangleMode=discardRectangleMode,discardRectangleCount=discardRectangleCount,pDiscardRectangles=pDiscardRectangles)

def VkPhysicalDeviceMultiviewPerViewAttributesPropertiesNVX(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PER_VIEW_ATTRIBUTES_PROPERTIES_NVX,pNext=None,perViewPositionAllComponents=None,):

    return _new('VkPhysicalDeviceMultiviewPerViewAttributesPropertiesNVX', sType=sType,pNext=pNext,perViewPositionAllComponents=perViewPositionAllComponents)

def VkInputAttachmentAspectReference(subpass=None,inputAttachmentIndex=None,aspectMask=None,):

    return _new('VkInputAttachmentAspectReference', subpass=subpass,inputAttachmentIndex=inputAttachmentIndex,aspectMask=aspectMask)

def VkRenderPassInputAttachmentAspectCreateInfo(sType=VK_STRUCTURE_TYPE_RENDER_PASS_INPUT_ATTACHMENT_ASPECT_CREATE_INFO,pNext=None,aspectReferenceCount=None,pAspectReferences=None,):
    if aspectReferenceCount is None and pAspectReferences is not None:
        aspectReferenceCount = len(pAspectReferences)

    return _new('VkRenderPassInputAttachmentAspectCreateInfo', sType=sType,pNext=pNext,aspectReferenceCount=aspectReferenceCount,pAspectReferences=pAspectReferences)

def VkPhysicalDeviceSurfaceInfo2KHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SURFACE_INFO_2_KHR,pNext=None,surface=None,):

    return _new('VkPhysicalDeviceSurfaceInfo2KHR', sType=sType,pNext=pNext,surface=surface)

def VkSurfaceCapabilities2KHR(sType=VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_KHR,pNext=None,surfaceCapabilities=None,):

    return _new('VkSurfaceCapabilities2KHR', sType=sType,pNext=pNext,surfaceCapabilities=surfaceCapabilities)

def VkSurfaceFormat2KHR(sType=VK_STRUCTURE_TYPE_SURFACE_FORMAT_2_KHR,pNext=None,surfaceFormat=None,):

    return _new('VkSurfaceFormat2KHR', sType=sType,pNext=pNext,surfaceFormat=surfaceFormat)

def VkDisplayProperties2KHR(sType=VK_STRUCTURE_TYPE_DISPLAY_PROPERTIES_2_KHR,pNext=None,displayProperties=None,):

    return _new('VkDisplayProperties2KHR', sType=sType,pNext=pNext,displayProperties=displayProperties)

def VkDisplayPlaneProperties2KHR(sType=VK_STRUCTURE_TYPE_DISPLAY_PLANE_PROPERTIES_2_KHR,pNext=None,displayPlaneProperties=None,):

    return _new('VkDisplayPlaneProperties2KHR', sType=sType,pNext=pNext,displayPlaneProperties=displayPlaneProperties)

def VkDisplayModeProperties2KHR(sType=VK_STRUCTURE_TYPE_DISPLAY_MODE_PROPERTIES_2_KHR,pNext=None,displayModeProperties=None,):

    return _new('VkDisplayModeProperties2KHR', sType=sType,pNext=pNext,displayModeProperties=displayModeProperties)

def VkDisplayPlaneInfo2KHR(sType=VK_STRUCTURE_TYPE_DISPLAY_PLANE_INFO_2_KHR,pNext=None,mode=None,planeIndex=None,):

    return _new('VkDisplayPlaneInfo2KHR', sType=sType,pNext=pNext,mode=mode,planeIndex=planeIndex)

def VkDisplayPlaneCapabilities2KHR(sType=VK_STRUCTURE_TYPE_DISPLAY_PLANE_CAPABILITIES_2_KHR,pNext=None,capabilities=None,):

    return _new('VkDisplayPlaneCapabilities2KHR', sType=sType,pNext=pNext,capabilities=capabilities)

def VkSharedPresentSurfaceCapabilitiesKHR(sType=VK_STRUCTURE_TYPE_SHARED_PRESENT_SURFACE_CAPABILITIES_KHR,pNext=None,sharedPresentSupportedUsageFlags=None,):

    return _new('VkSharedPresentSurfaceCapabilitiesKHR', sType=sType,pNext=pNext,sharedPresentSupportedUsageFlags=sharedPresentSupportedUsageFlags)

def VkPhysicalDevice16BitStorageFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_16BIT_STORAGE_FEATURES,pNext=None,storageBuffer16BitAccess=None,uniformAndStorageBuffer16BitAccess=None,storagePushConstant16=None,storageInputOutput16=None,):

    return _new('VkPhysicalDevice16BitStorageFeatures', sType=sType,pNext=pNext,storageBuffer16BitAccess=storageBuffer16BitAccess,uniformAndStorageBuffer16BitAccess=uniformAndStorageBuffer16BitAccess,storagePushConstant16=storagePushConstant16,storageInputOutput16=storageInputOutput16)

def VkPhysicalDeviceSubgroupProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_PROPERTIES,pNext=None,subgroupSize=None,supportedStages=None,supportedOperations=None,quadOperationsInAllStages=None,):

    return _new('VkPhysicalDeviceSubgroupProperties', sType=sType,pNext=pNext,subgroupSize=subgroupSize,supportedStages=supportedStages,supportedOperations=supportedOperations,quadOperationsInAllStages=quadOperationsInAllStages)

def VkPhysicalDeviceShaderSubgroupExtendedTypesFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_SUBGROUP_EXTENDED_TYPES_FEATURES,pNext=None,shaderSubgroupExtendedTypes=None,):

    return _new('VkPhysicalDeviceShaderSubgroupExtendedTypesFeatures', sType=sType,pNext=pNext,shaderSubgroupExtendedTypes=shaderSubgroupExtendedTypes)

def VkBufferMemoryRequirementsInfo2(sType=VK_STRUCTURE_TYPE_BUFFER_MEMORY_REQUIREMENTS_INFO_2,pNext=None,buffer=None,):

    return _new('VkBufferMemoryRequirementsInfo2', sType=sType,pNext=pNext,buffer=buffer)

def VkDeviceBufferMemoryRequirements(sType=VK_STRUCTURE_TYPE_DEVICE_BUFFER_MEMORY_REQUIREMENTS,pNext=None,pCreateInfo=None,):

    return _new('VkDeviceBufferMemoryRequirements', sType=sType,pNext=pNext,pCreateInfo=pCreateInfo)

def VkImageMemoryRequirementsInfo2(sType=VK_STRUCTURE_TYPE_IMAGE_MEMORY_REQUIREMENTS_INFO_2,pNext=None,image=None,):

    return _new('VkImageMemoryRequirementsInfo2', sType=sType,pNext=pNext,image=image)

def VkImageSparseMemoryRequirementsInfo2(sType=VK_STRUCTURE_TYPE_IMAGE_SPARSE_MEMORY_REQUIREMENTS_INFO_2,pNext=None,image=None,):

    return _new('VkImageSparseMemoryRequirementsInfo2', sType=sType,pNext=pNext,image=image)

def VkDeviceImageMemoryRequirements(sType=VK_STRUCTURE_TYPE_DEVICE_IMAGE_MEMORY_REQUIREMENTS,pNext=None,pCreateInfo=None,planeAspect=None,):

    return _new('VkDeviceImageMemoryRequirements', sType=sType,pNext=pNext,pCreateInfo=pCreateInfo,planeAspect=planeAspect)

def VkMemoryRequirements2(sType=VK_STRUCTURE_TYPE_MEMORY_REQUIREMENTS_2,pNext=None,memoryRequirements=None,):

    return _new('VkMemoryRequirements2', sType=sType,pNext=pNext,memoryRequirements=memoryRequirements)

def VkSparseImageMemoryRequirements2(sType=VK_STRUCTURE_TYPE_SPARSE_IMAGE_MEMORY_REQUIREMENTS_2,pNext=None,memoryRequirements=None,):

    return _new('VkSparseImageMemoryRequirements2', sType=sType,pNext=pNext,memoryRequirements=memoryRequirements)

def VkPhysicalDevicePointClippingProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_POINT_CLIPPING_PROPERTIES,pNext=None,pointClippingBehavior=None,):

    return _new('VkPhysicalDevicePointClippingProperties', sType=sType,pNext=pNext,pointClippingBehavior=pointClippingBehavior)

def VkMemoryDedicatedRequirements(sType=VK_STRUCTURE_TYPE_MEMORY_DEDICATED_REQUIREMENTS,pNext=None,prefersDedicatedAllocation=None,requiresDedicatedAllocation=None,):

    return _new('VkMemoryDedicatedRequirements', sType=sType,pNext=pNext,prefersDedicatedAllocation=prefersDedicatedAllocation,requiresDedicatedAllocation=requiresDedicatedAllocation)

def VkMemoryDedicatedAllocateInfo(sType=VK_STRUCTURE_TYPE_MEMORY_DEDICATED_ALLOCATE_INFO,pNext=None,image=None,buffer=None,):

    return _new('VkMemoryDedicatedAllocateInfo', sType=sType,pNext=pNext,image=image,buffer=buffer)

def VkImageViewUsageCreateInfo(sType=VK_STRUCTURE_TYPE_IMAGE_VIEW_USAGE_CREATE_INFO,pNext=None,usage=None,):

    return _new('VkImageViewUsageCreateInfo', sType=sType,pNext=pNext,usage=usage)

def VkImageViewSlicedCreateInfoEXT(sType=VK_STRUCTURE_TYPE_IMAGE_VIEW_SLICED_CREATE_INFO_EXT,pNext=None,sliceOffset=None,sliceCount=None,):

    return _new('VkImageViewSlicedCreateInfoEXT', sType=sType,pNext=pNext,sliceOffset=sliceOffset,sliceCount=sliceCount)

def VkPipelineTessellationDomainOriginStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_DOMAIN_ORIGIN_STATE_CREATE_INFO,pNext=None,domainOrigin=None,):

    return _new('VkPipelineTessellationDomainOriginStateCreateInfo', sType=sType,pNext=pNext,domainOrigin=domainOrigin)

def VkSamplerYcbcrConversionInfo(sType=VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_INFO,pNext=None,conversion=None,):

    return _new('VkSamplerYcbcrConversionInfo', sType=sType,pNext=pNext,conversion=conversion)

def VkSamplerYcbcrConversionCreateInfo(sType=VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_CREATE_INFO,pNext=None,format=None,ycbcrModel=None,ycbcrRange=None,components=None,xChromaOffset=None,yChromaOffset=None,chromaFilter=None,forceExplicitReconstruction=None,):

    return _new('VkSamplerYcbcrConversionCreateInfo', sType=sType,pNext=pNext,format=format,ycbcrModel=ycbcrModel,ycbcrRange=ycbcrRange,components=components,xChromaOffset=xChromaOffset,yChromaOffset=yChromaOffset,chromaFilter=chromaFilter,forceExplicitReconstruction=forceExplicitReconstruction)

def VkBindImagePlaneMemoryInfo(sType=VK_STRUCTURE_TYPE_BIND_IMAGE_PLANE_MEMORY_INFO,pNext=None,planeAspect=None,):

    return _new('VkBindImagePlaneMemoryInfo', sType=sType,pNext=pNext,planeAspect=planeAspect)

def VkImagePlaneMemoryRequirementsInfo(sType=VK_STRUCTURE_TYPE_IMAGE_PLANE_MEMORY_REQUIREMENTS_INFO,pNext=None,planeAspect=None,):

    return _new('VkImagePlaneMemoryRequirementsInfo', sType=sType,pNext=pNext,planeAspect=planeAspect)

def VkPhysicalDeviceSamplerYcbcrConversionFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_YCBCR_CONVERSION_FEATURES,pNext=None,samplerYcbcrConversion=None,):

    return _new('VkPhysicalDeviceSamplerYcbcrConversionFeatures', sType=sType,pNext=pNext,samplerYcbcrConversion=samplerYcbcrConversion)

def VkSamplerYcbcrConversionImageFormatProperties(sType=VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_IMAGE_FORMAT_PROPERTIES,pNext=None,combinedImageSamplerDescriptorCount=None,):

    return _new('VkSamplerYcbcrConversionImageFormatProperties', sType=sType,pNext=pNext,combinedImageSamplerDescriptorCount=combinedImageSamplerDescriptorCount)

def VkTextureLODGatherFormatPropertiesAMD(sType=VK_STRUCTURE_TYPE_TEXTURE_LOD_GATHER_FORMAT_PROPERTIES_AMD,pNext=None,supportsTextureGatherLODBiasAMD=None,):

    return _new('VkTextureLODGatherFormatPropertiesAMD', sType=sType,pNext=pNext,supportsTextureGatherLODBiasAMD=supportsTextureGatherLODBiasAMD)

def VkConditionalRenderingBeginInfoEXT(sType=VK_STRUCTURE_TYPE_CONDITIONAL_RENDERING_BEGIN_INFO_EXT,pNext=None,buffer=None,offset=None,flags=None,):

    return _new('VkConditionalRenderingBeginInfoEXT', sType=sType,pNext=pNext,buffer=buffer,offset=offset,flags=flags)

def VkProtectedSubmitInfo(sType=VK_STRUCTURE_TYPE_PROTECTED_SUBMIT_INFO,pNext=None,protectedSubmit=None,):

    return _new('VkProtectedSubmitInfo', sType=sType,pNext=pNext,protectedSubmit=protectedSubmit)

def VkPhysicalDeviceProtectedMemoryFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_FEATURES,pNext=None,protectedMemory=None,):

    return _new('VkPhysicalDeviceProtectedMemoryFeatures', sType=sType,pNext=pNext,protectedMemory=protectedMemory)

def VkPhysicalDeviceProtectedMemoryProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_PROPERTIES,pNext=None,protectedNoFault=None,):

    return _new('VkPhysicalDeviceProtectedMemoryProperties', sType=sType,pNext=pNext,protectedNoFault=protectedNoFault)

def VkDeviceQueueInfo2(sType=VK_STRUCTURE_TYPE_DEVICE_QUEUE_INFO_2,pNext=None,flags=None,queueFamilyIndex=None,queueIndex=None,):

    return _new('VkDeviceQueueInfo2', sType=sType,pNext=pNext,flags=flags,queueFamilyIndex=queueFamilyIndex,queueIndex=queueIndex)

def VkPipelineCoverageToColorStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_TO_COLOR_STATE_CREATE_INFO_NV,pNext=None,flags=None,coverageToColorEnable=None,coverageToColorLocation=None,):

    return _new('VkPipelineCoverageToColorStateCreateInfoNV', sType=sType,pNext=pNext,flags=flags,coverageToColorEnable=coverageToColorEnable,coverageToColorLocation=coverageToColorLocation)

def VkPhysicalDeviceSamplerFilterMinmaxProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_FILTER_MINMAX_PROPERTIES,pNext=None,filterMinmaxSingleComponentFormats=None,filterMinmaxImageComponentMapping=None,):

    return _new('VkPhysicalDeviceSamplerFilterMinmaxProperties', sType=sType,pNext=pNext,filterMinmaxSingleComponentFormats=filterMinmaxSingleComponentFormats,filterMinmaxImageComponentMapping=filterMinmaxImageComponentMapping)

def VkSampleLocationEXT(x=None,y=None,):

    return _new('VkSampleLocationEXT', x=x,y=y)

def VkSampleLocationsInfoEXT(sType=VK_STRUCTURE_TYPE_SAMPLE_LOCATIONS_INFO_EXT,pNext=None,sampleLocationsPerPixel=None,sampleLocationGridSize=None,sampleLocationsCount=None,pSampleLocations=None,):
    if sampleLocationsCount is None and pSampleLocations is not None:
        sampleLocationsCount = len(pSampleLocations)

    return _new('VkSampleLocationsInfoEXT', sType=sType,pNext=pNext,sampleLocationsPerPixel=sampleLocationsPerPixel,sampleLocationGridSize=sampleLocationGridSize,sampleLocationsCount=sampleLocationsCount,pSampleLocations=pSampleLocations)

def VkAttachmentSampleLocationsEXT(attachmentIndex=None,sampleLocationsInfo=None,):

    return _new('VkAttachmentSampleLocationsEXT', attachmentIndex=attachmentIndex,sampleLocationsInfo=sampleLocationsInfo)

def VkSubpassSampleLocationsEXT(subpassIndex=None,sampleLocationsInfo=None,):

    return _new('VkSubpassSampleLocationsEXT', subpassIndex=subpassIndex,sampleLocationsInfo=sampleLocationsInfo)

def VkRenderPassSampleLocationsBeginInfoEXT(sType=VK_STRUCTURE_TYPE_RENDER_PASS_SAMPLE_LOCATIONS_BEGIN_INFO_EXT,pNext=None,attachmentInitialSampleLocationsCount=None,pAttachmentInitialSampleLocations=None,postSubpassSampleLocationsCount=None,pPostSubpassSampleLocations=None,):
    if attachmentInitialSampleLocationsCount is None and pAttachmentInitialSampleLocations is not None:
        attachmentInitialSampleLocationsCount = len(pAttachmentInitialSampleLocations)
    if postSubpassSampleLocationsCount is None and pPostSubpassSampleLocations is not None:
        postSubpassSampleLocationsCount = len(pPostSubpassSampleLocations)

    return _new('VkRenderPassSampleLocationsBeginInfoEXT', sType=sType,pNext=pNext,attachmentInitialSampleLocationsCount=attachmentInitialSampleLocationsCount,pAttachmentInitialSampleLocations=pAttachmentInitialSampleLocations,postSubpassSampleLocationsCount=postSubpassSampleLocationsCount,pPostSubpassSampleLocations=pPostSubpassSampleLocations)

def VkPipelineSampleLocationsStateCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_SAMPLE_LOCATIONS_STATE_CREATE_INFO_EXT,pNext=None,sampleLocationsEnable=None,sampleLocationsInfo=None,):

    return _new('VkPipelineSampleLocationsStateCreateInfoEXT', sType=sType,pNext=pNext,sampleLocationsEnable=sampleLocationsEnable,sampleLocationsInfo=sampleLocationsInfo)

def VkPhysicalDeviceSampleLocationsPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLE_LOCATIONS_PROPERTIES_EXT,pNext=None,sampleLocationSampleCounts=None,maxSampleLocationGridSize=None,sampleLocationCoordinateRange=None,sampleLocationSubPixelBits=None,variableSampleLocations=None,):

    return _new('VkPhysicalDeviceSampleLocationsPropertiesEXT', sType=sType,pNext=pNext,sampleLocationSampleCounts=sampleLocationSampleCounts,maxSampleLocationGridSize=maxSampleLocationGridSize,sampleLocationCoordinateRange=sampleLocationCoordinateRange,sampleLocationSubPixelBits=sampleLocationSubPixelBits,variableSampleLocations=variableSampleLocations)

def VkMultisamplePropertiesEXT(sType=VK_STRUCTURE_TYPE_MULTISAMPLE_PROPERTIES_EXT,pNext=None,maxSampleLocationGridSize=None,):

    return _new('VkMultisamplePropertiesEXT', sType=sType,pNext=pNext,maxSampleLocationGridSize=maxSampleLocationGridSize)

def VkSamplerReductionModeCreateInfo(sType=VK_STRUCTURE_TYPE_SAMPLER_REDUCTION_MODE_CREATE_INFO,pNext=None,reductionMode=None,):

    return _new('VkSamplerReductionModeCreateInfo', sType=sType,pNext=pNext,reductionMode=reductionMode)

def VkPhysicalDeviceBlendOperationAdvancedFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_FEATURES_EXT,pNext=None,advancedBlendCoherentOperations=None,):

    return _new('VkPhysicalDeviceBlendOperationAdvancedFeaturesEXT', sType=sType,pNext=pNext,advancedBlendCoherentOperations=advancedBlendCoherentOperations)

def VkPhysicalDeviceMultiDrawFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTI_DRAW_FEATURES_EXT,pNext=None,multiDraw=None,):

    return _new('VkPhysicalDeviceMultiDrawFeaturesEXT', sType=sType,pNext=pNext,multiDraw=multiDraw)

def VkPhysicalDeviceBlendOperationAdvancedPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_PROPERTIES_EXT,pNext=None,advancedBlendMaxColorAttachments=None,advancedBlendIndependentBlend=None,advancedBlendNonPremultipliedSrcColor=None,advancedBlendNonPremultipliedDstColor=None,advancedBlendCorrelatedOverlap=None,advancedBlendAllOperations=None,):

    return _new('VkPhysicalDeviceBlendOperationAdvancedPropertiesEXT', sType=sType,pNext=pNext,advancedBlendMaxColorAttachments=advancedBlendMaxColorAttachments,advancedBlendIndependentBlend=advancedBlendIndependentBlend,advancedBlendNonPremultipliedSrcColor=advancedBlendNonPremultipliedSrcColor,advancedBlendNonPremultipliedDstColor=advancedBlendNonPremultipliedDstColor,advancedBlendCorrelatedOverlap=advancedBlendCorrelatedOverlap,advancedBlendAllOperations=advancedBlendAllOperations)

def VkPipelineColorBlendAdvancedStateCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_ADVANCED_STATE_CREATE_INFO_EXT,pNext=None,srcPremultiplied=None,dstPremultiplied=None,blendOverlap=None,):

    return _new('VkPipelineColorBlendAdvancedStateCreateInfoEXT', sType=sType,pNext=pNext,srcPremultiplied=srcPremultiplied,dstPremultiplied=dstPremultiplied,blendOverlap=blendOverlap)

def VkPhysicalDeviceInlineUniformBlockFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_FEATURES,pNext=None,inlineUniformBlock=None,descriptorBindingInlineUniformBlockUpdateAfterBind=None,):

    return _new('VkPhysicalDeviceInlineUniformBlockFeatures', sType=sType,pNext=pNext,inlineUniformBlock=inlineUniformBlock,descriptorBindingInlineUniformBlockUpdateAfterBind=descriptorBindingInlineUniformBlockUpdateAfterBind)

def VkPhysicalDeviceInlineUniformBlockProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_PROPERTIES,pNext=None,maxInlineUniformBlockSize=None,maxPerStageDescriptorInlineUniformBlocks=None,maxPerStageDescriptorUpdateAfterBindInlineUniformBlocks=None,maxDescriptorSetInlineUniformBlocks=None,maxDescriptorSetUpdateAfterBindInlineUniformBlocks=None,):

    return _new('VkPhysicalDeviceInlineUniformBlockProperties', sType=sType,pNext=pNext,maxInlineUniformBlockSize=maxInlineUniformBlockSize,maxPerStageDescriptorInlineUniformBlocks=maxPerStageDescriptorInlineUniformBlocks,maxPerStageDescriptorUpdateAfterBindInlineUniformBlocks=maxPerStageDescriptorUpdateAfterBindInlineUniformBlocks,maxDescriptorSetInlineUniformBlocks=maxDescriptorSetInlineUniformBlocks,maxDescriptorSetUpdateAfterBindInlineUniformBlocks=maxDescriptorSetUpdateAfterBindInlineUniformBlocks)

def VkWriteDescriptorSetInlineUniformBlock(sType=VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_INLINE_UNIFORM_BLOCK,pNext=None,dataSize=None,pData=None,):
    if dataSize is None and pData is not None:
        dataSize = len(pData)

    return _new('VkWriteDescriptorSetInlineUniformBlock', sType=sType,pNext=pNext,dataSize=dataSize,pData=pData)

def VkDescriptorPoolInlineUniformBlockCreateInfo(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_INLINE_UNIFORM_BLOCK_CREATE_INFO,pNext=None,maxInlineUniformBlockBindings=None,):

    return _new('VkDescriptorPoolInlineUniformBlockCreateInfo', sType=sType,pNext=pNext,maxInlineUniformBlockBindings=maxInlineUniformBlockBindings)

def VkPipelineCoverageModulationStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_MODULATION_STATE_CREATE_INFO_NV,pNext=None,flags=None,coverageModulationMode=None,coverageModulationTableEnable=None,coverageModulationTableCount=None,pCoverageModulationTable=None,):
    if coverageModulationTableCount is None and pCoverageModulationTable is not None:
        coverageModulationTableCount = len(pCoverageModulationTable)

    return _new('VkPipelineCoverageModulationStateCreateInfoNV', sType=sType,pNext=pNext,flags=flags,coverageModulationMode=coverageModulationMode,coverageModulationTableEnable=coverageModulationTableEnable,coverageModulationTableCount=coverageModulationTableCount,pCoverageModulationTable=pCoverageModulationTable)

def VkImageFormatListCreateInfo(sType=VK_STRUCTURE_TYPE_IMAGE_FORMAT_LIST_CREATE_INFO,pNext=None,viewFormatCount=None,pViewFormats=None,):
    if viewFormatCount is None and pViewFormats is not None:
        viewFormatCount = len(pViewFormats)

    return _new('VkImageFormatListCreateInfo', sType=sType,pNext=pNext,viewFormatCount=viewFormatCount,pViewFormats=pViewFormats)

def VkValidationCacheCreateInfoEXT(sType=VK_STRUCTURE_TYPE_VALIDATION_CACHE_CREATE_INFO_EXT,pNext=None,flags=None,initialDataSize=None,pInitialData=None,):
    if initialDataSize is None and pInitialData is not None:
        initialDataSize = len(pInitialData)

    return _new('VkValidationCacheCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,initialDataSize=initialDataSize,pInitialData=pInitialData)

def VkShaderModuleValidationCacheCreateInfoEXT(sType=VK_STRUCTURE_TYPE_SHADER_MODULE_VALIDATION_CACHE_CREATE_INFO_EXT,pNext=None,validationCache=None,):

    return _new('VkShaderModuleValidationCacheCreateInfoEXT', sType=sType,pNext=pNext,validationCache=validationCache)

def VkPhysicalDeviceMaintenance3Properties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_3_PROPERTIES,pNext=None,maxPerSetDescriptors=None,maxMemoryAllocationSize=None,):

    return _new('VkPhysicalDeviceMaintenance3Properties', sType=sType,pNext=pNext,maxPerSetDescriptors=maxPerSetDescriptors,maxMemoryAllocationSize=maxMemoryAllocationSize)

def VkPhysicalDeviceMaintenance4Features(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_4_FEATURES,pNext=None,maintenance4=None,):

    return _new('VkPhysicalDeviceMaintenance4Features', sType=sType,pNext=pNext,maintenance4=maintenance4)

def VkPhysicalDeviceMaintenance4Properties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_4_PROPERTIES,pNext=None,maxBufferSize=None,):

    return _new('VkPhysicalDeviceMaintenance4Properties', sType=sType,pNext=pNext,maxBufferSize=maxBufferSize)

def VkPhysicalDeviceMaintenance5FeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_5_FEATURES_KHR,pNext=None,maintenance5=None,):

    return _new('VkPhysicalDeviceMaintenance5FeaturesKHR', sType=sType,pNext=pNext,maintenance5=maintenance5)

def VkPhysicalDeviceMaintenance5PropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_5_PROPERTIES_KHR,pNext=None,earlyFragmentMultisampleCoverageAfterSampleCounting=None,earlyFragmentSampleMaskTestBeforeSampleCounting=None,depthStencilSwizzleOneSupport=None,polygonModePointSize=None,nonStrictSinglePixelWideLinesUseParallelogram=None,nonStrictWideLinesUseParallelogram=None,):

    return _new('VkPhysicalDeviceMaintenance5PropertiesKHR', sType=sType,pNext=pNext,earlyFragmentMultisampleCoverageAfterSampleCounting=earlyFragmentMultisampleCoverageAfterSampleCounting,earlyFragmentSampleMaskTestBeforeSampleCounting=earlyFragmentSampleMaskTestBeforeSampleCounting,depthStencilSwizzleOneSupport=depthStencilSwizzleOneSupport,polygonModePointSize=polygonModePointSize,nonStrictSinglePixelWideLinesUseParallelogram=nonStrictSinglePixelWideLinesUseParallelogram,nonStrictWideLinesUseParallelogram=nonStrictWideLinesUseParallelogram)

def VkPhysicalDeviceMaintenance6FeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_6_FEATURES_KHR,pNext=None,maintenance6=None,):

    return _new('VkPhysicalDeviceMaintenance6FeaturesKHR', sType=sType,pNext=pNext,maintenance6=maintenance6)

def VkPhysicalDeviceMaintenance6PropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_6_PROPERTIES_KHR,pNext=None,blockTexelViewCompatibleMultipleLayers=None,maxCombinedImageSamplerDescriptorCount=None,fragmentShadingRateClampCombinerInputs=None,):

    return _new('VkPhysicalDeviceMaintenance6PropertiesKHR', sType=sType,pNext=pNext,blockTexelViewCompatibleMultipleLayers=blockTexelViewCompatibleMultipleLayers,maxCombinedImageSamplerDescriptorCount=maxCombinedImageSamplerDescriptorCount,fragmentShadingRateClampCombinerInputs=fragmentShadingRateClampCombinerInputs)

def VkRenderingAreaInfoKHR(sType=VK_STRUCTURE_TYPE_RENDERING_AREA_INFO_KHR,pNext=None,viewMask=None,colorAttachmentCount=None,pColorAttachmentFormats=None,depthAttachmentFormat=None,stencilAttachmentFormat=None,):
    if colorAttachmentCount is None and pColorAttachmentFormats is not None:
        colorAttachmentCount = len(pColorAttachmentFormats)

    return _new('VkRenderingAreaInfoKHR', sType=sType,pNext=pNext,viewMask=viewMask,colorAttachmentCount=colorAttachmentCount,pColorAttachmentFormats=pColorAttachmentFormats,depthAttachmentFormat=depthAttachmentFormat,stencilAttachmentFormat=stencilAttachmentFormat)

def VkDescriptorSetLayoutSupport(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_SUPPORT,pNext=None,supported=None,):

    return _new('VkDescriptorSetLayoutSupport', sType=sType,pNext=pNext,supported=supported)

def VkPhysicalDeviceShaderDrawParametersFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_DRAW_PARAMETERS_FEATURES,pNext=None,shaderDrawParameters=None,):

    return _new('VkPhysicalDeviceShaderDrawParametersFeatures', sType=sType,pNext=pNext,shaderDrawParameters=shaderDrawParameters)

def VkPhysicalDeviceShaderFloat16Int8Features(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_FLOAT16_INT8_FEATURES,pNext=None,shaderFloat16=None,shaderInt8=None,):

    return _new('VkPhysicalDeviceShaderFloat16Int8Features', sType=sType,pNext=pNext,shaderFloat16=shaderFloat16,shaderInt8=shaderInt8)

def VkPhysicalDeviceFloatControlsProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT_CONTROLS_PROPERTIES,pNext=None,denormBehaviorIndependence=None,roundingModeIndependence=None,shaderSignedZeroInfNanPreserveFloat16=None,shaderSignedZeroInfNanPreserveFloat32=None,shaderSignedZeroInfNanPreserveFloat64=None,shaderDenormPreserveFloat16=None,shaderDenormPreserveFloat32=None,shaderDenormPreserveFloat64=None,shaderDenormFlushToZeroFloat16=None,shaderDenormFlushToZeroFloat32=None,shaderDenormFlushToZeroFloat64=None,shaderRoundingModeRTEFloat16=None,shaderRoundingModeRTEFloat32=None,shaderRoundingModeRTEFloat64=None,shaderRoundingModeRTZFloat16=None,shaderRoundingModeRTZFloat32=None,shaderRoundingModeRTZFloat64=None,):

    return _new('VkPhysicalDeviceFloatControlsProperties', sType=sType,pNext=pNext,denormBehaviorIndependence=denormBehaviorIndependence,roundingModeIndependence=roundingModeIndependence,shaderSignedZeroInfNanPreserveFloat16=shaderSignedZeroInfNanPreserveFloat16,shaderSignedZeroInfNanPreserveFloat32=shaderSignedZeroInfNanPreserveFloat32,shaderSignedZeroInfNanPreserveFloat64=shaderSignedZeroInfNanPreserveFloat64,shaderDenormPreserveFloat16=shaderDenormPreserveFloat16,shaderDenormPreserveFloat32=shaderDenormPreserveFloat32,shaderDenormPreserveFloat64=shaderDenormPreserveFloat64,shaderDenormFlushToZeroFloat16=shaderDenormFlushToZeroFloat16,shaderDenormFlushToZeroFloat32=shaderDenormFlushToZeroFloat32,shaderDenormFlushToZeroFloat64=shaderDenormFlushToZeroFloat64,shaderRoundingModeRTEFloat16=shaderRoundingModeRTEFloat16,shaderRoundingModeRTEFloat32=shaderRoundingModeRTEFloat32,shaderRoundingModeRTEFloat64=shaderRoundingModeRTEFloat64,shaderRoundingModeRTZFloat16=shaderRoundingModeRTZFloat16,shaderRoundingModeRTZFloat32=shaderRoundingModeRTZFloat32,shaderRoundingModeRTZFloat64=shaderRoundingModeRTZFloat64)

def VkPhysicalDeviceHostQueryResetFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_HOST_QUERY_RESET_FEATURES,pNext=None,hostQueryReset=None,):

    return _new('VkPhysicalDeviceHostQueryResetFeatures', sType=sType,pNext=pNext,hostQueryReset=hostQueryReset)

def VkNativeBufferUsage2ANDROID(consumer=None,producer=None,):

    return _new('VkNativeBufferUsage2ANDROID', consumer=consumer,producer=producer)

def VkNativeBufferANDROID(sType=VK_STRUCTURE_TYPE_NATIVE_BUFFER_ANDROID,pNext=None,handle=None,stride=None,format=None,usage=None,usage2=None,):

    return _new('VkNativeBufferANDROID', sType=sType,pNext=pNext,handle=handle,stride=stride,format=format,usage=usage,usage2=usage2)

def VkSwapchainImageCreateInfoANDROID(sType=VK_STRUCTURE_TYPE_SWAPCHAIN_IMAGE_CREATE_INFO_ANDROID,pNext=None,usage=None,):

    return _new('VkSwapchainImageCreateInfoANDROID', sType=sType,pNext=pNext,usage=usage)

def VkPhysicalDevicePresentationPropertiesANDROID(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRESENTATION_PROPERTIES_ANDROID,pNext=None,sharedImage=None,):

    return _new('VkPhysicalDevicePresentationPropertiesANDROID', sType=sType,pNext=pNext,sharedImage=sharedImage)

def VkShaderResourceUsageAMD(numUsedVgprs=None,numUsedSgprs=None,ldsSizePerLocalWorkGroup=None,ldsUsageSizeInBytes=None,scratchMemUsageInBytes=None,):

    return _new('VkShaderResourceUsageAMD', numUsedVgprs=numUsedVgprs,numUsedSgprs=numUsedSgprs,ldsSizePerLocalWorkGroup=ldsSizePerLocalWorkGroup,ldsUsageSizeInBytes=ldsUsageSizeInBytes,scratchMemUsageInBytes=scratchMemUsageInBytes)

def VkShaderStatisticsInfoAMD(shaderStageMask=None,resourceUsage=None,numPhysicalVgprs=None,numPhysicalSgprs=None,numAvailableVgprs=None,numAvailableSgprs=None,computeWorkGroupSize=None,):

    return _new('VkShaderStatisticsInfoAMD', shaderStageMask=shaderStageMask,resourceUsage=resourceUsage,numPhysicalVgprs=numPhysicalVgprs,numPhysicalSgprs=numPhysicalSgprs,numAvailableVgprs=numAvailableVgprs,numAvailableSgprs=numAvailableSgprs,computeWorkGroupSize=computeWorkGroupSize)

def VkDeviceQueueGlobalPriorityCreateInfoKHR(sType=VK_STRUCTURE_TYPE_DEVICE_QUEUE_GLOBAL_PRIORITY_CREATE_INFO_KHR,pNext=None,globalPriority=None,):

    return _new('VkDeviceQueueGlobalPriorityCreateInfoKHR', sType=sType,pNext=pNext,globalPriority=globalPriority)

def VkPhysicalDeviceGlobalPriorityQueryFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GLOBAL_PRIORITY_QUERY_FEATURES_KHR,pNext=None,globalPriorityQuery=None,):

    return _new('VkPhysicalDeviceGlobalPriorityQueryFeaturesKHR', sType=sType,pNext=pNext,globalPriorityQuery=globalPriorityQuery)

def VkQueueFamilyGlobalPriorityPropertiesKHR(sType=VK_STRUCTURE_TYPE_QUEUE_FAMILY_GLOBAL_PRIORITY_PROPERTIES_KHR,pNext=None,priorityCount=None,priorities=None,):

    return _new('VkQueueFamilyGlobalPriorityPropertiesKHR', sType=sType,pNext=pNext,priorityCount=priorityCount,priorities=priorities)

def VkDebugUtilsObjectNameInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_NAME_INFO_EXT,pNext=None,objectType=None,objectHandle=None,pObjectName=None,):

    return _new('VkDebugUtilsObjectNameInfoEXT', sType=sType,pNext=pNext,objectType=objectType,objectHandle=objectHandle,pObjectName=pObjectName)

def VkDebugUtilsObjectTagInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_TAG_INFO_EXT,pNext=None,objectType=None,objectHandle=None,tagName=None,tagSize=None,pTag=None,):
    if tagSize is None and pTag is not None:
        tagSize = len(pTag)

    return _new('VkDebugUtilsObjectTagInfoEXT', sType=sType,pNext=pNext,objectType=objectType,objectHandle=objectHandle,tagName=tagName,tagSize=tagSize,pTag=pTag)

def VkDebugUtilsLabelEXT(sType=VK_STRUCTURE_TYPE_DEBUG_UTILS_LABEL_EXT,pNext=None,pLabelName=None,color=None,):

    return _new('VkDebugUtilsLabelEXT', sType=sType,pNext=pNext,pLabelName=pLabelName,color=color)

def VkDebugUtilsMessengerCreateInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CREATE_INFO_EXT,pNext=None,flags=None,messageSeverity=None,messageType=None,pfnUserCallback=None,pUserData=None,):

    return _new('VkDebugUtilsMessengerCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,messageSeverity=messageSeverity,messageType=messageType,pfnUserCallback=pfnUserCallback,pUserData=pUserData)

def VkDebugUtilsMessengerCallbackDataEXT(sType=VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CALLBACK_DATA_EXT,pNext=None,flags=None,pMessageIdName=None,messageIdNumber=None,pMessage=None,queueLabelCount=None,pQueueLabels=None,cmdBufLabelCount=None,pCmdBufLabels=None,objectCount=None,pObjects=None,):
    if queueLabelCount is None and pQueueLabels is not None:
        queueLabelCount = len(pQueueLabels)
    if cmdBufLabelCount is None and pCmdBufLabels is not None:
        cmdBufLabelCount = len(pCmdBufLabels)
    if objectCount is None and pObjects is not None:
        objectCount = len(pObjects)

    return _new('VkDebugUtilsMessengerCallbackDataEXT', sType=sType,pNext=pNext,flags=flags,pMessageIdName=pMessageIdName,messageIdNumber=messageIdNumber,pMessage=pMessage,queueLabelCount=queueLabelCount,pQueueLabels=pQueueLabels,cmdBufLabelCount=cmdBufLabelCount,pCmdBufLabels=pCmdBufLabels,objectCount=objectCount,pObjects=pObjects)

def VkPhysicalDeviceDeviceMemoryReportFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEVICE_MEMORY_REPORT_FEATURES_EXT,pNext=None,deviceMemoryReport=None,):

    return _new('VkPhysicalDeviceDeviceMemoryReportFeaturesEXT', sType=sType,pNext=pNext,deviceMemoryReport=deviceMemoryReport)

def VkDeviceDeviceMemoryReportCreateInfoEXT(sType=VK_STRUCTURE_TYPE_DEVICE_DEVICE_MEMORY_REPORT_CREATE_INFO_EXT,pNext=None,flags=None,pfnUserCallback=None,pUserData=None,):

    return _new('VkDeviceDeviceMemoryReportCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,pfnUserCallback=pfnUserCallback,pUserData=pUserData)

def VkDeviceMemoryReportCallbackDataEXT(sType=VK_STRUCTURE_TYPE_DEVICE_MEMORY_REPORT_CALLBACK_DATA_EXT,pNext=None,flags=None,type=None,memoryObjectId=None,size=None,objectType=None,objectHandle=None,heapIndex=None,):

    return _new('VkDeviceMemoryReportCallbackDataEXT', sType=sType,pNext=pNext,flags=flags,type=type,memoryObjectId=memoryObjectId,size=size,objectType=objectType,objectHandle=objectHandle,heapIndex=heapIndex)

def VkImportMemoryHostPointerInfoEXT(sType=VK_STRUCTURE_TYPE_IMPORT_MEMORY_HOST_POINTER_INFO_EXT,pNext=None,handleType=None,pHostPointer=None,):

    return _new('VkImportMemoryHostPointerInfoEXT', sType=sType,pNext=pNext,handleType=handleType,pHostPointer=pHostPointer)

def VkMemoryHostPointerPropertiesEXT(sType=VK_STRUCTURE_TYPE_MEMORY_HOST_POINTER_PROPERTIES_EXT,pNext=None,memoryTypeBits=None,):

    return _new('VkMemoryHostPointerPropertiesEXT', sType=sType,pNext=pNext,memoryTypeBits=memoryTypeBits)

def VkPhysicalDeviceExternalMemoryHostPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_HOST_PROPERTIES_EXT,pNext=None,minImportedHostPointerAlignment=None,):

    return _new('VkPhysicalDeviceExternalMemoryHostPropertiesEXT', sType=sType,pNext=pNext,minImportedHostPointerAlignment=minImportedHostPointerAlignment)

def VkPhysicalDeviceConservativeRasterizationPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONSERVATIVE_RASTERIZATION_PROPERTIES_EXT,pNext=None,primitiveOverestimationSize=None,maxExtraPrimitiveOverestimationSize=None,extraPrimitiveOverestimationSizeGranularity=None,primitiveUnderestimation=None,conservativePointAndLineRasterization=None,degenerateTrianglesRasterized=None,degenerateLinesRasterized=None,fullyCoveredFragmentShaderInputVariable=None,conservativeRasterizationPostDepthCoverage=None,):

    return _new('VkPhysicalDeviceConservativeRasterizationPropertiesEXT', sType=sType,pNext=pNext,primitiveOverestimationSize=primitiveOverestimationSize,maxExtraPrimitiveOverestimationSize=maxExtraPrimitiveOverestimationSize,extraPrimitiveOverestimationSizeGranularity=extraPrimitiveOverestimationSizeGranularity,primitiveUnderestimation=primitiveUnderestimation,conservativePointAndLineRasterization=conservativePointAndLineRasterization,degenerateTrianglesRasterized=degenerateTrianglesRasterized,degenerateLinesRasterized=degenerateLinesRasterized,fullyCoveredFragmentShaderInputVariable=fullyCoveredFragmentShaderInputVariable,conservativeRasterizationPostDepthCoverage=conservativeRasterizationPostDepthCoverage)

def VkCalibratedTimestampInfoKHR(sType=VK_STRUCTURE_TYPE_CALIBRATED_TIMESTAMP_INFO_KHR,pNext=None,timeDomain=None,):

    return _new('VkCalibratedTimestampInfoKHR', sType=sType,pNext=pNext,timeDomain=timeDomain)

def VkPhysicalDeviceShaderCorePropertiesAMD(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_PROPERTIES_AMD,pNext=None,shaderEngineCount=None,shaderArraysPerEngineCount=None,computeUnitsPerShaderArray=None,simdPerComputeUnit=None,wavefrontsPerSimd=None,wavefrontSize=None,sgprsPerSimd=None,minSgprAllocation=None,maxSgprAllocation=None,sgprAllocationGranularity=None,vgprsPerSimd=None,minVgprAllocation=None,maxVgprAllocation=None,vgprAllocationGranularity=None,):

    return _new('VkPhysicalDeviceShaderCorePropertiesAMD', sType=sType,pNext=pNext,shaderEngineCount=shaderEngineCount,shaderArraysPerEngineCount=shaderArraysPerEngineCount,computeUnitsPerShaderArray=computeUnitsPerShaderArray,simdPerComputeUnit=simdPerComputeUnit,wavefrontsPerSimd=wavefrontsPerSimd,wavefrontSize=wavefrontSize,sgprsPerSimd=sgprsPerSimd,minSgprAllocation=minSgprAllocation,maxSgprAllocation=maxSgprAllocation,sgprAllocationGranularity=sgprAllocationGranularity,vgprsPerSimd=vgprsPerSimd,minVgprAllocation=minVgprAllocation,maxVgprAllocation=maxVgprAllocation,vgprAllocationGranularity=vgprAllocationGranularity)

def VkPhysicalDeviceShaderCoreProperties2AMD(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_PROPERTIES_2_AMD,pNext=None,shaderCoreFeatures=None,activeComputeUnitCount=None,):

    return _new('VkPhysicalDeviceShaderCoreProperties2AMD', sType=sType,pNext=pNext,shaderCoreFeatures=shaderCoreFeatures,activeComputeUnitCount=activeComputeUnitCount)

def VkPipelineRasterizationConservativeStateCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_CONSERVATIVE_STATE_CREATE_INFO_EXT,pNext=None,flags=None,conservativeRasterizationMode=None,extraPrimitiveOverestimationSize=None,):

    return _new('VkPipelineRasterizationConservativeStateCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,conservativeRasterizationMode=conservativeRasterizationMode,extraPrimitiveOverestimationSize=extraPrimitiveOverestimationSize)

def VkPhysicalDeviceDescriptorIndexingFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_FEATURES,pNext=None,shaderInputAttachmentArrayDynamicIndexing=None,shaderUniformTexelBufferArrayDynamicIndexing=None,shaderStorageTexelBufferArrayDynamicIndexing=None,shaderUniformBufferArrayNonUniformIndexing=None,shaderSampledImageArrayNonUniformIndexing=None,shaderStorageBufferArrayNonUniformIndexing=None,shaderStorageImageArrayNonUniformIndexing=None,shaderInputAttachmentArrayNonUniformIndexing=None,shaderUniformTexelBufferArrayNonUniformIndexing=None,shaderStorageTexelBufferArrayNonUniformIndexing=None,descriptorBindingUniformBufferUpdateAfterBind=None,descriptorBindingSampledImageUpdateAfterBind=None,descriptorBindingStorageImageUpdateAfterBind=None,descriptorBindingStorageBufferUpdateAfterBind=None,descriptorBindingUniformTexelBufferUpdateAfterBind=None,descriptorBindingStorageTexelBufferUpdateAfterBind=None,descriptorBindingUpdateUnusedWhilePending=None,descriptorBindingPartiallyBound=None,descriptorBindingVariableDescriptorCount=None,runtimeDescriptorArray=None,):

    return _new('VkPhysicalDeviceDescriptorIndexingFeatures', sType=sType,pNext=pNext,shaderInputAttachmentArrayDynamicIndexing=shaderInputAttachmentArrayDynamicIndexing,shaderUniformTexelBufferArrayDynamicIndexing=shaderUniformTexelBufferArrayDynamicIndexing,shaderStorageTexelBufferArrayDynamicIndexing=shaderStorageTexelBufferArrayDynamicIndexing,shaderUniformBufferArrayNonUniformIndexing=shaderUniformBufferArrayNonUniformIndexing,shaderSampledImageArrayNonUniformIndexing=shaderSampledImageArrayNonUniformIndexing,shaderStorageBufferArrayNonUniformIndexing=shaderStorageBufferArrayNonUniformIndexing,shaderStorageImageArrayNonUniformIndexing=shaderStorageImageArrayNonUniformIndexing,shaderInputAttachmentArrayNonUniformIndexing=shaderInputAttachmentArrayNonUniformIndexing,shaderUniformTexelBufferArrayNonUniformIndexing=shaderUniformTexelBufferArrayNonUniformIndexing,shaderStorageTexelBufferArrayNonUniformIndexing=shaderStorageTexelBufferArrayNonUniformIndexing,descriptorBindingUniformBufferUpdateAfterBind=descriptorBindingUniformBufferUpdateAfterBind,descriptorBindingSampledImageUpdateAfterBind=descriptorBindingSampledImageUpdateAfterBind,descriptorBindingStorageImageUpdateAfterBind=descriptorBindingStorageImageUpdateAfterBind,descriptorBindingStorageBufferUpdateAfterBind=descriptorBindingStorageBufferUpdateAfterBind,descriptorBindingUniformTexelBufferUpdateAfterBind=descriptorBindingUniformTexelBufferUpdateAfterBind,descriptorBindingStorageTexelBufferUpdateAfterBind=descriptorBindingStorageTexelBufferUpdateAfterBind,descriptorBindingUpdateUnusedWhilePending=descriptorBindingUpdateUnusedWhilePending,descriptorBindingPartiallyBound=descriptorBindingPartiallyBound,descriptorBindingVariableDescriptorCount=descriptorBindingVariableDescriptorCount,runtimeDescriptorArray=runtimeDescriptorArray)

def VkPhysicalDeviceDescriptorIndexingProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_PROPERTIES,pNext=None,maxUpdateAfterBindDescriptorsInAllPools=None,shaderUniformBufferArrayNonUniformIndexingNative=None,shaderSampledImageArrayNonUniformIndexingNative=None,shaderStorageBufferArrayNonUniformIndexingNative=None,shaderStorageImageArrayNonUniformIndexingNative=None,shaderInputAttachmentArrayNonUniformIndexingNative=None,robustBufferAccessUpdateAfterBind=None,quadDivergentImplicitLod=None,maxPerStageDescriptorUpdateAfterBindSamplers=None,maxPerStageDescriptorUpdateAfterBindUniformBuffers=None,maxPerStageDescriptorUpdateAfterBindStorageBuffers=None,maxPerStageDescriptorUpdateAfterBindSampledImages=None,maxPerStageDescriptorUpdateAfterBindStorageImages=None,maxPerStageDescriptorUpdateAfterBindInputAttachments=None,maxPerStageUpdateAfterBindResources=None,maxDescriptorSetUpdateAfterBindSamplers=None,maxDescriptorSetUpdateAfterBindUniformBuffers=None,maxDescriptorSetUpdateAfterBindUniformBuffersDynamic=None,maxDescriptorSetUpdateAfterBindStorageBuffers=None,maxDescriptorSetUpdateAfterBindStorageBuffersDynamic=None,maxDescriptorSetUpdateAfterBindSampledImages=None,maxDescriptorSetUpdateAfterBindStorageImages=None,maxDescriptorSetUpdateAfterBindInputAttachments=None,):

    return _new('VkPhysicalDeviceDescriptorIndexingProperties', sType=sType,pNext=pNext,maxUpdateAfterBindDescriptorsInAllPools=maxUpdateAfterBindDescriptorsInAllPools,shaderUniformBufferArrayNonUniformIndexingNative=shaderUniformBufferArrayNonUniformIndexingNative,shaderSampledImageArrayNonUniformIndexingNative=shaderSampledImageArrayNonUniformIndexingNative,shaderStorageBufferArrayNonUniformIndexingNative=shaderStorageBufferArrayNonUniformIndexingNative,shaderStorageImageArrayNonUniformIndexingNative=shaderStorageImageArrayNonUniformIndexingNative,shaderInputAttachmentArrayNonUniformIndexingNative=shaderInputAttachmentArrayNonUniformIndexingNative,robustBufferAccessUpdateAfterBind=robustBufferAccessUpdateAfterBind,quadDivergentImplicitLod=quadDivergentImplicitLod,maxPerStageDescriptorUpdateAfterBindSamplers=maxPerStageDescriptorUpdateAfterBindSamplers,maxPerStageDescriptorUpdateAfterBindUniformBuffers=maxPerStageDescriptorUpdateAfterBindUniformBuffers,maxPerStageDescriptorUpdateAfterBindStorageBuffers=maxPerStageDescriptorUpdateAfterBindStorageBuffers,maxPerStageDescriptorUpdateAfterBindSampledImages=maxPerStageDescriptorUpdateAfterBindSampledImages,maxPerStageDescriptorUpdateAfterBindStorageImages=maxPerStageDescriptorUpdateAfterBindStorageImages,maxPerStageDescriptorUpdateAfterBindInputAttachments=maxPerStageDescriptorUpdateAfterBindInputAttachments,maxPerStageUpdateAfterBindResources=maxPerStageUpdateAfterBindResources,maxDescriptorSetUpdateAfterBindSamplers=maxDescriptorSetUpdateAfterBindSamplers,maxDescriptorSetUpdateAfterBindUniformBuffers=maxDescriptorSetUpdateAfterBindUniformBuffers,maxDescriptorSetUpdateAfterBindUniformBuffersDynamic=maxDescriptorSetUpdateAfterBindUniformBuffersDynamic,maxDescriptorSetUpdateAfterBindStorageBuffers=maxDescriptorSetUpdateAfterBindStorageBuffers,maxDescriptorSetUpdateAfterBindStorageBuffersDynamic=maxDescriptorSetUpdateAfterBindStorageBuffersDynamic,maxDescriptorSetUpdateAfterBindSampledImages=maxDescriptorSetUpdateAfterBindSampledImages,maxDescriptorSetUpdateAfterBindStorageImages=maxDescriptorSetUpdateAfterBindStorageImages,maxDescriptorSetUpdateAfterBindInputAttachments=maxDescriptorSetUpdateAfterBindInputAttachments)

def VkDescriptorSetLayoutBindingFlagsCreateInfo(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_BINDING_FLAGS_CREATE_INFO,pNext=None,bindingCount=None,pBindingFlags=None,):
    if bindingCount is None and pBindingFlags is not None:
        bindingCount = len(pBindingFlags)

    return _new('VkDescriptorSetLayoutBindingFlagsCreateInfo', sType=sType,pNext=pNext,bindingCount=bindingCount,pBindingFlags=pBindingFlags)

def VkDescriptorSetVariableDescriptorCountAllocateInfo(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_ALLOCATE_INFO,pNext=None,descriptorSetCount=None,pDescriptorCounts=None,):
    if descriptorSetCount is None and pDescriptorCounts is not None:
        descriptorSetCount = len(pDescriptorCounts)

    return _new('VkDescriptorSetVariableDescriptorCountAllocateInfo', sType=sType,pNext=pNext,descriptorSetCount=descriptorSetCount,pDescriptorCounts=pDescriptorCounts)

def VkDescriptorSetVariableDescriptorCountLayoutSupport(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_LAYOUT_SUPPORT,pNext=None,maxVariableDescriptorCount=None,):

    return _new('VkDescriptorSetVariableDescriptorCountLayoutSupport', sType=sType,pNext=pNext,maxVariableDescriptorCount=maxVariableDescriptorCount)

def VkAttachmentDescription2(sType=VK_STRUCTURE_TYPE_ATTACHMENT_DESCRIPTION_2,pNext=None,flags=None,format=None,samples=None,loadOp=None,storeOp=None,stencilLoadOp=None,stencilStoreOp=None,initialLayout=None,finalLayout=None,):

    return _new('VkAttachmentDescription2', sType=sType,pNext=pNext,flags=flags,format=format,samples=samples,loadOp=loadOp,storeOp=storeOp,stencilLoadOp=stencilLoadOp,stencilStoreOp=stencilStoreOp,initialLayout=initialLayout,finalLayout=finalLayout)

def VkAttachmentReference2(sType=VK_STRUCTURE_TYPE_ATTACHMENT_REFERENCE_2,pNext=None,attachment=None,layout=None,aspectMask=None,):

    return _new('VkAttachmentReference2', sType=sType,pNext=pNext,attachment=attachment,layout=layout,aspectMask=aspectMask)

def VkSubpassDescription2(sType=VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_2,pNext=None,flags=None,pipelineBindPoint=None,viewMask=None,inputAttachmentCount=None,pInputAttachments=None,colorAttachmentCount=None,pColorAttachments=None,pResolveAttachments=None,pDepthStencilAttachment=None,preserveAttachmentCount=None,pPreserveAttachments=None,):
    if inputAttachmentCount is None and pInputAttachments is not None:
        inputAttachmentCount = len(pInputAttachments)
    if colorAttachmentCount is None and pColorAttachments is not None:
        colorAttachmentCount = len(pColorAttachments)
    if colorAttachmentCount is None and pResolveAttachments is not None:
        colorAttachmentCount = len(pResolveAttachments)
    if preserveAttachmentCount is None and pPreserveAttachments is not None:
        preserveAttachmentCount = len(pPreserveAttachments)

    return _new('VkSubpassDescription2', sType=sType,pNext=pNext,flags=flags,pipelineBindPoint=pipelineBindPoint,viewMask=viewMask,inputAttachmentCount=inputAttachmentCount,pInputAttachments=pInputAttachments,colorAttachmentCount=colorAttachmentCount,pColorAttachments=pColorAttachments,pResolveAttachments=pResolveAttachments,pDepthStencilAttachment=pDepthStencilAttachment,preserveAttachmentCount=preserveAttachmentCount,pPreserveAttachments=pPreserveAttachments)

def VkSubpassDependency2(sType=VK_STRUCTURE_TYPE_SUBPASS_DEPENDENCY_2,pNext=None,srcSubpass=None,dstSubpass=None,srcStageMask=None,dstStageMask=None,srcAccessMask=None,dstAccessMask=None,dependencyFlags=None,viewOffset=None,):

    return _new('VkSubpassDependency2', sType=sType,pNext=pNext,srcSubpass=srcSubpass,dstSubpass=dstSubpass,srcStageMask=srcStageMask,dstStageMask=dstStageMask,srcAccessMask=srcAccessMask,dstAccessMask=dstAccessMask,dependencyFlags=dependencyFlags,viewOffset=viewOffset)

def VkRenderPassCreateInfo2(sType=VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO_2,pNext=None,flags=None,attachmentCount=None,pAttachments=None,subpassCount=None,pSubpasses=None,dependencyCount=None,pDependencies=None,correlatedViewMaskCount=None,pCorrelatedViewMasks=None,):
    if attachmentCount is None and pAttachments is not None:
        attachmentCount = len(pAttachments)
    if subpassCount is None and pSubpasses is not None:
        subpassCount = len(pSubpasses)
    if dependencyCount is None and pDependencies is not None:
        dependencyCount = len(pDependencies)
    if correlatedViewMaskCount is None and pCorrelatedViewMasks is not None:
        correlatedViewMaskCount = len(pCorrelatedViewMasks)

    return _new('VkRenderPassCreateInfo2', sType=sType,pNext=pNext,flags=flags,attachmentCount=attachmentCount,pAttachments=pAttachments,subpassCount=subpassCount,pSubpasses=pSubpasses,dependencyCount=dependencyCount,pDependencies=pDependencies,correlatedViewMaskCount=correlatedViewMaskCount,pCorrelatedViewMasks=pCorrelatedViewMasks)

def VkSubpassBeginInfo(sType=VK_STRUCTURE_TYPE_SUBPASS_BEGIN_INFO,pNext=None,contents=None,):

    return _new('VkSubpassBeginInfo', sType=sType,pNext=pNext,contents=contents)

def VkSubpassEndInfo(sType=VK_STRUCTURE_TYPE_SUBPASS_END_INFO,pNext=None,):

    return _new('VkSubpassEndInfo', sType=sType,pNext=pNext)

def VkPhysicalDeviceTimelineSemaphoreFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TIMELINE_SEMAPHORE_FEATURES,pNext=None,timelineSemaphore=None,):

    return _new('VkPhysicalDeviceTimelineSemaphoreFeatures', sType=sType,pNext=pNext,timelineSemaphore=timelineSemaphore)

def VkPhysicalDeviceTimelineSemaphoreProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TIMELINE_SEMAPHORE_PROPERTIES,pNext=None,maxTimelineSemaphoreValueDifference=None,):

    return _new('VkPhysicalDeviceTimelineSemaphoreProperties', sType=sType,pNext=pNext,maxTimelineSemaphoreValueDifference=maxTimelineSemaphoreValueDifference)

def VkSemaphoreTypeCreateInfo(sType=VK_STRUCTURE_TYPE_SEMAPHORE_TYPE_CREATE_INFO,pNext=None,semaphoreType=None,initialValue=None,):

    return _new('VkSemaphoreTypeCreateInfo', sType=sType,pNext=pNext,semaphoreType=semaphoreType,initialValue=initialValue)

def VkTimelineSemaphoreSubmitInfo(sType=VK_STRUCTURE_TYPE_TIMELINE_SEMAPHORE_SUBMIT_INFO,pNext=None,waitSemaphoreValueCount=None,pWaitSemaphoreValues=None,signalSemaphoreValueCount=None,pSignalSemaphoreValues=None,):
    if waitSemaphoreValueCount is None and pWaitSemaphoreValues is not None:
        waitSemaphoreValueCount = len(pWaitSemaphoreValues)
    if signalSemaphoreValueCount is None and pSignalSemaphoreValues is not None:
        signalSemaphoreValueCount = len(pSignalSemaphoreValues)

    return _new('VkTimelineSemaphoreSubmitInfo', sType=sType,pNext=pNext,waitSemaphoreValueCount=waitSemaphoreValueCount,pWaitSemaphoreValues=pWaitSemaphoreValues,signalSemaphoreValueCount=signalSemaphoreValueCount,pSignalSemaphoreValues=pSignalSemaphoreValues)

def VkSemaphoreWaitInfo(sType=VK_STRUCTURE_TYPE_SEMAPHORE_WAIT_INFO,pNext=None,flags=None,semaphoreCount=None,pSemaphores=None,pValues=None,):
    if semaphoreCount is None and pSemaphores is not None:
        semaphoreCount = len(pSemaphores)
    if semaphoreCount is None and pValues is not None:
        semaphoreCount = len(pValues)

    return _new('VkSemaphoreWaitInfo', sType=sType,pNext=pNext,flags=flags,semaphoreCount=semaphoreCount,pSemaphores=pSemaphores,pValues=pValues)

def VkSemaphoreSignalInfo(sType=VK_STRUCTURE_TYPE_SEMAPHORE_SIGNAL_INFO,pNext=None,semaphore=None,value=None,):

    return _new('VkSemaphoreSignalInfo', sType=sType,pNext=pNext,semaphore=semaphore,value=value)

def VkVertexInputBindingDivisorDescriptionKHR(binding=None,divisor=None,):

    return _new('VkVertexInputBindingDivisorDescriptionKHR', binding=binding,divisor=divisor)

def VkPipelineVertexInputDivisorStateCreateInfoKHR(sType=VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_DIVISOR_STATE_CREATE_INFO_KHR,pNext=None,vertexBindingDivisorCount=None,pVertexBindingDivisors=None,):
    if vertexBindingDivisorCount is None and pVertexBindingDivisors is not None:
        vertexBindingDivisorCount = len(pVertexBindingDivisors)

    return _new('VkPipelineVertexInputDivisorStateCreateInfoKHR', sType=sType,pNext=pNext,vertexBindingDivisorCount=vertexBindingDivisorCount,pVertexBindingDivisors=pVertexBindingDivisors)

def VkPhysicalDeviceVertexAttributeDivisorPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_PROPERTIES_EXT,pNext=None,maxVertexAttribDivisor=None,):

    return _new('VkPhysicalDeviceVertexAttributeDivisorPropertiesEXT', sType=sType,pNext=pNext,maxVertexAttribDivisor=maxVertexAttribDivisor)

def VkPhysicalDeviceVertexAttributeDivisorPropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_PROPERTIES_KHR,pNext=None,maxVertexAttribDivisor=None,supportsNonZeroFirstInstance=None,):

    return _new('VkPhysicalDeviceVertexAttributeDivisorPropertiesKHR', sType=sType,pNext=pNext,maxVertexAttribDivisor=maxVertexAttribDivisor,supportsNonZeroFirstInstance=supportsNonZeroFirstInstance)

def VkPhysicalDevicePCIBusInfoPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PCI_BUS_INFO_PROPERTIES_EXT,pNext=None,pciDomain=None,pciBus=None,pciDevice=None,pciFunction=None,):

    return _new('VkPhysicalDevicePCIBusInfoPropertiesEXT', sType=sType,pNext=pNext,pciDomain=pciDomain,pciBus=pciBus,pciDevice=pciDevice,pciFunction=pciFunction)

def VkImportAndroidHardwareBufferInfoANDROID(sType=VK_STRUCTURE_TYPE_IMPORT_ANDROID_HARDWARE_BUFFER_INFO_ANDROID,pNext=None,buffer=None,):

    return _new('VkImportAndroidHardwareBufferInfoANDROID', sType=sType,pNext=pNext,buffer=buffer)

def VkAndroidHardwareBufferUsageANDROID(sType=VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_USAGE_ANDROID,pNext=None,androidHardwareBufferUsage=None,):

    return _new('VkAndroidHardwareBufferUsageANDROID', sType=sType,pNext=pNext,androidHardwareBufferUsage=androidHardwareBufferUsage)

def VkAndroidHardwareBufferPropertiesANDROID(sType=VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_PROPERTIES_ANDROID,pNext=None,allocationSize=None,memoryTypeBits=None,):

    return _new('VkAndroidHardwareBufferPropertiesANDROID', sType=sType,pNext=pNext,allocationSize=allocationSize,memoryTypeBits=memoryTypeBits)

def VkMemoryGetAndroidHardwareBufferInfoANDROID(sType=VK_STRUCTURE_TYPE_MEMORY_GET_ANDROID_HARDWARE_BUFFER_INFO_ANDROID,pNext=None,memory=None,):

    return _new('VkMemoryGetAndroidHardwareBufferInfoANDROID', sType=sType,pNext=pNext,memory=memory)

def VkAndroidHardwareBufferFormatPropertiesANDROID(sType=VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_FORMAT_PROPERTIES_ANDROID,pNext=None,format=None,externalFormat=None,formatFeatures=None,samplerYcbcrConversionComponents=None,suggestedYcbcrModel=None,suggestedYcbcrRange=None,suggestedXChromaOffset=None,suggestedYChromaOffset=None,):

    return _new('VkAndroidHardwareBufferFormatPropertiesANDROID', sType=sType,pNext=pNext,format=format,externalFormat=externalFormat,formatFeatures=formatFeatures,samplerYcbcrConversionComponents=samplerYcbcrConversionComponents,suggestedYcbcrModel=suggestedYcbcrModel,suggestedYcbcrRange=suggestedYcbcrRange,suggestedXChromaOffset=suggestedXChromaOffset,suggestedYChromaOffset=suggestedYChromaOffset)

def VkCommandBufferInheritanceConditionalRenderingInfoEXT(sType=VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_CONDITIONAL_RENDERING_INFO_EXT,pNext=None,conditionalRenderingEnable=None,):

    return _new('VkCommandBufferInheritanceConditionalRenderingInfoEXT', sType=sType,pNext=pNext,conditionalRenderingEnable=conditionalRenderingEnable)

def VkExternalFormatANDROID(sType=VK_STRUCTURE_TYPE_EXTERNAL_FORMAT_ANDROID,pNext=None,externalFormat=None,):

    return _new('VkExternalFormatANDROID', sType=sType,pNext=pNext,externalFormat=externalFormat)

def VkPhysicalDevice8BitStorageFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_8BIT_STORAGE_FEATURES,pNext=None,storageBuffer8BitAccess=None,uniformAndStorageBuffer8BitAccess=None,storagePushConstant8=None,):

    return _new('VkPhysicalDevice8BitStorageFeatures', sType=sType,pNext=pNext,storageBuffer8BitAccess=storageBuffer8BitAccess,uniformAndStorageBuffer8BitAccess=uniformAndStorageBuffer8BitAccess,storagePushConstant8=storagePushConstant8)

def VkPhysicalDeviceConditionalRenderingFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONDITIONAL_RENDERING_FEATURES_EXT,pNext=None,conditionalRendering=None,inheritedConditionalRendering=None,):

    return _new('VkPhysicalDeviceConditionalRenderingFeaturesEXT', sType=sType,pNext=pNext,conditionalRendering=conditionalRendering,inheritedConditionalRendering=inheritedConditionalRendering)

def VkPhysicalDeviceVulkanMemoryModelFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_MEMORY_MODEL_FEATURES,pNext=None,vulkanMemoryModel=None,vulkanMemoryModelDeviceScope=None,vulkanMemoryModelAvailabilityVisibilityChains=None,):

    return _new('VkPhysicalDeviceVulkanMemoryModelFeatures', sType=sType,pNext=pNext,vulkanMemoryModel=vulkanMemoryModel,vulkanMemoryModelDeviceScope=vulkanMemoryModelDeviceScope,vulkanMemoryModelAvailabilityVisibilityChains=vulkanMemoryModelAvailabilityVisibilityChains)

def VkPhysicalDeviceShaderAtomicInt64Features(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_INT64_FEATURES,pNext=None,shaderBufferInt64Atomics=None,shaderSharedInt64Atomics=None,):

    return _new('VkPhysicalDeviceShaderAtomicInt64Features', sType=sType,pNext=pNext,shaderBufferInt64Atomics=shaderBufferInt64Atomics,shaderSharedInt64Atomics=shaderSharedInt64Atomics)

def VkPhysicalDeviceShaderAtomicFloatFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_FLOAT_FEATURES_EXT,pNext=None,shaderBufferFloat32Atomics=None,shaderBufferFloat32AtomicAdd=None,shaderBufferFloat64Atomics=None,shaderBufferFloat64AtomicAdd=None,shaderSharedFloat32Atomics=None,shaderSharedFloat32AtomicAdd=None,shaderSharedFloat64Atomics=None,shaderSharedFloat64AtomicAdd=None,shaderImageFloat32Atomics=None,shaderImageFloat32AtomicAdd=None,sparseImageFloat32Atomics=None,sparseImageFloat32AtomicAdd=None,):

    return _new('VkPhysicalDeviceShaderAtomicFloatFeaturesEXT', sType=sType,pNext=pNext,shaderBufferFloat32Atomics=shaderBufferFloat32Atomics,shaderBufferFloat32AtomicAdd=shaderBufferFloat32AtomicAdd,shaderBufferFloat64Atomics=shaderBufferFloat64Atomics,shaderBufferFloat64AtomicAdd=shaderBufferFloat64AtomicAdd,shaderSharedFloat32Atomics=shaderSharedFloat32Atomics,shaderSharedFloat32AtomicAdd=shaderSharedFloat32AtomicAdd,shaderSharedFloat64Atomics=shaderSharedFloat64Atomics,shaderSharedFloat64AtomicAdd=shaderSharedFloat64AtomicAdd,shaderImageFloat32Atomics=shaderImageFloat32Atomics,shaderImageFloat32AtomicAdd=shaderImageFloat32AtomicAdd,sparseImageFloat32Atomics=sparseImageFloat32Atomics,sparseImageFloat32AtomicAdd=sparseImageFloat32AtomicAdd)

def VkPhysicalDeviceShaderAtomicFloat2FeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_FLOAT_2_FEATURES_EXT,pNext=None,shaderBufferFloat16Atomics=None,shaderBufferFloat16AtomicAdd=None,shaderBufferFloat16AtomicMinMax=None,shaderBufferFloat32AtomicMinMax=None,shaderBufferFloat64AtomicMinMax=None,shaderSharedFloat16Atomics=None,shaderSharedFloat16AtomicAdd=None,shaderSharedFloat16AtomicMinMax=None,shaderSharedFloat32AtomicMinMax=None,shaderSharedFloat64AtomicMinMax=None,shaderImageFloat32AtomicMinMax=None,sparseImageFloat32AtomicMinMax=None,):

    return _new('VkPhysicalDeviceShaderAtomicFloat2FeaturesEXT', sType=sType,pNext=pNext,shaderBufferFloat16Atomics=shaderBufferFloat16Atomics,shaderBufferFloat16AtomicAdd=shaderBufferFloat16AtomicAdd,shaderBufferFloat16AtomicMinMax=shaderBufferFloat16AtomicMinMax,shaderBufferFloat32AtomicMinMax=shaderBufferFloat32AtomicMinMax,shaderBufferFloat64AtomicMinMax=shaderBufferFloat64AtomicMinMax,shaderSharedFloat16Atomics=shaderSharedFloat16Atomics,shaderSharedFloat16AtomicAdd=shaderSharedFloat16AtomicAdd,shaderSharedFloat16AtomicMinMax=shaderSharedFloat16AtomicMinMax,shaderSharedFloat32AtomicMinMax=shaderSharedFloat32AtomicMinMax,shaderSharedFloat64AtomicMinMax=shaderSharedFloat64AtomicMinMax,shaderImageFloat32AtomicMinMax=shaderImageFloat32AtomicMinMax,sparseImageFloat32AtomicMinMax=sparseImageFloat32AtomicMinMax)

def VkPhysicalDeviceVertexAttributeDivisorFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_FEATURES_KHR,pNext=None,vertexAttributeInstanceRateDivisor=None,vertexAttributeInstanceRateZeroDivisor=None,):

    return _new('VkPhysicalDeviceVertexAttributeDivisorFeaturesKHR', sType=sType,pNext=pNext,vertexAttributeInstanceRateDivisor=vertexAttributeInstanceRateDivisor,vertexAttributeInstanceRateZeroDivisor=vertexAttributeInstanceRateZeroDivisor)

def VkQueueFamilyCheckpointPropertiesNV(sType=VK_STRUCTURE_TYPE_QUEUE_FAMILY_CHECKPOINT_PROPERTIES_NV,pNext=None,checkpointExecutionStageMask=None,):

    return _new('VkQueueFamilyCheckpointPropertiesNV', sType=sType,pNext=pNext,checkpointExecutionStageMask=checkpointExecutionStageMask)

def VkCheckpointDataNV(sType=VK_STRUCTURE_TYPE_CHECKPOINT_DATA_NV,pNext=None,stage=None,pCheckpointMarker=None,):

    return _new('VkCheckpointDataNV', sType=sType,pNext=pNext,stage=stage,pCheckpointMarker=pCheckpointMarker)

def VkPhysicalDeviceDepthStencilResolveProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_STENCIL_RESOLVE_PROPERTIES,pNext=None,supportedDepthResolveModes=None,supportedStencilResolveModes=None,independentResolveNone=None,independentResolve=None,):

    return _new('VkPhysicalDeviceDepthStencilResolveProperties', sType=sType,pNext=pNext,supportedDepthResolveModes=supportedDepthResolveModes,supportedStencilResolveModes=supportedStencilResolveModes,independentResolveNone=independentResolveNone,independentResolve=independentResolve)

def VkSubpassDescriptionDepthStencilResolve(sType=VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_DEPTH_STENCIL_RESOLVE,pNext=None,depthResolveMode=None,stencilResolveMode=None,pDepthStencilResolveAttachment=None,):

    return _new('VkSubpassDescriptionDepthStencilResolve', sType=sType,pNext=pNext,depthResolveMode=depthResolveMode,stencilResolveMode=stencilResolveMode,pDepthStencilResolveAttachment=pDepthStencilResolveAttachment)

def VkImageViewASTCDecodeModeEXT(sType=VK_STRUCTURE_TYPE_IMAGE_VIEW_ASTC_DECODE_MODE_EXT,pNext=None,decodeMode=None,):

    return _new('VkImageViewASTCDecodeModeEXT', sType=sType,pNext=pNext,decodeMode=decodeMode)

def VkPhysicalDeviceASTCDecodeFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ASTC_DECODE_FEATURES_EXT,pNext=None,decodeModeSharedExponent=None,):

    return _new('VkPhysicalDeviceASTCDecodeFeaturesEXT', sType=sType,pNext=pNext,decodeModeSharedExponent=decodeModeSharedExponent)

def VkPhysicalDeviceTransformFeedbackFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_FEATURES_EXT,pNext=None,transformFeedback=None,geometryStreams=None,):

    return _new('VkPhysicalDeviceTransformFeedbackFeaturesEXT', sType=sType,pNext=pNext,transformFeedback=transformFeedback,geometryStreams=geometryStreams)

def VkPhysicalDeviceTransformFeedbackPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_PROPERTIES_EXT,pNext=None,maxTransformFeedbackStreams=None,maxTransformFeedbackBuffers=None,maxTransformFeedbackBufferSize=None,maxTransformFeedbackStreamDataSize=None,maxTransformFeedbackBufferDataSize=None,maxTransformFeedbackBufferDataStride=None,transformFeedbackQueries=None,transformFeedbackStreamsLinesTriangles=None,transformFeedbackRasterizationStreamSelect=None,transformFeedbackDraw=None,):

    return _new('VkPhysicalDeviceTransformFeedbackPropertiesEXT', sType=sType,pNext=pNext,maxTransformFeedbackStreams=maxTransformFeedbackStreams,maxTransformFeedbackBuffers=maxTransformFeedbackBuffers,maxTransformFeedbackBufferSize=maxTransformFeedbackBufferSize,maxTransformFeedbackStreamDataSize=maxTransformFeedbackStreamDataSize,maxTransformFeedbackBufferDataSize=maxTransformFeedbackBufferDataSize,maxTransformFeedbackBufferDataStride=maxTransformFeedbackBufferDataStride,transformFeedbackQueries=transformFeedbackQueries,transformFeedbackStreamsLinesTriangles=transformFeedbackStreamsLinesTriangles,transformFeedbackRasterizationStreamSelect=transformFeedbackRasterizationStreamSelect,transformFeedbackDraw=transformFeedbackDraw)

def VkPipelineRasterizationStateStreamCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_STREAM_CREATE_INFO_EXT,pNext=None,flags=None,rasterizationStream=None,):

    return _new('VkPipelineRasterizationStateStreamCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,rasterizationStream=rasterizationStream)

def VkPhysicalDeviceRepresentativeFragmentTestFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_REPRESENTATIVE_FRAGMENT_TEST_FEATURES_NV,pNext=None,representativeFragmentTest=None,):

    return _new('VkPhysicalDeviceRepresentativeFragmentTestFeaturesNV', sType=sType,pNext=pNext,representativeFragmentTest=representativeFragmentTest)

def VkPipelineRepresentativeFragmentTestStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_REPRESENTATIVE_FRAGMENT_TEST_STATE_CREATE_INFO_NV,pNext=None,representativeFragmentTestEnable=None,):

    return _new('VkPipelineRepresentativeFragmentTestStateCreateInfoNV', sType=sType,pNext=pNext,representativeFragmentTestEnable=representativeFragmentTestEnable)

def VkPhysicalDeviceExclusiveScissorFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXCLUSIVE_SCISSOR_FEATURES_NV,pNext=None,exclusiveScissor=None,):

    return _new('VkPhysicalDeviceExclusiveScissorFeaturesNV', sType=sType,pNext=pNext,exclusiveScissor=exclusiveScissor)

def VkPipelineViewportExclusiveScissorStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_EXCLUSIVE_SCISSOR_STATE_CREATE_INFO_NV,pNext=None,exclusiveScissorCount=None,pExclusiveScissors=None,):
    if exclusiveScissorCount is None and pExclusiveScissors is not None:
        exclusiveScissorCount = len(pExclusiveScissors)

    return _new('VkPipelineViewportExclusiveScissorStateCreateInfoNV', sType=sType,pNext=pNext,exclusiveScissorCount=exclusiveScissorCount,pExclusiveScissors=pExclusiveScissors)

def VkPhysicalDeviceCornerSampledImageFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CORNER_SAMPLED_IMAGE_FEATURES_NV,pNext=None,cornerSampledImage=None,):

    return _new('VkPhysicalDeviceCornerSampledImageFeaturesNV', sType=sType,pNext=pNext,cornerSampledImage=cornerSampledImage)

def VkPhysicalDeviceComputeShaderDerivativesFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COMPUTE_SHADER_DERIVATIVES_FEATURES_NV,pNext=None,computeDerivativeGroupQuads=None,computeDerivativeGroupLinear=None,):

    return _new('VkPhysicalDeviceComputeShaderDerivativesFeaturesNV', sType=sType,pNext=pNext,computeDerivativeGroupQuads=computeDerivativeGroupQuads,computeDerivativeGroupLinear=computeDerivativeGroupLinear)

def VkPhysicalDeviceShaderImageFootprintFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_IMAGE_FOOTPRINT_FEATURES_NV,pNext=None,imageFootprint=None,):

    return _new('VkPhysicalDeviceShaderImageFootprintFeaturesNV', sType=sType,pNext=pNext,imageFootprint=imageFootprint)

def VkPhysicalDeviceDedicatedAllocationImageAliasingFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEDICATED_ALLOCATION_IMAGE_ALIASING_FEATURES_NV,pNext=None,dedicatedAllocationImageAliasing=None,):

    return _new('VkPhysicalDeviceDedicatedAllocationImageAliasingFeaturesNV', sType=sType,pNext=pNext,dedicatedAllocationImageAliasing=dedicatedAllocationImageAliasing)

def VkPhysicalDeviceCopyMemoryIndirectFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COPY_MEMORY_INDIRECT_FEATURES_NV,pNext=None,indirectCopy=None,):

    return _new('VkPhysicalDeviceCopyMemoryIndirectFeaturesNV', sType=sType,pNext=pNext,indirectCopy=indirectCopy)

def VkPhysicalDeviceCopyMemoryIndirectPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COPY_MEMORY_INDIRECT_PROPERTIES_NV,pNext=None,supportedQueues=None,):

    return _new('VkPhysicalDeviceCopyMemoryIndirectPropertiesNV', sType=sType,pNext=pNext,supportedQueues=supportedQueues)

def VkPhysicalDeviceMemoryDecompressionFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_DECOMPRESSION_FEATURES_NV,pNext=None,memoryDecompression=None,):

    return _new('VkPhysicalDeviceMemoryDecompressionFeaturesNV', sType=sType,pNext=pNext,memoryDecompression=memoryDecompression)

def VkPhysicalDeviceMemoryDecompressionPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_DECOMPRESSION_PROPERTIES_NV,pNext=None,decompressionMethods=None,maxDecompressionIndirectCount=None,):

    return _new('VkPhysicalDeviceMemoryDecompressionPropertiesNV', sType=sType,pNext=pNext,decompressionMethods=decompressionMethods,maxDecompressionIndirectCount=maxDecompressionIndirectCount)

def VkShadingRatePaletteNV(shadingRatePaletteEntryCount=None,pShadingRatePaletteEntries=None,):
    if shadingRatePaletteEntryCount is None and pShadingRatePaletteEntries is not None:
        shadingRatePaletteEntryCount = len(pShadingRatePaletteEntries)

    return _new('VkShadingRatePaletteNV', shadingRatePaletteEntryCount=shadingRatePaletteEntryCount,pShadingRatePaletteEntries=pShadingRatePaletteEntries)

def VkPipelineViewportShadingRateImageStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SHADING_RATE_IMAGE_STATE_CREATE_INFO_NV,pNext=None,shadingRateImageEnable=None,viewportCount=None,pShadingRatePalettes=None,):
    if viewportCount is None and pShadingRatePalettes is not None:
        viewportCount = len(pShadingRatePalettes)

    return _new('VkPipelineViewportShadingRateImageStateCreateInfoNV', sType=sType,pNext=pNext,shadingRateImageEnable=shadingRateImageEnable,viewportCount=viewportCount,pShadingRatePalettes=pShadingRatePalettes)

def VkPhysicalDeviceShadingRateImageFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_FEATURES_NV,pNext=None,shadingRateImage=None,shadingRateCoarseSampleOrder=None,):

    return _new('VkPhysicalDeviceShadingRateImageFeaturesNV', sType=sType,pNext=pNext,shadingRateImage=shadingRateImage,shadingRateCoarseSampleOrder=shadingRateCoarseSampleOrder)

def VkPhysicalDeviceShadingRateImagePropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_PROPERTIES_NV,pNext=None,shadingRateTexelSize=None,shadingRatePaletteSize=None,shadingRateMaxCoarseSamples=None,):

    return _new('VkPhysicalDeviceShadingRateImagePropertiesNV', sType=sType,pNext=pNext,shadingRateTexelSize=shadingRateTexelSize,shadingRatePaletteSize=shadingRatePaletteSize,shadingRateMaxCoarseSamples=shadingRateMaxCoarseSamples)

def VkPhysicalDeviceInvocationMaskFeaturesHUAWEI(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INVOCATION_MASK_FEATURES_HUAWEI,pNext=None,invocationMask=None,):

    return _new('VkPhysicalDeviceInvocationMaskFeaturesHUAWEI', sType=sType,pNext=pNext,invocationMask=invocationMask)

def VkCoarseSampleLocationNV(pixelX=None,pixelY=None,sample=None,):

    return _new('VkCoarseSampleLocationNV', pixelX=pixelX,pixelY=pixelY,sample=sample)

def VkCoarseSampleOrderCustomNV(shadingRate=None,sampleCount=None,sampleLocationCount=None,pSampleLocations=None,):
    if sampleLocationCount is None and pSampleLocations is not None:
        sampleLocationCount = len(pSampleLocations)

    return _new('VkCoarseSampleOrderCustomNV', shadingRate=shadingRate,sampleCount=sampleCount,sampleLocationCount=sampleLocationCount,pSampleLocations=pSampleLocations)

def VkPipelineViewportCoarseSampleOrderStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_COARSE_SAMPLE_ORDER_STATE_CREATE_INFO_NV,pNext=None,sampleOrderType=None,customSampleOrderCount=None,pCustomSampleOrders=None,):
    if customSampleOrderCount is None and pCustomSampleOrders is not None:
        customSampleOrderCount = len(pCustomSampleOrders)

    return _new('VkPipelineViewportCoarseSampleOrderStateCreateInfoNV', sType=sType,pNext=pNext,sampleOrderType=sampleOrderType,customSampleOrderCount=customSampleOrderCount,pCustomSampleOrders=pCustomSampleOrders)

def VkPhysicalDeviceMeshShaderFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_FEATURES_NV,pNext=None,taskShader=None,meshShader=None,):

    return _new('VkPhysicalDeviceMeshShaderFeaturesNV', sType=sType,pNext=pNext,taskShader=taskShader,meshShader=meshShader)

def VkPhysicalDeviceMeshShaderPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_PROPERTIES_NV,pNext=None,maxDrawMeshTasksCount=None,maxTaskWorkGroupInvocations=None,maxTaskWorkGroupSize=None,maxTaskTotalMemorySize=None,maxTaskOutputCount=None,maxMeshWorkGroupInvocations=None,maxMeshWorkGroupSize=None,maxMeshTotalMemorySize=None,maxMeshOutputVertices=None,maxMeshOutputPrimitives=None,maxMeshMultiviewViewCount=None,meshOutputPerVertexGranularity=None,meshOutputPerPrimitiveGranularity=None,):

    return _new('VkPhysicalDeviceMeshShaderPropertiesNV', sType=sType,pNext=pNext,maxDrawMeshTasksCount=maxDrawMeshTasksCount,maxTaskWorkGroupInvocations=maxTaskWorkGroupInvocations,maxTaskWorkGroupSize=maxTaskWorkGroupSize,maxTaskTotalMemorySize=maxTaskTotalMemorySize,maxTaskOutputCount=maxTaskOutputCount,maxMeshWorkGroupInvocations=maxMeshWorkGroupInvocations,maxMeshWorkGroupSize=maxMeshWorkGroupSize,maxMeshTotalMemorySize=maxMeshTotalMemorySize,maxMeshOutputVertices=maxMeshOutputVertices,maxMeshOutputPrimitives=maxMeshOutputPrimitives,maxMeshMultiviewViewCount=maxMeshMultiviewViewCount,meshOutputPerVertexGranularity=meshOutputPerVertexGranularity,meshOutputPerPrimitiveGranularity=meshOutputPerPrimitiveGranularity)

def VkDrawMeshTasksIndirectCommandNV(taskCount=None,firstTask=None,):

    return _new('VkDrawMeshTasksIndirectCommandNV', taskCount=taskCount,firstTask=firstTask)

def VkPhysicalDeviceMeshShaderFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_FEATURES_EXT,pNext=None,taskShader=None,meshShader=None,multiviewMeshShader=None,primitiveFragmentShadingRateMeshShader=None,meshShaderQueries=None,):

    return _new('VkPhysicalDeviceMeshShaderFeaturesEXT', sType=sType,pNext=pNext,taskShader=taskShader,meshShader=meshShader,multiviewMeshShader=multiviewMeshShader,primitiveFragmentShadingRateMeshShader=primitiveFragmentShadingRateMeshShader,meshShaderQueries=meshShaderQueries)

def VkPhysicalDeviceMeshShaderPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_PROPERTIES_EXT,pNext=None,maxTaskWorkGroupTotalCount=None,maxTaskWorkGroupCount=None,maxTaskWorkGroupInvocations=None,maxTaskWorkGroupSize=None,maxTaskPayloadSize=None,maxTaskSharedMemorySize=None,maxTaskPayloadAndSharedMemorySize=None,maxMeshWorkGroupTotalCount=None,maxMeshWorkGroupCount=None,maxMeshWorkGroupInvocations=None,maxMeshWorkGroupSize=None,maxMeshSharedMemorySize=None,maxMeshPayloadAndSharedMemorySize=None,maxMeshOutputMemorySize=None,maxMeshPayloadAndOutputMemorySize=None,maxMeshOutputComponents=None,maxMeshOutputVertices=None,maxMeshOutputPrimitives=None,maxMeshOutputLayers=None,maxMeshMultiviewViewCount=None,meshOutputPerVertexGranularity=None,meshOutputPerPrimitiveGranularity=None,maxPreferredTaskWorkGroupInvocations=None,maxPreferredMeshWorkGroupInvocations=None,prefersLocalInvocationVertexOutput=None,prefersLocalInvocationPrimitiveOutput=None,prefersCompactVertexOutput=None,prefersCompactPrimitiveOutput=None,):

    return _new('VkPhysicalDeviceMeshShaderPropertiesEXT', sType=sType,pNext=pNext,maxTaskWorkGroupTotalCount=maxTaskWorkGroupTotalCount,maxTaskWorkGroupCount=maxTaskWorkGroupCount,maxTaskWorkGroupInvocations=maxTaskWorkGroupInvocations,maxTaskWorkGroupSize=maxTaskWorkGroupSize,maxTaskPayloadSize=maxTaskPayloadSize,maxTaskSharedMemorySize=maxTaskSharedMemorySize,maxTaskPayloadAndSharedMemorySize=maxTaskPayloadAndSharedMemorySize,maxMeshWorkGroupTotalCount=maxMeshWorkGroupTotalCount,maxMeshWorkGroupCount=maxMeshWorkGroupCount,maxMeshWorkGroupInvocations=maxMeshWorkGroupInvocations,maxMeshWorkGroupSize=maxMeshWorkGroupSize,maxMeshSharedMemorySize=maxMeshSharedMemorySize,maxMeshPayloadAndSharedMemorySize=maxMeshPayloadAndSharedMemorySize,maxMeshOutputMemorySize=maxMeshOutputMemorySize,maxMeshPayloadAndOutputMemorySize=maxMeshPayloadAndOutputMemorySize,maxMeshOutputComponents=maxMeshOutputComponents,maxMeshOutputVertices=maxMeshOutputVertices,maxMeshOutputPrimitives=maxMeshOutputPrimitives,maxMeshOutputLayers=maxMeshOutputLayers,maxMeshMultiviewViewCount=maxMeshMultiviewViewCount,meshOutputPerVertexGranularity=meshOutputPerVertexGranularity,meshOutputPerPrimitiveGranularity=meshOutputPerPrimitiveGranularity,maxPreferredTaskWorkGroupInvocations=maxPreferredTaskWorkGroupInvocations,maxPreferredMeshWorkGroupInvocations=maxPreferredMeshWorkGroupInvocations,prefersLocalInvocationVertexOutput=prefersLocalInvocationVertexOutput,prefersLocalInvocationPrimitiveOutput=prefersLocalInvocationPrimitiveOutput,prefersCompactVertexOutput=prefersCompactVertexOutput,prefersCompactPrimitiveOutput=prefersCompactPrimitiveOutput)

def VkDrawMeshTasksIndirectCommandEXT(groupCountX=None,groupCountY=None,groupCountZ=None,):

    return _new('VkDrawMeshTasksIndirectCommandEXT', groupCountX=groupCountX,groupCountY=groupCountY,groupCountZ=groupCountZ)

def VkRayTracingShaderGroupCreateInfoNV(sType=VK_STRUCTURE_TYPE_RAY_TRACING_SHADER_GROUP_CREATE_INFO_NV,pNext=None,type=None,generalShader=None,closestHitShader=None,anyHitShader=None,intersectionShader=None,):

    return _new('VkRayTracingShaderGroupCreateInfoNV', sType=sType,pNext=pNext,type=type,generalShader=generalShader,closestHitShader=closestHitShader,anyHitShader=anyHitShader,intersectionShader=intersectionShader)

def VkRayTracingShaderGroupCreateInfoKHR(sType=VK_STRUCTURE_TYPE_RAY_TRACING_SHADER_GROUP_CREATE_INFO_KHR,pNext=None,type=None,generalShader=None,closestHitShader=None,anyHitShader=None,intersectionShader=None,pShaderGroupCaptureReplayHandle=None,):

    return _new('VkRayTracingShaderGroupCreateInfoKHR', sType=sType,pNext=pNext,type=type,generalShader=generalShader,closestHitShader=closestHitShader,anyHitShader=anyHitShader,intersectionShader=intersectionShader,pShaderGroupCaptureReplayHandle=pShaderGroupCaptureReplayHandle)

def VkRayTracingPipelineCreateInfoNV(sType=VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_CREATE_INFO_NV,pNext=None,flags=None,stageCount=None,pStages=None,groupCount=None,pGroups=None,maxRecursionDepth=None,layout=None,basePipelineHandle=None,basePipelineIndex=None,):
    if stageCount is None and pStages is not None:
        stageCount = len(pStages)
    if groupCount is None and pGroups is not None:
        groupCount = len(pGroups)

    return _new('VkRayTracingPipelineCreateInfoNV', sType=sType,pNext=pNext,flags=flags,stageCount=stageCount,pStages=pStages,groupCount=groupCount,pGroups=pGroups,maxRecursionDepth=maxRecursionDepth,layout=layout,basePipelineHandle=basePipelineHandle,basePipelineIndex=basePipelineIndex)

def VkRayTracingPipelineCreateInfoKHR(sType=VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_CREATE_INFO_KHR,pNext=None,flags=None,stageCount=None,pStages=None,groupCount=None,pGroups=None,maxPipelineRayRecursionDepth=None,pLibraryInfo=None,pLibraryInterface=None,pDynamicState=None,layout=None,basePipelineHandle=None,basePipelineIndex=None,):
    if stageCount is None and pStages is not None:
        stageCount = len(pStages)
    if groupCount is None and pGroups is not None:
        groupCount = len(pGroups)

    return _new('VkRayTracingPipelineCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,stageCount=stageCount,pStages=pStages,groupCount=groupCount,pGroups=pGroups,maxPipelineRayRecursionDepth=maxPipelineRayRecursionDepth,pLibraryInfo=pLibraryInfo,pLibraryInterface=pLibraryInterface,pDynamicState=pDynamicState,layout=layout,basePipelineHandle=basePipelineHandle,basePipelineIndex=basePipelineIndex)

def VkGeometryTrianglesNV(sType=VK_STRUCTURE_TYPE_GEOMETRY_TRIANGLES_NV,pNext=None,vertexData=None,vertexOffset=None,vertexCount=None,vertexStride=None,vertexFormat=None,indexData=None,indexOffset=None,indexCount=None,indexType=None,transformData=None,transformOffset=None,):

    return _new('VkGeometryTrianglesNV', sType=sType,pNext=pNext,vertexData=vertexData,vertexOffset=vertexOffset,vertexCount=vertexCount,vertexStride=vertexStride,vertexFormat=vertexFormat,indexData=indexData,indexOffset=indexOffset,indexCount=indexCount,indexType=indexType,transformData=transformData,transformOffset=transformOffset)

def VkGeometryAABBNV(sType=VK_STRUCTURE_TYPE_GEOMETRY_AABB_NV,pNext=None,aabbData=None,numAABBs=None,stride=None,offset=None,):

    return _new('VkGeometryAABBNV', sType=sType,pNext=pNext,aabbData=aabbData,numAABBs=numAABBs,stride=stride,offset=offset)

def VkGeometryDataNV(triangles=None,aabbs=None,):

    return _new('VkGeometryDataNV', triangles=triangles,aabbs=aabbs)

def VkGeometryNV(sType=VK_STRUCTURE_TYPE_GEOMETRY_NV,pNext=None,geometryType=None,geometry=None,flags=None,):

    return _new('VkGeometryNV', sType=sType,pNext=pNext,geometryType=geometryType,geometry=geometry,flags=flags)

def VkAccelerationStructureInfoNV(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_INFO_NV,pNext=None,type=None,flags=None,instanceCount=None,geometryCount=None,pGeometries=None,):
    if geometryCount is None and pGeometries is not None:
        geometryCount = len(pGeometries)

    return _new('VkAccelerationStructureInfoNV', sType=sType,pNext=pNext,type=type,flags=flags,instanceCount=instanceCount,geometryCount=geometryCount,pGeometries=pGeometries)

def VkAccelerationStructureCreateInfoNV(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_CREATE_INFO_NV,pNext=None,compactedSize=None,info=None,):

    return _new('VkAccelerationStructureCreateInfoNV', sType=sType,pNext=pNext,compactedSize=compactedSize,info=info)

def VkBindAccelerationStructureMemoryInfoNV(sType=VK_STRUCTURE_TYPE_BIND_ACCELERATION_STRUCTURE_MEMORY_INFO_NV,pNext=None,accelerationStructure=None,memory=None,memoryOffset=None,deviceIndexCount=None,pDeviceIndices=None,):
    if deviceIndexCount is None and pDeviceIndices is not None:
        deviceIndexCount = len(pDeviceIndices)

    return _new('VkBindAccelerationStructureMemoryInfoNV', sType=sType,pNext=pNext,accelerationStructure=accelerationStructure,memory=memory,memoryOffset=memoryOffset,deviceIndexCount=deviceIndexCount,pDeviceIndices=pDeviceIndices)

def VkWriteDescriptorSetAccelerationStructureKHR(sType=VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_ACCELERATION_STRUCTURE_KHR,pNext=None,accelerationStructureCount=None,pAccelerationStructures=None,):
    if accelerationStructureCount is None and pAccelerationStructures is not None:
        accelerationStructureCount = len(pAccelerationStructures)

    return _new('VkWriteDescriptorSetAccelerationStructureKHR', sType=sType,pNext=pNext,accelerationStructureCount=accelerationStructureCount,pAccelerationStructures=pAccelerationStructures)

def VkWriteDescriptorSetAccelerationStructureNV(sType=VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_ACCELERATION_STRUCTURE_NV,pNext=None,accelerationStructureCount=None,pAccelerationStructures=None,):
    if accelerationStructureCount is None and pAccelerationStructures is not None:
        accelerationStructureCount = len(pAccelerationStructures)

    return _new('VkWriteDescriptorSetAccelerationStructureNV', sType=sType,pNext=pNext,accelerationStructureCount=accelerationStructureCount,pAccelerationStructures=pAccelerationStructures)

def VkAccelerationStructureMemoryRequirementsInfoNV(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_INFO_NV,pNext=None,type=None,accelerationStructure=None,):

    return _new('VkAccelerationStructureMemoryRequirementsInfoNV', sType=sType,pNext=pNext,type=type,accelerationStructure=accelerationStructure)

def VkPhysicalDeviceAccelerationStructureFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ACCELERATION_STRUCTURE_FEATURES_KHR,pNext=None,accelerationStructure=None,accelerationStructureCaptureReplay=None,accelerationStructureIndirectBuild=None,accelerationStructureHostCommands=None,descriptorBindingAccelerationStructureUpdateAfterBind=None,):

    return _new('VkPhysicalDeviceAccelerationStructureFeaturesKHR', sType=sType,pNext=pNext,accelerationStructure=accelerationStructure,accelerationStructureCaptureReplay=accelerationStructureCaptureReplay,accelerationStructureIndirectBuild=accelerationStructureIndirectBuild,accelerationStructureHostCommands=accelerationStructureHostCommands,descriptorBindingAccelerationStructureUpdateAfterBind=descriptorBindingAccelerationStructureUpdateAfterBind)

def VkPhysicalDeviceRayTracingPipelineFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PIPELINE_FEATURES_KHR,pNext=None,rayTracingPipeline=None,rayTracingPipelineShaderGroupHandleCaptureReplay=None,rayTracingPipelineShaderGroupHandleCaptureReplayMixed=None,rayTracingPipelineTraceRaysIndirect=None,rayTraversalPrimitiveCulling=None,):

    return _new('VkPhysicalDeviceRayTracingPipelineFeaturesKHR', sType=sType,pNext=pNext,rayTracingPipeline=rayTracingPipeline,rayTracingPipelineShaderGroupHandleCaptureReplay=rayTracingPipelineShaderGroupHandleCaptureReplay,rayTracingPipelineShaderGroupHandleCaptureReplayMixed=rayTracingPipelineShaderGroupHandleCaptureReplayMixed,rayTracingPipelineTraceRaysIndirect=rayTracingPipelineTraceRaysIndirect,rayTraversalPrimitiveCulling=rayTraversalPrimitiveCulling)

def VkPhysicalDeviceRayQueryFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_QUERY_FEATURES_KHR,pNext=None,rayQuery=None,):

    return _new('VkPhysicalDeviceRayQueryFeaturesKHR', sType=sType,pNext=pNext,rayQuery=rayQuery)

def VkPhysicalDeviceAccelerationStructurePropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ACCELERATION_STRUCTURE_PROPERTIES_KHR,pNext=None,maxGeometryCount=None,maxInstanceCount=None,maxPrimitiveCount=None,maxPerStageDescriptorAccelerationStructures=None,maxPerStageDescriptorUpdateAfterBindAccelerationStructures=None,maxDescriptorSetAccelerationStructures=None,maxDescriptorSetUpdateAfterBindAccelerationStructures=None,minAccelerationStructureScratchOffsetAlignment=None,):

    return _new('VkPhysicalDeviceAccelerationStructurePropertiesKHR', sType=sType,pNext=pNext,maxGeometryCount=maxGeometryCount,maxInstanceCount=maxInstanceCount,maxPrimitiveCount=maxPrimitiveCount,maxPerStageDescriptorAccelerationStructures=maxPerStageDescriptorAccelerationStructures,maxPerStageDescriptorUpdateAfterBindAccelerationStructures=maxPerStageDescriptorUpdateAfterBindAccelerationStructures,maxDescriptorSetAccelerationStructures=maxDescriptorSetAccelerationStructures,maxDescriptorSetUpdateAfterBindAccelerationStructures=maxDescriptorSetUpdateAfterBindAccelerationStructures,minAccelerationStructureScratchOffsetAlignment=minAccelerationStructureScratchOffsetAlignment)

def VkPhysicalDeviceRayTracingPipelinePropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PIPELINE_PROPERTIES_KHR,pNext=None,shaderGroupHandleSize=None,maxRayRecursionDepth=None,maxShaderGroupStride=None,shaderGroupBaseAlignment=None,shaderGroupHandleCaptureReplaySize=None,maxRayDispatchInvocationCount=None,shaderGroupHandleAlignment=None,maxRayHitAttributeSize=None,):

    return _new('VkPhysicalDeviceRayTracingPipelinePropertiesKHR', sType=sType,pNext=pNext,shaderGroupHandleSize=shaderGroupHandleSize,maxRayRecursionDepth=maxRayRecursionDepth,maxShaderGroupStride=maxShaderGroupStride,shaderGroupBaseAlignment=shaderGroupBaseAlignment,shaderGroupHandleCaptureReplaySize=shaderGroupHandleCaptureReplaySize,maxRayDispatchInvocationCount=maxRayDispatchInvocationCount,shaderGroupHandleAlignment=shaderGroupHandleAlignment,maxRayHitAttributeSize=maxRayHitAttributeSize)

def VkPhysicalDeviceRayTracingPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PROPERTIES_NV,pNext=None,shaderGroupHandleSize=None,maxRecursionDepth=None,maxShaderGroupStride=None,shaderGroupBaseAlignment=None,maxGeometryCount=None,maxInstanceCount=None,maxTriangleCount=None,maxDescriptorSetAccelerationStructures=None,):

    return _new('VkPhysicalDeviceRayTracingPropertiesNV', sType=sType,pNext=pNext,shaderGroupHandleSize=shaderGroupHandleSize,maxRecursionDepth=maxRecursionDepth,maxShaderGroupStride=maxShaderGroupStride,shaderGroupBaseAlignment=shaderGroupBaseAlignment,maxGeometryCount=maxGeometryCount,maxInstanceCount=maxInstanceCount,maxTriangleCount=maxTriangleCount,maxDescriptorSetAccelerationStructures=maxDescriptorSetAccelerationStructures)

def VkStridedDeviceAddressRegionKHR(deviceAddress=None,stride=None,size=None,):

    return _new('VkStridedDeviceAddressRegionKHR', deviceAddress=deviceAddress,stride=stride,size=size)

def VkTraceRaysIndirectCommandKHR(width=None,height=None,depth=None,):

    return _new('VkTraceRaysIndirectCommandKHR', width=width,height=height,depth=depth)

def VkTraceRaysIndirectCommand2KHR(raygenShaderRecordAddress=None,raygenShaderRecordSize=None,missShaderBindingTableAddress=None,missShaderBindingTableSize=None,missShaderBindingTableStride=None,hitShaderBindingTableAddress=None,hitShaderBindingTableSize=None,hitShaderBindingTableStride=None,callableShaderBindingTableAddress=None,callableShaderBindingTableSize=None,callableShaderBindingTableStride=None,width=None,height=None,depth=None,):

    return _new('VkTraceRaysIndirectCommand2KHR', raygenShaderRecordAddress=raygenShaderRecordAddress,raygenShaderRecordSize=raygenShaderRecordSize,missShaderBindingTableAddress=missShaderBindingTableAddress,missShaderBindingTableSize=missShaderBindingTableSize,missShaderBindingTableStride=missShaderBindingTableStride,hitShaderBindingTableAddress=hitShaderBindingTableAddress,hitShaderBindingTableSize=hitShaderBindingTableSize,hitShaderBindingTableStride=hitShaderBindingTableStride,callableShaderBindingTableAddress=callableShaderBindingTableAddress,callableShaderBindingTableSize=callableShaderBindingTableSize,callableShaderBindingTableStride=callableShaderBindingTableStride,width=width,height=height,depth=depth)

def VkPhysicalDeviceRayTracingMaintenance1FeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_MAINTENANCE_1_FEATURES_KHR,pNext=None,rayTracingMaintenance1=None,rayTracingPipelineTraceRaysIndirect2=None,):

    return _new('VkPhysicalDeviceRayTracingMaintenance1FeaturesKHR', sType=sType,pNext=pNext,rayTracingMaintenance1=rayTracingMaintenance1,rayTracingPipelineTraceRaysIndirect2=rayTracingPipelineTraceRaysIndirect2)

def VkDrmFormatModifierPropertiesListEXT(sType=VK_STRUCTURE_TYPE_DRM_FORMAT_MODIFIER_PROPERTIES_LIST_EXT,pNext=None,drmFormatModifierCount=None,pDrmFormatModifierProperties=None,):
    if drmFormatModifierCount is None and pDrmFormatModifierProperties is not None:
        drmFormatModifierCount = len(pDrmFormatModifierProperties)

    return _new('VkDrmFormatModifierPropertiesListEXT', sType=sType,pNext=pNext,drmFormatModifierCount=drmFormatModifierCount,pDrmFormatModifierProperties=pDrmFormatModifierProperties)

def VkDrmFormatModifierPropertiesEXT(drmFormatModifier=None,drmFormatModifierPlaneCount=None,drmFormatModifierTilingFeatures=None,):

    return _new('VkDrmFormatModifierPropertiesEXT', drmFormatModifier=drmFormatModifier,drmFormatModifierPlaneCount=drmFormatModifierPlaneCount,drmFormatModifierTilingFeatures=drmFormatModifierTilingFeatures)

def VkPhysicalDeviceImageDrmFormatModifierInfoEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_DRM_FORMAT_MODIFIER_INFO_EXT,pNext=None,drmFormatModifier=None,sharingMode=None,queueFamilyIndexCount=None,pQueueFamilyIndices=None,):
    if queueFamilyIndexCount is None and pQueueFamilyIndices is not None:
        queueFamilyIndexCount = len(pQueueFamilyIndices)

    return _new('VkPhysicalDeviceImageDrmFormatModifierInfoEXT', sType=sType,pNext=pNext,drmFormatModifier=drmFormatModifier,sharingMode=sharingMode,queueFamilyIndexCount=queueFamilyIndexCount,pQueueFamilyIndices=pQueueFamilyIndices)

def VkImageDrmFormatModifierListCreateInfoEXT(sType=VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_LIST_CREATE_INFO_EXT,pNext=None,drmFormatModifierCount=None,pDrmFormatModifiers=None,):
    if drmFormatModifierCount is None and pDrmFormatModifiers is not None:
        drmFormatModifierCount = len(pDrmFormatModifiers)

    return _new('VkImageDrmFormatModifierListCreateInfoEXT', sType=sType,pNext=pNext,drmFormatModifierCount=drmFormatModifierCount,pDrmFormatModifiers=pDrmFormatModifiers)

def VkImageDrmFormatModifierExplicitCreateInfoEXT(sType=VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_EXPLICIT_CREATE_INFO_EXT,pNext=None,drmFormatModifier=None,drmFormatModifierPlaneCount=None,pPlaneLayouts=None,):
    if drmFormatModifierPlaneCount is None and pPlaneLayouts is not None:
        drmFormatModifierPlaneCount = len(pPlaneLayouts)

    return _new('VkImageDrmFormatModifierExplicitCreateInfoEXT', sType=sType,pNext=pNext,drmFormatModifier=drmFormatModifier,drmFormatModifierPlaneCount=drmFormatModifierPlaneCount,pPlaneLayouts=pPlaneLayouts)

def VkImageDrmFormatModifierPropertiesEXT(sType=VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_PROPERTIES_EXT,pNext=None,drmFormatModifier=None,):

    return _new('VkImageDrmFormatModifierPropertiesEXT', sType=sType,pNext=pNext,drmFormatModifier=drmFormatModifier)

def VkImageStencilUsageCreateInfo(sType=VK_STRUCTURE_TYPE_IMAGE_STENCIL_USAGE_CREATE_INFO,pNext=None,stencilUsage=None,):

    return _new('VkImageStencilUsageCreateInfo', sType=sType,pNext=pNext,stencilUsage=stencilUsage)

def VkDeviceMemoryOverallocationCreateInfoAMD(sType=VK_STRUCTURE_TYPE_DEVICE_MEMORY_OVERALLOCATION_CREATE_INFO_AMD,pNext=None,overallocationBehavior=None,):

    return _new('VkDeviceMemoryOverallocationCreateInfoAMD', sType=sType,pNext=pNext,overallocationBehavior=overallocationBehavior)

def VkPhysicalDeviceFragmentDensityMapFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_FEATURES_EXT,pNext=None,fragmentDensityMap=None,fragmentDensityMapDynamic=None,fragmentDensityMapNonSubsampledImages=None,):

    return _new('VkPhysicalDeviceFragmentDensityMapFeaturesEXT', sType=sType,pNext=pNext,fragmentDensityMap=fragmentDensityMap,fragmentDensityMapDynamic=fragmentDensityMapDynamic,fragmentDensityMapNonSubsampledImages=fragmentDensityMapNonSubsampledImages)

def VkPhysicalDeviceFragmentDensityMap2FeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_2_FEATURES_EXT,pNext=None,fragmentDensityMapDeferred=None,):

    return _new('VkPhysicalDeviceFragmentDensityMap2FeaturesEXT', sType=sType,pNext=pNext,fragmentDensityMapDeferred=fragmentDensityMapDeferred)

def VkPhysicalDeviceFragmentDensityMapOffsetFeaturesQCOM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_OFFSET_FEATURES_QCOM,pNext=None,fragmentDensityMapOffset=None,):

    return _new('VkPhysicalDeviceFragmentDensityMapOffsetFeaturesQCOM', sType=sType,pNext=pNext,fragmentDensityMapOffset=fragmentDensityMapOffset)

def VkPhysicalDeviceFragmentDensityMapPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_PROPERTIES_EXT,pNext=None,minFragmentDensityTexelSize=None,maxFragmentDensityTexelSize=None,fragmentDensityInvocations=None,):

    return _new('VkPhysicalDeviceFragmentDensityMapPropertiesEXT', sType=sType,pNext=pNext,minFragmentDensityTexelSize=minFragmentDensityTexelSize,maxFragmentDensityTexelSize=maxFragmentDensityTexelSize,fragmentDensityInvocations=fragmentDensityInvocations)

def VkPhysicalDeviceFragmentDensityMap2PropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_2_PROPERTIES_EXT,pNext=None,subsampledLoads=None,subsampledCoarseReconstructionEarlyAccess=None,maxSubsampledArrayLayers=None,maxDescriptorSetSubsampledSamplers=None,):

    return _new('VkPhysicalDeviceFragmentDensityMap2PropertiesEXT', sType=sType,pNext=pNext,subsampledLoads=subsampledLoads,subsampledCoarseReconstructionEarlyAccess=subsampledCoarseReconstructionEarlyAccess,maxSubsampledArrayLayers=maxSubsampledArrayLayers,maxDescriptorSetSubsampledSamplers=maxDescriptorSetSubsampledSamplers)

def VkPhysicalDeviceFragmentDensityMapOffsetPropertiesQCOM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_OFFSET_PROPERTIES_QCOM,pNext=None,fragmentDensityOffsetGranularity=None,):

    return _new('VkPhysicalDeviceFragmentDensityMapOffsetPropertiesQCOM', sType=sType,pNext=pNext,fragmentDensityOffsetGranularity=fragmentDensityOffsetGranularity)

def VkRenderPassFragmentDensityMapCreateInfoEXT(sType=VK_STRUCTURE_TYPE_RENDER_PASS_FRAGMENT_DENSITY_MAP_CREATE_INFO_EXT,pNext=None,fragmentDensityMapAttachment=None,):

    return _new('VkRenderPassFragmentDensityMapCreateInfoEXT', sType=sType,pNext=pNext,fragmentDensityMapAttachment=fragmentDensityMapAttachment)

def VkSubpassFragmentDensityMapOffsetEndInfoQCOM(sType=VK_STRUCTURE_TYPE_SUBPASS_FRAGMENT_DENSITY_MAP_OFFSET_END_INFO_QCOM,pNext=None,fragmentDensityOffsetCount=None,pFragmentDensityOffsets=None,):
    if fragmentDensityOffsetCount is None and pFragmentDensityOffsets is not None:
        fragmentDensityOffsetCount = len(pFragmentDensityOffsets)

    return _new('VkSubpassFragmentDensityMapOffsetEndInfoQCOM', sType=sType,pNext=pNext,fragmentDensityOffsetCount=fragmentDensityOffsetCount,pFragmentDensityOffsets=pFragmentDensityOffsets)

def VkPhysicalDeviceScalarBlockLayoutFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SCALAR_BLOCK_LAYOUT_FEATURES,pNext=None,scalarBlockLayout=None,):

    return _new('VkPhysicalDeviceScalarBlockLayoutFeatures', sType=sType,pNext=pNext,scalarBlockLayout=scalarBlockLayout)

def VkSurfaceProtectedCapabilitiesKHR(sType=VK_STRUCTURE_TYPE_SURFACE_PROTECTED_CAPABILITIES_KHR,pNext=None,supportsProtected=None,):

    return _new('VkSurfaceProtectedCapabilitiesKHR', sType=sType,pNext=pNext,supportsProtected=supportsProtected)

def VkPhysicalDeviceUniformBufferStandardLayoutFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_UNIFORM_BUFFER_STANDARD_LAYOUT_FEATURES,pNext=None,uniformBufferStandardLayout=None,):

    return _new('VkPhysicalDeviceUniformBufferStandardLayoutFeatures', sType=sType,pNext=pNext,uniformBufferStandardLayout=uniformBufferStandardLayout)

def VkPhysicalDeviceDepthClipEnableFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_CLIP_ENABLE_FEATURES_EXT,pNext=None,depthClipEnable=None,):

    return _new('VkPhysicalDeviceDepthClipEnableFeaturesEXT', sType=sType,pNext=pNext,depthClipEnable=depthClipEnable)

def VkPipelineRasterizationDepthClipStateCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_DEPTH_CLIP_STATE_CREATE_INFO_EXT,pNext=None,flags=None,depthClipEnable=None,):

    return _new('VkPipelineRasterizationDepthClipStateCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,depthClipEnable=depthClipEnable)

def VkPhysicalDeviceMemoryBudgetPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_BUDGET_PROPERTIES_EXT,pNext=None,heapBudget=None,heapUsage=None,):

    return _new('VkPhysicalDeviceMemoryBudgetPropertiesEXT', sType=sType,pNext=pNext,heapBudget=heapBudget,heapUsage=heapUsage)

def VkPhysicalDeviceMemoryPriorityFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PRIORITY_FEATURES_EXT,pNext=None,memoryPriority=None,):

    return _new('VkPhysicalDeviceMemoryPriorityFeaturesEXT', sType=sType,pNext=pNext,memoryPriority=memoryPriority)

def VkMemoryPriorityAllocateInfoEXT(sType=VK_STRUCTURE_TYPE_MEMORY_PRIORITY_ALLOCATE_INFO_EXT,pNext=None,priority=None,):

    return _new('VkMemoryPriorityAllocateInfoEXT', sType=sType,pNext=pNext,priority=priority)

def VkPhysicalDevicePageableDeviceLocalMemoryFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PAGEABLE_DEVICE_LOCAL_MEMORY_FEATURES_EXT,pNext=None,pageableDeviceLocalMemory=None,):

    return _new('VkPhysicalDevicePageableDeviceLocalMemoryFeaturesEXT', sType=sType,pNext=pNext,pageableDeviceLocalMemory=pageableDeviceLocalMemory)

def VkPhysicalDeviceBufferDeviceAddressFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BUFFER_DEVICE_ADDRESS_FEATURES,pNext=None,bufferDeviceAddress=None,bufferDeviceAddressCaptureReplay=None,bufferDeviceAddressMultiDevice=None,):

    return _new('VkPhysicalDeviceBufferDeviceAddressFeatures', sType=sType,pNext=pNext,bufferDeviceAddress=bufferDeviceAddress,bufferDeviceAddressCaptureReplay=bufferDeviceAddressCaptureReplay,bufferDeviceAddressMultiDevice=bufferDeviceAddressMultiDevice)

def VkPhysicalDeviceBufferDeviceAddressFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BUFFER_DEVICE_ADDRESS_FEATURES_EXT,pNext=None,bufferDeviceAddress=None,bufferDeviceAddressCaptureReplay=None,bufferDeviceAddressMultiDevice=None,):

    return _new('VkPhysicalDeviceBufferDeviceAddressFeaturesEXT', sType=sType,pNext=pNext,bufferDeviceAddress=bufferDeviceAddress,bufferDeviceAddressCaptureReplay=bufferDeviceAddressCaptureReplay,bufferDeviceAddressMultiDevice=bufferDeviceAddressMultiDevice)

def VkBufferDeviceAddressInfo(sType=VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_INFO,pNext=None,buffer=None,):

    return _new('VkBufferDeviceAddressInfo', sType=sType,pNext=pNext,buffer=buffer)

def VkBufferOpaqueCaptureAddressCreateInfo(sType=VK_STRUCTURE_TYPE_BUFFER_OPAQUE_CAPTURE_ADDRESS_CREATE_INFO,pNext=None,opaqueCaptureAddress=None,):

    return _new('VkBufferOpaqueCaptureAddressCreateInfo', sType=sType,pNext=pNext,opaqueCaptureAddress=opaqueCaptureAddress)

def VkBufferDeviceAddressCreateInfoEXT(sType=VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_CREATE_INFO_EXT,pNext=None,deviceAddress=None,):

    return _new('VkBufferDeviceAddressCreateInfoEXT', sType=sType,pNext=pNext,deviceAddress=deviceAddress)

def VkPhysicalDeviceImageViewImageFormatInfoEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_VIEW_IMAGE_FORMAT_INFO_EXT,pNext=None,imageViewType=None,):

    return _new('VkPhysicalDeviceImageViewImageFormatInfoEXT', sType=sType,pNext=pNext,imageViewType=imageViewType)

def VkFilterCubicImageViewImageFormatPropertiesEXT(sType=VK_STRUCTURE_TYPE_FILTER_CUBIC_IMAGE_VIEW_IMAGE_FORMAT_PROPERTIES_EXT,pNext=None,filterCubic=None,filterCubicMinmax=None,):

    return _new('VkFilterCubicImageViewImageFormatPropertiesEXT', sType=sType,pNext=pNext,filterCubic=filterCubic,filterCubicMinmax=filterCubicMinmax)

def VkPhysicalDeviceImagelessFramebufferFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGELESS_FRAMEBUFFER_FEATURES,pNext=None,imagelessFramebuffer=None,):

    return _new('VkPhysicalDeviceImagelessFramebufferFeatures', sType=sType,pNext=pNext,imagelessFramebuffer=imagelessFramebuffer)

def VkFramebufferAttachmentsCreateInfo(sType=VK_STRUCTURE_TYPE_FRAMEBUFFER_ATTACHMENTS_CREATE_INFO,pNext=None,attachmentImageInfoCount=None,pAttachmentImageInfos=None,):
    if attachmentImageInfoCount is None and pAttachmentImageInfos is not None:
        attachmentImageInfoCount = len(pAttachmentImageInfos)

    return _new('VkFramebufferAttachmentsCreateInfo', sType=sType,pNext=pNext,attachmentImageInfoCount=attachmentImageInfoCount,pAttachmentImageInfos=pAttachmentImageInfos)

def VkFramebufferAttachmentImageInfo(sType=VK_STRUCTURE_TYPE_FRAMEBUFFER_ATTACHMENT_IMAGE_INFO,pNext=None,flags=None,usage=None,width=None,height=None,layerCount=None,viewFormatCount=None,pViewFormats=None,):
    if viewFormatCount is None and pViewFormats is not None:
        viewFormatCount = len(pViewFormats)

    return _new('VkFramebufferAttachmentImageInfo', sType=sType,pNext=pNext,flags=flags,usage=usage,width=width,height=height,layerCount=layerCount,viewFormatCount=viewFormatCount,pViewFormats=pViewFormats)

def VkRenderPassAttachmentBeginInfo(sType=VK_STRUCTURE_TYPE_RENDER_PASS_ATTACHMENT_BEGIN_INFO,pNext=None,attachmentCount=None,pAttachments=None,):
    if attachmentCount is None and pAttachments is not None:
        attachmentCount = len(pAttachments)

    return _new('VkRenderPassAttachmentBeginInfo', sType=sType,pNext=pNext,attachmentCount=attachmentCount,pAttachments=pAttachments)

def VkPhysicalDeviceTextureCompressionASTCHDRFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXTURE_COMPRESSION_ASTC_HDR_FEATURES,pNext=None,textureCompressionASTC_HDR=None,):

    return _new('VkPhysicalDeviceTextureCompressionASTCHDRFeatures', sType=sType,pNext=pNext,textureCompressionASTC_HDR=textureCompressionASTC_HDR)

def VkPhysicalDeviceCooperativeMatrixFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COOPERATIVE_MATRIX_FEATURES_NV,pNext=None,cooperativeMatrix=None,cooperativeMatrixRobustBufferAccess=None,):

    return _new('VkPhysicalDeviceCooperativeMatrixFeaturesNV', sType=sType,pNext=pNext,cooperativeMatrix=cooperativeMatrix,cooperativeMatrixRobustBufferAccess=cooperativeMatrixRobustBufferAccess)

def VkPhysicalDeviceCooperativeMatrixPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COOPERATIVE_MATRIX_PROPERTIES_NV,pNext=None,cooperativeMatrixSupportedStages=None,):

    return _new('VkPhysicalDeviceCooperativeMatrixPropertiesNV', sType=sType,pNext=pNext,cooperativeMatrixSupportedStages=cooperativeMatrixSupportedStages)

def VkCooperativeMatrixPropertiesNV(sType=VK_STRUCTURE_TYPE_COOPERATIVE_MATRIX_PROPERTIES_NV,pNext=None,MSize=None,NSize=None,KSize=None,AType=None,BType=None,CType=None,DType=None,scope=None,):

    return _new('VkCooperativeMatrixPropertiesNV', sType=sType,pNext=pNext,MSize=MSize,NSize=NSize,KSize=KSize,AType=AType,BType=BType,CType=CType,DType=DType,scope=scope)

def VkPhysicalDeviceYcbcrImageArraysFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_YCBCR_IMAGE_ARRAYS_FEATURES_EXT,pNext=None,ycbcrImageArrays=None,):

    return _new('VkPhysicalDeviceYcbcrImageArraysFeaturesEXT', sType=sType,pNext=pNext,ycbcrImageArrays=ycbcrImageArrays)

def VkImageViewHandleInfoNVX(sType=VK_STRUCTURE_TYPE_IMAGE_VIEW_HANDLE_INFO_NVX,pNext=None,imageView=None,descriptorType=None,sampler=None,):

    return _new('VkImageViewHandleInfoNVX', sType=sType,pNext=pNext,imageView=imageView,descriptorType=descriptorType,sampler=sampler)

def VkImageViewAddressPropertiesNVX(sType=VK_STRUCTURE_TYPE_IMAGE_VIEW_ADDRESS_PROPERTIES_NVX,pNext=None,deviceAddress=None,size=None,):

    return _new('VkImageViewAddressPropertiesNVX', sType=sType,pNext=pNext,deviceAddress=deviceAddress,size=size)

def VkPresentFrameTokenGGP(sType=VK_STRUCTURE_TYPE_PRESENT_FRAME_TOKEN_GGP,pNext=None,frameToken=None,):

    return _new('VkPresentFrameTokenGGP', sType=sType,pNext=pNext,frameToken=frameToken)

def VkPipelineCreationFeedback(flags=None,duration=None,):

    return _new('VkPipelineCreationFeedback', flags=flags,duration=duration)

def VkPipelineCreationFeedbackCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_CREATION_FEEDBACK_CREATE_INFO,pNext=None,pPipelineCreationFeedback=None,pipelineStageCreationFeedbackCount=None,pPipelineStageCreationFeedbacks=None,):
    if pipelineStageCreationFeedbackCount is None and pPipelineStageCreationFeedbacks is not None:
        pipelineStageCreationFeedbackCount = len(pPipelineStageCreationFeedbacks)

    return _new('VkPipelineCreationFeedbackCreateInfo', sType=sType,pNext=pNext,pPipelineCreationFeedback=pPipelineCreationFeedback,pipelineStageCreationFeedbackCount=pipelineStageCreationFeedbackCount,pPipelineStageCreationFeedbacks=pPipelineStageCreationFeedbacks)

def VkSurfaceFullScreenExclusiveInfoEXT(sType=VK_STRUCTURE_TYPE_SURFACE_FULL_SCREEN_EXCLUSIVE_INFO_EXT,pNext=None,fullScreenExclusive=None,):

    return _new('VkSurfaceFullScreenExclusiveInfoEXT', sType=sType,pNext=pNext,fullScreenExclusive=fullScreenExclusive)

def VkSurfaceFullScreenExclusiveWin32InfoEXT(sType=VK_STRUCTURE_TYPE_SURFACE_FULL_SCREEN_EXCLUSIVE_WIN32_INFO_EXT,pNext=None,hmonitor=None,):

    return _new('VkSurfaceFullScreenExclusiveWin32InfoEXT', sType=sType,pNext=pNext,hmonitor=hmonitor)

def VkSurfaceCapabilitiesFullScreenExclusiveEXT(sType=VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_FULL_SCREEN_EXCLUSIVE_EXT,pNext=None,fullScreenExclusiveSupported=None,):

    return _new('VkSurfaceCapabilitiesFullScreenExclusiveEXT', sType=sType,pNext=pNext,fullScreenExclusiveSupported=fullScreenExclusiveSupported)

def VkPhysicalDevicePresentBarrierFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRESENT_BARRIER_FEATURES_NV,pNext=None,presentBarrier=None,):

    return _new('VkPhysicalDevicePresentBarrierFeaturesNV', sType=sType,pNext=pNext,presentBarrier=presentBarrier)

def VkSurfaceCapabilitiesPresentBarrierNV(sType=VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_PRESENT_BARRIER_NV,pNext=None,presentBarrierSupported=None,):

    return _new('VkSurfaceCapabilitiesPresentBarrierNV', sType=sType,pNext=pNext,presentBarrierSupported=presentBarrierSupported)

def VkSwapchainPresentBarrierCreateInfoNV(sType=VK_STRUCTURE_TYPE_SWAPCHAIN_PRESENT_BARRIER_CREATE_INFO_NV,pNext=None,presentBarrierEnable=None,):

    return _new('VkSwapchainPresentBarrierCreateInfoNV', sType=sType,pNext=pNext,presentBarrierEnable=presentBarrierEnable)

def VkPhysicalDevicePerformanceQueryFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PERFORMANCE_QUERY_FEATURES_KHR,pNext=None,performanceCounterQueryPools=None,performanceCounterMultipleQueryPools=None,):

    return _new('VkPhysicalDevicePerformanceQueryFeaturesKHR', sType=sType,pNext=pNext,performanceCounterQueryPools=performanceCounterQueryPools,performanceCounterMultipleQueryPools=performanceCounterMultipleQueryPools)

def VkPhysicalDevicePerformanceQueryPropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PERFORMANCE_QUERY_PROPERTIES_KHR,pNext=None,allowCommandBufferQueryCopies=None,):

    return _new('VkPhysicalDevicePerformanceQueryPropertiesKHR', sType=sType,pNext=pNext,allowCommandBufferQueryCopies=allowCommandBufferQueryCopies)

def VkPerformanceCounterKHR(sType=VK_STRUCTURE_TYPE_PERFORMANCE_COUNTER_KHR,pNext=None,unit=None,scope=None,storage=None,uuid=None,):

    return _new('VkPerformanceCounterKHR', sType=sType,pNext=pNext,unit=unit,scope=scope,storage=storage,uuid=uuid)

def VkPerformanceCounterDescriptionKHR(sType=VK_STRUCTURE_TYPE_PERFORMANCE_COUNTER_DESCRIPTION_KHR,pNext=None,flags=None,name=None,category=None,description=None,):

    return _new('VkPerformanceCounterDescriptionKHR', sType=sType,pNext=pNext,flags=flags,name=name,category=category,description=description)

def VkQueryPoolPerformanceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_QUERY_POOL_PERFORMANCE_CREATE_INFO_KHR,pNext=None,queueFamilyIndex=None,counterIndexCount=None,pCounterIndices=None,):
    if counterIndexCount is None and pCounterIndices is not None:
        counterIndexCount = len(pCounterIndices)

    return _new('VkQueryPoolPerformanceCreateInfoKHR', sType=sType,pNext=pNext,queueFamilyIndex=queueFamilyIndex,counterIndexCount=counterIndexCount,pCounterIndices=pCounterIndices)

def VkPerformanceCounterResultKHR(int32=None,int64=None,uint32=None,uint64=None,float32=None,float64=None,):

    return _new('VkPerformanceCounterResultKHR', int32=int32,int64=int64,uint32=uint32,uint64=uint64,float32=float32,float64=float64)

def VkAcquireProfilingLockInfoKHR(sType=VK_STRUCTURE_TYPE_ACQUIRE_PROFILING_LOCK_INFO_KHR,pNext=None,flags=None,timeout=None,):

    return _new('VkAcquireProfilingLockInfoKHR', sType=sType,pNext=pNext,flags=flags,timeout=timeout)

def VkPerformanceQuerySubmitInfoKHR(sType=VK_STRUCTURE_TYPE_PERFORMANCE_QUERY_SUBMIT_INFO_KHR,pNext=None,counterPassIndex=None,):

    return _new('VkPerformanceQuerySubmitInfoKHR', sType=sType,pNext=pNext,counterPassIndex=counterPassIndex)

def VkPerformanceQueryReservationInfoKHR(sType=VK_STRUCTURE_TYPE_PERFORMANCE_QUERY_RESERVATION_INFO_KHR,pNext=None,maxPerformanceQueriesPerPool=None,):

    return _new('VkPerformanceQueryReservationInfoKHR', sType=sType,pNext=pNext,maxPerformanceQueriesPerPool=maxPerformanceQueriesPerPool)

def VkHeadlessSurfaceCreateInfoEXT(sType=VK_STRUCTURE_TYPE_HEADLESS_SURFACE_CREATE_INFO_EXT,pNext=None,flags=None,):

    return _new('VkHeadlessSurfaceCreateInfoEXT', sType=sType,pNext=pNext,flags=flags)

def VkPhysicalDeviceCoverageReductionModeFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COVERAGE_REDUCTION_MODE_FEATURES_NV,pNext=None,coverageReductionMode=None,):

    return _new('VkPhysicalDeviceCoverageReductionModeFeaturesNV', sType=sType,pNext=pNext,coverageReductionMode=coverageReductionMode)

def VkPipelineCoverageReductionStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_REDUCTION_STATE_CREATE_INFO_NV,pNext=None,flags=None,coverageReductionMode=None,):

    return _new('VkPipelineCoverageReductionStateCreateInfoNV', sType=sType,pNext=pNext,flags=flags,coverageReductionMode=coverageReductionMode)

def VkFramebufferMixedSamplesCombinationNV(sType=VK_STRUCTURE_TYPE_FRAMEBUFFER_MIXED_SAMPLES_COMBINATION_NV,pNext=None,coverageReductionMode=None,rasterizationSamples=None,depthStencilSamples=None,colorSamples=None,):

    return _new('VkFramebufferMixedSamplesCombinationNV', sType=sType,pNext=pNext,coverageReductionMode=coverageReductionMode,rasterizationSamples=rasterizationSamples,depthStencilSamples=depthStencilSamples,colorSamples=colorSamples)

def VkPhysicalDeviceShaderIntegerFunctions2FeaturesINTEL(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_INTEGER_FUNCTIONS_2_FEATURES_INTEL,pNext=None,shaderIntegerFunctions2=None,):

    return _new('VkPhysicalDeviceShaderIntegerFunctions2FeaturesINTEL', sType=sType,pNext=pNext,shaderIntegerFunctions2=shaderIntegerFunctions2)

def VkPerformanceValueDataINTEL(value32=None,value64=None,valueFloat=None,valueBool=None,valueString=None,):

    return _new('VkPerformanceValueDataINTEL', value32=value32,value64=value64,valueFloat=valueFloat,valueBool=valueBool,valueString=valueString)

def VkPerformanceValueINTEL(type=None,data=None,):

    return _new('VkPerformanceValueINTEL', type=type,data=data)

def VkInitializePerformanceApiInfoINTEL(sType=VK_STRUCTURE_TYPE_INITIALIZE_PERFORMANCE_API_INFO_INTEL,pNext=None,pUserData=None,):

    return _new('VkInitializePerformanceApiInfoINTEL', sType=sType,pNext=pNext,pUserData=pUserData)

def VkQueryPoolPerformanceQueryCreateInfoINTEL(sType=VK_STRUCTURE_TYPE_QUERY_POOL_PERFORMANCE_QUERY_CREATE_INFO_INTEL,pNext=None,performanceCountersSampling=None,):

    return _new('VkQueryPoolPerformanceQueryCreateInfoINTEL', sType=sType,pNext=pNext,performanceCountersSampling=performanceCountersSampling)

def VkPerformanceMarkerInfoINTEL(sType=VK_STRUCTURE_TYPE_PERFORMANCE_MARKER_INFO_INTEL,pNext=None,marker=None,):

    return _new('VkPerformanceMarkerInfoINTEL', sType=sType,pNext=pNext,marker=marker)

def VkPerformanceStreamMarkerInfoINTEL(sType=VK_STRUCTURE_TYPE_PERFORMANCE_STREAM_MARKER_INFO_INTEL,pNext=None,marker=None,):

    return _new('VkPerformanceStreamMarkerInfoINTEL', sType=sType,pNext=pNext,marker=marker)

def VkPerformanceOverrideInfoINTEL(sType=VK_STRUCTURE_TYPE_PERFORMANCE_OVERRIDE_INFO_INTEL,pNext=None,type=None,enable=None,parameter=None,):

    return _new('VkPerformanceOverrideInfoINTEL', sType=sType,pNext=pNext,type=type,enable=enable,parameter=parameter)

def VkPerformanceConfigurationAcquireInfoINTEL(sType=VK_STRUCTURE_TYPE_PERFORMANCE_CONFIGURATION_ACQUIRE_INFO_INTEL,pNext=None,type=None,):

    return _new('VkPerformanceConfigurationAcquireInfoINTEL', sType=sType,pNext=pNext,type=type)

def VkPhysicalDeviceShaderClockFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CLOCK_FEATURES_KHR,pNext=None,shaderSubgroupClock=None,shaderDeviceClock=None,):

    return _new('VkPhysicalDeviceShaderClockFeaturesKHR', sType=sType,pNext=pNext,shaderSubgroupClock=shaderSubgroupClock,shaderDeviceClock=shaderDeviceClock)

def VkPhysicalDeviceIndexTypeUint8FeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INDEX_TYPE_UINT8_FEATURES_EXT,pNext=None,indexTypeUint8=None,):

    return _new('VkPhysicalDeviceIndexTypeUint8FeaturesEXT', sType=sType,pNext=pNext,indexTypeUint8=indexTypeUint8)

def VkPhysicalDeviceShaderSMBuiltinsPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_SM_BUILTINS_PROPERTIES_NV,pNext=None,shaderSMCount=None,shaderWarpsPerSM=None,):

    return _new('VkPhysicalDeviceShaderSMBuiltinsPropertiesNV', sType=sType,pNext=pNext,shaderSMCount=shaderSMCount,shaderWarpsPerSM=shaderWarpsPerSM)

def VkPhysicalDeviceShaderSMBuiltinsFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_SM_BUILTINS_FEATURES_NV,pNext=None,shaderSMBuiltins=None,):

    return _new('VkPhysicalDeviceShaderSMBuiltinsFeaturesNV', sType=sType,pNext=pNext,shaderSMBuiltins=shaderSMBuiltins)

def VkPhysicalDeviceFragmentShaderInterlockFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_INTERLOCK_FEATURES_EXT,pNext=None,fragmentShaderSampleInterlock=None,fragmentShaderPixelInterlock=None,fragmentShaderShadingRateInterlock=None,):

    return _new('VkPhysicalDeviceFragmentShaderInterlockFeaturesEXT', sType=sType,pNext=pNext,fragmentShaderSampleInterlock=fragmentShaderSampleInterlock,fragmentShaderPixelInterlock=fragmentShaderPixelInterlock,fragmentShaderShadingRateInterlock=fragmentShaderShadingRateInterlock)

def VkPhysicalDeviceSeparateDepthStencilLayoutsFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SEPARATE_DEPTH_STENCIL_LAYOUTS_FEATURES,pNext=None,separateDepthStencilLayouts=None,):

    return _new('VkPhysicalDeviceSeparateDepthStencilLayoutsFeatures', sType=sType,pNext=pNext,separateDepthStencilLayouts=separateDepthStencilLayouts)

def VkAttachmentReferenceStencilLayout(sType=VK_STRUCTURE_TYPE_ATTACHMENT_REFERENCE_STENCIL_LAYOUT,pNext=None,stencilLayout=None,):

    return _new('VkAttachmentReferenceStencilLayout', sType=sType,pNext=pNext,stencilLayout=stencilLayout)

def VkPhysicalDevicePrimitiveTopologyListRestartFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRIMITIVE_TOPOLOGY_LIST_RESTART_FEATURES_EXT,pNext=None,primitiveTopologyListRestart=None,primitiveTopologyPatchListRestart=None,):

    return _new('VkPhysicalDevicePrimitiveTopologyListRestartFeaturesEXT', sType=sType,pNext=pNext,primitiveTopologyListRestart=primitiveTopologyListRestart,primitiveTopologyPatchListRestart=primitiveTopologyPatchListRestart)

def VkAttachmentDescriptionStencilLayout(sType=VK_STRUCTURE_TYPE_ATTACHMENT_DESCRIPTION_STENCIL_LAYOUT,pNext=None,stencilInitialLayout=None,stencilFinalLayout=None,):

    return _new('VkAttachmentDescriptionStencilLayout', sType=sType,pNext=pNext,stencilInitialLayout=stencilInitialLayout,stencilFinalLayout=stencilFinalLayout)

def VkPhysicalDevicePipelineExecutablePropertiesFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_EXECUTABLE_PROPERTIES_FEATURES_KHR,pNext=None,pipelineExecutableInfo=None,):

    return _new('VkPhysicalDevicePipelineExecutablePropertiesFeaturesKHR', sType=sType,pNext=pNext,pipelineExecutableInfo=pipelineExecutableInfo)

def VkPipelineInfoKHR(sType=VK_STRUCTURE_TYPE_PIPELINE_INFO_KHR,pNext=None,pipeline=None,):

    return _new('VkPipelineInfoKHR', sType=sType,pNext=pNext,pipeline=pipeline)

def VkPipelineExecutablePropertiesKHR(sType=VK_STRUCTURE_TYPE_PIPELINE_EXECUTABLE_PROPERTIES_KHR,pNext=None,stages=None,name=None,description=None,subgroupSize=None,):

    return _new('VkPipelineExecutablePropertiesKHR', sType=sType,pNext=pNext,stages=stages,name=name,description=description,subgroupSize=subgroupSize)

def VkPipelineExecutableInfoKHR(sType=VK_STRUCTURE_TYPE_PIPELINE_EXECUTABLE_INFO_KHR,pNext=None,pipeline=None,executableIndex=None,):

    return _new('VkPipelineExecutableInfoKHR', sType=sType,pNext=pNext,pipeline=pipeline,executableIndex=executableIndex)

def VkPipelineExecutableStatisticValueKHR(b32=None,i64=None,u64=None,f64=None,):

    return _new('VkPipelineExecutableStatisticValueKHR', b32=b32,i64=i64,u64=u64,f64=f64)

def VkPipelineExecutableStatisticKHR(sType=VK_STRUCTURE_TYPE_PIPELINE_EXECUTABLE_STATISTIC_KHR,pNext=None,name=None,description=None,format=None,value=None,):

    return _new('VkPipelineExecutableStatisticKHR', sType=sType,pNext=pNext,name=name,description=description,format=format,value=value)

def VkPipelineExecutableInternalRepresentationKHR(sType=VK_STRUCTURE_TYPE_PIPELINE_EXECUTABLE_INTERNAL_REPRESENTATION_KHR,pNext=None,name=None,description=None,isText=None,dataSize=None,pData=None,):
    if dataSize is None and pData is not None:
        dataSize = len(pData)

    return _new('VkPipelineExecutableInternalRepresentationKHR', sType=sType,pNext=pNext,name=name,description=description,isText=isText,dataSize=dataSize,pData=pData)

def VkPhysicalDeviceShaderDemoteToHelperInvocationFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_DEMOTE_TO_HELPER_INVOCATION_FEATURES,pNext=None,shaderDemoteToHelperInvocation=None,):

    return _new('VkPhysicalDeviceShaderDemoteToHelperInvocationFeatures', sType=sType,pNext=pNext,shaderDemoteToHelperInvocation=shaderDemoteToHelperInvocation)

def VkPhysicalDeviceTexelBufferAlignmentFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXEL_BUFFER_ALIGNMENT_FEATURES_EXT,pNext=None,texelBufferAlignment=None,):

    return _new('VkPhysicalDeviceTexelBufferAlignmentFeaturesEXT', sType=sType,pNext=pNext,texelBufferAlignment=texelBufferAlignment)

def VkPhysicalDeviceTexelBufferAlignmentProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXEL_BUFFER_ALIGNMENT_PROPERTIES,pNext=None,storageTexelBufferOffsetAlignmentBytes=None,storageTexelBufferOffsetSingleTexelAlignment=None,uniformTexelBufferOffsetAlignmentBytes=None,uniformTexelBufferOffsetSingleTexelAlignment=None,):

    return _new('VkPhysicalDeviceTexelBufferAlignmentProperties', sType=sType,pNext=pNext,storageTexelBufferOffsetAlignmentBytes=storageTexelBufferOffsetAlignmentBytes,storageTexelBufferOffsetSingleTexelAlignment=storageTexelBufferOffsetSingleTexelAlignment,uniformTexelBufferOffsetAlignmentBytes=uniformTexelBufferOffsetAlignmentBytes,uniformTexelBufferOffsetSingleTexelAlignment=uniformTexelBufferOffsetSingleTexelAlignment)

def VkPhysicalDeviceSubgroupSizeControlFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_SIZE_CONTROL_FEATURES,pNext=None,subgroupSizeControl=None,computeFullSubgroups=None,):

    return _new('VkPhysicalDeviceSubgroupSizeControlFeatures', sType=sType,pNext=pNext,subgroupSizeControl=subgroupSizeControl,computeFullSubgroups=computeFullSubgroups)

def VkPhysicalDeviceSubgroupSizeControlProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_SIZE_CONTROL_PROPERTIES,pNext=None,minSubgroupSize=None,maxSubgroupSize=None,maxComputeWorkgroupSubgroups=None,requiredSubgroupSizeStages=None,):

    return _new('VkPhysicalDeviceSubgroupSizeControlProperties', sType=sType,pNext=pNext,minSubgroupSize=minSubgroupSize,maxSubgroupSize=maxSubgroupSize,maxComputeWorkgroupSubgroups=maxComputeWorkgroupSubgroups,requiredSubgroupSizeStages=requiredSubgroupSizeStages)

def VkPipelineShaderStageRequiredSubgroupSizeCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_REQUIRED_SUBGROUP_SIZE_CREATE_INFO,pNext=None,requiredSubgroupSize=None,):

    return _new('VkPipelineShaderStageRequiredSubgroupSizeCreateInfo', sType=sType,pNext=pNext,requiredSubgroupSize=requiredSubgroupSize)

def VkSubpassShadingPipelineCreateInfoHUAWEI(sType=VK_STRUCTURE_TYPE_SUBPASS_SHADING_PIPELINE_CREATE_INFO_HUAWEI,pNext=None,renderPass=None,subpass=None,):

    return _new('VkSubpassShadingPipelineCreateInfoHUAWEI', sType=sType,pNext=pNext,renderPass=renderPass,subpass=subpass)

def VkPhysicalDeviceSubpassShadingPropertiesHUAWEI(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBPASS_SHADING_PROPERTIES_HUAWEI,pNext=None,maxSubpassShadingWorkgroupSizeAspectRatio=None,):

    return _new('VkPhysicalDeviceSubpassShadingPropertiesHUAWEI', sType=sType,pNext=pNext,maxSubpassShadingWorkgroupSizeAspectRatio=maxSubpassShadingWorkgroupSizeAspectRatio)

def VkPhysicalDeviceClusterCullingShaderPropertiesHUAWEI(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CLUSTER_CULLING_SHADER_PROPERTIES_HUAWEI,pNext=None,maxWorkGroupCount=None,maxWorkGroupSize=None,maxOutputClusterCount=None,indirectBufferOffsetAlignment=None,):

    return _new('VkPhysicalDeviceClusterCullingShaderPropertiesHUAWEI', sType=sType,pNext=pNext,maxWorkGroupCount=maxWorkGroupCount,maxWorkGroupSize=maxWorkGroupSize,maxOutputClusterCount=maxOutputClusterCount,indirectBufferOffsetAlignment=indirectBufferOffsetAlignment)

def VkMemoryOpaqueCaptureAddressAllocateInfo(sType=VK_STRUCTURE_TYPE_MEMORY_OPAQUE_CAPTURE_ADDRESS_ALLOCATE_INFO,pNext=None,opaqueCaptureAddress=None,):

    return _new('VkMemoryOpaqueCaptureAddressAllocateInfo', sType=sType,pNext=pNext,opaqueCaptureAddress=opaqueCaptureAddress)

def VkDeviceMemoryOpaqueCaptureAddressInfo(sType=VK_STRUCTURE_TYPE_DEVICE_MEMORY_OPAQUE_CAPTURE_ADDRESS_INFO,pNext=None,memory=None,):

    return _new('VkDeviceMemoryOpaqueCaptureAddressInfo', sType=sType,pNext=pNext,memory=memory)

def VkPhysicalDeviceLineRasterizationFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LINE_RASTERIZATION_FEATURES_EXT,pNext=None,rectangularLines=None,bresenhamLines=None,smoothLines=None,stippledRectangularLines=None,stippledBresenhamLines=None,stippledSmoothLines=None,):

    return _new('VkPhysicalDeviceLineRasterizationFeaturesEXT', sType=sType,pNext=pNext,rectangularLines=rectangularLines,bresenhamLines=bresenhamLines,smoothLines=smoothLines,stippledRectangularLines=stippledRectangularLines,stippledBresenhamLines=stippledBresenhamLines,stippledSmoothLines=stippledSmoothLines)

def VkPhysicalDeviceLineRasterizationPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LINE_RASTERIZATION_PROPERTIES_EXT,pNext=None,lineSubPixelPrecisionBits=None,):

    return _new('VkPhysicalDeviceLineRasterizationPropertiesEXT', sType=sType,pNext=pNext,lineSubPixelPrecisionBits=lineSubPixelPrecisionBits)

def VkPipelineRasterizationLineStateCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_LINE_STATE_CREATE_INFO_EXT,pNext=None,lineRasterizationMode=None,stippledLineEnable=None,lineStippleFactor=None,lineStipplePattern=None,):

    return _new('VkPipelineRasterizationLineStateCreateInfoEXT', sType=sType,pNext=pNext,lineRasterizationMode=lineRasterizationMode,stippledLineEnable=stippledLineEnable,lineStippleFactor=lineStippleFactor,lineStipplePattern=lineStipplePattern)

def VkPhysicalDevicePipelineCreationCacheControlFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_CREATION_CACHE_CONTROL_FEATURES,pNext=None,pipelineCreationCacheControl=None,):

    return _new('VkPhysicalDevicePipelineCreationCacheControlFeatures', sType=sType,pNext=pNext,pipelineCreationCacheControl=pipelineCreationCacheControl)

def VkPhysicalDeviceVulkan11Features(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_1_FEATURES,pNext=None,storageBuffer16BitAccess=None,uniformAndStorageBuffer16BitAccess=None,storagePushConstant16=None,storageInputOutput16=None,multiview=None,multiviewGeometryShader=None,multiviewTessellationShader=None,variablePointersStorageBuffer=None,variablePointers=None,protectedMemory=None,samplerYcbcrConversion=None,shaderDrawParameters=None,):

    return _new('VkPhysicalDeviceVulkan11Features', sType=sType,pNext=pNext,storageBuffer16BitAccess=storageBuffer16BitAccess,uniformAndStorageBuffer16BitAccess=uniformAndStorageBuffer16BitAccess,storagePushConstant16=storagePushConstant16,storageInputOutput16=storageInputOutput16,multiview=multiview,multiviewGeometryShader=multiviewGeometryShader,multiviewTessellationShader=multiviewTessellationShader,variablePointersStorageBuffer=variablePointersStorageBuffer,variablePointers=variablePointers,protectedMemory=protectedMemory,samplerYcbcrConversion=samplerYcbcrConversion,shaderDrawParameters=shaderDrawParameters)

def VkPhysicalDeviceVulkan11Properties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_1_PROPERTIES,pNext=None,deviceUUID=None,driverUUID=None,deviceLUID=None,deviceNodeMask=None,deviceLUIDValid=None,subgroupSize=None,subgroupSupportedStages=None,subgroupSupportedOperations=None,subgroupQuadOperationsInAllStages=None,pointClippingBehavior=None,maxMultiviewViewCount=None,maxMultiviewInstanceIndex=None,protectedNoFault=None,maxPerSetDescriptors=None,maxMemoryAllocationSize=None,):

    return _new('VkPhysicalDeviceVulkan11Properties', sType=sType,pNext=pNext,deviceUUID=deviceUUID,driverUUID=driverUUID,deviceLUID=deviceLUID,deviceNodeMask=deviceNodeMask,deviceLUIDValid=deviceLUIDValid,subgroupSize=subgroupSize,subgroupSupportedStages=subgroupSupportedStages,subgroupSupportedOperations=subgroupSupportedOperations,subgroupQuadOperationsInAllStages=subgroupQuadOperationsInAllStages,pointClippingBehavior=pointClippingBehavior,maxMultiviewViewCount=maxMultiviewViewCount,maxMultiviewInstanceIndex=maxMultiviewInstanceIndex,protectedNoFault=protectedNoFault,maxPerSetDescriptors=maxPerSetDescriptors,maxMemoryAllocationSize=maxMemoryAllocationSize)

def VkPhysicalDeviceVulkan12Features(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_2_FEATURES,pNext=None,samplerMirrorClampToEdge=None,drawIndirectCount=None,storageBuffer8BitAccess=None,uniformAndStorageBuffer8BitAccess=None,storagePushConstant8=None,shaderBufferInt64Atomics=None,shaderSharedInt64Atomics=None,shaderFloat16=None,shaderInt8=None,descriptorIndexing=None,shaderInputAttachmentArrayDynamicIndexing=None,shaderUniformTexelBufferArrayDynamicIndexing=None,shaderStorageTexelBufferArrayDynamicIndexing=None,shaderUniformBufferArrayNonUniformIndexing=None,shaderSampledImageArrayNonUniformIndexing=None,shaderStorageBufferArrayNonUniformIndexing=None,shaderStorageImageArrayNonUniformIndexing=None,shaderInputAttachmentArrayNonUniformIndexing=None,shaderUniformTexelBufferArrayNonUniformIndexing=None,shaderStorageTexelBufferArrayNonUniformIndexing=None,descriptorBindingUniformBufferUpdateAfterBind=None,descriptorBindingSampledImageUpdateAfterBind=None,descriptorBindingStorageImageUpdateAfterBind=None,descriptorBindingStorageBufferUpdateAfterBind=None,descriptorBindingUniformTexelBufferUpdateAfterBind=None,descriptorBindingStorageTexelBufferUpdateAfterBind=None,descriptorBindingUpdateUnusedWhilePending=None,descriptorBindingPartiallyBound=None,descriptorBindingVariableDescriptorCount=None,runtimeDescriptorArray=None,samplerFilterMinmax=None,scalarBlockLayout=None,imagelessFramebuffer=None,uniformBufferStandardLayout=None,shaderSubgroupExtendedTypes=None,separateDepthStencilLayouts=None,hostQueryReset=None,timelineSemaphore=None,bufferDeviceAddress=None,bufferDeviceAddressCaptureReplay=None,bufferDeviceAddressMultiDevice=None,vulkanMemoryModel=None,vulkanMemoryModelDeviceScope=None,vulkanMemoryModelAvailabilityVisibilityChains=None,shaderOutputViewportIndex=None,shaderOutputLayer=None,subgroupBroadcastDynamicId=None,):

    return _new('VkPhysicalDeviceVulkan12Features', sType=sType,pNext=pNext,samplerMirrorClampToEdge=samplerMirrorClampToEdge,drawIndirectCount=drawIndirectCount,storageBuffer8BitAccess=storageBuffer8BitAccess,uniformAndStorageBuffer8BitAccess=uniformAndStorageBuffer8BitAccess,storagePushConstant8=storagePushConstant8,shaderBufferInt64Atomics=shaderBufferInt64Atomics,shaderSharedInt64Atomics=shaderSharedInt64Atomics,shaderFloat16=shaderFloat16,shaderInt8=shaderInt8,descriptorIndexing=descriptorIndexing,shaderInputAttachmentArrayDynamicIndexing=shaderInputAttachmentArrayDynamicIndexing,shaderUniformTexelBufferArrayDynamicIndexing=shaderUniformTexelBufferArrayDynamicIndexing,shaderStorageTexelBufferArrayDynamicIndexing=shaderStorageTexelBufferArrayDynamicIndexing,shaderUniformBufferArrayNonUniformIndexing=shaderUniformBufferArrayNonUniformIndexing,shaderSampledImageArrayNonUniformIndexing=shaderSampledImageArrayNonUniformIndexing,shaderStorageBufferArrayNonUniformIndexing=shaderStorageBufferArrayNonUniformIndexing,shaderStorageImageArrayNonUniformIndexing=shaderStorageImageArrayNonUniformIndexing,shaderInputAttachmentArrayNonUniformIndexing=shaderInputAttachmentArrayNonUniformIndexing,shaderUniformTexelBufferArrayNonUniformIndexing=shaderUniformTexelBufferArrayNonUniformIndexing,shaderStorageTexelBufferArrayNonUniformIndexing=shaderStorageTexelBufferArrayNonUniformIndexing,descriptorBindingUniformBufferUpdateAfterBind=descriptorBindingUniformBufferUpdateAfterBind,descriptorBindingSampledImageUpdateAfterBind=descriptorBindingSampledImageUpdateAfterBind,descriptorBindingStorageImageUpdateAfterBind=descriptorBindingStorageImageUpdateAfterBind,descriptorBindingStorageBufferUpdateAfterBind=descriptorBindingStorageBufferUpdateAfterBind,descriptorBindingUniformTexelBufferUpdateAfterBind=descriptorBindingUniformTexelBufferUpdateAfterBind,descriptorBindingStorageTexelBufferUpdateAfterBind=descriptorBindingStorageTexelBufferUpdateAfterBind,descriptorBindingUpdateUnusedWhilePending=descriptorBindingUpdateUnusedWhilePending,descriptorBindingPartiallyBound=descriptorBindingPartiallyBound,descriptorBindingVariableDescriptorCount=descriptorBindingVariableDescriptorCount,runtimeDescriptorArray=runtimeDescriptorArray,samplerFilterMinmax=samplerFilterMinmax,scalarBlockLayout=scalarBlockLayout,imagelessFramebuffer=imagelessFramebuffer,uniformBufferStandardLayout=uniformBufferStandardLayout,shaderSubgroupExtendedTypes=shaderSubgroupExtendedTypes,separateDepthStencilLayouts=separateDepthStencilLayouts,hostQueryReset=hostQueryReset,timelineSemaphore=timelineSemaphore,bufferDeviceAddress=bufferDeviceAddress,bufferDeviceAddressCaptureReplay=bufferDeviceAddressCaptureReplay,bufferDeviceAddressMultiDevice=bufferDeviceAddressMultiDevice,vulkanMemoryModel=vulkanMemoryModel,vulkanMemoryModelDeviceScope=vulkanMemoryModelDeviceScope,vulkanMemoryModelAvailabilityVisibilityChains=vulkanMemoryModelAvailabilityVisibilityChains,shaderOutputViewportIndex=shaderOutputViewportIndex,shaderOutputLayer=shaderOutputLayer,subgroupBroadcastDynamicId=subgroupBroadcastDynamicId)

def VkPhysicalDeviceVulkan12Properties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_2_PROPERTIES,pNext=None,driverID=None,driverName=None,driverInfo=None,conformanceVersion=None,denormBehaviorIndependence=None,roundingModeIndependence=None,shaderSignedZeroInfNanPreserveFloat16=None,shaderSignedZeroInfNanPreserveFloat32=None,shaderSignedZeroInfNanPreserveFloat64=None,shaderDenormPreserveFloat16=None,shaderDenormPreserveFloat32=None,shaderDenormPreserveFloat64=None,shaderDenormFlushToZeroFloat16=None,shaderDenormFlushToZeroFloat32=None,shaderDenormFlushToZeroFloat64=None,shaderRoundingModeRTEFloat16=None,shaderRoundingModeRTEFloat32=None,shaderRoundingModeRTEFloat64=None,shaderRoundingModeRTZFloat16=None,shaderRoundingModeRTZFloat32=None,shaderRoundingModeRTZFloat64=None,maxUpdateAfterBindDescriptorsInAllPools=None,shaderUniformBufferArrayNonUniformIndexingNative=None,shaderSampledImageArrayNonUniformIndexingNative=None,shaderStorageBufferArrayNonUniformIndexingNative=None,shaderStorageImageArrayNonUniformIndexingNative=None,shaderInputAttachmentArrayNonUniformIndexingNative=None,robustBufferAccessUpdateAfterBind=None,quadDivergentImplicitLod=None,maxPerStageDescriptorUpdateAfterBindSamplers=None,maxPerStageDescriptorUpdateAfterBindUniformBuffers=None,maxPerStageDescriptorUpdateAfterBindStorageBuffers=None,maxPerStageDescriptorUpdateAfterBindSampledImages=None,maxPerStageDescriptorUpdateAfterBindStorageImages=None,maxPerStageDescriptorUpdateAfterBindInputAttachments=None,maxPerStageUpdateAfterBindResources=None,maxDescriptorSetUpdateAfterBindSamplers=None,maxDescriptorSetUpdateAfterBindUniformBuffers=None,maxDescriptorSetUpdateAfterBindUniformBuffersDynamic=None,maxDescriptorSetUpdateAfterBindStorageBuffers=None,maxDescriptorSetUpdateAfterBindStorageBuffersDynamic=None,maxDescriptorSetUpdateAfterBindSampledImages=None,maxDescriptorSetUpdateAfterBindStorageImages=None,maxDescriptorSetUpdateAfterBindInputAttachments=None,supportedDepthResolveModes=None,supportedStencilResolveModes=None,independentResolveNone=None,independentResolve=None,filterMinmaxSingleComponentFormats=None,filterMinmaxImageComponentMapping=None,maxTimelineSemaphoreValueDifference=None,framebufferIntegerColorSampleCounts=None,):

    return _new('VkPhysicalDeviceVulkan12Properties', sType=sType,pNext=pNext,driverID=driverID,driverName=driverName,driverInfo=driverInfo,conformanceVersion=conformanceVersion,denormBehaviorIndependence=denormBehaviorIndependence,roundingModeIndependence=roundingModeIndependence,shaderSignedZeroInfNanPreserveFloat16=shaderSignedZeroInfNanPreserveFloat16,shaderSignedZeroInfNanPreserveFloat32=shaderSignedZeroInfNanPreserveFloat32,shaderSignedZeroInfNanPreserveFloat64=shaderSignedZeroInfNanPreserveFloat64,shaderDenormPreserveFloat16=shaderDenormPreserveFloat16,shaderDenormPreserveFloat32=shaderDenormPreserveFloat32,shaderDenormPreserveFloat64=shaderDenormPreserveFloat64,shaderDenormFlushToZeroFloat16=shaderDenormFlushToZeroFloat16,shaderDenormFlushToZeroFloat32=shaderDenormFlushToZeroFloat32,shaderDenormFlushToZeroFloat64=shaderDenormFlushToZeroFloat64,shaderRoundingModeRTEFloat16=shaderRoundingModeRTEFloat16,shaderRoundingModeRTEFloat32=shaderRoundingModeRTEFloat32,shaderRoundingModeRTEFloat64=shaderRoundingModeRTEFloat64,shaderRoundingModeRTZFloat16=shaderRoundingModeRTZFloat16,shaderRoundingModeRTZFloat32=shaderRoundingModeRTZFloat32,shaderRoundingModeRTZFloat64=shaderRoundingModeRTZFloat64,maxUpdateAfterBindDescriptorsInAllPools=maxUpdateAfterBindDescriptorsInAllPools,shaderUniformBufferArrayNonUniformIndexingNative=shaderUniformBufferArrayNonUniformIndexingNative,shaderSampledImageArrayNonUniformIndexingNative=shaderSampledImageArrayNonUniformIndexingNative,shaderStorageBufferArrayNonUniformIndexingNative=shaderStorageBufferArrayNonUniformIndexingNative,shaderStorageImageArrayNonUniformIndexingNative=shaderStorageImageArrayNonUniformIndexingNative,shaderInputAttachmentArrayNonUniformIndexingNative=shaderInputAttachmentArrayNonUniformIndexingNative,robustBufferAccessUpdateAfterBind=robustBufferAccessUpdateAfterBind,quadDivergentImplicitLod=quadDivergentImplicitLod,maxPerStageDescriptorUpdateAfterBindSamplers=maxPerStageDescriptorUpdateAfterBindSamplers,maxPerStageDescriptorUpdateAfterBindUniformBuffers=maxPerStageDescriptorUpdateAfterBindUniformBuffers,maxPerStageDescriptorUpdateAfterBindStorageBuffers=maxPerStageDescriptorUpdateAfterBindStorageBuffers,maxPerStageDescriptorUpdateAfterBindSampledImages=maxPerStageDescriptorUpdateAfterBindSampledImages,maxPerStageDescriptorUpdateAfterBindStorageImages=maxPerStageDescriptorUpdateAfterBindStorageImages,maxPerStageDescriptorUpdateAfterBindInputAttachments=maxPerStageDescriptorUpdateAfterBindInputAttachments,maxPerStageUpdateAfterBindResources=maxPerStageUpdateAfterBindResources,maxDescriptorSetUpdateAfterBindSamplers=maxDescriptorSetUpdateAfterBindSamplers,maxDescriptorSetUpdateAfterBindUniformBuffers=maxDescriptorSetUpdateAfterBindUniformBuffers,maxDescriptorSetUpdateAfterBindUniformBuffersDynamic=maxDescriptorSetUpdateAfterBindUniformBuffersDynamic,maxDescriptorSetUpdateAfterBindStorageBuffers=maxDescriptorSetUpdateAfterBindStorageBuffers,maxDescriptorSetUpdateAfterBindStorageBuffersDynamic=maxDescriptorSetUpdateAfterBindStorageBuffersDynamic,maxDescriptorSetUpdateAfterBindSampledImages=maxDescriptorSetUpdateAfterBindSampledImages,maxDescriptorSetUpdateAfterBindStorageImages=maxDescriptorSetUpdateAfterBindStorageImages,maxDescriptorSetUpdateAfterBindInputAttachments=maxDescriptorSetUpdateAfterBindInputAttachments,supportedDepthResolveModes=supportedDepthResolveModes,supportedStencilResolveModes=supportedStencilResolveModes,independentResolveNone=independentResolveNone,independentResolve=independentResolve,filterMinmaxSingleComponentFormats=filterMinmaxSingleComponentFormats,filterMinmaxImageComponentMapping=filterMinmaxImageComponentMapping,maxTimelineSemaphoreValueDifference=maxTimelineSemaphoreValueDifference,framebufferIntegerColorSampleCounts=framebufferIntegerColorSampleCounts)

def VkPhysicalDeviceVulkan13Features(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_3_FEATURES,pNext=None,robustImageAccess=None,inlineUniformBlock=None,descriptorBindingInlineUniformBlockUpdateAfterBind=None,pipelineCreationCacheControl=None,privateData=None,shaderDemoteToHelperInvocation=None,shaderTerminateInvocation=None,subgroupSizeControl=None,computeFullSubgroups=None,synchronization2=None,textureCompressionASTC_HDR=None,shaderZeroInitializeWorkgroupMemory=None,dynamicRendering=None,shaderIntegerDotProduct=None,maintenance4=None,):

    return _new('VkPhysicalDeviceVulkan13Features', sType=sType,pNext=pNext,robustImageAccess=robustImageAccess,inlineUniformBlock=inlineUniformBlock,descriptorBindingInlineUniformBlockUpdateAfterBind=descriptorBindingInlineUniformBlockUpdateAfterBind,pipelineCreationCacheControl=pipelineCreationCacheControl,privateData=privateData,shaderDemoteToHelperInvocation=shaderDemoteToHelperInvocation,shaderTerminateInvocation=shaderTerminateInvocation,subgroupSizeControl=subgroupSizeControl,computeFullSubgroups=computeFullSubgroups,synchronization2=synchronization2,textureCompressionASTC_HDR=textureCompressionASTC_HDR,shaderZeroInitializeWorkgroupMemory=shaderZeroInitializeWorkgroupMemory,dynamicRendering=dynamicRendering,shaderIntegerDotProduct=shaderIntegerDotProduct,maintenance4=maintenance4)

def VkPhysicalDeviceVulkan13Properties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_3_PROPERTIES,pNext=None,minSubgroupSize=None,maxSubgroupSize=None,maxComputeWorkgroupSubgroups=None,requiredSubgroupSizeStages=None,maxInlineUniformBlockSize=None,maxPerStageDescriptorInlineUniformBlocks=None,maxPerStageDescriptorUpdateAfterBindInlineUniformBlocks=None,maxDescriptorSetInlineUniformBlocks=None,maxDescriptorSetUpdateAfterBindInlineUniformBlocks=None,maxInlineUniformTotalSize=None,integerDotProduct8BitUnsignedAccelerated=None,integerDotProduct8BitSignedAccelerated=None,integerDotProduct8BitMixedSignednessAccelerated=None,integerDotProduct4x8BitPackedUnsignedAccelerated=None,integerDotProduct4x8BitPackedSignedAccelerated=None,integerDotProduct4x8BitPackedMixedSignednessAccelerated=None,integerDotProduct16BitUnsignedAccelerated=None,integerDotProduct16BitSignedAccelerated=None,integerDotProduct16BitMixedSignednessAccelerated=None,integerDotProduct32BitUnsignedAccelerated=None,integerDotProduct32BitSignedAccelerated=None,integerDotProduct32BitMixedSignednessAccelerated=None,integerDotProduct64BitUnsignedAccelerated=None,integerDotProduct64BitSignedAccelerated=None,integerDotProduct64BitMixedSignednessAccelerated=None,integerDotProductAccumulatingSaturating8BitUnsignedAccelerated=None,integerDotProductAccumulatingSaturating8BitSignedAccelerated=None,integerDotProductAccumulatingSaturating8BitMixedSignednessAccelerated=None,integerDotProductAccumulatingSaturating4x8BitPackedUnsignedAccelerated=None,integerDotProductAccumulatingSaturating4x8BitPackedSignedAccelerated=None,integerDotProductAccumulatingSaturating4x8BitPackedMixedSignednessAccelerated=None,integerDotProductAccumulatingSaturating16BitUnsignedAccelerated=None,integerDotProductAccumulatingSaturating16BitSignedAccelerated=None,integerDotProductAccumulatingSaturating16BitMixedSignednessAccelerated=None,integerDotProductAccumulatingSaturating32BitUnsignedAccelerated=None,integerDotProductAccumulatingSaturating32BitSignedAccelerated=None,integerDotProductAccumulatingSaturating32BitMixedSignednessAccelerated=None,integerDotProductAccumulatingSaturating64BitUnsignedAccelerated=None,integerDotProductAccumulatingSaturating64BitSignedAccelerated=None,integerDotProductAccumulatingSaturating64BitMixedSignednessAccelerated=None,storageTexelBufferOffsetAlignmentBytes=None,storageTexelBufferOffsetSingleTexelAlignment=None,uniformTexelBufferOffsetAlignmentBytes=None,uniformTexelBufferOffsetSingleTexelAlignment=None,maxBufferSize=None,):

    return _new('VkPhysicalDeviceVulkan13Properties', sType=sType,pNext=pNext,minSubgroupSize=minSubgroupSize,maxSubgroupSize=maxSubgroupSize,maxComputeWorkgroupSubgroups=maxComputeWorkgroupSubgroups,requiredSubgroupSizeStages=requiredSubgroupSizeStages,maxInlineUniformBlockSize=maxInlineUniformBlockSize,maxPerStageDescriptorInlineUniformBlocks=maxPerStageDescriptorInlineUniformBlocks,maxPerStageDescriptorUpdateAfterBindInlineUniformBlocks=maxPerStageDescriptorUpdateAfterBindInlineUniformBlocks,maxDescriptorSetInlineUniformBlocks=maxDescriptorSetInlineUniformBlocks,maxDescriptorSetUpdateAfterBindInlineUniformBlocks=maxDescriptorSetUpdateAfterBindInlineUniformBlocks,maxInlineUniformTotalSize=maxInlineUniformTotalSize,integerDotProduct8BitUnsignedAccelerated=integerDotProduct8BitUnsignedAccelerated,integerDotProduct8BitSignedAccelerated=integerDotProduct8BitSignedAccelerated,integerDotProduct8BitMixedSignednessAccelerated=integerDotProduct8BitMixedSignednessAccelerated,integerDotProduct4x8BitPackedUnsignedAccelerated=integerDotProduct4x8BitPackedUnsignedAccelerated,integerDotProduct4x8BitPackedSignedAccelerated=integerDotProduct4x8BitPackedSignedAccelerated,integerDotProduct4x8BitPackedMixedSignednessAccelerated=integerDotProduct4x8BitPackedMixedSignednessAccelerated,integerDotProduct16BitUnsignedAccelerated=integerDotProduct16BitUnsignedAccelerated,integerDotProduct16BitSignedAccelerated=integerDotProduct16BitSignedAccelerated,integerDotProduct16BitMixedSignednessAccelerated=integerDotProduct16BitMixedSignednessAccelerated,integerDotProduct32BitUnsignedAccelerated=integerDotProduct32BitUnsignedAccelerated,integerDotProduct32BitSignedAccelerated=integerDotProduct32BitSignedAccelerated,integerDotProduct32BitMixedSignednessAccelerated=integerDotProduct32BitMixedSignednessAccelerated,integerDotProduct64BitUnsignedAccelerated=integerDotProduct64BitUnsignedAccelerated,integerDotProduct64BitSignedAccelerated=integerDotProduct64BitSignedAccelerated,integerDotProduct64BitMixedSignednessAccelerated=integerDotProduct64BitMixedSignednessAccelerated,integerDotProductAccumulatingSaturating8BitUnsignedAccelerated=integerDotProductAccumulatingSaturating8BitUnsignedAccelerated,integerDotProductAccumulatingSaturating8BitSignedAccelerated=integerDotProductAccumulatingSaturating8BitSignedAccelerated,integerDotProductAccumulatingSaturating8BitMixedSignednessAccelerated=integerDotProductAccumulatingSaturating8BitMixedSignednessAccelerated,integerDotProductAccumulatingSaturating4x8BitPackedUnsignedAccelerated=integerDotProductAccumulatingSaturating4x8BitPackedUnsignedAccelerated,integerDotProductAccumulatingSaturating4x8BitPackedSignedAccelerated=integerDotProductAccumulatingSaturating4x8BitPackedSignedAccelerated,integerDotProductAccumulatingSaturating4x8BitPackedMixedSignednessAccelerated=integerDotProductAccumulatingSaturating4x8BitPackedMixedSignednessAccelerated,integerDotProductAccumulatingSaturating16BitUnsignedAccelerated=integerDotProductAccumulatingSaturating16BitUnsignedAccelerated,integerDotProductAccumulatingSaturating16BitSignedAccelerated=integerDotProductAccumulatingSaturating16BitSignedAccelerated,integerDotProductAccumulatingSaturating16BitMixedSignednessAccelerated=integerDotProductAccumulatingSaturating16BitMixedSignednessAccelerated,integerDotProductAccumulatingSaturating32BitUnsignedAccelerated=integerDotProductAccumulatingSaturating32BitUnsignedAccelerated,integerDotProductAccumulatingSaturating32BitSignedAccelerated=integerDotProductAccumulatingSaturating32BitSignedAccelerated,integerDotProductAccumulatingSaturating32BitMixedSignednessAccelerated=integerDotProductAccumulatingSaturating32BitMixedSignednessAccelerated,integerDotProductAccumulatingSaturating64BitUnsignedAccelerated=integerDotProductAccumulatingSaturating64BitUnsignedAccelerated,integerDotProductAccumulatingSaturating64BitSignedAccelerated=integerDotProductAccumulatingSaturating64BitSignedAccelerated,integerDotProductAccumulatingSaturating64BitMixedSignednessAccelerated=integerDotProductAccumulatingSaturating64BitMixedSignednessAccelerated,storageTexelBufferOffsetAlignmentBytes=storageTexelBufferOffsetAlignmentBytes,storageTexelBufferOffsetSingleTexelAlignment=storageTexelBufferOffsetSingleTexelAlignment,uniformTexelBufferOffsetAlignmentBytes=uniformTexelBufferOffsetAlignmentBytes,uniformTexelBufferOffsetSingleTexelAlignment=uniformTexelBufferOffsetSingleTexelAlignment,maxBufferSize=maxBufferSize)

def VkPipelineCompilerControlCreateInfoAMD(sType=VK_STRUCTURE_TYPE_PIPELINE_COMPILER_CONTROL_CREATE_INFO_AMD,pNext=None,compilerControlFlags=None,):

    return _new('VkPipelineCompilerControlCreateInfoAMD', sType=sType,pNext=pNext,compilerControlFlags=compilerControlFlags)

def VkPhysicalDeviceCoherentMemoryFeaturesAMD(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COHERENT_MEMORY_FEATURES_AMD,pNext=None,deviceCoherentMemory=None,):

    return _new('VkPhysicalDeviceCoherentMemoryFeaturesAMD', sType=sType,pNext=pNext,deviceCoherentMemory=deviceCoherentMemory)

def VkFaultData(sType=VK_STRUCTURE_TYPE_FAULT_DATA,pNext=None,faultLevel=None,faultType=None,):

    return _new('VkFaultData', sType=sType,pNext=pNext,faultLevel=faultLevel,faultType=faultType)

def VkFaultCallbackInfo(sType=VK_STRUCTURE_TYPE_FAULT_CALLBACK_INFO,pNext=None,faultCount=None,pFaults=None,pfnFaultCallback=None,):
    if faultCount is None and pFaults is not None:
        faultCount = len(pFaults)

    return _new('VkFaultCallbackInfo', sType=sType,pNext=pNext,faultCount=faultCount,pFaults=pFaults,pfnFaultCallback=pfnFaultCallback)

def VkPhysicalDeviceToolProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TOOL_PROPERTIES,pNext=None,name=None,version=None,purposes=None,description=None,layer=None,):

    return _new('VkPhysicalDeviceToolProperties', sType=sType,pNext=pNext,name=name,version=version,purposes=purposes,description=description,layer=layer)

def VkSamplerCustomBorderColorCreateInfoEXT(sType=VK_STRUCTURE_TYPE_SAMPLER_CUSTOM_BORDER_COLOR_CREATE_INFO_EXT,pNext=None,customBorderColor=None,format=None,):

    return _new('VkSamplerCustomBorderColorCreateInfoEXT', sType=sType,pNext=pNext,customBorderColor=customBorderColor,format=format)

def VkPhysicalDeviceCustomBorderColorPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUSTOM_BORDER_COLOR_PROPERTIES_EXT,pNext=None,maxCustomBorderColorSamplers=None,):

    return _new('VkPhysicalDeviceCustomBorderColorPropertiesEXT', sType=sType,pNext=pNext,maxCustomBorderColorSamplers=maxCustomBorderColorSamplers)

def VkPhysicalDeviceCustomBorderColorFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUSTOM_BORDER_COLOR_FEATURES_EXT,pNext=None,customBorderColors=None,customBorderColorWithoutFormat=None,):

    return _new('VkPhysicalDeviceCustomBorderColorFeaturesEXT', sType=sType,pNext=pNext,customBorderColors=customBorderColors,customBorderColorWithoutFormat=customBorderColorWithoutFormat)

def VkSamplerBorderColorComponentMappingCreateInfoEXT(sType=VK_STRUCTURE_TYPE_SAMPLER_BORDER_COLOR_COMPONENT_MAPPING_CREATE_INFO_EXT,pNext=None,components=None,srgb=None,):

    return _new('VkSamplerBorderColorComponentMappingCreateInfoEXT', sType=sType,pNext=pNext,components=components,srgb=srgb)

def VkPhysicalDeviceBorderColorSwizzleFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BORDER_COLOR_SWIZZLE_FEATURES_EXT,pNext=None,borderColorSwizzle=None,borderColorSwizzleFromImage=None,):

    return _new('VkPhysicalDeviceBorderColorSwizzleFeaturesEXT', sType=sType,pNext=pNext,borderColorSwizzle=borderColorSwizzle,borderColorSwizzleFromImage=borderColorSwizzleFromImage)

def VkDeviceOrHostAddressKHR(deviceAddress=None,hostAddress=None,):

    return _new('VkDeviceOrHostAddressKHR', deviceAddress=deviceAddress,hostAddress=hostAddress)

def VkDeviceOrHostAddressConstKHR(deviceAddress=None,hostAddress=None,):

    return _new('VkDeviceOrHostAddressConstKHR', deviceAddress=deviceAddress,hostAddress=hostAddress)

def VkDeviceOrHostAddressConstAMDX(deviceAddress=None,hostAddress=None,):

    return _new('VkDeviceOrHostAddressConstAMDX', deviceAddress=deviceAddress,hostAddress=hostAddress)

def VkAccelerationStructureGeometryTrianglesDataKHR(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_TRIANGLES_DATA_KHR,pNext=None,vertexFormat=None,vertexData=None,vertexStride=None,maxVertex=None,indexType=None,indexData=None,transformData=None,):

    return _new('VkAccelerationStructureGeometryTrianglesDataKHR', sType=sType,pNext=pNext,vertexFormat=vertexFormat,vertexData=vertexData,vertexStride=vertexStride,maxVertex=maxVertex,indexType=indexType,indexData=indexData,transformData=transformData)

def VkAccelerationStructureGeometryAabbsDataKHR(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_AABBS_DATA_KHR,pNext=None,data=None,stride=None,):

    return _new('VkAccelerationStructureGeometryAabbsDataKHR', sType=sType,pNext=pNext,data=data,stride=stride)

def VkAccelerationStructureGeometryInstancesDataKHR(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_INSTANCES_DATA_KHR,pNext=None,arrayOfPointers=None,data=None,):

    return _new('VkAccelerationStructureGeometryInstancesDataKHR', sType=sType,pNext=pNext,arrayOfPointers=arrayOfPointers,data=data)

def VkAccelerationStructureGeometryDataKHR(triangles=None,aabbs=None,instances=None,):

    return _new('VkAccelerationStructureGeometryDataKHR', triangles=triangles,aabbs=aabbs,instances=instances)

def VkAccelerationStructureGeometryKHR(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_KHR,pNext=None,geometryType=None,geometry=None,flags=None,):

    return _new('VkAccelerationStructureGeometryKHR', sType=sType,pNext=pNext,geometryType=geometryType,geometry=geometry,flags=flags)

def VkAccelerationStructureBuildGeometryInfoKHR(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_BUILD_GEOMETRY_INFO_KHR,pNext=None,type=None,flags=None,mode=None,srcAccelerationStructure=None,dstAccelerationStructure=None,geometryCount=None,pGeometries=None,ppGeometries=None,scratchData=None,):
    if geometryCount is None and pGeometries is not None:
        geometryCount = len(pGeometries)
    if geometryCount is None and ppGeometries is not None:
        geometryCount = len(ppGeometries)

    return _new('VkAccelerationStructureBuildGeometryInfoKHR', sType=sType,pNext=pNext,type=type,flags=flags,mode=mode,srcAccelerationStructure=srcAccelerationStructure,dstAccelerationStructure=dstAccelerationStructure,geometryCount=geometryCount,pGeometries=pGeometries,ppGeometries=ppGeometries,scratchData=scratchData)

def VkAccelerationStructureBuildRangeInfoKHR(primitiveCount=None,primitiveOffset=None,firstVertex=None,transformOffset=None,):

    return _new('VkAccelerationStructureBuildRangeInfoKHR', primitiveCount=primitiveCount,primitiveOffset=primitiveOffset,firstVertex=firstVertex,transformOffset=transformOffset)

def VkAccelerationStructureCreateInfoKHR(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_CREATE_INFO_KHR,pNext=None,createFlags=None,buffer=None,offset=None,size=None,type=None,deviceAddress=None,):

    return _new('VkAccelerationStructureCreateInfoKHR', sType=sType,pNext=pNext,createFlags=createFlags,buffer=buffer,offset=offset,size=size,type=type,deviceAddress=deviceAddress)

def VkAabbPositionsKHR(minX=None,minY=None,minZ=None,maxX=None,maxY=None,maxZ=None,):

    return _new('VkAabbPositionsKHR', minX=minX,minY=minY,minZ=minZ,maxX=maxX,maxY=maxY,maxZ=maxZ)

def VkTransformMatrixKHR(matrix=None,):

    return _new('VkTransformMatrixKHR', matrix=matrix)

def VkAccelerationStructureInstanceKHR(transform=None,instanceCustomIndex=None,mask=None,instanceShaderBindingTableRecordOffset=None,flags=None,accelerationStructureReference=None,):

    return _new('VkAccelerationStructureInstanceKHR', transform=transform,instanceCustomIndex=instanceCustomIndex,mask=mask,instanceShaderBindingTableRecordOffset=instanceShaderBindingTableRecordOffset,flags=flags,accelerationStructureReference=accelerationStructureReference)

def VkAccelerationStructureDeviceAddressInfoKHR(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_DEVICE_ADDRESS_INFO_KHR,pNext=None,accelerationStructure=None,):

    return _new('VkAccelerationStructureDeviceAddressInfoKHR', sType=sType,pNext=pNext,accelerationStructure=accelerationStructure)

def VkAccelerationStructureVersionInfoKHR(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_VERSION_INFO_KHR,pNext=None,pVersionData=None,):

    return _new('VkAccelerationStructureVersionInfoKHR', sType=sType,pNext=pNext,pVersionData=pVersionData)

def VkCopyAccelerationStructureInfoKHR(sType=VK_STRUCTURE_TYPE_COPY_ACCELERATION_STRUCTURE_INFO_KHR,pNext=None,src=None,dst=None,mode=None,):

    return _new('VkCopyAccelerationStructureInfoKHR', sType=sType,pNext=pNext,src=src,dst=dst,mode=mode)

def VkCopyAccelerationStructureToMemoryInfoKHR(sType=VK_STRUCTURE_TYPE_COPY_ACCELERATION_STRUCTURE_TO_MEMORY_INFO_KHR,pNext=None,src=None,dst=None,mode=None,):

    return _new('VkCopyAccelerationStructureToMemoryInfoKHR', sType=sType,pNext=pNext,src=src,dst=dst,mode=mode)

def VkCopyMemoryToAccelerationStructureInfoKHR(sType=VK_STRUCTURE_TYPE_COPY_MEMORY_TO_ACCELERATION_STRUCTURE_INFO_KHR,pNext=None,src=None,dst=None,mode=None,):

    return _new('VkCopyMemoryToAccelerationStructureInfoKHR', sType=sType,pNext=pNext,src=src,dst=dst,mode=mode)

def VkRayTracingPipelineInterfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_INTERFACE_CREATE_INFO_KHR,pNext=None,maxPipelineRayPayloadSize=None,maxPipelineRayHitAttributeSize=None,):

    return _new('VkRayTracingPipelineInterfaceCreateInfoKHR', sType=sType,pNext=pNext,maxPipelineRayPayloadSize=maxPipelineRayPayloadSize,maxPipelineRayHitAttributeSize=maxPipelineRayHitAttributeSize)

def VkPipelineLibraryCreateInfoKHR(sType=VK_STRUCTURE_TYPE_PIPELINE_LIBRARY_CREATE_INFO_KHR,pNext=None,libraryCount=None,pLibraries=None,):
    if libraryCount is None and pLibraries is not None:
        libraryCount = len(pLibraries)

    return _new('VkPipelineLibraryCreateInfoKHR', sType=sType,pNext=pNext,libraryCount=libraryCount,pLibraries=pLibraries)

def VkRefreshObjectKHR(objectType=None,objectHandle=None,flags=None,):

    return _new('VkRefreshObjectKHR', objectType=objectType,objectHandle=objectHandle,flags=flags)

def VkRefreshObjectListKHR(sType=VK_STRUCTURE_TYPE_REFRESH_OBJECT_LIST_KHR,pNext=None,objectCount=None,pObjects=None,):
    if objectCount is None and pObjects is not None:
        objectCount = len(pObjects)

    return _new('VkRefreshObjectListKHR', sType=sType,pNext=pNext,objectCount=objectCount,pObjects=pObjects)

def VkPhysicalDeviceExtendedDynamicStateFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_DYNAMIC_STATE_FEATURES_EXT,pNext=None,extendedDynamicState=None,):

    return _new('VkPhysicalDeviceExtendedDynamicStateFeaturesEXT', sType=sType,pNext=pNext,extendedDynamicState=extendedDynamicState)

def VkPhysicalDeviceExtendedDynamicState2FeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_DYNAMIC_STATE_2_FEATURES_EXT,pNext=None,extendedDynamicState2=None,extendedDynamicState2LogicOp=None,extendedDynamicState2PatchControlPoints=None,):

    return _new('VkPhysicalDeviceExtendedDynamicState2FeaturesEXT', sType=sType,pNext=pNext,extendedDynamicState2=extendedDynamicState2,extendedDynamicState2LogicOp=extendedDynamicState2LogicOp,extendedDynamicState2PatchControlPoints=extendedDynamicState2PatchControlPoints)

def VkPhysicalDeviceExtendedDynamicState3FeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_DYNAMIC_STATE_3_FEATURES_EXT,pNext=None,extendedDynamicState3TessellationDomainOrigin=None,extendedDynamicState3DepthClampEnable=None,extendedDynamicState3PolygonMode=None,extendedDynamicState3RasterizationSamples=None,extendedDynamicState3SampleMask=None,extendedDynamicState3AlphaToCoverageEnable=None,extendedDynamicState3AlphaToOneEnable=None,extendedDynamicState3LogicOpEnable=None,extendedDynamicState3ColorBlendEnable=None,extendedDynamicState3ColorBlendEquation=None,extendedDynamicState3ColorWriteMask=None,extendedDynamicState3RasterizationStream=None,extendedDynamicState3ConservativeRasterizationMode=None,extendedDynamicState3ExtraPrimitiveOverestimationSize=None,extendedDynamicState3DepthClipEnable=None,extendedDynamicState3SampleLocationsEnable=None,extendedDynamicState3ColorBlendAdvanced=None,extendedDynamicState3ProvokingVertexMode=None,extendedDynamicState3LineRasterizationMode=None,extendedDynamicState3LineStippleEnable=None,extendedDynamicState3DepthClipNegativeOneToOne=None,extendedDynamicState3ViewportWScalingEnable=None,extendedDynamicState3ViewportSwizzle=None,extendedDynamicState3CoverageToColorEnable=None,extendedDynamicState3CoverageToColorLocation=None,extendedDynamicState3CoverageModulationMode=None,extendedDynamicState3CoverageModulationTableEnable=None,extendedDynamicState3CoverageModulationTable=None,extendedDynamicState3CoverageReductionMode=None,extendedDynamicState3RepresentativeFragmentTestEnable=None,extendedDynamicState3ShadingRateImageEnable=None,):

    return _new('VkPhysicalDeviceExtendedDynamicState3FeaturesEXT', sType=sType,pNext=pNext,extendedDynamicState3TessellationDomainOrigin=extendedDynamicState3TessellationDomainOrigin,extendedDynamicState3DepthClampEnable=extendedDynamicState3DepthClampEnable,extendedDynamicState3PolygonMode=extendedDynamicState3PolygonMode,extendedDynamicState3RasterizationSamples=extendedDynamicState3RasterizationSamples,extendedDynamicState3SampleMask=extendedDynamicState3SampleMask,extendedDynamicState3AlphaToCoverageEnable=extendedDynamicState3AlphaToCoverageEnable,extendedDynamicState3AlphaToOneEnable=extendedDynamicState3AlphaToOneEnable,extendedDynamicState3LogicOpEnable=extendedDynamicState3LogicOpEnable,extendedDynamicState3ColorBlendEnable=extendedDynamicState3ColorBlendEnable,extendedDynamicState3ColorBlendEquation=extendedDynamicState3ColorBlendEquation,extendedDynamicState3ColorWriteMask=extendedDynamicState3ColorWriteMask,extendedDynamicState3RasterizationStream=extendedDynamicState3RasterizationStream,extendedDynamicState3ConservativeRasterizationMode=extendedDynamicState3ConservativeRasterizationMode,extendedDynamicState3ExtraPrimitiveOverestimationSize=extendedDynamicState3ExtraPrimitiveOverestimationSize,extendedDynamicState3DepthClipEnable=extendedDynamicState3DepthClipEnable,extendedDynamicState3SampleLocationsEnable=extendedDynamicState3SampleLocationsEnable,extendedDynamicState3ColorBlendAdvanced=extendedDynamicState3ColorBlendAdvanced,extendedDynamicState3ProvokingVertexMode=extendedDynamicState3ProvokingVertexMode,extendedDynamicState3LineRasterizationMode=extendedDynamicState3LineRasterizationMode,extendedDynamicState3LineStippleEnable=extendedDynamicState3LineStippleEnable,extendedDynamicState3DepthClipNegativeOneToOne=extendedDynamicState3DepthClipNegativeOneToOne,extendedDynamicState3ViewportWScalingEnable=extendedDynamicState3ViewportWScalingEnable,extendedDynamicState3ViewportSwizzle=extendedDynamicState3ViewportSwizzle,extendedDynamicState3CoverageToColorEnable=extendedDynamicState3CoverageToColorEnable,extendedDynamicState3CoverageToColorLocation=extendedDynamicState3CoverageToColorLocation,extendedDynamicState3CoverageModulationMode=extendedDynamicState3CoverageModulationMode,extendedDynamicState3CoverageModulationTableEnable=extendedDynamicState3CoverageModulationTableEnable,extendedDynamicState3CoverageModulationTable=extendedDynamicState3CoverageModulationTable,extendedDynamicState3CoverageReductionMode=extendedDynamicState3CoverageReductionMode,extendedDynamicState3RepresentativeFragmentTestEnable=extendedDynamicState3RepresentativeFragmentTestEnable,extendedDynamicState3ShadingRateImageEnable=extendedDynamicState3ShadingRateImageEnable)

def VkPhysicalDeviceExtendedDynamicState3PropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_DYNAMIC_STATE_3_PROPERTIES_EXT,pNext=None,dynamicPrimitiveTopologyUnrestricted=None,):

    return _new('VkPhysicalDeviceExtendedDynamicState3PropertiesEXT', sType=sType,pNext=pNext,dynamicPrimitiveTopologyUnrestricted=dynamicPrimitiveTopologyUnrestricted)

def VkColorBlendEquationEXT(srcColorBlendFactor=None,dstColorBlendFactor=None,colorBlendOp=None,srcAlphaBlendFactor=None,dstAlphaBlendFactor=None,alphaBlendOp=None,):

    return _new('VkColorBlendEquationEXT', srcColorBlendFactor=srcColorBlendFactor,dstColorBlendFactor=dstColorBlendFactor,colorBlendOp=colorBlendOp,srcAlphaBlendFactor=srcAlphaBlendFactor,dstAlphaBlendFactor=dstAlphaBlendFactor,alphaBlendOp=alphaBlendOp)

def VkColorBlendAdvancedEXT(advancedBlendOp=None,srcPremultiplied=None,dstPremultiplied=None,blendOverlap=None,clampResults=None,):

    return _new('VkColorBlendAdvancedEXT', advancedBlendOp=advancedBlendOp,srcPremultiplied=srcPremultiplied,dstPremultiplied=dstPremultiplied,blendOverlap=blendOverlap,clampResults=clampResults)

def VkRenderPassTransformBeginInfoQCOM(sType=VK_STRUCTURE_TYPE_RENDER_PASS_TRANSFORM_BEGIN_INFO_QCOM,pNext=None,transform=None,):

    return _new('VkRenderPassTransformBeginInfoQCOM', sType=sType,pNext=pNext,transform=transform)

def VkCopyCommandTransformInfoQCOM(sType=VK_STRUCTURE_TYPE_COPY_COMMAND_TRANSFORM_INFO_QCOM,pNext=None,transform=None,):

    return _new('VkCopyCommandTransformInfoQCOM', sType=sType,pNext=pNext,transform=transform)

def VkCommandBufferInheritanceRenderPassTransformInfoQCOM(sType=VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_RENDER_PASS_TRANSFORM_INFO_QCOM,pNext=None,transform=None,renderArea=None,):

    return _new('VkCommandBufferInheritanceRenderPassTransformInfoQCOM', sType=sType,pNext=pNext,transform=transform,renderArea=renderArea)

def VkPhysicalDeviceDiagnosticsConfigFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DIAGNOSTICS_CONFIG_FEATURES_NV,pNext=None,diagnosticsConfig=None,):

    return _new('VkPhysicalDeviceDiagnosticsConfigFeaturesNV', sType=sType,pNext=pNext,diagnosticsConfig=diagnosticsConfig)

def VkDeviceDiagnosticsConfigCreateInfoNV(sType=VK_STRUCTURE_TYPE_DEVICE_DIAGNOSTICS_CONFIG_CREATE_INFO_NV,pNext=None,flags=None,):

    return _new('VkDeviceDiagnosticsConfigCreateInfoNV', sType=sType,pNext=pNext,flags=flags)

def VkPipelineOfflineCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_OFFLINE_CREATE_INFO,pNext=None,pipelineIdentifier=None,matchControl=None,poolEntrySize=None,):

    return _new('VkPipelineOfflineCreateInfo', sType=sType,pNext=pNext,pipelineIdentifier=pipelineIdentifier,matchControl=matchControl,poolEntrySize=poolEntrySize)

def VkPhysicalDeviceZeroInitializeWorkgroupMemoryFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ZERO_INITIALIZE_WORKGROUP_MEMORY_FEATURES,pNext=None,shaderZeroInitializeWorkgroupMemory=None,):

    return _new('VkPhysicalDeviceZeroInitializeWorkgroupMemoryFeatures', sType=sType,pNext=pNext,shaderZeroInitializeWorkgroupMemory=shaderZeroInitializeWorkgroupMemory)

def VkPhysicalDeviceShaderSubgroupUniformControlFlowFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_SUBGROUP_UNIFORM_CONTROL_FLOW_FEATURES_KHR,pNext=None,shaderSubgroupUniformControlFlow=None,):

    return _new('VkPhysicalDeviceShaderSubgroupUniformControlFlowFeaturesKHR', sType=sType,pNext=pNext,shaderSubgroupUniformControlFlow=shaderSubgroupUniformControlFlow)

def VkPhysicalDeviceRobustness2FeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ROBUSTNESS_2_FEATURES_EXT,pNext=None,robustBufferAccess2=None,robustImageAccess2=None,nullDescriptor=None,):

    return _new('VkPhysicalDeviceRobustness2FeaturesEXT', sType=sType,pNext=pNext,robustBufferAccess2=robustBufferAccess2,robustImageAccess2=robustImageAccess2,nullDescriptor=nullDescriptor)

def VkPhysicalDeviceRobustness2PropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ROBUSTNESS_2_PROPERTIES_EXT,pNext=None,robustStorageBufferAccessSizeAlignment=None,robustUniformBufferAccessSizeAlignment=None,):

    return _new('VkPhysicalDeviceRobustness2PropertiesEXT', sType=sType,pNext=pNext,robustStorageBufferAccessSizeAlignment=robustStorageBufferAccessSizeAlignment,robustUniformBufferAccessSizeAlignment=robustUniformBufferAccessSizeAlignment)

def VkPhysicalDeviceImageRobustnessFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_ROBUSTNESS_FEATURES,pNext=None,robustImageAccess=None,):

    return _new('VkPhysicalDeviceImageRobustnessFeatures', sType=sType,pNext=pNext,robustImageAccess=robustImageAccess)

def VkPhysicalDeviceWorkgroupMemoryExplicitLayoutFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_WORKGROUP_MEMORY_EXPLICIT_LAYOUT_FEATURES_KHR,pNext=None,workgroupMemoryExplicitLayout=None,workgroupMemoryExplicitLayoutScalarBlockLayout=None,workgroupMemoryExplicitLayout8BitAccess=None,workgroupMemoryExplicitLayout16BitAccess=None,):

    return _new('VkPhysicalDeviceWorkgroupMemoryExplicitLayoutFeaturesKHR', sType=sType,pNext=pNext,workgroupMemoryExplicitLayout=workgroupMemoryExplicitLayout,workgroupMemoryExplicitLayoutScalarBlockLayout=workgroupMemoryExplicitLayoutScalarBlockLayout,workgroupMemoryExplicitLayout8BitAccess=workgroupMemoryExplicitLayout8BitAccess,workgroupMemoryExplicitLayout16BitAccess=workgroupMemoryExplicitLayout16BitAccess)

def VkPhysicalDevicePortabilitySubsetFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PORTABILITY_SUBSET_FEATURES_KHR,pNext=None,constantAlphaColorBlendFactors=None,events=None,imageViewFormatReinterpretation=None,imageViewFormatSwizzle=None,imageView2DOn3DImage=None,multisampleArrayImage=None,mutableComparisonSamplers=None,pointPolygons=None,samplerMipLodBias=None,separateStencilMaskRef=None,shaderSampleRateInterpolationFunctions=None,tessellationIsolines=None,tessellationPointMode=None,triangleFans=None,vertexAttributeAccessBeyondStride=None,):

    return _new('VkPhysicalDevicePortabilitySubsetFeaturesKHR', sType=sType,pNext=pNext,constantAlphaColorBlendFactors=constantAlphaColorBlendFactors,events=events,imageViewFormatReinterpretation=imageViewFormatReinterpretation,imageViewFormatSwizzle=imageViewFormatSwizzle,imageView2DOn3DImage=imageView2DOn3DImage,multisampleArrayImage=multisampleArrayImage,mutableComparisonSamplers=mutableComparisonSamplers,pointPolygons=pointPolygons,samplerMipLodBias=samplerMipLodBias,separateStencilMaskRef=separateStencilMaskRef,shaderSampleRateInterpolationFunctions=shaderSampleRateInterpolationFunctions,tessellationIsolines=tessellationIsolines,tessellationPointMode=tessellationPointMode,triangleFans=triangleFans,vertexAttributeAccessBeyondStride=vertexAttributeAccessBeyondStride)

def VkPhysicalDevicePortabilitySubsetPropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PORTABILITY_SUBSET_PROPERTIES_KHR,pNext=None,minVertexInputBindingStrideAlignment=None,):

    return _new('VkPhysicalDevicePortabilitySubsetPropertiesKHR', sType=sType,pNext=pNext,minVertexInputBindingStrideAlignment=minVertexInputBindingStrideAlignment)

def VkPhysicalDevice4444FormatsFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_4444_FORMATS_FEATURES_EXT,pNext=None,formatA4R4G4B4=None,formatA4B4G4R4=None,):

    return _new('VkPhysicalDevice4444FormatsFeaturesEXT', sType=sType,pNext=pNext,formatA4R4G4B4=formatA4R4G4B4,formatA4B4G4R4=formatA4B4G4R4)

def VkPhysicalDeviceSubpassShadingFeaturesHUAWEI(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBPASS_SHADING_FEATURES_HUAWEI,pNext=None,subpassShading=None,):

    return _new('VkPhysicalDeviceSubpassShadingFeaturesHUAWEI', sType=sType,pNext=pNext,subpassShading=subpassShading)

def VkPhysicalDeviceClusterCullingShaderFeaturesHUAWEI(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CLUSTER_CULLING_SHADER_FEATURES_HUAWEI,pNext=None,clustercullingShader=None,multiviewClusterCullingShader=None,):

    return _new('VkPhysicalDeviceClusterCullingShaderFeaturesHUAWEI', sType=sType,pNext=pNext,clustercullingShader=clustercullingShader,multiviewClusterCullingShader=multiviewClusterCullingShader)

def VkPhysicalDeviceClusterCullingShaderVrsFeaturesHUAWEI(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CLUSTER_CULLING_SHADER_VRS_FEATURES_HUAWEI,pNext=None,clusterShadingRate=None,):

    return _new('VkPhysicalDeviceClusterCullingShaderVrsFeaturesHUAWEI', sType=sType,pNext=pNext,clusterShadingRate=clusterShadingRate)

def VkBufferCopy2(sType=VK_STRUCTURE_TYPE_BUFFER_COPY_2,pNext=None,srcOffset=None,dstOffset=None,size=None,):

    return _new('VkBufferCopy2', sType=sType,pNext=pNext,srcOffset=srcOffset,dstOffset=dstOffset,size=size)

def VkImageCopy2(sType=VK_STRUCTURE_TYPE_IMAGE_COPY_2,pNext=None,srcSubresource=None,srcOffset=None,dstSubresource=None,dstOffset=None,extent=None,):

    return _new('VkImageCopy2', sType=sType,pNext=pNext,srcSubresource=srcSubresource,srcOffset=srcOffset,dstSubresource=dstSubresource,dstOffset=dstOffset,extent=extent)

def VkImageBlit2(sType=VK_STRUCTURE_TYPE_IMAGE_BLIT_2,pNext=None,srcSubresource=None,srcOffsets=None,dstSubresource=None,dstOffsets=None,):

    return _new('VkImageBlit2', sType=sType,pNext=pNext,srcSubresource=srcSubresource,srcOffsets=srcOffsets,dstSubresource=dstSubresource,dstOffsets=dstOffsets)

def VkBufferImageCopy2(sType=VK_STRUCTURE_TYPE_BUFFER_IMAGE_COPY_2,pNext=None,bufferOffset=None,bufferRowLength=None,bufferImageHeight=None,imageSubresource=None,imageOffset=None,imageExtent=None,):

    return _new('VkBufferImageCopy2', sType=sType,pNext=pNext,bufferOffset=bufferOffset,bufferRowLength=bufferRowLength,bufferImageHeight=bufferImageHeight,imageSubresource=imageSubresource,imageOffset=imageOffset,imageExtent=imageExtent)

def VkImageResolve2(sType=VK_STRUCTURE_TYPE_IMAGE_RESOLVE_2,pNext=None,srcSubresource=None,srcOffset=None,dstSubresource=None,dstOffset=None,extent=None,):

    return _new('VkImageResolve2', sType=sType,pNext=pNext,srcSubresource=srcSubresource,srcOffset=srcOffset,dstSubresource=dstSubresource,dstOffset=dstOffset,extent=extent)

def VkCopyBufferInfo2(sType=VK_STRUCTURE_TYPE_COPY_BUFFER_INFO_2,pNext=None,srcBuffer=None,dstBuffer=None,regionCount=None,pRegions=None,):
    if regionCount is None and pRegions is not None:
        regionCount = len(pRegions)

    return _new('VkCopyBufferInfo2', sType=sType,pNext=pNext,srcBuffer=srcBuffer,dstBuffer=dstBuffer,regionCount=regionCount,pRegions=pRegions)

def VkCopyImageInfo2(sType=VK_STRUCTURE_TYPE_COPY_IMAGE_INFO_2,pNext=None,srcImage=None,srcImageLayout=None,dstImage=None,dstImageLayout=None,regionCount=None,pRegions=None,):
    if regionCount is None and pRegions is not None:
        regionCount = len(pRegions)

    return _new('VkCopyImageInfo2', sType=sType,pNext=pNext,srcImage=srcImage,srcImageLayout=srcImageLayout,dstImage=dstImage,dstImageLayout=dstImageLayout,regionCount=regionCount,pRegions=pRegions)

def VkBlitImageInfo2(sType=VK_STRUCTURE_TYPE_BLIT_IMAGE_INFO_2,pNext=None,srcImage=None,srcImageLayout=None,dstImage=None,dstImageLayout=None,regionCount=None,pRegions=None,filter=None,):
    if regionCount is None and pRegions is not None:
        regionCount = len(pRegions)

    return _new('VkBlitImageInfo2', sType=sType,pNext=pNext,srcImage=srcImage,srcImageLayout=srcImageLayout,dstImage=dstImage,dstImageLayout=dstImageLayout,regionCount=regionCount,pRegions=pRegions,filter=filter)

def VkCopyBufferToImageInfo2(sType=VK_STRUCTURE_TYPE_COPY_BUFFER_TO_IMAGE_INFO_2,pNext=None,srcBuffer=None,dstImage=None,dstImageLayout=None,regionCount=None,pRegions=None,):
    if regionCount is None and pRegions is not None:
        regionCount = len(pRegions)

    return _new('VkCopyBufferToImageInfo2', sType=sType,pNext=pNext,srcBuffer=srcBuffer,dstImage=dstImage,dstImageLayout=dstImageLayout,regionCount=regionCount,pRegions=pRegions)

def VkCopyImageToBufferInfo2(sType=VK_STRUCTURE_TYPE_COPY_IMAGE_TO_BUFFER_INFO_2,pNext=None,srcImage=None,srcImageLayout=None,dstBuffer=None,regionCount=None,pRegions=None,):
    if regionCount is None and pRegions is not None:
        regionCount = len(pRegions)

    return _new('VkCopyImageToBufferInfo2', sType=sType,pNext=pNext,srcImage=srcImage,srcImageLayout=srcImageLayout,dstBuffer=dstBuffer,regionCount=regionCount,pRegions=pRegions)

def VkResolveImageInfo2(sType=VK_STRUCTURE_TYPE_RESOLVE_IMAGE_INFO_2,pNext=None,srcImage=None,srcImageLayout=None,dstImage=None,dstImageLayout=None,regionCount=None,pRegions=None,):
    if regionCount is None and pRegions is not None:
        regionCount = len(pRegions)

    return _new('VkResolveImageInfo2', sType=sType,pNext=pNext,srcImage=srcImage,srcImageLayout=srcImageLayout,dstImage=dstImage,dstImageLayout=dstImageLayout,regionCount=regionCount,pRegions=pRegions)

def VkPhysicalDeviceShaderImageAtomicInt64FeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_IMAGE_ATOMIC_INT64_FEATURES_EXT,pNext=None,shaderImageInt64Atomics=None,sparseImageInt64Atomics=None,):

    return _new('VkPhysicalDeviceShaderImageAtomicInt64FeaturesEXT', sType=sType,pNext=pNext,shaderImageInt64Atomics=shaderImageInt64Atomics,sparseImageInt64Atomics=sparseImageInt64Atomics)

def VkFragmentShadingRateAttachmentInfoKHR(sType=VK_STRUCTURE_TYPE_FRAGMENT_SHADING_RATE_ATTACHMENT_INFO_KHR,pNext=None,pFragmentShadingRateAttachment=None,shadingRateAttachmentTexelSize=None,):

    return _new('VkFragmentShadingRateAttachmentInfoKHR', sType=sType,pNext=pNext,pFragmentShadingRateAttachment=pFragmentShadingRateAttachment,shadingRateAttachmentTexelSize=shadingRateAttachmentTexelSize)

def VkPipelineFragmentShadingRateStateCreateInfoKHR(sType=VK_STRUCTURE_TYPE_PIPELINE_FRAGMENT_SHADING_RATE_STATE_CREATE_INFO_KHR,pNext=None,fragmentSize=None,combinerOps=None,):

    return _new('VkPipelineFragmentShadingRateStateCreateInfoKHR', sType=sType,pNext=pNext,fragmentSize=fragmentSize,combinerOps=combinerOps)

def VkPhysicalDeviceFragmentShadingRateFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_FEATURES_KHR,pNext=None,pipelineFragmentShadingRate=None,primitiveFragmentShadingRate=None,attachmentFragmentShadingRate=None,):

    return _new('VkPhysicalDeviceFragmentShadingRateFeaturesKHR', sType=sType,pNext=pNext,pipelineFragmentShadingRate=pipelineFragmentShadingRate,primitiveFragmentShadingRate=primitiveFragmentShadingRate,attachmentFragmentShadingRate=attachmentFragmentShadingRate)

def VkPhysicalDeviceFragmentShadingRatePropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_PROPERTIES_KHR,pNext=None,minFragmentShadingRateAttachmentTexelSize=None,maxFragmentShadingRateAttachmentTexelSize=None,maxFragmentShadingRateAttachmentTexelSizeAspectRatio=None,primitiveFragmentShadingRateWithMultipleViewports=None,layeredShadingRateAttachments=None,fragmentShadingRateNonTrivialCombinerOps=None,maxFragmentSize=None,maxFragmentSizeAspectRatio=None,maxFragmentShadingRateCoverageSamples=None,maxFragmentShadingRateRasterizationSamples=None,fragmentShadingRateWithShaderDepthStencilWrites=None,fragmentShadingRateWithSampleMask=None,fragmentShadingRateWithShaderSampleMask=None,fragmentShadingRateWithConservativeRasterization=None,fragmentShadingRateWithFragmentShaderInterlock=None,fragmentShadingRateWithCustomSampleLocations=None,fragmentShadingRateStrictMultiplyCombiner=None,):

    return _new('VkPhysicalDeviceFragmentShadingRatePropertiesKHR', sType=sType,pNext=pNext,minFragmentShadingRateAttachmentTexelSize=minFragmentShadingRateAttachmentTexelSize,maxFragmentShadingRateAttachmentTexelSize=maxFragmentShadingRateAttachmentTexelSize,maxFragmentShadingRateAttachmentTexelSizeAspectRatio=maxFragmentShadingRateAttachmentTexelSizeAspectRatio,primitiveFragmentShadingRateWithMultipleViewports=primitiveFragmentShadingRateWithMultipleViewports,layeredShadingRateAttachments=layeredShadingRateAttachments,fragmentShadingRateNonTrivialCombinerOps=fragmentShadingRateNonTrivialCombinerOps,maxFragmentSize=maxFragmentSize,maxFragmentSizeAspectRatio=maxFragmentSizeAspectRatio,maxFragmentShadingRateCoverageSamples=maxFragmentShadingRateCoverageSamples,maxFragmentShadingRateRasterizationSamples=maxFragmentShadingRateRasterizationSamples,fragmentShadingRateWithShaderDepthStencilWrites=fragmentShadingRateWithShaderDepthStencilWrites,fragmentShadingRateWithSampleMask=fragmentShadingRateWithSampleMask,fragmentShadingRateWithShaderSampleMask=fragmentShadingRateWithShaderSampleMask,fragmentShadingRateWithConservativeRasterization=fragmentShadingRateWithConservativeRasterization,fragmentShadingRateWithFragmentShaderInterlock=fragmentShadingRateWithFragmentShaderInterlock,fragmentShadingRateWithCustomSampleLocations=fragmentShadingRateWithCustomSampleLocations,fragmentShadingRateStrictMultiplyCombiner=fragmentShadingRateStrictMultiplyCombiner)

def VkPhysicalDeviceFragmentShadingRateKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_KHR,pNext=None,sampleCounts=None,fragmentSize=None,):

    return _new('VkPhysicalDeviceFragmentShadingRateKHR', sType=sType,pNext=pNext,sampleCounts=sampleCounts,fragmentSize=fragmentSize)

def VkPhysicalDeviceShaderTerminateInvocationFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_TERMINATE_INVOCATION_FEATURES,pNext=None,shaderTerminateInvocation=None,):

    return _new('VkPhysicalDeviceShaderTerminateInvocationFeatures', sType=sType,pNext=pNext,shaderTerminateInvocation=shaderTerminateInvocation)

def VkPhysicalDeviceFragmentShadingRateEnumsFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_ENUMS_FEATURES_NV,pNext=None,fragmentShadingRateEnums=None,supersampleFragmentShadingRates=None,noInvocationFragmentShadingRates=None,):

    return _new('VkPhysicalDeviceFragmentShadingRateEnumsFeaturesNV', sType=sType,pNext=pNext,fragmentShadingRateEnums=fragmentShadingRateEnums,supersampleFragmentShadingRates=supersampleFragmentShadingRates,noInvocationFragmentShadingRates=noInvocationFragmentShadingRates)

def VkPhysicalDeviceFragmentShadingRateEnumsPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_ENUMS_PROPERTIES_NV,pNext=None,maxFragmentShadingRateInvocationCount=None,):

    return _new('VkPhysicalDeviceFragmentShadingRateEnumsPropertiesNV', sType=sType,pNext=pNext,maxFragmentShadingRateInvocationCount=maxFragmentShadingRateInvocationCount)

def VkPipelineFragmentShadingRateEnumStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_FRAGMENT_SHADING_RATE_ENUM_STATE_CREATE_INFO_NV,pNext=None,shadingRateType=None,shadingRate=None,combinerOps=None,):

    return _new('VkPipelineFragmentShadingRateEnumStateCreateInfoNV', sType=sType,pNext=pNext,shadingRateType=shadingRateType,shadingRate=shadingRate,combinerOps=combinerOps)

def VkAccelerationStructureBuildSizesInfoKHR(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_BUILD_SIZES_INFO_KHR,pNext=None,accelerationStructureSize=None,updateScratchSize=None,buildScratchSize=None,):

    return _new('VkAccelerationStructureBuildSizesInfoKHR', sType=sType,pNext=pNext,accelerationStructureSize=accelerationStructureSize,updateScratchSize=updateScratchSize,buildScratchSize=buildScratchSize)

def VkPhysicalDeviceImage2DViewOf3DFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_2D_VIEW_OF_3D_FEATURES_EXT,pNext=None,image2DViewOf3D=None,sampler2DViewOf3D=None,):

    return _new('VkPhysicalDeviceImage2DViewOf3DFeaturesEXT', sType=sType,pNext=pNext,image2DViewOf3D=image2DViewOf3D,sampler2DViewOf3D=sampler2DViewOf3D)

def VkPhysicalDeviceImageSlicedViewOf3DFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_SLICED_VIEW_OF_3D_FEATURES_EXT,pNext=None,imageSlicedViewOf3D=None,):

    return _new('VkPhysicalDeviceImageSlicedViewOf3DFeaturesEXT', sType=sType,pNext=pNext,imageSlicedViewOf3D=imageSlicedViewOf3D)

def VkPhysicalDeviceAttachmentFeedbackLoopDynamicStateFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ATTACHMENT_FEEDBACK_LOOP_DYNAMIC_STATE_FEATURES_EXT,pNext=None,attachmentFeedbackLoopDynamicState=None,):

    return _new('VkPhysicalDeviceAttachmentFeedbackLoopDynamicStateFeaturesEXT', sType=sType,pNext=pNext,attachmentFeedbackLoopDynamicState=attachmentFeedbackLoopDynamicState)

def VkPhysicalDeviceMutableDescriptorTypeFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MUTABLE_DESCRIPTOR_TYPE_FEATURES_EXT,pNext=None,mutableDescriptorType=None,):

    return _new('VkPhysicalDeviceMutableDescriptorTypeFeaturesEXT', sType=sType,pNext=pNext,mutableDescriptorType=mutableDescriptorType)

def VkMutableDescriptorTypeListEXT(descriptorTypeCount=None,pDescriptorTypes=None,):
    if descriptorTypeCount is None and pDescriptorTypes is not None:
        descriptorTypeCount = len(pDescriptorTypes)

    return _new('VkMutableDescriptorTypeListEXT', descriptorTypeCount=descriptorTypeCount,pDescriptorTypes=pDescriptorTypes)

def VkMutableDescriptorTypeCreateInfoEXT(sType=VK_STRUCTURE_TYPE_MUTABLE_DESCRIPTOR_TYPE_CREATE_INFO_EXT,pNext=None,mutableDescriptorTypeListCount=None,pMutableDescriptorTypeLists=None,):
    if mutableDescriptorTypeListCount is None and pMutableDescriptorTypeLists is not None:
        mutableDescriptorTypeListCount = len(pMutableDescriptorTypeLists)

    return _new('VkMutableDescriptorTypeCreateInfoEXT', sType=sType,pNext=pNext,mutableDescriptorTypeListCount=mutableDescriptorTypeListCount,pMutableDescriptorTypeLists=pMutableDescriptorTypeLists)

def VkPhysicalDeviceDepthClipControlFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_CLIP_CONTROL_FEATURES_EXT,pNext=None,depthClipControl=None,):

    return _new('VkPhysicalDeviceDepthClipControlFeaturesEXT', sType=sType,pNext=pNext,depthClipControl=depthClipControl)

def VkPipelineViewportDepthClipControlCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_DEPTH_CLIP_CONTROL_CREATE_INFO_EXT,pNext=None,negativeOneToOne=None,):

    return _new('VkPipelineViewportDepthClipControlCreateInfoEXT', sType=sType,pNext=pNext,negativeOneToOne=negativeOneToOne)

def VkPhysicalDeviceVertexInputDynamicStateFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_INPUT_DYNAMIC_STATE_FEATURES_EXT,pNext=None,vertexInputDynamicState=None,):

    return _new('VkPhysicalDeviceVertexInputDynamicStateFeaturesEXT', sType=sType,pNext=pNext,vertexInputDynamicState=vertexInputDynamicState)

def VkPhysicalDeviceExternalMemoryRDMAFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_RDMA_FEATURES_NV,pNext=None,externalMemoryRDMA=None,):

    return _new('VkPhysicalDeviceExternalMemoryRDMAFeaturesNV', sType=sType,pNext=pNext,externalMemoryRDMA=externalMemoryRDMA)

def VkVertexInputBindingDescription2EXT(sType=VK_STRUCTURE_TYPE_VERTEX_INPUT_BINDING_DESCRIPTION_2_EXT,pNext=None,binding=None,stride=None,inputRate=None,divisor=None,):

    return _new('VkVertexInputBindingDescription2EXT', sType=sType,pNext=pNext,binding=binding,stride=stride,inputRate=inputRate,divisor=divisor)

def VkVertexInputAttributeDescription2EXT(sType=VK_STRUCTURE_TYPE_VERTEX_INPUT_ATTRIBUTE_DESCRIPTION_2_EXT,pNext=None,location=None,binding=None,format=None,offset=None,):

    return _new('VkVertexInputAttributeDescription2EXT', sType=sType,pNext=pNext,location=location,binding=binding,format=format,offset=offset)

def VkPhysicalDeviceColorWriteEnableFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COLOR_WRITE_ENABLE_FEATURES_EXT,pNext=None,colorWriteEnable=None,):

    return _new('VkPhysicalDeviceColorWriteEnableFeaturesEXT', sType=sType,pNext=pNext,colorWriteEnable=colorWriteEnable)

def VkPipelineColorWriteCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_COLOR_WRITE_CREATE_INFO_EXT,pNext=None,attachmentCount=None,pColorWriteEnables=None,):
    if attachmentCount is None and pColorWriteEnables is not None:
        attachmentCount = len(pColorWriteEnables)

    return _new('VkPipelineColorWriteCreateInfoEXT', sType=sType,pNext=pNext,attachmentCount=attachmentCount,pColorWriteEnables=pColorWriteEnables)

def VkMemoryBarrier2(sType=VK_STRUCTURE_TYPE_MEMORY_BARRIER_2,pNext=None,srcStageMask=None,srcAccessMask=None,dstStageMask=None,dstAccessMask=None,):

    return _new('VkMemoryBarrier2', sType=sType,pNext=pNext,srcStageMask=srcStageMask,srcAccessMask=srcAccessMask,dstStageMask=dstStageMask,dstAccessMask=dstAccessMask)

def VkImageMemoryBarrier2(sType=VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER_2,pNext=None,srcStageMask=None,srcAccessMask=None,dstStageMask=None,dstAccessMask=None,oldLayout=None,newLayout=None,srcQueueFamilyIndex=None,dstQueueFamilyIndex=None,image=None,subresourceRange=None,):

    return _new('VkImageMemoryBarrier2', sType=sType,pNext=pNext,srcStageMask=srcStageMask,srcAccessMask=srcAccessMask,dstStageMask=dstStageMask,dstAccessMask=dstAccessMask,oldLayout=oldLayout,newLayout=newLayout,srcQueueFamilyIndex=srcQueueFamilyIndex,dstQueueFamilyIndex=dstQueueFamilyIndex,image=image,subresourceRange=subresourceRange)

def VkBufferMemoryBarrier2(sType=VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER_2,pNext=None,srcStageMask=None,srcAccessMask=None,dstStageMask=None,dstAccessMask=None,srcQueueFamilyIndex=None,dstQueueFamilyIndex=None,buffer=None,offset=None,size=None,):

    return _new('VkBufferMemoryBarrier2', sType=sType,pNext=pNext,srcStageMask=srcStageMask,srcAccessMask=srcAccessMask,dstStageMask=dstStageMask,dstAccessMask=dstAccessMask,srcQueueFamilyIndex=srcQueueFamilyIndex,dstQueueFamilyIndex=dstQueueFamilyIndex,buffer=buffer,offset=offset,size=size)

def VkDependencyInfo(sType=VK_STRUCTURE_TYPE_DEPENDENCY_INFO,pNext=None,dependencyFlags=None,memoryBarrierCount=None,pMemoryBarriers=None,bufferMemoryBarrierCount=None,pBufferMemoryBarriers=None,imageMemoryBarrierCount=None,pImageMemoryBarriers=None,):
    if memoryBarrierCount is None and pMemoryBarriers is not None:
        memoryBarrierCount = len(pMemoryBarriers)
    if bufferMemoryBarrierCount is None and pBufferMemoryBarriers is not None:
        bufferMemoryBarrierCount = len(pBufferMemoryBarriers)
    if imageMemoryBarrierCount is None and pImageMemoryBarriers is not None:
        imageMemoryBarrierCount = len(pImageMemoryBarriers)

    return _new('VkDependencyInfo', sType=sType,pNext=pNext,dependencyFlags=dependencyFlags,memoryBarrierCount=memoryBarrierCount,pMemoryBarriers=pMemoryBarriers,bufferMemoryBarrierCount=bufferMemoryBarrierCount,pBufferMemoryBarriers=pBufferMemoryBarriers,imageMemoryBarrierCount=imageMemoryBarrierCount,pImageMemoryBarriers=pImageMemoryBarriers)

def VkSemaphoreSubmitInfo(sType=VK_STRUCTURE_TYPE_SEMAPHORE_SUBMIT_INFO,pNext=None,semaphore=None,value=None,stageMask=None,deviceIndex=None,):

    return _new('VkSemaphoreSubmitInfo', sType=sType,pNext=pNext,semaphore=semaphore,value=value,stageMask=stageMask,deviceIndex=deviceIndex)

def VkCommandBufferSubmitInfo(sType=VK_STRUCTURE_TYPE_COMMAND_BUFFER_SUBMIT_INFO,pNext=None,commandBuffer=None,deviceMask=None,):

    return _new('VkCommandBufferSubmitInfo', sType=sType,pNext=pNext,commandBuffer=commandBuffer,deviceMask=deviceMask)

def VkSubmitInfo2(sType=VK_STRUCTURE_TYPE_SUBMIT_INFO_2,pNext=None,flags=None,waitSemaphoreInfoCount=None,pWaitSemaphoreInfos=None,commandBufferInfoCount=None,pCommandBufferInfos=None,signalSemaphoreInfoCount=None,pSignalSemaphoreInfos=None,):
    if waitSemaphoreInfoCount is None and pWaitSemaphoreInfos is not None:
        waitSemaphoreInfoCount = len(pWaitSemaphoreInfos)
    if commandBufferInfoCount is None and pCommandBufferInfos is not None:
        commandBufferInfoCount = len(pCommandBufferInfos)
    if signalSemaphoreInfoCount is None and pSignalSemaphoreInfos is not None:
        signalSemaphoreInfoCount = len(pSignalSemaphoreInfos)

    return _new('VkSubmitInfo2', sType=sType,pNext=pNext,flags=flags,waitSemaphoreInfoCount=waitSemaphoreInfoCount,pWaitSemaphoreInfos=pWaitSemaphoreInfos,commandBufferInfoCount=commandBufferInfoCount,pCommandBufferInfos=pCommandBufferInfos,signalSemaphoreInfoCount=signalSemaphoreInfoCount,pSignalSemaphoreInfos=pSignalSemaphoreInfos)

def VkQueueFamilyCheckpointProperties2NV(sType=VK_STRUCTURE_TYPE_QUEUE_FAMILY_CHECKPOINT_PROPERTIES_2_NV,pNext=None,checkpointExecutionStageMask=None,):

    return _new('VkQueueFamilyCheckpointProperties2NV', sType=sType,pNext=pNext,checkpointExecutionStageMask=checkpointExecutionStageMask)

def VkCheckpointData2NV(sType=VK_STRUCTURE_TYPE_CHECKPOINT_DATA_2_NV,pNext=None,stage=None,pCheckpointMarker=None,):

    return _new('VkCheckpointData2NV', sType=sType,pNext=pNext,stage=stage,pCheckpointMarker=pCheckpointMarker)

def VkPhysicalDeviceSynchronization2Features(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SYNCHRONIZATION_2_FEATURES,pNext=None,synchronization2=None,):

    return _new('VkPhysicalDeviceSynchronization2Features', sType=sType,pNext=pNext,synchronization2=synchronization2)

def VkPhysicalDeviceHostImageCopyFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_HOST_IMAGE_COPY_FEATURES_EXT,pNext=None,hostImageCopy=None,):

    return _new('VkPhysicalDeviceHostImageCopyFeaturesEXT', sType=sType,pNext=pNext,hostImageCopy=hostImageCopy)

def VkPhysicalDeviceHostImageCopyPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_HOST_IMAGE_COPY_PROPERTIES_EXT,pNext=None,copySrcLayoutCount=None,pCopySrcLayouts=None,copyDstLayoutCount=None,pCopyDstLayouts=None,optimalTilingLayoutUUID=None,identicalMemoryTypeRequirements=None,):
    if copySrcLayoutCount is None and pCopySrcLayouts is not None:
        copySrcLayoutCount = len(pCopySrcLayouts)
    if copyDstLayoutCount is None and pCopyDstLayouts is not None:
        copyDstLayoutCount = len(pCopyDstLayouts)

    return _new('VkPhysicalDeviceHostImageCopyPropertiesEXT', sType=sType,pNext=pNext,copySrcLayoutCount=copySrcLayoutCount,pCopySrcLayouts=pCopySrcLayouts,copyDstLayoutCount=copyDstLayoutCount,pCopyDstLayouts=pCopyDstLayouts,optimalTilingLayoutUUID=optimalTilingLayoutUUID,identicalMemoryTypeRequirements=identicalMemoryTypeRequirements)

def VkMemoryToImageCopyEXT(sType=VK_STRUCTURE_TYPE_MEMORY_TO_IMAGE_COPY_EXT,pNext=None,pHostPointer=None,memoryRowLength=None,memoryImageHeight=None,imageSubresource=None,imageOffset=None,imageExtent=None,):

    return _new('VkMemoryToImageCopyEXT', sType=sType,pNext=pNext,pHostPointer=pHostPointer,memoryRowLength=memoryRowLength,memoryImageHeight=memoryImageHeight,imageSubresource=imageSubresource,imageOffset=imageOffset,imageExtent=imageExtent)

def VkImageToMemoryCopyEXT(sType=VK_STRUCTURE_TYPE_IMAGE_TO_MEMORY_COPY_EXT,pNext=None,pHostPointer=None,memoryRowLength=None,memoryImageHeight=None,imageSubresource=None,imageOffset=None,imageExtent=None,):

    return _new('VkImageToMemoryCopyEXT', sType=sType,pNext=pNext,pHostPointer=pHostPointer,memoryRowLength=memoryRowLength,memoryImageHeight=memoryImageHeight,imageSubresource=imageSubresource,imageOffset=imageOffset,imageExtent=imageExtent)

def VkCopyMemoryToImageInfoEXT(sType=VK_STRUCTURE_TYPE_COPY_MEMORY_TO_IMAGE_INFO_EXT,pNext=None,flags=None,dstImage=None,dstImageLayout=None,regionCount=None,pRegions=None,):
    if regionCount is None and pRegions is not None:
        regionCount = len(pRegions)

    return _new('VkCopyMemoryToImageInfoEXT', sType=sType,pNext=pNext,flags=flags,dstImage=dstImage,dstImageLayout=dstImageLayout,regionCount=regionCount,pRegions=pRegions)

def VkCopyImageToMemoryInfoEXT(sType=VK_STRUCTURE_TYPE_COPY_IMAGE_TO_MEMORY_INFO_EXT,pNext=None,flags=None,srcImage=None,srcImageLayout=None,regionCount=None,pRegions=None,):
    if regionCount is None and pRegions is not None:
        regionCount = len(pRegions)

    return _new('VkCopyImageToMemoryInfoEXT', sType=sType,pNext=pNext,flags=flags,srcImage=srcImage,srcImageLayout=srcImageLayout,regionCount=regionCount,pRegions=pRegions)

def VkCopyImageToImageInfoEXT(sType=VK_STRUCTURE_TYPE_COPY_IMAGE_TO_IMAGE_INFO_EXT,pNext=None,flags=None,srcImage=None,srcImageLayout=None,dstImage=None,dstImageLayout=None,regionCount=None,pRegions=None,):
    if regionCount is None and pRegions is not None:
        regionCount = len(pRegions)

    return _new('VkCopyImageToImageInfoEXT', sType=sType,pNext=pNext,flags=flags,srcImage=srcImage,srcImageLayout=srcImageLayout,dstImage=dstImage,dstImageLayout=dstImageLayout,regionCount=regionCount,pRegions=pRegions)

def VkHostImageLayoutTransitionInfoEXT(sType=VK_STRUCTURE_TYPE_HOST_IMAGE_LAYOUT_TRANSITION_INFO_EXT,pNext=None,image=None,oldLayout=None,newLayout=None,subresourceRange=None,):

    return _new('VkHostImageLayoutTransitionInfoEXT', sType=sType,pNext=pNext,image=image,oldLayout=oldLayout,newLayout=newLayout,subresourceRange=subresourceRange)

def VkSubresourceHostMemcpySizeEXT(sType=VK_STRUCTURE_TYPE_SUBRESOURCE_HOST_MEMCPY_SIZE_EXT,pNext=None,size=None,):

    return _new('VkSubresourceHostMemcpySizeEXT', sType=sType,pNext=pNext,size=size)

def VkHostImageCopyDevicePerformanceQueryEXT(sType=VK_STRUCTURE_TYPE_HOST_IMAGE_COPY_DEVICE_PERFORMANCE_QUERY_EXT,pNext=None,optimalDeviceAccess=None,identicalMemoryLayout=None,):

    return _new('VkHostImageCopyDevicePerformanceQueryEXT', sType=sType,pNext=pNext,optimalDeviceAccess=optimalDeviceAccess,identicalMemoryLayout=identicalMemoryLayout)

def VkPhysicalDeviceVulkanSC10Properties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_SC_1_0_PROPERTIES,pNext=None,deviceNoDynamicHostAllocations=None,deviceDestroyFreesMemory=None,commandPoolMultipleCommandBuffersRecording=None,commandPoolResetCommandBuffer=None,commandBufferSimultaneousUse=None,secondaryCommandBufferNullOrImagelessFramebuffer=None,recycleDescriptorSetMemory=None,recyclePipelineMemory=None,maxRenderPassSubpasses=None,maxRenderPassDependencies=None,maxSubpassInputAttachments=None,maxSubpassPreserveAttachments=None,maxFramebufferAttachments=None,maxDescriptorSetLayoutBindings=None,maxQueryFaultCount=None,maxCallbackFaultCount=None,maxCommandPoolCommandBuffers=None,maxCommandBufferSize=None,):

    return _new('VkPhysicalDeviceVulkanSC10Properties', sType=sType,pNext=pNext,deviceNoDynamicHostAllocations=deviceNoDynamicHostAllocations,deviceDestroyFreesMemory=deviceDestroyFreesMemory,commandPoolMultipleCommandBuffersRecording=commandPoolMultipleCommandBuffersRecording,commandPoolResetCommandBuffer=commandPoolResetCommandBuffer,commandBufferSimultaneousUse=commandBufferSimultaneousUse,secondaryCommandBufferNullOrImagelessFramebuffer=secondaryCommandBufferNullOrImagelessFramebuffer,recycleDescriptorSetMemory=recycleDescriptorSetMemory,recyclePipelineMemory=recyclePipelineMemory,maxRenderPassSubpasses=maxRenderPassSubpasses,maxRenderPassDependencies=maxRenderPassDependencies,maxSubpassInputAttachments=maxSubpassInputAttachments,maxSubpassPreserveAttachments=maxSubpassPreserveAttachments,maxFramebufferAttachments=maxFramebufferAttachments,maxDescriptorSetLayoutBindings=maxDescriptorSetLayoutBindings,maxQueryFaultCount=maxQueryFaultCount,maxCallbackFaultCount=maxCallbackFaultCount,maxCommandPoolCommandBuffers=maxCommandPoolCommandBuffers,maxCommandBufferSize=maxCommandBufferSize)

def VkPipelinePoolSize(sType=VK_STRUCTURE_TYPE_PIPELINE_POOL_SIZE,pNext=None,poolEntrySize=None,poolEntryCount=None,):

    return _new('VkPipelinePoolSize', sType=sType,pNext=pNext,poolEntrySize=poolEntrySize,poolEntryCount=poolEntryCount)

def VkDeviceObjectReservationCreateInfo(sType=VK_STRUCTURE_TYPE_DEVICE_OBJECT_RESERVATION_CREATE_INFO,pNext=None,pipelineCacheCreateInfoCount=None,pPipelineCacheCreateInfos=None,pipelinePoolSizeCount=None,pPipelinePoolSizes=None,semaphoreRequestCount=None,commandBufferRequestCount=None,fenceRequestCount=None,deviceMemoryRequestCount=None,bufferRequestCount=None,imageRequestCount=None,eventRequestCount=None,queryPoolRequestCount=None,bufferViewRequestCount=None,imageViewRequestCount=None,layeredImageViewRequestCount=None,pipelineCacheRequestCount=None,pipelineLayoutRequestCount=None,renderPassRequestCount=None,graphicsPipelineRequestCount=None,computePipelineRequestCount=None,descriptorSetLayoutRequestCount=None,samplerRequestCount=None,descriptorPoolRequestCount=None,descriptorSetRequestCount=None,framebufferRequestCount=None,commandPoolRequestCount=None,samplerYcbcrConversionRequestCount=None,surfaceRequestCount=None,swapchainRequestCount=None,displayModeRequestCount=None,subpassDescriptionRequestCount=None,attachmentDescriptionRequestCount=None,descriptorSetLayoutBindingRequestCount=None,descriptorSetLayoutBindingLimit=None,maxImageViewMipLevels=None,maxImageViewArrayLayers=None,maxLayeredImageViewMipLevels=None,maxOcclusionQueriesPerPool=None,maxPipelineStatisticsQueriesPerPool=None,maxTimestampQueriesPerPool=None,maxImmutableSamplersPerDescriptorSetLayout=None,):
    if pipelineCacheCreateInfoCount is None and pPipelineCacheCreateInfos is not None:
        pipelineCacheCreateInfoCount = len(pPipelineCacheCreateInfos)
    if pipelinePoolSizeCount is None and pPipelinePoolSizes is not None:
        pipelinePoolSizeCount = len(pPipelinePoolSizes)

    return _new('VkDeviceObjectReservationCreateInfo', sType=sType,pNext=pNext,pipelineCacheCreateInfoCount=pipelineCacheCreateInfoCount,pPipelineCacheCreateInfos=pPipelineCacheCreateInfos,pipelinePoolSizeCount=pipelinePoolSizeCount,pPipelinePoolSizes=pPipelinePoolSizes,semaphoreRequestCount=semaphoreRequestCount,commandBufferRequestCount=commandBufferRequestCount,fenceRequestCount=fenceRequestCount,deviceMemoryRequestCount=deviceMemoryRequestCount,bufferRequestCount=bufferRequestCount,imageRequestCount=imageRequestCount,eventRequestCount=eventRequestCount,queryPoolRequestCount=queryPoolRequestCount,bufferViewRequestCount=bufferViewRequestCount,imageViewRequestCount=imageViewRequestCount,layeredImageViewRequestCount=layeredImageViewRequestCount,pipelineCacheRequestCount=pipelineCacheRequestCount,pipelineLayoutRequestCount=pipelineLayoutRequestCount,renderPassRequestCount=renderPassRequestCount,graphicsPipelineRequestCount=graphicsPipelineRequestCount,computePipelineRequestCount=computePipelineRequestCount,descriptorSetLayoutRequestCount=descriptorSetLayoutRequestCount,samplerRequestCount=samplerRequestCount,descriptorPoolRequestCount=descriptorPoolRequestCount,descriptorSetRequestCount=descriptorSetRequestCount,framebufferRequestCount=framebufferRequestCount,commandPoolRequestCount=commandPoolRequestCount,samplerYcbcrConversionRequestCount=samplerYcbcrConversionRequestCount,surfaceRequestCount=surfaceRequestCount,swapchainRequestCount=swapchainRequestCount,displayModeRequestCount=displayModeRequestCount,subpassDescriptionRequestCount=subpassDescriptionRequestCount,attachmentDescriptionRequestCount=attachmentDescriptionRequestCount,descriptorSetLayoutBindingRequestCount=descriptorSetLayoutBindingRequestCount,descriptorSetLayoutBindingLimit=descriptorSetLayoutBindingLimit,maxImageViewMipLevels=maxImageViewMipLevels,maxImageViewArrayLayers=maxImageViewArrayLayers,maxLayeredImageViewMipLevels=maxLayeredImageViewMipLevels,maxOcclusionQueriesPerPool=maxOcclusionQueriesPerPool,maxPipelineStatisticsQueriesPerPool=maxPipelineStatisticsQueriesPerPool,maxTimestampQueriesPerPool=maxTimestampQueriesPerPool,maxImmutableSamplersPerDescriptorSetLayout=maxImmutableSamplersPerDescriptorSetLayout)

def VkCommandPoolMemoryReservationCreateInfo(sType=VK_STRUCTURE_TYPE_COMMAND_POOL_MEMORY_RESERVATION_CREATE_INFO,pNext=None,commandPoolReservedSize=None,commandPoolMaxCommandBuffers=None,):

    return _new('VkCommandPoolMemoryReservationCreateInfo', sType=sType,pNext=pNext,commandPoolReservedSize=commandPoolReservedSize,commandPoolMaxCommandBuffers=commandPoolMaxCommandBuffers)

def VkCommandPoolMemoryConsumption(sType=VK_STRUCTURE_TYPE_COMMAND_POOL_MEMORY_CONSUMPTION,pNext=None,commandPoolAllocated=None,commandPoolReservedSize=None,commandBufferAllocated=None,):

    return _new('VkCommandPoolMemoryConsumption', sType=sType,pNext=pNext,commandPoolAllocated=commandPoolAllocated,commandPoolReservedSize=commandPoolReservedSize,commandBufferAllocated=commandBufferAllocated)

def VkPhysicalDeviceVulkanSC10Features(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_SC_1_0_FEATURES,pNext=None,shaderAtomicInstructions=None,):

    return _new('VkPhysicalDeviceVulkanSC10Features', sType=sType,pNext=pNext,shaderAtomicInstructions=shaderAtomicInstructions)

def VkPhysicalDevicePrimitivesGeneratedQueryFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRIMITIVES_GENERATED_QUERY_FEATURES_EXT,pNext=None,primitivesGeneratedQuery=None,primitivesGeneratedQueryWithRasterizerDiscard=None,primitivesGeneratedQueryWithNonZeroStreams=None,):

    return _new('VkPhysicalDevicePrimitivesGeneratedQueryFeaturesEXT', sType=sType,pNext=pNext,primitivesGeneratedQuery=primitivesGeneratedQuery,primitivesGeneratedQueryWithRasterizerDiscard=primitivesGeneratedQueryWithRasterizerDiscard,primitivesGeneratedQueryWithNonZeroStreams=primitivesGeneratedQueryWithNonZeroStreams)

def VkPhysicalDeviceLegacyDitheringFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LEGACY_DITHERING_FEATURES_EXT,pNext=None,legacyDithering=None,):

    return _new('VkPhysicalDeviceLegacyDitheringFeaturesEXT', sType=sType,pNext=pNext,legacyDithering=legacyDithering)

def VkPhysicalDeviceMultisampledRenderToSingleSampledFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTISAMPLED_RENDER_TO_SINGLE_SAMPLED_FEATURES_EXT,pNext=None,multisampledRenderToSingleSampled=None,):

    return _new('VkPhysicalDeviceMultisampledRenderToSingleSampledFeaturesEXT', sType=sType,pNext=pNext,multisampledRenderToSingleSampled=multisampledRenderToSingleSampled)

def VkSubpassResolvePerformanceQueryEXT(sType=VK_STRUCTURE_TYPE_SUBPASS_RESOLVE_PERFORMANCE_QUERY_EXT,pNext=None,optimal=None,):

    return _new('VkSubpassResolvePerformanceQueryEXT', sType=sType,pNext=pNext,optimal=optimal)

def VkMultisampledRenderToSingleSampledInfoEXT(sType=VK_STRUCTURE_TYPE_MULTISAMPLED_RENDER_TO_SINGLE_SAMPLED_INFO_EXT,pNext=None,multisampledRenderToSingleSampledEnable=None,rasterizationSamples=None,):

    return _new('VkMultisampledRenderToSingleSampledInfoEXT', sType=sType,pNext=pNext,multisampledRenderToSingleSampledEnable=multisampledRenderToSingleSampledEnable,rasterizationSamples=rasterizationSamples)

def VkPhysicalDevicePipelineProtectedAccessFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_PROTECTED_ACCESS_FEATURES_EXT,pNext=None,pipelineProtectedAccess=None,):

    return _new('VkPhysicalDevicePipelineProtectedAccessFeaturesEXT', sType=sType,pNext=pNext,pipelineProtectedAccess=pipelineProtectedAccess)

def VkQueueFamilyVideoPropertiesKHR(sType=VK_STRUCTURE_TYPE_QUEUE_FAMILY_VIDEO_PROPERTIES_KHR,pNext=None,videoCodecOperations=None,):

    return _new('VkQueueFamilyVideoPropertiesKHR', sType=sType,pNext=pNext,videoCodecOperations=videoCodecOperations)

def VkQueueFamilyQueryResultStatusPropertiesKHR(sType=VK_STRUCTURE_TYPE_QUEUE_FAMILY_QUERY_RESULT_STATUS_PROPERTIES_KHR,pNext=None,queryResultStatusSupport=None,):

    return _new('VkQueueFamilyQueryResultStatusPropertiesKHR', sType=sType,pNext=pNext,queryResultStatusSupport=queryResultStatusSupport)

def VkVideoProfileListInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_PROFILE_LIST_INFO_KHR,pNext=None,profileCount=None,pProfiles=None,):
    if profileCount is None and pProfiles is not None:
        profileCount = len(pProfiles)

    return _new('VkVideoProfileListInfoKHR', sType=sType,pNext=pNext,profileCount=profileCount,pProfiles=pProfiles)

def VkPhysicalDeviceVideoFormatInfoKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VIDEO_FORMAT_INFO_KHR,pNext=None,imageUsage=None,):

    return _new('VkPhysicalDeviceVideoFormatInfoKHR', sType=sType,pNext=pNext,imageUsage=imageUsage)

def VkVideoFormatPropertiesKHR(sType=VK_STRUCTURE_TYPE_VIDEO_FORMAT_PROPERTIES_KHR,pNext=None,format=None,componentMapping=None,imageCreateFlags=None,imageType=None,imageTiling=None,imageUsageFlags=None,):

    return _new('VkVideoFormatPropertiesKHR', sType=sType,pNext=pNext,format=format,componentMapping=componentMapping,imageCreateFlags=imageCreateFlags,imageType=imageType,imageTiling=imageTiling,imageUsageFlags=imageUsageFlags)

def VkVideoProfileInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_PROFILE_INFO_KHR,pNext=None,videoCodecOperation=None,chromaSubsampling=None,lumaBitDepth=None,chromaBitDepth=None,):

    return _new('VkVideoProfileInfoKHR', sType=sType,pNext=pNext,videoCodecOperation=videoCodecOperation,chromaSubsampling=chromaSubsampling,lumaBitDepth=lumaBitDepth,chromaBitDepth=chromaBitDepth)

def VkVideoCapabilitiesKHR(sType=VK_STRUCTURE_TYPE_VIDEO_CAPABILITIES_KHR,pNext=None,flags=None,minBitstreamBufferOffsetAlignment=None,minBitstreamBufferSizeAlignment=None,pictureAccessGranularity=None,minCodedExtent=None,maxCodedExtent=None,maxDpbSlots=None,maxActiveReferencePictures=None,stdHeaderVersion=None,):

    return _new('VkVideoCapabilitiesKHR', sType=sType,pNext=pNext,flags=flags,minBitstreamBufferOffsetAlignment=minBitstreamBufferOffsetAlignment,minBitstreamBufferSizeAlignment=minBitstreamBufferSizeAlignment,pictureAccessGranularity=pictureAccessGranularity,minCodedExtent=minCodedExtent,maxCodedExtent=maxCodedExtent,maxDpbSlots=maxDpbSlots,maxActiveReferencePictures=maxActiveReferencePictures,stdHeaderVersion=stdHeaderVersion)

def VkVideoSessionMemoryRequirementsKHR(sType=VK_STRUCTURE_TYPE_VIDEO_SESSION_MEMORY_REQUIREMENTS_KHR,pNext=None,memoryBindIndex=None,memoryRequirements=None,):

    return _new('VkVideoSessionMemoryRequirementsKHR', sType=sType,pNext=pNext,memoryBindIndex=memoryBindIndex,memoryRequirements=memoryRequirements)

def VkBindVideoSessionMemoryInfoKHR(sType=VK_STRUCTURE_TYPE_BIND_VIDEO_SESSION_MEMORY_INFO_KHR,pNext=None,memoryBindIndex=None,memory=None,memoryOffset=None,memorySize=None,):

    return _new('VkBindVideoSessionMemoryInfoKHR', sType=sType,pNext=pNext,memoryBindIndex=memoryBindIndex,memory=memory,memoryOffset=memoryOffset,memorySize=memorySize)

def VkVideoPictureResourceInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_PICTURE_RESOURCE_INFO_KHR,pNext=None,codedOffset=None,codedExtent=None,baseArrayLayer=None,imageViewBinding=None,):

    return _new('VkVideoPictureResourceInfoKHR', sType=sType,pNext=pNext,codedOffset=codedOffset,codedExtent=codedExtent,baseArrayLayer=baseArrayLayer,imageViewBinding=imageViewBinding)

def VkVideoReferenceSlotInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_REFERENCE_SLOT_INFO_KHR,pNext=None,slotIndex=None,pPictureResource=None,):

    return _new('VkVideoReferenceSlotInfoKHR', sType=sType,pNext=pNext,slotIndex=slotIndex,pPictureResource=pPictureResource)

def VkVideoDecodeCapabilitiesKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_CAPABILITIES_KHR,pNext=None,flags=None,):

    return _new('VkVideoDecodeCapabilitiesKHR', sType=sType,pNext=pNext,flags=flags)

def VkVideoDecodeUsageInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_USAGE_INFO_KHR,pNext=None,videoUsageHints=None,):

    return _new('VkVideoDecodeUsageInfoKHR', sType=sType,pNext=pNext,videoUsageHints=videoUsageHints)

def VkVideoDecodeInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_INFO_KHR,pNext=None,flags=None,srcBuffer=None,srcBufferOffset=None,srcBufferRange=None,dstPictureResource=None,pSetupReferenceSlot=None,referenceSlotCount=None,pReferenceSlots=None,):
    if referenceSlotCount is None and pReferenceSlots is not None:
        referenceSlotCount = len(pReferenceSlots)

    return _new('VkVideoDecodeInfoKHR', sType=sType,pNext=pNext,flags=flags,srcBuffer=srcBuffer,srcBufferOffset=srcBufferOffset,srcBufferRange=srcBufferRange,dstPictureResource=dstPictureResource,pSetupReferenceSlot=pSetupReferenceSlot,referenceSlotCount=referenceSlotCount,pReferenceSlots=pReferenceSlots)

def VkPhysicalDeviceVideoMaintenance1FeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VIDEO_MAINTENANCE_1_FEATURES_KHR,pNext=None,videoMaintenance1=None,):

    return _new('VkPhysicalDeviceVideoMaintenance1FeaturesKHR', sType=sType,pNext=pNext,videoMaintenance1=videoMaintenance1)

def VkVideoInlineQueryInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_INLINE_QUERY_INFO_KHR,pNext=None,queryPool=None,firstQuery=None,queryCount=None,):

    return _new('VkVideoInlineQueryInfoKHR', sType=sType,pNext=pNext,queryPool=queryPool,firstQuery=firstQuery,queryCount=queryCount)

def VkVideoDecodeH264ProfileInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_H264_PROFILE_INFO_KHR,pNext=None,stdProfileIdc=None,pictureLayout=None,):

    return _new('VkVideoDecodeH264ProfileInfoKHR', sType=sType,pNext=pNext,stdProfileIdc=stdProfileIdc,pictureLayout=pictureLayout)

def VkVideoDecodeH264CapabilitiesKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_H264_CAPABILITIES_KHR,pNext=None,maxLevelIdc=None,fieldOffsetGranularity=None,):

    return _new('VkVideoDecodeH264CapabilitiesKHR', sType=sType,pNext=pNext,maxLevelIdc=maxLevelIdc,fieldOffsetGranularity=fieldOffsetGranularity)

def VkVideoDecodeH264SessionParametersAddInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_H264_SESSION_PARAMETERS_ADD_INFO_KHR,pNext=None,stdSPSCount=None,pStdSPSs=None,stdPPSCount=None,pStdPPSs=None,):
    if stdSPSCount is None and pStdSPSs is not None:
        stdSPSCount = len(pStdSPSs)
    if stdPPSCount is None and pStdPPSs is not None:
        stdPPSCount = len(pStdPPSs)

    return _new('VkVideoDecodeH264SessionParametersAddInfoKHR', sType=sType,pNext=pNext,stdSPSCount=stdSPSCount,pStdSPSs=pStdSPSs,stdPPSCount=stdPPSCount,pStdPPSs=pStdPPSs)

def VkVideoDecodeH264SessionParametersCreateInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_H264_SESSION_PARAMETERS_CREATE_INFO_KHR,pNext=None,maxStdSPSCount=None,maxStdPPSCount=None,pParametersAddInfo=None,):

    return _new('VkVideoDecodeH264SessionParametersCreateInfoKHR', sType=sType,pNext=pNext,maxStdSPSCount=maxStdSPSCount,maxStdPPSCount=maxStdPPSCount,pParametersAddInfo=pParametersAddInfo)

def VkVideoDecodeH264PictureInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_H264_PICTURE_INFO_KHR,pNext=None,pStdPictureInfo=None,sliceCount=None,pSliceOffsets=None,):
    if sliceCount is None and pSliceOffsets is not None:
        sliceCount = len(pSliceOffsets)

    return _new('VkVideoDecodeH264PictureInfoKHR', sType=sType,pNext=pNext,pStdPictureInfo=pStdPictureInfo,sliceCount=sliceCount,pSliceOffsets=pSliceOffsets)

def VkVideoDecodeH264DpbSlotInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_H264_DPB_SLOT_INFO_KHR,pNext=None,pStdReferenceInfo=None,):

    return _new('VkVideoDecodeH264DpbSlotInfoKHR', sType=sType,pNext=pNext,pStdReferenceInfo=pStdReferenceInfo)

def VkVideoDecodeH265ProfileInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_H265_PROFILE_INFO_KHR,pNext=None,stdProfileIdc=None,):

    return _new('VkVideoDecodeH265ProfileInfoKHR', sType=sType,pNext=pNext,stdProfileIdc=stdProfileIdc)

def VkVideoDecodeH265CapabilitiesKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_H265_CAPABILITIES_KHR,pNext=None,maxLevelIdc=None,):

    return _new('VkVideoDecodeH265CapabilitiesKHR', sType=sType,pNext=pNext,maxLevelIdc=maxLevelIdc)

def VkVideoDecodeH265SessionParametersAddInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_H265_SESSION_PARAMETERS_ADD_INFO_KHR,pNext=None,stdVPSCount=None,pStdVPSs=None,stdSPSCount=None,pStdSPSs=None,stdPPSCount=None,pStdPPSs=None,):
    if stdVPSCount is None and pStdVPSs is not None:
        stdVPSCount = len(pStdVPSs)
    if stdSPSCount is None and pStdSPSs is not None:
        stdSPSCount = len(pStdSPSs)
    if stdPPSCount is None and pStdPPSs is not None:
        stdPPSCount = len(pStdPPSs)

    return _new('VkVideoDecodeH265SessionParametersAddInfoKHR', sType=sType,pNext=pNext,stdVPSCount=stdVPSCount,pStdVPSs=pStdVPSs,stdSPSCount=stdSPSCount,pStdSPSs=pStdSPSs,stdPPSCount=stdPPSCount,pStdPPSs=pStdPPSs)

def VkVideoDecodeH265SessionParametersCreateInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_H265_SESSION_PARAMETERS_CREATE_INFO_KHR,pNext=None,maxStdVPSCount=None,maxStdSPSCount=None,maxStdPPSCount=None,pParametersAddInfo=None,):

    return _new('VkVideoDecodeH265SessionParametersCreateInfoKHR', sType=sType,pNext=pNext,maxStdVPSCount=maxStdVPSCount,maxStdSPSCount=maxStdSPSCount,maxStdPPSCount=maxStdPPSCount,pParametersAddInfo=pParametersAddInfo)

def VkVideoDecodeH265PictureInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_H265_PICTURE_INFO_KHR,pNext=None,pStdPictureInfo=None,sliceSegmentCount=None,pSliceSegmentOffsets=None,):
    if sliceSegmentCount is None and pSliceSegmentOffsets is not None:
        sliceSegmentCount = len(pSliceSegmentOffsets)

    return _new('VkVideoDecodeH265PictureInfoKHR', sType=sType,pNext=pNext,pStdPictureInfo=pStdPictureInfo,sliceSegmentCount=sliceSegmentCount,pSliceSegmentOffsets=pSliceSegmentOffsets)

def VkVideoDecodeH265DpbSlotInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_DECODE_H265_DPB_SLOT_INFO_KHR,pNext=None,pStdReferenceInfo=None,):

    return _new('VkVideoDecodeH265DpbSlotInfoKHR', sType=sType,pNext=pNext,pStdReferenceInfo=pStdReferenceInfo)

def VkVideoSessionCreateInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_SESSION_CREATE_INFO_KHR,pNext=None,queueFamilyIndex=None,flags=None,pVideoProfile=None,pictureFormat=None,maxCodedExtent=None,referencePictureFormat=None,maxDpbSlots=None,maxActiveReferencePictures=None,pStdHeaderVersion=None,):

    return _new('VkVideoSessionCreateInfoKHR', sType=sType,pNext=pNext,queueFamilyIndex=queueFamilyIndex,flags=flags,pVideoProfile=pVideoProfile,pictureFormat=pictureFormat,maxCodedExtent=maxCodedExtent,referencePictureFormat=referencePictureFormat,maxDpbSlots=maxDpbSlots,maxActiveReferencePictures=maxActiveReferencePictures,pStdHeaderVersion=pStdHeaderVersion)

def VkVideoSessionParametersCreateInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_SESSION_PARAMETERS_CREATE_INFO_KHR,pNext=None,flags=None,videoSessionParametersTemplate=None,videoSession=None,):

    return _new('VkVideoSessionParametersCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,videoSessionParametersTemplate=videoSessionParametersTemplate,videoSession=videoSession)

def VkVideoSessionParametersUpdateInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_SESSION_PARAMETERS_UPDATE_INFO_KHR,pNext=None,updateSequenceCount=None,):

    return _new('VkVideoSessionParametersUpdateInfoKHR', sType=sType,pNext=pNext,updateSequenceCount=updateSequenceCount)

def VkVideoEncodeSessionParametersGetInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_SESSION_PARAMETERS_GET_INFO_KHR,pNext=None,videoSessionParameters=None,):

    return _new('VkVideoEncodeSessionParametersGetInfoKHR', sType=sType,pNext=pNext,videoSessionParameters=videoSessionParameters)

def VkVideoEncodeSessionParametersFeedbackInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_SESSION_PARAMETERS_FEEDBACK_INFO_KHR,pNext=None,hasOverrides=None,):

    return _new('VkVideoEncodeSessionParametersFeedbackInfoKHR', sType=sType,pNext=pNext,hasOverrides=hasOverrides)

def VkVideoBeginCodingInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_BEGIN_CODING_INFO_KHR,pNext=None,flags=None,videoSession=None,videoSessionParameters=None,referenceSlotCount=None,pReferenceSlots=None,):
    if referenceSlotCount is None and pReferenceSlots is not None:
        referenceSlotCount = len(pReferenceSlots)

    return _new('VkVideoBeginCodingInfoKHR', sType=sType,pNext=pNext,flags=flags,videoSession=videoSession,videoSessionParameters=videoSessionParameters,referenceSlotCount=referenceSlotCount,pReferenceSlots=pReferenceSlots)

def VkVideoEndCodingInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_END_CODING_INFO_KHR,pNext=None,flags=None,):

    return _new('VkVideoEndCodingInfoKHR', sType=sType,pNext=pNext,flags=flags)

def VkVideoCodingControlInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_CODING_CONTROL_INFO_KHR,pNext=None,flags=None,):

    return _new('VkVideoCodingControlInfoKHR', sType=sType,pNext=pNext,flags=flags)

def VkVideoEncodeUsageInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_USAGE_INFO_KHR,pNext=None,videoUsageHints=None,videoContentHints=None,tuningMode=None,):

    return _new('VkVideoEncodeUsageInfoKHR', sType=sType,pNext=pNext,videoUsageHints=videoUsageHints,videoContentHints=videoContentHints,tuningMode=tuningMode)

def VkVideoEncodeInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_INFO_KHR,pNext=None,flags=None,dstBuffer=None,dstBufferOffset=None,dstBufferRange=None,srcPictureResource=None,pSetupReferenceSlot=None,referenceSlotCount=None,pReferenceSlots=None,precedingExternallyEncodedBytes=None,):
    if referenceSlotCount is None and pReferenceSlots is not None:
        referenceSlotCount = len(pReferenceSlots)

    return _new('VkVideoEncodeInfoKHR', sType=sType,pNext=pNext,flags=flags,dstBuffer=dstBuffer,dstBufferOffset=dstBufferOffset,dstBufferRange=dstBufferRange,srcPictureResource=srcPictureResource,pSetupReferenceSlot=pSetupReferenceSlot,referenceSlotCount=referenceSlotCount,pReferenceSlots=pReferenceSlots,precedingExternallyEncodedBytes=precedingExternallyEncodedBytes)

def VkQueryPoolVideoEncodeFeedbackCreateInfoKHR(sType=VK_STRUCTURE_TYPE_QUERY_POOL_VIDEO_ENCODE_FEEDBACK_CREATE_INFO_KHR,pNext=None,encodeFeedbackFlags=None,):

    return _new('VkQueryPoolVideoEncodeFeedbackCreateInfoKHR', sType=sType,pNext=pNext,encodeFeedbackFlags=encodeFeedbackFlags)

def VkVideoEncodeQualityLevelInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_QUALITY_LEVEL_INFO_KHR,pNext=None,qualityLevel=None,):

    return _new('VkVideoEncodeQualityLevelInfoKHR', sType=sType,pNext=pNext,qualityLevel=qualityLevel)

def VkPhysicalDeviceVideoEncodeQualityLevelInfoKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VIDEO_ENCODE_QUALITY_LEVEL_INFO_KHR,pNext=None,pVideoProfile=None,qualityLevel=None,):

    return _new('VkPhysicalDeviceVideoEncodeQualityLevelInfoKHR', sType=sType,pNext=pNext,pVideoProfile=pVideoProfile,qualityLevel=qualityLevel)

def VkVideoEncodeQualityLevelPropertiesKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_QUALITY_LEVEL_PROPERTIES_KHR,pNext=None,preferredRateControlMode=None,preferredRateControlLayerCount=None,):

    return _new('VkVideoEncodeQualityLevelPropertiesKHR', sType=sType,pNext=pNext,preferredRateControlMode=preferredRateControlMode,preferredRateControlLayerCount=preferredRateControlLayerCount)

def VkVideoEncodeRateControlInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_RATE_CONTROL_INFO_KHR,pNext=None,flags=None,rateControlMode=None,layerCount=None,pLayers=None,virtualBufferSizeInMs=None,initialVirtualBufferSizeInMs=None,):
    if layerCount is None and pLayers is not None:
        layerCount = len(pLayers)

    return _new('VkVideoEncodeRateControlInfoKHR', sType=sType,pNext=pNext,flags=flags,rateControlMode=rateControlMode,layerCount=layerCount,pLayers=pLayers,virtualBufferSizeInMs=virtualBufferSizeInMs,initialVirtualBufferSizeInMs=initialVirtualBufferSizeInMs)

def VkVideoEncodeRateControlLayerInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_RATE_CONTROL_LAYER_INFO_KHR,pNext=None,averageBitrate=None,maxBitrate=None,frameRateNumerator=None,frameRateDenominator=None,):

    return _new('VkVideoEncodeRateControlLayerInfoKHR', sType=sType,pNext=pNext,averageBitrate=averageBitrate,maxBitrate=maxBitrate,frameRateNumerator=frameRateNumerator,frameRateDenominator=frameRateDenominator)

def VkVideoEncodeCapabilitiesKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_CAPABILITIES_KHR,pNext=None,flags=None,rateControlModes=None,maxRateControlLayers=None,maxBitrate=None,maxQualityLevels=None,encodeInputPictureGranularity=None,supportedEncodeFeedbackFlags=None,):

    return _new('VkVideoEncodeCapabilitiesKHR', sType=sType,pNext=pNext,flags=flags,rateControlModes=rateControlModes,maxRateControlLayers=maxRateControlLayers,maxBitrate=maxBitrate,maxQualityLevels=maxQualityLevels,encodeInputPictureGranularity=encodeInputPictureGranularity,supportedEncodeFeedbackFlags=supportedEncodeFeedbackFlags)

def VkVideoEncodeH264CapabilitiesKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_CAPABILITIES_KHR,pNext=None,flags=None,maxLevelIdc=None,maxSliceCount=None,maxPPictureL0ReferenceCount=None,maxBPictureL0ReferenceCount=None,maxL1ReferenceCount=None,maxTemporalLayerCount=None,expectDyadicTemporalLayerPattern=None,minQp=None,maxQp=None,prefersGopRemainingFrames=None,requiresGopRemainingFrames=None,stdSyntaxFlags=None,):

    return _new('VkVideoEncodeH264CapabilitiesKHR', sType=sType,pNext=pNext,flags=flags,maxLevelIdc=maxLevelIdc,maxSliceCount=maxSliceCount,maxPPictureL0ReferenceCount=maxPPictureL0ReferenceCount,maxBPictureL0ReferenceCount=maxBPictureL0ReferenceCount,maxL1ReferenceCount=maxL1ReferenceCount,maxTemporalLayerCount=maxTemporalLayerCount,expectDyadicTemporalLayerPattern=expectDyadicTemporalLayerPattern,minQp=minQp,maxQp=maxQp,prefersGopRemainingFrames=prefersGopRemainingFrames,requiresGopRemainingFrames=requiresGopRemainingFrames,stdSyntaxFlags=stdSyntaxFlags)

def VkVideoEncodeH264QualityLevelPropertiesKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_QUALITY_LEVEL_PROPERTIES_KHR,pNext=None,preferredRateControlFlags=None,preferredGopFrameCount=None,preferredIdrPeriod=None,preferredConsecutiveBFrameCount=None,preferredTemporalLayerCount=None,preferredConstantQp=None,preferredMaxL0ReferenceCount=None,preferredMaxL1ReferenceCount=None,preferredStdEntropyCodingModeFlag=None,):

    return _new('VkVideoEncodeH264QualityLevelPropertiesKHR', sType=sType,pNext=pNext,preferredRateControlFlags=preferredRateControlFlags,preferredGopFrameCount=preferredGopFrameCount,preferredIdrPeriod=preferredIdrPeriod,preferredConsecutiveBFrameCount=preferredConsecutiveBFrameCount,preferredTemporalLayerCount=preferredTemporalLayerCount,preferredConstantQp=preferredConstantQp,preferredMaxL0ReferenceCount=preferredMaxL0ReferenceCount,preferredMaxL1ReferenceCount=preferredMaxL1ReferenceCount,preferredStdEntropyCodingModeFlag=preferredStdEntropyCodingModeFlag)

def VkVideoEncodeH264SessionCreateInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_SESSION_CREATE_INFO_KHR,pNext=None,useMaxLevelIdc=None,maxLevelIdc=None,):

    return _new('VkVideoEncodeH264SessionCreateInfoKHR', sType=sType,pNext=pNext,useMaxLevelIdc=useMaxLevelIdc,maxLevelIdc=maxLevelIdc)

def VkVideoEncodeH264SessionParametersAddInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_SESSION_PARAMETERS_ADD_INFO_KHR,pNext=None,stdSPSCount=None,pStdSPSs=None,stdPPSCount=None,pStdPPSs=None,):
    if stdSPSCount is None and pStdSPSs is not None:
        stdSPSCount = len(pStdSPSs)
    if stdPPSCount is None and pStdPPSs is not None:
        stdPPSCount = len(pStdPPSs)

    return _new('VkVideoEncodeH264SessionParametersAddInfoKHR', sType=sType,pNext=pNext,stdSPSCount=stdSPSCount,pStdSPSs=pStdSPSs,stdPPSCount=stdPPSCount,pStdPPSs=pStdPPSs)

def VkVideoEncodeH264SessionParametersCreateInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_SESSION_PARAMETERS_CREATE_INFO_KHR,pNext=None,maxStdSPSCount=None,maxStdPPSCount=None,pParametersAddInfo=None,):

    return _new('VkVideoEncodeH264SessionParametersCreateInfoKHR', sType=sType,pNext=pNext,maxStdSPSCount=maxStdSPSCount,maxStdPPSCount=maxStdPPSCount,pParametersAddInfo=pParametersAddInfo)

def VkVideoEncodeH264SessionParametersGetInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_SESSION_PARAMETERS_GET_INFO_KHR,pNext=None,writeStdSPS=None,writeStdPPS=None,stdSPSId=None,stdPPSId=None,):

    return _new('VkVideoEncodeH264SessionParametersGetInfoKHR', sType=sType,pNext=pNext,writeStdSPS=writeStdSPS,writeStdPPS=writeStdPPS,stdSPSId=stdSPSId,stdPPSId=stdPPSId)

def VkVideoEncodeH264SessionParametersFeedbackInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_SESSION_PARAMETERS_FEEDBACK_INFO_KHR,pNext=None,hasStdSPSOverrides=None,hasStdPPSOverrides=None,):

    return _new('VkVideoEncodeH264SessionParametersFeedbackInfoKHR', sType=sType,pNext=pNext,hasStdSPSOverrides=hasStdSPSOverrides,hasStdPPSOverrides=hasStdPPSOverrides)

def VkVideoEncodeH264DpbSlotInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_DPB_SLOT_INFO_KHR,pNext=None,pStdReferenceInfo=None,):

    return _new('VkVideoEncodeH264DpbSlotInfoKHR', sType=sType,pNext=pNext,pStdReferenceInfo=pStdReferenceInfo)

def VkVideoEncodeH264PictureInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_PICTURE_INFO_KHR,pNext=None,naluSliceEntryCount=None,pNaluSliceEntries=None,pStdPictureInfo=None,generatePrefixNalu=None,):
    if naluSliceEntryCount is None and pNaluSliceEntries is not None:
        naluSliceEntryCount = len(pNaluSliceEntries)

    return _new('VkVideoEncodeH264PictureInfoKHR', sType=sType,pNext=pNext,naluSliceEntryCount=naluSliceEntryCount,pNaluSliceEntries=pNaluSliceEntries,pStdPictureInfo=pStdPictureInfo,generatePrefixNalu=generatePrefixNalu)

def VkVideoEncodeH264ProfileInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_PROFILE_INFO_KHR,pNext=None,stdProfileIdc=None,):

    return _new('VkVideoEncodeH264ProfileInfoKHR', sType=sType,pNext=pNext,stdProfileIdc=stdProfileIdc)

def VkVideoEncodeH264NaluSliceInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_NALU_SLICE_INFO_KHR,pNext=None,constantQp=None,pStdSliceHeader=None,):

    return _new('VkVideoEncodeH264NaluSliceInfoKHR', sType=sType,pNext=pNext,constantQp=constantQp,pStdSliceHeader=pStdSliceHeader)

def VkVideoEncodeH264RateControlInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_RATE_CONTROL_INFO_KHR,pNext=None,flags=None,gopFrameCount=None,idrPeriod=None,consecutiveBFrameCount=None,temporalLayerCount=None,):

    return _new('VkVideoEncodeH264RateControlInfoKHR', sType=sType,pNext=pNext,flags=flags,gopFrameCount=gopFrameCount,idrPeriod=idrPeriod,consecutiveBFrameCount=consecutiveBFrameCount,temporalLayerCount=temporalLayerCount)

def VkVideoEncodeH264QpKHR(qpI=None,qpP=None,qpB=None,):

    return _new('VkVideoEncodeH264QpKHR', qpI=qpI,qpP=qpP,qpB=qpB)

def VkVideoEncodeH264FrameSizeKHR(frameISize=None,framePSize=None,frameBSize=None,):

    return _new('VkVideoEncodeH264FrameSizeKHR', frameISize=frameISize,framePSize=framePSize,frameBSize=frameBSize)

def VkVideoEncodeH264GopRemainingFrameInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_GOP_REMAINING_FRAME_INFO_KHR,pNext=None,useGopRemainingFrames=None,gopRemainingI=None,gopRemainingP=None,gopRemainingB=None,):

    return _new('VkVideoEncodeH264GopRemainingFrameInfoKHR', sType=sType,pNext=pNext,useGopRemainingFrames=useGopRemainingFrames,gopRemainingI=gopRemainingI,gopRemainingP=gopRemainingP,gopRemainingB=gopRemainingB)

def VkVideoEncodeH264RateControlLayerInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H264_RATE_CONTROL_LAYER_INFO_KHR,pNext=None,useMinQp=None,minQp=None,useMaxQp=None,maxQp=None,useMaxFrameSize=None,maxFrameSize=None,):

    return _new('VkVideoEncodeH264RateControlLayerInfoKHR', sType=sType,pNext=pNext,useMinQp=useMinQp,minQp=minQp,useMaxQp=useMaxQp,maxQp=maxQp,useMaxFrameSize=useMaxFrameSize,maxFrameSize=maxFrameSize)

def VkVideoEncodeH265CapabilitiesKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_CAPABILITIES_KHR,pNext=None,flags=None,maxLevelIdc=None,maxSliceSegmentCount=None,maxTiles=None,ctbSizes=None,transformBlockSizes=None,maxPPictureL0ReferenceCount=None,maxBPictureL0ReferenceCount=None,maxL1ReferenceCount=None,maxSubLayerCount=None,expectDyadicTemporalSubLayerPattern=None,minQp=None,maxQp=None,prefersGopRemainingFrames=None,requiresGopRemainingFrames=None,stdSyntaxFlags=None,):

    return _new('VkVideoEncodeH265CapabilitiesKHR', sType=sType,pNext=pNext,flags=flags,maxLevelIdc=maxLevelIdc,maxSliceSegmentCount=maxSliceSegmentCount,maxTiles=maxTiles,ctbSizes=ctbSizes,transformBlockSizes=transformBlockSizes,maxPPictureL0ReferenceCount=maxPPictureL0ReferenceCount,maxBPictureL0ReferenceCount=maxBPictureL0ReferenceCount,maxL1ReferenceCount=maxL1ReferenceCount,maxSubLayerCount=maxSubLayerCount,expectDyadicTemporalSubLayerPattern=expectDyadicTemporalSubLayerPattern,minQp=minQp,maxQp=maxQp,prefersGopRemainingFrames=prefersGopRemainingFrames,requiresGopRemainingFrames=requiresGopRemainingFrames,stdSyntaxFlags=stdSyntaxFlags)

def VkVideoEncodeH265QualityLevelPropertiesKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_QUALITY_LEVEL_PROPERTIES_KHR,pNext=None,preferredRateControlFlags=None,preferredGopFrameCount=None,preferredIdrPeriod=None,preferredConsecutiveBFrameCount=None,preferredSubLayerCount=None,preferredConstantQp=None,preferredMaxL0ReferenceCount=None,preferredMaxL1ReferenceCount=None,):

    return _new('VkVideoEncodeH265QualityLevelPropertiesKHR', sType=sType,pNext=pNext,preferredRateControlFlags=preferredRateControlFlags,preferredGopFrameCount=preferredGopFrameCount,preferredIdrPeriod=preferredIdrPeriod,preferredConsecutiveBFrameCount=preferredConsecutiveBFrameCount,preferredSubLayerCount=preferredSubLayerCount,preferredConstantQp=preferredConstantQp,preferredMaxL0ReferenceCount=preferredMaxL0ReferenceCount,preferredMaxL1ReferenceCount=preferredMaxL1ReferenceCount)

def VkVideoEncodeH265SessionCreateInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_SESSION_CREATE_INFO_KHR,pNext=None,useMaxLevelIdc=None,maxLevelIdc=None,):

    return _new('VkVideoEncodeH265SessionCreateInfoKHR', sType=sType,pNext=pNext,useMaxLevelIdc=useMaxLevelIdc,maxLevelIdc=maxLevelIdc)

def VkVideoEncodeH265SessionParametersAddInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_SESSION_PARAMETERS_ADD_INFO_KHR,pNext=None,stdVPSCount=None,pStdVPSs=None,stdSPSCount=None,pStdSPSs=None,stdPPSCount=None,pStdPPSs=None,):
    if stdVPSCount is None and pStdVPSs is not None:
        stdVPSCount = len(pStdVPSs)
    if stdSPSCount is None and pStdSPSs is not None:
        stdSPSCount = len(pStdSPSs)
    if stdPPSCount is None and pStdPPSs is not None:
        stdPPSCount = len(pStdPPSs)

    return _new('VkVideoEncodeH265SessionParametersAddInfoKHR', sType=sType,pNext=pNext,stdVPSCount=stdVPSCount,pStdVPSs=pStdVPSs,stdSPSCount=stdSPSCount,pStdSPSs=pStdSPSs,stdPPSCount=stdPPSCount,pStdPPSs=pStdPPSs)

def VkVideoEncodeH265SessionParametersCreateInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_SESSION_PARAMETERS_CREATE_INFO_KHR,pNext=None,maxStdVPSCount=None,maxStdSPSCount=None,maxStdPPSCount=None,pParametersAddInfo=None,):

    return _new('VkVideoEncodeH265SessionParametersCreateInfoKHR', sType=sType,pNext=pNext,maxStdVPSCount=maxStdVPSCount,maxStdSPSCount=maxStdSPSCount,maxStdPPSCount=maxStdPPSCount,pParametersAddInfo=pParametersAddInfo)

def VkVideoEncodeH265SessionParametersGetInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_SESSION_PARAMETERS_GET_INFO_KHR,pNext=None,writeStdVPS=None,writeStdSPS=None,writeStdPPS=None,stdVPSId=None,stdSPSId=None,stdPPSId=None,):

    return _new('VkVideoEncodeH265SessionParametersGetInfoKHR', sType=sType,pNext=pNext,writeStdVPS=writeStdVPS,writeStdSPS=writeStdSPS,writeStdPPS=writeStdPPS,stdVPSId=stdVPSId,stdSPSId=stdSPSId,stdPPSId=stdPPSId)

def VkVideoEncodeH265SessionParametersFeedbackInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_SESSION_PARAMETERS_FEEDBACK_INFO_KHR,pNext=None,hasStdVPSOverrides=None,hasStdSPSOverrides=None,hasStdPPSOverrides=None,):

    return _new('VkVideoEncodeH265SessionParametersFeedbackInfoKHR', sType=sType,pNext=pNext,hasStdVPSOverrides=hasStdVPSOverrides,hasStdSPSOverrides=hasStdSPSOverrides,hasStdPPSOverrides=hasStdPPSOverrides)

def VkVideoEncodeH265PictureInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_PICTURE_INFO_KHR,pNext=None,naluSliceSegmentEntryCount=None,pNaluSliceSegmentEntries=None,pStdPictureInfo=None,):
    if naluSliceSegmentEntryCount is None and pNaluSliceSegmentEntries is not None:
        naluSliceSegmentEntryCount = len(pNaluSliceSegmentEntries)

    return _new('VkVideoEncodeH265PictureInfoKHR', sType=sType,pNext=pNext,naluSliceSegmentEntryCount=naluSliceSegmentEntryCount,pNaluSliceSegmentEntries=pNaluSliceSegmentEntries,pStdPictureInfo=pStdPictureInfo)

def VkVideoEncodeH265NaluSliceSegmentInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_NALU_SLICE_SEGMENT_INFO_KHR,pNext=None,constantQp=None,pStdSliceSegmentHeader=None,):

    return _new('VkVideoEncodeH265NaluSliceSegmentInfoKHR', sType=sType,pNext=pNext,constantQp=constantQp,pStdSliceSegmentHeader=pStdSliceSegmentHeader)

def VkVideoEncodeH265RateControlInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_RATE_CONTROL_INFO_KHR,pNext=None,flags=None,gopFrameCount=None,idrPeriod=None,consecutiveBFrameCount=None,subLayerCount=None,):

    return _new('VkVideoEncodeH265RateControlInfoKHR', sType=sType,pNext=pNext,flags=flags,gopFrameCount=gopFrameCount,idrPeriod=idrPeriod,consecutiveBFrameCount=consecutiveBFrameCount,subLayerCount=subLayerCount)

def VkVideoEncodeH265QpKHR(qpI=None,qpP=None,qpB=None,):

    return _new('VkVideoEncodeH265QpKHR', qpI=qpI,qpP=qpP,qpB=qpB)

def VkVideoEncodeH265FrameSizeKHR(frameISize=None,framePSize=None,frameBSize=None,):

    return _new('VkVideoEncodeH265FrameSizeKHR', frameISize=frameISize,framePSize=framePSize,frameBSize=frameBSize)

def VkVideoEncodeH265GopRemainingFrameInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_GOP_REMAINING_FRAME_INFO_KHR,pNext=None,useGopRemainingFrames=None,gopRemainingI=None,gopRemainingP=None,gopRemainingB=None,):

    return _new('VkVideoEncodeH265GopRemainingFrameInfoKHR', sType=sType,pNext=pNext,useGopRemainingFrames=useGopRemainingFrames,gopRemainingI=gopRemainingI,gopRemainingP=gopRemainingP,gopRemainingB=gopRemainingB)

def VkVideoEncodeH265RateControlLayerInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_RATE_CONTROL_LAYER_INFO_KHR,pNext=None,useMinQp=None,minQp=None,useMaxQp=None,maxQp=None,useMaxFrameSize=None,maxFrameSize=None,):

    return _new('VkVideoEncodeH265RateControlLayerInfoKHR', sType=sType,pNext=pNext,useMinQp=useMinQp,minQp=minQp,useMaxQp=useMaxQp,maxQp=maxQp,useMaxFrameSize=useMaxFrameSize,maxFrameSize=maxFrameSize)

def VkVideoEncodeH265ProfileInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_PROFILE_INFO_KHR,pNext=None,stdProfileIdc=None,):

    return _new('VkVideoEncodeH265ProfileInfoKHR', sType=sType,pNext=pNext,stdProfileIdc=stdProfileIdc)

def VkVideoEncodeH265DpbSlotInfoKHR(sType=VK_STRUCTURE_TYPE_VIDEO_ENCODE_H265_DPB_SLOT_INFO_KHR,pNext=None,pStdReferenceInfo=None,):

    return _new('VkVideoEncodeH265DpbSlotInfoKHR', sType=sType,pNext=pNext,pStdReferenceInfo=pStdReferenceInfo)

def VkPhysicalDeviceInheritedViewportScissorFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INHERITED_VIEWPORT_SCISSOR_FEATURES_NV,pNext=None,inheritedViewportScissor2D=None,):

    return _new('VkPhysicalDeviceInheritedViewportScissorFeaturesNV', sType=sType,pNext=pNext,inheritedViewportScissor2D=inheritedViewportScissor2D)

def VkCommandBufferInheritanceViewportScissorInfoNV(sType=VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_VIEWPORT_SCISSOR_INFO_NV,pNext=None,viewportScissor2D=None,viewportDepthCount=None,pViewportDepths=None,):

    return _new('VkCommandBufferInheritanceViewportScissorInfoNV', sType=sType,pNext=pNext,viewportScissor2D=viewportScissor2D,viewportDepthCount=viewportDepthCount,pViewportDepths=pViewportDepths)

def VkPhysicalDeviceYcbcr2Plane444FormatsFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_YCBCR_2_PLANE_444_FORMATS_FEATURES_EXT,pNext=None,ycbcr2plane444Formats=None,):

    return _new('VkPhysicalDeviceYcbcr2Plane444FormatsFeaturesEXT', sType=sType,pNext=pNext,ycbcr2plane444Formats=ycbcr2plane444Formats)

def VkPhysicalDeviceProvokingVertexFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROVOKING_VERTEX_FEATURES_EXT,pNext=None,provokingVertexLast=None,transformFeedbackPreservesProvokingVertex=None,):

    return _new('VkPhysicalDeviceProvokingVertexFeaturesEXT', sType=sType,pNext=pNext,provokingVertexLast=provokingVertexLast,transformFeedbackPreservesProvokingVertex=transformFeedbackPreservesProvokingVertex)

def VkPhysicalDeviceProvokingVertexPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROVOKING_VERTEX_PROPERTIES_EXT,pNext=None,provokingVertexModePerPipeline=None,transformFeedbackPreservesTriangleFanProvokingVertex=None,):

    return _new('VkPhysicalDeviceProvokingVertexPropertiesEXT', sType=sType,pNext=pNext,provokingVertexModePerPipeline=provokingVertexModePerPipeline,transformFeedbackPreservesTriangleFanProvokingVertex=transformFeedbackPreservesTriangleFanProvokingVertex)

def VkPipelineRasterizationProvokingVertexStateCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_PROVOKING_VERTEX_STATE_CREATE_INFO_EXT,pNext=None,provokingVertexMode=None,):

    return _new('VkPipelineRasterizationProvokingVertexStateCreateInfoEXT', sType=sType,pNext=pNext,provokingVertexMode=provokingVertexMode)

def VkCuModuleCreateInfoNVX(sType=VK_STRUCTURE_TYPE_CU_MODULE_CREATE_INFO_NVX,pNext=None,dataSize=None,pData=None,):
    if dataSize is None and pData is not None:
        dataSize = len(pData)

    return _new('VkCuModuleCreateInfoNVX', sType=sType,pNext=pNext,dataSize=dataSize,pData=pData)

def VkCuFunctionCreateInfoNVX(sType=VK_STRUCTURE_TYPE_CU_FUNCTION_CREATE_INFO_NVX,pNext=None,module=None,pName=None,):

    return _new('VkCuFunctionCreateInfoNVX', sType=sType,pNext=pNext,module=module,pName=pName)

def VkCuLaunchInfoNVX(sType=VK_STRUCTURE_TYPE_CU_LAUNCH_INFO_NVX,pNext=None,function=None,gridDimX=None,gridDimY=None,gridDimZ=None,blockDimX=None,blockDimY=None,blockDimZ=None,sharedMemBytes=None,paramCount=None,pParams=None,extraCount=None,pExtras=None,):
    if paramCount is None and pParams is not None:
        paramCount = len(pParams)
    if extraCount is None and pExtras is not None:
        extraCount = len(pExtras)

    return _new('VkCuLaunchInfoNVX', sType=sType,pNext=pNext,function=function,gridDimX=gridDimX,gridDimY=gridDimY,gridDimZ=gridDimZ,blockDimX=blockDimX,blockDimY=blockDimY,blockDimZ=blockDimZ,sharedMemBytes=sharedMemBytes,paramCount=paramCount,pParams=pParams,extraCount=extraCount,pExtras=pExtras)

def VkPhysicalDeviceDescriptorBufferFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_BUFFER_FEATURES_EXT,pNext=None,descriptorBuffer=None,descriptorBufferCaptureReplay=None,descriptorBufferImageLayoutIgnored=None,descriptorBufferPushDescriptors=None,):

    return _new('VkPhysicalDeviceDescriptorBufferFeaturesEXT', sType=sType,pNext=pNext,descriptorBuffer=descriptorBuffer,descriptorBufferCaptureReplay=descriptorBufferCaptureReplay,descriptorBufferImageLayoutIgnored=descriptorBufferImageLayoutIgnored,descriptorBufferPushDescriptors=descriptorBufferPushDescriptors)

def VkPhysicalDeviceDescriptorBufferPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_BUFFER_PROPERTIES_EXT,pNext=None,combinedImageSamplerDescriptorSingleArray=None,bufferlessPushDescriptors=None,allowSamplerImageViewPostSubmitCreation=None,descriptorBufferOffsetAlignment=None,maxDescriptorBufferBindings=None,maxResourceDescriptorBufferBindings=None,maxSamplerDescriptorBufferBindings=None,maxEmbeddedImmutableSamplerBindings=None,maxEmbeddedImmutableSamplers=None,bufferCaptureReplayDescriptorDataSize=None,imageCaptureReplayDescriptorDataSize=None,imageViewCaptureReplayDescriptorDataSize=None,samplerCaptureReplayDescriptorDataSize=None,accelerationStructureCaptureReplayDescriptorDataSize=None,samplerDescriptorSize=None,combinedImageSamplerDescriptorSize=None,sampledImageDescriptorSize=None,storageImageDescriptorSize=None,uniformTexelBufferDescriptorSize=None,robustUniformTexelBufferDescriptorSize=None,storageTexelBufferDescriptorSize=None,robustStorageTexelBufferDescriptorSize=None,uniformBufferDescriptorSize=None,robustUniformBufferDescriptorSize=None,storageBufferDescriptorSize=None,robustStorageBufferDescriptorSize=None,inputAttachmentDescriptorSize=None,accelerationStructureDescriptorSize=None,maxSamplerDescriptorBufferRange=None,maxResourceDescriptorBufferRange=None,samplerDescriptorBufferAddressSpaceSize=None,resourceDescriptorBufferAddressSpaceSize=None,descriptorBufferAddressSpaceSize=None,):

    return _new('VkPhysicalDeviceDescriptorBufferPropertiesEXT', sType=sType,pNext=pNext,combinedImageSamplerDescriptorSingleArray=combinedImageSamplerDescriptorSingleArray,bufferlessPushDescriptors=bufferlessPushDescriptors,allowSamplerImageViewPostSubmitCreation=allowSamplerImageViewPostSubmitCreation,descriptorBufferOffsetAlignment=descriptorBufferOffsetAlignment,maxDescriptorBufferBindings=maxDescriptorBufferBindings,maxResourceDescriptorBufferBindings=maxResourceDescriptorBufferBindings,maxSamplerDescriptorBufferBindings=maxSamplerDescriptorBufferBindings,maxEmbeddedImmutableSamplerBindings=maxEmbeddedImmutableSamplerBindings,maxEmbeddedImmutableSamplers=maxEmbeddedImmutableSamplers,bufferCaptureReplayDescriptorDataSize=bufferCaptureReplayDescriptorDataSize,imageCaptureReplayDescriptorDataSize=imageCaptureReplayDescriptorDataSize,imageViewCaptureReplayDescriptorDataSize=imageViewCaptureReplayDescriptorDataSize,samplerCaptureReplayDescriptorDataSize=samplerCaptureReplayDescriptorDataSize,accelerationStructureCaptureReplayDescriptorDataSize=accelerationStructureCaptureReplayDescriptorDataSize,samplerDescriptorSize=samplerDescriptorSize,combinedImageSamplerDescriptorSize=combinedImageSamplerDescriptorSize,sampledImageDescriptorSize=sampledImageDescriptorSize,storageImageDescriptorSize=storageImageDescriptorSize,uniformTexelBufferDescriptorSize=uniformTexelBufferDescriptorSize,robustUniformTexelBufferDescriptorSize=robustUniformTexelBufferDescriptorSize,storageTexelBufferDescriptorSize=storageTexelBufferDescriptorSize,robustStorageTexelBufferDescriptorSize=robustStorageTexelBufferDescriptorSize,uniformBufferDescriptorSize=uniformBufferDescriptorSize,robustUniformBufferDescriptorSize=robustUniformBufferDescriptorSize,storageBufferDescriptorSize=storageBufferDescriptorSize,robustStorageBufferDescriptorSize=robustStorageBufferDescriptorSize,inputAttachmentDescriptorSize=inputAttachmentDescriptorSize,accelerationStructureDescriptorSize=accelerationStructureDescriptorSize,maxSamplerDescriptorBufferRange=maxSamplerDescriptorBufferRange,maxResourceDescriptorBufferRange=maxResourceDescriptorBufferRange,samplerDescriptorBufferAddressSpaceSize=samplerDescriptorBufferAddressSpaceSize,resourceDescriptorBufferAddressSpaceSize=resourceDescriptorBufferAddressSpaceSize,descriptorBufferAddressSpaceSize=descriptorBufferAddressSpaceSize)

def VkPhysicalDeviceDescriptorBufferDensityMapPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_BUFFER_DENSITY_MAP_PROPERTIES_EXT,pNext=None,combinedImageSamplerDensityMapDescriptorSize=None,):

    return _new('VkPhysicalDeviceDescriptorBufferDensityMapPropertiesEXT', sType=sType,pNext=pNext,combinedImageSamplerDensityMapDescriptorSize=combinedImageSamplerDensityMapDescriptorSize)

def VkDescriptorAddressInfoEXT(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_ADDRESS_INFO_EXT,pNext=None,address=None,range=None,format=None,):

    return _new('VkDescriptorAddressInfoEXT', sType=sType,pNext=pNext,address=address,range=range,format=format)

def VkDescriptorBufferBindingInfoEXT(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_BUFFER_BINDING_INFO_EXT,pNext=None,address=None,usage=None,):

    return _new('VkDescriptorBufferBindingInfoEXT', sType=sType,pNext=pNext,address=address,usage=usage)

def VkDescriptorBufferBindingPushDescriptorBufferHandleEXT(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_BUFFER_BINDING_PUSH_DESCRIPTOR_BUFFER_HANDLE_EXT,pNext=None,buffer=None,):

    return _new('VkDescriptorBufferBindingPushDescriptorBufferHandleEXT', sType=sType,pNext=pNext,buffer=buffer)

def VkDescriptorDataEXT(pSampler=None,pCombinedImageSampler=None,pInputAttachmentImage=None,pSampledImage=None,pStorageImage=None,pUniformTexelBuffer=None,pStorageTexelBuffer=None,pUniformBuffer=None,pStorageBuffer=None,accelerationStructure=None,):

    return _new('VkDescriptorDataEXT', pSampler=pSampler,pCombinedImageSampler=pCombinedImageSampler,pInputAttachmentImage=pInputAttachmentImage,pSampledImage=pSampledImage,pStorageImage=pStorageImage,pUniformTexelBuffer=pUniformTexelBuffer,pStorageTexelBuffer=pStorageTexelBuffer,pUniformBuffer=pUniformBuffer,pStorageBuffer=pStorageBuffer,accelerationStructure=accelerationStructure)

def VkDescriptorGetInfoEXT(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_GET_INFO_EXT,pNext=None,type=None,data=None,):

    return _new('VkDescriptorGetInfoEXT', sType=sType,pNext=pNext,type=type,data=data)

def VkBufferCaptureDescriptorDataInfoEXT(sType=VK_STRUCTURE_TYPE_BUFFER_CAPTURE_DESCRIPTOR_DATA_INFO_EXT,pNext=None,buffer=None,):

    return _new('VkBufferCaptureDescriptorDataInfoEXT', sType=sType,pNext=pNext,buffer=buffer)

def VkImageCaptureDescriptorDataInfoEXT(sType=VK_STRUCTURE_TYPE_IMAGE_CAPTURE_DESCRIPTOR_DATA_INFO_EXT,pNext=None,image=None,):

    return _new('VkImageCaptureDescriptorDataInfoEXT', sType=sType,pNext=pNext,image=image)

def VkImageViewCaptureDescriptorDataInfoEXT(sType=VK_STRUCTURE_TYPE_IMAGE_VIEW_CAPTURE_DESCRIPTOR_DATA_INFO_EXT,pNext=None,imageView=None,):

    return _new('VkImageViewCaptureDescriptorDataInfoEXT', sType=sType,pNext=pNext,imageView=imageView)

def VkSamplerCaptureDescriptorDataInfoEXT(sType=VK_STRUCTURE_TYPE_SAMPLER_CAPTURE_DESCRIPTOR_DATA_INFO_EXT,pNext=None,sampler=None,):

    return _new('VkSamplerCaptureDescriptorDataInfoEXT', sType=sType,pNext=pNext,sampler=sampler)

def VkAccelerationStructureCaptureDescriptorDataInfoEXT(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_CAPTURE_DESCRIPTOR_DATA_INFO_EXT,pNext=None,accelerationStructure=None,accelerationStructureNV=None,):

    return _new('VkAccelerationStructureCaptureDescriptorDataInfoEXT', sType=sType,pNext=pNext,accelerationStructure=accelerationStructure,accelerationStructureNV=accelerationStructureNV)

def VkOpaqueCaptureDescriptorDataCreateInfoEXT(sType=VK_STRUCTURE_TYPE_OPAQUE_CAPTURE_DESCRIPTOR_DATA_CREATE_INFO_EXT,pNext=None,opaqueCaptureDescriptorData=None,):

    return _new('VkOpaqueCaptureDescriptorDataCreateInfoEXT', sType=sType,pNext=pNext,opaqueCaptureDescriptorData=opaqueCaptureDescriptorData)

def VkPhysicalDeviceShaderIntegerDotProductFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_INTEGER_DOT_PRODUCT_FEATURES,pNext=None,shaderIntegerDotProduct=None,):

    return _new('VkPhysicalDeviceShaderIntegerDotProductFeatures', sType=sType,pNext=pNext,shaderIntegerDotProduct=shaderIntegerDotProduct)

def VkPhysicalDeviceShaderIntegerDotProductProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_INTEGER_DOT_PRODUCT_PROPERTIES,pNext=None,integerDotProduct8BitUnsignedAccelerated=None,integerDotProduct8BitSignedAccelerated=None,integerDotProduct8BitMixedSignednessAccelerated=None,integerDotProduct4x8BitPackedUnsignedAccelerated=None,integerDotProduct4x8BitPackedSignedAccelerated=None,integerDotProduct4x8BitPackedMixedSignednessAccelerated=None,integerDotProduct16BitUnsignedAccelerated=None,integerDotProduct16BitSignedAccelerated=None,integerDotProduct16BitMixedSignednessAccelerated=None,integerDotProduct32BitUnsignedAccelerated=None,integerDotProduct32BitSignedAccelerated=None,integerDotProduct32BitMixedSignednessAccelerated=None,integerDotProduct64BitUnsignedAccelerated=None,integerDotProduct64BitSignedAccelerated=None,integerDotProduct64BitMixedSignednessAccelerated=None,integerDotProductAccumulatingSaturating8BitUnsignedAccelerated=None,integerDotProductAccumulatingSaturating8BitSignedAccelerated=None,integerDotProductAccumulatingSaturating8BitMixedSignednessAccelerated=None,integerDotProductAccumulatingSaturating4x8BitPackedUnsignedAccelerated=None,integerDotProductAccumulatingSaturating4x8BitPackedSignedAccelerated=None,integerDotProductAccumulatingSaturating4x8BitPackedMixedSignednessAccelerated=None,integerDotProductAccumulatingSaturating16BitUnsignedAccelerated=None,integerDotProductAccumulatingSaturating16BitSignedAccelerated=None,integerDotProductAccumulatingSaturating16BitMixedSignednessAccelerated=None,integerDotProductAccumulatingSaturating32BitUnsignedAccelerated=None,integerDotProductAccumulatingSaturating32BitSignedAccelerated=None,integerDotProductAccumulatingSaturating32BitMixedSignednessAccelerated=None,integerDotProductAccumulatingSaturating64BitUnsignedAccelerated=None,integerDotProductAccumulatingSaturating64BitSignedAccelerated=None,integerDotProductAccumulatingSaturating64BitMixedSignednessAccelerated=None,):

    return _new('VkPhysicalDeviceShaderIntegerDotProductProperties', sType=sType,pNext=pNext,integerDotProduct8BitUnsignedAccelerated=integerDotProduct8BitUnsignedAccelerated,integerDotProduct8BitSignedAccelerated=integerDotProduct8BitSignedAccelerated,integerDotProduct8BitMixedSignednessAccelerated=integerDotProduct8BitMixedSignednessAccelerated,integerDotProduct4x8BitPackedUnsignedAccelerated=integerDotProduct4x8BitPackedUnsignedAccelerated,integerDotProduct4x8BitPackedSignedAccelerated=integerDotProduct4x8BitPackedSignedAccelerated,integerDotProduct4x8BitPackedMixedSignednessAccelerated=integerDotProduct4x8BitPackedMixedSignednessAccelerated,integerDotProduct16BitUnsignedAccelerated=integerDotProduct16BitUnsignedAccelerated,integerDotProduct16BitSignedAccelerated=integerDotProduct16BitSignedAccelerated,integerDotProduct16BitMixedSignednessAccelerated=integerDotProduct16BitMixedSignednessAccelerated,integerDotProduct32BitUnsignedAccelerated=integerDotProduct32BitUnsignedAccelerated,integerDotProduct32BitSignedAccelerated=integerDotProduct32BitSignedAccelerated,integerDotProduct32BitMixedSignednessAccelerated=integerDotProduct32BitMixedSignednessAccelerated,integerDotProduct64BitUnsignedAccelerated=integerDotProduct64BitUnsignedAccelerated,integerDotProduct64BitSignedAccelerated=integerDotProduct64BitSignedAccelerated,integerDotProduct64BitMixedSignednessAccelerated=integerDotProduct64BitMixedSignednessAccelerated,integerDotProductAccumulatingSaturating8BitUnsignedAccelerated=integerDotProductAccumulatingSaturating8BitUnsignedAccelerated,integerDotProductAccumulatingSaturating8BitSignedAccelerated=integerDotProductAccumulatingSaturating8BitSignedAccelerated,integerDotProductAccumulatingSaturating8BitMixedSignednessAccelerated=integerDotProductAccumulatingSaturating8BitMixedSignednessAccelerated,integerDotProductAccumulatingSaturating4x8BitPackedUnsignedAccelerated=integerDotProductAccumulatingSaturating4x8BitPackedUnsignedAccelerated,integerDotProductAccumulatingSaturating4x8BitPackedSignedAccelerated=integerDotProductAccumulatingSaturating4x8BitPackedSignedAccelerated,integerDotProductAccumulatingSaturating4x8BitPackedMixedSignednessAccelerated=integerDotProductAccumulatingSaturating4x8BitPackedMixedSignednessAccelerated,integerDotProductAccumulatingSaturating16BitUnsignedAccelerated=integerDotProductAccumulatingSaturating16BitUnsignedAccelerated,integerDotProductAccumulatingSaturating16BitSignedAccelerated=integerDotProductAccumulatingSaturating16BitSignedAccelerated,integerDotProductAccumulatingSaturating16BitMixedSignednessAccelerated=integerDotProductAccumulatingSaturating16BitMixedSignednessAccelerated,integerDotProductAccumulatingSaturating32BitUnsignedAccelerated=integerDotProductAccumulatingSaturating32BitUnsignedAccelerated,integerDotProductAccumulatingSaturating32BitSignedAccelerated=integerDotProductAccumulatingSaturating32BitSignedAccelerated,integerDotProductAccumulatingSaturating32BitMixedSignednessAccelerated=integerDotProductAccumulatingSaturating32BitMixedSignednessAccelerated,integerDotProductAccumulatingSaturating64BitUnsignedAccelerated=integerDotProductAccumulatingSaturating64BitUnsignedAccelerated,integerDotProductAccumulatingSaturating64BitSignedAccelerated=integerDotProductAccumulatingSaturating64BitSignedAccelerated,integerDotProductAccumulatingSaturating64BitMixedSignednessAccelerated=integerDotProductAccumulatingSaturating64BitMixedSignednessAccelerated)

def VkPhysicalDeviceDrmPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DRM_PROPERTIES_EXT,pNext=None,hasPrimary=None,hasRender=None,primaryMajor=None,primaryMinor=None,renderMajor=None,renderMinor=None,):

    return _new('VkPhysicalDeviceDrmPropertiesEXT', sType=sType,pNext=pNext,hasPrimary=hasPrimary,hasRender=hasRender,primaryMajor=primaryMajor,primaryMinor=primaryMinor,renderMajor=renderMajor,renderMinor=renderMinor)

def VkPhysicalDeviceFragmentShaderBarycentricFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_BARYCENTRIC_FEATURES_KHR,pNext=None,fragmentShaderBarycentric=None,):

    return _new('VkPhysicalDeviceFragmentShaderBarycentricFeaturesKHR', sType=sType,pNext=pNext,fragmentShaderBarycentric=fragmentShaderBarycentric)

def VkPhysicalDeviceFragmentShaderBarycentricPropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_BARYCENTRIC_PROPERTIES_KHR,pNext=None,triStripVertexOrderIndependentOfProvokingVertex=None,):

    return _new('VkPhysicalDeviceFragmentShaderBarycentricPropertiesKHR', sType=sType,pNext=pNext,triStripVertexOrderIndependentOfProvokingVertex=triStripVertexOrderIndependentOfProvokingVertex)

def VkPhysicalDeviceRayTracingMotionBlurFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_MOTION_BLUR_FEATURES_NV,pNext=None,rayTracingMotionBlur=None,rayTracingMotionBlurPipelineTraceRaysIndirect=None,):

    return _new('VkPhysicalDeviceRayTracingMotionBlurFeaturesNV', sType=sType,pNext=pNext,rayTracingMotionBlur=rayTracingMotionBlur,rayTracingMotionBlurPipelineTraceRaysIndirect=rayTracingMotionBlurPipelineTraceRaysIndirect)

def VkAccelerationStructureGeometryMotionTrianglesDataNV(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_MOTION_TRIANGLES_DATA_NV,pNext=None,vertexData=None,):

    return _new('VkAccelerationStructureGeometryMotionTrianglesDataNV', sType=sType,pNext=pNext,vertexData=vertexData)

def VkAccelerationStructureMotionInfoNV(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_MOTION_INFO_NV,pNext=None,maxInstances=None,flags=None,):

    return _new('VkAccelerationStructureMotionInfoNV', sType=sType,pNext=pNext,maxInstances=maxInstances,flags=flags)

def VkSRTDataNV(sx=None,a=None,b=None,pvx=None,sy=None,c=None,pvy=None,sz=None,pvz=None,qx=None,qy=None,qz=None,qw=None,tx=None,ty=None,tz=None,):

    return _new('VkSRTDataNV', sx=sx,a=a,b=b,pvx=pvx,sy=sy,c=c,pvy=pvy,sz=sz,pvz=pvz,qx=qx,qy=qy,qz=qz,qw=qw,tx=tx,ty=ty,tz=tz)

def VkAccelerationStructureSRTMotionInstanceNV(transformT0=None,transformT1=None,instanceCustomIndex=None,mask=None,instanceShaderBindingTableRecordOffset=None,flags=None,accelerationStructureReference=None,):

    return _new('VkAccelerationStructureSRTMotionInstanceNV', transformT0=transformT0,transformT1=transformT1,instanceCustomIndex=instanceCustomIndex,mask=mask,instanceShaderBindingTableRecordOffset=instanceShaderBindingTableRecordOffset,flags=flags,accelerationStructureReference=accelerationStructureReference)

def VkAccelerationStructureMatrixMotionInstanceNV(transformT0=None,transformT1=None,instanceCustomIndex=None,mask=None,instanceShaderBindingTableRecordOffset=None,flags=None,accelerationStructureReference=None,):

    return _new('VkAccelerationStructureMatrixMotionInstanceNV', transformT0=transformT0,transformT1=transformT1,instanceCustomIndex=instanceCustomIndex,mask=mask,instanceShaderBindingTableRecordOffset=instanceShaderBindingTableRecordOffset,flags=flags,accelerationStructureReference=accelerationStructureReference)

def VkAccelerationStructureMotionInstanceDataNV(staticInstance=None,matrixMotionInstance=None,srtMotionInstance=None,):

    return _new('VkAccelerationStructureMotionInstanceDataNV', staticInstance=staticInstance,matrixMotionInstance=matrixMotionInstance,srtMotionInstance=srtMotionInstance)

def VkAccelerationStructureMotionInstanceNV(type=None,flags=None,data=None,):

    return _new('VkAccelerationStructureMotionInstanceNV', type=type,flags=flags,data=data)

def VkMemoryGetRemoteAddressInfoNV(sType=VK_STRUCTURE_TYPE_MEMORY_GET_REMOTE_ADDRESS_INFO_NV,pNext=None,memory=None,handleType=None,):

    return _new('VkMemoryGetRemoteAddressInfoNV', sType=sType,pNext=pNext,memory=memory,handleType=handleType)

def VkImportMemoryBufferCollectionFUCHSIA(sType=VK_STRUCTURE_TYPE_IMPORT_MEMORY_BUFFER_COLLECTION_FUCHSIA,pNext=None,collection=None,index=None,):

    return _new('VkImportMemoryBufferCollectionFUCHSIA', sType=sType,pNext=pNext,collection=collection,index=index)

def VkBufferCollectionImageCreateInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_BUFFER_COLLECTION_IMAGE_CREATE_INFO_FUCHSIA,pNext=None,collection=None,index=None,):

    return _new('VkBufferCollectionImageCreateInfoFUCHSIA', sType=sType,pNext=pNext,collection=collection,index=index)

def VkBufferCollectionBufferCreateInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_BUFFER_COLLECTION_BUFFER_CREATE_INFO_FUCHSIA,pNext=None,collection=None,index=None,):

    return _new('VkBufferCollectionBufferCreateInfoFUCHSIA', sType=sType,pNext=pNext,collection=collection,index=index)

def VkBufferCollectionCreateInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_BUFFER_COLLECTION_CREATE_INFO_FUCHSIA,pNext=None,collectionToken=None,):

    return _new('VkBufferCollectionCreateInfoFUCHSIA', sType=sType,pNext=pNext,collectionToken=collectionToken)

def VkBufferCollectionPropertiesFUCHSIA(sType=VK_STRUCTURE_TYPE_BUFFER_COLLECTION_PROPERTIES_FUCHSIA,pNext=None,memoryTypeBits=None,bufferCount=None,createInfoIndex=None,sysmemPixelFormat=None,formatFeatures=None,sysmemColorSpaceIndex=None,samplerYcbcrConversionComponents=None,suggestedYcbcrModel=None,suggestedYcbcrRange=None,suggestedXChromaOffset=None,suggestedYChromaOffset=None,):

    return _new('VkBufferCollectionPropertiesFUCHSIA', sType=sType,pNext=pNext,memoryTypeBits=memoryTypeBits,bufferCount=bufferCount,createInfoIndex=createInfoIndex,sysmemPixelFormat=sysmemPixelFormat,formatFeatures=formatFeatures,sysmemColorSpaceIndex=sysmemColorSpaceIndex,samplerYcbcrConversionComponents=samplerYcbcrConversionComponents,suggestedYcbcrModel=suggestedYcbcrModel,suggestedYcbcrRange=suggestedYcbcrRange,suggestedXChromaOffset=suggestedXChromaOffset,suggestedYChromaOffset=suggestedYChromaOffset)

def VkBufferConstraintsInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_BUFFER_CONSTRAINTS_INFO_FUCHSIA,pNext=None,createInfo=None,requiredFormatFeatures=None,bufferCollectionConstraints=None,):

    return _new('VkBufferConstraintsInfoFUCHSIA', sType=sType,pNext=pNext,createInfo=createInfo,requiredFormatFeatures=requiredFormatFeatures,bufferCollectionConstraints=bufferCollectionConstraints)

def VkSysmemColorSpaceFUCHSIA(sType=VK_STRUCTURE_TYPE_SYSMEM_COLOR_SPACE_FUCHSIA,pNext=None,colorSpace=None,):

    return _new('VkSysmemColorSpaceFUCHSIA', sType=sType,pNext=pNext,colorSpace=colorSpace)

def VkImageFormatConstraintsInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_IMAGE_FORMAT_CONSTRAINTS_INFO_FUCHSIA,pNext=None,imageCreateInfo=None,requiredFormatFeatures=None,flags=None,sysmemPixelFormat=None,colorSpaceCount=None,pColorSpaces=None,):
    if colorSpaceCount is None and pColorSpaces is not None:
        colorSpaceCount = len(pColorSpaces)

    return _new('VkImageFormatConstraintsInfoFUCHSIA', sType=sType,pNext=pNext,imageCreateInfo=imageCreateInfo,requiredFormatFeatures=requiredFormatFeatures,flags=flags,sysmemPixelFormat=sysmemPixelFormat,colorSpaceCount=colorSpaceCount,pColorSpaces=pColorSpaces)

def VkImageConstraintsInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_IMAGE_CONSTRAINTS_INFO_FUCHSIA,pNext=None,formatConstraintsCount=None,pFormatConstraints=None,bufferCollectionConstraints=None,flags=None,):
    if formatConstraintsCount is None and pFormatConstraints is not None:
        formatConstraintsCount = len(pFormatConstraints)

    return _new('VkImageConstraintsInfoFUCHSIA', sType=sType,pNext=pNext,formatConstraintsCount=formatConstraintsCount,pFormatConstraints=pFormatConstraints,bufferCollectionConstraints=bufferCollectionConstraints,flags=flags)

def VkBufferCollectionConstraintsInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_BUFFER_COLLECTION_CONSTRAINTS_INFO_FUCHSIA,pNext=None,minBufferCount=None,maxBufferCount=None,minBufferCountForCamping=None,minBufferCountForDedicatedSlack=None,minBufferCountForSharedSlack=None,):

    return _new('VkBufferCollectionConstraintsInfoFUCHSIA', sType=sType,pNext=pNext,minBufferCount=minBufferCount,maxBufferCount=maxBufferCount,minBufferCountForCamping=minBufferCountForCamping,minBufferCountForDedicatedSlack=minBufferCountForDedicatedSlack,minBufferCountForSharedSlack=minBufferCountForSharedSlack)

def VkCudaModuleCreateInfoNV(sType=VK_STRUCTURE_TYPE_CUDA_MODULE_CREATE_INFO_NV,pNext=None,dataSize=None,pData=None,):
    if dataSize is None and pData is not None:
        dataSize = len(pData)

    return _new('VkCudaModuleCreateInfoNV', sType=sType,pNext=pNext,dataSize=dataSize,pData=pData)

def VkCudaFunctionCreateInfoNV(sType=VK_STRUCTURE_TYPE_CUDA_FUNCTION_CREATE_INFO_NV,pNext=None,module=None,pName=None,):

    return _new('VkCudaFunctionCreateInfoNV', sType=sType,pNext=pNext,module=module,pName=pName)

def VkCudaLaunchInfoNV(sType=VK_STRUCTURE_TYPE_CUDA_LAUNCH_INFO_NV,pNext=None,function=None,gridDimX=None,gridDimY=None,gridDimZ=None,blockDimX=None,blockDimY=None,blockDimZ=None,sharedMemBytes=None,paramCount=None,pParams=None,extraCount=None,pExtras=None,):
    if paramCount is None and pParams is not None:
        paramCount = len(pParams)
    if extraCount is None and pExtras is not None:
        extraCount = len(pExtras)

    return _new('VkCudaLaunchInfoNV', sType=sType,pNext=pNext,function=function,gridDimX=gridDimX,gridDimY=gridDimY,gridDimZ=gridDimZ,blockDimX=blockDimX,blockDimY=blockDimY,blockDimZ=blockDimZ,sharedMemBytes=sharedMemBytes,paramCount=paramCount,pParams=pParams,extraCount=extraCount,pExtras=pExtras)

def VkPhysicalDeviceRGBA10X6FormatsFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RGBA10X6_FORMATS_FEATURES_EXT,pNext=None,formatRgba10x6WithoutYCbCrSampler=None,):

    return _new('VkPhysicalDeviceRGBA10X6FormatsFeaturesEXT', sType=sType,pNext=pNext,formatRgba10x6WithoutYCbCrSampler=formatRgba10x6WithoutYCbCrSampler)

def VkFormatProperties3(sType=VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_3,pNext=None,linearTilingFeatures=None,optimalTilingFeatures=None,bufferFeatures=None,):

    return _new('VkFormatProperties3', sType=sType,pNext=pNext,linearTilingFeatures=linearTilingFeatures,optimalTilingFeatures=optimalTilingFeatures,bufferFeatures=bufferFeatures)

def VkDrmFormatModifierPropertiesList2EXT(sType=VK_STRUCTURE_TYPE_DRM_FORMAT_MODIFIER_PROPERTIES_LIST_2_EXT,pNext=None,drmFormatModifierCount=None,pDrmFormatModifierProperties=None,):
    if drmFormatModifierCount is None and pDrmFormatModifierProperties is not None:
        drmFormatModifierCount = len(pDrmFormatModifierProperties)

    return _new('VkDrmFormatModifierPropertiesList2EXT', sType=sType,pNext=pNext,drmFormatModifierCount=drmFormatModifierCount,pDrmFormatModifierProperties=pDrmFormatModifierProperties)

def VkDrmFormatModifierProperties2EXT(drmFormatModifier=None,drmFormatModifierPlaneCount=None,drmFormatModifierTilingFeatures=None,):

    return _new('VkDrmFormatModifierProperties2EXT', drmFormatModifier=drmFormatModifier,drmFormatModifierPlaneCount=drmFormatModifierPlaneCount,drmFormatModifierTilingFeatures=drmFormatModifierTilingFeatures)

def VkAndroidHardwareBufferFormatProperties2ANDROID(sType=VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_FORMAT_PROPERTIES_2_ANDROID,pNext=None,format=None,externalFormat=None,formatFeatures=None,samplerYcbcrConversionComponents=None,suggestedYcbcrModel=None,suggestedYcbcrRange=None,suggestedXChromaOffset=None,suggestedYChromaOffset=None,):

    return _new('VkAndroidHardwareBufferFormatProperties2ANDROID', sType=sType,pNext=pNext,format=format,externalFormat=externalFormat,formatFeatures=formatFeatures,samplerYcbcrConversionComponents=samplerYcbcrConversionComponents,suggestedYcbcrModel=suggestedYcbcrModel,suggestedYcbcrRange=suggestedYcbcrRange,suggestedXChromaOffset=suggestedXChromaOffset,suggestedYChromaOffset=suggestedYChromaOffset)

def VkPipelineRenderingCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_RENDERING_CREATE_INFO,pNext=None,viewMask=None,colorAttachmentCount=None,pColorAttachmentFormats=None,depthAttachmentFormat=None,stencilAttachmentFormat=None,):
    if colorAttachmentCount is None and pColorAttachmentFormats is not None:
        colorAttachmentCount = len(pColorAttachmentFormats)

    return _new('VkPipelineRenderingCreateInfo', sType=sType,pNext=pNext,viewMask=viewMask,colorAttachmentCount=colorAttachmentCount,pColorAttachmentFormats=pColorAttachmentFormats,depthAttachmentFormat=depthAttachmentFormat,stencilAttachmentFormat=stencilAttachmentFormat)

def VkRenderingInfo(sType=VK_STRUCTURE_TYPE_RENDERING_INFO,pNext=None,flags=None,renderArea=None,layerCount=None,viewMask=None,colorAttachmentCount=None,pColorAttachments=None,pDepthAttachment=None,pStencilAttachment=None,):
    if colorAttachmentCount is None and pColorAttachments is not None:
        colorAttachmentCount = len(pColorAttachments)

    return _new('VkRenderingInfo', sType=sType,pNext=pNext,flags=flags,renderArea=renderArea,layerCount=layerCount,viewMask=viewMask,colorAttachmentCount=colorAttachmentCount,pColorAttachments=pColorAttachments,pDepthAttachment=pDepthAttachment,pStencilAttachment=pStencilAttachment)

def VkRenderingAttachmentInfo(sType=VK_STRUCTURE_TYPE_RENDERING_ATTACHMENT_INFO,pNext=None,imageView=None,imageLayout=None,resolveMode=None,resolveImageView=None,resolveImageLayout=None,loadOp=None,storeOp=None,clearValue=None,):

    return _new('VkRenderingAttachmentInfo', sType=sType,pNext=pNext,imageView=imageView,imageLayout=imageLayout,resolveMode=resolveMode,resolveImageView=resolveImageView,resolveImageLayout=resolveImageLayout,loadOp=loadOp,storeOp=storeOp,clearValue=clearValue)

def VkRenderingFragmentShadingRateAttachmentInfoKHR(sType=VK_STRUCTURE_TYPE_RENDERING_FRAGMENT_SHADING_RATE_ATTACHMENT_INFO_KHR,pNext=None,imageView=None,imageLayout=None,shadingRateAttachmentTexelSize=None,):

    return _new('VkRenderingFragmentShadingRateAttachmentInfoKHR', sType=sType,pNext=pNext,imageView=imageView,imageLayout=imageLayout,shadingRateAttachmentTexelSize=shadingRateAttachmentTexelSize)

def VkRenderingFragmentDensityMapAttachmentInfoEXT(sType=VK_STRUCTURE_TYPE_RENDERING_FRAGMENT_DENSITY_MAP_ATTACHMENT_INFO_EXT,pNext=None,imageView=None,imageLayout=None,):

    return _new('VkRenderingFragmentDensityMapAttachmentInfoEXT', sType=sType,pNext=pNext,imageView=imageView,imageLayout=imageLayout)

def VkPhysicalDeviceDynamicRenderingFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DYNAMIC_RENDERING_FEATURES,pNext=None,dynamicRendering=None,):

    return _new('VkPhysicalDeviceDynamicRenderingFeatures', sType=sType,pNext=pNext,dynamicRendering=dynamicRendering)

def VkCommandBufferInheritanceRenderingInfo(sType=VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_RENDERING_INFO,pNext=None,flags=None,viewMask=None,colorAttachmentCount=None,pColorAttachmentFormats=None,depthAttachmentFormat=None,stencilAttachmentFormat=None,rasterizationSamples=None,):
    if colorAttachmentCount is None and pColorAttachmentFormats is not None:
        colorAttachmentCount = len(pColorAttachmentFormats)

    return _new('VkCommandBufferInheritanceRenderingInfo', sType=sType,pNext=pNext,flags=flags,viewMask=viewMask,colorAttachmentCount=colorAttachmentCount,pColorAttachmentFormats=pColorAttachmentFormats,depthAttachmentFormat=depthAttachmentFormat,stencilAttachmentFormat=stencilAttachmentFormat,rasterizationSamples=rasterizationSamples)

def VkAttachmentSampleCountInfoAMD(sType=VK_STRUCTURE_TYPE_ATTACHMENT_SAMPLE_COUNT_INFO_AMD,pNext=None,colorAttachmentCount=None,pColorAttachmentSamples=None,depthStencilAttachmentSamples=None,):
    if colorAttachmentCount is None and pColorAttachmentSamples is not None:
        colorAttachmentCount = len(pColorAttachmentSamples)

    return _new('VkAttachmentSampleCountInfoAMD', sType=sType,pNext=pNext,colorAttachmentCount=colorAttachmentCount,pColorAttachmentSamples=pColorAttachmentSamples,depthStencilAttachmentSamples=depthStencilAttachmentSamples)

def VkMultiviewPerViewAttributesInfoNVX(sType=VK_STRUCTURE_TYPE_MULTIVIEW_PER_VIEW_ATTRIBUTES_INFO_NVX,pNext=None,perViewAttributes=None,perViewAttributesPositionXOnly=None,):

    return _new('VkMultiviewPerViewAttributesInfoNVX', sType=sType,pNext=pNext,perViewAttributes=perViewAttributes,perViewAttributesPositionXOnly=perViewAttributesPositionXOnly)

def VkPhysicalDeviceImageViewMinLodFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_VIEW_MIN_LOD_FEATURES_EXT,pNext=None,minLod=None,):

    return _new('VkPhysicalDeviceImageViewMinLodFeaturesEXT', sType=sType,pNext=pNext,minLod=minLod)

def VkImageViewMinLodCreateInfoEXT(sType=VK_STRUCTURE_TYPE_IMAGE_VIEW_MIN_LOD_CREATE_INFO_EXT,pNext=None,minLod=None,):

    return _new('VkImageViewMinLodCreateInfoEXT', sType=sType,pNext=pNext,minLod=minLod)

def VkPhysicalDeviceRasterizationOrderAttachmentAccessFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RASTERIZATION_ORDER_ATTACHMENT_ACCESS_FEATURES_EXT,pNext=None,rasterizationOrderColorAttachmentAccess=None,rasterizationOrderDepthAttachmentAccess=None,rasterizationOrderStencilAttachmentAccess=None,):

    return _new('VkPhysicalDeviceRasterizationOrderAttachmentAccessFeaturesEXT', sType=sType,pNext=pNext,rasterizationOrderColorAttachmentAccess=rasterizationOrderColorAttachmentAccess,rasterizationOrderDepthAttachmentAccess=rasterizationOrderDepthAttachmentAccess,rasterizationOrderStencilAttachmentAccess=rasterizationOrderStencilAttachmentAccess)

def VkPhysicalDeviceLinearColorAttachmentFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LINEAR_COLOR_ATTACHMENT_FEATURES_NV,pNext=None,linearColorAttachment=None,):

    return _new('VkPhysicalDeviceLinearColorAttachmentFeaturesNV', sType=sType,pNext=pNext,linearColorAttachment=linearColorAttachment)

def VkPhysicalDeviceGraphicsPipelineLibraryFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GRAPHICS_PIPELINE_LIBRARY_FEATURES_EXT,pNext=None,graphicsPipelineLibrary=None,):

    return _new('VkPhysicalDeviceGraphicsPipelineLibraryFeaturesEXT', sType=sType,pNext=pNext,graphicsPipelineLibrary=graphicsPipelineLibrary)

def VkPhysicalDeviceGraphicsPipelineLibraryPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GRAPHICS_PIPELINE_LIBRARY_PROPERTIES_EXT,pNext=None,graphicsPipelineLibraryFastLinking=None,graphicsPipelineLibraryIndependentInterpolationDecoration=None,):

    return _new('VkPhysicalDeviceGraphicsPipelineLibraryPropertiesEXT', sType=sType,pNext=pNext,graphicsPipelineLibraryFastLinking=graphicsPipelineLibraryFastLinking,graphicsPipelineLibraryIndependentInterpolationDecoration=graphicsPipelineLibraryIndependentInterpolationDecoration)

def VkGraphicsPipelineLibraryCreateInfoEXT(sType=VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_LIBRARY_CREATE_INFO_EXT,pNext=None,flags=None,):

    return _new('VkGraphicsPipelineLibraryCreateInfoEXT', sType=sType,pNext=pNext,flags=flags)

def VkPhysicalDeviceDescriptorSetHostMappingFeaturesVALVE(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_SET_HOST_MAPPING_FEATURES_VALVE,pNext=None,descriptorSetHostMapping=None,):

    return _new('VkPhysicalDeviceDescriptorSetHostMappingFeaturesVALVE', sType=sType,pNext=pNext,descriptorSetHostMapping=descriptorSetHostMapping)

def VkDescriptorSetBindingReferenceVALVE(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_BINDING_REFERENCE_VALVE,pNext=None,descriptorSetLayout=None,binding=None,):

    return _new('VkDescriptorSetBindingReferenceVALVE', sType=sType,pNext=pNext,descriptorSetLayout=descriptorSetLayout,binding=binding)

def VkDescriptorSetLayoutHostMappingInfoVALVE(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_HOST_MAPPING_INFO_VALVE,pNext=None,descriptorOffset=None,descriptorSize=None,):

    return _new('VkDescriptorSetLayoutHostMappingInfoVALVE', sType=sType,pNext=pNext,descriptorOffset=descriptorOffset,descriptorSize=descriptorSize)

def VkPhysicalDeviceNestedCommandBufferFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_NESTED_COMMAND_BUFFER_FEATURES_EXT,pNext=None,nestedCommandBuffer=None,nestedCommandBufferRendering=None,nestedCommandBufferSimultaneousUse=None,):

    return _new('VkPhysicalDeviceNestedCommandBufferFeaturesEXT', sType=sType,pNext=pNext,nestedCommandBuffer=nestedCommandBuffer,nestedCommandBufferRendering=nestedCommandBufferRendering,nestedCommandBufferSimultaneousUse=nestedCommandBufferSimultaneousUse)

def VkPhysicalDeviceNestedCommandBufferPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_NESTED_COMMAND_BUFFER_PROPERTIES_EXT,pNext=None,maxCommandBufferNestingLevel=None,):

    return _new('VkPhysicalDeviceNestedCommandBufferPropertiesEXT', sType=sType,pNext=pNext,maxCommandBufferNestingLevel=maxCommandBufferNestingLevel)

def VkPhysicalDeviceShaderModuleIdentifierFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_MODULE_IDENTIFIER_FEATURES_EXT,pNext=None,shaderModuleIdentifier=None,):

    return _new('VkPhysicalDeviceShaderModuleIdentifierFeaturesEXT', sType=sType,pNext=pNext,shaderModuleIdentifier=shaderModuleIdentifier)

def VkPhysicalDeviceShaderModuleIdentifierPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_MODULE_IDENTIFIER_PROPERTIES_EXT,pNext=None,shaderModuleIdentifierAlgorithmUUID=None,):

    return _new('VkPhysicalDeviceShaderModuleIdentifierPropertiesEXT', sType=sType,pNext=pNext,shaderModuleIdentifierAlgorithmUUID=shaderModuleIdentifierAlgorithmUUID)

def VkPipelineShaderStageModuleIdentifierCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_MODULE_IDENTIFIER_CREATE_INFO_EXT,pNext=None,identifierSize=None,pIdentifier=None,):
    if identifierSize is None and pIdentifier is not None:
        identifierSize = len(pIdentifier)

    return _new('VkPipelineShaderStageModuleIdentifierCreateInfoEXT', sType=sType,pNext=pNext,identifierSize=identifierSize,pIdentifier=pIdentifier)

def VkShaderModuleIdentifierEXT(sType=VK_STRUCTURE_TYPE_SHADER_MODULE_IDENTIFIER_EXT,pNext=None,identifierSize=None,identifier=None,):

    return _new('VkShaderModuleIdentifierEXT', sType=sType,pNext=pNext,identifierSize=identifierSize,identifier=identifier)

def VkImageCompressionControlEXT(sType=VK_STRUCTURE_TYPE_IMAGE_COMPRESSION_CONTROL_EXT,pNext=None,flags=None,compressionControlPlaneCount=None,pFixedRateFlags=None,):
    if compressionControlPlaneCount is None and pFixedRateFlags is not None:
        compressionControlPlaneCount = len(pFixedRateFlags)

    return _new('VkImageCompressionControlEXT', sType=sType,pNext=pNext,flags=flags,compressionControlPlaneCount=compressionControlPlaneCount,pFixedRateFlags=pFixedRateFlags)

def VkPhysicalDeviceImageCompressionControlFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_COMPRESSION_CONTROL_FEATURES_EXT,pNext=None,imageCompressionControl=None,):

    return _new('VkPhysicalDeviceImageCompressionControlFeaturesEXT', sType=sType,pNext=pNext,imageCompressionControl=imageCompressionControl)

def VkImageCompressionPropertiesEXT(sType=VK_STRUCTURE_TYPE_IMAGE_COMPRESSION_PROPERTIES_EXT,pNext=None,imageCompressionFlags=None,imageCompressionFixedRateFlags=None,):

    return _new('VkImageCompressionPropertiesEXT', sType=sType,pNext=pNext,imageCompressionFlags=imageCompressionFlags,imageCompressionFixedRateFlags=imageCompressionFixedRateFlags)

def VkPhysicalDeviceImageCompressionControlSwapchainFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_COMPRESSION_CONTROL_SWAPCHAIN_FEATURES_EXT,pNext=None,imageCompressionControlSwapchain=None,):

    return _new('VkPhysicalDeviceImageCompressionControlSwapchainFeaturesEXT', sType=sType,pNext=pNext,imageCompressionControlSwapchain=imageCompressionControlSwapchain)

def VkImageSubresource2KHR(sType=VK_STRUCTURE_TYPE_IMAGE_SUBRESOURCE_2_KHR,pNext=None,imageSubresource=None,):

    return _new('VkImageSubresource2KHR', sType=sType,pNext=pNext,imageSubresource=imageSubresource)

def VkSubresourceLayout2KHR(sType=VK_STRUCTURE_TYPE_SUBRESOURCE_LAYOUT_2_KHR,pNext=None,subresourceLayout=None,):

    return _new('VkSubresourceLayout2KHR', sType=sType,pNext=pNext,subresourceLayout=subresourceLayout)

def VkRenderPassCreationControlEXT(sType=VK_STRUCTURE_TYPE_RENDER_PASS_CREATION_CONTROL_EXT,pNext=None,disallowMerging=None,):

    return _new('VkRenderPassCreationControlEXT', sType=sType,pNext=pNext,disallowMerging=disallowMerging)

def VkRenderPassCreationFeedbackInfoEXT(postMergeSubpassCount=None,):

    return _new('VkRenderPassCreationFeedbackInfoEXT', postMergeSubpassCount=postMergeSubpassCount)

def VkRenderPassCreationFeedbackCreateInfoEXT(sType=VK_STRUCTURE_TYPE_RENDER_PASS_CREATION_FEEDBACK_CREATE_INFO_EXT,pNext=None,pRenderPassFeedback=None,):

    return _new('VkRenderPassCreationFeedbackCreateInfoEXT', sType=sType,pNext=pNext,pRenderPassFeedback=pRenderPassFeedback)

def VkRenderPassSubpassFeedbackInfoEXT(subpassMergeStatus=None,description=None,postMergeIndex=None,):

    return _new('VkRenderPassSubpassFeedbackInfoEXT', subpassMergeStatus=subpassMergeStatus,description=description,postMergeIndex=postMergeIndex)

def VkRenderPassSubpassFeedbackCreateInfoEXT(sType=VK_STRUCTURE_TYPE_RENDER_PASS_SUBPASS_FEEDBACK_CREATE_INFO_EXT,pNext=None,pSubpassFeedback=None,):

    return _new('VkRenderPassSubpassFeedbackCreateInfoEXT', sType=sType,pNext=pNext,pSubpassFeedback=pSubpassFeedback)

def VkPhysicalDeviceSubpassMergeFeedbackFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBPASS_MERGE_FEEDBACK_FEATURES_EXT,pNext=None,subpassMergeFeedback=None,):

    return _new('VkPhysicalDeviceSubpassMergeFeedbackFeaturesEXT', sType=sType,pNext=pNext,subpassMergeFeedback=subpassMergeFeedback)

def VkMicromapBuildInfoEXT(sType=VK_STRUCTURE_TYPE_MICROMAP_BUILD_INFO_EXT,pNext=None,type=None,flags=None,mode=None,dstMicromap=None,usageCountsCount=None,pUsageCounts=None,ppUsageCounts=None,data=None,scratchData=None,triangleArray=None,triangleArrayStride=None,):
    if usageCountsCount is None and pUsageCounts is not None:
        usageCountsCount = len(pUsageCounts)
    if usageCountsCount is None and ppUsageCounts is not None:
        usageCountsCount = len(ppUsageCounts)

    return _new('VkMicromapBuildInfoEXT', sType=sType,pNext=pNext,type=type,flags=flags,mode=mode,dstMicromap=dstMicromap,usageCountsCount=usageCountsCount,pUsageCounts=pUsageCounts,ppUsageCounts=ppUsageCounts,data=data,scratchData=scratchData,triangleArray=triangleArray,triangleArrayStride=triangleArrayStride)

def VkMicromapCreateInfoEXT(sType=VK_STRUCTURE_TYPE_MICROMAP_CREATE_INFO_EXT,pNext=None,createFlags=None,buffer=None,offset=None,size=None,type=None,deviceAddress=None,):

    return _new('VkMicromapCreateInfoEXT', sType=sType,pNext=pNext,createFlags=createFlags,buffer=buffer,offset=offset,size=size,type=type,deviceAddress=deviceAddress)

def VkMicromapVersionInfoEXT(sType=VK_STRUCTURE_TYPE_MICROMAP_VERSION_INFO_EXT,pNext=None,pVersionData=None,):

    return _new('VkMicromapVersionInfoEXT', sType=sType,pNext=pNext,pVersionData=pVersionData)

def VkCopyMicromapInfoEXT(sType=VK_STRUCTURE_TYPE_COPY_MICROMAP_INFO_EXT,pNext=None,src=None,dst=None,mode=None,):

    return _new('VkCopyMicromapInfoEXT', sType=sType,pNext=pNext,src=src,dst=dst,mode=mode)

def VkCopyMicromapToMemoryInfoEXT(sType=VK_STRUCTURE_TYPE_COPY_MICROMAP_TO_MEMORY_INFO_EXT,pNext=None,src=None,dst=None,mode=None,):

    return _new('VkCopyMicromapToMemoryInfoEXT', sType=sType,pNext=pNext,src=src,dst=dst,mode=mode)

def VkCopyMemoryToMicromapInfoEXT(sType=VK_STRUCTURE_TYPE_COPY_MEMORY_TO_MICROMAP_INFO_EXT,pNext=None,src=None,dst=None,mode=None,):

    return _new('VkCopyMemoryToMicromapInfoEXT', sType=sType,pNext=pNext,src=src,dst=dst,mode=mode)

def VkMicromapBuildSizesInfoEXT(sType=VK_STRUCTURE_TYPE_MICROMAP_BUILD_SIZES_INFO_EXT,pNext=None,micromapSize=None,buildScratchSize=None,discardable=None,):

    return _new('VkMicromapBuildSizesInfoEXT', sType=sType,pNext=pNext,micromapSize=micromapSize,buildScratchSize=buildScratchSize,discardable=discardable)

def VkMicromapUsageEXT(count=None,subdivisionLevel=None,format=None,):

    return _new('VkMicromapUsageEXT', count=count,subdivisionLevel=subdivisionLevel,format=format)

def VkMicromapTriangleEXT(dataOffset=None,subdivisionLevel=None,format=None,):

    return _new('VkMicromapTriangleEXT', dataOffset=dataOffset,subdivisionLevel=subdivisionLevel,format=format)

def VkPhysicalDeviceOpacityMicromapFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_OPACITY_MICROMAP_FEATURES_EXT,pNext=None,micromap=None,micromapCaptureReplay=None,micromapHostCommands=None,):

    return _new('VkPhysicalDeviceOpacityMicromapFeaturesEXT', sType=sType,pNext=pNext,micromap=micromap,micromapCaptureReplay=micromapCaptureReplay,micromapHostCommands=micromapHostCommands)

def VkPhysicalDeviceOpacityMicromapPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_OPACITY_MICROMAP_PROPERTIES_EXT,pNext=None,maxOpacity2StateSubdivisionLevel=None,maxOpacity4StateSubdivisionLevel=None,):

    return _new('VkPhysicalDeviceOpacityMicromapPropertiesEXT', sType=sType,pNext=pNext,maxOpacity2StateSubdivisionLevel=maxOpacity2StateSubdivisionLevel,maxOpacity4StateSubdivisionLevel=maxOpacity4StateSubdivisionLevel)

def VkAccelerationStructureTrianglesOpacityMicromapEXT(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_TRIANGLES_OPACITY_MICROMAP_EXT,pNext=None,indexType=None,indexBuffer=None,indexStride=None,baseTriangle=None,usageCountsCount=None,pUsageCounts=None,ppUsageCounts=None,micromap=None,):
    if usageCountsCount is None and pUsageCounts is not None:
        usageCountsCount = len(pUsageCounts)
    if usageCountsCount is None and ppUsageCounts is not None:
        usageCountsCount = len(ppUsageCounts)

    return _new('VkAccelerationStructureTrianglesOpacityMicromapEXT', sType=sType,pNext=pNext,indexType=indexType,indexBuffer=indexBuffer,indexStride=indexStride,baseTriangle=baseTriangle,usageCountsCount=usageCountsCount,pUsageCounts=pUsageCounts,ppUsageCounts=ppUsageCounts,micromap=micromap)

def VkPhysicalDeviceDisplacementMicromapFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DISPLACEMENT_MICROMAP_FEATURES_NV,pNext=None,displacementMicromap=None,):

    return _new('VkPhysicalDeviceDisplacementMicromapFeaturesNV', sType=sType,pNext=pNext,displacementMicromap=displacementMicromap)

def VkPhysicalDeviceDisplacementMicromapPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DISPLACEMENT_MICROMAP_PROPERTIES_NV,pNext=None,maxDisplacementMicromapSubdivisionLevel=None,):

    return _new('VkPhysicalDeviceDisplacementMicromapPropertiesNV', sType=sType,pNext=pNext,maxDisplacementMicromapSubdivisionLevel=maxDisplacementMicromapSubdivisionLevel)

def VkAccelerationStructureTrianglesDisplacementMicromapNV(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_TRIANGLES_DISPLACEMENT_MICROMAP_NV,pNext=None,displacementBiasAndScaleFormat=None,displacementVectorFormat=None,displacementBiasAndScaleBuffer=None,displacementBiasAndScaleStride=None,displacementVectorBuffer=None,displacementVectorStride=None,displacedMicromapPrimitiveFlags=None,displacedMicromapPrimitiveFlagsStride=None,indexType=None,indexBuffer=None,indexStride=None,baseTriangle=None,usageCountsCount=None,pUsageCounts=None,ppUsageCounts=None,micromap=None,):
    if usageCountsCount is None and pUsageCounts is not None:
        usageCountsCount = len(pUsageCounts)
    if usageCountsCount is None and ppUsageCounts is not None:
        usageCountsCount = len(ppUsageCounts)

    return _new('VkAccelerationStructureTrianglesDisplacementMicromapNV', sType=sType,pNext=pNext,displacementBiasAndScaleFormat=displacementBiasAndScaleFormat,displacementVectorFormat=displacementVectorFormat,displacementBiasAndScaleBuffer=displacementBiasAndScaleBuffer,displacementBiasAndScaleStride=displacementBiasAndScaleStride,displacementVectorBuffer=displacementVectorBuffer,displacementVectorStride=displacementVectorStride,displacedMicromapPrimitiveFlags=displacedMicromapPrimitiveFlags,displacedMicromapPrimitiveFlagsStride=displacedMicromapPrimitiveFlagsStride,indexType=indexType,indexBuffer=indexBuffer,indexStride=indexStride,baseTriangle=baseTriangle,usageCountsCount=usageCountsCount,pUsageCounts=pUsageCounts,ppUsageCounts=ppUsageCounts,micromap=micromap)

def VkPipelinePropertiesIdentifierEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_PROPERTIES_IDENTIFIER_EXT,pNext=None,pipelineIdentifier=None,):

    return _new('VkPipelinePropertiesIdentifierEXT', sType=sType,pNext=pNext,pipelineIdentifier=pipelineIdentifier)

def VkPhysicalDevicePipelinePropertiesFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_PROPERTIES_FEATURES_EXT,pNext=None,pipelinePropertiesIdentifier=None,):

    return _new('VkPhysicalDevicePipelinePropertiesFeaturesEXT', sType=sType,pNext=pNext,pipelinePropertiesIdentifier=pipelinePropertiesIdentifier)

def VkPhysicalDeviceShaderEarlyAndLateFragmentTestsFeaturesAMD(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_EARLY_AND_LATE_FRAGMENT_TESTS_FEATURES_AMD,pNext=None,shaderEarlyAndLateFragmentTests=None,):

    return _new('VkPhysicalDeviceShaderEarlyAndLateFragmentTestsFeaturesAMD', sType=sType,pNext=pNext,shaderEarlyAndLateFragmentTests=shaderEarlyAndLateFragmentTests)

def VkExternalMemoryAcquireUnmodifiedEXT(sType=VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_ACQUIRE_UNMODIFIED_EXT,pNext=None,acquireUnmodifiedMemory=None,):

    return _new('VkExternalMemoryAcquireUnmodifiedEXT', sType=sType,pNext=pNext,acquireUnmodifiedMemory=acquireUnmodifiedMemory)

def VkExportMetalObjectCreateInfoEXT(sType=VK_STRUCTURE_TYPE_EXPORT_METAL_OBJECT_CREATE_INFO_EXT,pNext=None,exportObjectType=None,):

    return _new('VkExportMetalObjectCreateInfoEXT', sType=sType,pNext=pNext,exportObjectType=exportObjectType)

def VkExportMetalObjectsInfoEXT(sType=VK_STRUCTURE_TYPE_EXPORT_METAL_OBJECTS_INFO_EXT,pNext=None,):

    return _new('VkExportMetalObjectsInfoEXT', sType=sType,pNext=pNext)

def VkExportMetalDeviceInfoEXT(sType=VK_STRUCTURE_TYPE_EXPORT_METAL_DEVICE_INFO_EXT,pNext=None,mtlDevice=None,):

    return _new('VkExportMetalDeviceInfoEXT', sType=sType,pNext=pNext,mtlDevice=mtlDevice)

def VkExportMetalCommandQueueInfoEXT(sType=VK_STRUCTURE_TYPE_EXPORT_METAL_COMMAND_QUEUE_INFO_EXT,pNext=None,queue=None,mtlCommandQueue=None,):

    return _new('VkExportMetalCommandQueueInfoEXT', sType=sType,pNext=pNext,queue=queue,mtlCommandQueue=mtlCommandQueue)

def VkExportMetalBufferInfoEXT(sType=VK_STRUCTURE_TYPE_EXPORT_METAL_BUFFER_INFO_EXT,pNext=None,memory=None,mtlBuffer=None,):

    return _new('VkExportMetalBufferInfoEXT', sType=sType,pNext=pNext,memory=memory,mtlBuffer=mtlBuffer)

def VkImportMetalBufferInfoEXT(sType=VK_STRUCTURE_TYPE_IMPORT_METAL_BUFFER_INFO_EXT,pNext=None,mtlBuffer=None,):

    return _new('VkImportMetalBufferInfoEXT', sType=sType,pNext=pNext,mtlBuffer=mtlBuffer)

def VkExportMetalTextureInfoEXT(sType=VK_STRUCTURE_TYPE_EXPORT_METAL_TEXTURE_INFO_EXT,pNext=None,image=None,imageView=None,bufferView=None,plane=None,mtlTexture=None,):

    return _new('VkExportMetalTextureInfoEXT', sType=sType,pNext=pNext,image=image,imageView=imageView,bufferView=bufferView,plane=plane,mtlTexture=mtlTexture)

def VkImportMetalTextureInfoEXT(sType=VK_STRUCTURE_TYPE_IMPORT_METAL_TEXTURE_INFO_EXT,pNext=None,plane=None,mtlTexture=None,):

    return _new('VkImportMetalTextureInfoEXT', sType=sType,pNext=pNext,plane=plane,mtlTexture=mtlTexture)

def VkExportMetalIOSurfaceInfoEXT(sType=VK_STRUCTURE_TYPE_EXPORT_METAL_IO_SURFACE_INFO_EXT,pNext=None,image=None,ioSurface=None,):

    return _new('VkExportMetalIOSurfaceInfoEXT', sType=sType,pNext=pNext,image=image,ioSurface=ioSurface)

def VkImportMetalIOSurfaceInfoEXT(sType=VK_STRUCTURE_TYPE_IMPORT_METAL_IO_SURFACE_INFO_EXT,pNext=None,ioSurface=None,):

    return _new('VkImportMetalIOSurfaceInfoEXT', sType=sType,pNext=pNext,ioSurface=ioSurface)

def VkExportMetalSharedEventInfoEXT(sType=VK_STRUCTURE_TYPE_EXPORT_METAL_SHARED_EVENT_INFO_EXT,pNext=None,semaphore=None,event=None,mtlSharedEvent=None,):

    return _new('VkExportMetalSharedEventInfoEXT', sType=sType,pNext=pNext,semaphore=semaphore,event=event,mtlSharedEvent=mtlSharedEvent)

def VkImportMetalSharedEventInfoEXT(sType=VK_STRUCTURE_TYPE_IMPORT_METAL_SHARED_EVENT_INFO_EXT,pNext=None,mtlSharedEvent=None,):

    return _new('VkImportMetalSharedEventInfoEXT', sType=sType,pNext=pNext,mtlSharedEvent=mtlSharedEvent)

def VkPhysicalDeviceNonSeamlessCubeMapFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_NON_SEAMLESS_CUBE_MAP_FEATURES_EXT,pNext=None,nonSeamlessCubeMap=None,):

    return _new('VkPhysicalDeviceNonSeamlessCubeMapFeaturesEXT', sType=sType,pNext=pNext,nonSeamlessCubeMap=nonSeamlessCubeMap)

def VkPhysicalDevicePipelineRobustnessFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_ROBUSTNESS_FEATURES_EXT,pNext=None,pipelineRobustness=None,):

    return _new('VkPhysicalDevicePipelineRobustnessFeaturesEXT', sType=sType,pNext=pNext,pipelineRobustness=pipelineRobustness)

def VkPipelineRobustnessCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_ROBUSTNESS_CREATE_INFO_EXT,pNext=None,storageBuffers=None,uniformBuffers=None,vertexInputs=None,images=None,):

    return _new('VkPipelineRobustnessCreateInfoEXT', sType=sType,pNext=pNext,storageBuffers=storageBuffers,uniformBuffers=uniformBuffers,vertexInputs=vertexInputs,images=images)

def VkPhysicalDevicePipelineRobustnessPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_ROBUSTNESS_PROPERTIES_EXT,pNext=None,defaultRobustnessStorageBuffers=None,defaultRobustnessUniformBuffers=None,defaultRobustnessVertexInputs=None,defaultRobustnessImages=None,):

    return _new('VkPhysicalDevicePipelineRobustnessPropertiesEXT', sType=sType,pNext=pNext,defaultRobustnessStorageBuffers=defaultRobustnessStorageBuffers,defaultRobustnessUniformBuffers=defaultRobustnessUniformBuffers,defaultRobustnessVertexInputs=defaultRobustnessVertexInputs,defaultRobustnessImages=defaultRobustnessImages)

def VkImageViewSampleWeightCreateInfoQCOM(sType=VK_STRUCTURE_TYPE_IMAGE_VIEW_SAMPLE_WEIGHT_CREATE_INFO_QCOM,pNext=None,filterCenter=None,filterSize=None,numPhases=None,):

    return _new('VkImageViewSampleWeightCreateInfoQCOM', sType=sType,pNext=pNext,filterCenter=filterCenter,filterSize=filterSize,numPhases=numPhases)

def VkPhysicalDeviceImageProcessingFeaturesQCOM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_PROCESSING_FEATURES_QCOM,pNext=None,textureSampleWeighted=None,textureBoxFilter=None,textureBlockMatch=None,):

    return _new('VkPhysicalDeviceImageProcessingFeaturesQCOM', sType=sType,pNext=pNext,textureSampleWeighted=textureSampleWeighted,textureBoxFilter=textureBoxFilter,textureBlockMatch=textureBlockMatch)

def VkPhysicalDeviceImageProcessingPropertiesQCOM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_PROCESSING_PROPERTIES_QCOM,pNext=None,maxWeightFilterPhases=None,maxWeightFilterDimension=None,maxBlockMatchRegion=None,maxBoxFilterBlockSize=None,):

    return _new('VkPhysicalDeviceImageProcessingPropertiesQCOM', sType=sType,pNext=pNext,maxWeightFilterPhases=maxWeightFilterPhases,maxWeightFilterDimension=maxWeightFilterDimension,maxBlockMatchRegion=maxBlockMatchRegion,maxBoxFilterBlockSize=maxBoxFilterBlockSize)

def VkPhysicalDeviceTilePropertiesFeaturesQCOM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TILE_PROPERTIES_FEATURES_QCOM,pNext=None,tileProperties=None,):

    return _new('VkPhysicalDeviceTilePropertiesFeaturesQCOM', sType=sType,pNext=pNext,tileProperties=tileProperties)

def VkTilePropertiesQCOM(sType=VK_STRUCTURE_TYPE_TILE_PROPERTIES_QCOM,pNext=None,tileSize=None,apronSize=None,origin=None,):

    return _new('VkTilePropertiesQCOM', sType=sType,pNext=pNext,tileSize=tileSize,apronSize=apronSize,origin=origin)

def VkPhysicalDeviceAmigoProfilingFeaturesSEC(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_AMIGO_PROFILING_FEATURES_SEC,pNext=None,amigoProfiling=None,):

    return _new('VkPhysicalDeviceAmigoProfilingFeaturesSEC', sType=sType,pNext=pNext,amigoProfiling=amigoProfiling)

def VkAmigoProfilingSubmitInfoSEC(sType=VK_STRUCTURE_TYPE_AMIGO_PROFILING_SUBMIT_INFO_SEC,pNext=None,firstDrawTimestamp=None,swapBufferTimestamp=None,):

    return _new('VkAmigoProfilingSubmitInfoSEC', sType=sType,pNext=pNext,firstDrawTimestamp=firstDrawTimestamp,swapBufferTimestamp=swapBufferTimestamp)

def VkPhysicalDeviceAttachmentFeedbackLoopLayoutFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ATTACHMENT_FEEDBACK_LOOP_LAYOUT_FEATURES_EXT,pNext=None,attachmentFeedbackLoopLayout=None,):

    return _new('VkPhysicalDeviceAttachmentFeedbackLoopLayoutFeaturesEXT', sType=sType,pNext=pNext,attachmentFeedbackLoopLayout=attachmentFeedbackLoopLayout)

def VkPhysicalDeviceDepthClampZeroOneFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_CLAMP_ZERO_ONE_FEATURES_EXT,pNext=None,depthClampZeroOne=None,):

    return _new('VkPhysicalDeviceDepthClampZeroOneFeaturesEXT', sType=sType,pNext=pNext,depthClampZeroOne=depthClampZeroOne)

def VkPhysicalDeviceAddressBindingReportFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ADDRESS_BINDING_REPORT_FEATURES_EXT,pNext=None,reportAddressBinding=None,):

    return _new('VkPhysicalDeviceAddressBindingReportFeaturesEXT', sType=sType,pNext=pNext,reportAddressBinding=reportAddressBinding)

def VkDeviceAddressBindingCallbackDataEXT(sType=VK_STRUCTURE_TYPE_DEVICE_ADDRESS_BINDING_CALLBACK_DATA_EXT,pNext=None,flags=None,baseAddress=None,size=None,bindingType=None,):

    return _new('VkDeviceAddressBindingCallbackDataEXT', sType=sType,pNext=pNext,flags=flags,baseAddress=baseAddress,size=size,bindingType=bindingType)

def VkPhysicalDeviceOpticalFlowFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_OPTICAL_FLOW_FEATURES_NV,pNext=None,opticalFlow=None,):

    return _new('VkPhysicalDeviceOpticalFlowFeaturesNV', sType=sType,pNext=pNext,opticalFlow=opticalFlow)

def VkPhysicalDeviceOpticalFlowPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_OPTICAL_FLOW_PROPERTIES_NV,pNext=None,supportedOutputGridSizes=None,supportedHintGridSizes=None,hintSupported=None,costSupported=None,bidirectionalFlowSupported=None,globalFlowSupported=None,minWidth=None,minHeight=None,maxWidth=None,maxHeight=None,maxNumRegionsOfInterest=None,):

    return _new('VkPhysicalDeviceOpticalFlowPropertiesNV', sType=sType,pNext=pNext,supportedOutputGridSizes=supportedOutputGridSizes,supportedHintGridSizes=supportedHintGridSizes,hintSupported=hintSupported,costSupported=costSupported,bidirectionalFlowSupported=bidirectionalFlowSupported,globalFlowSupported=globalFlowSupported,minWidth=minWidth,minHeight=minHeight,maxWidth=maxWidth,maxHeight=maxHeight,maxNumRegionsOfInterest=maxNumRegionsOfInterest)

def VkOpticalFlowImageFormatInfoNV(sType=VK_STRUCTURE_TYPE_OPTICAL_FLOW_IMAGE_FORMAT_INFO_NV,pNext=None,usage=None,):

    return _new('VkOpticalFlowImageFormatInfoNV', sType=sType,pNext=pNext,usage=usage)

def VkOpticalFlowImageFormatPropertiesNV(sType=VK_STRUCTURE_TYPE_OPTICAL_FLOW_IMAGE_FORMAT_PROPERTIES_NV,pNext=None,format=None,):

    return _new('VkOpticalFlowImageFormatPropertiesNV', sType=sType,pNext=pNext,format=format)

def VkOpticalFlowSessionCreateInfoNV(sType=VK_STRUCTURE_TYPE_OPTICAL_FLOW_SESSION_CREATE_INFO_NV,pNext=None,width=None,height=None,imageFormat=None,flowVectorFormat=None,costFormat=None,outputGridSize=None,hintGridSize=None,performanceLevel=None,flags=None,):

    return _new('VkOpticalFlowSessionCreateInfoNV', sType=sType,pNext=pNext,width=width,height=height,imageFormat=imageFormat,flowVectorFormat=flowVectorFormat,costFormat=costFormat,outputGridSize=outputGridSize,hintGridSize=hintGridSize,performanceLevel=performanceLevel,flags=flags)

def VkOpticalFlowSessionCreatePrivateDataInfoNV(sType=VK_STRUCTURE_TYPE_OPTICAL_FLOW_SESSION_CREATE_PRIVATE_DATA_INFO_NV,pNext=None,id=None,size=None,pPrivateData=None,):

    return _new('VkOpticalFlowSessionCreatePrivateDataInfoNV', sType=sType,pNext=pNext,id=id,size=size,pPrivateData=pPrivateData)

def VkOpticalFlowExecuteInfoNV(sType=VK_STRUCTURE_TYPE_OPTICAL_FLOW_EXECUTE_INFO_NV,pNext=None,flags=None,regionCount=None,pRegions=None,):
    if regionCount is None and pRegions is not None:
        regionCount = len(pRegions)

    return _new('VkOpticalFlowExecuteInfoNV', sType=sType,pNext=pNext,flags=flags,regionCount=regionCount,pRegions=pRegions)

def VkPhysicalDeviceFaultFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FAULT_FEATURES_EXT,pNext=None,deviceFault=None,deviceFaultVendorBinary=None,):

    return _new('VkPhysicalDeviceFaultFeaturesEXT', sType=sType,pNext=pNext,deviceFault=deviceFault,deviceFaultVendorBinary=deviceFaultVendorBinary)

def VkDeviceFaultAddressInfoEXT(addressType=None,reportedAddress=None,addressPrecision=None,):

    return _new('VkDeviceFaultAddressInfoEXT', addressType=addressType,reportedAddress=reportedAddress,addressPrecision=addressPrecision)

def VkDeviceFaultVendorInfoEXT(description=None,vendorFaultCode=None,vendorFaultData=None,):

    return _new('VkDeviceFaultVendorInfoEXT', description=description,vendorFaultCode=vendorFaultCode,vendorFaultData=vendorFaultData)

def VkDeviceFaultCountsEXT(sType=VK_STRUCTURE_TYPE_DEVICE_FAULT_COUNTS_EXT,pNext=None,addressInfoCount=None,vendorInfoCount=None,vendorBinarySize=None,):

    return _new('VkDeviceFaultCountsEXT', sType=sType,pNext=pNext,addressInfoCount=addressInfoCount,vendorInfoCount=vendorInfoCount,vendorBinarySize=vendorBinarySize)

def VkDeviceFaultInfoEXT(sType=VK_STRUCTURE_TYPE_DEVICE_FAULT_INFO_EXT,pNext=None,description=None,pAddressInfos=None,pVendorInfos=None,pVendorBinaryData=None,):

    return _new('VkDeviceFaultInfoEXT', sType=sType,pNext=pNext,description=description,pAddressInfos=pAddressInfos,pVendorInfos=pVendorInfos,pVendorBinaryData=pVendorBinaryData)

def VkDeviceFaultVendorBinaryHeaderVersionOneEXT(headerSize=None,headerVersion=None,vendorID=None,deviceID=None,driverVersion=None,pipelineCacheUUID=None,applicationNameOffset=None,applicationVersion=None,engineNameOffset=None,engineVersion=None,apiVersion=None,):

    return _new('VkDeviceFaultVendorBinaryHeaderVersionOneEXT', headerSize=headerSize,headerVersion=headerVersion,vendorID=vendorID,deviceID=deviceID,driverVersion=driverVersion,pipelineCacheUUID=pipelineCacheUUID,applicationNameOffset=applicationNameOffset,applicationVersion=applicationVersion,engineNameOffset=engineNameOffset,engineVersion=engineVersion,apiVersion=apiVersion)

def VkPhysicalDevicePipelineLibraryGroupHandlesFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_LIBRARY_GROUP_HANDLES_FEATURES_EXT,pNext=None,pipelineLibraryGroupHandles=None,):

    return _new('VkPhysicalDevicePipelineLibraryGroupHandlesFeaturesEXT', sType=sType,pNext=pNext,pipelineLibraryGroupHandles=pipelineLibraryGroupHandles)

def VkDepthBiasInfoEXT(sType=VK_STRUCTURE_TYPE_DEPTH_BIAS_INFO_EXT,pNext=None,depthBiasConstantFactor=None,depthBiasClamp=None,depthBiasSlopeFactor=None,):

    return _new('VkDepthBiasInfoEXT', sType=sType,pNext=pNext,depthBiasConstantFactor=depthBiasConstantFactor,depthBiasClamp=depthBiasClamp,depthBiasSlopeFactor=depthBiasSlopeFactor)

def VkDepthBiasRepresentationInfoEXT(sType=VK_STRUCTURE_TYPE_DEPTH_BIAS_REPRESENTATION_INFO_EXT,pNext=None,depthBiasRepresentation=None,depthBiasExact=None,):

    return _new('VkDepthBiasRepresentationInfoEXT', sType=sType,pNext=pNext,depthBiasRepresentation=depthBiasRepresentation,depthBiasExact=depthBiasExact)

def VkDecompressMemoryRegionNV(srcAddress=None,dstAddress=None,compressedSize=None,decompressedSize=None,decompressionMethod=None,):

    return _new('VkDecompressMemoryRegionNV', srcAddress=srcAddress,dstAddress=dstAddress,compressedSize=compressedSize,decompressedSize=decompressedSize,decompressionMethod=decompressionMethod)

def VkPhysicalDeviceShaderCoreBuiltinsPropertiesARM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_BUILTINS_PROPERTIES_ARM,pNext=None,shaderCoreMask=None,shaderCoreCount=None,shaderWarpsPerCore=None,):

    return _new('VkPhysicalDeviceShaderCoreBuiltinsPropertiesARM', sType=sType,pNext=pNext,shaderCoreMask=shaderCoreMask,shaderCoreCount=shaderCoreCount,shaderWarpsPerCore=shaderWarpsPerCore)

def VkPhysicalDeviceShaderCoreBuiltinsFeaturesARM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_BUILTINS_FEATURES_ARM,pNext=None,shaderCoreBuiltins=None,):

    return _new('VkPhysicalDeviceShaderCoreBuiltinsFeaturesARM', sType=sType,pNext=pNext,shaderCoreBuiltins=shaderCoreBuiltins)

def VkFrameBoundaryEXT(sType=VK_STRUCTURE_TYPE_FRAME_BOUNDARY_EXT,pNext=None,flags=None,frameID=None,imageCount=None,pImages=None,bufferCount=None,pBuffers=None,tagName=None,tagSize=None,pTag=None,):
    if imageCount is None and pImages is not None:
        imageCount = len(pImages)
    if bufferCount is None and pBuffers is not None:
        bufferCount = len(pBuffers)
    if tagSize is None and pTag is not None:
        tagSize = len(pTag)

    return _new('VkFrameBoundaryEXT', sType=sType,pNext=pNext,flags=flags,frameID=frameID,imageCount=imageCount,pImages=pImages,bufferCount=bufferCount,pBuffers=pBuffers,tagName=tagName,tagSize=tagSize,pTag=pTag)

def VkPhysicalDeviceFrameBoundaryFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAME_BOUNDARY_FEATURES_EXT,pNext=None,frameBoundary=None,):

    return _new('VkPhysicalDeviceFrameBoundaryFeaturesEXT', sType=sType,pNext=pNext,frameBoundary=frameBoundary)

def VkPhysicalDeviceDynamicRenderingUnusedAttachmentsFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DYNAMIC_RENDERING_UNUSED_ATTACHMENTS_FEATURES_EXT,pNext=None,dynamicRenderingUnusedAttachments=None,):

    return _new('VkPhysicalDeviceDynamicRenderingUnusedAttachmentsFeaturesEXT', sType=sType,pNext=pNext,dynamicRenderingUnusedAttachments=dynamicRenderingUnusedAttachments)

def VkSurfacePresentModeEXT(sType=VK_STRUCTURE_TYPE_SURFACE_PRESENT_MODE_EXT,pNext=None,presentMode=None,):

    return _new('VkSurfacePresentModeEXT', sType=sType,pNext=pNext,presentMode=presentMode)

def VkSurfacePresentScalingCapabilitiesEXT(sType=VK_STRUCTURE_TYPE_SURFACE_PRESENT_SCALING_CAPABILITIES_EXT,pNext=None,supportedPresentScaling=None,supportedPresentGravityX=None,supportedPresentGravityY=None,minScaledImageExtent=None,maxScaledImageExtent=None,):

    return _new('VkSurfacePresentScalingCapabilitiesEXT', sType=sType,pNext=pNext,supportedPresentScaling=supportedPresentScaling,supportedPresentGravityX=supportedPresentGravityX,supportedPresentGravityY=supportedPresentGravityY,minScaledImageExtent=minScaledImageExtent,maxScaledImageExtent=maxScaledImageExtent)

def VkSurfacePresentModeCompatibilityEXT(sType=VK_STRUCTURE_TYPE_SURFACE_PRESENT_MODE_COMPATIBILITY_EXT,pNext=None,presentModeCount=None,pPresentModes=None,):
    if presentModeCount is None and pPresentModes is not None:
        presentModeCount = len(pPresentModes)

    return _new('VkSurfacePresentModeCompatibilityEXT', sType=sType,pNext=pNext,presentModeCount=presentModeCount,pPresentModes=pPresentModes)

def VkPhysicalDeviceSwapchainMaintenance1FeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SWAPCHAIN_MAINTENANCE_1_FEATURES_EXT,pNext=None,swapchainMaintenance1=None,):

    return _new('VkPhysicalDeviceSwapchainMaintenance1FeaturesEXT', sType=sType,pNext=pNext,swapchainMaintenance1=swapchainMaintenance1)

def VkSwapchainPresentFenceInfoEXT(sType=VK_STRUCTURE_TYPE_SWAPCHAIN_PRESENT_FENCE_INFO_EXT,pNext=None,swapchainCount=None,pFences=None,):
    if swapchainCount is None and pFences is not None:
        swapchainCount = len(pFences)

    return _new('VkSwapchainPresentFenceInfoEXT', sType=sType,pNext=pNext,swapchainCount=swapchainCount,pFences=pFences)

def VkSwapchainPresentModesCreateInfoEXT(sType=VK_STRUCTURE_TYPE_SWAPCHAIN_PRESENT_MODES_CREATE_INFO_EXT,pNext=None,presentModeCount=None,pPresentModes=None,):
    if presentModeCount is None and pPresentModes is not None:
        presentModeCount = len(pPresentModes)

    return _new('VkSwapchainPresentModesCreateInfoEXT', sType=sType,pNext=pNext,presentModeCount=presentModeCount,pPresentModes=pPresentModes)

def VkSwapchainPresentModeInfoEXT(sType=VK_STRUCTURE_TYPE_SWAPCHAIN_PRESENT_MODE_INFO_EXT,pNext=None,swapchainCount=None,pPresentModes=None,):
    if swapchainCount is None and pPresentModes is not None:
        swapchainCount = len(pPresentModes)

    return _new('VkSwapchainPresentModeInfoEXT', sType=sType,pNext=pNext,swapchainCount=swapchainCount,pPresentModes=pPresentModes)

def VkSwapchainPresentScalingCreateInfoEXT(sType=VK_STRUCTURE_TYPE_SWAPCHAIN_PRESENT_SCALING_CREATE_INFO_EXT,pNext=None,scalingBehavior=None,presentGravityX=None,presentGravityY=None,):

    return _new('VkSwapchainPresentScalingCreateInfoEXT', sType=sType,pNext=pNext,scalingBehavior=scalingBehavior,presentGravityX=presentGravityX,presentGravityY=presentGravityY)

def VkReleaseSwapchainImagesInfoEXT(sType=VK_STRUCTURE_TYPE_RELEASE_SWAPCHAIN_IMAGES_INFO_EXT,pNext=None,swapchain=None,imageIndexCount=None,pImageIndices=None,):
    if imageIndexCount is None and pImageIndices is not None:
        imageIndexCount = len(pImageIndices)

    return _new('VkReleaseSwapchainImagesInfoEXT', sType=sType,pNext=pNext,swapchain=swapchain,imageIndexCount=imageIndexCount,pImageIndices=pImageIndices)

def VkPhysicalDeviceDepthBiasControlFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_BIAS_CONTROL_FEATURES_EXT,pNext=None,depthBiasControl=None,leastRepresentableValueForceUnormRepresentation=None,floatRepresentation=None,depthBiasExact=None,):

    return _new('VkPhysicalDeviceDepthBiasControlFeaturesEXT', sType=sType,pNext=pNext,depthBiasControl=depthBiasControl,leastRepresentableValueForceUnormRepresentation=leastRepresentableValueForceUnormRepresentation,floatRepresentation=floatRepresentation,depthBiasExact=depthBiasExact)

def VkPhysicalDeviceRayTracingInvocationReorderFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_INVOCATION_REORDER_FEATURES_NV,pNext=None,rayTracingInvocationReorder=None,):

    return _new('VkPhysicalDeviceRayTracingInvocationReorderFeaturesNV', sType=sType,pNext=pNext,rayTracingInvocationReorder=rayTracingInvocationReorder)

def VkPhysicalDeviceRayTracingInvocationReorderPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_INVOCATION_REORDER_PROPERTIES_NV,pNext=None,rayTracingInvocationReorderReorderingHint=None,):

    return _new('VkPhysicalDeviceRayTracingInvocationReorderPropertiesNV', sType=sType,pNext=pNext,rayTracingInvocationReorderReorderingHint=rayTracingInvocationReorderReorderingHint)

def VkPhysicalDeviceExtendedSparseAddressSpaceFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_SPARSE_ADDRESS_SPACE_FEATURES_NV,pNext=None,extendedSparseAddressSpace=None,):

    return _new('VkPhysicalDeviceExtendedSparseAddressSpaceFeaturesNV', sType=sType,pNext=pNext,extendedSparseAddressSpace=extendedSparseAddressSpace)

def VkPhysicalDeviceExtendedSparseAddressSpacePropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_SPARSE_ADDRESS_SPACE_PROPERTIES_NV,pNext=None,extendedSparseAddressSpaceSize=None,extendedSparseImageUsageFlags=None,extendedSparseBufferUsageFlags=None,):

    return _new('VkPhysicalDeviceExtendedSparseAddressSpacePropertiesNV', sType=sType,pNext=pNext,extendedSparseAddressSpaceSize=extendedSparseAddressSpaceSize,extendedSparseImageUsageFlags=extendedSparseImageUsageFlags,extendedSparseBufferUsageFlags=extendedSparseBufferUsageFlags)

def VkDirectDriverLoadingInfoLUNARG(sType=VK_STRUCTURE_TYPE_DIRECT_DRIVER_LOADING_INFO_LUNARG,pNext=None,flags=None,pfnGetInstanceProcAddr=None,):

    return _new('VkDirectDriverLoadingInfoLUNARG', sType=sType,pNext=pNext,flags=flags,pfnGetInstanceProcAddr=pfnGetInstanceProcAddr)

def VkDirectDriverLoadingListLUNARG(sType=VK_STRUCTURE_TYPE_DIRECT_DRIVER_LOADING_LIST_LUNARG,pNext=None,mode=None,driverCount=None,pDrivers=None,):
    if driverCount is None and pDrivers is not None:
        driverCount = len(pDrivers)

    return _new('VkDirectDriverLoadingListLUNARG', sType=sType,pNext=pNext,mode=mode,driverCount=driverCount,pDrivers=pDrivers)

def VkPhysicalDeviceMultiviewPerViewViewportsFeaturesQCOM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PER_VIEW_VIEWPORTS_FEATURES_QCOM,pNext=None,multiviewPerViewViewports=None,):

    return _new('VkPhysicalDeviceMultiviewPerViewViewportsFeaturesQCOM', sType=sType,pNext=pNext,multiviewPerViewViewports=multiviewPerViewViewports)

def VkPhysicalDeviceRayTracingPositionFetchFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_POSITION_FETCH_FEATURES_KHR,pNext=None,rayTracingPositionFetch=None,):

    return _new('VkPhysicalDeviceRayTracingPositionFetchFeaturesKHR', sType=sType,pNext=pNext,rayTracingPositionFetch=rayTracingPositionFetch)

def VkDeviceImageSubresourceInfoKHR(sType=VK_STRUCTURE_TYPE_DEVICE_IMAGE_SUBRESOURCE_INFO_KHR,pNext=None,pCreateInfo=None,pSubresource=None,):

    return _new('VkDeviceImageSubresourceInfoKHR', sType=sType,pNext=pNext,pCreateInfo=pCreateInfo,pSubresource=pSubresource)

def VkPhysicalDeviceShaderCorePropertiesARM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_PROPERTIES_ARM,pNext=None,pixelRate=None,texelRate=None,fmaRate=None,):

    return _new('VkPhysicalDeviceShaderCorePropertiesARM', sType=sType,pNext=pNext,pixelRate=pixelRate,texelRate=texelRate,fmaRate=fmaRate)

def VkPhysicalDeviceMultiviewPerViewRenderAreasFeaturesQCOM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PER_VIEW_RENDER_AREAS_FEATURES_QCOM,pNext=None,multiviewPerViewRenderAreas=None,):

    return _new('VkPhysicalDeviceMultiviewPerViewRenderAreasFeaturesQCOM', sType=sType,pNext=pNext,multiviewPerViewRenderAreas=multiviewPerViewRenderAreas)

def VkMultiviewPerViewRenderAreasRenderPassBeginInfoQCOM(sType=VK_STRUCTURE_TYPE_MULTIVIEW_PER_VIEW_RENDER_AREAS_RENDER_PASS_BEGIN_INFO_QCOM,pNext=None,perViewRenderAreaCount=None,pPerViewRenderAreas=None,):
    if perViewRenderAreaCount is None and pPerViewRenderAreas is not None:
        perViewRenderAreaCount = len(pPerViewRenderAreas)

    return _new('VkMultiviewPerViewRenderAreasRenderPassBeginInfoQCOM', sType=sType,pNext=pNext,perViewRenderAreaCount=perViewRenderAreaCount,pPerViewRenderAreas=pPerViewRenderAreas)

def VkQueryLowLatencySupportNV(sType=VK_STRUCTURE_TYPE_QUERY_LOW_LATENCY_SUPPORT_NV,pNext=None,pQueriedLowLatencyData=None,):

    return _new('VkQueryLowLatencySupportNV', sType=sType,pNext=pNext,pQueriedLowLatencyData=pQueriedLowLatencyData)

def VkMemoryMapInfoKHR(sType=VK_STRUCTURE_TYPE_MEMORY_MAP_INFO_KHR,pNext=None,flags=None,memory=None,offset=None,size=None,):

    return _new('VkMemoryMapInfoKHR', sType=sType,pNext=pNext,flags=flags,memory=memory,offset=offset,size=size)

def VkMemoryUnmapInfoKHR(sType=VK_STRUCTURE_TYPE_MEMORY_UNMAP_INFO_KHR,pNext=None,flags=None,memory=None,):

    return _new('VkMemoryUnmapInfoKHR', sType=sType,pNext=pNext,flags=flags,memory=memory)

def VkPhysicalDeviceShaderObjectFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_OBJECT_FEATURES_EXT,pNext=None,shaderObject=None,):

    return _new('VkPhysicalDeviceShaderObjectFeaturesEXT', sType=sType,pNext=pNext,shaderObject=shaderObject)

def VkPhysicalDeviceShaderObjectPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_OBJECT_PROPERTIES_EXT,pNext=None,shaderBinaryUUID=None,shaderBinaryVersion=None,):

    return _new('VkPhysicalDeviceShaderObjectPropertiesEXT', sType=sType,pNext=pNext,shaderBinaryUUID=shaderBinaryUUID,shaderBinaryVersion=shaderBinaryVersion)

def VkShaderCreateInfoEXT(sType=VK_STRUCTURE_TYPE_SHADER_CREATE_INFO_EXT,pNext=None,flags=None,stage=None,nextStage=None,codeType=None,codeSize=None,pCode=None,pName=None,setLayoutCount=None,pSetLayouts=None,pushConstantRangeCount=None,pPushConstantRanges=None,pSpecializationInfo=None,):
    if codeSize is None and pCode is not None:
        codeSize = len(pCode)
    if setLayoutCount is None and pSetLayouts is not None:
        setLayoutCount = len(pSetLayouts)
    if pushConstantRangeCount is None and pPushConstantRanges is not None:
        pushConstantRangeCount = len(pPushConstantRanges)

    return _new('VkShaderCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,stage=stage,nextStage=nextStage,codeType=codeType,codeSize=codeSize,pCode=pCode,pName=pName,setLayoutCount=setLayoutCount,pSetLayouts=pSetLayouts,pushConstantRangeCount=pushConstantRangeCount,pPushConstantRanges=pPushConstantRanges,pSpecializationInfo=pSpecializationInfo)

def VkPhysicalDeviceShaderTileImageFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_TILE_IMAGE_FEATURES_EXT,pNext=None,shaderTileImageColorReadAccess=None,shaderTileImageDepthReadAccess=None,shaderTileImageStencilReadAccess=None,):

    return _new('VkPhysicalDeviceShaderTileImageFeaturesEXT', sType=sType,pNext=pNext,shaderTileImageColorReadAccess=shaderTileImageColorReadAccess,shaderTileImageDepthReadAccess=shaderTileImageDepthReadAccess,shaderTileImageStencilReadAccess=shaderTileImageStencilReadAccess)

def VkPhysicalDeviceShaderTileImagePropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_TILE_IMAGE_PROPERTIES_EXT,pNext=None,shaderTileImageCoherentReadAccelerated=None,shaderTileImageReadSampleFromPixelRateInvocation=None,shaderTileImageReadFromHelperInvocation=None,):

    return _new('VkPhysicalDeviceShaderTileImagePropertiesEXT', sType=sType,pNext=pNext,shaderTileImageCoherentReadAccelerated=shaderTileImageCoherentReadAccelerated,shaderTileImageReadSampleFromPixelRateInvocation=shaderTileImageReadSampleFromPixelRateInvocation,shaderTileImageReadFromHelperInvocation=shaderTileImageReadFromHelperInvocation)

def VkImportScreenBufferInfoQNX(sType=VK_STRUCTURE_TYPE_IMPORT_SCREEN_BUFFER_INFO_QNX,pNext=None,buffer=None,):

    return _new('VkImportScreenBufferInfoQNX', sType=sType,pNext=pNext,buffer=buffer)

def VkScreenBufferPropertiesQNX(sType=VK_STRUCTURE_TYPE_SCREEN_BUFFER_PROPERTIES_QNX,pNext=None,allocationSize=None,memoryTypeBits=None,):

    return _new('VkScreenBufferPropertiesQNX', sType=sType,pNext=pNext,allocationSize=allocationSize,memoryTypeBits=memoryTypeBits)

def VkScreenBufferFormatPropertiesQNX(sType=VK_STRUCTURE_TYPE_SCREEN_BUFFER_FORMAT_PROPERTIES_QNX,pNext=None,format=None,externalFormat=None,screenUsage=None,formatFeatures=None,samplerYcbcrConversionComponents=None,suggestedYcbcrModel=None,suggestedYcbcrRange=None,suggestedXChromaOffset=None,suggestedYChromaOffset=None,):

    return _new('VkScreenBufferFormatPropertiesQNX', sType=sType,pNext=pNext,format=format,externalFormat=externalFormat,screenUsage=screenUsage,formatFeatures=formatFeatures,samplerYcbcrConversionComponents=samplerYcbcrConversionComponents,suggestedYcbcrModel=suggestedYcbcrModel,suggestedYcbcrRange=suggestedYcbcrRange,suggestedXChromaOffset=suggestedXChromaOffset,suggestedYChromaOffset=suggestedYChromaOffset)

def VkExternalFormatQNX(sType=VK_STRUCTURE_TYPE_EXTERNAL_FORMAT_QNX,pNext=None,externalFormat=None,):

    return _new('VkExternalFormatQNX', sType=sType,pNext=pNext,externalFormat=externalFormat)

def VkPhysicalDeviceExternalMemoryScreenBufferFeaturesQNX(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_SCREEN_BUFFER_FEATURES_QNX,pNext=None,screenBufferImport=None,):

    return _new('VkPhysicalDeviceExternalMemoryScreenBufferFeaturesQNX', sType=sType,pNext=pNext,screenBufferImport=screenBufferImport)

def VkPhysicalDeviceCooperativeMatrixFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COOPERATIVE_MATRIX_FEATURES_KHR,pNext=None,cooperativeMatrix=None,cooperativeMatrixRobustBufferAccess=None,):

    return _new('VkPhysicalDeviceCooperativeMatrixFeaturesKHR', sType=sType,pNext=pNext,cooperativeMatrix=cooperativeMatrix,cooperativeMatrixRobustBufferAccess=cooperativeMatrixRobustBufferAccess)

def VkCooperativeMatrixPropertiesKHR(sType=VK_STRUCTURE_TYPE_COOPERATIVE_MATRIX_PROPERTIES_KHR,pNext=None,MSize=None,NSize=None,KSize=None,AType=None,BType=None,CType=None,ResultType=None,saturatingAccumulation=None,scope=None,):

    return _new('VkCooperativeMatrixPropertiesKHR', sType=sType,pNext=pNext,MSize=MSize,NSize=NSize,KSize=KSize,AType=AType,BType=BType,CType=CType,ResultType=ResultType,saturatingAccumulation=saturatingAccumulation,scope=scope)

def VkPhysicalDeviceCooperativeMatrixPropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COOPERATIVE_MATRIX_PROPERTIES_KHR,pNext=None,cooperativeMatrixSupportedStages=None,):

    return _new('VkPhysicalDeviceCooperativeMatrixPropertiesKHR', sType=sType,pNext=pNext,cooperativeMatrixSupportedStages=cooperativeMatrixSupportedStages)

def VkPhysicalDeviceShaderEnqueuePropertiesAMDX(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ENQUEUE_PROPERTIES_AMDX,pNext=None,maxExecutionGraphDepth=None,maxExecutionGraphShaderOutputNodes=None,maxExecutionGraphShaderPayloadSize=None,maxExecutionGraphShaderPayloadCount=None,executionGraphDispatchAddressAlignment=None,):

    return _new('VkPhysicalDeviceShaderEnqueuePropertiesAMDX', sType=sType,pNext=pNext,maxExecutionGraphDepth=maxExecutionGraphDepth,maxExecutionGraphShaderOutputNodes=maxExecutionGraphShaderOutputNodes,maxExecutionGraphShaderPayloadSize=maxExecutionGraphShaderPayloadSize,maxExecutionGraphShaderPayloadCount=maxExecutionGraphShaderPayloadCount,executionGraphDispatchAddressAlignment=executionGraphDispatchAddressAlignment)

def VkPhysicalDeviceShaderEnqueueFeaturesAMDX(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ENQUEUE_FEATURES_AMDX,pNext=None,shaderEnqueue=None,):

    return _new('VkPhysicalDeviceShaderEnqueueFeaturesAMDX', sType=sType,pNext=pNext,shaderEnqueue=shaderEnqueue)

def VkExecutionGraphPipelineCreateInfoAMDX(sType=VK_STRUCTURE_TYPE_EXECUTION_GRAPH_PIPELINE_CREATE_INFO_AMDX,pNext=None,flags=None,stageCount=None,pStages=None,pLibraryInfo=None,layout=None,basePipelineHandle=None,basePipelineIndex=None,):
    if stageCount is None and pStages is not None:
        stageCount = len(pStages)

    return _new('VkExecutionGraphPipelineCreateInfoAMDX', sType=sType,pNext=pNext,flags=flags,stageCount=stageCount,pStages=pStages,pLibraryInfo=pLibraryInfo,layout=layout,basePipelineHandle=basePipelineHandle,basePipelineIndex=basePipelineIndex)

def VkPipelineShaderStageNodeCreateInfoAMDX(sType=VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_NODE_CREATE_INFO_AMDX,pNext=None,pName=None,index=None,):

    return _new('VkPipelineShaderStageNodeCreateInfoAMDX', sType=sType,pNext=pNext,pName=pName,index=index)

def VkExecutionGraphPipelineScratchSizeAMDX(sType=VK_STRUCTURE_TYPE_EXECUTION_GRAPH_PIPELINE_SCRATCH_SIZE_AMDX,pNext=None,size=None,):

    return _new('VkExecutionGraphPipelineScratchSizeAMDX', sType=sType,pNext=pNext,size=size)

def VkDispatchGraphInfoAMDX(nodeIndex=None,payloadCount=None,payloads=None,payloadStride=None,):

    return _new('VkDispatchGraphInfoAMDX', nodeIndex=nodeIndex,payloadCount=payloadCount,payloads=payloads,payloadStride=payloadStride)

def VkDispatchGraphCountInfoAMDX(count=None,infos=None,stride=None,):

    return _new('VkDispatchGraphCountInfoAMDX', count=count,infos=infos,stride=stride)

def VkBindMemoryStatusKHR(sType=VK_STRUCTURE_TYPE_BIND_MEMORY_STATUS_KHR,pNext=None,pResult=None,):

    return _new('VkBindMemoryStatusKHR', sType=sType,pNext=pNext,pResult=pResult)

def VkBindDescriptorSetsInfoKHR(sType=VK_STRUCTURE_TYPE_BIND_DESCRIPTOR_SETS_INFO_KHR,pNext=None,stageFlags=None,layout=None,firstSet=None,descriptorSetCount=None,pDescriptorSets=None,dynamicOffsetCount=None,pDynamicOffsets=None,):
    if descriptorSetCount is None and pDescriptorSets is not None:
        descriptorSetCount = len(pDescriptorSets)
    if dynamicOffsetCount is None and pDynamicOffsets is not None:
        dynamicOffsetCount = len(pDynamicOffsets)

    return _new('VkBindDescriptorSetsInfoKHR', sType=sType,pNext=pNext,stageFlags=stageFlags,layout=layout,firstSet=firstSet,descriptorSetCount=descriptorSetCount,pDescriptorSets=pDescriptorSets,dynamicOffsetCount=dynamicOffsetCount,pDynamicOffsets=pDynamicOffsets)

def VkPushConstantsInfoKHR(sType=VK_STRUCTURE_TYPE_PUSH_CONSTANTS_INFO_KHR,pNext=None,layout=None,stageFlags=None,offset=None,size=None,pValues=None,):
    if size is None and pValues is not None:
        size = len(pValues)

    return _new('VkPushConstantsInfoKHR', sType=sType,pNext=pNext,layout=layout,stageFlags=stageFlags,offset=offset,size=size,pValues=pValues)

def VkPushDescriptorSetInfoKHR(sType=VK_STRUCTURE_TYPE_PUSH_DESCRIPTOR_SET_INFO_KHR,pNext=None,stageFlags=None,layout=None,set=None,descriptorWriteCount=None,pDescriptorWrites=None,):
    if descriptorWriteCount is None and pDescriptorWrites is not None:
        descriptorWriteCount = len(pDescriptorWrites)

    return _new('VkPushDescriptorSetInfoKHR', sType=sType,pNext=pNext,stageFlags=stageFlags,layout=layout,set=set,descriptorWriteCount=descriptorWriteCount,pDescriptorWrites=pDescriptorWrites)

def VkPushDescriptorSetWithTemplateInfoKHR(sType=VK_STRUCTURE_TYPE_PUSH_DESCRIPTOR_SET_WITH_TEMPLATE_INFO_KHR,pNext=None,descriptorUpdateTemplate=None,layout=None,set=None,pData=None,):

    return _new('VkPushDescriptorSetWithTemplateInfoKHR', sType=sType,pNext=pNext,descriptorUpdateTemplate=descriptorUpdateTemplate,layout=layout,set=set,pData=pData)

def VkSetDescriptorBufferOffsetsInfoEXT(sType=VK_STRUCTURE_TYPE_SET_DESCRIPTOR_BUFFER_OFFSETS_INFO_EXT,pNext=None,stageFlags=None,layout=None,firstSet=None,setCount=None,pBufferIndices=None,pOffsets=None,):
    if setCount is None and pBufferIndices is not None:
        setCount = len(pBufferIndices)
    if setCount is None and pOffsets is not None:
        setCount = len(pOffsets)

    return _new('VkSetDescriptorBufferOffsetsInfoEXT', sType=sType,pNext=pNext,stageFlags=stageFlags,layout=layout,firstSet=firstSet,setCount=setCount,pBufferIndices=pBufferIndices,pOffsets=pOffsets)

def VkBindDescriptorBufferEmbeddedSamplersInfoEXT(sType=VK_STRUCTURE_TYPE_BIND_DESCRIPTOR_BUFFER_EMBEDDED_SAMPLERS_INFO_EXT,pNext=None,stageFlags=None,layout=None,set=None,):

    return _new('VkBindDescriptorBufferEmbeddedSamplersInfoEXT', sType=sType,pNext=pNext,stageFlags=stageFlags,layout=layout,set=set)

def VkPhysicalDeviceCubicClampFeaturesQCOM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUBIC_CLAMP_FEATURES_QCOM,pNext=None,cubicRangeClamp=None,):

    return _new('VkPhysicalDeviceCubicClampFeaturesQCOM', sType=sType,pNext=pNext,cubicRangeClamp=cubicRangeClamp)

def VkPhysicalDeviceYcbcrDegammaFeaturesQCOM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_YCBCR_DEGAMMA_FEATURES_QCOM,pNext=None,ycbcrDegamma=None,):

    return _new('VkPhysicalDeviceYcbcrDegammaFeaturesQCOM', sType=sType,pNext=pNext,ycbcrDegamma=ycbcrDegamma)

def VkSamplerYcbcrConversionYcbcrDegammaCreateInfoQCOM(sType=VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_YCBCR_DEGAMMA_CREATE_INFO_QCOM,pNext=None,enableYDegamma=None,enableCbCrDegamma=None,):

    return _new('VkSamplerYcbcrConversionYcbcrDegammaCreateInfoQCOM', sType=sType,pNext=pNext,enableYDegamma=enableYDegamma,enableCbCrDegamma=enableCbCrDegamma)

def VkPhysicalDeviceCubicWeightsFeaturesQCOM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUBIC_WEIGHTS_FEATURES_QCOM,pNext=None,selectableCubicWeights=None,):

    return _new('VkPhysicalDeviceCubicWeightsFeaturesQCOM', sType=sType,pNext=pNext,selectableCubicWeights=selectableCubicWeights)

def VkSamplerCubicWeightsCreateInfoQCOM(sType=VK_STRUCTURE_TYPE_SAMPLER_CUBIC_WEIGHTS_CREATE_INFO_QCOM,pNext=None,cubicWeights=None,):

    return _new('VkSamplerCubicWeightsCreateInfoQCOM', sType=sType,pNext=pNext,cubicWeights=cubicWeights)

def VkBlitImageCubicWeightsInfoQCOM(sType=VK_STRUCTURE_TYPE_BLIT_IMAGE_CUBIC_WEIGHTS_INFO_QCOM,pNext=None,cubicWeights=None,):

    return _new('VkBlitImageCubicWeightsInfoQCOM', sType=sType,pNext=pNext,cubicWeights=cubicWeights)

def VkPhysicalDeviceImageProcessing2FeaturesQCOM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_PROCESSING_2_FEATURES_QCOM,pNext=None,textureBlockMatch2=None,):

    return _new('VkPhysicalDeviceImageProcessing2FeaturesQCOM', sType=sType,pNext=pNext,textureBlockMatch2=textureBlockMatch2)

def VkPhysicalDeviceImageProcessing2PropertiesQCOM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_PROCESSING_2_PROPERTIES_QCOM,pNext=None,maxBlockMatchWindow=None,):

    return _new('VkPhysicalDeviceImageProcessing2PropertiesQCOM', sType=sType,pNext=pNext,maxBlockMatchWindow=maxBlockMatchWindow)

def VkSamplerBlockMatchWindowCreateInfoQCOM(sType=VK_STRUCTURE_TYPE_SAMPLER_BLOCK_MATCH_WINDOW_CREATE_INFO_QCOM,pNext=None,windowExtent=None,windowCompareMode=None,):

    return _new('VkSamplerBlockMatchWindowCreateInfoQCOM', sType=sType,pNext=pNext,windowExtent=windowExtent,windowCompareMode=windowCompareMode)

def VkPhysicalDeviceDescriptorPoolOverallocationFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_POOL_OVERALLOCATION_FEATURES_NV,pNext=None,descriptorPoolOverallocation=None,):

    return _new('VkPhysicalDeviceDescriptorPoolOverallocationFeaturesNV', sType=sType,pNext=pNext,descriptorPoolOverallocation=descriptorPoolOverallocation)

def VkPhysicalDeviceLayeredDriverPropertiesMSFT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LAYERED_DRIVER_PROPERTIES_MSFT,pNext=None,underlyingAPI=None,):

    return _new('VkPhysicalDeviceLayeredDriverPropertiesMSFT', sType=sType,pNext=pNext,underlyingAPI=underlyingAPI)

def VkPhysicalDevicePerStageDescriptorSetFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PER_STAGE_DESCRIPTOR_SET_FEATURES_NV,pNext=None,perStageDescriptorSet=None,dynamicPipelineLayout=None,):

    return _new('VkPhysicalDevicePerStageDescriptorSetFeaturesNV', sType=sType,pNext=pNext,perStageDescriptorSet=perStageDescriptorSet,dynamicPipelineLayout=dynamicPipelineLayout)

def VkPhysicalDeviceExternalFormatResolveFeaturesANDROID(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FORMAT_RESOLVE_FEATURES_ANDROID,pNext=None,externalFormatResolve=None,):

    return _new('VkPhysicalDeviceExternalFormatResolveFeaturesANDROID', sType=sType,pNext=pNext,externalFormatResolve=externalFormatResolve)

def VkPhysicalDeviceExternalFormatResolvePropertiesANDROID(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FORMAT_RESOLVE_PROPERTIES_ANDROID,pNext=None,nullColorAttachmentWithExternalFormatResolve=None,externalFormatResolveChromaOffsetX=None,externalFormatResolveChromaOffsetY=None,):

    return _new('VkPhysicalDeviceExternalFormatResolvePropertiesANDROID', sType=sType,pNext=pNext,nullColorAttachmentWithExternalFormatResolve=nullColorAttachmentWithExternalFormatResolve,externalFormatResolveChromaOffsetX=externalFormatResolveChromaOffsetX,externalFormatResolveChromaOffsetY=externalFormatResolveChromaOffsetY)

def VkAndroidHardwareBufferFormatResolvePropertiesANDROID(sType=VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_FORMAT_RESOLVE_PROPERTIES_ANDROID,pNext=None,colorAttachmentFormat=None,):

    return _new('VkAndroidHardwareBufferFormatResolvePropertiesANDROID', sType=sType,pNext=pNext,colorAttachmentFormat=colorAttachmentFormat)

def VkLatencySleepModeInfoNV(sType=VK_STRUCTURE_TYPE_LATENCY_SLEEP_MODE_INFO_NV,pNext=None,lowLatencyMode=None,lowLatencyBoost=None,minimumIntervalUs=None,):

    return _new('VkLatencySleepModeInfoNV', sType=sType,pNext=pNext,lowLatencyMode=lowLatencyMode,lowLatencyBoost=lowLatencyBoost,minimumIntervalUs=minimumIntervalUs)

def VkLatencySleepInfoNV(sType=VK_STRUCTURE_TYPE_LATENCY_SLEEP_INFO_NV,pNext=None,signalSemaphore=None,value=None,):

    return _new('VkLatencySleepInfoNV', sType=sType,pNext=pNext,signalSemaphore=signalSemaphore,value=value)

def VkSetLatencyMarkerInfoNV(sType=VK_STRUCTURE_TYPE_SET_LATENCY_MARKER_INFO_NV,pNext=None,presentID=None,marker=None,):

    return _new('VkSetLatencyMarkerInfoNV', sType=sType,pNext=pNext,presentID=presentID,marker=marker)

def VkGetLatencyMarkerInfoNV(sType=VK_STRUCTURE_TYPE_GET_LATENCY_MARKER_INFO_NV,pNext=None,timingCount=None,pTimings=None,):
    if timingCount is None and pTimings is not None:
        timingCount = len(pTimings)

    return _new('VkGetLatencyMarkerInfoNV', sType=sType,pNext=pNext,timingCount=timingCount,pTimings=pTimings)

def VkLatencyTimingsFrameReportNV(sType=VK_STRUCTURE_TYPE_LATENCY_TIMINGS_FRAME_REPORT_NV,pNext=None,presentID=None,inputSampleTimeUs=None,simStartTimeUs=None,simEndTimeUs=None,renderSubmitStartTimeUs=None,renderSubmitEndTimeUs=None,presentStartTimeUs=None,presentEndTimeUs=None,driverStartTimeUs=None,driverEndTimeUs=None,osRenderQueueStartTimeUs=None,osRenderQueueEndTimeUs=None,gpuRenderStartTimeUs=None,gpuRenderEndTimeUs=None,):

    return _new('VkLatencyTimingsFrameReportNV', sType=sType,pNext=pNext,presentID=presentID,inputSampleTimeUs=inputSampleTimeUs,simStartTimeUs=simStartTimeUs,simEndTimeUs=simEndTimeUs,renderSubmitStartTimeUs=renderSubmitStartTimeUs,renderSubmitEndTimeUs=renderSubmitEndTimeUs,presentStartTimeUs=presentStartTimeUs,presentEndTimeUs=presentEndTimeUs,driverStartTimeUs=driverStartTimeUs,driverEndTimeUs=driverEndTimeUs,osRenderQueueStartTimeUs=osRenderQueueStartTimeUs,osRenderQueueEndTimeUs=osRenderQueueEndTimeUs,gpuRenderStartTimeUs=gpuRenderStartTimeUs,gpuRenderEndTimeUs=gpuRenderEndTimeUs)

def VkOutOfBandQueueTypeInfoNV(sType=VK_STRUCTURE_TYPE_OUT_OF_BAND_QUEUE_TYPE_INFO_NV,pNext=None,queueType=None,):

    return _new('VkOutOfBandQueueTypeInfoNV', sType=sType,pNext=pNext,queueType=queueType)

def VkLatencySubmissionPresentIdNV(sType=VK_STRUCTURE_TYPE_LATENCY_SUBMISSION_PRESENT_ID_NV,pNext=None,presentID=None,):

    return _new('VkLatencySubmissionPresentIdNV', sType=sType,pNext=pNext,presentID=presentID)

def VkSwapchainLatencyCreateInfoNV(sType=VK_STRUCTURE_TYPE_SWAPCHAIN_LATENCY_CREATE_INFO_NV,pNext=None,latencyModeEnable=None,):

    return _new('VkSwapchainLatencyCreateInfoNV', sType=sType,pNext=pNext,latencyModeEnable=latencyModeEnable)

def VkLatencySurfaceCapabilitiesNV(sType=VK_STRUCTURE_TYPE_LATENCY_SURFACE_CAPABILITIES_NV,pNext=None,presentModeCount=None,pPresentModes=None,):
    if presentModeCount is None and pPresentModes is not None:
        presentModeCount = len(pPresentModes)

    return _new('VkLatencySurfaceCapabilitiesNV', sType=sType,pNext=pNext,presentModeCount=presentModeCount,pPresentModes=pPresentModes)

def VkPhysicalDeviceCudaKernelLaunchFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUDA_KERNEL_LAUNCH_FEATURES_NV,pNext=None,cudaKernelLaunchFeatures=None,):

    return _new('VkPhysicalDeviceCudaKernelLaunchFeaturesNV', sType=sType,pNext=pNext,cudaKernelLaunchFeatures=cudaKernelLaunchFeatures)

def VkPhysicalDeviceCudaKernelLaunchPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUDA_KERNEL_LAUNCH_PROPERTIES_NV,pNext=None,computeCapabilityMinor=None,computeCapabilityMajor=None,):

    return _new('VkPhysicalDeviceCudaKernelLaunchPropertiesNV', sType=sType,pNext=pNext,computeCapabilityMinor=computeCapabilityMinor,computeCapabilityMajor=computeCapabilityMajor)

def VkDeviceQueueShaderCoreControlCreateInfoARM(sType=VK_STRUCTURE_TYPE_DEVICE_QUEUE_SHADER_CORE_CONTROL_CREATE_INFO_ARM,pNext=None,shaderCoreCount=None,):

    return _new('VkDeviceQueueShaderCoreControlCreateInfoARM', sType=sType,pNext=pNext,shaderCoreCount=shaderCoreCount)

def VkPhysicalDeviceSchedulingControlsFeaturesARM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SCHEDULING_CONTROLS_FEATURES_ARM,pNext=None,schedulingControls=None,):

    return _new('VkPhysicalDeviceSchedulingControlsFeaturesARM', sType=sType,pNext=pNext,schedulingControls=schedulingControls)

def VkPhysicalDeviceSchedulingControlsPropertiesARM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SCHEDULING_CONTROLS_PROPERTIES_ARM,pNext=None,schedulingControlsFlags=None,):

    return _new('VkPhysicalDeviceSchedulingControlsPropertiesARM', sType=sType,pNext=pNext,schedulingControlsFlags=schedulingControlsFlags)

def VkPhysicalDeviceRelaxedLineRasterizationFeaturesIMG(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RELAXED_LINE_RASTERIZATION_FEATURES_IMG,pNext=None,relaxedLineRasterization=None,):

    return _new('VkPhysicalDeviceRelaxedLineRasterizationFeaturesIMG', sType=sType,pNext=pNext,relaxedLineRasterization=relaxedLineRasterization)

def VkPhysicalDeviceRenderPassStripedFeaturesARM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RENDER_PASS_STRIPED_FEATURES_ARM,pNext=None,renderPassStriped=None,):

    return _new('VkPhysicalDeviceRenderPassStripedFeaturesARM', sType=sType,pNext=pNext,renderPassStriped=renderPassStriped)

def VkPhysicalDeviceRenderPassStripedPropertiesARM(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RENDER_PASS_STRIPED_PROPERTIES_ARM,pNext=None,renderPassStripeGranularity=None,maxRenderPassStripes=None,):

    return _new('VkPhysicalDeviceRenderPassStripedPropertiesARM', sType=sType,pNext=pNext,renderPassStripeGranularity=renderPassStripeGranularity,maxRenderPassStripes=maxRenderPassStripes)

def VkRenderPassStripeInfoARM(sType=VK_STRUCTURE_TYPE_RENDER_PASS_STRIPE_INFO_ARM,pNext=None,stripeArea=None,):

    return _new('VkRenderPassStripeInfoARM', sType=sType,pNext=pNext,stripeArea=stripeArea)

def VkRenderPassStripeBeginInfoARM(sType=VK_STRUCTURE_TYPE_RENDER_PASS_STRIPE_BEGIN_INFO_ARM,pNext=None,stripeInfoCount=None,pStripeInfos=None,):
    if stripeInfoCount is None and pStripeInfos is not None:
        stripeInfoCount = len(pStripeInfos)

    return _new('VkRenderPassStripeBeginInfoARM', sType=sType,pNext=pNext,stripeInfoCount=stripeInfoCount,pStripeInfos=pStripeInfos)

def VkRenderPassStripeSubmitInfoARM(sType=VK_STRUCTURE_TYPE_RENDER_PASS_STRIPE_SUBMIT_INFO_ARM,pNext=None,stripeSemaphoreInfoCount=None,pStripeSemaphoreInfos=None,):
    if stripeSemaphoreInfoCount is None and pStripeSemaphoreInfos is not None:
        stripeSemaphoreInfoCount = len(pStripeSemaphoreInfos)

    return _new('VkRenderPassStripeSubmitInfoARM', sType=sType,pNext=pNext,stripeSemaphoreInfoCount=stripeSemaphoreInfoCount,pStripeSemaphoreInfos=pStripeSemaphoreInfos)



def _auto_handle(x, _type):
    if x is None:
        return ffi.NULL
    if _type.kind == 'pointer':
        ptr, _ = _cast_ptr(x, _type)
        return ptr
    return x


def _callApi(fn, *args):
    fn_args = [_auto_handle(i, j) for i, j in zip(args, ffi.typeof(fn).args)]
    return fn(*fn_args)







def vkCreateInstance(
pCreateInfo
        ,pAllocator
        ,pInstance=None
        ,):

    custom_return = True
    if not pInstance:
        pInstance = ffi.new('VkInstance*')
        custom_return = False

    result = _callApi(lib.vkCreateInstance, pCreateInfo,pAllocator,pInstance)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pInstance

    return pInstance[0]



def vkDestroyInstance(
instance
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyInstance, instance,pAllocator)


def vkEnumeratePhysicalDevices(
instance
        ,):

    pPhysicalDeviceCount = ffi.new('uint32_t*')
    pPhysicalDevices = ffi.NULL

    result = _callApi(lib.vkEnumeratePhysicalDevices, instance,pPhysicalDeviceCount,pPhysicalDevices)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pPhysicalDevices = ffi.new('VkPhysicalDevice[]', pPhysicalDeviceCount[0])
    result = _callApi(lib.vkEnumeratePhysicalDevices, instance,pPhysicalDeviceCount,pPhysicalDevices)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    return pPhysicalDevices


def vkGetPhysicalDeviceProperties(
physicalDevice
        ,pProperties=None
        ,):

    custom_return = True
    if not pProperties:
        pProperties = ffi.new('VkPhysicalDeviceProperties*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceProperties, physicalDevice,pProperties)

    if custom_return:
        return pProperties

    return StrWrap(pProperties[0])



def vkGetPhysicalDeviceQueueFamilyProperties(
physicalDevice
        ,):

    pQueueFamilyPropertyCount = ffi.new('uint32_t*')
    pQueueFamilyProperties = ffi.NULL

    result = _callApi(lib.vkGetPhysicalDeviceQueueFamilyProperties, physicalDevice,pQueueFamilyPropertyCount,pQueueFamilyProperties)

    pQueueFamilyProperties = ffi.new('VkQueueFamilyProperties[]', pQueueFamilyPropertyCount[0])
    result = _callApi(lib.vkGetPhysicalDeviceQueueFamilyProperties, physicalDevice,pQueueFamilyPropertyCount,pQueueFamilyProperties)

    return pQueueFamilyProperties


def vkGetPhysicalDeviceMemoryProperties(
physicalDevice
        ,pMemoryProperties=None
        ,):

    custom_return = True
    if not pMemoryProperties:
        pMemoryProperties = ffi.new('VkPhysicalDeviceMemoryProperties*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceMemoryProperties, physicalDevice,pMemoryProperties)

    if custom_return:
        return pMemoryProperties

    return pMemoryProperties[0]



def vkGetPhysicalDeviceFeatures(
physicalDevice
        ,pFeatures=None
        ,):

    custom_return = True
    if not pFeatures:
        pFeatures = ffi.new('VkPhysicalDeviceFeatures*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceFeatures, physicalDevice,pFeatures)

    if custom_return:
        return pFeatures

    return pFeatures[0]



def vkGetPhysicalDeviceFormatProperties(
physicalDevice
        ,format
        ,pFormatProperties=None
        ,):

    custom_return = True
    if not pFormatProperties:
        pFormatProperties = ffi.new('VkFormatProperties*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceFormatProperties, physicalDevice,format,pFormatProperties)

    if custom_return:
        return pFormatProperties

    return pFormatProperties[0]



def vkGetPhysicalDeviceImageFormatProperties(
physicalDevice
        ,format
        ,type
        ,tiling
        ,usage
        ,flags
        ,pImageFormatProperties=None
        ,):

    custom_return = True
    if not pImageFormatProperties:
        pImageFormatProperties = ffi.new('VkImageFormatProperties*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceImageFormatProperties, physicalDevice,format,type,tiling,usage,flags,pImageFormatProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pImageFormatProperties

    return pImageFormatProperties[0]



def vkCreateDevice(
physicalDevice
        ,pCreateInfo
        ,pAllocator
        ,pDevice=None
        ,):

    custom_return = True
    if not pDevice:
        pDevice = ffi.new('VkDevice*')
        custom_return = False

    result = _callApi(lib.vkCreateDevice, physicalDevice,pCreateInfo,pAllocator,pDevice)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pDevice

    return pDevice[0]



def vkCreateDevice(
physicalDevice
        ,pCreateInfo
        ,pAllocator
        ,pDevice=None
        ,):

    custom_return = True
    if not pDevice:
        pDevice = ffi.new('VkDevice*')
        custom_return = False

    result = _callApi(lib.vkCreateDevice, physicalDevice,pCreateInfo,pAllocator,pDevice)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pDevice

    return pDevice[0]



def vkDestroyDevice(
device
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyDevice, device,pAllocator)


def vkEnumerateInstanceVersion(
pApiVersion=None
        ,):

    custom_return = True
    if not pApiVersion:
        pApiVersion = ffi.new('uint32_t*')
        custom_return = False

    result = _callApi(lib.vkEnumerateInstanceVersion, pApiVersion)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pApiVersion

    return pApiVersion[0]



def vkEnumerateInstanceLayerProperties(
):

    pPropertyCount = ffi.new('uint32_t*')
    pProperties = ffi.NULL

    result = _callApi(lib.vkEnumerateInstanceLayerProperties, pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pProperties = ffi.new('VkLayerProperties[]', pPropertyCount[0])
    result = _callApi(lib.vkEnumerateInstanceLayerProperties, pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    result = (StrWrap(x) for x in pProperties)
    _weakkey_dict[result] = pProperties
    return result


def vkEnumerateInstanceExtensionProperties(
pLayerName
        ,):

    pPropertyCount = ffi.new('uint32_t*')
    pProperties = ffi.NULL

    result = _callApi(lib.vkEnumerateInstanceExtensionProperties, pLayerName,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pProperties = ffi.new('VkExtensionProperties[]', pPropertyCount[0])
    result = _callApi(lib.vkEnumerateInstanceExtensionProperties, pLayerName,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    result = (StrWrap(x) for x in pProperties)
    _weakkey_dict[result] = pProperties
    return result


def vkEnumerateDeviceLayerProperties(
physicalDevice
        ,):

    pPropertyCount = ffi.new('uint32_t*')
    pProperties = ffi.NULL

    result = _callApi(lib.vkEnumerateDeviceLayerProperties, physicalDevice,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pProperties = ffi.new('VkLayerProperties[]', pPropertyCount[0])
    result = _callApi(lib.vkEnumerateDeviceLayerProperties, physicalDevice,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    result = (StrWrap(x) for x in pProperties)
    _weakkey_dict[result] = pProperties
    return result


def vkEnumerateDeviceLayerProperties(
physicalDevice
        ,):

    pPropertyCount = ffi.new('uint32_t*')
    pProperties = ffi.NULL

    result = _callApi(lib.vkEnumerateDeviceLayerProperties, physicalDevice,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pProperties = ffi.new('VkLayerProperties[]', pPropertyCount[0])
    result = _callApi(lib.vkEnumerateDeviceLayerProperties, physicalDevice,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    result = (StrWrap(x) for x in pProperties)
    _weakkey_dict[result] = pProperties
    return result


def vkEnumerateDeviceExtensionProperties(
physicalDevice
        ,pLayerName
        ,):

    pPropertyCount = ffi.new('uint32_t*')
    pProperties = ffi.NULL

    result = _callApi(lib.vkEnumerateDeviceExtensionProperties, physicalDevice,pLayerName,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pProperties = ffi.new('VkExtensionProperties[]', pPropertyCount[0])
    result = _callApi(lib.vkEnumerateDeviceExtensionProperties, physicalDevice,pLayerName,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    result = (StrWrap(x) for x in pProperties)
    _weakkey_dict[result] = pProperties
    return result


def vkGetDeviceQueue(
device
        ,queueFamilyIndex
        ,queueIndex
        ,pQueue=None
        ,):

    custom_return = True
    if not pQueue:
        pQueue = ffi.new('VkQueue*')
        custom_return = False

    result = _callApi(lib.vkGetDeviceQueue, device,queueFamilyIndex,queueIndex,pQueue)

    if custom_return:
        return pQueue

    return pQueue[0]



def vkQueueSubmit(
queue
        ,submitCount
        ,pSubmits
        ,fence
        ,):
    result = _callApi(lib.vkQueueSubmit, queue,submitCount,pSubmits,fence)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkQueueWaitIdle(
queue
        ,):
    result = _callApi(lib.vkQueueWaitIdle, queue)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkDeviceWaitIdle(
device
        ,):
    result = _callApi(lib.vkDeviceWaitIdle, device)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkAllocateMemory(
device
        ,pAllocateInfo
        ,pAllocator
        ,pMemory=None
        ,):

    custom_return = True
    if not pMemory:
        pMemory = ffi.new('VkDeviceMemory*')
        custom_return = False

    result = _callApi(lib.vkAllocateMemory, device,pAllocateInfo,pAllocator,pMemory)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pMemory

    return pMemory[0]



def vkFreeMemory(
device
        ,memory
        ,pAllocator
        ,):
    result = _callApi(lib.vkFreeMemory, device,memory,pAllocator)


def vkUnmapMemory(
device
        ,memory
        ,):
    result = _callApi(lib.vkUnmapMemory, device,memory)


def vkFlushMappedMemoryRanges(
device
        ,memoryRangeCount
        ,pMemoryRanges
        ,):
    result = _callApi(lib.vkFlushMappedMemoryRanges, device,memoryRangeCount,pMemoryRanges)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkInvalidateMappedMemoryRanges(
device
        ,memoryRangeCount
        ,pMemoryRanges
        ,):
    result = _callApi(lib.vkInvalidateMappedMemoryRanges, device,memoryRangeCount,pMemoryRanges)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkGetDeviceMemoryCommitment(
device
        ,memory
        ,pCommittedMemoryInBytes=None
        ,):

    custom_return = True
    if not pCommittedMemoryInBytes:
        pCommittedMemoryInBytes = ffi.new('VkDeviceSize*')
        custom_return = False

    result = _callApi(lib.vkGetDeviceMemoryCommitment, device,memory,pCommittedMemoryInBytes)

    if custom_return:
        return pCommittedMemoryInBytes

    return pCommittedMemoryInBytes[0]



def vkGetBufferMemoryRequirements(
device
        ,buffer
        ,pMemoryRequirements=None
        ,):

    custom_return = True
    if not pMemoryRequirements:
        pMemoryRequirements = ffi.new('VkMemoryRequirements*')
        custom_return = False

    result = _callApi(lib.vkGetBufferMemoryRequirements, device,buffer,pMemoryRequirements)

    if custom_return:
        return pMemoryRequirements

    return pMemoryRequirements[0]



def vkBindBufferMemory(
device
        ,buffer
        ,memory
        ,memoryOffset
        ,):
    result = _callApi(lib.vkBindBufferMemory, device,buffer,memory,memoryOffset)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkGetImageMemoryRequirements(
device
        ,image
        ,pMemoryRequirements=None
        ,):

    custom_return = True
    if not pMemoryRequirements:
        pMemoryRequirements = ffi.new('VkMemoryRequirements*')
        custom_return = False

    result = _callApi(lib.vkGetImageMemoryRequirements, device,image,pMemoryRequirements)

    if custom_return:
        return pMemoryRequirements

    return pMemoryRequirements[0]



def vkBindImageMemory(
device
        ,image
        ,memory
        ,memoryOffset
        ,):
    result = _callApi(lib.vkBindImageMemory, device,image,memory,memoryOffset)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkGetImageSparseMemoryRequirements(
device
        ,image
        ,):

    pSparseMemoryRequirementCount = ffi.new('uint32_t*')
    pSparseMemoryRequirements = ffi.NULL

    result = _callApi(lib.vkGetImageSparseMemoryRequirements, device,image,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

    pSparseMemoryRequirements = ffi.new('VkSparseImageMemoryRequirements[]', pSparseMemoryRequirementCount[0])
    result = _callApi(lib.vkGetImageSparseMemoryRequirements, device,image,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

    return pSparseMemoryRequirements


def vkGetPhysicalDeviceSparseImageFormatProperties(
physicalDevice
        ,format
        ,type
        ,samples
        ,usage
        ,tiling
        ,):

    pPropertyCount = ffi.new('uint32_t*')
    pProperties = ffi.NULL

    result = _callApi(lib.vkGetPhysicalDeviceSparseImageFormatProperties, physicalDevice,format,type,samples,usage,tiling,pPropertyCount,pProperties)

    pProperties = ffi.new('VkSparseImageFormatProperties[]', pPropertyCount[0])
    result = _callApi(lib.vkGetPhysicalDeviceSparseImageFormatProperties, physicalDevice,format,type,samples,usage,tiling,pPropertyCount,pProperties)

    return pProperties


def vkQueueBindSparse(
queue
        ,bindInfoCount
        ,pBindInfo
        ,fence
        ,):
    result = _callApi(lib.vkQueueBindSparse, queue,bindInfoCount,pBindInfo,fence)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkCreateFence(
device
        ,pCreateInfo
        ,pAllocator
        ,pFence=None
        ,):

    custom_return = True
    if not pFence:
        pFence = ffi.new('VkFence*')
        custom_return = False

    result = _callApi(lib.vkCreateFence, device,pCreateInfo,pAllocator,pFence)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pFence

    return pFence[0]



def vkDestroyFence(
device
        ,fence
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyFence, device,fence,pAllocator)


def vkResetFences(
device
        ,fenceCount
        ,pFences
        ,):
    result = _callApi(lib.vkResetFences, device,fenceCount,pFences)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkGetFenceStatus(
device
        ,fence
        ,):
    result = _callApi(lib.vkGetFenceStatus, device,fence)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkWaitForFences(
device
        ,fenceCount
        ,pFences
        ,waitAll
        ,timeout
        ,):
    result = _callApi(lib.vkWaitForFences, device,fenceCount,pFences,waitAll,timeout)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkCreateSemaphore(
device
        ,pCreateInfo
        ,pAllocator
        ,pSemaphore=None
        ,):

    custom_return = True
    if not pSemaphore:
        pSemaphore = ffi.new('VkSemaphore*')
        custom_return = False

    result = _callApi(lib.vkCreateSemaphore, device,pCreateInfo,pAllocator,pSemaphore)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pSemaphore

    return pSemaphore[0]



def vkDestroySemaphore(
device
        ,semaphore
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroySemaphore, device,semaphore,pAllocator)


def vkCreateEvent(
device
        ,pCreateInfo
        ,pAllocator
        ,pEvent=None
        ,):

    custom_return = True
    if not pEvent:
        pEvent = ffi.new('VkEvent*')
        custom_return = False

    result = _callApi(lib.vkCreateEvent, device,pCreateInfo,pAllocator,pEvent)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pEvent

    return pEvent[0]



def vkDestroyEvent(
device
        ,event
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyEvent, device,event,pAllocator)


def vkGetEventStatus(
device
        ,event
        ,):
    result = _callApi(lib.vkGetEventStatus, device,event)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkSetEvent(
device
        ,event
        ,):
    result = _callApi(lib.vkSetEvent, device,event)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkResetEvent(
device
        ,event
        ,):
    result = _callApi(lib.vkResetEvent, device,event)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkCreateQueryPool(
device
        ,pCreateInfo
        ,pAllocator
        ,pQueryPool=None
        ,):

    custom_return = True
    if not pQueryPool:
        pQueryPool = ffi.new('VkQueryPool*')
        custom_return = False

    result = _callApi(lib.vkCreateQueryPool, device,pCreateInfo,pAllocator,pQueryPool)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pQueryPool

    return pQueryPool[0]



def vkDestroyQueryPool(
device
        ,queryPool
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyQueryPool, device,queryPool,pAllocator)


def vkGetQueryPoolResults(
device
        ,queryPool
        ,firstQuery
        ,queryCount
        ,dataSize
        ,pData
        ,stride
        ,flags
        ,):
    result = _callApi(lib.vkGetQueryPoolResults, device,queryPool,firstQuery,queryCount,dataSize,pData,stride,flags)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkResetQueryPool(
device
        ,queryPool
        ,firstQuery
        ,queryCount
        ,):
    result = _callApi(lib.vkResetQueryPool, device,queryPool,firstQuery,queryCount)


def _wrap_vkResetQueryPool(fn):
    def vkResetQueryPool(
    device
            ,queryPool
            ,firstQuery
            ,queryCount
            ,):
        result = _callApi(fn, device,queryPool,firstQuery,queryCount)


    return vkResetQueryPool
def vkCreateBuffer(
device
        ,pCreateInfo
        ,pAllocator
        ,pBuffer=None
        ,):

    custom_return = True
    if not pBuffer:
        pBuffer = ffi.new('VkBuffer*')
        custom_return = False

    result = _callApi(lib.vkCreateBuffer, device,pCreateInfo,pAllocator,pBuffer)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pBuffer

    return pBuffer[0]



def vkDestroyBuffer(
device
        ,buffer
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyBuffer, device,buffer,pAllocator)


def vkCreateBufferView(
device
        ,pCreateInfo
        ,pAllocator
        ,pView=None
        ,):

    custom_return = True
    if not pView:
        pView = ffi.new('VkBufferView*')
        custom_return = False

    result = _callApi(lib.vkCreateBufferView, device,pCreateInfo,pAllocator,pView)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pView

    return pView[0]



def vkDestroyBufferView(
device
        ,bufferView
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyBufferView, device,bufferView,pAllocator)


def vkCreateImage(
device
        ,pCreateInfo
        ,pAllocator
        ,pImage=None
        ,):

    custom_return = True
    if not pImage:
        pImage = ffi.new('VkImage*')
        custom_return = False

    result = _callApi(lib.vkCreateImage, device,pCreateInfo,pAllocator,pImage)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pImage

    return pImage[0]



def vkDestroyImage(
device
        ,image
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyImage, device,image,pAllocator)


def vkGetImageSubresourceLayout(
device
        ,image
        ,pSubresource
        ,pLayout=None
        ,):

    custom_return = True
    if not pLayout:
        pLayout = ffi.new('VkSubresourceLayout*')
        custom_return = False

    result = _callApi(lib.vkGetImageSubresourceLayout, device,image,pSubresource,pLayout)

    if custom_return:
        return pLayout

    return pLayout[0]



def vkCreateImageView(
device
        ,pCreateInfo
        ,pAllocator
        ,pView=None
        ,):

    custom_return = True
    if not pView:
        pView = ffi.new('VkImageView*')
        custom_return = False

    result = _callApi(lib.vkCreateImageView, device,pCreateInfo,pAllocator,pView)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pView

    return pView[0]



def vkDestroyImageView(
device
        ,imageView
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyImageView, device,imageView,pAllocator)


def vkCreateShaderModule(
device
        ,pCreateInfo
        ,pAllocator
        ,pShaderModule=None
        ,):

    custom_return = True
    if not pShaderModule:
        pShaderModule = ffi.new('VkShaderModule*')
        custom_return = False

    result = _callApi(lib.vkCreateShaderModule, device,pCreateInfo,pAllocator,pShaderModule)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pShaderModule

    return pShaderModule[0]



def vkDestroyShaderModule(
device
        ,shaderModule
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyShaderModule, device,shaderModule,pAllocator)


def vkCreatePipelineCache(
device
        ,pCreateInfo
        ,pAllocator
        ,pPipelineCache=None
        ,):

    custom_return = True
    if not pPipelineCache:
        pPipelineCache = ffi.new('VkPipelineCache*')
        custom_return = False

    result = _callApi(lib.vkCreatePipelineCache, device,pCreateInfo,pAllocator,pPipelineCache)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pPipelineCache

    return pPipelineCache[0]



def vkCreatePipelineCache(
device
        ,pCreateInfo
        ,pAllocator
        ,pPipelineCache=None
        ,):

    custom_return = True
    if not pPipelineCache:
        pPipelineCache = ffi.new('VkPipelineCache*')
        custom_return = False

    result = _callApi(lib.vkCreatePipelineCache, device,pCreateInfo,pAllocator,pPipelineCache)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pPipelineCache

    return pPipelineCache[0]



def vkDestroyPipelineCache(
device
        ,pipelineCache
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyPipelineCache, device,pipelineCache,pAllocator)


def vkMergePipelineCaches(
device
        ,dstCache
        ,srcCacheCount
        ,pSrcCaches
        ,):
    result = _callApi(lib.vkMergePipelineCaches, device,dstCache,srcCacheCount,pSrcCaches)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkCreateGraphicsPipelines(
device
        ,pipelineCache
        ,createInfoCount
        ,pCreateInfos
        ,pAllocator
        ,pPipelines=None
        ,):

    custom_return = True
    if not pPipelines:
        pPipelines = ffi.new('VkPipeline[%d]' % createInfoCount)
        custom_return = False

    result = _callApi(lib.vkCreateGraphicsPipelines, device,pipelineCache,createInfoCount,pCreateInfos,pAllocator,pPipelines)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pPipelines

    return pPipelines



def vkCreateGraphicsPipelines(
device
        ,pipelineCache
        ,createInfoCount
        ,pCreateInfos
        ,pAllocator
        ,pPipelines=None
        ,):

    custom_return = True
    if not pPipelines:
        pPipelines = ffi.new('VkPipeline[%d]' % createInfoCount)
        custom_return = False

    result = _callApi(lib.vkCreateGraphicsPipelines, device,pipelineCache,createInfoCount,pCreateInfos,pAllocator,pPipelines)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pPipelines

    return pPipelines



def vkCreateComputePipelines(
device
        ,pipelineCache
        ,createInfoCount
        ,pCreateInfos
        ,pAllocator
        ,pPipelines=None
        ,):

    custom_return = True
    if not pPipelines:
        pPipelines = ffi.new('VkPipeline[%d]' % createInfoCount)
        custom_return = False

    result = _callApi(lib.vkCreateComputePipelines, device,pipelineCache,createInfoCount,pCreateInfos,pAllocator,pPipelines)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pPipelines

    return pPipelines



def vkCreateComputePipelines(
device
        ,pipelineCache
        ,createInfoCount
        ,pCreateInfos
        ,pAllocator
        ,pPipelines=None
        ,):

    custom_return = True
    if not pPipelines:
        pPipelines = ffi.new('VkPipeline[%d]' % createInfoCount)
        custom_return = False

    result = _callApi(lib.vkCreateComputePipelines, device,pipelineCache,createInfoCount,pCreateInfos,pAllocator,pPipelines)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pPipelines

    return pPipelines



def _wrap_vkGetDeviceSubpassShadingMaxWorkgroupSizeHUAWEI(fn):
    def vkGetDeviceSubpassShadingMaxWorkgroupSizeHUAWEI(
    device
            ,renderpass
            ,pMaxWorkgroupSize=None
            ,):

        custom_return = True
        if not pMaxWorkgroupSize:
            pMaxWorkgroupSize = ffi.new('VkExtent2D*')
            custom_return = False

        result = _callApi(fn, device,renderpass,pMaxWorkgroupSize)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMaxWorkgroupSize

        return pMaxWorkgroupSize[0]



    return vkGetDeviceSubpassShadingMaxWorkgroupSizeHUAWEI
def vkDestroyPipeline(
device
        ,pipeline
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyPipeline, device,pipeline,pAllocator)


def vkCreatePipelineLayout(
device
        ,pCreateInfo
        ,pAllocator
        ,pPipelineLayout=None
        ,):

    custom_return = True
    if not pPipelineLayout:
        pPipelineLayout = ffi.new('VkPipelineLayout*')
        custom_return = False

    result = _callApi(lib.vkCreatePipelineLayout, device,pCreateInfo,pAllocator,pPipelineLayout)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pPipelineLayout

    return pPipelineLayout[0]



def vkDestroyPipelineLayout(
device
        ,pipelineLayout
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyPipelineLayout, device,pipelineLayout,pAllocator)


def vkCreateSampler(
device
        ,pCreateInfo
        ,pAllocator
        ,pSampler=None
        ,):

    custom_return = True
    if not pSampler:
        pSampler = ffi.new('VkSampler*')
        custom_return = False

    result = _callApi(lib.vkCreateSampler, device,pCreateInfo,pAllocator,pSampler)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pSampler

    return pSampler[0]



def vkDestroySampler(
device
        ,sampler
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroySampler, device,sampler,pAllocator)


def vkCreateDescriptorSetLayout(
device
        ,pCreateInfo
        ,pAllocator
        ,pSetLayout=None
        ,):

    custom_return = True
    if not pSetLayout:
        pSetLayout = ffi.new('VkDescriptorSetLayout*')
        custom_return = False

    result = _callApi(lib.vkCreateDescriptorSetLayout, device,pCreateInfo,pAllocator,pSetLayout)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pSetLayout

    return pSetLayout[0]



def vkDestroyDescriptorSetLayout(
device
        ,descriptorSetLayout
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyDescriptorSetLayout, device,descriptorSetLayout,pAllocator)


def vkCreateDescriptorPool(
device
        ,pCreateInfo
        ,pAllocator
        ,pDescriptorPool=None
        ,):

    custom_return = True
    if not pDescriptorPool:
        pDescriptorPool = ffi.new('VkDescriptorPool*')
        custom_return = False

    result = _callApi(lib.vkCreateDescriptorPool, device,pCreateInfo,pAllocator,pDescriptorPool)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pDescriptorPool

    return pDescriptorPool[0]



def vkDestroyDescriptorPool(
device
        ,descriptorPool
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyDescriptorPool, device,descriptorPool,pAllocator)


def vkResetDescriptorPool(
device
        ,descriptorPool
        ,flags
        ,):
    result = _callApi(lib.vkResetDescriptorPool, device,descriptorPool,flags)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkAllocateDescriptorSets(
device
        ,pAllocateInfo
        ,pDescriptorSets=None
        ,):

    custom_return = True
    if not pDescriptorSets:
        pDescriptorSets = ffi.new('VkDescriptorSet[%d]' % pAllocateInfo.descriptorSetCount)
        custom_return = False

    result = _callApi(lib.vkAllocateDescriptorSets, device,pAllocateInfo,pDescriptorSets)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pDescriptorSets

    return pDescriptorSets



def vkFreeDescriptorSets(
device
        ,descriptorPool
        ,descriptorSetCount
        ,pDescriptorSets
        ,):
    result = _callApi(lib.vkFreeDescriptorSets, device,descriptorPool,descriptorSetCount,pDescriptorSets)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkUpdateDescriptorSets(
device
        ,descriptorWriteCount
        ,pDescriptorWrites
        ,descriptorCopyCount
        ,pDescriptorCopies
        ,):
    result = _callApi(lib.vkUpdateDescriptorSets, device,descriptorWriteCount,pDescriptorWrites,descriptorCopyCount,pDescriptorCopies)


def vkCreateFramebuffer(
device
        ,pCreateInfo
        ,pAllocator
        ,pFramebuffer=None
        ,):

    custom_return = True
    if not pFramebuffer:
        pFramebuffer = ffi.new('VkFramebuffer*')
        custom_return = False

    result = _callApi(lib.vkCreateFramebuffer, device,pCreateInfo,pAllocator,pFramebuffer)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pFramebuffer

    return pFramebuffer[0]



def vkDestroyFramebuffer(
device
        ,framebuffer
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyFramebuffer, device,framebuffer,pAllocator)


def vkCreateRenderPass(
device
        ,pCreateInfo
        ,pAllocator
        ,pRenderPass=None
        ,):

    custom_return = True
    if not pRenderPass:
        pRenderPass = ffi.new('VkRenderPass*')
        custom_return = False

    result = _callApi(lib.vkCreateRenderPass, device,pCreateInfo,pAllocator,pRenderPass)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pRenderPass

    return pRenderPass[0]



def vkDestroyRenderPass(
device
        ,renderPass
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyRenderPass, device,renderPass,pAllocator)


def vkGetRenderAreaGranularity(
device
        ,renderPass
        ,pGranularity=None
        ,):

    custom_return = True
    if not pGranularity:
        pGranularity = ffi.new('VkExtent2D*')
        custom_return = False

    result = _callApi(lib.vkGetRenderAreaGranularity, device,renderPass,pGranularity)

    if custom_return:
        return pGranularity

    return pGranularity[0]



def _wrap_vkGetRenderingAreaGranularityKHR(fn):
    def vkGetRenderingAreaGranularityKHR(
    device
            ,pRenderingAreaInfo
            ,pGranularity=None
            ,):

        custom_return = True
        if not pGranularity:
            pGranularity = ffi.new('VkExtent2D*')
            custom_return = False

        result = _callApi(fn, device,pRenderingAreaInfo,pGranularity)

        if custom_return:
            return pGranularity

        return pGranularity[0]



    return vkGetRenderingAreaGranularityKHR
def vkCreateCommandPool(
device
        ,pCreateInfo
        ,pAllocator
        ,pCommandPool=None
        ,):

    custom_return = True
    if not pCommandPool:
        pCommandPool = ffi.new('VkCommandPool*')
        custom_return = False

    result = _callApi(lib.vkCreateCommandPool, device,pCreateInfo,pAllocator,pCommandPool)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pCommandPool

    return pCommandPool[0]



def vkDestroyCommandPool(
device
        ,commandPool
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyCommandPool, device,commandPool,pAllocator)


def vkResetCommandPool(
device
        ,commandPool
        ,flags
        ,):
    result = _callApi(lib.vkResetCommandPool, device,commandPool,flags)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkAllocateCommandBuffers(
device
        ,pAllocateInfo
        ,pCommandBuffers=None
        ,):

    custom_return = True
    if not pCommandBuffers:
        pCommandBuffers = ffi.new('VkCommandBuffer[%d]' % pAllocateInfo.commandBufferCount)
        custom_return = False

    result = _callApi(lib.vkAllocateCommandBuffers, device,pAllocateInfo,pCommandBuffers)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pCommandBuffers

    return pCommandBuffers



def vkFreeCommandBuffers(
device
        ,commandPool
        ,commandBufferCount
        ,pCommandBuffers
        ,):
    result = _callApi(lib.vkFreeCommandBuffers, device,commandPool,commandBufferCount,pCommandBuffers)


def vkBeginCommandBuffer(
commandBuffer
        ,pBeginInfo
        ,):
    result = _callApi(lib.vkBeginCommandBuffer, commandBuffer,pBeginInfo)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkEndCommandBuffer(
commandBuffer
        ,):
    result = _callApi(lib.vkEndCommandBuffer, commandBuffer)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkResetCommandBuffer(
commandBuffer
        ,flags
        ,):
    result = _callApi(lib.vkResetCommandBuffer, commandBuffer,flags)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkCmdBindPipeline(
commandBuffer
        ,pipelineBindPoint
        ,pipeline
        ,):
    result = _callApi(lib.vkCmdBindPipeline, commandBuffer,pipelineBindPoint,pipeline)


def _wrap_vkCmdSetAttachmentFeedbackLoopEnableEXT(fn):
    def vkCmdSetAttachmentFeedbackLoopEnableEXT(
    commandBuffer
            ,aspectMask
            ,):
        result = _callApi(fn, commandBuffer,aspectMask)


    return vkCmdSetAttachmentFeedbackLoopEnableEXT
def vkCmdSetViewport(
commandBuffer
        ,firstViewport
        ,viewportCount
        ,pViewports
        ,):
    result = _callApi(lib.vkCmdSetViewport, commandBuffer,firstViewport,viewportCount,pViewports)


def vkCmdSetScissor(
commandBuffer
        ,firstScissor
        ,scissorCount
        ,pScissors
        ,):
    result = _callApi(lib.vkCmdSetScissor, commandBuffer,firstScissor,scissorCount,pScissors)


def vkCmdSetLineWidth(
commandBuffer
        ,lineWidth
        ,):
    result = _callApi(lib.vkCmdSetLineWidth, commandBuffer,lineWidth)


def vkCmdSetDepthBias(
commandBuffer
        ,depthBiasConstantFactor
        ,depthBiasClamp
        ,depthBiasSlopeFactor
        ,):
    result = _callApi(lib.vkCmdSetDepthBias, commandBuffer,depthBiasConstantFactor,depthBiasClamp,depthBiasSlopeFactor)


def vkCmdSetBlendConstants(
commandBuffer
        ,blendConstants
        ,):
    result = _callApi(lib.vkCmdSetBlendConstants, commandBuffer,blendConstants)


def vkCmdSetDepthBounds(
commandBuffer
        ,minDepthBounds
        ,maxDepthBounds
        ,):
    result = _callApi(lib.vkCmdSetDepthBounds, commandBuffer,minDepthBounds,maxDepthBounds)


def vkCmdSetStencilCompareMask(
commandBuffer
        ,faceMask
        ,compareMask
        ,):
    result = _callApi(lib.vkCmdSetStencilCompareMask, commandBuffer,faceMask,compareMask)


def vkCmdSetStencilWriteMask(
commandBuffer
        ,faceMask
        ,writeMask
        ,):
    result = _callApi(lib.vkCmdSetStencilWriteMask, commandBuffer,faceMask,writeMask)


def vkCmdSetStencilReference(
commandBuffer
        ,faceMask
        ,reference
        ,):
    result = _callApi(lib.vkCmdSetStencilReference, commandBuffer,faceMask,reference)


def vkCmdBindDescriptorSets(
commandBuffer
        ,pipelineBindPoint
        ,layout
        ,firstSet
        ,descriptorSetCount
        ,pDescriptorSets
        ,dynamicOffsetCount
        ,pDynamicOffsets
        ,):
    result = _callApi(lib.vkCmdBindDescriptorSets, commandBuffer,pipelineBindPoint,layout,firstSet,descriptorSetCount,pDescriptorSets,dynamicOffsetCount,pDynamicOffsets)


def vkCmdBindIndexBuffer(
commandBuffer
        ,buffer
        ,offset
        ,indexType
        ,):
    result = _callApi(lib.vkCmdBindIndexBuffer, commandBuffer,buffer,offset,indexType)


def vkCmdBindVertexBuffers(
commandBuffer
        ,firstBinding
        ,bindingCount
        ,pBuffers
        ,pOffsets
        ,):
    result = _callApi(lib.vkCmdBindVertexBuffers, commandBuffer,firstBinding,bindingCount,pBuffers,pOffsets)


def vkCmdDraw(
commandBuffer
        ,vertexCount
        ,instanceCount
        ,firstVertex
        ,firstInstance
        ,):
    result = _callApi(lib.vkCmdDraw, commandBuffer,vertexCount,instanceCount,firstVertex,firstInstance)


def vkCmdDrawIndexed(
commandBuffer
        ,indexCount
        ,instanceCount
        ,firstIndex
        ,vertexOffset
        ,firstInstance
        ,):
    result = _callApi(lib.vkCmdDrawIndexed, commandBuffer,indexCount,instanceCount,firstIndex,vertexOffset,firstInstance)


def _wrap_vkCmdDrawMultiEXT(fn):
    def vkCmdDrawMultiEXT(
    commandBuffer
            ,drawCount
            ,pVertexInfo
            ,instanceCount
            ,firstInstance
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,drawCount,pVertexInfo,instanceCount,firstInstance,stride)


    return vkCmdDrawMultiEXT
def _wrap_vkCmdDrawMultiIndexedEXT(fn):
    def vkCmdDrawMultiIndexedEXT(
    commandBuffer
            ,drawCount
            ,pIndexInfo
            ,instanceCount
            ,firstInstance
            ,stride
            ,pVertexOffset
            ,):
        result = _callApi(fn, commandBuffer,drawCount,pIndexInfo,instanceCount,firstInstance,stride,pVertexOffset)


    return vkCmdDrawMultiIndexedEXT
def vkCmdDrawIndirect(
commandBuffer
        ,buffer
        ,offset
        ,drawCount
        ,stride
        ,):
    result = _callApi(lib.vkCmdDrawIndirect, commandBuffer,buffer,offset,drawCount,stride)


def vkCmdDrawIndexedIndirect(
commandBuffer
        ,buffer
        ,offset
        ,drawCount
        ,stride
        ,):
    result = _callApi(lib.vkCmdDrawIndexedIndirect, commandBuffer,buffer,offset,drawCount,stride)


def vkCmdDispatch(
commandBuffer
        ,groupCountX
        ,groupCountY
        ,groupCountZ
        ,):
    result = _callApi(lib.vkCmdDispatch, commandBuffer,groupCountX,groupCountY,groupCountZ)


def vkCmdDispatchIndirect(
commandBuffer
        ,buffer
        ,offset
        ,):
    result = _callApi(lib.vkCmdDispatchIndirect, commandBuffer,buffer,offset)


def _wrap_vkCmdSubpassShadingHUAWEI(fn):
    def vkCmdSubpassShadingHUAWEI(
    commandBuffer
            ,):
        result = _callApi(fn, commandBuffer)


    return vkCmdSubpassShadingHUAWEI
def _wrap_vkCmdDrawClusterHUAWEI(fn):
    def vkCmdDrawClusterHUAWEI(
    commandBuffer
            ,groupCountX
            ,groupCountY
            ,groupCountZ
            ,):
        result = _callApi(fn, commandBuffer,groupCountX,groupCountY,groupCountZ)


    return vkCmdDrawClusterHUAWEI
def _wrap_vkCmdDrawClusterIndirectHUAWEI(fn):
    def vkCmdDrawClusterIndirectHUAWEI(
    commandBuffer
            ,buffer
            ,offset
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset)


    return vkCmdDrawClusterIndirectHUAWEI
def _wrap_vkCmdUpdatePipelineIndirectBufferNV(fn):
    def vkCmdUpdatePipelineIndirectBufferNV(
    commandBuffer
            ,pipelineBindPoint
            ,pipeline
            ,):
        result = _callApi(fn, commandBuffer,pipelineBindPoint,pipeline)


    return vkCmdUpdatePipelineIndirectBufferNV
def vkCmdCopyBuffer(
commandBuffer
        ,srcBuffer
        ,dstBuffer
        ,regionCount
        ,pRegions
        ,):
    result = _callApi(lib.vkCmdCopyBuffer, commandBuffer,srcBuffer,dstBuffer,regionCount,pRegions)


def vkCmdCopyImage(
commandBuffer
        ,srcImage
        ,srcImageLayout
        ,dstImage
        ,dstImageLayout
        ,regionCount
        ,pRegions
        ,):
    result = _callApi(lib.vkCmdCopyImage, commandBuffer,srcImage,srcImageLayout,dstImage,dstImageLayout,regionCount,pRegions)


def vkCmdBlitImage(
commandBuffer
        ,srcImage
        ,srcImageLayout
        ,dstImage
        ,dstImageLayout
        ,regionCount
        ,pRegions
        ,filter
        ,):
    result = _callApi(lib.vkCmdBlitImage, commandBuffer,srcImage,srcImageLayout,dstImage,dstImageLayout,regionCount,pRegions,filter)


def vkCmdCopyBufferToImage(
commandBuffer
        ,srcBuffer
        ,dstImage
        ,dstImageLayout
        ,regionCount
        ,pRegions
        ,):
    result = _callApi(lib.vkCmdCopyBufferToImage, commandBuffer,srcBuffer,dstImage,dstImageLayout,regionCount,pRegions)


def vkCmdCopyImageToBuffer(
commandBuffer
        ,srcImage
        ,srcImageLayout
        ,dstBuffer
        ,regionCount
        ,pRegions
        ,):
    result = _callApi(lib.vkCmdCopyImageToBuffer, commandBuffer,srcImage,srcImageLayout,dstBuffer,regionCount,pRegions)


def _wrap_vkCmdCopyMemoryIndirectNV(fn):
    def vkCmdCopyMemoryIndirectNV(
    commandBuffer
            ,copyBufferAddress
            ,copyCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,copyBufferAddress,copyCount,stride)


    return vkCmdCopyMemoryIndirectNV
def _wrap_vkCmdCopyMemoryToImageIndirectNV(fn):
    def vkCmdCopyMemoryToImageIndirectNV(
    commandBuffer
            ,copyBufferAddress
            ,copyCount
            ,stride
            ,dstImage
            ,dstImageLayout
            ,pImageSubresources
            ,):
        result = _callApi(fn, commandBuffer,copyBufferAddress,copyCount,stride,dstImage,dstImageLayout,pImageSubresources)


    return vkCmdCopyMemoryToImageIndirectNV
def vkCmdUpdateBuffer(
commandBuffer
        ,dstBuffer
        ,dstOffset
        ,dataSize
        ,pData
        ,):
    result = _callApi(lib.vkCmdUpdateBuffer, commandBuffer,dstBuffer,dstOffset,dataSize,pData)


def vkCmdFillBuffer(
commandBuffer
        ,dstBuffer
        ,dstOffset
        ,size
        ,data
        ,):
    result = _callApi(lib.vkCmdFillBuffer, commandBuffer,dstBuffer,dstOffset,size,data)


def vkCmdClearColorImage(
commandBuffer
        ,image
        ,imageLayout
        ,pColor
        ,rangeCount
        ,pRanges
        ,):
    result = _callApi(lib.vkCmdClearColorImage, commandBuffer,image,imageLayout,pColor,rangeCount,pRanges)


def vkCmdClearDepthStencilImage(
commandBuffer
        ,image
        ,imageLayout
        ,pDepthStencil
        ,rangeCount
        ,pRanges
        ,):
    result = _callApi(lib.vkCmdClearDepthStencilImage, commandBuffer,image,imageLayout,pDepthStencil,rangeCount,pRanges)


def vkCmdClearAttachments(
commandBuffer
        ,attachmentCount
        ,pAttachments
        ,rectCount
        ,pRects
        ,):
    result = _callApi(lib.vkCmdClearAttachments, commandBuffer,attachmentCount,pAttachments,rectCount,pRects)


def vkCmdResolveImage(
commandBuffer
        ,srcImage
        ,srcImageLayout
        ,dstImage
        ,dstImageLayout
        ,regionCount
        ,pRegions
        ,):
    result = _callApi(lib.vkCmdResolveImage, commandBuffer,srcImage,srcImageLayout,dstImage,dstImageLayout,regionCount,pRegions)


def vkCmdSetEvent(
commandBuffer
        ,event
        ,stageMask
        ,):
    result = _callApi(lib.vkCmdSetEvent, commandBuffer,event,stageMask)


def vkCmdResetEvent(
commandBuffer
        ,event
        ,stageMask
        ,):
    result = _callApi(lib.vkCmdResetEvent, commandBuffer,event,stageMask)


def vkCmdWaitEvents(
commandBuffer
        ,eventCount
        ,pEvents
        ,srcStageMask
        ,dstStageMask
        ,memoryBarrierCount
        ,pMemoryBarriers
        ,bufferMemoryBarrierCount
        ,pBufferMemoryBarriers
        ,imageMemoryBarrierCount
        ,pImageMemoryBarriers
        ,):
    result = _callApi(lib.vkCmdWaitEvents, commandBuffer,eventCount,pEvents,srcStageMask,dstStageMask,memoryBarrierCount,pMemoryBarriers,bufferMemoryBarrierCount,pBufferMemoryBarriers,imageMemoryBarrierCount,pImageMemoryBarriers)


def vkCmdPipelineBarrier(
commandBuffer
        ,srcStageMask
        ,dstStageMask
        ,dependencyFlags
        ,memoryBarrierCount
        ,pMemoryBarriers
        ,bufferMemoryBarrierCount
        ,pBufferMemoryBarriers
        ,imageMemoryBarrierCount
        ,pImageMemoryBarriers
        ,):
    result = _callApi(lib.vkCmdPipelineBarrier, commandBuffer,srcStageMask,dstStageMask,dependencyFlags,memoryBarrierCount,pMemoryBarriers,bufferMemoryBarrierCount,pBufferMemoryBarriers,imageMemoryBarrierCount,pImageMemoryBarriers)


def vkCmdBeginQuery(
commandBuffer
        ,queryPool
        ,query
        ,flags
        ,):
    result = _callApi(lib.vkCmdBeginQuery, commandBuffer,queryPool,query,flags)


def vkCmdEndQuery(
commandBuffer
        ,queryPool
        ,query
        ,):
    result = _callApi(lib.vkCmdEndQuery, commandBuffer,queryPool,query)


def _wrap_vkCmdBeginConditionalRenderingEXT(fn):
    def vkCmdBeginConditionalRenderingEXT(
    commandBuffer
            ,pConditionalRenderingBegin
            ,):
        result = _callApi(fn, commandBuffer,pConditionalRenderingBegin)


    return vkCmdBeginConditionalRenderingEXT
def _wrap_vkCmdEndConditionalRenderingEXT(fn):
    def vkCmdEndConditionalRenderingEXT(
    commandBuffer
            ,):
        result = _callApi(fn, commandBuffer)


    return vkCmdEndConditionalRenderingEXT
def vkCmdResetQueryPool(
commandBuffer
        ,queryPool
        ,firstQuery
        ,queryCount
        ,):
    result = _callApi(lib.vkCmdResetQueryPool, commandBuffer,queryPool,firstQuery,queryCount)


def vkCmdWriteTimestamp(
commandBuffer
        ,pipelineStage
        ,queryPool
        ,query
        ,):
    result = _callApi(lib.vkCmdWriteTimestamp, commandBuffer,pipelineStage,queryPool,query)


def vkCmdCopyQueryPoolResults(
commandBuffer
        ,queryPool
        ,firstQuery
        ,queryCount
        ,dstBuffer
        ,dstOffset
        ,stride
        ,flags
        ,):
    result = _callApi(lib.vkCmdCopyQueryPoolResults, commandBuffer,queryPool,firstQuery,queryCount,dstBuffer,dstOffset,stride,flags)


def vkCmdPushConstants(
commandBuffer
        ,layout
        ,stageFlags
        ,offset
        ,size
        ,pValues
        ,):
    result = _callApi(lib.vkCmdPushConstants, commandBuffer,layout,stageFlags,offset,size,pValues)


def vkCmdBeginRenderPass(
commandBuffer
        ,pRenderPassBegin
        ,contents
        ,):
    result = _callApi(lib.vkCmdBeginRenderPass, commandBuffer,pRenderPassBegin,contents)


def vkCmdNextSubpass(
commandBuffer
        ,contents
        ,):
    result = _callApi(lib.vkCmdNextSubpass, commandBuffer,contents)


def vkCmdEndRenderPass(
commandBuffer
        ,):
    result = _callApi(lib.vkCmdEndRenderPass, commandBuffer)


def vkCmdExecuteCommands(
commandBuffer
        ,commandBufferCount
        ,pCommandBuffers
        ,):
    result = _callApi(lib.vkCmdExecuteCommands, commandBuffer,commandBufferCount,pCommandBuffers)


def _wrap_vkCreateAndroidSurfaceKHR(fn):
    def vkCreateAndroidSurfaceKHR(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateAndroidSurfaceKHR
def _wrap_vkGetPhysicalDeviceDisplayPropertiesKHR(fn):
    def vkGetPhysicalDeviceDisplayPropertiesKHR(
    physicalDevice
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkDisplayPropertiesKHR[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        result = (StrWrap(x) for x in pProperties)
        _weakkey_dict[result] = pProperties
        return result


    return vkGetPhysicalDeviceDisplayPropertiesKHR
def _wrap_vkGetPhysicalDeviceDisplayPlanePropertiesKHR(fn):
    def vkGetPhysicalDeviceDisplayPlanePropertiesKHR(
    physicalDevice
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkDisplayPlanePropertiesKHR[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pProperties


    return vkGetPhysicalDeviceDisplayPlanePropertiesKHR
def _wrap_vkGetDisplayPlaneSupportedDisplaysKHR(fn):
    def vkGetDisplayPlaneSupportedDisplaysKHR(
    physicalDevice
            ,planeIndex
            ,):

        pDisplayCount = ffi.new('uint32_t*')
        pDisplays = ffi.NULL

        result = _callApi(fn, physicalDevice,planeIndex,pDisplayCount,pDisplays)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pDisplays = ffi.new('VkDisplayKHR[]', pDisplayCount[0])
        result = _callApi(fn, physicalDevice,planeIndex,pDisplayCount,pDisplays)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pDisplays


    return vkGetDisplayPlaneSupportedDisplaysKHR
def _wrap_vkGetDisplayModePropertiesKHR(fn):
    def vkGetDisplayModePropertiesKHR(
    physicalDevice
            ,display
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,display,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkDisplayModePropertiesKHR[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,display,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pProperties


    return vkGetDisplayModePropertiesKHR
def _wrap_vkCreateDisplayModeKHR(fn):
    def vkCreateDisplayModeKHR(
    physicalDevice
            ,display
            ,pCreateInfo
            ,pAllocator
            ,pMode=None
            ,):

        custom_return = True
        if not pMode:
            pMode = ffi.new('VkDisplayModeKHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,display,pCreateInfo,pAllocator,pMode)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMode

        return pMode[0]



    return vkCreateDisplayModeKHR
def _wrap_vkGetDisplayPlaneCapabilitiesKHR(fn):
    def vkGetDisplayPlaneCapabilitiesKHR(
    physicalDevice
            ,mode
            ,planeIndex
            ,pCapabilities=None
            ,):

        custom_return = True
        if not pCapabilities:
            pCapabilities = ffi.new('VkDisplayPlaneCapabilitiesKHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,mode,planeIndex,pCapabilities)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pCapabilities

        return pCapabilities[0]



    return vkGetDisplayPlaneCapabilitiesKHR
def _wrap_vkCreateDisplayPlaneSurfaceKHR(fn):
    def vkCreateDisplayPlaneSurfaceKHR(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateDisplayPlaneSurfaceKHR
def _wrap_vkCreateSharedSwapchainsKHR(fn):
    def vkCreateSharedSwapchainsKHR(
    device
            ,swapchainCount
            ,pCreateInfos
            ,pAllocator
            ,pSwapchains=None
            ,):

        custom_return = True
        if not pSwapchains:
            pSwapchains = ffi.new('VkSwapchainKHR[%d]' % swapchainCount)
            custom_return = False

        result = _callApi(fn, device,swapchainCount,pCreateInfos,pAllocator,pSwapchains)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSwapchains

        return pSwapchains



    return vkCreateSharedSwapchainsKHR
def _wrap_vkDestroySurfaceKHR(fn):
    def vkDestroySurfaceKHR(
    instance
            ,surface
            ,pAllocator
            ,):
        result = _callApi(fn, instance,surface,pAllocator)


    return vkDestroySurfaceKHR
def _wrap_vkGetPhysicalDeviceSurfaceSupportKHR(fn):
    def vkGetPhysicalDeviceSurfaceSupportKHR(
    physicalDevice
            ,queueFamilyIndex
            ,surface
            ,pSupported=None
            ,):

        custom_return = True
        if not pSupported:
            pSupported = ffi.new('VkBool32*')
            custom_return = False

        result = _callApi(fn, physicalDevice,queueFamilyIndex,surface,pSupported)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSupported

        return pSupported[0]



    return vkGetPhysicalDeviceSurfaceSupportKHR
def _wrap_vkGetPhysicalDeviceSurfaceCapabilitiesKHR(fn):
    def vkGetPhysicalDeviceSurfaceCapabilitiesKHR(
    physicalDevice
            ,surface
            ,pSurfaceCapabilities=None
            ,):

        custom_return = True
        if not pSurfaceCapabilities:
            pSurfaceCapabilities = ffi.new('VkSurfaceCapabilitiesKHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,surface,pSurfaceCapabilities)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurfaceCapabilities

        return pSurfaceCapabilities[0]



    return vkGetPhysicalDeviceSurfaceCapabilitiesKHR
def _wrap_vkGetPhysicalDeviceSurfaceFormatsKHR(fn):
    def vkGetPhysicalDeviceSurfaceFormatsKHR(
    physicalDevice
            ,surface
            ,):

        pSurfaceFormatCount = ffi.new('uint32_t*')
        pSurfaceFormats = ffi.NULL

        result = _callApi(fn, physicalDevice,surface,pSurfaceFormatCount,pSurfaceFormats)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pSurfaceFormats = ffi.new('VkSurfaceFormatKHR[]', pSurfaceFormatCount[0])
        result = _callApi(fn, physicalDevice,surface,pSurfaceFormatCount,pSurfaceFormats)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pSurfaceFormats


    return vkGetPhysicalDeviceSurfaceFormatsKHR
def _wrap_vkGetPhysicalDeviceSurfacePresentModesKHR(fn):
    def vkGetPhysicalDeviceSurfacePresentModesKHR(
    physicalDevice
            ,surface
            ,):

        pPresentModeCount = ffi.new('uint32_t*')
        pPresentModes = ffi.NULL

        result = _callApi(fn, physicalDevice,surface,pPresentModeCount,pPresentModes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pPresentModes = ffi.new('VkPresentModeKHR[]', pPresentModeCount[0])
        result = _callApi(fn, physicalDevice,surface,pPresentModeCount,pPresentModes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pPresentModes


    return vkGetPhysicalDeviceSurfacePresentModesKHR
def _wrap_vkCreateSwapchainKHR(fn):
    def vkCreateSwapchainKHR(
    device
            ,pCreateInfo
            ,pAllocator
            ,pSwapchain=None
            ,):

        custom_return = True
        if not pSwapchain:
            pSwapchain = ffi.new('VkSwapchainKHR*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pSwapchain)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSwapchain

        return pSwapchain[0]



    return vkCreateSwapchainKHR
def _wrap_vkDestroySwapchainKHR(fn):
    def vkDestroySwapchainKHR(
    device
            ,swapchain
            ,pAllocator
            ,):
        result = _callApi(fn, device,swapchain,pAllocator)


    return vkDestroySwapchainKHR
def _wrap_vkGetSwapchainImagesKHR(fn):
    def vkGetSwapchainImagesKHR(
    device
            ,swapchain
            ,):

        pSwapchainImageCount = ffi.new('uint32_t*')
        pSwapchainImages = ffi.NULL

        result = _callApi(fn, device,swapchain,pSwapchainImageCount,pSwapchainImages)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pSwapchainImages = ffi.new('VkImage[]', pSwapchainImageCount[0])
        result = _callApi(fn, device,swapchain,pSwapchainImageCount,pSwapchainImages)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pSwapchainImages


    return vkGetSwapchainImagesKHR
def _wrap_vkAcquireNextImageKHR(fn):
    def vkAcquireNextImageKHR(
    device
            ,swapchain
            ,timeout
            ,semaphore
            ,fence
            ,pImageIndex=None
            ,):

        custom_return = True
        if not pImageIndex:
            pImageIndex = ffi.new('uint32_t*')
            custom_return = False

        result = _callApi(fn, device,swapchain,timeout,semaphore,fence,pImageIndex)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pImageIndex

        return pImageIndex[0]



    return vkAcquireNextImageKHR
def _wrap_vkQueuePresentKHR(fn):
    def vkQueuePresentKHR(
    queue
            ,pPresentInfo
            ,):
        result = _callApi(fn, queue,pPresentInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkQueuePresentKHR
def _wrap_vkCreateViSurfaceNN(fn):
    def vkCreateViSurfaceNN(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateViSurfaceNN
def _wrap_vkCreateWaylandSurfaceKHR(fn):
    def vkCreateWaylandSurfaceKHR(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateWaylandSurfaceKHR
def _wrap_vkGetPhysicalDeviceWaylandPresentationSupportKHR(fn):
    def vkGetPhysicalDeviceWaylandPresentationSupportKHR(
    physicalDevice
            ,queueFamilyIndex
            ,display
            ,):
        result = _callApi(fn, physicalDevice,queueFamilyIndex,display)


    return vkGetPhysicalDeviceWaylandPresentationSupportKHR
def _wrap_vkCreateWin32SurfaceKHR(fn):
    def vkCreateWin32SurfaceKHR(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateWin32SurfaceKHR
def _wrap_vkGetPhysicalDeviceWin32PresentationSupportKHR(fn):
    def vkGetPhysicalDeviceWin32PresentationSupportKHR(
    physicalDevice
            ,queueFamilyIndex
            ,):
        result = _callApi(fn, physicalDevice,queueFamilyIndex)


    return vkGetPhysicalDeviceWin32PresentationSupportKHR
def _wrap_vkCreateXlibSurfaceKHR(fn):
    def vkCreateXlibSurfaceKHR(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateXlibSurfaceKHR
def _wrap_vkGetPhysicalDeviceXlibPresentationSupportKHR(fn):
    def vkGetPhysicalDeviceXlibPresentationSupportKHR(
    physicalDevice
            ,queueFamilyIndex
            ,dpy
            ,visualID
            ,):
        result = _callApi(fn, physicalDevice,queueFamilyIndex,dpy,visualID)


    return vkGetPhysicalDeviceXlibPresentationSupportKHR
def _wrap_vkCreateXcbSurfaceKHR(fn):
    def vkCreateXcbSurfaceKHR(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateXcbSurfaceKHR
def _wrap_vkGetPhysicalDeviceXcbPresentationSupportKHR(fn):
    def vkGetPhysicalDeviceXcbPresentationSupportKHR(
    physicalDevice
            ,queueFamilyIndex
            ,connection
            ,visual_id
            ,):
        result = _callApi(fn, physicalDevice,queueFamilyIndex,connection,visual_id)


    return vkGetPhysicalDeviceXcbPresentationSupportKHR
def _wrap_vkCreateDirectFBSurfaceEXT(fn):
    def vkCreateDirectFBSurfaceEXT(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateDirectFBSurfaceEXT
def _wrap_vkGetPhysicalDeviceDirectFBPresentationSupportEXT(fn):
    def vkGetPhysicalDeviceDirectFBPresentationSupportEXT(
    physicalDevice
            ,queueFamilyIndex
            ,dfb
            ,):
        result = _callApi(fn, physicalDevice,queueFamilyIndex,dfb)


    return vkGetPhysicalDeviceDirectFBPresentationSupportEXT
def _wrap_vkCreateImagePipeSurfaceFUCHSIA(fn):
    def vkCreateImagePipeSurfaceFUCHSIA(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateImagePipeSurfaceFUCHSIA
def _wrap_vkCreateStreamDescriptorSurfaceGGP(fn):
    def vkCreateStreamDescriptorSurfaceGGP(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateStreamDescriptorSurfaceGGP
def _wrap_vkCreateScreenSurfaceQNX(fn):
    def vkCreateScreenSurfaceQNX(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateScreenSurfaceQNX
def _wrap_vkGetPhysicalDeviceScreenPresentationSupportQNX(fn):
    def vkGetPhysicalDeviceScreenPresentationSupportQNX(
    physicalDevice
            ,queueFamilyIndex
            ,window
            ,):
        result = _callApi(fn, physicalDevice,queueFamilyIndex,window)


    return vkGetPhysicalDeviceScreenPresentationSupportQNX
def _wrap_vkCreateDebugReportCallbackEXT(fn):
    def vkCreateDebugReportCallbackEXT(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pCallback=None
            ,):

        custom_return = True
        if not pCallback:
            pCallback = ffi.new('VkDebugReportCallbackEXT*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pCallback)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pCallback

        return pCallback[0]



    return vkCreateDebugReportCallbackEXT
def _wrap_vkDestroyDebugReportCallbackEXT(fn):
    def vkDestroyDebugReportCallbackEXT(
    instance
            ,callback
            ,pAllocator
            ,):
        result = _callApi(fn, instance,callback,pAllocator)


    return vkDestroyDebugReportCallbackEXT
def _wrap_vkDebugReportMessageEXT(fn):
    def vkDebugReportMessageEXT(
    instance
            ,flags
            ,objectType
            ,object
            ,location
            ,messageCode
            ,pLayerPrefix
            ,pMessage
            ,):
        result = _callApi(fn, instance,flags,objectType,object,location,messageCode,pLayerPrefix,pMessage)


    return vkDebugReportMessageEXT
def _wrap_vkDebugMarkerSetObjectNameEXT(fn):
    def vkDebugMarkerSetObjectNameEXT(
    device
            ,pNameInfo
            ,):
        result = _callApi(fn, device,pNameInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkDebugMarkerSetObjectNameEXT
def _wrap_vkDebugMarkerSetObjectTagEXT(fn):
    def vkDebugMarkerSetObjectTagEXT(
    device
            ,pTagInfo
            ,):
        result = _callApi(fn, device,pTagInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkDebugMarkerSetObjectTagEXT
def _wrap_vkCmdDebugMarkerBeginEXT(fn):
    def vkCmdDebugMarkerBeginEXT(
    commandBuffer
            ,pMarkerInfo
            ,):
        result = _callApi(fn, commandBuffer,pMarkerInfo)


    return vkCmdDebugMarkerBeginEXT
def _wrap_vkCmdDebugMarkerEndEXT(fn):
    def vkCmdDebugMarkerEndEXT(
    commandBuffer
            ,):
        result = _callApi(fn, commandBuffer)


    return vkCmdDebugMarkerEndEXT
def _wrap_vkCmdDebugMarkerInsertEXT(fn):
    def vkCmdDebugMarkerInsertEXT(
    commandBuffer
            ,pMarkerInfo
            ,):
        result = _callApi(fn, commandBuffer,pMarkerInfo)


    return vkCmdDebugMarkerInsertEXT
def _wrap_vkGetPhysicalDeviceExternalImageFormatPropertiesNV(fn):
    def vkGetPhysicalDeviceExternalImageFormatPropertiesNV(
    physicalDevice
            ,format
            ,type
            ,tiling
            ,usage
            ,flags
            ,externalHandleType
            ,pExternalImageFormatProperties=None
            ,):

        custom_return = True
        if not pExternalImageFormatProperties:
            pExternalImageFormatProperties = ffi.new('VkExternalImageFormatPropertiesNV*')
            custom_return = False

        result = _callApi(fn, physicalDevice,format,type,tiling,usage,flags,externalHandleType,pExternalImageFormatProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pExternalImageFormatProperties

        return pExternalImageFormatProperties[0]



    return vkGetPhysicalDeviceExternalImageFormatPropertiesNV
def _wrap_vkGetMemoryWin32HandleNV(fn):
    def vkGetMemoryWin32HandleNV(
    device
            ,memory
            ,handleType
            ,pHandle=None
            ,):

        custom_return = True
        if not pHandle:
            pHandle = ffi.new('HANDLE*')
            custom_return = False

        result = _callApi(fn, device,memory,handleType,pHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pHandle

        return pHandle[0]



    return vkGetMemoryWin32HandleNV
def _wrap_vkCmdExecuteGeneratedCommandsNV(fn):
    def vkCmdExecuteGeneratedCommandsNV(
    commandBuffer
            ,isPreprocessed
            ,pGeneratedCommandsInfo
            ,):
        result = _callApi(fn, commandBuffer,isPreprocessed,pGeneratedCommandsInfo)


    return vkCmdExecuteGeneratedCommandsNV
def _wrap_vkCmdPreprocessGeneratedCommandsNV(fn):
    def vkCmdPreprocessGeneratedCommandsNV(
    commandBuffer
            ,pGeneratedCommandsInfo
            ,):
        result = _callApi(fn, commandBuffer,pGeneratedCommandsInfo)


    return vkCmdPreprocessGeneratedCommandsNV
def _wrap_vkCmdBindPipelineShaderGroupNV(fn):
    def vkCmdBindPipelineShaderGroupNV(
    commandBuffer
            ,pipelineBindPoint
            ,pipeline
            ,groupIndex
            ,):
        result = _callApi(fn, commandBuffer,pipelineBindPoint,pipeline,groupIndex)


    return vkCmdBindPipelineShaderGroupNV
def _wrap_vkGetGeneratedCommandsMemoryRequirementsNV(fn):
    def vkGetGeneratedCommandsMemoryRequirementsNV(
    device
            ,pInfo
            ,pMemoryRequirements=None
            ,):

        custom_return = True
        if not pMemoryRequirements:
            pMemoryRequirements = ffi.new('VkMemoryRequirements2*')
            custom_return = False

        result = _callApi(fn, device,pInfo,pMemoryRequirements)

        if custom_return:
            return pMemoryRequirements

        return pMemoryRequirements[0]



    return vkGetGeneratedCommandsMemoryRequirementsNV
def _wrap_vkCreateIndirectCommandsLayoutNV(fn):
    def vkCreateIndirectCommandsLayoutNV(
    device
            ,pCreateInfo
            ,pAllocator
            ,pIndirectCommandsLayout=None
            ,):

        custom_return = True
        if not pIndirectCommandsLayout:
            pIndirectCommandsLayout = ffi.new('VkIndirectCommandsLayoutNV*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pIndirectCommandsLayout)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pIndirectCommandsLayout

        return pIndirectCommandsLayout[0]



    return vkCreateIndirectCommandsLayoutNV
def _wrap_vkDestroyIndirectCommandsLayoutNV(fn):
    def vkDestroyIndirectCommandsLayoutNV(
    device
            ,indirectCommandsLayout
            ,pAllocator
            ,):
        result = _callApi(fn, device,indirectCommandsLayout,pAllocator)


    return vkDestroyIndirectCommandsLayoutNV
def vkGetPhysicalDeviceFeatures2(
physicalDevice
        ,pFeatures=None
        ,):

    custom_return = True
    if not pFeatures:
        pFeatures = ffi.new('VkPhysicalDeviceFeatures2*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceFeatures2, physicalDevice,pFeatures)

    if custom_return:
        return pFeatures

    return pFeatures[0]



def _wrap_vkGetPhysicalDeviceFeatures2(fn):
    def vkGetPhysicalDeviceFeatures2(
    physicalDevice
            ,pFeatures=None
            ,):

        custom_return = True
        if not pFeatures:
            pFeatures = ffi.new('VkPhysicalDeviceFeatures2*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pFeatures)

        if custom_return:
            return pFeatures

        return pFeatures[0]



    return vkGetPhysicalDeviceFeatures2
def vkGetPhysicalDeviceProperties2(
physicalDevice
        ,pProperties=None
        ,):

    custom_return = True
    if not pProperties:
        pProperties = ffi.new('VkPhysicalDeviceProperties2*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceProperties2, physicalDevice,pProperties)

    if custom_return:
        return pProperties

    return pProperties[0]



def _wrap_vkGetPhysicalDeviceProperties2(fn):
    def vkGetPhysicalDeviceProperties2(
    physicalDevice
            ,pProperties=None
            ,):

        custom_return = True
        if not pProperties:
            pProperties = ffi.new('VkPhysicalDeviceProperties2*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pProperties)

        if custom_return:
            return pProperties

        return pProperties[0]



    return vkGetPhysicalDeviceProperties2
def vkGetPhysicalDeviceFormatProperties2(
physicalDevice
        ,format
        ,pFormatProperties=None
        ,):

    custom_return = True
    if not pFormatProperties:
        pFormatProperties = ffi.new('VkFormatProperties2*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceFormatProperties2, physicalDevice,format,pFormatProperties)

    if custom_return:
        return pFormatProperties

    return pFormatProperties[0]



def _wrap_vkGetPhysicalDeviceFormatProperties2(fn):
    def vkGetPhysicalDeviceFormatProperties2(
    physicalDevice
            ,format
            ,pFormatProperties=None
            ,):

        custom_return = True
        if not pFormatProperties:
            pFormatProperties = ffi.new('VkFormatProperties2*')
            custom_return = False

        result = _callApi(fn, physicalDevice,format,pFormatProperties)

        if custom_return:
            return pFormatProperties

        return pFormatProperties[0]



    return vkGetPhysicalDeviceFormatProperties2
def vkGetPhysicalDeviceImageFormatProperties2(
physicalDevice
        ,pImageFormatInfo
        ,pImageFormatProperties=None
        ,):

    custom_return = True
    if not pImageFormatProperties:
        pImageFormatProperties = ffi.new('VkImageFormatProperties2*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceImageFormatProperties2, physicalDevice,pImageFormatInfo,pImageFormatProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pImageFormatProperties

    return pImageFormatProperties[0]



def _wrap_vkGetPhysicalDeviceImageFormatProperties2(fn):
    def vkGetPhysicalDeviceImageFormatProperties2(
    physicalDevice
            ,pImageFormatInfo
            ,pImageFormatProperties=None
            ,):

        custom_return = True
        if not pImageFormatProperties:
            pImageFormatProperties = ffi.new('VkImageFormatProperties2*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pImageFormatInfo,pImageFormatProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pImageFormatProperties

        return pImageFormatProperties[0]



    return vkGetPhysicalDeviceImageFormatProperties2
def vkGetPhysicalDeviceQueueFamilyProperties2(
physicalDevice
        ,):

    pQueueFamilyPropertyCount = ffi.new('uint32_t*')
    pQueueFamilyProperties = ffi.NULL

    result = _callApi(lib.vkGetPhysicalDeviceQueueFamilyProperties2, physicalDevice,pQueueFamilyPropertyCount,pQueueFamilyProperties)

    pQueueFamilyProperties = ffi.new('VkQueueFamilyProperties2[]', pQueueFamilyPropertyCount[0])
    result = _callApi(lib.vkGetPhysicalDeviceQueueFamilyProperties2, physicalDevice,pQueueFamilyPropertyCount,pQueueFamilyProperties)

    return pQueueFamilyProperties


def _wrap_vkGetPhysicalDeviceQueueFamilyProperties2(fn):
    def vkGetPhysicalDeviceQueueFamilyProperties2(
    physicalDevice
            ,):

        pQueueFamilyPropertyCount = ffi.new('uint32_t*')
        pQueueFamilyProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pQueueFamilyPropertyCount,pQueueFamilyProperties)

        pQueueFamilyProperties = ffi.new('VkQueueFamilyProperties2[]', pQueueFamilyPropertyCount[0])
        result = _callApi(fn, physicalDevice,pQueueFamilyPropertyCount,pQueueFamilyProperties)

        return pQueueFamilyProperties


    return vkGetPhysicalDeviceQueueFamilyProperties2
def vkGetPhysicalDeviceMemoryProperties2(
physicalDevice
        ,pMemoryProperties=None
        ,):

    custom_return = True
    if not pMemoryProperties:
        pMemoryProperties = ffi.new('VkPhysicalDeviceMemoryProperties2*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceMemoryProperties2, physicalDevice,pMemoryProperties)

    if custom_return:
        return pMemoryProperties

    return pMemoryProperties[0]



def _wrap_vkGetPhysicalDeviceMemoryProperties2(fn):
    def vkGetPhysicalDeviceMemoryProperties2(
    physicalDevice
            ,pMemoryProperties=None
            ,):

        custom_return = True
        if not pMemoryProperties:
            pMemoryProperties = ffi.new('VkPhysicalDeviceMemoryProperties2*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pMemoryProperties)

        if custom_return:
            return pMemoryProperties

        return pMemoryProperties[0]



    return vkGetPhysicalDeviceMemoryProperties2
def vkGetPhysicalDeviceSparseImageFormatProperties2(
physicalDevice
        ,pFormatInfo
        ,):

    pPropertyCount = ffi.new('uint32_t*')
    pProperties = ffi.NULL

    result = _callApi(lib.vkGetPhysicalDeviceSparseImageFormatProperties2, physicalDevice,pFormatInfo,pPropertyCount,pProperties)

    pProperties = ffi.new('VkSparseImageFormatProperties2[]', pPropertyCount[0])
    result = _callApi(lib.vkGetPhysicalDeviceSparseImageFormatProperties2, physicalDevice,pFormatInfo,pPropertyCount,pProperties)

    return pProperties


def _wrap_vkGetPhysicalDeviceSparseImageFormatProperties2(fn):
    def vkGetPhysicalDeviceSparseImageFormatProperties2(
    physicalDevice
            ,pFormatInfo
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pFormatInfo,pPropertyCount,pProperties)

        pProperties = ffi.new('VkSparseImageFormatProperties2[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,pFormatInfo,pPropertyCount,pProperties)

        return pProperties


    return vkGetPhysicalDeviceSparseImageFormatProperties2
def _wrap_vkCmdPushDescriptorSetKHR(fn):
    def vkCmdPushDescriptorSetKHR(
    commandBuffer
            ,pipelineBindPoint
            ,layout
            ,set
            ,descriptorWriteCount
            ,pDescriptorWrites
            ,):
        result = _callApi(fn, commandBuffer,pipelineBindPoint,layout,set,descriptorWriteCount,pDescriptorWrites)


    return vkCmdPushDescriptorSetKHR
def vkTrimCommandPool(
device
        ,commandPool
        ,flags
        ,):
    result = _callApi(lib.vkTrimCommandPool, device,commandPool,flags)


def _wrap_vkTrimCommandPool(fn):
    def vkTrimCommandPool(
    device
            ,commandPool
            ,flags
            ,):
        result = _callApi(fn, device,commandPool,flags)


    return vkTrimCommandPool
def vkGetPhysicalDeviceExternalBufferProperties(
physicalDevice
        ,pExternalBufferInfo
        ,pExternalBufferProperties=None
        ,):

    custom_return = True
    if not pExternalBufferProperties:
        pExternalBufferProperties = ffi.new('VkExternalBufferProperties*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceExternalBufferProperties, physicalDevice,pExternalBufferInfo,pExternalBufferProperties)

    if custom_return:
        return pExternalBufferProperties

    return pExternalBufferProperties[0]



def _wrap_vkGetPhysicalDeviceExternalBufferProperties(fn):
    def vkGetPhysicalDeviceExternalBufferProperties(
    physicalDevice
            ,pExternalBufferInfo
            ,pExternalBufferProperties=None
            ,):

        custom_return = True
        if not pExternalBufferProperties:
            pExternalBufferProperties = ffi.new('VkExternalBufferProperties*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pExternalBufferInfo,pExternalBufferProperties)

        if custom_return:
            return pExternalBufferProperties

        return pExternalBufferProperties[0]



    return vkGetPhysicalDeviceExternalBufferProperties
def _wrap_vkGetMemoryWin32HandleKHR(fn):
    def vkGetMemoryWin32HandleKHR(
    device
            ,pGetWin32HandleInfo
            ,pHandle=None
            ,):

        custom_return = True
        if not pHandle:
            pHandle = ffi.new('HANDLE*')
            custom_return = False

        result = _callApi(fn, device,pGetWin32HandleInfo,pHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pHandle

        return pHandle[0]



    return vkGetMemoryWin32HandleKHR
def _wrap_vkGetMemoryWin32HandlePropertiesKHR(fn):
    def vkGetMemoryWin32HandlePropertiesKHR(
    device
            ,handleType
            ,handle
            ,pMemoryWin32HandleProperties=None
            ,):

        custom_return = True
        if not pMemoryWin32HandleProperties:
            pMemoryWin32HandleProperties = ffi.new('VkMemoryWin32HandlePropertiesKHR*')
            custom_return = False

        result = _callApi(fn, device,handleType,handle,pMemoryWin32HandleProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMemoryWin32HandleProperties

        return pMemoryWin32HandleProperties[0]



    return vkGetMemoryWin32HandlePropertiesKHR
def _wrap_vkGetMemoryFdKHR(fn):
    def vkGetMemoryFdKHR(
    device
            ,pGetFdInfo
            ,pFd=None
            ,):

        custom_return = True
        if not pFd:
            pFd = ffi.new('int*')
            custom_return = False

        result = _callApi(fn, device,pGetFdInfo,pFd)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pFd

        return pFd[0]



    return vkGetMemoryFdKHR
def _wrap_vkGetMemoryFdPropertiesKHR(fn):
    def vkGetMemoryFdPropertiesKHR(
    device
            ,handleType
            ,fd
            ,pMemoryFdProperties=None
            ,):

        custom_return = True
        if not pMemoryFdProperties:
            pMemoryFdProperties = ffi.new('VkMemoryFdPropertiesKHR*')
            custom_return = False

        result = _callApi(fn, device,handleType,fd,pMemoryFdProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMemoryFdProperties

        return pMemoryFdProperties[0]



    return vkGetMemoryFdPropertiesKHR
def _wrap_vkGetMemoryZirconHandleFUCHSIA(fn):
    def vkGetMemoryZirconHandleFUCHSIA(
    device
            ,pGetZirconHandleInfo
            ,pZirconHandle=None
            ,):

        custom_return = True
        if not pZirconHandle:
            pZirconHandle = ffi.new('zx_handle_t*')
            custom_return = False

        result = _callApi(fn, device,pGetZirconHandleInfo,pZirconHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pZirconHandle

        return pZirconHandle[0]



    return vkGetMemoryZirconHandleFUCHSIA
def _wrap_vkGetMemoryZirconHandlePropertiesFUCHSIA(fn):
    def vkGetMemoryZirconHandlePropertiesFUCHSIA(
    device
            ,handleType
            ,zirconHandle
            ,pMemoryZirconHandleProperties=None
            ,):

        custom_return = True
        if not pMemoryZirconHandleProperties:
            pMemoryZirconHandleProperties = ffi.new('VkMemoryZirconHandlePropertiesFUCHSIA*')
            custom_return = False

        result = _callApi(fn, device,handleType,zirconHandle,pMemoryZirconHandleProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMemoryZirconHandleProperties

        return pMemoryZirconHandleProperties[0]



    return vkGetMemoryZirconHandlePropertiesFUCHSIA
def _wrap_vkGetMemoryRemoteAddressNV(fn):
    def vkGetMemoryRemoteAddressNV(
    device
            ,pMemoryGetRemoteAddressInfo
            ,pAddress=None
            ,):

        custom_return = True
        if not pAddress:
            pAddress = ffi.new('VkRemoteAddressNV*')
            custom_return = False

        result = _callApi(fn, device,pMemoryGetRemoteAddressInfo,pAddress)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pAddress

        return pAddress[0]



    return vkGetMemoryRemoteAddressNV
def _wrap_vkGetMemorySciBufNV(fn):
    def vkGetMemorySciBufNV(
    device
            ,pGetSciBufInfo
            ,pHandle=None
            ,):

        custom_return = True
        if not pHandle:
            pHandle = ffi.new('NvSciBufObj*')
            custom_return = False

        result = _callApi(fn, device,pGetSciBufInfo,pHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pHandle

        return pHandle[0]



    return vkGetMemorySciBufNV
def _wrap_vkGetPhysicalDeviceExternalMemorySciBufPropertiesNV(fn):
    def vkGetPhysicalDeviceExternalMemorySciBufPropertiesNV(
    physicalDevice
            ,handleType
            ,handle
            ,pMemorySciBufProperties=None
            ,):

        custom_return = True
        if not pMemorySciBufProperties:
            pMemorySciBufProperties = ffi.new('VkMemorySciBufPropertiesNV*')
            custom_return = False

        result = _callApi(fn, physicalDevice,handleType,handle,pMemorySciBufProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMemorySciBufProperties

        return pMemorySciBufProperties[0]



    return vkGetPhysicalDeviceExternalMemorySciBufPropertiesNV
def _wrap_vkGetPhysicalDeviceSciBufAttributesNV(fn):
    def vkGetPhysicalDeviceSciBufAttributesNV(
    physicalDevice
            ,pAttributes=None
            ,):

        custom_return = True
        if not pAttributes:
            pAttributes = ffi.new('NvSciBufAttrList*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pAttributes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pAttributes

        return pAttributes[0]



    return vkGetPhysicalDeviceSciBufAttributesNV
def vkGetPhysicalDeviceExternalSemaphoreProperties(
physicalDevice
        ,pExternalSemaphoreInfo
        ,pExternalSemaphoreProperties=None
        ,):

    custom_return = True
    if not pExternalSemaphoreProperties:
        pExternalSemaphoreProperties = ffi.new('VkExternalSemaphoreProperties*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceExternalSemaphoreProperties, physicalDevice,pExternalSemaphoreInfo,pExternalSemaphoreProperties)

    if custom_return:
        return pExternalSemaphoreProperties

    return pExternalSemaphoreProperties[0]



def _wrap_vkGetPhysicalDeviceExternalSemaphoreProperties(fn):
    def vkGetPhysicalDeviceExternalSemaphoreProperties(
    physicalDevice
            ,pExternalSemaphoreInfo
            ,pExternalSemaphoreProperties=None
            ,):

        custom_return = True
        if not pExternalSemaphoreProperties:
            pExternalSemaphoreProperties = ffi.new('VkExternalSemaphoreProperties*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pExternalSemaphoreInfo,pExternalSemaphoreProperties)

        if custom_return:
            return pExternalSemaphoreProperties

        return pExternalSemaphoreProperties[0]



    return vkGetPhysicalDeviceExternalSemaphoreProperties
def _wrap_vkGetSemaphoreWin32HandleKHR(fn):
    def vkGetSemaphoreWin32HandleKHR(
    device
            ,pGetWin32HandleInfo
            ,pHandle=None
            ,):

        custom_return = True
        if not pHandle:
            pHandle = ffi.new('HANDLE*')
            custom_return = False

        result = _callApi(fn, device,pGetWin32HandleInfo,pHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pHandle

        return pHandle[0]



    return vkGetSemaphoreWin32HandleKHR
def _wrap_vkImportSemaphoreWin32HandleKHR(fn):
    def vkImportSemaphoreWin32HandleKHR(
    device
            ,pImportSemaphoreWin32HandleInfo
            ,):
        result = _callApi(fn, device,pImportSemaphoreWin32HandleInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkImportSemaphoreWin32HandleKHR
def _wrap_vkGetSemaphoreFdKHR(fn):
    def vkGetSemaphoreFdKHR(
    device
            ,pGetFdInfo
            ,pFd=None
            ,):

        custom_return = True
        if not pFd:
            pFd = ffi.new('int*')
            custom_return = False

        result = _callApi(fn, device,pGetFdInfo,pFd)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pFd

        return pFd[0]



    return vkGetSemaphoreFdKHR
def _wrap_vkImportSemaphoreFdKHR(fn):
    def vkImportSemaphoreFdKHR(
    device
            ,pImportSemaphoreFdInfo
            ,):
        result = _callApi(fn, device,pImportSemaphoreFdInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkImportSemaphoreFdKHR
def _wrap_vkGetSemaphoreZirconHandleFUCHSIA(fn):
    def vkGetSemaphoreZirconHandleFUCHSIA(
    device
            ,pGetZirconHandleInfo
            ,pZirconHandle=None
            ,):

        custom_return = True
        if not pZirconHandle:
            pZirconHandle = ffi.new('zx_handle_t*')
            custom_return = False

        result = _callApi(fn, device,pGetZirconHandleInfo,pZirconHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pZirconHandle

        return pZirconHandle[0]



    return vkGetSemaphoreZirconHandleFUCHSIA
def _wrap_vkImportSemaphoreZirconHandleFUCHSIA(fn):
    def vkImportSemaphoreZirconHandleFUCHSIA(
    device
            ,pImportSemaphoreZirconHandleInfo
            ,):
        result = _callApi(fn, device,pImportSemaphoreZirconHandleInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkImportSemaphoreZirconHandleFUCHSIA
def vkGetPhysicalDeviceExternalFenceProperties(
physicalDevice
        ,pExternalFenceInfo
        ,pExternalFenceProperties=None
        ,):

    custom_return = True
    if not pExternalFenceProperties:
        pExternalFenceProperties = ffi.new('VkExternalFenceProperties*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceExternalFenceProperties, physicalDevice,pExternalFenceInfo,pExternalFenceProperties)

    if custom_return:
        return pExternalFenceProperties

    return pExternalFenceProperties[0]



def _wrap_vkGetPhysicalDeviceExternalFenceProperties(fn):
    def vkGetPhysicalDeviceExternalFenceProperties(
    physicalDevice
            ,pExternalFenceInfo
            ,pExternalFenceProperties=None
            ,):

        custom_return = True
        if not pExternalFenceProperties:
            pExternalFenceProperties = ffi.new('VkExternalFenceProperties*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pExternalFenceInfo,pExternalFenceProperties)

        if custom_return:
            return pExternalFenceProperties

        return pExternalFenceProperties[0]



    return vkGetPhysicalDeviceExternalFenceProperties
def _wrap_vkGetFenceWin32HandleKHR(fn):
    def vkGetFenceWin32HandleKHR(
    device
            ,pGetWin32HandleInfo
            ,pHandle=None
            ,):

        custom_return = True
        if not pHandle:
            pHandle = ffi.new('HANDLE*')
            custom_return = False

        result = _callApi(fn, device,pGetWin32HandleInfo,pHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pHandle

        return pHandle[0]



    return vkGetFenceWin32HandleKHR
def _wrap_vkImportFenceWin32HandleKHR(fn):
    def vkImportFenceWin32HandleKHR(
    device
            ,pImportFenceWin32HandleInfo
            ,):
        result = _callApi(fn, device,pImportFenceWin32HandleInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkImportFenceWin32HandleKHR
def _wrap_vkGetFenceFdKHR(fn):
    def vkGetFenceFdKHR(
    device
            ,pGetFdInfo
            ,pFd=None
            ,):

        custom_return = True
        if not pFd:
            pFd = ffi.new('int*')
            custom_return = False

        result = _callApi(fn, device,pGetFdInfo,pFd)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pFd

        return pFd[0]



    return vkGetFenceFdKHR
def _wrap_vkImportFenceFdKHR(fn):
    def vkImportFenceFdKHR(
    device
            ,pImportFenceFdInfo
            ,):
        result = _callApi(fn, device,pImportFenceFdInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkImportFenceFdKHR
def _wrap_vkGetFenceSciSyncFenceNV(fn):
    def vkGetFenceSciSyncFenceNV(
    device
            ,pGetSciSyncHandleInfo
            ,pHandle=None
            ,):

        custom_return = True
        if not pHandle:
            pHandle = ffi.new('void*')
            custom_return = False

        result = _callApi(fn, device,pGetSciSyncHandleInfo,pHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pHandle

        return pHandle[0]



    return vkGetFenceSciSyncFenceNV
def _wrap_vkGetFenceSciSyncObjNV(fn):
    def vkGetFenceSciSyncObjNV(
    device
            ,pGetSciSyncHandleInfo
            ,pHandle=None
            ,):

        custom_return = True
        if not pHandle:
            pHandle = ffi.new('void*')
            custom_return = False

        result = _callApi(fn, device,pGetSciSyncHandleInfo,pHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pHandle

        return pHandle[0]



    return vkGetFenceSciSyncObjNV
def _wrap_vkImportFenceSciSyncFenceNV(fn):
    def vkImportFenceSciSyncFenceNV(
    device
            ,pImportFenceSciSyncInfo
            ,):
        result = _callApi(fn, device,pImportFenceSciSyncInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkImportFenceSciSyncFenceNV
def _wrap_vkImportFenceSciSyncObjNV(fn):
    def vkImportFenceSciSyncObjNV(
    device
            ,pImportFenceSciSyncInfo
            ,):
        result = _callApi(fn, device,pImportFenceSciSyncInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkImportFenceSciSyncObjNV
def _wrap_vkGetSemaphoreSciSyncObjNV(fn):
    def vkGetSemaphoreSciSyncObjNV(
    device
            ,pGetSciSyncInfo
            ,pHandle=None
            ,):

        custom_return = True
        if not pHandle:
            pHandle = ffi.new('void*')
            custom_return = False

        result = _callApi(fn, device,pGetSciSyncInfo,pHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pHandle

        return pHandle[0]



    return vkGetSemaphoreSciSyncObjNV
def _wrap_vkImportSemaphoreSciSyncObjNV(fn):
    def vkImportSemaphoreSciSyncObjNV(
    device
            ,pImportSemaphoreSciSyncInfo
            ,):
        result = _callApi(fn, device,pImportSemaphoreSciSyncInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkImportSemaphoreSciSyncObjNV
def _wrap_vkGetPhysicalDeviceSciSyncAttributesNV(fn):
    def vkGetPhysicalDeviceSciSyncAttributesNV(
    physicalDevice
            ,pSciSyncAttributesInfo
            ,pAttributes=None
            ,):

        custom_return = True
        if not pAttributes:
            pAttributes = ffi.new('NvSciSyncAttrList*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pSciSyncAttributesInfo,pAttributes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pAttributes

        return pAttributes[0]



    return vkGetPhysicalDeviceSciSyncAttributesNV
def _wrap_vkCreateSemaphoreSciSyncPoolNV(fn):
    def vkCreateSemaphoreSciSyncPoolNV(
    device
            ,pCreateInfo
            ,pAllocator
            ,pSemaphorePool=None
            ,):

        custom_return = True
        if not pSemaphorePool:
            pSemaphorePool = ffi.new('VkSemaphoreSciSyncPoolNV*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pSemaphorePool)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSemaphorePool

        return pSemaphorePool[0]



    return vkCreateSemaphoreSciSyncPoolNV
def _wrap_vkDestroySemaphoreSciSyncPoolNV(fn):
    def vkDestroySemaphoreSciSyncPoolNV(
    device
            ,semaphorePool
            ,pAllocator
            ,):
        result = _callApi(fn, device,semaphorePool,pAllocator)


    return vkDestroySemaphoreSciSyncPoolNV
def _wrap_vkReleaseDisplayEXT(fn):
    def vkReleaseDisplayEXT(
    physicalDevice
            ,display
            ,):
        result = _callApi(fn, physicalDevice,display)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkReleaseDisplayEXT
def _wrap_vkAcquireXlibDisplayEXT(fn):
    def vkAcquireXlibDisplayEXT(
    physicalDevice
            ,dpy
            ,display=None
            ,):

        custom_return = True
        if not display:
            display = ffi.new('VkDisplayKHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,dpy,display)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return display

        return display[0]



    return vkAcquireXlibDisplayEXT
def _wrap_vkGetRandROutputDisplayEXT(fn):
    def vkGetRandROutputDisplayEXT(
    physicalDevice
            ,dpy
            ,rrOutput
            ,pDisplay=None
            ,):

        custom_return = True
        if not pDisplay:
            pDisplay = ffi.new('VkDisplayKHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,dpy,rrOutput,pDisplay)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pDisplay

        return pDisplay[0]



    return vkGetRandROutputDisplayEXT
def _wrap_vkAcquireWinrtDisplayNV(fn):
    def vkAcquireWinrtDisplayNV(
    physicalDevice
            ,display=None
            ,):

        custom_return = True
        if not display:
            display = ffi.new('VkDisplayKHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,display)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return display

        return display[0]



    return vkAcquireWinrtDisplayNV
def _wrap_vkGetWinrtDisplayNV(fn):
    def vkGetWinrtDisplayNV(
    physicalDevice
            ,deviceRelativeId
            ,pDisplay=None
            ,):

        custom_return = True
        if not pDisplay:
            pDisplay = ffi.new('VkDisplayKHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,deviceRelativeId,pDisplay)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pDisplay

        return pDisplay[0]



    return vkGetWinrtDisplayNV
def _wrap_vkDisplayPowerControlEXT(fn):
    def vkDisplayPowerControlEXT(
    device
            ,display
            ,pDisplayPowerInfo
            ,):
        result = _callApi(fn, device,display,pDisplayPowerInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkDisplayPowerControlEXT
def _wrap_vkRegisterDeviceEventEXT(fn):
    def vkRegisterDeviceEventEXT(
    device
            ,pDeviceEventInfo
            ,pAllocator
            ,pFence
            ,):
        result = _callApi(fn, device,pDeviceEventInfo,pAllocator,pFence)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkRegisterDeviceEventEXT
def _wrap_vkRegisterDisplayEventEXT(fn):
    def vkRegisterDisplayEventEXT(
    device
            ,display
            ,pDisplayEventInfo
            ,pAllocator
            ,pFence
            ,):
        result = _callApi(fn, device,display,pDisplayEventInfo,pAllocator,pFence)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkRegisterDisplayEventEXT
def _wrap_vkGetSwapchainCounterEXT(fn):
    def vkGetSwapchainCounterEXT(
    device
            ,swapchain
            ,counter
            ,pCounterValue=None
            ,):

        custom_return = True
        if not pCounterValue:
            pCounterValue = ffi.new('uint64_t*')
            custom_return = False

        result = _callApi(fn, device,swapchain,counter,pCounterValue)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pCounterValue

        return pCounterValue[0]



    return vkGetSwapchainCounterEXT
def _wrap_vkGetPhysicalDeviceSurfaceCapabilities2EXT(fn):
    def vkGetPhysicalDeviceSurfaceCapabilities2EXT(
    physicalDevice
            ,surface
            ,pSurfaceCapabilities=None
            ,):

        custom_return = True
        if not pSurfaceCapabilities:
            pSurfaceCapabilities = ffi.new('VkSurfaceCapabilities2EXT*')
            custom_return = False

        result = _callApi(fn, physicalDevice,surface,pSurfaceCapabilities)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurfaceCapabilities

        return pSurfaceCapabilities[0]



    return vkGetPhysicalDeviceSurfaceCapabilities2EXT
def vkEnumeratePhysicalDeviceGroups(
instance
        ,):

    pPhysicalDeviceGroupCount = ffi.new('uint32_t*')
    pPhysicalDeviceGroupProperties = ffi.NULL

    result = _callApi(lib.vkEnumeratePhysicalDeviceGroups, instance,pPhysicalDeviceGroupCount,pPhysicalDeviceGroupProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pPhysicalDeviceGroupProperties = ffi.new('VkPhysicalDeviceGroupProperties[]', pPhysicalDeviceGroupCount[0])
    result = _callApi(lib.vkEnumeratePhysicalDeviceGroups, instance,pPhysicalDeviceGroupCount,pPhysicalDeviceGroupProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    return pPhysicalDeviceGroupProperties


def _wrap_vkEnumeratePhysicalDeviceGroups(fn):
    def vkEnumeratePhysicalDeviceGroups(
    instance
            ,):

        pPhysicalDeviceGroupCount = ffi.new('uint32_t*')
        pPhysicalDeviceGroupProperties = ffi.NULL

        result = _callApi(fn, instance,pPhysicalDeviceGroupCount,pPhysicalDeviceGroupProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pPhysicalDeviceGroupProperties = ffi.new('VkPhysicalDeviceGroupProperties[]', pPhysicalDeviceGroupCount[0])
        result = _callApi(fn, instance,pPhysicalDeviceGroupCount,pPhysicalDeviceGroupProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pPhysicalDeviceGroupProperties


    return vkEnumeratePhysicalDeviceGroups
def vkGetDeviceGroupPeerMemoryFeatures(
device
        ,heapIndex
        ,localDeviceIndex
        ,remoteDeviceIndex
        ,pPeerMemoryFeatures=None
        ,):

    custom_return = True
    if not pPeerMemoryFeatures:
        pPeerMemoryFeatures = ffi.new('VkPeerMemoryFeatureFlags*')
        custom_return = False

    result = _callApi(lib.vkGetDeviceGroupPeerMemoryFeatures, device,heapIndex,localDeviceIndex,remoteDeviceIndex,pPeerMemoryFeatures)

    if custom_return:
        return pPeerMemoryFeatures

    return pPeerMemoryFeatures[0]



def _wrap_vkGetDeviceGroupPeerMemoryFeatures(fn):
    def vkGetDeviceGroupPeerMemoryFeatures(
    device
            ,heapIndex
            ,localDeviceIndex
            ,remoteDeviceIndex
            ,pPeerMemoryFeatures=None
            ,):

        custom_return = True
        if not pPeerMemoryFeatures:
            pPeerMemoryFeatures = ffi.new('VkPeerMemoryFeatureFlags*')
            custom_return = False

        result = _callApi(fn, device,heapIndex,localDeviceIndex,remoteDeviceIndex,pPeerMemoryFeatures)

        if custom_return:
            return pPeerMemoryFeatures

        return pPeerMemoryFeatures[0]



    return vkGetDeviceGroupPeerMemoryFeatures
def vkBindBufferMemory2(
device
        ,bindInfoCount
        ,pBindInfos
        ,):
    result = _callApi(lib.vkBindBufferMemory2, device,bindInfoCount,pBindInfos)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def _wrap_vkBindBufferMemory2(fn):
    def vkBindBufferMemory2(
    device
            ,bindInfoCount
            ,pBindInfos
            ,):
        result = _callApi(fn, device,bindInfoCount,pBindInfos)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkBindBufferMemory2
def vkBindImageMemory2(
device
        ,bindInfoCount
        ,pBindInfos
        ,):
    result = _callApi(lib.vkBindImageMemory2, device,bindInfoCount,pBindInfos)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def _wrap_vkBindImageMemory2(fn):
    def vkBindImageMemory2(
    device
            ,bindInfoCount
            ,pBindInfos
            ,):
        result = _callApi(fn, device,bindInfoCount,pBindInfos)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkBindImageMemory2
def vkCmdSetDeviceMask(
commandBuffer
        ,deviceMask
        ,):
    result = _callApi(lib.vkCmdSetDeviceMask, commandBuffer,deviceMask)


def _wrap_vkCmdSetDeviceMask(fn):
    def vkCmdSetDeviceMask(
    commandBuffer
            ,deviceMask
            ,):
        result = _callApi(fn, commandBuffer,deviceMask)


    return vkCmdSetDeviceMask
def _wrap_vkGetDeviceGroupPresentCapabilitiesKHR(fn):
    def vkGetDeviceGroupPresentCapabilitiesKHR(
    device
            ,pDeviceGroupPresentCapabilities=None
            ,):

        custom_return = True
        if not pDeviceGroupPresentCapabilities:
            pDeviceGroupPresentCapabilities = ffi.new('VkDeviceGroupPresentCapabilitiesKHR*')
            custom_return = False

        result = _callApi(fn, device,pDeviceGroupPresentCapabilities)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pDeviceGroupPresentCapabilities

        return pDeviceGroupPresentCapabilities[0]



    return vkGetDeviceGroupPresentCapabilitiesKHR
def _wrap_vkGetDeviceGroupSurfacePresentModesKHR(fn):
    def vkGetDeviceGroupSurfacePresentModesKHR(
    device
            ,surface
            ,pModes=None
            ,):

        custom_return = True
        if not pModes:
            pModes = ffi.new('VkDeviceGroupPresentModeFlagsKHR*')
            custom_return = False

        result = _callApi(fn, device,surface,pModes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pModes

        return pModes[0]



    return vkGetDeviceGroupSurfacePresentModesKHR
def _wrap_vkAcquireNextImage2KHR(fn):
    def vkAcquireNextImage2KHR(
    device
            ,):

        pAcquireInfo = ffi.new('VkAcquireNextImageInfoKHR*')
        pImageIndex = ffi.NULL

        result = _callApi(fn, device,pAcquireInfo,pImageIndex)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pImageIndex = ffi.new('uint32_t[]', pAcquireInfo[0])
        result = _callApi(fn, device,pAcquireInfo,pImageIndex)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pImageIndex


    return vkAcquireNextImage2KHR
def vkCmdDispatchBase(
commandBuffer
        ,baseGroupX
        ,baseGroupY
        ,baseGroupZ
        ,groupCountX
        ,groupCountY
        ,groupCountZ
        ,):
    result = _callApi(lib.vkCmdDispatchBase, commandBuffer,baseGroupX,baseGroupY,baseGroupZ,groupCountX,groupCountY,groupCountZ)


def _wrap_vkCmdDispatchBase(fn):
    def vkCmdDispatchBase(
    commandBuffer
            ,baseGroupX
            ,baseGroupY
            ,baseGroupZ
            ,groupCountX
            ,groupCountY
            ,groupCountZ
            ,):
        result = _callApi(fn, commandBuffer,baseGroupX,baseGroupY,baseGroupZ,groupCountX,groupCountY,groupCountZ)


    return vkCmdDispatchBase
def _wrap_vkGetPhysicalDevicePresentRectanglesKHR(fn):
    def vkGetPhysicalDevicePresentRectanglesKHR(
    physicalDevice
            ,surface
            ,):

        pRectCount = ffi.new('uint32_t*')
        pRects = ffi.NULL

        result = _callApi(fn, physicalDevice,surface,pRectCount,pRects)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pRects = ffi.new('VkRect2D[]', pRectCount[0])
        result = _callApi(fn, physicalDevice,surface,pRectCount,pRects)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pRects


    return vkGetPhysicalDevicePresentRectanglesKHR
def vkCreateDescriptorUpdateTemplate(
device
        ,pCreateInfo
        ,pAllocator
        ,pDescriptorUpdateTemplate=None
        ,):

    custom_return = True
    if not pDescriptorUpdateTemplate:
        pDescriptorUpdateTemplate = ffi.new('VkDescriptorUpdateTemplate*')
        custom_return = False

    result = _callApi(lib.vkCreateDescriptorUpdateTemplate, device,pCreateInfo,pAllocator,pDescriptorUpdateTemplate)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pDescriptorUpdateTemplate

    return pDescriptorUpdateTemplate[0]



def _wrap_vkCreateDescriptorUpdateTemplate(fn):
    def vkCreateDescriptorUpdateTemplate(
    device
            ,pCreateInfo
            ,pAllocator
            ,pDescriptorUpdateTemplate=None
            ,):

        custom_return = True
        if not pDescriptorUpdateTemplate:
            pDescriptorUpdateTemplate = ffi.new('VkDescriptorUpdateTemplate*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pDescriptorUpdateTemplate)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pDescriptorUpdateTemplate

        return pDescriptorUpdateTemplate[0]



    return vkCreateDescriptorUpdateTemplate
def vkDestroyDescriptorUpdateTemplate(
device
        ,descriptorUpdateTemplate
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyDescriptorUpdateTemplate, device,descriptorUpdateTemplate,pAllocator)


def _wrap_vkDestroyDescriptorUpdateTemplate(fn):
    def vkDestroyDescriptorUpdateTemplate(
    device
            ,descriptorUpdateTemplate
            ,pAllocator
            ,):
        result = _callApi(fn, device,descriptorUpdateTemplate,pAllocator)


    return vkDestroyDescriptorUpdateTemplate
def vkUpdateDescriptorSetWithTemplate(
device
        ,descriptorSet
        ,descriptorUpdateTemplate
        ,pData
        ,):
    result = _callApi(lib.vkUpdateDescriptorSetWithTemplate, device,descriptorSet,descriptorUpdateTemplate,pData)


def _wrap_vkUpdateDescriptorSetWithTemplate(fn):
    def vkUpdateDescriptorSetWithTemplate(
    device
            ,descriptorSet
            ,descriptorUpdateTemplate
            ,pData
            ,):
        result = _callApi(fn, device,descriptorSet,descriptorUpdateTemplate,pData)


    return vkUpdateDescriptorSetWithTemplate
def _wrap_vkCmdPushDescriptorSetWithTemplateKHR(fn):
    def vkCmdPushDescriptorSetWithTemplateKHR(
    commandBuffer
            ,descriptorUpdateTemplate
            ,layout
            ,set
            ,pData
            ,):
        result = _callApi(fn, commandBuffer,descriptorUpdateTemplate,layout,set,pData)


    return vkCmdPushDescriptorSetWithTemplateKHR
def _wrap_vkSetHdrMetadataEXT(fn):
    def vkSetHdrMetadataEXT(
    device
            ,swapchainCount
            ,pSwapchains
            ,pMetadata
            ,):
        result = _callApi(fn, device,swapchainCount,pSwapchains,pMetadata)


    return vkSetHdrMetadataEXT
def _wrap_vkGetSwapchainStatusKHR(fn):
    def vkGetSwapchainStatusKHR(
    device
            ,swapchain=None
            ,):

        custom_return = True
        if not swapchain:
            swapchain = ffi.new('VkSwapchainKHR*')
            custom_return = False

        result = _callApi(fn, device,swapchain)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return swapchain

        return swapchain[0]



    return vkGetSwapchainStatusKHR
def _wrap_vkGetRefreshCycleDurationGOOGLE(fn):
    def vkGetRefreshCycleDurationGOOGLE(
    device
            ,swapchain
            ,pDisplayTimingProperties=None
            ,):

        custom_return = True
        if not pDisplayTimingProperties:
            pDisplayTimingProperties = ffi.new('VkRefreshCycleDurationGOOGLE*')
            custom_return = False

        result = _callApi(fn, device,swapchain,pDisplayTimingProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pDisplayTimingProperties

        return pDisplayTimingProperties[0]



    return vkGetRefreshCycleDurationGOOGLE
def _wrap_vkGetPastPresentationTimingGOOGLE(fn):
    def vkGetPastPresentationTimingGOOGLE(
    device
            ,swapchain
            ,):

        pPresentationTimingCount = ffi.new('uint32_t*')
        pPresentationTimings = ffi.NULL

        result = _callApi(fn, device,swapchain,pPresentationTimingCount,pPresentationTimings)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pPresentationTimings = ffi.new('VkPastPresentationTimingGOOGLE[]', pPresentationTimingCount[0])
        result = _callApi(fn, device,swapchain,pPresentationTimingCount,pPresentationTimings)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pPresentationTimings


    return vkGetPastPresentationTimingGOOGLE
def _wrap_vkCreateIOSSurfaceMVK(fn):
    def vkCreateIOSSurfaceMVK(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateIOSSurfaceMVK
def _wrap_vkCreateMacOSSurfaceMVK(fn):
    def vkCreateMacOSSurfaceMVK(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateMacOSSurfaceMVK
def _wrap_vkCreateMetalSurfaceEXT(fn):
    def vkCreateMetalSurfaceEXT(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateMetalSurfaceEXT
def _wrap_vkCmdSetViewportWScalingNV(fn):
    def vkCmdSetViewportWScalingNV(
    commandBuffer
            ,firstViewport
            ,viewportCount
            ,pViewportWScalings
            ,):
        result = _callApi(fn, commandBuffer,firstViewport,viewportCount,pViewportWScalings)


    return vkCmdSetViewportWScalingNV
def _wrap_vkCmdSetDiscardRectangleEXT(fn):
    def vkCmdSetDiscardRectangleEXT(
    commandBuffer
            ,firstDiscardRectangle
            ,discardRectangleCount
            ,pDiscardRectangles
            ,):
        result = _callApi(fn, commandBuffer,firstDiscardRectangle,discardRectangleCount,pDiscardRectangles)


    return vkCmdSetDiscardRectangleEXT
def _wrap_vkCmdSetDiscardRectangleEnableEXT(fn):
    def vkCmdSetDiscardRectangleEnableEXT(
    commandBuffer
            ,discardRectangleEnable
            ,):
        result = _callApi(fn, commandBuffer,discardRectangleEnable)


    return vkCmdSetDiscardRectangleEnableEXT
def _wrap_vkCmdSetDiscardRectangleModeEXT(fn):
    def vkCmdSetDiscardRectangleModeEXT(
    commandBuffer
            ,discardRectangleMode
            ,):
        result = _callApi(fn, commandBuffer,discardRectangleMode)


    return vkCmdSetDiscardRectangleModeEXT
def _wrap_vkCmdSetSampleLocationsEXT(fn):
    def vkCmdSetSampleLocationsEXT(
    commandBuffer
            ,pSampleLocationsInfo
            ,):
        result = _callApi(fn, commandBuffer,pSampleLocationsInfo)


    return vkCmdSetSampleLocationsEXT
def _wrap_vkGetPhysicalDeviceMultisamplePropertiesEXT(fn):
    def vkGetPhysicalDeviceMultisamplePropertiesEXT(
    physicalDevice
            ,samples
            ,pMultisampleProperties=None
            ,):

        custom_return = True
        if not pMultisampleProperties:
            pMultisampleProperties = ffi.new('VkMultisamplePropertiesEXT*')
            custom_return = False

        result = _callApi(fn, physicalDevice,samples,pMultisampleProperties)

        if custom_return:
            return pMultisampleProperties

        return pMultisampleProperties[0]



    return vkGetPhysicalDeviceMultisamplePropertiesEXT
def _wrap_vkGetPhysicalDeviceSurfaceCapabilities2KHR(fn):
    def vkGetPhysicalDeviceSurfaceCapabilities2KHR(
    physicalDevice
            ,pSurfaceInfo
            ,pSurfaceCapabilities=None
            ,):

        custom_return = True
        if not pSurfaceCapabilities:
            pSurfaceCapabilities = ffi.new('VkSurfaceCapabilities2KHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pSurfaceInfo,pSurfaceCapabilities)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurfaceCapabilities

        return pSurfaceCapabilities[0]



    return vkGetPhysicalDeviceSurfaceCapabilities2KHR
def _wrap_vkGetPhysicalDeviceSurfaceFormats2KHR(fn):
    def vkGetPhysicalDeviceSurfaceFormats2KHR(
    physicalDevice
            ,pSurfaceInfo
            ,):

        pSurfaceFormatCount = ffi.new('uint32_t*')
        pSurfaceFormats = ffi.NULL

        result = _callApi(fn, physicalDevice,pSurfaceInfo,pSurfaceFormatCount,pSurfaceFormats)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pSurfaceFormats = ffi.new('VkSurfaceFormat2KHR[]', pSurfaceFormatCount[0])
        result = _callApi(fn, physicalDevice,pSurfaceInfo,pSurfaceFormatCount,pSurfaceFormats)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pSurfaceFormats


    return vkGetPhysicalDeviceSurfaceFormats2KHR
def _wrap_vkGetPhysicalDeviceDisplayProperties2KHR(fn):
    def vkGetPhysicalDeviceDisplayProperties2KHR(
    physicalDevice
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkDisplayProperties2KHR[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pProperties


    return vkGetPhysicalDeviceDisplayProperties2KHR
def _wrap_vkGetPhysicalDeviceDisplayPlaneProperties2KHR(fn):
    def vkGetPhysicalDeviceDisplayPlaneProperties2KHR(
    physicalDevice
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkDisplayPlaneProperties2KHR[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pProperties


    return vkGetPhysicalDeviceDisplayPlaneProperties2KHR
def _wrap_vkGetDisplayModeProperties2KHR(fn):
    def vkGetDisplayModeProperties2KHR(
    physicalDevice
            ,display
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,display,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkDisplayModeProperties2KHR[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,display,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pProperties


    return vkGetDisplayModeProperties2KHR
def _wrap_vkGetDisplayPlaneCapabilities2KHR(fn):
    def vkGetDisplayPlaneCapabilities2KHR(
    physicalDevice
            ,pDisplayPlaneInfo
            ,pCapabilities=None
            ,):

        custom_return = True
        if not pCapabilities:
            pCapabilities = ffi.new('VkDisplayPlaneCapabilities2KHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pDisplayPlaneInfo,pCapabilities)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pCapabilities

        return pCapabilities[0]



    return vkGetDisplayPlaneCapabilities2KHR
def vkGetBufferMemoryRequirements2(
device
        ,pInfo
        ,pMemoryRequirements=None
        ,):

    custom_return = True
    if not pMemoryRequirements:
        pMemoryRequirements = ffi.new('VkMemoryRequirements2*')
        custom_return = False

    result = _callApi(lib.vkGetBufferMemoryRequirements2, device,pInfo,pMemoryRequirements)

    if custom_return:
        return pMemoryRequirements

    return pMemoryRequirements[0]



def _wrap_vkGetBufferMemoryRequirements2(fn):
    def vkGetBufferMemoryRequirements2(
    device
            ,pInfo
            ,pMemoryRequirements=None
            ,):

        custom_return = True
        if not pMemoryRequirements:
            pMemoryRequirements = ffi.new('VkMemoryRequirements2*')
            custom_return = False

        result = _callApi(fn, device,pInfo,pMemoryRequirements)

        if custom_return:
            return pMemoryRequirements

        return pMemoryRequirements[0]



    return vkGetBufferMemoryRequirements2
def vkGetImageMemoryRequirements2(
device
        ,pInfo
        ,pMemoryRequirements=None
        ,):

    custom_return = True
    if not pMemoryRequirements:
        pMemoryRequirements = ffi.new('VkMemoryRequirements2*')
        custom_return = False

    result = _callApi(lib.vkGetImageMemoryRequirements2, device,pInfo,pMemoryRequirements)

    if custom_return:
        return pMemoryRequirements

    return pMemoryRequirements[0]



def _wrap_vkGetImageMemoryRequirements2(fn):
    def vkGetImageMemoryRequirements2(
    device
            ,pInfo
            ,pMemoryRequirements=None
            ,):

        custom_return = True
        if not pMemoryRequirements:
            pMemoryRequirements = ffi.new('VkMemoryRequirements2*')
            custom_return = False

        result = _callApi(fn, device,pInfo,pMemoryRequirements)

        if custom_return:
            return pMemoryRequirements

        return pMemoryRequirements[0]



    return vkGetImageMemoryRequirements2
def vkGetImageSparseMemoryRequirements2(
device
        ,pInfo
        ,):

    pSparseMemoryRequirementCount = ffi.new('uint32_t*')
    pSparseMemoryRequirements = ffi.NULL

    result = _callApi(lib.vkGetImageSparseMemoryRequirements2, device,pInfo,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

    pSparseMemoryRequirements = ffi.new('VkSparseImageMemoryRequirements2[]', pSparseMemoryRequirementCount[0])
    result = _callApi(lib.vkGetImageSparseMemoryRequirements2, device,pInfo,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

    return pSparseMemoryRequirements


def _wrap_vkGetImageSparseMemoryRequirements2(fn):
    def vkGetImageSparseMemoryRequirements2(
    device
            ,pInfo
            ,):

        pSparseMemoryRequirementCount = ffi.new('uint32_t*')
        pSparseMemoryRequirements = ffi.NULL

        result = _callApi(fn, device,pInfo,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

        pSparseMemoryRequirements = ffi.new('VkSparseImageMemoryRequirements2[]', pSparseMemoryRequirementCount[0])
        result = _callApi(fn, device,pInfo,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

        return pSparseMemoryRequirements


    return vkGetImageSparseMemoryRequirements2
def vkGetDeviceBufferMemoryRequirements(
device
        ,pInfo
        ,pMemoryRequirements=None
        ,):

    custom_return = True
    if not pMemoryRequirements:
        pMemoryRequirements = ffi.new('VkMemoryRequirements2*')
        custom_return = False

    result = _callApi(lib.vkGetDeviceBufferMemoryRequirements, device,pInfo,pMemoryRequirements)

    if custom_return:
        return pMemoryRequirements

    return pMemoryRequirements[0]



def _wrap_vkGetDeviceBufferMemoryRequirements(fn):
    def vkGetDeviceBufferMemoryRequirements(
    device
            ,pInfo
            ,pMemoryRequirements=None
            ,):

        custom_return = True
        if not pMemoryRequirements:
            pMemoryRequirements = ffi.new('VkMemoryRequirements2*')
            custom_return = False

        result = _callApi(fn, device,pInfo,pMemoryRequirements)

        if custom_return:
            return pMemoryRequirements

        return pMemoryRequirements[0]



    return vkGetDeviceBufferMemoryRequirements
def vkGetDeviceImageMemoryRequirements(
device
        ,pInfo
        ,pMemoryRequirements=None
        ,):

    custom_return = True
    if not pMemoryRequirements:
        pMemoryRequirements = ffi.new('VkMemoryRequirements2*')
        custom_return = False

    result = _callApi(lib.vkGetDeviceImageMemoryRequirements, device,pInfo,pMemoryRequirements)

    if custom_return:
        return pMemoryRequirements

    return pMemoryRequirements[0]



def _wrap_vkGetDeviceImageMemoryRequirements(fn):
    def vkGetDeviceImageMemoryRequirements(
    device
            ,pInfo
            ,pMemoryRequirements=None
            ,):

        custom_return = True
        if not pMemoryRequirements:
            pMemoryRequirements = ffi.new('VkMemoryRequirements2*')
            custom_return = False

        result = _callApi(fn, device,pInfo,pMemoryRequirements)

        if custom_return:
            return pMemoryRequirements

        return pMemoryRequirements[0]



    return vkGetDeviceImageMemoryRequirements
def vkGetDeviceImageSparseMemoryRequirements(
device
        ,pInfo
        ,):

    pSparseMemoryRequirementCount = ffi.new('uint32_t*')
    pSparseMemoryRequirements = ffi.NULL

    result = _callApi(lib.vkGetDeviceImageSparseMemoryRequirements, device,pInfo,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

    pSparseMemoryRequirements = ffi.new('VkSparseImageMemoryRequirements2[]', pSparseMemoryRequirementCount[0])
    result = _callApi(lib.vkGetDeviceImageSparseMemoryRequirements, device,pInfo,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

    return pSparseMemoryRequirements


def _wrap_vkGetDeviceImageSparseMemoryRequirements(fn):
    def vkGetDeviceImageSparseMemoryRequirements(
    device
            ,pInfo
            ,):

        pSparseMemoryRequirementCount = ffi.new('uint32_t*')
        pSparseMemoryRequirements = ffi.NULL

        result = _callApi(fn, device,pInfo,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

        pSparseMemoryRequirements = ffi.new('VkSparseImageMemoryRequirements2[]', pSparseMemoryRequirementCount[0])
        result = _callApi(fn, device,pInfo,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

        return pSparseMemoryRequirements


    return vkGetDeviceImageSparseMemoryRequirements
def vkCreateSamplerYcbcrConversion(
device
        ,pCreateInfo
        ,pAllocator
        ,pYcbcrConversion=None
        ,):

    custom_return = True
    if not pYcbcrConversion:
        pYcbcrConversion = ffi.new('VkSamplerYcbcrConversion*')
        custom_return = False

    result = _callApi(lib.vkCreateSamplerYcbcrConversion, device,pCreateInfo,pAllocator,pYcbcrConversion)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pYcbcrConversion

    return pYcbcrConversion[0]



def _wrap_vkCreateSamplerYcbcrConversion(fn):
    def vkCreateSamplerYcbcrConversion(
    device
            ,pCreateInfo
            ,pAllocator
            ,pYcbcrConversion=None
            ,):

        custom_return = True
        if not pYcbcrConversion:
            pYcbcrConversion = ffi.new('VkSamplerYcbcrConversion*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pYcbcrConversion)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pYcbcrConversion

        return pYcbcrConversion[0]



    return vkCreateSamplerYcbcrConversion
def vkDestroySamplerYcbcrConversion(
device
        ,ycbcrConversion
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroySamplerYcbcrConversion, device,ycbcrConversion,pAllocator)


def _wrap_vkDestroySamplerYcbcrConversion(fn):
    def vkDestroySamplerYcbcrConversion(
    device
            ,ycbcrConversion
            ,pAllocator
            ,):
        result = _callApi(fn, device,ycbcrConversion,pAllocator)


    return vkDestroySamplerYcbcrConversion
def vkGetDeviceQueue2(
device
        ,pQueueInfo
        ,pQueue=None
        ,):

    custom_return = True
    if not pQueue:
        pQueue = ffi.new('VkQueue*')
        custom_return = False

    result = _callApi(lib.vkGetDeviceQueue2, device,pQueueInfo,pQueue)

    if custom_return:
        return pQueue

    return pQueue[0]



def _wrap_vkCreateValidationCacheEXT(fn):
    def vkCreateValidationCacheEXT(
    device
            ,pCreateInfo
            ,pAllocator
            ,pValidationCache=None
            ,):

        custom_return = True
        if not pValidationCache:
            pValidationCache = ffi.new('VkValidationCacheEXT*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pValidationCache)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pValidationCache

        return pValidationCache[0]



    return vkCreateValidationCacheEXT
def _wrap_vkDestroyValidationCacheEXT(fn):
    def vkDestroyValidationCacheEXT(
    device
            ,validationCache
            ,pAllocator
            ,):
        result = _callApi(fn, device,validationCache,pAllocator)


    return vkDestroyValidationCacheEXT
def _wrap_vkGetValidationCacheDataEXT(fn):
    def vkGetValidationCacheDataEXT(
    device
            ,validationCache
            ,pDataSize
            ,pData=None
            ,):

        custom_return = True
        if not pData:
            pData = ffi.new('void[%d]' % pDataSize)
            custom_return = False

        result = _callApi(fn, device,validationCache,pDataSize,pData)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pData

        return pData



    return vkGetValidationCacheDataEXT
def _wrap_vkMergeValidationCachesEXT(fn):
    def vkMergeValidationCachesEXT(
    device
            ,dstCache
            ,srcCacheCount
            ,pSrcCaches
            ,):
        result = _callApi(fn, device,dstCache,srcCacheCount,pSrcCaches)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkMergeValidationCachesEXT
def vkGetDescriptorSetLayoutSupport(
device
        ,pCreateInfo
        ,pSupport=None
        ,):

    custom_return = True
    if not pSupport:
        pSupport = ffi.new('VkDescriptorSetLayoutSupport*')
        custom_return = False

    result = _callApi(lib.vkGetDescriptorSetLayoutSupport, device,pCreateInfo,pSupport)

    if custom_return:
        return pSupport

    return pSupport[0]



def _wrap_vkGetDescriptorSetLayoutSupport(fn):
    def vkGetDescriptorSetLayoutSupport(
    device
            ,pCreateInfo
            ,pSupport=None
            ,):

        custom_return = True
        if not pSupport:
            pSupport = ffi.new('VkDescriptorSetLayoutSupport*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pSupport)

        if custom_return:
            return pSupport

        return pSupport[0]



    return vkGetDescriptorSetLayoutSupport
def _wrap_vkGetSwapchainGrallocUsageANDROID(fn):
    def vkGetSwapchainGrallocUsageANDROID(
    device
            ,format
            ,imageUsage
            ,grallocUsage=None
            ,):

        custom_return = True
        if not grallocUsage:
            grallocUsage = ffi.new('int*')
            custom_return = False

        result = _callApi(fn, device,format,imageUsage,grallocUsage)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return grallocUsage

        return grallocUsage[0]



    return vkGetSwapchainGrallocUsageANDROID
def _wrap_vkGetSwapchainGrallocUsage2ANDROID(fn):
    def vkGetSwapchainGrallocUsage2ANDROID(
    device
            ,format
            ,imageUsage
            ,swapchainImageUsage
            ,grallocConsumerUsage
            ,grallocProducerUsage=None
            ,):

        custom_return = True
        if not grallocProducerUsage:
            grallocProducerUsage = ffi.new('uint64_t*')
            custom_return = False

        result = _callApi(fn, device,format,imageUsage,swapchainImageUsage,grallocConsumerUsage,grallocProducerUsage)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return grallocProducerUsage

        return grallocProducerUsage[0]



    return vkGetSwapchainGrallocUsage2ANDROID
def _wrap_vkAcquireImageANDROID(fn):
    def vkAcquireImageANDROID(
    device
            ,image
            ,nativeFenceFd
            ,semaphore
            ,fence=None
            ,):

        custom_return = True
        if not fence:
            fence = ffi.new('VkFence*')
            custom_return = False

        result = _callApi(fn, device,image,nativeFenceFd,semaphore,fence)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return fence

        return fence[0]



    return vkAcquireImageANDROID
def _wrap_vkQueueSignalReleaseImageANDROID(fn):
    def vkQueueSignalReleaseImageANDROID(
    queue
            ,waitSemaphoreCount
            ,pWaitSemaphores
            ,image
            ,pNativeFenceFd
            ,):
        result = _callApi(fn, queue,waitSemaphoreCount,pWaitSemaphores,image,pNativeFenceFd)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkQueueSignalReleaseImageANDROID
def _wrap_vkGetShaderInfoAMD(fn):
    def vkGetShaderInfoAMD(
    device
            ,pipeline
            ,shaderStage
            ,infoType
            ,pInfoSize
            ,pInfo=None
            ,):

        custom_return = True
        if not pInfo:
            pInfo = ffi.new('void[%d]' % pInfoSize)
            custom_return = False

        result = _callApi(fn, device,pipeline,shaderStage,infoType,pInfoSize,pInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pInfo

        return pInfo



    return vkGetShaderInfoAMD
def _wrap_vkSetLocalDimmingAMD(fn):
    def vkSetLocalDimmingAMD(
    device
            ,swapChain
            ,localDimmingEnable
            ,):
        result = _callApi(fn, device,swapChain,localDimmingEnable)


    return vkSetLocalDimmingAMD
def _wrap_vkGetPhysicalDeviceCalibrateableTimeDomainsKHR(fn):
    def vkGetPhysicalDeviceCalibrateableTimeDomainsKHR(
    physicalDevice
            ,):

        pTimeDomainCount = ffi.new('uint32_t*')
        pTimeDomains = ffi.NULL

        result = _callApi(fn, physicalDevice,pTimeDomainCount,pTimeDomains)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pTimeDomains = ffi.new('VkTimeDomainKHR[]', pTimeDomainCount[0])
        result = _callApi(fn, physicalDevice,pTimeDomainCount,pTimeDomains)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pTimeDomains


    return vkGetPhysicalDeviceCalibrateableTimeDomainsKHR
def _wrap_vkGetCalibratedTimestampsKHR(fn):
    def vkGetCalibratedTimestampsKHR(
    device
            ,timestampCount
            ,pTimestampInfos
            ,pTimestamps
            ,pMaxDeviation=None
            ,):

        custom_return = True
        if not pMaxDeviation:
            pMaxDeviation = ffi.new('uint64_t*')
            custom_return = False

        result = _callApi(fn, device,timestampCount,pTimestampInfos,pTimestamps,pMaxDeviation)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMaxDeviation

        return pMaxDeviation[0]



    return vkGetCalibratedTimestampsKHR
def _wrap_vkSetDebugUtilsObjectNameEXT(fn):
    def vkSetDebugUtilsObjectNameEXT(
    device
            ,pNameInfo
            ,):
        result = _callApi(fn, device,pNameInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkSetDebugUtilsObjectNameEXT
def _wrap_vkSetDebugUtilsObjectTagEXT(fn):
    def vkSetDebugUtilsObjectTagEXT(
    device
            ,pTagInfo
            ,):
        result = _callApi(fn, device,pTagInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkSetDebugUtilsObjectTagEXT
def _wrap_vkQueueBeginDebugUtilsLabelEXT(fn):
    def vkQueueBeginDebugUtilsLabelEXT(
    queue
            ,pLabelInfo
            ,):
        result = _callApi(fn, queue,pLabelInfo)


    return vkQueueBeginDebugUtilsLabelEXT
def _wrap_vkQueueEndDebugUtilsLabelEXT(fn):
    def vkQueueEndDebugUtilsLabelEXT(
    queue
            ,):
        result = _callApi(fn, queue)


    return vkQueueEndDebugUtilsLabelEXT
def _wrap_vkQueueInsertDebugUtilsLabelEXT(fn):
    def vkQueueInsertDebugUtilsLabelEXT(
    queue
            ,pLabelInfo
            ,):
        result = _callApi(fn, queue,pLabelInfo)


    return vkQueueInsertDebugUtilsLabelEXT
def _wrap_vkCmdBeginDebugUtilsLabelEXT(fn):
    def vkCmdBeginDebugUtilsLabelEXT(
    commandBuffer
            ,pLabelInfo
            ,):
        result = _callApi(fn, commandBuffer,pLabelInfo)


    return vkCmdBeginDebugUtilsLabelEXT
def _wrap_vkCmdEndDebugUtilsLabelEXT(fn):
    def vkCmdEndDebugUtilsLabelEXT(
    commandBuffer
            ,):
        result = _callApi(fn, commandBuffer)


    return vkCmdEndDebugUtilsLabelEXT
def _wrap_vkCmdInsertDebugUtilsLabelEXT(fn):
    def vkCmdInsertDebugUtilsLabelEXT(
    commandBuffer
            ,pLabelInfo
            ,):
        result = _callApi(fn, commandBuffer,pLabelInfo)


    return vkCmdInsertDebugUtilsLabelEXT
def _wrap_vkCreateDebugUtilsMessengerEXT(fn):
    def vkCreateDebugUtilsMessengerEXT(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pMessenger=None
            ,):

        custom_return = True
        if not pMessenger:
            pMessenger = ffi.new('VkDebugUtilsMessengerEXT*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pMessenger)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMessenger

        return pMessenger[0]



    return vkCreateDebugUtilsMessengerEXT
def _wrap_vkDestroyDebugUtilsMessengerEXT(fn):
    def vkDestroyDebugUtilsMessengerEXT(
    instance
            ,messenger
            ,pAllocator
            ,):
        result = _callApi(fn, instance,messenger,pAllocator)


    return vkDestroyDebugUtilsMessengerEXT
def _wrap_vkSubmitDebugUtilsMessageEXT(fn):
    def vkSubmitDebugUtilsMessageEXT(
    instance
            ,messageSeverity
            ,messageTypes
            ,pCallbackData
            ,):
        result = _callApi(fn, instance,messageSeverity,messageTypes,pCallbackData)


    return vkSubmitDebugUtilsMessageEXT
def _wrap_vkGetMemoryHostPointerPropertiesEXT(fn):
    def vkGetMemoryHostPointerPropertiesEXT(
    device
            ,handleType
            ,pHostPointer
            ,pMemoryHostPointerProperties=None
            ,):

        custom_return = True
        if not pMemoryHostPointerProperties:
            pMemoryHostPointerProperties = ffi.new('VkMemoryHostPointerPropertiesEXT*')
            custom_return = False

        result = _callApi(fn, device,handleType,pHostPointer,pMemoryHostPointerProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMemoryHostPointerProperties

        return pMemoryHostPointerProperties[0]



    return vkGetMemoryHostPointerPropertiesEXT
def _wrap_vkCmdWriteBufferMarkerAMD(fn):
    def vkCmdWriteBufferMarkerAMD(
    commandBuffer
            ,pipelineStage
            ,dstBuffer
            ,dstOffset
            ,marker
            ,):
        result = _callApi(fn, commandBuffer,pipelineStage,dstBuffer,dstOffset,marker)


    return vkCmdWriteBufferMarkerAMD
def vkCreateRenderPass2(
device
        ,pCreateInfo
        ,pAllocator
        ,pRenderPass=None
        ,):

    custom_return = True
    if not pRenderPass:
        pRenderPass = ffi.new('VkRenderPass*')
        custom_return = False

    result = _callApi(lib.vkCreateRenderPass2, device,pCreateInfo,pAllocator,pRenderPass)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pRenderPass

    return pRenderPass[0]



def _wrap_vkCreateRenderPass2(fn):
    def vkCreateRenderPass2(
    device
            ,pCreateInfo
            ,pAllocator
            ,pRenderPass=None
            ,):

        custom_return = True
        if not pRenderPass:
            pRenderPass = ffi.new('VkRenderPass*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pRenderPass)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pRenderPass

        return pRenderPass[0]



    return vkCreateRenderPass2
def vkCmdBeginRenderPass2(
commandBuffer
        ,pRenderPassBegin
        ,pSubpassBeginInfo
        ,):
    result = _callApi(lib.vkCmdBeginRenderPass2, commandBuffer,pRenderPassBegin,pSubpassBeginInfo)


def _wrap_vkCmdBeginRenderPass2(fn):
    def vkCmdBeginRenderPass2(
    commandBuffer
            ,pRenderPassBegin
            ,pSubpassBeginInfo
            ,):
        result = _callApi(fn, commandBuffer,pRenderPassBegin,pSubpassBeginInfo)


    return vkCmdBeginRenderPass2
def vkCmdNextSubpass2(
commandBuffer
        ,pSubpassBeginInfo
        ,pSubpassEndInfo
        ,):
    result = _callApi(lib.vkCmdNextSubpass2, commandBuffer,pSubpassBeginInfo,pSubpassEndInfo)


def _wrap_vkCmdNextSubpass2(fn):
    def vkCmdNextSubpass2(
    commandBuffer
            ,pSubpassBeginInfo
            ,pSubpassEndInfo
            ,):
        result = _callApi(fn, commandBuffer,pSubpassBeginInfo,pSubpassEndInfo)


    return vkCmdNextSubpass2
def vkCmdEndRenderPass2(
commandBuffer
        ,pSubpassEndInfo
        ,):
    result = _callApi(lib.vkCmdEndRenderPass2, commandBuffer,pSubpassEndInfo)


def _wrap_vkCmdEndRenderPass2(fn):
    def vkCmdEndRenderPass2(
    commandBuffer
            ,pSubpassEndInfo
            ,):
        result = _callApi(fn, commandBuffer,pSubpassEndInfo)


    return vkCmdEndRenderPass2
def vkGetSemaphoreCounterValue(
device
        ,semaphore
        ,pValue=None
        ,):

    custom_return = True
    if not pValue:
        pValue = ffi.new('uint64_t*')
        custom_return = False

    result = _callApi(lib.vkGetSemaphoreCounterValue, device,semaphore,pValue)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pValue

    return pValue[0]



def _wrap_vkGetSemaphoreCounterValue(fn):
    def vkGetSemaphoreCounterValue(
    device
            ,semaphore
            ,pValue=None
            ,):

        custom_return = True
        if not pValue:
            pValue = ffi.new('uint64_t*')
            custom_return = False

        result = _callApi(fn, device,semaphore,pValue)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pValue

        return pValue[0]



    return vkGetSemaphoreCounterValue
def vkWaitSemaphores(
device
        ,pWaitInfo
        ,timeout
        ,):
    result = _callApi(lib.vkWaitSemaphores, device,pWaitInfo,timeout)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def _wrap_vkWaitSemaphores(fn):
    def vkWaitSemaphores(
    device
            ,pWaitInfo
            ,timeout
            ,):
        result = _callApi(fn, device,pWaitInfo,timeout)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkWaitSemaphores
def vkSignalSemaphore(
device
        ,pSignalInfo
        ,):
    result = _callApi(lib.vkSignalSemaphore, device,pSignalInfo)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def _wrap_vkSignalSemaphore(fn):
    def vkSignalSemaphore(
    device
            ,pSignalInfo
            ,):
        result = _callApi(fn, device,pSignalInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkSignalSemaphore
def _wrap_vkGetAndroidHardwareBufferPropertiesANDROID(fn):
    def vkGetAndroidHardwareBufferPropertiesANDROID(
    device
            ,buffer
            ,pProperties=None
            ,):

        custom_return = True
        if not pProperties:
            pProperties = ffi.new('VkAndroidHardwareBufferPropertiesANDROID*')
            custom_return = False

        result = _callApi(fn, device,buffer,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pProperties

        return pProperties[0]



    return vkGetAndroidHardwareBufferPropertiesANDROID
def _wrap_vkGetMemoryAndroidHardwareBufferANDROID(fn):
    def vkGetMemoryAndroidHardwareBufferANDROID(
    device
            ,pInfo
            ,pBuffer=None
            ,):

        custom_return = True
        if not pBuffer:
            pBuffer = ffi.new('AHardwareBuffer*')
            custom_return = False

        result = _callApi(fn, device,pInfo,pBuffer)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pBuffer

        return pBuffer[0]



    return vkGetMemoryAndroidHardwareBufferANDROID
def vkCmdDrawIndirectCount(
commandBuffer
        ,buffer
        ,offset
        ,countBuffer
        ,countBufferOffset
        ,maxDrawCount
        ,stride
        ,):
    result = _callApi(lib.vkCmdDrawIndirectCount, commandBuffer,buffer,offset,countBuffer,countBufferOffset,maxDrawCount,stride)


def _wrap_vkCmdDrawIndirectCount(fn):
    def vkCmdDrawIndirectCount(
    commandBuffer
            ,buffer
            ,offset
            ,countBuffer
            ,countBufferOffset
            ,maxDrawCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,countBuffer,countBufferOffset,maxDrawCount,stride)


    return vkCmdDrawIndirectCount
def vkCmdDrawIndexedIndirectCount(
commandBuffer
        ,buffer
        ,offset
        ,countBuffer
        ,countBufferOffset
        ,maxDrawCount
        ,stride
        ,):
    result = _callApi(lib.vkCmdDrawIndexedIndirectCount, commandBuffer,buffer,offset,countBuffer,countBufferOffset,maxDrawCount,stride)


def _wrap_vkCmdDrawIndexedIndirectCount(fn):
    def vkCmdDrawIndexedIndirectCount(
    commandBuffer
            ,buffer
            ,offset
            ,countBuffer
            ,countBufferOffset
            ,maxDrawCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,countBuffer,countBufferOffset,maxDrawCount,stride)


    return vkCmdDrawIndexedIndirectCount
def _wrap_vkCmdSetCheckpointNV(fn):
    def vkCmdSetCheckpointNV(
    commandBuffer
            ,pCheckpointMarker
            ,):
        result = _callApi(fn, commandBuffer,pCheckpointMarker)


    return vkCmdSetCheckpointNV
def _wrap_vkGetQueueCheckpointDataNV(fn):
    def vkGetQueueCheckpointDataNV(
    queue
            ,):

        pCheckpointDataCount = ffi.new('uint32_t*')
        pCheckpointData = ffi.NULL

        result = _callApi(fn, queue,pCheckpointDataCount,pCheckpointData)

        pCheckpointData = ffi.new('VkCheckpointDataNV[]', pCheckpointDataCount[0])
        result = _callApi(fn, queue,pCheckpointDataCount,pCheckpointData)

        return pCheckpointData


    return vkGetQueueCheckpointDataNV
def _wrap_vkCmdBindTransformFeedbackBuffersEXT(fn):
    def vkCmdBindTransformFeedbackBuffersEXT(
    commandBuffer
            ,firstBinding
            ,bindingCount
            ,pBuffers
            ,pOffsets
            ,pSizes
            ,):
        result = _callApi(fn, commandBuffer,firstBinding,bindingCount,pBuffers,pOffsets,pSizes)


    return vkCmdBindTransformFeedbackBuffersEXT
def _wrap_vkCmdBeginTransformFeedbackEXT(fn):
    def vkCmdBeginTransformFeedbackEXT(
    commandBuffer
            ,firstCounterBuffer
            ,counterBufferCount
            ,pCounterBuffers
            ,pCounterBufferOffsets
            ,):
        result = _callApi(fn, commandBuffer,firstCounterBuffer,counterBufferCount,pCounterBuffers,pCounterBufferOffsets)


    return vkCmdBeginTransformFeedbackEXT
def _wrap_vkCmdEndTransformFeedbackEXT(fn):
    def vkCmdEndTransformFeedbackEXT(
    commandBuffer
            ,firstCounterBuffer
            ,counterBufferCount
            ,pCounterBuffers
            ,pCounterBufferOffsets
            ,):
        result = _callApi(fn, commandBuffer,firstCounterBuffer,counterBufferCount,pCounterBuffers,pCounterBufferOffsets)


    return vkCmdEndTransformFeedbackEXT
def _wrap_vkCmdBeginQueryIndexedEXT(fn):
    def vkCmdBeginQueryIndexedEXT(
    commandBuffer
            ,queryPool
            ,query
            ,flags
            ,index
            ,):
        result = _callApi(fn, commandBuffer,queryPool,query,flags,index)


    return vkCmdBeginQueryIndexedEXT
def _wrap_vkCmdEndQueryIndexedEXT(fn):
    def vkCmdEndQueryIndexedEXT(
    commandBuffer
            ,queryPool
            ,query
            ,index
            ,):
        result = _callApi(fn, commandBuffer,queryPool,query,index)


    return vkCmdEndQueryIndexedEXT
def _wrap_vkCmdDrawIndirectByteCountEXT(fn):
    def vkCmdDrawIndirectByteCountEXT(
    commandBuffer
            ,instanceCount
            ,firstInstance
            ,counterBuffer
            ,counterBufferOffset
            ,counterOffset
            ,vertexStride
            ,):
        result = _callApi(fn, commandBuffer,instanceCount,firstInstance,counterBuffer,counterBufferOffset,counterOffset,vertexStride)


    return vkCmdDrawIndirectByteCountEXT
def _wrap_vkCmdSetExclusiveScissorNV(fn):
    def vkCmdSetExclusiveScissorNV(
    commandBuffer
            ,firstExclusiveScissor
            ,exclusiveScissorCount
            ,pExclusiveScissors
            ,):
        result = _callApi(fn, commandBuffer,firstExclusiveScissor,exclusiveScissorCount,pExclusiveScissors)


    return vkCmdSetExclusiveScissorNV
def _wrap_vkCmdSetExclusiveScissorEnableNV(fn):
    def vkCmdSetExclusiveScissorEnableNV(
    commandBuffer
            ,firstExclusiveScissor
            ,exclusiveScissorCount
            ,pExclusiveScissorEnables
            ,):
        result = _callApi(fn, commandBuffer,firstExclusiveScissor,exclusiveScissorCount,pExclusiveScissorEnables)


    return vkCmdSetExclusiveScissorEnableNV
def _wrap_vkCmdBindShadingRateImageNV(fn):
    def vkCmdBindShadingRateImageNV(
    commandBuffer
            ,imageView
            ,imageLayout
            ,):
        result = _callApi(fn, commandBuffer,imageView,imageLayout)


    return vkCmdBindShadingRateImageNV
def _wrap_vkCmdSetViewportShadingRatePaletteNV(fn):
    def vkCmdSetViewportShadingRatePaletteNV(
    commandBuffer
            ,firstViewport
            ,viewportCount
            ,pShadingRatePalettes
            ,):
        result = _callApi(fn, commandBuffer,firstViewport,viewportCount,pShadingRatePalettes)


    return vkCmdSetViewportShadingRatePaletteNV
def _wrap_vkCmdSetCoarseSampleOrderNV(fn):
    def vkCmdSetCoarseSampleOrderNV(
    commandBuffer
            ,sampleOrderType
            ,customSampleOrderCount
            ,pCustomSampleOrders
            ,):
        result = _callApi(fn, commandBuffer,sampleOrderType,customSampleOrderCount,pCustomSampleOrders)


    return vkCmdSetCoarseSampleOrderNV
def _wrap_vkCmdDrawMeshTasksNV(fn):
    def vkCmdDrawMeshTasksNV(
    commandBuffer
            ,taskCount
            ,firstTask
            ,):
        result = _callApi(fn, commandBuffer,taskCount,firstTask)


    return vkCmdDrawMeshTasksNV
def _wrap_vkCmdDrawMeshTasksIndirectNV(fn):
    def vkCmdDrawMeshTasksIndirectNV(
    commandBuffer
            ,buffer
            ,offset
            ,drawCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,drawCount,stride)


    return vkCmdDrawMeshTasksIndirectNV
def _wrap_vkCmdDrawMeshTasksIndirectCountNV(fn):
    def vkCmdDrawMeshTasksIndirectCountNV(
    commandBuffer
            ,buffer
            ,offset
            ,countBuffer
            ,countBufferOffset
            ,maxDrawCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,countBuffer,countBufferOffset,maxDrawCount,stride)


    return vkCmdDrawMeshTasksIndirectCountNV
def _wrap_vkCmdDrawMeshTasksEXT(fn):
    def vkCmdDrawMeshTasksEXT(
    commandBuffer
            ,groupCountX
            ,groupCountY
            ,groupCountZ
            ,):
        result = _callApi(fn, commandBuffer,groupCountX,groupCountY,groupCountZ)


    return vkCmdDrawMeshTasksEXT
def _wrap_vkCmdDrawMeshTasksIndirectEXT(fn):
    def vkCmdDrawMeshTasksIndirectEXT(
    commandBuffer
            ,buffer
            ,offset
            ,drawCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,drawCount,stride)


    return vkCmdDrawMeshTasksIndirectEXT
def _wrap_vkCmdDrawMeshTasksIndirectCountEXT(fn):
    def vkCmdDrawMeshTasksIndirectCountEXT(
    commandBuffer
            ,buffer
            ,offset
            ,countBuffer
            ,countBufferOffset
            ,maxDrawCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,countBuffer,countBufferOffset,maxDrawCount,stride)


    return vkCmdDrawMeshTasksIndirectCountEXT
def _wrap_vkCompileDeferredNV(fn):
    def vkCompileDeferredNV(
    device
            ,pipeline
            ,shader
            ,):
        result = _callApi(fn, device,pipeline,shader)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkCompileDeferredNV
def _wrap_vkCreateAccelerationStructureNV(fn):
    def vkCreateAccelerationStructureNV(
    device
            ,pCreateInfo
            ,pAllocator
            ,pAccelerationStructure=None
            ,):

        custom_return = True
        if not pAccelerationStructure:
            pAccelerationStructure = ffi.new('VkAccelerationStructureNV*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pAccelerationStructure)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pAccelerationStructure

        return pAccelerationStructure[0]



    return vkCreateAccelerationStructureNV
def _wrap_vkCmdBindInvocationMaskHUAWEI(fn):
    def vkCmdBindInvocationMaskHUAWEI(
    commandBuffer
            ,imageView
            ,imageLayout
            ,):
        result = _callApi(fn, commandBuffer,imageView,imageLayout)


    return vkCmdBindInvocationMaskHUAWEI
def _wrap_vkDestroyAccelerationStructureKHR(fn):
    def vkDestroyAccelerationStructureKHR(
    device
            ,accelerationStructure
            ,pAllocator
            ,):
        result = _callApi(fn, device,accelerationStructure,pAllocator)


    return vkDestroyAccelerationStructureKHR
def _wrap_vkDestroyAccelerationStructureNV(fn):
    def vkDestroyAccelerationStructureNV(
    device
            ,accelerationStructure
            ,pAllocator
            ,):
        result = _callApi(fn, device,accelerationStructure,pAllocator)


    return vkDestroyAccelerationStructureNV
def _wrap_vkGetAccelerationStructureMemoryRequirementsNV(fn):
    def vkGetAccelerationStructureMemoryRequirementsNV(
    device
            ,pInfo
            ,pMemoryRequirements=None
            ,):

        custom_return = True
        if not pMemoryRequirements:
            pMemoryRequirements = ffi.new('VkMemoryRequirements2KHR*')
            custom_return = False

        result = _callApi(fn, device,pInfo,pMemoryRequirements)

        if custom_return:
            return pMemoryRequirements

        return pMemoryRequirements[0]



    return vkGetAccelerationStructureMemoryRequirementsNV
def _wrap_vkBindAccelerationStructureMemoryNV(fn):
    def vkBindAccelerationStructureMemoryNV(
    device
            ,bindInfoCount
            ,pBindInfos
            ,):
        result = _callApi(fn, device,bindInfoCount,pBindInfos)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkBindAccelerationStructureMemoryNV
def _wrap_vkCmdCopyAccelerationStructureNV(fn):
    def vkCmdCopyAccelerationStructureNV(
    commandBuffer
            ,dst
            ,src
            ,mode
            ,):
        result = _callApi(fn, commandBuffer,dst,src,mode)


    return vkCmdCopyAccelerationStructureNV
def _wrap_vkCmdCopyAccelerationStructureKHR(fn):
    def vkCmdCopyAccelerationStructureKHR(
    commandBuffer
            ,pInfo
            ,):
        result = _callApi(fn, commandBuffer,pInfo)


    return vkCmdCopyAccelerationStructureKHR
def _wrap_vkCopyAccelerationStructureKHR(fn):
    def vkCopyAccelerationStructureKHR(
    device
            ,deferredOperation
            ,pInfo
            ,):
        result = _callApi(fn, device,deferredOperation,pInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkCopyAccelerationStructureKHR
def _wrap_vkCmdCopyAccelerationStructureToMemoryKHR(fn):
    def vkCmdCopyAccelerationStructureToMemoryKHR(
    commandBuffer
            ,pInfo
            ,):
        result = _callApi(fn, commandBuffer,pInfo)


    return vkCmdCopyAccelerationStructureToMemoryKHR
def _wrap_vkCopyAccelerationStructureToMemoryKHR(fn):
    def vkCopyAccelerationStructureToMemoryKHR(
    device
            ,deferredOperation
            ,pInfo
            ,):
        result = _callApi(fn, device,deferredOperation,pInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkCopyAccelerationStructureToMemoryKHR
def _wrap_vkCmdCopyMemoryToAccelerationStructureKHR(fn):
    def vkCmdCopyMemoryToAccelerationStructureKHR(
    commandBuffer
            ,pInfo
            ,):
        result = _callApi(fn, commandBuffer,pInfo)


    return vkCmdCopyMemoryToAccelerationStructureKHR
def _wrap_vkCopyMemoryToAccelerationStructureKHR(fn):
    def vkCopyMemoryToAccelerationStructureKHR(
    device
            ,deferredOperation
            ,pInfo
            ,):
        result = _callApi(fn, device,deferredOperation,pInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkCopyMemoryToAccelerationStructureKHR
def _wrap_vkCmdWriteAccelerationStructuresPropertiesKHR(fn):
    def vkCmdWriteAccelerationStructuresPropertiesKHR(
    commandBuffer
            ,accelerationStructureCount
            ,pAccelerationStructures
            ,queryType
            ,queryPool
            ,firstQuery
            ,):
        result = _callApi(fn, commandBuffer,accelerationStructureCount,pAccelerationStructures,queryType,queryPool,firstQuery)


    return vkCmdWriteAccelerationStructuresPropertiesKHR
def _wrap_vkCmdWriteAccelerationStructuresPropertiesNV(fn):
    def vkCmdWriteAccelerationStructuresPropertiesNV(
    commandBuffer
            ,accelerationStructureCount
            ,pAccelerationStructures
            ,queryType
            ,queryPool
            ,firstQuery
            ,):
        result = _callApi(fn, commandBuffer,accelerationStructureCount,pAccelerationStructures,queryType,queryPool,firstQuery)


    return vkCmdWriteAccelerationStructuresPropertiesNV
def _wrap_vkCmdBuildAccelerationStructureNV(fn):
    def vkCmdBuildAccelerationStructureNV(
    commandBuffer
            ,pInfo
            ,instanceData
            ,instanceOffset
            ,update
            ,dst
            ,src
            ,scratch
            ,scratchOffset
            ,):
        result = _callApi(fn, commandBuffer,pInfo,instanceData,instanceOffset,update,dst,src,scratch,scratchOffset)


    return vkCmdBuildAccelerationStructureNV
def _wrap_vkWriteAccelerationStructuresPropertiesKHR(fn):
    def vkWriteAccelerationStructuresPropertiesKHR(
    device
            ,accelerationStructureCount
            ,pAccelerationStructures
            ,queryType
            ,dataSize
            ,pData
            ,stride
            ,):
        result = _callApi(fn, device,accelerationStructureCount,pAccelerationStructures,queryType,dataSize,pData,stride)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkWriteAccelerationStructuresPropertiesKHR
def _wrap_vkCmdTraceRaysKHR(fn):
    def vkCmdTraceRaysKHR(
    commandBuffer
            ,pRaygenShaderBindingTable
            ,pMissShaderBindingTable
            ,pHitShaderBindingTable
            ,pCallableShaderBindingTable
            ,width
            ,height
            ,depth
            ,):
        result = _callApi(fn, commandBuffer,pRaygenShaderBindingTable,pMissShaderBindingTable,pHitShaderBindingTable,pCallableShaderBindingTable,width,height,depth)


    return vkCmdTraceRaysKHR
def _wrap_vkCmdTraceRaysNV(fn):
    def vkCmdTraceRaysNV(
    commandBuffer
            ,raygenShaderBindingTableBuffer
            ,raygenShaderBindingOffset
            ,missShaderBindingTableBuffer
            ,missShaderBindingOffset
            ,missShaderBindingStride
            ,hitShaderBindingTableBuffer
            ,hitShaderBindingOffset
            ,hitShaderBindingStride
            ,callableShaderBindingTableBuffer
            ,callableShaderBindingOffset
            ,callableShaderBindingStride
            ,width
            ,height
            ,depth
            ,):
        result = _callApi(fn, commandBuffer,raygenShaderBindingTableBuffer,raygenShaderBindingOffset,missShaderBindingTableBuffer,missShaderBindingOffset,missShaderBindingStride,hitShaderBindingTableBuffer,hitShaderBindingOffset,hitShaderBindingStride,callableShaderBindingTableBuffer,callableShaderBindingOffset,callableShaderBindingStride,width,height,depth)


    return vkCmdTraceRaysNV
def _wrap_vkGetRayTracingShaderGroupHandlesKHR(fn):
    def vkGetRayTracingShaderGroupHandlesKHR(
    device
            ,pipeline
            ,firstGroup
            ,groupCount
            ,dataSize
            ,pData=None
            ,):

        custom_return = True
        if not pData:
            pData = ffi.new('uint64_t[%d]' % dataSize)
            custom_return = False

        result = _callApi(fn, device,pipeline,firstGroup,groupCount,dataSize,pData)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pData

        return pData



    return vkGetRayTracingShaderGroupHandlesKHR
def _wrap_vkGetRayTracingCaptureReplayShaderGroupHandlesKHR(fn):
    def vkGetRayTracingCaptureReplayShaderGroupHandlesKHR(
    device
            ,pipeline
            ,firstGroup
            ,groupCount
            ,dataSize
            ,pData=None
            ,):

        custom_return = True
        if not pData:
            pData = ffi.new('uint64_t[%d]' % dataSize)
            custom_return = False

        result = _callApi(fn, device,pipeline,firstGroup,groupCount,dataSize,pData)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pData

        return pData



    return vkGetRayTracingCaptureReplayShaderGroupHandlesKHR
def _wrap_vkGetAccelerationStructureHandleNV(fn):
    def vkGetAccelerationStructureHandleNV(
    device
            ,accelerationStructure
            ,dataSize
            ,pData=None
            ,):

        custom_return = True
        if not pData:
            pData = ffi.new('uint64_t[%d]' % dataSize)
            custom_return = False

        result = _callApi(fn, device,accelerationStructure,dataSize,pData)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pData

        return pData



    return vkGetAccelerationStructureHandleNV
def _wrap_vkCreateRayTracingPipelinesNV(fn):
    def vkCreateRayTracingPipelinesNV(
    device
            ,pipelineCache
            ,createInfoCount
            ,pCreateInfos
            ,pAllocator
            ,pPipelines=None
            ,):

        custom_return = True
        if not pPipelines:
            pPipelines = ffi.new('VkPipeline[%d]' % createInfoCount)
            custom_return = False

        result = _callApi(fn, device,pipelineCache,createInfoCount,pCreateInfos,pAllocator,pPipelines)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pPipelines

        return pPipelines



    return vkCreateRayTracingPipelinesNV
def _wrap_vkCreateRayTracingPipelinesNV(fn):
    def vkCreateRayTracingPipelinesNV(
    device
            ,pipelineCache
            ,createInfoCount
            ,pCreateInfos
            ,pAllocator
            ,pPipelines=None
            ,):

        custom_return = True
        if not pPipelines:
            pPipelines = ffi.new('VkPipeline[%d]' % createInfoCount)
            custom_return = False

        result = _callApi(fn, device,pipelineCache,createInfoCount,pCreateInfos,pAllocator,pPipelines)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pPipelines

        return pPipelines



    return vkCreateRayTracingPipelinesNV
def _wrap_vkCreateRayTracingPipelinesKHR(fn):
    def vkCreateRayTracingPipelinesKHR(
    device
            ,deferredOperation
            ,pipelineCache
            ,createInfoCount
            ,pCreateInfos
            ,pAllocator
            ,pPipelines=None
            ,):

        custom_return = True
        if not pPipelines:
            pPipelines = ffi.new('VkPipeline[%d]' % createInfoCount)
            custom_return = False

        result = _callApi(fn, device,deferredOperation,pipelineCache,createInfoCount,pCreateInfos,pAllocator,pPipelines)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pPipelines

        return pPipelines



    return vkCreateRayTracingPipelinesKHR
def _wrap_vkCreateRayTracingPipelinesKHR(fn):
    def vkCreateRayTracingPipelinesKHR(
    device
            ,deferredOperation
            ,pipelineCache
            ,createInfoCount
            ,pCreateInfos
            ,pAllocator
            ,pPipelines=None
            ,):

        custom_return = True
        if not pPipelines:
            pPipelines = ffi.new('VkPipeline[%d]' % createInfoCount)
            custom_return = False

        result = _callApi(fn, device,deferredOperation,pipelineCache,createInfoCount,pCreateInfos,pAllocator,pPipelines)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pPipelines

        return pPipelines



    return vkCreateRayTracingPipelinesKHR
def _wrap_vkGetPhysicalDeviceCooperativeMatrixPropertiesNV(fn):
    def vkGetPhysicalDeviceCooperativeMatrixPropertiesNV(
    physicalDevice
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkCooperativeMatrixPropertiesNV[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pProperties


    return vkGetPhysicalDeviceCooperativeMatrixPropertiesNV
def _wrap_vkCmdTraceRaysIndirectKHR(fn):
    def vkCmdTraceRaysIndirectKHR(
    commandBuffer
            ,pRaygenShaderBindingTable
            ,pMissShaderBindingTable
            ,pHitShaderBindingTable
            ,pCallableShaderBindingTable
            ,indirectDeviceAddress
            ,):
        result = _callApi(fn, commandBuffer,pRaygenShaderBindingTable,pMissShaderBindingTable,pHitShaderBindingTable,pCallableShaderBindingTable,indirectDeviceAddress)


    return vkCmdTraceRaysIndirectKHR
def _wrap_vkCmdTraceRaysIndirect2KHR(fn):
    def vkCmdTraceRaysIndirect2KHR(
    commandBuffer
            ,indirectDeviceAddress
            ,):
        result = _callApi(fn, commandBuffer,indirectDeviceAddress)


    return vkCmdTraceRaysIndirect2KHR
def _wrap_vkGetDeviceAccelerationStructureCompatibilityKHR(fn):
    def vkGetDeviceAccelerationStructureCompatibilityKHR(
    device
            ,pVersionInfo
            ,pCompatibility=None
            ,):

        custom_return = True
        if not pCompatibility:
            pCompatibility = ffi.new('VkAccelerationStructureCompatibilityKHR*')
            custom_return = False

        result = _callApi(fn, device,pVersionInfo,pCompatibility)

        if custom_return:
            return pCompatibility

        return pCompatibility[0]



    return vkGetDeviceAccelerationStructureCompatibilityKHR
def _wrap_vkGetRayTracingShaderGroupStackSizeKHR(fn):
    def vkGetRayTracingShaderGroupStackSizeKHR(
    device
            ,pipeline
            ,group
            ,groupShader=None
            ,):

        custom_return = True
        if not groupShader:
            groupShader = ffi.new('VkShaderGroupShaderKHR*')
            custom_return = False

        result = _callApi(fn, device,pipeline,group,groupShader)

        if custom_return:
            return groupShader

        return groupShader[0]



    return vkGetRayTracingShaderGroupStackSizeKHR
def _wrap_vkCmdSetRayTracingPipelineStackSizeKHR(fn):
    def vkCmdSetRayTracingPipelineStackSizeKHR(
    commandBuffer
            ,pipelineStackSize
            ,):
        result = _callApi(fn, commandBuffer,pipelineStackSize)


    return vkCmdSetRayTracingPipelineStackSizeKHR
def _wrap_vkGetImageViewHandleNVX(fn):
    def vkGetImageViewHandleNVX(
    device
            ,pInfo=None
            ,):

        custom_return = True
        if not pInfo:
            pInfo = ffi.new('VkImageViewHandleInfoNVX*')
            custom_return = False

        result = _callApi(fn, device,pInfo)

        if custom_return:
            return pInfo

        return pInfo[0]



    return vkGetImageViewHandleNVX
def _wrap_vkGetImageViewAddressNVX(fn):
    def vkGetImageViewAddressNVX(
    device
            ,imageView
            ,pProperties=None
            ,):

        custom_return = True
        if not pProperties:
            pProperties = ffi.new('VkImageViewAddressPropertiesNVX*')
            custom_return = False

        result = _callApi(fn, device,imageView,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pProperties

        return pProperties[0]



    return vkGetImageViewAddressNVX
def _wrap_vkGetPhysicalDeviceSurfacePresentModes2EXT(fn):
    def vkGetPhysicalDeviceSurfacePresentModes2EXT(
    physicalDevice
            ,pSurfaceInfo
            ,):

        pPresentModeCount = ffi.new('uint32_t*')
        pPresentModes = ffi.NULL

        result = _callApi(fn, physicalDevice,pSurfaceInfo,pPresentModeCount,pPresentModes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pPresentModes = ffi.new('VkPresentModeKHR[]', pPresentModeCount[0])
        result = _callApi(fn, physicalDevice,pSurfaceInfo,pPresentModeCount,pPresentModes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pPresentModes


    return vkGetPhysicalDeviceSurfacePresentModes2EXT
def _wrap_vkGetDeviceGroupSurfacePresentModes2EXT(fn):
    def vkGetDeviceGroupSurfacePresentModes2EXT(
    device
            ,pSurfaceInfo
            ,pModes=None
            ,):

        custom_return = True
        if not pModes:
            pModes = ffi.new('VkDeviceGroupPresentModeFlagsKHR*')
            custom_return = False

        result = _callApi(fn, device,pSurfaceInfo,pModes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pModes

        return pModes[0]



    return vkGetDeviceGroupSurfacePresentModes2EXT
def _wrap_vkAcquireFullScreenExclusiveModeEXT(fn):
    def vkAcquireFullScreenExclusiveModeEXT(
    device
            ,swapchain=None
            ,):

        custom_return = True
        if not swapchain:
            swapchain = ffi.new('VkSwapchainKHR*')
            custom_return = False

        result = _callApi(fn, device,swapchain)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return swapchain

        return swapchain[0]



    return vkAcquireFullScreenExclusiveModeEXT
def _wrap_vkReleaseFullScreenExclusiveModeEXT(fn):
    def vkReleaseFullScreenExclusiveModeEXT(
    device
            ,swapchain
            ,):
        result = _callApi(fn, device,swapchain)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkReleaseFullScreenExclusiveModeEXT
def _wrap_vkEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR(fn):
    def vkEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR(
    physicalDevice
            ,queueFamilyIndex
            ,pCounterCount
            ,):

        pCounters = ffi.new('VkPerformanceCounterKHR*')
        pCounterDescriptions = ffi.NULL

        result = _callApi(fn, physicalDevice,queueFamilyIndex,pCounterCount,pCounters,pCounterDescriptions)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pCounterDescriptions = ffi.new('VkPerformanceCounterDescriptionKHR[]', pCounters[0])
        result = _callApi(fn, physicalDevice,queueFamilyIndex,pCounterCount,pCounters,pCounterDescriptions)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        result = (StrWrap(x) for x in pCounterDescriptions)
        _weakkey_dict[result] = pCounterDescriptions
        return result


    return vkEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR
def _wrap_vkGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR(fn):
    def vkGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR(
    physicalDevice
            ,):

        pPerformanceQueryCreateInfo = ffi.new('VkQueryPoolPerformanceCreateInfoKHR*')
        pNumPasses = ffi.NULL

        result = _callApi(fn, physicalDevice,pPerformanceQueryCreateInfo,pNumPasses)

        pNumPasses = ffi.new('uint32_t[]', pPerformanceQueryCreateInfo[0])
        result = _callApi(fn, physicalDevice,pPerformanceQueryCreateInfo,pNumPasses)

        return pNumPasses


    return vkGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR
def _wrap_vkAcquireProfilingLockKHR(fn):
    def vkAcquireProfilingLockKHR(
    device
            ,pInfo=None
            ,):

        custom_return = True
        if not pInfo:
            pInfo = ffi.new('VkAcquireProfilingLockInfoKHR*')
            custom_return = False

        result = _callApi(fn, device,pInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pInfo

        return pInfo[0]



    return vkAcquireProfilingLockKHR
def _wrap_vkReleaseProfilingLockKHR(fn):
    def vkReleaseProfilingLockKHR(
    device
            ,):
        result = _callApi(fn, device)


    return vkReleaseProfilingLockKHR
def _wrap_vkGetImageDrmFormatModifierPropertiesEXT(fn):
    def vkGetImageDrmFormatModifierPropertiesEXT(
    device
            ,image
            ,pProperties=None
            ,):

        custom_return = True
        if not pProperties:
            pProperties = ffi.new('VkImageDrmFormatModifierPropertiesEXT*')
            custom_return = False

        result = _callApi(fn, device,image,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pProperties

        return pProperties[0]



    return vkGetImageDrmFormatModifierPropertiesEXT
def vkGetBufferOpaqueCaptureAddress(
device
        ,pInfo=None
        ,):

    custom_return = True
    if not pInfo:
        pInfo = ffi.new('VkBufferDeviceAddressInfo*')
        custom_return = False

    result = _callApi(lib.vkGetBufferOpaqueCaptureAddress, device,pInfo)

    if custom_return:
        return pInfo

    return pInfo[0]



def _wrap_vkGetBufferOpaqueCaptureAddress(fn):
    def vkGetBufferOpaqueCaptureAddress(
    device
            ,pInfo=None
            ,):

        custom_return = True
        if not pInfo:
            pInfo = ffi.new('VkBufferDeviceAddressInfo*')
            custom_return = False

        result = _callApi(fn, device,pInfo)

        if custom_return:
            return pInfo

        return pInfo[0]



    return vkGetBufferOpaqueCaptureAddress
def vkGetBufferDeviceAddress(
device
        ,pInfo=None
        ,):

    custom_return = True
    if not pInfo:
        pInfo = ffi.new('VkBufferDeviceAddressInfo*')
        custom_return = False

    result = _callApi(lib.vkGetBufferDeviceAddress, device,pInfo)

    if custom_return:
        return pInfo

    return pInfo[0]



def _wrap_vkGetBufferDeviceAddress(fn):
    def vkGetBufferDeviceAddress(
    device
            ,pInfo=None
            ,):

        custom_return = True
        if not pInfo:
            pInfo = ffi.new('VkBufferDeviceAddressInfo*')
            custom_return = False

        result = _callApi(fn, device,pInfo)

        if custom_return:
            return pInfo

        return pInfo[0]



    return vkGetBufferDeviceAddress
def _wrap_vkCreateHeadlessSurfaceEXT(fn):
    def vkCreateHeadlessSurfaceEXT(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateHeadlessSurfaceEXT
def _wrap_vkGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV(fn):
    def vkGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV(
    physicalDevice
            ,):

        pCombinationCount = ffi.new('uint32_t*')
        pCombinations = ffi.NULL

        result = _callApi(fn, physicalDevice,pCombinationCount,pCombinations)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pCombinations = ffi.new('VkFramebufferMixedSamplesCombinationNV[]', pCombinationCount[0])
        result = _callApi(fn, physicalDevice,pCombinationCount,pCombinations)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pCombinations


    return vkGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV
def _wrap_vkInitializePerformanceApiINTEL(fn):
    def vkInitializePerformanceApiINTEL(
    device
            ,pInitializeInfo
            ,):
        result = _callApi(fn, device,pInitializeInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkInitializePerformanceApiINTEL
def _wrap_vkUninitializePerformanceApiINTEL(fn):
    def vkUninitializePerformanceApiINTEL(
    device
            ,):
        result = _callApi(fn, device)


    return vkUninitializePerformanceApiINTEL
def _wrap_vkCmdSetPerformanceMarkerINTEL(fn):
    def vkCmdSetPerformanceMarkerINTEL(
    commandBuffer
            ,pMarkerInfo
            ,):
        result = _callApi(fn, commandBuffer,pMarkerInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkCmdSetPerformanceMarkerINTEL
def _wrap_vkCmdSetPerformanceStreamMarkerINTEL(fn):
    def vkCmdSetPerformanceStreamMarkerINTEL(
    commandBuffer
            ,pMarkerInfo
            ,):
        result = _callApi(fn, commandBuffer,pMarkerInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkCmdSetPerformanceStreamMarkerINTEL
def _wrap_vkCmdSetPerformanceOverrideINTEL(fn):
    def vkCmdSetPerformanceOverrideINTEL(
    commandBuffer
            ,pOverrideInfo
            ,):
        result = _callApi(fn, commandBuffer,pOverrideInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkCmdSetPerformanceOverrideINTEL
def _wrap_vkAcquirePerformanceConfigurationINTEL(fn):
    def vkAcquirePerformanceConfigurationINTEL(
    device
            ,pAcquireInfo
            ,pConfiguration=None
            ,):

        custom_return = True
        if not pConfiguration:
            pConfiguration = ffi.new('VkPerformanceConfigurationINTEL*')
            custom_return = False

        result = _callApi(fn, device,pAcquireInfo,pConfiguration)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pConfiguration

        return pConfiguration[0]



    return vkAcquirePerformanceConfigurationINTEL
def _wrap_vkReleasePerformanceConfigurationINTEL(fn):
    def vkReleasePerformanceConfigurationINTEL(
    device
            ,configuration
            ,):
        result = _callApi(fn, device,configuration)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkReleasePerformanceConfigurationINTEL
def _wrap_vkQueueSetPerformanceConfigurationINTEL(fn):
    def vkQueueSetPerformanceConfigurationINTEL(
    queue
            ,configuration
            ,):
        result = _callApi(fn, queue,configuration)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkQueueSetPerformanceConfigurationINTEL
def _wrap_vkGetPerformanceParameterINTEL(fn):
    def vkGetPerformanceParameterINTEL(
    device
            ,parameter
            ,pValue=None
            ,):

        custom_return = True
        if not pValue:
            pValue = ffi.new('VkPerformanceValueINTEL*')
            custom_return = False

        result = _callApi(fn, device,parameter,pValue)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pValue

        return pValue[0]



    return vkGetPerformanceParameterINTEL
def vkGetDeviceMemoryOpaqueCaptureAddress(
device
        ,pInfo=None
        ,):

    custom_return = True
    if not pInfo:
        pInfo = ffi.new('VkDeviceMemoryOpaqueCaptureAddressInfo*')
        custom_return = False

    result = _callApi(lib.vkGetDeviceMemoryOpaqueCaptureAddress, device,pInfo)

    if custom_return:
        return pInfo

    return pInfo[0]



def _wrap_vkGetDeviceMemoryOpaqueCaptureAddress(fn):
    def vkGetDeviceMemoryOpaqueCaptureAddress(
    device
            ,pInfo=None
            ,):

        custom_return = True
        if not pInfo:
            pInfo = ffi.new('VkDeviceMemoryOpaqueCaptureAddressInfo*')
            custom_return = False

        result = _callApi(fn, device,pInfo)

        if custom_return:
            return pInfo

        return pInfo[0]



    return vkGetDeviceMemoryOpaqueCaptureAddress
def _wrap_vkGetPipelineExecutablePropertiesKHR(fn):
    def vkGetPipelineExecutablePropertiesKHR(
    device
            ,pPipelineInfo
            ,):

        pExecutableCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, device,pPipelineInfo,pExecutableCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkPipelineExecutablePropertiesKHR[]', pExecutableCount[0])
        result = _callApi(fn, device,pPipelineInfo,pExecutableCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        result = (StrWrap(x) for x in pProperties)
        _weakkey_dict[result] = pProperties
        return result


    return vkGetPipelineExecutablePropertiesKHR
def _wrap_vkGetPipelineExecutableStatisticsKHR(fn):
    def vkGetPipelineExecutableStatisticsKHR(
    device
            ,pExecutableInfo
            ,):

        pStatisticCount = ffi.new('uint32_t*')
        pStatistics = ffi.NULL

        result = _callApi(fn, device,pExecutableInfo,pStatisticCount,pStatistics)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pStatistics = ffi.new('VkPipelineExecutableStatisticKHR[]', pStatisticCount[0])
        result = _callApi(fn, device,pExecutableInfo,pStatisticCount,pStatistics)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        result = (StrWrap(x) for x in pStatistics)
        _weakkey_dict[result] = pStatistics
        return result


    return vkGetPipelineExecutableStatisticsKHR
def _wrap_vkGetPipelineExecutableInternalRepresentationsKHR(fn):
    def vkGetPipelineExecutableInternalRepresentationsKHR(
    device
            ,pExecutableInfo
            ,):

        pInternalRepresentationCount = ffi.new('uint32_t*')
        pInternalRepresentations = ffi.NULL

        result = _callApi(fn, device,pExecutableInfo,pInternalRepresentationCount,pInternalRepresentations)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pInternalRepresentations = ffi.new('VkPipelineExecutableInternalRepresentationKHR[]', pInternalRepresentationCount[0])
        result = _callApi(fn, device,pExecutableInfo,pInternalRepresentationCount,pInternalRepresentations)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        result = (StrWrap(x) for x in pInternalRepresentations)
        _weakkey_dict[result] = pInternalRepresentations
        return result


    return vkGetPipelineExecutableInternalRepresentationsKHR
def _wrap_vkCmdSetLineStippleEXT(fn):
    def vkCmdSetLineStippleEXT(
    commandBuffer
            ,lineStippleFactor
            ,lineStipplePattern
            ,):
        result = _callApi(fn, commandBuffer,lineStippleFactor,lineStipplePattern)


    return vkCmdSetLineStippleEXT
def vkGetFaultData(
device
        ,faultQueryBehavior
        ,pUnrecordedFaults
        ,):

    pFaultCount = ffi.new('uint32_t*')
    pFaults = ffi.NULL

    result = _callApi(lib.vkGetFaultData, device,faultQueryBehavior,pUnrecordedFaults,pFaultCount,pFaults)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pFaults = ffi.new('VkFaultData[]', pFaultCount[0])
    result = _callApi(lib.vkGetFaultData, device,faultQueryBehavior,pUnrecordedFaults,pFaultCount,pFaults)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    return pFaults


def vkGetPhysicalDeviceToolProperties(
physicalDevice
        ,):

    pToolCount = ffi.new('uint32_t*')
    pToolProperties = ffi.NULL

    result = _callApi(lib.vkGetPhysicalDeviceToolProperties, physicalDevice,pToolCount,pToolProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pToolProperties = ffi.new('VkPhysicalDeviceToolProperties[]', pToolCount[0])
    result = _callApi(lib.vkGetPhysicalDeviceToolProperties, physicalDevice,pToolCount,pToolProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    result = (StrWrap(x) for x in pToolProperties)
    _weakkey_dict[result] = pToolProperties
    return result


def _wrap_vkGetPhysicalDeviceToolProperties(fn):
    def vkGetPhysicalDeviceToolProperties(
    physicalDevice
            ,):

        pToolCount = ffi.new('uint32_t*')
        pToolProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pToolCount,pToolProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pToolProperties = ffi.new('VkPhysicalDeviceToolProperties[]', pToolCount[0])
        result = _callApi(fn, physicalDevice,pToolCount,pToolProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        result = (StrWrap(x) for x in pToolProperties)
        _weakkey_dict[result] = pToolProperties
        return result


    return vkGetPhysicalDeviceToolProperties
def _wrap_vkCreateAccelerationStructureKHR(fn):
    def vkCreateAccelerationStructureKHR(
    device
            ,pCreateInfo
            ,pAllocator
            ,pAccelerationStructure=None
            ,):

        custom_return = True
        if not pAccelerationStructure:
            pAccelerationStructure = ffi.new('VkAccelerationStructureKHR*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pAccelerationStructure)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pAccelerationStructure

        return pAccelerationStructure[0]



    return vkCreateAccelerationStructureKHR
def _wrap_vkCmdBuildAccelerationStructuresKHR(fn):
    def vkCmdBuildAccelerationStructuresKHR(
    commandBuffer
            ,infoCount
            ,pInfos
            ,ppBuildRangeInfos
            ,):
        result = _callApi(fn, commandBuffer,infoCount,pInfos,ppBuildRangeInfos)


    return vkCmdBuildAccelerationStructuresKHR
def _wrap_vkCmdBuildAccelerationStructuresIndirectKHR(fn):
    def vkCmdBuildAccelerationStructuresIndirectKHR(
    commandBuffer
            ,infoCount
            ,pInfos
            ,pIndirectDeviceAddresses
            ,pIndirectStrides
            ,ppMaxPrimitiveCounts
            ,):
        result = _callApi(fn, commandBuffer,infoCount,pInfos,pIndirectDeviceAddresses,pIndirectStrides,ppMaxPrimitiveCounts)


    return vkCmdBuildAccelerationStructuresIndirectKHR
def _wrap_vkBuildAccelerationStructuresKHR(fn):
    def vkBuildAccelerationStructuresKHR(
    device
            ,deferredOperation
            ,infoCount
            ,pInfos
            ,ppBuildRangeInfos
            ,):
        result = _callApi(fn, device,deferredOperation,infoCount,pInfos,ppBuildRangeInfos)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkBuildAccelerationStructuresKHR
def _wrap_vkGetAccelerationStructureDeviceAddressKHR(fn):
    def vkGetAccelerationStructureDeviceAddressKHR(
    device
            ,pInfo=None
            ,):

        custom_return = True
        if not pInfo:
            pInfo = ffi.new('VkAccelerationStructureDeviceAddressInfoKHR*')
            custom_return = False

        result = _callApi(fn, device,pInfo)

        if custom_return:
            return pInfo

        return pInfo[0]



    return vkGetAccelerationStructureDeviceAddressKHR
def _wrap_vkCreateDeferredOperationKHR(fn):
    def vkCreateDeferredOperationKHR(
    device
            ,pAllocator
            ,pDeferredOperation=None
            ,):

        custom_return = True
        if not pDeferredOperation:
            pDeferredOperation = ffi.new('VkDeferredOperationKHR*')
            custom_return = False

        result = _callApi(fn, device,pAllocator,pDeferredOperation)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pDeferredOperation

        return pDeferredOperation[0]



    return vkCreateDeferredOperationKHR
def _wrap_vkDestroyDeferredOperationKHR(fn):
    def vkDestroyDeferredOperationKHR(
    device
            ,operation
            ,pAllocator
            ,):
        result = _callApi(fn, device,operation,pAllocator)


    return vkDestroyDeferredOperationKHR
def _wrap_vkGetDeferredOperationMaxConcurrencyKHR(fn):
    def vkGetDeferredOperationMaxConcurrencyKHR(
    device
            ,operation=None
            ,):

        custom_return = True
        if not operation:
            operation = ffi.new('VkDeferredOperationKHR*')
            custom_return = False

        result = _callApi(fn, device,operation)

        if custom_return:
            return operation

        return operation[0]



    return vkGetDeferredOperationMaxConcurrencyKHR
def _wrap_vkGetDeferredOperationResultKHR(fn):
    def vkGetDeferredOperationResultKHR(
    device
            ,operation=None
            ,):

        custom_return = True
        if not operation:
            operation = ffi.new('VkDeferredOperationKHR*')
            custom_return = False

        result = _callApi(fn, device,operation)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return operation

        return operation[0]



    return vkGetDeferredOperationResultKHR
def _wrap_vkDeferredOperationJoinKHR(fn):
    def vkDeferredOperationJoinKHR(
    device
            ,operation
            ,):
        result = _callApi(fn, device,operation)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkDeferredOperationJoinKHR
def _wrap_vkGetPipelineIndirectMemoryRequirementsNV(fn):
    def vkGetPipelineIndirectMemoryRequirementsNV(
    device
            ,pCreateInfo
            ,pMemoryRequirements=None
            ,):

        custom_return = True
        if not pMemoryRequirements:
            pMemoryRequirements = ffi.new('VkMemoryRequirements2*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pMemoryRequirements)

        if custom_return:
            return pMemoryRequirements

        return pMemoryRequirements[0]



    return vkGetPipelineIndirectMemoryRequirementsNV
def _wrap_vkGetPipelineIndirectDeviceAddressNV(fn):
    def vkGetPipelineIndirectDeviceAddressNV(
    device
            ,pInfo=None
            ,):

        custom_return = True
        if not pInfo:
            pInfo = ffi.new('VkPipelineIndirectDeviceAddressInfoNV*')
            custom_return = False

        result = _callApi(fn, device,pInfo)

        if custom_return:
            return pInfo

        return pInfo[0]



    return vkGetPipelineIndirectDeviceAddressNV
def vkCmdSetCullMode(
commandBuffer
        ,cullMode
        ,):
    result = _callApi(lib.vkCmdSetCullMode, commandBuffer,cullMode)


def _wrap_vkCmdSetCullMode(fn):
    def vkCmdSetCullMode(
    commandBuffer
            ,cullMode
            ,):
        result = _callApi(fn, commandBuffer,cullMode)


    return vkCmdSetCullMode
def vkCmdSetFrontFace(
commandBuffer
        ,frontFace
        ,):
    result = _callApi(lib.vkCmdSetFrontFace, commandBuffer,frontFace)


def _wrap_vkCmdSetFrontFace(fn):
    def vkCmdSetFrontFace(
    commandBuffer
            ,frontFace
            ,):
        result = _callApi(fn, commandBuffer,frontFace)


    return vkCmdSetFrontFace
def vkCmdSetPrimitiveTopology(
commandBuffer
        ,primitiveTopology
        ,):
    result = _callApi(lib.vkCmdSetPrimitiveTopology, commandBuffer,primitiveTopology)


def _wrap_vkCmdSetPrimitiveTopology(fn):
    def vkCmdSetPrimitiveTopology(
    commandBuffer
            ,primitiveTopology
            ,):
        result = _callApi(fn, commandBuffer,primitiveTopology)


    return vkCmdSetPrimitiveTopology
def vkCmdSetViewportWithCount(
commandBuffer
        ,viewportCount
        ,pViewports
        ,):
    result = _callApi(lib.vkCmdSetViewportWithCount, commandBuffer,viewportCount,pViewports)


def _wrap_vkCmdSetViewportWithCount(fn):
    def vkCmdSetViewportWithCount(
    commandBuffer
            ,viewportCount
            ,pViewports
            ,):
        result = _callApi(fn, commandBuffer,viewportCount,pViewports)


    return vkCmdSetViewportWithCount
def vkCmdSetScissorWithCount(
commandBuffer
        ,scissorCount
        ,pScissors
        ,):
    result = _callApi(lib.vkCmdSetScissorWithCount, commandBuffer,scissorCount,pScissors)


def _wrap_vkCmdSetScissorWithCount(fn):
    def vkCmdSetScissorWithCount(
    commandBuffer
            ,scissorCount
            ,pScissors
            ,):
        result = _callApi(fn, commandBuffer,scissorCount,pScissors)


    return vkCmdSetScissorWithCount
def _wrap_vkCmdBindIndexBuffer2KHR(fn):
    def vkCmdBindIndexBuffer2KHR(
    commandBuffer
            ,buffer
            ,offset
            ,size
            ,indexType
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,size,indexType)


    return vkCmdBindIndexBuffer2KHR
def vkCmdBindVertexBuffers2(
commandBuffer
        ,firstBinding
        ,bindingCount
        ,pBuffers
        ,pOffsets
        ,pSizes
        ,pStrides
        ,):
    result = _callApi(lib.vkCmdBindVertexBuffers2, commandBuffer,firstBinding,bindingCount,pBuffers,pOffsets,pSizes,pStrides)


def _wrap_vkCmdBindVertexBuffers2(fn):
    def vkCmdBindVertexBuffers2(
    commandBuffer
            ,firstBinding
            ,bindingCount
            ,pBuffers
            ,pOffsets
            ,pSizes
            ,pStrides
            ,):
        result = _callApi(fn, commandBuffer,firstBinding,bindingCount,pBuffers,pOffsets,pSizes,pStrides)


    return vkCmdBindVertexBuffers2
def vkCmdSetDepthTestEnable(
commandBuffer
        ,depthTestEnable
        ,):
    result = _callApi(lib.vkCmdSetDepthTestEnable, commandBuffer,depthTestEnable)


def _wrap_vkCmdSetDepthTestEnable(fn):
    def vkCmdSetDepthTestEnable(
    commandBuffer
            ,depthTestEnable
            ,):
        result = _callApi(fn, commandBuffer,depthTestEnable)


    return vkCmdSetDepthTestEnable
def vkCmdSetDepthWriteEnable(
commandBuffer
        ,depthWriteEnable
        ,):
    result = _callApi(lib.vkCmdSetDepthWriteEnable, commandBuffer,depthWriteEnable)


def _wrap_vkCmdSetDepthWriteEnable(fn):
    def vkCmdSetDepthWriteEnable(
    commandBuffer
            ,depthWriteEnable
            ,):
        result = _callApi(fn, commandBuffer,depthWriteEnable)


    return vkCmdSetDepthWriteEnable
def vkCmdSetDepthCompareOp(
commandBuffer
        ,depthCompareOp
        ,):
    result = _callApi(lib.vkCmdSetDepthCompareOp, commandBuffer,depthCompareOp)


def _wrap_vkCmdSetDepthCompareOp(fn):
    def vkCmdSetDepthCompareOp(
    commandBuffer
            ,depthCompareOp
            ,):
        result = _callApi(fn, commandBuffer,depthCompareOp)


    return vkCmdSetDepthCompareOp
def vkCmdSetDepthBoundsTestEnable(
commandBuffer
        ,depthBoundsTestEnable
        ,):
    result = _callApi(lib.vkCmdSetDepthBoundsTestEnable, commandBuffer,depthBoundsTestEnable)


def _wrap_vkCmdSetDepthBoundsTestEnable(fn):
    def vkCmdSetDepthBoundsTestEnable(
    commandBuffer
            ,depthBoundsTestEnable
            ,):
        result = _callApi(fn, commandBuffer,depthBoundsTestEnable)


    return vkCmdSetDepthBoundsTestEnable
def vkCmdSetStencilTestEnable(
commandBuffer
        ,stencilTestEnable
        ,):
    result = _callApi(lib.vkCmdSetStencilTestEnable, commandBuffer,stencilTestEnable)


def _wrap_vkCmdSetStencilTestEnable(fn):
    def vkCmdSetStencilTestEnable(
    commandBuffer
            ,stencilTestEnable
            ,):
        result = _callApi(fn, commandBuffer,stencilTestEnable)


    return vkCmdSetStencilTestEnable
def vkCmdSetStencilOp(
commandBuffer
        ,faceMask
        ,failOp
        ,passOp
        ,depthFailOp
        ,compareOp
        ,):
    result = _callApi(lib.vkCmdSetStencilOp, commandBuffer,faceMask,failOp,passOp,depthFailOp,compareOp)


def _wrap_vkCmdSetStencilOp(fn):
    def vkCmdSetStencilOp(
    commandBuffer
            ,faceMask
            ,failOp
            ,passOp
            ,depthFailOp
            ,compareOp
            ,):
        result = _callApi(fn, commandBuffer,faceMask,failOp,passOp,depthFailOp,compareOp)


    return vkCmdSetStencilOp
def _wrap_vkCmdSetPatchControlPointsEXT(fn):
    def vkCmdSetPatchControlPointsEXT(
    commandBuffer
            ,patchControlPoints
            ,):
        result = _callApi(fn, commandBuffer,patchControlPoints)


    return vkCmdSetPatchControlPointsEXT
def vkCmdSetRasterizerDiscardEnable(
commandBuffer
        ,rasterizerDiscardEnable
        ,):
    result = _callApi(lib.vkCmdSetRasterizerDiscardEnable, commandBuffer,rasterizerDiscardEnable)


def _wrap_vkCmdSetRasterizerDiscardEnable(fn):
    def vkCmdSetRasterizerDiscardEnable(
    commandBuffer
            ,rasterizerDiscardEnable
            ,):
        result = _callApi(fn, commandBuffer,rasterizerDiscardEnable)


    return vkCmdSetRasterizerDiscardEnable
def vkCmdSetDepthBiasEnable(
commandBuffer
        ,depthBiasEnable
        ,):
    result = _callApi(lib.vkCmdSetDepthBiasEnable, commandBuffer,depthBiasEnable)


def _wrap_vkCmdSetDepthBiasEnable(fn):
    def vkCmdSetDepthBiasEnable(
    commandBuffer
            ,depthBiasEnable
            ,):
        result = _callApi(fn, commandBuffer,depthBiasEnable)


    return vkCmdSetDepthBiasEnable
def _wrap_vkCmdSetLogicOpEXT(fn):
    def vkCmdSetLogicOpEXT(
    commandBuffer
            ,logicOp
            ,):
        result = _callApi(fn, commandBuffer,logicOp)


    return vkCmdSetLogicOpEXT
def vkCmdSetPrimitiveRestartEnable(
commandBuffer
        ,primitiveRestartEnable
        ,):
    result = _callApi(lib.vkCmdSetPrimitiveRestartEnable, commandBuffer,primitiveRestartEnable)


def _wrap_vkCmdSetPrimitiveRestartEnable(fn):
    def vkCmdSetPrimitiveRestartEnable(
    commandBuffer
            ,primitiveRestartEnable
            ,):
        result = _callApi(fn, commandBuffer,primitiveRestartEnable)


    return vkCmdSetPrimitiveRestartEnable
def _wrap_vkCmdSetTessellationDomainOriginEXT(fn):
    def vkCmdSetTessellationDomainOriginEXT(
    commandBuffer
            ,domainOrigin
            ,):
        result = _callApi(fn, commandBuffer,domainOrigin)


    return vkCmdSetTessellationDomainOriginEXT
def _wrap_vkCmdSetDepthClampEnableEXT(fn):
    def vkCmdSetDepthClampEnableEXT(
    commandBuffer
            ,depthClampEnable
            ,):
        result = _callApi(fn, commandBuffer,depthClampEnable)


    return vkCmdSetDepthClampEnableEXT
def _wrap_vkCmdSetPolygonModeEXT(fn):
    def vkCmdSetPolygonModeEXT(
    commandBuffer
            ,polygonMode
            ,):
        result = _callApi(fn, commandBuffer,polygonMode)


    return vkCmdSetPolygonModeEXT
def _wrap_vkCmdSetRasterizationSamplesEXT(fn):
    def vkCmdSetRasterizationSamplesEXT(
    commandBuffer
            ,rasterizationSamples
            ,):
        result = _callApi(fn, commandBuffer,rasterizationSamples)


    return vkCmdSetRasterizationSamplesEXT
def _wrap_vkCmdSetSampleMaskEXT(fn):
    def vkCmdSetSampleMaskEXT(
    commandBuffer
            ,samples
            ,pSampleMask
            ,):
        result = _callApi(fn, commandBuffer,samples,pSampleMask)


    return vkCmdSetSampleMaskEXT
def _wrap_vkCmdSetAlphaToCoverageEnableEXT(fn):
    def vkCmdSetAlphaToCoverageEnableEXT(
    commandBuffer
            ,alphaToCoverageEnable
            ,):
        result = _callApi(fn, commandBuffer,alphaToCoverageEnable)


    return vkCmdSetAlphaToCoverageEnableEXT
def _wrap_vkCmdSetAlphaToOneEnableEXT(fn):
    def vkCmdSetAlphaToOneEnableEXT(
    commandBuffer
            ,alphaToOneEnable
            ,):
        result = _callApi(fn, commandBuffer,alphaToOneEnable)


    return vkCmdSetAlphaToOneEnableEXT
def _wrap_vkCmdSetLogicOpEnableEXT(fn):
    def vkCmdSetLogicOpEnableEXT(
    commandBuffer
            ,logicOpEnable
            ,):
        result = _callApi(fn, commandBuffer,logicOpEnable)


    return vkCmdSetLogicOpEnableEXT
def _wrap_vkCmdSetColorBlendEnableEXT(fn):
    def vkCmdSetColorBlendEnableEXT(
    commandBuffer
            ,firstAttachment
            ,attachmentCount
            ,pColorBlendEnables
            ,):
        result = _callApi(fn, commandBuffer,firstAttachment,attachmentCount,pColorBlendEnables)


    return vkCmdSetColorBlendEnableEXT
def _wrap_vkCmdSetColorBlendEquationEXT(fn):
    def vkCmdSetColorBlendEquationEXT(
    commandBuffer
            ,firstAttachment
            ,attachmentCount
            ,pColorBlendEquations
            ,):
        result = _callApi(fn, commandBuffer,firstAttachment,attachmentCount,pColorBlendEquations)


    return vkCmdSetColorBlendEquationEXT
def _wrap_vkCmdSetColorWriteMaskEXT(fn):
    def vkCmdSetColorWriteMaskEXT(
    commandBuffer
            ,firstAttachment
            ,attachmentCount
            ,pColorWriteMasks
            ,):
        result = _callApi(fn, commandBuffer,firstAttachment,attachmentCount,pColorWriteMasks)


    return vkCmdSetColorWriteMaskEXT
def _wrap_vkCmdSetRasterizationStreamEXT(fn):
    def vkCmdSetRasterizationStreamEXT(
    commandBuffer
            ,rasterizationStream
            ,):
        result = _callApi(fn, commandBuffer,rasterizationStream)


    return vkCmdSetRasterizationStreamEXT
def _wrap_vkCmdSetConservativeRasterizationModeEXT(fn):
    def vkCmdSetConservativeRasterizationModeEXT(
    commandBuffer
            ,conservativeRasterizationMode
            ,):
        result = _callApi(fn, commandBuffer,conservativeRasterizationMode)


    return vkCmdSetConservativeRasterizationModeEXT
def _wrap_vkCmdSetExtraPrimitiveOverestimationSizeEXT(fn):
    def vkCmdSetExtraPrimitiveOverestimationSizeEXT(
    commandBuffer
            ,extraPrimitiveOverestimationSize
            ,):
        result = _callApi(fn, commandBuffer,extraPrimitiveOverestimationSize)


    return vkCmdSetExtraPrimitiveOverestimationSizeEXT
def _wrap_vkCmdSetDepthClipEnableEXT(fn):
    def vkCmdSetDepthClipEnableEXT(
    commandBuffer
            ,depthClipEnable
            ,):
        result = _callApi(fn, commandBuffer,depthClipEnable)


    return vkCmdSetDepthClipEnableEXT
def _wrap_vkCmdSetSampleLocationsEnableEXT(fn):
    def vkCmdSetSampleLocationsEnableEXT(
    commandBuffer
            ,sampleLocationsEnable
            ,):
        result = _callApi(fn, commandBuffer,sampleLocationsEnable)


    return vkCmdSetSampleLocationsEnableEXT
def _wrap_vkCmdSetColorBlendAdvancedEXT(fn):
    def vkCmdSetColorBlendAdvancedEXT(
    commandBuffer
            ,firstAttachment
            ,attachmentCount
            ,pColorBlendAdvanced
            ,):
        result = _callApi(fn, commandBuffer,firstAttachment,attachmentCount,pColorBlendAdvanced)


    return vkCmdSetColorBlendAdvancedEXT
def _wrap_vkCmdSetProvokingVertexModeEXT(fn):
    def vkCmdSetProvokingVertexModeEXT(
    commandBuffer
            ,provokingVertexMode
            ,):
        result = _callApi(fn, commandBuffer,provokingVertexMode)


    return vkCmdSetProvokingVertexModeEXT
def _wrap_vkCmdSetLineRasterizationModeEXT(fn):
    def vkCmdSetLineRasterizationModeEXT(
    commandBuffer
            ,lineRasterizationMode
            ,):
        result = _callApi(fn, commandBuffer,lineRasterizationMode)


    return vkCmdSetLineRasterizationModeEXT
def _wrap_vkCmdSetLineStippleEnableEXT(fn):
    def vkCmdSetLineStippleEnableEXT(
    commandBuffer
            ,stippledLineEnable
            ,):
        result = _callApi(fn, commandBuffer,stippledLineEnable)


    return vkCmdSetLineStippleEnableEXT
def _wrap_vkCmdSetDepthClipNegativeOneToOneEXT(fn):
    def vkCmdSetDepthClipNegativeOneToOneEXT(
    commandBuffer
            ,negativeOneToOne
            ,):
        result = _callApi(fn, commandBuffer,negativeOneToOne)


    return vkCmdSetDepthClipNegativeOneToOneEXT
def _wrap_vkCmdSetViewportWScalingEnableNV(fn):
    def vkCmdSetViewportWScalingEnableNV(
    commandBuffer
            ,viewportWScalingEnable
            ,):
        result = _callApi(fn, commandBuffer,viewportWScalingEnable)


    return vkCmdSetViewportWScalingEnableNV
def _wrap_vkCmdSetViewportSwizzleNV(fn):
    def vkCmdSetViewportSwizzleNV(
    commandBuffer
            ,firstViewport
            ,viewportCount
            ,pViewportSwizzles
            ,):
        result = _callApi(fn, commandBuffer,firstViewport,viewportCount,pViewportSwizzles)


    return vkCmdSetViewportSwizzleNV
def _wrap_vkCmdSetCoverageToColorEnableNV(fn):
    def vkCmdSetCoverageToColorEnableNV(
    commandBuffer
            ,coverageToColorEnable
            ,):
        result = _callApi(fn, commandBuffer,coverageToColorEnable)


    return vkCmdSetCoverageToColorEnableNV
def _wrap_vkCmdSetCoverageToColorLocationNV(fn):
    def vkCmdSetCoverageToColorLocationNV(
    commandBuffer
            ,coverageToColorLocation
            ,):
        result = _callApi(fn, commandBuffer,coverageToColorLocation)


    return vkCmdSetCoverageToColorLocationNV
def _wrap_vkCmdSetCoverageModulationModeNV(fn):
    def vkCmdSetCoverageModulationModeNV(
    commandBuffer
            ,coverageModulationMode
            ,):
        result = _callApi(fn, commandBuffer,coverageModulationMode)


    return vkCmdSetCoverageModulationModeNV
def _wrap_vkCmdSetCoverageModulationTableEnableNV(fn):
    def vkCmdSetCoverageModulationTableEnableNV(
    commandBuffer
            ,coverageModulationTableEnable
            ,):
        result = _callApi(fn, commandBuffer,coverageModulationTableEnable)


    return vkCmdSetCoverageModulationTableEnableNV
def _wrap_vkCmdSetCoverageModulationTableNV(fn):
    def vkCmdSetCoverageModulationTableNV(
    commandBuffer
            ,coverageModulationTableCount
            ,pCoverageModulationTable
            ,):
        result = _callApi(fn, commandBuffer,coverageModulationTableCount,pCoverageModulationTable)


    return vkCmdSetCoverageModulationTableNV
def _wrap_vkCmdSetShadingRateImageEnableNV(fn):
    def vkCmdSetShadingRateImageEnableNV(
    commandBuffer
            ,shadingRateImageEnable
            ,):
        result = _callApi(fn, commandBuffer,shadingRateImageEnable)


    return vkCmdSetShadingRateImageEnableNV
def _wrap_vkCmdSetCoverageReductionModeNV(fn):
    def vkCmdSetCoverageReductionModeNV(
    commandBuffer
            ,coverageReductionMode
            ,):
        result = _callApi(fn, commandBuffer,coverageReductionMode)


    return vkCmdSetCoverageReductionModeNV
def _wrap_vkCmdSetRepresentativeFragmentTestEnableNV(fn):
    def vkCmdSetRepresentativeFragmentTestEnableNV(
    commandBuffer
            ,representativeFragmentTestEnable
            ,):
        result = _callApi(fn, commandBuffer,representativeFragmentTestEnable)


    return vkCmdSetRepresentativeFragmentTestEnableNV
def vkCreatePrivateDataSlot(
device
        ,pCreateInfo
        ,pAllocator
        ,pPrivateDataSlot=None
        ,):

    custom_return = True
    if not pPrivateDataSlot:
        pPrivateDataSlot = ffi.new('VkPrivateDataSlot*')
        custom_return = False

    result = _callApi(lib.vkCreatePrivateDataSlot, device,pCreateInfo,pAllocator,pPrivateDataSlot)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pPrivateDataSlot

    return pPrivateDataSlot[0]



def _wrap_vkCreatePrivateDataSlot(fn):
    def vkCreatePrivateDataSlot(
    device
            ,pCreateInfo
            ,pAllocator
            ,pPrivateDataSlot=None
            ,):

        custom_return = True
        if not pPrivateDataSlot:
            pPrivateDataSlot = ffi.new('VkPrivateDataSlot*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pPrivateDataSlot)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pPrivateDataSlot

        return pPrivateDataSlot[0]



    return vkCreatePrivateDataSlot
def vkDestroyPrivateDataSlot(
device
        ,privateDataSlot
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyPrivateDataSlot, device,privateDataSlot,pAllocator)


def _wrap_vkDestroyPrivateDataSlot(fn):
    def vkDestroyPrivateDataSlot(
    device
            ,privateDataSlot
            ,pAllocator
            ,):
        result = _callApi(fn, device,privateDataSlot,pAllocator)


    return vkDestroyPrivateDataSlot
def vkSetPrivateData(
device
        ,objectType
        ,objectHandle
        ,privateDataSlot
        ,data
        ,):
    result = _callApi(lib.vkSetPrivateData, device,objectType,objectHandle,privateDataSlot,data)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def _wrap_vkSetPrivateData(fn):
    def vkSetPrivateData(
    device
            ,objectType
            ,objectHandle
            ,privateDataSlot
            ,data
            ,):
        result = _callApi(fn, device,objectType,objectHandle,privateDataSlot,data)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkSetPrivateData
def vkGetPrivateData(
device
        ,objectType
        ,objectHandle
        ,privateDataSlot
        ,pData=None
        ,):

    custom_return = True
    if not pData:
        pData = ffi.new('uint64_t*')
        custom_return = False

    result = _callApi(lib.vkGetPrivateData, device,objectType,objectHandle,privateDataSlot,pData)

    if custom_return:
        return pData

    return pData[0]



def _wrap_vkGetPrivateData(fn):
    def vkGetPrivateData(
    device
            ,objectType
            ,objectHandle
            ,privateDataSlot
            ,pData=None
            ,):

        custom_return = True
        if not pData:
            pData = ffi.new('uint64_t*')
            custom_return = False

        result = _callApi(fn, device,objectType,objectHandle,privateDataSlot,pData)

        if custom_return:
            return pData

        return pData[0]



    return vkGetPrivateData
def vkCmdCopyBuffer2(
commandBuffer
        ,pCopyBufferInfo
        ,):
    result = _callApi(lib.vkCmdCopyBuffer2, commandBuffer,pCopyBufferInfo)


def _wrap_vkCmdCopyBuffer2(fn):
    def vkCmdCopyBuffer2(
    commandBuffer
            ,pCopyBufferInfo
            ,):
        result = _callApi(fn, commandBuffer,pCopyBufferInfo)


    return vkCmdCopyBuffer2
def vkCmdCopyImage2(
commandBuffer
        ,pCopyImageInfo
        ,):
    result = _callApi(lib.vkCmdCopyImage2, commandBuffer,pCopyImageInfo)


def _wrap_vkCmdCopyImage2(fn):
    def vkCmdCopyImage2(
    commandBuffer
            ,pCopyImageInfo
            ,):
        result = _callApi(fn, commandBuffer,pCopyImageInfo)


    return vkCmdCopyImage2
def vkCmdBlitImage2(
commandBuffer
        ,pBlitImageInfo
        ,):
    result = _callApi(lib.vkCmdBlitImage2, commandBuffer,pBlitImageInfo)


def _wrap_vkCmdBlitImage2(fn):
    def vkCmdBlitImage2(
    commandBuffer
            ,pBlitImageInfo
            ,):
        result = _callApi(fn, commandBuffer,pBlitImageInfo)


    return vkCmdBlitImage2
def vkCmdCopyBufferToImage2(
commandBuffer
        ,pCopyBufferToImageInfo
        ,):
    result = _callApi(lib.vkCmdCopyBufferToImage2, commandBuffer,pCopyBufferToImageInfo)


def _wrap_vkCmdCopyBufferToImage2(fn):
    def vkCmdCopyBufferToImage2(
    commandBuffer
            ,pCopyBufferToImageInfo
            ,):
        result = _callApi(fn, commandBuffer,pCopyBufferToImageInfo)


    return vkCmdCopyBufferToImage2
def vkCmdCopyImageToBuffer2(
commandBuffer
        ,pCopyImageToBufferInfo
        ,):
    result = _callApi(lib.vkCmdCopyImageToBuffer2, commandBuffer,pCopyImageToBufferInfo)


def _wrap_vkCmdCopyImageToBuffer2(fn):
    def vkCmdCopyImageToBuffer2(
    commandBuffer
            ,pCopyImageToBufferInfo
            ,):
        result = _callApi(fn, commandBuffer,pCopyImageToBufferInfo)


    return vkCmdCopyImageToBuffer2
def vkCmdResolveImage2(
commandBuffer
        ,pResolveImageInfo
        ,):
    result = _callApi(lib.vkCmdResolveImage2, commandBuffer,pResolveImageInfo)


def _wrap_vkCmdResolveImage2(fn):
    def vkCmdResolveImage2(
    commandBuffer
            ,pResolveImageInfo
            ,):
        result = _callApi(fn, commandBuffer,pResolveImageInfo)


    return vkCmdResolveImage2
def _wrap_vkCmdRefreshObjectsKHR(fn):
    def vkCmdRefreshObjectsKHR(
    commandBuffer
            ,pRefreshObjects
            ,):
        result = _callApi(fn, commandBuffer,pRefreshObjects)


    return vkCmdRefreshObjectsKHR
def _wrap_vkGetPhysicalDeviceRefreshableObjectTypesKHR(fn):
    def vkGetPhysicalDeviceRefreshableObjectTypesKHR(
    physicalDevice
            ,):

        pRefreshableObjectTypeCount = ffi.new('uint32_t*')
        pRefreshableObjectTypes = ffi.NULL

        result = _callApi(fn, physicalDevice,pRefreshableObjectTypeCount,pRefreshableObjectTypes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pRefreshableObjectTypes = ffi.new('VkObjectType[]', pRefreshableObjectTypeCount[0])
        result = _callApi(fn, physicalDevice,pRefreshableObjectTypeCount,pRefreshableObjectTypes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pRefreshableObjectTypes


    return vkGetPhysicalDeviceRefreshableObjectTypesKHR
def _wrap_vkCmdSetFragmentShadingRateKHR(fn):
    def vkCmdSetFragmentShadingRateKHR(
    commandBuffer
            ,pFragmentSize
            ,combinerOps
            ,):
        result = _callApi(fn, commandBuffer,pFragmentSize,combinerOps)


    return vkCmdSetFragmentShadingRateKHR
def _wrap_vkGetPhysicalDeviceFragmentShadingRatesKHR(fn):
    def vkGetPhysicalDeviceFragmentShadingRatesKHR(
    physicalDevice
            ,):

        pFragmentShadingRateCount = ffi.new('uint32_t*')
        pFragmentShadingRates = ffi.NULL

        result = _callApi(fn, physicalDevice,pFragmentShadingRateCount,pFragmentShadingRates)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pFragmentShadingRates = ffi.new('VkPhysicalDeviceFragmentShadingRateKHR[]', pFragmentShadingRateCount[0])
        result = _callApi(fn, physicalDevice,pFragmentShadingRateCount,pFragmentShadingRates)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pFragmentShadingRates


    return vkGetPhysicalDeviceFragmentShadingRatesKHR
def _wrap_vkCmdSetFragmentShadingRateEnumNV(fn):
    def vkCmdSetFragmentShadingRateEnumNV(
    commandBuffer
            ,shadingRate
            ,combinerOps
            ,):
        result = _callApi(fn, commandBuffer,shadingRate,combinerOps)


    return vkCmdSetFragmentShadingRateEnumNV
def _wrap_vkGetAccelerationStructureBuildSizesKHR(fn):
    def vkGetAccelerationStructureBuildSizesKHR(
    device
            ,buildType
            ,pBuildInfo
            ,pMaxPrimitiveCounts
            ,pSizeInfo=None
            ,):

        custom_return = True
        if not pSizeInfo:
            pSizeInfo = ffi.new('VkAccelerationStructureBuildSizesInfoKHR*')
            custom_return = False

        result = _callApi(fn, device,buildType,pBuildInfo,pMaxPrimitiveCounts,pSizeInfo)

        if custom_return:
            return pSizeInfo

        return pSizeInfo[0]



    return vkGetAccelerationStructureBuildSizesKHR
def _wrap_vkCmdSetVertexInputEXT(fn):
    def vkCmdSetVertexInputEXT(
    commandBuffer
            ,vertexBindingDescriptionCount
            ,pVertexBindingDescriptions
            ,vertexAttributeDescriptionCount
            ,pVertexAttributeDescriptions
            ,):
        result = _callApi(fn, commandBuffer,vertexBindingDescriptionCount,pVertexBindingDescriptions,vertexAttributeDescriptionCount,pVertexAttributeDescriptions)


    return vkCmdSetVertexInputEXT
def _wrap_vkCmdSetColorWriteEnableEXT(fn):
    def vkCmdSetColorWriteEnableEXT(
    commandBuffer
            ,attachmentCount
            ,pColorWriteEnables
            ,):
        result = _callApi(fn, commandBuffer,attachmentCount,pColorWriteEnables)


    return vkCmdSetColorWriteEnableEXT
def vkCmdSetEvent2(
commandBuffer
        ,event
        ,pDependencyInfo
        ,):
    result = _callApi(lib.vkCmdSetEvent2, commandBuffer,event,pDependencyInfo)


def _wrap_vkCmdSetEvent2(fn):
    def vkCmdSetEvent2(
    commandBuffer
            ,event
            ,pDependencyInfo
            ,):
        result = _callApi(fn, commandBuffer,event,pDependencyInfo)


    return vkCmdSetEvent2
def vkCmdResetEvent2(
commandBuffer
        ,event
        ,stageMask
        ,):
    result = _callApi(lib.vkCmdResetEvent2, commandBuffer,event,stageMask)


def _wrap_vkCmdResetEvent2(fn):
    def vkCmdResetEvent2(
    commandBuffer
            ,event
            ,stageMask
            ,):
        result = _callApi(fn, commandBuffer,event,stageMask)


    return vkCmdResetEvent2
def vkCmdWaitEvents2(
commandBuffer
        ,eventCount
        ,pEvents
        ,pDependencyInfos
        ,):
    result = _callApi(lib.vkCmdWaitEvents2, commandBuffer,eventCount,pEvents,pDependencyInfos)


def _wrap_vkCmdWaitEvents2(fn):
    def vkCmdWaitEvents2(
    commandBuffer
            ,eventCount
            ,pEvents
            ,pDependencyInfos
            ,):
        result = _callApi(fn, commandBuffer,eventCount,pEvents,pDependencyInfos)


    return vkCmdWaitEvents2
def vkCmdPipelineBarrier2(
commandBuffer
        ,pDependencyInfo
        ,):
    result = _callApi(lib.vkCmdPipelineBarrier2, commandBuffer,pDependencyInfo)


def _wrap_vkCmdPipelineBarrier2(fn):
    def vkCmdPipelineBarrier2(
    commandBuffer
            ,pDependencyInfo
            ,):
        result = _callApi(fn, commandBuffer,pDependencyInfo)


    return vkCmdPipelineBarrier2
def vkQueueSubmit2(
queue
        ,submitCount
        ,pSubmits
        ,fence
        ,):
    result = _callApi(lib.vkQueueSubmit2, queue,submitCount,pSubmits,fence)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def _wrap_vkQueueSubmit2(fn):
    def vkQueueSubmit2(
    queue
            ,submitCount
            ,pSubmits
            ,fence
            ,):
        result = _callApi(fn, queue,submitCount,pSubmits,fence)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkQueueSubmit2
def vkCmdWriteTimestamp2(
commandBuffer
        ,stage
        ,queryPool
        ,query
        ,):
    result = _callApi(lib.vkCmdWriteTimestamp2, commandBuffer,stage,queryPool,query)


def _wrap_vkCmdWriteTimestamp2(fn):
    def vkCmdWriteTimestamp2(
    commandBuffer
            ,stage
            ,queryPool
            ,query
            ,):
        result = _callApi(fn, commandBuffer,stage,queryPool,query)


    return vkCmdWriteTimestamp2
def _wrap_vkCmdWriteBufferMarker2AMD(fn):
    def vkCmdWriteBufferMarker2AMD(
    commandBuffer
          