import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Gio from 'gi://Gio';

import { ClipboardHistory, ItemType, getDataPath } from '../common/constants.js';
import { int32ParamSpec, registerClass } from '../common/gjs.js';
import {
	add_expr_value,
	async_statement_execute_non_select,
	async_statement_execute_select,
	convert_datetime,
	new_connection,
	open_async,
	unescape_sql,
} from './gda.js';
import { getLinkImagePath } from './link.js';

var __decorate =
	(this && this.__decorate) ||
	function (decorators, target, key, desc) {
		var c = arguments.length,
			r = c < 3 ? target : desc === null ? (desc = Object.getOwnPropertyDescriptor(target, key)) : desc,
			d;
		if (typeof Reflect === 'object' && typeof Reflect.decorate === 'function')
			r = Reflect.decorate(decorators, target, key, desc);
		else
			for (var i = decorators.length - 1; i >= 0; i--)
				if ((d = decorators[i])) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
		return (c > 3 && r && Object.defineProperty(target, key, r), r);
	};

export const FileOperation = {
	Copy: 'COPY',
	Cut: 'CUT',
};

let ClipboardEntry = class ClipboardEntry extends GObject.Object {
	_id;

	constructor(id, type, content, pinned, tag, datetime, metadata = null) {
		super();
		this._id = id;
		this.type = type;
		this.content = content;
		this.pinned = pinned;
		this.tag = tag;
		this.datetime = datetime;
		this.metadata = metadata;
	}

	get id() {
		return this._id;
	}
};
ClipboardEntry = __decorate(
	[
		registerClass({
			Properties: {
				id: int32ParamSpec('id', GObject.ParamFlags.READABLE, 0),
				type: GObject.ParamSpec.string('type', null, null, GObject.ParamFlags.READWRITE, ItemType.Text),
				content: GObject.ParamSpec.string('content', null, null, GObject.ParamFlags.READWRITE, ''),
				pinned: GObject.ParamSpec.boolean('pinned', null, null, GObject.ParamFlags.READWRITE, false),
				tag: GObject.ParamSpec.string('tag', null, null, GObject.ParamFlags.READWRITE, ''),
				datetime: GObject.ParamSpec.boxed('datetime', null, null, GObject.ParamFlags.READWRITE, GLib.DateTime),
				metadata: GObject.ParamSpec.jsobject('metadata', null, null, GObject.ParamFlags.READWRITE),
			},
			Signals: {
				delete: {},
			},
		}),
	],
	ClipboardEntry,
);

export { ClipboardEntry };

export class ClipboardEntryTracker {
	ext;
	_database;
	_entries = new Map();

	constructor(ext) {
		this.ext = ext;
	}

	async init() {
		if (this._database) {
			await this.clear();
			await this.destroy();
		}
		try {
			// Check if DEBUG_COPYOUS_GDA_VERSION is set
			const environment = GLib.get_environ();
			const gdaVersion = GLib.environ_getenv(environment, 'DEBUG_COPYOUS_GDA_VERSION');
			if (gdaVersion) {
				imports.package.require({ Gda: gdaVersion });
			}
		} catch (err) {
			this.ext.logger.warn(err);
		}
		let gda = null;
		try {
			gda = (await import('gi://Gda')).default;
		} catch {
			this.ext.logger.error('Failed to load Gda');
			if (!this.ext.settings.get_boolean('disable-gda-warning')) {
				this.ext.notificationManager?.warning(
					_('Failed to load Gda'),
					_('Clipboard history will be disabled'),
					[_('Disable Warning'), () => this.ext.settings.set_boolean('disable-gda-warning', true)],
				);
			}
		}
		if (gda) {
			try {
				// Create database
				this.ext.logger.log('Using Gda', gda.__version__, 'database');
				this._database = new GdaDatabase(this.ext, gda);
				await this._database.init();

				// First get entries and track them
				const entries = await this._database.entries();
				entries.forEach((entry) => this.track(entry));

				// Then delete the oldest entries so that images are deleted
				await this.deleteOldest();
				return entries;
			} catch (e) {
				this.ext.logger.error('Failed to initialize Gda database', e);
				this.ext.notificationManager?.warning(
					_('Failed to load database'),
					_('Clipboard history will be disabled'),
				);
			}
		}
		this.ext.logger.log('Using in-memory database');
		this._database = new MemoryDatabase();
		await this._database.init();
		return [];
	}

	async clear(history = null) {
		if (!this._database) return;
		history ??= this.ext.settings.get_enum('clipboard-history');
		const deleted = await this._database.clear(history);
		deleted.forEach((id) => this.deleteFromDatabase(id));
	}

	async destroy() {
		await this._database?.close();
		this._database = undefined;
	}

	/**
	 * Inserts an entry into the database
	 * @param type The type of the entry
	 * @param content The content of the entry
	 * @param metadata The metadata of the entry
	 * @returns The inserted entry or null if the entry could not be inserted or is already tracked
	 */
	async insert(type, content, metadata = null) {
		const id = await this._database?.selectConflict({ type, content });
		if (id) {
			// Check if the entry is already tracked
			const trackedEntry = this._entries.get(id);
			if (trackedEntry) {
				trackedEntry.datetime = GLib.DateTime.new_now_utc();
				return null;
			}
		}
		const entry = await this._database?.insert(type, content, metadata);
		if (!entry) return null;

		// Start tracking it
		this.track(entry);

		// Also delete oldest entries
		await this.deleteOldest();
		return entry;
	}

	checkOldest() {
		const M = this.ext.settings.get_int('history-time');
		if (M === 0) return false;
		const now = GLib.DateTime.new_now_utc();
		const olderThan = now.add_minutes(-M);
		for (const entry of this._entries.values()) {
			if (entry.pinned || entry.tag) continue;
			if (entry.datetime.compare(olderThan) < 0) return true;
		}
		return false;
	}

	async deleteOldest() {
		const N = this.ext.settings.get_int('history-length');
		const M = this.ext.settings.get_int('history-time');
		const deleted = await this._database?.deleteOldest(N, M);
		if (deleted) deleted.forEach((id) => this.deleteFromDatabase(id));
	}

	track(entry) {
		entry.connect('notify::content', async () => {
			const id = await this._database?.updateProperty(entry, 'content');

			// If entry conflicts with another entry, delete it
			if (id !== undefined && id >= 0) {
				entry.emit('delete');

				// Update the date of the other entry
				const conflicted = this._entries.get(id);
				if (conflicted) {
					conflicted.datetime = entry.datetime;
				}
			}
		});
		entry.connect('notify::pinned', () => this._database?.updateProperty(entry, 'pinned'));
		entry.connect('notify::tag', () => this._database?.updateProperty(entry, 'tag'));
		entry.connect('notify::datetime', () => this._database?.updateProperty(entry, 'datetime'));
		entry.connect('notify::metadata', () => this._database?.updateProperty(entry, 'metadata'));
		entry.connect('delete', () => this.delete(entry));
		this._entries?.set(entry.id, entry);
	}

	async delete(entry) {
		if (entry.type === ItemType.Image) {
			// Delete image
			try {
				const file = Gio.File.new_for_uri(entry.content);
				if (file.query_exists(null)) {
					file.delete(null);
				}
			} catch {
				this.ext.logger.error('Failed to delete image', entry.content);
			}
		} else if (entry.type === ItemType.Link && entry.metadata) {
			// Delete thumbnail image
			const metadata = { image: null, ...entry.metadata };
			if (metadata.image) {
				try {
					const file = getLinkImagePath(this.ext, metadata.image);
					if (file?.query_exists(null)) {
						file.delete(null);
					}
				} catch {
					this.ext.logger.error('Failed to delete thumbnail image', metadata.image);
				}
			}
		}

		// Delete from database if not deleted already
		if (this._entries.has(entry.id)) {
			await this._database?.delete(entry);
			this._entries.delete(entry.id);
		}
	}

	deleteFromDatabase(id) {
		const entry = this._entries.get(id);
		if (entry) {
			this._entries.delete(id);
			entry.emit('delete');
		}
	}
}

/**
 * In memory database
 */
class MemoryDatabase {
	_entries = new Map();
	_keys = new Map();
	_id = 0;

	constructor() {}

	async init() {}

	clear(history) {
		let deleted = [];
		switch (history) {
			case ClipboardHistory.Clear:
				deleted = Array.from(this._keys.keys());
				this._entries.clear();
				this._keys.clear();
				break;
			case ClipboardHistory.KeepPinnedAndTagged:
				deleted = [];
				for (const [key, entry] of this._entries) {
					if (!(entry.pinned || entry.tag)) {
						this._entries.delete(key);
						this._keys.delete(entry.id);
						deleted.push(entry.id);
					}
				}
				break;
			case ClipboardHistory.KeepAll:
				break;
		}
		return Promise.resolve(deleted);
	}

	async close() {
		await this.clear(ClipboardHistory.Clear);
	}

	entries() {
		const entries = Array.from(this._entries.values()).sort((a, b) => b.datetime.compare(a.datetime));
		return Promise.resolve(entries);
	}

	selectConflict(entry) {
		const key = `${entry.type}:${entry.content}`;
		return Promise.resolve(this._entries.get(key)?.id ?? null);
	}

	insert(type, content, metadata = null) {
		const key = `${type}:${content}`;
		const entry = this._entries.get(key);
		if (entry) {
			return Promise.resolve(null);
		} else {
			const newEntry = new ClipboardEntry(
				this._id++,
				type,
				content,
				false,
				null,
				GLib.DateTime.new_now_utc(),
				metadata,
			);
			this._entries.set(key, newEntry);
			this._keys.set(newEntry.id, key);
			return Promise.resolve(newEntry);
		}
	}

	updateProperty(entry, property) {
		if (property !== 'content') return Promise.resolve(-1);
		const key = `${entry.type}:${entry.content}`;
		const existingEntry = this._entries.get(key);
		if (existingEntry) {
			return Promise.resolve(existingEntry.id);
		} else {
			const prevKey = this._keys.get(entry.id);
			if (prevKey) this._entries.delete(prevKey);
			this._entries.set(key, entry);
			this._keys.set(entry.id, key);
			return Promise.resolve(-1);
		}
	}

	delete(entry) {
		const key = this._keys.get(entry.id);
		this._keys.delete(entry.id);
		if (key) this._entries.delete(key);
		return Promise.resolve();
	}

	async deleteOldest(offset, olderThanMinutes) {
		const entries = await this.entries();
		let deleted = entries
			.filter((e) => !(e.pinned || e.tag))
			.map((e) => e.id)
			.slice(offset);
		if (olderThanMinutes > 0) {
			const now = GLib.DateTime.new_now_utc();
			const olderThan = now.add_minutes(-olderThanMinutes);
			deleted = [
				...new Set([
					...deleted,
					...entries
						.filter((e) => !(e.pinned || e.tag) && e.datetime.compare(olderThan) < 0)
						.map((e) => e.id),
				]),
			];
		}
		for (const id of deleted) {
			const key = this._keys.get(id);
			this._keys.delete(id);
			if (key) this._entries.delete(key);
		}
		return deleted;
	}
}

// Remove double backslashes since libgda's sqlite escaping is broken
function unescapeContent(content) {
	return content.replace(/\\\\/g, '\\');
}

/**
 * Database with Gda backend
 */
export class GdaDatabase {
	ext;
	_Gda;
	_connection;
	_cancellable = new Gio.Cancellable();

	constructor(ext, gda) {
		this.ext = ext;
		this._Gda = gda;
		const location = this.ext.settings.get_string('database-location');
		const inMemory = this.ext.settings.get_boolean('in-memory-database');

		// Check if DEBUG_COPYOUS_DBPATH is set
		const environment = GLib.get_environ();
		const debugPath = GLib.environ_getenv(environment, 'DEBUG_COPYOUS_DBPATH');
		if (debugPath) ext.logger.log('Using debug database');

		// Get database location or use default ${XDG_DATA_HOME}/${EXTENSION UUID}
		const database = Gio.File.new_for_path(debugPath ?? location);
		const dir = database.get_parent() ?? getDataPath(ext);

		// Use in memory database :memory:, database name, or default clipboard.db
		const file = inMemory ? ':memory:' : (database.get_basename() ?? 'clipboard.db');

		// Create database directory
		if (!dir.query_exists(null)) {
			dir.make_directory_with_parents(null);
		}

		// Establish connection
		const cncString = `DB_DIR=${dir.get_path()};DB_NAME=${file.replace(/\.db$/, '')}`;
		this._connection = new_connection(this._Gda, cncString);
	}

	async init() {
		await open_async(this._connection);
		const [stmt] = this._connection.parse_sql_string(`
			CREATE TABLE IF NOT EXISTS 'clipboard' (
				'id'       integer   NOT NULL UNIQUE PRIMARY KEY AUTOINCREMENT,
				'type'     text      NOT NULL,
				'content'  text      NOT NULL,
				'pinned'   boolean   NOT NULL,
				'tag'      text,
				'datetime' timestamp NOT NULL,
				'metadata' text,
				UNIQUE ('type', 'content')
			);
		`);
		await async_statement_execute_non_select(this._Gda, this._connection, stmt, this._cancellable);
	}

	async clear(history) {
		try {
			if (history === ClipboardHistory.KeepAll) {
				return [];
			}

			// SELECT id FROM table (WHERE NOT (pinned == true OR tag IS NOT NULL))?
			const [selectBuilder, where] = this.selectToDeleteBuilder(history === ClipboardHistory.KeepPinnedAndTagged);
			const selectStmt = selectBuilder.get_statement();
			const datamodel = await async_statement_execute_select(
				this._Gda,
				this._connection,
				selectStmt,
				this._cancellable,
			);
			const deleted = [];
			const iter = datamodel.create_iter();
			while (iter.move_next()) {
				deleted.push(iter.get_value_for_field('id'));
			}

			// Only delete if there are entries to delete
			if (deleted.length > 0) {
				// DELETE FROM table (WHERE ...)? RETURNING id;
				const deleteBuilder = new this._Gda.SqlBuilder({ stmt_type: this._Gda.SqlStatementType.DELETE });
				deleteBuilder.set_table('clipboard');
				if (where) {
					deleteBuilder.set_where(deleteBuilder.import_expression_from_builder(selectBuilder, where));
				}
				const deleteStmt = deleteBuilder.get_statement();
				await async_statement_execute_non_select(this._Gda, this._connection, deleteStmt, this._cancellable);
			}
			return deleted;
		} catch (e) {
			this.ext.logger.error('Failed to clear clipboard', e);
		}
		return [];
	}

	close() {
		this._connection.close();
		this._cancellable.cancel();
		return Promise.resolve();
	}

	async entries() {
		try {
			// SELECT * FROM clipboard
			const builder = new this._Gda.SqlBuilder({ stmt_type: this._Gda.SqlStatementType.SELECT });
			builder.select_add_target('clipboard', null);
			builder.select_add_field('id', null, null);
			builder.select_add_field('type', null, null);
			builder.select_add_field('content', null, null);
			builder.select_add_field('pinned', null, null);
			builder.select_add_field('tag', null, null);
			const datetimeId = builder.select_add_field('datetime', null, null);
			builder.select_add_field('metadata', null, null);
			builder.select_order_by(datetimeId, false, null);
			const stmt = builder.get_statement();
			const dataModel = await async_statement_execute_select(
				this._Gda,
				this._connection,
				stmt,
				this._cancellable,
			);
			const entries = [];
			const iter = dataModel.create_iter();
			while (iter.move_next()) {
				const id = iter.get_value_for_field('id');
				const type = iter.get_value_for_field('type');
				const content = unescapeContent(iter.get_value_for_field('content'));
				const pinned = iter.get_value_for_field('pinned');
				const tag = iter.get_value_for_field('tag');
				let datetime = iter.get_value_for_field('datetime');
				const metadata = iter.get_value_for_field('metadata');
				if ('Timestamp' in this._Gda && datetime instanceof this._Gda.Timestamp) {
					const timezone = GLib.TimeZone.new_offset(datetime.timezone);
					datetime = GLib.DateTime.new(
						timezone,
						datetime.year,
						datetime.month,
						datetime.day,
						datetime.hour,
						datetime.minute,
						datetime.second,
					);
				}
				let metadataObj = null;
				if (metadata) {
					try {
						const json = JSON.parse(metadata);
						if (json) {
							metadataObj = json;
						}
					} catch {
						this.ext.logger.error('Failed to parse metadata');
					}
				}
				entries.push(new ClipboardEntry(id, type, content, pinned, tag, datetime, metadataObj));
			}
			return entries;
		} catch (e) {
			this.ext.logger.error('Failed to get clipboard entries', e);
		}
		return [];
	}

	async selectConflict(entry) {
		try {
			// SELECT id FROM table WHERE type == entry.type AND content == entry.content LIMIT 1
			const builder = new this._Gda.SqlBuilder({ stmt_type: this._Gda.SqlStatementType.SELECT });
			builder.select_add_target('clipboard', null);
			builder.select_add_field('id', null, null);
			builder.set_where(
				builder.add_cond(
					this._Gda.SqlOperatorType.AND,
					builder.add_cond(
						this._Gda.SqlOperatorType.EQ,
						builder.add_id('type'),
						add_expr_value(builder, entry.type),
						0,
					),
					builder.add_cond(
						this._Gda.SqlOperatorType.EQ,
						builder.add_id('content'),
						add_expr_value(builder, entry.content),
						0,
					),
					0,
				),
			);
			builder.select_set_limit(add_expr_value(builder, 1), add_expr_value(builder, 0));

			// Get id
			const stmt = builder.get_statement();
			const datamodel = await async_statement_execute_select(
				this._Gda,
				this._connection,
				stmt,
				this._cancellable,
			);
			const iter = datamodel.create_iter();
			if (iter.move_next()) {
				return iter.get_value_for_field('id');
			}
			return null;
		} catch (e) {
			this.ext.logger.error('Failed to select conflicting entry', e);
		}
		return null;
	}

	/// Note: content will be inserted incorrectly into the database since libgda escapes sqlite incorrectly
	async insert(type, content, metadata = null) {
		try {
			// INSERT INTO table (type, content, pinned, tag, datetime, metadata)
			// VALUES (entry.type, entry.content, entry.pinned, entry.tag, entry.datetime, entry.metadata)
			const builder = new this._Gda.SqlBuilder({
				stmt_type: this._Gda.SqlStatementType.INSERT,
			});
			builder.set_table('clipboard');
			builder.add_field_value_as_gvalue('type', type);

			// NOTE: content will be inserted incorrectly
			builder.add_field_value_as_gvalue('content', content);
			builder.add_field_value_as_gvalue('pinned', false);
			const datetime = GLib.DateTime.new_now_utc();
			builder.add_field_value_as_gvalue('datetime', convert_datetime(datetime));
			if (metadata) builder.add_field_value_as_gvalue('metadata', JSON.stringify(metadata));

			// Execute
			const stmt = builder.get_statement();
			const [, row] = await async_statement_execute_non_select(
				this._Gda,
				this._connection,
				stmt,
				this._cancellable,
			);
			const id = row?.get_nth_holder(0).get_value();
			if (id == null) return null;
			return new ClipboardEntry(id, type, content, false, null, datetime, metadata);
		} catch (e) {
			this.ext.logger.error('Failed to insert entry', e);
		}
		return null;
	}

	async updateProperty(entry, property) {
		try {
			let value = entry[property] ?? 'NULL';
			if (property === 'metadata') value = JSON.stringify(entry[property]);
			else if (property === 'datetime') value = convert_datetime(entry[property]);

			// UPDATE table
			// SET property = entry.property
			// WHERE id == entry.id
			const builder = new this._Gda.SqlBuilder({
				stmt_type: this._Gda.SqlStatementType.UPDATE,
			});
			builder.set_table('clipboard');
			builder.add_field_value_as_gvalue(property, value);
			builder.set_where(
				builder.add_cond(
					this._Gda.SqlOperatorType.EQ,
					builder.add_id('id'),
					add_expr_value(builder, entry.id),
					0,
				),
			);

			// Escape the null value since the bindings for Gda5 do not support Gda.Null
			const stmt = unescape_sql(this._connection, builder);
			const [rows] = await async_statement_execute_non_select(
				this._Gda,
				this._connection,
				stmt,
				this._cancellable,
			);
			if (rows !== -1 || (property !== 'type' && property !== 'content')) {
				return -1; // success
			}

			// Return the id of the conflicting entry
			const id = await this.selectConflict(entry);
			return id ?? -1;
		} catch (e) {
			this.ext.logger.error(`Failed to update property "${property}" for entry ${entry.id}`, e);
		}
		return -1;
	}

	async delete(entry) {
		try {
			// DELETE FROM table WHERE id == entry.id
			const builder = new this._Gda.SqlBuilder({
				stmt_type: this._Gda.SqlStatementType.DELETE,
			});
			builder.set_table('clipboard');
			builder.set_where(
				builder.add_cond(
					this._Gda.SqlOperatorType.EQ,
					builder.add_id('id'),
					add_expr_value(builder, entry.id),
					0,
				),
			);
			const stmt = builder.get_statement();
			await async_statement_execute_non_select(this._Gda, this._connection, stmt, this._cancellable);
		} catch (e) {
			this.ext.logger.error(`Failed to delete entry ${entry.id}`, e);
		}
	}

	async deleteOldest(offset, olderThanMinutes) {
		try {
			// WITH select1 AS (...) (SELECT id FROM select1) UNION (select2)
			const selectBuilder = new this._Gda.SqlBuilder({
				stmt_type: this._Gda.SqlStatementType.COMPOUND,
			});
			selectBuilder.compound_set_type(this._Gda.SqlStatementCompoundType.UNION);

			// SELECT id FROM table WHERE NOT (pinned == true OR tag IS NOT NULL) ORDER BY datetime LIMIT -1 OFFSET offset
			const [select1Builder] = this.selectToDeleteBuilder();
			select1Builder.select_order_by(select1Builder.add_id('datetime'), false, null);
			select1Builder.select_set_limit(add_expr_value(select1Builder, -1), add_expr_value(select1Builder, offset));

			// SELECT id FROM table WHERE NOT (pinned == true OR tag IS NOT NULL) AND datetime < DATETIME('now', '-n minutes')
			let selectStmt;
			if (olderThanMinutes > 0) {
				// Workaround for ORDER BY not working inside compound selector and add_subselect not being exposed in Gda 5.0
				const workAroundBuilder = new this._Gda.SqlBuilder({ stmt_type: this._Gda.SqlStatementType.SELECT });
				workAroundBuilder.select_add_field('id', null, null);
				workAroundBuilder.select_add_target('select1', null);
				selectBuilder.compound_add_sub_select_from_builder(workAroundBuilder);
				const [select2Builder] = this.selectToDeleteBuilder(true, olderThanMinutes);
				selectBuilder.compound_add_sub_select_from_builder(select2Builder);

				// SELECT id FROM (SELECT id FROM table WHERE ...)
				const select1Sql = this._connection.statement_to_sql(select1Builder.get_statement(), null, null)[0];
				const selectSql = this._connection.statement_to_sql(selectBuilder.get_statement(), null, null)[0];
				selectStmt = this._connection.parse_sql_string(selectSql.replace('select1', `(${select1Sql})`))[0];
			} else {
				// Ignore compound selector
				selectStmt = select1Builder.get_statement();
			}

			// Run select
			const datamodel = await async_statement_execute_select(
				this._Gda,
				this._connection,
				selectStmt,
				this._cancellable,
			);

			// DELETE FROM table WHERE id IN (select)
			// add_subselect is not exposed as a javascript binding in Gda 5.0
			const selectSql = this._connection.statement_to_sql(selectStmt, selectStmt.get_parameters()[1], null)[0];
			const [deleteStmt] = this._connection.parse_sql_string(`DELETE FROM clipboard WHERE id IN (${selectSql})`);
			const [rows] = await async_statement_execute_non_select(
				this._Gda,
				this._connection,
				deleteStmt,
				this._cancellable,
			);

			// Get ids
			const deleted = [];
			if (rows > 0) {
				const iter = datamodel.create_iter();
				while (iter.move_next()) {
					deleted.push(iter.get_value_for_field('id'));
				}
			}
			return deleted;
		} catch (e) {
			this.ext.logger.error('Failed to delete oldest entries', e);
		}
		return [];
	}

	selectToDeleteBuilder(includeWhere = true, olderThanMinutes = 0) {
		// SELECT id FROM table (WHERE NOT (pinned == true OR tag IS NOT NULL) (AND datetime < DATETIME('now', '-n minutes'))?)?
		const builder = new this._Gda.SqlBuilder({
			stmt_type: this._Gda.SqlStatementType.SELECT,
		});
		builder.select_add_field('id', null, null);
		builder.select_add_target('clipboard', null);
		let where = null;
		if (includeWhere) {
			// WHERE NOT (pinned == true OR tag IS NOT NULL)
			where = builder.add_cond(
				this._Gda.SqlOperatorType.NOT,
				builder.add_cond(
					this._Gda.SqlOperatorType.OR,
					builder.add_cond(
						this._Gda.SqlOperatorType.EQ,
						builder.add_id('pinned'),
						add_expr_value(builder, true),
						0,
					),
					builder.add_cond(this._Gda.SqlOperatorType.ISNOTNULL, builder.add_id('tag'), 0, 0),
					0,
				),
				0,
				0,
			);

			// AND datetime < DATETIME('now', '-n minutes')
			if (olderThanMinutes > 0) {
				where = builder.add_cond(
					this._Gda.SqlOperatorType.AND,
					where,
					builder.add_cond(
						this._Gda.SqlOperatorType.LT,
						builder.add_id('datetime'),
						builder.add_function('DATETIME', [
							add_expr_value(builder, 'now'),
							add_expr_value(builder, `-${olderThanMinutes} minutes`),
						]),
						0,
					),
					0,
				);
			}
			builder.set_where(where);
		}
		return [builder, where];
	}
}
