/*
 * Decompiled with CFR 0.152.
 */
package org.jackhuang.hmcl.util.logging;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.jackhuang.hmcl.util.logging.CallerFinder;
import org.jackhuang.hmcl.util.logging.LogEvent;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.tukaani.xz.LZMA2Options;
import org.tukaani.xz.XZOutputStream;

public final class Logger {
    public static final Logger LOG = new Logger();
    private static volatile String[] accessTokens = new String[0];
    static final String PACKAGE_PREFIX = "org.jackhuang.hmcl.";
    static final String CLASS_NAME = Logger.class.getName();
    private static final DateTimeFormatter TIME_FORMATTER = DateTimeFormatter.ofPattern("HH:mm:ss").withZone(ZoneId.systemDefault());
    private final BlockingQueue<LogEvent> queue = new LinkedBlockingQueue<LogEvent>();
    private final StringBuilder builder = new StringBuilder(512);
    private Path logFile;
    private ByteArrayOutputStream rawLogs;
    private PrintWriter logWriter;
    private Thread loggerThread;
    private boolean shutdown = false;
    private int logRetention = 0;

    public static synchronized void registerAccessToken(String token) {
        if (token == null || token.length() <= 1) {
            return;
        }
        String[] oldAccessTokens = accessTokens;
        String[] newAccessTokens = Arrays.copyOf(oldAccessTokens, oldAccessTokens.length + 1);
        newAccessTokens[oldAccessTokens.length] = token;
        accessTokens = newAccessTokens;
    }

    public static String filterForbiddenToken(String message) {
        for (String token : accessTokens) {
            message = message.replace(token, "<access token>");
        }
        return message;
    }

    public void setLogRetention(int logRetention) {
        this.logRetention = Math.max(0, logRetention);
    }

    private String format(LogEvent.DoLog event) {
        StringBuilder builder = this.builder;
        builder.setLength(0);
        builder.append('[');
        TIME_FORMATTER.formatTo(Instant.ofEpochMilli(event.time()), builder);
        builder.append("] [");
        if (event.caller() != null && event.caller().startsWith(PACKAGE_PREFIX)) {
            builder.append("@.").append(event.caller(), PACKAGE_PREFIX.length(), event.caller().length());
        } else {
            builder.append(event.caller());
        }
        builder.append('/').append((Object)event.level()).append("] ").append(Logger.filterForbiddenToken(event.message()));
        return builder.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handle(LogEvent event) {
        if (event instanceof LogEvent.DoLog) {
            LogEvent.DoLog doLog = (LogEvent.DoLog)event;
            String log = this.format(doLog);
            Throwable exception = doLog.exception();
            System.out.println(log);
            if (exception != null) {
                exception.printStackTrace(System.out);
            }
            this.logWriter.println(log);
            if (exception != null) {
                exception.printStackTrace(this.logWriter);
            }
        } else if (event instanceof LogEvent.ExportLog) {
            LogEvent.ExportLog exportEvent = (LogEvent.ExportLog)event;
            this.logWriter.flush();
            try {
                if (this.logFile != null) {
                    Files.copy(this.logFile, exportEvent.output);
                }
                this.rawLogs.writeTo(exportEvent.output);
            }
            catch (IOException e) {
                exportEvent.exception = e;
            }
            finally {
                exportEvent.latch.countDown();
            }
        } else if (event instanceof LogEvent.Shutdown) {
            this.shutdown = true;
        } else {
            throw new AssertionError((Object)("Unknown event: " + String.valueOf(event)));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void onExit() {
        List<Path> list;
        this.shutdown();
        try {
            this.loggerThread.join();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        String caller = CLASS_NAME + ".onExit";
        if (this.logRetention > 0 && this.logFile != null && (list = this.findRecentLogFiles(Integer.MAX_VALUE)).size() > this.logRetention) {
            int end = list.size() - this.logRetention;
            for (int i = 0; i < end; ++i) {
                Path file = list.get(i);
                try {
                    if (Files.isSameFile(file, this.logFile)) continue;
                    this.log(System.Logger.Level.INFO, caller, "Delete old log file " + String.valueOf(file), null);
                    Files.delete(file);
                    continue;
                }
                catch (IOException e) {
                    this.log(System.Logger.Level.WARNING, caller, "Failed to delete log file " + String.valueOf(file), e);
                }
            }
        }
        ArrayList logs = new ArrayList();
        this.queue.drainTo(logs);
        for (LogEvent log : logs) {
            this.handle(log);
        }
        if (this.logFile == null) {
            return;
        }
        boolean failed = false;
        Path xzFile = this.logFile.resolveSibling(String.valueOf(this.logFile.getFileName()) + ".xz");
        try (XZOutputStream output = new XZOutputStream(Files.newOutputStream(xzFile, new OpenOption[0]), new LZMA2Options());){
            this.logWriter.flush();
            Files.copy(this.logFile, output);
        }
        catch (IOException e) {
            failed = true;
            this.handle(new LogEvent.DoLog(System.currentTimeMillis(), caller, System.Logger.Level.WARNING, "Failed to dump log file to xz format", e));
        }
        finally {
            this.logWriter.close();
        }
        if (!failed) {
            try {
                Files.delete(this.logFile);
            }
            catch (IOException e) {
                System.err.println("An exception occurred while deleting raw log file");
                e.printStackTrace(System.err);
            }
        }
    }

    public void start(Path logFolder) {
        if (logFolder != null) {
            String time = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH-mm-ss"));
            try {
                Files.createDirectories(logFolder, new FileAttribute[0]);
                int n = 0;
                while (true) {
                    Path file = logFolder.resolve(time + (String)(n == 0 ? "" : "." + n) + ".log").toAbsolutePath().normalize();
                    try {
                        this.logWriter = new PrintWriter(Files.newBufferedWriter(file, StandardCharsets.UTF_8, StandardOpenOption.CREATE_NEW));
                        this.logFile = file;
                    }
                    catch (FileAlreadyExistsException fileAlreadyExistsException) {
                        ++n;
                        continue;
                    }
                    break;
                }
            }
            catch (IOException e) {
                this.log(System.Logger.Level.WARNING, CLASS_NAME + ".start", "Failed to create log file", e);
            }
        }
        if (this.logWriter == null) {
            this.rawLogs = new ByteArrayOutputStream(262144);
            this.logWriter = new PrintWriter(new OutputStreamWriter((OutputStream)this.rawLogs, StandardCharsets.UTF_8));
        }
        this.loggerThread = new Thread(() -> {
            ArrayList logs = new ArrayList();
            try {
                while (!this.shutdown) {
                    if (this.queue.drainTo(logs) > 0) {
                        for (LogEvent log : logs) {
                            this.handle(log);
                        }
                        logs.clear();
                        continue;
                    }
                    this.logWriter.flush();
                    this.handle(this.queue.take());
                }
                while (this.queue.drainTo(logs) > 0) {
                    for (LogEvent log : logs) {
                        this.handle(log);
                    }
                    logs.clear();
                }
            }
            catch (InterruptedException e) {
                throw new AssertionError("This thread cannot be interrupted", e);
            }
        });
        this.loggerThread.setName("HMCL Logger Thread");
        this.loggerThread.start();
        Thread cleanerThread = new Thread(this::onExit);
        cleanerThread.setName("HMCL Logger Shutdown Hook");
        Runtime.getRuntime().addShutdownHook(cleanerThread);
    }

    public void shutdown() {
        this.queue.add(new LogEvent.Shutdown());
    }

    public Path getLogFile() {
        return this.logFile;
    }

    @NotNull
    public List<Path> findRecentLogFiles(int n) {
        if (n <= 0 || this.logFile == null) {
            return List.of();
        }
        LogFile currentLogFile = LogFile.ofFile(this.logFile);
        Path logDir = this.logFile.getParent();
        if (logDir == null || !Files.isDirectory(logDir, new LinkOption[0])) {
            return List.of();
        }
        ArrayList<LogFile> logFiles = new ArrayList<LogFile>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(logDir);){
            for (Path path : stream) {
                LogFile item = LogFile.ofFile(path);
                if (item == null || currentLogFile != null && item.compareTo(currentLogFile) >= 0) continue;
                logFiles.add(item);
            }
        }
        catch (IOException e) {
            this.log(System.Logger.Level.WARNING, CLASS_NAME + ".findRecentLogFiles", "Failed to list log files in " + String.valueOf(logDir), e);
            return List.of();
        }
        logFiles.sort(Comparator.naturalOrder());
        int resultLength = Math.min(n, logFiles.size());
        int offset = logFiles.size() - resultLength;
        Path[] result = new Path[resultLength];
        for (int i = 0; i < resultLength; ++i) {
            result[i] = ((LogFile)logFiles.get((int)(i + offset))).file;
        }
        return List.of(result);
    }

    public void exportLogs(OutputStream output) throws IOException {
        Objects.requireNonNull(output);
        LogEvent.ExportLog event = new LogEvent.ExportLog(output);
        try {
            this.queue.put(event);
            event.await();
        }
        catch (InterruptedException e) {
            throw new AssertionError("This thread cannot be interrupted", e);
        }
        if (event.exception != null) {
            throw event.exception;
        }
    }

    public String getLogs() {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        try {
            this.exportLogs(output);
            return output.toString(StandardCharsets.UTF_8);
        }
        catch (IOException e) {
            this.log(System.Logger.Level.WARNING, CLASS_NAME + ".getLogs", "Failed to export logs", e);
            return "";
        }
    }

    private void log(System.Logger.Level level, String caller, String msg, Throwable exception) {
        this.queue.add(new LogEvent.DoLog(System.currentTimeMillis(), caller, level, msg, exception));
    }

    public void log(System.Logger.Level level, String msg) {
        this.log(level, CallerFinder.getCaller(), msg, null);
    }

    public void log(System.Logger.Level level, String msg, Throwable exception) {
        this.log(level, CallerFinder.getCaller(), msg, exception);
    }

    public void error(String msg) {
        this.log(System.Logger.Level.ERROR, CallerFinder.getCaller(), msg, null);
    }

    public void error(String msg, Throwable exception) {
        this.log(System.Logger.Level.ERROR, CallerFinder.getCaller(), msg, exception);
    }

    public void warning(String msg) {
        this.log(System.Logger.Level.WARNING, CallerFinder.getCaller(), msg, null);
    }

    public void warning(String msg, Throwable exception) {
        this.log(System.Logger.Level.WARNING, CallerFinder.getCaller(), msg, exception);
    }

    public void info(String msg) {
        this.log(System.Logger.Level.INFO, CallerFinder.getCaller(), msg, null);
    }

    public void info(String msg, Throwable exception) {
        this.log(System.Logger.Level.INFO, CallerFinder.getCaller(), msg, exception);
    }

    public void debug(String msg) {
        this.log(System.Logger.Level.DEBUG, CallerFinder.getCaller(), msg, null);
    }

    public void debug(String msg, Throwable exception) {
        this.log(System.Logger.Level.DEBUG, CallerFinder.getCaller(), msg, exception);
    }

    public void trace(String msg) {
        this.log(System.Logger.Level.TRACE, CallerFinder.getCaller(), msg, null);
    }

    public void trace(String msg, Throwable exception) {
        this.log(System.Logger.Level.TRACE, CallerFinder.getCaller(), msg, exception);
    }

    private record LogFile(Path file, int year, int month, int day, int hour, int minute, int second, int n) implements Comparable<LogFile>
    {
        private static final Pattern FILE_NAME_PATTERN = Pattern.compile("(?<year>\\d{4})-(?<month>\\d{2})-(?<day>\\d{2})T(?<hour>\\d{2})-(?<minute>\\d{2})-(?<second>\\d{2})(\\.(?<n>\\d+))?\\.log(\\.(gz|xz))?");

        @Nullable
        private static LogFile ofFile(Path file) {
            if (!Files.isRegularFile(file, new LinkOption[0])) {
                return null;
            }
            Matcher matcher = FILE_NAME_PATTERN.matcher(file.getFileName().toString());
            if (!matcher.matches()) {
                return null;
            }
            int year = Integer.parseInt(matcher.group("year"));
            int month = Integer.parseInt(matcher.group("month"));
            int day = Integer.parseInt(matcher.group("day"));
            int hour = Integer.parseInt(matcher.group("hour"));
            int minute = Integer.parseInt(matcher.group("minute"));
            int second = Integer.parseInt(matcher.group("second"));
            int n = Optional.ofNullable(matcher.group("n")).map(Integer::parseInt).orElse(0);
            return new LogFile(file, year, month, day, hour, minute, second, n);
        }

        @Override
        public int compareTo(@NotNull LogFile that) {
            if (this.year != that.year) {
                return Integer.compare(this.year, that.year);
            }
            if (this.month != that.month) {
                return Integer.compare(this.month, that.month);
            }
            if (this.day != that.day) {
                return Integer.compare(this.day, that.day);
            }
            if (this.hour != that.hour) {
                return Integer.compare(this.hour, that.hour);
            }
            if (this.minute != that.minute) {
                return Integer.compare(this.minute, that.minute);
            }
            if (this.second != that.second) {
                return Integer.compare(this.second, that.second);
            }
            if (this.n != that.n) {
                return Integer.compare(this.n, that.n);
            }
            return 0;
        }

        @Override
        public int hashCode() {
            return Objects.hash(this.year, this.month, this.day, this.hour, this.minute, this.second, this.n);
        }

        @Override
        public boolean equals(Object obj) {
            LogFile that;
            return obj instanceof LogFile && this.compareTo(that = (LogFile)obj) == 0;
        }
    }
}

