var documenterSearchIndex = {"docs":
[{"title":"Constructors","page":"Constructors","location":"manual/constructors.html#man-constructors","category":"section","text":"Constructors [1] are functions that create new objects – specifically, instances of Composite Types.\nIn Julia, type objects also serve as constructor functions: they create new instances of themselves\nwhen applied to an argument tuple as a function. This much was already mentioned briefly when\ncomposite types were introduced. For example:\n\njulia> struct Foo\n           bar\n           baz\n       end\n\njulia> foo = Foo(1, 2)\nFoo(1, 2)\n\njulia> foo.bar\n1\n\njulia> foo.baz\n2\n\nFor many types, forming new objects by binding their field values together is all that is ever\nneeded to create instances. However, in some cases more functionality is required when\ncreating composite objects. Sometimes invariants must be enforced, either by checking arguments\nor by transforming them. Recursive data structures,\nespecially those that may be self-referential, often cannot be constructed cleanly without first\nbeing created in an incomplete state and then altered programmatically to be made whole, as a\nseparate step from object creation. Sometimes, it's just convenient to be able to construct objects\nwith fewer or different types of parameters than they have fields. Julia's system for object construction\naddresses all of these cases and more.\n\n[1]: Nomenclature: while the term \"constructor\" generally refers to the entire function which constructs\nobjects of a type, it is common to abuse terminology slightly and refer to specific constructor\nmethods as \"constructors\". In such situations, it is generally clear from the context that the term\nis used to mean \"constructor method\" rather than \"constructor function\", especially as it is often\nused in the sense of singling out a particular method of the constructor from all of the others."},{"title":"Outer Constructor Methods","page":"Constructors","location":"manual/constructors.html#man-outer-constructor-methods","category":"section","text":"A constructor is just like any other function in Julia in that its overall behavior is defined\nby the combined behavior of its methods. Accordingly, you can add functionality to a constructor\nby simply defining new methods. For example, let's say you want to add a constructor method for\nFoo objects that takes only one argument and uses the given value for both the bar and baz\nfields. This is simple:\n\njulia> Foo(x) = Foo(x,x)\nFoo\n\njulia> Foo(1)\nFoo(1, 1)\n\nYou could also add a zero-argument Foo constructor method that supplies default values for both\nof the bar and baz fields:\n\njulia> Foo() = Foo(0)\nFoo\n\njulia> Foo()\nFoo(0, 0)\n\nHere the zero-argument constructor method calls the single-argument constructor method, which\nin turn calls the automatically provided two-argument constructor method. For reasons that will\nbecome clear very shortly, additional constructor methods declared as normal methods like this\nare called outer constructor methods. Outer constructor methods can only ever create a new instance\nby calling another constructor method, such as the automatically provided default ones."},{"title":"Inner Constructor Methods","page":"Constructors","location":"manual/constructors.html#man-inner-constructor-methods","category":"section","text":"While outer constructor methods succeed in addressing the problem of providing additional convenience\nmethods for constructing objects, they fail to address the other two use cases mentioned in the\nintroduction of this chapter: enforcing invariants, and allowing construction of self-referential\nobjects. For these problems, one needs inner constructor methods. An inner constructor method\nis like an outer constructor method, except for two differences:\n\nIt is declared inside the block of a type declaration, rather than outside of it like normal methods.\nIt has access to a special locally existent function called new that creates objects of the\nblock's type.\n\nFor example, suppose one wants to declare a type that holds a pair of real numbers, subject to\nthe constraint that the first number is not greater than the second one. One could declare it\nlike this:\n\njulia> struct OrderedPair\n           x::Real\n           y::Real\n           OrderedPair(x,y) = x > y ? error(\"out of order\") : new(x,y)\n       end\n\nNow OrderedPair objects can only be constructed such that x <= y:\n\njulia> OrderedPair(1, 2)\nOrderedPair(1, 2)\n\njulia> OrderedPair(2,1)\nERROR: out of order\nStacktrace:\n [1] error at ./error.jl:33 [inlined]\n [2] OrderedPair(::Int64, ::Int64) at ./none:4\n [3] top-level scope\n\nIf the type were declared mutable, you could reach in and directly change the field values to\nviolate this invariant. Of course, messing around with an object's internals uninvited is bad practice.\nYou (or someone else) can also provide additional outer constructor methods at any later point, but\nonce a type is declared, there is no way to add more inner constructor methods. Since outer constructor\nmethods can only create objects by calling other constructor methods, ultimately, some inner constructor\nmust be called to create an object. This guarantees that all objects of the declared type must come into\nexistence by a call to one of the inner constructor methods provided with the type, thereby giving\nsome degree of enforcement of a type's invariants.\n\nIf any inner constructor method is defined, no default constructor method is provided: it is presumed\nthat you have supplied yourself with all the inner constructors you need. The default constructor\nis equivalent to writing your own inner constructor method that takes all of the object's fields\nas parameters (constrained to be of the correct type, if the corresponding field has a type),\nand passes them to new, returning the resulting object:\n\njulia> struct Foo\n           bar\n           baz\n           Foo(bar,baz) = new(bar,baz)\n       end\n\n\nThis declaration has the same effect as the earlier definition of the Foo type without an explicit\ninner constructor method. The following two types are equivalent – one with a default constructor,\nthe other with an explicit constructor:\n\njulia> struct T1\n           x::Int64\n       end\n\njulia> struct T2\n           x::Int64\n           T2(x) = new(x)\n       end\n\njulia> T1(1)\nT1(1)\n\njulia> T2(1)\nT2(1)\n\njulia> T1(1.0)\nT1(1)\n\njulia> T2(1.0)\nT2(1)\n\nIt is good practice to provide as few inner constructor methods as possible: only those\ntaking all arguments explicitly and enforcing essential error checking and transformation. Additional\nconvenience constructor methods, supplying default values or auxiliary transformations, should\nbe provided as outer constructors that call the inner constructors to do the heavy lifting. This\nseparation is typically quite natural."},{"title":"Incomplete Initialization","page":"Constructors","location":"manual/constructors.html#Incomplete-Initialization","category":"section","text":"The final problem which has still not been addressed is construction of self-referential objects,\nor more generally, recursive data structures. Since the fundamental difficulty may not be immediately\nobvious, let us briefly explain it. Consider the following recursive type declaration:\n\njulia> mutable struct SelfReferential\n           obj::SelfReferential\n       end\n\n\nThis type may appear innocuous enough, until one considers how to construct an instance of it.\nIf a is an instance of SelfReferential, then a second instance can be created by the call:\n\njulia> b = SelfReferential(a)\n\nBut how does one construct the first instance when no instance exists to provide as a valid value\nfor its obj field? The only solution is to allow creating an incompletely initialized instance\nof SelfReferential with an unassigned obj field, and using that incomplete instance as a valid\nvalue for the obj field of another instance, such as, for example, itself.\n\nTo allow for the creation of incompletely initialized objects, Julia allows the new function\nto be called with fewer than the number of fields that the type has, returning an object with\nthe unspecified fields uninitialized. The inner constructor method can then use the incomplete\nobject, finishing its initialization before returning it. Here, for example, is another attempt\nat defining the SelfReferential type, this time using a zero-argument inner constructor returning instances\nhaving obj fields pointing to themselves:\n\njulia> mutable struct SelfReferential\n           obj::SelfReferential\n           SelfReferential() = (x = new(); x.obj = x)\n       end\n\n\nWe can verify that this constructor works and constructs objects that are, in fact, self-referential:\n\njulia> x = SelfReferential();\n\njulia> x === x\ntrue\n\njulia> x === x.obj\ntrue\n\njulia> x === x.obj.obj\ntrue\n\nAlthough it is generally a good idea to return a fully initialized object from an inner constructor,\nit is possible to return incompletely initialized objects:\n\njulia> mutable struct Incomplete\n           data\n           Incomplete() = new()\n       end\n\njulia> z = Incomplete();\n\nWhile you are allowed to create objects with uninitialized fields, any access to an uninitialized\nreference is an immediate error:\n\njulia> z.data\nERROR: UndefRefError: access to undefined reference\n\nThis avoids the need to continually check for null values. However, not all object fields are\nreferences. Julia considers some types to be \"plain data\", meaning all of their data is self-contained\nand does not reference other objects. The plain data types consist of primitive types (e.g. Int)\nand immutable structs of other plain data types (see also: isbits, isbitstype).\nThe initial contents of a plain data type is undefined:\n\njulia> struct HasPlain\n           n::Int\n           HasPlain() = new()\n       end\n\njulia> HasPlain()\nHasPlain(438103441441)\n\nArrays of plain data types exhibit the same behavior.\n\nYou can pass incomplete objects to other functions from inner constructors to delegate their completion:\n\njulia> mutable struct Lazy\n           data\n           Lazy(v) = complete_me(new(), v)\n       end\n\nAs with incomplete objects returned from constructors, if complete_me or any of its callees\ntry to access the data field of the Lazy object before it has been initialized, an error will\nbe thrown immediately."},{"title":"Parametric Constructors","page":"Constructors","location":"manual/constructors.html#Parametric-Constructors","category":"section","text":"Parametric types add a few wrinkles to the constructor story. Recall from Parametric Types\nthat, by default, instances of parametric composite types can be constructed either with explicitly\ngiven type parameters or with type parameters implied by the types of the arguments given to the\nconstructor. Here are some examples:\n\njulia> struct Point{T<:Real}\n           x::T\n           y::T\n       end\n\njulia> Point(1,2) ## implicit T ##\nPoint{Int64}(1, 2)\n\njulia> Point(1.0,2.5) ## implicit T ##\nPoint{Float64}(1.0, 2.5)\n\njulia> Point(1,2.5) ## implicit T ##\nERROR: MethodError: no method matching Point(::Int64, ::Float64)\nThe type `Point` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  Point(::T, ::T) where T<:Real at none:2\n\njulia> Point{Int64}(1, 2) ## explicit T ##\nPoint{Int64}(1, 2)\n\njulia> Point{Int64}(1.0,2.5) ## explicit T ##\nERROR: InexactError: Int64(2.5)\nStacktrace:\n[...]\n\njulia> Point{Float64}(1.0, 2.5) ## explicit T ##\nPoint{Float64}(1.0, 2.5)\n\njulia> Point{Float64}(1,2) ## explicit T ##\nPoint{Float64}(1.0, 2.0)\n\nAs you can see, for constructor calls with explicit type parameters, the arguments are converted\nto the implied field types: Point{Int64}(1,2) works, but Point{Int64}(1.0,2.5) raises an\nInexactError when converting 2.5 to Int64. When the type is implied\nby the arguments to the constructor call, as in Point(1,2), then the types of the\narguments must agree – otherwise the T cannot be determined – but any pair of real\narguments with matching type may be given to the generic Point constructor.\n\nWhat's really going on here is that Point, Point{Float64} and Point{Int64} are all different\nconstructor functions. In fact, Point{T} is a distinct constructor function for each type T.\nWithout any explicitly provided inner constructors, the declaration of the composite type Point{T<:Real}\nautomatically provides an inner constructor, Point{T}, for each possible type T<:Real, that\nbehaves just like non-parametric default inner constructors do. It also provides a single general\nouter Point constructor that takes pairs of real arguments, which must be of the same type.\nThis automatic provision of constructors is equivalent to the following explicit declaration:\n\njulia> struct Point{T<:Real}\n           x::T\n           y::T\n           Point{T}(x,y) where {T<:Real} = new(x,y)\n       end\n\njulia> Point(x::T, y::T) where {T<:Real} = Point{T}(x,y);\n\nNotice that each definition looks like the form of constructor call that it handles.\nThe call Point{Int64}(1,2) will invoke the definition Point{T}(x,y) inside the\nstruct block.\nThe outer constructor declaration, on the other hand, defines a\nmethod for the general Point constructor which only applies to pairs of values of the same real\ntype. This declaration makes constructor calls without explicit type parameters, like Point(1,2)\nand Point(1.0,2.5), work. Since the method declaration restricts the arguments to being of the\nsame type, calls like Point(1,2.5), with arguments of different types, result in \"no method\"\nerrors.\n\nSuppose we wanted to make the constructor call Point(1,2.5) work by \"promoting\" the integer\nvalue 1 to the floating-point value 1.0. The simplest way to achieve this is to define the\nfollowing additional outer constructor method:\n\njulia> Point(x::Int64, y::Float64) = Point(convert(Float64,x),y);\n\nThis method uses the convert function to explicitly convert x to Float64\nand then delegates construction to the general constructor for the case where both arguments are\nFloat64. With this method definition what was previously a MethodError now\nsuccessfully creates a point of type Point{Float64}:\n\njulia> p = Point(1,2.5)\nPoint{Float64}(1.0, 2.5)\n\njulia> typeof(p)\nPoint{Float64}\n\nHowever, other similar calls still don't work:\n\njulia> Point(1.5,2)\nERROR: MethodError: no method matching Point(::Float64, ::Int64)\nThe type `Point` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  Point(::T, !Matched::T) where T<:Real\n   @ Main none:1\n  Point(!Matched::Int64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\nFor a more general way to make all such calls work sensibly, see Conversion and Promotion.\nAt the risk of spoiling the suspense, we can reveal here that all it takes is the following outer\nmethod definition to make all calls to the general Point constructor work as one would expect:\n\njulia> Point(x::Real, y::Real) = Point(promote(x,y)...);\n\nThe promote function converts all its arguments to a common type – in this case Float64.\nWith this method definition, the Point constructor promotes its arguments the same way that\nnumeric operators like + do, and works for all kinds of real numbers:\n\njulia> Point(1.5,2)\nPoint{Float64}(1.5, 2.0)\n\njulia> Point(1,1//2)\nPoint{Rational{Int64}}(1//1, 1//2)\n\njulia> Point(1.0,1//2)\nPoint{Float64}(1.0, 0.5)\n\nThus, while the implicit type parameter constructors provided by default in Julia are fairly strict,\nit is possible to make them behave in a more relaxed but sensible manner quite easily. Moreover,\nsince constructors can leverage all of the power of the type system, methods, and multiple dispatch,\ndefining sophisticated behavior is typically quite simple."},{"title":"Case Study: Rational","page":"Constructors","location":"manual/constructors.html#Case-Study:-Rational","category":"section","text":"Perhaps the best way to tie all these pieces together is to present a real world example of a\nparametric composite type and its constructor methods. To that end, we implement our own rational number type\nOurRational, similar to Julia's built-in Rational type, defined in\nrational.jl:\n\njulia> struct OurRational{T<:Integer} <: Real\n           num::T\n           den::T\n           function OurRational{T}(num::T, den::T) where T<:Integer\n               if num == 0 && den == 0\n                    error(\"invalid rational: 0//0\")\n               end\n               num = flipsign(num, den)\n               den = flipsign(den, den)\n               g = gcd(num, den)\n               num = div(num, g)\n               den = div(den, g)\n               new(num, den)\n           end\n       end\n\njulia> OurRational(n::T, d::T) where {T<:Integer} = OurRational{T}(n,d)\nOurRational\n\njulia> OurRational(n::Integer, d::Integer) = OurRational(promote(n,d)...)\nOurRational\n\njulia> OurRational(n::Integer) = OurRational(n,one(n))\nOurRational\n\njulia> ⊘(n::Integer, d::Integer) = OurRational(n,d)\n⊘ (generic function with 1 method)\n\njulia> ⊘(x::OurRational, y::Integer) = x.num ⊘ (x.den*y)\n⊘ (generic function with 2 methods)\n\njulia> ⊘(x::Integer, y::OurRational) = (x*y.den) ⊘ y.num\n⊘ (generic function with 3 methods)\n\njulia> ⊘(x::Complex, y::Real) = complex(real(x) ⊘ y, imag(x) ⊘ y)\n⊘ (generic function with 4 methods)\n\njulia> ⊘(x::Real, y::Complex) = (x*y') ⊘ real(y*y')\n⊘ (generic function with 5 methods)\n\njulia> function ⊘(x::Complex, y::Complex)\n           xy = x*y'\n           yy = real(y*y')\n           complex(real(xy) ⊘ yy, imag(xy) ⊘ yy)\n       end\n⊘ (generic function with 6 methods)\n\nThe first line – struct OurRational{T<:Integer} <: Real – declares that OurRational takes one\ntype parameter of an integer type, and is itself a real type. The field declarations num::T\nand den::T indicate that the data held in a OurRational{T} object are a pair of integers of type\nT, one representing the rational value's numerator and the other representing its denominator.\n\nNow things get interesting. OurRational has a single inner constructor method which checks that\nnum and den aren't both zero and ensures that every rational is constructed in \"lowest\nterms\" with a non-negative denominator. This is accomplished by first flipping the signs of numerator\nand denominator if the denominator is negative. Then, both are divided by their greatest common\ndivisor (gcd always returns a non-negative number, regardless of the sign of its arguments). Because\nthis is the only inner constructor for OurRational, we can be certain that OurRational objects are\nalways constructed in this normalized form.\n\nOurRational also provides several outer constructor methods for convenience. The first is the \"standard\"\ngeneral constructor that infers the type parameter T from the type of the numerator and denominator\nwhen they have the same type. The second applies when the given numerator and denominator values\nhave different types: it promotes them to a common type and then delegates construction to the\nouter constructor for arguments of matching type. The third outer constructor turns integer values\ninto rationals by supplying a value of 1 as the denominator.\n\nFollowing the outer constructor definitions, we defined a number of methods for the ⊘\noperator, which provides a syntax for writing rationals (e.g. 1 ⊘ 2). Julia's Rational\ntype uses the // operator for this purpose. Before these definitions, ⊘\nis a completely undefined operator with only syntax and no meaning. Afterwards, it behaves just\nas described in Rational Numbers – its entire behavior is defined in these few lines.\nNote that the infix use of ⊘ works because Julia has a set of symbols that are recognized to be infix operators.\nThe first and most basic definition just makes a ⊘ b construct a OurRational by applying the\nOurRational constructor to a and b when they are integers. When one of the operands of ⊘\nis already a rational number, we construct a new rational for the resulting ratio slightly differently;\nthis behavior is actually identical to division of a rational with an integer.\nFinally, applying\n⊘ to complex integral values creates an instance of Complex{<:OurRational} – a complex\nnumber whose real and imaginary parts are rationals:\n\njulia> z = (1 + 2im) ⊘ (1 - 2im);\n\njulia> typeof(z)\nComplex{OurRational{Int64}}\n\njulia> typeof(z) <: Complex{<:OurRational}\ntrue\n\nThus, although the ⊘ operator usually returns an instance of OurRational, if either\nof its arguments are complex integers, it will return an instance of Complex{<:OurRational} instead.\nThe interested reader should consider perusing the rest of rational.jl:\nit is short, self-contained, and implements an entire basic Julia type."},{"title":"Outer-only constructors","page":"Constructors","location":"manual/constructors.html#Outer-only-constructors","category":"section","text":"As we have seen, a typical parametric type has inner constructors that are called when type parameters\nare known; e.g. they apply to Point{Int} but not to Point. Optionally, outer constructors\nthat determine type parameters automatically can be added, for example constructing a Point{Int}\nfrom the call Point(1,2). Outer constructors call inner constructors to actually\nmake instances. However, in some cases one would rather not provide inner constructors, so\nthat specific type parameters cannot be requested manually.\n\nFor example, say we define a type that stores a vector along with an accurate representation of\nits sum:\n\njulia> struct SummedArray{T<:Number,S<:Number}\n           data::Vector{T}\n           sum::S\n       end\n\njulia> SummedArray(Int32[1; 2; 3], Int32(6))\nSummedArray{Int32, Int32}(Int32[1, 2, 3], 6)\n\nThe problem is that we want S to be a larger type than T, so that we can sum many elements\nwith less information loss. For example, when T is Int32, we would like S to\nbe Int64. Therefore we want to avoid an interface that allows the user to construct\ninstances of the type SummedArray{Int32,Int32}. One way to do this is to provide a\nconstructor only for SummedArray, but inside the struct definition block to suppress\ngeneration of default constructors:\n\njulia> struct SummedArray{T<:Number,S<:Number}\n           data::Vector{T}\n           sum::S\n           function SummedArray(a::Vector{T}) where T\n               S = widen(T)\n               new{T,S}(a, sum(S, a))\n           end\n       end\n\njulia> SummedArray(Int32[1; 2; 3], Int32(6))\nERROR: MethodError: no method matching SummedArray(::Vector{Int32}, ::Int32)\nThe type `SummedArray` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  SummedArray(::Vector{T}) where T\n   @ Main none:4\n\nStacktrace:\n[...]\n\nThis constructor will be invoked by the syntax SummedArray(a). The syntax new{T,S} allows\nspecifying parameters for the type to be constructed, i.e. this call will return a SummedArray{T,S}.\nnew{T,S} can be used in any constructor definition, but for convenience the parameters\nto new{} are automatically derived from the type being constructed when possible."},{"title":"Constructors are just callable objects","page":"Constructors","location":"manual/constructors.html#Constructors-are-just-callable-objects","category":"section","text":"An object of any type may be made callable by defining a\nmethod. This includes types, i.e., objects of type Type; and constructors may,\nin fact, be viewed as just callable type objects. For example, there are many methods\ndefined on Bool and various supertypes of it:\n\nmethods(Bool)\n\nThe usual constructor syntax is exactly equivalent to the function-like object\nsyntax, so trying to define a method with each syntax will cause the first method\nto be overwritten by the next one:\n\njulia> struct S\n           f::Int\n       end\n\njulia> S() = S(7)\nS\n\njulia> (::Type{S})() = S(8)  # overwrites the previous constructor method\n\njulia> S()\nS(8)"},{"title":"SubArrays","page":"SubArrays","location":"devdocs/subarrays.html#SubArrays","category":"section","text":"Julia's SubArray type is a container encoding a \"view\" of a parent AbstractArray. This page\ndocuments some of the design principles and implementation of SubArrays.\n\nOne of the major design goals is to ensure high performance for views of both IndexLinear and\nIndexCartesian arrays. Furthermore, views of IndexLinear arrays should themselves be\nIndexLinear to the extent that it is possible."},{"title":"Index replacement","page":"SubArrays","location":"devdocs/subarrays.html#Index-replacement","category":"section","text":"Consider making 2d slices of a 3d array:\n\njulia> A = rand(2,3,4);\n\njulia> S1 = view(A, :, 1, 2:3)\n2×2 view(::Array{Float64, 3}, :, 1, 2:3) with eltype Float64:\n 0.839622  0.711389\n 0.967143  0.103929\n\njulia> S2 = view(A, 1, :, 2:3)\n3×2 view(::Array{Float64, 3}, 1, :, 2:3) with eltype Float64:\n 0.839622  0.711389\n 0.789764  0.806704\n 0.566704  0.962715\n\nview drops \"singleton\" dimensions (ones that are specified by an Int), so both S1 and S2\nare two-dimensional SubArrays. Consequently, the natural way to index these is with S1[i,j].\nTo extract the value from the parent array A, the natural approach is to replace S1[i,j]\nwith A[i,1,(2:3)[j]] and S2[i,j] with A[1,i,(2:3)[j]].\n\nThe key feature of the design of SubArrays is that this index replacement can be performed without\nany runtime overhead."},{"title":"SubArray design","page":"SubArrays","location":"devdocs/subarrays.html#SubArray-design","category":"section","text":""},{"title":"Type parameters and fields","page":"SubArrays","location":"devdocs/subarrays.html#Type-parameters-and-fields","category":"section","text":"The strategy adopted is first and foremost expressed in the definition of the type:\n\nstruct SubArray{T,N,P,I,L} <: AbstractArray{T,N}\n    parent::P\n    indices::I\n    offset1::Int       # for linear indexing and pointer, only valid when L==true\n    stride1::Int       # used only for linear indexing\n    ...\nend\n\nSubArray has 5 type parameters. The first two are the standard element type and dimensionality.\n The next is the type of the parent AbstractArray. The most heavily-used is the fourth parameter,\na Tuple of the types of the indices for each dimension. The final one, L, is only provided\nas a convenience for dispatch; it's a boolean that represents whether the index types support\nfast linear indexing. More on that later.\n\nIf in our example above A is a Array{Float64, 3}, our S1 case above would be a\nSubArray{Float64,2,Array{Float64,3},Tuple{Base.Slice{Base.OneTo{Int64}},Int64,UnitRange{Int64}},false}.\nNote in particular the tuple parameter, which stores the types of the indices used to create\nS1. Likewise,\n\njulia> S1.indices\n(Base.Slice(Base.OneTo(2)), 1, 2:3)\n\nStoring these values allows index replacement, and having the types encoded as parameters allows\none to dispatch to efficient algorithms."},{"title":"Index translation","page":"SubArrays","location":"devdocs/subarrays.html#Index-translation","category":"section","text":"Performing index translation requires that you do different things for different concrete SubArray\ntypes. For example, for S1, one needs to apply the i,j indices to the first and third dimensions\nof the parent array, whereas for S2 one needs to apply them to the second and third. The simplest\napproach to indexing would be to do the type-analysis at runtime:\n\nparentindices = Vector{Any}()\nfor thisindex in S.indices\n    ...\n    if isa(thisindex, Int)\n        # Don't consume one of the input indices\n        push!(parentindices, thisindex)\n    elseif isa(thisindex, AbstractVector)\n        # Consume an input index\n        push!(parentindices, thisindex[inputindex[j]])\n        j += 1\n    elseif isa(thisindex, AbstractMatrix)\n        # Consume two input indices\n        push!(parentindices, thisindex[inputindex[j], inputindex[j+1]])\n        j += 2\n    elseif ...\nend\nS.parent[parentindices...]\n\nUnfortunately, this would be disastrous in terms of performance: each element access would allocate\nmemory, and involves the running of a lot of poorly-typed code.\n\nThe better approach is to dispatch to specific methods to handle each type of stored index. That's\nwhat reindex does: it dispatches on the type of the first stored index and consumes the appropriate\nnumber of input indices, and then it recurses on the remaining indices. In the case of S1, this\nexpands to\n\nBase.reindex(S1, S1.indices, (i, j)) == (i, S1.indices[2], S1.indices[3][j])\n\nfor any pair of indices (i,j) (except CartesianIndexs and arrays thereof, see below).\n\nThis is the core of a SubArray; indexing methods depend upon reindex to do this index translation.\nSometimes, though, we can avoid the indirection and make it even faster."},{"title":"Linear indexing","page":"SubArrays","location":"devdocs/subarrays.html#Linear-indexing","category":"section","text":"Linear indexing can be implemented efficiently when the entire array has a single stride that\nseparates successive elements, starting from some offset. This means that we can pre-compute these\nvalues and represent linear indexing simply as an addition and multiplication, avoiding the indirection\nof reindex and (more importantly) the slow computation of the cartesian coordinates entirely.\n\nFor SubArray types, the availability of efficient linear indexing is based purely on the types\nof the indices, and does not depend on values like the size of the parent array. You can ask whether\na given set of indices supports fast linear indexing with the internal Base.viewindexing function:\n\njulia> Base.viewindexing(S1.indices)\nIndexCartesian()\n\njulia> Base.viewindexing(S2.indices)\nIndexLinear()\n\nThis is computed during construction of the SubArray and stored in the L type parameter as\na boolean that encodes fast linear indexing support. While not strictly necessary, it means that\nwe can define dispatch directly on SubArray{T,N,A,I,true} without any intermediaries.\n\nSince this computation doesn't depend on runtime values, it can miss some cases in which the stride\nhappens to be uniform:\n\njulia> A = reshape(1:4*2, 4, 2)\n4×2 reshape(::UnitRange{Int64}, 4, 2) with eltype Int64:\n 1  5\n 2  6\n 3  7\n 4  8\n\njulia> diff(A[2:2:4,:][:])\n3-element Vector{Int64}:\n 2\n 2\n 2\n\nA view constructed as view(A, 2:2:4, :) happens to have uniform stride, and therefore linear\nindexing indeed could be performed efficiently. However, success in this case depends on the\nsize of the array: if the first dimension instead were odd,\n\njulia> A = reshape(1:5*2, 5, 2)\n5×2 reshape(::UnitRange{Int64}, 5, 2) with eltype Int64:\n 1   6\n 2   7\n 3   8\n 4   9\n 5  10\n\njulia> diff(A[2:2:4,:][:])\n3-element Vector{Int64}:\n 2\n 3\n 2\n\nthen A[2:2:4,:] does not have uniform stride, so we cannot guarantee efficient linear indexing.\n Since we have to base this decision based purely on types encoded in the parameters of the SubArray,\nS = view(A, 2:2:4, :) cannot implement efficient linear indexing."},{"title":"A few details","page":"SubArrays","location":"devdocs/subarrays.html#A-few-details","category":"section","text":"Note that the Base.reindex function is agnostic to the types of the input indices; it simply\ndetermines how and where the stored indices should be reindexed. It not only supports integer\nindices, but it supports non-scalar indexing, too. This means that views of views don't need two\nlevels of indirection; they can simply re-compute the indices into the original parent array!\nHopefully by now it's fairly clear that supporting slices means that the dimensionality, given\nby the parameter N, is not necessarily equal to the dimensionality of the parent array or the\nlength of the indices tuple. Neither do user-supplied indices necessarily line up with entries\nin the indices tuple (e.g., the second user-supplied index might correspond to the third dimension\nof the parent array, and the third element in the indices tuple).\nWhat might be less obvious is that the dimensionality of the stored parent array must be equal\nto the number of effective indices in the indices tuple. Some examples:\nA = reshape(1:35, 5, 7) # A 2d parent Array\nS = view(A, 2:7)         # A 1d view created by linear indexing\nS = view(A, :, :, 1:1)   # Appending extra indices is supported\nNaively, you'd think you could just set S.parent = A and S.indices = (:,:,1:1), but supporting\nthis dramatically complicates the reindexing process, especially for views of views. Not only\ndo you need to dispatch on the types of the stored indices, but you need to examine whether a\ngiven index is the final one and \"merge\" any remaining stored indices together. This is not an\neasy task, and even worse: it's slow since it implicitly depends upon linear indexing.\nFortunately, this is precisely the computation that ReshapedArray performs, and it does so linearly\nif possible. Consequently, view ensures that the parent array is the appropriate dimensionality\nfor the given indices by reshaping it if needed. The inner SubArray constructor ensures that\nthis invariant is satisfied.\nCartesianIndex and arrays thereof throw a nasty wrench into the reindex scheme. Recall that\nreindex simply dispatches on the type of the stored indices in order to determine how many passed\nindices should be used and where they should go. But with CartesianIndex, there's no longer\na one-to-one correspondence between the number of passed arguments and the number of dimensions\nthat they index into. If we return to the above example of Base.reindex(S1, S1.indices, (i, j)),\nyou can see that the expansion is incorrect for i, j = CartesianIndex(), CartesianIndex(2,1).\nIt should skip the CartesianIndex() entirely and return:\n(CartesianIndex(2,1)[1], S1.indices[2], S1.indices[3][CartesianIndex(2,1)[2]])\nInstead, though, we get:\n(CartesianIndex(), S1.indices[2], S1.indices[3][CartesianIndex(2,1)])\nDoing this correctly would require combined dispatch on both the stored and passed indices across\nall combinations of dimensionalities in an intractable manner. As such, reindex must never be\ncalled with CartesianIndex indices. Fortunately, the scalar case is easily handled by first\nflattening the CartesianIndex arguments to plain integers. Arrays of CartesianIndex, however,\ncannot be split apart into orthogonal pieces so easily. Before attempting to use reindex, view\nmust ensure that there are no arrays of CartesianIndex in the argument list. If there are, it\ncan simply \"punt\" by avoiding the reindex calculation entirely, constructing a nested SubArray\nwith two levels of indirection instead."},{"title":"High-level Overview of the Native-Code Generation Process","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html#High-level-Overview-of-the-Native-Code-Generation-Process","category":"section","text":""},{"title":"Representation of Pointers","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html#Representation-of-Pointers","category":"section","text":"When emitting code to an object file, pointers will be emitted as relocations.\nThe deserialization code will ensure any object that pointed to one of these constants\ngets recreated and contains the right runtime pointer.\n\nOtherwise, they will be emitted as literal constants.\n\nTo emit one of these objects, call literal_pointer_val.\nIt'll handle tracking the Julia value and the LLVM global,\nensuring they are valid both for the current runtime and after deserialization.\n\nWhen emitted into the object file, these globals are stored as references\nin a large gvals table. This allows the deserializer to reference them by index,\nand implement a custom manual mechanism similar to a Global Offset Table (GOT) to restore them.\n\nFunction pointers are handled similarly.\nThey are stored as values in a large fvals table.\nLike globals, this allows the deserializer to reference them by index.\n\nNote that extern functions are handled separately,\nwith names, via the usual symbol resolution mechanism in the linker.\n\nNote too that ccall functions are also handled separately,\nvia a manual GOT and Procedure Linkage Table (PLT)."},{"title":"Representation of Intermediate Values","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html#Representation-of-Intermediate-Values","category":"section","text":"Values are passed around in a jl_cgval_t struct.\nThis represents an R-value, and includes enough information to\ndetermine how to assign or pass it somewhere.\n\nThey are created via one of the helper constructors, usually:\nmark_julia_type (for immediate values) and mark_julia_slot (for pointers to values)."},{"title":"Union representation","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html#Union-representation","category":"section","text":"Inferred union types may be stack allocated via a tagged type representation.\n\nThe primitive routines that need to be able to handle tagged unions are:\n\nmark-type\nload-local\nstore-local\nisa\nis\nemit_typeof\nemit_sizeof\nboxed\nunbox\nspecialized cc-ret\n\nEverything else should be possible to handle in inference by using these\nprimitives to implement union-splitting.\n\nThe representation of the tagged-union is as a pair\nof < void* union, byte selector >.\nThe selector is fixed-size as byte & 0x7f,\nand will union-tag the first 126 isbits.\nIt records the one-based depth-first count into the type-union of the\nisbits objects inside. An index of zero indicates that the union* is\nactually a tagged heap-allocated jl_value_t*,\nand needs to be treated as normal for a boxed object rather than as a\ntagged union.\n\nThe high bit of the selector (byte & 0x80) can be tested to determine if the\nvoid* is actually a heap-allocated (jl_value_t*) box,\nthus avoiding the cost of re-allocating a box,\nwhile maintaining the ability to efficiently handle union-splitting based on the low bits.\n\nIt is guaranteed that byte & 0x7f is an exact test for the type,\nif the value can be represented by a tag – it will never be marked byte = 0x80.\nIt is not necessary to also test the type-tag when testing isa.\n\nThe union* memory region may be allocated at any size.\nThe only constraint is that it is big enough to contain the data\ncurrently specified by selector.\nIt might not be big enough to contain the union of all types that\ncould be stored there according to the associated Union type field.\nUse appropriate care when copying."},{"title":"Specialized Calling Convention Signature Representation","page":"High-level Overview of the Native-Code Generation Process","location":"devdocs/compiler.html#Specialized-Calling-Convention-Signature-Representation","category":"section","text":"A jl_returninfo_t object describes the specialized calling convention details of any\ncallable. It can be generated from any (specTypes, rettype) pair, such as a CodeInstance, or\nother place they are declared. This is the expected calling convention for specptr, but\nother data may be stored there. Only if the function pointer stored there has the\nexpected specialized calling convention will the corresponding flag be set in specsigflags\nto indicate it is useable.\n\nIf any of the arguments or return type of a method can be represented unboxed, and none are\nunable to be represented unboxed (such as an unbounded vararg), it will be given an\noptimized calling convention signature based on the specTypes and rettype values.\n\nThe general principles are that:\n\nPrimitive types get passed in int/float registers.\nTuples of VecElement types get passed in vector registers.\nStructs get passed on the stack.\nReturn values are handled similarly to arguments,\nwith a size-cutoff at which they will instead be returned via a hidden sret argument.\n\nThe total logic for this is implemented by get_specsig_function and deserves_sret.\n\nAdditionally, if the return type is a union, it may be returned as a pair of values (a pointer and a tag).\nIf the union values can be stack-allocated, then sufficient space to store them will also be passed as a hidden first argument.\nIf the struct to return needs gc roots, space for those will be passed as a hidden second argument.\nIt is up to the callee whether the returned pointer will point to this space, a boxed object, or even other constant memory."},{"title":"Code changes","page":"Code changes","location":"devdocs/contributing/code-changes.html#Code-changes","category":"section","text":""},{"title":"Contributing to core functionality or base libraries","page":"Code changes","location":"devdocs/contributing/code-changes.html#Contributing-to-core-functionality-or-base-libraries","category":"section","text":"By contributing code to Julia, you are agreeing to release it under the MIT License.\n\nThe Julia community uses GitHub issues to track and discuss problems, feature requests, and pull requests (PR).\n\nIssues and pull requests should have self explanatory titles such that they can be understood from the list of PRs and Issues.\ni.e. Add {feature} and Fix {bug} are good, Fix #12345. Corrects the bug. is bad.\n\nYou can make pull requests for incomplete features to get code review. The convention is to open these as draft PRs and prefix\nthe pull request title with \"WIP:\" for Work In Progress, or \"RFC:\" for Request for Comments when work is completed and ready\nfor merging. This will prevent accidental merging of work that is in progress.\n\nNote: These instructions are for adding to or improving functionality in the base library. Before getting started, it can be helpful to discuss the proposed changes or additions on the Julia Discourse forum or in a GitHub issue—it's possible your proposed change belongs in a package rather than the core language. Also, keep in mind that changing stuff in the base can potentially break a lot of things. Finally, because of the time required to build Julia, note that it's usually faster to develop your code in stand-alone files, get it working, and then migrate it into the base libraries.\n\nAdd new code to Julia's base libraries as follows (this is the \"basic\" approach; see a more efficient approach in the next section):\n\nEdit the appropriate file in the base/ directory, or add new files if necessary. Create tests for your functionality and add them to files in the test/ directory. If you're editing C or Scheme code, most likely it lives in src/ or one of its subdirectories, although some aspects of Julia's REPL initialization live in cli/.\nAdd any new files to sysimg.jl in order to build them into the Julia system image.\nAdd any necessary export symbols in exports.jl.\nInclude your tests in test/Makefile and test/choosetests.jl.\n\nBuild as usual, and do make clean testall to test your contribution. If your contribution includes changes to Makefiles or external dependencies, make sure you can build Julia from a clean tree using git clean -fdx or equivalent (be careful – this command will delete any files lying around that aren't checked into git)."},{"title":"Running specific tests","page":"Code changes","location":"devdocs/contributing/code-changes.html#Running-specific-tests","category":"section","text":"There are make targets for running specific tests:\n\nmake test-bitarray\n\nYou can also use the runtests.jl script, e.g. to run test/bitarray.jl and test/math.jl:\n\n./usr/bin/julia test/runtests.jl bitarray math"},{"title":"Modifying base more efficiently with Revise.jl","page":"Code changes","location":"devdocs/contributing/code-changes.html#Modifying-base-more-efficiently-with-Revise.jl","category":"section","text":"Revise is a package that\ntracks changes in source files and automatically updates function\ndefinitions in your running Julia session. Using it, you can make\nextensive changes to Base without needing to rebuild in order to test\nyour changes.\n\nHere is the standard procedure:\n\nIf you are planning changes to any types or macros, make those\nchanges and build julia using make. (This is\nnecessary because Revise cannot handle changes to type\ndefinitions or macros.) Unless it's\nrequired to get Julia to build, you do not have to add any\nfunctionality based on the new types, just the type definitions\nthemselves.\nStart a Julia REPL session. Then issue the following commands:\n\nusing Revise    # if you aren't launching it in your `.julia/config/startup.jl`\nRevise.track(Base)\n\nEdit files in base/, save your edits, and test the\nfunctionality.\n\nIf you need to restart your Julia session, just start at step 2 above.\nRevise.track(Base) will note any changes from when Julia was last\nbuilt and incorporate them automatically. You only need to rebuild\nJulia if you made code-changes that Revise cannot handle.\n\nFor convenience, there are also test-revise-* targets for every [test-*\ntarget](https://github.com/JuliaLang/julia/blob/master/CONTRIBUTING.md#running-specific-tests) that use Revise to load any modifications to Base into the current\nsystem image before running the corresponding test. This can be useful as a shortcut\non the command line (since tests aren't always designed to be run outside the\nruntest harness)."},{"title":"Contributing to the standard library","page":"Code changes","location":"devdocs/contributing/code-changes.html#Contributing-to-the-standard-library","category":"section","text":"The standard library (stdlib) packages are baked into the Julia system image.\nWhen running the ordinary test workflow on the stdlib packages, the system image\nversion overrides the version you are developing.\nTo test stdlib packages, you can do the following steps:\n\nEdit the UUID field of the Project.toml in the stdlib package\nChange the current directory to the directory of the stdlib you are developing\nStart julia with julia --project=.\nYou can now test the package by running pkg> test in Pkg mode.\n\nBecause you changed the UUID, the package manager treats the stdlib package as\ndifferent from the one in the system image, and the system image version will\nnot override the package.\n\nBe sure to change the UUID value back before making the pull request."},{"title":"News-worthy changes","page":"Code changes","location":"devdocs/contributing/code-changes.html#News-worthy-changes","category":"section","text":"For new functionality and other substantial changes, add a brief summary to NEWS.md. The news item should cross reference the pull request (PR) parenthetically, in the form ([#pr]). To add the PR reference number, first create the PR, then push an additional commit updating NEWS.md with the PR reference number. We periodically run ./julia doc/NEWS-update.jl from the julia directory to update the cross-reference links, but this should not be done in a typical PR in order to avoid conflicting commits."},{"title":"Annotations for new features, deprecations and behavior changes","page":"Code changes","location":"devdocs/contributing/code-changes.html#Annotations-for-new-features,-deprecations-and-behavior-changes","category":"section","text":"API additions and deprecations, and minor behavior changes are allowed in minor version releases.\nFor documented features that are part of the public API, a compatibility note should be added into\nthe manual or the docstring. It should state the Julia minor version that changed the behavior\nand have a brief message describing the change.\n\nAt the moment, this should always be done with the following compat admonition\n(so that it would be possible to programmatically find the annotations in the future):\n\n!!! compat \"Julia 1.X\"       This method was added in Julia 1.X."},{"title":"Mathematical Operations and Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Mathematical-Operations-and-Elementary-Functions","category":"section","text":"Julia provides a complete collection of basic arithmetic and bitwise operators across all of its\nnumeric primitive types, as well as providing portable, efficient implementations of a comprehensive\ncollection of standard mathematical functions."},{"title":"Arithmetic Operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Arithmetic-Operators","category":"section","text":"The following arithmetic operators\nare supported on all primitive numeric types:\n\nExpression Name Description\n+x unary plus the identity operation\n-x unary minus maps values to their additive inverses\nx + y binary plus performs addition\nx - y binary minus performs subtraction\nx * y times performs multiplication\nx / y divide performs division\nx ÷ y integer divide x / y, truncated to an integer\nx \\ y inverse divide equivalent to y / x\nx ^ y power raises x to the yth power\nx % y remainder equivalent to rem(x, y)\n\nA numeric literal placed directly before an identifier or parentheses, e.g. 2x or 2(x + y), is treated as a multiplication, except with higher precedence than other binary operations. See Numeric Literal Coefficients for details.\n\nJulia's promotion system makes arithmetic operations on mixtures of argument types \"just work\"\nnaturally and automatically. See Conversion and Promotion for details of the promotion\nsystem.\n\nThe ÷ sign can be conveniently typed by writing \\div<tab> to the REPL or Julia IDE. See the manual section on Unicode input for more information.\n\nHere are some simple examples using arithmetic operators:\n\njulia> 1 + 2 + 3\n6\n\njulia> 1 - 2\n-1\n\njulia> 3*2/12\n0.5\n\n(By convention, we tend to space operators more tightly if they get applied before other nearby\noperators. For instance, we would generally write -x + 2 to reflect that first x gets negated,\nand then 2 is added to that result.)"},{"title":"Boolean Operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Boolean-Operators","category":"section","text":"The following Boolean operators are supported on Bool types:\n\nExpression Name\n!x negation\nx && y short-circuiting and\nx || y short-circuiting or\n\nNegation changes true to false and vice versa. The short-circuiting operations are explained on the linked page."},{"title":"Arithmetic operations with Bool values","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Arithmetic-operations-with-Bool-values","category":"section","text":"Note that Bool is an integer type, such that false is numerically equal to 0 and true is numerically equal to 1. All the usual promotion rules and numeric operators are also defined on it, with a special behavior of arithmetic (non-Boolean) operations when all the arguments are Bool: in those cases, the arguments are promoted to Int instead of keeping their type. Compare e.g. the following equivalent operations with Bool and with a different numeric type (UInt8):\n\njulia> true - true\n0\n\njulia> 0x01 - 0x01\n0x00\n\nAlso, when used in multiplication, false acts as a strong zero:\n\njulia> NaN * false\n0.0\n\njulia> false * Inf\n0.0\n\nThis is useful for preventing the propagation of NaN values in quantities that are known to be zero. See Knuth (1992) for motivation."},{"title":"Bitwise Operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Bitwise-Operators","category":"section","text":"The following bitwise operators\nare supported on all primitive integer types:\n\nExpression Name\n~x bitwise not\nx & y bitwise and\nx | y bitwise or\nx ⊻ y bitwise xor (exclusive or)\nx ⊼ y bitwise nand (not and)\nx ⊽ y bitwise nor (not or)\nx >>> y logical shift right\nx >> y arithmetic shift right\nx << y logical/arithmetic shift left\n\nHere are some examples with bitwise operators:\n\njulia> ~123\n-124\n\njulia> 123 & 234\n106\n\njulia> 123 | 234\n251\n\njulia> 123 ⊻ 234\n145\n\njulia> xor(123, 234)\n145\n\njulia> nand(123, 123)\n-124\n\njulia> 123 ⊼ 123\n-124\n\njulia> nor(123, 124)\n-128\n\njulia> 123 ⊽ 124\n-128\n\njulia> ~UInt32(123)\n0xffffff84\n\njulia> ~UInt8(123)\n0x84"},{"title":"Updating operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Updating-operators","category":"section","text":"Every binary arithmetic and bitwise operator also has an updating version that assigns the result\nof the operation back into its left operand. The updating version of the binary operator is formed\nby placing a = immediately after the operator. For example, writing x += 3 is equivalent to\nwriting x = x + 3:\n\njulia> x = 1\n1\n\njulia> x += 3\n4\n\njulia> x\n4\n\nThe updating versions of all the binary arithmetic and bitwise operators are:\n\n+=  -=  *=  /=  \\=  ÷=  %=  ^=  &=  |=  ⊻=  >>>=  >>=  <<=\n\nnote: Note\nAn updating operator rebinds the variable on the left-hand side. As a result, the type of the\nvariable may change.julia> x = 0x01; typeof(x)\nUInt8\n\njulia> x *= 2 # Same as x = x * 2\n2\n\njulia> typeof(x)\nInt64"},{"title":"Vectorized \"dot\" operators","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#man-dot-operators","category":"section","text":"For every binary operation like ^, there is a corresponding\n\"dot\" operation .^ that is automatically defined\nto perform ^ element-by-element on arrays. For example,\n[1, 2, 3] ^ 3 is not defined, since there is no standard\nmathematical meaning to \"cubing\" a (non-square) array, but\n[1, 2, 3] .^ 3 is defined as computing the elementwise\n(or \"vectorized\") result [1^3, 2^3, 3^3]. Similarly for unary\noperators like ! or √, there is a corresponding .√ that\napplies the operator elementwise.\n\njulia> [1, 2, 3] .^ 3\n3-element Vector{Int64}:\n  1\n  8\n 27\n\nMore specifically, a .^ b is parsed as the \"dot\" call\n(^).(a,b), which performs a broadcast operation:\nit can combine arrays and scalars, arrays of the same size (performing\nthe operation elementwise), and even arrays of different shapes (e.g.\ncombining row and column vectors to produce a matrix). Moreover, like\nall vectorized \"dot calls,\" these \"dot operators\" are\nfusing. For example, if you compute 2 .* A.^2 .+ sin.(A) (or\nequivalently @. 2A^2 + sin(A), using the @. macro) for\nan array A, it performs a single loop over A, computing 2a^2 + sin(a)\nfor each element a of A. In particular, nested dot calls like f.(g.(x))\nare fused, and \"adjacent\" binary operators like x .+ 3 .* x.^2 are\nequivalent to nested dot calls (+).(x, (*).(3, (^).(x, 2))).\n\nFurthermore, \"dotted\" updating operators like a .+= b (or @. a += b) are parsed\nas a .= a .+ b, where .= is a fused in-place assignment operation\n(see the dot syntax documentation).\n\nNote the dot syntax is also applicable to user-defined operators.\nFor example, if you define ⊗(A, B) = kron(A, B) to give a convenient\ninfix syntax A ⊗ B for Kronecker products (kron), then\n[A, B] .⊗ [C, D] will compute [A⊗C, B⊗D] with no additional coding.\n\nCombining dot operators with numeric literals can be ambiguous.\nFor example, it is not clear whether 1.+x means 1. + x or 1 .+ x.\nTherefore this syntax is disallowed, and spaces must be used around\nthe operator in such cases."},{"title":"Numeric Comparisons","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Numeric-Comparisons","category":"section","text":"Standard comparison operations are defined for all the primitive numeric types:\n\nOperator Name\n== equality\n!=, ≠ inequality\n< less than\n<=, ≤ less than or equal to\n> greater than\n>=, ≥ greater than or equal to\n\nHere are some simple examples:\n\njulia> 1 == 1\ntrue\n\njulia> 1 == 2\nfalse\n\njulia> 1 != 2\ntrue\n\njulia> 1 == 1.0\ntrue\n\njulia> 1 < 2\ntrue\n\njulia> 1.0 > 3\nfalse\n\njulia> 1 >= 1.0\ntrue\n\njulia> -1 <= 1\ntrue\n\njulia> -1 <= -1\ntrue\n\njulia> -1 <= -2\nfalse\n\njulia> 3 < -0.5\nfalse\n\nIntegers are compared in the standard manner – by comparison of bits. Floating-point numbers\nare compared according to the IEEE 754 standard:\n\nFinite numbers are ordered in the usual manner.\nPositive zero is equal but not greater than negative zero.\nInf is equal to itself and greater than everything else except NaN.\n-Inf is equal to itself and less than everything else except NaN.\nNaN is not equal to, not less than, and not greater than anything, including itself.\n\nThe last point is potentially surprising and thus worth noting:\n\njulia> NaN == NaN\nfalse\n\njulia> NaN != NaN\ntrue\n\njulia> NaN < NaN\nfalse\n\njulia> NaN > NaN\nfalse\n\nand can cause headaches when working with arrays:\n\njulia> [1 NaN] == [1 NaN]\nfalse\n\nJulia provides additional functions to test numbers for special values, which can be useful in\nsituations like hash key comparisons:\n\nFunction Tests if\nisequal(x, y) x and y are identical\nisfinite(x) x is a finite number\nisinf(x) x is infinite\nisnan(x) x is not a number\n\nisequal considers NaNs equal to each other:\n\njulia> isequal(NaN, NaN)\ntrue\n\njulia> isequal([1 NaN], [1 NaN])\ntrue\n\njulia> isequal(NaN, NaN32)\ntrue\n\nisequal can also be used to distinguish signed zeros:\n\njulia> -0.0 == 0.0\ntrue\n\njulia> isequal(-0.0, 0.0)\nfalse\n\nMixed-type comparisons between signed integers, unsigned integers, and floats can be tricky. A\ngreat deal of care has been taken to ensure that Julia does them correctly.\n\nFor other types, isequal defaults to calling ==, so if you want to define\nequality for your own types then you only need to add a == method. If you define\nyour own equality function, you should probably define a corresponding hash method\nto ensure that isequal(x,y) implies hash(x) == hash(y)."},{"title":"Chaining comparisons","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Chaining-comparisons","category":"section","text":"Unlike most languages, with the notable exception of Python,\ncomparisons can be arbitrarily chained:\n\njulia> 1 < 2 <= 2 < 3 == 3 > 2 >= 1 == 1 < 3 != 5\ntrue\n\nChaining comparisons is often quite convenient in numerical code. Chained comparisons use the\n&& operator for scalar comparisons, and the & operator for elementwise comparisons,\nwhich allows them to work on arrays. For example, 0 .< A .< 1 gives a boolean array whose entries\nare true where the corresponding elements of A are between 0 and 1.\n\nNote the evaluation behavior of chained comparisons:\n\njulia> v(x) = (println(x); x)\nv (generic function with 1 method)\n\njulia> v(1) < v(2) <= v(3)\n2\n1\n3\ntrue\n\njulia> v(1) > v(2) <= v(3)\n2\n1\nfalse\n\nThe middle expression is only evaluated once, rather than twice as it would be if the expression\nwere written as v(1) < v(2) && v(2) <= v(3). However, the order of evaluations in a chained\ncomparison is undefined. It is strongly recommended not to use expressions with side effects (such\nas printing) in chained comparisons. If side effects are required, the short-circuit && operator\nshould be used explicitly (see Short-Circuit Evaluation)."},{"title":"Elementary Functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Elementary-Functions","category":"section","text":"Julia provides a comprehensive collection of mathematical functions and operators. These mathematical\noperations are defined over as broad a class of numerical values as permit sensible definitions,\nincluding integers, floating-point numbers, rationals, and complex numbers,\nwherever such definitions make sense.\n\nMoreover, these functions (like any Julia function) can be applied in \"vectorized\" fashion to\narrays and other collections with the dot syntax f.(A),\ne.g. sin.(A) will compute the sine of each element of an array A."},{"title":"Operator Precedence and Associativity","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Operator-Precedence-and-Associativity","category":"section","text":"Julia applies the following order and associativity of operations, from highest precedence to lowest:\n\nCategory Operators Associativity\nSyntax . followed by :: Left\nExponentiation ^ Right\nUnary + - ! ~ ¬ √ ∛ ∜ ⋆ ± ∓ <: >: Right[1]\nBitshifts << >> >>> Left\nFractions // Left\nMultiplication * / % & \\ ÷ Left[2]\nAddition + - | ⊻ Left[2]\nSyntax : .. Left\nSyntax |> Left\nSyntax <| Right\nComparisons > < >= <= == === != !== <: Non-associative\nControl flow && followed by || followed by ? Right\nPair => Right\nAssignments = += -= *= /= //= \\= ^= ÷= %= |= &= ⊻= <<= >>= >>>= Right\n\n[1]: The unary operators + and - require explicit parentheses around their argument to disambiguate them from the operator ++, etc. Other compositions of unary operators are parsed with right-associativity, e. g., √√-a as √(√(-a)).\n\n[2]: The operators +, ++ and * are non-associative. a + b + c is parsed as +(a, b, c) not +(+(a, b), c). However, the fallback methods for +(a, b, c, d...) and *(a, b, c, d...) both default to left-associative evaluation.\n\nFor a complete list of every Julia operator's precedence, see the top of this file:\nsrc/julia-parser.scm. Note that some of the operators there are not defined\nin the Base module but may be given definitions by standard libraries, packages or user code.\n\nYou can also find the numerical precedence for any given operator via the built-in function Base.operator_precedence, where higher numbers take precedence:\n\njulia> Base.operator_precedence(:+), Base.operator_precedence(:*), Base.operator_precedence(:.)\n(11, 12, 17)\n\njulia> Base.operator_precedence(:sin), Base.operator_precedence(:+=), Base.operator_precedence(:(=))  # (Note the necessary parens on `:(=)`)\n(0, 1, 1)\n\nA symbol representing the operator associativity can also be found by calling the built-in function Base.operator_associativity:\n\njulia> Base.operator_associativity(:-), Base.operator_associativity(:+), Base.operator_associativity(:^)\n(:left, :none, :right)\n\njulia> Base.operator_associativity(:⊗), Base.operator_associativity(:sin), Base.operator_associativity(:→)\n(:left, :none, :right)\n\nNote that symbols such as :sin return precedence 0. This value represents invalid operators and not\noperators of lowest precedence. Similarly, such operators are assigned associativity :none.\n\nNumeric literal coefficients, e.g. 2x, are treated as multiplications with higher precedence than any other binary operation, with the exception of ^ where they have higher precedence only as the exponent.\n\njulia> x = 3; 2x^2\n18\n\njulia> x = 3; 2^2x\n64\n\nJuxtaposition parses like a unary operator, which has the same natural asymmetry around exponents: -x^y and 2x^y parse as -(x^y) and 2(x^y) whereas x^-y and x^2y parse as x^(-y) and x^(2y)."},{"title":"Numerical Conversions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Numerical-Conversions","category":"section","text":"Julia supports three forms of numerical conversion, which differ in their handling of inexact\nconversions.\n\nThe notation T(x) or convert(T, x) converts x to a value of type T.\nIf T is a floating-point type, the result is the nearest representable value, which could be\npositive or negative infinity.\nIf T is an integer type, an InexactError is raised if x is not representable by T.\nx % T converts an integer x to a value of integer type T congruent to x modulo 2^n,\nwhere n is the number of bits in T. In other words, the binary representation is truncated\nto fit.\nThe Rounding functions take a type T as an optional argument. For example, round(Int,x)\nis a shorthand for Int(round(x)).\n\nThe following examples show the different forms.\n\njulia> Int8(127)\n127\n\njulia> Int8(128)\nERROR: InexactError: trunc(Int8, 128)\nStacktrace:\n[...]\n\njulia> Int8(127.0)\n127\n\njulia> Int8(3.14)\nERROR: InexactError: Int8(3.14)\nStacktrace:\n[...]\n\njulia> Int8(128.0)\nERROR: InexactError: Int8(128.0)\nStacktrace:\n[...]\n\njulia> 127 % Int8\n127\n\njulia> 128 % Int8\n-128\n\njulia> round(Int8,127.4)\n127\n\njulia> round(Int8,127.6)\nERROR: InexactError: Int8(128.0)\nStacktrace:\n[...]\n\nSee Conversion and Promotion for how to define your own conversions and promotions."},{"title":"Rounding functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Rounding-functions","category":"section","text":"Function Description Return type\nround(x) round x to the nearest integer typeof(x)\nround(T, x) round x to the nearest integer T\nfloor(x) round x towards -Inf typeof(x)\nfloor(T, x) round x towards -Inf T\nceil(x) round x towards +Inf typeof(x)\nceil(T, x) round x towards +Inf T\ntrunc(x) round x towards zero typeof(x)\ntrunc(T, x) round x towards zero T"},{"title":"Division functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Division-functions","category":"section","text":"Function Description\ndiv(x, y), x÷y truncated division; quotient rounded towards zero\nfld(x, y) floored division; quotient rounded towards -Inf\ncld(x, y) ceiling division; quotient rounded towards +Inf\nrem(x, y), x%y remainder; satisfies x == div(x, y)*y + rem(x, y); sign matches x\nmod(x, y) modulus; satisfies x == fld(x, y)*y + mod(x, y); sign matches y\nmod1(x, y) mod with offset 1; returns r∈(0, y] for y>0 or r∈[y, 0) for y<0, where mod(r, y) == mod(x, y)\nmod2pi(x) modulus with respect to 2pi;  0 <= mod2pi(x) < 2pi\ndivrem(x, y) returns (div(x, y),rem(x, y))\nfldmod(x, y) returns (fld(x, y), mod(x, y))\ngcd(x, y...) greatest positive common divisor of x, y,...\nlcm(x, y...) least positive common multiple of x, y,..."},{"title":"Sign and absolute value functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Sign-and-absolute-value-functions","category":"section","text":"Function Description\nabs(x) a positive value with the magnitude of x\nabs2(x) the squared magnitude of x\nsign(x) indicates the sign of x, returning -1, 0, or +1\nsignbit(x) indicates whether the sign bit is on (true) or off (false)\ncopysign(x, y) a value with the magnitude of x and the sign of y\nflipsign(x, y) a value with the magnitude of x and the sign of x*y"},{"title":"Powers, logs and roots","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Powers,-logs-and-roots","category":"section","text":"Function Description\nsqrt(x), √x square root of x\ncbrt(x), ∛x cube root of x\nfourthroot(x), ∜x fourth root of x\nhypot(x, y) hypotenuse of right-angled triangle with other sides of length x and y\nexp(x) natural exponential function at x\nexpm1(x) accurate exp(x) - 1 for x near zero\nldexp(x, n) x * 2^n computed efficiently for integer values of n\nlog(x) natural logarithm of x\nlog(b, x) base b logarithm of x\nlog2(x) base 2 logarithm of x\nlog10(x) base 10 logarithm of x\nlog1p(x) accurate log(1 + x) for x near zero\nexponent(x) binary exponent of x\nsignificand(x) binary significand (a.k.a. mantissa) of a floating-point number x\n\nFor an overview of why functions like hypot, expm1, and log1p\nare necessary and useful, see John D. Cook's excellent pair of blog posts on the subject: expm1, log1p, erfc,\nand hypot."},{"title":"Trigonometric and hyperbolic functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Trigonometric-and-hyperbolic-functions","category":"section","text":"All the standard trigonometric and hyperbolic functions are also defined:\n\nsin    cos    tan    cot    sec    csc\nsinh   cosh   tanh   coth   sech   csch\nasin   acos   atan   acot   asec   acsc\nasinh  acosh  atanh  acoth  asech  acsch\nsinc   cosc\n\nThese are all single-argument functions, with atan also accepting two arguments\ncorresponding to a traditional atan2 function.\n\nAdditionally, sinpi(x) and cospi(x) are provided for more accurate computations\nof sin(pi * x) and cos(pi * x) respectively.\n\nIn order to compute trigonometric functions with degrees instead of radians, suffix the function\nwith d. For example, sind(x) computes the sine of x where x is specified in degrees.\nThe complete list of trigonometric functions with degree variants is:\n\nsind   cosd   tand   cotd   secd   cscd\nasind  acosd  atand  acotd  asecd  acscd"},{"title":"Special functions","page":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Special-functions","category":"section","text":"Many other special mathematical functions are provided by the package\nSpecialFunctions.jl."},{"title":"Markdown","page":"Markdown","location":"stdlib/Markdown.html#markdown_stdlib","category":"section","text":"This section describes Julia's markdown syntax, which is enabled by the\nMarkdown standard library. The following Markdown elements are supported:"},{"title":"Inline elements","page":"Markdown","location":"stdlib/Markdown.html#Inline-elements","category":"section","text":"Here \"inline\" refers to elements that can be found within blocks of text, i.e. paragraphs. These\ninclude the following elements."},{"title":"Bold","page":"Markdown","location":"stdlib/Markdown.html#Bold","category":"section","text":"Surround words with two asterisks, **, to display the enclosed text in boldface.\n\nA paragraph containing a **bold** word."},{"title":"Italics","page":"Markdown","location":"stdlib/Markdown.html#Italics","category":"section","text":"Surround words with one asterisk, *, to display the enclosed text in italics.\n\nA paragraph containing an *italicized* word."},{"title":"Strikethrough","page":"Markdown","location":"stdlib/Markdown.html#Strikethrough","category":"section","text":"Surround words with one or two tildes, ~ or~~, to display the enclosed text with a strikethrough, or a line through it.\n\nA paragraph containing a ~~struck~~ word."},{"title":"Literals","page":"Markdown","location":"stdlib/Markdown.html#Literals","category":"section","text":"Surround text that should be displayed exactly as written with single backticks, ` .\n\nA paragraph containing a `literal` word.\n\nLiterals should be used when writing text that refers to names of variables, functions, or other\nparts of a Julia program.\n\ntip: Tip\nTo include a backtick character within literal text use three backticks rather than one to enclose\nthe text.A paragraph containing ``` `backtick` characters ```.By extension any odd number of backticks may be used to enclose a lesser number of backticks."},{"title":"LaTeX","page":"Markdown","location":"stdlib/Markdown.html#\\LaTeX","category":"section","text":"Surround text that should be displayed as mathematics using LaTeX syntax with double backticks:\n``.\n\nA paragraph containing some ``\\LaTeX`` markup.\n\nSee the Display equations section for non-inline LaTeX.\n\ntip: Tip\nAs with literals in the previous section, if literal backticks need to be written within double\nbackticks use an even number greater than two. Note that if a single literal backtick needs to\nbe included within LaTeX markup then two enclosing backticks is sufficient.\n\nnote: Note\nThe \\ character should be escaped appropriately if the text is embedded in a Julia source code,\nfor example, \"``\\\\LaTeX`` syntax in a docstring.\", since it is interpreted as a string\nliteral. Alternatively, in order to avoid escaping, it is possible to use the raw string macro\ntogether with the @doc macro:@doc raw\"``\\LaTeX`` syntax in a docstring.\" functionname\n\nnote: Note\nInline LaTeX can also be created using a pair of single $ characters. However\nthis is not recommended as $ is also used for string interpolation. Thus using it can\neasily lead to unintended results. For example, inside an md string the parser will\ninterpret an unmatched $ as string interpolation, leading to unexpected errors or\nworse, undesired behavior, when a variable is unexpectedly interpolated into a string.\nIn contrast, an unmatched `` results in a helpful error message."},{"title":"Links","page":"Markdown","location":"stdlib/Markdown.html#Links","category":"section","text":"Links to either external or internal targets can be written using the following syntax, where\nthe text enclosed in square brackets, [ ], is the name of the link and the text enclosed in\nparentheses, ( ), is the URL.\n\nA paragraph containing a link to [Julia](https://www.julialang.org).\n\nIt's also possible to add cross-references to other documented functions/methods/variables within\nthe Julia documentation itself. For example:\n\n\"\"\"\n    tryparse(type, str; base)\n\nLike [`parse`](@ref), but returns either a value of the requested type,\nor [`nothing`](@ref) if the string does not contain a valid number.\n\"\"\"\n\nThis will create a link in the generated docs to the parse documentation\n(which has more information about what this function actually does), and to the\nnothing documentation. It's good to include cross references to mutating/non-mutating\nversions of a function, or to highlight a difference between two similar-seeming functions.\n\nnote: Note\nThe above cross referencing is not a Markdown feature, and relies on\nDocumenter.jl, which is\nused to build base Julia's documentation."},{"title":"Footnote references","page":"Markdown","location":"stdlib/Markdown.html#Footnote-references","category":"section","text":"Named and numbered footnote references can be written using the following syntax. A footnote name\nmust be a single alphanumeric word containing no punctuation.\n\nA paragraph containing a numbered footnote [^1] and a named one [^named].\n\nnote: Note\nThe text associated with a footnote can be written anywhere within the same page as the footnote\nreference. The syntax used to define the footnote text is discussed in the Footnotes section\nbelow."},{"title":"Toplevel elements","page":"Markdown","location":"stdlib/Markdown.html#Toplevel-elements","category":"section","text":"The following elements can be written either at the \"toplevel\" of a document or within another\n\"toplevel\" element."},{"title":"Paragraphs","page":"Markdown","location":"stdlib/Markdown.html#Paragraphs","category":"section","text":"A paragraph is a block of plain text, possibly containing any number of inline elements defined\nin the Inline elements section above, with one or more blank lines above and below it.\n\nThis is a paragraph.\n\nAnd this is *another* paragraph containing some emphasized text.\nA new line, but still part of the same paragraph."},{"title":"Headers","page":"Markdown","location":"stdlib/Markdown.html#Headers","category":"section","text":"A document can be split up into different sections using headers. Headers use the following syntax:\n\n# Level One\n## Level Two\n### Level Three\n#### Level Four\n##### Level Five\n###### Level Six\n\nA header line can contain any inline syntax in the same way as a paragraph can.\n\ntip: Tip\nTry to avoid using too many levels of header within a single document. A heavily nested document\nmay be indicative of a need to restructure it or split it into several pages covering separate\ntopics."},{"title":"Code blocks","page":"Markdown","location":"stdlib/Markdown.html#Code-blocks","category":"section","text":"Source code can be displayed as a literal block using an indent of four spaces or one tab as shown\nin the following example.\n\nThis is a paragraph.\n\n    function func(x)\n        # ...\n    end\n\nAnother paragraph.\n\nAdditionally, code blocks can be enclosed using triple backticks with an optional \"language\" to\nspecify how a block of code should be highlighted.\n\nA code block without a \"language\":\n\n```\nfunction func(x)\n    # ...\nend\n```\n\nand another one with the \"language\" specified as `julia`:\n\n```julia\nfunction func(x)\n    # ...\nend\n```\n\nnote: Note\n\"Fenced\" code blocks, as shown in the last example, should be preferred over indented code blocks\nsince there is no way to specify what language an indented code block is written in."},{"title":"Block quotes","page":"Markdown","location":"stdlib/Markdown.html#Block-quotes","category":"section","text":"Text from external sources, such as quotations from books or websites, can be quoted using >\ncharacters prepended to each line of the quote as follows.\n\nHere's a quote:\n\n> Julia is a high-level, high-performance dynamic programming language for\n> technical computing, with syntax that is familiar to users of other\n> technical computing environments.\n\nNote that a single space must appear after the > character on each line. Quoted blocks may themselves\ncontain other toplevel or inline elements."},{"title":"Images","page":"Markdown","location":"stdlib/Markdown.html#Images","category":"section","text":"The syntax for images is similar to the link syntax mentioned above. Prepending a ! character\nto a link will display an image from the specified URL rather than a link to it.\n\n![alternative text](link/to/image.png)"},{"title":"Lists","page":"Markdown","location":"stdlib/Markdown.html#Lists","category":"section","text":"Unordered lists can be written by prepending each item in a list with either *, +, or -.\n\nA list of items:\n\n  * item one\n  * item two\n  * item three\n\nNote the two spaces before each * and the single space after each one.\n\nLists can contain other nested toplevel elements such as lists, code blocks, or quoteblocks. A\nblank line should be left between each list item when including any toplevel elements within a\nlist.\n\nAnother list:\n\n  * item one\n\n  * item two\n\n    ```\n    f(x) = x\n    ```\n\n  * And a sublist:\n\n      + sub-item one\n      + sub-item two\n\nnote: Note\nThe contents of each item in the list must line up with the first line of the item. In the above\nexample the fenced code block must be indented by four spaces to align with the i in item two.\n\nOrdered lists are written by replacing the \"bullet\" character, either *, +, or -, with a\npositive integer followed by either . or ).\n\nTwo ordered lists:\n\n 1. item one\n 2. item two\n 3. item three\n\n 5) item five\n 6) item six\n 7) item seven\n\nAn ordered list may start from a number other than one, as in the second list of the above example,\nwhere it is numbered from five. As with unordered lists, ordered lists can contain nested toplevel\nelements."},{"title":"Display equations","page":"Markdown","location":"stdlib/Markdown.html#Display-equations","category":"section","text":"Large LaTeX equations that do not fit inline within a paragraph may be written as display\nequations using a fenced code block with the \"language\" math as in the example below.\n\n```math\nf(a) = \\frac{1}{2\\pi}\\int_{0}^{2\\pi} (\\alpha+R\\cos(\\theta))d\\theta\n```\n\nNote that the \\$\\$ version is available but deprecated:\n\nraw\"$$H = - \\sum p(x) \\log p(x)$$\"\n\nSee the Inline elements for inline LaTeX and a note about the status of the dollar-sign (\\$) deprecated form."},{"title":"Footnotes","page":"Markdown","location":"stdlib/Markdown.html#Footnotes","category":"section","text":"This syntax is paired with the inline syntax for Footnote references. Make sure to read\nthat section as well.\n\nFootnote text is defined using the following syntax, which is similar to footnote reference syntax,\naside from the : character that is appended to the footnote label.\n\n[^1]: Numbered footnote text.\n\n[^note]:\n\n    Named footnote text containing several toplevel elements\n    indented by 4 spaces or one tab.\n\n      * item one\n      * item two\n      * item three\n\n    ```julia\n    function func(x)\n        # ...\n    end\n    ```\n\nnote: Note\nNo checks are done during parsing to make sure that all footnote references have matching footnotes."},{"title":"Horizontal rules","page":"Markdown","location":"stdlib/Markdown.html#Horizontal-rules","category":"section","text":"The equivalent of an <hr> HTML tag can be achieved using three hyphens (---).\nFor example:\n\nText above the line.\n\n---\n\nAnd text below the line."},{"title":"Tables","page":"Markdown","location":"stdlib/Markdown.html#Tables","category":"section","text":"Basic tables can be written using the syntax described below. Note that markdown tables have limited\nfeatures and cannot contain nested toplevel elements unlike other elements discussed above –\nonly inline elements are allowed. Tables must always contain a header row with column names. Cells\ncannot span multiple rows or columns of the table.\n\n| Column One | Column Two | Column Three |\n|:---------- | ---------- |:------------:|\n| Row `1`    | Column `2` |              |\n| *Row* 2    | **Row** 2  | Column ``3`` |\n\nnote: Note\nAs illustrated in the above example each column of | characters must be aligned vertically.A : character on either end of a column's header separator (the row containing - characters)\nspecifies whether the row is left-aligned, right-aligned, or (when : appears on both ends) center-aligned.\nProviding no : characters will default to right-aligning the column."},{"title":"Admonitions","page":"Markdown","location":"stdlib/Markdown.html#Admonitions","category":"section","text":"Specially formatted blocks, known as admonitions, can be used to highlight particular remarks.\nThey can be defined using the following !!! syntax:\n\n!!! note\n\n    This is the content of the note.\n    It is indented by 4 spaces. A tab would work as well.\n\n!!! warning \"Beware!\"\n\n    And this is another one.\n\n    This warning admonition has a custom title: `\"Beware!\"`.\n\nThe first word after !!! declares the type of the admonition.\nThere are standard admonition types that should produce special styling.\nNamely (in order of decreasing severity): danger, warning, info/note, and tip.\n\nYou can also use your own admonition types, as long as the type name only contains lowercase Latin characters (a-z).\nFor example, you could have a terminology block like this:\n\n!!! terminology \"julia vs Julia\"\n\n    Strictly speaking, \"Julia\" refers to the language,\n    and \"julia\" to the standard implementation.\n\nHowever, unless the code rendering the Markdown special-cases that particular admonition type, it will get the default styling.\n\nA custom title for the box can be provided as a string (in double quotes) after the admonition type.\nIf no title text is specified after the admonition type, then the type name will be used as the title (e.g. \"Note\" for the note admonition).\n\nAdmonitions, like most other toplevel elements, can contain other toplevel elements (e.g. lists, images)."},{"title":"Markdown String Literals","page":"Markdown","location":"stdlib/Markdown.html#stdlib-markdown-literals","category":"section","text":"The md\"\" macro allows you to embed Markdown strings directly into your Julia code.\nThis macro is designed to simplify the inclusion of Markdown-formatted text within your Julia source files."},{"title":"Usage","page":"Markdown","location":"stdlib/Markdown.html#Usage","category":"section","text":"result = md\"This is a **custom** Markdown string with [a link](http://example.com).\""},{"title":"Markdown Syntax Extensions","page":"Markdown","location":"stdlib/Markdown.html#Markdown-Syntax-Extensions","category":"section","text":"Julia's markdown supports interpolation in a very similar way to basic string literals, with the\ndifference that it will store the object itself in the Markdown tree (as opposed to converting\nit to a string). When the Markdown content is rendered the usual show methods will be called,\nand these can be overridden as usual. This design allows the Markdown to be extended with arbitrarily\ncomplex features (such as references) without cluttering the basic syntax.\n\nIn principle, the Markdown parser itself can also be arbitrarily extended by packages, or an entirely\ncustom flavour of Markdown can be used, but this should generally be unnecessary."},{"title":"API reference","page":"Markdown","location":"stdlib/Markdown.html#stdlib-markdown-api","category":"section","text":""},{"title":"Markdown.MD","page":"Markdown","location":"stdlib/Markdown.html#Markdown.MD","category":"type","text":"MD\n\nMD represents a Markdown document. Note that the MD constructor should not generally be\nused directly, since it constructs the internal data structures. Instead, you can construct\nMD objects using the exported macros @md_str and @doc_str.\n\n\n\n\n\n"},{"title":"Markdown.@md_str","page":"Markdown","location":"stdlib/Markdown.html#Markdown.@md_str","category":"macro","text":"@md_str -> MD\n\nParse the given string as Markdown text and return a corresponding MD object.\n\nSee also Markdown.parse.\n\nExamples\n\njulia> s = md\"# Hello, world!\"\n  Hello, world!\n  ≡≡≡≡≡≡≡≡≡≡≡≡≡\n\njulia> typeof(s)\nMarkdown.MD\n\n\n\n\n\n\n"},{"title":"Markdown.@doc_str","page":"Markdown","location":"stdlib/Markdown.html#Markdown.@doc_str","category":"macro","text":"@doc_str -> MD\n\nParse the given string as Markdown text, add line and module information and return a\ncorresponding MD object.\n\n@doc_str can be used in conjunction with the Base.Docs module. Please also refer to\nthe manual section on documentation for more information.\n\nExamples\n\njulia> s = doc\"f(x) = 2*x\"\n  f(x) = 2*x\n\njulia> typeof(s)\nMarkdown.MD\n\n\n\n\n\n\n"},{"title":"Markdown.parse","page":"Markdown","location":"stdlib/Markdown.html#Markdown.parse","category":"function","text":"parse(stream::IO)::MD\n\nParse the content of stream as Julia-flavored Markdown text and return the corresponding MD object.\n\n\n\n\n\nMarkdown.parse(markdown::AbstractString)::MD\n\nParse markdown as Julia-flavored Markdown text and return the corresponding MD object.\n\nSee also @md_str.\n\n\n\n\n\n"},{"title":"Markdown.html","page":"Markdown","location":"stdlib/Markdown.html#Markdown.html","category":"function","text":"html([io::IO], md)\n\nOutput the contents of the Markdown object md in HTML format, either\nwriting to an (optional) io stream or returning a string.\n\nOne can alternatively use show(io, \"text/html\", md) or repr(\"text/html\", md), which\ndiffer in that they wrap the output in a <div class=\"markdown\"> ... </div> element.\n\nExamples\n\njulia> html(md\"hello _world_\")\n\"<p>hello <em>world</em></p>\\n\"\n\n\n\n\n\n"},{"title":"Markdown.latex","page":"Markdown","location":"stdlib/Markdown.html#Markdown.latex","category":"function","text":"latex([io::IO], md)\n\nOutput the contents of the Markdown object md in LaTeX format, either\nwriting to an (optional) io stream or returning a string.\n\nOne can alternatively use show(io, \"text/latex\", md) or repr(\"text/latex\", md).\n\nExamples\n\njulia> latex(md\"hello _world_\")\n\"hello \\\\emph{world}\\n\\n\"\n\n\n\n\n\n"},{"title":"TOML","page":"TOML","location":"stdlib/TOML.html#TOML","category":"section","text":"TOML.jl is a Julia standard library for parsing and writing [TOML\nv1.0](https://toml.io/en/) files."},{"title":"Parsing TOML data","page":"TOML","location":"stdlib/TOML.html#Parsing-TOML-data","category":"section","text":"julia> using TOML\n\njulia> data = \"\"\"\n           [database]\n           server = \"192.168.1.1\"\n           ports = [ 8001, 8001, 8002 ]\n       \"\"\";\n\njulia> TOML.parse(data)\nDict{String, Any} with 1 entry:\n  \"database\" => Dict{String, Any}(\"server\"=>\"192.168.1.1\", \"ports\"=>[8001, 8001…\n\nTo parse a file, use TOML.parsefile. If the file has a syntax error,\nan exception is thrown:\n\njulia> using TOML\n\njulia> TOML.parse(\"\"\"\n           value = 0.0.0\n       \"\"\")\nERROR: TOML Parser error:\nnone:1:16 error: failed to parse value\n      value = 0.0.0\n                 ^\n[...]\n\nThere are other versions of the parse functions (TOML.tryparse\nand TOML.tryparsefile) that instead of throwing exceptions on parser error\nreturns a TOML.ParserError with information:\n\njulia> using TOML\n\njulia> err = TOML.tryparse(\"\"\"\n           value = 0.0.0\n       \"\"\");\n\njulia> err.type\nErrGenericValueError::ErrorType = 14\n\njulia> err.line\n1\n\njulia> err.column\n16"},{"title":"Exporting data to TOML file","page":"TOML","location":"stdlib/TOML.html#Exporting-data-to-TOML-file","category":"section","text":"The TOML.print function is used to print (or serialize) data into TOML\nformat.\n\njulia> using TOML\n\njulia> data = Dict(\n          \"names\" => [\"Julia\", \"Julio\"],\n          \"age\" => [10, 20],\n       );\n\njulia> TOML.print(data)\nnames = [\"Julia\", \"Julio\"]\nage = [10, 20]\n\njulia> fname = tempname();\n\njulia> open(fname, \"w\") do io\n           TOML.print(io, data)\n       end\n\njulia> TOML.parsefile(fname)\nDict{String, Any} with 2 entries:\n  \"names\" => [\"Julia\", \"Julio\"]\n  \"age\"   => [10, 20]\n\nKeys can be sorted according to some value\n\njulia> using TOML\n\njulia> TOML.print(Dict(\n       \"abc\"  => 1,\n       \"ab\"   => 2,\n       \"abcd\" => 3,\n       ); sorted=true, by=length)\nab = 2\nabc = 1\nabcd = 3\n\nFor custom structs, pass a function that converts the struct to a supported\ntype\n\njulia> using TOML\n\njulia> struct MyStruct\n           a::Int\n           b::String\n       end\n\njulia> TOML.print(Dict(\"foo\" => MyStruct(5, \"bar\"))) do x\n           x isa MyStruct && return [x.a, x.b]\n           error(\"unhandled type $(typeof(x))\")\n       end\nfoo = [5, \"bar\"]"},{"title":"References","page":"TOML","location":"stdlib/TOML.html#References","category":"section","text":""},{"title":"TOML.parse","page":"TOML","location":"stdlib/TOML.html#TOML.parse","category":"function","text":"parse(x::Union{AbstractString, IO})\nparse(p::Parser, x::Union{AbstractString, IO})\n\nParse the string  or stream x, and return the resulting table (dictionary).\nThrow a ParserError upon failure.\n\nSee also TOML.tryparse.\n\n\n\n\n\n"},{"title":"TOML.parsefile","page":"TOML","location":"stdlib/TOML.html#TOML.parsefile","category":"function","text":"parsefile(f::AbstractString)\nparsefile(p::Parser, f::AbstractString)\n\nParse file f and return the resulting table (dictionary). Throw a\nParserError upon failure.\n\nSee also TOML.tryparsefile.\n\n\n\n\n\n"},{"title":"TOML.tryparse","page":"TOML","location":"stdlib/TOML.html#TOML.tryparse","category":"function","text":"tryparse(x::Union{AbstractString, IO})\ntryparse(p::Parser, x::Union{AbstractString, IO})\n\nParse the string or stream x, and return the resulting table (dictionary).\nReturn a ParserError upon failure.\n\nSee also TOML.parse.\n\n\n\n\n\n"},{"title":"TOML.tryparsefile","page":"TOML","location":"stdlib/TOML.html#TOML.tryparsefile","category":"function","text":"tryparsefile(f::AbstractString)\ntryparsefile(p::Parser, f::AbstractString)\n\nParse file f and return the resulting table (dictionary). Return a\nParserError upon failure.\n\nSee also TOML.parsefile.\n\n\n\n\n\n"},{"title":"TOML.print","page":"TOML","location":"stdlib/TOML.html#TOML.print","category":"function","text":"print([to_toml::Function], io::IO [=stdout], data::AbstractDict; sorted=false, by=identity, inline_tables::IdSet{<:AbstractDict})\n\nWrite data as TOML syntax to the stream io. If the keyword argument sorted is set to true,\nsort tables according to the function given by the keyword argument by. If the keyword argument\ninline_tables is given, it should be a set of tables that should be printed \"inline\".\n\ncompat: Julia 1.11\nThe inline_tables keyword argument is supported by Julia 1.11 or later.\n\nThe following data types are supported: AbstractDict, AbstractVector, AbstractString, Integer, AbstractFloat, Bool,\nDates.DateTime, Dates.Time, Dates.Date. Note that the integers and floats\nneed to be convertible to Float64 and Int64 respectively. For other data types,\npass the function to_toml that takes the data types and returns a value of a\nsupported type.\n\n\n\n\n\n"},{"title":"TOML.Parser","page":"TOML","location":"stdlib/TOML.html#TOML.Parser","category":"type","text":"Parser()\n\nConstructor for a TOML Parser.  Note that in most cases one does not need to\nexplicitly create a Parser but instead one directly uses\nTOML.parsefile or TOML.parse.  Using an explicit parser\nwill however reuse some internal data structures which can be beneficial for\nperformance if a larger number of small files are parsed.\n\n\n\n\n\n"},{"title":"TOML.ParserError","page":"TOML","location":"stdlib/TOML.html#TOML.ParserError","category":"type","text":"ParserError\n\nType that is returned from tryparse and tryparsefile\nwhen parsing fails. It contains (among others) the following fields:\n\npos, the position in the string when the error happened\ntable, the result that so far was successfully parsed\ntype, an error type, different for different types of errors\n\n\n\n\n\n"},{"title":"Mathematics","page":"Mathematics","location":"base/math.html#Mathematics","category":"section","text":""},{"title":"Mathematical Operators","page":"Mathematics","location":"base/math.html#math-ops","category":"section","text":""},{"title":"Comparisons","page":"Mathematics","location":"base/math.html#Comparisons","category":"section","text":""},{"title":"Bit Operators","page":"Mathematics","location":"base/math.html#Bit-Operators","category":"section","text":""},{"title":"Logical Operators","page":"Mathematics","location":"base/math.html#Logical-Operators","category":"section","text":""},{"title":"Trigonometry","page":"Mathematics","location":"base/math.html#Trigonometry","category":"section","text":""},{"title":"... in Radians","page":"Mathematics","location":"base/math.html#...-in-Radians","category":"section","text":""},{"title":"... in Degrees","page":"Mathematics","location":"base/math.html#...-in-Degrees","category":"section","text":""},{"title":"... in Multiples of π","page":"Mathematics","location":"base/math.html#...-in-Multiples-of-π","category":"section","text":""},{"title":"Hyperbolic","page":"Mathematics","location":"base/math.html#Hyperbolic","category":"section","text":""},{"title":"Other Trigonometric Functions","page":"Mathematics","location":"base/math.html#Other-Trigonometric-Functions","category":"section","text":""},{"title":"Logarithms","page":"Mathematics","location":"base/math.html#Logarithms","category":"section","text":""},{"title":"Exponentiation","page":"Mathematics","location":"base/math.html#Exponentiation","category":"section","text":""},{"title":"Rounding","page":"Mathematics","location":"base/math.html#Rounding","category":"section","text":""},{"title":"Extrema","page":"Mathematics","location":"base/math.html#Extrema","category":"section","text":""},{"title":"Clamping","page":"Mathematics","location":"base/math.html#Clamping","category":"section","text":""},{"title":"Checked Operations","page":"Mathematics","location":"base/math.html#Checked-Operations","category":"section","text":""},{"title":"Signs","page":"Mathematics","location":"base/math.html#Signs","category":"section","text":""},{"title":"Roots","page":"Mathematics","location":"base/math.html#Roots","category":"section","text":""},{"title":"Complex Functions","page":"Mathematics","location":"base/math.html#Complex-Functions","category":"section","text":""},{"title":"Combinatorics and Number Theory","page":"Mathematics","location":"base/math.html#Combinatorics-and-Number-Theory","category":"section","text":""},{"title":"Other Mathematical Functions","page":"Mathematics","location":"base/math.html#Other-Mathematical-Functions","category":"section","text":""},{"title":"Customizable binary operators","page":"Mathematics","location":"base/math.html#Customizable-binary-operators","category":"section","text":"Some unicode characters can be used to define new binary operators\nthat support infix notation.\nFor example\n⊗(x,y) = kron(x,y)\ndefines the ⊗ (otimes) function to be the Kronecker product,\nand one can call it as binary operator using infix syntax:\nC = A ⊗ B\nas well as with the usual prefix syntax\nC = ⊗(A,B).\n\nOther characters that support such extensions include\n\\odot ⊙\nand\n\\oplus ⊕\n\nThe complete list is in the parser code:\nhttps://github.com/JuliaLang/julia/blob/master/src/julia-parser.scm\n\nThose that are parsed like * (in terms of precedence) include\n* / ÷ % & ⋅ ∘ × |\\\\| ∩ ∧ ⊗ ⊘ ⊙ ⊚ ⊛ ⊠ ⊡ ⊓ ∗ ∙ ∤ ⅋ ≀ ⊼ ⋄ ⋆ ⋇ ⋉ ⋊ ⋋ ⋌ ⋏ ⋒ ⟑ ⦸ ⦼ ⦾ ⦿ ⧶ ⧷ ⨇ ⨰ ⨱ ⨲ ⨳ ⨴ ⨵ ⨶ ⨷ ⨸ ⨻ ⨼ ⨽ ⩀ ⩃ ⩄ ⩋ ⩍ ⩎ ⩑ ⩓ ⩕ ⩘ ⩚ ⩜ ⩞ ⩟ ⩠ ⫛ ⊍ ▷ ⨝ ⟕ ⟖ ⟗\nand those that are parsed like + include\n+ - |\\|| ⊕ ⊖ ⊞ ⊟ |++| ∪ ∨ ⊔ ± ∓ ∔ ∸ ≏ ⊎ ⊻ ⊽ ⋎ ⋓ ⟇ ⧺ ⧻ ⨈ ⨢ ⨣ ⨤ ⨥ ⨦ ⨧ ⨨ ⨩ ⨪ ⨫ ⨬ ⨭ ⨮ ⨹ ⨺ ⩁ ⩂ ⩅ ⩊ ⩌ ⩏ ⩐ ⩒ ⩔ ⩖ ⩗ ⩛ ⩝ ⩡ ⩢ ⩣\nThere are many others that are related to arrows, comparisons, and powers."},{"title":"Base.:-","page":"Mathematics","location":"base/math.html#Base.:--Tuple{Any}","category":"method","text":"-(x)\n\nUnary minus operator.\n\nSee also: abs, flipsign.\n\nExamples\n\njulia> -1\n-1\n\njulia> -(2)\n-2\n\njulia> -[1 2; 3 4]\n2×2 Matrix{Int64}:\n -1  -2\n -3  -4\n\njulia> -(true)  # promotes to Int\n-1\n\njulia> -(0x003)\n0xfffd\n\n\n\n\n\n"},{"title":"Base.:+","page":"Mathematics","location":"base/math.html#Base.:+","category":"function","text":"+(x, y...)\n\nAddition operator.\n\nInfix x+y+z+... calls this function with all arguments, i.e. +(x, y, z, ...),\nwhich by default then calls (x+y) + z + ... starting from the left.\n\nNote that overflow is possible for most integer types, including the\ndefault Int, when adding large numbers.\n\nExamples\n\njulia> 1 + 20 + 4\n25\n\njulia> +(1, 20, 4)\n25\n\njulia> [1,2] + [3,4]\n2-element Vector{Int64}:\n 4\n 6\n\njulia> typemax(Int) + 1 < 0\ntrue\n\n\n\n\n\ndt::Date + t::Time -> DateTime\n\nThe addition of a Date with a Time produces a DateTime. The hour, minute, second, and millisecond parts of\nthe Time are used along with the year, month, and day of the Date to create the new DateTime.\nNon-zero microseconds or nanoseconds in the Time type will result in an InexactError being thrown.\n\n\n\n\n\n"},{"title":"Base.:-","page":"Mathematics","location":"base/math.html#Base.:--Tuple{Any, Any}","category":"method","text":"-(x, y)\n\nSubtraction operator.\n\nExamples\n\njulia> 2 - 3\n-1\n\njulia> -(2, 4.5)\n-2.5\n\n\n\n\n\n"},{"title":"Base.:*","page":"Mathematics","location":"base/math.html#Base.:*-Tuple{Any, Vararg{Any}}","category":"method","text":"*(x, y...)\n\nMultiplication operator.\n\nInfix x*y*z*... calls this function with all arguments, i.e. *(x, y, z, ...),\nwhich by default then calls (x*y) * z * ... starting from the left.\n\nJuxtaposition such as 2pi also calls *(2, pi). Note that this operation\nhas higher precedence than a literal *. Note also that juxtaposition \"0x...\"\n(integer zero times a variable whose name starts with x) is forbidden as\nit clashes with unsigned integer literals: 0x01 isa UInt8.\n\nNote that overflow is possible for most integer types, including the default Int,\nwhen multiplying large numbers.\n\nExamples\n\njulia> 2 * 7 * 8\n112\n\njulia> *(2, 7, 8)\n112\n\njulia> [2 0; 0 3] * [1, 10]  # matrix * vector\n2-element Vector{Int64}:\n  2\n 30\n\njulia> 1/2pi, 1/2*pi  # juxtaposition has higher precedence\n(0.15915494309189535, 1.5707963267948966)\n\njulia> x = [1, 2]; x'x  # adjoint vector * vector\n5\n\n\n\n\n\n"},{"title":"Base.:/","page":"Mathematics","location":"base/math.html#Base.:/","category":"function","text":"/(x, y)\n\nRight division operator: multiplication of x by the inverse of y on the right.\n\nGives floating-point results for integer arguments.\nSee ÷ for integer division, or // for Rational results.\n\nExamples\n\njulia> 1/2\n0.5\n\njulia> 4/2\n2.0\n\njulia> 4.5/2\n2.25\n\n\n\n\n\nA / B\n\nMatrix right-division: A / B is equivalent to (B' \\ A')' where \\ is the left-division operator.\nFor square matrices, the result X is such that A == X*B.\n\nSee also: rdiv!.\n\nExamples\n\njulia> A = Float64[1 4 5; 3 9 2]; B = Float64[1 4 2; 3 4 2; 8 7 1];\n\njulia> X = A / B\n2×3 Matrix{Float64}:\n -0.65   3.75  -1.2\n  3.25  -2.75   1.0\n\njulia> isapprox(A, X*B)\ntrue\n\njulia> isapprox(X, A*pinv(B))\ntrue\n\n\n\n\n\n"},{"title":"Base.:\\","page":"Mathematics","location":"base/math.html#Base.:\\-Tuple{Any, Any}","category":"method","text":"\\(x, y)\n\nLeft division operator: multiplication of y by the inverse of x on the left. Gives\nfloating-point results for integer arguments.\n\nExamples\n\njulia> 3 \\ 6\n2.0\n\njulia> inv(3) * 6\n2.0\n\njulia> A = [4 3; 2 1]; x = [5, 6];\n\njulia> A \\ x\n2-element Vector{Float64}:\n  6.5\n -7.0\n\njulia> inv(A) * x\n2-element Vector{Float64}:\n  6.5\n -7.0\n\n\n\n\n\n"},{"title":"Base.:^","page":"Mathematics","location":"base/math.html#Base.:^-Tuple{Number, Number}","category":"method","text":"^(x, y)\n\nExponentiation operator.\n\nIf x and y are integers, the result may overflow.\nTo enter numbers in scientific notation, use Float64 literals\nsuch as 1.2e3 rather than 1.2 * 10^3.\n\nIf y is an Int literal (e.g. 2 in x^2 or -3 in x^-3), the Julia code\nx^y is transformed by the compiler to Base.literal_pow(^, x, Val(y)), to\nenable compile-time specialization on the value of the exponent.\n(As a default fallback we have Base.literal_pow(^, x, Val(y)) = ^(x,y),\nwhere usually ^ == Base.^ unless ^ has been defined in the calling\nnamespace.) If y is a negative integer literal, then Base.literal_pow\ntransforms the operation to inv(x)^-y by default, where -y is positive.\n\nSee also exp2, <<.\n\nExamples\n\njulia> 3^5\n243\n\njulia> 3^-1  # uses Base.literal_pow\n0.3333333333333333\n\njulia> p = -1;\n\njulia> 3^p\nERROR: DomainError with -1:\nCannot raise an integer x to a negative power -1.\n[...]\n\njulia> 3.0^p\n0.3333333333333333\n\njulia> 10^19 > 0  # integer overflow\nfalse\n\njulia> big(10)^19 == 1e19\ntrue\n\n\n\n\n\n"},{"title":"Base.fma","page":"Mathematics","location":"base/math.html#Base.fma","category":"function","text":"fma(x, y, z)\n\nCompute x*y+z without rounding the intermediate result x*y. On some systems this is\nsignificantly more expensive than x*y+z. fma is used to improve accuracy in certain\nalgorithms. See muladd.\n\n\n\n\n\n"},{"title":"Base.muladd","page":"Mathematics","location":"base/math.html#Base.muladd","category":"function","text":"muladd(x, y, z)\n\nCombined multiply-add: computes x*y+z, but allowing the add and multiply to be merged\nwith each other or with surrounding operations for performance.\nFor example, this may be implemented as an fma if the hardware supports it\nefficiently.\nThe result can be different on different machines and can also be different on the same machine\ndue to constant propagation or other optimizations.\nSee fma.\n\nExamples\n\njulia> muladd(3, 2, 1)\n7\n\njulia> 3 * 2 + 1\n7\n\n\n\n\n\nmuladd(A, y, z)\n\nCombined multiply-add, A*y .+ z, for matrix-matrix or matrix-vector multiplication.\nThe result is always the same size as A*y, but z may be smaller, or a scalar.\n\ncompat: Julia 1.6\nThese methods require Julia 1.6 or later.\n\nExamples\n\njulia> A=[1.0 2.0; 3.0 4.0]; B=[1.0 1.0; 1.0 1.0]; z=[0, 100];\n\njulia> muladd(A, B, z)\n2×2 Matrix{Float64}:\n   3.0    3.0\n 107.0  107.0\n\n\n\n\n\n"},{"title":"Base.inv","page":"Mathematics","location":"base/math.html#Base.inv-Tuple{Number}","category":"method","text":"inv(x)\n\nReturn the multiplicative inverse of x, such that x*inv(x) or inv(x)*x\nyields one(x) (the multiplicative identity) up to roundoff errors.\n\nIf x is a number, this is essentially the same as one(x)/x, but for\nsome types inv(x) may be slightly more efficient.\n\nExamples\n\njulia> inv(2)\n0.5\n\njulia> inv(1 + 2im)\n0.2 - 0.4im\n\njulia> inv(1 + 2im) * (1 + 2im)\n1.0 + 0.0im\n\njulia> inv(2//3)\n3//2\n\ncompat: Julia 1.2\ninv(::Missing) requires at least Julia 1.2.\n\n\n\n\n\n"},{"title":"Base.div","page":"Mathematics","location":"base/math.html#Base.div","category":"function","text":"div(x, y)\n÷(x, y)\n\nThe quotient from Euclidean (integer) division. Generally equivalent\nto a mathematical operation x/y without a fractional part.\n\nSee also: cld, fld, rem, divrem.\n\nExamples\n\njulia> 9 ÷ 4\n2\n\njulia> -5 ÷ 3\n-1\n\njulia> 5.0 ÷ 2\n2.0\n\njulia> div.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -1  -1  -1  0  0  0  0  0  1  1  1\n\n\n\n\n\n"},{"title":"Base.div","page":"Mathematics","location":"base/math.html#Base.div-Tuple{Any, Any, RoundingMode}","category":"method","text":"div(x, y, r::RoundingMode=RoundToZero)\n\nThe quotient from Euclidean (integer) division. Computes x / y, rounded to\nan integer according to the rounding mode r. In other words, the quantity\n\nround(x / y, r)\n\nwithout any intermediate rounding.\n\ncompat: Julia 1.4\nThe three-argument method taking a RoundingMode requires Julia 1.4 or later.\n\nSee also fld and cld, which are special cases of this function.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9.\n\nExamples:\n\njulia> div(4, 3, RoundToZero) # Matches div(4, 3)\n1\njulia> div(4, 3, RoundDown) # Matches fld(4, 3)\n1\njulia> div(4, 3, RoundUp) # Matches cld(4, 3)\n2\njulia> div(5, 2, RoundNearest)\n2\njulia> div(5, 2, RoundNearestTiesAway)\n3\njulia> div(-5, 2, RoundNearest)\n-2\njulia> div(-5, 2, RoundNearestTiesAway)\n-3\njulia> div(-5, 2, RoundNearestTiesUp)\n-2\njulia> div(4, 3, RoundFromZero)\n2\njulia> div(-4, 3, RoundFromZero)\n-2\n\nBecause div(x, y) implements strictly correct truncated rounding based on the true\nvalue of floating-point numbers, unintuitive situations can arise. For example:\n\njulia> div(6.0, 0.1)\n59.0\njulia> 6.0 / 0.1\n60.0\njulia> 6.0 / big(0.1)\n59.99999999999999666933092612453056361837965690217069245739573412231113406246995\n\nWhat is happening here is that the true value of the floating-point number written\nas 0.1 is slightly larger than the numerical value 1/10 while 6.0 represents\nthe number 6 precisely. Therefore the true value of 6.0 / 0.1 is slightly less\nthan 60. When doing division, this is rounded to precisely 60.0, but\ndiv(6.0, 0.1, RoundToZero) always truncates the true value, so the result is 59.0.\n\n\n\n\n\n"},{"title":"Base.fld","page":"Mathematics","location":"base/math.html#Base.fld","category":"function","text":"fld(x, y)\n\nLargest integer less than or equal to x / y. Equivalent to div(x, y, RoundDown).\n\nSee also div, cld, fld1.\n\nExamples\n\njulia> fld(7.3, 5.5)\n1.0\n\njulia> fld.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -2  -2  -1  -1  -1  0  0  0  1  1  1\n\nBecause fld(x, y) implements strictly correct floored rounding based on the true\nvalue of floating-point numbers, unintuitive situations can arise. For example:\n\njulia> fld(6.0, 0.1)\n59.0\njulia> 6.0 / 0.1\n60.0\njulia> 6.0 / big(0.1)\n59.99999999999999666933092612453056361837965690217069245739573412231113406246995\n\nWhat is happening here is that the true value of the floating-point number written\nas 0.1 is slightly larger than the numerical value 1/10 while 6.0 represents\nthe number 6 precisely. Therefore the true value of 6.0 / 0.1 is slightly less\nthan 60. When doing division, this is rounded to precisely 60.0, but\nfld(6.0, 0.1) always takes the floor of the true value, so the result is 59.0.\n\n\n\n\n\n"},{"title":"Base.cld","page":"Mathematics","location":"base/math.html#Base.cld","category":"function","text":"cld(x, y)\n\nSmallest integer larger than or equal to x / y. Equivalent to div(x, y, RoundUp).\n\nSee also div, fld.\n\nExamples\n\njulia> cld(5.5, 2.2)\n3.0\n\njulia> cld.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -1  -1  -1  0  0  0  1  1  1  2  2\n\n\n\n\n\n"},{"title":"Base.mod","page":"Mathematics","location":"base/math.html#Base.mod","category":"function","text":"rem(x::Integer, T::Type{<:Integer})::T\nmod(x::Integer, T::Type{<:Integer})::T\n%(x::Integer, T::Type{<:Integer})::T\n\nFind y::T such that x ≡ y (mod n), where n is the number of integers representable\nin T, and y is an integer in [typemin(T),typemax(T)].\nIf T can represent any integer (e.g. T == BigInt), then this operation corresponds to\na conversion to T.\n\nExamples\n\njulia> x = 129 % Int8\n-127\n\njulia> typeof(x)\nInt8\n\njulia> x = 129 % BigInt\n129\n\njulia> typeof(x)\nBigInt\n\n\n\n\n\nmod(x, y)\nrem(x, y, RoundDown)\n\nThe reduction of x modulo y, or equivalently, the remainder of x after floored\ndivision by y, i.e. x - y*fld(x,y) if computed without intermediate rounding.\n\nThe result will have the same sign as y if isfinite(y), and magnitude less than abs(y) (with some\nexceptions, see note below).\n\nnote: Note\nWhen used with floating point values, the exact result may not be representable by the\ntype, and so rounding error may occur. In particular, if the exact result is very\nclose to y, then it may be rounded to y.\n\nSee also: rem, div, fld, mod1, invmod.\n\njulia> mod(8, 3)\n2\n\njulia> mod(9, 3)\n0\n\njulia> mod(8.9, 3)\n2.9000000000000004\n\njulia> mod(eps(), 3)\n2.220446049250313e-16\n\njulia> mod(-eps(), 3)\n3.0\n\njulia> mod.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n 1  2  0  1  2  0  1  2  0  1  2\n\n\n\n\n\nmod(x::Integer, r::AbstractUnitRange)\n\nFind y in the range r such that x ≡ y (mod n), where n = length(r),\ni.e. y = mod(x - first(r), n) + first(r).\n\nSee also mod1.\n\nExamples\n\njulia> mod(0, Base.OneTo(3))  # mod1(0, 3)\n3\n\njulia> mod(3, 0:2)  # mod(3, 3)\n0\n\ncompat: Julia 1.3\nThis method requires at least Julia 1.3.\n\n\n\n\n\n"},{"title":"Base.rem","page":"Mathematics","location":"base/math.html#Base.rem","category":"function","text":"rem(x, y)\n%(x, y)\n\nRemainder from Euclidean division, returning a value of the same sign as x, and smaller in\nmagnitude than y. This value is always exact.\n\nSee also: div, mod, mod1, divrem.\n\nExamples\n\njulia> x = 15; y = 4;\n\njulia> x % y\n3\n\njulia> x == div(x, y) * y + rem(x, y)\ntrue\n\njulia> rem.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -2  -1  0  -2  -1  0  1  2  0  1  2\n\n\n\n\n\n"},{"title":"Base.rem","page":"Mathematics","location":"base/math.html#Base.rem-Tuple{Any, Any, RoundingMode}","category":"method","text":"rem(x, y, r::RoundingMode=RoundToZero)\n\nCompute the remainder of x after integer division by y, with the quotient rounded\naccording to the rounding mode r. In other words, the quantity\n\nx - y * round(x / y, r)\n\nwithout any intermediate rounding.\n\nif r == RoundNearest, then the result is exact, and in the interval\n-y  2 y  2. See also RoundNearest.\nif r == RoundToZero (default), then the result is exact, and in the interval\n0 y) if x is positive, or (-y 0 otherwise. See also RoundToZero.\nif r == RoundDown, then the result is in the interval 0 y) if y is positive, or\n(y 0 otherwise. The result may not be exact if x and y have different signs, and\nabs(x) < abs(y). See also RoundDown.\nif r == RoundUp, then the result is in the interval (-y 0 if y is positive, or\n0 -y) otherwise. The result may not be exact if x and y have the same sign, and\nabs(x) < abs(y). See also RoundUp.\nif r == RoundFromZero, then the result is in the interval (-y 0 if y is positive, or\n0 -y) otherwise. The result may not be exact if x and y have the same sign, and\nabs(x) < abs(y). See also RoundFromZero.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9.\n\nExamples:\n\njulia> x = 9; y = 4;\n\njulia> x % y  # same as rem(x, y)\n1\n\njulia> x ÷ y  # same as div(x, y)\n2\n\njulia> x == div(x, y) * y + rem(x, y)\ntrue\n\n\n\n\n\n"},{"title":"Base.Math.rem2pi","page":"Mathematics","location":"base/math.html#Base.Math.rem2pi","category":"function","text":"rem2pi(x, r::RoundingMode)\n\nCompute the remainder of x after integer division by 2π, with the quotient rounded\naccording to the rounding mode r. In other words, the quantity\n\nx - 2π*round(x/(2π),r)\n\nwithout any intermediate rounding. This internally uses a high precision approximation of\n2π, and so will give a more accurate result than rem(x,2π,r)\n\nif r == RoundNearest, then the result is in the interval -π π. This will generally\nbe the most accurate result. See also RoundNearest.\nif r == RoundToZero, then the result is in the interval 0 2π if x is positive,.\nor -2π 0 otherwise. See also RoundToZero.\nif r == RoundDown, then the result is in the interval 0 2π.\nSee also RoundDown.\nif r == RoundUp, then the result is in the interval -2π 0.\nSee also RoundUp.\n\nExamples\n\njulia> rem2pi(7pi/4, RoundNearest)\n-0.7853981633974485\n\njulia> rem2pi(7pi/4, RoundDown)\n5.497787143782138\n\n\n\n\n\n"},{"title":"Base.Math.mod2pi","page":"Mathematics","location":"base/math.html#Base.Math.mod2pi","category":"function","text":"mod2pi(x)\n\nModulus after division by 2π, returning in the range 02π).\n\nThis function computes a floating point representation of the modulus after division by\nnumerically exact 2π, and is therefore not exactly the same as mod(x,2π), which would\ncompute the modulus of x relative to division by the floating-point number 2π.\n\nnote: Note\nDepending on the format of the input value, the closest representable value to 2π may\nbe less than 2π. For example, the expression mod2pi(2π) will not return 0, because\nthe intermediate value of 2*π is a Float64 and 2*Float64(π) < 2*big(π). See\nrem2pi for more refined control of this behavior.\n\nExamples\n\njulia> mod2pi(9*pi/4)\n0.7853981633974481\n\n\n\n\n\n"},{"title":"Base.divrem","page":"Mathematics","location":"base/math.html#Base.divrem","category":"function","text":"divrem(x, y, r::RoundingMode=RoundToZero)\n\nThe quotient and remainder from Euclidean division.\nEquivalent to (div(x, y, r), rem(x, y, r)). Equivalently, with the default\nvalue of r, this call is equivalent to (x ÷ y, x % y).\n\nSee also: fldmod, cld.\n\nExamples\n\njulia> divrem(3, 7)\n(0, 3)\n\njulia> divrem(7, 3)\n(2, 1)\n\n\n\n\n\n"},{"title":"Base.fldmod","page":"Mathematics","location":"base/math.html#Base.fldmod","category":"function","text":"fldmod(x, y)\n\nThe floored quotient and modulus after division. A convenience wrapper for\ndivrem(x, y, RoundDown). Equivalent to (fld(x, y), mod(x, y)).\n\nSee also: fld, cld, fldmod1.\n\n\n\n\n\n"},{"title":"Base.fld1","page":"Mathematics","location":"base/math.html#Base.fld1","category":"function","text":"fld1(x, y)\n\nFlooring division, returning a value consistent with mod1(x,y)\n\nSee also mod1, fldmod1.\n\nExamples\n\njulia> x = 15; y = 4;\n\njulia> fld1(x, y)\n4\n\njulia> x == fld(x, y) * y + mod(x, y)\ntrue\n\njulia> x == (fld1(x, y) - 1) * y + mod1(x, y)\ntrue\n\n\n\n\n\n"},{"title":"Base.mod1","page":"Mathematics","location":"base/math.html#Base.mod1","category":"function","text":"mod1(x, y)\n\nModulus after flooring division, returning a value r such that mod(r, y) == mod(x, y)\nin the range (0 y for positive y and in the range y0) for negative y.\n\nWith integer arguments and positive y, this is equal to mod(x, 1:y), and hence natural\nfor 1-based indexing. By comparison, mod(x, y) == mod(x, 0:y-1) is natural for computations with\noffsets or strides.\n\nSee also mod, fld1, fldmod1.\n\nExamples\n\njulia> mod1(4, 2)\n2\n\njulia> mod1.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n 1  2  3  1  2  3  1  2  3  1  2\n\njulia> mod1.([-0.1, 0, 0.1, 1, 2, 2.9, 3, 3.1]', 3)\n1×8 Matrix{Float64}:\n 2.9  3.0  0.1  1.0  2.0  2.9  3.0  0.1\n\n\n\n\n\n"},{"title":"Base.fldmod1","page":"Mathematics","location":"base/math.html#Base.fldmod1","category":"function","text":"fldmod1(x, y)\n\nReturn (fld1(x,y), mod1(x,y)).\n\nSee also fld1, mod1.\n\n\n\n\n\n"},{"title":"Base.://","page":"Mathematics","location":"base/math.html#Base.://","category":"function","text":"//(num, den)\n\nDivide two integers or rational numbers, giving a Rational result.\nMore generally, // can be used for exact rational division of other numeric types\nwith integer or rational components, such as complex numbers with integer components.\n\nNote that floating-point (AbstractFloat) arguments are not permitted by //\n(even if the values are rational).\nThe arguments must be subtypes of Integer, Rational, or composites thereof.\n\nExamples\n\njulia> 3 // 5\n3//5\n\njulia> (3 // 5) // (2 // 1)\n3//10\n\njulia> (1+2im) // (3+4im)\n11//25 + 2//25*im\n\njulia> 1.0 // 2\nERROR: MethodError: no method matching //(::Float64, ::Int64)\n[...]\n\n\n\n\n\n"},{"title":"Base.rationalize","page":"Mathematics","location":"base/math.html#Base.rationalize","category":"function","text":"rationalize([T<:Integer=Int,] x; tol::Real=eps(x))\n\nApproximate floating point number x as a Rational number with components\nof the given integer type. The result will differ from x by no more than tol.\n\nExamples\n\njulia> rationalize(5.6)\n28//5\n\njulia> a = rationalize(BigInt, 10.3)\n103//10\n\njulia> typeof(numerator(a))\nBigInt\n\n\n\n\n\n"},{"title":"Base.numerator","page":"Mathematics","location":"base/math.html#Base.numerator","category":"function","text":"numerator(x)\n\nNumerator of the rational representation of x.\n\nExamples\n\njulia> numerator(2//3)\n2\n\njulia> numerator(4)\n4\n\n\n\n\n\n"},{"title":"Base.denominator","page":"Mathematics","location":"base/math.html#Base.denominator","category":"function","text":"denominator(x)\n\nDenominator of the rational representation of x.\n\nExamples\n\njulia> denominator(2//3)\n3\n\njulia> denominator(4)\n1\n\n\n\n\n\n"},{"title":"Base.:==","page":"Mathematics","location":"base/math.html#Base.:==","category":"function","text":"==(x, y)\n\nGeneric equality operator. Falls back to ===.\nShould be implemented for all types with a notion of equality, based on the abstract value\nthat an instance represents. For example, all numeric types are compared by numeric value,\nignoring type. Strings are compared as sequences of characters, ignoring encoding.\nCollections of the same type generally compare their key sets, and if those are ==, then compare the values\nfor each of those keys, returning true if all such pairs are ==.\nOther properties are typically not taken into account (such as the exact type).\n\nThis operator follows IEEE semantics for floating-point numbers: 0.0 == -0.0 and\nNaN != NaN.\n\nThe result is of type Bool, except when one of the operands is missing,\nin which case missing is returned\n(three-valued logic).\nCollections generally implement three-valued logic akin to all, returning\nmissing if any operands contain missing values and all other pairs are equal.\nUse isequal or === to always get a Bool result.\n\nImplementation\n\nNew numeric types should implement this function for two arguments of the new type, and\nhandle comparison to other types via promotion rules where possible.\n\nEquality and hashing are intimately related; two values that are considered isequal must\nhave the same hash and by default isequal falls back to ==. If a type customizes the behavior of == and/or isequal,\nthen hash must be similarly implemented to ensure isequal and hash agree. Sets, Dicts, and many other internal\nimplementations assume that this invariant holds.\n\nIf some type defines ==, isequal, and isless then it should\nalso implement < to ensure consistency of comparisons.\n\n\n\n\n\n"},{"title":"Base.:!=","page":"Mathematics","location":"base/math.html#Base.:!=","category":"function","text":"!=(x)\n\nCreate a function that compares its argument to x using !=, i.e.\na function equivalent to y -> y != x.\nThe returned function is of type Base.Fix2{typeof(!=)}, which can be\nused to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n!=(x, y)\n≠(x,y)\n\nNot-equals comparison operator. Always gives the opposite answer as ==.\n\nImplementation\n\nNew types should generally not implement this, and rely on the fallback definition\n!=(x,y) = !(x==y) instead.\n\nExamples\n\njulia> 3 != 2\ntrue\n\njulia> \"foo\" ≠ \"foo\"\nfalse\n\n\n\n\n\n"},{"title":"Core.:!==","page":"Mathematics","location":"base/math.html#Core.:!==","category":"function","text":"!==(x, y)\n≢(x,y)\n\nAlways gives the opposite answer as ===.\n\nExamples\n\njulia> a = [1, 2]; b = [1, 2];\n\njulia> a ≢ b\ntrue\n\njulia> a ≢ a\nfalse\n\n\n\n\n\n"},{"title":"Base.:<","page":"Mathematics","location":"base/math.html#Base.:<","category":"function","text":"<(x)\n\nCreate a function that compares its argument to x using <, i.e.\na function equivalent to y -> y < x.\nThe returned function is of type Base.Fix2{typeof(<)}, which can be\nused to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n<(x, y)\n\nLess-than comparison operator. Falls back to isless.\nBecause of the behavior of floating-point NaN values, this operator implements\na partial order.\n\nImplementation\n\nNew types with a canonical partial order should implement this function for\ntwo arguments of the new type.\nTypes with a canonical total order should implement isless instead.\n\nSee also isunordered.\n\nExamples\n\njulia> 'a' < 'b'\ntrue\n\njulia> \"abc\" < \"abd\"\ntrue\n\njulia> 5 < 3\nfalse\n\n\n\n\n\n"},{"title":"Base.:<=","page":"Mathematics","location":"base/math.html#Base.:<=","category":"function","text":"<=(x)\n\nCreate a function that compares its argument to x using <=, i.e.\na function equivalent to y -> y <= x.\nThe returned function is of type Base.Fix2{typeof(<=)}, which can be\nused to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n<=(x, y)\n≤(x,y)\n\nLess-than-or-equals comparison operator. Falls back to (x < y) | (x == y).\n\nExamples\n\njulia> 'a' <= 'b'\ntrue\n\njulia> 7 ≤ 7 ≤ 9\ntrue\n\njulia> \"abc\" ≤ \"abc\"\ntrue\n\njulia> 5 <= 3\nfalse\n\n\n\n\n\n"},{"title":"Base.:>","page":"Mathematics","location":"base/math.html#Base.:>","category":"function","text":">(x)\n\nCreate a function that compares its argument to x using >, i.e.\na function equivalent to y -> y > x.\nThe returned function is of type Base.Fix2{typeof(>)}, which can be\nused to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n>(x, y)\n\nGreater-than comparison operator. Falls back to y < x.\n\nImplementation\n\nGenerally, new types should implement < instead of this function,\nand rely on the fallback definition >(x, y) = y < x.\n\nExamples\n\njulia> 'a' > 'b'\nfalse\n\njulia> 7 > 3 > 1\ntrue\n\njulia> \"abc\" > \"abd\"\nfalse\n\njulia> 5 > 3\ntrue\n\n\n\n\n\n"},{"title":"Base.:>=","page":"Mathematics","location":"base/math.html#Base.:>=","category":"function","text":">=(x)\n\nCreate a function that compares its argument to x using >=, i.e.\na function equivalent to y -> y >= x.\nThe returned function is of type Base.Fix2{typeof(>=)}, which can be\nused to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n>=(x, y)\n≥(x,y)\n\nGreater-than-or-equals comparison operator. Falls back to y <= x.\n\nImplementation\n\nNew types should prefer to implement <= instead of this function,\nand rely on the fallback definition >=(x, y) = y <= x.\n\nFurthermore, in many cases it is enough to implement just < and\n==, relying on the fallback definitions of both <= and >=.\n\nExamples\n\njulia> 'a' >= 'b'\nfalse\n\njulia> 7 ≥ 7 ≥ 3\ntrue\n\njulia> \"abc\" ≥ \"abc\"\ntrue\n\njulia> 5 >= 3\ntrue\n\n\n\n\n\n"},{"title":"Base.cmp","page":"Mathematics","location":"base/math.html#Base.cmp","category":"function","text":"cmp(a::AbstractString, b::AbstractString)::Int\n\nCompare two strings. Return 0 if both strings have the same length and the character\nat each index is the same in both strings. Return -1 if a is a prefix of b, or if\na comes before b in alphabetical order. Return 1 if b is a prefix of a, or if\nb comes before a in alphabetical order (technically, lexicographical order by Unicode\ncode points).\n\nExamples\n\njulia> cmp(\"abc\", \"abc\")\n0\n\njulia> cmp(\"ab\", \"abc\")\n-1\n\njulia> cmp(\"abc\", \"ab\")\n1\n\njulia> cmp(\"ab\", \"ac\")\n-1\n\njulia> cmp(\"ac\", \"ab\")\n1\n\njulia> cmp(\"α\", \"a\")\n1\n\njulia> cmp(\"b\", \"β\")\n-1\n\n\n\n\n\ncmp(<, x, y)\n\nReturn -1, 0, or 1 depending on whether x is less than, equal to, or greater than y,\nrespectively. The first argument specifies a less-than comparison function to use.\n\n\n\n\n\ncmp(x,y)\n\nReturn -1, 0, or 1 depending on whether x is less than, equal to, or greater than y,\nrespectively. Uses the total order implemented by isless.\n\nExamples\n\njulia> cmp(1, 2)\n-1\n\njulia> cmp(2, 1)\n1\n\njulia> cmp(2+im, 3-im)\nERROR: MethodError: no method matching isless(::Complex{Int64}, ::Complex{Int64})\n[...]\n\n\n\n\n\n"},{"title":"Base.isapprox","page":"Mathematics","location":"base/math.html#Base.isapprox","category":"function","text":"isapprox(x; kwargs...) / ≈(x; kwargs...)\n\nCreate a function that compares its argument to x using ≈, i.e. a function equivalent to y -> y ≈ x.\n\nThe keyword arguments supported here are the same as those in the 2-argument isapprox.\n\ncompat: Julia 1.5\nThis method requires Julia 1.5 or later.\n\n\n\n\n\nisapprox(x, y; atol::Real=0, rtol::Real=atol>0 ? 0 : √eps, nans::Bool=false[, norm::Function])\n\nInexact equality comparison. Two numbers compare equal if their relative distance or their\nabsolute distance is within tolerance bounds: isapprox returns true if\nnorm(x-y) <= max(atol, rtol*max(norm(x), norm(y))). The default atol (absolute tolerance) is zero and the\ndefault rtol (relative tolerance) depends on the types of x and y. The keyword argument nans determines\nwhether or not NaN values are considered equal (defaults to false).\n\nFor real or complex floating-point values, if an atol > 0 is not specified, rtol defaults to\nthe square root of eps of the type of x or y, whichever is bigger (least precise).\nThis corresponds to requiring equality of about half of the significant digits. Otherwise,\ne.g. for integer arguments or if an atol > 0 is supplied, rtol defaults to zero.\n\nThe norm keyword defaults to abs for numeric (x,y) and to LinearAlgebra.norm for\narrays (where an alternative norm choice is sometimes useful).\nWhen x and y are arrays, if norm(x-y) is not finite (i.e. ±Inf\nor NaN), the comparison falls back to checking whether all elements of x and y are\napproximately equal component-wise.\n\nThe binary operator ≈ is equivalent to isapprox with the default arguments, and x ≉ y\nis equivalent to !isapprox(x,y).\n\nNote that x ≈ 0 (i.e., comparing to zero with the default tolerances) is\nequivalent to x == 0 since the default atol is 0.  In such cases, you should either\nsupply an appropriate atol (or use norm(x) ≤ atol) or rearrange your code (e.g.\nuse x ≈ y rather than x - y ≈ 0).   It is not possible to pick a nonzero atol\nautomatically because it depends on the overall scaling (the \"units\") of your problem:\nfor example, in x - y ≈ 0, atol=1e-9 is an absurdly small tolerance if x is the\nradius of the Earth in meters,\nbut an absurdly large tolerance if x is the\nradius of a Hydrogen atom in meters.\n\ncompat: Julia 1.6\nPassing the norm keyword argument when comparing numeric (non-array) arguments\nrequires Julia 1.6 or later.\n\nExamples\n\njulia> isapprox(0.1, 0.15; atol=0.05)\ntrue\n\njulia> isapprox(0.1, 0.15; rtol=0.34)\ntrue\n\njulia> isapprox(0.1, 0.15; rtol=0.33)\nfalse\n\njulia> 0.1 + 1e-10 ≈ 0.1\ntrue\n\njulia> 1e-10 ≈ 0\nfalse\n\njulia> isapprox(1e-10, 0, atol=1e-8)\ntrue\n\njulia> isapprox([10.0^9, 1.0], [10.0^9, 2.0]) # using `norm`\ntrue\n\n\n\n\n\n"},{"title":"Base.:<<","page":"Mathematics","location":"base/math.html#Base.:<<","category":"function","text":"<<(B::BitVector, n)::BitVector\n\nLeft bit shift operator, B << n. For n >= 0, the result is B\nwith elements shifted n positions backwards, filling with false\nvalues. If n < 0, elements are shifted forwards. Equivalent to\nB >> -n.\n\nExamples\n\njulia> B = BitVector([true, false, true, false, false])\n5-element BitVector:\n 1\n 0\n 1\n 0\n 0\n\njulia> B << 1\n5-element BitVector:\n 0\n 1\n 0\n 0\n 0\n\njulia> B << -1\n5-element BitVector:\n 0\n 1\n 0\n 1\n 0\n\n\n\n\n\n<<(x, n)\n\nLeft bit shift operator, x << n. For n >= 0, the result is x shifted left\nby n bits, filling with 0s. This is equivalent to x * 2^n. For n < 0,\nthis is equivalent to x >> -n.\n\nExamples\n\njulia> Int8(3) << 2\n12\n\njulia> bitstring(Int8(3))\n\"00000011\"\n\njulia> bitstring(Int8(12))\n\"00001100\"\n\nSee also >>, >>>, exp2, ldexp.\n\n\n\n\n\n"},{"title":"Base.:>>","page":"Mathematics","location":"base/math.html#Base.:>>","category":"function","text":">>(B::BitVector, n)::BitVector\n\nRight bit shift operator, B >> n. For n >= 0, the result is B\nwith elements shifted n positions forward, filling with false\nvalues. If n < 0, elements are shifted backwards. Equivalent to\nB << -n.\n\nExamples\n\njulia> B = BitVector([true, false, true, false, false])\n5-element BitVector:\n 1\n 0\n 1\n 0\n 0\n\njulia> B >> 1\n5-element BitVector:\n 0\n 1\n 0\n 1\n 0\n\njulia> B >> -1\n5-element BitVector:\n 0\n 1\n 0\n 0\n 0\n\n\n\n\n\n>>(x, n)\n\nRight bit shift operator, x >> n. For n >= 0, the result is x shifted\nright by n bits, filling with 0s if x >= 0, 1s if x < 0, preserving\nthe sign of x. This is equivalent to fld(x, 2^n). For n < 0, this is\nequivalent to x << -n.\n\nExamples\n\njulia> Int8(13) >> 2\n3\n\njulia> bitstring(Int8(13))\n\"00001101\"\n\njulia> bitstring(Int8(3))\n\"00000011\"\n\njulia> Int8(-14) >> 2\n-4\n\njulia> bitstring(Int8(-14))\n\"11110010\"\n\njulia> bitstring(Int8(-4))\n\"11111100\"\n\nSee also >>>, <<.\n\n\n\n\n\n"},{"title":"Base.:>>>","page":"Mathematics","location":"base/math.html#Base.:>>>","category":"function","text":">>>(B::BitVector, n)::BitVector\n\nUnsigned right bitshift operator, B >>> n. Equivalent to B >> n. See >> for\ndetails and examples.\n\n\n\n\n\n>>>(x, n)\n\nUnsigned right bit shift operator, x >>> n. For n >= 0, the result is x\nshifted right by n bits, filling with 0s. For n < 0, this is equivalent\nto x << -n.\n\nFor Unsigned integer types, this is equivalent to >>. For\nSigned integer types, this is equivalent to signed(unsigned(x) >> n).\n\nExamples\n\njulia> Int8(-14) >>> 2\n60\n\njulia> bitstring(Int8(-14))\n\"11110010\"\n\njulia> bitstring(Int8(60))\n\"00111100\"\n\nBigInts are treated as if having infinite size, so no filling is required and this\nis equivalent to >>.\n\nSee also >>, <<.\n\n\n\n\n\n"},{"title":"Base.bitrotate","page":"Mathematics","location":"base/math.html#Base.bitrotate","category":"function","text":"bitrotate(x::Base.BitInteger, k::Integer)\n\nbitrotate(x, k) implements bitwise rotation.\nIt returns the value of x with its bits rotated left k times.\nA negative value of k will rotate to the right instead.\n\ncompat: Julia 1.5\nThis function requires Julia 1.5 or later.\n\nSee also: <<, circshift, BitArray.\n\njulia> bitrotate(UInt8(114), 2)\n0xc9\n\njulia> bitstring(bitrotate(0b01110010, 2))\n\"11001001\"\n\njulia> bitstring(bitrotate(0b01110010, -2))\n\"10011100\"\n\njulia> bitstring(bitrotate(0b01110010, 8))\n\"01110010\"\n\n\n\n\n\n"},{"title":"Base.:~","page":"Mathematics","location":"base/math.html#Base.:~","category":"function","text":"~(x)\n\nBitwise not.\n\nSee also: !, &, |.\n\nExamples\n\njulia> ~4\n-5\n\njulia> ~10\n-11\n\njulia> ~true\nfalse\n\n\n\n\n\n"},{"title":"Base.:&","page":"Mathematics","location":"base/math.html#Base.:&","category":"function","text":"x & y\n\nBitwise and. Implements three-valued logic,\nreturning missing if one operand is missing and the other is true. Add parentheses for\nfunction application form: (&)(x, y).\n\nSee also: |, xor, &&.\n\nExamples\n\njulia> 4 & 10\n0\n\njulia> 4 & 12\n4\n\njulia> true & missing\nmissing\n\njulia> false & missing\nfalse\n\n\n\n\n\n"},{"title":"Base.:|","page":"Mathematics","location":"base/math.html#Base.:|","category":"function","text":"x | y\n\nBitwise or. Implements three-valued logic,\nreturning missing if one operand is missing and the other is false.\n\nSee also: &, xor, ||.\n\nExamples\n\njulia> 4 | 10\n14\n\njulia> 4 | 1\n5\n\njulia> true | missing\ntrue\n\njulia> false | missing\nmissing\n\n\n\n\n\n"},{"title":"Base.xor","page":"Mathematics","location":"base/math.html#Base.xor","category":"function","text":"xor(x, y)\n⊻(x, y)\n\nBitwise exclusive or of x and y. Implements\nthree-valued logic,\nreturning missing if one of the arguments is missing.\n\nThe infix operation a ⊻ b is a synonym for xor(a,b), and\n⊻ can be typed by tab-completing \\xor or \\veebar in the Julia REPL.\n\nExamples\n\njulia> xor(true, false)\ntrue\n\njulia> xor(true, true)\nfalse\n\njulia> xor(true, missing)\nmissing\n\njulia> false ⊻ false\nfalse\n\njulia> [true; true; false] .⊻ [true; false; false]\n3-element BitVector:\n 0\n 1\n 0\n\n\n\n\n\n"},{"title":"Base.nand","page":"Mathematics","location":"base/math.html#Base.nand","category":"function","text":"nand(x, y)\n⊼(x, y)\n\nBitwise nand (not and) of x and y. Implements\nthree-valued logic,\nreturning missing if one of the arguments is missing.\n\nThe infix operation a ⊼ b is a synonym for nand(a,b), and\n⊼ can be typed by tab-completing \\nand or \\barwedge in the Julia REPL.\n\nExamples\n\njulia> nand(true, false)\ntrue\n\njulia> nand(true, true)\nfalse\n\njulia> nand(true, missing)\nmissing\n\njulia> false ⊼ false\ntrue\n\njulia> [true; true; false] .⊼ [true; false; false]\n3-element BitVector:\n 0\n 1\n 1\n\n\n\n\n\n"},{"title":"Base.nor","page":"Mathematics","location":"base/math.html#Base.nor","category":"function","text":"nor(x, y)\n⊽(x, y)\n\nBitwise nor (not or) of x and y. Implements\nthree-valued logic,\nreturning missing if one of the arguments is missing and the\nother is not true.\n\nThe infix operation a ⊽ b is a synonym for nor(a,b), and\n⊽ can be typed by tab-completing \\nor or \\barvee in the Julia REPL.\n\nExamples\n\njulia> nor(true, false)\nfalse\n\njulia> nor(true, true)\nfalse\n\njulia> nor(true, missing)\nfalse\n\njulia> false ⊽ false\ntrue\n\njulia> false ⊽ missing\nmissing\n\njulia> [true; true; false] .⊽ [true; false; false]\n3-element BitVector:\n 0\n 0\n 1\n\n\n\n\n\n"},{"title":"Base.:!","page":"Mathematics","location":"base/math.html#Base.:!","category":"function","text":"!f::Function\n\nPredicate function negation: when the argument of ! is a function, it returns a composed function which computes the boolean negation of f.\n\nSee also ∘.\n\nExamples\n\njulia> str = \"∀ ε > 0, ∃ δ > 0: |x-y| < δ ⇒ |f(x)-f(y)| < ε\"\n\"∀ ε > 0, ∃ δ > 0: |x-y| < δ ⇒ |f(x)-f(y)| < ε\"\n\njulia> filter(isletter, str)\n\"εδxyδfxfyε\"\n\njulia> filter(!isletter, str)\n\"∀  > 0, ∃  > 0: |-| <  ⇒ |()-()| < \"\n\ncompat: Julia 1.9\nStarting with Julia 1.9, !f returns a ComposedFunction instead of an anonymous function.\n\n\n\n\n\n!(x)\n\nBoolean not. Implements three-valued logic,\nreturning missing if x is missing.\n\nSee also ~ for bitwise not.\n\nExamples\n\njulia> !true\nfalse\n\njulia> !false\ntrue\n\njulia> !missing\nmissing\n\njulia> .![true false true]\n1×3 BitMatrix:\n 0  1  0\n\n\n\n\n\n"},{"title":"&&","page":"Mathematics","location":"base/math.html#&&","category":"keyword","text":"x && y\n\nShort-circuiting boolean AND.\n\nThis is equivalent to x ? y : false: it returns false if x is false and the result of evaluating y if x is true.\nNote that if y is an expression, it is only evaluated when x is true, which is called \"short-circuiting\" behavior.\n\nAlso, y does not need to have a boolean value.  This means that (condition) && (statement) can be used as shorthand for\nif condition; statement; end for an arbitrary statement.\n\nSee also &, the ternary operator ? :, and the manual section on control flow.\n\nExamples\n\njulia> x = 3;\n\njulia> x > 1 && x < 10 && x isa Int\ntrue\n\njulia> x < 0 && error(\"expected positive x\")\nfalse\n\njulia> x > 0 && \"not a boolean\"\n\"not a boolean\"\n\n\n\n\n\n"},{"title":"||","page":"Mathematics","location":"base/math.html#||","category":"keyword","text":"x || y\n\nShort-circuiting boolean OR.\n\nThis is equivalent to x ? true : y: it returns true if x is true and the result of evaluating y if x is false.\nNote that if y is an expression, it is only evaluated when x is false, which is called \"short-circuiting\" behavior.\n\nAlso, y does not need to have a boolean value.  This means that (condition) || (statement) can be used as shorthand for\nif !(condition); statement; end for an arbitrary statement.\n\nSee also: |, xor, &&.\n\nExamples\n\njulia> pi < 3 || ℯ < 3\ntrue\n\njulia> false || true || println(\"neither is true!\")\ntrue\n\njulia> pi < 3 || \"not a boolean\"\n\"not a boolean\"\n\n\n\n\n\n"},{"title":"Base.sin","page":"Mathematics","location":"base/math.html#Base.sin-Tuple{Number}","category":"method","text":"sin(x::T) where {T <: Number} -> float(T)\n\nCompute sine of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also sind, sinpi, sincos, cis, asin.\n\nExamples\n\njulia> round.(sin.(range(0, 2pi, length=9)'), digits=3)\n1×9 Matrix{Float64}:\n 0.0  0.707  1.0  0.707  0.0  -0.707  -1.0  -0.707  -0.0\n\njulia> sind(45)\n0.7071067811865476\n\njulia> sinpi(1/4)\n0.7071067811865476\n\njulia> round.(sincos(pi/6), digits=3)\n(0.5, 0.866)\n\njulia> round(cis(pi/6), digits=3)\n0.866 + 0.5im\n\njulia> round(exp(im*pi/6), digits=3)\n0.866 + 0.5im\n\n\n\n\n\n"},{"title":"Base.cos","page":"Mathematics","location":"base/math.html#Base.cos-Tuple{Number}","category":"method","text":"cos(x::T) where {T <: Number} -> float(T)\n\nCompute cosine of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also cosd, cospi, sincos, cis.\n\n\n\n\n\n"},{"title":"Base.Math.sincos","page":"Mathematics","location":"base/math.html#Base.Math.sincos-Tuple{Float64}","category":"method","text":"sincos(x::T) where T -> Tuple{float(T),float(T)}\n\nSimultaneously compute the sine and cosine of x, where x is in radians, returning\na tuple (sine, cosine).\n\nThrow a DomainError if isinf(x), return a (T(NaN), T(NaN)) if isnan(x).\n\nSee also cis, sincospi, sincosd.\n\n\n\n\n\n"},{"title":"Base.tan","page":"Mathematics","location":"base/math.html#Base.tan-Tuple{Number}","category":"method","text":"tan(x::T) where {T <: Number} -> float(T)\n\nCompute tangent of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also tanh.\n\n\n\n\n\n"},{"title":"Base.asin","page":"Mathematics","location":"base/math.html#Base.asin-Tuple{Number}","category":"method","text":"asin(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse sine of x, where the output is in radians.\n\nReturn a T(NaN) if isnan(x).\n\nSee also asind for output in degrees.\n\nExamples\n\njulia> asin.((0, 1/2, 1))\n(0.0, 0.5235987755982989, 1.5707963267948966)\n\njulia> asind.((0, 1/2, 1))\n(0.0, 30.000000000000004, 90.0)\n\n\n\n\n\n"},{"title":"Base.acos","page":"Mathematics","location":"base/math.html#Base.acos-Tuple{Number}","category":"method","text":"acos(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse cosine of x, where the output is in radians.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.atan","page":"Mathematics","location":"base/math.html#Base.atan-Tuple{Number}","category":"method","text":"atan(y)\natan(y, x)\n\nCompute the inverse tangent of y or y/x, respectively.\n\nFor one real argument, this is the angle in radians between the positive x-axis and the point\n(1, y), returning a value in the interval -pi2 pi2.\n\nFor two arguments, this is the angle in radians between the positive x-axis and the\npoint (x, y), returning a value in the interval -pi pi. This corresponds to a\nstandard atan2 function. Note that by convention\natan(0.0,x) is defined as pi and atan(-0.0,x) is defined as -pi when x < 0.\n\nSee also atand for degrees.\n\nExamples\n\njulia> rad2deg(atan(-1/√3))\n-30.000000000000004\n\njulia> rad2deg(atan(-1, √3))\n-30.000000000000004\n\njulia> rad2deg(atan(1, -√3))\n150.0\n\n\n\n\n\n"},{"title":"Base.Math.sec","page":"Mathematics","location":"base/math.html#Base.Math.sec-Tuple{Number}","category":"method","text":"sec(x::T) where {T <: Number} -> float(T)\n\nCompute the secant of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.csc","page":"Mathematics","location":"base/math.html#Base.Math.csc-Tuple{Number}","category":"method","text":"csc(x::T) where {T <: Number} -> float(T)\n\nCompute the cosecant of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.cot","page":"Mathematics","location":"base/math.html#Base.Math.cot-Tuple{Number}","category":"method","text":"cot(x::T) where {T <: Number} -> float(T)\n\nCompute the cotangent of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.asec","page":"Mathematics","location":"base/math.html#Base.Math.asec-Tuple{Number}","category":"method","text":"asec(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse secant of x, where the output is in radians.\n\n\n\n\n\n"},{"title":"Base.Math.acsc","page":"Mathematics","location":"base/math.html#Base.Math.acsc-Tuple{Number}","category":"method","text":"acsc(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse cosecant of x, where the output is in radians.\n\n\n\n\n\n"},{"title":"Base.Math.acot","page":"Mathematics","location":"base/math.html#Base.Math.acot-Tuple{Number}","category":"method","text":"acot(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse cotangent of x, where the output is in radians.\n\n\n\n\n\n"},{"title":"Base.Math.sind","page":"Mathematics","location":"base/math.html#Base.Math.sind","category":"function","text":"sind(x::T) where T -> float(T)\n\nCompute sine of x, where x is in degrees.\nIf x is a matrix, x needs to be a square matrix.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.cosd","page":"Mathematics","location":"base/math.html#Base.Math.cosd","category":"function","text":"cosd(x::T) where T -> float(T)\n\nCompute cosine of x, where x is in degrees.\nIf x is a matrix, x needs to be a square matrix.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.tand","page":"Mathematics","location":"base/math.html#Base.Math.tand","category":"function","text":"tand(x::T) where T -> float(T)\n\nCompute tangent of x, where x is in degrees.\nIf x is a matrix, x needs to be a square matrix.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.sincosd","page":"Mathematics","location":"base/math.html#Base.Math.sincosd","category":"function","text":"sincosd(x::T) where T -> Tuple{float(T),float(T)}\n\nSimultaneously compute the sine and cosine of x, where x is in degrees, returning\na tuple (sine, cosine).\n\nThrow a DomainError if isinf(x), return a (T(NaN), T(NaN)) tuple if isnan(x).\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n"},{"title":"Base.Math.asind","page":"Mathematics","location":"base/math.html#Base.Math.asind","category":"function","text":"asind(x)\n\nCompute the inverse sine of x, where the output is in degrees.\nIf x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.acosd","page":"Mathematics","location":"base/math.html#Base.Math.acosd","category":"function","text":"acosd(x)\n\nCompute the inverse cosine of x, where the output is in degrees.\nIf x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.atand","page":"Mathematics","location":"base/math.html#Base.Math.atand","category":"function","text":"atand(y::T) where T -> float(T)\natand(y::T, x::S) where {T,S} -> promote_type(T,S)\natand(y::AbstractMatrix{T}) where T -> AbstractMatrix{Complex{float(T)}}\n\nCompute the inverse tangent of y or y/x, respectively, where the output is in degrees.\n\nReturn a NaN if isnan(y) or isnan(x). The returned NaN is either a T in the single\nargument version, or a promote_type(T,S) in the two argument version.\n\ncompat: Julia 1.7\nThe one-argument method supports square matrix arguments as of Julia 1.7.\n\n\n\n\n\n"},{"title":"Base.Math.secd","page":"Mathematics","location":"base/math.html#Base.Math.secd","category":"function","text":"secd(x::T) where {T <: Number} -> float(T)\n\nCompute the secant of x, where x is in degrees.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.cscd","page":"Mathematics","location":"base/math.html#Base.Math.cscd","category":"function","text":"cscd(x::T) where {T <: Number} -> float(T)\n\nCompute the cosecant of x, where x is in degrees.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.cotd","page":"Mathematics","location":"base/math.html#Base.Math.cotd","category":"function","text":"cotd(x::T) where {T <: Number} -> float(T)\n\nCompute the cotangent of x, where x is in degrees.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.asecd","page":"Mathematics","location":"base/math.html#Base.Math.asecd","category":"function","text":"asecd(x)\n\nCompute the inverse secant of x, where the output is in degrees.\nIf x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.acscd","page":"Mathematics","location":"base/math.html#Base.Math.acscd","category":"function","text":"acscd(x)\n\nCompute the inverse cosecant of x, where the output is in degrees.\nIf x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.acotd","page":"Mathematics","location":"base/math.html#Base.Math.acotd","category":"function","text":"acotd(x)\n\nCompute the inverse cotangent of x, where the output is in degrees.\nIf x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n"},{"title":"Base.Math.sinpi","page":"Mathematics","location":"base/math.html#Base.Math.sinpi","category":"function","text":"sinpi(x::T) where T -> float(T)\n\nCompute sin(pi x) more accurately than sin(pi*x), especially for large x.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also sind, cospi, sincospi.\n\n\n\n\n\n"},{"title":"Base.Math.cospi","page":"Mathematics","location":"base/math.html#Base.Math.cospi","category":"function","text":"cospi(x::T) where T -> float(T)\n\nCompute cos(pi x) more accurately than cos(pi*x), especially for large x.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also: cispi, sincosd, sinpi.\n\n\n\n\n\n"},{"title":"Base.Math.tanpi","page":"Mathematics","location":"base/math.html#Base.Math.tanpi","category":"function","text":"tanpi(x::T) where T -> float(T)\n\nCompute tan(pi x) more accurately than tan(pi*x), especially for large x.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\nSee also tand, sinpi, cospi, sincospi.\n\n\n\n\n\n"},{"title":"Base.Math.sincospi","page":"Mathematics","location":"base/math.html#Base.Math.sincospi","category":"function","text":"sincospi(x::T) where T -> Tuple{float(T),float(T)}\n\nSimultaneously compute sinpi(x) and cospi(x) (the sine and cosine of π*x,\nwhere x is in radians), returning a tuple (sine, cosine).\n\nThrow a DomainError if isinf(x), return a (T(NaN), T(NaN)) tuple if isnan(x).\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\nSee also: cispi, sincosd, sinpi.\n\n\n\n\n\n"},{"title":"Base.sinh","page":"Mathematics","location":"base/math.html#Base.sinh-Tuple{Number}","category":"method","text":"sinh(x)\n\nCompute hyperbolic sine of x.\n\nSee also sin.\n\n\n\n\n\n"},{"title":"Base.cosh","page":"Mathematics","location":"base/math.html#Base.cosh-Tuple{Number}","category":"method","text":"cosh(x)\n\nCompute hyperbolic cosine of x.\n\nSee also cos.\n\n\n\n\n\n"},{"title":"Base.tanh","page":"Mathematics","location":"base/math.html#Base.tanh-Tuple{Number}","category":"method","text":"tanh(x)\n\nCompute hyperbolic tangent of x.\n\nSee also tan, atanh.\n\nExamples\n\njulia> tanh.(-3:3f0)  # Here 3f0 isa Float32\n7-element Vector{Float32}:\n -0.9950548\n -0.9640276\n -0.7615942\n  0.0\n  0.7615942\n  0.9640276\n  0.9950548\n\njulia> tan.(im .* (1:3))\n3-element Vector{ComplexF64}:\n 0.0 + 0.7615941559557649im\n 0.0 + 0.9640275800758169im\n 0.0 + 0.9950547536867306im\n\n\n\n\n\n"},{"title":"Base.asinh","page":"Mathematics","location":"base/math.html#Base.asinh-Tuple{Number}","category":"method","text":"asinh(x)\n\nCompute the inverse hyperbolic sine of x.\n\n\n\n\n\n"},{"title":"Base.acosh","page":"Mathematics","location":"base/math.html#Base.acosh-Tuple{Number}","category":"method","text":"acosh(x)\n\nCompute the inverse hyperbolic cosine of x.\n\n\n\n\n\n"},{"title":"Base.atanh","page":"Mathematics","location":"base/math.html#Base.atanh-Tuple{Number}","category":"method","text":"atanh(x)\n\nCompute the inverse hyperbolic tangent of x.\n\n\n\n\n\n"},{"title":"Base.Math.sech","page":"Mathematics","location":"base/math.html#Base.Math.sech-Tuple{Number}","category":"method","text":"sech(x::T) where {T <: Number} -> float(T)\n\nCompute the hyperbolic secant of x.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.csch","page":"Mathematics","location":"base/math.html#Base.Math.csch-Tuple{Number}","category":"method","text":"csch(x::T) where {T <: Number} -> float(T)\n\nCompute the hyperbolic cosecant of x.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.coth","page":"Mathematics","location":"base/math.html#Base.Math.coth-Tuple{Number}","category":"method","text":"coth(x::T) where {T <: Number} -> float(T)\n\nCompute the hyperbolic cotangent of x.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n"},{"title":"Base.Math.asech","page":"Mathematics","location":"base/math.html#Base.Math.asech-Tuple{Number}","category":"method","text":"asech(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse hyperbolic secant of x.\n\n\n\n\n\n"},{"title":"Base.Math.acsch","page":"Mathematics","location":"base/math.html#Base.Math.acsch-Tuple{Number}","category":"method","text":"acsch(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse hyperbolic cosecant of x.\n\n\n\n\n\n"},{"title":"Base.Math.acoth","page":"Mathematics","location":"base/math.html#Base.Math.acoth-Tuple{Number}","category":"method","text":"acoth(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse hyperbolic cotangent of x.\n\n\n\n\n\n"},{"title":"Base.Math.sinc","page":"Mathematics","location":"base/math.html#Base.Math.sinc","category":"function","text":"sinc(x::T) where {T <: Number} -> float(T)\n\nCompute normalized sinc function operatornamesinc(x) = sin(pi x)  (pi x) if x neq 0, and 1 if x = 0.\n\nReturn a T(NaN) if isnan(x).\n\nSee also cosc, its derivative.\n\n\n\n\n\n"},{"title":"Base.Math.cosc","page":"Mathematics","location":"base/math.html#Base.Math.cosc","category":"function","text":"cosc(x::T) where {T <: Number} -> float(T)\n\nCompute cos(pi x)  x - sin(pi x)  (pi x^2) if x neq 0, and 0 if\nx = 0. This is the derivative of sinc(x).\n\nReturn a T(NaN) if isnan(x).\n\nSee also sinc.\n\n\n\n\n\n"},{"title":"Base.Math.deg2rad","page":"Mathematics","location":"base/math.html#Base.Math.deg2rad","category":"function","text":"deg2rad(x)\n\nConvert x from degrees to radians.\n\nSee also rad2deg, sind, pi.\n\nExamples\n\njulia> deg2rad(90)\n1.5707963267948966\n\n\n\n\n\n"},{"title":"Base.Math.rad2deg","page":"Mathematics","location":"base/math.html#Base.Math.rad2deg","category":"function","text":"rad2deg(x)\n\nConvert x from radians to degrees.\n\nSee also deg2rad.\n\nExamples\n\njulia> rad2deg(pi)\n180.0\n\n\n\n\n\n"},{"title":"Base.Math.hypot","page":"Mathematics","location":"base/math.html#Base.Math.hypot","category":"function","text":"hypot(x, y)\n\nCompute the hypotenuse sqrtx^2+y^2 avoiding overflow and underflow.\n\nThis code is an implementation of the algorithm described in:\nAn Improved Algorithm for hypot(a,b) by Carlos F. Borges\n\nhypot(x...)\n\nCompute the hypotenuse sqrtsum x_i^2 avoiding overflow and underflow.\n\nSee also norm in the LinearAlgebra standard library.\n\nExamples\n\njulia> a = Int64(10)^10;\n\njulia> hypot(a, a)\n1.4142135623730951e10\n\njulia> √(a^2 + a^2) # a^2 overflows\nERROR: DomainError with -2.914184810805068e18:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\njulia> hypot(3, 4im)\n5.0\n\njulia> hypot(-5.7)\n5.7\n\njulia> hypot(3, 4im, 12.0)\n13.0\n\njulia> using LinearAlgebra\n\njulia> norm([a, a, a, a]) == hypot(a, a, a, a)\ntrue\n\n\n\n\n\n"},{"title":"Base.log","page":"Mathematics","location":"base/math.html#Base.log-Tuple{Number}","category":"method","text":"log(x)\n\nCompute the natural logarithm of x.\n\nThrow a DomainError for negative Real arguments.\nUse Complex arguments to obtain Complex results.\n\nnote: Branch cut\nlog has a branch cut along the negative real axis; -0.0im is taken\nto be below the axis.\n\nSee also ℯ, log1p, log2, log10.\n\nExamples\n\njulia> log(2)\n0.6931471805599453\n\njulia> log(-3)\nERROR: DomainError with -3.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\njulia> log(-3 + 0im)\n1.0986122886681098 + 3.141592653589793im\n\njulia> log(-3 - 0.0im)\n1.0986122886681098 - 3.141592653589793im\n\njulia> log.(exp.(-1:1))\n3-element Vector{Float64}:\n -1.0\n  0.0\n  1.0\n\n\n\n\n\n"},{"title":"Base.log","page":"Mathematics","location":"base/math.html#Base.log-Tuple{Number, Number}","category":"method","text":"log(b,x)\n\nCompute the base b logarithm of x. Throw a DomainError for negative\nReal arguments.\n\nExamples\n\njulia> log(4,8)\n1.5\n\njulia> log(4,2)\n0.5\n\njulia> log(-2, 3)\nERROR: DomainError with -2.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\njulia> log(2, -3)\nERROR: DomainError with -3.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\nnote: Note\nIf b is a power of 2 or 10, log2 or log10 should be used, as these will\ntypically be faster and more accurate. For example,julia> log(100,1000000)\n2.9999999999999996\n\njulia> log10(1000000)/2\n3.0\n\n\n\n\n\n"},{"title":"Base.log2","page":"Mathematics","location":"base/math.html#Base.log2","category":"function","text":"log2(x)\n\nCompute the logarithm of x to base 2. Throw a DomainError for negative\nReal arguments.\n\nSee also: exp2, ldexp, ispow2.\n\nExamples\n\njulia> log2(4)\n2.0\n\njulia> log2(10)\n3.321928094887362\n\njulia> log2(-2)\nERROR: DomainError with -2.0:\nlog2 was called with a negative real argument but will only return a complex result if called with a complex argument. Try log2(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(f::Symbol, x::Float64) at ./math.jl:31\n[...]\n\njulia> log2.(2.0 .^ (-1:1))\n3-element Vector{Float64}:\n -1.0\n  0.0\n  1.0\n\n\n\n\n\n"},{"title":"Base.log10","page":"Mathematics","location":"base/math.html#Base.log10","category":"function","text":"log10(x)\n\nCompute the logarithm of x to base 10.\nThrow a DomainError for negative Real arguments.\n\nExamples\n\njulia> log10(100)\n2.0\n\njulia> log10(2)\n0.3010299956639812\n\njulia> log10(-2)\nERROR: DomainError with -2.0:\nlog10 was called with a negative real argument but will only return a complex result if called with a complex argument. Try log10(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(f::Symbol, x::Float64) at ./math.jl:31\n[...]\n\n\n\n\n\n"},{"title":"Base.log1p","page":"Mathematics","location":"base/math.html#Base.log1p","category":"function","text":"log1p(x)\n\nAccurate natural logarithm of 1+x. Throw a DomainError for Real\narguments less than -1.\n\nExamples\n\njulia> log1p(-0.5)\n-0.6931471805599453\n\njulia> log1p(0)\n0.0\n\njulia> log1p(-2)\nERROR: DomainError with -2.0:\nlog1p was called with a real argument < -1 but will only return a complex result if called with a complex argument. Try log1p(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\n\n\n\n\n"},{"title":"Base.exp","page":"Mathematics","location":"base/math.html#Base.exp-Tuple{Float64}","category":"method","text":"exp(x)\n\nCompute the natural base exponential of x, in other words ℯ^x.\n\nSee also exp2, exp10 and cis.\n\nExamples\n\njulia> exp(1.0)\n2.718281828459045\n\njulia> exp(im * pi) ≈ cis(pi)\ntrue\n\n\n\n\n\n"},{"title":"Base.exp2","page":"Mathematics","location":"base/math.html#Base.exp2","category":"function","text":"exp2(x)\n\nCompute the base 2 exponential of x, in other words 2^x.\n\nSee also ldexp, <<.\n\nExamples\n\njulia> exp2(5)\n32.0\n\njulia> 2^5\n32\n\njulia> exp2(63) > typemax(Int)\ntrue\n\n\n\n\n\n"},{"title":"Base.exp10","page":"Mathematics","location":"base/math.html#Base.exp10","category":"function","text":"exp10(x)\n\nCompute the base 10 exponential of x, in other words 10^x.\n\nExamples\n\njulia> exp10(2)\n100.0\n\njulia> 10^2\n100\n\n\n\n\n\n"},{"title":"Base.expm1","page":"Mathematics","location":"base/math.html#Base.expm1","category":"function","text":"expm1(x)\n\nAccurately compute e^x-1. It avoids the loss of precision involved in the direct\nevaluation of exp(x) - 1 for small values of x.\n\nExamples\n\njulia> expm1(1e-16)\n1.0e-16\n\njulia> exp(1e-16) - 1\n0.0\n\n\n\n\n\n"},{"title":"Base.round","page":"Mathematics","location":"base/math.html#Base.round","category":"function","text":"round([T,] x, [r::RoundingMode])\nround(x, [r::RoundingMode]; digits::Integer=0, base = 10)\nround(x, [r::RoundingMode]; sigdigits::Integer, base = 10)\n\nRounds the number x.\n\nWithout keyword arguments, x is rounded to an integer value, returning a value of type\nT, or of the same type of x if no T is provided. An InexactError will be\nthrown if the value is not representable by T, similar to convert.\n\nIf the digits keyword argument is provided, it rounds to the specified number of digits\nafter the decimal place (or before if digits is negative), in base base.\n\nIf the sigdigits keyword argument is provided, it rounds to the specified number of\nsignificant digits, in base base.\n\nThe RoundingMode r controls the direction of the rounding; the default is\nRoundNearest, which rounds to the nearest integer, with ties (fractional values\nof 0.5) being rounded to the nearest even integer. Note that round may give incorrect\nresults if the global rounding mode is changed (see rounding).\n\nWhen rounding to a floating point type, will round to integers representable by that type\n(and Inf) rather than true integers. Inf is treated as one ulp greater than the\nfloatmax(T) for purposes of determining \"nearest\", similar to convert.\n\nExamples\n\njulia> round(1.7)\n2.0\n\njulia> round(Int, 1.7)\n2\n\njulia> round(1.5)\n2.0\n\njulia> round(2.5)\n2.0\n\njulia> round(pi; digits=2)\n3.14\n\njulia> round(pi; digits=3, base=2)\n3.125\n\njulia> round(123.456; sigdigits=2)\n120.0\n\njulia> round(357.913; sigdigits=4, base=2)\n352.0\n\njulia> round(Float16, typemax(UInt128))\nInf16\n\njulia> floor(Float16, typemax(UInt128))\nFloat16(6.55e4)\n\nnote: Note\nRounding to specified digits in bases other than 2 can be inexact when\noperating on binary floating point numbers. For example, the Float64\nvalue represented by 1.15 is actually less than 1.15, yet will be\nrounded to 1.2. For example:julia> x = 1.15\n1.15\n\njulia> big(1.15)\n1.149999999999999911182158029987476766109466552734375\n\njulia> x < 115//100\ntrue\n\njulia> round(x, digits=1)\n1.2\n\nExtensions\n\nTo extend round to new numeric types, it is typically sufficient to define Base.round(x::NewType, r::RoundingMode).\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundingMode","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundingMode","category":"type","text":"RoundingMode\n\nA type used for controlling the rounding mode of floating point operations (via\nrounding/setrounding functions), or as\noptional arguments for rounding to the nearest integer (via the round\nfunction).\n\nCurrently supported rounding modes are:\n\nRoundNearest (default)\nRoundNearestTiesAway\nRoundNearestTiesUp\nRoundToZero\nRoundFromZero\nRoundUp\nRoundDown\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9. Prior versions support\nRoundFromZero for BigFloats only.\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundNearest","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundNearest","category":"constant","text":"RoundNearest\n\nThe default rounding mode. Rounds to the nearest integer, with ties (fractional values of\n0.5) being rounded to the nearest even integer.\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundNearestTiesAway","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundNearestTiesAway","category":"constant","text":"RoundNearestTiesAway\n\nRounds to nearest integer, with ties rounded away from zero (C/C++\nround behaviour).\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundNearestTiesUp","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundNearestTiesUp","category":"constant","text":"RoundNearestTiesUp\n\nRounds to nearest integer, with ties rounded toward positive infinity (Java/JavaScript\nround behaviour).\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundToZero","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundToZero","category":"constant","text":"RoundToZero\n\nround using this rounding mode is an alias for trunc.\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundFromZero","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundFromZero","category":"constant","text":"RoundFromZero\n\nRounds away from zero.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9. Prior versions support\nRoundFromZero for BigFloats only.\n\nExamples\n\njulia> BigFloat(\"1.0000000000000001\", 5, RoundFromZero)\n1.06\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundUp","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundUp","category":"constant","text":"RoundUp\n\nround using this rounding mode is an alias for ceil.\n\n\n\n\n\n"},{"title":"Base.Rounding.RoundDown","page":"Mathematics","location":"base/math.html#Base.Rounding.RoundDown","category":"constant","text":"RoundDown\n\nround using this rounding mode is an alias for floor.\n\n\n\n\n\n"},{"title":"Base.round","page":"Mathematics","location":"base/math.html#Base.round-Tuple{Complex{<:AbstractFloat}, RoundingMode, RoundingMode}","category":"method","text":"round(z::Complex[, RoundingModeReal, [RoundingModeImaginary]])\nround(z::Complex[, RoundingModeReal, [RoundingModeImaginary]]; digits=0, base=10)\nround(z::Complex[, RoundingModeReal, [RoundingModeImaginary]]; sigdigits, base=10)\n\nReturn the nearest integral value of the same type as the complex-valued z to z,\nbreaking ties using the specified RoundingModes. The first\nRoundingMode is used for rounding the real components while the\nsecond is used for rounding the imaginary components.\n\nRoundingModeReal and RoundingModeImaginary default to RoundNearest,\nwhich rounds to the nearest integer, with ties (fractional values of 0.5)\nbeing rounded to the nearest even integer.\n\nExamples\n\njulia> round(3.14 + 4.5im)\n3.0 + 4.0im\n\njulia> round(3.14 + 4.5im, RoundUp, RoundNearestTiesUp)\n4.0 + 5.0im\n\njulia> round(3.14159 + 4.512im; digits = 1)\n3.1 + 4.5im\n\njulia> round(3.14159 + 4.512im; sigdigits = 3)\n3.14 + 4.51im\n\n\n\n\n\n"},{"title":"Base.ceil","page":"Mathematics","location":"base/math.html#Base.ceil","category":"function","text":"ceil([T,] x)\nceil(x; digits::Integer= [, base = 10])\nceil(x; sigdigits::Integer= [, base = 10])\n\nceil(x) returns the nearest integral value of the same type as x that is greater than or\nequal to x.\n\nceil(T, x) converts the result to type T, throwing an InexactError if the ceiled\nvalue is not representable as a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support ceil for a new type, define Base.round(x::NewType, ::RoundingMode{:Up}).\n\n\n\n\n\n"},{"title":"Base.floor","page":"Mathematics","location":"base/math.html#Base.floor","category":"function","text":"floor([T,] x)\nfloor(x; digits::Integer= [, base = 10])\nfloor(x; sigdigits::Integer= [, base = 10])\n\nfloor(x) returns the nearest integral value of the same type as x that is less than or\nequal to x.\n\nfloor(T, x) converts the result to type T, throwing an InexactError if the floored\nvalue is not representable a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support floor for a new type, define Base.round(x::NewType, ::RoundingMode{:Down}).\n\n\n\n\n\n"},{"title":"Base.trunc","page":"Mathematics","location":"base/math.html#Base.trunc","category":"function","text":"trunc([T,] x)\ntrunc(x; digits::Integer= [, base = 10])\ntrunc(x; sigdigits::Integer= [, base = 10])\n\ntrunc(x) returns the nearest integral value of the same type as x whose absolute value\nis less than or equal to the absolute value of x.\n\ntrunc(T, x) converts the result to type T, throwing an InexactError if the truncated\nvalue is not representable a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support trunc for a new type, define Base.round(x::NewType, ::RoundingMode{:ToZero}).\n\nSee also: %, floor, unsigned, unsafe_trunc.\n\nExamples\n\njulia> trunc(2.22)\n2.0\n\njulia> trunc(-2.22, digits=1)\n-2.2\n\njulia> trunc(Int, -2.22)\n-2\n\n\n\n\n\n"},{"title":"Base.unsafe_trunc","page":"Mathematics","location":"base/math.html#Base.unsafe_trunc","category":"function","text":"unsafe_trunc(T, x)\n\nReturn the nearest integral value of type T whose absolute value is\nless than or equal to the absolute value of x. If the value is not representable by T,\nan arbitrary value will be returned.\nSee also trunc.\n\nExamples\n\njulia> unsafe_trunc(Int, -2.2)\n-2\n\njulia> unsafe_trunc(Int, NaN) isa Int\ntrue\n\n\n\n\n\n"},{"title":"Base.Math.modf","page":"Mathematics","location":"base/math.html#Base.Math.modf","category":"function","text":"modf(x)\n\nReturn a tuple (fpart, ipart) of the fractional and integral parts of a number. Both parts\nhave the same sign as the argument.\n\nExamples\n\njulia> modf(3.5)\n(0.5, 3.0)\n\njulia> modf(-3.5)\n(-0.5, -3.0)\n\n\n\n\n\n"},{"title":"Base.min","page":"Mathematics","location":"base/math.html#Base.min","category":"function","text":"min(x, y, ...)\n\nReturn the minimum of the arguments, with respect to isless.\nIf any of the arguments is missing, return missing.\nSee also the minimum function to take the minimum element from a collection.\n\nExamples\n\njulia> min(2, 5, 1)\n1\n\njulia> min(4, missing, 6)\nmissing\n\n\n\n\n\n"},{"title":"Base.max","page":"Mathematics","location":"base/math.html#Base.max","category":"function","text":"max(x, y, ...)\n\nReturn the maximum of the arguments, with respect to isless.\nIf any of the arguments is missing, return missing.\nSee also the maximum function to take the maximum element from a collection.\n\nExamples\n\njulia> max(2, 5, 1)\n5\n\njulia> max(5, missing, 6)\nmissing\n\n\n\n\n\n"},{"title":"Base.minmax","page":"Mathematics","location":"base/math.html#Base.minmax","category":"function","text":"minmax(x, y)\n\nReturn (min(x,y), max(x,y)).\n\nSee also extrema that returns (minimum(x), maximum(x)).\n\nExamples\n\njulia> minmax('c','b')\n('b', 'c')\n\n\n\n\n\n"},{"title":"Base.clamp","page":"Mathematics","location":"base/math.html#Base.clamp","category":"function","text":"clamp(x::Integer, r::AbstractUnitRange)\n\nClamp x to lie within range r.\n\ncompat: Julia 1.6\nThis method requires at least Julia 1.6.\n\n\n\n\n\nclamp(x, T)::T\n\nClamp x between typemin(T) and typemax(T) and convert the result to type T.\n\nSee also trunc.\n\nExamples\n\njulia> clamp(200, Int8)\n127\n\njulia> clamp(-200, Int8)\n-128\n\njulia> trunc(Int, 4pi^2)\n39\n\n\n\n\n\nclamp(x, lo, hi)\n\nReturn x if lo <= x <= hi. If x > hi, return hi. If x < lo, return lo. Arguments\nare promoted to a common type.\n\nSee also clamp!, min, max.\n\ncompat: Julia 1.3\nmissing as the first argument requires at least Julia 1.3.\n\nExamples\n\njulia> clamp.([pi, 1.0, big(10)], 2.0, 9.0)\n3-element Vector{BigFloat}:\n 3.141592653589793238462643383279502884197169399375105820974944592307816406286198\n 2.0\n 9.0\n\njulia> clamp.([11, 8, 5], 10, 6)  # an example where lo > hi\n3-element Vector{Int64}:\n  6\n  6\n 10\n\n\n\n\n\n"},{"title":"Base.clamp!","page":"Mathematics","location":"base/math.html#Base.clamp!","category":"function","text":"clamp!(array::AbstractArray, lo, hi)\n\nRestrict values in array to the specified range, in-place.\nSee also clamp.\n\ncompat: Julia 1.3\nmissing entries in array require at least Julia 1.3.\n\nExamples\n\njulia> row = collect(-4:4)';\n\njulia> clamp!(row, 0, Inf)\n1×9 adjoint(::Vector{Int64}) with eltype Int64:\n 0  0  0  0  0  1  2  3  4\n\njulia> clamp.((-4:4)', 0, Inf)\n1×9 Matrix{Float64}:\n 0.0  0.0  0.0  0.0  0.0  1.0  2.0  3.0  4.0\n\n\n\n\n\n"},{"title":"Base.Checked","page":"Mathematics","location":"base/math.html#Base.Checked","category":"module","text":"Checked\n\nThe Checked module provides arithmetic functions for the built-in signed and unsigned\nInteger types which throw an error when an overflow occurs. They are named like checked_sub,\nchecked_div, etc. In addition, add_with_overflow, sub_with_overflow, mul_with_overflow\nreturn both the unchecked results and a boolean value denoting the presence of an overflow.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_abs","page":"Mathematics","location":"base/math.html#Base.Checked.checked_abs","category":"function","text":"Base.checked_abs(x)\n\nCalculates abs(x), checking for overflow errors where applicable.\nFor example, standard two's complement signed integers (e.g. Int)\ncannot represent abs(typemin(Int)), thus leading to an overflow.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_neg","page":"Mathematics","location":"base/math.html#Base.Checked.checked_neg","category":"function","text":"Base.checked_neg(x)\n\nCalculates -x, checking for overflow errors where applicable. For\nexample, standard two's complement signed integers (e.g. Int) cannot\nrepresent -typemin(Int), thus leading to an overflow.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_add","page":"Mathematics","location":"base/math.html#Base.Checked.checked_add","category":"function","text":"Base.checked_add(x, y)\n\nCalculates x+y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_sub","page":"Mathematics","location":"base/math.html#Base.Checked.checked_sub","category":"function","text":"Base.checked_sub(x, y)\n\nCalculates x-y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_mul","page":"Mathematics","location":"base/math.html#Base.Checked.checked_mul","category":"function","text":"Base.checked_mul(x, y)\n\nCalculates x*y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_div","page":"Mathematics","location":"base/math.html#Base.Checked.checked_div","category":"function","text":"Base.checked_div(x, y)\n\nCalculates div(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_rem","page":"Mathematics","location":"base/math.html#Base.Checked.checked_rem","category":"function","text":"Base.checked_rem(x, y)\n\nCalculates x%y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_fld","page":"Mathematics","location":"base/math.html#Base.Checked.checked_fld","category":"function","text":"Base.checked_fld(x, y)\n\nCalculates fld(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_mod","page":"Mathematics","location":"base/math.html#Base.Checked.checked_mod","category":"function","text":"Base.checked_mod(x, y)\n\nCalculates mod(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_cld","page":"Mathematics","location":"base/math.html#Base.Checked.checked_cld","category":"function","text":"Base.checked_cld(x, y)\n\nCalculates cld(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.checked_pow","page":"Mathematics","location":"base/math.html#Base.Checked.checked_pow","category":"function","text":"Base.checked_pow(x, y)\n\nCalculates ^(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n"},{"title":"Base.Checked.add_with_overflow","page":"Mathematics","location":"base/math.html#Base.Checked.add_with_overflow","category":"function","text":"Base.add_with_overflow(x, y) -> (r, f)\n\nCalculates r = x+y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n"},{"title":"Base.Checked.sub_with_overflow","page":"Mathematics","location":"base/math.html#Base.Checked.sub_with_overflow","category":"function","text":"Base.sub_with_overflow(x, y) -> (r, f)\n\nCalculates r = x-y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n"},{"title":"Base.Checked.mul_with_overflow","page":"Mathematics","location":"base/math.html#Base.Checked.mul_with_overflow","category":"function","text":"Base.mul_with_overflow(x, y) -> (r, f)\n\nCalculates r = x*y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n"},{"title":"Base.abs","page":"Mathematics","location":"base/math.html#Base.abs","category":"function","text":"abs(x)\n\nThe absolute value of x.\n\nWhen abs is applied to signed integers, overflow may occur,\nresulting in the return of a negative value. This overflow occurs only\nwhen abs is applied to the minimum representable value of a signed\ninteger. That is, when x == typemin(typeof(x)), abs(x) == x < 0,\nnot -x as might be expected.\n\nSee also: abs2, unsigned, sign.\n\nExamples\n\njulia> abs(-3)\n3\n\njulia> abs(1 + im)\n1.4142135623730951\n\njulia> abs.(Int8[-128 -127 -126 0 126 127])  # overflow at typemin(Int8)\n1×6 Matrix{Int8}:\n -128  127  126  0  126  127\n\njulia> maximum(abs, [1, -2, 3, -4])\n4\n\n\n\n\n\n"},{"title":"Base.abs2","page":"Mathematics","location":"base/math.html#Base.abs2","category":"function","text":"abs2(x)\n\nSquared absolute value of x.\n\nThis can be faster than abs(x)^2, especially for complex\nnumbers where abs(x) requires a square root via hypot.\n\nSee also abs, conj, real.\n\nExamples\n\njulia> abs2(-3)\n9\n\njulia> abs2(3.0 + 4.0im)\n25.0\n\njulia> sum(abs2, [1+2im, 3+4im])  # LinearAlgebra.norm(x)^2\n30\n\n\n\n\n\n"},{"title":"Base.uabs","page":"Mathematics","location":"base/math.html#Base.uabs","category":"function","text":"Base.uabs(x::Integer)\n\nReturn the absolute value of x, possibly returning a different type should the\noperation be susceptible to overflow. This typically arises when x is a two's complement\nsigned integer, so that abs(typemin(x)) == typemin(x) < 0, in which case the result of\nuabs(x) will be an unsigned integer of the same size.\n\n\n\n\n\n"},{"title":"Base.copysign","page":"Mathematics","location":"base/math.html#Base.copysign","category":"function","text":"copysign(x, y) -> z\n\nReturn z which has the magnitude of x and the same sign as y.\n\nExamples\n\njulia> copysign(1, -2)\n-1\n\njulia> copysign(-1, 2)\n1\n\n\n\n\n\n"},{"title":"Base.sign","page":"Mathematics","location":"base/math.html#Base.sign","category":"function","text":"sign(x)\n\nReturn zero if x==0 and xx otherwise (i.e., ±1 for real x).\n\nSee also signbit, zero, copysign, flipsign.\n\nExamples\n\njulia> sign(-4.0)\n-1.0\n\njulia> sign(99)\n1\n\njulia> sign(-0.0)\n-0.0\n\njulia> sign(0 + im)\n0.0 + 1.0im\n\n\n\n\n\n"},{"title":"Base.signbit","page":"Mathematics","location":"base/math.html#Base.signbit","category":"function","text":"signbit(x)\n\nReturn true if the value of the sign of x is negative, otherwise false.\n\nSee also sign and copysign.\n\nExamples\n\njulia> signbit(-4)\ntrue\n\njulia> signbit(5)\nfalse\n\njulia> signbit(5.5)\nfalse\n\njulia> signbit(-4.1)\ntrue\n\n\n\n\n\n"},{"title":"Base.flipsign","page":"Mathematics","location":"base/math.html#Base.flipsign","category":"function","text":"flipsign(x, y)\n\nReturn x with its sign flipped if y is negative. For example abs(x) = flipsign(x,x).\n\nExamples\n\njulia> flipsign(5, 3)\n5\n\njulia> flipsign(5, -3)\n-5\n\n\n\n\n\n"},{"title":"Base.sqrt","page":"Mathematics","location":"base/math.html#Base.sqrt-Tuple{Number}","category":"method","text":"sqrt(x)\n\nReturn sqrtx.\n\nThrow a DomainError for negative Real arguments.\nUse Complex negative arguments instead to obtain a Complex result.\n\nThe prefix operator √ is equivalent to sqrt.\n\nnote: Branch cut\nsqrt has a branch cut along the negative real axis; -0.0im is taken\nto be below the axis.\n\nSee also: hypot.\n\nExamples\n\njulia> sqrt(big(81))\n9.0\n\njulia> sqrt(big(-81))\nERROR: DomainError with -81.0:\nNaN result for non-NaN input.\nStacktrace:\n [1] sqrt(::BigFloat) at ./mpfr.jl:501\n[...]\n\njulia> sqrt(big(complex(-81)))\n0.0 + 9.0im\n\njulia> sqrt(-81 - 0.0im)  # -0.0im is below the branch cut\n0.0 - 9.0im\n\njulia> .√(1:4)\n4-element Vector{Float64}:\n 1.0\n 1.4142135623730951\n 1.7320508075688772\n 2.0\n\n\n\n\n\n"},{"title":"Base.isqrt","page":"Mathematics","location":"base/math.html#Base.isqrt","category":"function","text":"isqrt(n::Integer)\n\nInteger square root: the largest integer m such that m*m <= n.\n\njulia> isqrt(5)\n2\n\n\n\n\n\n"},{"title":"Base.Math.cbrt","page":"Mathematics","location":"base/math.html#Base.Math.cbrt-Tuple{AbstractFloat}","category":"method","text":"cbrt(x::Real)\n\nReturn the cube root of x, i.e. x^13. Negative values are accepted\n(returning the negative real root when x  0).\n\nThe prefix operator ∛ is equivalent to cbrt.\n\nExamples\n\njulia> cbrt(big(27))\n3.0\n\njulia> cbrt(big(-27))\n-3.0\n\n\n\n\n\n"},{"title":"Base.Math.fourthroot","page":"Mathematics","location":"base/math.html#Base.Math.fourthroot-Tuple{Number}","category":"method","text":"fourthroot(x)\n\nReturn the fourth root of x by applying sqrt twice successively.\n\n\n\n\n\n"},{"title":"Base.real","page":"Mathematics","location":"base/math.html#Base.real","category":"function","text":"real(A::AbstractArray)\n\nReturn an array containing the real part of each entry in array A.\n\nEquivalent to real.(A), except that when eltype(A) <: Real\nA is returned without copying, and that when A has zero dimensions,\na 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> real([1, 2im, 3 + 4im])\n3-element Vector{Int64}:\n 1\n 0\n 3\n\njulia> real(fill(2 - im))\n0-dimensional Array{Int64, 0}:\n2\n\n\n\n\n\nreal(T::Type)\n\nReturn the type that represents the real part of a value of type T.\ne.g: for T == Complex{R}, returns R.\nEquivalent to typeof(real(zero(T))).\n\nExamples\n\njulia> real(Complex{Int})\nInt64\n\njulia> real(Float64)\nFloat64\n\n\n\n\n\nreal(z)\n\nReturn the real part of the complex number z.\n\nSee also: imag, reim, complex, isreal, Real.\n\nExamples\n\njulia> real(1 + 3im)\n1\n\n\n\n\n\n"},{"title":"Base.imag","page":"Mathematics","location":"base/math.html#Base.imag","category":"function","text":"imag(A::AbstractArray)\n\nReturn an array containing the imaginary part of each entry in array A.\n\nEquivalent to imag.(A), except that when A has zero dimensions,\na 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> imag([1, 2im, 3 + 4im])\n3-element Vector{Int64}:\n 0\n 2\n 4\n\njulia> imag(fill(2 - im))\n0-dimensional Array{Int64, 0}:\n-1\n\n\n\n\n\nimag(z)\n\nReturn the imaginary part of the complex number z.\n\nSee also: conj, reim, adjoint, angle.\n\nExamples\n\njulia> imag(1 + 3im)\n3\n\n\n\n\n\n"},{"title":"Base.reim","page":"Mathematics","location":"base/math.html#Base.reim","category":"function","text":"reim(A::AbstractArray)\n\nReturn a tuple of two arrays containing respectively the real and the imaginary\npart of each entry in A.\n\nEquivalent to (real.(A), imag.(A)), except that when eltype(A) <: Real\nA is returned without copying to represent the real part, and that when A has\nzero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> reim([1, 2im, 3 + 4im])\n([1, 0, 3], [0, 2, 4])\n\njulia> reim(fill(2 - im))\n(fill(2), fill(-1))\n\n\n\n\n\nreim(z)\n\nReturn a tuple of the real and imaginary parts of the complex number z.\n\nExamples\n\njulia> reim(1 + 3im)\n(1, 3)\n\n\n\n\n\n"},{"title":"Base.conj","page":"Mathematics","location":"base/math.html#Base.conj","category":"function","text":"conj(A::AbstractArray)\n\nReturn an array containing the complex conjugate of each entry in array A.\n\nEquivalent to conj.(A), except that when eltype(A) <: Real\nA is returned without copying, and that when A has zero dimensions,\na 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> conj([1, 2im, 3 + 4im])\n3-element Vector{Complex{Int64}}:\n 1 + 0im\n 0 - 2im\n 3 - 4im\n\njulia> conj(fill(2 - im))\n0-dimensional Array{Complex{Int64}, 0}:\n2 + 1im\n\n\n\n\n\nconj(z)\n\nCompute the complex conjugate of a complex number z.\n\nSee also: angle, adjoint.\n\nExamples\n\njulia> conj(1 + 3im)\n1 - 3im\n\n\n\n\n\n"},{"title":"Base.angle","page":"Mathematics","location":"base/math.html#Base.angle","category":"function","text":"angle(z)\n\nCompute the phase angle in radians of a complex number z.\n\nReturns a number -pi ≤ angle(z) ≤ pi, and is thus discontinuous\nalong the negative real axis.\n\nSee also: atan, cis, rad2deg.\n\nExamples\n\njulia> rad2deg(angle(1 + im))\n45.0\n\njulia> rad2deg(angle(1 - im))\n-45.0\n\njulia> rad2deg(angle(-1 + 1e-20im))\n180.0\n\njulia> rad2deg(angle(-1 - 1e-20im))\n-180.0\n\n\n\n\n\n"},{"title":"Base.cis","page":"Mathematics","location":"base/math.html#Base.cis","category":"function","text":"cis(x)\n\nMore efficient method for exp(im*x) by using Euler's formula: cos(x) + i sin(x) = exp(i x).\n\nSee also cispi, sincos, exp, angle.\n\nExamples\n\njulia> cis(π) ≈ -1\ntrue\n\n\n\n\n\n"},{"title":"Base.cispi","page":"Mathematics","location":"base/math.html#Base.cispi","category":"function","text":"cispi(x)\n\nMore accurate method for cis(pi*x) (especially for large x).\n\nSee also cis, sincospi, exp, angle.\n\nExamples\n\njulia> cispi(10000)\n1.0 + 0.0im\n\njulia> cispi(0.25 + 1im)\n0.030556854645954562 + 0.03055685464595456im\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n"},{"title":"Base.binomial","page":"Mathematics","location":"base/math.html#Base.binomial","category":"function","text":"binomial(x::Number, k::Integer)\n\nThe generalized binomial coefficient, defined for k ≥ 0 by\nthe polynomial\n\nfrac1k prod_j=0^k-1 (x - j)\n\nWhen k < 0 it returns zero.\n\nFor the case of integer x, this is equivalent to the ordinary\ninteger binomial coefficient\n\nbinomnk = fracnk (n-k)\n\nFurther generalizations to non-integer k are mathematically possible, but\ninvolve the Gamma function and/or the beta function, which are\nnot provided by the Julia standard library but are available\nin external packages such as SpecialFunctions.jl.\n\nExternal links\n\nBinomial coefficient on Wikipedia.\n\n\n\n\n\nbinomial(n::Integer, k::Integer)\n\nThe binomial coefficient binomnk, being the coefficient of the kth term in\nthe polynomial expansion of (1+x)^n.\n\nIf n is non-negative, then it is the number of ways to choose k out of n items:\n\nbinomnk = fracnk (n-k)\n\nwhere n is the factorial function.\n\nIf n is negative, then it is defined in terms of the identity\n\nbinomnk = (-1)^k binomk-n-1k\n\nSee also factorial.\n\nExamples\n\njulia> binomial(5, 3)\n10\n\njulia> factorial(5) ÷ (factorial(5-3) * factorial(3))\n10\n\njulia> binomial(-5, 3)\n-35\n\nExternal links\n\nBinomial coefficient on Wikipedia.\n\n\n\n\n\n"},{"title":"Base.factorial","page":"Mathematics","location":"base/math.html#Base.factorial","category":"function","text":"factorial(n::Integer)\n\nFactorial of n. If n is an Integer, the factorial is computed as an\ninteger (promoted to at least 64 bits). Note that this may overflow if n is not small,\nbut you can use factorial(big(n)) to compute the result exactly in arbitrary precision.\n\nSee also binomial.\n\nExamples\n\njulia> factorial(6)\n720\n\njulia> factorial(21)\nERROR: OverflowError: 21 is too large to look up in the table; consider using `factorial(big(21))` instead\nStacktrace:\n[...]\n\njulia> factorial(big(21))\n51090942171709440000\n\nExternal links\n\nFactorial on Wikipedia.\n\n\n\n\n\n"},{"title":"Base.gcd","page":"Mathematics","location":"base/math.html#Base.gcd","category":"function","text":"gcd(x, y...)\n\nGreatest common (positive) divisor (or zero if all arguments are zero).\nThe arguments may be integer and rational numbers.\n\na is a divisor of b if there exists an integer m such\nthat ma=b.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> gcd(6, 9)\n3\n\njulia> gcd(6, -9)\n3\n\njulia> gcd(6, 0)\n6\n\njulia> gcd(0, 0)\n0\n\njulia> gcd(1//3, 2//3)\n1//3\n\njulia> gcd(1//3, -2//3)\n1//3\n\njulia> gcd(1//3, 2)\n1//3\n\njulia> gcd(0, 0, 10, 15)\n5\n\n\n\n\n\n"},{"title":"Base.lcm","page":"Mathematics","location":"base/math.html#Base.lcm","category":"function","text":"lcm(x, y...)\n\nLeast common (positive) multiple (or zero if any argument is zero).\nThe arguments may be integer and rational numbers.\n\na is a multiple of b if there exists an integer m such\nthat a=mb.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> lcm(2, 3)\n6\n\njulia> lcm(-2, 3)\n6\n\njulia> lcm(0, 3)\n0\n\njulia> lcm(0, 0)\n0\n\njulia> lcm(1//3, 2//3)\n2//3\n\njulia> lcm(1//3, -2//3)\n2//3\n\njulia> lcm(1//3, 2)\n2//1\n\njulia> lcm(1, 3, 5, 7)\n105\n\n\n\n\n\n"},{"title":"Base.gcdx","page":"Mathematics","location":"base/math.html#Base.gcdx","category":"function","text":"gcdx(a, b...)\n\nCompute the greatest common (positive) divisor of a and b and their Bézout\ncoefficients, i.e. the integer coefficients u and v that satisfy\nu*a + v*b = d = gcd(a b). gcdx(a b) returns (d u v).\n\nFor more arguments than two, i.e., gcdx(a, b, c, ...) the Bézout coefficients are computed\nrecursively, returning a solution (d, u, v, w, ...) to\nu*a + v*b + w*c +  = d = gcd(a b c ).\n\nThe arguments may be integer and rational numbers.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\ncompat: Julia 1.12\nMore or fewer arguments than two require Julia 1.12 or later.\n\nExamples\n\njulia> gcdx(12, 42)\n(6, -3, 1)\n\njulia> gcdx(240, 46)\n(2, -9, 47)\n\njulia> gcdx(15, 12, 20)\n(1, 7, -7, -1)\n\nnote: Note\nBézout coefficients are not uniquely defined. gcdx returns the minimal\nBézout coefficients that are computed by the extended Euclidean algorithm.\n(Ref: D. Knuth, TAoCP, 2/e, p. 325, Algorithm X.)\nFor signed integers, these coefficients u and v are minimal in\nthe sense that u  bd and v  ad. Furthermore,\nthe signs of u and v are chosen so that d is positive.\nFor unsigned integers, the coefficients u and v might be near\ntheir typemax, and the identity then holds only via the unsigned\nintegers' modulo arithmetic.\n\n\n\n\n\n"},{"title":"Base.ispow2","page":"Mathematics","location":"base/math.html#Base.ispow2","category":"function","text":"ispow2(n::Number)::Bool\n\nTest whether n is an integer power of two.\n\nSee also count_ones, prevpow, nextpow.\n\nExamples\n\njulia> ispow2(4)\ntrue\n\njulia> ispow2(5)\nfalse\n\njulia> ispow2(4.5)\nfalse\n\njulia> ispow2(0.25)\ntrue\n\njulia> ispow2(1//8)\ntrue\n\ncompat: Julia 1.6\nSupport for non-Integer arguments was added in Julia 1.6.\n\n\n\n\n\n"},{"title":"Base.nextpow","page":"Mathematics","location":"base/math.html#Base.nextpow","category":"function","text":"nextpow(a, x)\n\nThe smallest a^n not less than x, where n is a non-negative integer. a must be\ngreater than 1, and x must be greater than 0.\n\nSee also prevpow.\n\nExamples\n\njulia> nextpow(2, 7)\n8\n\njulia> nextpow(2, 9)\n16\n\njulia> nextpow(5, 20)\n25\n\njulia> nextpow(4, 16)\n16\n\n\n\n\n\n"},{"title":"Base.prevpow","page":"Mathematics","location":"base/math.html#Base.prevpow","category":"function","text":"prevpow(a, x)\n\nThe largest a^n not greater than x, where n is a non-negative integer.\na must be greater than 1, and x must not be less than 1.\n\nSee also nextpow, isqrt.\n\nExamples\n\njulia> prevpow(2, 7)\n4\n\njulia> prevpow(2, 9)\n8\n\njulia> prevpow(5, 20)\n5\n\njulia> prevpow(4, 16)\n16\n\n\n\n\n\n"},{"title":"Base.nextprod","page":"Mathematics","location":"base/math.html#Base.nextprod","category":"function","text":"nextprod(factors::Union{Tuple,AbstractVector}, n)\n\nNext integer greater than or equal to n that can be written as prod k_i^p_i for integers\np_1, p_2, etcetera, for factors k_i in factors.\n\nExamples\n\njulia> nextprod((2, 3), 105)\n108\n\njulia> 2^2 * 3^3\n108\n\ncompat: Julia 1.6\nThe method that accepts a tuple requires Julia 1.6 or later.\n\n\n\n\n\n"},{"title":"Base.invmod","page":"Mathematics","location":"base/math.html#Base.invmod","category":"function","text":"invmod(n::Integer, T) where {T <: Base.BitInteger}\ninvmod(n::T) where {T <: Base.BitInteger}\n\nCompute the modular inverse of n in the integer ring of type T, i.e. modulo\n2^N where N = 8*sizeof(T) (e.g. N = 32 for Int32). In other words, these\nmethods satisfy the following identities:\n\nn * invmod(n) == 1\n(n * invmod(n, T)) % T == 1\n(n % T) * invmod(n, T) == 1\n\nNote that * here is modular multiplication in the integer ring, T.  This will\nthrow an error if n is even, because then it is not relatively prime with 2^N\nand thus has no such inverse.\n\nSpecifying the modulus implied by an integer type as an explicit value is often\ninconvenient since the modulus is by definition too big to be represented by the\ntype.\n\nThe modular inverse is computed much more efficiently than the general case\nusing the algorithm described in [An Improved Integer Modular Multiplicative\nInverse (modulo 2^w) by Jeffrey Hurchalla](https://arxiv.org/abs/2204.04342).\n\ncompat: Julia 1.11\nThe invmod(n) and invmod(n, T) methods require Julia 1.11 or later.\n\n\n\n\n\ninvmod(n::Integer, m::Integer)\n\nTake the inverse of n modulo m: y such that n y = 1 pmod m,\nand div(ym) = 0. This will throw an error if m = 0, or if\ngcd(nm) neq 1.\n\nExamples\n\njulia> invmod(2, 5)\n3\n\njulia> invmod(2, 3)\n2\n\njulia> invmod(5, 6)\n5\n\n\n\n\n\n"},{"title":"Base.powermod","page":"Mathematics","location":"base/math.html#Base.powermod","category":"function","text":"powermod(x::Integer, p::Integer, m)\n\nCompute x^p pmod m.\n\nExamples\n\njulia> powermod(2, 6, 5)\n4\n\njulia> mod(2^6, 5)\n4\n\njulia> powermod(5, 2, 20)\n5\n\njulia> powermod(5, 2, 19)\n6\n\njulia> powermod(5, 3, 19)\n11\n\n\n\n\n\n"},{"title":"Base.add_sum","page":"Mathematics","location":"base/math.html#Base.add_sum","category":"function","text":"Base.add_sum(x, y)\n\nThe reduction operator used in sum. The main difference from + is that small\nintegers are promoted to Int/UInt.\n\n\n\n\n\n"},{"title":"Base.widemul","page":"Mathematics","location":"base/math.html#Base.widemul","category":"function","text":"widemul(x, y)\n\nMultiply x and y, giving the result as a larger type.\n\nSee also promote, Base.add_sum.\n\nExamples\n\njulia> widemul(Float32(3.0), 4.0) isa BigFloat\ntrue\n\njulia> typemax(Int8) * typemax(Int8)\n1\n\njulia> widemul(typemax(Int8), typemax(Int8))  # == 127^2\n16129\n\n\n\n\n\n"},{"title":"Base.Math.evalpoly","page":"Mathematics","location":"base/math.html#Base.Math.evalpoly","category":"function","text":"evalpoly(x, p)\n\nEvaluate the polynomial sum_k x^k-1 pk for the coefficients p[1], p[2], ...;\nthat is, the coefficients are given in ascending order by power of x.\nLoops are unrolled at compile time if the number of coefficients is statically known, i.e.\nwhen p is a Tuple.\nThis function generates efficient code using Horner's method if x is real, or using\na Goertzel-like [DK62] algorithm if x is complex.\n\n[DK62]: Donald Knuth, Art of Computer Programming, Volume 2: Seminumerical Algorithms, Sec. 4.6.4.\n\ncompat: Julia 1.4\nThis function requires Julia 1.4 or later.\n\nExamples\n\njulia> evalpoly(2, (1, 2, 3))\n17\n\n\n\n\n\n"},{"title":"Base.Math.@evalpoly","page":"Mathematics","location":"base/math.html#Base.Math.@evalpoly","category":"macro","text":"@evalpoly(z, c...)\n\nEvaluate the polynomial sum_k z^k-1 ck for the coefficients c[1], c[2], ...;\nthat is, the coefficients are given in ascending order by power of z.  This macro expands\nto efficient inline code that uses either Horner's method or, for complex z, a more\nefficient Goertzel-like algorithm.\n\nSee also evalpoly.\n\nExamples\n\njulia> @evalpoly(3, 1, 0, 1)\n10\n\njulia> @evalpoly(2, 1, 0, 1)\n5\n\njulia> @evalpoly(2, 1, 1, 1)\n7\n\n\n\n\n\n"},{"title":"Base.FastMath.@fastmath","page":"Mathematics","location":"base/math.html#Base.FastMath.@fastmath","category":"macro","text":"@fastmath expr\n\nExecute a transformed version of the expression, which calls functions that\nmay violate strict IEEE semantics. This allows the fastest possible operation,\nbut results are undefined – be careful when doing this, as it may change numerical\nresults.\n\nThis sets the LLVM Fast-Math flags,\nand corresponds to the -ffast-math option in clang. See [the notes on performance\nannotations](@ref man-performance-annotations) for more details.\n\nExamples\n\njulia> @fastmath 1+2\n3\n\njulia> @fastmath(sin(3))\n0.1411200080598672\n\n\n\n\n\n"},{"title":"External Profiler Support","page":"External Profiler Support","location":"devdocs/external_profilers.html#External-Profiler-Support","category":"section","text":"Julia provides explicit support for some external tracing profilers, enabling you to obtain a high-level overview of the runtime's execution behavior.\n\nThe currently supported profilers are:\n\nTracy\nIntel VTune (ITTAPI)"},{"title":"Adding New Zones","page":"External Profiler Support","location":"devdocs/external_profilers.html#Adding-New-Zones","category":"section","text":""},{"title":"From C/C++ code","page":"External Profiler Support","location":"devdocs/external_profilers.html#From-C/C-code","category":"section","text":"To add new zones, use the JL_TIMING macro. You can find numerous examples throughout the codebase by searching for JL_TIMING. To add a new type of zone\nyou add it to JL_TIMING_OWNERS (and possibly JL_TIMING_EVENTS)."},{"title":"From Julia code","page":"External Profiler Support","location":"devdocs/external_profilers.html#From-Julia-code","category":"section","text":"The Compiler.@zone macro can be used to add a zone from Julia code, it is used as:\n\nCompiler.@zone \"ZONE NAME\" begin\n    ...\nend"},{"title":"Dynamically Enabling and Disabling Zones","page":"External Profiler Support","location":"devdocs/external_profilers.html#Dynamically-Enabling-and-Disabling-Zones","category":"section","text":"The JULIA_TIMING_SUBSYSTEMS environment variable allows you to enable or disable zones for a specific Julia run. For instance, setting the variable to +GC,-INFERENCE will enable the GC zones and disable the INFERENCE\nzones."},{"title":"Tracy Profiler","page":"External Profiler Support","location":"devdocs/external_profilers.html#Tracy-Profiler","category":"section","text":"Tracy  is a flexible profiler that can be optionally integrated with Julia.\n\nA typical Tracy session might look like this:\n\n(Image: Typical Tracy usage)"},{"title":"Building Julia with Tracy","page":"External Profiler Support","location":"devdocs/external_profilers.html#Building-Julia-with-Tracy","category":"section","text":"To enable Tracy integration, build Julia with the extra option WITH_TRACY=1 in the Make.user file."},{"title":"Installing the Tracy Profile Viewer","page":"External Profiler Support","location":"devdocs/external_profilers.html#Installing-the-Tracy-Profile-Viewer","category":"section","text":"The easiest way to obtain the profile viewer is by adding the TracyProfiler_jll package and launching the profiler with:\n\nrun(TracyProfiler_jll.tracy())\n\nnote: Note\nOn macOS, you may want to set the TRACY_DPI_SCALE environment variable to 1.0 if the UI elements in the profiler appear excessively large.\n\nTo run a \"headless\" instance that saves the trace to disk, use\n\nrun(`$(TracyProfiler_jll.capture()) -o mytracefile.tracy`)\n\ninstead.\n\nFor information on using the Tracy UI, refer to the Tracy manual."},{"title":"Profiling Julia with Tracy","page":"External Profiler Support","location":"devdocs/external_profilers.html#Profiling-Julia-with-Tracy","category":"section","text":"A typical workflow for profiling Julia with Tracy involves starting Julia using:\n\nJULIA_WAIT_FOR_TRACY=1 ./julia -e '...'\n\nThe environment variable ensures that Julia waits until it has successfully connected to the Tracy profiler before continuing execution. Afterward, use the Tracy profiler UI, click Connect, and Julia execution should resume and profiling should start."},{"title":"Profiling package precompilation with Tracy","page":"External Profiler Support","location":"devdocs/external_profilers.html#Profiling-package-precompilation-with-Tracy","category":"section","text":"To profile a package precompilation process it is easiest to explicitly call into Base.compilecache with the package you want to precompile:\n\npkg = Base.identify_package(\"SparseArrays\")\nwithenv(\"JULIA_WAIT_FOR_TRACY\" => 1, \"TRACY_PORT\" => 9001) do\n    Base.compilecache(pkg)\nend\n\nHere, we use a custom port for tracy which makes it easier to find the correct client in the Tracy UI to connect to."},{"title":"Adding metadata to zones","page":"External Profiler Support","location":"devdocs/external_profilers.html#Adding-metadata-to-zones","category":"section","text":"The various jl_timing_show_* and jl_timing_printf functions can be used to attach a string (or strings) to a zone. For example, the trace zone for inference shows the method instance that is being inferred.\n\nThe TracyCZoneColor function can be used to set the color of a certain zone. Search through the codebase to see how it is used."},{"title":"Viewing Tracy files in your browser","page":"External Profiler Support","location":"devdocs/external_profilers.html#Viewing-Tracy-files-in-your-browser","category":"section","text":"Visit https://topolarity.github.io/trace-viewer/ for an (experimental) web viewer for Tracy traces.\n\nYou can open a local .tracy file or provide a URL from the web (e.g. a file in a Github repo). If you load a trace file from the web, you can also share the page URL directly with others, enabling them to view the same trace."},{"title":"Enabling stack trace samples","page":"External Profiler Support","location":"devdocs/external_profilers.html#Enabling-stack-trace-samples","category":"section","text":"To enable call stack sampling in Tracy, build Julia with these options in your Make.user file:\n\nWITH_TRACY := 1\nWITH_TRACY_CALLSTACKS := 1\nUSE_BINARYBUILDER_LIBTRACYCLIENT := 0\n\nYou may also need to run make -C deps clean-libtracyclient to force a re-build of Tracy.\n\nThis feature has a significant impact on trace size and profiling overhead, so it is recommended to leave call stack sampling off when possible, especially if you intend to share your trace files online.\n\nNote that the Julia JIT runtime does not yet have integration for Tracy's symbolification, so Julia functions will typically be unknown in these stack traces."},{"title":"Intel VTune (ITTAPI) Profiler","page":"External Profiler Support","location":"devdocs/external_profilers.html#Intel-VTune-(ITTAPI)-Profiler","category":"section","text":"This section is yet to be written."},{"title":"FreeBSD","page":"FreeBSD","location":"devdocs/build/freebsd.html#FreeBSD","category":"section","text":"Clang is the default compiler on FreeBSD 11.0-RELEASE and above.\nThe remaining build tools are available from the Ports Collection, and can be installed using\npkg install git gcc gmake cmake pkgconf.\nTo build Julia, simply run gmake.\n(Note that gmake must be used rather than make, since make on FreeBSD corresponds to the incompatible BSD Make rather than GNU Make.)\n\nAs mentioned above, it is important to note that the USE_SYSTEM_* flags should be used with caution on FreeBSD.\nThis is because many system libraries, and even libraries from the Ports Collection, link to the system's libgcc_s.so.1,\nor to another library which links to the system libgcc_s.\nThis library declares its GCC version to be 4.6, which is too old to build Julia, and conflicts with other libraries when linking.\nThus it is highly recommended to simply allow Julia to build all of its dependencies.\nIf you do choose to use the USE_SYSTEM_* flags, note that /usr/local is not on the compiler path by default, so you may need\nto add LDFLAGS=-L/usr/local/lib and CPPFLAGS=-I/usr/local/include to your Make.user, though doing so may interfere with\nother dependencies.\n\nNote that the x86 architecture does not support threading due to lack of compiler runtime library support, so you may need to\nset JULIA_THREADS=0 in your Make.user if you're on a 32-bit system."},{"title":"Julia ASTs","page":"Julia ASTs","location":"devdocs/ast.html#Julia-ASTs","category":"section","text":"Julia has two representations of code. First there is a surface syntax AST returned by the parser\n(e.g. the Meta.parse function), and manipulated by macros. It is a structured representation\nof code as it is written, constructed by julia-parser.scm from a character stream. Next there\nis a lowered form, or IR (intermediate representation), which is used by type inference and code\ngeneration. In the lowered form there are fewer types of nodes, all macros are expanded, and all\ncontrol flow is converted to explicit branches and sequences of statements. The lowered form is\nconstructed by julia-syntax.scm.\n\nFirst we will focus on the AST, since it is needed to write macros."},{"title":"Surface syntax AST","page":"Julia ASTs","location":"devdocs/ast.html#Surface-syntax-AST","category":"section","text":"Front end ASTs consist almost entirely of Exprs and atoms (e.g. symbols, numbers).\nThere is generally a different expression head for each visually distinct syntactic form.\nExamples will be given in s-expression syntax.\nEach parenthesized list corresponds to an Expr, where the first element is the head.\nFor example (call f x) corresponds to Expr(:call, :f, :x) in Julia."},{"title":"Calls","page":"Julia ASTs","location":"devdocs/ast.html#Calls","category":"section","text":"Input AST\nf(x) (call f x)\nf(x, y=1, z=2) (call f x (kw y 1) (kw z 2))\nf(x; y=1) (call f (parameters (kw y 1)) x)\nf(x...) (call f (... x))\n\ndo syntax:\n\nf(x) do a,b\n    body\nend\n\nparses as (do (call f x) (-> (tuple a b) (block body)))."},{"title":"Operators","page":"Julia ASTs","location":"devdocs/ast.html#Operators","category":"section","text":"Most uses of operators are just function calls, so they are parsed with the head call. However\nsome operators are special forms (not necessarily function calls), and in those cases the operator\nitself is the expression head. In julia-parser.scm these are referred to as \"syntactic operators\".\nSome operators (+ and *) use N-ary parsing; chained calls are parsed as a single N-argument\ncall. Finally, chains of comparisons have their own special expression structure.\n\nInput AST\nx+y (call + x y)\na+b+c+d (call + a b c d)\n2x (call * 2 x)\na&&b (&& a b)\nx += 1 (+= x 1)\na ? 1 : 2 (if a 1 2)\na,b (tuple a b)\na==b (call == a b)\n1<i<=n (comparison 1 < i <= n)\na.b (. a (quote b))\na.(b) (. a (tuple b))"},{"title":"Bracketed forms","page":"Julia ASTs","location":"devdocs/ast.html#Bracketed-forms","category":"section","text":"Input AST\na[i] (ref a i)\nt[i;j] (typed_vcat t i j)\nt[i j] (typed_hcat t i j)\nt[a b; c d] (typed_vcat t (row a b) (row c d))\nt[a b;;; c d] (typed_ncat t 3 (row a b) (row c d))\na{b} (curly a b)\na{b;c} (curly a (parameters c) b)\n[x] (vect x)\n[x,y] (vect x y)\n[x;y] (vcat x y)\n[x y] (hcat x y)\n[x y; z t] (vcat (row x y) (row z t))\n[x;y;; z;t;;;] (ncat 3 (nrow 2 (nrow 1 x y) (nrow 1 z t)))\n[x for y in z, a in b] (comprehension (generator x (= y z) (= a b)))\nT[x for y in z] (typed_comprehension T (generator x (= y z)))\n(a, b, c) (tuple a b c)\n(a; b; c) (block a b c)"},{"title":"Macros","page":"Julia ASTs","location":"devdocs/ast.html#Macros","category":"section","text":"Input AST\n@m x y (macrocall @m (line) x y)\nBase.@m x y (macrocall (. Base (quote @m)) (line) x y)\n@Base.m x y (macrocall (. Base (quote @m)) (line) x y)"},{"title":"Strings","page":"Julia ASTs","location":"devdocs/ast.html#Strings","category":"section","text":"Input AST\n\"a\" \"a\"\nx\"y\" (macrocall @x_str (line) \"y\")\nx\"y\"z (macrocall @x_str (line) \"y\" \"z\")\n\"x = $x\" (string \"x = \" x)\n`a b c` (macrocall @cmd (line) \"a b c\")\n\nDoc string syntax:\n\n\"some docs\"\nf(x) = x\n\nparses as (macrocall (|.| Core '@doc) (line) \"some docs\" (= (call f x) (block x)))."},{"title":"Imports and such","page":"Julia ASTs","location":"devdocs/ast.html#Imports-and-such","category":"section","text":"Input AST\nimport a (import (. a))\nimport a.b.c (import (. a b c))\nimport ...a (import (. . . . a))\nimport a.b, c.d (import (. a b) (. c d))\nimport Base: x (import (: (. Base) (. x)))\nimport Base: x, y (import (: (. Base) (. x) (. y)))\nexport a, b (export a b)\npublic a, b (public a b)\n\nusing has the same representation as import, but with expression head :using\ninstead of :import.\n\nTo programmatically create a public statement, you can use Expr(:public, :a, :b) or,\ncloser to regular code, Meta.parse(\"public a, b\"). This approach is necessary due to\ncurrent limitations on public. The public keyword is only\nrecognized at the syntactic top level within a file (parse_stmts) or module. This\nrestriction was implemented to prevent breaking existing code that used public as an\nidentifier when it was introduced in Julia 1.11."},{"title":"Numbers","page":"Julia ASTs","location":"devdocs/ast.html#Numbers","category":"section","text":"Julia supports more number types than many scheme implementations, so not all numbers are represented\ndirectly as scheme numbers in the AST.\n\nInput AST\n11111111111111111111 (macrocall @int128_str nothing \"11111111111111111111\")\n0xfffffffffffffffff (macrocall @uint128_str nothing \"0xfffffffffffffffff\")\n1111...many digits... (macrocall @big_str nothing \"1111....\")"},{"title":"Block forms","page":"Julia ASTs","location":"devdocs/ast.html#Block-forms","category":"section","text":"A block of statements is parsed as (block stmt1 stmt2 ...).\n\nIf statement:\n\nif a\n    b\nelseif c\n    d\nelse\n    e\nend\n\nparses as:\n\n(if a (block (line 2) b)\n    (elseif (block (line 3) c) (block (line 4) d)\n            (block (line 6) e)))\n\nA while loop parses as (while condition body).\n\nA for loop parses as (for (= var iter) body). If there is more than one iteration specification,\nthey are parsed as a block: (for (block (= v1 iter1) (= v2 iter2)) body).\n\nbreak and continue are parsed as 0-argument expressions (break) and (continue).\n\nlet is parsed as (let (= var val) body) or (let (block (= var1 val1) (= var2 val2) ...) body),\nlike for loops.\n\nA basic function definition is parsed as (function (call f x) body). A more complex example:\n\nfunction f(x::T; k = 1) where T\n    return x+1\nend\n\nparses as:\n\n(function (where (call f (parameters (kw k 1))\n                       (:: x T))\n                 T)\n          (block (line 2) (return (call + x 1))))\n\nType definition:\n\nmutable struct Foo{T<:S}\n    x::T\nend\n\nparses as:\n\n(struct true (curly Foo (<: T S))\n        (block (line 2) (:: x T)))\n\nThe first argument is a boolean telling whether the type is mutable.\n\ntry blocks parse as (try try_block var catch_block finally_block). If no variable is present\nafter catch, var is #f. If there is no finally clause, then the last argument is not present."},{"title":"Quote expressions","page":"Julia ASTs","location":"devdocs/ast.html#Quote-expressions","category":"section","text":"Julia source syntax forms for code quoting (quote and :( )) support interpolation with $.\nIn Lisp terminology, this means they are actually \"backquote\" or \"quasiquote\" forms.\nInternally, there is also a need for code quoting without interpolation.\nIn Julia's scheme code, non-interpolating quote is represented with the expression head inert.\n\ninert expressions are converted to Julia QuoteNode objects.\nThese objects wrap a single value of any type, and when evaluated simply return that value.\n\nA quote expression whose argument is an atom also gets converted to a QuoteNode."},{"title":"Line numbers","page":"Julia ASTs","location":"devdocs/ast.html#Line-numbers","category":"section","text":"Source location information is represented as (line line_num file_name) where the third\ncomponent is optional (and omitted when the current line number, but not file name,\nchanges).\n\nThese expressions are represented as LineNumberNodes in Julia."},{"title":"Macros","page":"Julia ASTs","location":"devdocs/ast.html#Macros-2","category":"section","text":"Macro hygiene is represented through the expression head pair escape and hygienic-scope.\nThe result of a macro expansion is automatically wrapped in (hygienic-scope block module [lno]),\nto represent the result of the new scope. The user can insert (escape block) inside\nto interpolate code from the caller. The lno is the __source__ argument of the macro, if included."},{"title":"Lowered form","page":"Julia ASTs","location":"devdocs/ast.html#Lowered-form","category":"section","text":"Lowered form (IR) is more important to the compiler, since it is used for type inference,\noptimizations like inlining, and code generation. It is also less obvious to the human,\nsince it results from a significant rearrangement of the input syntax.\n\nIn addition to Symbols and some number types, the following data\ntypes exist in lowered form:\n\nExpr\nHas a node type indicated by the head field, and an args field which is a Vector{Any} of\nsubexpressions.\nWhile almost every part of a surface AST is represented by an Expr, the IR uses only a\nlimited number of Exprs, mostly for calls and some top-level-only forms.\nSlotNumber\nIdentifies arguments and local variables by consecutive numbering. It has an\ninteger-valued id field giving the slot index.\nThe types of these slots can be found in the slottypes field of their CodeInfo object.\nArgument\nThe same as SlotNumber, but appears only post-optimization. Indicates that the\nreferenced slot is an argument of the enclosing function.\nCodeInfo\nWraps the IR of a group of statements. Its code field is an array of expressions to execute.\nGotoNode\nUnconditional branch. The argument is the branch target, represented as an index in\nthe code array to jump to.\nGotoIfNot\nConditional branch. If the cond field evaluates to false, goes to the index identified\nby the dest field.\nReturnNode\nReturns its argument (the val field) as the value of the enclosing function.\nIf the val field is undefined, then this represents an unreachable statement.\nQuoteNode\nWraps an arbitrary value to reference as data. For example, the function f() = :a contains a\nQuoteNode whose value field is the symbol a, in order to return the symbol itself instead\nof evaluating it.\nGlobalRef\nRefers to global variable name in module mod.\nSSAValue\nRefers to a consecutively-numbered (starting at 1) static single assignment (SSA) variable inserted\nby the compiler. The number (id) of an SSAValue is the code array index of the expression whose\nvalue it represents.\nNewvarNode\nMarks a point where a variable (slot) is created. This has the effect of resetting a variable to undefined."},{"title":"Expr types","page":"Julia ASTs","location":"devdocs/ast.html#Expr-types","category":"section","text":"These symbols appear in the head field of Exprs in lowered form.\n\ncall\nFunction call (dynamic dispatch). args[1] is the function to call, args[2:end] are the arguments.\ninvoke\nFunction call (static dispatch). args[1] is the MethodInstance to call, args[2:end] are the\narguments (including the function that is being called, at args[2]).\nstatic_parameter\nReference a static parameter by index.\n=\nAssignment. In the IR, the first argument is always a SlotNumber or a GlobalRef.\nmethod\nAdds a method to a generic function and assigns the result if necessary.\nHas a 1-argument form and a 3-argument form. The 1-argument form arises from the syntax function foo end.\nIn the 1-argument form, the argument is a symbol. If this symbol already names a function in the\ncurrent scope, nothing happens. If the symbol is undefined, a new function is created and assigned\nto the identifier specified by the symbol. If the symbol is defined but names a non-function,\nan error is raised. The definition of \"names a function\" is that the binding is constant, and\nrefers to an object of singleton type. The rationale for this is that an instance of a singleton\ntype uniquely identifies the type to add the method to. When the type has fields, it wouldn't\nbe clear whether the method was being added to the instance or its type.\nThe 3-argument form has the following arguments:\nargs[1]\nA function name, or nothing if unknown or unneeded. If a symbol, then the expression\nfirst behaves like the 1-argument form above. This argument is ignored from then on.\nIt can be nothing when methods are added strictly by type, (::T)(x) = x,\nor when a method is being added to an existing function, MyModule.f(x) = x.\nargs[2]\nA SimpleVector of argument type data. args[2][1] is a SimpleVector of the\nargument types, and args[2][2] is a SimpleVector of type variables corresponding\nto the method's static parameters.\nargs[3]\nA CodeInfo of the method itself. For \"out of scope\" method definitions (adding a\nmethod to a function that also has methods defined in different scopes) this is an\nexpression that evaluates to a :lambda expression.\nstruct_type\nA 7-argument expression that defines a new struct:\nargs[1]\nThe name of the struct\nargs[2]\nA call expression that creates a SimpleVector specifying its parameters\nargs[3]\nA call expression that creates a SimpleVector specifying its fieldnames\nargs[4]\nA Symbol, GlobalRef, or Expr specifying the supertype (e.g., :Integer,\nGlobalRef(Core, :Any), or :(Core.apply_type(AbstractArray, T, N)))\nargs[5]\nA call expression that creates a SimpleVector specifying its fieldtypes\nargs[6]\nA Bool, true if mutable\nargs[7]\nThe number of arguments to initialize. This will be the number\nof fields, or the minimum number of fields called by an inner\nconstructor's new statement.\nabstract_type\nA 3-argument expression that defines a new abstract type. The\narguments are the same as arguments 1, 2, and 4 of\nstruct_type expressions.\nprimitive_type\nA 4-argument expression that defines a new primitive type. Arguments 1, 2, and 4\nare the same as struct_type. Argument 3 is the number of bits.\ncompat: Julia 1.5\nstruct_type, abstract_type, and primitive_type were removed in Julia 1.5\nand replaced by calls to new builtins.\nglobal\nDeclares a global binding.\nconst\nDeclares a (global) variable as constant.\nnew\nAllocates a new struct-like object. First argument is the type. The new pseudo-function is lowered\nto this, and the type is always inserted by the compiler. This is very much an internal-only\nfeature, and does no checking. Evaluating arbitrary new expressions can easily segfault.\nsplatnew\nSimilar to new, except field values are passed as a single tuple. Works similarly to\nsplat(new) if new were a first-class function, hence the name.\nisdefined\nExpr(:isdefined, :x) returns a Bool indicating whether x has\nalready been defined in the current scope.\nthe_exception\nYields the caught exception inside a catch block, as returned by jl_current_exception(ct).\nenter\nEnters an exception handler (setjmp). args[1] is the label of the catch block to jump to on\nerror. Yields a token which is consumed by pop_exception.\nleave\nPop exception handlers. args[1] is the number of handlers to pop.\npop_exception\nPop the stack of current exceptions back to the state at the associated enter when leaving a\ncatch block. args[1] contains the token from the associated enter.\ncompat: Julia 1.1\npop_exception is new in Julia 1.1.\ninbounds\nControls turning bounds checks on or off. A stack is maintained; if the first argument of this\nexpression is true or false (true means bounds checks are disabled), it is pushed onto the stack.\nIf the first argument is :pop, the stack is popped.\nboundscheck\nHas the value false if inlined into a section of code marked with @inbounds,\notherwise has the value true.\nloopinfo\nMarks the end of the a loop. Contains metadata that is passed to LowerSimdLoop to either mark\nthe inner loop of @simd expression, or to propagate information to LLVM loop passes.\ncopyast\nPart of the implementation of quasi-quote. The argument is a surface syntax AST that is simply\ncopied recursively and returned at run time.\nmeta\nMetadata. args[1] is typically a symbol specifying the kind of metadata, and the rest of the\narguments are free-form. The following kinds of metadata are commonly used:\n:inline and :noinline: Inlining hints.\nforeigncall\nStatically-computed container for ccall information. The fields are:\nargs[1] : name\nThe expression that'll be parsed for the foreign function.\nargs[2]::Type : RT\nThe (literal) return type, computed statically when the containing method was defined.\nargs[3]::SimpleVector (of Types) : AT\nThe (literal) vector of argument types, computed statically when the containing method was defined.\nargs[4]::Int : nreq\nThe number of required arguments for a varargs function definition.\nargs[5]::QuoteNode{<:Union{Symbol,Tuple{Symbol,UInt16}, Tuple{Symbol,UInt16,Bool}}: calling convention\nThe calling convention for the call, optionally with effects, and gc_safe (safe to execute concurrently to GC.).\nargs[6:5+length(args[3])] : arguments\nThe values for all the arguments (with types of each given in args[3]).\nargs[6+length(args[3])+1:end] : gc-roots\nThe additional objects that may need to be gc-rooted for the duration of the call.\nSee Working with LLVM for where these are derived from and how they get handled.\nnew_opaque_closure\nConstructs a new opaque closure. The fields are:\nargs[1] : signature\nThe function signature of the opaque closure. Opaque closures don't participate in dispatch, but the input types can be restricted.\nargs[2] : lb\nLower bound on the output type. (Defaults to Union{})\nargs[3] : ub\nUpper bound on the output type. (Defaults to Any)\nargs[4] : constprop\nIndicates whether the opaque closure's identity may be used for constant\npropagation. The @opaque macro enables this by default, but this will\ncause additional inference which may be undesirable and prevents the\ncode from running during precompile.\nIf args[4] is a method, the argument is considered skipped.\nargs[5] : method\nThe actual method as an opaque_closure_method expression.\nargs[6:end] : captures\nThe values captured by the opaque closure.\ncompat: Julia 1.7\nOpaque closures were added in Julia 1.7"},{"title":"Method","page":"Julia ASTs","location":"devdocs/ast.html#ast-lowered-method","category":"section","text":"A unique'd container describing the shared metadata for a single method.\n\nname, module, file, line, sig\nMetadata to uniquely identify the method for the computer and the human.\nambig\nCache of other methods that may be ambiguous with this one.\nspecializations\nCache of all MethodInstance ever created for this Method, used to ensure uniqueness.\nUniqueness is required for efficiency, especially for incremental precompile and\ntracking of method invalidation.\nsource\nThe original source code (if available, usually compressed).\ngenerator\nA callable object which can be executed to get specialized source for a specific method signature.\nroots\nPointers to non-AST things that have been interpolated into the AST, required by\ncompression of the AST, type-inference, or the generation of native code.\nnargs, isva, called, is_for_opaque_closure,\nDescriptive bit-fields for the source code of this Method.\nprimary_world\nThe world age that \"owns\" this Method."},{"title":"MethodInstance","page":"Julia ASTs","location":"devdocs/ast.html#MethodInstance","category":"section","text":"A unique'd container describing a single callable signature for a Method.\nSee especially Proper maintenance and care of multi-threading locks\nfor important details on how to modify these fields safely.\n\nspecTypes\nThe primary key for this MethodInstance. Uniqueness is guaranteed through a\ndef.specializations lookup.\ndef\nThe Method that this function describes a specialization of. Or a Module,\nif this is a top-level Lambda expanded in Module, and which is not part of a Method.\nsparam_vals\nThe values of the static parameters in specTypes.\nFor the MethodInstance at Method.unspecialized, this is the empty SimpleVector.\nBut for a runtime MethodInstance from the MethodTable cache, this will always be defined and indexable.\nbackedges\nWe store the reverse-list of cache dependencies for efficient tracking of incremental reanalysis/recompilation work that may be needed after a new method definitions.\nThis works by keeping a list of the other MethodInstance that have been inferred or optimized to contain a possible call to this MethodInstance.\nThose optimization results might be stored somewhere in the cache, or it might have been the result of something we didn't want to cache, such as constant propagation.\nThus we merge all of those backedges to various cache entries here (there's almost always only the one applicable cache entry with a sentinel value for max_world anyways).\ncache\nCache of CodeInstance objects that share this template instantiation."},{"title":"CodeInstance","page":"Julia ASTs","location":"devdocs/ast.html#CodeInstance","category":"section","text":"def\nThe MethodInstance that this cache entry is derived from.\nowner\nA token that represents the owner of this CodeInstance. Will use jl_egal to match.\nrettype/rettype_const\nThe inferred return type for the specFunctionObject field, which (in most cases) is\nalso the computed return type for the function in general.\ninferred\nMay contain a cache of the inferred source for this function,\nor it could be set to nothing to just indicate rettype is inferred.\nftpr\nThe generic jlcall entry point.\njlcall_api\nThe ABI to use when calling fptr. Some significant ones include:\n0 - Not compiled yet\n1 - JL_CALLABLE jl_value_t *(*)(jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\n2 - Constant (value stored in rettype_const)\n3 - With Static-parameters forwarded jl_value_t *(*)(jl_svec_t *sparams, jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\n4 - Run in interpreter jl_value_t *(*)(jl_method_instance_t *meth, jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\nmin_world / max_world\nThe range of world ages for which this method instance is valid to be called.\nIf max_world is the special token value -1, the value is not yet known.\nIt may continue to be used until we encounter a backedge that requires us to reconsider.\nTiming fields\ntime_infer_total: Total cost of computing inferred originally as wall-time from start to finish.\ntime_infer_cache_saved: The cost saved from time_infer_total by having caching.\nAdding this to time_infer_total should give a stable estimate for comparing the cost\nof two implementations or one implementation over time. This is generally an\nover-estimate of the time to infer something, since the cache is frequently effective\nat handling repeated work.\ntime_infer_self: Self cost of julia inference for inferred (a portion of\ntime_infer_total). This is simply the incremental cost of compiling this one method,\nif given a fully populated cache of all call targets, even including constant\ninference results and LimitedAccuracy results, which generally are not in a cache.\ntime_compile: Self cost of llvm JIT compilation (e.g. of computing invoke from\ninferred). A total cost estimate can be computed by walking all of the edges\ncontents and summing those, while accounting for cycles and duplicates. (This field\ncurrently does not include any measured AOT compile times.)"},{"title":"CodeInfo","page":"Julia ASTs","location":"devdocs/ast.html#CodeInfo","category":"section","text":"A (usually temporary) container for holding lowered (and possibly inferred) source code.\n\ncode\nAn Any array of statements\nslotnames\nAn array of symbols giving names for each slot (argument or local variable).\nslotflags\nA UInt8 array of slot properties, represented as bit flags:\n0x02 - assigned (only false if there are no assignment statements with this var on the left)\n0x08 - used (if there is any read or write of the slot)\n0x10 - statically assigned once\n0x20 - might be used before assigned. This flag is only valid after type inference.\nssavaluetypes\nEither an array or an Int.\nIf an Int, it gives the number of compiler-inserted temporary locations in the\nfunction (the length of code array). If an array, specifies a type for each location.\nssaflags\nStatement-level 32 bits flags for each expression in the function.\nSee the definition of jl_code_info_t in julia.h for more details.\n\nThese are only populated after inference (or by generated functions in some cases):\n\ndebuginfo\nAn object to retrieve source information for each statements, see\nHow to interpret line numbers in a CodeInfo object.\nrettype\nThe inferred return type of the lowered form (IR). Default value is Any. This is\nmostly present for convenience, as (due to the way OpaqueClosures work) it is not\nnecessarily the rettype used by codegen.\nparent\nThe MethodInstance that \"owns\" this object (if applicable).\nedges\nForward edges to method instances that must be invalidated.\nmin_world/max_world\nThe range of world ages for which this code was valid at the time when it had been inferred.\n\nOptional Fields:\n\nslottypes\nAn array of types for the slots.\nmethod_for_inference_limit_heuristics\nThe method_for_inference_heuristics will expand the given method's generator if\nnecessary during inference.\n\nBoolean properties:\n\npropagate_inbounds\nWhether this should propagate @inbounds when inlined for the purpose of eliding\n@boundscheck blocks.\n\nUInt8 settings:\n\nconstprop, inlineable\n0 = use heuristic\n1 = aggressive\n2 = none\npurity\nConstructed from 5 bit flags:\n0x01 << 0 = this method is guaranteed to return or terminate consistently (:consistent)\n0x01 << 1 = this method is free from externally semantically visible side effects (:effect_free)\n0x01 << 2 = this method is guaranteed to not throw an exception (:nothrow)\n0x01 << 3 = this method is guaranteed to terminate (:terminates_globally)\n0x01 << 4 = the syntactic control flow within this method is guaranteed to terminate (:terminates_locally)\nSee the documentation of Base.@assume_effects for more details."},{"title":"How to interpret line numbers in a CodeInfo object","page":"Julia ASTs","location":"devdocs/ast.html#How-to-interpret-line-numbers-in-a-CodeInfo-object","category":"section","text":"There are 2 common forms for this data: one used internally that compresses the data somewhat and one used in the compiler.\nThey contain the same basic info, but the compiler version is all mutable while the version used internally is not.\n\nMany consumers may be able to call Base.IRShow.buildLineInfoNode,\nBase.IRShow.append_scopes!, or Stacktraces.lookup(::InterpreterIP) to avoid needing to\n(re-)implement these details specifically.\n\nThe definitions of each of these are:\n\nstruct Core.DebugInfo\n    @noinline\n    def::Union{Method,MethodInstance,Symbol}\n    linetable::Union{Nothing,DebugInfo}\n    edges::SimpleVector{DebugInfo}\n    codelocs::String # compressed data\nend\nmutable struct Core.Compiler.DebugInfoStream\n    def::Union{Method,MethodInstance,Symbol}\n    linetable::Union{Nothing,DebugInfo}\n    edges::Vector{DebugInfo}\n    firstline::Int32 # the starting line for this block (specified by an index of 0)\n    codelocs::Vector{Int32} # for each statement:\n        # index into linetable (if defined), else a line number (in the file represented by def)\n        # then index into edges\n        # then index into edges[linetable]\nend\n\ndef : where this DebugInfo was defined (the Method, MethodInstance, or Symbol of file scope, for example)\nlinetable\nAnother DebugInfo that this was derived from, which contains the actual line numbers,\nsuch that this DebugInfo contains only the indexes into it. This avoids making copies,\nas well as makes it possible to track how each individual statement transformed from\nsource to optimized, not just the separate line numbers. If def is not a Symbol, then\nthat object replaces the current function object for the metadata on what function is\nconceptually being executed (e.g. think Cassette transforms here). The codelocs values\ndescribed below also are interpreted as an index into the codelocs in this object,\ninstead of being a line number itself.\nedges : Vector of the unique DebugInfo for every function inlined into this (which\nrecursively have the edges for everything inlined into them).\nfirstline (when uncompressed to DebugInfoStream)\nThe line number associated with the begin statement (or other keyword such as\nfunction or quote) that delineates where this code definition \"starts\".\ncodelocs (when uncompressed to DebugInfoStream)\nA vector of indices, with 3 values for each statement in the IR plus one for the\nstarting point of the block, that describe the stacktrace from that point:\nthe integer index into the linetable.codelocs field, giving the\noriginal location associated with each statement (including its syntactic edges),\nor zero indicating no change to the line number from the previously\nexecuted statement (which is not necessarily syntactic or lexical prior),\nor the line number itself if the linetable field is nothing.\nthe integer index into edges, giving the DebugInfo inlined there,\nor zero if there are no edges.\n(if entry 2 is non-zero) the integer index into edges[].codelocs,\nto interpret recursively for each function in the inlining stack,\nor zero indicating to use edges[].firstline as the line number.\nSpecial codes include:\n(zero, zero, *): no change to the line number or edges from the previous statement\n(you may choose to interpret this either syntactically or lexically). The inlining\ndepth also might have changed, though most callers should ignore that.\n(zero, non-zero, *) : no line number, just edges (usually because of\nmacro-expansion into top-level code)."},{"title":"Methods","page":"Methods","location":"manual/methods.html#Methods","category":"section","text":"Recall from Functions that a function is an object that maps a tuple of arguments to a\nreturn value, or throws an exception if no appropriate value can be returned. It is common for\nthe same conceptual function or operation to be implemented quite differently for different types\nof arguments: adding two integers is very different from adding two floating-point numbers, both\nof which are distinct from adding an integer to a floating-point number. Despite their implementation\ndifferences, these operations all fall under the general concept of \"addition\". Accordingly, in\nJulia, these behaviors all belong to a single object: the + function.\n\nTo facilitate using many different implementations of the same concept smoothly, functions need\nnot be defined all at once, but can rather be defined piecewise by providing specific behaviors\nfor certain combinations of argument types and counts. A definition of one possible behavior for\na function is called a method. Thus far, we have presented only examples of functions defined\nwith a single method, applicable to all types of arguments. However, the signatures of method\ndefinitions can be annotated to indicate the types of arguments in addition to their number, and\nmore than a single method definition may be provided. When a function is applied to a particular\ntuple of arguments, the most specific method applicable to those arguments is applied. Thus, the\noverall behavior of a function is a patchwork of the behaviors of its various method definitions.\nIf the patchwork is well designed, even though the implementations of the methods may be quite\ndifferent, the outward behavior of the function will appear seamless and consistent.\n\nThe choice of which method to execute when a function is applied is called dispatch. Julia allows\nthe dispatch process to choose which of a function's methods to call based on the number of arguments\ngiven, and on the types of all of the function's arguments. This is different than traditional\nobject-oriented languages, where dispatch occurs based only on the first argument, which often\nhas a special argument syntax, and is sometimes implied rather than explicitly written as an argument.\n[1] Using all of a function's arguments to choose which method should be invoked, rather than\njust the first, is known as multiple dispatch.\nMultiple dispatch is particularly useful for mathematical code, where it makes little sense to\nartificially deem the operations to \"belong\" to one argument more than any of the others: does\nthe addition operation in x + y belong to x any more than it does to y? The implementation\nof a mathematical operator generally depends on the types of all of its arguments. Even beyond\nmathematical operations, however, multiple dispatch ends up being a powerful and convenient paradigm\nfor structuring and organizing programs.\n\n[1]: In C++ or Java, for example, in a method call like obj.meth(arg1,arg2), the object obj \"receives\"\nthe method call and is implicitly passed to the method via the this keyword, rather than as\nan explicit method argument. When the current this object is the receiver of a method call,\nit can be omitted altogether, writing just meth(arg1,arg2), with this implied as the receiving\nobject.\n\nnote: Note\nAll the examples in this chapter assume that you are defining methods for a function in the same\nmodule. If you want to add methods to a function in another module, you have to import it or\nuse the name qualified with module names. See the section on [namespace management](@ref\nnamespace-management)."},{"title":"Defining Methods","page":"Methods","location":"manual/methods.html#Defining-Methods","category":"section","text":"Until now, we have, in our examples, defined only functions with a single method having unconstrained\nargument types. Such functions behave just like they would in traditional dynamically typed languages.\nNevertheless, we have used multiple dispatch and methods almost continually without being aware\nof it: all of Julia's standard functions and operators, like the aforementioned + function,\nhave many methods defining their behavior over various possible combinations of argument type\nand count.\n\nWhen defining a function, one can optionally constrain the types of parameters it is applicable\nto, using the :: type-assertion operator, introduced in the section on Composite Types:\n\njulia> f(x::Float64, y::Float64) = 2x + y\nf (generic function with 1 method)\n\nThis function definition applies only to calls where x and y are both values of type\nFloat64:\n\njulia> f(2.0, 3.0)\n7.0\n\nApplying it to any other types of arguments will result in a MethodError:\n\njulia> f(2.0, 3)\nERROR: MethodError: no method matching f(::Float64, ::Int64)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(::Float64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> f(Float32(2.0), 3.0)\nERROR: MethodError: no method matching f(::Float32, ::Float64)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(!Matched::Float64, ::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> f(2.0, \"3.0\")\nERROR: MethodError: no method matching f(::Float64, ::String)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(::Float64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> f(\"2.0\", \"3.0\")\nERROR: MethodError: no method matching f(::String, ::String)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nAs you can see, the arguments must be precisely of type Float64. Other numeric\ntypes, such as integers or 32-bit floating-point values, are not automatically converted\nto 64-bit floating-point, nor are strings parsed as numbers. Because Float64 is a concrete\ntype and concrete types cannot be subclassed in Julia, such a definition can only be applied\nto arguments that are exactly of type Float64. It may often be useful, however, to write\nmore general methods where the declared parameter types are abstract:\n\njulia> f(x::Number, y::Number) = 2x - y\nf (generic function with 2 methods)\n\njulia> f(2.0, 3)\n1.0\n\nThis method definition applies to any pair of arguments that are instances of Number.\nThey need not be of the same type, so long as they are each numeric values. The problem of\nhandling disparate numeric types is delegated to the arithmetic operations in the\nexpression 2x - y.\n\nTo define a function with multiple methods, one simply defines the function multiple times, with\ndifferent numbers and types of arguments. The first method definition for a function creates the\nfunction object, and subsequent method definitions add new methods to the existing function object.\nThe most specific method definition matching the number and types of the arguments will be executed\nwhen the function is applied. Thus, the two method definitions above, taken together, define the\nbehavior for f over all pairs of instances of the abstract type Number – but with a different\nbehavior specific to pairs of Float64 values. If one of the arguments is a 64-bit\nfloat but the other one is not, then the f(Float64,Float64) method cannot be called and\nthe more general f(Number,Number) method must be used:\n\njulia> f(2.0, 3.0)\n7.0\n\njulia> f(2, 3.0)\n1.0\n\njulia> f(2.0, 3)\n1.0\n\njulia> f(2, 3)\n1\n\nThe 2x + y definition is only used in the first case, while the 2x - y definition is used\nin the others. No automatic casting or conversion of function arguments is ever performed: all\nconversion in Julia is non-magical and completely explicit. Conversion and Promotion,\nhowever, shows how clever application of sufficiently advanced technology can be indistinguishable\nfrom magic. [Clarke61]\n\nFor non-numeric values, and for fewer or more than two arguments, the function f remains undefined,\nand applying it will still result in a MethodError:\n\njulia> f(\"foo\", 3)\nERROR: MethodError: no method matching f(::String, ::Int64)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(!Matched::Number, ::Number)\n   @ Main none:1\n  f(!Matched::Float64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> f()\nERROR: MethodError: no method matching f()\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(!Matched::Float64, !Matched::Float64)\n   @ Main none:1\n  f(!Matched::Number, !Matched::Number)\n   @ Main none:1\n\nStacktrace:\n[...]\n\nYou can easily see which methods exist for a function by entering the function object itself in\nan interactive session:\n\njulia> f\nf (generic function with 2 methods)\n\nThis output tells us that f is a function object with two methods. To find out what the signatures\nof those methods are, use the methods function:\n\njulia> methods(f)\n# 2 methods for generic function \"f\" from Main:\n [1] f(x::Float64, y::Float64)\n     @ none:1\n [2] f(x::Number, y::Number)\n     @ none:1\n\nwhich shows that f has two methods, one taking two Float64 arguments and one taking arguments\nof type Number. It also indicates the file and line number where the methods were defined: because\nthese methods were defined at the REPL, we get the apparent line number none:1.\n\nIn the absence of a type declaration with ::, the type of a method parameter is Any by default,\nmeaning that it is unconstrained since all values in Julia are instances of the abstract type\nAny. Thus, we can define a catch-all method for f like so:\n\njulia> f(x,y) = println(\"Whoa there, Nelly.\")\nf (generic function with 3 methods)\n\njulia> methods(f)\n# 3 methods for generic function \"f\" from Main:\n [1] f(x::Float64, y::Float64)\n     @ none:1\n [2] f(x::Number, y::Number)\n     @ none:1\n [3] f(x, y)\n     @ none:1\n\njulia> f(\"foo\", 1)\nWhoa there, Nelly.\n\nThis catch-all is less specific than any other possible method definition for a pair of parameter\nvalues, so it will only be called on pairs of arguments to which no other method definition applies.\n\nNote that in the signature of the third method, there is no type specified for the arguments x and y.\nThis is a shortened way of expressing f(x::Any, y::Any).\n\nAlthough it seems a simple concept, multiple dispatch on the types of values is perhaps the single\nmost powerful and central feature of the Julia language. Core operations typically have dozens\nof methods:\n\njulia> methods(+)\n# 180 methods for generic function \"+\":\n[1] +(x::Bool, z::Complex{Bool}) in Base at complex.jl:227\n[2] +(x::Bool, y::Bool) in Base at bool.jl:89\n[3] +(x::Bool) in Base at bool.jl:86\n[4] +(x::Bool, y::T) where T<:AbstractFloat in Base at bool.jl:96\n[5] +(x::Bool, z::Complex) in Base at complex.jl:234\n[6] +(a::Float16, b::Float16) in Base at float.jl:373\n[7] +(x::Float32, y::Float32) in Base at float.jl:375\n[8] +(x::Float64, y::Float64) in Base at float.jl:376\n[9] +(z::Complex{Bool}, x::Bool) in Base at complex.jl:228\n[10] +(z::Complex{Bool}, x::Real) in Base at complex.jl:242\n[11] +(x::Char, y::Integer) in Base at char.jl:40\n[12] +(c::BigInt, x::BigFloat) in Base.MPFR at mpfr.jl:307\n[13] +(a::BigInt, b::BigInt, c::BigInt, d::BigInt, e::BigInt) in Base.GMP at gmp.jl:392\n[14] +(a::BigInt, b::BigInt, c::BigInt, d::BigInt) in Base.GMP at gmp.jl:391\n[15] +(a::BigInt, b::BigInt, c::BigInt) in Base.GMP at gmp.jl:390\n[16] +(x::BigInt, y::BigInt) in Base.GMP at gmp.jl:361\n[17] +(x::BigInt, c::Union{UInt16, UInt32, UInt64, UInt8}) in Base.GMP at gmp.jl:398\n...\n[180] +(a, b, c, xs...) in Base at operators.jl:424\n\nMultiple dispatch together with the flexible parametric type system give Julia its ability to\nabstractly express high-level algorithms decoupled from implementation details."},{"title":"Method specializations","page":"Methods","location":"manual/methods.html#man-method-specializations","category":"section","text":"When you create multiple methods of the same function, this is sometimes called\n\"specialization.\" In this case, you're specializing the function by adding additional\nmethods to it: each new method is a new specialization of the function.\nAs shown above, these specializations are returned by methods.\n\nThere's another kind of specialization that occurs without programmer intervention:\nJulia's compiler can automatically specialize the method for the specific argument types used.\nSuch specializations are not listed by methods, as this doesn't create new Methods, but tools like @code_typed allow you to inspect such specializations.\n\nFor example, if you create a method\n\nmysum(x::Real, y::Real) = x + y\n\nyou've given the function mysum one new method (possibly its only method), and that method takes any pair of Real number inputs. But if you then execute\n\njulia> mysum(1, 2)\n3\n\njulia> mysum(1.0, 2.0)\n3.0\n\nJulia will compile mysum twice, once for x::Int, y::Int and again for x::Float64, y::Float64.\nThe point of compiling twice is performance: the methods that get called for + (which mysum uses) vary depending on the specific types of x and y, and by compiling different specializations Julia can do all the method lookup ahead of time. This allows the program to run much more quickly, since it does not have to bother with method lookup while it is running.\nJulia's automatic specialization allows you to write generic algorithms and expect that the compiler will generate efficient, specialized code to handle each case you need.\n\nIn cases where the number of potential specializations might be effectively unlimited, Julia may avoid this default specialization. See Be aware of when Julia avoids specializing for more information."},{"title":"Method Ambiguities","page":"Methods","location":"manual/methods.html#man-ambiguities","category":"section","text":"It is possible to define a set of function methods such that there is no unique most specific\nmethod applicable to some combinations of arguments:\n\njulia> g(x::Float64, y) = 2x + y\ng (generic function with 1 method)\n\njulia> g(x, y::Float64) = x + 2y\ng (generic function with 2 methods)\n\njulia> g(2.0, 3)\n7.0\n\njulia> g(2, 3.0)\n8.0\n\njulia> g(2.0, 3.0)\nERROR: MethodError: g(::Float64, ::Float64) is ambiguous.\n\nCandidates:\n  g(x, y::Float64)\n    @ Main none:1\n  g(x::Float64, y)\n    @ Main none:1\n\nPossible fix, define\n  g(::Float64, ::Float64)\n\nStacktrace:\n[...]\n\nHere the call g(2.0, 3.0) could be handled by either the g(::Float64, ::Any) or the\ng(::Any, ::Float64) method. The order in which the methods are defined does not matter and\nneither is more specific than the other. In such cases, Julia raises a\nMethodError rather than arbitrarily picking a method. You can avoid method\nambiguities by specifying an appropriate method for the intersection case:\n\njulia> g(x::Float64, y::Float64) = 2x + 2y\ng (generic function with 3 methods)\n\njulia> g(2.0, 3)\n7.0\n\njulia> g(2, 3.0)\n8.0\n\njulia> g(2.0, 3.0)\n10.0\n\nIt is recommended that the disambiguating method be defined first, since otherwise the ambiguity\nexists, if transiently, until the more specific method is defined.\n\nIn more complex cases, resolving method ambiguities involves a certain\nelement of design; this topic is explored further below."},{"title":"Parametric Methods","page":"Methods","location":"manual/methods.html#Parametric-Methods","category":"section","text":"Method definitions can optionally have type parameters qualifying the signature:\n\njulia> same_type(x::T, y::T) where {T} = true\nsame_type (generic function with 1 method)\n\njulia> same_type(x,y) = false\nsame_type (generic function with 2 methods)\n\nThe first method applies whenever both arguments are of the same concrete type, regardless of\nwhat type that is, while the second method acts as a catch-all, covering all other cases. Thus,\noverall, this defines a boolean function that checks whether its two arguments are of the same\ntype:\n\njulia> same_type(1, 2)\ntrue\n\njulia> same_type(1, 2.0)\nfalse\n\njulia> same_type(1.0, 2.0)\ntrue\n\njulia> same_type(\"foo\", 2.0)\nfalse\n\njulia> same_type(\"foo\", \"bar\")\ntrue\n\njulia> same_type(Int32(1), Int64(2))\nfalse\n\nSuch definitions correspond to methods whose type signatures are UnionAll types\n(see UnionAll Types).\n\nThis kind of definition of function behavior by dispatch is quite common – idiomatic, even –\nin Julia. Method type parameters are not restricted to being used as the types of arguments:\nthey can be used anywhere a value would be in the signature of the function or body of the function.\nHere's an example where the method type parameter T is used as the type parameter to the parametric\ntype Vector{T} in the method signature:\n\njulia> function myappend(v::Vector{T}, x::T) where {T}\n           return [v..., x]\n       end\nmyappend (generic function with 1 method)\n\nThe type parameter T in this example ensures that the added element x is a subtype of the\nexisting eltype of the vector v.\nThe where keyword introduces a list of those constraints after the method signature definition.\nThis works the same for one-line definitions, as seen above, and must appear before the [return\ntype declaration](@ref man-functions-return-type), if present, as illustrated below:\n\njulia> (myappend(v::Vector{T}, x::T)::Vector) where {T} = [v..., x]\nmyappend (generic function with 1 method)\n\njulia> myappend([1,2,3],4)\n4-element Vector{Int64}:\n 1\n 2\n 3\n 4\n\njulia> myappend([1,2,3],2.5)\nERROR: MethodError: no method matching myappend(::Vector{Int64}, ::Float64)\nThe function `myappend` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  myappend(::Vector{T}, !Matched::T) where T\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> myappend([1.0,2.0,3.0],4.0)\n4-element Vector{Float64}:\n 1.0\n 2.0\n 3.0\n 4.0\n\njulia> myappend([1.0,2.0,3.0],4)\nERROR: MethodError: no method matching myappend(::Vector{Float64}, ::Int64)\nThe function `myappend` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  myappend(::Vector{T}, !Matched::T) where T\n   @ Main none:1\n\nStacktrace:\n[...]\n\nIf the type of the appended element does not match the element type of the vector it is appended to,\na MethodError is raised.\nIn the following example, the method's type parameter T is used as the return value:\n\njulia> mytypeof(x::T) where {T} = T\nmytypeof (generic function with 1 method)\n\njulia> mytypeof(1)\nInt64\n\njulia> mytypeof(1.0)\nFloat64\n\nJust as you can put subtype constraints on type parameters in type declarations (see Parametric Types),\nyou can also constrain type parameters of methods:\n\njulia> same_type_numeric(x::T, y::T) where {T<:Number} = true\nsame_type_numeric (generic function with 1 method)\n\njulia> same_type_numeric(x::Number, y::Number) = false\nsame_type_numeric (generic function with 2 methods)\n\njulia> same_type_numeric(1, 2)\ntrue\n\njulia> same_type_numeric(1, 2.0)\nfalse\n\njulia> same_type_numeric(1.0, 2.0)\ntrue\n\njulia> same_type_numeric(\"foo\", 2.0)\nERROR: MethodError: no method matching same_type_numeric(::String, ::Float64)\nThe function `same_type_numeric` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  same_type_numeric(!Matched::T, ::T) where T<:Number\n   @ Main none:1\n  same_type_numeric(!Matched::Number, ::Number)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> same_type_numeric(\"foo\", \"bar\")\nERROR: MethodError: no method matching same_type_numeric(::String, ::String)\nThe function `same_type_numeric` exists, but no method is defined for this combination of argument types.\n\njulia> same_type_numeric(Int32(1), Int64(2))\nfalse\n\nThe same_type_numeric function behaves much like the same_type function defined above, but\nis only defined for pairs of numbers.\n\nParametric methods allow the same syntax as where expressions used to write types\n(see UnionAll Types).\nIf there is only a single parameter, the enclosing curly braces (in where {T}) can be omitted,\nbut are often preferred for clarity.\nMultiple parameters can be separated with commas, e.g. where {T, S<:Real}, or written using\nnested where, e.g. where S<:Real where T."},{"title":"Redefining Methods","page":"Methods","location":"manual/methods.html#Redefining-Methods","category":"section","text":"When redefining a method or adding new methods,\nit is important to realize that these changes don't take effect immediately.\nThis is key to Julia's ability to statically infer and compile code to run fast,\nwithout the usual JIT tricks and overhead.\nIndeed, any new method definition won't be visible to the current runtime environment,\nincluding Tasks and Threads (and any previously defined @generated functions).\nLet's start with an example to see what this means:\n\njulia> function tryeval()\n           @eval newfun() = 1\n           newfun()\n       end\ntryeval (generic function with 1 method)\n\njulia> tryeval()\nERROR: MethodError: no method matching newfun()\nThe applicable method may be too new: running in world age xxxx1, while current world is xxxx2.\nClosest candidates are:\n  newfun() at none:1 (method too new to be called from this world context.)\n in tryeval() at none:1\n ...\n\njulia> newfun()\n1\n\nIn this example, observe that the new definition for newfun has been created,\nbut can't be immediately called.\nThe new global is immediately visible to the tryeval function,\nso you could write return newfun (without parentheses).\nBut neither you, nor any of your callers, nor the functions they call, or etc.\ncan call this new method definition!\n\nBut there's an exception: future calls to newfun from the REPL work as expected,\nbeing able to both see and call the new definition of newfun.\n\nHowever, future calls to tryeval will continue to see the definition of newfun as it was\nat the previous statement at the REPL, and thus before that call to tryeval.\n\nYou may want to try this for yourself to see how it works.\n\nThe implementation of this behavior is a \"world age counter\", which is further described in the World Age\nmanual chapter."},{"title":"Design Patterns with Parametric Methods","page":"Methods","location":"manual/methods.html#Design-Patterns-with-Parametric-Methods","category":"section","text":"While complex dispatch logic is not required for performance or usability,\nsometimes it can be the best way to express some algorithm.\nHere are a few common design patterns that come up sometimes when using dispatch in this way."},{"title":"Extracting the type parameter from a super-type","page":"Methods","location":"manual/methods.html#Extracting-the-type-parameter-from-a-super-type","category":"section","text":"Here is a correct code template for returning the element-type T\nof any arbitrary subtype of AbstractArray that has well-defined\nelement type:\n\nabstract type AbstractArray{T, N} end\neltype(::Type{<:AbstractArray{T}}) where {T} = T\n\nusing so-called triangular dispatch. Note that UnionAll types, for\nexample eltype(AbstractArray{T} where T <: Integer), do not match the\nabove method. The implementation of eltype in Base adds a fallback\nmethod to Any for such cases.\n\nOne common mistake is to try and get the element-type by using introspection:\n\neltype_wrong(::Type{A}) where {A<:AbstractArray} = A.parameters[1]\n\nHowever, it is not hard to construct cases where this will fail:\n\nstruct BitVector <: AbstractArray{Bool, 1}; end\n\nHere we have created a type BitVector which has no parameters,\nbut where the element-type is still fully specified, with T equal to Bool!\n\nAnother mistake is to try to walk up the type hierarchy using\nsupertype:\n\neltype_wrong(::Type{AbstractArray{T}}) where {T} = T\neltype_wrong(::Type{AbstractArray{T, N}}) where {T, N} = T\neltype_wrong(::Type{A}) where {A<:AbstractArray} = eltype_wrong(supertype(A))\n\nWhile this works for declared types, it fails for types without\nsupertypes:\n\njulia> eltype_wrong(Union{Vector{Int}, Matrix{Int}})\nERROR: MethodError: no method matching supertype(::Type{VecOrMat{Int64}})\n\nClosest candidates are:\n  supertype(::UnionAll)\n   @ Base operators.jl:44\n  supertype(::DataType)\n   @ Base operators.jl:43"},{"title":"Building a similar type with a different type parameter","page":"Methods","location":"manual/methods.html#Building-a-similar-type-with-a-different-type-parameter","category":"section","text":"When building generic code, there is often a need for constructing a similar\nobject with some change made to the layout of the type, also\nnecessitating a change of the type parameters.\nFor instance, you might have some sort of abstract array with an arbitrary element type\nand want to write your computation on it with a specific element type.\nWe must implement a method for each AbstractArray{T} subtype that describes how to compute this type transform.\nThere is no general transform of one subtype into another subtype with a different parameter.\n\nThe subtypes of AbstractArray typically implement two methods to\nachieve this:\nA method to convert the input array to a subtype of a specific AbstractArray{T, N} abstract type;\nand a method to make a new uninitialized array with a specific element type.\nSample implementations of these can be found in Julia Base.\nHere is a basic example usage of them, guaranteeing that input and\noutput are of the same type:\n\ninput = convert(AbstractArray{Eltype}, input)\noutput = similar(input, Eltype)\n\nAs an extension of this, in cases where the algorithm needs a copy of\nthe input array,\nconvert is insufficient as the return value may alias the original input.\nCombining similar (to make the output array) and copyto! (to fill it with the input data)\nis a generic way to express the requirement for a mutable copy of the input argument:\n\ncopy_with_eltype(input, Eltype) = copyto!(similar(input, Eltype), input)"},{"title":"Iterated dispatch","page":"Methods","location":"manual/methods.html#Iterated-dispatch","category":"section","text":"In order to dispatch a multi-level parametric argument list,\noften it is best to separate each level of dispatch into distinct functions.\nThis may sound similar in approach to single-dispatch, but as we shall see below, it is still more flexible.\n\nFor example, trying to dispatch on the element-type of an array will often run into ambiguous situations.\nInstead, common code will dispatch first on the container type,\nthen recurse down to a more specific method based on eltype.\nIn most cases, the algorithms lend themselves conveniently to this hierarchical approach,\nwhile in other cases, this rigor must be resolved manually.\nThis dispatching branching can be observed, for example, in the logic to sum two matrices:\n\n# First dispatch selects the map algorithm for element-wise summation.\n+(a::Matrix, b::Matrix) = map(+, a, b)\n# Then dispatch handles each element and selects the appropriate\n# common element type for the computation.\n+(a, b) = +(promote(a, b)...)\n# Once the elements have the same type, they can be added.\n# For example, via primitive operations exposed by the processor.\n+(a::Float64, b::Float64) = Core.add(a, b)"},{"title":"Trait-based dispatch","page":"Methods","location":"manual/methods.html#Trait-based-dispatch","category":"section","text":"A natural extension to the iterated dispatch above is to add a layer to\nmethod selection that allows to dispatch on sets of types which are\nindependent from the sets defined by the type hierarchy.\nWe could construct such a set by writing out a Union of the types in question,\nbut then this set would not be extensible as Union-types cannot be\naltered after creation.\nHowever, such an extensible set can be programmed with a design pattern\noften referred to as a\n\"Holy-trait\".\n\nThis pattern is implemented by defining a generic function which\ncomputes a different singleton value (or type) for each trait-set to which the\nfunction arguments may belong to. If this function is pure there is\nno impact on performance compared to normal dispatch.\n\nThe example in the previous section glossed over the implementation details of\nmap and promote, which both operate in terms of these traits.\nWhen iterating over a matrix, such as in the implementation of map,\none important question is what order to use to traverse the data.\nWhen AbstractArray subtypes implement the Base.IndexStyle trait,\nother functions such as map can dispatch on this information to pick\nthe best algorithm (see Abstract Array Interface).\nThis means that each subtype does not need to implement a custom version of map,\nsince the generic definitions + trait classes will enable the system to select the fastest version.\nHere is a toy implementation of map illustrating the trait-based dispatch:\n\nmap(f, a::AbstractArray, b::AbstractArray) = map(Base.IndexStyle(a, b), f, a, b)\n# generic implementation:\nmap(::Base.IndexCartesian, f, a::AbstractArray, b::AbstractArray) = ...\n# linear-indexing implementation (faster)\nmap(::Base.IndexLinear, f, a::AbstractArray, b::AbstractArray) = ...\n\nThis trait-based approach is also present in the promote\nmechanism employed by the scalar +.\nIt uses promote_type, which returns the optimal common type to\ncompute the operation given the two types of the operands.\nThis makes it possible to reduce the problem of implementing every function for every pair of possible type arguments,\nto the much smaller problem of implementing a conversion operation from each type to a common type,\nplus a table of preferred pair-wise promotion rules."},{"title":"Output-type computation","page":"Methods","location":"manual/methods.html#Output-type-computation","category":"section","text":"The discussion of trait-based promotion provides a transition into our next design pattern:\ncomputing the output element type for a matrix operation.\n\nFor implementing primitive operations, such as addition,\nwe use the promote_type function to compute the desired output type.\n(As before, we saw this at work in the promote call in the call to +).\n\nFor more complex functions on matrices, it may be necessary to compute the expected return\ntype for a more complex sequence of operations.\nThis is often performed by the following steps:\n\nWrite a small function op that expresses the set of operations performed by the kernel of the algorithm.\nCompute the element type R of the result matrix as promote_op(op, argument_types...),\nwhere argument_types is computed from eltype applied to each input array.\nBuild the output matrix as similar(R, dims), where dims are the desired dimensions of the output array.\n\nFor a more specific example, a generic square-matrix multiply pseudo-code might look like:\n\nfunction matmul(a::AbstractMatrix, b::AbstractMatrix)\n    op = (ai, bi) -> ai * bi + ai * bi\n\n    ## this is insufficient because it assumes `one(eltype(a))` is constructable:\n    # R = typeof(op(one(eltype(a)), one(eltype(b))))\n\n    ## this fails because it assumes `a[1]` exists and is representative of all elements of the array\n    # R = typeof(op(a[1], b[1]))\n\n    ## this is incorrect because it assumes that `+` calls `promote_type`\n    ## but this is not true for some types, such as Bool:\n    # R = promote_type(ai, bi)\n\n    # this is wrong, since depending on the return value\n    # of type-inference is very brittle (as well as not being optimizable):\n    # R = Base.return_types(op, (eltype(a), eltype(b)))\n\n    ## but, finally, this works:\n    R = promote_op(op, eltype(a), eltype(b))\n    ## although sometimes it may give a larger type than desired\n    ## it will always give a correct type\n\n    output = similar(b, R, (size(a, 1), size(b, 2)))\n    if size(a, 2) > 0\n        for j in 1:size(b, 2)\n            for i in 1:size(a, 1)\n                ## here we don't use `ab = zero(R)`,\n                ## since `R` might be `Any` and `zero(Any)` is not defined\n                ## we also must declare `ab::R` to make the type of `ab` constant in the loop,\n                ## since it is possible that typeof(a * b) != typeof(a * b + a * b) == R\n                ab::R = a[i, 1] * b[1, j]\n                for k in 2:size(a, 2)\n                    ab += a[i, k] * b[k, j]\n                end\n                output[i, j] = ab\n            end\n        end\n    end\n    return output\nend"},{"title":"Separate convert and kernel logic","page":"Methods","location":"manual/methods.html#Separate-convert-and-kernel-logic","category":"section","text":"One way to significantly cut down on compile-times and testing complexity is to isolate\nthe logic for converting to the desired type and the computation.\nThis lets the compiler specialize and inline the conversion logic independent\nfrom the rest of the body of the larger kernel.\n\nThis is a common pattern seen when converting from a larger class of types\nto the one specific argument type that is actually supported by the algorithm:\n\ncomplexfunction(arg::Int) = ...\ncomplexfunction(arg::Any) = complexfunction(convert(Int, arg))\n\nmatmul(a::T, b::T) = ...\nmatmul(a, b) = matmul(promote(a, b)...)"},{"title":"Parametrically-constrained Varargs methods","page":"Methods","location":"manual/methods.html#Parametrically-constrained-Varargs-methods","category":"section","text":"Function parameters can also be used to constrain the number of arguments that may be supplied\nto a \"varargs\" function (Varargs Functions). The notation Vararg{T,N} is used to indicate\nsuch a constraint. For example:\n\njulia> bar(a,b,x::Vararg{Any,2}) = (a,b,x)\nbar (generic function with 1 method)\n\njulia> bar(1,2,3)\nERROR: MethodError: no method matching bar(::Int64, ::Int64, ::Int64)\nThe function `bar` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  bar(::Any, ::Any, ::Any, !Matched::Any)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> bar(1,2,3,4)\n(1, 2, (3, 4))\n\njulia> bar(1,2,3,4,5)\nERROR: MethodError: no method matching bar(::Int64, ::Int64, ::Int64, ::Int64, ::Int64)\nThe function `bar` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  bar(::Any, ::Any, ::Any, ::Any)\n   @ Main none:1\n\nStacktrace:\n[...]\n\nMore usefully, it is possible to constrain varargs methods by a parameter. For example:\n\nfunction getindex(A::AbstractArray{T,N}, indices::Vararg{Number,N}) where {T,N}\n\nwould be called only when the number of indices matches the dimensionality of the array.\n\nWhen only the type of supplied arguments needs to be constrained Vararg{T} can be equivalently\nwritten as T.... For instance f(x::Int...) = x is a shorthand for f(x::Vararg{Int}) = x."},{"title":"Note on Optional and keyword Arguments","page":"Methods","location":"manual/methods.html#Note-on-Optional-and-keyword-Arguments","category":"section","text":"As mentioned briefly in Functions, optional arguments are implemented as syntax for multiple\nmethod definitions. For example, this definition:\n\nf(a=1,b=2) = a+2b\n\ntranslates to the following three methods:\n\nf(a,b) = a+2b\nf(a) = f(a,2)\nf() = f(1,2)\n\nThis means that calling f() is equivalent to calling f(1,2). In this case the result is 5,\nbecause f(1,2) invokes the first method of f above. However, this need not always be the case.\nIf you define a fourth method that is more specialized for integers:\n\nf(a::Int,b::Int) = a-2b\n\nthen the result of both f() and f(1,2) is -3. In other words, optional arguments are tied\nto a function, not to any specific method of that function. It depends on the types of the optional\narguments which method is invoked. When optional arguments are defined in terms of a global variable,\nthe type of the optional argument may even change at run-time.\n\nKeyword arguments behave quite differently from ordinary positional arguments. In particular,\nthey do not participate in method dispatch. Methods are dispatched based only on positional arguments,\nwith keyword arguments processed after the matching method is identified."},{"title":"Function-like objects","page":"Methods","location":"manual/methods.html#Function-like-objects","category":"section","text":"Methods are associated with types, so it is possible to make any arbitrary Julia object \"callable\"\nby adding methods to its type.\n\nFor example, you can define a type that stores the coefficients of a polynomial, but behaves like\na function evaluating the polynomial:\n\njulia> struct Polynomial{R}\n           coeffs::Vector{R}\n       end\n\njulia> function (p::Polynomial)(x)\n           v = p.coeffs[end]\n           for i = (length(p.coeffs)-1):-1:1\n               v = v*x + p.coeffs[i]\n           end\n           return v\n       end\n\njulia> (p::Polynomial)() = p(5)\n\nNotice that the function is specified by type instead of by name. As with normal functions\nthere is a terse syntax form. In the function body, p will refer to the object that was\ncalled. A Polynomial can be used as follows:\n\njulia> poly = Polynomial([1,10,100])\nPolynomial{Int64}([1, 10, 100])\n\njulia> poly(3)\n931\n\njulia> poly()\n2551\n\nThis mechanism is also the key to how type constructors and closures (inner functions that refer\nto their surrounding environment) work in Julia."},{"title":"Empty generic functions","page":"Methods","location":"manual/methods.html#Empty-generic-functions","category":"section","text":"Occasionally it is useful to introduce a generic function without yet adding methods. This can\nbe used to separate interface definitions from implementations. It might also be done for the\npurpose of documentation or code readability. The syntax for this is an empty function block\nwithout a tuple of arguments:\n\nfunction emptyfunc end"},{"title":"Method design and the avoidance of ambiguities","page":"Methods","location":"manual/methods.html#man-method-design-ambiguities","category":"section","text":"Julia's method polymorphism is one of its most powerful features, yet\nexploiting this power can pose design challenges. In particular, in\nmore complex method hierarchies it is not uncommon for\nambiguities to arise.\n\nAbove, it was pointed out that one can resolve ambiguities like\n\nf(x, y::Int) = 1\nf(x::Int, y) = 2\n\nby defining a method\n\nf(x::Int, y::Int) = 3\n\nThis is often the right strategy; however, there are circumstances\nwhere following this advice mindlessly can be counterproductive. In\nparticular, the more methods a generic function has, the more\npossibilities there are for ambiguities. When your method hierarchies\nget more complicated than this simple example, it can be worth your\nwhile to think carefully about alternative strategies.\n\nBelow we discuss particular challenges and some alternative ways to resolve such issues."},{"title":"Tuple and NTuple arguments","page":"Methods","location":"manual/methods.html#Tuple-and-NTuple-arguments","category":"section","text":"Tuple (and NTuple) arguments present special challenges. For example,\n\nf(x::NTuple{N,Int}) where {N} = 1\nf(x::NTuple{N,Float64}) where {N} = 2\n\nare ambiguous because of the possibility that N == 0: there are no\nelements to determine whether the Int or Float64 variant should be\ncalled. To resolve the ambiguity, one approach is define a method for\nthe empty tuple:\n\nf(x::Tuple{}) = 3\n\nAlternatively, for all methods but one you can insist that there is at\nleast one element in the tuple:\n\nf(x::NTuple{N,Int}) where {N} = 1           # this is the fallback\nf(x::Tuple{Float64, Vararg{Float64}}) = 2   # this requires at least one Float64"},{"title":"Orthogonalize your design","page":"Methods","location":"manual/methods.html#man-methods-orthogonalize","category":"section","text":"When you might be tempted to dispatch on two or more arguments,\nconsider whether a \"wrapper\" function might make for a simpler\ndesign. For example, instead of writing multiple variants:\n\nf(x::A, y::A) = ...\nf(x::A, y::B) = ...\nf(x::B, y::A) = ...\nf(x::B, y::B) = ...\n\nyou might consider defining\n\nf(x::A, y::A) = ...\nf(x, y) = f(g(x), g(y))\n\nwhere g converts the argument to type A. This is a very specific\nexample of the more general principle of\northogonal design,\nin which separate concepts are assigned to separate methods. Here, g\nwill most likely need a fallback definition\n\ng(x::A) = x\n\nA related strategy exploits promote to bring x and y to a common\ntype:\n\nf(x::T, y::T) where {T} = ...\nf(x, y) = f(promote(x, y)...)\n\nOne risk with this design is the possibility that if there is no\nsuitable promotion method converting x and y to the same type, the\nsecond method will recurse on itself infinitely and trigger a stack\noverflow."},{"title":"Dispatch on one argument at a time","page":"Methods","location":"manual/methods.html#Dispatch-on-one-argument-at-a-time","category":"section","text":"If you need to dispatch on multiple arguments, and there are many\nfallbacks with too many combinations to make it practical to define\nall possible variants, then consider introducing a \"name cascade\"\nwhere (for example) you dispatch on the first argument and then call\nan internal method:\n\nf(x::A, y) = _fA(x, y)\nf(x::B, y) = _fB(x, y)\n\nThen the internal methods _fA and _fB can dispatch on y without\nconcern about ambiguities with each other with respect to x.\n\nBe aware that this strategy has at least one major disadvantage: in\nmany cases, it is not possible for users to further customize the\nbehavior of f by defining further specializations of your exported\nfunction f. Instead, they have to define specializations for your\ninternal methods _fA and _fB, and this blurs the lines between\nexported and internal methods."},{"title":"Abstract containers and element types","page":"Methods","location":"manual/methods.html#Abstract-containers-and-element-types","category":"section","text":"Where possible, try to avoid defining methods that dispatch on\nspecific element types of abstract containers. For example,\n\n-(A::AbstractArray{T}, b::Date) where {T<:Date}\n\ngenerates ambiguities for anyone who defines a method\n\n-(A::MyArrayType{T}, b::T) where {T}\n\nThe best approach is to avoid defining either of these methods:\ninstead, rely on a generic method -(A::AbstractArray, b) and make\nsure this method is implemented with generic calls (like similar and\n-) that do the right thing for each container type and element type\nseparately. This is just a more complex variant of the advice to\northogonalize your methods.\n\nWhen this approach is not possible, it may be worth starting a\ndiscussion with other developers about resolving the ambiguity; just\nbecause one method was defined first does not necessarily mean that it\ncan't be modified or eliminated. As a last resort, one developer can\ndefine the \"band-aid\" method\n\n-(A::MyArrayType{T}, b::Date) where {T<:Date} = ...\n\nthat resolves the ambiguity by brute force."},{"title":"Complex method \"cascades\" with default arguments","page":"Methods","location":"manual/methods.html#Complex-method-\"cascades\"-with-default-arguments","category":"section","text":"If you are defining a method \"cascade\" that supplies defaults, be\ncareful about dropping any arguments that correspond to potential\ndefaults. For example, suppose you're writing a digital filtering\nalgorithm and you have a method that handles the edges of the signal\nby applying padding:\n\nfunction myfilter(A, kernel, ::Replicate)\n    Apadded = replicate_edges(A, size(kernel))\n    myfilter(Apadded, kernel)  # now perform the \"real\" computation\nend\n\nThis will run afoul of a method that supplies default padding:\n\nmyfilter(A, kernel) = myfilter(A, kernel, Replicate()) # replicate the edge by default\n\nTogether, these two methods generate an infinite recursion with A constantly growing bigger.\n\nThe better design would be to define your call hierarchy like this:\n\nstruct NoPad end  # indicate that no padding is desired, or that it's already applied\n\nmyfilter(A, kernel) = myfilter(A, kernel, Replicate())  # default boundary conditions\n\nfunction myfilter(A, kernel, ::Replicate)\n    Apadded = replicate_edges(A, size(kernel))\n    myfilter(Apadded, kernel, NoPad())  # indicate the new boundary conditions\nend\n\n# other padding methods go here\n\nfunction myfilter(A, kernel, ::NoPad)\n    # Here's the \"real\" implementation of the core computation\nend\n\nNoPad is supplied in the same argument position as any other kind of\npadding, so it keeps the dispatch hierarchy well organized and with\nreduced likelihood of ambiguities. Moreover, it extends the \"public\"\nmyfilter interface: a user who wants to control the padding\nexplicitly can call the NoPad variant directly."},{"title":"Defining methods in local scope","page":"Methods","location":"manual/methods.html#Defining-methods-in-local-scope","category":"section","text":"You can define methods within a local scope, for example\n\njulia> function f(x)\n           g(y::Int) = y + x\n           g(y) = y - x\n           g\n       end\nf (generic function with 1 method)\n\njulia> h = f(3);\n\njulia> h(4)\n7\n\njulia> h(4.0)\n1.0\n\nHowever, you should not define local methods conditionally or subject to control flow, as in\n\nfunction f2(inc)\n    if inc\n        g(x) = x + 1\n    else\n        g(x) = x - 1\n    end\nend\n\nfunction f3()\n    function g end\n    return g\n    g() = 0\nend\n\nas it is not clear what function will end up getting defined. In the future, it might be an error to define local methods in this manner.\n\nFor cases like this use anonymous functions instead:\n\nfunction f2(inc)\n    g = if inc\n        x -> x + 1\n    else\n        x -> x - 1\n    end\nend\n\n[Clarke61]: Arthur C. Clarke, Profiles of the Future (1961): Clarke's Third Law."},{"title":"Multi-Threading","page":"Multi-Threading","location":"base/multi-threading.html#lib-multithreading","category":"section","text":"See also Multi-Threading."},{"title":"Atomic operations","page":"Multi-Threading","location":"base/multi-threading.html#Atomic-operations","category":"section","text":"There are also optional memory ordering parameters for the unsafe set of functions, that\nselect the C/C++-compatible versions of these atomic operations, if that parameter is specified to\nunsafe_load, unsafe_store!, unsafe_swap!, unsafe_replace!, and unsafe_modify!.\n\nwarning: Warning\nThe following APIs are deprecated, though support for them is likely to remain for several releases."},{"title":"ccall using a libuv threadpool (Experimental)","page":"Multi-Threading","location":"base/multi-threading.html#ccall-using-a-libuv-threadpool-(Experimental)","category":"section","text":""},{"title":"Low-level synchronization primitives","page":"Multi-Threading","location":"base/multi-threading.html#Low-level-synchronization-primitives","category":"section","text":"These building blocks are used to create the regular synchronization objects."},{"title":"Task metrics (Experimental)","page":"Multi-Threading","location":"base/multi-threading.html#Task-metrics-(Experimental)","category":"section","text":""},{"title":"Base.Threads.@threads","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.@threads","category":"macro","text":"Threads.@threads [schedule] for ... end\n\nA macro to execute a for loop in parallel. The iteration space is distributed to\ncoarse-grained tasks. This policy can be specified by the schedule argument. The\nexecution of the loop waits for the evaluation of all iterations.\n\nTasks spawned by @threads are scheduled on the :default threadpool. This means that\n@threads will not use threads from the :interactive threadpool, even if called from\nthe main thread or from a task in the interactive pool. The :default threadpool is\nintended for compute-intensive parallel workloads.\n\nSee also: @spawn and\npmap in Distributed.\nFor more information on threadpools, see the chapter on threadpools.\n\nExtended help\n\nSemantics\n\nUnless stronger guarantees are specified by the scheduling option, the loop executed by\n@threads macro have the following semantics.\n\nThe @threads macro executes the loop body in an unspecified order and potentially\nconcurrently. It does not specify the exact assignments of the tasks and the worker threads.\nThe assignments can be different for each execution. The loop body code (including any code\ntransitively called from it) must not make any assumptions about the distribution of\niterations to tasks or the worker thread in which they are executed. The loop body for each\niteration must be able to make forward progress independent of other iterations and be free\nfrom data races. As such, invalid synchronizations across iterations may deadlock while\nunsynchronized memory accesses may result in undefined behavior.\n\nFor example, the above conditions imply that:\n\nA lock taken in an iteration must be released within the same iteration.\nCommunicating between iterations using blocking primitives like Channels is incorrect.\nWrite only to locations not shared across iterations (unless a lock or atomic operation is\nused).\nUnless the :static schedule is used, the value of threadid()\nmay change even within a single iteration. See Task Migration.\n\nSchedulers\n\nWithout the scheduler argument, the exact scheduling is unspecified and varies across Julia\nreleases. Currently, :dynamic is used when the scheduler is not specified.\n\ncompat: Julia 1.5\nThe schedule argument is available as of Julia 1.5.\n\n:dynamic (default)\n\n:dynamic scheduler executes iterations dynamically to available worker threads. Current\nimplementation assumes that the workload for each iteration is uniform. However, this\nassumption may be removed in the future.\n\nThis scheduling option is merely a hint to the underlying execution mechanism. However, a\nfew properties can be expected. The number of Tasks used by :dynamic scheduler is\nbounded by a small constant multiple of the number of available worker threads\n(Threads.threadpoolsize()). Each task processes contiguous regions of the\niteration space. Thus, @threads :dynamic for x in xs; f(x); end is typically more\nefficient than @sync for x in xs; @spawn f(x); end if length(xs) is significantly\nlarger than the number of the worker threads and the run-time of f(x) is relatively\nsmaller than the cost of spawning and synchronizing a task (typically less than 10\nmicroseconds).\n\ncompat: Julia 1.8\nThe :dynamic option for the schedule argument is available and the default as of Julia 1.8.\n\n:greedy\n\n:greedy scheduler spawns up to Threads.threadpoolsize() tasks, each greedily working on\nthe given iterated values as they are produced. As soon as one task finishes its work, it takes\nthe next value from the iterator. Work done by any individual task is not necessarily on\ncontiguous values from the iterator. The given iterator may produce values forever, only the\niterator interface is required (no indexing).\n\nThis scheduling option is generally a good choice if the workload of individual iterations\nis not uniform/has a large spread.\n\ncompat: Julia 1.11\nThe :greedy option for the schedule argument is available as of Julia 1.11.\n\n:static\n\n:static scheduler creates one task per thread and divides the iterations equally among\nthem, assigning each task specifically to each thread. In particular, the value of\nthreadid() is guaranteed to be constant within one iteration.\nSpecifying :static is an error if used from inside another @threads loop or from a\nthread other than 1.\n\nnote: Note\n:static scheduling exists for supporting transition of code written before Julia 1.3.\nIn newly written library functions, :static scheduling is discouraged because the\nfunctions using this option cannot be called from arbitrary worker threads.\n\nExamples\n\nTo illustrate of the different scheduling strategies, consider the following function\nbusywait containing a non-yielding timed loop that runs for a given number of seconds.\n\njulia> function busywait(seconds)\n            tstart = time_ns()\n            while (time_ns() - tstart) / 1e9 < seconds\n            end\n        end\n\njulia> @time begin\n            Threads.@spawn busywait(5)\n            Threads.@threads :static for i in 1:Threads.threadpoolsize()\n                busywait(1)\n            end\n        end\n6.003001 seconds (16.33 k allocations: 899.255 KiB, 0.25% compilation time)\n\njulia> @time begin\n            Threads.@spawn busywait(5)\n            Threads.@threads :dynamic for i in 1:Threads.threadpoolsize()\n                busywait(1)\n            end\n        end\n2.012056 seconds (16.05 k allocations: 883.919 KiB, 0.66% compilation time)\n\nThe :dynamic example takes 2 seconds since one of the non-occupied threads is able\nto run two of the 1-second iterations to complete the for loop.\n\n\n\n\n\n"},{"title":"Base.Threads.foreach","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.foreach","category":"function","text":"Threads.foreach(f, channel::Channel;\n                schedule::Threads.AbstractSchedule=Threads.FairSchedule(),\n                ntasks=Threads.threadpoolsize())\n\nSimilar to foreach(f, channel), but iteration over channel and calls to\nf are split across ntasks tasks spawned by Threads.@spawn. This function\nwill wait for all internally spawned tasks to complete before returning.\n\nIf schedule isa FairSchedule, Threads.foreach will attempt to spawn tasks in a\nmanner that enables Julia's scheduler to more freely load-balance work items across\nthreads. This approach generally has higher per-item overhead, but may perform\nbetter than StaticSchedule in concurrence with other multithreaded workloads.\n\nIf schedule isa StaticSchedule, Threads.foreach will spawn tasks in a manner\nthat incurs lower per-item overhead than FairSchedule, but is less amenable\nto load-balancing. This approach thus may be more suitable for fine-grained,\nuniform workloads, but may perform worse than FairSchedule in concurrence\nwith other multithreaded workloads.\n\nExamples\n\njulia> n = 20\n\njulia> c = Channel{Int}(ch -> foreach(i -> put!(ch, i), 1:n), 1)\n\njulia> d = Channel{Int}(n) do ch\n           f = i -> put!(ch, i^2)\n           Threads.foreach(f, c)\n       end\n\njulia> collect(d)\ncollect(d) = [1, 4, 9, 16, 25, 36, 49, 64, 81, 100, 121, 144, 169, 196, 225, 256, 289, 324, 361, 400]\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n"},{"title":"Base.Threads.@spawn","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.@spawn","category":"macro","text":"Threads.@spawn [:default|:interactive|:samepool] expr\n\nCreate a Task and schedule it to run on any available\nthread in the specified threadpool: :default, :interactive, or :samepool\nto use the same as the caller. :default is used if unspecified. The task is\nallocated to a thread once one becomes available. To wait for the task to\nfinish, call wait on the result of this macro, or call\nfetch to wait and then obtain its return value.\n\nValues can be interpolated into @spawn via $, which copies the value\ndirectly into the constructed underlying closure. This allows you to insert\nthe value of a variable, isolating the asynchronous code from changes to\nthe variable's value in the current task.\n\nnote: Note\nThe thread that the task runs on may change if the task yields, therefore threadid() should not\nbe treated as constant for a task. See Task Migration, and the broader\nmulti-threading manual for further important caveats.\nSee also the chapter on threadpools.\n\ncompat: Julia 1.3\nThis macro is available as of Julia 1.3.\n\ncompat: Julia 1.4\nInterpolating values via $ is available as of Julia 1.4.\n\ncompat: Julia 1.9\nA threadpool may be specified as of Julia 1.9.\n\ncompat: Julia 1.12\nThe same threadpool may be specified as of Julia 1.12.\n\nExamples\n\njulia> t() = println(\"Hello from \", Threads.threadid());\n\njulia> tasks = fetch.([Threads.@spawn t() for i in 1:4]);\nHello from 1\nHello from 1\nHello from 3\nHello from 4\n\n\n\n\n\n"},{"title":"Base.Threads.threadid","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.threadid","category":"function","text":"Threads.threadid([t::Task])::Int\n\nGet the ID number of the current thread of execution, or the thread of task\nt. The master thread has ID 1.\n\nExamples\n\njulia> Threads.threadid()\n1\n\njulia> Threads.@threads for i in 1:4\n          println(Threads.threadid())\n       end\n4\n2\n5\n4\n\njulia> Threads.threadid(Threads.@spawn \"foo\")\n2\n\nnote: Note\nThe thread that a task runs on may change if the task yields, which is known as Task Migration.\nFor this reason in most cases it is not safe to use threadid([task]) to index into, say, a vector of buffers or stateful\nobjects.\n\n\n\n\n\n"},{"title":"Base.Threads.maxthreadid","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.maxthreadid","category":"function","text":"Threads.maxthreadid()::Int\n\nGet a lower bound on the number of threads (across all thread pools) available\nto the Julia process, with atomic-acquire semantics. The result will always be\ngreater than or equal to threadid() as well as threadid(task) for\nany task you were able to observe before calling maxthreadid.\n\n\n\n\n\n"},{"title":"Base.Threads.nthreads","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.nthreads","category":"function","text":"Threads.nthreads(:default | :interactive)::Int\n\nGet the current number of threads within the specified thread pool. The threads in :interactive\nhave id numbers 1:nthreads(:interactive), and the threads in :default have id numbers in\nnthreads(:interactive) .+ (1:nthreads(:default)).\n\nSee also BLAS.get_num_threads and BLAS.set_num_threads in the [LinearAlgebra](@ref\nman-linalg) standard library, and nprocs() in the Distributed\nstandard library and Threads.maxthreadid().\n\n\n\n\n\n"},{"title":"Base.Threads.threadpool","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.threadpool","category":"function","text":"Threads.threadpool(tid = threadid())::Symbol\n\nReturn the specified thread's threadpool; either :default, :interactive, or :foreign.\n\n\n\n\n\n"},{"title":"Base.Threads.nthreadpools","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.nthreadpools","category":"function","text":"Threads.nthreadpools()::Int\n\nReturn the number of threadpools currently configured.\n\n\n\n\n\n"},{"title":"Base.Threads.threadpoolsize","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.threadpoolsize","category":"function","text":"Threads.threadpoolsize(pool::Symbol = :default)::Int\n\nGet the number of threads available to the default thread pool (or to the\nspecified thread pool).\n\nSee also: BLAS.get_num_threads and BLAS.set_num_threads in the\nLinearAlgebra standard library, and nprocs() in the\nDistributed standard library.\n\n\n\n\n\n"},{"title":"Base.Threads.ngcthreads","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.ngcthreads","category":"function","text":"Threads.ngcthreads()::Int\n\nReturn the number of GC threads currently configured.\nThis includes both mark threads and concurrent sweep threads.\n\n\n\n\n\n"},{"title":"atomic","page":"Multi-Threading","location":"base/multi-threading.html#atomic","category":"keyword","text":"Unsafe pointer operations are compatible with loading and storing pointers declared with\n_Atomic and std::atomic type in C11 and C++23 respectively. An error may be thrown if\nthere is not support for atomically loading the Julia type T.\n\nSee also: unsafe_load, unsafe_modify!, unsafe_replace!, unsafe_store!, unsafe_swap!\n\n\n\n\n\n"},{"title":"Base.@atomic","page":"Multi-Threading","location":"base/multi-threading.html#Base.@atomic","category":"macro","text":"@atomic var\n@atomic order ex\n\nMark var or ex as being performed atomically, if ex is a supported expression.\nIf no order is specified it defaults to :sequentially_consistent.\n\n@atomic a.b.x = new\n@atomic a.b.x += addend\n@atomic :release a.b.x = new\n@atomic :acquire_release a.b.x += addend\n@atomic m[idx] = new\n@atomic m[idx] += addend\n@atomic :release m[idx] = new\n@atomic :acquire_release m[idx] += addend\n\nPerform the store operation expressed on the right atomically and return the\nnew value.\n\nWith assignment (=), this operation translates to a setproperty!(a.b, :x, new)\nor, in case of reference, to a setindex_atomic!(m, order, new, idx) call,\nwith order defaulting to :sequentially_consistent.\n\nWith any modifying operator this operation translates to a\nmodifyproperty!(a.b, :x, op, addend)[2] or, in case of reference, to a\nmodifyindex_atomic!(m, order, op, addend, idx...)[2] call,\nwith order defaulting to :sequentially_consistent.\n\n@atomic a.b.x max arg2\n@atomic a.b.x + arg2\n@atomic max(a.b.x, arg2)\n@atomic :acquire_release max(a.b.x, arg2)\n@atomic :acquire_release a.b.x + arg2\n@atomic :acquire_release a.b.x max arg2\n@atomic m[idx] max arg2\n@atomic m[idx] + arg2\n@atomic max(m[idx], arg2)\n@atomic :acquire_release max(m[idx], arg2)\n@atomic :acquire_release m[idx] + arg2\n@atomic :acquire_release m[idx] max arg2\n\nPerform the binary operation expressed on the right atomically. Store the\nresult into the field or the reference in the first argument, and return the values\n(old, new).\n\nThis operation translates to a modifyproperty!(a.b, :x, func, arg2) or,\nin case of reference to a modifyindex_atomic!(m, order, func, arg2, idx) call,\nwith order defaulting to :sequentially_consistent.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n1\n\njulia> @atomic :sequentially_consistent a.x = 2 # set field x of a, with sequential consistency\n2\n\njulia> @atomic a.x += 1 # increment field x of a, with sequential consistency\n3\n\njulia> @atomic a.x + 1 # increment field x of a, with sequential consistency\n3 => 4\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n4\n\njulia> @atomic max(a.x, 10) # change field x of a to the max value, with sequential consistency\n4 => 10\n\njulia> @atomic a.x max 5 # again change field x of a to the max value, with sequential consistency\n10 => 10\n\njulia> mem = AtomicMemory{Int}(undef, 2);\n\njulia> @atomic mem[1] = 2 # set mem[1] to value 2 with sequential consistency\n2\n\njulia> @atomic :monotonic mem[1] # fetch the first value of mem, with monotonic consistency\n2\n\njulia> @atomic mem[1] += 1 # increment the first value of mem, with sequential consistency\n3\n\njulia> @atomic mem[1] + 1 # increment the first value of mem, with sequential consistency\n3 => 4\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n4\n\njulia> @atomic max(mem[1], 10) # change the first value of mem to the max value, with sequential consistency\n4 => 10\n\njulia> @atomic mem[1] max 5 # again change the first value of mem to the max value, with sequential consistency\n10 => 10\n\ncompat: Julia 1.7\nAtomic fields functionality requires at least Julia 1.7.\n\ncompat: Julia 1.12\nAtomic reference functionality requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.@atomicswap","page":"Multi-Threading","location":"base/multi-threading.html#Base.@atomicswap","category":"macro","text":"@atomicswap a.b.x = new\n@atomicswap :sequentially_consistent a.b.x = new\n@atomicswap m[idx] = new\n@atomicswap :sequentially_consistent m[idx] = new\n\nStores new into a.b.x (m[idx] in case of reference) and returns the old\nvalue of a.b.x (the old value stored at m[idx], respectively).\n\nThis operation translates to a swapproperty!(a.b, :x, new) or,\nin case of reference, swapindex_atomic!(mem, order, new, idx) call,\nwith order defaulting to :sequentially_consistent.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomicswap a.x = 2+2 # replace field x of a with 4, with sequential consistency\n1\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n4\n\njulia> mem = AtomicMemory{Int}(undef, 2);\n\njulia> @atomic mem[1] = 1;\n\njulia> @atomicswap mem[1] = 4 # replace the first value of `mem` with 4, with sequential consistency\n1\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n4\n\ncompat: Julia 1.7\nAtomic fields functionality requires at least Julia 1.7.\n\ncompat: Julia 1.12\nAtomic reference functionality requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.@atomicreplace","page":"Multi-Threading","location":"base/multi-threading.html#Base.@atomicreplace","category":"macro","text":"@atomicreplace a.b.x expected => desired\n@atomicreplace :sequentially_consistent a.b.x expected => desired\n@atomicreplace :sequentially_consistent :monotonic a.b.x expected => desired\n@atomicreplace m[idx] expected => desired\n@atomicreplace :sequentially_consistent m[idx] expected => desired\n@atomicreplace :sequentially_consistent :monotonic m[idx] expected => desired\n\nPerform the conditional replacement expressed by the pair atomically, returning\nthe values (old, success::Bool). Where success indicates whether the\nreplacement was completed.\n\nThis operation translates to a replaceproperty!(a.b, :x, expected, desired) or,\nin case of reference, to a\nreplaceindex_atomic!(mem, success_order, fail_order, expected, desired, idx) call,\nwith both orders defaulting to :sequentially_consistent.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomicreplace a.x 1 => 2 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 1, success = true)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n2\n\njulia> @atomicreplace a.x 1 => 3 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 2, success = false)\n\njulia> xchg = 2 => 0; # replace field x of a with 0 if it was 2, with sequential consistency\n\njulia> @atomicreplace a.x xchg\n(old = 2, success = true)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n0\n\njulia> mem = AtomicMemory{Int}(undef, 2);\n\njulia> @atomic mem[1] = 1;\n\njulia> @atomicreplace mem[1] 1 => 2 # replace the first value of mem with 2 if it was 1, with sequential consistency\n(old = 1, success = true)\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n2\n\njulia> @atomicreplace mem[1] 1 => 3 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 2, success = false)\n\njulia> xchg = 2 => 0; # replace field x of a with 0 if it was 2, with sequential consistency\n\njulia> @atomicreplace mem[1] xchg\n(old = 2, success = true)\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n0\n\ncompat: Julia 1.7\nAtomic fields functionality requires at least Julia 1.7.\n\ncompat: Julia 1.12\nAtomic reference functionality requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.@atomiconce","page":"Multi-Threading","location":"base/multi-threading.html#Base.@atomiconce","category":"macro","text":"@atomiconce a.b.x = value\n@atomiconce :sequentially_consistent a.b.x = value\n@atomiconce :sequentially_consistent :monotonic a.b.x = value\n@atomiconce m[idx] = value\n@atomiconce :sequentially_consistent m[idx] = value\n@atomiconce :sequentially_consistent :monotonic m[idx] = value\n\nPerform the conditional assignment of value atomically if it was previously\nunset. Returned value success::Bool indicates whether the assignment was completed.\n\nThis operation translates to a setpropertyonce!(a.b, :x, value) or,\nin case of reference, to a setindexonce_atomic!(m, success_order, fail_order, value, idx) call,\nwith both orders defaulting to :sequentially_consistent.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct AtomicOnce\n           @atomic x\n           AtomicOnce() = new()\n       end\n\njulia> a = AtomicOnce()\nAtomicOnce(#undef)\n\njulia> @atomiconce a.x = 1 # set field x of a to 1, if unset, with sequential consistency\ntrue\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n1\n\njulia> @atomiconce :monotonic a.x = 2 # set field x of a to 1, if unset, with monotonic consistence\nfalse\n\njulia> mem = AtomicMemory{Vector{Int}}(undef, 1);\n\njulia> isassigned(mem, 1)\nfalse\n\njulia> @atomiconce mem[1] = [1] # set the first value of mem to [1], if unset, with sequential consistency\ntrue\n\njulia> isassigned(mem, 1)\ntrue\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n1-element Vector{Int64}:\n 1\n\njulia> @atomiconce :monotonic mem[1] = [2] # set the first value of mem to [2], if unset, with monotonic\nfalse\n\njulia> @atomic mem[1]\n1-element Vector{Int64}:\n 1\n\ncompat: Julia 1.11\nAtomic fields functionality requires at least Julia 1.11.\n\ncompat: Julia 1.12\nAtomic reference functionality requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Core.AtomicMemory","page":"Multi-Threading","location":"base/multi-threading.html#Core.AtomicMemory","category":"type","text":"AtomicMemory{T} == GenericMemory{:atomic, T, Core.CPU}\n\nFixed-size DenseVector{T}.\nFetching of any of its individual elements is performed atomically\n(with :monotonic ordering by default).\n\nwarning: Warning\nThe access to AtomicMemory must be done by either using the @atomic\nmacro or the lower level interface functions: Base.getindex_atomic,\nBase.setindex_atomic!, Base.setindexonce_atomic!,\nBase.swapindex_atomic!, Base.modifyindex_atomic!, and Base.replaceindex_atomic!.\n\nFor details, see Atomic Operations as well as macros\n@atomic, @atomiconce, @atomicswap, and @atomicreplace.\n\ncompat: Julia 1.11\nThis type requires Julia 1.11 or later.\n\ncompat: Julia 1.12\nLower level interface functions or @atomic macro requires Julia 1.12 or later.\n\n\n\n\n\n"},{"title":"Base.Threads.Atomic","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.Atomic","category":"type","text":"Threads.Atomic{T}\n\nHolds a reference to an object of type T, ensuring that it is only\naccessed atomically, i.e. in a thread-safe manner.\n\nNew atomic objects can be created from a non-atomic values; if none is\nspecified, the atomic object is initialized with zero.\n\nAtomic objects can be accessed using the [] notation:\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> x[] = 1\n1\n\njulia> x[]\n1\n\nAtomic operations use an atomic_ prefix, such as atomic_add!,\natomic_xchg!, etc.\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_cas!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_cas!","category":"function","text":"Threads.atomic_cas!(x::Atomic{T}, cmp::T, newval::T) where T\n\nAtomically compare-and-set x\n\nAtomically compares the value in x with cmp. If equal, write\nnewval to x. Otherwise, leaves x unmodified. Returns the old\nvalue in x. By comparing the returned value to cmp (via ===) one\nknows whether x was modified and now holds the new value newval.\n\nFor further details, see LLVM's cmpxchg instruction.\n\nThis function can be used to implement transactional semantics. Before\nthe transaction, one records the value in x. After the transaction,\nthe new value is stored only if x has not been modified in the mean\ntime.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_cas!(x, 4, 2);\n\njulia> x\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_cas!(x, 3, 2);\n\njulia> x\nBase.Threads.Atomic{Int64}(2)\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_xchg!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_xchg!","category":"function","text":"Threads.atomic_xchg!(x::Atomic{T}, newval::T) where T\n\nAtomically exchange the value in x\n\nAtomically exchanges the value in x with newval. Returns the old\nvalue.\n\nFor further details, see LLVM's atomicrmw xchg instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_xchg!(x, 2)\n3\n\njulia> x[]\n2\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_add!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_add!","category":"function","text":"Threads.atomic_add!(x::Atomic{T}, val::T) where T <: ArithmeticTypes\n\nAtomically add val to x\n\nPerforms x[] += val atomically. Returns the old value. Not defined for\nAtomic{Bool}.\n\nFor further details, see LLVM's atomicrmw add instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_add!(x, 2)\n3\n\njulia> x[]\n5\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_sub!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_sub!","category":"function","text":"Threads.atomic_sub!(x::Atomic{T}, val::T) where T <: ArithmeticTypes\n\nAtomically subtract val from x\n\nPerforms x[] -= val atomically. Returns the old value. Not defined for\nAtomic{Bool}.\n\nFor further details, see LLVM's atomicrmw sub instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_sub!(x, 2)\n3\n\njulia> x[]\n1\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_and!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_and!","category":"function","text":"Threads.atomic_and!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-and x with val\n\nPerforms x[] &= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw and instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_and!(x, 2)\n3\n\njulia> x[]\n2\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_nand!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_nand!","category":"function","text":"Threads.atomic_nand!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-nand (not-and) x with val\n\nPerforms x[] = ~(x[] & val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw nand instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_nand!(x, 2)\n3\n\njulia> x[]\n-3\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_or!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_or!","category":"function","text":"Threads.atomic_or!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-or x with val\n\nPerforms x[] |= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw or instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_or!(x, 7)\n5\n\njulia> x[]\n7\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_xor!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_xor!","category":"function","text":"Threads.atomic_xor!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-xor (exclusive-or) x with val\n\nPerforms x[] $= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw xor instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_xor!(x, 7)\n5\n\njulia> x[]\n2\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_max!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_max!","category":"function","text":"Threads.atomic_max!(x::Atomic{T}, val::T) where T\n\nAtomically store the maximum of x and val in x\n\nPerforms x[] = max(x[], val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw max instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_max!(x, 7)\n5\n\njulia> x[]\n7\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_min!","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_min!","category":"function","text":"Threads.atomic_min!(x::Atomic{T}, val::T) where T\n\nAtomically store the minimum of x and val in x\n\nPerforms x[] = min(x[], val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw min instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(7)\nBase.Threads.Atomic{Int64}(7)\n\njulia> Threads.atomic_min!(x, 5)\n7\n\njulia> x[]\n5\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_fence","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_fence","category":"function","text":"Threads.atomic_fence()\n\nInsert a sequential-consistency memory fence\n\nInserts a memory fence with sequentially-consistent ordering\nsemantics. There are algorithms where this is needed, i.e. where an\nacquire/release ordering is insufficient.\n\nThis is likely a very expensive operation. Given that all other atomic\noperations in Julia already have acquire/release semantics, explicit\nfences should not be necessary in most cases.\n\nFor further details, see LLVM's fence instruction.\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_fence_heavy","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_fence_heavy","category":"function","text":"Threads.atomic_fence_heavy()\n\nInsert the heavy side of an asymmetric sequential-consistency memory fence.\nUse this function on the side that runs rarely.\nSee atomic_fence_light for more details.\n\n\n\n\n\n"},{"title":"Base.Threads.atomic_fence_light","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.atomic_fence_light","category":"function","text":"Threads.atomic_fence_light()\n\nInsert the light side of an asymmetric sequential-consistency memory fence.\nAsymmetric memory fences are useful in scenarios where one side of the\nsynchronization runs significantly less often than the other side. Use this\nfunction on the side that runs often and atomic_fence_heavy on the\nside that runs rarely.\n\nOn supported operating systems and architectures this fence is cheaper than\nThreads.atomic_fence(), but synchronizes only with atomic_fence_heavy\ncalls from other threads.\n\n\n\n\n\n"},{"title":"Base.@threadcall","page":"Multi-Threading","location":"base/multi-threading.html#Base.@threadcall","category":"macro","text":"@threadcall((cfunc, clib), rettype, (argtypes...), argvals...)\n\nThe @threadcall macro is called in the same way as ccall but does the work\nin a different thread. This is useful when you want to call a blocking C\nfunction without causing the current julia thread to become blocked. Concurrency\nis limited by size of the libuv thread pool, which defaults to 4 threads but\ncan be increased by setting the UV_THREADPOOL_SIZE environment variable and\nrestarting the julia process.\n\nNote that the called function should never call back into Julia.\n\n\n\n\n\n"},{"title":"Base.Threads.AbstractSpinLock","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.AbstractSpinLock","category":"type","text":"abstract type AbstractSpinLock <: AbstractLock end\n\nA non-reentrant, test-and-test-and-set spin lock.\nRecursive use will result in a deadlock.\nThis kind of lock should only be used around code that takes little time\nto execute and does not block (e.g. perform I/O).\nIn general, ReentrantLock should be used instead.\n\nEach lock must be matched with an unlock.\nIf !islocked(lck::AbstractSpinLock) holds, trylock(lck)\nsucceeds unless there are other tasks attempting to hold the lock \"at the same time.\"\n\nTest-and-test-and-set spin locks are quickest up to about 30ish\ncontending threads. If you have more contention than that, different\nsynchronization approaches should be considered.\n\n\n\n\n\n"},{"title":"Base.Threads.SpinLock","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.SpinLock","category":"type","text":"SpinLock() <: AbstractSpinLock\n\nSpinlocks are not padded, and so may suffer from false sharing.\nSee also PaddedSpinLock.\n\nSee the documentation for AbstractSpinLock regarding correct usage.\n\n\n\n\n\n"},{"title":"Base.Threads.PaddedSpinLock","page":"Multi-Threading","location":"base/multi-threading.html#Base.Threads.PaddedSpinLock","category":"type","text":"PaddedSpinLock() <: AbstractSpinLock\n\nPaddedSpinLocks are padded so that each is guaranteed to be on its own cache line, to avoid\nfalse sharing.\nSee also SpinLock.\n\nSee the documentation for AbstractSpinLock regarding correct usage.\n\n\n\n\n\n"},{"title":"Base.Experimental.task_metrics","page":"Multi-Threading","location":"base/multi-threading.html#Base.Experimental.task_metrics","category":"function","text":"Base.Experimental.task_metrics(::Bool)\n\nEnable or disable the collection of per-task metrics.\nA Task created when Base.Experimental.task_metrics(true) is in effect will have\nBase.Experimental.task_running_time_ns and Base.Experimental.task_wall_time_ns\ntiming information available.\n\nnote: Note\nTask metrics can be enabled at start-up via the --task-metrics=yes command line option.\n\n\n\n\n\n"},{"title":"Base.Experimental.task_running_time_ns","page":"Multi-Threading","location":"base/multi-threading.html#Base.Experimental.task_running_time_ns","category":"function","text":"Base.Experimental.task_running_time_ns(t::Task)::Union{UInt64, Nothing}\n\nReturn the total nanoseconds that the task t has spent running.\nThis metric is only updated when t yields or completes unless t is the current task, in\nwhich it will be updated continuously.\nSee also Base.Experimental.task_wall_time_ns.\n\nReturn nothing if task timings are not enabled.\nSee Base.Experimental.task_metrics.\n\nnote: This metric is from the Julia scheduler\nA task may be running on an OS thread that is descheduled by the OS\nscheduler, this time still counts towards the metric.\n\ncompat: Julia 1.12\nThis method was added in Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.Experimental.task_wall_time_ns","page":"Multi-Threading","location":"base/multi-threading.html#Base.Experimental.task_wall_time_ns","category":"function","text":"Base.Experimental.task_wall_time_ns(t::Task)::Union{UInt64, Nothing}\n\nReturn the total nanoseconds that the task t was runnable.\nThis is the time since the task first entered the run queue until the time at which it\ncompleted, or until the current time if the task has not yet completed.\nSee also Base.Experimental.task_running_time_ns.\n\nReturn nothing if task timings are not enabled.\nSee Base.Experimental.task_metrics.\n\ncompat: Julia 1.12\nThis method was added in Julia 1.12.\n\n\n\n\n\n"},{"title":"Metaprogramming","page":"Metaprogramming","location":"manual/metaprogramming.html#Metaprogramming","category":"section","text":"The strongest legacy of Lisp in the Julia language is its metaprogramming support. Like Lisp,\nJulia represents its own code as a data structure of the language itself. Since code is represented\nby objects that can be created and manipulated from within the language, it is possible for a\nprogram to transform and generate its own code. This allows sophisticated code generation without\nextra build steps, and also allows true Lisp-style macros operating at the level of abstract syntax trees.\nIn contrast, preprocessor \"macro\" systems, like that of C and C++, perform textual manipulation\nand substitution before any actual parsing or interpretation occurs. Because all data types and\ncode in Julia are represented by Julia data structures, powerful reflection\ncapabilities are available to explore the internals of a program and its types just like any other\ndata.\n\nwarning: Warning\nMetaprogramming is a powerful tool, but it introduces complexity that can make code more\ndifficult to understand. For example, it can be surprisingly hard to get scope rules\ncorrect. Metaprogramming should typically be used only when other approaches such as\nhigher order functions and\nclosures cannot be applied.eval and defining new macros should be typically used as a last resort. It is almost\nnever a good idea to use Meta.parse or convert an arbitrary string into Julia code. For\nmanipulating Julia code, use the Expr data structure directly to avoid the complexity\nof how Julia syntax is parsed.The best uses of metaprogramming often implement most of their functionality in runtime\nhelper functions, striving to minimize the amount of code they generate."},{"title":"Program representation","page":"Metaprogramming","location":"manual/metaprogramming.html#Program-representation","category":"section","text":"Every Julia program starts life as a string:\n\njulia> prog = \"1 + 1\"\n\"1 + 1\"\n\nWhat happens next?\n\nThe next step is to parse each string\ninto an object called an expression, represented by the Julia type Expr:\n\njulia> ex1 = Meta.parse(prog)\n:(1 + 1)\n\njulia> typeof(ex1)\nExpr\n\nExpr objects contain two parts:\n\na Symbol identifying the kind of expression. A symbol is an interned string\nidentifier (more discussion below).\n\njulia> ex1.head\n:call\n\nthe expression arguments, which may be symbols, other expressions, or literal values:\n\njulia> ex1.args\n3-element Vector{Any}:\n  :+\n 1\n 1\n\nExpressions may also be constructed directly in prefix notation:\n\njulia> ex2 = Expr(:call, :+, 1, 1)\n:(1 + 1)\n\nThe two expressions constructed above – by parsing and by direct construction – are equivalent:\n\njulia> ex1 == ex2\ntrue\n\nThe key point here is that Julia code is internally represented as a data structure that is accessible\nfrom the language itself.\n\nThe dump function provides indented and annotated display of Expr objects:\n\njulia> dump(ex2)\nExpr\n  head: Symbol call\n  args: Array{Any}((3,))\n    1: Symbol +\n    2: Int64 1\n    3: Int64 1\n\nExpr objects may also be nested:\n\njulia> ex3 = Meta.parse(\"(4 + 4) / 2\")\n:((4 + 4) / 2)\n\nAnother way to view expressions is with Meta.show_sexpr, which displays the S-expression\nform of a given Expr, which may look very familiar to users of Lisp. Here's an example illustrating\nthe display on a nested Expr:\n\njulia> Meta.show_sexpr(ex3)\n(:call, :/, (:call, :+, 4, 4), 2)"},{"title":"Symbols","page":"Metaprogramming","location":"manual/metaprogramming.html#Symbols","category":"section","text":"The : character has two syntactic purposes in Julia. The first form creates a Symbol,\nan interned string used as one building-block\nof expressions, from valid identifiers:\n\njulia> s = :foo\n:foo\n\njulia> typeof(s)\nSymbol\n\nThe Symbol constructor takes any number of arguments and creates a new symbol by concatenating\ntheir string representations together:\n\njulia> :foo === Symbol(\"foo\")\ntrue\n\njulia> Symbol(\"1foo\") # `:1foo` would not work, as `1foo` is not a valid identifier\nSymbol(\"1foo\")\n\njulia> Symbol(\"func\",10)\n:func10\n\njulia> Symbol(:var,'_',\"sym\")\n:var_sym\n\nIn the context of an expression, symbols are used to indicate access to variables; when an expression\nis evaluated, a symbol is replaced with the value bound to that symbol in the appropriate scope.\n\nSometimes extra parentheses around the argument to : are needed to avoid ambiguity in parsing:\n\njulia> :(:)\n:(:)\n\njulia> :(::)\n:(::)"},{"title":"Expressions and evaluation","page":"Metaprogramming","location":"manual/metaprogramming.html#Expressions-and-evaluation","category":"section","text":""},{"title":"Quoting","page":"Metaprogramming","location":"manual/metaprogramming.html#Quoting","category":"section","text":"The second syntactic purpose of the : character is to create expression objects without using\nthe explicit Expr constructor. This is referred to as quoting. The : character, followed\nby paired parentheses around a single statement of Julia code, produces an Expr object based\non the enclosed code. Here is an example of the short form used to quote an arithmetic expression:\n\njulia> ex = :(a+b*c+1)\n:(a + b * c + 1)\n\njulia> typeof(ex)\nExpr\n\n(to view the structure of this expression, try ex.head and ex.args, or use dump\nas above or Meta.@dump)\n\nNote that equivalent expressions may be constructed using Meta.parse or the direct Expr\nform:\n\njulia>      :(a + b*c + 1)       ==\n       Meta.parse(\"a + b*c + 1\") ==\n       Expr(:call, :+, :a, Expr(:call, :*, :b, :c), 1)\ntrue\n\nExpressions provided by the parser generally only have symbols, other expressions, and literal\nvalues as their args, whereas expressions constructed by Julia code can have arbitrary run-time\nvalues without literal forms as args. In this specific example, + and a are symbols, *(b,c)\nis a subexpression, and 1 is a literal 64-bit signed integer.\n\nThere is a second syntactic form of quoting for multiple expressions: blocks of code enclosed\nin quote ... end.\n\njulia> ex = quote\n           x = 1\n           y = 2\n           x + y\n       end\nquote\n    #= none:2 =#\n    x = 1\n    #= none:3 =#\n    y = 2\n    #= none:4 =#\n    x + y\nend\n\njulia> typeof(ex)\nExpr"},{"title":"Interpolation","page":"Metaprogramming","location":"manual/metaprogramming.html#man-expression-interpolation","category":"section","text":"Direct construction of Expr objects with value arguments is powerful, but Expr constructors\ncan be tedious compared to \"normal\" Julia syntax. As an alternative, Julia allows interpolation of\nliterals or expressions into quoted expressions. Interpolation is indicated by a prefix $.\n\nIn this example, the value of variable a is interpolated:\n\njulia> a = 1;\n\njulia> ex = :($a + b)\n:(1 + b)\n\nInterpolating into an unquoted expression is not supported and will cause a compile-time error:\n\njulia> $a + b\nERROR: syntax: \"$\" expression outside quote\n\nIn this example, the tuple (1,2,3) is interpolated as an expression into a conditional test:\n\njulia> ex = :(a in $:((1,2,3)) )\n:(a in (1, 2, 3))\n\nThe use of $ for expression interpolation is intentionally reminiscent of string interpolation\nand command interpolation. Expression interpolation allows convenient, readable programmatic\nconstruction of complex Julia expressions."},{"title":"Splatting interpolation","page":"Metaprogramming","location":"manual/metaprogramming.html#Splatting-interpolation","category":"section","text":"Notice that the $ interpolation syntax allows inserting only a single expression into an\nenclosing expression.\nOccasionally, you have an array of expressions and need them all to become arguments of\nthe surrounding expression.\nThis can be done with the syntax $(xs...).\nFor example, the following code generates a function call where the number of arguments is\ndetermined programmatically:\n\njulia> args = [:x, :y, :z];\n\njulia> :(f(1, $(args...)))\n:(f(1, x, y, z))"},{"title":"Nested quote","page":"Metaprogramming","location":"manual/metaprogramming.html#Nested-quote","category":"section","text":"Naturally, it is possible for quote expressions to contain other quote expressions.\nUnderstanding how interpolation works in these cases can be a bit tricky.\nConsider this example:\n\njulia> x = :(1 + 2);\n\njulia> e = quote quote $x end end\nquote\n    #= none:1 =#\n    $(Expr(:quote, quote\n    #= none:1 =#\n    $(Expr(:$, :x))\nend))\nend\n\nNotice that the result contains $x, which means that x has not been\nevaluated yet.\nIn other words, the $ expression \"belongs to\" the inner quote expression, and\nso its argument is only evaluated when the inner quote expression is:\n\njulia> eval(e)\nquote\n    #= none:1 =#\n    1 + 2\nend\n\nHowever, the outer quote expression is able to interpolate values inside the $\nin the inner quote.\nThis is done with multiple $s:\n\njulia> e = quote quote $$x end end\nquote\n    #= none:1 =#\n    $(Expr(:quote, quote\n    #= none:1 =#\n    $(Expr(:$, :(1 + 2)))\nend))\nend\n\nNotice that (1 + 2) now appears in the result instead of the symbol x.\nEvaluating this expression yields an interpolated 3:\n\njulia> eval(e)\nquote\n    #= none:1 =#\n    3\nend\n\nThe intuition behind this behavior is that x is evaluated once for each $:\none $ works similarly to eval(:x), giving x's value, while two $s do the\nequivalent of eval(eval(:x))."},{"title":"QuoteNode","page":"Metaprogramming","location":"manual/metaprogramming.html#man-quote-node","category":"section","text":"The usual representation of a quote form in an AST is an Expr with head :quote:\n\njulia> dump(Meta.parse(\":(1+2)\"))\nExpr\n  head: Symbol quote\n  args: Array{Any}((1,))\n    1: Expr\n      head: Symbol call\n      args: Array{Any}((3,))\n        1: Symbol +\n        2: Int64 1\n        3: Int64 2\n\nAs we have seen, such expressions support interpolation with $.\nHowever, in some situations it is necessary to quote code without performing interpolation.\nThis kind of quoting does not yet have syntax, but is represented internally\nas an object of type QuoteNode:\n\njulia> eval(Meta.quot(Expr(:$, :(1+2))))\n3\n\njulia> eval(QuoteNode(Expr(:$, :(1+2))))\n:($(Expr(:$, :(1 + 2))))\n\nThe parser yields QuoteNodes for simple quoted items like symbols:\n\njulia> dump(Meta.parse(\":x\"))\nQuoteNode\n  value: Symbol x\n\nQuoteNode can also be used for certain advanced metaprogramming tasks.\n\nNote that while it does not support $, it also does not prevent it, nor does\nit preserve the identity of the wrapped object:\n\njulia> b = 2; eval(Expr(:quote, QuoteNode(Expr(:$, :b))))\n:($(QuoteNode(2)))"},{"title":"Evaluating expressions","page":"Metaprogramming","location":"manual/metaprogramming.html#Evaluating-expressions","category":"section","text":"Given an expression object, one can cause Julia to evaluate (execute) it at global scope using\neval:\n\njulia> ex1 = :(1 + 2)\n:(1 + 2)\n\njulia> eval(ex1)\n3\n\njulia> ex = :(a + b)\n:(a + b)\n\njulia> eval(ex)\nERROR: UndefVarError: `b` not defined in `Main`\n[...]\n\njulia> a = 1; b = 2;\n\njulia> eval(ex)\n3\n\nEvery module has its own eval function that evaluates expressions in its global\nscope. Expressions passed to eval are not limited to returning values – they can\nalso have side-effects that alter the state of the enclosing module's environment:\n\njulia> ex = :(x = 1)\n:(x = 1)\n\njulia> x\nERROR: UndefVarError: `x` not defined in `Main`\n\njulia> eval(ex)\n1\n\njulia> x\n1\n\nHere, the evaluation of an expression object causes a value to be assigned to the global variable\nx.\n\nSince expressions are just Expr objects which can be constructed programmatically and then evaluated,\nit is possible to dynamically generate arbitrary code which can then be run using eval.\nHere is a simple example:\n\njulia> a = 1;\n\njulia> ex = Expr(:call, :+, a, :b)\n:(1 + b)\n\njulia> a = 0; b = 2;\n\njulia> eval(ex)\n3\n\nThe value of a is used to construct the expression ex which applies the + function to the\nvalue 1 and the variable b. Note the important distinction between the way a and b are used:\n\nThe value of the variable a at expression construction time is used as an immediate value in\nthe expression. Thus, the value of a when the expression is evaluated no longer matters: the\nvalue in the expression is already 1, independent of whatever the value of a might be.\nOn the other hand, the symbol :b is used in the expression construction, so the value of the\nvariable b at that time is irrelevant – :b is just a symbol and the variable b need not\neven be defined. At expression evaluation time, however, the value of the symbol :b is resolved\nby looking up the value of the variable b."},{"title":"Functions on Expressions","page":"Metaprogramming","location":"manual/metaprogramming.html#Functions-on-Expressions","category":"section","text":"As hinted above, one extremely useful feature of Julia is the capability to generate and manipulate\nJulia code within Julia itself. We have already seen one example of a function returning Expr\nobjects: the Meta.parse function, which takes a string of Julia code and returns the corresponding\nExpr. A function can also take one or more Expr objects as arguments, and return another\nExpr. Here is a simple, motivating example:\n\njulia> function math_expr(op, op1, op2)\n           expr = Expr(:call, op, op1, op2)\n           return expr\n       end\nmath_expr (generic function with 1 method)\n\njulia>  ex = math_expr(:+, 1, Expr(:call, :*, 4, 5))\n:(1 + 4 * 5)\n\njulia> eval(ex)\n21\n\nAs another example, here is a function that doubles any numeric argument, but leaves expressions\nalone:\n\njulia> function make_expr2(op, opr1, opr2)\n           opr1f, opr2f = map(x -> isa(x, Number) ? 2*x : x, (opr1, opr2))\n           retexpr = Expr(:call, op, opr1f, opr2f)\n           return retexpr\n       end\nmake_expr2 (generic function with 1 method)\n\njulia> make_expr2(:+, 1, 2)\n:(2 + 4)\n\njulia> ex = make_expr2(:+, 1, Expr(:call, :*, 5, 8))\n:(2 + 5 * 8)\n\njulia> eval(ex)\n42"},{"title":"Macros","page":"Metaprogramming","location":"manual/metaprogramming.html#man-macros","category":"section","text":"Macros provide a mechanism to include generated code in the final body of a program. A macro maps\na tuple of arguments to a returned expression, and the resulting expression is compiled directly\nrather than requiring a runtime eval call. Macro arguments may include expressions,\nliteral values, and symbols."},{"title":"Basics","page":"Metaprogramming","location":"manual/metaprogramming.html#Basics","category":"section","text":"Here is an extraordinarily simple macro:\n\njulia> macro sayhello()\n           return :( println(\"Hello, world!\") )\n       end\n@sayhello (macro with 1 method)\n\nMacros have a dedicated character in Julia's syntax: the @ (at-sign), followed by the unique\nname declared in a macro NAME ... end block. In this example, the compiler will replace all\ninstances of @sayhello with:\n\n:( println(\"Hello, world!\") )\n\nWhen @sayhello is entered in the REPL, the expression executes immediately, thus we only see the\nevaluation result:\n\njulia> @sayhello()\nHello, world!\n\nNow, consider a slightly more complex macro:\n\njulia> macro sayhello(name)\n           return :( println(\"Hello, \", $name) )\n       end\n@sayhello (macro with 1 method)\n\nThis macro takes one argument: name. When @sayhello is encountered, the quoted expression\nis expanded to interpolate the value of the argument into the final expression:\n\njulia> @sayhello(\"human\")\nHello, human\n\nWe can view the quoted return expression using the function macroexpand (important note:\nthis is an extremely useful tool for debugging macros):\n\njulia> ex = macroexpand(Main, :(@sayhello(\"human\")) )\n:(Main.println(\"Hello, \", \"human\"))\n\njulia> typeof(ex)\nExpr\n\nWe can see that the \"human\" literal has been interpolated into the expression.\n\nThere also exists a macro @macroexpand that is perhaps a bit more convenient than the macroexpand function:\n\njulia> @macroexpand @sayhello \"human\"\n:(println(\"Hello, \", \"human\"))"},{"title":"Hold up: why macros?","page":"Metaprogramming","location":"manual/metaprogramming.html#Hold-up:-why-macros?","category":"section","text":"We have already seen a function f(::Expr...) -> Expr in a previous section. In fact, macroexpand\nis also such a function. So, why do macros exist?\n\nMacros are necessary because they execute when code is parsed, therefore, macros allow the programmer\nto generate and include fragments of customized code before the full program is run. To illustrate\nthe difference, consider the following example:\n\njulia> macro twostep(arg)\n           println(\"I execute at parse time. The argument is: \", arg)\n           return :(println(\"I execute at runtime. The argument is: \", $arg))\n       end\n@twostep (macro with 1 method)\n\njulia> ex = macroexpand(Main, :(@twostep :(1, 2, 3)) );\nI execute at parse time. The argument is: :((1, 2, 3))\n\nThe first call to println is executed when macroexpand is called. The\nresulting expression contains only the second println:\n\njulia> typeof(ex)\nExpr\n\njulia> ex\n:(println(\"I execute at runtime. The argument is: \", $(Expr(:copyast, :($(QuoteNode(:((1, 2, 3)))))))))\n\njulia> eval(ex)\nI execute at runtime. The argument is: (1, 2, 3)"},{"title":"Macro invocation","page":"Metaprogramming","location":"manual/metaprogramming.html#Macro-invocation","category":"section","text":"Macros are invoked with the following general syntax:\n\n@name expr1 expr2 ...\n@name(expr1, expr2, ...)\n\nNote the distinguishing @ before the macro name and the lack of commas between the argument\nexpressions in the first form, and the lack of whitespace after @name in the second form. The\ntwo styles should not be mixed. For example, the following syntax is different from the examples\nabove; it passes the tuple (expr1, expr2, ...) as one argument to the macro:\n\n@name (expr1, expr2, ...)\n\nAn alternative way to invoke a macro over an array literal (or comprehension) is to juxtapose both without using parentheses. In this case, the array will be the only expression fed to the macro. The following syntax is equivalent (and different from @name [a b] * v):\n\n@name[a b] * v\n@name([a b]) * v\n\nIt is important to emphasize that macros receive their arguments as expressions, literals, or\nsymbols. One way to explore macro arguments is to call the show function within the\nmacro body:\n\njulia> macro showarg(x)\n           show(x)\n           # ... remainder of macro, returning an expression\n       end\n@showarg (macro with 1 method)\n\njulia> @showarg(a)\n:a\n\njulia> @showarg(1+1)\n:(1 + 1)\n\njulia> @showarg(println(\"Yo!\"))\n:(println(\"Yo!\"))\n\njulia> @showarg(1)        # Numeric literal\n1\n\njulia> @showarg(\"Yo!\")    # String literal\n\"Yo!\"\n\njulia> @showarg(\"Yo! $(\"hello\")\")    # String with interpolation is an Expr rather than a String\n:(\"Yo! $(\"hello\")\")\n\nIn addition to the given argument list, every macro is passed extra arguments named __source__ and __module__.\n\nThe argument __source__ provides information (in the form of a LineNumberNode object) about the parser location\nof the @ sign from the macro invocation.\nThis allows macros to include better error diagnostic information,\nand is commonly used by logging, string-parser macros, and docs, for example,\nas well as to implement the @__LINE__, @__FILE__, and @__DIR__ macros.\n\nThe location information can be accessed by referencing __source__.line and __source__.file:\n\njulia> macro __LOCATION__(); return QuoteNode(__source__); end\n@__LOCATION__ (macro with 1 method)\n\njulia> dump(\n            @__LOCATION__(\n       ))\nLineNumberNode\n  line: Int64 2\n  file: Symbol none\n\nThe argument __module__ provides information (in the form of a Module object)\nabout the expansion context of the macro invocation.\nThis allows macros to look up contextual information, such as existing bindings,\nor to insert the value as an extra argument to a runtime function call doing self-reflection\nin the current module."},{"title":"Building an advanced macro","page":"Metaprogramming","location":"manual/metaprogramming.html#Building-an-advanced-macro","category":"section","text":"Here is a simplified definition of Julia's @assert macro:\n\njulia> macro assert(ex)\n           return :( $ex ? nothing : throw(AssertionError($(string(ex)))) )\n       end\n@assert (macro with 1 method)\n\nThis macro can be used like this:\n\njulia> @assert 1 == 1.0\n\njulia> @assert 1 == 0\nERROR: AssertionError: 1 == 0\n\nIn place of the written syntax, the macro call is expanded at parse time to its returned result.\nThis is equivalent to writing:\n\n1 == 1.0 ? nothing : throw(AssertionError(\"1 == 1.0\"))\n1 == 0 ? nothing : throw(AssertionError(\"1 == 0\"))\n\nThat is, in the first call, the expression :(1 == 1.0) is spliced into the test condition slot,\nwhile the value of string(:(1 == 1.0)) is spliced into the assertion message slot. The entire\nexpression, thus constructed, is placed into the syntax tree where the @assert macro call occurs.\nThen at execution time, if the test expression evaluates to true, then nothing is returned,\nwhereas if the test is false, an error is raised indicating the asserted expression that was false.\nNotice that it would not be possible to write this as a function, since only the value of the\ncondition is available and it would be impossible to display the expression that computed it in\nthe error message.\n\nThe actual definition of @assert in Julia Base is more complicated. It allows the\nuser to optionally specify their own error message, instead of just printing the failed expression.\nJust like in functions with a variable number of arguments (Varargs Functions), this is specified with an ellipses\nfollowing the last argument:\n\njulia> macro assert(ex, msgs...)\n           msg_body = isempty(msgs) ? ex : msgs[1]\n           msg = string(msg_body)\n           return :($ex ? nothing : throw(AssertionError($msg)))\n       end\n@assert (macro with 1 method)\n\nNow @assert has two modes of operation, depending upon the number of arguments it receives!\nIf there's only one argument, the tuple of expressions captured by msgs will be empty and it\nwill behave the same as the simpler definition above. But now if the user specifies a second argument,\nit is printed in the message body instead of the failing expression. You can inspect the result\nof a macro expansion with the aptly named @macroexpand macro:\n\njulia> @macroexpand @assert a == b\n:(if Main.a == Main.b\n        Main.nothing\n    else\n        Main.throw(Main.AssertionError(\"a == b\"))\n    end)\n\njulia> @macroexpand @assert a==b \"a should equal b!\"\n:(if Main.a == Main.b\n        Main.nothing\n    else\n        Main.throw(Main.AssertionError(\"a should equal b!\"))\n    end)\n\nThere is yet another case that the actual @assert macro handles: what if, in addition to printing\n\"a should equal b,\" we wanted to print their values? One might naively try to use string interpolation\nin the custom message, e.g., @assert a==b \"a ($a) should equal b ($b)!\", but this won't work\nas expected with the above macro. Can you see why? Recall from string interpolation that\nan interpolated string is rewritten to a call to string. Compare:\n\njulia> typeof(:(\"a should equal b\"))\nString\n\njulia> typeof(:(\"a ($a) should equal b ($b)!\"))\nExpr\n\njulia> dump(:(\"a ($a) should equal b ($b)!\"))\nExpr\n  head: Symbol string\n  args: Array{Any}((5,))\n    1: String \"a (\"\n    2: Symbol a\n    3: String \") should equal b (\"\n    4: Symbol b\n    5: String \")!\"\n\nSo now instead of getting a plain string in msg_body, the macro is receiving a full expression\nthat will need to be evaluated in order to display as expected. This can be spliced directly into\nthe returned expression as an argument to the string call; see error.jl\nfor the complete implementation.\n\nThe @assert macro makes great use of splicing into quoted expressions to simplify the manipulation\nof expressions inside the macro body."},{"title":"Hygiene","page":"Metaprogramming","location":"manual/metaprogramming.html#Hygiene","category":"section","text":"An issue that arises in more complex macros is that of hygiene.\nIn short, macros must ensure that the variables they introduce in their returned expressions do\nnot accidentally clash with existing variables in the surrounding code they expand into. Conversely,\nthe expressions that are passed into a macro as arguments are often expected to evaluate in\nthe context of the surrounding code, interacting with and modifying the existing variables. Another\nconcern arises from the fact that a macro may be called in a different module from where it was\ndefined. In this case we need to ensure that all global variables are resolved to the correct\nmodule. Julia already has a major advantage over languages with textual macro expansion (like\nC) in that it only needs to consider the returned expression. All the other variables (such as\nmsg in @assert above) follow the normal scoping block behavior.\n\nTo demonstrate these issues, let us consider writing a @time macro that takes an expression\nas its argument, records the time, evaluates the expression, records the time again, prints the\ndifference between the before and after times, and then has the value of the expression as its\nfinal value. The macro might look like this:\n\nmacro time(ex)\n    return quote\n        local t0 = time_ns()\n        local val = $ex\n        local t1 = time_ns()\n        println(\"elapsed time: \", (t1-t0)/1e9, \" seconds\")\n        val\n    end\nend\n\nHere, we want t0, t1, and val to be private temporary variables, and we want time_ns to refer\nto the time_ns function in Julia Base, not to any time_ns variable the user\nmight have (the same applies to println). Imagine the problems that could occur if the user\nexpression ex also contained assignments to a variable called t0, or defined its own time_ns\nvariable. We might get errors, or mysteriously incorrect behavior.\n\nJulia's macro expander solves these problems in the following way. First, variables within a macro\nresult are classified as either local or global. A variable is considered local if it is assigned\nto (and not declared global), declared local, or used as a function argument name. Otherwise,\nit is considered global. Local variables are then renamed to be unique (using the gensym\nfunction, which generates new symbols), and global variables are resolved within the macro definition\nenvironment. Therefore both of the above concerns are handled; the macro's locals will not conflict\nwith any user variables, and time_ns and println will refer to the Julia Base definitions.\n\nOne problem remains however. Consider the following use of this macro:\n\nmodule MyModule\nimport Base.@time\n\ntime_ns() = ... # compute something\n\n@time time_ns()\nend\n\nHere the user expression ex is a call to time_ns, but not the same time_ns function that the macro\nuses. It clearly refers to MyModule.time_ns. Therefore we must arrange for the code in ex to\nbe resolved in the macro call environment. This is done by \"escaping\" the expression with esc:\n\nmacro time(ex)\n    ...\n    local val = $(esc(ex))\n    ...\nend\n\nAn expression wrapped in this manner is left alone by the macro expander and simply pasted into\nthe output verbatim. Therefore it will be resolved in the macro call environment.\n\nThis escaping mechanism can be used to \"violate\" hygiene when necessary, in order to introduce\nor manipulate user variables. For example, the following macro sets x to zero in the call environment:\n\njulia> macro zerox()\n           return esc(:(x = 0))\n       end\n@zerox (macro with 1 method)\n\njulia> function foo()\n           x = 1\n           @zerox\n           return x # is zero\n       end\nfoo (generic function with 1 method)\n\njulia> foo()\n0\n\nThis kind of manipulation of variables should be used judiciously, but is occasionally quite handy.\n\nGetting the hygiene rules correct can be a formidable challenge.\nBefore using a macro, you might want to consider whether a function closure\nwould be sufficient. Another useful strategy is to defer as much work as possible to runtime.\nFor example, many macros simply wrap their arguments in a QuoteNode or other similar Expr.\nSome examples of this include @task body which simply returns schedule(Task(() -> $body)),\nand @eval expr, which simply returns eval(QuoteNode(expr)).\n\nTo demonstrate, we might rewrite the @time example above as:\n\nmacro time(expr)\n    return :(timeit(() -> $(esc(expr))))\nend\nfunction timeit(f)\n    t0 = time_ns()\n    val = f()\n    t1 = time_ns()\n    println(\"elapsed time: \", (t1-t0)/1e9, \" seconds\")\n    return val\nend\n\nHowever, we don't do this for a good reason: wrapping the expr in a new scope block (the anonymous function)\nalso slightly changes the meaning of the expression (the scope of any variables in it),\nwhile we want @time to be usable with minimum impact on the wrapped code."},{"title":"Macros and dispatch","page":"Metaprogramming","location":"manual/metaprogramming.html#Macros-and-dispatch","category":"section","text":"Macros, just like Julia functions, are generic. This means they can also have multiple method definitions, thanks to multiple dispatch:\n\njulia> macro m end\n@m (macro with 0 methods)\n\njulia> macro m(args...)\n           println(\"$(length(args)) arguments\")\n       end\n@m (macro with 1 method)\n\njulia> macro m(x,y)\n           println(\"Two arguments\")\n       end\n@m (macro with 2 methods)\n\njulia> @m \"asd\"\n1 arguments\n\njulia> @m 1 2\nTwo arguments\n\nHowever one should keep in mind, that macro dispatch is based on the types of AST\nthat are handed to the macro, not the types that the AST evaluates to at runtime:\n\njulia> macro m(::Int)\n           println(\"An Integer\")\n       end\n@m (macro with 3 methods)\n\njulia> @m 2\nAn Integer\n\njulia> x = 2\n2\n\njulia> @m x\n1 arguments"},{"title":"Code Generation","page":"Metaprogramming","location":"manual/metaprogramming.html#Code-Generation","category":"section","text":"When a significant amount of repetitive boilerplate code is required, it is common to generate\nit programmatically to avoid redundancy. In most languages, this requires an extra build step,\nand a separate program to generate the repetitive code. In Julia, expression interpolation and\neval allow such code generation to take place in the normal course of program execution.\nFor example, consider the following custom type\n\nstruct MyNumber\n    x::Float64\nend\n# output\n\n\nfor which we want to add a number of methods to. We can do this programmatically in the\nfollowing loop:\n\nfor op = (:sin, :cos, :tan, :log, :exp)\n    eval(quote\n        Base.$op(a::MyNumber) = MyNumber($op(a.x))\n    end)\nend\n# output\n\n\nand we can now use those functions with our custom type:\n\njulia> x = MyNumber(π)\nMyNumber(3.141592653589793)\n\njulia> sin(x)\nMyNumber(1.2246467991473532e-16)\n\njulia> cos(x)\nMyNumber(-1.0)\n\nIn this manner, Julia acts as its own preprocessor,\nand allows code generation from inside the language. The above code could be written slightly\nmore tersely using the : prefix quoting form:\n\nfor op = (:sin, :cos, :tan, :log, :exp)\n    eval(:(Base.$op(a::MyNumber) = MyNumber($op(a.x))))\nend\n\nThis sort of in-language code generation, however, using the eval(quote(...)) pattern, is common\nenough that Julia comes with a macro to abbreviate this pattern:\n\nfor op = (:sin, :cos, :tan, :log, :exp)\n    @eval Base.$op(a::MyNumber) = MyNumber($op(a.x))\nend\n\nThe @eval macro rewrites this call to be precisely equivalent to the above longer versions.\nFor longer blocks of generated code, the expression argument given to @eval can be a\nblock:\n\n@eval begin\n    # multiple lines\nend"},{"title":"Non-Standard String Literals","page":"Metaprogramming","location":"manual/metaprogramming.html#meta-non-standard-string-literals","category":"section","text":"Recall from Strings that string literals prefixed by an identifier are called non-standard\nstring literals, and can have different semantics than un-prefixed string literals. For example:\n\nr\"^\\s*(?:#|$)\" produces a regular expression object rather than a string\nb\"DATA\\xff\\u2200\" is a byte array literal for [68,65,84,65,255,226,136,128].\n\nPerhaps surprisingly, these behaviors are not hard-coded into the Julia parser or compiler. Instead,\nthey are custom behaviors provided by a general mechanism that anyone can use: prefixed string\nliterals are parsed as calls to specially-named macros. For example, the regular expression macro\nis just the following:\n\nmacro r_str(p)\n    Regex(p)\nend\n\nThat's all. This macro says that the literal contents of the string literal r\"^\\s*(?:#|$)\" should\nbe passed to the @r_str macro and the result of that expansion should be placed in the syntax\ntree where the string literal occurs. In other words, the expression r\"^\\s*(?:#|$)\" is equivalent\nto placing the following object directly into the syntax tree:\n\nRegex(\"^\\\\s*(?:#|\\$)\")\n\nNot only is the string literal form shorter and far more convenient, but it is also more efficient:\nsince the regular expression is compiled and the Regex object is actually created when the code is compiled,\nthe compilation occurs only once, rather than every time the code is executed. Consider if the\nregular expression occurs in a loop:\n\nfor line = lines\n    m = match(r\"^\\s*(?:#|$)\", line)\n    if m === nothing\n        # non-comment\n    else\n        # comment\n    end\nend\n\nSince the regular expression r\"^\\s*(?:#|$)\" is compiled and inserted into the syntax tree when\nthis code is parsed, the expression is only compiled once instead of each time the loop is executed.\nIn order to accomplish this without macros, one would have to write this loop like this:\n\nre = Regex(\"^\\\\s*(?:#|\\$)\")\nfor line = lines\n    m = match(re, line)\n    if m === nothing\n        # non-comment\n    else\n        # comment\n    end\nend\n\nMoreover, if the compiler could not determine that the regex object was constant over all loops,\ncertain optimizations might not be possible, making this version still less efficient than the\nmore convenient literal form above. Of course, there are still situations where the non-literal\nform is more convenient: if one needs to interpolate a variable into the regular expression, one\nmust take this more verbose approach; in cases where the regular expression pattern itself is\ndynamic, potentially changing upon each loop iteration, a new regular expression object must be\nconstructed on each iteration. In the vast majority of use cases, however, regular expressions\nare not constructed based on run-time data. In this majority of cases, the ability to write regular\nexpressions as compile-time values is invaluable.\n\nThe mechanism for user-defined string literals is deeply, profoundly powerful. Not only are Julia's\nnon-standard literals implemented using it, but the command literal syntax (`echo \"Hello, $person\"`)\nis also implemented using the following innocuous-looking macro:\n\nmacro cmd(str)\n    :(cmd_gen($(shell_parse(str)[1])))\nend\n\nOf course, a large amount of complexity is hidden in the functions used in this macro definition,\nbut they are just functions, written entirely in Julia. You can read their source and see precisely\nwhat they do – and all they do is construct expression objects to be inserted into your program's\nsyntax tree.\n\nLike string literals, command literals can also be prefixed by an identifier\nto form what are called non-standard command literals. These command literals are parsed\nas calls to specially-named macros. For example, the syntax custom`literal` is parsed\nas @custom_cmd \"literal\".\nJulia itself does not contain any non-standard command literals, but packages can make use of\nthis syntax. Aside from the different syntax and the _cmd suffix instead of the _str suffix,\nnon-standard command literals behave exactly like non-standard string literals.\n\nIn the event that two modules provide non-standard string or command literals with the same name,\nit is possible to qualify the string or command literal with a module name. For instance, if both\nFoo and Bar provide non-standard string literal @x_str, then one can write Foo.x\"literal\"\nor Bar.x\"literal\" to disambiguate between the two.\n\nAnother way to define a macro would be like this:\n\nmacro foo_str(str, flag)\n    # do stuff\nend\n\nThis macro can then be called with the following syntax:\n\nfoo\"str\"flag\n\nThe type of flag in the above mentioned syntax would be a String with contents of whatever trails after the string literal."},{"title":"Generated functions","page":"Metaprogramming","location":"manual/metaprogramming.html#Generated-functions","category":"section","text":"A very special macro is @generated, which allows you to define so-called generated functions.\nThese have the capability to generate specialized code depending on the types of their arguments\nwith more flexibility and/or less code than what can be achieved with multiple dispatch. While\nmacros work with expressions at parse time and cannot access the types of their inputs, a generated\nfunction gets expanded at a time when the types of the arguments are known, but the function is\nnot yet compiled.\n\nInstead of performing some calculation or action, a generated function declaration returns a quoted\nexpression which then forms the body for the method corresponding to the types of the arguments.\nWhen a generated function is called, the expression it returns is compiled and then run.\nTo make this efficient, the result is usually cached. And to make this inferable, only a limited\nsubset of the language is usable. Thus, generated functions provide a flexible way to move work from\nrun time to compile time, at the expense of greater restrictions on allowed constructs.\n\nWhen defining generated functions, there are five main differences to ordinary functions:\n\nYou annotate the function declaration with the @generated macro. This adds some information\nto the AST that lets the compiler know that this is a generated function.\nIn the body of the generated function you only have access to the types of the arguments –\nnot their values.\nInstead of calculating something or performing some action, you return a quoted expression which,\nwhen evaluated, does what you want.\nGenerated functions are only permitted to call functions that were defined before the definition of the generated\nfunction. (Failure to follow this may result in getting MethodErrors referring to functions from a future world-age.)\nGenerated functions must not mutate or observe any non-constant global state (including,\nfor example, IO, locks, non-local dictionaries, or using hasmethod).\nThis means they can only read global constants, and cannot have any side effects.\nIn other words, they must be completely pure.\nDue to an implementation limitation, this also means that they currently cannot define a closure\nor generator.\n\nIt's easiest to illustrate this with an example. We can declare a generated function foo as\n\njulia> @generated function foo(x)\n           Core.println(x)\n           return :(x * x)\n       end\nfoo (generic function with 1 method)\n\nNote that the body returns a quoted expression, namely :(x * x), rather than just the value\nof x * x.\n\nFrom the caller's perspective, this is identical to a regular function; in fact, you don't\nhave to know whether you're calling a regular or generated function. Let's see how foo behaves:\n\njulia> x = foo(2); # note: output is from println() statement in the body\nInt64\n\njulia> x           # now we print x\n4\n\njulia> y = foo(\"bar\");\nString\n\njulia> y\n\"barbar\"\n\nSo, we see that in the body of the generated function, x is the type of the passed argument,\nand the value returned by the generated function, is the result of evaluating the quoted expression\nwe returned from the definition, now with the value of x.\n\nWhat happens if we evaluate foo again with a type that we have already used?\n\njulia> foo(4)\n16\n\nNote that there is no printout of Int64. We can see that the body of the generated function\nwas only executed once here, for the specific set of argument types, and the result was cached.\nAfter that, for this example, the expression returned from the generated function on the first\ninvocation was re-used as the method body. However, the actual caching behavior is an implementation-defined\nperformance optimization, so it is invalid to depend too closely on this behavior.\n\nThe number of times a generated function is generated might be only once, but it might also\nbe more often, or appear to not happen at all. As a consequence, you should never write a generated\nfunction with side effects - when, and how often, the side effects occur is undefined. (This is\ntrue for macros too - and just like for macros, the use of eval in a generated function\nis a sign that you're doing something the wrong way.) However, unlike macros, the runtime system\ncannot correctly handle a call to eval, so it is disallowed.\n\nIt is also important to see how @generated functions interact with method redefinition.\nFollowing the principle that a correct @generated function must not observe any\nmutable state or cause any mutation of global state, we see the following behavior.\nObserve that the generated function cannot call any method that was not defined\nprior to the definition of the generated function itself.\n\nInitially f(x) has one definition\n\njulia> f(x) = \"original definition\";\n\nDefine other operations that use f(x):\n\njulia> g(x) = f(x);\n\njulia> @generated gen1(x) = f(x);\n\njulia> @generated gen2(x) = :(f(x));\n\nWe now add some new definitions for f(x):\n\njulia> f(x::Int) = \"definition for Int\";\n\njulia> f(x::Type{Int}) = \"definition for Type{Int}\";\n\nand compare how these results differ:\n\njulia> f(1)\n\"definition for Int\"\n\njulia> g(1)\n\"definition for Int\"\n\njulia> gen1(1)\n\"original definition\"\n\njulia> gen2(1)\n\"definition for Int\"\n\nEach method of a generated function has its own view of defined functions:\n\njulia> @generated gen1(x::Real) = f(x);\n\njulia> gen1(1)\n\"definition for Type{Int}\"\n\nThe example generated function foo above did not do anything a normal function foo(x) = x * x\ncould not do (except printing the type on the first invocation, and incurring higher overhead).\nHowever, the power of a generated function lies in its ability to compute different quoted expressions\ndepending on the types passed to it:\n\njulia> @generated function bar(x)\n           if x <: Integer\n               return :(x ^ 2)\n           else\n               return :(x)\n           end\n       end\nbar (generic function with 1 method)\n\njulia> bar(4)\n16\n\njulia> bar(\"baz\")\n\"baz\"\n\n(although of course this contrived example would be more easily implemented using multiple dispatch...)\n\nAbusing this will corrupt the runtime system and cause undefined behavior:\n\njulia> @generated function baz(x)\n           if rand() < .9\n               return :(x^2)\n           else\n               return :(\"boo!\")\n           end\n       end\nbaz (generic function with 1 method)\n\nSince the body of the generated function is non-deterministic, its behavior, and the behavior of all subsequent code\nis undefined.\n\nDon't copy these examples!\n\nThese examples are hopefully helpful to illustrate how generated functions work, both in the definition\nend and at the call site; however, don't copy them, for the following reasons:\n\nthe foo function has side-effects (the call to Core.println), and it is undefined exactly\nwhen, how often or how many times these side-effects will occur\nthe bar function solves a problem that is better solved with multiple dispatch - defining bar(x) = x\nand bar(x::Integer) = x ^ 2 will do the same thing, but it is both simpler and faster.\nthe baz function is pathological\n\nNote that the set of operations that should not be attempted in a generated function is unbounded,\nand the runtime system can currently only detect a subset of the invalid operations. There are\nmany other operations that will simply corrupt the runtime system without notification, usually\nin subtle ways not obviously connected to the bad definition. Because the function generator is\nrun during inference, it must respect all of the limitations of that code.\n\nSome operations that should not be attempted include:\n\nCaching of native pointers.\nInteracting with the contents or methods of Core.Compiler in any way.\nObserving any mutable state.\nInference on the generated function may be run at any time, including while your code is attempting\nto observe or mutate this state.\nTaking any locks: C code you call out to may use locks internally, (for example, it is not problematic\nto call malloc, even though most implementations require locks internally) but don't attempt\nto hold or acquire any while executing Julia code.\nCalling any function that is defined after the body of the generated function. This condition\nis relaxed for incrementally-loaded precompiled modules to allow calling any function in the module.\n\nAlright, now that we have a better understanding of how generated functions work, let's use them\nto build some more advanced (and valid) functionality..."},{"title":"An advanced example","page":"Metaprogramming","location":"manual/metaprogramming.html#An-advanced-example","category":"section","text":"Julia's base library has an internal sub2ind function to calculate a linear index into an n-dimensional\narray, based on a set of n multilinear indices - in other words, to calculate the index i that\ncan be used to index into an array A using A[i], instead of A[x,y,z,...]. One possible implementation\nis the following:\n\njulia> function sub2ind_loop(dims::NTuple{N}, I::Integer...) where N\n           ind = I[N] - 1\n           for i = N-1:-1:1\n               ind = I[i]-1 + dims[i]*ind\n           end\n           return ind + 1\n       end;\n\njulia> sub2ind_loop((3, 5), 1, 2)\n4\n\nThe same thing can be done using recursion:\n\njulia> sub2ind_rec(dims::Tuple{}) = 1;\n\njulia> sub2ind_rec(dims::Tuple{}, i1::Integer, I::Integer...) =\n           i1 == 1 ? sub2ind_rec(dims, I...) : throw(BoundsError());\n\njulia> sub2ind_rec(dims::Tuple{Integer, Vararg{Integer}}, i1::Integer) = i1;\n\njulia> sub2ind_rec(dims::Tuple{Integer, Vararg{Integer}}, i1::Integer, I::Integer...) =\n           i1 + dims[1] * (sub2ind_rec(Base.tail(dims), I...) - 1);\n\njulia> sub2ind_rec((3, 5), 1, 2)\n4\n\nBoth these implementations, although different, do essentially the same thing: a runtime loop\nover the dimensions of the array, collecting the offset in each dimension into the final index.\n\nHowever, all the information we need for the loop is embedded in the type information of the arguments.\nThis allows the compiler to move the iteration to compile time and eliminate the runtime loops\naltogether. We can utilize generated functions to achieve a similar effect; in compiler parlance,\nwe use generated functions to manually unroll the loop. The body becomes almost identical, but\ninstead of calculating the linear index, we build up an expression that calculates the index:\n\njulia> @generated function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end;\n\njulia> sub2ind_gen((3, 5), 1, 2)\n4\n\nWhat code will this generate?\n\nAn easy way to find out is to extract the body into another (regular) function:\n\njulia> function sub2ind_gen_impl(dims::Type{T}, I...) where T <: NTuple{N,Any} where N\n           length(I) == N || return :(error(\"partial indexing is unsupported\"))\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end;\n\njulia> @generated function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           return sub2ind_gen_impl(dims, I...)\n       end;\n\njulia> sub2ind_gen((3, 5), 1, 2)\n4\n\nWe can now execute sub2ind_gen_impl and examine the expression it returns:\n\njulia> sub2ind_gen_impl(Tuple{Int,Int}, Int, Int)\n:(((I[1] - 1) + dims[1] * (I[2] - 1)) + 1)\n\nSo, the method body that will be used here doesn't include a loop at all - just indexing into\nthe two tuples, multiplication and addition/subtraction. All the looping is performed compile-time,\nand we avoid looping during execution entirely. Thus, we only loop once per type, in this case\nonce per N (except in edge cases where the function is generated more than once - see disclaimer\nabove)."},{"title":"Optionally-generated functions","page":"Metaprogramming","location":"manual/metaprogramming.html#Optionally-generated-functions","category":"section","text":"Generated functions can achieve high efficiency at run time, but come with a compile time cost:\na new function body must be generated for every combination of concrete argument types.\nTypically, Julia is able to compile \"generic\" versions of functions that will work for any\narguments, but with generated functions this is impossible.\nThis means that programs making heavy use of generated functions might be impossible to\nstatically compile.\n\nTo solve this problem, the language provides syntax for writing normal, non-generated\nalternative implementations of generated functions.\nApplied to the sub2ind example above, it would look like this:\n\njulia> function sub2ind_gen_impl(dims::Type{T}, I...) where T <: NTuple{N,Any} where N\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end;\n\njulia> function sub2ind_gen_fallback(dims::NTuple{N}, I) where N\n           ind = I[N] - 1\n           for i = (N - 1):-1:1\n               ind = I[i] - 1 + dims[i]*ind\n           end\n           return ind + 1\n       end;\n\njulia> function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           length(I) == N || error(\"partial indexing is unsupported\")\n           if @generated\n               return sub2ind_gen_impl(dims, I...)\n           else\n               return sub2ind_gen_fallback(dims, I)\n           end\n       end;\n\njulia> sub2ind_gen((3, 5), 1, 2)\n4\n\nInternally, this code creates two implementations of the function: a generated one where\nthe first block in if @generated is used, and a normal one where the else block is used.\nInside the then part of the if @generated block, code has the same semantics as other\ngenerated functions: argument names refer to types, and the code should return an expression.\nMultiple if @generated blocks may occur, in which case the generated implementation uses\nall of the then blocks and the alternate implementation uses all of the else blocks.\n\nNotice that we added an error check to the top of the function.\nThis code will be common to both versions, and is run-time code in both versions\n(it will be quoted and returned as an expression from the generated version).\nThat means that the values and types of local variables are not available at code generation\ntime — the code-generation code can only see the types of arguments.\n\nIn this style of definition, the code generation feature is essentially an optional\noptimization.\nThe compiler will use it if convenient, but otherwise may choose to use the normal\nimplementation instead.\nThis style is preferred, since it allows the compiler to make more decisions and compile\nprograms in more ways, and since normal code is more readable than code-generating code.\nHowever, which implementation is used depends on compiler implementation details, so it\nis essential for the two implementations to behave identically."},{"title":"Git workflow recommendations","page":"Git workflow recommendations","location":"devdocs/contributing/git-workflow.html#Git-workflow-recommendations","category":"section","text":""},{"title":"Git Recommendations For Pull Requests","page":"Git workflow recommendations","location":"devdocs/contributing/git-workflow.html#Git-Recommendations-For-Pull-Requests","category":"section","text":"Avoid working from the master branch of your fork. Create a new branch as it will make it easier to update your pull request if Julia's master changes.\nTry to squash together small commits that make repeated changes to the same section of code, so your pull request is easier to review. A reasonable number of separate well-factored commits is fine, especially for larger changes.\nIf any conflicts arise due to changes in Julia's master, prefer updating your pull request branch with git rebase versus git merge or git pull, since the latter will introduce merge commits that clutter the git history with noise that makes your changes more difficult to review.\nDescriptive commit messages are good.\nUsing git add -p or git add -i can be useful to avoid accidentally committing unrelated changes.\nWhen linking to specific lines of code in discussion of an issue or pull request, hit the y key while viewing code on GitHub to reload the page with a URL that includes the specific version that you're viewing. That way any lines of code that you refer to will still make sense in the future, even if the content of the file changes.\nWhitespace can be automatically removed from existing commits with git rebase.\nTo remove whitespace for the previous commit, run\ngit rebase --whitespace=fix HEAD~1.\nTo remove whitespace relative to the master branch, run\ngit rebase --whitespace=fix master."},{"title":"Git Recommendations For Pull Request Reviewers","page":"Git workflow recommendations","location":"devdocs/contributing/git-workflow.html#Git-Recommendations-For-Pull-Request-Reviewers","category":"section","text":"When merging, we generally like squash+merge. Unless it is the rare case of a PR with carefully staged individual commits that you want in the history separately, in which case merge is acceptable, but usually prefer squash+merge."},{"title":"Punctuation","page":"Punctuation","location":"base/punctuation.html#man-punctuation","category":"section","text":"See also the extended documentation for mathematical symbols & functions.\n\nsymbol meaning\n@ the at-sign marks a macro invocation; optionally followed by an argument list\n! an exclamation mark is a prefix operator for logical negation (\"not\")\na! function names that end with an exclamation mark modify one or more of their arguments by convention\n# the number sign (or hash or pound) character begins single line comments\n#= when followed by an equals sign, it begins a multi-line comment (these are nestable)\n=# end a multi-line comment by immediately preceding the number sign with an equals sign\n$ the dollar sign is used for string and expression interpolation\n% the percent symbol is the remainder operator\n^ the caret is the exponentiation operator\n& single ampersand is bitwise and\n&& double ampersands is short-circuiting boolean and\n| single pipe character is bitwise or\n|| double pipe characters is short-circuiting boolean or\n⊻ the unicode xor character is bitwise exclusive or\n~ the tilde is an operator for bitwise not\n' a trailing apostrophe is the adjoint (that is, the complex transpose) operator Aᴴ\n* the asterisk is used for multiplication, including matrix multiplication and string concatenation\n/ forward slash divides the argument on its left by the one on its right\n// double forward slash performs exact, rational division\n\\ backslash operator divides the argument on its right by the one on its left, commonly used to solve matrix equations\n() parentheses with no arguments constructs an empty Tuple\n(a,...) parentheses with comma-separated arguments constructs a tuple containing its arguments\n(a=1,...) parentheses with comma-separated assignments constructs a NamedTuple\n(x;y) parentheses can also be used to group one or more semicolon separated expressions\na[] array indexing (calling getindex or setindex!)\n[,] vector literal constructor (calling vect)\n[;] vertical concatenation (calling vcat or hvcat)\n[    ] with space-separated expressions, horizontal concatenation (calling hcat or hvcat)\nT{ } curly braces following a type list that type's parameters\n{} curly braces can also be used to group multiple where expressions in function declarations\n; semicolons separate statements, begin a list of keyword arguments in function declarations or calls, or are used to separate array literals for vertical concatenation\n, commas separate function arguments or tuple or array components\n? the question mark delimits the ternary conditional operator (used like: conditional ? if_true : if_false)\n\" \" the single double-quote character delimits String literals\n\"\"\" \"\"\" three double-quote characters delimits string literals that may contain \" and ignore leading indentation\n' ' the single-quote character delimits Char (that is, character) literals\n` ` the backtick character delimits external process (Cmd) literals\nA... triple periods are a postfix operator that \"splat\" their arguments' contents into many arguments of a function call or declare a varargs function that \"slurps\" up many arguments into a single tuple\na.b single periods access named fields in objects/modules (calling getproperty or setproperty!)\nf.() periods may also prefix parentheses (like f.(...)) or infix operators (like .+) to perform the function element-wise (calling broadcast)\na:b colons (:) used as a binary infix operator construct a range from a to b (inclusive) with fixed step size 1\na:s:b colons (:) used as a ternary infix operator construct a range from a to b (inclusive) with step size s\n: when used by themselves, Colons represent all indices within a dimension, frequently combined with indexing\n:: double-colons represent a type annotation or typeassert, depending on context, frequently used when declaring function arguments\n:( ) quoted expression\n:a Symbol a\n<: subtype operator\n>: supertype operator (reverse of subtype operator)\n= single equals sign is assignment\n== double equals sign is value equality comparison\n=== triple equals sign is programmatically identical equality comparison\n=> right arrow using an equals sign defines a Pair typically used to populate dictionaries\n-> right arrow using a hyphen defines an anonymous function on a single line\n|> pipe operator passes output from the left argument to input of the right argument, usually a function\n∘ function composition operator (typed with \\circ{tab}) combines two functions as though they are a single larger function\n_ underscores may be assigned values which will not be saved, often used to ignore multiple return values or create repetitive comprehensions"},{"title":"Talking to the compiler (the :meta mechanism)","page":"Talking to the compiler (the :meta mechanism)","location":"devdocs/meta.html#Talking-to-the-compiler-(the-:meta-mechanism)","category":"section","text":"In some circumstances, one might wish to provide hints or instructions that a given block of code\nhas special properties: you might always want to inline it, or you might want to turn on special\ncompiler optimization passes. Starting with version 0.4, Julia has a convention that these instructions\ncan be placed inside a :meta expression, which is typically (but not necessarily) the first\nexpression in the body of a function.\n\n:meta expressions are created with macros. As an example, consider the implementation of the\n@inline macro:\n\nmacro inline(ex)\n    esc(isa(ex, Expr) ? pushmeta!(ex, :inline) : ex)\nend\n\nHere, ex is expected to be an expression defining a function. A statement like this:\n\n@inline function myfunction(x)\n    x*(x+3)\nend\n\ngets turned into an expression like this:\n\nquote\n    function myfunction(x)\n        Expr(:meta, :inline)\n        x*(x+3)\n    end\nend\n\nBase.pushmeta!(ex, tag::Union{Symbol,Expr}) appends :tag to the end of the :meta expression,\ncreating a new :meta expression if necessary.\n\nTo use the metadata, you have to parse these :meta expressions. If your implementation can be\nperformed within Julia, Base.popmeta! is very handy: Base.popmeta!(body, :symbol) will scan\na function body expression (one without the function signature) for the first :meta expression\ncontaining :symbol, extract any arguments, and return a tuple (found::Bool, args::Array{Any}).\nIf the metadata did not have any arguments, or :symbol was not found, the args array will\nbe empty.\n\nNot yet provided is a convenient infrastructure for parsing :meta expressions from C++."},{"title":"Conversion and Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#conversion-and-promotion","category":"section","text":"Julia has a system for promoting arguments of mathematical operators to a common type, which has\nbeen mentioned in various other sections, including Integers and Floating-Point Numbers,\nMathematical Operations and Elementary Functions, Types, and Methods.\nIn this section, we explain how this promotion system works, as well as how to extend it to new\ntypes and apply it to functions besides built-in mathematical operators. Traditionally, programming\nlanguages fall into two camps with respect to promotion of arithmetic arguments:\n\nAutomatic promotion for built-in arithmetic types and operators. In most languages, built-in\nnumeric types, when used as operands to arithmetic operators with infix syntax, such as +,\n-, *, and /, are automatically promoted to a common type to produce the expected results.\nC, Java, Perl, and Python, to name a few, all correctly compute the sum 1 + 1.5 as the floating-point\nvalue 2.5, even though one of the operands to + is an integer. These systems are convenient\nand designed carefully enough that they are generally all-but-invisible to the programmer: hardly\nanyone consciously thinks of this promotion taking place when writing such an expression, but\ncompilers and interpreters must perform conversion before addition since integers and floating-point\nvalues cannot be added as-is. Complex rules for such automatic conversions are thus inevitably\npart of specifications and implementations for such languages.\nNo automatic promotion. This camp includes Ada and ML – very \"strict\" statically typed languages.\nIn these languages, every conversion must be explicitly specified by the programmer. Thus, the\nexample expression 1 + 1.5 would be a compilation error in both Ada and ML. Instead one must\nwrite real(1) + 1.5, explicitly converting the integer 1 to a floating-point value before\nperforming addition. Explicit conversion everywhere is so inconvenient, however, that even Ada\nhas some degree of automatic conversion: integer literals are promoted to the expected integer\ntype automatically, and floating-point literals are similarly promoted to appropriate floating-point\ntypes.\n\nIn a sense, Julia falls into the \"no automatic promotion\" category: mathematical operators are\njust functions with special syntax, and the arguments of functions are never automatically converted.\nHowever, one may observe that applying mathematical operations to a wide variety of mixed argument\ntypes is just an extreme case of polymorphic multiple dispatch – something which Julia's dispatch\nand type systems are particularly well-suited to handle. \"Automatic\" promotion of mathematical\noperands simply emerges as a special application: Julia comes with pre-defined catch-all dispatch\nrules for mathematical operators, invoked when no specific implementation exists for some combination\nof operand types. These catch-all rules first promote all operands to a common type using user-definable\npromotion rules, and then invoke a specialized implementation of the operator in question for\nthe resulting values, now of the same type. User-defined types can easily participate in this\npromotion system by defining methods for conversion to and from other types, and providing a handful\nof promotion rules defining what types they should promote to when mixed with other types."},{"title":"Conversion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Conversion","category":"section","text":"The standard way to obtain a value of a certain type T is to call the type's constructor, T(x).\nHowever, there are cases where it's convenient to convert a value from one type to another\nwithout the programmer asking for it explicitly.\nOne example is assigning a value into an array: if A is a Vector{Float64}, the expression\nA[1] = 2 should work by automatically converting the 2 from Int to Float64, and\nstoring the result in the array.\nThis is done via the convert function.\n\nThe convert function generally takes two arguments: the first is a type object and the second is\na value to convert to that type. The returned value is the value converted to an instance of given type.\nThe simplest way to understand this function is to see it in action:\n\njulia> x = 12\n12\n\njulia> typeof(x)\nInt64\n\njulia> xu = convert(UInt8, x)\n0x0c\n\njulia> typeof(xu)\nUInt8\n\njulia> xf = convert(AbstractFloat, x)\n12.0\n\njulia> typeof(xf)\nFloat64\n\njulia> a = Any[1 2 3; 4 5 6]\n2×3 Matrix{Any}:\n 1  2  3\n 4  5  6\n\njulia> convert(Array{Float64}, a)\n2×3 Matrix{Float64}:\n 1.0  2.0  3.0\n 4.0  5.0  6.0\n\nConversion isn't always possible, in which case a MethodError is thrown indicating that convert\ndoesn't know how to perform the requested conversion:\n\njulia> convert(AbstractFloat, \"foo\")\nERROR: MethodError: Cannot `convert` an object of type String to an object of type AbstractFloat\n[...]\n\nSome languages consider parsing strings as numbers or formatting numbers as strings to be conversions\n(many dynamic languages will even perform conversion for you automatically). This is not the case in Julia.\nEven though some strings can be parsed as numbers, most strings are not valid representations\nof numbers, and only a very limited subset of them are. Therefore in Julia the dedicated parse\nfunction must be used to perform this operation, making it more explicit."},{"title":"When is convert called?","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#When-is-convert-called?","category":"section","text":"The following language constructs call convert:\n\nAssigning to an array converts to the array's element type.\nAssigning to a field of an object converts to the declared type of the field.\nConstructing an object with new converts to the object's declared field types.\nAssigning to a variable with a declared type (e.g. local x::T) converts to that type.\nA function with a declared return type converts its return value to that type.\nPassing a value to ccall converts it to the corresponding argument type."},{"title":"Conversion vs. Construction","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Conversion-vs.-Construction","category":"section","text":"Note that the behavior of convert(T, x) appears to be nearly identical to T(x).\nIndeed, it usually is.\nHowever, there is a key semantic difference: since convert can be called implicitly,\nits methods are restricted to cases that are considered \"safe\" or \"unsurprising\".\nconvert will only convert between types that represent the same basic kind of thing\n(e.g. different representations of numbers, or different string encodings).\nIt is also usually lossless; converting a value to a different type and back again\nshould result in the exact same value.\n\nThere are four general kinds of cases where constructors differ from convert:"},{"title":"Constructors for types unrelated to their arguments","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Constructors-for-types-unrelated-to-their-arguments","category":"section","text":"Some constructors don't implement the concept of \"conversion\".\nFor example, Timer(2) creates a 2-second timer, which is not really a\n\"conversion\" from an integer to a timer."},{"title":"Mutable collections","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Mutable-collections","category":"section","text":"convert(T, x) is expected to return the original x if x is already of type T.\nIn contrast, if T is a mutable collection type then T(x) should always make a new\ncollection (copying elements from x)."},{"title":"Wrapper types","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Wrapper-types","category":"section","text":"For some types which \"wrap\" other values, the constructor may wrap its argument inside\na new object even if it is already of the requested type.\nFor example Some(x) wraps x to indicate that a value is present (in a context\nwhere the result might be a Some or nothing).\nHowever, x itself might be the object Some(y), in which case the result is\nSome(Some(y)), with two levels of wrapping.\nconvert(Some, x), on the other hand, would just return x since it is already\na Some."},{"title":"Constructors that don't return instances of their own type","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Constructors-that-don't-return-instances-of-their-own-type","category":"section","text":"In very rare cases it might make sense for the constructor T(x) to return\nan object not of type T.\nThis could happen if a wrapper type is its own inverse (e.g. Flip(Flip(x)) === x),\nor to support an old calling syntax for backwards compatibility when a library is\nrestructured.\nBut convert(T, x) should always return a value of type T."},{"title":"Defining New Conversions","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Defining-New-Conversions","category":"section","text":"When defining a new type, initially all ways of creating it should be defined as\nconstructors.\nIf it becomes clear that implicit conversion would be useful, and that some\nconstructors meet the above \"safety\" criteria, then convert methods can be added.\nThese methods are typically quite simple, as they only need to call the appropriate\nconstructor.\nSuch a definition might look like this:\n\nimport Base: convert\nconvert(::Type{MyType}, x) = MyType(x)\n\nThe type of the first argument of this method is Type{MyType},\nthe only instance of which is MyType. Thus, this method is only invoked\nwhen the first argument is the type value MyType. Notice the syntax used for the first\nargument: the argument name is omitted prior to the :: symbol, and only the type is given.\nThis is the syntax in Julia for a function argument whose type is specified but whose value\ndoes not need to be referenced by name.\n\nAll instances of some abstract types are by default considered \"sufficiently similar\"\nthat a universal convert definition is provided in Julia Base.\nFor example, this definition states that it's valid to convert any Number type to\nany other by calling a 1-argument constructor:\n\nconvert(::Type{T}, x::Number) where {T<:Number} = T(x)::T\n\nThis means that new Number types only need to define constructors, since this\ndefinition will handle convert for them.\nAn identity conversion is also provided to handle the case where the argument is\nalready of the requested type:\n\nconvert(::Type{T}, x::T) where {T<:Number} = x\n\nSimilar definitions exist for AbstractString, AbstractArray, and AbstractDict."},{"title":"Promotion","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Promotion","category":"section","text":"Promotion refers to converting values of mixed types to a single common type. Although it is not\nstrictly necessary, it is generally implied that the common type to which the values are converted\ncan faithfully represent all of the original values. In this sense, the term \"promotion\" is appropriate\nsince the values are converted to a \"greater\" type – i.e. one which can represent all of the\ninput values in a single common type. It is important, however, not to confuse this with object-oriented\n(structural) super-typing, or Julia's notion of abstract super-types: promotion has nothing to\ndo with the type hierarchy, and everything to do with converting between alternate representations.\nFor instance, although every Int32 value can also be represented as a Float64 value,\nInt32 is not a subtype of Float64.\n\nPromotion to a common \"greater\" type is performed in Julia by the promote function, which takes\nany number of arguments, and returns a tuple of the same number of values, converted to a common\ntype, or throws an exception if promotion is not possible. The most common use case for promotion\nis to convert numeric arguments to a common type:\n\njulia> promote(1, 2.5)\n(1.0, 2.5)\n\njulia> promote(1, 2.5, 3)\n(1.0, 2.5, 3.0)\n\njulia> promote(2, 3//4)\n(2//1, 3//4)\n\njulia> promote(1, 2.5, 3, 3//4)\n(1.0, 2.5, 3.0, 0.75)\n\njulia> promote(1.5, im)\n(1.5 + 0.0im, 0.0 + 1.0im)\n\njulia> promote(1 + 2im, 3//4)\n(1//1 + 2//1*im, 3//4 + 0//1*im)\n\nFloating-point values are promoted to the largest of the floating-point argument types. Integer\nvalues are promoted to the largest of the integer argument types. If the types are the same size\nbut differ in signedness, the unsigned type is chosen. Mixtures of integers and floating-point\nvalues are promoted to a floating-point type big enough to hold all the values. Integers mixed\nwith rationals are promoted to rationals. Rationals mixed with floats are promoted to floats.\nComplex values mixed with real values are promoted to the appropriate kind of complex value.\n\nThat is really all there is to using promotions. The rest is just a matter of clever application,\nthe most typical \"clever\" application being the definition of catch-all methods for numeric operations\nlike the arithmetic operators +, -, * and /. Here are some of the catch-all method definitions\ngiven in promotion.jl:\n\n+(x::Number, y::Number) = +(promote(x,y)...)\n-(x::Number, y::Number) = -(promote(x,y)...)\n*(x::Number, y::Number) = *(promote(x,y)...)\n/(x::Number, y::Number) = /(promote(x,y)...)\n\nThese method definitions say that in the absence of more specific rules for adding, subtracting,\nmultiplying and dividing pairs of numeric values, promote the values to a common type and then\ntry again. That's all there is to it: nowhere else does one ever need to worry about promotion\nto a common numeric type for arithmetic operations – it just happens automatically. There are\ndefinitions of catch-all promotion methods for a number of other arithmetic and mathematical functions\nin promotion.jl, but beyond\nthat, there are hardly any calls to promote required in Julia Base. The most\ncommon usages of promote occur in outer constructors methods, provided for convenience, to allow\nconstructor calls with mixed types to delegate to an inner type with fields promoted to an appropriate\ncommon type. For example, recall that rational.jl\nprovides the following outer constructor method:\n\nRational(n::Integer, d::Integer) = Rational(promote(n,d)...)\n\nThis allows calls like the following to work:\n\njulia> x = Rational(Int8(15),Int32(-5))\n-3//1\n\njulia> typeof(x)\nRational{Int32}\n\nFor most user-defined types, it is better practice to require programmers to supply the expected\ntypes to constructor functions explicitly, but sometimes, especially for numeric problems, it\ncan be convenient to do promotion automatically."},{"title":"Defining Promotion Rules","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Defining-Promotion-Rules","category":"section","text":"Although one could, in principle, define methods for the promote function directly, this would\nrequire many redundant definitions for all possible permutations of argument types. Instead, the\nbehavior of promote is defined in terms of an auxiliary function called promote_rule, which\none can provide methods for. The promote_rule function takes a pair of type objects and returns\nanother type object, such that instances of the argument types will be promoted to the returned\ntype. Thus, by defining the rule:\n\nimport Base: promote_rule\npromote_rule(::Type{Float64}, ::Type{Float32}) = Float64\n\none declares that when 64-bit and 32-bit floating-point values are promoted together, they should\nbe promoted to 64-bit floating-point. The promotion type does not need to be one of the argument\ntypes. For example, the following promotion rules both occur in Julia Base:\n\npromote_rule(::Type{BigInt}, ::Type{Float64}) = BigFloat\npromote_rule(::Type{BigInt}, ::Type{Int8}) = BigInt\n\nIn the latter case, the result type is BigInt since BigInt is the only type\nlarge enough to hold integers for arbitrary-precision integer arithmetic. Also note that\none does not need to define both promote_rule(::Type{A}, ::Type{B}) and\npromote_rule(::Type{B}, ::Type{A}) – the symmetry is implied by the way promote_rule\nis used in the promotion process.\n\nThe promote_rule function is used as a building block to define a second function called promote_type,\nwhich, given any number of type objects, returns the common type to which those values, as arguments\nto promote should be promoted. Thus, if one wants to know, in absence of actual values, what\ntype a collection of values of certain types would promote to, one can use promote_type:\n\njulia> promote_type(Int8, Int64)\nInt64\n\nNote that we do not overload promote_type directly: we overload promote_rule instead.\npromote_type uses promote_rule, and adds the symmetry.\nOverloading it directly can cause ambiguity errors.\nWe overload promote_rule to define how things should be promoted, and we use promote_type\nto query that.\n\nInternally, promote_type is used inside of promote to determine what type argument values\nshould be converted to for promotion. The curious reader can read the code in\npromotion.jl,\nwhich defines the complete promotion mechanism in about 35 lines."},{"title":"Case Study: Rational Promotions","page":"Conversion and Promotion","location":"manual/conversion-and-promotion.html#Case-Study:-Rational-Promotions","category":"section","text":"Finally, we finish off our ongoing case study of Julia's rational number type, which makes relatively\nsophisticated use of the promotion mechanism with the following promotion rules:\n\nimport Base: promote_rule\npromote_rule(::Type{Rational{T}}, ::Type{S}) where {T<:Integer,S<:Integer} = Rational{promote_type(T,S)}\npromote_rule(::Type{Rational{T}}, ::Type{Rational{S}}) where {T<:Integer,S<:Integer} = Rational{promote_type(T,S)}\npromote_rule(::Type{Rational{T}}, ::Type{S}) where {T<:Integer,S<:AbstractFloat} = promote_type(T,S)\n\nThe first rule says that promoting a rational number with any other integer type promotes to a\nrational type whose numerator/denominator type is the result of promotion of its numerator/denominator\ntype with the other integer type. The second rule applies the same logic to two different types\nof rational numbers, resulting in a rational of the promotion of their respective numerator/denominator\ntypes. The third and final rule dictates that promoting a rational with a float results in the\nsame type as promoting the numerator/denominator type with the float.\n\nThis small handful of promotion rules, together with the type's constructors and the default\nconvert method for numbers, are sufficient to make rational numbers interoperate completely\nnaturally with all of Julia's other numeric types – integers, floating-point numbers, and complex\nnumbers. By providing appropriate conversion methods and promotion rules in the same manner, any\nuser-defined numeric type can interoperate just as naturally with Julia's predefined numerics."},{"title":"Strings","page":"Strings","location":"manual/strings.html#man-strings","category":"section","text":"Strings are finite sequences of characters. Of course, the real trouble comes when one asks what\na character is. The characters that English speakers are familiar with are the letters A, B,\nC, etc., together with numerals and common punctuation symbols. These characters are standardized\ntogether with a mapping to integer values between 0 and 127 by the ASCII\nstandard. There are, of course, many other characters used in non-English languages, including\nvariants of the ASCII characters with accents and other modifications, related scripts such as\nCyrillic and Greek, and scripts completely unrelated to ASCII and English, including Arabic, Chinese,\nHebrew, Hindi, Japanese, and Korean. The Unicode standard\ntackles the complexities of what exactly a character is, and is generally accepted as the definitive\nstandard addressing this problem. Depending on your needs, you can either ignore these complexities\nentirely and just pretend that only ASCII characters exist, or you can write code that can handle\nany of the characters or encodings that one may encounter when handling non-ASCII text. Julia\nmakes dealing with plain ASCII text simple and efficient, and handling Unicode is as simple and\nefficient as possible. In particular, you can write C-style string code to process ASCII strings,\nand they will work as expected, both in terms of performance and semantics. If such code encounters\nnon-ASCII text, it will gracefully fail with a clear error message, rather than silently introducing\ncorrupt results. When this happens, modifying the code to handle non-ASCII data is straightforward.\n\nThere are a few noteworthy high-level features about Julia's strings:\n\nThe built-in concrete type used for strings (and string literals) in Julia is String.\nThis supports the full range of Unicode characters via\nthe UTF-8 encoding. (A transcode function is\nprovided to convert to/from other Unicode encodings.)\nAll string types are subtypes of the abstract type AbstractString, and external packages define\nadditional AbstractString subtypes (e.g. for other encodings). If you define a function expecting\na string argument, you should declare the type as AbstractString in order to accept any string\ntype.\nLike C and Java, but unlike most dynamic languages, Julia has a first-class type for representing\na single character, called AbstractChar. The built-in Char subtype of AbstractChar\nis a 32-bit primitive type that can represent any Unicode character (and which is based\non the UTF-8 encoding).\nAs in Java, strings are immutable: the value of an AbstractString object cannot be changed.\nTo construct a different string value, you construct a new string from parts of other strings.\nConceptually, a string is a partial function from indices to characters: for some index values,\nno character value is returned, and instead an exception is thrown. This allows for efficient\nindexing into strings by the byte index of an encoded representation rather than by a character\nindex, which cannot be implemented both efficiently and simply for variable-width encodings of\nUnicode strings."},{"title":"Characters","page":"Strings","location":"manual/strings.html#man-characters","category":"section","text":"A Char value represents a single character: it is just a 32-bit primitive type with a special literal\nrepresentation and appropriate arithmetic behaviors, and which can be converted\nto a numeric value representing a\nUnicode code point.  (Julia packages may define\nother subtypes of AbstractChar, e.g. to optimize operations for other\ntext encodings.) Here is how Char values are\ninput and shown (note that character literals are delimited with single quotes, not double quotes):\n\njulia> c = 'x'\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)\n\njulia> typeof(c)\nChar\n\nYou can easily convert a Char to its integer value, i.e. code point:\n\njulia> c = Int('x')\n120\n\njulia> typeof(c)\nInt64\n\nOn 32-bit architectures, typeof(c) will be Int32. You can convert an\ninteger value back to a Char just as easily:\n\njulia> Char(120)\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)\n\nNot all integer values are valid Unicode code points, but for performance, the Char conversion\ndoes not check that every character value is valid. If you want to check that each converted value\nis a valid code point, use the isvalid function:\n\njulia> Char(0x110000)\n'\\U110000': Unicode U+110000 (category In: Invalid, too high)\n\njulia> isvalid(Char, 0x110000)\nfalse\n\nAs of this writing, the valid Unicode code points are U+0000 through U+D7FF and U+E000 through\nU+10FFFF. These have not all been assigned intelligible meanings yet, nor are they necessarily\ninterpretable by applications, but all of these values are considered to be valid Unicode characters.\n\nYou can input any Unicode character in single quotes using \\u followed by up to four hexadecimal\ndigits or \\U followed by up to eight hexadecimal digits (the longest valid value only requires\nsix):\n\njulia> '\\u0'\n'\\0': ASCII/Unicode U+0000 (category Cc: Other, control)\n\njulia> '\\u78'\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)\n\njulia> '\\u2200'\n'∀': Unicode U+2200 (category Sm: Symbol, math)\n\njulia> '\\U10ffff'\n'\\U10ffff': Unicode U+10FFFF (category Cn: Other, not assigned)\n\nJulia uses your system's locale and language settings to determine which characters can be printed\nas-is and which must be output using the generic, escaped \\u or \\U input forms. In addition\nto these Unicode escape forms, all of C's traditional escaped input forms\ncan also be used:\n\njulia> Int('\\0')\n0\n\njulia> Int('\\t')\n9\n\njulia> Int('\\n')\n10\n\njulia> Int('\\e')\n27\n\njulia> Int('\\x7f')\n127\n\njulia> Int('\\177')\n127\n\nYou can do comparisons and a limited amount of arithmetic with Char values:\n\njulia> 'A' < 'a'\ntrue\n\njulia> 'A' <= 'a' <= 'Z'\nfalse\n\njulia> 'A' <= 'X' <= 'Z'\ntrue\n\njulia> 'x' - 'a'\n23\n\njulia> 'A' + 1\n'B': ASCII/Unicode U+0042 (category Lu: Letter, uppercase)"},{"title":"String Basics","page":"Strings","location":"manual/strings.html#String-Basics","category":"section","text":"String literals are delimited by double quotes or triple double quotes (not single quotes):\n\njulia> str = \"Hello, world.\\n\"\n\"Hello, world.\\n\"\n\njulia> \"\"\"Contains \"quote\" characters\"\"\"\n\"Contains \\\"quote\\\" characters\"\n\nLong lines in strings can be broken up by preceding the newline with a backslash (\\):\n\njulia> \"This is a long \\\n       line\"\n\"This is a long line\"\n\nIf you want to extract a character from a string, you index into it:\n\njulia> str[begin]\n'H': ASCII/Unicode U+0048 (category Lu: Letter, uppercase)\n\njulia> str[1]\n'H': ASCII/Unicode U+0048 (category Lu: Letter, uppercase)\n\njulia> str[6]\n',': ASCII/Unicode U+002C (category Po: Punctuation, other)\n\njulia> str[end]\n'\\n': ASCII/Unicode U+000A (category Cc: Other, control)\n\nMany Julia objects, including strings, can be indexed with integers. The index of the\nfirst element (the first character of a string) is returned by firstindex(str),\nand the index of the last element (character) with lastindex(str). The keywords\nbegin and end can be used inside an indexing operation as shorthand for the first and\nlast indices, respectively, along the given dimension. String indexing, like most indexing\nin Julia, is 1-based: firstindex always returns 1 for any AbstractString. As we will\nsee below, however, lastindex(str) is not in general the same as length(str) for a\nstring, because some Unicode characters can occupy multiple \"code units\".\n\nYou can perform arithmetic and other operations with end, just like\na normal value:\n\njulia> str[end-1]\n'.': ASCII/Unicode U+002E (category Po: Punctuation, other)\n\njulia> str[end÷2]\n' ': ASCII/Unicode U+0020 (category Zs: Separator, space)\n\nUsing an index less than begin (1) or greater than end raises an error:\n\njulia> str[begin-1]\nERROR: BoundsError: attempt to access 14-codeunit String at index [0]\n[...]\n\njulia> str[end+1]\nERROR: BoundsError: attempt to access 14-codeunit String at index [15]\n[...]\n\nYou can also extract a substring using range indexing:\n\njulia> str[4:9]\n\"lo, wo\"\n\nNotice that the expressions str[k] and str[k:k] do not give the same result:\n\njulia> str[6]\n',': ASCII/Unicode U+002C (category Po: Punctuation, other)\n\njulia> str[6:6]\n\",\"\n\nThe former is a single character value of type Char, while the latter is a string value that\nhappens to contain only a single character. In Julia these are very different things.\n\nRange indexing makes a copy of the selected part of the original string.\nAlternatively, it is possible to create a view into a string using the type SubString.\nMore simply, using the @views macro on a block of code converts all string slices\ninto substrings. For example:\n\njulia> str = \"long string\"\n\"long string\"\n\njulia> substr = SubString(str, 1, 4)\n\"long\"\n\njulia> typeof(substr)\nSubString{String}\n\njulia> @views typeof(str[1:4]) # @views converts slices to SubStrings\nSubString{String}\n\nSeveral standard functions like chop, chomp or strip\nreturn a SubString."},{"title":"Unicode and UTF-8","page":"Strings","location":"manual/strings.html#Unicode-and-UTF-8","category":"section","text":"Julia fully supports Unicode characters and strings. As discussed above, in character\nliterals, Unicode code points can be represented using Unicode \\u and \\U escape sequences,\nas well as all the standard C escape sequences. These can likewise be used to write string literals:\n\njulia> s = \"\\u2200 x \\u2203 y\"\n\"∀ x ∃ y\"\n\nWhether these Unicode characters are displayed as escapes or shown as special characters depends\non your terminal's locale settings and its support for Unicode. String literals are encoded using\nthe UTF-8 encoding. UTF-8 is a variable-width encoding, meaning that not all characters are encoded\nin the same number of bytes (\"code units\"). In UTF-8, ASCII characters — i.e. those with code points\nless than 0x80 (128) – are encoded as they are in ASCII, using a single byte, while code points\n0x80 and above are encoded using multiple bytes — up to four per character.\n\nString indices in Julia refer to code units (= bytes for UTF-8), the fixed-width building blocks that\nare used to encode arbitrary characters (code points). This means that not every\nindex into a String is necessarily a valid index for a character. If you index into\na string at such an invalid byte index, an error is thrown:\n\njulia> s[1]\n'∀': Unicode U+2200 (category Sm: Symbol, math)\n\njulia> s[2]\nERROR: StringIndexError: invalid index [2], valid nearby indices [1]=>'∀', [4]=>' '\nStacktrace:\n[...]\n\njulia> s[3]\nERROR: StringIndexError: invalid index [3], valid nearby indices [1]=>'∀', [4]=>' '\nStacktrace:\n[...]\n\njulia> s[4]\n' ': ASCII/Unicode U+0020 (category Zs: Separator, space)\n\nIn this case, the character ∀ is a three-byte character, so the indices 2 and 3 are invalid\nand the next character's index is 4; this next valid index can be computed by nextind(s,1),\nand the next index after that by nextind(s,4) and so on.\n\nSince end is always the last valid index into a collection, end-1 references an invalid\nbyte index if the second-to-last character is multibyte.\n\njulia> s[end-1]\n' ': ASCII/Unicode U+0020 (category Zs: Separator, space)\n\njulia> s[end-2]\nERROR: StringIndexError: invalid index [9], valid nearby indices [7]=>'∃', [10]=>' '\nStacktrace:\n[...]\n\njulia> s[prevind(s, end, 2)]\n'∃': Unicode U+2203 (category Sm: Symbol, math)\n\nThe first case works, because the last character y and the space are one-byte characters,\nwhereas end-2 indexes into the middle of the ∃ multibyte representation. The correct\nway for this case is using prevind(s, lastindex(s), 2) or, if you're using that value to index\ninto s you can write s[prevind(s, end, 2)] and end expands to lastindex(s).\n\nExtraction of a substring using range indexing also expects valid byte indices or an error is thrown:\n\njulia> s[1:1]\n\"∀\"\n\njulia> s[1:2]\nERROR: StringIndexError: invalid index [2], valid nearby indices [1]=>'∀', [4]=>' '\nStacktrace:\n[...]\n\njulia> s[1:4]\n\"∀ \"\n\nBecause of variable-length encodings, the number of characters in a string (given by length(s))\nis not always the same as the last index. If you iterate through the indices 1 through lastindex(s)\nand index into s, the sequence of characters returned when errors aren't thrown is the sequence\nof characters comprising the string s. Thus length(s) <= lastindex(s),\nsince each character in a string must have its own index. The following is an inefficient and\nverbose way to iterate through the characters of s:\n\njulia> for i = firstindex(s):lastindex(s)\n           try\n               println(s[i])\n           catch\n               # ignore the index error\n           end\n       end\n∀\n\nx\n\n∃\n\ny\n\nThe blank lines actually have spaces on them. Fortunately, the above awkward idiom is unnecessary\nfor iterating through the characters in a string, since you can just use the string as an iterable\nobject, no exception handling required:\n\njulia> for c in s\n           println(c)\n       end\n∀\n\nx\n\n∃\n\ny\n\nIf you need to obtain valid indices for a string, you can use the nextind and\nprevind functions to increment/decrement to the next/previous valid index, as mentioned above.\nYou can also use the eachindex function to iterate over the valid character indices:\n\njulia> collect(eachindex(s))\n7-element Vector{Int64}:\n  1\n  4\n  5\n  6\n  7\n 10\n 11\n\nTo access the raw code units (bytes for UTF-8) of the encoding, you can use the codeunit(s,i)\nfunction, where the index i runs consecutively from 1 to ncodeunits(s). The\ncodeunits(s) function returns an AbstractVector{UInt8} wrapper that lets you\naccess these raw codeunits (bytes) as an array.\n\nStrings in Julia can contain invalid UTF-8 code unit sequences. This convention allows to\ntreat any byte sequence as a String. In such situations a rule is that when parsing\na sequence of code units from left to right characters are formed by the longest sequence of\n8-bit code units that matches the start of one of the following bit patterns\n(each x can be 0 or 1):\n\n0xxxxxxx;\n110xxxxx 10xxxxxx;\n1110xxxx 10xxxxxx 10xxxxxx;\n11110xxx 10xxxxxx 10xxxxxx 10xxxxxx;\n10xxxxxx;\n11111xxx.\n\nIn particular this means that overlong and too-high code unit sequences and prefixes thereof are treated\nas a single invalid character rather than multiple invalid characters.\nThis rule may be best explained with an example:\n\njulia> s = \"\\xc0\\xa0\\xe2\\x88\\xe2|\"\n\"\\xc0\\xa0\\xe2\\x88\\xe2|\"\n\njulia> foreach(display, s)\n'\\xc0\\xa0': [overlong] ASCII/Unicode U+0020 (category Zs: Separator, space)\n'\\xe2\\x88': Malformed UTF-8 (category Ma: Malformed, bad data)\n'\\xe2': Malformed UTF-8 (category Ma: Malformed, bad data)\n'|': ASCII/Unicode U+007C (category Sm: Symbol, math)\n\njulia> isvalid.(collect(s))\n4-element BitArray{1}:\n 0\n 0\n 0\n 1\n\njulia> s2 = \"\\xf7\\xbf\\xbf\\xbf\"\n\"\\U1fffff\"\n\njulia> foreach(display, s2)\n'\\U1fffff': Unicode U+1FFFFF (category In: Invalid, too high)\n\nWe can see that the first two code units in the string s form an overlong encoding of\nspace character. It is invalid, but is accepted in a string as a single character.\nThe next two code units form a valid start of a three-byte UTF-8 sequence. However, the fifth\ncode unit \\xe2 is not its valid continuation. Therefore code units 3 and 4 are also\ninterpreted as malformed characters in this string. Similarly code unit 5 forms a malformed\ncharacter because | is not a valid continuation to it. Finally the string s2 contains\none too high code point.\n\nJulia uses the UTF-8 encoding by default, and support for new encodings can be added by packages.\nFor example, the LegacyStrings.jl package\nimplements UTF16String and UTF32String types. Additional discussion of other encodings and\nhow to implement support for them is beyond the scope of this document for the time being. For\nfurther discussion of UTF-8 encoding issues, see the section below on byte array literals.\nThe transcode function is provided to convert data between the various UTF-xx encodings,\nprimarily for working with external data and libraries."},{"title":"Concatenation","page":"Strings","location":"manual/strings.html#man-concatenation","category":"section","text":"One of the most common and useful string operations is concatenation:\n\njulia> greet = \"Hello\"\n\"Hello\"\n\njulia> whom = \"world\"\n\"world\"\n\njulia> string(greet, \", \", whom, \".\\n\")\n\"Hello, world.\\n\"\n\nIt is important to be aware of potentially dangerous situations such as concatenation of\ninvalid UTF-8 strings. The resulting string may contain different characters than the\ninput strings, and its number of characters may be lower than sum of numbers of characters\nof the concatenated strings, e.g.:\n\njulia> a, b = \"\\xe2\\x88\", \"\\x80\"\n(\"\\xe2\\x88\", \"\\x80\")\n\njulia> c = string(a, b)\n\"∀\"\n\njulia> collect.([a, b, c])\n3-element Vector{Vector{Char}}:\n ['\\xe2\\x88']\n ['\\x80']\n ['∀']\n\njulia> length.([a, b, c])\n3-element Vector{Int64}:\n 1\n 1\n 1\n\nThis situation can happen only for invalid UTF-8 strings. For valid UTF-8 strings\nconcatenation preserves all characters in strings and additivity of string lengths.\n\nJulia also provides * for string concatenation:\n\njulia> greet * \", \" * whom * \".\\n\"\n\"Hello, world.\\n\"\n\nWhile * may seem like a surprising choice to users of languages that provide + for string\nconcatenation, this use of * has precedent in mathematics, particularly in abstract algebra.\n\nIn mathematics, + usually denotes a commutative operation, where the order of the operands does\nnot matter. An example of this is matrix addition, where A + B == B + A for any matrices A and B\nthat have the same shape. In contrast, * typically denotes a noncommutative operation, where the\norder of the operands does matter. An example of this is matrix multiplication, where in general\nA * B != B * A. As with matrix multiplication, string concatenation is noncommutative:\ngreet * whom != whom * greet. As such, * is a more natural choice for an infix string concatenation\noperator, consistent with common mathematical use.\n\nMore precisely, the set of all finite-length strings S together with the string concatenation operator\n* forms a free monoid (S, *). The identity element\nof this set is the empty string, \"\". Whenever a free monoid is not commutative, the operation is\ntypically represented as \\cdot, *, or a similar symbol, rather than +, which as stated usually\nimplies commutativity."},{"title":"Interpolation","page":"Strings","location":"manual/strings.html#string-interpolation","category":"section","text":"Constructing strings using concatenation can become a bit cumbersome, however. To reduce\nthe need for these verbose calls to string or repeated multiplications, Julia\nallows interpolation into string literals using $, as in Perl:\n\njulia> greet = \"Hello\"; whom = \"world\";\n\njulia> \"$greet, $whom.\\n\"\n\"Hello, world.\\n\"\n\nThis is more readable and convenient and equivalent to the above string concatenation – the system\nrewrites this apparent single string literal into the call string(greet, \", \", whom, \".\\n\").\n\nThe shortest complete expression after the $ is taken as the expression whose value is to be\ninterpolated into the string. Thus, you can interpolate any expression into a string using parentheses:\n\njulia> \"1 + 2 = $(1 + 2)\"\n\"1 + 2 = 3\"\n\nBoth concatenation and string interpolation call string to convert objects into string\nform. However, string actually just returns the output of print, so new types\nshould add methods to print or show instead of string.\n\nMost non-AbstractString objects are converted to strings closely corresponding to how\nthey are entered as literal expressions:\n\njulia> v = [1,2,3]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> \"v: $v\"\n\"v: [1, 2, 3]\"\n\nstring is the identity for AbstractString and AbstractChar values, so these are interpolated\ninto strings as themselves, unquoted and unescaped:\n\njulia> c = 'x'\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)\n\njulia> \"hi, $c\"\n\"hi, x\"\n\nTo include a literal $ in a string literal, escape it with a backslash:\n\njulia> print(\"I have \\$100 in my account.\\n\")\nI have $100 in my account."},{"title":"Triple-Quoted String Literals","page":"Strings","location":"manual/strings.html#Triple-Quoted-String-Literals","category":"section","text":"When strings are created using triple-quotes (\"\"\"...\"\"\") they have some special behavior that\ncan be useful for creating longer blocks of text.\n\nFirst, triple-quoted strings are also dedented to the level of the least-indented line.\nThis is useful for defining strings within code that is indented. For example:\n\njulia> str = \"\"\"\n           Hello,\n           world.\n         \"\"\"\n\"  Hello,\\n  world.\\n\"\n\nIn this case the final (empty) line before the closing \"\"\" sets the indentation level.\n\nThe dedentation level is determined as the longest common starting sequence of spaces or\ntabs in all lines, excluding the line following the opening \"\"\" and lines containing\nonly spaces or tabs (the line containing the closing \"\"\" is always included).\nThen for all lines, excluding the text following the opening \"\"\", the common starting\nsequence is removed (including lines containing only spaces and tabs if they start with\nthis sequence), e.g.:\n\njulia> \"\"\"    This\n         is\n           a test\"\"\"\n\"    This\\nis\\n  a test\"\n\nNext, if the opening \"\"\" is followed by a newline,\nthe newline is stripped from the resulting string.\n\n\"\"\"hello\"\"\"\n\nis equivalent to\n\n\"\"\"\nhello\"\"\"\n\nbut\n\n\"\"\"\n\nhello\"\"\"\n\nwill contain a literal newline at the beginning.\n\nStripping of the newline is performed after the dedentation. For example:\n\njulia> \"\"\"\n         Hello,\n         world.\"\"\"\n\"Hello,\\nworld.\"\n\nIf the newline is removed using a backslash, dedentation will be respected as well:\n\njulia> \"\"\"\n         Averylong\\\n         word\"\"\"\n\"Averylongword\"\n\nTrailing whitespace is left unaltered.\n\nTriple-quoted string literals can contain \" characters without escaping.\n\nNote that line breaks in literal strings, whether single- or triple-quoted, result in a newline\n(LF) character \\n in the string, even if your editor uses a carriage return \\r (CR) or CRLF\ncombination to end lines. To include a CR in a string, use an explicit escape \\r; for example,\nyou can enter the literal string \"a CRLF line ending\\r\\n\"."},{"title":"Common Operations","page":"Strings","location":"manual/strings.html#Common-Operations","category":"section","text":"You can lexicographically compare strings using the standard comparison operators:\n\njulia> \"abracadabra\" < \"xylophone\"\ntrue\n\njulia> \"abracadabra\" == \"xylophone\"\nfalse\n\njulia> \"Hello, world.\" != \"Goodbye, world.\"\ntrue\n\njulia> \"1 + 2 = 3\" == \"1 + 2 = $(1 + 2)\"\ntrue\n\nYou can search for the index of a particular character using the\nfindfirst and findlast functions:\n\njulia> findfirst('o', \"xylophone\")\n4\n\njulia> findlast('o', \"xylophone\")\n7\n\njulia> findfirst('z', \"xylophone\")\n\nYou can start the search for a character at a given offset by using\nthe functions findnext and findprev:\n\njulia> findnext('o', \"xylophone\", 1)\n4\n\njulia> findnext('o', \"xylophone\", 5)\n7\n\njulia> findprev('o', \"xylophone\", 5)\n4\n\njulia> findnext('o', \"xylophone\", 8)\n\nYou can use the occursin function to check if a substring is found within a string:\n\njulia> occursin(\"world\", \"Hello, world.\")\ntrue\n\njulia> occursin(\"o\", \"Xylophon\")\ntrue\n\njulia> occursin(\"a\", \"Xylophon\")\nfalse\n\njulia> occursin('o', \"Xylophon\")\ntrue\n\nThe last example shows that occursin can also look for a character literal.\n\nTwo other handy string functions are repeat and join:\n\njulia> repeat(\".:Z:.\", 10)\n\".:Z:..:Z:..:Z:..:Z:..:Z:..:Z:..:Z:..:Z:..:Z:..:Z:.\"\n\njulia> join([\"apples\", \"bananas\", \"pineapples\"], \", \", \" and \")\n\"apples, bananas and pineapples\"\n\nSome other useful functions include:\n\nfirstindex(str) gives the minimal (byte) index that can be used to index into str\n(always 1 for strings, not necessarily true for other containers).\nlastindex(str) gives the maximal (byte) index that can be used to index into str.\nlength(str) the number of characters in str.\nlength(str, i, j) the number of valid character indices in str from i to j.\nncodeunits(str) number of code units in a string.\ncodeunit(str, i) gives the code unit value in the string str at index i.\nthisind(str, i) given an arbitrary index into a string find the first index of the character into which the index points.\nnextind(str, i, n=1) find the start of the nth character starting after index i.\nprevind(str, i, n=1) find the start of the nth character starting before index i."},{"title":"Non-Standard String Literals","page":"Strings","location":"manual/strings.html#non-standard-string-literals","category":"section","text":"There are situations when you want to construct a string or use string semantics, but the behavior\nof the standard string construct is not quite what is needed. For these kinds of situations, Julia\nprovides non-standard string literals. A non-standard string literal looks like a regular\ndouble-quoted string literal,\nbut is immediately prefixed by an identifier, and may behave differently from a normal string literal.\n\nRegular expressions, byte array literals,\nand version number literals, as described below,\nare some examples of non-standard string literals. Users and packages may also define new non-standard string literals.\nFurther documentation is given in the Metaprogramming section."},{"title":"Regular Expressions","page":"Strings","location":"manual/strings.html#man-regex-literals","category":"section","text":"Sometimes you are not looking for an exact string, but a particular pattern. For\nexample, suppose you are trying to extract a single date from a large text file. You don’t\nknow what that date is (that’s why you are searching for it), but you do know it will look\nsomething like YYYY-MM-DD. Regular expressions allow you to specify these patterns and\nsearch for them.\n\nJulia uses version 2 of Perl-compatible regular expressions (regexes), as provided by the PCRE\nlibrary (see the PCRE2 syntax description\nfor more details). Regular expressions are related to strings in two ways: the obvious connection\nis that regular expressions are used to find regular patterns in strings; the other connection is\nthat regular expressions are themselves input as strings, which are parsed into a state machine that\ncan be used to efficiently search for patterns in strings. In Julia, regular expressions are input\nusing non-standard string literals prefixed with various identifiers beginning with r. The most\nbasic regular expression literal without any options turned on just uses r\"...\":\n\njulia> re = r\"^\\s*(?:#|$)\"\nr\"^\\s*(?:#|$)\"\n\njulia> typeof(re)\nRegex\n\nTo check if a regex matches a string, use occursin:\n\njulia> occursin(r\"^\\s*(?:#|$)\", \"not a comment\")\nfalse\n\njulia> occursin(r\"^\\s*(?:#|$)\", \"# a comment\")\ntrue\n\nAs one can see here, occursin simply returns true or false, indicating whether a\nmatch for the given regex occurs in the string. Commonly, however, one wants to know not\njust whether a string matched, but also how it matched. To capture this information about\na match, use the match function instead:\n\njulia> match(r\"^\\s*(?:#|$)\", \"not a comment\")\n\njulia> match(r\"^\\s*(?:#|$)\", \"# a comment\")\nRegexMatch(\"#\")\n\nIf the regular expression does not match the given string, match returns nothing\n– a special value that does not print anything at the interactive prompt. Other than not printing,\nit is a completely normal value and you can test for it programmatically:\n\nm = match(r\"^\\s*(?:#|$)\", line)\nif m === nothing\n    println(\"not a comment\")\nelse\n    println(\"blank or comment\")\nend\n\nIf a regular expression does match, the value returned by match is a RegexMatch\nobject. These objects record how the expression matches, including the substring that the pattern\nmatches and any captured substrings, if there are any. This example only captures the portion\nof the substring that matches, but perhaps we want to capture any non-blank text after the comment\ncharacter. We could do the following:\n\njulia> m = match(r\"^\\s*(?:#\\s*(.*?)\\s*$)\", \"# a comment \")\nRegexMatch(\"# a comment \", 1=\"a comment\")\n\nWhen calling match, you have the option to specify an index at which to start the\nsearch. For example:\n\njulia> m = match(r\"[0-9]\",\"aaaa1aaaa2aaaa3\",1)\nRegexMatch(\"1\")\n\njulia> m = match(r\"[0-9]\",\"aaaa1aaaa2aaaa3\",6)\nRegexMatch(\"2\")\n\njulia> m = match(r\"[0-9]\",\"aaaa1aaaa2aaaa3\",11)\nRegexMatch(\"3\")\n\nYou can extract the following info from a RegexMatch object:\n\nthe entire substring matched: m.match\nthe captured substrings as an array of strings: m.captures\nthe offset at which the whole match begins: m.offset\nthe offsets of the captured substrings as a vector: m.offsets\n\nFor when a capture doesn't match, instead of a substring, m.captures contains nothing in that\nposition, and m.offsets has a zero offset (recall that indices in Julia are 1-based, so a zero\noffset into a string is invalid). Here is a pair of somewhat contrived examples:\n\njulia> m = match(r\"(a|b)(c)?(d)\", \"acd\")\nRegexMatch(\"acd\", 1=\"a\", 2=\"c\", 3=\"d\")\n\njulia> m.match\n\"acd\"\n\njulia> m.captures\n3-element Vector{Union{Nothing, SubString{String}}}:\n \"a\"\n \"c\"\n \"d\"\n\njulia> m.offset\n1\n\njulia> m.offsets\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> m = match(r\"(a|b)(c)?(d)\", \"ad\")\nRegexMatch(\"ad\", 1=\"a\", 2=nothing, 3=\"d\")\n\njulia> m.match\n\"ad\"\n\njulia> m.captures\n3-element Vector{Union{Nothing, SubString{String}}}:\n \"a\"\n nothing\n \"d\"\n\njulia> m.offset\n1\n\njulia> m.offsets\n3-element Vector{Int64}:\n 1\n 0\n 2\n\nIt is convenient to have captures returned as an array so that one can use destructuring\nsyntax to bind them to local variables. As a convenience, the RegexMatch object\nimplements iterator methods that pass through to the captures field, so you can\ndestructure the match object directly:\n\njulia> first, second, third = m; first\n\"a\"\n\nCaptures can also be accessed by indexing the RegexMatch object with the number or name of the\ncapture group:\n\njulia> m=match(r\"(?<hour>\\d+):(?<minute>\\d+)\",\"12:45\")\nRegexMatch(\"12:45\", hour=\"12\", minute=\"45\")\n\njulia> m[:minute]\n\"45\"\n\njulia> m[2]\n\"45\"\n\nCaptures can be referenced in a substitution string when using replace by using \\n\nto refer to the nth capture group and prefixing the substitution string with s. Capture group\n0 refers to the entire match object. Named capture groups can be referenced in the substitution\nwith \\g<groupname>. For example:\n\njulia> replace(\"first second\", r\"(\\w+) (?<agroup>\\w+)\" => s\"\\g<agroup> \\1\")\n\"second first\"\n\nNumbered capture groups can also be referenced as \\g<n> for disambiguation, as in:\n\njulia> replace(\"a\", r\".\" => s\"\\g<0>1\")\n\"a1\"\n\nYou can modify the behavior of regular expressions by some combination of the flags i, m,\ns, and x after the closing double quote mark. These flags have the same meaning as they do\nin Perl, as explained in this excerpt from the perlre manpage:\n\ni   Do case-insensitive pattern matching.\n\n    If locale matching rules are in effect, the case map is taken\n    from the current locale for code points less than 255, and\n    from Unicode rules for larger code points. However, matches\n    that would cross the Unicode rules/non-Unicode rules boundary\n    (ords 255/256) will not succeed.\n\nm   Treat string as multiple lines. That is, change \"^\" and \"$\"\n    from matching the start or end of the string to matching the\n    start or end of any line anywhere within the string.\n\ns   Treat string as single line. That is, change \".\" to match any\n    character whatsoever, even a newline, which normally it would\n    not match.\n\n    Used together, as r\"\"ms, they let the \".\" match any character\n    whatsoever, while still allowing \"^\" and \"$\" to match,\n    respectively, just after and just before newlines within the\n    string.\n\nx   Tells the regular expression parser to ignore most whitespace\n    that is neither backslashed nor within a character class. You\n    can use this to break up your regular expression into\n    (slightly) more readable parts. The '#' character is also\n    treated as a metacharacter introducing a comment, just as in\n    ordinary code.\n\nFor example, the following regex has all three flags turned on:\n\njulia> r\"a+.*b+.*d$\"ism\nr\"a+.*b+.*d$\"ims\n\njulia> match(r\"a+.*b+.*d$\"ism, \"Goodbye,\\nOh, angry,\\nBad world\\n\")\nRegexMatch(\"angry,\\nBad world\")\n\nThe r\"...\" literal is constructed without interpolation and unescaping (except for\nquotation mark \" which still has to be escaped). Here is an example\nshowing the difference from standard string literals:\n\njulia> x = 10\n10\n\njulia> r\"$x\"\nr\"$x\"\n\njulia> \"$x\"\n\"10\"\n\njulia> r\"\\x\"\nr\"\\x\"\n\njulia> \"\\x\"\nERROR: ParseError:\n# Error @ none:1:2\n\"\\x\"\n#└┘ ── invalid hex escape sequence\n\nTriple-quoted regex strings, of the form r\"\"\"...\"\"\", are also supported (and may be convenient\nfor regular expressions containing quotation marks or newlines).\n\nThe Regex() constructor may be used to create a valid regex string programmatically.\nThis permits using the contents of string variables and other string operations when\nconstructing the regex string. Any of the regex codes above can be used within the single\nstring argument to Regex(). Here are some examples:\n\njulia> using Dates\n\njulia> d = Date(1962,7,10)\n1962-07-10\n\njulia> regex_d = Regex(\"Day \" * string(day(d)))\nr\"Day 10\"\n\njulia> match(regex_d, \"It happened on Day 10\")\nRegexMatch(\"Day 10\")\n\njulia> name = \"Jon\"\n\"Jon\"\n\njulia> regex_name = Regex(\"[\\\"( ]\\\\Q$name\\\\E[\\\") ]\")  # interpolate value of name\nr\"[\\\"( ]\\QJon\\E[\\\") ]\"\n\njulia> match(regex_name, \" Jon \")\nRegexMatch(\" Jon \")\n\njulia> match(regex_name, \"[Jon]\") === nothing\ntrue\n\nNote the use of the \\Q...\\E escape sequence. All characters between the \\Q and the \\E\nare interpreted as literal characters. This is convenient for matching characters that\nwould otherwise be regex metacharacters. However, caution is needed when using this feature\ntogether with string interpolation, since the interpolated string might itself contain\nthe \\E sequence, unexpectedly terminating literal matching. User inputs need to be sanitized\nbefore inclusion in a regex."},{"title":"Byte Array Literals","page":"Strings","location":"manual/strings.html#man-byte-array-literals","category":"section","text":"Another useful non-standard string literal is the byte-array string literal: b\"...\". This\nform lets you use string notation to express read only literal byte arrays – i.e. arrays of\nUInt8 values. The type of those objects is CodeUnits{UInt8, String}.\nThe rules for byte array literals are the following:\n\nASCII characters and ASCII escapes produce a single byte.\n\\x and octal escape sequences produce the byte corresponding to the escape value.\nUnicode escape sequences produce a sequence of bytes encoding that code point in UTF-8.\n\nThere is some overlap between these rules since the behavior of \\x and octal escapes less than\n0x80 (128) are covered by both of the first two rules, but here these rules agree. Together, these\nrules allow one to easily use ASCII characters, arbitrary byte values, and UTF-8 sequences to\nproduce arrays of bytes. Here is an example using all three:\n\njulia> b\"DATA\\xff\\u2200\"\n8-element Base.CodeUnits{UInt8, String}:\n 0x44\n 0x41\n 0x54\n 0x41\n 0xff\n 0xe2\n 0x88\n 0x80\n\nThe ASCII string \"DATA\" corresponds to the bytes 68, 65, 84, 65. \\xff produces the single byte 255.\nThe Unicode escape \\u2200 is encoded in UTF-8 as the three bytes 226, 136, 128. Note that the\nresulting byte array does not correspond to a valid UTF-8 string:\n\njulia> isvalid(\"DATA\\xff\\u2200\")\nfalse\n\nAs it was mentioned CodeUnits{UInt8, String} type behaves like read only array of UInt8 and\nif you need a standard vector you can convert it using Vector{UInt8}:\n\njulia> x = b\"123\"\n3-element Base.CodeUnits{UInt8, String}:\n 0x31\n 0x32\n 0x33\n\njulia> x[1]\n0x31\n\njulia> x[1] = 0x32\nERROR: CanonicalIndexError: setindex! not defined for Base.CodeUnits{UInt8, String}\n[...]\n\njulia> Vector{UInt8}(x)\n3-element Vector{UInt8}:\n 0x31\n 0x32\n 0x33\n\nAlso observe the significant distinction between \\xff and \\uff: the former escape sequence\nencodes the byte 255, whereas the latter escape sequence represents the code point 255, which\nis encoded as two bytes in UTF-8:\n\njulia> b\"\\xff\"\n1-element Base.CodeUnits{UInt8, String}:\n 0xff\n\njulia> b\"\\uff\"\n2-element Base.CodeUnits{UInt8, String}:\n 0xc3\n 0xbf\n\nCharacter literals use the same behavior.\n\nFor code points less than \\u80, it happens that the\nUTF-8 encoding of each code point is just the single byte produced by the corresponding \\x escape,\nso the distinction can safely be ignored. For the escapes \\x80 through \\xff as compared to\n\\u80 through \\uff, however, there is a major difference: the former escapes all encode single\nbytes, which – unless followed by very specific continuation bytes – do not form valid UTF-8\ndata, whereas the latter escapes all represent Unicode code points with two-byte encodings.\n\nIf this is all extremely confusing, try reading [\"The Absolute Minimum Every\nSoftware Developer Absolutely, Positively Must Know About Unicode and Character\nSets\"](https://www.joelonsoftware.com/2003/10/08/the-absolute-minimum-every-software-developer-absolutely-positively-must-know-about-unicode-and-character-sets-no-excuses/).\nIt is an excellent introduction to Unicode and UTF-8, and may help alleviate\nsome confusion regarding the matter."},{"title":"Version Number Literals","page":"Strings","location":"manual/strings.html#man-version-number-literals","category":"section","text":"Version numbers can easily be expressed with non-standard string literals of the form v\"...\".\nVersion number literals create VersionNumber objects which follow the\nspecifications of semantic versioning 2.0.0-rc2,\nand therefore are composed of major, minor and patch numeric values, followed by pre-release and\nbuild alphanumeric annotations. For example, v\"0.2.1-rc1+win64\" is broken into major version\n0, minor version 2, patch version 1, pre-release rc1 and build win64. When entering\na version literal, everything except the major version number is optional, therefore e.g.  v\"0.2\"\nis equivalent to v\"0.2.0\" (with empty pre-release/build annotations), v\"2\" is equivalent to\nv\"2.0.0\", and so on.\n\nVersionNumber objects are mostly useful to easily and correctly compare two (or more) versions.\nFor example, the constant VERSION holds Julia version number as a VersionNumber object, and\ntherefore one can define some version-specific behavior using simple statements as:\n\nif v\"0.2\" <= VERSION < v\"0.3-\"\n    # do something specific to 0.2 release series\nend\n\nNote that in the above example the non-standard version number v\"0.3-\" is used, with a trailing\n-: this notation is a Julia extension of the standard, and it is used to indicate a version which\nis lower than any 0.3 release, including all of its pre-releases. So in the above example the\ncode would only run with stable 0.2 versions, and exclude such versions as v\"0.3.0-rc1\". In\norder to also allow for unstable (i.e. pre-release) 0.2 versions, the lower bound check should\nbe modified like this: v\"0.2-\" <= VERSION.\n\nAnother non-standard version specification extension allows one to use a trailing + to express\nan upper limit on build versions, e.g.  VERSION > v\"0.2-rc1+\" can be used to mean any version\nabove 0.2-rc1 and any of its builds: it will return false for version v\"0.2-rc1+win64\" and\ntrue for v\"0.2-rc2\".\n\nIt is good practice to use such special versions in comparisons (particularly, the trailing -\nshould always be used on upper bounds unless there's a good reason not to), but they must not\nbe used as the actual version number of anything, as they are invalid in the semantic versioning\nscheme.\n\nBesides being used for the VERSION constant, VersionNumber objects are widely used\nin the Pkg module, to specify packages versions and their dependencies."},{"title":"Raw String Literals","page":"Strings","location":"manual/strings.html#man-raw-string-literals","category":"section","text":"Raw strings without interpolation or unescaping can be expressed with\nnon-standard string literals of the form raw\"...\". Raw string literals create\nordinary String objects which contain the enclosed contents exactly as\nentered with no interpolation or unescaping. This is useful for strings which\ncontain code or markup in other languages which use $ or \\ as special\ncharacters.\n\nThe exception is that quotation marks still must be escaped, e.g. raw\"\\\"\" is equivalent\nto \"\\\"\".\nTo make it possible to express all strings, backslashes then also must be escaped, but\nonly when appearing right before a quote character:\n\njulia> println(raw\"\\\\ \\\\\\\"\")\n\\\\ \\\"\n\nNotice that the first two backslashes appear verbatim in the output, since they do not\nprecede a quote character.\nHowever, the next backslash character escapes the backslash that follows it, and the\nlast backslash escapes a quote, since these backslashes appear before a quote."},{"title":"Annotated Strings","page":"Strings","location":"manual/strings.html#man-annotated-strings","category":"section","text":"note: Note\nThe API for AnnotatedStrings is considered experimental and is subject to change between\nJulia versions.\n\nIt is sometimes useful to be able to hold metadata relating to regions of a\nstring. A AnnotatedString wraps another string and\nallows for regions of it to be annotated with labelled values (:label => value).\nAll generic string operations are applied to the underlying string. However,\nwhen possible, styling information is preserved. This means you can manipulate a\nAnnotatedString —taking substrings, padding them,\nconcatenating them with other strings— and the metadata annotations will \"come\nalong for the ride\".\n\nThis string type is fundamental to the [StyledStrings stdlib](@ref\nstdlib-styledstrings), which uses :face-labelled annotations to hold styling\ninformation.\n\nWhen concatenating a AnnotatedString, take care to use\nannotatedstring instead of string if you want\nto keep the string annotations.\n\njulia> str = Base.AnnotatedString(\"hello there\",\n               [(1:5, :word, :greeting), (7:11, :label, 1)])\n\"hello there\"\n\njulia> length(str)\n11\n\njulia> lpad(str, 14)\n\"   hello there\"\n\njulia> typeof(lpad(str, 7))\nBase.AnnotatedString{String}\n\njulia> str2 = Base.AnnotatedString(\" julia\", [(2:6, :face, :magenta)])\n\" julia\"\n\njulia> Base.annotatedstring(str, str2)\n\"hello there julia\"\n\njulia> str * str2 == Base.annotatedstring(str, str2) # *-concatenation still works\ntrue\n\nThe annotations of a AnnotatedString can be accessed\nand modified via the annotations and\nannotate! functions."},{"title":"Unicode Input","page":"Unicode Input","location":"manual/unicode-input.html#Unicode-Input","category":"section","text":"The following table lists Unicode characters that can be entered via\ntab completion of LaTeX-like abbreviations in the Julia REPL (and\nin various other editing environments). You can also get information on how to\ntype a symbol by entering it in the REPL help, i.e. by typing ? and then\nentering the symbol in the REPL (e.g., by copy-paste from somewhere you saw\nthe symbol).\n\nwarning: Warning\nThis table may appear to contain missing characters in the second column, or even\nshow characters that are inconsistent with the characters as they are rendered in\nthe Julia REPL. In these cases, users are strongly advised to check their choice\nof fonts in their browser and REPL environment, as there are known issues with\nglyphs in many fonts.\n\n#\n# Generate a table containing all LaTeX and Emoji tab completions available in the REPL.\n#\nimport REPL, Markdown\nconst NBSP = '\\u00A0'\n\nfunction tab_completions(symbols...)\n    completions = Dict{String, Vector{String}}()\n    for each in symbols, (k, v) in each\n        completions[v] = push!(get!(completions, v, String[]), k)\n    end\n    return completions\nend\n\nfunction unicode_data()\n    file = normpath(@__DIR__, \"..\", \"..\", \"..\", \"..\", \"..\", \"doc\", \"UnicodeData.txt\")\n    names = Dict{UInt32, String}()\n    open(file) do unidata\n        for line in readlines(unidata)\n            id, name, desc = split(line, \";\")[[1, 2, 11]]\n            codepoint = parse(UInt32, \"0x$id\")\n            names[codepoint] = titlecase(lowercase(\n                name == \"\" ? desc : desc == \"\" ? name : \"$name / $desc\"))\n        end\n    end\n    return names\nend\n\n# Surround combining characters with no-break spaces (i.e '\\u00A0'). Follows the same format\n# for how unicode is displayed on the unicode.org website:\n# https://util.unicode.org/UnicodeJsps/character.jsp?a=0300\nfunction fix_combining_chars(char)\n    cat = Base.Unicode.category_code(char)\n    return cat == 6 || cat == 8 ? \"$NBSP$char$NBSP\" : \"$char\"\nend\n\nfunction table_entries(completions, unicode_dict)\n    entries = Any[Any[\n        [\"Code point(s)\"],\n        [\"Character(s)\"],\n        [\"Tab completion sequence(s)\"],\n        [\"Unicode name(s)\"],\n    ]]\n    for (chars, inputs) in sort!(collect(completions), by = first)\n        code_points, unicode_names, characters = String[], String[], String[]\n        for char in chars\n            push!(code_points, \"U+$(uppercase(string(UInt32(char), base = 16, pad = 5)))\")\n            push!(unicode_names, get(unicode_dict, UInt32(char), \"(No Unicode name)\"))\n            push!(characters, isempty(characters) ? fix_combining_chars(char) : \"$char\")\n        end\n        inputs_md = []\n        for (i, input) in enumerate(inputs)\n            i > 1 && push!(inputs_md, \", \")\n            push!(inputs_md, Markdown.Code(\"\", input))\n        end\n        push!(entries, [\n            [join(code_points, \" + \")],\n            [join(characters)],\n            inputs_md,\n            [join(unicode_names, \" + \")],\n        ])\n    end\n    table = Markdown.Table(entries, [:l, :c, :l, :l])\n    # We also need to wrap the Table in a Markdown.MD \"document\"\n    return Markdown.MD([table])\nend\n\ntable_entries(\n    tab_completions(\n        REPL.REPLCompletions.latex_symbols,\n        REPL.REPLCompletions.emoji_symbols\n    ),\n    unicode_data()\n)"},{"title":"Contributing to patch releases","page":"Contributing to patch releases","location":"devdocs/contributing/patch-releases.html#Contributing-to-patch-releases","category":"section","text":"The process of creating a patch release is roughly as follows:\n\nCreate a new branch (e.g. backports-release-1.10) against the relevant minor release\nbranch (e.g. release-1.10). Usually a corresponding pull request is created as well.\nAdd commits, nominally from master (hence \"backports\"), to that branch.\nSee below for more information on this process.\nRun the BaseBenchmarks.jl benchmark\nsuite and PkgEval.jl package ecosystem\nexerciser against that branch. Nominally BaseBenchmarks.jl and PkgEval.jl are\ninvoked via Nanosoldier.jl from\nthe pull request associated with the backports branch. Fix any issues.\nOnce all test and benchmark reports look good, merge the backports branch into\nthe corresponding release branch (e.g. merge backports-release-1.10 into\nrelease-1.10).\nOpen a pull request that bumps the version of the relevant minor release to the\nnext patch version, e.g. as in this pull request.\nPing @JuliaLang/releases to tag the patch release and update the website.\nOpen a pull request that bumps the version of the relevant minor release to the\nnext prerelease patch version, e.g. as in this pull request.\n\nStep 2 above, i.e. backporting commits to the backports-release-X.Y branch, has largely\nbeen automated via Backporter: Backporter\nsearches for merged pull requests with the relevant backport-X.Y tag, and attempts to\ncherry-pick the commits from those pull requests onto the backports-release-X.Y branch.\nSome commits apply successfully without intervention, others not so much. The latter\ncommits require \"manual\" backporting, with which help is generally much appreciated.\nBackporter generates a report identifying those commits it managed to backport automatically\nand those that require manual backporting; this report is usually copied into the first\npost of the pull request associated with backports-release-X.Y and maintained as\nadditional commits are automatically and/or manually backported.\n\nWhen contributing a manual backport, if you have the necessary permissions, please push the\nbackport directly to the backports-release-X.Y branch. If you lack the relevant\npermissions, please open a pull request against the backports-release-X.Y branch with the\nmanual backport. Once the manual backport is live on the backports-release-X.Y branch,\nplease remove the backport-X.Y tag from the originating pull request for the commits."},{"title":"Memory-mapped I/O","page":"Memory-mapped I/O","location":"stdlib/Mmap.html#Memory-mapped-I/O","category":"section","text":"Low level module for mmap (memory mapping of files)."},{"title":"Mmap.Anonymous","page":"Memory-mapped I/O","location":"stdlib/Mmap.html#Mmap.Anonymous","category":"type","text":"Mmap.Anonymous(name::AbstractString=\"\", readonly::Bool=false, create::Bool=true)\n\nCreate an IO-like object for creating zeroed-out mmapped-memory that is not tied to a file\nfor use in mmap. Used by SharedArray for creating shared memory arrays.\n\nExamples\n\njulia> using Mmap\n\njulia> anon = Mmap.Anonymous();\n\njulia> isreadable(anon)\ntrue\n\njulia> iswritable(anon)\ntrue\n\njulia> isopen(anon)\ntrue\n\n\n\n\n\n"},{"title":"Mmap.mmap","page":"Memory-mapped I/O","location":"stdlib/Mmap.html#Mmap.mmap","category":"function","text":"mmap(io::Union{IOStream,AbstractString,Mmap.AnonymousMmap}[, type::Type{Array{T,N}}, dims, offset]; grow::Bool=true, shared::Bool=true)\nmmap(type::Type{Array{T,N}}, dims)\n\nCreate an Array whose values are linked to a file, using memory-mapping. This provides a\nconvenient way of working with data too large to fit in the computer's memory.\n\nThe type is an Array{T,N} with a bits-type element of T and dimension N that\ndetermines how the bytes of the array are interpreted. Note that the file must be stored in\nbinary format, and no format conversions are possible (this is a limitation of operating\nsystems, not Julia).\n\ndims is a tuple or single Integer specifying the size or length of the array.\n\nThe file is passed via the stream argument, either as an open IOStream or filename string.\nWhen you initialize the stream, use \"r\" for a \"read-only\" array, and \"w+\" to create a\nnew array used to write values to disk.\n\nIf no type argument is specified, the default is Vector{UInt8}.\n\nOptionally, you can specify an offset (in bytes) if, for example, you want to skip over a\nheader in the file. The default value for the offset is the current stream position for an\nIOStream.\n\nThe grow keyword argument specifies whether the disk file should be grown to accommodate\nthe requested size of array (if the total file size is < requested array size). Write\nprivileges are required to grow the file.\n\nThe shared keyword argument specifies whether the resulting Array and changes made to it\nwill be visible to other processes mapping the same file.\n\nFor example, the following code\n\n# Create a file for mmapping\n# (you could alternatively use mmap to do this step, too)\nusing Mmap\nA = rand(1:20, 5, 30)\ns = open(\"/tmp/mmap.bin\", \"w+\")\n# We'll write the dimensions of the array as the first two Ints in the file\nwrite(s, size(A,1))\nwrite(s, size(A,2))\n# Now write the data\nwrite(s, A)\nclose(s)\n\n# Test by reading it back in\ns = open(\"/tmp/mmap.bin\")   # default is read-only\nm = read(s, Int)\nn = read(s, Int)\nA2 = mmap(s, Matrix{Int}, (m,n))\n\ncreates a m-by-n Matrix{Int}, linked to the file associated with stream s.\n\nA more portable file would need to encode the word size – 32 bit or 64 bit – and endianness\ninformation in the header. In practice, consider encoding binary data using standard formats\nlike HDF5 (which can be used with memory-mapping).\n\n\n\n\n\nmmap(io, BitArray, [dims, offset])\n\nCreate a BitArray whose values are linked to a file, using memory-mapping; it has the same\npurpose, works in the same way, and has the same arguments, as mmap, but\nthe byte representation is different.\n\nExamples\n\njulia> using Mmap\n\njulia> io = open(\"mmap.bin\", \"w+\");\n\njulia> B = mmap(io, BitArray, (25,30000));\n\njulia> B[3, 4000] = true;\n\njulia> Mmap.sync!(B);\n\njulia> close(io);\n\njulia> io = open(\"mmap.bin\", \"r+\");\n\njulia> C = mmap(io, BitArray, (25,30000));\n\njulia> C[3, 4000]\ntrue\n\njulia> C[2, 4000]\nfalse\n\njulia> close(io)\n\njulia> rm(\"mmap.bin\")\n\nThis creates a 25-by-30000 BitArray, linked to the file associated with stream io.\n\n\n\n\n\n"},{"title":"Mmap.sync!","page":"Memory-mapped I/O","location":"stdlib/Mmap.html#Mmap.sync!","category":"function","text":"Mmap.sync!(array)\n\nForces synchronization between the in-memory version of a memory-mapped Array or\nBitArray and the on-disk version.\n\n\n\n\n\n"},{"title":"macOS","page":"macOS","location":"devdocs/build/macos.html#macOS","category":"section","text":"You need to have the current Xcode command line utilities installed: run xcode-select --install in the terminal. You will need to rerun this terminal command after each macOS update, otherwise you may run into errors involving missing libraries or headers.\n\nThe dependent libraries are now built with BinaryBuilder and will be automatically downloaded. This is the preferred way to build Julia source. In case you want to build them all on your own, you will need a 64-bit gfortran to compile Julia dependencies.\n\nbrew install gcc\n\nIf you have set LD_LIBRARY_PATH or DYLD_LIBRARY_PATH in your .bashrc or equivalent, Julia may be unable to find various libraries that come bundled with it. These environment variables need to be unset for Julia to work."},{"title":"Serialization","page":"Serialization","location":"stdlib/Serialization.html#Serialization","category":"section","text":"Provides serialization of Julia objects."},{"title":"Recommended File Extension","page":"Serialization","location":"stdlib/Serialization.html#Recommended-File-Extension","category":"section","text":"While the Serialization module does not mandate a specific file extension, the Julia community commonly uses the .jls extension for serialized Julia files.\n\nExample:\n\nopen(\"model.jls\", \"w\") do io\n    serialize(io, my_model)\nend"},{"title":"Serialization.serialize","page":"Serialization","location":"stdlib/Serialization.html#Serialization.serialize","category":"function","text":"serialize(stream::IO, value)\n\nWrite an arbitrary value to a stream in an opaque format, such that it can be read back by\ndeserialize. The read-back value will be as identical as possible to the original,\nbut note that Ptr values are serialized as all-zero bit patterns (NULL).\n\nAn 8-byte identifying header is written to the stream first. To avoid writing the header,\nconstruct a Serializer and use it as the first argument to serialize instead.\nSee also Serialization.writeheader.\n\nThe data format can change in minor (1.x) Julia releases, but files written by prior 1.x\nversions will remain readable. The main exception to this is when the definition of a\ntype in an external package changes. If that occurs, it may be necessary to specify\nan explicit compatible version of the affected package in your environment.\nRenaming functions, even private functions, inside packages can also put existing files\nout of sync. Anonymous functions require special care: because their names are automatically\ngenerated, minor code changes can cause them to be renamed.\nSerializing anonymous functions should be avoided in files intended for long-term storage.\n\nIn some cases, the word size (32- or 64-bit) of the reading and writing machines must match.\nIn rarer cases the OS or architecture must also match, for example when using packages\nthat contain platform-dependent code.\n\n\n\n\n\nserialize(filename::AbstractString, value)\n\nOpen a file and serialize the given value to it.\n\ncompat: Julia 1.1\nThis method is available as of Julia 1.1.\n\n\n\n\n\n"},{"title":"Serialization.deserialize","page":"Serialization","location":"stdlib/Serialization.html#Serialization.deserialize","category":"function","text":"deserialize(stream)\n\nRead a value written by serialize. deserialize assumes the binary data read from\nstream is correct and has been serialized by a compatible implementation of serialize.\ndeserialize is designed for simplicity and performance, and so does not validate\nthe data read. Malformed data can result in process termination. The caller must ensure\nthe integrity and correctness of data read from stream.\n\n\n\n\n\ndeserialize(filename::AbstractString)\n\nOpen a file and deserialize its contents.\n\ncompat: Julia 1.1\nThis method is available as of Julia 1.1.\n\n\n\n\n\n"},{"title":"Serialization.writeheader","page":"Serialization","location":"stdlib/Serialization.html#Serialization.writeheader","category":"function","text":"Serialization.writeheader(s::AbstractSerializer)\n\nWrite an identifying header to the specified serializer. The header consists of\n8 bytes as follows:\n\nOffset Description\n0 tag byte (0x37)\n1-2 signature bytes \"JL\"\n3 protocol version\n4 bits 0-1: endianness: 0 = little, 1 = big\n4 bits 2-3: platform: 0 = 32-bit, 1 = 64-bit\n5-7 reserved\n\n\n\n\n\n"},{"title":"Binary distributions","page":"Binary distributions","location":"devdocs/build/distributing.html#Binary-distributions","category":"section","text":"These notes are for those wishing to compile a binary distribution of Julia\nfor distribution on various platforms. We love users spreading Julia as\nfar and wide as they can, trying it out on as wide an array of\noperating systems and hardware configurations as possible. As each\nplatform has specific gotchas and processes that must be followed in\norder to create a portable, working Julia distribution, we have\nseparated most of the notes by OS.\n\nNote that while the code for Julia is\nMIT-licensed, with a few exceptions,\nthe distribution created by the techniques described herein will be\nGPL licensed, as various dependent libraries such as SuiteSparse are\nGPL licensed. We do hope to have a non-GPL distribution of Julia in the future."},{"title":"Versioning and Git","page":"Binary distributions","location":"devdocs/build/distributing.html#Versioning-and-Git","category":"section","text":"The Makefile uses both the VERSION file and commit hashes and tags from the\ngit repository to generate the base/version_git.jl with information we use to\nfill the splash screen and the versioninfo() output. If you for some reason\ndon't want to have the git repository available when building you should\npregenerate the base/version_git.jl file with:\n\nmake -C base version_git.jl.phony\n\nJulia has lots of build dependencies where we use patched versions that has not\nyet been included by the popular package managers. These dependencies will usually\nbe automatically downloaded when you build, but if you want to be able to build\nJulia on a computer without internet access you should create a full-source-dist\narchive with the special make target\n\nmake full-source-dist\n\nthat creates a julia-version-commit.tar.gz archive with all required dependencies.\n\nWhen compiling a tagged release in the git repository, we don't display the\nbranch/commit hash info in the splash screen. You can use this line to show\na release description of up to 45 characters. To set this line you have\nto create a Make.user file containing:\n\noverride TAGGED_RELEASE_BANNER = \"my-package-repository build\""},{"title":"Target Architectures","page":"Binary distributions","location":"devdocs/build/distributing.html#Target-Architectures","category":"section","text":"By default, Julia optimizes its system image to the native architecture of\nthe build machine. This is usually not what you want when building packages,\nas it will make Julia fail at startup on any machine with incompatible CPUs\n(in particular older ones with more restricted instruction sets).\n\nWe therefore recommend that you pass the MARCH variable when calling make,\nsetting it to the baseline target you intend to support. This will determine\nthe target CPU for both the Julia executable and libraries, and the system\nimage (the latter can also be set using JULIA_CPU_TARGET). Typically useful\nvalues for x86 CPUs are x86-64 and core2 (for 64-bit builds) and\npentium4 (for 32-bit builds). Unfortunately, CPUs older than Pentium 4\nare currently not supported (see\nthis issue).\n\nThe full list of CPU targets supported by LLVM can be obtained by running\nllc -mattr=help."},{"title":"Linux","page":"Binary distributions","location":"devdocs/build/distributing.html#Linux","category":"section","text":"On Linux, make binary-dist creates a tarball that contains a fully\nfunctional Julia installation. If you wish to create a distribution\npackage such as a .deb, or .rpm, some extra effort is needed. See the\njulia-debian repository\nfor an example of what metadata is needed for creating .deb packages\nfor Debian and Ubuntu-based systems. See the\nFedora package\nfor RPM-based distributions. Although we have not yet experimented\nwith it, Alien could be used to\ngenerate Julia packages for various Linux distributions.\n\nJulia supports overriding standard installation directories via prefix\nand other environment variables you can pass when calling make and\nmake install. See Make.inc for their list. DESTDIR can also be used\nto force the installation into a temporary directory.\n\nBy default, Julia loads $prefix/etc/julia/startup.jl as an\ninstallation-wide initialization file. This file can be used by\ndistribution managers to set up custom paths or initialization code.\nFor Linux distribution packages, if $prefix is\nset to /usr, there is no /usr/etc to look into. This requires\nthe path to Julia's private etc directory to be changed. This can\nbe done via the sysconfdir make variable when building. Simply\npass sysconfdir=/etc to make when building and Julia will first\ncheck /etc/julia/startup.jl before trying\n$prefix/etc/julia/startup.jl."},{"title":"OS X","page":"Binary distributions","location":"devdocs/build/distributing.html#OS-X","category":"section","text":"To create a binary distribution on OSX, build Julia first, then cd to\ncontrib/mac/app, and run make with the same makevars that were used\nwith make when building Julia proper. This will then\ncreate a .dmg file in the contrib/mac/app directory holding a\ncompletely self-contained Julia.app.\n\nAlternatively, Julia may be built as a framework by invoking make with the\ndarwinframework target and DARWIN_FRAMEWORK=1 set. For example,\nmake DARWIN_FRAMEWORK=1 darwinframework."},{"title":"Windows","page":"Binary distributions","location":"devdocs/build/distributing.html#Windows","category":"section","text":"Instructions for creating a Julia distribution on Windows are described in the\nbuild devdocs for Windows."},{"title":"Notes on BLAS and LAPACK","page":"Binary distributions","location":"devdocs/build/distributing.html#Notes-on-BLAS-and-LAPACK","category":"section","text":"Julia builds OpenBLAS by default, which includes the BLAS and LAPACK\nlibraries. On 32-bit architectures, Julia builds OpenBLAS to use\n32-bit integers, while on 64-bit architectures, Julia builds OpenBLAS\nto use 64-bit integers (ILP64). It is essential that all Julia functions\nthat call BLAS and LAPACK API routines use integers of the correct width.\n\nMost BLAS and LAPACK distributions provided on linux distributions,\nand even commercial implementations ship libraries that use 32-bit\nAPIs. In many cases, a 64-bit API is provided as a separate library.\n\nWhen using vendor provided or OS provided libraries, a make option\ncalled USE_BLAS64 is available as part of the Julia build. When doing\nmake USE_BLAS64=0, Julia will call BLAS and LAPACK assuming a 32-bit\nAPI, where all integers are 32-bit wide, even on a 64-bit architecture.\n\nOther libraries that Julia uses, such as SuiteSparse also\nuse BLAS and LAPACK internally. The APIs need to be consistent across\nall libraries that depend on BLAS and LAPACK. The Julia build process\nwill build all these libraries correctly, but when overriding defaults\nand using system provided libraries, this consistency must be ensured.\n\nAlso note that Linux distributions sometimes ship several versions of\nOpenBLAS, some of which enable multithreading, and others only working\nin a serial fashion. For example, in Fedora, libopenblasp.so is threaded,\nbut libopenblas.so is not. We recommend using the former for optimal\nperformance. To choose an OpenBLAS library whose name is different from\nthe default libopenblas.so, pass LIBBLAS=-l$(YOURBLAS) and\nLIBBLASNAME=lib$(YOURBLAS) to make, replacing $(YOURBLAS) with the\nname of your library. You can also add .so.0 to the name of the library\nif you want your package to work without requiring the unversioned .so\nsymlink.\n\nFinally, OpenBLAS includes its own optimized version of LAPACK. If you\nset USE_SYSTEM_BLAS=1 and USE_SYSTEM_LAPACK=1, you should also set\nLIBLAPACK=-l$(YOURBLAS) and LIBLAPACKNAME=lib$(YOURBLAS). Else, the\nreference LAPACK will be used and performance will typically be much lower.\n\nStarting with Julia 1.7, Julia uses libblastrampoline\nto pick a different BLAS at runtime."},{"title":"Point releasing 101","page":"Binary distributions","location":"devdocs/build/distributing.html#Point-releasing-101","category":"section","text":"Creating a point/patch release consists of several distinct steps."},{"title":"Backporting commits","page":"Binary distributions","location":"devdocs/build/distributing.html#Backporting-commits","category":"section","text":"Some pull requests are labeled \"backport pending x.y\", e.g. \"backport pending 0.6\".\nThis designates that the next subsequent release tagged from the release-x.y branch\nshould include the commit(s) in that pull request.\nOnce the pull request is merged into master, each of the commits should be [cherry\npicked](https://git-scm.com/docs/git-cherry-pick) to a dedicated branch that will\nultimately be merged into release-x.y."},{"title":"Creating a backports branch","page":"Binary distributions","location":"devdocs/build/distributing.html#Creating-a-backports-branch","category":"section","text":"First, create a new branch based on release-x.y.\nThe typical convention for Julia branches is to prefix the branch name with your\ninitials if it's intended to be a personal branch.\nFor the sake of example, we'll say that the author of the branch is Jane Smith.\n\ngit fetch origin\ngit checkout release-x.y\ngit rebase origin/release-x.y\ngit checkout -b js/backport-x.y\n\nThis ensures that your local copy of release-x.y is up to date with origin before\nyou create a new branch from it."},{"title":"Cherry picking commits","page":"Binary distributions","location":"devdocs/build/distributing.html#Cherry-picking-commits","category":"section","text":"Now we do the actual backporting.\nFind all merged pull requests labeled \"backport pending x.y\" in the GitHub web UI.\nFor each of these, scroll to the bottom where it says \"someperson merged commit\n123abc into master XX minutes ago\".\nNote that the commit name is a link; if you click it, you'll be shown the contents\nof the commit.\nIf this page shows that 123abc is a merge commit, go back to the PR page—we\ndon't want merge commits, we want the actual commits.\nHowever, if this does not show a merge commit, it means that the PR was squash-merged.\nIn that case, use the git SHA of the commit, listed next to commit on this page.\n\nOnce you have the SHA of the commit, cherry-pick it onto the backporting branch:\n\ngit cherry-pick -x -e <sha>\n\nThere may be conflicts which need to be resolved manually.\nOnce conflicts are resolved (if applicable), add a reference to the GitHub pull\nrequest that introduced the commit in the body of the commit message.\n\nAfter all of the relevant commits are on the backports branch, push the branch to\nGitHub."},{"title":"Checking for performance regressions","page":"Binary distributions","location":"devdocs/build/distributing.html#Checking-for-performance-regressions","category":"section","text":"Point releases should never introduce performance regressions.\nLuckily the Julia benchmarking bot, Nanosoldier, can run benchmarks against any\nbranch, not just master.\nIn this case we want to check the benchmark results of js/backport-x.y against\nrelease-x.y.\nTo do this, awaken the Nanosoldier from his robotic slumber using a comment on\nyour backporting pull request:\n\n@nanosoldier `runbenchmarks(ALL, vs=\":release-x.y\")`\n\nThis will run all registered benchmarks on release-x.y and js/backport-x.y and\nproduce a summary of results, marking all improvements and regressions.\n\nIf Nanosoldier finds any regressions, try verifying locally and rerun Nanosoldier\nif necessary.\nIf the regressions are deemed to be real rather than just noise, you'll have to\nfind a commit on master to backport that fixes it if one exists, otherwise you\nshould determine what caused the regression and submit a patch (or get someone who\nknows the code to submit a patch) to master, then backport the commit once that's\nmerged.\n(Or submit a patch directly to the backport branch if appropriate.)"},{"title":"Building test binaries","page":"Binary distributions","location":"devdocs/build/distributing.html#Building-test-binaries","category":"section","text":"After the backport PR has been merged into the release-x.y branch, update your local\nclone of Julia, then get the SHA of the branch using\n\ngit rev-parse origin/release-x.y\n\nKeep that handy, as it's what you'll enter in the \"Revision\" field in the buildbot UI.\n\nFor now, all you need are binaries for Linux x86-64, since this is what's used for\nrunning PackageEvaluator.\nGo to https://buildog.julialang.org, submit a job for nuke_linux64, then queue up a\njob for package_linux64, providing the SHA as the revision.\nWhen the packaging job completes, it will upload the binary to the julialang2 bucket\non AWS.\nRetrieve the URL, as it will be used for PackageEvaluator."},{"title":"Checking for package breakages","page":"Binary distributions","location":"devdocs/build/distributing.html#Checking-for-package-breakages","category":"section","text":"Point releases should never break packages, with the possible exception of packages\nthat are doing some seriously questionable hacks using Base internals that are\nnot intended to be user-facing.\n(In those cases, maybe have a word with the package author.)\n\nChecking whether changes made in the forthcoming new version will break packages can\nbe accomplished using PackageEvaluator,\noften called \"PkgEval\" for short.\nPkgEval is what populates the status badges on GitHub repos and on pkg.julialang.org.\nIt typically runs on one of the non-benchmarking nodes of Nanosoldier and uses Vagrant\nto perform its duties in separate, parallel VirtualBox virtual machines."},{"title":"Setting up PackageEvaluator","page":"Binary distributions","location":"devdocs/build/distributing.html#Setting-up-PackageEvaluator","category":"section","text":"Clone PackageEvaluator and create a branch called backport-x.y.z, and check it out.\nNote that the required changes are a little hacky and confusing, and hopefully that will\nbe addressed in a future version of PackageEvaluator.\nThe changes to make will be modeled off of\nthis commit.\n\nThe setup script takes its first argument as the version of Julia to run and the second\nas the range of package names (AK for packages named A-K, LZ for L-Z).\nThe basic idea is that we're going to tweak that a bit to run only two versions of Julia,\nthe current x.y release and our backport version, each with three ranges of packages.\n\nIn the linked diff, we're saying that if the second argument is LZ, use the binaries\nbuilt from our backport branch, otherwise (AK) use the release binaries.\nThen we're using the first argument to run a section of the package list: A-F for input\n0.4, G-N for 0.5, and O-Z for 0.6."},{"title":"Running PackageEvaluator","page":"Binary distributions","location":"devdocs/build/distributing.html#Running-PackageEvaluator","category":"section","text":"To run PkgEval, find a hefty enough machine (such as Nanosoldier node 1), then run\n\ngit clone https://github.com/JuliaCI/PackageEvaluator.jl.git\ncd PackageEvaluator.jl/scripts\ngit checkout backport-x.y.z\n./runvagrant.sh\n\nThis produces some folders in the scripts/ directory.\nThe folder names and their contents are decoded below:\n\nFolder name Julia version Package range\n0.4AK Release A-F\n0.4LZ Backport A-F\n0.5AK Release G-N\n0.5LZ Backport G-N\n0.6AK Release O-Z\n0.6LZ Backport O-Z"},{"title":"Investigating results","page":"Binary distributions","location":"devdocs/build/distributing.html#Investigating-results","category":"section","text":"Once that's done, you can use ./summary.sh from that same directory to produce\na summary report of the findings.\nWe'll do so for each of the folders to aggregate overall results by version.\n\n./summary.sh 0.4AK/*.json > summary_release.txt\n./summary.sh 0.5AK/*.json >> summary_release.txt\n./summary.sh 0.6AK/*.json >> summary_release.txt\n./summary.sh 0.4LZ/*.json > summary_backport.txt\n./summary.sh 0.5LZ/*.json >> summary_backport.txt\n./summary.sh 0.6LZ/*.json >> summary_backport.txt\n\nNow we have two files, summary_release.txt and summary_backport.txt, containing\nthe PackageEvaluator test results (pass/fail) for each package for the two versions.\n\nTo make these easier to ingest into a Julia, we'll convert them into CSV files then\nuse the DataFrames package to process the results.\nTo convert to CSV, copy each .txt file to a corresponding .csv file, then enter Vim\nand execute ggVGI\"<esc> then :%s/\\.json /\",/g.\n(You don't have to use Vim; this just is one way to do it.)\nNow process the results with Julia code similar to the following.\n\nusing DataFrames\n\nrelease = readtable(\"summary_release.csv\", header=false, names=[:package, :release])\nbackport = readtable(\"summary_backport.csv\", header=false, names=[:package, :backport])\n\nresults = join(release, backport, on=:package, kind=:outer)\n\nfor result in eachrow(results)\n    a = result[:release]\n    b = result[:backport]\n    if (isna(a) && !isna(b)) || (isna(b) && !isna(a))\n        color = :yellow\n    elseif a != b && occursin(\"pass\", b)\n        color = :green\n    elseif a != b\n        color = :red\n    else\n        continue\n    end\n    printstyled(result[:package], \": Release \", a, \" -> Backport \", b, \"\\n\", color=color)\nend\n\nThis will write color-coded lines to stdout.\nAll lines in red must be investigated as they signify potential breakages caused by the\nbackport version.\nLines in yellow should be looked into since it means a package ran on one version but\nnot on the other for some reason.\nIf you find that your backported branch is causing breakages, use git bisect to\nidentify the problematic commits, git revert those commits, and repeat the process."},{"title":"Merging backports into the release branch","page":"Binary distributions","location":"devdocs/build/distributing.html#Merging-backports-into-the-release-branch","category":"section","text":"After you have ensured that\n\nthe backported commits pass all of Julia's unit tests,\nthere are no performance regressions introduced by the backported commits as compared\nto the release branch, and\nthe backported commits do not break any registered packages,\n\nthen the backport branch is ready to be merged into release-x.y.\nOnce it's merged, go through and remove the \"backport pending x.y\" label from all pull\nrequests containing the commits that have been backported.\nDo not remove the label from PRs that have not been backported.\n\nThe release-x.y branch should now contain all of the new commits.\nThe last thing we want to do to the branch is to adjust the version number.\nTo do this, submit a PR against release-x.y that edits the VERSION file to remove -pre\nfrom the version number.\nOnce that's merged, we're ready to tag."},{"title":"Tagging the release","page":"Binary distributions","location":"devdocs/build/distributing.html#Tagging-the-release","category":"section","text":"It's time!\nCheck out the release-x.y branch and make sure that your local copy of the branch is\nup to date with the remote branch.\nAt the command line, run\n\ngit tag v$(cat VERSION)\ngit push --tags\n\nThis creates the tag locally and pushes it to GitHub.\n\nAfter tagging the release, submit another PR to release-x.y to bump the patch number\nand add -pre back to the end.\nThis denotes that the branch state reflects a prerelease version of the next point\nrelease in the x.y series.\n\nFollow the remaining directions in the Makefile."},{"title":"Signing binaries","page":"Binary distributions","location":"devdocs/build/distributing.html#Signing-binaries","category":"section","text":"Some of these steps will require secure passwords.\nTo obtain the appropriate passwords, contact Elliot Saba (staticfloat) or Alex Arslan\n(ararslan).\nNote that code signing for each platform must be performed on that platform (e.g. Windows\nsigning must be done on Windows, etc.)."},{"title":"Linux","page":"Binary distributions","location":"devdocs/build/distributing.html#Linux-2","category":"section","text":"Code signing must be done manually on Linux, but it's quite simple.\nFirst obtain the file julia.key from the CodeSigning folder in the juliasecure AWS\nbucket.\nAdd this to your GnuPG keyring using\n\ngpg --import julia.key\n\nThis will require entering a password that you must obtain from Elliot or Alex.\nNext, set the trust level for the key to maximum.\nStart by entering a gpg session:\n\ngpg --edit-key julia\n\nAt the prompt, type trust, then when asked for a trust level, provide the maximum\navailable (likely 5).\nExit GnuPG.\n\nNow, for each of the Linux tarballs that were built on the buildbots, enter\n\ngpg -u julia --armor --detach-sig julia-x.y.z-linux-<arch>.tar.gz\n\nThis will produce a corresponding .asc file for each tarball.\nAnd that's it!"},{"title":"macOS","page":"Binary distributions","location":"devdocs/build/distributing.html#macOS","category":"section","text":"Code signing should happen automatically on the macOS buildbots.\nHowever, it's important to verify that it was successful.\nOn a system or virtual machine running macOS, download the .dmg file that was built on\nthe buildbots.\nFor the sake of example, say that the .dmg file is called julia-x.y.z-osx.dmg.\nRun\n\nmkdir ./jlmnt\nhdiutil mount -readonly -mountpoint ./jlmnt julia-x.y.z-osx.dmg\ncodesign -v jlmnt/Julia-x.y.app\n\nBe sure to note the name of the mounted disk listed when mounting!\nFor the sake of example, we'll assume this is disk3.\nIf the code signing verification exited successfully, there will be no output from the\ncodesign step.\nIf it was indeed successful, you can detach the .dmg now:\n\nhdiutil eject /dev/disk3\nrm -rf ./jlmnt\n\nIf you get a message like\n\nJulia-x.y.app: code object is not signed at all\n\nthen you'll need to sign manually.\n\nTo sign manually, first retrieve the OS X certificates from the CodeSigning folder\nin the juliasecure bucket on AWS.\nAdd the .p12 file to your keychain using Keychain.app.\nAsk Elliot Saba (staticfloat) or Alex Arslan (ararslan) for the password for the key.\nNow run\n\nhdiutil convert julia-x.y.z-osx.dmg -format UDRW -o julia-x.y.z-osx_writable.dmg\nmkdir ./jlmnt\nhdiutil mount -mountpoint julia-x.y.z-osx_writable.dmg\ncodesign -s \"AFB379C0B4CBD9DB9A762797FC2AB5460A2B0DBE\" --deep jlmnt/Julia-x.y.app\n\nThis may fail with a message like\n\nJulia-x.y.app: resource fork, Finder information, or similar detritus not allowed\n\nIf that's the case, you'll need to remove extraneous attributes:\n\nxattr -cr jlmnt/Julia-x.y.app\n\nThen retry code signing.\nIf that produces no errors, retry verification.\nIf all is now well, unmount the writable .dmg and convert it back to read-only:\n\nhdiutil eject /dev/disk3\nrm -rf ./jlmnt\nhdiutil convert julia-x.y.z-osx_writable.dmg -format UDZO -o julia-x.y.z-osx_fixed.dmg\n\nVerify that the resulting .dmg is in fact fixed by double clicking it.\nIf everything looks good, eject it then drop the _fixed suffix from the name.\nAnd that's it!"},{"title":"Windows","page":"Binary distributions","location":"devdocs/build/distributing.html#Windows-2","category":"section","text":"Signing must be performed manually on Windows.\nFirst obtain the Windows 10 SDK, which contains the necessary signing utilities, from\nthe Microsoft website.\nWe need the SignTool utility which should have been installed somewhere like\nC:\\Program Files (x86)\\Windows Kits\\10\\App Certification Kit.\nGrab the Windows certificate files from CodeSigning on juliasecure and put them\nin the same directory as the executables.\nOpen a Windows CMD window, cd to where all the files are, and run\n\nset PATH=%PATH%;C:\\Program Files (x86)\\Windows Kits\\10\\App Certification Kit;\nsigntool sign /f julia-windows-code-sign_2017.p12 /p \"PASSWORD\" ^\n   /t http://timestamp.verisign.com/scripts/timstamp.dll ^\n   /v julia-x.y.z-win32.exe\n\nNote that ^ is a line continuation character in Windows CMD and PASSWORD is a\nplaceholder for the password for this certificate.\nAs usual, contact Elliot or Alex for passwords.\nIf there are no errors, we're all good!"},{"title":"Uploading binaries","page":"Binary distributions","location":"devdocs/build/distributing.html#Uploading-binaries","category":"section","text":"Now that everything is signed, we need to upload the binaries to AWS.\nYou can use a program like Cyberduck or the aws command line utility.\nThe binaries should go in the julialang2 bucket in the appropriate folders.\nFor example, Linux x86-64 goes in julialang2/bin/linux/x.y.\nBe sure to delete the current julia-x.y-latest-linux-<arch>.tar.gz file and replace\nit with a duplicate of julia-x.y.z-linux-<arch>.tar.gz.\n\nWe also need to upload the checksums for everything we've built, including the source\ntarballs and all release binaries.\nThis is simple:\n\nshasum -a 256 julia-x.y.z* | grep -v -e sha256 -e md5 -e asc > julia-x.y.z.sha256\nmd5sum julia-x.y.z* | grep -v -e sha256 -e md5 -e asc > julia-x.y.z.md5\n\nNote that if you're running those commands on macOS, you'll get very slightly different\noutput, which can be reformatted by looking at an existing file.\nMac users will also need to use md5 -r instead of md5sum.\nUpload the .md5 and .sha256 files to julialang2/bin/checksums on AWS.\n\nEnsure that the permissions on AWS for all uploaded files are set to \"Everyone: READ.\"\n\nFor each file we've uploaded, we need to purge the Fastly cache so that the links on\nthe website point to the updated files.\nAs an example:\n\ncurl -X PURGE https://julialang-s3.julialang.org/bin/checksums/julia-x.y.z.sha256\n\nSometimes this isn't necessary but it's good to do anyway."},{"title":"Iteration utilities","page":"Iteration utilities","location":"base/iterators.html#Iteration-utilities","category":"section","text":""},{"title":"Base.Iterators.Stateful","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.Stateful","category":"type","text":"Stateful(itr)\n\nThere are several different ways to think about this iterator wrapper:\n\nIt provides a mutable wrapper around an iterator and\nits iteration state.\nIt turns an iterator-like abstraction into a Channel-like\nabstraction.\nIt's an iterator that mutates to become its own rest iterator\nwhenever an item is produced.\n\nStateful provides the regular iterator interface. Like other mutable iterators\n(e.g. Base.Channel), if iteration is stopped early (e.g. by a break in a for loop),\niteration can be resumed from the same spot by continuing to iterate over the\nsame iterator object (in contrast, an immutable iterator would restart from the\nbeginning).\n\nExamples\n\njulia> a = Iterators.Stateful(\"abcdef\");\n\njulia> isempty(a)\nfalse\n\njulia> popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(Iterators.take(a, 3))\n3-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n 'd': ASCII/Unicode U+0064 (category Ll: Letter, lowercase)\n\njulia> collect(a)\n2-element Vector{Char}:\n 'e': ASCII/Unicode U+0065 (category Ll: Letter, lowercase)\n 'f': ASCII/Unicode U+0066 (category Ll: Letter, lowercase)\n\njulia> Iterators.reset!(a); popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> Iterators.reset!(a, \"hello\"); popfirst!(a)\n'h': ASCII/Unicode U+0068 (category Ll: Letter, lowercase)\n\njulia> a = Iterators.Stateful([1,1,1,2,3,4]);\n\njulia> for x in a; x == 1 || break; end\n\njulia> peek(a)\n3\n\njulia> sum(a) # Sum the remaining elements\n7\n\n\n\n\n\n"},{"title":"Base.Iterators.zip","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.zip","category":"function","text":"zip(iters...)\n\nRun multiple iterators at the same time, until any of them is exhausted. The value type of\nthe zip iterator is a tuple of values of its subiterators.\n\nnote: Note\nzip orders the calls to its subiterators in such a way that stateful iterators will\nnot advance when another iterator finishes in the current iteration.\n\nnote: Note\nzip() with no arguments yields an infinite iterator of empty tuples.\n\nSee also: enumerate, Base.splat.\n\nExamples\n\njulia> a = 1:5\n1:5\n\njulia> b = [\"e\",\"d\",\"b\",\"c\",\"a\"]\n5-element Vector{String}:\n \"e\"\n \"d\"\n \"b\"\n \"c\"\n \"a\"\n\njulia> c = zip(a,b)\nzip(1:5, [\"e\", \"d\", \"b\", \"c\", \"a\"])\n\njulia> length(c)\n5\n\njulia> first(c)\n(1, \"e\")\n\n\n\n\n\n"},{"title":"Base.Iterators.enumerate","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.enumerate","category":"function","text":"enumerate(iter)\n\nAn iterator that yields (i, x) where i is a counter starting at 1,\nand x is the ith value from the given iterator. It's useful when\nyou need not only the values x over which you are iterating, but\nalso the number of iterations so far.\n\nNote that i may not be valid for indexing iter, or may index a\ndifferent element. This will happen if iter has indices that do not\nstart at 1, and may happen for strings, dictionaries, etc.\nSee the pairs(IndexLinear(), iter) method if you want to ensure that i is an index.\n\nExamples\n\njulia> a = [\"a\", \"b\", \"c\"];\n\njulia> for (index, value) in enumerate(a)\n           println(\"$index $value\")\n       end\n1 a\n2 b\n3 c\n\njulia> str = \"naïve\";\n\njulia> for (i, val) in enumerate(str)\n           print(\"i = \", i, \", val = \", val, \", \")\n           try @show(str[i]) catch e println(e) end\n       end\ni = 1, val = n, str[i] = 'n'\ni = 2, val = a, str[i] = 'a'\ni = 3, val = ï, str[i] = 'ï'\ni = 4, val = v, StringIndexError(\"naïve\", 4)\ni = 5, val = e, str[i] = 'v'\n\n\n\n\n\n"},{"title":"Base.Iterators.rest","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.rest","category":"function","text":"rest(iter, state)\n\nAn iterator that yields the same elements as iter, but starting at the given state, which\nmust be a state obtainable via a sequence of one or more calls to iterate(iter[, state])\n\nSee also: Iterators.drop, Iterators.peel, Base.rest.\n\nExamples\n\njulia> iter = [1,2,3,4];\n\njulia> val, state = iterate(iter)\n(1, 2)\n\njulia> collect(Iterators.rest(iter, state))\n3-element Vector{Int64}:\n 2\n 3\n 4\n\n\n\n\n\n"},{"title":"Base.Iterators.countfrom","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.countfrom","category":"function","text":"countfrom(start=1, step=1)\n\nAn iterator that counts forever, starting at start and incrementing by step.\n\nExamples\n\njulia> for v in Iterators.countfrom(5, 2)\n           v > 10 && break\n           println(v)\n       end\n5\n7\n9\n\n\n\n\n\n"},{"title":"Base.Iterators.take","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.take","category":"function","text":"take(iter, n)\n\nAn iterator that generates at most the first n elements of iter.\n\nSee also: drop, peel, first, Base.take!.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.take(a,3))\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n"},{"title":"Base.Iterators.takewhile","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.takewhile","category":"function","text":"takewhile(pred, iter)\n\nAn iterator that generates element from iter as long as predicate pred is true,\nafterwards, drops every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.takewhile(<(3),s))\n2-element Vector{Int64}:\n 1\n 2\n\n\n\n\n\n"},{"title":"Base.Iterators.drop","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.drop","category":"function","text":"drop(iter, n)\n\nAn iterator that generates all but the first n elements of iter.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.drop(a,4))\n2-element Vector{Int64}:\n  9\n 11\n\n\n\n\n\n"},{"title":"Base.Iterators.dropwhile","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.dropwhile","category":"function","text":"dropwhile(pred, iter)\n\nAn iterator that drops element from iter as long as predicate pred is true,\nafterwards, returns every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.dropwhile(<(3),s))\n3-element Vector{Int64}:\n 3\n 4\n 5\n\n\n\n\n\n"},{"title":"Base.Iterators.findeach","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.findeach","category":"function","text":"findeach(f, it)\nfindeach(it)\n\nAn iterator that generates every key from the key/value pairs of pairs(it),\nwhere f(value) returns true.\n\nIf f is not specified, default to identity.\n\nIterators.findeach is the lazy equivalent of findall.\n\ncompat: Julia 1.13\nfindeach requires at least Julia 1.13.\n\nExamples\n\njulia> collect(Iterators.findeach(isodd, Dict(2 => 3, 3 => 2)))\n1-element Vector{Int64}:\n 2\n\njulia> only(Iterators.findeach(==(1), [3,6,2,1]))\n4\n\n\n\n\n\n"},{"title":"Base.Iterators.cycle","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.cycle","category":"function","text":"cycle(iter[, n::Int])\n\nAn iterator that cycles through iter forever.\nIf n is specified, then it cycles through iter that many times.\nWhen iter is empty, so are cycle(iter) and cycle(iter, n).\n\nIterators.cycle(iter, n) is the lazy equivalent of Base.repeat(vector, n),\nwhile Iterators.repeated(iter, n) is the lazy Base.fill(item, n).\n\ncompat: Julia 1.11\nThe method cycle(iter, n) was added in Julia 1.11.\n\nExamples\n\njulia> for (i, v) in enumerate(Iterators.cycle(\"hello\"))\n           print(v)\n           i > 10 && break\n       end\nhellohelloh\n\njulia> foreach(print, Iterators.cycle(['j', 'u', 'l', 'i', 'a'], 3))\njuliajuliajulia\n\njulia> repeat([1,2,3], 4) == collect(Iterators.cycle([1,2,3], 4))\ntrue\n\njulia> fill([1,2,3], 4) == collect(Iterators.repeated([1,2,3], 4))\ntrue\n\n\n\n\n\n"},{"title":"Base.Iterators.repeated","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.repeated","category":"function","text":"repeated(x[, n::Int])\n\nAn iterator that generates the value x forever. If n is specified, generates x that\nmany times (equivalent to take(repeated(x), n)).\n\nSee also fill, and compare Iterators.cycle.\n\nExamples\n\njulia> a = Iterators.repeated([1 2], 4);\n\njulia> collect(a)\n4-element Vector{Matrix{Int64}}:\n [1 2]\n [1 2]\n [1 2]\n [1 2]\n\njulia> ans == fill([1 2], 4)\ntrue\n\njulia> Iterators.cycle([1 2], 4) |> collect |> println\n[1, 2, 1, 2, 1, 2, 1, 2]\n\n\n\n\n\n"},{"title":"Base.Iterators.product","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.product","category":"function","text":"product(iters...)\n\nReturn an iterator over the product of several iterators. Each generated element is\na tuple whose ith element comes from the ith argument iterator. The first iterator\nchanges the fastest.\n\nSee also: zip, Iterators.flatten.\n\nExamples\n\njulia> collect(Iterators.product(1:2, 3:5))\n2×3 Matrix{Tuple{Int64, Int64}}:\n (1, 3)  (1, 4)  (1, 5)\n (2, 3)  (2, 4)  (2, 5)\n\njulia> ans == [(x,y) for x in 1:2, y in 3:5]  # collects a generator involving Iterators.product\ntrue\n\n\n\n\n\n"},{"title":"Base.Iterators.flatten","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.flatten","category":"function","text":"flatten(iter)\n\nGiven an iterator that yields iterators, return an iterator that yields the\nelements of those iterators.\nPut differently, the elements of the argument iterator are concatenated.\n\nExamples\n\njulia> collect(Iterators.flatten((1:2, 8:9)))\n4-element Vector{Int64}:\n 1\n 2\n 8\n 9\n\njulia> [(x,y) for x in 0:1 for y in 'a':'c']  # collects generators involving Iterators.flatten\n6-element Vector{Tuple{Int64, Char}}:\n (0, 'a')\n (0, 'b')\n (0, 'c')\n (1, 'a')\n (1, 'b')\n (1, 'c')\n\n\n\n\n\n"},{"title":"Base.Iterators.flatmap","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.flatmap","category":"function","text":"Iterators.flatmap(f, iterators...)\n\nEquivalent to flatten(map(f, iterators...)).\n\nSee also Iterators.flatten, Iterators.map.\n\ncompat: Julia 1.9\nThis function was added in Julia 1.9.\n\nExamples\n\njulia> Iterators.flatmap(n -> -n:2:n, 1:3) |> collect\n9-element Vector{Int64}:\n -1\n  1\n -2\n  0\n  2\n -3\n -1\n  1\n  3\n\njulia> stack(n -> -n:2:n, 1:3)\nERROR: DimensionMismatch: stack expects uniform slices, got axes(x) == (1:3,) while first had (1:2,)\n[...]\n\njulia> Iterators.flatmap(n -> (-n, 10n), 1:2) |> collect\n4-element Vector{Int64}:\n -1\n 10\n -2\n 20\n\njulia> ans == vec(stack(n -> (-n, 10n), 1:2))\ntrue\n\n\n\n\n\n"},{"title":"Base.Iterators.partition","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.partition","category":"function","text":"partition(collection, n)\n\nIterate over a collection n elements at a time.\n\nExamples\n\njulia> collect(Iterators.partition([1,2,3,4,5], 2))\n3-element Vector{SubArray{Int64, 1, Vector{Int64}, Tuple{UnitRange{Int64}}, true}}:\n [1, 2]\n [3, 4]\n [5]\n\n\n\n\n\n"},{"title":"Base.Iterators.map","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.map","category":"function","text":"Iterators.map(f, iterators...)\n\nCreate a lazy mapping.  This is another syntax for writing\n(f(args...) for args in zip(iterators...)).\n\ncompat: Julia 1.6\nThis function requires at least Julia 1.6.\n\nExamples\n\njulia> collect(Iterators.map(x -> x^2, 1:3))\n3-element Vector{Int64}:\n 1\n 4\n 9\n\n\n\n\n\n"},{"title":"Base.Iterators.filter","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.filter","category":"function","text":"Iterators.filter(flt, itr)\n\nGiven a predicate function flt and an iterable object itr, return an\niterable object which upon iteration yields the elements x of itr that\nsatisfy flt(x). The order of the original iterator is preserved.\n\nThis function is lazy; that is, it is guaranteed to return in Θ(1) time\nand use Θ(1) additional space, and flt will not be called by an\ninvocation of filter. Calls to flt will be made when iterating over the\nreturned iterable object. These calls are not cached and repeated calls will be\nmade when reiterating.\n\nwarning: Warning\nSubsequent lazy transformations on the iterator returned from filter, such\nas those performed by Iterators.reverse or cycle, will also delay calls to flt\nuntil collecting or iterating over the returned iterable object. If the filter\npredicate is nondeterministic or its return values depend on the order of iteration\nover the elements of itr, composition with lazy transformations may result in\nsurprising behavior. If this is undesirable, either ensure that flt is a pure\nfunction or collect intermediate filter iterators before further transformations.\n\nSee Base.filter for an eager implementation of filtering for arrays.\n\nExamples\n\njulia> f = Iterators.filter(isodd, [1, 2, 3, 4, 5])\nBase.Iterators.Filter{typeof(isodd), Vector{Int64}}(isodd, [1, 2, 3, 4, 5])\n\njulia> foreach(println, f)\n1\n3\n5\n\njulia> [x for x in [1, 2, 3, 4, 5] if isodd(x)]  # collects a generator over Iterators.filter\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n"},{"title":"Base.Iterators.accumulate","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.accumulate","category":"function","text":"Iterators.accumulate(f, itr; [init])\n\nGiven a 2-argument function f and an iterator itr, return a new\niterator that successively applies f to the previous value and the\nnext element of itr.\n\nThis is effectively a lazy version of Base.accumulate.\n\ncompat: Julia 1.5\nKeyword argument init is added in Julia 1.5.\n\nExamples\n\njulia> a = Iterators.accumulate(+, [1,2,3,4]);\n\njulia> foreach(println, a)\n1\n3\n6\n10\n\njulia> b = Iterators.accumulate(/, (2, 5, 2, 5); init = 100);\n\njulia> collect(b)\n4-element Vector{Float64}:\n 50.0\n 10.0\n  5.0\n  1.0\n\n\n\n\n\n"},{"title":"Base.Iterators.reverse","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.reverse","category":"function","text":"Iterators.reverse(itr)\n\nGiven an iterator itr, then reverse(itr) is an iterator over the\nsame collection but in the reverse order.\nThis iterator is \"lazy\" in that it does not make a copy of the collection in\norder to reverse it; see Base.reverse for an eager implementation.\n\n(By default, this returns\nan Iterators.Reverse object wrapping itr, which is iterable\nif the corresponding iterate methods are defined, but some itr types\nmay implement more specialized Iterators.reverse behaviors.)\n\nNot all iterator types T support reverse-order iteration.  If T\ndoesn't, then iterating over Iterators.reverse(itr::T) will throw a MethodError\nbecause of the missing iterate methods for Iterators.Reverse{T}.\n(To implement these methods, the original iterator\nitr::T can be obtained from an r::Iterators.Reverse{T} object by r.itr;\nmore generally, one can use Iterators.reverse(r).)\n\nExamples\n\njulia> foreach(println, Iterators.reverse(1:5))\n5\n4\n3\n2\n1\n\n\n\n\n\n"},{"title":"Base.Iterators.only","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.only","category":"function","text":"only(x)\n\nReturn the one and only element of collection x, or throw an ArgumentError if the\ncollection has zero or multiple elements.\n\nSee also first, last.\n\ncompat: Julia 1.4\nThis method requires at least Julia 1.4.\n\nExamples\n\njulia> only([\"a\"])\n\"a\"\n\njulia> only(\"a\")\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> only(())\nERROR: ArgumentError: Tuple contains 0 elements, must contain exactly 1 element\nStacktrace:\n[...]\n\njulia> only(('a', 'b'))\nERROR: ArgumentError: Tuple contains 2 elements, must contain exactly 1 element\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Base.Iterators.peel","page":"Iteration utilities","location":"base/iterators.html#Base.Iterators.peel","category":"function","text":"peel(iter)\n\nReturns the first element and an iterator over the remaining elements.\n\nIf the iterator is empty return nothing (like iterate).\n\ncompat: Julia 1.7\nPrior versions throw a BoundsError if the iterator is empty.\n\nSee also: Iterators.drop, Iterators.take.\n\nExamples\n\njulia> (a, rest) = Iterators.peel(\"abc\");\n\njulia> a\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(rest)\n2-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n\n\n\n\n\n"},{"title":"Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html#Environment-Variables","category":"section","text":"Julia can be configured with a number of environment variables, set either in\nthe usual way for each operating system, or in a portable way from within Julia.\nSupposing that you want to set the environment variable JULIA_EDITOR to vim,\nyou can type ENV[\"JULIA_EDITOR\"] = \"vim\" (for instance, in the REPL) to make\nthis change on a case by case basis, or add the same to the user configuration\nfile ~/.julia/config/startup.jl in the user's home directory to have a\npermanent effect. The current value of the same environment variable can be\ndetermined by evaluating ENV[\"JULIA_EDITOR\"].\n\nThe environment variables that Julia uses generally start with JULIA. If\nInteractiveUtils.versioninfo is called with the keyword verbose=true, then the\noutput will list any defined environment variables relevant for Julia,\nincluding those which include JULIA in their names.\n\nnote: Note\nIt is recommended to avoid changing environment variables during runtime,\nsuch as within a ~/.julia/config/startup.jl.One reason is that some julia language variables, such as JULIA_NUM_THREADS\nand JULIA_PROJECT, need to be set before Julia starts.Similarly, __init__() functions of user modules in the sysimage (via PackageCompiler) are\nrun before startup.jl, so setting environment variables in a startup.jl may be too late for\nuser code.Further, changing environment variables during runtime can introduce data races into\notherwise benign code.In Bash, environment variables can either be set manually by running, e.g.,\nexport JULIA_NUM_THREADS=4 before starting Julia, or by adding the same command to\n~/.bashrc or ~/.bash_profile to set the variable each time Bash is started."},{"title":"File locations","page":"Environment Variables","location":"manual/environment-variables.html#File-locations","category":"section","text":""},{"title":"JULIA_BINDIR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_BINDIR","category":"section","text":"The absolute path of the directory containing the Julia executable, which sets\nthe global variable Sys.BINDIR. If $JULIA_BINDIR is not set, then\nJulia determines the value Sys.BINDIR at run-time.\n\nThe executable itself is one of\n\n$JULIA_BINDIR/julia\n$JULIA_BINDIR/julia-debug\n\nby default.\n\nThe global variable Base.DATAROOTDIR determines a relative path from\nSys.BINDIR to the data directory associated with Julia. Then the path\n\n$JULIA_BINDIR/$DATAROOTDIR/julia/base\n\ndetermines the directory in which Julia initially searches for source files (via\nBase.find_source_file()).\n\nLikewise, the global variable Base.SYSCONFDIR determines a relative path to the\nconfiguration file directory. Then Julia searches for a startup.jl file at\n\n$JULIA_BINDIR/$SYSCONFDIR/julia/startup.jl\n$JULIA_BINDIR/../etc/julia/startup.jl\n\nby default (via Base.load_julia_startup()).\n\nFor example, a Linux installation with a Julia executable located at\n/bin/julia, a DATAROOTDIR of ../share, and a SYSCONFDIR of ../etc will\nhave JULIA_BINDIR set to /bin, a source-file search path of\n\n/share/julia/base\n\nand a global configuration search path of\n\n/etc/julia/startup.jl"},{"title":"JULIA_PROJECT","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PROJECT","category":"section","text":"A directory path that indicates which project should be the initial active project.\nSetting this environment variable has the same effect as specifying the --project\nstart-up option, but --project has higher precedence. If the variable is set to @.\n(note the trailing dot)\nthen Julia tries to find a project directory that contains Project.toml or\nJuliaProject.toml file from the current directory and its parents. See also\nthe chapter on Code Loading.\n\nnote: Note\nJULIA_PROJECT must be defined before starting julia; defining it in startup.jl\nis too late in the startup process."},{"title":"JULIA_LOAD_PATH","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_LOAD_PATH","category":"section","text":"The JULIA_LOAD_PATH environment variable is used to populate the global Julia\nLOAD_PATH variable, which determines which packages can be loaded via\nimport and using (see Code Loading).\n\nUnlike the shell PATH variable, empty entries in JULIA_LOAD_PATH are expanded to\nthe default value of LOAD_PATH, [\"@\", \"@v#.#\", \"@stdlib\"] when populating\nLOAD_PATH. This allows easy appending, prepending, etc. of the load path value in\nshell scripts regardless of whether JULIA_LOAD_PATH is already set or not. For\nexample, to prepend the directory /foo/bar to LOAD_PATH just do\n\nexport JULIA_LOAD_PATH=\"/foo/bar:$JULIA_LOAD_PATH\"\n\nIf the JULIA_LOAD_PATH environment variable is already set, its old value will be\nprepended with /foo/bar. On the other hand, if JULIA_LOAD_PATH is not set, then\nit will be set to /foo/bar: which will expand to a LOAD_PATH value of\n[\"/foo/bar\", \"@\", \"@v#.#\", \"@stdlib\"]. If JULIA_LOAD_PATH is set to the empty\nstring, it expands to an empty LOAD_PATH array. In other words, the empty string\nis interpreted as a zero-element array, not a one-element array of the empty string.\nThis behavior was chosen so that it would be possible to set an empty load path via\nthe environment variable. If you want the default load path, either unset the\nenvironment variable or if it must have a value, set it to the string :.\n\nnote: Note\nOn Windows, path elements are separated by the ; character, as is the case with\nmost path lists on Windows. Replace : with ; in the above paragraph."},{"title":"JULIA_DEPOT_PATH","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_DEPOT_PATH","category":"section","text":"The JULIA_DEPOT_PATH environment variable is used to populate the\nglobal Julia DEPOT_PATH variable, which controls where the package manager, as well\nas Julia's code loading mechanisms, look for package registries, installed packages, named\nenvironments, repo clones, cached compiled package images, configuration files, and the default\nlocation of the REPL's history file.\n\nUnlike the shell PATH variable but similar to JULIA_LOAD_PATH,\nempty entries in JULIA_DEPOT_PATH have special behavior:\n\nAt the end, it is expanded to the default value of DEPOT_PATH, excluding the user depot.\nAt the start, it is expanded to the default value of DEPOT_PATH, including the user depot.\n\nThis allows easy overriding of the user depot, while still retaining access to resources that\nare bundled with Julia, like cache files, artifacts, etc. For example, to switch the user depot\nto /foo/bar use a trailing :\n\nexport JULIA_DEPOT_PATH=\"/foo/bar:\"\n\nAll package operations, like cloning registries or installing packages, will now write to\n/foo/bar, but since the empty entry is expanded to the default system depot, any bundled\nresources will still be available. If you really only want to use the depot at /foo/bar,\nand not load any bundled resources, simply set the environment variable to /foo/bar\nwithout the trailing colon.\n\nTo append a depot at the end of the full default list, including the default user depot, use a\nleading :\n\nexport JULIA_DEPOT_PATH=\":/foo/bar\"\n\nThere are two exceptions to the above rule. First, if [JULIA_DEPOT_PATH](@ref\nJULIADEPOTPATH) is set to the empty string, it expands to an empty DEPOT_PATH array. In\nother words, the empty string is interpreted as a zero-element array, not a one-element\narray of the empty string. This behavior was chosen so that it would be possible to set an\nempty depot path via the environment variable.\n\nSecond, if no user depot is specified in JULIA_DEPOT_PATH, then\nthe empty entry is expanded to the default depot including the user depot. This makes\nit possible to use the default depot, as if the environment variable was unset, by setting\nit to the string :.\n\nnote: Note\nOn Windows, path elements are separated by the ; character, as is the case with\nmost path lists on Windows. Replace : with ; in the above paragraph.\n\nnote: Note\nJULIA_DEPOT_PATH must be defined before starting julia; defining it in\nstartup.jl is too late in the startup process; at that point you can instead\ndirectly modify the DEPOT_PATH array, which is populated from the environment\nvariable."},{"title":"JULIA_HISTORY","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_HISTORY","category":"section","text":"The absolute path REPL.find_hist_file() of the REPL's history file. If\n$JULIA_HISTORY is not set, then REPL.find_hist_file() defaults to\n\n$(DEPOT_PATH[1])/logs/repl_history.jl"},{"title":"JULIA_MAX_NUM_PRECOMPILE_FILES","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_MAX_NUM_PRECOMPILE_FILES","category":"section","text":"Sets the maximum number of different instances of a single package that are to be stored in the precompile cache (default = 10)."},{"title":"JULIA_VERBOSE_LINKING","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_VERBOSE_LINKING","category":"section","text":"If set to true, linker commands will be displayed during precompilation."},{"title":"Pkg.jl","page":"Environment Variables","location":"manual/environment-variables.html#Pkg.jl","category":"section","text":""},{"title":"JULIA_CI","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_CI","category":"section","text":"If set to true, this indicates to the package server that any package operations are part of a continuous integration (CI) system for the purposes of gathering package usage statistics."},{"title":"JULIA_NUM_PRECOMPILE_TASKS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_NUM_PRECOMPILE_TASKS","category":"section","text":"The number of parallel tasks to use when precompiling packages. See Pkg.precompile."},{"title":"JULIA_PKG_DEVDIR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_DEVDIR","category":"section","text":"The default directory used by Pkg.develop for downloading packages."},{"title":"JULIA_PKG_IGNORE_HASHES","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_IGNORE_HASHES","category":"section","text":"If set to 1, this will ignore incorrect hashes in artifacts. This should be used carefully, as it disables verification of downloads, but can resolve issues when moving files across different types of file systems. See Pkg.jl issue #2317 for more details.\n\ncompat: Julia 1.6\nThis is only supported in Julia 1.6 and above."},{"title":"JULIA_PKG_OFFLINE","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_OFFLINE","category":"section","text":"If set to true, this will enable offline mode: see Pkg.offline.\n\ncompat: Julia 1.5\nPkg's offline mode requires Julia 1.5 or later."},{"title":"JULIA_PKG_PRECOMPILE_AUTO","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_PRECOMPILE_AUTO","category":"section","text":"If set to 0, this will disable automatic precompilation by package actions which change the manifest. See Pkg.precompile."},{"title":"JULIA_PKG_SERVER","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_SERVER","category":"section","text":"Specifies the URL of the package registry to use. By default, Pkg uses\nhttps://pkg.julialang.org to fetch Julia packages. In addition, you can disable the use of the PkgServer\nprotocol, and instead access the packages directly from their hosts (GitHub, GitLab, etc.)\nby setting: export JULIA_PKG_SERVER=\"\""},{"title":"JULIA_PKG_SERVER_REGISTRY_PREFERENCE","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_SERVER_REGISTRY_PREFERENCE","category":"section","text":"Specifies the preferred registry flavor. Currently supported values are conservative\n(the default), which will only publish resources that have been processed by the storage\nserver (and thereby have a higher probability of being available from the PkgServers),\nwhereas eager will publish registries whose resources have not necessarily been\nprocessed by the storage servers. Users behind restrictive firewalls that do not allow\ndownloading from arbitrary servers should not use the eager flavor.\n\ncompat: Julia 1.7\nThis only affects Julia 1.7 and above."},{"title":"JULIA_PKG_UNPACK_REGISTRY","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_UNPACK_REGISTRY","category":"section","text":"If set to true, this will unpack the registry instead of storing it as a compressed tarball.\n\ncompat: Julia 1.7\nThis only affects Julia 1.7 and above. Earlier versions will always unpack the registry."},{"title":"JULIA_PKG_USE_CLI_GIT","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_USE_CLI_GIT","category":"section","text":"If set to true, Pkg operations which use the git protocol will use an external git executable instead of the default libgit2 library.\n\ncompat: Julia 1.7\nUse of the git executable is only supported on Julia 1.7 and above."},{"title":"JULIA_PKGRESOLVE_ACCURACY","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKGRESOLVE_ACCURACY","category":"section","text":"The accuracy of the package resolver. This should be a positive integer, the default is 1."},{"title":"JULIA_PKG_PRESERVE_TIERED_INSTALLED","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_PRESERVE_TIERED_INSTALLED","category":"section","text":"Change the default package installation strategy to Pkg.PRESERVE_TIERED_INSTALLED\nto let the package manager try to install versions of packages while keeping as many\nversions of packages already installed as possible.\n\ncompat: Julia 1.9\nThis only affects Julia 1.9 and above."},{"title":"JULIA_PKG_GC_AUTO","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PKG_GC_AUTO","category":"section","text":"If set to false, automatic garbage collection of packages and artifacts will be disabled;\nsee Pkg.gc for more details.\n\ncompat: Julia 1.12\nThis environment variable is only supported on Julia 1.12 and above."},{"title":"Network transport","page":"Environment Variables","location":"manual/environment-variables.html#Network-transport","category":"section","text":""},{"title":"JULIA_NO_VERIFY_HOSTS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_NO_VERIFY_HOSTS","category":"section","text":""},{"title":"JULIA_SSL_NO_VERIFY_HOSTS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_SSL_NO_VERIFY_HOSTS","category":"section","text":""},{"title":"JULIA_SSH_NO_VERIFY_HOSTS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_SSH_NO_VERIFY_HOSTS","category":"section","text":""},{"title":"JULIA_ALWAYS_VERIFY_HOSTS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_ALWAYS_VERIFY_HOSTS","category":"section","text":"Specify hosts whose identity should or should not be verified for specific transport layers. See NetworkOptions.verify_host"},{"title":"JULIA_SSL_CA_ROOTS_PATH","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_SSL_CA_ROOTS_PATH","category":"section","text":"Specify the file or directory containing the certificate authority roots. See NetworkOptions.ca_roots"},{"title":"External applications","page":"Environment Variables","location":"manual/environment-variables.html#External-applications","category":"section","text":""},{"title":"JULIA_SHELL","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_SHELL","category":"section","text":"The absolute path of the shell with which Julia should execute external commands\n(via Base.repl_cmd()). Defaults to the environment variable $SHELL, and\nfalls back to /bin/sh if $SHELL is unset.\n\nnote: Note\nOn Windows, this environment variable is ignored, and external commands are\nexecuted directly."},{"title":"JULIA_EDITOR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_EDITOR","category":"section","text":"The editor returned by InteractiveUtils.editor() and used in, e.g., InteractiveUtils.edit,\nreferring to the command of the preferred editor, for instance vim.\n\n$JULIA_EDITOR takes precedence over $VISUAL, which in turn takes precedence\nover $EDITOR. If none of these environment variables is set, then the editor\nis taken to be open on Windows and OS X, or /etc/alternatives/editor if it\nexists, or emacs otherwise.\n\nTo use Visual Studio Code on Windows, set $JULIA_EDITOR to code.cmd."},{"title":"Parallelization","page":"Environment Variables","location":"manual/environment-variables.html#Parallelization","category":"section","text":""},{"title":"JULIA_CPU_THREADS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_CPU_THREADS","category":"section","text":"Overrides the global variable Base.Sys.CPU_THREADS, the number of\nlogical CPU cores available."},{"title":"JULIA_WORKER_TIMEOUT","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_WORKER_TIMEOUT","category":"section","text":"A Float64 that sets the value of Distributed.worker_timeout() (default: 60.0).\nThis function gives the number of seconds a worker process will wait for\na master process to establish a connection before dying."},{"title":"JULIA_NUM_THREADS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_NUM_THREADS","category":"section","text":"An unsigned 64-bit integer (uint64_t) or string that sets the maximum number\nof threads available to Julia. If $JULIA_NUM_THREADS is not set or is a\nnon-positive integer, or if the number of CPU threads cannot be determined\nthrough system calls, then the number of threads is set to 1.\n\nIf $JULIA_NUM_THREADS is set to auto, then the number of threads will be set\nto the number of CPU threads. It can also be set to a comma-separated string to\nspecify the size of the :default and :interactive [threadpools](@ref\nman-threadpools), respectively:\n\n# 5 threads in the :default pool and 2 in the :interactive pool\nexport JULIA_NUM_THREADS=5,2\n\n# `auto` threads in the :default pool and 1 in the :interactive pool\nexport JULIA_NUM_THREADS=auto,1\n\nnote: Note\nJULIA_NUM_THREADS must be defined before starting Julia; defining it in\nstartup.jl is too late in the startup process.\n\ncompat: Julia 1.5\nIn Julia 1.5 and above the number of threads can also be specified on startup\nusing the -t/--threads command line argument.\n\ncompat: Julia 1.7\nThe auto value for $JULIA_NUM_THREADS requires Julia 1.7 or above.\n\ncompat: Julia 1.9\nThe x,y format for threadpools requires Julia 1.9 or above."},{"title":"JULIA_THREAD_SLEEP_THRESHOLD","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_THREAD_SLEEP_THRESHOLD","category":"section","text":"If set to a string that starts with the case-insensitive substring \"infinite\",\nthen spinning threads never sleep. Otherwise, $JULIA_THREAD_SLEEP_THRESHOLD is\ninterpreted as an unsigned 64-bit integer (uint64_t) and gives, in\nnanoseconds, the amount of time after which spinning threads should sleep."},{"title":"JULIA_NUM_GC_THREADS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_NUM_GC_THREADS","category":"section","text":"Sets the number of threads used by Garbage Collection. If unspecified is set to the number of worker threads.\n\ncompat: Julia 1.10\nThe environment variable was added in 1.10"},{"title":"JULIA_IMAGE_THREADS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_IMAGE_THREADS","category":"section","text":"An unsigned 32-bit integer that sets the number of threads used by image\ncompilation in this Julia process. The value of this variable may be\nignored if the module is a small module. If left unspecified, the smaller\nof the value of JULIA_CPU_THREADS or half the\nnumber of logical CPU cores is used in its place."},{"title":"JULIA_IMAGE_TIMINGS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_IMAGE_TIMINGS","category":"section","text":"A boolean value that determines if detailed timing information is printed during\nduring image compilation. Defaults to 0."},{"title":"JULIA_EXCLUSIVE","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_EXCLUSIVE","category":"section","text":"If set to anything besides 0, then Julia's thread policy is consistent with\nrunning on a dedicated machine: each thread in the default threadpool is\naffinitized.  Interactive threads remain under the\ncontrol of the operating system scheduler.\n\nOtherwise, Julia lets the operating system handle thread policy."},{"title":"Garbage Collection","page":"Environment Variables","location":"manual/environment-variables.html#Garbage-Collection","category":"section","text":""},{"title":"JULIA_HEAP_SIZE_HINT","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_HEAP_SIZE_HINT","category":"section","text":"Environment variable equivalent to the --heap-size-hint=<size>[<unit>] command line option.\n\nForces garbage collection if memory usage is higher than the given value. The value may be specified as a number of bytes, optionally in units of:\n\n- B  (bytes)\n- K  (kibibytes)\n- M  (mebibytes)\n- G  (gibibytes)\n- T  (tebibytes)\n- %  (percentage of physical memory)\n\nFor example, JULIA_HEAP_SIZE_HINT=1G would provide a 1 GB heap size hint to the garbage collector."},{"title":"REPL formatting","page":"Environment Variables","location":"manual/environment-variables.html#REPL-formatting","category":"section","text":"Environment variables that determine how REPL output should be formatted at the\nterminal. The JULIA_*_COLOR variables should be set to [ANSI terminal escape\nsequences](https://en.wikipedia.org/wiki/ANSIescapecode). Julia provides\na high-level interface with much of the same functionality; see the section on\nThe Julia REPL."},{"title":"JULIA_ERROR_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_ERROR_COLOR","category":"section","text":"The formatting Base.error_color() (default: light red, \"\\033[91m\") that\nerrors should have at the terminal."},{"title":"JULIA_WARN_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_WARN_COLOR","category":"section","text":"The formatting Base.warn_color() (default: yellow, \"\\033[93m\") that warnings\nshould have at the terminal."},{"title":"JULIA_INFO_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_INFO_COLOR","category":"section","text":"The formatting Base.info_color() (default: cyan, \"\\033[36m\") that info\nshould have at the terminal."},{"title":"JULIA_INPUT_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_INPUT_COLOR","category":"section","text":"The formatting Base.input_color() (default: normal, \"\\033[0m\") that input\nshould have at the terminal."},{"title":"JULIA_ANSWER_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_ANSWER_COLOR","category":"section","text":"The formatting Base.answer_color() (default: normal, \"\\033[0m\") that output\nshould have at the terminal."},{"title":"NO_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#NO_COLOR","category":"section","text":"When this variable is present and not an empty string (regardless of its value) then colored\ntext will be disabled on the REPL. Can be overridden with the flag --color=yes or with the\nenvironment variable FORCE_COLOR. This environment variable is\ncommonly recognized by command-line applications."},{"title":"FORCE_COLOR","page":"Environment Variables","location":"manual/environment-variables.html#FORCE_COLOR","category":"section","text":"When this variable is present and not an empty string (regardless of its value) then\ncolored text will be enabled on the REPL. Can be overridden with the flag --color=no. This\nenvironment variable is commonly recognized by command-line applications."},{"title":"System and Package Image Building","page":"Environment Variables","location":"manual/environment-variables.html#System-and-Package-Image-Building","category":"section","text":""},{"title":"JULIA_CPU_TARGET","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_CPU_TARGET","category":"section","text":"Modify the target machine architecture for (pre)compiling\nsystem and package images.\nJULIA_CPU_TARGET only affects machine code image generation being output to a disk cache.\nUnlike the --cpu-target, or -C, command line option, it does not influence\njust-in-time (JIT) code generation within a Julia session where machine code is only\nstored in memory.\n\nValid values for JULIA_CPU_TARGET can be obtained by executing julia -C help.\n\nTo get the CPU target string that was used to build the current system image,\nuse Sys.sysimage_target(). This can be useful for reproducing\nthe same system image or understanding what CPU features were enabled during compilation.\n\nSetting JULIA_CPU_TARGET is important for heterogeneous compute systems where processors of\ndistinct types or features may be present. This is commonly encountered in high performance\ncomputing (HPC) clusters since the component nodes may be using distinct processors. In this case,\nyou may want to use the sysimage CPU target to maintain the same configuration as the sysimage.\nSee below for more details.\n\nThe CPU target string is a list of strings separated by ; each string starts with a CPU\nor architecture name and followed by an optional list of features separated by ,.\nA generic or empty CPU name means the basic required feature set of the target ISA\nwhich is at least the architecture the C/C++ runtime is compiled with. Each string\nis interpreted by LLVM.\n\nnote: Note\nPackage images can only target the same or more specific CPU features than\ntheir base system image.\n\nA few special features are supported:\n\nsysimage\nA special keyword that can be used as a CPU target name, which will be replaced\n  with the CPU target string that was used to build the current system image. This allows\n  you to specify CPU targets that build upon or extend the current sysimage's target, which\n  is particularly helpful for creating package images that are as flexible as the sysimage.\nclone_all\nThis forces the target to have all functions in sysimg cloned.\n  When used in negative form (i.e. -clone_all), this disables full clone that's\n  enabled by default for certain targets.\nbase([0-9]*)\nThis specifies the (0-based) base target index. The base target is the target\n  that the current target is based on, i.e. the functions that are not being cloned\n  will use the version in the base target. This option causes the base target to be\n  fully cloned (as if clone_all is specified for it) if it is not the default target (0).\n  The index can only be smaller than the current index.\nopt_size\nOptimize for size with minimum performance impact. Clang/GCC's -Os.\nmin_size\nOptimize only for size. Clang's -Oz."},{"title":"Debugging and profiling","page":"Environment Variables","location":"manual/environment-variables.html#Debugging-and-profiling","category":"section","text":""},{"title":"JULIA_DEBUG","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_DEBUG","category":"section","text":"Enable debug logging for a file or module, see Logging for more information."},{"title":"CI Debug Environment Variables","page":"Environment Variables","location":"manual/environment-variables.html#CI-Debug-Environment-Variables","category":"section","text":"Julia automatically enables verbose debugging options when certain continuous integration (CI) debug environment variables are set. This improves the debugging experience when CI jobs are re-run with debug logging enabled, by automatically:\n\nEnabling --trace-eval (location mode) to show expressions being evaluated\nSetting JULIA_TEST_VERBOSE=true to enable verbose test output\n\nThis allows developers to get detailed debugging information from CI runs without modifying their scripts or workflow files."},{"title":"JULIA_PROFILE_PEEK_HEAP_SNAPSHOT","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_PROFILE_PEEK_HEAP_SNAPSHOT","category":"section","text":"Enable collecting of a heap snapshot during execution via the profiling peek mechanism.\nSee Triggered During Execution."},{"title":"JULIA_TIMING_SUBSYSTEMS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_TIMING_SUBSYSTEMS","category":"section","text":"Allows you to enable or disable zones for a specific Julia run.\nFor instance, setting the variable to +GC,-INFERENCE will enable the GC zones and disable\nthe INFERENCE zones. See Dynamically Enabling and Disabling Zones."},{"title":"JULIA_GC_WAIT_FOR_DEBUGGER","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_GC_WAIT_FOR_DEBUGGER","category":"section","text":"If set to anything besides 0, then the Julia garbage collector will wait for\na debugger to attach instead of aborting whenever there's a critical error.\n\nnote: Note\nThis environment variable only has an effect if Julia was compiled with\ngarbage-collection debugging (that is, if WITH_GC_DEBUG_ENV is set to 1\nin the build configuration)."},{"title":"ENABLE_JITPROFILING","page":"Environment Variables","location":"manual/environment-variables.html#ENABLE_JITPROFILING","category":"section","text":"If set to anything besides 0, then the compiler will create and register an\nevent listener for just-in-time (JIT) profiling.\n\nnote: Note\nThis environment variable only has an effect if Julia was compiled with JIT\nprofiling support, using eitherIntel's VTune™ Amplifier\n(USE_INTEL_JITEVENTS set to 1 in the build configuration), or\nOProfile (USE_OPROFILE_JITEVENTS set to 1\nin the build configuration).\nPerf (USE_PERF_JITEVENTS set to 1\nin the build configuration). This integration is enabled by default."},{"title":"ENABLE_GDBLISTENER","page":"Environment Variables","location":"manual/environment-variables.html#ENABLE_GDBLISTENER","category":"section","text":"If set to anything besides 0 enables GDB registration of Julia code on release builds.\nOn debug builds of Julia this is always enabled. Recommended to use with -g 2."},{"title":"JULIA_LLVM_ARGS","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_LLVM_ARGS","category":"section","text":"Arguments to be passed to the LLVM backend."},{"title":"JULIA_FALLBACK_REPL","page":"Environment Variables","location":"manual/environment-variables.html#JULIA_FALLBACK_REPL","category":"section","text":"Forces the fallback repl instead of REPL.jl."},{"title":"Writing tests","page":"Writing tests","location":"devdocs/contributing/tests.html#Writing-tests","category":"section","text":"There are never enough tests. Track code coverage at Codecov, and help improve it.\n\nGo visit https://codecov.io/github/JuliaLang/julia.\nBrowse through the source files and find some untested functionality (highlighted in red) that you think you might be able to write a test for.\nWrite a test that exercises this functionality—you can add your test to one of the existing files, or start a new one, whichever seems most appropriate to you. If you're adding a new test file, make sure you include it in the list of tests in test/choosetests.jl. https://docs.julialang.org/en/v1/stdlib/Test/ may be helpful in explaining how the testing infrastructure works.\nRun make test-all to rebuild Julia and run your new test(s). If you had to fix a bug or add functionality in base, this will ensure that your test passes and that you have not introduced extraneous whitespace.\nSubmit the test as a pull request (PR).\n\nCode for the buildbot configuration is maintained at: https://github.com/staticfloat/julia-buildbot\nYou can see the current buildbot setup at: https://build.julialang.org/builders\nIssue 9493 and issue 11885 have more detailed discussion on code coverage.\n\nCode coverage shows functionality that still needs \"proof of concept\" tests. These are important, as are tests for tricky edge cases, such as converting between integer types when the number to convert is near the maximum of the range of one of the integer types. Even if a function already has some coverage on Codecov, it may still benefit from tests for edge cases."},{"title":"Base.Cartesian","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian","category":"section","text":"The (non-exported) Cartesian module provides macros that facilitate writing multidimensional algorithms.\nMost often you can write such algorithms with straightforward techniques; however, there are a few cases where Base.Cartesian is still useful or necessary."},{"title":"Principles of usage","page":"Base.Cartesian","location":"devdocs/cartesian.html#Principles-of-usage","category":"section","text":"A simple example of usage is:\n\n@nloops 3 i A begin\n    s += @nref 3 A i\nend\n\nwhich generates the following code:\n\nfor i_3 = axes(A, 3)\n    for i_2 = axes(A, 2)\n        for i_1 = axes(A, 1)\n            s += A[i_1, i_2, i_3]\n        end\n    end\nend\n\nIn general, Cartesian allows you to write generic code that contains repetitive elements, like\nthe nested loops in this example. Other applications include repeated expressions (e.g., loop\nunwinding) or creating function calls with variable numbers of arguments without using the \"splat\"\nconstruct (i...)."},{"title":"Basic syntax","page":"Base.Cartesian","location":"devdocs/cartesian.html#Basic-syntax","category":"section","text":"The (basic) syntax of @nloops is as follows:\n\nThe first argument must be an integer (not a variable) specifying the number of loops.\nThe second argument is the symbol-prefix used for the iterator variable. Here we used i, and\nvariables i_1, i_2, i_3 were generated.\nThe third argument specifies the range for each iterator variable. If you use a variable (symbol)\nhere, it's taken as axes(A, dim). More flexibly, you can use the anonymous-function expression\nsyntax described below.\nThe last argument is the body of the loop. Here, that's what appears between the begin...end.\n\nThere are some additional features of @nloops described in the reference section.\n\n@nref follows a similar pattern, generating A[i_1,i_2,i_3] from @nref 3 A i. The general\npractice is to read from left to right, which is why @nloops is @nloops 3 i A expr (as in\nfor i_2 = axes(A, 2), where i_2 is to the left and the range is to the right) whereas @nref\nis @nref 3 A i (as in A[i_1,i_2,i_3], where the array comes first).\n\nIf you're developing code with Cartesian, you may find that debugging is easier when you examine\nthe generated code, using @macroexpand:\n\njulia> @macroexpand @nref 2 A i\n:(A[i_1, i_2])"},{"title":"Supplying the number of expressions","page":"Base.Cartesian","location":"devdocs/cartesian.html#Supplying-the-number-of-expressions","category":"section","text":"The first argument to both of these macros is the number of expressions, which must be an integer.\nWhen you're writing a function that you intend to work in multiple dimensions, this may not be\nsomething you want to hard-code. The recommended approach is to use a @generated function. Here's\nan example:\n\n@generated function mysum(A::Array{T,N}) where {T,N}\n    quote\n        s = zero(T)\n        @nloops $N i A begin\n            s += @nref $N A i\n        end\n        s\n    end\nend\n\nNaturally, you can also prepare expressions or perform calculations before the quote block."},{"title":"Anonymous-function expressions as macro arguments","page":"Base.Cartesian","location":"devdocs/cartesian.html#Anonymous-function-expressions-as-macro-arguments","category":"section","text":"Perhaps the single most powerful feature in Cartesian is the ability to supply anonymous-function\nexpressions that get evaluated at parsing time. Let's consider a simple example:\n\n@nexprs 2 j->(i_j = 1)\n\n@nexprs generates n expressions that follow a pattern. This code would generate the following\nstatements:\n\ni_1 = 1\ni_2 = 1\n\nIn each generated statement, an \"isolated\" j (the variable of the anonymous function) gets replaced\nby values in the range 1:2. Generally speaking, Cartesian employs a LaTeX-like syntax. This\nallows you to do math on the index j. Here's an example computing the strides of an array:\n\ns_1 = 1\n@nexprs 3 j->(s_{j+1} = s_j * size(A, j))\n\nwould generate expressions\n\ns_1 = 1\ns_2 = s_1 * size(A, 1)\ns_3 = s_2 * size(A, 2)\ns_4 = s_3 * size(A, 3)\n\nAnonymous-function expressions have many uses in practice."},{"title":"Macro reference","page":"Base.Cartesian","location":"devdocs/cartesian.html#dev-cartesian-reference","category":"section","text":""},{"title":"Base.Cartesian.@nloops","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nloops","category":"macro","text":"@nloops N itersym rangeexpr bodyexpr\n@nloops N itersym rangeexpr preexpr bodyexpr\n@nloops N itersym rangeexpr preexpr postexpr bodyexpr\n\nGenerate N nested loops, using itersym as the prefix for the iteration variables.\nrangeexpr may be an anonymous-function expression, or a simple symbol var in which case\nthe range is axes(var, d) for dimension d.\n\nOptionally, you can provide \"pre\" and \"post\" expressions. These get executed first and last,\nrespectively, in the body of each loop. For example:\n\n@nloops 2 i A d -> j_d = min(i_d, 5) begin\n    s += @nref 2 A j\nend\n\nwould generate:\n\nfor i_2 = axes(A, 2)\n    j_2 = min(i_2, 5)\n    for i_1 = axes(A, 1)\n        j_1 = min(i_1, 5)\n        s += A[j_1, j_2]\n    end\nend\n\nIf you want just a post-expression, supply nothing for the pre-expression. Using\nparentheses and semicolons, you can supply multi-statement expressions.\n\n\n\n\n\n"},{"title":"Base.Cartesian.@nref","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nref","category":"macro","text":"@nref N A indexexpr\n\nGenerate expressions like A[i_1, i_2, ...]. indexexpr can either be an iteration-symbol\nprefix, or an anonymous-function expression.\n\nExamples\n\njulia> @macroexpand Base.Cartesian.@nref 3 A i\n:(A[i_1, i_2, i_3])\n\n\n\n\n\n"},{"title":"Base.Cartesian.@nextract","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nextract","category":"macro","text":"@nextract N esym isym\n\nGenerate N variables esym_1, esym_2, ..., esym_N to extract values from isym.\nisym can be either a Symbol or anonymous-function expression.\n\n@nextract 2 x y would generate\n\nx_1 = y[1]\nx_2 = y[2]\n\nwhile @nextract 3 x d->y[2d-1] yields\n\nx_1 = y[1]\nx_2 = y[3]\nx_3 = y[5]\n\n\n\n\n\n"},{"title":"Base.Cartesian.@nexprs","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nexprs","category":"macro","text":"@nexprs N expr\n\nGenerate N expressions. expr should be an anonymous-function expression.\n\nExamples\n\njulia> @macroexpand Base.Cartesian.@nexprs 4 i -> y[i] = A[i+j]\nquote\n    y[1] = A[1 + j]\n    y[2] = A[2 + j]\n    y[3] = A[3 + j]\n    y[4] = A[4 + j]\nend\n\n\n\n\n\n"},{"title":"Base.Cartesian.@ncall","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@ncall","category":"macro","text":"@ncall N f sym...\n\nGenerate a function call expression. sym represents any number of function arguments, the\nlast of which may be an anonymous-function expression and is expanded into N arguments.\n\nFor example, @ncall 3 func a generates\n\nfunc(a_1, a_2, a_3)\n\nwhile @ncall 2 func a b i->c[i] yields\n\nfunc(a, b, c[1], c[2])\n\n\n\n\n\n"},{"title":"Base.Cartesian.@ncallkw","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@ncallkw","category":"macro","text":"@ncallkw N f kw sym...\n\nGenerate a function call expression with keyword arguments kw.... As\nin the case of @ncall, sym represents any number of function arguments, the\nlast of which may be an anonymous-function expression and is expanded into N arguments.\n\nExamples\n\njulia> using Base.Cartesian\n\njulia> f(x...; a, b = 1, c = 2, d = 3) = +(x..., a, b, c, d);\n\njulia> x_1, x_2 = (-1, -2); b = 0; kw = (c = 0, d = 0);\n\njulia> @ncallkw 2 f (; a = 0, b, kw...) x\n-3\n\n\n\n\n\n\n"},{"title":"Base.Cartesian.@ntuple","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@ntuple","category":"macro","text":"@ntuple N expr\n\nGenerates an N-tuple. @ntuple 2 i would generate (i_1, i_2), and @ntuple 2 k->k+1\nwould generate (2,3).\n\n\n\n\n\n"},{"title":"Base.Cartesian.@nall","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nall","category":"macro","text":"@nall N expr\n\nCheck whether all of the expressions generated by the anonymous-function expression expr\nevaluate to true.\n\n@nall 3 d->(i_d > 1) would generate the expression (i_1 > 1 && i_2 > 1 && i_3 > 1). This\ncan be convenient for bounds-checking.\n\n\n\n\n\n"},{"title":"Base.Cartesian.@nany","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nany","category":"macro","text":"@nany N expr\n\nCheck whether any of the expressions generated by the anonymous-function expression expr\nevaluate to true.\n\n@nany 3 d->(i_d > 1) would generate the expression (i_1 > 1 || i_2 > 1 || i_3 > 1).\n\n\n\n\n\n"},{"title":"Base.Cartesian.@nif","page":"Base.Cartesian","location":"devdocs/cartesian.html#Base.Cartesian.@nif","category":"macro","text":"@nif N conditionexpr expr\n@nif N conditionexpr expr elseexpr\n\nGenerates a sequence of if ... elseif ... else ... end statements. For example:\n\n@nif 3 d->(i_d >= size(A,d)) d->(error(\"Dimension \", d, \" too big\")) d->println(\"All OK\")\n\nwould generate:\n\nif i_1 > size(A, 1)\n    error(\"Dimension \", 1, \" too big\")\nelseif i_2 > size(A, 2)\n    error(\"Dimension \", 2, \" too big\")\nelse\n    println(\"All OK\")\nend\n\n\n\n\n\n"},{"title":"Pkg","page":"Pkg","location":"stdlib/Pkg.html#Pkg","category":"section","text":"Pkg is Julia's built-in package manager, and handles operations\nsuch as installing, updating and removing packages.\n\nnote: Note\nWhat follows is a very brief introduction to Pkg. For more\ninformation on Project.toml files, Manifest.toml files, package\nversion compatibility ([compat]), environments, registries, etc.,\nit is highly recommended to read the full manual, which is available here:\nhttps://pkgdocs.julialang.org.\n\nimport Markdown\nfile = joinpath(Sys.STDLIB, \"Pkg\", \"docs\", \"src\", \"getting-started.md\")\nstr = read(file, String)\nstr = replace(str, r\"^#.*$\"m => \"\")\nstr = replace(str, \"[API Reference](@ref)\" => \"[API Reference](https://pkgdocs.julialang.org/v1/api/)\")\nstr = replace(str, \"(@ref Working-with-Environments)\" => \"(https://pkgdocs.julialang.org/v1/environments/)\")\nstr = replace(str, \"(@ref Managing-Packages)\" => \"(https://pkgdocs.julialang.org/v1/managing-packages/)\")\nMarkdown.parse(str)"},{"title":"Using AI agents to work on Julia","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html#Using-AI-agents-to-work-on-Julia","category":"section","text":"![WARNING]\nYou are responsible for the code you submit in PRs. Do not submit PRs\ncontaining AI-generated code that you do not understand or that does not\nmeet the ordinary quality bar for PRs to julia.\n\nThis page documents best practices for setting up AI agents to work with Julia.\nIf you find additional prompt instructions that work well for common tasks,\nconsider submitting a PR to add these to AGENTS.md."},{"title":"Google Jules","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html#Google-Jules","category":"section","text":"Use the following for your Initial Setup configuration.\n\ncurl -fsSL https://install.julialang.org | sh -s -- -y --default-channel nightly\n. /home/swebot/.profile\n\nJules has access to the internet, so you can give it links to issues or additional\ndocumentation in your prompting."},{"title":"OpenAI Codex","page":"Using AI agents to work on Julia","location":"devdocs/contributing/aiagents.html#OpenAI-Codex","category":"section","text":"Configure the following:\n\nSetup Script\n\napt update\napt install less\ncurl -fsSL https://install.julialang.org | sh -s -- -y --default-channel nightly\nsource /root/.bashrc\nmake -C /workspace/julia/doc alldeps JULIA_EXECUTABLE=\"/root/.juliaup/bin/julia\"\nmake -C /workspace/julia/test install-revise-deps JULIA_EXECUTABLE=\"/root/.juliaup/bin/julia\"\n\nEnvironment Variables\n\nJULIA_PKG_OFFLINE=true\n\nCodex does not have internet access after initial setup, so you cannot give it\nadditional information as links - you will need to copy any relevant text into\nthe prompt.\n\nNote that Codex rebuilds the environment after every invocation. This can\nadd significant latency. Codex work best for well-defined tasks that can\nbe solved in a single shot."},{"title":"SIMD Support","page":"SIMD Support","location":"base/simd-types.html#SIMD-Support","category":"section","text":"Type VecElement{T} is intended for building libraries of SIMD operations. Practical use of it\nrequires using llvmcall. The type is defined as:\n\nstruct VecElement{T}\n    value::T\nend\n\nIt has a special compilation rule: a homogeneous tuple of VecElement{T} maps to an LLVM vector\ntype when T is a primitive bits type.\n\nAt -O3, the compiler might automatically vectorize operations on such tuples. For example,\nthe following program, when compiled with julia -O3 generates two SIMD addition instructions\n(addps) on x86 systems:\n\nconst m128 = NTuple{4,VecElement{Float32}}\n\nfunction add(a::m128, b::m128)\n    (VecElement(a[1].value+b[1].value),\n     VecElement(a[2].value+b[2].value),\n     VecElement(a[3].value+b[3].value),\n     VecElement(a[4].value+b[4].value))\nend\n\ntriple(c::m128) = add(add(c,c),c)\n\ncode_native(triple,(m128,))\n\nHowever, since the automatic vectorization cannot be relied upon, future use will mostly be via\nlibraries that use llvmcall."},{"title":"Julia Syntax Highlighting","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html#Julia-Syntax-Highlighting","category":"section","text":"The JuliaSyntaxHighlighting library serves as a small convenience package to\nsyntax highlight Julia code using JuliaSyntax and StyledStrings.\n\nIt is intended for use across the standard library, and the wider ecosystem."},{"title":"Functions","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html#stdlib-jsh-api","category":"section","text":""},{"title":"Faces","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html#stdlib-jsh-faces","category":"section","text":"The highlight/highlight! methods work by applying custom faces to Julia\ncode. As part of the standard library, these faces use privileged face names, of\nthe form julia_*. These can be re-used in other packages, and customised with\nfaces.toml configuration.\n\nwarning: Unstable faces\nThe particular faces used by JuliaSyntaxHighlighting are liable to change\nwithout warning in point releases. As the syntax highlighting rules are refined\nover time, changes should become less and less frequent though.\n\nThe current set of faces, and their default values are as follows:\n\njulia_macro: magenta\njulia_symbol: magenta\njulia_singleton_identifier: inherits from julia_symbol\njulia_type: yellow\njulia_typedec: bright blue\njulia_comment: grey\njulia_string: green\njulia_regex: inherits from julia_string\njulia_backslash_literal: magenta, inherits from julia_string\njulia_string_delim: bright green\njulia_cmdstring: inherits from julia_string\njulia_char: inherits from julia_string\njulia_char_delim: inherits from julia_string_delim\njulia_number: bright magenta\njulia_bool: inherits from julia_number\njulia_funcall: cyan\njulia_funcdef: cyan\njulia_broadcast: bright blue, bold\njulia_builtin: bright blue\njulia_operator: blue\njulia_comparator: inherits from julia_operator\njulia_assignment: bright red\njulia_keyword: red\njulia_parentheses: unstyled\njulia_unpaired_parentheses: inherit from julia_error and julia_parentheses\njulia_error: red background\njulia_rainbow_paren_1: bright green, inherits from julia_parentheses\njulia_rainbow_paren_2: bright blue, inherits from julia_parentheses\njulia_rainbow_paren_3: bright red, inherits from julia_parentheses\njulia_rainbow_paren_4: inherits from julia_rainbow_paren_1\njulia_rainbow_paren_5: inherits from julia_rainbow_paren_2\njulia_rainbow_paren_6: inherits from julia_rainbow_paren_3\njulia_rainbow_bracket_1: blue, inherits from julia_parentheses\njulia_rainbow_bracket_2: brightmagenta, inherits from `juliaparentheses`\njulia_rainbow_bracket_3: inherits from julia_rainbow_bracket_1\njulia_rainbow_bracket_4: inherits from julia_rainbow_bracket_2\njulia_rainbow_bracket_5: inherits from julia_rainbow_bracket_1\njulia_rainbow_bracket_6: inherits from julia_rainbow_bracket_2\njulia_rainbow_curly_1: bright yellow, inherits from julia_parentheses\njulia_rainbow_curly_2: yellow, inherits from julia_parentheses\njulia_rainbow_curly_3: inherits from julia_rainbow_curly_1\njulia_rainbow_curly_4: inherits from julia_rainbow_curly_2\njulia_rainbow_curly_5: inherits from julia_rainbow_curly_1\njulia_rainbow_curly_6: inherits from julia_rainbow_curly_2"},{"title":"JuliaSyntaxHighlighting.highlight","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html#JuliaSyntaxHighlighting.highlight","category":"function","text":"highlight(content::Union{AbstractString, IO},\n          ast::JuliaSyntax.GreenNode = <parsed content>;\n          syntax_errors::Bool = false) -> AnnotatedString{String}\n\nApply syntax highlighting to content using JuliaSyntax.\n\nBy default, JuliaSyntax.parseall is used to generate to ast with the\nignore_errors keyword argument set to true. Alternatively, one may provide a\npre-generated ast.\n\nWhen syntax_errors is set, the julia_error face is applied to detected syntax errors.\n\nwarning: Warning\nNote that the particular faces used by JuliaSyntax, and the way they\nare applied, is subject to change.\n\nExamples\n\njulia> JuliaSyntaxHighlighting.highlight(\"sum(1:8)\")\n\"sum(1:8)\"\n\njulia> JuliaSyntaxHighlighting.highlight(\"sum(1:8)\") |> Base.annotations\n6-element Vector{@NamedTuple{region::UnitRange{Int64}, label::Symbol, value}}:\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((1:3, :face, :julia_funcall))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((4:4, :face, :julia_rainbow_paren_1))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((5:5, :face, :julia_number))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((6:6, :face, :julia_operator))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((7:7, :face, :julia_number))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((8:8, :face, :julia_rainbow_paren_1))\n\n\n\n\n\n"},{"title":"JuliaSyntaxHighlighting.highlight!","page":"Julia Syntax Highlighting","location":"stdlib/JuliaSyntaxHighlighting.html#JuliaSyntaxHighlighting.highlight!","category":"function","text":"highlight!(content::Union{AnnotatedString, SubString{AnnotatedString}},\n           ast::JuliaSyntax.GreenNode = <parsed content>;\n           syntax_errors::Bool = false) -> content\n\nModify content by applying syntax highlighting using JuliaSyntax.\n\nBy default, JuliaSyntax.parseall is used to generate to ast with the\nignore_errors keyword argument set to true. Alternatively, one may provide a\npre-generated ast.\n\nWhen syntax_errors is set, the julia_error face is applied to detected syntax errors.\n\nwarning: Warning\nNote that the particular faces used by JuliaSyntax, and the way they\nare applied, is subject to change.\n\nExamples\n\njulia> str = Base.AnnotatedString(\"sum(1:8)\")\n\"sum(1:8)\"\n\njulia> JuliaSyntaxHighlighting.highlight!(str)\n\"sum(1:8)\"\n\njulia> Base.annotations(str)\n6-element Vector{@NamedTuple{region::UnitRange{Int64}, label::Symbol, value}}:\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((1:3, :face, :julia_funcall))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((4:4, :face, :julia_rainbow_paren_1))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((5:5, :face, :julia_number))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((6:6, :face, :julia_operator))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((7:7, :face, :julia_number))\n @NamedTuple{region::UnitRange{Int64}, label::Symbol, value}((8:8, :face, :julia_rainbow_paren_1))\n\n\n\n\n\n"},{"title":"Tasks","page":"Tasks","location":"base/parallel.html#Tasks","category":"section","text":""},{"title":"Scheduling","page":"Tasks","location":"base/parallel.html#Scheduling","category":"section","text":""},{"title":"Synchronization","page":"Tasks","location":"base/parallel.html#lib-task-sync","category":"section","text":""},{"title":"Channels","page":"Tasks","location":"base/parallel.html#Channels","category":"section","text":""},{"title":"Low-level synchronization using schedule and wait","page":"Tasks","location":"base/parallel.html#low-level-schedule-wait","category":"section","text":"The easiest correct use of schedule is on a Task that is not started (scheduled)\nyet. However, it is possible to use schedule and wait as a very\nlow-level building block for constructing synchronization interfaces. A crucial\npre-condition of calling schedule(task) is that the caller must \"own\" the task; i.e., it\nmust know that the call to wait in the given task is happening at the locations known to\nthe code calling schedule(task). One strategy for ensuring such pre-condition is to use\natomics, as demonstrated in the following example:\n\n@enum OWEState begin\n    OWE_EMPTY\n    OWE_WAITING\n    OWE_NOTIFYING\nend\n\nmutable struct OneWayEvent\n    @atomic state::OWEState\n    task::Task\n    OneWayEvent() = new(OWE_EMPTY)\nend\n\nfunction Base.notify(ev::OneWayEvent)\n    state = @atomic ev.state\n    while state !== OWE_NOTIFYING\n        # Spin until we successfully update the state to OWE_NOTIFYING:\n        state, ok = @atomicreplace(ev.state, state => OWE_NOTIFYING)\n        if ok\n            if state == OWE_WAITING\n                # OWE_WAITING -> OWE_NOTIFYING transition means that the waiter task is\n                # already waiting or about to call `wait`. The notifier task must wake up\n                # the waiter task.\n                schedule(ev.task)\n            else\n                @assert state == OWE_EMPTY\n                # Since we are assuming that there is only one notifier task (for\n                # simplicity), we know that the other possible case here is OWE_EMPTY.\n                # We do not need to do anything because we know that the waiter task has\n                # not called `wait(ev::OneWayEvent)` yet.\n            end\n            break\n        end\n    end\n    return\nend\n\nfunction Base.wait(ev::OneWayEvent)\n    ev.task = current_task()\n    state, ok = @atomicreplace(ev.state, OWE_EMPTY => OWE_WAITING)\n    if ok\n        # OWE_EMPTY -> OWE_WAITING transition means that the notifier task is guaranteed to\n        # invoke OWE_WAITING -> OWE_NOTIFYING transition. The waiter task must call\n        # `wait()` immediately. In particular, it MUST NOT invoke any function that may\n        # yield to the scheduler at this point in code.\n        wait()\n    else\n        @assert state == OWE_NOTIFYING\n        # Otherwise, the `state` must have already been moved to OWE_NOTIFYING by the\n        # notifier task.\n    end\n    return\nend\n\nev = OneWayEvent()\n@sync begin\n    Threads.@spawn begin\n        wait(ev)\n        println(\"done\")\n    end\n    println(\"notifying...\")\n    notify(ev)\nend\n\n# output\nnotifying...\ndone\n\nOneWayEvent lets one task to wait for another task's notify. It is a limited\ncommunication interface since wait can only be used once from a single task (note the\nnon-atomic assignment of ev.task)\n\nIn this example, notify(ev::OneWayEvent) is allowed to call schedule(ev.task) if and\nonly if it modifies the state from OWE_WAITING to OWE_NOTIFYING. This lets us know that\nthe task executing wait(ev::OneWayEvent) is now in the ok branch and that there cannot be\nother tasks that try to schedule(ev.task) since their\n@atomicreplace(ev.state, state => OWE_NOTIFYING) will fail."},{"title":"Core.Task","page":"Tasks","location":"base/parallel.html#Core.Task","category":"type","text":"Task(func[, reserved_stack::Int])\n\nCreate a Task (i.e. coroutine) to execute the given function func (which\nmust be callable with no arguments). The task exits when this function returns.\nThe task will run in the \"world age\" from the parent at construction when scheduled.\n\nThe optional reserved_stack argument specifies the size of the stack available\nfor this task, in bytes. The default, 0, uses the system-dependent stack size default.\n\nwarning: Warning\nBy default tasks will have the sticky bit set to true t.sticky. This models the\nhistoric default for @async. Sticky tasks can only be run on the worker thread\nthey are first scheduled on, and when scheduled will make the task that they were scheduled\nfrom sticky. To obtain the behavior of Threads.@spawn set the sticky\nbit manually to false.\n\nExamples\n\njulia> a() = sum(i for i in 1:1000);\n\njulia> b = Task(a);\n\nIn this example, b is a runnable Task that hasn't started yet.\n\n\n\n\n\n"},{"title":"Base.@task","page":"Tasks","location":"base/parallel.html#Base.@task","category":"macro","text":"@task\n\nWrap an expression in a Task without executing it, and return the Task. This only\ncreates a task, and does not run it.\n\nwarning: Warning\nBy default tasks will have the sticky bit set to true t.sticky. This models the\nhistoric default for @async. Sticky tasks can only be run on the worker thread\nthey are first scheduled on, and when scheduled will make the task that they were scheduled\nfrom sticky. To obtain the behavior of Threads.@spawn set the sticky\nbit manually to false.\n\nExamples\n\njulia> a1() = sum(i for i in 1:1000);\n\njulia> b = @task a1();\n\njulia> istaskstarted(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n"},{"title":"Base.@async","page":"Tasks","location":"base/parallel.html#Base.@async","category":"macro","text":"@async\n\nWrap an expression in a Task and add it to the local machine's scheduler queue.\n\nValues can be interpolated into @async via $, which copies the value directly into the\nconstructed underlying closure. This allows you to insert the value of a variable,\nisolating the asynchronous code from changes to the variable's value in the current task.\n\nwarning: Warning\nIt is strongly encouraged to favor Threads.@spawn over @async always even when no\nparallelism is required especially in publicly distributed libraries.  This is\nbecause a use of @async disables the migration of the parent task across worker\nthreads in the current implementation of Julia.  Thus, seemingly innocent use of\n@async in a library function can have a large impact on the performance of very\ndifferent parts of user applications.\n\ncompat: Julia 1.4\nInterpolating values via $ is available as of Julia 1.4.\n\n\n\n\n\n"},{"title":"Base.asyncmap","page":"Tasks","location":"base/parallel.html#Base.asyncmap","category":"function","text":"asyncmap(f, c...; ntasks=0, batch_size=nothing)\n\nUses multiple concurrent tasks to map f over a collection (or multiple\nequal length collections). For multiple collection arguments, f is\napplied elementwise.\n\nThe output is guaranteed to be the same order as the elements of the collection(s) c.\n\nntasks specifies the number of tasks to run concurrently.\nDepending on the length of the collections, if ntasks is unspecified,\nup to 100 tasks will be used for concurrent mapping.\n\nntasks can also be specified as a zero-arg function. In this case, the\nnumber of tasks to run in parallel is checked before processing every element and a new\ntask started if the value of ntasks_func is greater than the current number\nof tasks.\n\nIf batch_size is specified, the collection is processed in batch mode. f must\nthen be a function that must accept a Vector of argument tuples and must\nreturn a vector of results. The input vector will have a length of batch_size or less.\n\nThe following examples highlight execution in different tasks by returning\nthe objectid of the tasks in which the mapping function is executed.\n\nFirst, with ntasks undefined, each element is processed in a different task.\n\njulia> tskoid() = objectid(current_task());\n\njulia> asyncmap(x->tskoid(), 1:5)\n5-element Vector{UInt64}:\n 0x6e15e66c75c75853\n 0x440f8819a1baa682\n 0x9fb3eeadd0c83985\n 0xebd3e35fe90d4050\n 0x29efc93edce2b961\n\njulia> length(unique(asyncmap(x->tskoid(), 1:5)))\n5\n\nWith ntasks=2 all elements are processed in 2 tasks.\n\njulia> asyncmap(x->tskoid(), 1:5; ntasks=2)\n5-element Vector{UInt64}:\n 0x027ab1680df7ae94\n 0xa23d2f80cd7cf157\n 0x027ab1680df7ae94\n 0xa23d2f80cd7cf157\n 0x027ab1680df7ae94\n\njulia> length(unique(asyncmap(x->tskoid(), 1:5; ntasks=2)))\n2\n\nWith batch_size defined, the mapping function needs to be changed to accept an array\nof argument tuples and return an array of results. map is used in the modified mapping\nfunction to achieve this.\n\njulia> batch_func(input) = map(x->string(\"args_tuple: \", x, \", element_val: \", x[1], \", task: \", tskoid()), input)\nbatch_func (generic function with 1 method)\n\njulia> asyncmap(batch_func, 1:5; ntasks=2, batch_size=2)\n5-element Vector{String}:\n \"args_tuple: (1,), element_val: 1, task: 9118321258196414413\"\n \"args_tuple: (2,), element_val: 2, task: 4904288162898683522\"\n \"args_tuple: (3,), element_val: 3, task: 9118321258196414413\"\n \"args_tuple: (4,), element_val: 4, task: 4904288162898683522\"\n \"args_tuple: (5,), element_val: 5, task: 9118321258196414413\"\n\n\n\n\n\n"},{"title":"Base.asyncmap!","page":"Tasks","location":"base/parallel.html#Base.asyncmap!","category":"function","text":"asyncmap!(f, results, c...; ntasks=0, batch_size=nothing)\n\nLike asyncmap, but stores output in results rather than\nreturning a collection.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\n\n\n\n\n"},{"title":"Base.current_task","page":"Tasks","location":"base/parallel.html#Base.current_task","category":"function","text":"current_task()\n\nGet the currently running Task.\n\n\n\n\n\n"},{"title":"Base.istaskdone","page":"Tasks","location":"base/parallel.html#Base.istaskdone","category":"function","text":"istaskdone(t::Task)::Bool\n\nDetermine whether a task has exited.\n\nExamples\n\njulia> a2() = sum(i for i in 1:1000);\n\njulia> b = Task(a2);\n\njulia> istaskdone(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n"},{"title":"Base.istaskstarted","page":"Tasks","location":"base/parallel.html#Base.istaskstarted","category":"function","text":"istaskstarted(t::Task)::Bool\n\nDetermine whether a task has started executing.\n\nExamples\n\njulia> a3() = sum(i for i in 1:1000);\n\njulia> b = Task(a3);\n\njulia> istaskstarted(b)\nfalse\n\n\n\n\n\n"},{"title":"Base.istaskfailed","page":"Tasks","location":"base/parallel.html#Base.istaskfailed","category":"function","text":"istaskfailed(t::Task)::Bool\n\nDetermine whether a task has exited because an exception was thrown.\n\nExamples\n\njulia> a4() = error(\"task failed\");\n\njulia> b = Task(a4);\n\njulia> istaskfailed(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskfailed(b)\ntrue\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n"},{"title":"Base.task_local_storage","page":"Tasks","location":"base/parallel.html#Base.task_local_storage-Tuple{Any}","category":"method","text":"task_local_storage(key)\n\nLook up the value of a key in the current task's task-local storage.\n\n\n\n\n\n"},{"title":"Base.task_local_storage","page":"Tasks","location":"base/parallel.html#Base.task_local_storage-Tuple{Any, Any}","category":"method","text":"task_local_storage(key, value)\n\nAssign a value to a key in the current task's task-local storage.\n\n\n\n\n\n"},{"title":"Base.task_local_storage","page":"Tasks","location":"base/parallel.html#Base.task_local_storage-Tuple{Function, Any, Any}","category":"method","text":"task_local_storage(body, key, value)\n\nCall the function body with a modified task-local storage, in which value is assigned to\nkey; the previous value of key, or lack thereof, is restored afterwards. Useful\nfor emulating dynamic scoping.\n\n\n\n\n\n"},{"title":"Core.ConcurrencyViolationError","page":"Tasks","location":"base/parallel.html#Core.ConcurrencyViolationError","category":"type","text":"ConcurrencyViolationError(msg) <: Exception\n\nAn error thrown when a detectable violation of concurrent semantics has occurred.\n\nA non-exhaustive list of examples of when this is used include:\n\nThrowing when a deadlock has been detected (e.g. wait(current_task()))\nKnown-unsafe behavior is attempted (e.g. yield(current_task))\nA known non-threadsafe datastructure is attempted to be modified from multiple concurrent tasks\nA lock is being unlocked that wasn't locked by this task\n\n\n\n\n\n"},{"title":"Base.yield","page":"Tasks","location":"base/parallel.html#Base.yield","category":"function","text":"yield(t::Task, arg = nothing)\n\nA fast, unfair-scheduling version of schedule(t, arg); yield() which\nimmediately yields to t before calling the scheduler.\n\nThrows a ConcurrencyViolationError if t is the currently running task.\n\n\n\n\n\nyield()\n\nSwitch to the scheduler to allow another scheduled task to run. A task that calls this\nfunction is still runnable, and will be restarted immediately if there are no other runnable\ntasks.\n\n\n\n\n\n"},{"title":"Base.yieldto","page":"Tasks","location":"base/parallel.html#Base.yieldto","category":"function","text":"yieldto(t::Task, arg = nothing)\n\nSwitch to the given task. The first time a task is switched to, the task's function is\ncalled with no arguments. On subsequent switches, arg is returned from the task's last\ncall to yieldto. This is a low-level call that only switches tasks, not considering states\nor scheduling in any way. Its use is discouraged.\n\n\n\n\n\n"},{"title":"Base.sleep","page":"Tasks","location":"base/parallel.html#Base.sleep","category":"function","text":"sleep(seconds)\n\nBlock the current task for a specified number of seconds. The minimum sleep time is 1\nmillisecond or input of 0.001.\n\n\n\n\n\n"},{"title":"Base.schedule","page":"Tasks","location":"base/parallel.html#Base.schedule","category":"function","text":"schedule(t::Task, [val]; error=false)\n\nAdd a Task to the scheduler's queue. This causes the task to run constantly when the system\nis otherwise idle, unless the task performs a blocking operation such as wait.\n\nIf a second argument val is provided, it will be passed to the task (via the return value of\nyieldto) when it runs again. If error is true, the value is raised as an exception in\nthe woken task.\n\nwarning: Warning\nIt is incorrect to use schedule on an arbitrary Task that has already been started.\nSee the API reference for more information.\n\nwarning: Warning\nBy default tasks will have the sticky bit set to true t.sticky. This models the\nhistoric default for @async. Sticky tasks can only be run on the worker thread\nthey are first scheduled on, and when scheduled will make the task that they were scheduled\nfrom sticky. To obtain the behavior of Threads.@spawn set the sticky\nbit manually to false.\n\nExamples\n\njulia> a5() = sum(i for i in 1:1000);\n\njulia> b = Task(a5);\n\njulia> istaskstarted(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskstarted(b)\ntrue\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n"},{"title":"Base.errormonitor","page":"Tasks","location":"base/parallel.html#Base.errormonitor","category":"function","text":"errormonitor(t::Task)\n\nPrint an error log to stderr if task t fails.\n\nExamples\n\njulia> wait(errormonitor(Threads.@spawn error(\"task failed\")); throw = false)\nUnhandled Task ERROR: task failed\nStacktrace:\n[...]\n\n\n\n\n\n"},{"title":"Base.@sync","page":"Tasks","location":"base/parallel.html#Base.@sync","category":"macro","text":"@sync\n\nWait until all lexically-enclosed uses of @async, @spawn,\nDistributed.@spawnat and Distributed.@distributed\nare complete. All exceptions thrown by enclosed async operations are collected and thrown as\na CompositeException.\n\nExamples\n\njulia> Threads.nthreads()\n4\n\njulia> @sync begin\n           Threads.@spawn println(\"Thread-id $(Threads.threadid()), task 1\")\n           Threads.@spawn println(\"Thread-id $(Threads.threadid()), task 2\")\n       end;\nThread-id 3, task 1\nThread-id 1, task 2\n\n\n\n\n\n"},{"title":"Base.wait","page":"Tasks","location":"base/parallel.html#Base.wait","category":"function","text":"wait([x])\n\nBlock the current task until some event occurs.\n\nChannel: Wait for a value to be appended to the channel.\nCondition: Wait for notify on a condition and return the val\nparameter passed to notify. See the Condition-specific docstring of wait for\nthe exact behavior.\nProcess: Wait for a process or process chain to exit. The exitcode field of a process\ncan be used to determine success or failure.\nTask: Wait for a Task to finish. See the Task-specific docstring of wait for\nthe exact behavior.\nRawFD: Wait for changes on a file descriptor (see the FileWatching package).\n\nIf no argument is passed, the task blocks for an undefined period. A task can only be\nrestarted by an explicit call to schedule or yieldto.\n\nOften wait is called within a while loop to ensure a waited-for condition is met before\nproceeding.\n\n\n\n\n\n"},{"title":"Base.waitany","page":"Tasks","location":"base/parallel.html#Base.waitany","category":"function","text":"waitany(tasks; throw=true) -> (done_tasks, remaining_tasks)\n\nWait until at least one of the given tasks have been completed.\n\nIf throw is true, throw CompositeException when one of the\ncompleted tasks completes with an exception.\n\nThe return value consists of two task vectors. The first one consists of\ncompleted tasks, and the other consists of uncompleted tasks.\n\nwarning: Warning\nThis may scale poorly compared to writing code that uses multiple individual tasks that\neach runs serially, since this needs to scan the list of tasks each time and\nsynchronize with each one every time this is called. Or consider using\nwaitall(tasks; failfast=true) instead.\n\ncompat: Julia 1.12\nThis function requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.waitall","page":"Tasks","location":"base/parallel.html#Base.waitall","category":"function","text":"waitall(tasks; failfast=true, throw=true) -> (done_tasks, remaining_tasks)\n\nWait until all the given tasks have been completed.\n\nIf failfast is true, the function will return when at least one of the\ngiven tasks is finished by exception. If throw is true, throw\nCompositeException when one of the completed tasks has failed.\n\nfailfast and throw keyword arguments work independently; when only\nthrow=true is specified, this function waits for all the tasks to complete.\n\nThe return value consists of two task vectors. The first one consists of\ncompleted tasks, and the other consists of uncompleted tasks.\n\ncompat: Julia 1.12\nThis function requires at least Julia 1.12.\n\n\n\n\n\n"},{"title":"Base.fetch","page":"Tasks","location":"base/parallel.html#Base.fetch-Tuple{Task}","category":"method","text":"fetch(t::Task)\n\nWait for a Task to finish, then return its result value.\nIf the task fails with an exception, a TaskFailedException (which wraps the failed task)\nis thrown.\n\n\n\n\n\n"},{"title":"Base.fetch","page":"Tasks","location":"base/parallel.html#Base.fetch-Tuple{Any}","category":"method","text":"fetch(x::Any)\n\nReturn x.\n\n\n\n\n\n"},{"title":"Base.timedwait","page":"Tasks","location":"base/parallel.html#Base.timedwait","category":"function","text":"timedwait(testcb, timeout::Real; pollint::Real=0.1)\n\nWait until testcb() returns true or timeout seconds have passed, whichever is earlier.\nThe test function is polled every pollint seconds. The minimum value for pollint is 0.001 seconds,\nthat is, 1 millisecond.\n\nReturn :ok or :timed_out.\n\nExamples\n\njulia> cb() = (sleep(5); return);\n\njulia> t = @async cb();\n\njulia> timedwait(()->istaskdone(t), 1)\n:timed_out\n\njulia> timedwait(()->istaskdone(t), 6.5)\n:ok\n\n\n\n\n\n"},{"title":"Base.Condition","page":"Tasks","location":"base/parallel.html#Base.Condition","category":"type","text":"Condition()\n\nCreate an edge-triggered event source that tasks can wait for. Tasks that call wait on a\nCondition are suspended and queued. Tasks are woken up when notify is later called on\nthe Condition. Waiting on a condition can return a value or raise an error if the optional arguments\nof notify are used. Edge triggering means that only tasks waiting at the time notify\nis called can be woken up. For level-triggered notifications, you must keep extra state to keep\ntrack of whether a notification has happened. The Channel and Threads.Event types do\nthis, and can be used for level-triggered events.\n\nThis object is NOT thread-safe. See Threads.Condition for a thread-safe version.\n\n\n\n\n\n"},{"title":"Base.Threads.Condition","page":"Tasks","location":"base/parallel.html#Base.Threads.Condition","category":"type","text":"Threads.Condition([lock])\n\nA thread-safe version of Base.Condition.\n\nTo call wait or notify on a Threads.Condition, you must first call\nlock on it. When wait is called, the lock is atomically released during\nblocking, and will be reacquired before wait returns. Therefore idiomatic use\nof a Threads.Condition c looks like the following:\n\nlock(c)\ntry\n    while !thing_we_are_waiting_for\n        wait(c)\n    end\nfinally\n    unlock(c)\nend\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n"},{"title":"Base.Event","page":"Tasks","location":"base/parallel.html#Base.Event","category":"type","text":"Event([autoreset=false])\n\nCreate a level-triggered event source. Tasks that call wait on an\nEvent are suspended and queued until notify is called on the Event.\nAfter notify is called, the Event remains in a signaled state and\ntasks will no longer block when waiting for it, until reset is called.\n\nIf autoreset is true, at most one task will be released from wait for\neach call to notify.\n\nThis provides an acquire & release memory ordering on notify/wait.\n\ncompat: Julia 1.1\nThis functionality requires at least Julia 1.1.\n\ncompat: Julia 1.8\nThe autoreset functionality and memory ordering guarantee requires at least Julia 1.8.\n\n\n\n\n\n"},{"title":"Base.notify","page":"Tasks","location":"base/parallel.html#Base.notify","category":"function","text":"notify(condition, val=nothing; all=true, error=false)\n\nWake up tasks waiting for a condition, passing them val. If all is true (the default),\nall waiting tasks are woken, otherwise only one is. If error is true, the passed value\nis raised as an exception in the woken tasks.\n\nReturn the count of tasks woken up. Return 0 if no tasks are waiting on condition.\n\n\n\n\n\n"},{"title":"Base.reset","page":"Tasks","location":"base/parallel.html#Base.reset-Tuple{Base.Event}","category":"method","text":"reset(::Event)\n\nReset an Event back into an un-set state. Then any future calls to wait will\nblock until notify is called again.\n\n\n\n\n\n"},{"title":"Base.Semaphore","page":"Tasks","location":"base/parallel.html#Base.Semaphore","category":"type","text":"Semaphore(sem_size)\n\nCreate a counting semaphore that allows at most sem_size\nacquires to be in use at any time.\nEach acquire must be matched with a release.\n\nThis provides a acquire & release memory ordering on acquire/release calls.\n\n\n\n\n\n"},{"title":"Base.acquire","page":"Tasks","location":"base/parallel.html#Base.acquire","category":"function","text":"acquire(f, s::Semaphore)\n\nExecute f after acquiring from Semaphore s,\nand release on completion or error.\n\nFor example, a do-block form that ensures only 2\ncalls of foo will be active at the same time:\n\ns = Base.Semaphore(2)\n@sync for _ in 1:100\n    Threads.@spawn begin\n        Base.acquire(s) do\n            foo()\n        end\n    end\nend\n\ncompat: Julia 1.8\nThis method requires at least Julia 1.8.\n\n\n\n\n\nacquire(s::Semaphore)\n\nWait for one of the sem_size permits to be available,\nblocking until one can be acquired.\n\n\n\n\n\n"},{"title":"Base.release","page":"Tasks","location":"base/parallel.html#Base.release","category":"function","text":"release(s::Semaphore)\n\nReturn one permit to the pool,\npossibly allowing another task to acquire it\nand resume execution.\n\n\n\n\n\n"},{"title":"Base.AbstractLock","page":"Tasks","location":"base/parallel.html#Base.AbstractLock","category":"type","text":"AbstractLock\n\nAbstract supertype describing types that\nimplement the synchronization primitives:\nlock, trylock, unlock, and islocked.\n\n\n\n\n\n"},{"title":"Base.lock","page":"Tasks","location":"base/parallel.html#Base.lock","category":"function","text":"lock(f::Function, l::Lockable)\n\nAcquire the lock associated with l, execute f with the lock held,\nand release the lock when f returns. f will receive one positional\nargument: the value wrapped by l. If the lock is already locked by a\ndifferent task/thread, wait for it to become available.\nWhen this function returns, the lock has been released, so the caller should\nnot attempt to unlock it.\n\ncompat: Julia 1.11\nRequires at least Julia 1.11.\n\n\n\n\n\nlock(f::Function, lock)\n\nAcquire the lock, execute f with the lock held, and release the lock when f\nreturns. If the lock is already locked by a different task/thread, wait for it to become\navailable.\n\nWhen this function returns, the lock has been released, so the caller should\nnot attempt to unlock it.\n\nSee also: @lock.\n\ncompat: Julia 1.7\nUsing a Channel as the second argument requires Julia 1.7 or later.\n\n\n\n\n\nlock(lock)\n\nAcquire the lock when it becomes available.\nIf the lock is already locked by a different task/thread,\nwait for it to become available.\n\nEach lock must be matched by an unlock.\n\n\n\n\n\n"},{"title":"Base.unlock","page":"Tasks","location":"base/parallel.html#Base.unlock","category":"function","text":"unlock(lock)\n\nReleases ownership of the lock.\n\nIf this is a recursive lock which has been acquired before, decrement an\ninternal counter and return immediately.\n\n\n\n\n\n"},{"title":"Base.trylock","page":"Tasks","location":"base/parallel.html#Base.trylock","category":"function","text":"trylock(lock) -> Success (Boolean)\n\nAcquire the lock if it is available,\nand return true if successful.\nIf the lock is already locked by a different task/thread,\nreturn false.\n\nEach successful trylock must be matched by an unlock.\n\nFunction trylock combined with islocked can be used for writing the\ntest-and-test-and-set or exponential backoff algorithms if it is supported by the\ntypeof(lock) (read its documentation).\n\n\n\n\n\n"},{"title":"Base.islocked","page":"Tasks","location":"base/parallel.html#Base.islocked","category":"function","text":"islocked(lock) -> Status (Boolean)\n\nCheck whether the lock is held by any task/thread.\nThis function alone should not be used for synchronization. However, islocked combined\nwith trylock can be used for writing the test-and-test-and-set or exponential\nbackoff algorithms if it is supported by the typeof(lock) (read its documentation).\n\nExtended help\n\nFor example, an exponential backoff can be implemented as follows if the lock\nimplementation satisfied the properties documented below.\n\nnspins = 0\nwhile true\n    while islocked(lock)\n        GC.safepoint()\n        nspins += 1\n        nspins > LIMIT && error(\"timeout\")\n    end\n    trylock(lock) && break\n    backoff()\nend\n\nImplementation\n\nA lock implementation is advised to define islocked with the following properties and note\nit in its docstring.\n\nislocked(lock) is data-race-free.\nIf islocked(lock) returns false, an immediate invocation of trylock(lock) must\nsucceed (returns true) if there is no interference from other tasks.\n\n\n\n\n\n"},{"title":"Base.ReentrantLock","page":"Tasks","location":"base/parallel.html#Base.ReentrantLock","category":"type","text":"ReentrantLock()\n\nCreate a re-entrant lock for synchronizing Tasks. The same task can\nacquire the lock as many times as required (this is what the \"Reentrant\" part\nof the name means). Each lock must be matched with an unlock.\n\nCalling lock will also inhibit running of finalizers on that thread until the\ncorresponding unlock. Use of the standard lock pattern illustrated below\nshould naturally be supported, but beware of inverting the try/lock order or\nmissing the try block entirely (e.g. attempting to return with the lock still\nheld):\n\nThis provides a acquire/release memory ordering on lock/unlock calls.\n\nlock(l)\ntry\n    <atomic work>\nfinally\n    unlock(l)\nend\n\nIf !islocked(lck::ReentrantLock) holds, trylock(lck)\nsucceeds unless there are other tasks attempting to hold the lock \"at the same time.\"\n\n\n\n\n\n"},{"title":"Base.@lock","page":"Tasks","location":"base/parallel.html#Base.@lock","category":"macro","text":"@lock l expr\n\nMacro version of lock(f, l::AbstractLock) but with expr instead of f function.\nExpands to:\n\nlock(l)\ntry\n    expr\nfinally\n    unlock(l)\nend\n\nThis is similar to using lock with a do block, but avoids creating a closure\nand thus can improve the performance.\n\ncompat: Compat\n@lock was added in Julia 1.3, and exported in Julia 1.7.\n\n\n\n\n\n"},{"title":"Base.Lockable","page":"Tasks","location":"base/parallel.html#Base.Lockable","category":"type","text":"Lockable(value, lock = ReentrantLock())\n\nCreate a Lockable object that wraps value and\nassociates it with the provided lock. This object\nsupports @lock, lock, trylock,\nunlock. To access the value, index the lockable object while\nholding the lock.\n\ncompat: Julia 1.11\nRequires at least Julia 1.11.\n\nExample\n\njulia> locked_list = Base.Lockable(Int[]);\n\njulia> @lock(locked_list, push!(locked_list[], 1)) # must hold the lock to access the value\n1-element Vector{Int64}:\n 1\n\njulia> lock(summary, locked_list)\n\"1-element Vector{Int64}\"\n\n\n\n\n\n"},{"title":"Base.AbstractChannel","page":"Tasks","location":"base/parallel.html#Base.AbstractChannel","category":"type","text":"AbstractChannel{T}\n\nRepresentation of a channel passing objects of type T.\n\n\n\n\n\n"},{"title":"Base.Channel","page":"Tasks","location":"base/parallel.html#Base.Channel","category":"type","text":"Channel{T=Any}(size::Int=0)\n\nConstruct a Channel with an internal buffer that can hold a maximum of size objects\nof type T.\nput! calls on a full channel block until an object is removed with take!.\n\nChannel(0) constructs an unbuffered channel. put! blocks until a matching take! is called.\nAnd vice-versa.\n\nOther constructors:\n\nChannel(): default constructor, equivalent to Channel{Any}(0)\nChannel(Inf): equivalent to Channel{Any}(typemax(Int))\nChannel(sz): equivalent to Channel{Any}(sz)\n\ncompat: Julia 1.3\nThe default constructor Channel() and default size=0 were added in Julia 1.3.\n\n\n\n\n\n"},{"title":"Base.Channel","page":"Tasks","location":"base/parallel.html#Base.Channel-Tuple{Function}","category":"method","text":"Channel{T=Any}(func::Function, size=0; taskref=nothing, spawn=false, threadpool=nothing)\n\nCreate a new task from func, bind it to a new channel of type\nT and size size, and schedule the task, all in a single call.\nThe channel is automatically closed when the task terminates.\n\nfunc must accept the bound channel as its only argument.\n\nIf you need a reference to the created task, pass a Ref{Task} object via\nthe keyword argument taskref.\n\nIf spawn=true, the Task created for func may be scheduled on another thread\nin parallel, equivalent to creating a task via Threads.@spawn.\n\nIf spawn=true and the threadpool argument is not set, it defaults to :default.\n\nIf the threadpool argument is set (to :default or :interactive), this implies\nthat spawn=true and the new Task is spawned to the specified threadpool.\n\nReturn a Channel.\n\nExamples\n\njulia> chnl = Channel() do ch\n           foreach(i -> put!(ch, i), 1:4)\n       end;\n\njulia> typeof(chnl)\nChannel{Any}\n\njulia> for i in chnl\n           @show i\n       end;\ni = 1\ni = 2\ni = 3\ni = 4\n\nReferencing the created task:\n\njulia> taskref = Ref{Task}();\n\njulia> chnl = Channel(taskref=taskref) do ch\n           println(take!(ch))\n       end;\n\njulia> istaskdone(taskref[])\nfalse\n\njulia> put!(chnl, \"Hello\");\nHello\n\njulia> istaskdone(taskref[])\ntrue\n\ncompat: Julia 1.3\nThe spawn= parameter was added in Julia 1.3. This constructor was added in Julia 1.3.\nIn earlier versions of Julia, Channel used keyword arguments to set size and T, but\nthose constructors are deprecated.\n\ncompat: Julia 1.9\nThe threadpool= argument was added in Julia 1.9.\n\njulia> chnl = Channel{Char}(1, spawn=true) do ch\n           for c in \"hello world\"\n               put!(ch, c)\n           end\n       end;\n\njulia> String(collect(chnl))\n\"hello world\"\n\n\n\n\n\n"},{"title":"Base.put!","page":"Tasks","location":"base/parallel.html#Base.put!-Tuple{Channel, Any}","category":"method","text":"put!(c::Channel, v)\n\nAppend an item v to the channel c. Blocks if the channel is full.\n\nFor unbuffered channels, blocks until a take! is performed by a different\ntask.\n\ncompat: Julia 1.1\nv now gets converted to the channel's type with convert as put! is called.\n\n\n\n\n\n"},{"title":"Base.take!","page":"Tasks","location":"base/parallel.html#Base.take!-Tuple{Channel}","category":"method","text":"take!(c::Channel)\n\nRemoves and returns a value from a Channel in order. Blocks until data is available.\nFor unbuffered channels, blocks until a put! is performed by a different task.\n\nExamples\n\nBuffered channel\n\njulia> c = Channel(1);\n\njulia> put!(c, 1);\n\njulia> take!(c)\n1\n\nUnbuffered channel\n\njulia> c = Channel(0);\n\njulia> task = Task(() -> put!(c, 1));\n\njulia> schedule(task);\n\njulia> take!(c)\n1\n\n\n\n\n\n"},{"title":"Base.isfull","page":"Tasks","location":"base/parallel.html#Base.isfull-Tuple{Channel}","category":"method","text":"isfull(c::Channel)\n\nDetermine if a Channel is full, in the sense\nthat calling put!(c, some_value) would have blocked.\nReturns immediately, does not block.\n\nNote that it may frequently be the case that put! will\nnot block after this returns true. Users must take\nprecautions not to accidentally create live-lock bugs\nin their code by calling this method, as these are\ngenerally harder to debug than deadlocks. It is also\npossible that put! will block after this call\nreturns false, if there are multiple producer\ntasks calling put! in parallel.\n\nExamples\n\nBuffered channel\n\njulia> c = Channel(1); # capacity = 1\n\njulia> isfull(c)\nfalse\n\njulia> put!(c, 1);\n\njulia> isfull(c)\ntrue\n\nUnbuffered channel\n\njulia> c = Channel(); # capacity = 0\n\njulia> isfull(c) # unbuffered channel is always full\ntrue\n\n\n\n\n\n"},{"title":"Base.isready","page":"Tasks","location":"base/parallel.html#Base.isready-Tuple{Channel}","category":"method","text":"isready(c::Channel)\n\nDetermine whether a Channel has a value stored in it.\nReturns immediately, does not block.\n\nFor unbuffered channels, return true if there are tasks waiting on a put!.\n\nExamples\n\nBuffered channel\n\njulia> c = Channel(1);\n\njulia> isready(c)\nfalse\n\njulia> put!(c, 1);\n\njulia> isready(c)\ntrue\n\nUnbuffered channel\n\njulia> c = Channel();\n\njulia> isready(c)  # no tasks waiting to put!\nfalse\n\njulia> task = Task(() -> put!(c, 1));\n\njulia> schedule(task);  # schedule a put! task\n\njulia> isready(c)\ntrue\n\n\n\n\n\n"},{"title":"Base.isopen","page":"Tasks","location":"base/parallel.html#Base.isopen-Tuple{Channel}","category":"method","text":"isopen(c::Channel)\n\nDetermine whether a Channel is open for new put! operations.\nNotice that a Channel can be closed and still have buffered elements which can be\nconsumed with take!.\n\nExamples\n\nBuffered channel with task\n\njulia> c = Channel(ch -> put!(ch, 1), 1);\n\njulia> isopen(c) # The channel is closed to new `put!`s\nfalse\n\njulia> isready(c) # The channel is closed but still contains elements\ntrue\n\njulia> take!(c)\n1\n\njulia> isready(c)\nfalse\n\nUnbuffered channel\n\njulia> c = Channel{Int}();\n\njulia> isopen(c)\ntrue\n\njulia> close(c)\n\njulia> isopen(c)\nfalse\n\n\n\n\n\n"},{"title":"Base.fetch","page":"Tasks","location":"base/parallel.html#Base.fetch-Tuple{Channel}","category":"method","text":"fetch(c::Channel)\n\nWaits for and returns (without removing) the first available item from the Channel.\nNote: fetch is unsupported on an unbuffered (0-size) Channel.\n\nExamples\n\nBuffered channel\n\njulia> c = Channel(3) do ch\n           foreach(i -> put!(ch, i), 1:3)\n       end;\n\njulia> fetch(c)\n1\n\njulia> collect(c)  # item is not removed\n3-element Vector{Any}:\n 1\n 2\n 3\n\n\n\n\n\n"},{"title":"Base.close","page":"Tasks","location":"base/parallel.html#Base.close-Tuple{Channel}","category":"method","text":"close(c::Channel[, excp::Exception])\n\nClose a channel. An exception (optionally given by excp), is thrown by:\n\nput! on a closed channel.\ntake! and fetch on an empty, closed channel.\n\n\n\n\n\n"},{"title":"Base.bind","page":"Tasks","location":"base/parallel.html#Base.bind-Tuple{Channel, Task}","category":"method","text":"bind(chnl::Channel, task::Task)\n\nAssociate the lifetime of chnl with a task.\nChannel chnl is automatically closed when the task terminates.\nAny uncaught exception in the task is propagated to all waiters on chnl.\n\nThe chnl object can be explicitly closed independent of task termination.\nTerminating tasks have no effect on already closed Channel objects.\n\nWhen a channel is bound to multiple tasks, the first task to terminate will\nclose the channel. When multiple channels are bound to the same task,\ntermination of the task will close all of the bound channels.\n\nExamples\n\njulia> c = Channel(0);\n\njulia> task = @async foreach(i->put!(c, i), 1:4);\n\njulia> bind(c,task);\n\njulia> for i in c\n           @show i\n       end;\ni = 1\ni = 2\ni = 3\ni = 4\n\njulia> isopen(c)\nfalse\n\njulia> c = Channel(0);\n\njulia> task = @async (put!(c, 1); error(\"foo\"));\n\njulia> bind(c, task);\n\njulia> take!(c)\n1\n\njulia> put!(c, 1);\nERROR: TaskFailedException\nStacktrace:\n[...]\n    nested task error: foo\n[...]\n\n\n\n\n\n"},{"title":"Scope of Variables","page":"Scope of Variables","location":"manual/variables-and-scoping.html#scope-of-variables","category":"section","text":"The scope of a variable is the region of code within which a variable is accessible. Variable\nscoping helps avoid variable naming conflicts. The concept is intuitive: two functions can both have\narguments called x without the two x's referring to the same thing. Similarly, there are many\nother cases where different blocks of code can use the same name without referring to the same\nthing. The rules for when the same variable name does or doesn't refer to the same thing are called\nscope rules; this section spells them out in detail.\n\nCertain constructs in the language introduce scope blocks, which are regions of code that are\neligible to be the scope of some set of variables. The scope of a variable cannot be an arbitrary\nset of source lines; instead, it will always line up with one of these blocks. There are two main\ntypes of scopes in Julia, global scope and local scope. The latter can be nested. There is also\na distinction in Julia between constructs which introduce a \"hard scope\" and those which only\nintroduce a \"soft scope\", which affects whether\nshadowing\na global variable by the same name is allowed or not.\n\ninfo: Summary\nVariables defined in global scope may be undefined in inner local scopes,\ndepending on where the code is run, in order to balance safety and convenience.\nThe hard and soft local scoping rules define the interplay between global and local variables.However, variables defined only in local scope behave consistently in all contexts.\nIf the variable is already defined, it will be reused. If the variable is not defined,\nit will be made available to the current and inner scopes (but not outer scopes).\n\ntip: A Common Confusion\nIf you run into an unexpectedly undefined variable,# Print the numbers 1 through 5\ni = 0\nwhile i < 5\n    i += 1     # ERROR: UndefVarError: `i` not defined\n    println(i)\nenda simple fix is to change all global variable definitions into local definitions\nby wrapping the code in a let block or function.# Print the numbers 1 through 5\nlet i = 0\n    while i < 5\n        i += 1     # Now outer `i` is defined in the inner scope of the while loop\n        println(i)\n    end\nendThis is a common source of confusion when writing procedural scripts,\nbut it becomes a non-issue if code is moved inside functions\nor executed interactively in the REPL.See also the global and local keywords\nto explicitly achieve any desired scoping behavior."},{"title":"Scope Constructs","page":"Scope of Variables","location":"manual/variables-and-scoping.html#man-scope-table","category":"section","text":"The constructs introducing scope blocks are:\n\nConstruct Scope Type Introduced Scope Types Able to Contain Construct\nmodule, baremodule global global\nstruct local (hard) global\nmacro local (hard) global\nfor, while, try local (soft) global, local\nfunction, do, let, comprehensions, generators local (hard) global, local\n\nNotably missing from this table are\nbegin blocks and if blocks\nwhich do not introduce new scopes.\nThe three types of scopes follow somewhat different rules which will be explained below.\n\nJulia uses lexical scoping,\nmeaning that a function's scope does not inherit from its caller's scope, but from the scope in\nwhich the function was defined. For example, in the following code the x inside foo refers\nto the x in the global scope of its module Bar:\n\njulia> module Bar\n           x = 1\n           foo() = x\n       end;\n\nand not a x in the scope where foo is used:\n\njulia> import .Bar\n\njulia> x = -1;\n\njulia> Bar.foo()\n1\n\nThus lexical scope means that what a variable in a particular piece of code refers to can be\ndeduced from the code in which it appears alone and does not depend on how the program executes. A\nscope nested inside another scope can \"see\" variables in all the outer scopes in which it is\ncontained. Outer scopes, on the other hand, cannot see variables in inner scopes."},{"title":"Global Scope","page":"Scope of Variables","location":"manual/variables-and-scoping.html#Global-Scope","category":"section","text":"Each module introduces a new global scope, separate from the global scope of all other modules—there\nis no all-encompassing global scope. Modules can introduce variables of other modules into their\nscope through the using or import statements or through qualified access using the\ndot-notation, i.e. each module is a so-called namespace as well as a first-class data structure\nassociating names with values.\n\nIf a top-level expression contains a variable declaration with keyword local,\nthen that variable is not accessible outside that expression.\nThe variable inside the expression does not affect global variables of the same name.\nAn example is to declare local x in a begin or if block at the top-level:\n\njulia> x = 1\n       begin\n           local x = 0\n           @show x\n       end\n       @show x;\nx = 0\nx = 1\n\nNote that the interactive prompt (aka REPL) is in the global scope of the module Main."},{"title":"Local Scope","page":"Scope of Variables","location":"manual/variables-and-scoping.html#local-scope","category":"section","text":"A new local scope is introduced by most code blocks (see above [table](@ref\nman-scope-table) for a complete list). If such a block is syntactically nested\ninside of another local scope, the scope it creates is nested inside of all the\nlocal scopes that it appears within, which are all ultimately nested inside of\nthe global scope of the module in which the code is evaluated. Variables in\nouter scopes are visible from any scope they contain — meaning that they can be\nread and written in inner scopes — unless there is a variable with the same name\nthat \"shadows\" the outer variable of the same name. This is true even if the\nouter local is declared after (in the sense of textually below) an inner\nblock. When we say that a variable \"exists\" in a given scope, this means that a\nvariable by that name exists in any of the scopes that the current scope is\nnested inside of, including the current one. If a variable's value is used in a\nlocal scope, but nothing with its name exists in this scope, it is assumed to be\na global.\n\nSome programming languages require explicitly declaring new variables before\nusing them. Explicit declaration works in Julia too: in any local scope, writing\nlocal x declares a new local variable in that scope, regardless of whether\nthere is already a variable named x in an outer scope or not. Declaring each\nnew variable like this is somewhat verbose and tedious, however, so Julia, like\nmany other languages, considers assignment to a variable name that doesn't\nalready exist to implicitly declare that variable. If the current scope is\nglobal, the new variable is global; if the current scope is local, the new\nvariable is local to the innermost local scope and will be visible inside of\nthat scope but not outside of it. If you assign to an existing local, it\nalways updates that existing local: you can only shadow a local by explicitly\ndeclaring a new local in a nested scope with the local keyword. In particular,\nthis applies to variables assigned in inner functions, which may surprise users\ncoming from Python where assignment in an inner function creates a new local\nunless the variable is explicitly declared to be non-local.\n\nMostly this is pretty intuitive, but as with many things that behave\nintuitively, the details are more subtle than one might naïvely imagine.\n\nWhen x = <value> occurs in a local scope, Julia applies the following rules to decide what the\nexpression means based on where the assignment expression occurs and what x already refers to at\nthat location:\n\nExisting local: If x is already a local variable, then the existing local x is\nassigned;\nHard scope: If x is not already a local variable and assignment occurs inside of any\nhard scope construct (i.e. within a let block, function, struct or macro body, comprehension, or\ngenerator), a new local named x is created in the scope of the assignment;\nSoft scope: If x is not already a local variable and all of the scope constructs\ncontaining the assignment are soft scopes (loops, try/catch blocks), the\nbehavior depends on whether the global variable x is defined:\nif global x is undefined, a new local named x is created in the scope of the\nassignment;\nif global x is defined, the assignment is considered ambiguous:\nin non-interactive contexts (files, eval), an ambiguity warning is printed and a new\nlocal is created;\nin interactive contexts (REPL, notebooks), the global variable x is assigned.\n\nYou may note that in non-interactive contexts the hard and soft scope behaviors are identical except\nthat a warning is printed when an implicitly local variable (i.e. not declared with local x)\nshadows a global. In interactive contexts, the rules follow a more complex heuristic for the sake of\nconvenience. This is covered in depth in examples that follow.\n\nNow that you know the rules, let's look at some examples. Each example is assumed to be evaluated in\na fresh REPL session so that the only globals in each snippet are the ones that are assigned in that\nblock of code.\n\nWe'll begin with a nice and clear-cut situation—assignment inside of a hard scope, in this case a\nfunction body, when no local variable by that name already exists:\n\njulia> function greet()\n           x = \"hello\" # new local\n           println(x)\n       end\ngreet (generic function with 1 method)\n\njulia> greet()\nhello\n\njulia> x # global\nERROR: UndefVarError: `x` not defined in `Main`\n\nInside of the greet function, the assignment x = \"hello\" causes x to be a new local variable\nin the function's scope. There are two relevant facts: the assignment occurs in local scope and\nthere is no existing local x variable. Since x is local, it doesn't matter if there is a global\nnamed x or not. Here for example we define x = 123 before defining and calling greet:\n\njulia> x = 123 # global\n123\n\njulia> function greet()\n           x = \"hello\" # new local\n           println(x)\n       end\ngreet (generic function with 1 method)\n\njulia> greet()\nhello\n\njulia> x # global\n123\n\nSince the x in greet is local, the value (or lack thereof) of the global x is unaffected by\ncalling greet. The hard scope rule doesn't care whether a global named x exists or not:\nassignment to x in a hard scope is local (unless x is declared global).\n\nThe next clear cut situation we'll consider is when there is already a local\nvariable named x, in which case x = <value> always assigns to this existing\nlocal x. This is true whether the assignment occurs in the same local scope,\nan inner local scope in the same function body, or in the body of a function\nnested inside of another function, also known as a\nclosure.\n\nWe'll use the sum_to function, which computes the sum of integers from one up\nto n, as an example:\n\nfunction sum_to(n)\n    s = 0 # new local\n    for i = 1:n\n        s = s + i # assign existing local\n    end\n    return s # same local\nend\n\nAs in the previous example, the first assignment to s at the top of sum_to causes s to be a\nnew local variable in the body of the function. The for loop has its own inner local scope within\nthe function scope. At the point where s = s + i occurs, s is already a local variable, so the\nassignment updates the existing s instead of creating a new local. We can test this out by calling\nsum_to in the REPL:\n\njulia> function sum_to(n)\n           s = 0 # new local\n           for i = 1:n\n               s = s + i # assign existing local\n           end\n           return s # same local\n       end\nsum_to (generic function with 1 method)\n\njulia> sum_to(10)\n55\n\njulia> s # global\nERROR: UndefVarError: `s` not defined in `Main`\n\nSince s is local to the function sum_to, calling the function has no effect on the global\nvariable s. We can also see that the update s = s + i in the for loop must have updated the same\ns created by the initialization s = 0 since we get the correct sum of 55 for the integers 1\nthrough 10.\n\nLet's dig into the fact that the for loop body has its own scope for a second by writing a slightly\nmore verbose variation which we'll call sum_to_def, in which we save the sum s + i in a variable t\nbefore updating s:\n\njulia> function sum_to_def(n)\n           s = 0 # new local\n           for i = 1:n\n               t = s + i # new local `t`\n               s = t # assign existing local `s`\n           end\n           return s, @isdefined(t)\n       end\nsum_to_def (generic function with 1 method)\n\njulia> sum_to_def(10)\n(55, false)\n\nThis version returns s as before but it also uses the @isdefined macro to return a boolean\nindicating whether there is a local variable named t defined in the function's outermost local\nscope. As you can see, there is no t defined outside of the for loop body. This is because of the\nhard scope rule again: since the assignment to t occurs inside of a function, which\nintroduces a hard scope, the assignment causes t to become a new local variable in the local scope\nwhere it appears, i.e. inside of the loop body. Even if there were a global named t, it would make\nno difference—the hard scope rule isn't affected by anything in global scope.\n\nNote that the local scope of a for loop body is no different from the local\nscope of an inner function. This means that we could rewrite this example so\nthat the loop body is implemented as a call to an inner helper function and it\nbehaves the same way:\n\njulia> function sum_to_def_closure(n)\n           function loop_body(i)\n               t = s + i # new local `t`\n               s = t # assign same local `s` as below\n           end\n           s = 0 # new local\n           for i = 1:n\n               loop_body(i)\n           end\n           return s, @isdefined(t)\n       end\nsum_to_def_closure (generic function with 1 method)\n\njulia> sum_to_def_closure(10)\n(55, false)\n\nThis example illustrates a couple of key points:\n\nInner function scopes are just like any other nested local scope. In\nparticular, if a variable is already a local outside of an inner function and\nyou assign to it in the inner function, the outer local variable is updated.\nIt doesn't matter if the definition of an outer local happens below where it\nis updated, the rule remains the same. The entire enclosing local scope is\nparsed and its locals determined before inner local meanings are resolved.\n\nThis design means that you can generally move code in or out of an inner\nfunction without changing its meaning, which facilitates a number of common\nidioms in the language using closures (see [do blocks](@ref\nDo-Block-Syntax-for-Function-Arguments)).\n\nLet's move onto some more ambiguous cases covered by the soft scope rule. We'll explore this by\nextracting the bodies of the greet and sum_to_def functions into soft scope contexts. First, let's put the\nbody of greet in a for loop—which is soft, rather than hard—and evaluate it in the REPL:\n\njulia> for i = 1:3\n           x = \"hello\" # new local\n           println(x)\n       end\nhello\nhello\nhello\n\njulia> x\nERROR: UndefVarError: `x` not defined in `Main`\n\nSince the global x is not defined when the for loop is evaluated, the first clause of the soft\nscope rule applies and x is created as local to the for loop and therefore global x remains\nundefined after the loop executes. Next, let's consider the body of sum_to_def extracted into global\nscope, fixing its argument to n = 10\n\ns = 0\nfor i = 1:10\n    t = s + i\n    s = t\nend\ns\n@isdefined(t)\n\nWhat does this code do? Hint: it's a trick question. The answer is \"it depends.\" If this code is\nentered interactively, it behaves the same way it does in a function body. But if the code appears\nin a file, it  prints an ambiguity warning and throws an undefined variable error. Let's see it\nworking in the REPL first:\n\njulia> s = 0 # global\n0\n\njulia> for i = 1:10\n           t = s + i # new local `t`\n           s = t # assign global `s`\n       end\n\njulia> s # global\n55\n\njulia> @isdefined(t) # global\nfalse\n\nThe REPL approximates being in the body of a function by deciding whether assignment inside the loop\nassigns to a global or creates new local based on whether a global variable by that name is defined\nor not. If a global by the name exists, then the assignment updates it. If no global exists, then\nthe assignment creates a new local variable. In this example we see both cases in action:\n\nThere is no global named t, so t = s + i creates a new t that is local to the for loop;\nThere is a global named s, so s = t assigns to it.\n\nThe second fact is why execution of the loop changes the global value of s and the first fact is\nwhy t is still undefined after the loop executes. Now, let's try evaluating this same code as\nthough it were in a file instead:\n\njulia> code = \"\"\"\n       s = 0 # global\n       for i = 1:10\n           t = s + i # new local `t`\n           s = t # new local `s` with warning\n       end\n       s, # global\n       @isdefined(t) # global\n       \"\"\";\n\njulia> include_string(Main, code)\n┌ Warning: Assignment to `s` in soft scope is ambiguous because a global variable by the same name exists: `s` will be treated as a new local. Disambiguate by using `local s` to suppress this warning or `global s` to assign to the existing global variable.\n└ @ string:4\nERROR: LoadError: UndefVarError: `s` not defined in local scope\n\nHere we use include_string, to evaluate code as though it were the contents of a file.\nWe could also save code to a file and then call include on that file—the result would be the\nsame. As you can see, this behaves quite different from evaluating the same code in the REPL. Let's\nbreak down what's happening here:\n\nglobal s is defined with the value 0 before the loop is evaluated\nthe assignment s = t occurs in a soft scope—a for loop outside of any function body or other hard\nscope construct\ntherefore the second clause of the soft scope rule applies, and the assignment is ambiguous so a\nwarning is emitted\nexecution continues, making s local to the for loop body\nsince s is local to the for loop, it is undefined when t = s + i is evaluated, causing an error\nevaluation stops there, but if it got to s and @isdefined(t), it would return 0 and false.\n\nThis demonstrates some important aspects of scope: in a scope, each variable can only have one\nmeaning, and that meaning is determined regardless of the order of expressions. The presence of the\nexpression s = t in the loop causes s to be local to the loop, which means that it is also local\nwhen it appears on the right hand side of t = s + i, even though that expression appears first and is\nevaluated first. One might imagine that the s on the first line of the loop could be global while\nthe s on the second line of the loop is local, but that's not possible since the two lines are in\nthe same scope block and each variable can only mean one thing in a given scope."},{"title":"On Soft Scope","page":"Scope of Variables","location":"manual/variables-and-scoping.html#on-soft-scope","category":"section","text":"We have now covered all the local scope rules, but before wrapping up this section, perhaps a few\nwords should be said about why the ambiguous soft scope case is handled differently in interactive\nand non-interactive contexts. There are two obvious questions one could ask:\n\nWhy doesn't it just work like the REPL everywhere?\nWhy doesn't it just work like in files everywhere? And maybe skip the warning?\n\nIn Julia ≤ 0.6, all global scopes did work like the current REPL: when x = <value> occurred in a\nloop (or try/catch, or struct body) but outside of a function body (or let block or comprehension),\nit was decided based on whether a global named x was defined or not whether x should be local to\nthe loop. This behavior has the advantage of being intuitive and convenient since it approximates\nthe behavior inside of a function body as closely as possible. In particular, it makes it easy to\nmove code back and forth between a function body and the REPL when trying to debug the behavior of a\nfunction. However, it has some downsides. First, it's quite a complex behavior: many people over the\nyears were confused about this behavior and complained that it was complicated and hard both to\nexplain and understand. Fair point. Second, and arguably worse, is that it's bad for programming \"at\nscale.\" When you see a small piece of code in one place like this, it's quite clear what's going on:\n\ns = 0\nfor i = 1:10\n    s += i\nend\n\nObviously the intention is to modify the existing global variable s. What else could it mean?\nHowever, not all real world code is so short or so clear. We found that code like the following\noften occurs in the wild:\n\nx = 123\n\n# much later\n# maybe in a different file\n\nfor i = 1:10\n    x = \"hello\"\n    println(x)\nend\n\n# much later\n# maybe in yet another file\n# or maybe back in the first one where `x = 123`\n\ny = x + 234\n\nIt's far less clear what should happen here. Since x + \"hello\" is a method error, it seems\nprobable that the intention is for x to be local to the for loop. But runtime values and what\nmethods happen to exist cannot be used to determine the scopes of variables. With the Julia ≤ 0.6\nbehavior, it's especially concerning that someone might have written the for loop first, had it\nworking just fine, but later when someone else adds a new global far away—possibly in a different\nfile—the code suddenly changes meaning and either breaks noisily or, worse still, silently does the\nwrong thing. This kind of \"spooky action at a distance\" is something that good programming language\ndesigns should prevent.\n\nSo in Julia 1.0, we simplified the rules for scope: in any local scope, assignment to a name that\nwasn't already a local variable created a new local variable. This eliminated the notion of soft\nscope entirely as well as removing the potential for spooky action. We uncovered and fixed a significant number of bugs due to the removal of soft scope, vindicating the choice to get rid of it.\nAnd there was much rejoicing! Well, no, not really. Because some people were\nangry that they now had to write:\n\ns = 0\nfor i = 1:10\n    global s += i\nend\n\nDo you see that global annotation in there? Hideous. Obviously this situation could not be\ntolerated. But seriously, there are two main issues with requiring global for this kind of\ntop-level code:\n\nIt's no longer convenient to copy and paste the code from inside a function body into the REPL\nto debug it—you have to add global annotations and then remove them again to go back;\nBeginners will write this kind of code without the global and have no idea why their code\ndoesn't work—the error that they get is that s is undefined, which does not seem to enlighten\nanyone who happens to make this mistake.\n\nAs of Julia 1.5, this code works without the global annotation in interactive contexts like the\nREPL or Jupyter notebooks (just like Julia 0.6) and in files and other non-interactive contexts, it\nprints this very direct warning:\n\nAssignment to s in soft scope is ambiguous because a global variable by the same name exists:\ns will be treated as a new local. Disambiguate by using local s to suppress this warning or\nglobal s to assign to the existing global variable.\n\nThis addresses both issues while preserving the \"programming at scale\" benefits of the 1.0 behavior:\nglobal variables have no spooky effect on the meaning of code that may be far away; in the REPL\ncopy-and-paste debugging works and beginners don't have any issues; any time someone either forgets\na global annotation or accidentally shadows an existing global with a local in a soft scope,\nwhich would be confusing anyway, they get a nice clear warning.\n\nAn important property of this design is that any code that executes in a file without a warning will\nbehave the same way in a fresh REPL. And on the flip side, if you take a REPL session and save it to\nfile, if it behaves differently than it did in the REPL, then you will get a warning."},{"title":"Let Blocks","page":"Scope of Variables","location":"manual/variables-and-scoping.html#Let-Blocks","category":"section","text":"let statements create a new hard scope block (see above) and introduce new variable\nbindings each time they run. The variable need not be immediately assigned:\n\njulia> var1 = let x\n           for i in 1:5\n               (i == 4) && (x = i; break)\n           end\n           x\n       end\n4\n\nWhereas assignments might reassign a new value to an existing value location, let always creates a\nnew location. This difference is usually not important, and is only detectable in the case of\nvariables that outlive their scope via closures. The let syntax accepts a comma-separated series of\nassignments and variable names:\n\njulia> x, y, z = -1, -1, -1;\n\njulia> let x = 1, z\n           println(\"x: $x, y: $y\") # x is local variable, y the global\n           println(\"z: $z\") # errors as z has not been assigned yet but is local\n       end\nx: 1, y: -1\nERROR: UndefVarError: `z` not defined in local scope\n\nThe assignments are evaluated in order, with each right-hand side evaluated in the scope before\nthe new variable on the left-hand side has been introduced. Therefore it makes sense to write\nsomething like let x = x since the two x variables are distinct and have separate storage.\nHere is an example where the behavior of let is needed:\n\njulia> Fs = Vector{Any}(undef, 2); i = 1;\n\njulia> while i <= 2\n           Fs[i] = ()->i\n           global i += 1\n       end\n\njulia> Fs[1]()\n3\n\njulia> Fs[2]()\n3\n\nHere we create and store two closures that return variable i. However, it is always the same\nvariable i, so the two closures behave identically. We can use let to create a new binding\nfor i:\n\njulia> Fs = Vector{Any}(undef, 2); i = 1;\n\njulia> while i <= 2\n           let i = i\n               Fs[i] = ()->i\n           end\n           global i += 1\n       end\n\njulia> Fs[1]()\n1\n\njulia> Fs[2]()\n2\n\nSince the begin construct does not introduce a new scope, it can be useful to use a zero-argument\nlet to just introduce a new scope block without creating any new bindings immediately:\n\njulia> let\n           local x = 1\n           let\n               local x = 2\n           end\n           x\n       end\n1\n\nSince let introduces a new scope block, the inner local x is a different variable than the\nouter local x. This particular example is equivalent to:\n\njulia> let x = 1\n           let x = 2\n           end\n           x\n       end\n1"},{"title":"Loops and Comprehensions","page":"Scope of Variables","location":"manual/variables-and-scoping.html#Loops-and-Comprehensions","category":"section","text":"In loops and comprehensions, new variables\nintroduced in their body scopes are freshly allocated for each loop iteration, as if the loop body\nwere surrounded by a let block, as demonstrated by this example:\n\njulia> Fs = Vector{Any}(undef, 2);\n\njulia> for j = 1:2\n           Fs[j] = ()->j\n       end\n\njulia> Fs[1]()\n1\n\njulia> Fs[2]()\n2\n\nA for loop or comprehension iteration variable is always a new variable:\n\njulia> function f()\n           i = 0\n           for i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n0\n\nHowever, it is occasionally useful to reuse an existing local variable as the iteration variable.\nThis can be done conveniently by adding the keyword outer:\n\njulia> function f()\n           i = 0\n           for outer i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n3"},{"title":"Constants","page":"Scope of Variables","location":"manual/variables-and-scoping.html#Constants","category":"section","text":"A common use of variables is giving names to specific, unchanging values. Such variables are only\nassigned once. This intent can be conveyed to the compiler using the const keyword:\n\njulia> const e  = 2.71828182845904523536;\n\njulia> const pi = 3.14159265358979323846;\n\nMultiple variables can be declared in a single const statement:\n\njulia> const a, b = 1, 2\n(1, 2)\n\nThe const declaration should only be used in global scope on globals.\nIt is difficult for the compiler to optimize code involving global variables, since\ntheir values (or even their types) might change at almost any time. If a global variable will\nnot change, adding a const declaration solves this performance problem.\n\nLocal constants are quite different. The compiler is able to determine automatically when a local\nvariable is constant, so local constant declarations are not necessary, and in fact are currently\nnot supported.\n\nSpecial top-level assignments, such as those performed by the function and struct keywords,\nare constant by default.\n\nNote that const only affects the variable binding; the variable may be bound to a mutable\nobject (such as an array), and that object may still be modified. Additionally when one tries\nto assign a value to a variable that is declared constant the following scenarios are possible:\n\nAttempting to replace a constant without the const keyword is disallowed:\njulia> const x = 1.0\n1.0\n\njulia> x = 1\nERROR: invalid assignment to constant x. This redefinition may be permitted using the `const` keyword.\nAll other definitions of constants are permitted, but may cause significant re-compilation:\njulia> const y = 1.0\n1.0\n\njulia> const y = 2.0\n2.0\n\ncompat: Julia 1.12\nPrior to julia 1.12, redefinition of constants was poorly supported. It was restricted to\nredefinition of constants of the same type and could lead to observably incorrect behavior\nor crashes. Constant redefinition is highly discouraged in versions of julia prior to 1.12.\nSee the manual for prior julia versions for further information."},{"title":"Typed Globals","page":"Scope of Variables","location":"manual/variables-and-scoping.html#man-typed-globals","category":"section","text":"compat: Julia 1.8\nSupport for typed globals was added in Julia 1.8\n\nSimilar to being declared as constants, global bindings can also be declared to always be of a\nconstant type. This can either be done without assigning an actual value using the syntax\nglobal x::T or upon assignment as x::T = 123.\n\njulia> x::Float64 = 2.718\n2.718\n\njulia> f() = x\nf (generic function with 1 method)\n\njulia> Base.return_types(f)\n1-element Vector{Any}:\n Float64\n\nFor any assignment to a global, Julia will first try to convert it to the appropriate type using\nconvert:\n\njulia> global y::Int\n\njulia> y = 1.0\n1.0\n\njulia> y\n1\n\njulia> y = 3.14\nERROR: InexactError: Int64(3.14)\nStacktrace:\n[...]\n\nThe type does not need to be concrete, but annotations with abstract types typically have little\nperformance benefit.\n\nOnce a global has either been assigned to or its type has been set, the binding type is not allowed\nto change:\n\njulia> x = 1\n1\n\njulia> global x::Int\nERROR: cannot set type for global x. It already has a value or is already set to a different type.\nStacktrace:\n[...]"},{"title":"Downloads","page":"Downloads","location":"stdlib/Downloads.html#Downloads","category":"section","text":""},{"title":"Downloads.download","page":"Downloads","location":"stdlib/Downloads.html#Downloads.download","category":"function","text":"download(url, [ output = tempname() ];\n    [ method = \"GET\", ]\n    [ headers = <none>, ]\n    [ timeout = <none>, ]\n    [ progress = <none>, ]\n    [ verbose = false, ]\n    [ debug = <none>, ]\n    [ downloader = <default>, ]\n) -> output\n\n    url        :: AbstractString\n    output     :: Union{AbstractString, AbstractCmd, IO}\n    method     :: AbstractString\n    headers    :: Union{AbstractVector, AbstractDict}\n    timeout    :: Real\n    progress   :: (total::Integer, now::Integer) --> Any\n    verbose    :: Bool\n    debug      :: (type, message) --> Any\n    downloader :: Downloader\n\nDownload a file from the given url, saving it to output or if not specified, a\ntemporary path. The output can also be an IO handle, in which case the body\nof the response is streamed to that handle and the handle is returned. If\noutput is a command, the command is run and output is sent to it on stdin.\n\nIf the downloader keyword argument is provided, it must be a Downloader\nobject. Resources and connections will be shared between downloads performed by\nthe same Downloader and cleaned up automatically when the object is garbage\ncollected or there have been no downloads performed with it for a grace period.\nSee Downloader for more info about configuration and usage.\n\nIf the headers keyword argument is provided, it must be a vector or dictionary\nwhose elements are all pairs of strings. These pairs are passed as headers when\ndownloading URLs with protocols that supports them, such as HTTP/S.\n\nThe timeout keyword argument specifies a timeout for the download to complete in\nseconds, with a resolution of milliseconds. By default no timeout is set, but this\ncan also be explicitly requested by passing a timeout value of Inf. Separately,\nif 20 seconds elapse without receiving any data, the download will timeout. See\nextended help for how to disable this timeout.\n\nIf the progress keyword argument is provided, it must be a callback function\nwhich will be called whenever there are updates about the size and status of the\nongoing download. The callback must take two integer arguments: total and\nnow which are the total size of the download in bytes, and the number of bytes\nwhich have been downloaded so far. Note that total starts out as zero and\nremains zero until the server gives an indication of the total size of the\ndownload (e.g. with a Content-Length header), which may never happen. So a\nwell-behaved progress callback should handle a total size of zero gracefully.\n\nIf the verbose option is set to true, libcurl, which is used to implement\nthe download functionality will print debugging information to stderr. If the\ndebug option is set to a function accepting two String arguments, then the\nverbose option is ignored and instead the data that would have been printed to\nstderr is passed to the debug callback with type and message arguments.\nThe type argument indicates what kind of event has occurred, and is one of:\nTEXT, HEADER IN, HEADER OUT, DATA IN, DATA OUT, SSL DATA IN or SSL DATA OUT. The message argument is the description of the debug event.\n\nExtended Help\n\nFor further customization, use a Downloader and\neasy_hooks.\nFor example, to disable the 20 second timeout when no data is received, you may\nuse the following:\n\ndownloader = Downloads.Downloader()\ndownloader.easy_hook = (easy, info) -> Downloads.Curl.setopt(easy, Downloads.Curl.CURLOPT_LOW_SPEED_TIME, 0)\n\nDownloads.download(\"https://httpbingo.julialang.org/delay/30\"; downloader)\n\n\n\n\n\n"},{"title":"Downloads.request","page":"Downloads","location":"stdlib/Downloads.html#Downloads.request","category":"function","text":"request(url;\n    [ input = <none>, ]\n    [ output = <none>, ]\n    [ method = input ? \"PUT\" : output ? \"GET\" : \"HEAD\", ]\n    [ headers = <none>, ]\n    [ timeout = <none>, ]\n    [ progress = <none>, ]\n    [ verbose = false, ]\n    [ debug = <none>, ]\n    [ throw = true, ]\n    [ downloader = <default>, ]\n    [ interrupt = <none>, ]\n) -> Union{Response, RequestError}\n\n    url        :: AbstractString\n    input      :: Union{AbstractString, AbstractCmd, IO}\n    output     :: Union{AbstractString, AbstractCmd, IO}\n    method     :: AbstractString\n    headers    :: Union{AbstractVector, AbstractDict}\n    timeout    :: Real\n    progress   :: (dl_total, dl_now, ul_total, ul_now) --> Any\n    verbose    :: Bool\n    debug      :: (type, message) --> Any\n    throw      :: Bool\n    downloader :: Downloader\n    interrupt  :: Base.Event\n\nMake a request to the given url, returning a Response object capturing the\nstatus, headers and other information about the response. The body of the\nresponse is written to output if specified and discarded otherwise. For HTTP/S\nrequests, if an input stream is given, a PUT request is made; otherwise if\nan output stream is given, a GET request is made; if neither is given a\nHEAD request is made. For other protocols, appropriate default methods are\nused based on what combination of input and output are requested. The following\noptions differ from the download function:\n\ninput allows providing a request body; if provided default to PUT request\nprogress is a callback taking four integers for upload and download progress\nthrow controls whether to throw or return a RequestError on request error\n\nNote that unlike download which throws an error if the requested URL could not\nbe downloaded (indicated by non-2xx status code), request returns a Response\nobject no matter what the status code of the response is. If there is an error\nwith getting a response at all, then a RequestError is thrown or returned.\n\nIf the interrupt keyword argument is provided, it must be a Base.Event object.\nIf the event is triggered while the request is in progress, the request will be\ncancelled and an error will be thrown. This can be used to interrupt a long\nrunning request, for example if the user wants to cancel a download.\n\n\n\n\n\n"},{"title":"Downloads.Response","page":"Downloads","location":"stdlib/Downloads.html#Downloads.Response","category":"type","text":"struct Response\n    proto   :: String\n    url     :: String\n    status  :: Int\n    message :: String\n    headers :: Vector{Pair{String,String}}\nend\n\nResponse is a type capturing the properties of a successful response to a\nrequest as an object. It has the following fields:\n\nproto: the protocol that was used to get the response\nurl: the URL that was ultimately requested after following redirects\nstatus: the status code of the response, indicating success, failure, etc.\nmessage: a textual message describing the nature of the response\nheaders: any headers that were returned with the response\n\nThe meaning and availability of some of these responses depends on the protocol\nused for the request. For many protocols, including HTTP/S and S/FTP, a 2xx\nstatus code indicates a successful response. For responses in protocols that do\nnot support headers, the headers vector will be empty. HTTP/2 does not include a\nstatus message, only a status code, so the message will be empty.\n\n\n\n\n\n"},{"title":"Downloads.RequestError","page":"Downloads","location":"stdlib/Downloads.html#Downloads.RequestError","category":"type","text":"struct RequestError <: ErrorException\n    url      :: String\n    code     :: Int\n    message  :: String\n    response :: Response\nend\n\nRequestError is a type capturing the properties of a failed response to a\nrequest as an exception object:\n\nurl: the original URL that was requested without any redirects\ncode: the libcurl error code; 0 if a protocol-only error occurred\nmessage: the libcurl error message indicating what went wrong\nresponse: response object capturing what response info is available\n\nThe same RequestError type is thrown by download if the request was\nsuccessful but there was a protocol-level error indicated by a status code that\nis not in the 2xx range, in which case code will be zero and the message\nfield will be the empty string. The request API only throws a RequestError\nif the libcurl error code is non-zero, in which case the included response\nobject is likely to have a status of zero and an empty message. There are,\nhowever, situations where a curl-level error is thrown due to a protocol error,\nin which case both the inner and outer code and message may be of interest.\n\n\n\n\n\n"},{"title":"Downloads.Downloader","page":"Downloads","location":"stdlib/Downloads.html#Downloads.Downloader","category":"type","text":"Downloader(; [ grace::Real = 30 ])\n\nDownloader objects are used to perform individual download operations.\nConnections, name lookups and other resources are shared within a Downloader.\nThese connections and resources are cleaned up after a configurable grace period\n(default: 30 seconds) since anything was downloaded with it, or when it is\ngarbage collected, whichever comes first. If the grace period is set to zero,\nall resources will be cleaned up immediately as soon as there are no more\nongoing downloads in progress. If the grace period is set to Inf then\nresources are not cleaned up until Downloader is garbage collected.\n\n\n\n\n\n"},{"title":"Package Images","page":"Package Images","location":"devdocs/pkgimg.html#pkgimages","category":"section","text":"Julia package images provide object (native code) caches for Julia packages.\nThey are similar to Julia's system image and support many of the same features.\nIn fact the underlying serialization format is the same, and the system image is the base image that the package images are build against."},{"title":"High-level overview","page":"Package Images","location":"devdocs/pkgimg.html#High-level-overview","category":"section","text":"Package images are shared libraries that contain both code and data. Like .ji cache files, they are generated per package. The data section contains both global data (global variables in the package) as well as the necessary metadata about what methods and types are defined by the package. The code section contains native objects that cache the final output of Julia's LLVM-based compiler.\n\nThe command line option --pkgimages=no can be used to turn off object caching for this session. Note that this means that cache files have to likely be regenerated.\nSee JULIA_MAX_NUM_PRECOMPILE_FILES for the upper limit of variants Julia caches per default.\n\nnote: Note\nWhile the package images present themselves as native shared libraries, they are only an approximation thereof. You will not be able to link against them from a native program and they must be loaded from Julia."},{"title":"Linking","page":"Package Images","location":"devdocs/pkgimg.html#Linking","category":"section","text":"Since the package images contain native code, we must run a linker over them before we can use them. You can set the environment variable JULIA_VERBOSE_LINKING to true to make the package image linking process verbose.\n\nFurthermore, we cannot assume that the user has a working system linker installed. Therefore, Julia ships with LLD, the LLVM linker, to provide a working out of the box experience. In base/linking.jl, we implement a limited interface to be able to link package images on all supported platforms."},{"title":"Quirks","page":"Package Images","location":"devdocs/pkgimg.html#Quirks","category":"section","text":"Despite LLD being a multi-platform linker, it does not provide a consistent interface across platforms. Furthermore, it is meant to be used from clang or\nanother compiler driver, we therefore reimplement some of the logic from llvm-project/clang/lib/Driver/ToolChains. Thankfully one can use lld -flavor to set lld to the right platform"},{"title":"Windows","page":"Package Images","location":"devdocs/pkgimg.html#Windows","category":"section","text":"To avoid having to deal with link.exe we use -flavor gnu, effectively turning lld into a cross-linker from a mingw32 environment. Windows DLLs are required to contain a _DllMainCRTStartup function and to minimize our dependence on mingw32 libraries, we inject a stub definition ourselves."},{"title":"MacOS","page":"Package Images","location":"devdocs/pkgimg.html#MacOS","category":"section","text":"Dynamic libraries on macOS need to link against -lSystem. On recent macOS versions, -lSystem is only available for linking when Xcode is available.\nTo that effect we link with -undefined dynamic_lookup."},{"title":"Package images optimized for multiple microarchitectures","page":"Package Images","location":"devdocs/pkgimg.html#pkgimgs-multi-versioning","category":"section","text":"Similar to multi-versioning for system images, package images support multi-versioning. This allows creating package caches that can run efficiently on different CPU architectures within the same environment.\n\nSee the JULIA_CPU_TARGET environment variable for more information on how to set the CPU target for package images."},{"title":"Flags that impact package image creation and selection","page":"Package Images","location":"devdocs/pkgimg.html#Flags-that-impact-package-image-creation-and-selection","category":"section","text":"These are the Julia command line flags that impact cache selection. Package images\nthat were created with different flags will be rejected.\n\n-g, --debug-info: Exact match required since it changes code generation.\n--check-bounds: Exact match required since it changes code generation.\n--inline: Exact match required since it changes code generation.\n--pkgimages: To allow running without object caching enabled.\n-O, --optimize: Reject package images generated for a lower optimization level,\nbut allow for higher optimization levels to be loaded."},{"title":"Dynamic Linker","page":"Dynamic Linker","location":"stdlib/Libdl.html#Dynamic-Linker","category":"section","text":""},{"title":"Lazy Library Loading","page":"Dynamic Linker","location":"stdlib/Libdl.html#Lazy-Library-Loading","category":"section","text":""},{"title":"Libdl","page":"Dynamic Linker","location":"stdlib/Libdl.html#Libdl","category":"module","text":"The Libdl module in Julia provides specialized and lower-level facilities for dynamic linking with shared libraries. While Julia\ninherently supports linking to runtime shared libraries through the ccall intrinsic, Libdl extends this capability by offering additional, more\ngranular control. It enables users to search for shared libraries both in memory and the filesystem, manually load them with specific runtime linker options, and look up\nlibrary symbols as low-level pointers.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlopen","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlopen","category":"function","text":"dlopen(libfile::AbstractString [, flags::Integer]; throw_error:Bool = true)\n\nLoad a shared library, returning an opaque handle.\n\nThe extension given by the constant dlext (.so, .dll, or .dylib)\ncan be omitted from the libfile string, as it is automatically appended\nif needed.   If libfile is not an absolute path name, then the paths\nin the array DL_LOAD_PATH are searched for libfile, followed by the\nsystem load path.\n\nThe optional flags argument is a bitwise-or of zero or more of RTLD_LOCAL, RTLD_GLOBAL,\nRTLD_LAZY, RTLD_NOW, RTLD_NODELETE, RTLD_NOLOAD, RTLD_DEEPBIND, and RTLD_FIRST.\nThese are converted to the corresponding flags of the POSIX (and/or GNU libc and/or MacOS)\ndlopen command, if possible, or are ignored if the specified functionality is not available\non the current platform. The default flags are platform specific. On MacOS the default\ndlopen flags are RTLD_LAZY|RTLD_DEEPBIND|RTLD_GLOBAL while on other platforms the\ndefaults are RTLD_LAZY|RTLD_DEEPBIND|RTLD_LOCAL. An important usage of these flags is to\nspecify non default behavior for when the dynamic library loader binds library references to\nexported symbols and if the bound references are put into process local or global scope. For\ninstance RTLD_LAZY|RTLD_DEEPBIND|RTLD_GLOBAL allows the library's symbols to be available\nfor usage in other shared libraries, addressing situations where there are dependencies\nbetween shared libraries.\n\nIf the library cannot be found, this method throws an error, unless the keyword argument\nthrow_error is set to false, in which case this method returns nothing.\n\nnote: Note\nFrom Julia 1.6 on, this method replaces paths starting with @executable_path/ with\n the path to the Julia executable, allowing for relocatable relative-path loads. In\n Julia 1.5 and earlier, this only worked on macOS.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlopen_e","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlopen_e","category":"function","text":"dlopen_e(libfile::AbstractString [, flags::Integer])\n\nSimilar to dlopen, except returns C_NULL instead of raising errors.\nThis method is now deprecated in favor of dlopen(libfile::AbstractString [, flags::Integer]; throw_error=false).\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.RTLD_NOW","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.RTLD_NOW","category":"constant","text":"RTLD_DEEPBIND\nRTLD_FIRST\nRTLD_GLOBAL\nRTLD_LAZY\nRTLD_LOCAL\nRTLD_NODELETE\nRTLD_NOLOAD\nRTLD_NOW\n\nEnum constant for dlopen. See your platform man page for details, if\napplicable.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlsym","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlsym","category":"function","text":"dlsym(handle, sym; throw_error::Bool = true)\n\nLook up a symbol from a shared library handle, return callable function pointer on success.\n\nIf the symbol cannot be found, this method throws an error, unless the keyword argument\nthrow_error is set to false, in which case this method returns nothing.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlsym_e","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlsym_e","category":"function","text":"dlsym_e(handle, sym)\n\nLook up a symbol from a shared library handle, silently return C_NULL on lookup failure.\nThis method is now deprecated in favor of dlsym(handle, sym; throw_error=false).\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlclose","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlclose","category":"function","text":"dlclose(::Nothing)\n\nFor the very common pattern usage pattern of\n\ntry\n    hdl = dlopen(library_name)\n    ... do something\nfinally\n    dlclose(hdl)\nend\n\nWe define a dlclose() method that accepts a parameter of type Nothing, so\nthat user code does not have to change its behavior for the case that library_name\nwas not found.\n\n\n\n\n\ndlclose(handle)\n\nClose shared library referenced by handle.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlext","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlext","category":"constant","text":"dlext\n\nFile extension for dynamic libraries (e.g. dll, dylib, so) on the current platform.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dllist","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dllist","category":"function","text":"dllist()\n\nReturn the paths of dynamic libraries currently loaded in a Vector{String}.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.dlpath","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.dlpath","category":"function","text":"dlpath(libname::Union{AbstractString, Symbol})\n\nGet the full path of the library libname.\n\nExamples\n\njulia> dlpath(\"libjulia\")\n\n\n\n\n\ndlpath(handle::Ptr{Cvoid})\n\nGiven a library handle from dlopen, return the full path.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.find_library","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.find_library","category":"function","text":"find_library(names [, locations])\n\nSearches for the first library in names in the paths in the locations list,\nDL_LOAD_PATH, or system library paths (in that order) which can successfully be dlopen'd.\nOn success, the return value will be one of the names (potentially prefixed by one of the\npaths in locations). This string can be assigned to a global const and used as the library\nname in future ccall's. On failure, it returns the empty string.\n\n\n\n\n\n"},{"title":"Base.DL_LOAD_PATH","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.DL_LOAD_PATH","category":"constant","text":"DL_LOAD_PATH\n\nWhen calling dlopen, the paths in this list will be searched first, in\norder, before searching the system locations for a valid library handle.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.LazyLibrary","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.LazyLibrary","category":"type","text":"LazyLibrary(name; flags = <default dlopen flags>,\n            dependencies = LazyLibrary[], on_load_callback = nothing)\n\nRepresents a lazily-loaded shared library that delays loading itself and its dependencies\nuntil first use in a ccall(), @ccall, dlopen(), dlsym(), dlpath(), or cglobal().\nThis is a thread-safe mechanism for on-demand library initialization.\n\nArguments\n\nname: Library name (or lazy path computation) as a String,\nLazyLibraryPath, or BundledLazyLibraryPath.\nflags: Optional dlopen flags (default: RTLD_LAZY | RTLD_DEEPBIND). See dlopen.\ndependencies: Vector of LazyLibrary object references to load before this one.\non_load_callback: Optional function to run arbitrary code on first load (use sparingly,\nas it is not expected that ccall() should result in large amounts of Julia code being run.\nYou may call ccall() from within the on_load_callback but only for the current library\nand its dependencies, and user should not call wait() on any tasks within the on load\ncallback as they may deadlock).\n\nThe dlopen operation is thread-safe: only one thread loads the library, acquired after the\nrelease store of the reference to each dependency from loading of each dependency. Other\ntasks block until loading completes. The handle is then cached and reused for all subsequent\ncalls (there is no dlclose for lazy library and dlclose should not be called on the returned\nhandled).\n\nExamples\n\n# Basic usage\nconst mylib = LazyLibrary(\"libmylib\")\n@ccall mylib.myfunc(42::Cint)::Cint\n\n# With dependencies\nconst libfoo = LazyLibrary(\"libfoo\")\nconst libbar = LazyLibrary(\"libbar\"; dependencies=[libfoo])\n\nFor more examples including platform-specific libraries, lazy path construction, and\nmigration from __init__() patterns, see the manual section on\nUsing LazyLibrary for Lazy Loading.\n\ncompat: Julia 1.11\nLazyLibrary was added in Julia 1.11.\n\nSee also LazyLibraryPath, BundledLazyLibraryPath, dlopen,\ndlsym, add_dependency!.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.LazyLibraryPath","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.LazyLibraryPath","category":"type","text":"LazyLibraryPath(path_pieces...)\n\nHelper type for lazily constructed library paths for use with LazyLibrary.\nPath pieces are stored unevaluated and joined with joinpath() when the library is first\naccessed. Arguments must be able to have string() called on them.\n\nExample\n\nconst mylib = LazyLibrary(LazyLibraryPath(artifact_dir, \"lib\", \"libmylib.so.1.2.3\"))\n\ncompat: Julia 1.11\nLazyLibraryPath was added in Julia 1.11.\n\nSee also LazyLibrary, BundledLazyLibraryPath.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.BundledLazyLibraryPath","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.BundledLazyLibraryPath","category":"function","text":"BundledLazyLibraryPath(subpath)\n\nHelper type for lazily constructed library paths within the Julia distribution.\nConstructs paths relative to Julia's private shared library directory.\n\nPrimarily used by Julia's standard library. For example:\n\nconst libgmp = LazyLibrary(BundledLazyLibraryPath(\"libgmp.so.10\"))\n\ncompat: Julia 1.11\nBundledLazyLibraryPath was added in Julia 1.11.\n\nSee also LazyLibrary, LazyLibraryPath.\n\n\n\n\n\n"},{"title":"Base.Libc.Libdl.add_dependency!","page":"Dynamic Linker","location":"stdlib/Libdl.html#Base.Libc.Libdl.add_dependency!","category":"function","text":"add_dependency!(library::LazyLibrary, dependency::LazyLibrary)\n\nDynamically add a dependency that must be loaded before library. Only needed when\ndependencies cannot be determined at construction time.\n\nwarning: Warning\nDependencies added with this function are ephemeral and only persist within the\ncurrent process. They will not persist across precompilation boundaries.\n\nPrefer specifying dependencies in the LazyLibrary constructor when possible.\n\ncompat: Julia 1.11\nadd_dependency! was added in Julia 1.11.\n\nSee also LazyLibrary.\n\n\n\n\n\n"},{"title":"Logging","page":"Logging","location":"stdlib/Logging.html#man-logging","category":"section","text":"The Logging module provides a way to record the history and progress of a\ncomputation as a log of events. Events are created by inserting a logging\nstatement into the source code, for example:\n\n@warn \"Abandon printf debugging, all ye who enter here!\"\n┌ Warning: Abandon printf debugging, all ye who enter here!\n└ @ Main REPL[1]:1\n\nThe system provides several advantages over peppering your source code with\ncalls to println(). First, it allows you to control the visibility and\npresentation of messages without editing the source code. For example, in\ncontrast to the @warn above\n\n@debug \"The sum of some values $(sum(rand(100)))\"\n\nwill produce no output by default. Furthermore, it's very cheap to leave debug\nstatements like this in the source code because the system avoids evaluating\nthe message if it would later be ignored. In this case sum(rand(100)) and\nthe associated string processing will never be executed unless debug logging is\nenabled.\n\nSecond, the logging tools allow you to attach arbitrary data to each event as a\nset of key–value pairs. This allows you to capture local variables and other\nprogram state for later analysis. For example, to attach the local array\nvariable A and the sum of a vector v as the key s you can use\n\nA = ones(Int, 4, 4)\nv = ones(100)\n@info \"Some variables\"  A  s=sum(v)\n\n# output\n┌ Info: Some variables\n│   A =\n│    4×4 Matrix{Int64}:\n│     1  1  1  1\n│     1  1  1  1\n│     1  1  1  1\n│     1  1  1  1\n└   s = 100.0\n\nAll of the logging macros @debug, @info, @warn and @error share common\nfeatures that are described in detail in the documentation for the more\ngeneral macro @logmsg."},{"title":"Log event structure","page":"Logging","location":"stdlib/Logging.html#Log-event-structure","category":"section","text":"Each event generates several pieces of data, some provided by the user and some\nautomatically extracted. Let's examine the user-defined data first:\n\nThe log level is a broad category for the message that is used for early\nfiltering. There are several standard levels of type LogLevel;\nuser-defined levels are also possible.\nEach is distinct in purpose:\nLogging.Debug (log level -1000) is information intended for the developer of\nthe program. These events are disabled by default.\nLogging.Info (log level 0) is for general information to the user.\nThink of it as an alternative to using println directly.\nLogging.Warn (log level 1000) means something is wrong and action is likely\nrequired but that for now the program is still working.\nLogging.Error (log level 2000) means something is wrong and it is unlikely to\nbe recovered, at least by this part of the code.\nOften this log-level is unneeded as throwing an exception can convey\nall the required information.\nThe message  is an object describing the event. By convention\nAbstractStrings passed as messages are assumed to be in markdown format.\nOther types will be displayed using print(io, obj) or string(obj) for\ntext-based output and possibly show(io,mime,obj) for other multimedia\ndisplays used in the installed logger.\nOptional key–value pairs allow arbitrary data to be attached to each event.\nSome keys have conventional meaning that can affect the way an event is\ninterpreted (see @logmsg).\n\nThe system also generates some standard information for each event:\n\nThe module in which the logging macro was expanded.\nThe file and line where the logging macro occurs in the source code.\nA message id that is a unique, fixed identifier for the source code\nstatement where the logging macro appears. This identifier is designed to be\nfairly stable even if the source code of the file changes, as long as the\nlogging statement itself remains the same.\nA group for the event, which is set to the base name of the file by default,\nwithout extension. This can be used to group messages into categories more\nfinely than the log level (for example, all deprecation warnings have group\n:depwarn), or into logical groupings across or within modules.\n\nNotice that some useful information such as the event time is not included by\ndefault. This is because such information can be expensive to extract and is\nalso dynamically available to the current logger. It's simple to define a\ncustom logger to augment event data with the\ntime, backtrace, values of global variables and other useful information as\nrequired."},{"title":"Processing log events","page":"Logging","location":"stdlib/Logging.html#Processing-log-events","category":"section","text":"As you can see in the examples, logging statements make no mention of\nwhere log events go or how they are processed. This is a key design feature\nthat makes the system composable and natural for concurrent use. It does this\nby separating two different concerns:\n\nCreating log events is the concern of the module author who needs to\ndecide where events are triggered and which information to include.\nProcessing of log events — that is, display, filtering, aggregation and\nrecording — is the concern of the application author who needs to bring\nmultiple modules together into a cooperating application."},{"title":"Loggers","page":"Logging","location":"stdlib/Logging.html#Loggers","category":"section","text":"Processing of events is performed by a logger, which is the first piece of\nuser configurable code to see the event. All loggers must be subtypes of\nAbstractLogger.\n\nWhen an event is triggered, the appropriate logger is found by looking for a\ntask-local logger with the global logger as fallback. The idea here is that\nthe application code knows how log events should be processed and exists\nsomewhere at the top of the call stack. So we should look up through the call\nstack to discover the logger — that is, the logger should be dynamically\nscoped. (This is a point of contrast with logging frameworks where the\nlogger is lexically scoped; provided explicitly by the module author or as a\nsimple global variable. In such a system it's awkward to control logging while\ncomposing functionality from multiple modules.)\n\nThe global logger may be set with global_logger, and task-local\nloggers controlled using with_logger. Newly spawned tasks inherit\nthe logger of the parent task.\n\nThere are three logger types provided by the library.  ConsoleLogger\nis the default logger you see when starting the REPL. It displays events in a\nreadable text format and tries to give simple but user friendly control over\nformatting and filtering.  NullLogger is a convenient way to drop all\nmessages where necessary; it is the logging equivalent of the devnull\nstream.  SimpleLogger is a very simplistic text formatting logger,\nmainly useful for debugging the logging system itself.\n\nCustom loggers should come with overloads for the functions described in the\nreference section."},{"title":"Early filtering and message handling","page":"Logging","location":"stdlib/Logging.html#Early-filtering-and-message-handling","category":"section","text":"When an event occurs, a few steps of early filtering occur to avoid generating\nmessages that will be discarded:\n\nThe message log level is checked against a global minimum level (set via\ndisable_logging). This is a crude but extremely cheap global\nsetting.\nThe current logger state is looked up and the message level checked against the\nlogger's cached minimum level, as found by calling Logging.min_enabled_level.\nThis behavior can be overridden via environment variables (more on this later).\nThe Logging.shouldlog function is called with the current logger, taking\nsome minimal information (level, module, group, id) which can be computed\nstatically. Most usefully, shouldlog is passed an event id which can be\nused to discard events early based on a cached predicate.\n\nIf all these checks pass, the message and key–value pairs are evaluated in full\nand passed to the current logger via the Logging.handle_message function.\nhandle_message() may perform additional filtering as required and display the\nevent to the screen, save it to a file, etc.\n\nExceptions that occur while generating the log event are captured and logged\nby default. This prevents individual broken events from crashing the\napplication, which is helpful when enabling little-used debug events in a\nproduction system. This behavior can be customized per logger type by\nextending Logging.catch_exceptions."},{"title":"Testing log events","page":"Logging","location":"stdlib/Logging.html#Testing-log-events","category":"section","text":"Log events are a side effect of running normal code, but you might find\nyourself wanting to test particular informational messages and warnings. The\nTest module provides a @test_logs macro that can be used to\npattern match against the log event stream."},{"title":"Environment variables","page":"Logging","location":"stdlib/Logging.html#Environment-variables","category":"section","text":"Message filtering can be influenced through the JULIA_DEBUG environment\nvariable, and serves as an easy way to enable debug logging for a file or\nmodule. Loading julia with JULIA_DEBUG=loading will activate\n@debug log messages in loading.jl. For example, in Linux shells:\n\n$ JULIA_DEBUG=loading julia -e 'using OhMyREPL'\n┌ Debug: Rejecting cache file /home/user/.julia/compiled/v0.7/OhMyREPL.ji due to it containing an incompatible cache header\n└ @ Base loading.jl:1328\n[ Info: Recompiling stale cache file /home/user/.julia/compiled/v0.7/OhMyREPL.ji for module OhMyREPL\n┌ Debug: Rejecting cache file /home/user/.julia/compiled/v0.7/Tokenize.ji due to it containing an incompatible cache header\n└ @ Base loading.jl:1328\n...\n\nOn windows, the same can be achieved in CMD via first running set JULIA_DEBUG=\"loading\" and in Powershell via\n$env:JULIA_DEBUG=\"loading\".\n\nSimilarly, the environment variable can be used to enable debug logging of\nmodules, such as Pkg, or module roots (see Base.moduleroot). To\nenable all debug logging, use the special value all.\n\nTo turn debug logging on from the REPL, set ENV[\"JULIA_DEBUG\"] to the\nname of the module of interest. Functions defined in the REPL belong to\nmodule Main; logging for them can be enabled like this:\n\njulia> foo() = @debug \"foo\"\nfoo (generic function with 1 method)\n\njulia> foo()\n\njulia> ENV[\"JULIA_DEBUG\"] = Main\nMain\n\njulia> foo()\n┌ Debug: foo\n└ @ Main REPL[1]:1\n\n\nUse a comma separator to enable debug for multiple\nmodules: JULIA_DEBUG=loading,Main."},{"title":"Examples","page":"Logging","location":"stdlib/Logging.html#Examples","category":"section","text":""},{"title":"Example: Writing log events to a file","page":"Logging","location":"stdlib/Logging.html#Example:-Writing-log-events-to-a-file","category":"section","text":"Sometimes it can be useful to write log events to a file. Here is an example\nof how to use a task-local and global logger to write information to a text\nfile:\n\n# Load the logging module\njulia> using Logging\n\n# Open a textfile for writing\njulia> io = open(\"log.txt\", \"w+\")\nIOStream(<file log.txt>)\n\n# Create a simple logger\njulia> logger = SimpleLogger(io)\nSimpleLogger(IOStream(<file log.txt>), Info, Dict{Any,Int64}())\n\n# Log a task-specific message\njulia> with_logger(logger) do\n           @info(\"a context specific log message\")\n       end\n\n# Write all buffered messages to the file\njulia> flush(io)\n\n# Set the global logger to logger\njulia> global_logger(logger)\nSimpleLogger(IOStream(<file log.txt>), Info, Dict{Any,Int64}())\n\n# This message will now also be written to the file\njulia> @info(\"a global log message\")\n\n# Close the file\njulia> close(io)"},{"title":"Example: Enable debug-level messages","page":"Logging","location":"stdlib/Logging.html#Example:-Enable-debug-level-messages","category":"section","text":"Here is an example of creating a ConsoleLogger that lets through any messages\nwith log level higher than, or equal, to Logging.Debug.\n\njulia> using Logging\n\n# Create a ConsoleLogger that prints any log messages with level >= Debug to stderr\njulia> debuglogger = ConsoleLogger(stderr, Logging.Debug)\n\n# Enable debuglogger for a task\njulia> with_logger(debuglogger) do\n           @debug \"a context specific log message\"\n       end\n\n# Set the global logger\njulia> global_logger(debuglogger)"},{"title":"Reference","page":"Logging","location":"stdlib/Logging.html#Reference","category":"section","text":""},{"title":"Logging module","page":"Logging","location":"stdlib/Logging.html#Logging-module","category":"section","text":""},{"title":"Creating events","page":"Logging","location":"stdlib/Logging.html#Creating-events","category":"section","text":""},{"title":"Processing events with AbstractLogger","page":"Logging","location":"stdlib/Logging.html#AbstractLogger-interface","category":"section","text":"Event processing is controlled by overriding functions associated with\nAbstractLogger:\n\nMethods to implement  Brief description\nLogging.handle_message  Handle a log event\nLogging.shouldlog  Early filtering of events\nLogging.min_enabled_level  Lower bound for log level of accepted events\nOptional methods Default definition Brief description\nLogging.catch_exceptions true Catch exceptions during event evaluation"},{"title":"Using Loggers","page":"Logging","location":"stdlib/Logging.html#Using-Loggers","category":"section","text":"Logger installation and inspection:\n\nLoggers that are supplied with the system:"},{"title":"Logging.Logging","page":"Logging","location":"stdlib/Logging.html#Logging.Logging","category":"module","text":"Utilities for capturing, filtering and presenting streams of log events.\nNormally you don't need to import Logging to create log events; for this\nthe standard logging macros such as @info are already exported by Base\nand available by default.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.@logmsg","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.@logmsg","category":"macro","text":"@debug message  [key=value | value ...]\n@info  message  [key=value | value ...]\n@warn  message  [key=value | value ...]\n@error message  [key=value | value ...]\n\n@logmsg level message [key=value | value ...]\n\nCreate a log record with an informational message.  For convenience, four\nlogging macros @debug, @info, @warn and @error are defined which log at\nthe standard severity levels Debug, Info, Warn and Error.  @logmsg\nallows level to be set programmatically to any LogLevel or custom log level\ntypes.\n\nmessage should be an expression which evaluates to a string which is a human\nreadable description of the log event.  By convention, this string will be\nformatted as markdown when presented.\n\nThe optional list of key=value pairs supports arbitrary user defined\nmetadata which will be passed through to the logging backend as part of the\nlog record.  If only a value expression is supplied, a key representing the\nexpression will be generated using Symbol. For example, x becomes x=x,\nand foo(10) becomes Symbol(\"foo(10)\")=foo(10).  For splatting a list of\nkey value pairs, use the normal splatting syntax, @info \"blah\" kws....\n\nThere are some keys which allow automatically generated log data to be\noverridden:\n\n_module=mod can be used to specify a different originating module from\nthe source location of the message.\n_group=symbol can be used to override the message group (this is\nnormally derived from the base name of the source file).\n_id=symbol can be used to override the automatically generated unique\nmessage identifier.  This is useful if you need to very closely associate\nmessages generated on different source lines.\n_file=string and _line=integer can be used to override the apparent\nsource location of a log message.\n\nThere's also some key value pairs which have conventional meaning:\n\nmaxlog=integer should be used as a hint to the backend that the message\nshould be displayed no more than maxlog times.\nexception=ex should be used to transport an exception with a log message,\noften used with @error. An associated backtrace bt may be attached\nusing the tuple exception=(ex,bt).\n\nExamples\n\n@debug \"Verbose debugging information.  Invisible by default\"\n@info  \"An informational message\"\n@warn  \"Something was odd.  You should pay attention\"\n@error \"A non fatal error occurred\"\n\nx = 10\n@info \"Some variables attached to the message\" x a=42.0\n\n@debug begin\n    sA = sum(A)\n    \"sum(A) = $sA is an expensive operation, evaluated only when `shouldlog` returns true\"\nend\n\nfor i=1:10000\n    @info \"With the default backend, you will only see (i = $i) ten times\"  maxlog=10\n    @debug \"Algorithm1\" i progress=i/10000\nend\n\n\n\n\n\n"},{"title":"Base.CoreLogging.LogLevel","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.LogLevel","category":"type","text":"LogLevel(level)\n\nSeverity/verbosity of a log record.\n\nThe log level provides a key against which potential log records may be\nfiltered, before any other work is done to construct the log record data\nstructure itself.\n\nExamples\n\njulia> Logging.LogLevel(0) == Logging.Info\ntrue\n\n\n\n\n\n"},{"title":"Base.CoreLogging.Debug","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.Debug","category":"constant","text":"Debug\n\nAlias for LogLevel(-1000).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.Info","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.Info","category":"constant","text":"Info\n\nAlias for LogLevel(0).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.Warn","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.Warn","category":"constant","text":"Warn\n\nAlias for LogLevel(1000).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.Error","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.Error","category":"constant","text":"Error\n\nAlias for LogLevel(2000).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.BelowMinLevel","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.BelowMinLevel","category":"constant","text":"BelowMinLevel\n\nAlias for LogLevel(-1_000_001).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.AboveMaxLevel","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.AboveMaxLevel","category":"constant","text":"AboveMaxLevel\n\nAlias for LogLevel(1_000_001).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.AbstractLogger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.AbstractLogger","category":"type","text":"A logger controls how log records are filtered and dispatched.  When a log\nrecord is generated, the logger is the first piece of user configurable code\nwhich gets to inspect the record and decide what to do with it.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.handle_message","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.handle_message","category":"function","text":"handle_message(logger, level, message, _module, group, id, file, line; key1=val1, ...)\n\nLog a message to logger at level.  The logical location at which the\nmessage was generated is given by module _module and group; the source\nlocation by file and line. id is an arbitrary unique value (typically a\nSymbol) to be used as a key to identify the log statement when\nfiltering.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.shouldlog","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.shouldlog","category":"function","text":"shouldlog(logger, level, _module, group, id)\n\nReturn true when logger accepts a message at level, generated for\n_module, group and with unique log identifier id.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.min_enabled_level","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.min_enabled_level","category":"function","text":"min_enabled_level(logger)\n\nReturn the minimum enabled level for logger for early filtering.  That is,\nthe log level below or equal to which all messages are filtered.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.catch_exceptions","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.catch_exceptions","category":"function","text":"catch_exceptions(logger)\n\nReturn true if the logger should catch exceptions which happen during log\nrecord construction.  By default, messages are caught.\n\nBy default all exceptions are caught to prevent log message generation from\ncrashing the program.  This lets users confidently toggle little-used\nfunctionality - such as debug logging - in a production system.\n\nIf you want to use logging as an audit trail you should disable this for your\nlogger type.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.disable_logging","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.disable_logging","category":"function","text":"disable_logging(level)\n\nDisable all log messages at log levels equal to or less than level.  This is\na global setting, intended to make debug logging extremely cheap when\ndisabled. Note that this cannot be used to enable logging that is currently disabled\nby other mechanisms.\n\nExamples\n\nLogging.disable_logging(Logging.Info) # Disable debug and info\n\n\n\n\n\n"},{"title":"Base.CoreLogging.global_logger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.global_logger","category":"function","text":"global_logger()\n\nReturn the global logger, used to receive messages when no specific logger\nexists for the current task.\n\nglobal_logger(logger)\n\nSet the global logger to logger, and return the previous global logger.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.with_logger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.with_logger","category":"function","text":"with_logger(function, logger)\n\nExecute function, directing all log messages to logger.\n\nExamples\n\nfunction test(x)\n    @info \"x = $x\"\nend\n\nwith_logger(logger) do\n    test(1)\n    test([1,2])\nend\n\n\n\n\n\n"},{"title":"Base.CoreLogging.current_logger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.current_logger","category":"function","text":"current_logger()\n\nReturn the logger for the current task, or the global logger if none is\nattached to the task.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.NullLogger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.NullLogger","category":"type","text":"NullLogger()\n\nLogger which disables all messages and produces no output - the logger\nequivalent of /dev/null.\n\n\n\n\n\n"},{"title":"Base.CoreLogging.ConsoleLogger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.ConsoleLogger","category":"type","text":"ConsoleLogger([stream,] min_level=Info; meta_formatter=default_metafmt,\n              show_limited=true, right_justify=0)\n\nLogger with formatting optimized for readability in a text console, for example\ninteractive work with the Julia REPL.\n\nLog levels less than min_level are filtered out.\n\nThis Logger is thread-safe, with locks for both orchestration of message\nlimits i.e. maxlog, and writes to the stream.\n\nMessage formatting can be controlled by setting keyword arguments:\n\nmeta_formatter is a function which takes the log event metadata\n(level, _module, group, id, file, line) and returns a color (as would be\npassed to printstyled), prefix and suffix for the log message.  The\ndefault is to prefix with the log level and a suffix containing the module,\nfile and line location.\nshow_limited limits the printing of large data structures to something\nwhich can fit on the screen by setting the :limit IOContext key during\nformatting.\nright_justify is the integer column which log metadata is right justified\nat. The default is zero (metadata goes on its own line).\n\n\n\n\n\n"},{"title":"Base.CoreLogging.SimpleLogger","page":"Logging","location":"stdlib/Logging.html#Base.CoreLogging.SimpleLogger","category":"type","text":"SimpleLogger([stream,] min_level=Info)\n\nSimplistic logger for logging all messages with level greater than or equal to\nmin_level to stream. If stream is closed then messages with log level\ngreater or equal to Warn will be logged to stderr and below to stdout.\n\nThis Logger is thread-safe, with a lock taken around orchestration of message\nlimits i.e. maxlog, and writes to the stream.\n\n\n\n\n\n"},{"title":"Fixing precompilation hangs due to open tasks or IO","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html#Fixing-precompilation-hangs-due-to-open-tasks-or-IO","category":"section","text":"On Julia 1.10 or higher, you might see the following message:\n\n(Image: Screenshot of precompilation hang)\n\nThis may repeat. If it continues to repeat with no hints that it will\nresolve itself, you may have a \"precompilation hang\" that requires\nfixing. Even if it's transient, you might prefer to resolve it so that\nusers will not be bothered by this warning. This page walks you\nthrough how to analyze and fix such issues.\n\nIf you follow the advice and hit Ctrl-C, you might see\n\n^C Interrupted: Exiting precompilation...\n\n  1 dependency had warnings during precompilation:\n┌ Test1 [ac89d554-e2ba-40bc-bc5c-de68b658c982]\n│  [pid 2745] Waiting for background task / IO / timer to finish:\n│   Handle type        uv_handle_t->data\n│   timer              0x55580decd1e0->0x7f94c3a4c340\n\nThis message conveys two key pieces of information:\n\nthe hang is occurring during precompilation of Test1, a dependency of Test2 (the package we were trying to load with using Test2)\nduring precompilation of Test1, Julia created a Timer object (use ?Timer if you're unfamiliar with Timers) which is still open; until that closes, the process is hung\n\nIf this is enough of a hint for you to figure out how timer = Timer(args...) is being created, one good solution is to add wait(timer) if timer eventually finishes on its own, or close(timer) if you need to force-close it, before the final end of the module.\n\nHowever, there are cases that may not be that straightforward. Usually the best option is to start by determining whether the hang is due to code in Test1 or whether it is due to one of Test1's dependencies:\n\nOption 1: Pkg.add(\"Aqua\") and use Aqua.test_persistent_tasks. This should help you identify which package is causing the problem, after which the instructions below should be followed. If needed, you can create a PkgId as Base.PkgId(UUID(\"...\"), \"Test1\"), where ... comes from the uuid entry in Test1/Project.toml.\nOption 2: manually diagnose the source of the hang.\n\nTo manually diagnose:\n\nPkg.develop(\"Test1\")\nComment out all the code included or defined in Test1, except the using/import statements.\nTry using Test2 (or even using Test1 assuming that hangs too) again\n\nNow we arrive at a fork in the road: either\n\nthe hang persists, indicating it is due to one of your dependencies\nthe hang disappears, indicating that it is due to something in your code."},{"title":"Diagnosing and fixing hangs due to a package dependency","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html#pchang_deps","category":"section","text":"Use a binary search to identify the problematic dependency: start by commenting out half your dependencies, then when you isolate which half is responsible comment out half of that half, etc. (You don't have to remove them from the project, just comment out the using/import statements.)\n\nOnce you've identified a suspect (here we'll call it ThePackageYouThinkIsCausingTheProblem), first try precompiling that package. If it also hangs during precompilation, continue chasing the problem backwards.\n\nHowever, most likely ThePackageYouThinkIsCausingTheProblem will precompile fine. This suggests it's in the function ThePackageYouThinkIsCausingTheProblem.__init__, which does not run during precompilation of ThePackageYouThinkIsCausingTheProblem but does in any package that loads ThePackageYouThinkIsCausingTheProblem. To test this theory, set up a minimal working example (MWE), something like\n\n(@v1.10) pkg> generate MWE\n  Generating  project MWE:\n    MWE\\Project.toml\n    MWE\\src\\MWE.jl\n\nwhere the source code of MWE.jl is\n\nmodule MWE\nusing ThePackageYouThinkIsCausingTheProblem\nend\n\nand you've added ThePackageYouThinkIsCausingTheProblem to MWE's dependencies.\n\nIf that MWE reproduces the hang, you've found your culprit:\nThePackageYouThinkIsCausingTheProblem.__init__ must be creating the Timer object. If the timer object can be safely closed, that's a good option. Otherwise, the most common solution is to avoid creating the timer while any package is being precompiled: add\n\nccall(:jl_generating_output, Cint, ()) == 1 && return nothing\n\nas the first line of ThePackageYouThinkIsCausingTheProblem.__init__, and it will avoid doing any initialization in any Julia process whose purpose is to precompile packages."},{"title":"Fixing package code to avoid hangs","page":"Fixing precompilation hangs due to open tasks or IO","location":"devdocs/precompile_hang.html#pchang_fix","category":"section","text":"Search your package for suggestive words (here like \"Timer\") and see if you can identify where the problem is being created. Note that a method definition like\n\nmaketimer() = Timer(timer -> println(\"hi\"), 0; interval=1)\n\nis not problematic in and of itself: it can cause this problem only if maketimer gets called while the module is being defined. This might be happening from a top-level statement such as\n\nconst GLOBAL_TIMER = maketimer()\n\nor it might conceivably occur in a precompile workload.\n\nIf you struggle to identify the causative lines, then consider doing a binary search: comment out sections of your package (or include lines to omit entire files) until you've reduced the problem in scope."},{"title":"Using Valgrind with Julia","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Using-Valgrind-with-Julia","category":"section","text":"Valgrind is a tool for memory debugging, memory leak detection, and profiling.\n This section describes things to keep in mind when using Valgrind to debug memory issues with\nJulia."},{"title":"General considerations","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#General-considerations","category":"section","text":"By default, Valgrind assumes that there is no self modifying code in the programs it runs. This\nassumption works fine in most instances but fails miserably for a just-in-time compiler like\njulia. For this reason it is crucial to pass --smc-check=all-non-file to valgrind, else\ncode may crash or behave unexpectedly (often in subtle ways).\n\nIn some cases, to better detect memory errors using Valgrind, it can help to compile julia with\nmemory pools disabled. The compile-time flag MEMDEBUG disables memory pools in Julia, and\nMEMDEBUG2 disables memory pools in FemtoLisp. To build julia with both flags, add the following\nline to Make.user:\n\nCFLAGS = -DMEMDEBUG -DMEMDEBUG2\n\nAnother thing to note: if your program uses multiple worker processes, it is likely that you\nwant all such worker processes to run under Valgrind, not just the parent process. To do this,\npass --trace-children=yes to valgrind.\n\nYet another thing to note: if using valgrind errors with Unable to find compatible target in system image,\ntry rebuilding the sysimage with target generic or julia with JULIA_CPU_TARGET=generic."},{"title":"Suppressions","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Suppressions","category":"section","text":"Valgrind will typically display spurious warnings as it runs. To reduce the number of such warnings,\nit helps to provide a suppressions file\nto Valgrind. A sample suppressions file is included in the Julia source distribution at contrib/valgrind-julia.supp.\n\nThe suppressions file can be used from the julia/ source directory as follows:\n\n$ valgrind --smc-check=all-non-file --suppressions=contrib/valgrind-julia.supp ./julia progname.jl\n\nAny memory errors that are displayed should either be reported as bugs or contributed as additional\nsuppressions. Note that some versions of Valgrind are shipped with insufficient default suppressions,\nso that may be one thing to consider before submitting any bugs."},{"title":"Running the Julia test suite under Valgrind","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Running-the-Julia-test-suite-under-Valgrind","category":"section","text":"It is possible to run the entire Julia test suite under Valgrind, but it does take quite some\ntime (typically several hours). To do so, run the following command from the julia/test/ directory:\n\nvalgrind --smc-check=all-non-file --trace-children=yes --suppressions=$PWD/../contrib/valgrind-julia.supp ../julia runtests.jl all\n\nIf you would like to see a report of \"definite\" memory leaks, pass the flags --leak-check=full --show-leak-kinds=definite\nto valgrind as well."},{"title":"Additional spurious warnings","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Additional-spurious-warnings","category":"section","text":"This section covers Valgrind warnings that cannot be added to the\nsuppressions file yet are nonetheless safe to ignore."},{"title":"Unhandled rr system calls","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Unhandled-rr-system-calls","category":"section","text":"Valgrind will emit a warning if it encounters any of the [system calls\nthat are specific to\nrr](https://github.com/rr-debugger/rr/blob/master/src/preload/rrcalls.h),\nthe Record and Replay Framework. In\nparticular, a warning about an unhandled 1008 syscall will be shown\nwhen julia tries to detect whether it is running under rr:\n\n--xxxxxx-- WARNING: unhandled amd64-linux syscall: 1008\n--xxxxxx-- You may be able to write your own handler.\n--xxxxxx-- Read the file README_MISSING_SYSCALL_OR_IOCTL.\n--xxxxxx-- Nevertheless we consider this a bug.  Please report\n--xxxxxx-- it at http://valgrind.org/support/bug_reports.html.\n\nThis issue\nhas been reported\nto the Valgrind developers as they have requested."},{"title":"Caveats","page":"Using Valgrind with Julia","location":"devdocs/valgrind.html#Caveats","category":"section","text":"Valgrind currently does not support multiple rounding modes,\nso code that adjusts the rounding mode will behave differently when run under Valgrind.\n\nIn general, if after setting --smc-check=all-non-file you find that your program behaves differently\nwhen run under Valgrind, it may help to pass --tool=none to valgrind as you investigate further.\n This will enable the minimal Valgrind machinery but will also run much faster than when the full\nmemory checker is enabled."},{"title":"Workflow Tips","page":"Workflow Tips","location":"manual/workflow-tips.html#man-workflow-tips","category":"section","text":"Here are some tips for working with Julia efficiently."},{"title":"REPL-based workflow","page":"Workflow Tips","location":"manual/workflow-tips.html#REPL-based-workflow","category":"section","text":"As already elaborated in The Julia REPL, Julia's REPL provides rich functionality\nthat facilitates an efficient interactive workflow. Here are some tips that might further enhance\nyour experience at the command line."},{"title":"A basic editor/REPL workflow","page":"Workflow Tips","location":"manual/workflow-tips.html#A-basic-editor/REPL-workflow","category":"section","text":"The most basic Julia workflows involve using a text editor in conjunction with the julia command line.\n\nCreate a file, say Tmp.jl, and include within it\n\nmodule Tmp\n\nsay_hello() = println(\"Hello!\")\n\n# Your other definitions here\n\nend # module\n\nusing .Tmp\n\nThen, in the same directory, start the Julia REPL (using the julia command).\nRun the new file as follows:\n\njulia> include(\"Tmp.jl\")\n\njulia> Tmp.say_hello()\nHello!\n\nExplore ideas in the REPL. Save good ideas in Tmp.jl.\nTo reload the file after it has been changed, just include it again.\n\nThe key in the above is that your code is encapsulated in a module.\nThat allows you to edit struct definitions and remove methods, without restarting Julia.\n\n(Explanation: structs cannot be edited after definition, nor can methods be deleted.\nBut you can overwrite the definition of a module, which is what we do when we re-include(\"Tmp.jl\")).\n\nIn addition, the encapsulation of code in a module protects it from being influenced\nby previous state in the REPL, protecting you from hard-to-detect errors."},{"title":"Browser-based workflow","page":"Workflow Tips","location":"manual/workflow-tips.html#Browser-based-workflow","category":"section","text":"There are a few ways to interact with Julia in a browser:\n\nUsing Pluto notebooks through Pluto.jl\nUsing Jupyter notebooks through IJulia.jl"},{"title":"Revise-based workflows","page":"Workflow Tips","location":"manual/workflow-tips.html#Revise-based-workflows","category":"section","text":"Whether you're at the REPL or in IJulia, you can typically improve\nyour development experience with\nRevise.\nIt is common to configure Revise to start whenever julia is started,\nas per the instructions in the Revise documentation.\nOnce configured, Revise will track changes to files in any loaded modules,\nand to any files loaded in to the REPL with includet (but not with plain include);\nyou can then edit the files and the changes take effect without restarting your julia session.\nA standard workflow is similar to the REPL-based workflow above, with\nthe following modifications:\n\nPut your code in a module somewhere on your load path. There are\nseveral options for achieving this, of which two recommended choices are:\nFor long-term projects, use\nPkgTemplates:\nusing PkgTemplates\nt = Template()\nt(\"MyPkg\")\nThis will create a blank package, \"MyPkg\", in your .julia/dev directory.\nNote that PkgTemplates allows you to control many different options\nthrough its Template constructor.\nIn step 2 below, edit MyPkg/src/MyPkg.jl to change the source code, and\nMyPkg/test/runtests.jl for the tests.\nFor \"throw-away\" projects, you can avoid any need for cleanup\nby doing your work in your temporary directory (e.g., /tmp).\nNavigate to your temporary directory and launch Julia, then do the following:\npkg> generate MyPkg            # type ] to enter pkg mode\njulia> push!(LOAD_PATH, pwd())   # hit backspace to exit pkg mode\nIf you restart your Julia session you'll have to re-issue that command\nmodifying LOAD_PATH.\nIn step 2 below, edit MyPkg/src/MyPkg.jl to change the source code, and create any\ntest file of your choosing.\nDevelop your package\nBefore loading any code, make sure you're running Revise: say\nusing Revise or follow its documentation on configuring it to run\nautomatically.\nThen navigate to the directory containing your test file (here\nassumed to be \"runtests.jl\") and do the following:\njulia> using MyPkg\n\njulia> include(\"runtests.jl\")\nYou can iteratively modify the code in MyPkg in your editor and re-run the\ntests with include(\"runtests.jl\"). You generally should not need to restart\nyour Julia session to see the changes take effect (subject to a few limitations)."},{"title":"Complex and Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html#Complex-and-Rational-Numbers","category":"section","text":"Julia includes predefined types for both complex and rational numbers, and supports\nall the standard Mathematical Operations and Elementary Functions on them. Conversion and Promotion are defined\nso that operations on any combination of predefined numeric types, whether primitive or composite,\nbehave as expected."},{"title":"Complex Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html#Complex-Numbers","category":"section","text":"The global constant im is bound to the complex number i, representing the principal\nsquare root of -1. (Using mathematicians' i or engineers' j for this global constant was rejected since they are such popular index variable names.) Since Julia allows numeric literals to be juxtaposed with identifiers as coefficients,\nthis binding suffices to provide convenient syntax for complex numbers, similar to the traditional\nmathematical notation:\n\njulia> 1+2im\n1 + 2im\n\nYou can perform all the standard arithmetic operations with complex numbers:\n\njulia> (1 + 2im)*(2 - 3im)\n8 + 1im\n\njulia> (1 + 2im)/(1 - 2im)\n-0.6 + 0.8im\n\njulia> (1 + 2im) + (1 - 2im)\n2 + 0im\n\njulia> (-3 + 2im) - (5 - 1im)\n-8 + 3im\n\njulia> (-1 + 2im)^2\n-3 - 4im\n\njulia> (-1 + 2im)^2.5\n2.729624464784009 - 6.9606644595719im\n\njulia> (-1 + 2im)^(1 + 1im)\n-0.27910381075826657 + 0.08708053414102428im\n\njulia> 3(2 - 5im)\n6 - 15im\n\njulia> 3(2 - 5im)^2\n-63 - 60im\n\njulia> 3(2 - 5im)^-1.0\n0.20689655172413793 + 0.5172413793103449im\n\nThe promotion mechanism ensures that combinations of operands of different types just work:\n\njulia> 2(1 - 1im)\n2 - 2im\n\njulia> (2 + 3im) - 1\n1 + 3im\n\njulia> (1 + 2im) + 0.5\n1.5 + 2.0im\n\njulia> (2 + 3im) - 0.5im\n2.0 + 2.5im\n\njulia> 0.75(1 + 2im)\n0.75 + 1.5im\n\njulia> (2 + 3im) / 2\n1.0 + 1.5im\n\njulia> (1 - 3im) / (2 + 2im)\n-0.5 - 1.0im\n\njulia> 2im^2\n-2 + 0im\n\njulia> 1 + 3/4im\n1.0 - 0.75im\n\nNote that 3/4im == 3/(4*im) == -(3/4*im), since a literal coefficient binds more tightly than\ndivision.\n\nStandard functions to manipulate complex values are provided:\n\njulia> z = 1 + 2im\n1 + 2im\n\njulia> real(1 + 2im) # real part of z\n1\n\njulia> imag(1 + 2im) # imaginary part of z\n2\n\njulia> conj(1 + 2im) # complex conjugate of z\n1 - 2im\n\njulia> abs(1 + 2im) # absolute value of z\n2.23606797749979\n\njulia> abs2(1 + 2im) # squared absolute value\n5\n\njulia> angle(1 + 2im) # phase angle in radians\n1.1071487177940904\n\nAs usual, the absolute value (abs) of a complex number is its distance from zero.\nabs2 gives the square of the absolute value, and is of particular use for complex\nnumbers since it avoids taking a square root. angle returns the phase angle in radians\n(also known as the argument or arg function). The full gamut of other Elementary Functions\nis also defined for complex numbers:\n\njulia> sqrt(1im)\n0.7071067811865476 + 0.7071067811865475im\n\njulia> sqrt(1 + 2im)\n1.272019649514069 + 0.7861513777574233im\n\njulia> cos(1 + 2im)\n2.0327230070196656 - 3.0518977991517997im\n\njulia> exp(1 + 2im)\n-1.1312043837568135 + 2.4717266720048188im\n\njulia> sinh(1 + 2im)\n-0.4890562590412937 + 1.4031192506220405im\n\nNote that mathematical functions typically return real values when applied to real numbers and\ncomplex values when applied to complex numbers. For example, sqrt behaves differently\nwhen applied to -1 versus -1 + 0im even though -1 == -1 + 0im:\n\njulia> sqrt(-1)\nERROR: DomainError with -1.0:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\njulia> sqrt(-1 + 0im)\n0.0 + 1.0im\n\nThe literal numeric coefficient notation does not work when constructing a complex number\nfrom variables. Instead, the multiplication must be explicitly written out:\n\njulia> a = 1; b = 2; a + b*im\n1 + 2im\n\nHowever, this is not recommended. Instead, use the more efficient complex function to construct\na complex value directly from its real and imaginary parts:\n\njulia> a = 1; b = 2; complex(a, b)\n1 + 2im\n\nThis construction avoids the multiplication and addition operations.\n\nInf and NaN propagate through complex numbers in the real and imaginary parts\nof a complex number as described in the Special floating-point values section:\n\njulia> 1 + Inf*im\n1.0 + Inf*im\n\njulia> 1 + NaN*im\n1.0 + NaN*im"},{"title":"Rational Numbers","page":"Complex and Rational Numbers","location":"manual/complex-and-rational-numbers.html#Rational-Numbers","category":"section","text":"Julia has a rational number type to represent exact ratios of integers. Rationals are constructed\nusing the // operator:\n\njulia> 2//3\n2//3\n\nIf the numerator and denominator of a rational have common factors, they are reduced to lowest\nterms such that the denominator is non-negative:\n\njulia> 6//9\n2//3\n\njulia> -4//8\n-1//2\n\njulia> 5//-15\n-1//3\n\njulia> -4//-12\n1//3\n\nThis normalized form for a ratio of integers is unique, so equality of rational values can be\ntested by checking for equality of the numerator and denominator. The standardized numerator and\ndenominator of a rational value can be extracted using the numerator and denominator\nfunctions:\n\njulia> numerator(2//3)\n2\n\njulia> denominator(2//3)\n3\n\nDirect comparison of the numerator and denominator is generally not necessary, since the standard\narithmetic and comparison operations are defined for rational values:\n\njulia> 2//3 == 6//9\ntrue\n\njulia> 2//3 == 9//27\nfalse\n\njulia> 3//7 < 1//2\ntrue\n\njulia> 3//4 > 2//3\ntrue\n\njulia> 2//4 + 1//6\n2//3\n\njulia> 5//12 - 1//4\n1//6\n\njulia> 5//8 * 3//12\n5//32\n\njulia> 6//5 / 10//7\n21//25\n\nRationals can easily be converted to floating-point numbers:\n\njulia> float(3//4)\n0.75\n\nConversion from rational to floating-point respects the following identity for any integral values\nof a and b, except when a==0 && b <= 0:\n\njulia> a = 1; b = 2;\n\njulia> isequal(float(a//b), a/b)\ntrue\n\njulia> a, b = 0, 0\n(0, 0)\n\njulia> float(a//b)\nERROR: ArgumentError: invalid rational: zero(Int64)//zero(Int64)\nStacktrace:\n[...]\n\njulia> a/b\nNaN\n\njulia> a, b = 0, -1\n(0, -1)\n\njulia> float(a//b), a/b\n(0.0, -0.0)\n\nConstructing infinite rational values is acceptable:\n\njulia> 5//0\n1//0\n\njulia> x = -3//0\n-1//0\n\njulia> typeof(x)\nRational{Int64}\n\nTrying to construct a NaN rational value, however, is invalid:\n\njulia> 0//0\nERROR: ArgumentError: invalid rational: zero(Int64)//zero(Int64)\nStacktrace:\n[...]\n\nAs usual, the promotion system makes interactions with other numeric types effortless:\n\njulia> 3//5 + 1\n8//5\n\njulia> 3//5 - 0.5\n0.09999999999999998\n\njulia> 2//7 * (1 + 2im)\n2//7 + 4//7*im\n\njulia> 2//7 * (1.5 + 2im)\n0.42857142857142855 + 0.5714285714285714im\n\njulia> 3//2 / (1 + 2im)\n3//10 - 3//5*im\n\njulia> 1//2 + 2im\n1//2 + 2//1*im\n\njulia> 1 + 2//3im\n1//1 - 2//3*im\n\njulia> 0.5 == 1//2\ntrue\n\njulia> 0.33 == 1//3\nfalse\n\njulia> 0.33 < 1//3\ntrue\n\njulia> 1//3 - 0.33\n0.0033333333333332993"},{"title":"SHA","page":"SHA","location":"stdlib/SHA.html#SHA","category":"section","text":""},{"title":"SHA functions","page":"SHA","location":"stdlib/SHA.html#SHA-functions","category":"section","text":"Usage is very straightforward:\n\njulia> using SHA\n\njulia> bytes2hex(sha256(\"test\"))\n\"9f86d081884c7d659a2feaa0c55ad015a3bf4f1b2b0b822cd15d6c15b0f00a08\"\n\nEach exported function (at the time of this writing, SHA-1, SHA-2 224, 256, 384 and 512, and SHA-3 224, 256, 384 and 512 functions are implemented) takes in either an AbstractVector{UInt8}, an AbstractString or an IO object.  This makes it trivial to checksum a file:\n\nshell> cat /tmp/test.txt\ntest\njulia> using SHA\n\njulia> open(\"/tmp/test.txt\") do f\n           sha2_256(f)\n       end\n32-element Vector{UInt8}:\n 0x9f\n 0x86\n 0xd0\n 0x81\n 0x88\n 0x4c\n 0x7d\n 0x65\n    ⋮\n 0x5d\n 0x6c\n 0x15\n 0xb0\n 0xf0\n 0x0a\n 0x08"},{"title":"All SHA functions","page":"SHA","location":"stdlib/SHA.html#All-SHA-functions","category":"section","text":"Due to the colloquial usage of sha256 to refer to sha2_256, convenience functions are provided, mapping shaxxx() function calls to sha2_xxx().\nFor SHA-3, no such colloquialisms exist and the user must use the full sha3_xxx() names.\n\nshaxxx() takes AbstractString and array-like objects (NTuple and Vector) with elements of type UInt8.\n\nSHA-1\n\nSHA-2\n\nSHA-3"},{"title":"Working with context","page":"SHA","location":"stdlib/SHA.html#Working-with-context","category":"section","text":"To create a hash from multiple items the SHAX_XXX_CTX() types can be used to create a stateful hash object that\nis updated with update! and finalized with digest!\n\njulia> using SHA\n\njulia> ctx = SHA2_256_CTX()\nSHA2 256-bit hash state\n\njulia> update!(ctx, b\"some data\")\n0x0000000000000009\n\njulia> update!(ctx, b\"some more data\")\n0x0000000000000017\n\njulia> digest!(ctx)\n32-element Vector{UInt8}:\n 0xbe\n 0xcf\n 0x23\n 0xda\n 0xaf\n 0x02\n 0xf7\n 0xa3\n 0x57\n 0x92\n    ⋮\n 0x89\n 0x4f\n 0x59\n 0xd8\n 0xb3\n 0xb4\n 0x81\n 0x8b\n 0xc5\n\nNote that, at the time of this writing, the SHA3 code is not optimized, and as such is roughly an order of magnitude slower than SHA2."},{"title":"All SHA context types","page":"SHA","location":"stdlib/SHA.html#All-SHA-context-types","category":"section","text":"SHA-1\n\nSHA-2\n\nConvenience types are also provided, where SHAXXX_CTX is a type alias for SHA2_XXX_CTX.\n\nSHA-3"},{"title":"HMAC functions","page":"SHA","location":"stdlib/SHA.html#HMAC-functions","category":"section","text":"julia> using SHA\n\njulia> key = collect(codeunits(\"key_string\"))\n10-element Vector{UInt8}:\n 0x6b\n 0x65\n 0x79\n 0x5f\n 0x73\n 0x74\n 0x72\n 0x69\n 0x6e\n 0x67\n\njulia> bytes2hex(hmac_sha3_256(key, \"test-message\"))\n\"bc49a6f2aa29b27ee5ed1e944edd7f3d153e8a01535d98b5e24dac9a589a6248\"\n\nTo create a hash from multiple items, the HMAC_CTX() types can be used to create a stateful hash object that\nis updated with update! and finalized with digest!.\n\njulia> using SHA\n\njulia> key = collect(codeunits(\"key_string\"))\n10-element Vector{UInt8}:\n 0x6b\n 0x65\n 0x79\n 0x5f\n 0x73\n 0x74\n 0x72\n 0x69\n 0x6e\n 0x67\n\njulia> ctx = HMAC_CTX(SHA3_256_CTX(), key);\n\njulia> update!(ctx, b\"test-\")\n0x0000000000000000000000000000008d\n\njulia> update!(ctx, b\"message\")\n0x00000000000000000000000000000094\n\njulia> bytes2hex(digest!(ctx))\n\"bc49a6f2aa29b27ee5ed1e944edd7f3d153e8a01535d98b5e24dac9a589a6248\""},{"title":"All HMAC functions","page":"SHA","location":"stdlib/SHA.html#All-HMAC-functions","category":"section","text":"HMAC context type\n\nSHA-1\n\nSHA-2\n\nSHA-3"},{"title":"SHA.sha1","page":"SHA","location":"stdlib/SHA.html#SHA.sha1","category":"function","text":"sha1(data)\n\nHash data using the sha1 algorithm and return the resulting digest.\nSee also SHA1_CTX.\n\n\n\n\n\nsha1(io::IO)\n\nHash data from io using sha1 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha224","page":"SHA","location":"stdlib/SHA.html#SHA.sha224","category":"function","text":"sha224(data)\n\nHash data using the sha224 algorithm and return the resulting digest.\nSee also SHA2_224_CTX.\n\n\n\n\n\nsha224(io::IO)\n\nHash data from io using sha224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha256","page":"SHA","location":"stdlib/SHA.html#SHA.sha256","category":"function","text":"sha256(data)\n\nHash data using the sha256 algorithm and return the resulting digest.\nSee also SHA2_256_CTX.\n\n\n\n\n\nsha256(io::IO)\n\nHash data from io using sha256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha384","page":"SHA","location":"stdlib/SHA.html#SHA.sha384","category":"function","text":"sha384(data)\n\nHash data using the sha384 algorithm and return the resulting digest.\nSee also SHA2_384_CTX.\n\n\n\n\n\nsha384(io::IO)\n\nHash data from io using sha384 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha512","page":"SHA","location":"stdlib/SHA.html#SHA.sha512","category":"function","text":"sha512(data)\n\nHash data using the sha512 algorithm and return the resulting digest.\nSee also SHA2_512_CTX.\n\n\n\n\n\nsha512(io::IO)\n\nHash data from io using sha512 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha2_224","page":"SHA","location":"stdlib/SHA.html#SHA.sha2_224","category":"function","text":"sha2_224(data)\n\nHash data using the sha2_224 algorithm and return the resulting digest.\nSee also SHA2_224_CTX.\n\n\n\n\n\nsha2_224(io::IO)\n\nHash data from io using sha2_224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha2_256","page":"SHA","location":"stdlib/SHA.html#SHA.sha2_256","category":"function","text":"sha2_256(data)\n\nHash data using the sha2_256 algorithm and return the resulting digest.\nSee also SHA2_256_CTX.\n\n\n\n\n\nsha2_256(io::IO)\n\nHash data from io using sha2_256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha2_384","page":"SHA","location":"stdlib/SHA.html#SHA.sha2_384","category":"function","text":"sha2_384(data)\n\nHash data using the sha2_384 algorithm and return the resulting digest.\nSee also SHA2_384_CTX.\n\n\n\n\n\nsha2_384(io::IO)\n\nHash data from io using sha2_384 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha2_512","page":"SHA","location":"stdlib/SHA.html#SHA.sha2_512","category":"function","text":"sha2_512(data)\n\nHash data using the sha2_512 algorithm and return the resulting digest.\nSee also SHA2_512_CTX.\n\n\n\n\n\nsha2_512(io::IO)\n\nHash data from io using sha2_512 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha2_512_224","page":"SHA","location":"stdlib/SHA.html#SHA.sha2_512_224","category":"function","text":"sha2_512_224(data)\n\nHash data using the sha2_512_224 algorithm and return the resulting digest.\nSee also SHA2_512_224_CTX.\n\n\n\n\n\nsha2_512_224(io::IO)\n\nHash data from io using sha2_512_224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha2_512_256","page":"SHA","location":"stdlib/SHA.html#SHA.sha2_512_256","category":"function","text":"sha2_512_256(data)\n\nHash data using the sha2_512_256 algorithm and return the resulting digest.\nSee also SHA2_512_256_CTX.\n\n\n\n\n\nsha2_512_256(io::IO)\n\nHash data from io using sha2_512_256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha3_224","page":"SHA","location":"stdlib/SHA.html#SHA.sha3_224","category":"function","text":"sha3_224(data)\n\nHash data using the sha3_224 algorithm and return the resulting digest.\nSee also SHA3_224_CTX.\n\n\n\n\n\nsha3_224(io::IO)\n\nHash data from io using sha3_224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha3_256","page":"SHA","location":"stdlib/SHA.html#SHA.sha3_256","category":"function","text":"sha3_256(data)\n\nHash data using the sha3_256 algorithm and return the resulting digest.\nSee also SHA3_256_CTX.\n\n\n\n\n\nsha3_256(io::IO)\n\nHash data from io using sha3_256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha3_384","page":"SHA","location":"stdlib/SHA.html#SHA.sha3_384","category":"function","text":"sha3_384(data)\n\nHash data using the sha3_384 algorithm and return the resulting digest.\nSee also SHA3_384_CTX.\n\n\n\n\n\nsha3_384(io::IO)\n\nHash data from io using sha3_384 algorithm.\n\n\n\n\n\n"},{"title":"SHA.sha3_512","page":"SHA","location":"stdlib/SHA.html#SHA.sha3_512","category":"function","text":"sha3_512(data)\n\nHash data using the sha3_512 algorithm and return the resulting digest.\nSee also SHA3_512_CTX.\n\n\n\n\n\nsha3_512(io::IO)\n\nHash data from io using sha3_512 algorithm.\n\n\n\n\n\n"},{"title":"SHA.update!","page":"SHA","location":"stdlib/SHA.html#SHA.update!","category":"function","text":"update!(context, data[, datalen])\n\nUpdate the SHA context with the bytes in data. See also digest! for\nfinalizing the hash.\n\nExamples\n\njulia> ctx = SHA1_CTX()\nSHA1 hash state\n\njulia> update!(ctx, b\"data to to be hashed\")\n\n\n\n\n\n"},{"title":"SHA.digest!","page":"SHA","location":"stdlib/SHA.html#SHA.digest!","category":"function","text":"digest!(context)\n\nFinalize the SHA context and return the hash as array of bytes (Vector{Uint8}).\nUpdating the context after calling digest! on it will error.\n\nExamples\n\njulia> ctx = SHA1_CTX()\nSHA1 hash state\n\njulia> update!(ctx, b\"data to to be hashed\")\n\njulia> digest!(ctx)\n20-element Vector{UInt8}:\n 0x83\n 0xe4\n ⋮\n 0x89\n 0xf5\n\njulia> update!(ctx, b\"more data\")\nERROR: Cannot update CTX after `digest!` has been called on it\n[...]\n\n\n\n\n\n"},{"title":"SHA.SHA1_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA1_CTX","category":"type","text":"SHA1_CTX()\n\nConstruct an empty SHA1 context.\n\n\n\n\n\n"},{"title":"SHA.SHA224_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA224_CTX","category":"type","text":"SHA2_224_CTX()\n\nConstruct an empty SHA2_224 context.\n\n\n\n\n\n"},{"title":"SHA.SHA256_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA256_CTX","category":"type","text":"SHA2_256_CTX()\n\nConstruct an empty SHA2_256 context.\n\n\n\n\n\n"},{"title":"SHA.SHA384_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA384_CTX","category":"type","text":"SHA2_384()\n\nConstruct an empty SHA2_384 context.\n\n\n\n\n\n"},{"title":"SHA.SHA512_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA512_CTX","category":"type","text":"SHA2_512_CTX()\n\nConstruct an empty SHA2_512 context.\n\n\n\n\n\n"},{"title":"SHA.SHA2_224_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA2_224_CTX","category":"type","text":"SHA2_224_CTX()\n\nConstruct an empty SHA2_224 context.\n\n\n\n\n\n"},{"title":"SHA.SHA2_256_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA2_256_CTX","category":"type","text":"SHA2_256_CTX()\n\nConstruct an empty SHA2_256 context.\n\n\n\n\n\n"},{"title":"SHA.SHA2_384_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA2_384_CTX","category":"type","text":"SHA2_384()\n\nConstruct an empty SHA2_384 context.\n\n\n\n\n\n"},{"title":"SHA.SHA2_512_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA2_512_CTX","category":"type","text":"SHA2_512_CTX()\n\nConstruct an empty SHA2_512 context.\n\n\n\n\n\n"},{"title":"SHA.SHA2_512_224_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA2_512_224_CTX","category":"type","text":"SHA2_512_224_CTX()\n\nConstruct an empty SHA2_512/224 context and set the initial hash value.\n\nFor the source of the initial value,\nrefer to FIPS 180-4, 5.3.6.1 SHA-512/224\n\n\n\n\n\n"},{"title":"SHA.SHA2_512_256_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA2_512_256_CTX","category":"type","text":"SHA2_512_256_CTX()\n\nConstruct an empty SHA2_512/256 context and set the initial hash value.\n\nFor the source of the initial value,\nrefer to FIPS 180-4, 5.3.6.2 SHA-512/256\n\n\n\n\n\n"},{"title":"SHA.SHA3_224_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA3_224_CTX","category":"type","text":"SHA3_224_CTX()\n\nConstruct an empty SHA3_224 context.\n\n\n\n\n\n"},{"title":"SHA.SHA3_256_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA3_256_CTX","category":"type","text":"SHA3_256_CTX()\n\nConstruct an empty SHA3_256 context.\n\n\n\n\n\n"},{"title":"SHA.SHA3_384_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA3_384_CTX","category":"type","text":"SHA3_384_CTX()\n\nConstruct an empty SHA3_384 context.\n\n\n\n\n\n"},{"title":"SHA.SHA3_512_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.SHA3_512_CTX","category":"type","text":"SHA3_512_CTX()\n\nConstruct an empty SHA3_512 context.\n\n\n\n\n\n"},{"title":"SHA.HMAC_CTX","page":"SHA","location":"stdlib/SHA.html#SHA.HMAC_CTX","category":"type","text":"HMAC_CTX(ctx::CTX, key::Vector{UInt8}) where {CTX<:SHA_CTX}\n\nConstruct an empty HMAC_CTX context.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha1","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha1","category":"function","text":"hmac_sha1(key, data)\n\nHash data using the sha1 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha1(key, io::IO)\n\nHash data from io with the passed key using sha1 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha224","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha224","category":"function","text":"hmac_sha224(key, data)\n\nHash data using the sha224 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha224(key, io::IO)\n\nHash data from io with the passed key using sha224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha256","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha256","category":"function","text":"hmac_sha256(key, data)\n\nHash data using the sha256 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha256(key, io::IO)\n\nHash data from io with the passed key using sha256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha384","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha384","category":"function","text":"hmac_sha384(key, data)\n\nHash data using the sha384 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha384(key, io::IO)\n\nHash data from io with the passed key using sha384 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha512","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha512","category":"function","text":"hmac_sha512(key, data)\n\nHash data using the sha512 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha512(key, io::IO)\n\nHash data from io with the passed key using sha512 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha2_224","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha2_224","category":"function","text":"hmac_sha2_224(key, data)\n\nHash data using the sha2_224 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_224(key, io::IO)\n\nHash data from io with the passed key using sha2_224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha2_256","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha2_256","category":"function","text":"hmac_sha2_256(key, data)\n\nHash data using the sha2_256 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_256(key, io::IO)\n\nHash data from io with the passed key using sha2_256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha2_384","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha2_384","category":"function","text":"hmac_sha2_384(key, data)\n\nHash data using the sha2_384 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_384(key, io::IO)\n\nHash data from io with the passed key using sha2_384 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha2_512","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha2_512","category":"function","text":"hmac_sha2_512(key, data)\n\nHash data using the sha2_512 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_512(key, io::IO)\n\nHash data from io with the passed key using sha2_512 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha3_224","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha3_224","category":"function","text":"hmac_sha3_224(key, data)\n\nHash data using the sha3_224 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_224(key, io::IO)\n\nHash data from io with the passed key using sha3_224 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha3_256","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha3_256","category":"function","text":"hmac_sha3_256(key, data)\n\nHash data using the sha3_256 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_256(key, io::IO)\n\nHash data from io with the passed key using sha3_256 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha3_384","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha3_384","category":"function","text":"hmac_sha3_384(key, data)\n\nHash data using the sha3_384 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_384(key, io::IO)\n\nHash data from io with the passed key using sha3_384 algorithm.\n\n\n\n\n\n"},{"title":"SHA.hmac_sha3_512","page":"SHA","location":"stdlib/SHA.html#SHA.hmac_sha3_512","category":"function","text":"hmac_sha3_512(key, data)\n\nHash data using the sha3_512 algorithm using the passed key.\nSee also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_512(key, io::IO)\n\nHash data from io with the passed key using sha3_512 algorithm.\n\n\n\n\n\n"},{"title":"LibCURL","page":"LibCURL","location":"stdlib/LibCURL.html#LibCURL","category":"section","text":"This is a simple Julia wrapper around http://curl.haxx.se/libcurl/ generated using Clang.jl. \nPlease see the libcurl API documentation for help on how to use this package."},{"title":"JIT Design and Implementation","page":"JIT Design and Implementation","location":"devdocs/jit.html#JIT-Design-and-Implementation","category":"section","text":"This document explains the design and implementation of Julia's JIT, after codegen has finished and unoptimized LLVM IR has been produced. The JIT is responsible for optimizing and compiling this IR to machine code, and for linking it into the current process and making the code available for execution."},{"title":"Introduction","page":"JIT Design and Implementation","location":"devdocs/jit.html#Introduction","category":"section","text":"The JIT is responsible for managing compilation resources, looking up previously compiled code, and compiling new code. It is primarily built on LLVM's On-Request-Compilation (ORCv2) technology, which provides support for a number of useful features such as concurrent compilation, lazy compilation, and the ability to compile code in a separate process. Though LLVM provides a basic JIT compiler in the form of LLJIT, Julia uses many ORCv2 APIs directly to create its own custom JIT compiler."},{"title":"Overview","page":"JIT Design and Implementation","location":"devdocs/jit.html#Overview","category":"section","text":"(Image: Diagram of the compiler flow)\n\nCodegen produces an LLVM module containing IR for one or more Julia functions from the original Julia SSA IR produced by type inference (labeled as translate on the compiler diagram above). It also produces a mapping of code-instance to LLVM function name. However, though some optimizations have been applied by the Julia-based compiler on Julia IR, the LLVM IR produced by codegen still contains many opportunities for optimization. Thus, the first step the JIT takes is to run a target-independent optimization pipeline[tdp] on the LLVM module. Then, the JIT runs a target-dependent optimization pipeline, which includes target-specific optimizations and code generation, and outputs an object file. Finally, the JIT links the resulting object file into the current process and makes the code available for execution. All of this is controlled by code in src/jitlayers.cpp.\n\n[tdp]: This is not a totally-target independent pipeline, as transformations such as vectorization rely upon target information such as vector register width and cost modeling. Additionally, codegen itself makes a few target-dependent assumptions, and the optimization pipeline will take advantage of that knowledge.\n\nCurrently, only one thread at a time is permitted to enter the optimize-compile-link pipeline at a time, due to restrictions imposed by one of our linkers (RuntimeDyld). However, the JIT is designed to support concurrent optimization and compilation, and the linker restriction is expected to be lifted in the future when RuntimeDyld has been fully superseded on all platforms."},{"title":"Optimization Pipeline","page":"JIT Design and Implementation","location":"devdocs/jit.html#Optimization-Pipeline","category":"section","text":"The optimization pipeline is based off LLVM's new pass manager, but the pipeline is customized for Julia's needs. The pipeline is defined in src/pipeline.cpp, and broadly proceeds through a number of stages as detailed below.\n\nEarly Simplification\nThese passes are mainly used to simplify the IR and canonicalize patterns so that later passes can identify those patterns more easily. Additionally, various intrinsic calls such as branch prediction hints and annotations are lowered into other metadata or other IR features. SimplifyCFG (simplify control flow graph), DCE (dead code elimination), and SROA (scalar replacement of aggregates) are some of the key players here.\nEarly Optimization\nThese passes are typically cheap and are primarily focused around reducing the number of instructions in the IR and propagating knowledge to other instructions. For example, EarlyCSE is used to perform common subexpression elimination, and InstCombine and InstSimplify perform a number of small peephole optimizations to make operations less expensive.\nLoop Optimization\nThese passes canonicalize and simplify loops. Loops are often hot code, which makes loop optimization extremely important for performance. Key players here include LoopRotate, LICM, and LoopFullUnroll. Some bounds check elimination also happens here, as a result of the IRCE pass which can prove certain bounds are never exceeded.\nScalar Optimization\nThe scalar optimization pipeline contains a number of more expensive, but more powerful passes such as GVN (global value numbering), SCCP (sparse conditional constant propagation), and another round of bounds check elimination. These passes are expensive, but they can often remove large amounts of code and make vectorization much more successful and effective. Several other simplification and optimization passes intersperse the more expensive ones to reduce the amount of work they have to do.\nVectorization\nAutomatic vectorization is an extremely powerful transformation for CPU-intensive code. Briefly, vectorization allows execution of a single instruction on multiple data (SIMD), e.g. performing 8 addition operations at the same time. However, proving code to be both capable of vectorization and profitable to vectorize is difficult, and this relies heavily on the prior optimization passes to massage the IR into a state where vectorization is worth it.\nIntrinsic Lowering\nJulia inserts a number of custom intrinsics, for reasons such as object allocation, garbage collection, and exception handling. These intrinsics were originally placed to make optimization opportunities more obvious, but they are now lowered into LLVM IR to enable the IR to be emitted as machine code.\nCleanup\nThese passes are last-chance optimizations, and perform small optimizations such as fused multiply-add propagation and division-remainder simplification. Additionally, targets that do not support half-precision floating point numbers will have their half-precision instructions lowered into single-precision instructions here, and passes are added to provide sanitizer support."},{"title":"Target-Dependent Optimization and Code Generation","page":"JIT Design and Implementation","location":"devdocs/jit.html#Target-Dependent-Optimization-and-Code-Generation","category":"section","text":"LLVM provides target-dependent optimization and machine code generation in the same pipeline, located in the TargetMachine for a given platform. These passes include instruction selection, instruction scheduling, register allocation, and machine code emission. The LLVM documentation provides a good overview of the process, and the LLVM source code is the best place to look for details on the pipeline and passes."},{"title":"Linking","page":"JIT Design and Implementation","location":"devdocs/jit.html#Linking","category":"section","text":"Currently, Julia is transitioning between two linkers: the older RuntimeDyld linker, and the newer JITLink linker. JITLink contains a number of features that RuntimeDyld does not have, such as concurrent and reentrant linking, but currently lacks good support for profiling integrations and does not yet support all of the platforms that RuntimeDyld supports. Over time, JITLink is expected to replace RuntimeDyld entirely. Further details on JITLink can be found in the LLVM documentation."},{"title":"Execution","page":"JIT Design and Implementation","location":"devdocs/jit.html#Execution","category":"section","text":"Once the code has been linked into the current process, it is available for execution. This fact is made known to the generating codeinst by updating the invoke, specsigflags, and specptr fields appropriately. Codeinsts support upgrading invoke, specsigflags, and specptr fields, so long as every combination of these fields that exists at any given point in time is valid to be called. This allows the JIT to update these fields without invalidating existing codeinsts, supporting a potential future concurrent JIT. Specifically, the following states may be valid:\n\ninvoke is NULL, specsigflags is 0b00, specptr is NULL\nThis is the initial state of a codeinst, and indicates that the codeinst has not yet been compiled.\ninvoke is non-null, specsigflags is 0b00, specptr is NULL\nThis indicates that the codeinst was not compiled with any specialization, and that the codeinst should be invoked directly. Note that in this instance, invoke does not read either the specsigflags or specptr fields, and therefore they may be modified without invalidating the invoke pointer.\ninvoke is non-null, specsigflags is 0b10, specptr is non-null\nThis indicates that the codeinst was compiled, but a specialized function signature was deemed unnecessary by codegen.\ninvoke is non-null, specsigflags is 0b11, specptr is non-null\nThis indicates that the codeinst was compiled, and a specialized function signature was deemed necessary by codegen. The specptr field contains a pointer to the specialized function signature. The invoke pointer is permitted to read both specsigflags and specptr fields.\n\nIn addition, there are a number of different transitional states that occur during the update process. To account for these potential situations, the following write and read patterns should be used when dealing with these codeinst fields.\n\nWhen writing invoke, specsigflags, and specptr:\nPerform an atomic compare-exchange operation of specptr assuming the old value was NULL. This compare-exchange operation should have at least acquire-release ordering, to provide ordering guarantees of the remaining memory operations in the write.\nIf specptr was non-null, cease the write operation and wait for bit 0b10 of specsigflags to be written, then restart from step 1 if desired.\nWrite the new low bit of specsigflags to its final value. This may be a relaxed write.\nWrite the new invoke pointer to its final value. This must have at least a release memory ordering to synchronize with reads of invoke.\nSet the second bit of specsigflags to 1. This must be at least a release memory ordering to synchronize with reads of specsigflags. This step completes the write operation and announces to all other threads that all fields have been set.\nWhen reading all of invoke, specsigflags, and specptr:\nRead the specptr field with any memory ordering.\nRead the invoke field with at least an acquire memory ordering. This load will be referred to as initial_invoke.\nIf initial_invoke is NULL, the codeinst is not yet executable. invoke is NULL, specsigflags may be treated as 0b00, specptr may be treated as NULL.\nIf specptr is NULL, then the initial_invoke pointer must not be relying on specptr to guarantee correct execution. Therefore, invoke is non-null, specsigflags may be treated as 0b00, specptr may be treated as NULL.\nIf specptr is non-null, then initial_invoke might not be the final invoke field that uses specptr. This can occur if specptr has been written, but invoke has not yet been written. Therefore, spin on the second bit of specsigflags until it is set to 1 with at least acquire memory ordering.\nRe-read the invoke field with any memory ordering. This load will be referred to as final_invoke.\nRead the specsigflags field with any memory ordering.\ninvoke is final_invoke, specsigflags is the value read in step 7, specptr is the value read in step 3.\nWhen updating a specptr to a different but equivalent function pointer:\nPerform a release store of the new function pointer to specptr. Races here must be benign, as the old function pointer is required to still be valid, and any new ones are also required to be valid as well. Once a pointer has been written to specptr, it must always be callable whether or not it is later overwritten.\n\nCorrectly reading these fields is implemented in jl_read_codeinst_invoke.\n\nAlthough these write, read, and update steps are complicated, they ensure that the JIT can update codeinsts without invalidating existing codeinsts, and that the JIT can update codeinsts without invalidating existing invoke pointers. This allows the JIT to potentially reoptimize functions at higher optimization levels in the future, and also will allow the JIT to support concurrent compilation of functions in the future."},{"title":"Working with LLVM","page":"Working with LLVM","location":"devdocs/llvm.html#Working-with-LLVM","category":"section","text":"This is not a replacement for the LLVM documentation, but a collection of tips for working on\nLLVM for Julia."},{"title":"Overview of Julia to LLVM Interface","page":"Working with LLVM","location":"devdocs/llvm.html#Overview-of-Julia-to-LLVM-Interface","category":"section","text":"Julia dynamically links against LLVM by default. Build with USE_LLVM_SHLIB=0 to link statically.\n\nThe code for lowering Julia AST to LLVM IR or interpreting it directly is in directory src/.\n\nFile Description\naotcompile.cpp Compiler C-interface entry and object file emission\nbuiltins.c Builtin functions\nccall.cpp Lowering ccall\ncgutils.cpp Lowering utilities, notably for array and tuple accesses\ncodegen.cpp Top-level of code generation, pass list, lowering builtins\ndebuginfo.cpp Tracks debug information for JIT code\ndisasm.cpp Handles native object file and JIT code disassembly\ngf.c Generic functions\nintrinsics.cpp Lowering intrinsics\njitlayers.cpp JIT-specific code, ORC compilation layers/utilities\nllvm-alloc-helpers.cpp Julia-specific escape analysis\nllvm-alloc-opt.cpp Custom LLVM pass to demote heap allocations to the stack\nllvm-cpufeatures.cpp Custom LLVM pass to lower CPU-based functions (e.g. haveFMA)\nllvm-demote-float16.cpp Custom LLVM pass to lower 16b float ops to 32b float ops\nllvm-final-gc-lowering.cpp Custom LLVM pass to lower GC calls to their final form\nllvm-gc-invariant-verifier.cpp Custom LLVM pass to verify Julia GC invariants\nllvm-julia-licm.cpp Custom LLVM pass to hoist/sink Julia-specific intrinsics\nllvm-late-gc-lowering.cpp Custom LLVM pass to root GC-tracked values\nllvm-multiversioning.cpp Custom LLVM pass to generate sysimg code on multiple architectures\nllvm-propagate-addrspaces.cpp Custom LLVM pass to canonicalize addrspaces\nllvm-ptls.cpp Custom LLVM pass to lower TLS operations\nllvm-remove-addrspaces.cpp Custom LLVM pass to remove Julia addrspaces\nllvm-remove-ni.cpp Custom LLVM pass to remove Julia non-integral addrspaces\nllvm-simdloop.cpp Custom LLVM pass for @simd\npipeline.cpp New pass manager pipeline, pass pipeline parsing\nsys.c I/O and operating system utility functions\n\nSome of the .cpp files form a group that compile to a single object.\n\nThe difference between an intrinsic and a builtin is that a builtin is a first class function\nthat can be used like any other Julia function. An intrinsic can operate only on unboxed data,\nand therefore its arguments must be statically typed."},{"title":"Alias Analysis","page":"Working with LLVM","location":"devdocs/llvm.html#LLVM-Alias-Analysis","category":"section","text":"Julia currently uses LLVM's Type Based Alias Analysis.\nTo find the comments that document the inclusion relationships, look for static MDNode* in\nsrc/codegen.cpp.\n\nThe -O option enables LLVM's Basic Alias Analysis."},{"title":"Building Julia with a different version of LLVM","page":"Working with LLVM","location":"devdocs/llvm.html#Building-Julia-with-a-different-version-of-LLVM","category":"section","text":"The default version of LLVM is specified in deps/llvm.version. You can override it by creating\na file called Make.user in the top-level directory and adding a line to it such as:\n\nLLVM_VER = 13.0.0\n\nBesides the LLVM release numerals, you can also use DEPS_GIT = llvm in combination with\nUSE_BINARYBUILDER_LLVM = 0 to build against the latest development version of LLVM.\n\nYou can also specify to build a debug version of LLVM, by setting either LLVM_DEBUG = 1 or\nLLVM_DEBUG = Release in your Make.user file. The former will be a fully unoptimized build\nof LLVM and the latter will produce an optimized build of LLVM. Depending on your needs the\nlatter will suffice and it quite a bit faster. If you use LLVM_DEBUG = Release you will also\nwant to set LLVM_ASSERTIONS = 1 to enable diagnostics for different passes. Only LLVM_DEBUG = 1\nimplies that option by default."},{"title":"Passing options to LLVM","page":"Working with LLVM","location":"devdocs/llvm.html#Passing-options-to-LLVM","category":"section","text":"You can pass options to LLVM via the environment variable JULIA_LLVM_ARGS.\nHere are example settings using bash syntax:\n\nexport JULIA_LLVM_ARGS=-print-after-all dumps IR after each pass.\nexport JULIA_LLVM_ARGS=-debug-only=loop-vectorize dumps LLVM DEBUG(...) diagnostics for\nloop vectorizer. If you get warnings about \"Unknown command line argument\", rebuild LLVM with\nLLVM_ASSERTIONS = 1.\nexport JULIA_LLVM_ARGS=-help shows a list of available options. export JULIA_LLVM_ARGS=-help-hidden shows even more.\nexport JULIA_LLVM_ARGS=\"-fatal-warnings -print-options\" is an example how to use multiple options."},{"title":"Useful JULIA_LLVM_ARGS parameters","page":"Working with LLVM","location":"devdocs/llvm.html#Useful-JULIA_LLVM_ARGS-parameters","category":"section","text":"-print-after=PASS: prints the IR after any execution of PASS, useful for checking changes done by a pass.\n-print-before=PASS: prints the IR before any execution of PASS, useful for checking the input to a pass.\n-print-changed: prints the IR whenever a pass changes the IR, useful for narrowing down which passes are causing problems.\n-print-(before|after)=MARKER-PASS: the Julia pipeline ships with a number of marker passes in the pipeline, which can be used to identify where problems or optimizations are occurring. A marker pass is defined as a pass which appears once in the pipeline and performs no transformations on the IR, and is only useful for targeting print-before/print-after. Currently, the following marker passes exist in the pipeline:\nBeforeOptimization\nBeforeEarlySimplification\nAfterEarlySimplification\nBeforeEarlyOptimization\nAfterEarlyOptimization\nBeforeLoopOptimization\nBeforeLICM\nAfterLICM\nBeforeLoopSimplification\nAfterLoopSimplification\nAfterLoopOptimization\nBeforeScalarOptimization\nAfterScalarOptimization\nBeforeVectorization\nAfterVectorization\nBeforeIntrinsicLowering\nAfterIntrinsicLowering\nBeforeCleanup\nAfterCleanup\nAfterOptimization\n-time-passes: prints the time spent in each pass, useful for identifying which passes are taking a long time.\n-print-module-scope: used in conjunction with -print-(before|after), gets the entire module rather than the IR unit received by the pass\n-debug: prints out a lot of debugging information throughout LLVM\n-debug-only=NAME, prints out debugging statements from files with DEBUG_TYPE defined to NAME, useful for getting additional context about a problem"},{"title":"Debugging LLVM transformations in isolation","page":"Working with LLVM","location":"devdocs/llvm.html#Debugging-LLVM-transformations-in-isolation","category":"section","text":"On occasion, it can be useful to debug LLVM's transformations in isolation from\nthe rest of the Julia system, e.g. because reproducing the issue inside julia\nwould take too long, or because one wants to take advantage of LLVM's tooling\n(e.g. bugpoint).\n\nTo start with, you can install the developer tools to work with LLVM via:\n\nmake -C deps install-llvm-tools\n\nTo get unoptimized IR for the entire system image, pass the\n--output-unopt-bc unopt.bc option to the system image build process, which will\noutput the unoptimized IR to an unopt.bc file. This file can then be passed to\nLLVM tools as usual. libjulia can function as an LLVM pass plugin and can be\nloaded into LLVM tools, to make julia-specific passes available in this\nenvironment. In addition, it exposes the -julia meta-pass, which runs the\nentire Julia pass-pipeline over the IR. As an example, to generate a system\nimage with the old pass manager, one could do:\n\n\nllc -o sys.o opt.bc\ncc -shared -o sys.so sys.o\n\nTo generate a system image with the new pass manager, one could do:\n\n./usr/tools/opt -load-pass-plugin=libjulia-codegen.so --passes='julia' -o opt.bc unopt.bc\n./usr/tools/llc -o sys.o opt.bc\n./usr/tools/cc -shared -o sys.so sys.o\n\nThis system image can then be loaded by julia as usual.\n\nIt is also possible to dump an LLVM IR module for just one Julia function,\nusing:\n\nfun, T = +, Tuple{Int,Int} # Substitute your function of interest here\noptimize = false\nopen(\"plus.ll\", \"w\") do file\n    code_llvm(file, fun, T; raw=true, dump_module=true, optimize)\nend\n\nThese files can be processed the same way as the unoptimized sysimg IR shown\nabove, or if you want to see the LLVM IR yourself and get extra verification run, you can use\n\n./usr/tools/opt -load-pass-plugin=libjulia-codegen.so --passes='julia' -S -verify-each plus.ll\n\n(note on MacOS this would be libjulia-codegen.dylib and on Windows libjulia-codegen.dll)"},{"title":"Running the LLVM test suite","page":"Working with LLVM","location":"devdocs/llvm.html#Running-the-LLVM-test-suite","category":"section","text":"To run the llvm tests locally, you need to first install the tools, build julia, then you\ncan run the tests:\n\nmake -C deps install-llvm-tools\nmake -j julia-src-release\nmake -C test/llvmpasses\n\nIf you want to run the individual test files directly, via the commands at the top of each\ntest file, the first step here will have installed the tools into ./usr/tools/opt. Then\nyou'll want to manually replace %s with the name of the test file."},{"title":"Improving LLVM optimizations for Julia","page":"Working with LLVM","location":"devdocs/llvm.html#Improving-LLVM-optimizations-for-Julia","category":"section","text":"Improving LLVM code generation usually involves either changing Julia lowering to be more friendly\nto LLVM's passes, or improving a pass.\n\nIf you are planning to improve a pass, be sure to read the LLVM developer policy.\nThe best strategy is to create a code example in a form where you can use LLVM's opt tool to\nstudy it and the pass of interest in isolation.\n\nCreate an example Julia code of interest.\nUse JULIA_LLVM_ARGS=-print-after-all to dump the IR.\nPick out the IR at the point just before the pass of interest runs.\nStrip the debug metadata and fix up the TBAA metadata by hand.\n\nThe last step is labor intensive. Suggestions on a better way would be appreciated."},{"title":"The jlcall calling convention","page":"Working with LLVM","location":"devdocs/llvm.html#The-jlcall-calling-convention","category":"section","text":"Julia has a generic calling convention for unoptimized code, which looks somewhat\nas follows:\n\njl_value_t *any_unoptimized_call(jl_value_t *, jl_value_t **, int);\n\nwhere the first argument is the boxed function object, the second argument is\nan on-stack array of arguments and the third is the number of arguments. Now,\nwe could perform a straightforward lowering and emit an alloca for the argument\narray. However, this would betray the SSA nature of the uses at the call site,\nmaking optimizations (including GC root placement), significantly harder.\nInstead, we emit it as follows:\n\ncall %jl_value_t *@julia.call(jl_value_t *(*)(...) @any_unoptimized_call, %jl_value_t *%arg1, %jl_value_t *%arg2)\n\nThis allows us to retain the SSA-ness of the\nuses throughout the optimizer. GC root placement will later lower this call to\nthe original C ABI."},{"title":"GC root placement","page":"Working with LLVM","location":"devdocs/llvm.html#GC-root-placement","category":"section","text":"GC root placement is done by an LLVM pass late in the pass pipeline. Doing GC root\nplacement this late enables LLVM to make more aggressive optimizations around\ncode that requires GC roots, as well as allowing us to reduce the number of\nrequired GC roots and GC root store operations (since LLVM doesn't understand\nour GC, it wouldn't otherwise know what it is and is not allowed to do with\nvalues stored to the GC frame, so it'll conservatively do very little). As an\nexample, consider an error path\n\nif some_condition()\n    #= Use some variables maybe =#\n    error(\"An error occurred\")\nend\n\nDuring constant folding, LLVM may discover that the condition is always false,\nand can remove the basic block. However, if GC root lowering is done early,\nthe GC root slots used in the deleted block, as well as any values kept alive\nin those slots only because they were used in the error path, would be kept\nalive by LLVM. By doing GC root lowering late, we give LLVM the license to do\nany of its usual optimizations (constant folding, dead code elimination, etc.),\nwithout having to worry (too much) about which values may or may not be GC\ntracked.\n\nHowever, in order to be able to do late GC root placement, we need to be able to\nidentify a) which pointers are GC tracked and b) all uses of such pointers. The\ngoal of the GC placement pass is thus simple:\n\nMinimize the number of needed GC roots/stores to them subject to the constraint\nthat at every safepoint, any live GC-tracked pointer (i.e. for which there is\na path after this point that contains a use of this pointer) is in some GC slot."},{"title":"Representation","page":"Working with LLVM","location":"devdocs/llvm.html#Representation","category":"section","text":"The primary difficulty is thus choosing an IR representation that allows us to\nidentify GC-tracked pointers and their uses, even after the program has been\nrun through the optimizer. Our design makes use of three LLVM features to achieve\nthis:\n\nCustom address spaces\nOperand Bundles\nNon-integral pointers\n\nCustom address spaces allow us to tag every point with an integer that needs\nto be preserved through optimizations. The compiler may not insert casts between\naddress spaces that did not exist in the original program and it must never\nchange the address space of a pointer on a load/store/etc operation. This allows\nus to annotate which pointers are GC-tracked in an optimizer-resistant way. Note\nthat metadata would not be able to achieve the same purpose. Metadata is supposed\nto always be discardable without altering the semantics of the program. However,\nfailing to identify a GC-tracked pointer alters the resulting program behavior\ndramatically - it'll probably crash or return wrong results. We currently use\nthree different address spaces (their numbers are defined in src/codegen_shared.cpp):\n\nGC Tracked Pointers (currently 10): These are pointers to boxed values that may be put\ninto a GC frame. It is loosely equivalent to a jl_value_t* pointer on the C\nside. N.B. It is illegal to ever have a pointer in this address space that may\nnot be stored to a GC slot.\nDerived Pointers (currently 11): These are pointers that are derived from some GC\ntracked pointer. Uses of these pointers generate uses of the original pointer.\nHowever, they need not themselves be known to the GC. The GC root placement\npass MUST always find the GC tracked pointer from which this pointer is\nderived and use that as the pointer to root.\nCallee Rooted Pointers (currently 12): This is a utility address space to express the\nnotion of a callee rooted value. All values of this address space MUST be\nstorable to a GC root (though it is possible to relax this condition in the\nfuture), but unlike the other pointers need not be rooted if passed to a\ncall (they do still need to be rooted if they are live across another safepoint\nbetween the definition and the call).\nPointers loaded from tracked object (currently 13): This is used by arrays,\nwhich themselves contain a pointer to the managed data. This data area is owned\nby the array, but is not a GC-tracked object by itself. The compiler guarantees\nthat as long as this pointer is live, the object that this pointer was loaded\nfrom will keep being live."},{"title":"Invariants","page":"Working with LLVM","location":"devdocs/llvm.html#Invariants","category":"section","text":"The GC root placement pass makes use of several invariants, which need\nto be observed by the frontend and are preserved by the optimizer.\n\nFirst, only the following address space casts are allowed:\n\n0->{Tracked,Derived,CalleeRooted}: It is allowable to decay an untracked pointer to any of the\nothers. However, do note that the optimizer has broad license to not root\nsuch a value. It is never safe to have a value in address space 0 in any part\nof the program if it is (or is derived from) a value that requires a GC root.\nTracked->Derived: This is the standard decay route for interior values. The placement\npass will look for these to identify the base pointer for any use.\nTracked->CalleeRooted: Addrspace CalleeRooted serves merely as a hint that a GC root is not\nrequired. However, do note that the Derived->CalleeRooted decay is prohibited, since\npointers should generally be storable to a GC slot, even in this address space.\n\nNow let us consider what constitutes a use:\n\nLoads whose loaded values is in one of the address spaces\nStores of a value in one of the address spaces to a location\nStores to a pointer in one of the address spaces\nCalls for which a value in one of the address spaces is an operand\nCalls in jlcall ABI, for which the argument array contains a value\nReturn instructions.\n\nWe explicitly allow load/stores and simple calls in address spaces Tracked/Derived. Elements of jlcall\nargument arrays must always be in address space Tracked (it is required by the ABI that\nthey are valid jl_value_t* pointers). The same is true for return instructions\n(though note that struct return arguments are allowed to have any of the address\nspaces). The only allowable use of an address space CalleeRooted pointer is to pass it to\na call (which must have an appropriately typed operand).\n\nFurther, we disallow getelementptr in addrspace Tracked. This is because unless\nthe operation is a noop, the resulting pointer will not be validly storable\nto a GC slot and may thus not be in this address space. If such a pointer\nis required, it should be decayed to addrspace Derived first.\n\nLastly, we disallow inttoptr/ptrtoint instructions in these address spaces.\nHaving these instructions would mean that some i64 values are really GC tracked.\nThis is problematic, because it breaks that stated requirement that we're able\nto identify GC-relevant pointers. This invariant is accomplished using the LLVM\n\"non-integral pointers\" feature, which is new in LLVM 5.0. It prohibits the\noptimizer from making optimizations that would introduce these operations. Note\nwe can still insert static constants at JIT time by using inttoptr in address\nspace 0 and then decaying to the appropriate address space afterwards."},{"title":"Supporting ccall","page":"Working with LLVM","location":"devdocs/llvm.html#Supporting-[ccall](@ref)","category":"section","text":"One important aspect missing from the discussion so far is the handling of\nccall. ccall has the peculiar feature that the location and\nscope of a use do not coincide. As an example consider:\n\nA = randn(1024)\nccall(:foo, Cvoid, (Ptr{Float64},), A)\n\nIn lowering, the compiler will insert a conversion from the array to the\npointer which drops the reference to the array value. However, we of course\nneed to make sure that the array does stay alive while we're doing the\nccall. To understand how this is done, lets look at a hypothetical\napproximate possible lowering of the above code:\n\nreturn $(Expr(:foreigncall, Expr(:tuple, :(:foo)), Cvoid, svec(Ptr{Float64}), 0, :(:ccall), Expr(:foreigncall, Expr(:tuple, :(:jl_array_ptr)), Ptr{Float64}, svec(Any), 0, :(:ccall), :(A)), :(A)))\n\nThe last :(A), is an extra argument list inserted during lowering that informs\nthe code generator which Julia level values need to be kept alive for the\nduration of this ccall. We then take this information and represent\nit in an \"operand bundle\" at the IR level. An operand bundle is essentially a fake\nuse that is attached to the call site. At the IR level, this looks like so:\n\ncall void inttoptr (i64 ... to void (double*)*)(double* %5) [ \"jl_roots\"(%jl_value_t addrspace(10)* %A) ]\n\nThe GC root placement pass will treat the jl_roots operand bundle as if it were\na regular operand. However, as a final step, after the GC roots are inserted,\nit will drop the operand bundle to avoid confusing instruction selection."},{"title":"Supporting pointer_from_objref","page":"Working with LLVM","location":"devdocs/llvm.html#Supporting-[pointer_from_objref](@ref)","category":"section","text":"pointer_from_objref is special because it requires the user to take\nexplicit control of GC rooting. By our above invariants, this function is illegal,\nbecause it performs an address space cast from 10 to 0. However, it can be useful,\nin certain situations, so we provide a special intrinsic:\n\ndeclared %jl_value_t *julia.pointer_from_objref(%jl_value_t addrspace(10)*)\n\nwhich is lowered to the corresponding address space cast after GC root lowering.\nDo note however that by using this intrinsic, the caller assumes all responsibility\nfor making sure that the value in question is rooted. Further this intrinsic is\nnot considered a use, so the GC root placement pass will not provide a GC root\nfor the function. As a result, the external rooting must be arranged while the\nvalue is still tracked by the system. I.e. it is not valid to attempt to use the\nresult of this operation to establish a global root - the optimizer may have\nalready dropped the value."},{"title":"Keeping values alive in the absence of uses","page":"Working with LLVM","location":"devdocs/llvm.html#Keeping-values-alive-in-the-absence-of-uses","category":"section","text":"In certain cases it is necessary to keep an object alive, even though there is\nno compiler-visible use of said object. This may be case for low level code\nthat operates on the memory-representation of an object directly or code that\nneeds to interface with C code. In order to allow this, we provide the following\nintrinsics at the LLVM level:\n\ntoken @llvm.julia.gc_preserve_begin(...)\nvoid @llvm.julia.gc_preserve_end(token)\n\n(The llvm. in the name is required in order to be able to use the token\ntype). The semantics of these intrinsics are as follows:\nAt any safepoint that is dominated by a gc_preserve_begin call, but that is not\nnot dominated by a corresponding gc_preserve_end call (i.e. a call whose argument\nis the token returned by a gc_preserve_begin call), the values passed as\narguments to that gc_preserve_begin will be kept live. Note that the\ngc_preserve_begin still counts as a regular use of those values, so the\nstandard lifetime semantics will ensure that the values will be kept alive\nbefore entering the preserve region."},{"title":"Printf","page":"Printf","location":"stdlib/Printf.html#man-printf","category":"section","text":"The Printf module provides formatted output functions similar to the C standard library's printf. It allows formatted printing to an output stream or to a string."},{"title":"Printf.@printf","page":"Printf","location":"stdlib/Printf.html#Printf.@printf","category":"macro","text":"@printf([io::IO], \"%Fmt\", args...)\n\nPrint args using C printf style format specification string.\nOptionally, an IO may be passed as the first argument to redirect output.\n\nExamples\n\njulia> @printf \"Hello %s\" \"world\"\nHello world\n\njulia> @printf \"Scientific notation %e\" 1.234\nScientific notation 1.234000e+00\n\njulia> @printf \"Scientific notation three digits %.3e\" 1.23456\nScientific notation three digits 1.235e+00\n\njulia> @printf \"Decimal two digits %.2f\" 1.23456\nDecimal two digits 1.23\n\njulia> @printf \"Padded to length 5 %5i\" 123\nPadded to length 5   123\n\njulia> @printf \"Padded with zeros to length 6 %06i\" 123\nPadded with zeros to length 6 000123\n\njulia> @printf \"Use shorter of decimal or scientific %g %g\" 1.23 12300000.0\nUse shorter of decimal or scientific 1.23 1.23e+07\n\njulia> @printf \"Use dynamic width and precision  %*.*f\" 10 2 0.12345\nUse dynamic width and precision        0.12\n\nFor a systematic specification of the format, see here.\nSee also @sprintf to get the result as a String instead of it being printed.\n\nCaveats\n\nInf and NaN are printed consistently as Inf and NaN for flags %a, %A,\n%e, %E, %f, %F, %g, and %G. Furthermore, if a floating point number is\nequally close to the numeric values of two possible output strings, the output\nstring further away from zero is chosen.\n\nExamples\n\njulia> @printf(\"%f %F %f %F\", Inf, Inf, NaN, NaN)\nInf Inf NaN NaN\n\njulia> @printf \"%.0f %.1f %f\" 0.5 0.025 -0.0078125\n0 0.0 -0.007812\n\ncompat: Julia 1.8\nStarting in Julia 1.8, %s (string) and %c (character) widths are computed\nusing textwidth, which e.g. ignores zero-width characters\n(such as combining characters for diacritical marks) and treats certain\n\"wide\" characters (e.g. emoji) as width 2.\n\ncompat: Julia 1.10\nDynamic width specifiers like %*s and %0*.*f require Julia 1.10.\n\n\n\n\n\n"},{"title":"Printf.@sprintf","page":"Printf","location":"stdlib/Printf.html#Printf.@sprintf","category":"macro","text":"@sprintf(\"%Fmt\", args...)\n\nReturn @printf formatted output as string.\n\nExamples\n\njulia> @sprintf \"this is a %s %15.1f\" \"test\" 34.567\n\"this is a test            34.6\"\n\n\n\n\n\n"},{"title":"Printf.Format","page":"Printf","location":"stdlib/Printf.html#Printf.Format","category":"type","text":"Printf.Format(format_str)\n\nCreate a C printf-compatible format object that can be used for formatting values.\n\nThe input format_str can include any valid format specifier character and modifiers.\n\nA Format object can be passed to Printf.format(f::Format, args...) to produce a\nformatted string, or Printf.format(io::IO, f::Format, args...) to print the\nformatted string directly to io.\n\nFor convenience, the Printf.format\"...\" string macro form can be used for building\na Printf.Format object at macro-expansion-time.\n\ncompat: Julia 1.6\nPrintf.Format requires Julia 1.6 or later.\n\n\n\n\n\n"},{"title":"Printf.format","page":"Printf","location":"stdlib/Printf.html#Printf.format","category":"function","text":"Printf.format(f::Printf.Format, args...) => String\nPrintf.format(io::IO, f::Printf.Format, args...)\n\nApply a printf format object f to provided args and return the formatted string\n(1st method), or print directly to an io object (2nd method). See @printf\nfor more details on C printf support.\n\n\n\n\n\n"},{"title":"Sparse Arrays","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Sparse-Arrays","category":"section","text":"Julia has support for sparse vectors and sparse matrices\nin the SparseArrays stdlib module. Sparse arrays are arrays that contain enough zeros that storing them in a special data structure leads to savings in space and execution time, compared to dense arrays.\n\nExternal packages which implement different sparse storage types, multidimensional sparse arrays, and more can be found in Noteworthy External Sparse Packages"},{"title":"Compressed Sparse Column (CSC) Sparse Matrix Storage","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#man-csc","category":"section","text":"In Julia, sparse matrices are stored in the Compressed Sparse Column (CSC) format.\nJulia sparse matrices have the type SparseMatrixCSC{Tv,Ti}, where Tv is the\ntype of the stored values, and Ti is the integer type for storing column pointers and\nrow indices. The internal representation of SparseMatrixCSC is as follows:\n\nstruct SparseMatrixCSC{Tv,Ti<:Integer} <: AbstractSparseMatrixCSC{Tv,Ti}\n    m::Int                  # Number of rows\n    n::Int                  # Number of columns\n    colptr::Vector{Ti}      # Column j is in colptr[j]:(colptr[j+1]-1)\n    rowval::Vector{Ti}      # Row indices of stored values\n    nzval::Vector{Tv}       # Stored values, typically nonzeros\nend\n\nThe compressed sparse column storage makes it easy and quick to access the elements in the column\nof a sparse matrix, whereas accessing the sparse matrix by rows is considerably slower. Operations\nsuch as insertion of previously unstored entries one at a time in the CSC structure tend to be slow. This is\nbecause all elements of the sparse matrix that are beyond the point of insertion have to be moved\none place over.\n\nAll operations on sparse matrices are carefully implemented to exploit the CSC data structure\nfor performance, and to avoid expensive operations.\n\nIf you have data in CSC format from a different application or\nlibrary, and wish to import it in Julia, make sure that you use\n1-based indexing. The row indices in every column need to be sorted,\nand if they are not, the matrix will display incorrectly.  If your\nSparseMatrixCSC object contains unsorted row indices, one quick way\nto sort them is by doing a double transpose. Since the transpose operation\nis lazy, make a copy to materialize each transpose.\n\nIn some applications, it is convenient to store explicit zero values in a SparseMatrixCSC. These\nare accepted by functions in Base (but there is no guarantee that they will be preserved in\nmutating operations). Such explicitly stored zeros are treated as structural nonzeros by many\nroutines. The nnz function returns the number of elements explicitly stored in the\nsparse data structure, including non-structural zeros. In order to count the exact number of\nnumerical nonzeros, use count(!iszero, x), which inspects every stored element of a sparse\nmatrix. dropzeros, and the in-place dropzeros!, can be used to\nremove stored zeros from the sparse matrix.\n\njulia> A = sparse([1, 1, 2, 3], [1, 3, 2, 3], [0, 1, 2, 0])\n3×3 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n 0  ⋅  1\n ⋅  2  ⋅\n ⋅  ⋅  0\n\njulia> dropzeros(A)\n3×3 SparseMatrixCSC{Int64, Int64} with 2 stored entries:\n ⋅  ⋅  1\n ⋅  2  ⋅\n ⋅  ⋅  ⋅"},{"title":"Sparse Vector Storage","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Sparse-Vector-Storage","category":"section","text":"Sparse vectors are stored in a close analog to compressed sparse column format for sparse\nmatrices. In Julia, sparse vectors have the type SparseVector{Tv,Ti} where Tv\nis the type of the stored values and Ti the integer type for the indices. The internal\nrepresentation is as follows:\n\nstruct SparseVector{Tv,Ti<:Integer} <: AbstractSparseVector{Tv,Ti}\n    n::Int              # Length of the sparse vector\n    nzind::Vector{Ti}   # Indices of stored values\n    nzval::Vector{Tv}   # Stored values, typically nonzeros\nend\n\nLike SparseMatrixCSC, the SparseVector type can also contain explicitly\nstored zeros. (See Sparse Matrix Storage.)."},{"title":"Sparse Vector and Matrix Constructors","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Sparse-Vector-and-Matrix-Constructors","category":"section","text":"The simplest way to create a sparse array is to use a function equivalent to the zeros\nfunction that Julia provides for working with dense arrays. To produce a\nsparse array instead, you can use the same name with an sp prefix:\n\njulia> spzeros(3)\n3-element SparseVector{Float64, Int64} with 0 stored entries\n\nThe sparse function is often a handy way to construct sparse arrays. For\nexample, to construct a sparse matrix we can input a vector I of row indices, a vector\nJ of column indices, and a vector V of stored values (this is also known as the\nCOO (coordinate) format).\nsparse(I,J,V) then constructs a sparse matrix such that S[I[k], J[k]] = V[k]. The\nequivalent sparse vector constructor is sparsevec, which takes the (row) index\nvector I and the vector V with the stored values and constructs a sparse vector R\nsuch that R[I[k]] = V[k].\n\njulia> I = [1, 4, 3, 5]; J = [4, 7, 18, 9]; V = [1, 2, -5, 3];\n\njulia> S = sparse(I,J,V)\n5×18 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n⎡⠀⠈⠀⠀⠀⠀⠀⠀⢀⎤\n⎣⠀⠀⠀⠂⡀⠀⠀⠀⠀⎦\n\njulia> R = sparsevec(I,V)\n5-element SparseVector{Int64, Int64} with 4 stored entries:\n  [1]  =  1\n  [3]  =  -5\n  [4]  =  2\n  [5]  =  3\n\nThe inverse of the sparse and sparsevec functions is\nfindnz, which retrieves the inputs used to create the sparse array (including stored entries equal to zero).\nfindall(!iszero, x) returns the Cartesian indices of non-zero entries in x\n(not including stored entries equal to zero).\n\njulia> findnz(S)\n([1, 4, 5, 3], [4, 7, 9, 18], [1, 2, 3, -5])\n\njulia> findall(!iszero, S)\n4-element Vector{CartesianIndex{2}}:\n CartesianIndex(1, 4)\n CartesianIndex(4, 7)\n CartesianIndex(5, 9)\n CartesianIndex(3, 18)\n\njulia> findnz(R)\n([1, 3, 4, 5], [1, -5, 2, 3])\n\njulia> findall(!iszero, R)\n4-element Vector{Int64}:\n 1\n 3\n 4\n 5\n\nAnother way to create a sparse array is to convert a dense array into a sparse array using\nthe sparse function:\n\njulia> sparse(Matrix(1.0I, 5, 5))\n5×5 SparseMatrixCSC{Float64, Int64} with 5 stored entries:\n 1.0   ⋅    ⋅    ⋅    ⋅\n  ⋅   1.0   ⋅    ⋅    ⋅\n  ⋅    ⋅   1.0   ⋅    ⋅\n  ⋅    ⋅    ⋅   1.0   ⋅\n  ⋅    ⋅    ⋅    ⋅   1.0\n\njulia> sparse([1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0\n\nYou can go in the other direction using the Array constructor. The issparse\nfunction can be used to query if a matrix is sparse.\n\njulia> issparse(spzeros(5))\ntrue"},{"title":"Sparse matrix operations","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Sparse-matrix-operations","category":"section","text":"Arithmetic operations on sparse matrices also work as they do on dense matrices. Indexing of,\nassignment into, and concatenation of sparse matrices work in the same way as dense matrices.\nIndexing operations, especially assignment, are expensive, when carried out one element at a time.\nIn many cases it may be better to convert the sparse matrix into (I,J,V) format using findnz,\nmanipulate the values or the structure in the dense vectors (I,J,V), and then reconstruct\nthe sparse matrix."},{"title":"Correspondence of dense and sparse methods","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Correspondence-of-dense-and-sparse-methods","category":"section","text":"The following table gives a correspondence between built-in methods on sparse matrices and their\ncorresponding methods on dense matrix types. In general, methods that generate sparse matrices\ndiffer from their dense counterparts in that the resulting matrix follows the same sparsity pattern\nas a given sparse matrix S, or that the resulting sparse matrix has density d, i.e. each matrix\nelement has a probability d of being non-zero.\n\nDetails can be found in the Sparse Vectors and Matrices\nsection of the standard library reference.\n\nSparse Dense Description\nspzeros(m,n) zeros(m,n) Creates a m-by-n matrix of zeros. (spzeros(m,n) is empty.)\nsparse(I,n,n) Matrix(I,n,n) Creates a n-by-n identity matrix.\nsparse(A) Array(S) Interconverts between dense and sparse formats.\nsprand(m,n,d) rand(m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed uniformly on the half-open interval 0 1).\nsprandn(m,n,d) randn(m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed according to the standard normal (Gaussian) distribution.\nsprandn(rng,m,n,d) randn(rng,m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements generated with the rng random number generator"},{"title":"SparseArrays API","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#stdlib-sparse-arrays","category":"section","text":""},{"title":"Noteworthy External Sparse Packages","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Noteworthy-External-Sparse-Packages","category":"section","text":"Several other Julia packages provide sparse matrix implementations that should be mentioned:\n\nSuiteSparseGraphBLAS.jl is a wrapper over the fast, multithreaded SuiteSparse:GraphBLAS C library. On CPU this is typically the fastest option, often significantly outperforming MKLSparse.\nCUDA.jl exposes the CUSPARSE library for GPU sparse matrix operations.\nSparseMatricesCSR.jl provides a Julia native implementation of the Compressed Sparse Rows (CSR) format.\nMKLSparse.jl accelerates SparseArrays sparse-dense matrix operations using Intel's MKL library.\nSparseArrayKit.jl available for multidimensional sparse arrays.\nLuxurySparse.jl provides static sparse array formats, as well as a coordinate format.\nExtendableSparse.jl enables fast insertion into sparse matrices using a lazy approach to new stored indices.\nFinch.jl supports extensive multidimensional sparse array formats and operations through a mini tensor language and compiler, all in native Julia. Support for COO, CSF, CSR, CSC and more, as well as operations like broadcast, reduce, etc. and custom operations.\n\nExternal packages providing sparse direct solvers:\n\nKLU.jl\nPardiso.jl\n\nExternal packages providing solvers for iterative solution of eigensystems and singular value decompositions:\n\nArnoldiMethods.jl\nKrylovKit\nArpack.jl\n\nExternal packages for working with graphs:\n\nGraphs.jl"},{"title":"SparseArrays.AbstractSparseArray","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseArray","category":"type","text":"AbstractSparseArray{Tv,Ti,N}\n\nSupertype for N-dimensional sparse arrays (or array-like types) with elements\nof type Tv and index type Ti. SparseMatrixCSC, SparseVector\nand SuiteSparse.CHOLMOD.Sparse are subtypes of this.\n\n\n\n\n\n"},{"title":"SparseArrays.AbstractSparseVector","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseVector","category":"type","text":"AbstractSparseVector{Tv,Ti}\n\nSupertype for one-dimensional sparse arrays (or array-like types) with elements\nof type Tv and index type Ti. Alias for AbstractSparseArray{Tv,Ti,1}.\n\n\n\n\n\n"},{"title":"SparseArrays.AbstractSparseMatrix","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseMatrix","category":"type","text":"AbstractSparseMatrix{Tv,Ti}\n\nSupertype for two-dimensional sparse arrays (or array-like types) with elements\nof type Tv and index type Ti. Alias for AbstractSparseArray{Tv,Ti,2}.\n\n\n\n\n\n"},{"title":"SparseArrays.SparseVector","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.SparseVector","category":"type","text":"SparseVector{Tv,Ti<:Integer} <: AbstractSparseVector{Tv,Ti}\n\nVector type for storing sparse vectors. Can be created by passing the length of the vector,\na sorted vector of non-zero indices, and a vector of non-zero values.\n\nFor instance, the vector [5, 6, 0, 7] can be represented as\n\nSparseVector(4, [1, 2, 4], [5, 6, 7])\n\nThis indicates that the element at index 1 is 5, at index 2 is 6, at index 3 is zero(Int),\nand at index 4 is 7.\n\nIt may be more convenient to create sparse vectors directly from dense vectors using sparse as\n\nsparse([5, 6, 0, 7])\n\nyields the same sparse vector.\n\n\n\n\n\n"},{"title":"SparseArrays.SparseMatrixCSC","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.SparseMatrixCSC","category":"type","text":"SparseMatrixCSC{Tv,Ti<:Integer} <: AbstractSparseMatrixCSC{Tv,Ti}\n\nMatrix type for storing sparse matrices in the\nCompressed Sparse Column format. The standard way\nof constructing SparseMatrixCSC is through the sparse function.\nSee also spzeros, spdiagm and sprand.\n\n\n\n\n\n"},{"title":"SparseArrays.sparse","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sparse","category":"function","text":"sparse(A::Union{AbstractVector, AbstractMatrix})\n\nConvert a vector or matrix A into a sparse array.\nNumerical zeros in A are turned into structural zeros.\n\nExamples\n\njulia> A = Matrix(1.0I, 3, 3)\n3×3 Matrix{Float64}:\n 1.0  0.0  0.0\n 0.0  1.0  0.0\n 0.0  0.0  1.0\n\njulia> sparse(A)\n3×3 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n 1.0   ⋅    ⋅\n  ⋅   1.0   ⋅\n  ⋅    ⋅   1.0\n\njulia> [1.0, 0.0, 1.0]\n3-element Vector{Float64}:\n 1.0\n 0.0\n 1.0\n\njulia> sparse([1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0\n\n\n\n\n\nsparse(I, J, V,[ m, n, combine])\n\nCreate a sparse matrix S of dimensions m x n such that S[I[k], J[k]] = V[k]. The\ncombine function is used to combine duplicates. If m and n are not specified, they\nare set to maximum(I) and maximum(J) respectively. If the combine function is not\nsupplied, combine defaults to + unless the elements of V are Booleans in which case\ncombine defaults to |. All elements of I must satisfy 1 <= I[k] <= m, and all\nelements of J must satisfy 1 <= J[k] <= n. Numerical zeros in (I, J, V) are\nretained as structural nonzeros; to drop numerical zeros, use dropzeros!.\n\nFor additional documentation and an expert driver, see SparseArrays.sparse!.\n\nExamples\n\njulia> Is = [1; 2; 3];\n\njulia> Js = [1; 2; 3];\n\njulia> Vs = [1; 2; 3];\n\njulia> sparse(Is, Js, Vs)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 1  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  3\n\n\n\n\n\n"},{"title":"SparseArrays.sparse!","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sparse!","category":"function","text":"sparse!(I::AbstractVector{Ti}, J::AbstractVector{Ti}, V::AbstractVector{Tv},\n        m::Integer, n::Integer, combine, klasttouch::Vector{Ti},\n        csrrowptr::Vector{Ti}, csrcolval::Vector{Ti}, csrnzval::Vector{Tv},\n        [csccolptr::Vector{Ti}], [cscrowval::Vector{Ti}, cscnzval::Vector{Tv}] ) where {Tv,Ti<:Integer}\n\nParent of and expert driver for sparse;\nsee sparse for basic usage. This method\nallows the user to provide preallocated storage for sparse's intermediate objects and\nresult as described below. This capability enables more efficient successive construction\nof SparseMatrixCSCs from coordinate representations, and also enables extraction\nof an unsorted-column representation of the result's transpose at no additional cost.\n\nThis method consists of three major steps: (1) Counting-sort the provided coordinate\nrepresentation into an unsorted-row CSR form including repeated entries. (2) Sweep through\nthe CSR form, simultaneously calculating the desired CSC form's column-pointer array,\ndetecting repeated entries, and repacking the CSR form with repeated entries combined;\nthis stage yields an unsorted-row CSR form with no repeated entries. (3) Counting-sort the\npreceding CSR form into a fully-sorted CSC form with no repeated entries.\n\nInput arrays csrrowptr, csrcolval, and csrnzval constitute storage for the\nintermediate CSR forms and require length(csrrowptr) >= m + 1,\nlength(csrcolval) >= length(I), and length(csrnzval >= length(I)). Input\narray klasttouch, workspace for the second stage, requires length(klasttouch) >= n.\nOptional input arrays csccolptr, cscrowval, and cscnzval constitute storage for the\nreturned CSC form S. If necessary, these are resized automatically to satisfy\nlength(csccolptr) = n + 1, length(cscrowval) = nnz(S) and length(cscnzval) = nnz(S); hence, if nnz(S) is\nunknown at the outset, passing in empty vectors of the appropriate type (Vector{Ti}()\nand Vector{Tv}() respectively) suffices, or calling the sparse! method\nneglecting cscrowval and cscnzval.\n\nOn return, csrrowptr, csrcolval, and csrnzval contain an unsorted-column\nrepresentation of the result's transpose.\n\nYou may reuse the input arrays' storage (I, J, V) for the output arrays\n(csccolptr, cscrowval, cscnzval). For example, you may call\nsparse!(I, J, V, csrrowptr, csrcolval, csrnzval, I, J, V).\nNote that they will be resized to satisfy the conditions above.\n\nFor the sake of efficiency, this method performs no argument checking beyond\n1 <= I[k] <= m and 1 <= J[k] <= n. Use with care. Testing with --check-bounds=yes\nis wise.\n\nThis method runs in O(m, n, length(I)) time. The HALFPERM algorithm described in\nF. Gustavson, \"Two fast algorithms for sparse matrices: multiplication and permuted\ntransposition,\" ACM TOMS 4(3), 250-269 (1978) inspired this method's use of a pair of\ncounting sorts.\n\n\n\n\n\nSparseArrays.sparse!(I, J, V, [m, n, combine]) -> SparseMatrixCSC\n\nVariant of sparse! that re-uses the input vectors (I, J, V) for the final matrix\nstorage. After construction the input vectors will alias the matrix buffers; S.colptr === I, S.rowval === J, and S.nzval === V holds, and they will be resize!d as necessary.\n\nNote that some work buffers will still be allocated. Specifically, this method is a\nconvenience wrapper around sparse!(I, J, V, m, n, combine, klasttouch, csrrowptr, csrcolval, csrnzval, csccolptr, cscrowval, cscnzval) where this method allocates\nklasttouch, csrrowptr, csrcolval, and csrnzval of appropriate size, but reuses I,\nJ, and V for csccolptr, cscrowval, and cscnzval.\n\nArguments m, n, and combine defaults to maximum(I), maximum(J), and +,\nrespectively.\n\ncompat: Julia 1.10\nThis method requires Julia version 1.10 or later.\n\n\n\n\n\n"},{"title":"SparseArrays.sparsevec","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sparsevec","category":"function","text":"sparsevec(I, V, [m, combine])\n\nCreate a sparse vector S of length m such that S[I[k]] = V[k].\nDuplicates are combined using the combine function, which defaults to\n+ if no combine argument is provided, unless the elements of V are Booleans\nin which case combine defaults to |.\n\nExamples\n\njulia> II = [1, 3, 3, 5]; V = [0.1, 0.2, 0.3, 0.2];\n\njulia> sparsevec(II, V)\n5-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  0.1\n  [3]  =  0.5\n  [5]  =  0.2\n\njulia> sparsevec(II, V, 8, -)\n8-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  0.1\n  [3]  =  -0.1\n  [5]  =  0.2\n\njulia> sparsevec([1, 3, 1, 2, 2], [true, true, false, false, false])\n3-element SparseVector{Bool, Int64} with 3 stored entries:\n  [1]  =  1\n  [2]  =  0\n  [3]  =  1\n\n\n\n\n\nsparsevec(d::Dict, [m])\n\nCreate a sparse vector of length m where the nonzero indices are keys from\nthe dictionary, and the nonzero values are the values from the dictionary.\n\nExamples\n\njulia> sparsevec(Dict(1 => 3, 2 => 2))\n2-element SparseVector{Int64, Int64} with 2 stored entries:\n  [1]  =  3\n  [2]  =  2\n\n\n\n\n\nsparsevec(A)\n\nConvert a vector A into a sparse vector of length m.\nNumerical zeros in A are turned into structural zeros.\n\nExamples\n\njulia> sparsevec([1.0, 2.0, 0.0, 0.0, 3.0, 0.0])\n6-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  1.0\n  [2]  =  2.0\n  [5]  =  3.0\n\n\n\n\n\n"},{"title":"Base.similar","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Base.similar-Tuple{SparseArrays.AbstractSparseMatrixCSC, Type}","category":"method","text":"similar(A::AbstractSparseMatrixCSC{Tv,Ti}, [::Type{TvNew}, ::Type{TiNew}, m::Integer, n::Integer]) where {Tv,Ti}\n\nCreate an uninitialized mutable array with the given element type,\nindex type, and size, based upon the given source\nSparseMatrixCSC. The new sparse matrix maintains the structure of\nthe original sparse matrix, except in the case where dimensions of the\noutput matrix are different from the output.\n\nThe output matrix has zeros in the same locations as the input, but\nuninitialized values for the nonzero locations.\n\n\n\n\n\n"},{"title":"SparseArrays.issparse","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.issparse","category":"function","text":"issparse(S)\n\nReturns true if S is sparse, and false otherwise.\n\nExamples\n\njulia> sv = sparsevec([1, 4], [2.3, 2.2], 10)\n10-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  2.3\n  [4]  =  2.2\n\njulia> issparse(sv)\ntrue\n\njulia> issparse(Array(sv))\nfalse\n\n\n\n\n\n"},{"title":"SparseArrays.nnz","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.nnz","category":"function","text":"nnz(A)\n\nReturns the number of stored (filled) elements in a sparse array.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> nnz(A)\n3\n\n\n\n\n\n"},{"title":"SparseArrays.findnz","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.findnz","category":"function","text":"findnz(A::SparseMatrixCSC)\n\nReturn a tuple (I, J, V) where I and J are the row and column indices of the stored\n(\"structurally non-zero\") values in sparse matrix A, and V is a vector of the values.\n\nExamples\n\njulia> A = sparse([1 2 0; 0 0 3; 0 4 0])\n3×3 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n 1  2  ⋅\n ⋅  ⋅  3\n ⋅  4  ⋅\n\njulia> findnz(A)\n([1, 1, 3, 2], [1, 2, 2, 3], [1, 2, 4, 3])\n\n\n\n\n\n"},{"title":"SparseArrays.spzeros","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.spzeros","category":"function","text":"spzeros([type,]m[,n])\n\nCreate a sparse vector of length m or sparse matrix of size m x n. This\nsparse array will not contain any nonzero values. No storage will be allocated\nfor nonzero values during construction. The type defaults to Float64 if not\nspecified.\n\nExamples\n\njulia> spzeros(3, 3)\n3×3 SparseMatrixCSC{Float64, Int64} with 0 stored entries:\n  ⋅    ⋅    ⋅\n  ⋅    ⋅    ⋅\n  ⋅    ⋅    ⋅\n\njulia> spzeros(Float32, 4)\n4-element SparseVector{Float32, Int64} with 0 stored entries\n\n\n\n\n\nspzeros([type], I::AbstractVector, J::AbstractVector, [m, n])\n\nCreate a sparse matrix S of dimensions m x n with structural zeros at S[I[k], J[k]].\n\nThis method can be used to construct the sparsity pattern of the matrix, and is more\nefficient than using e.g. sparse(I, J, zeros(length(I))).\n\nFor additional documentation and an expert driver, see SparseArrays.spzeros!.\n\ncompat: Julia 1.10\nThis methods requires Julia version 1.10 or later.\n\n\n\n\n\n"},{"title":"SparseArrays.spzeros!","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.spzeros!","category":"function","text":"spzeros!(::Type{Tv}, I::AbstractVector{Ti}, J::AbstractVector{Ti}, m::Integer, n::Integer,\n         klasttouch::Vector{Ti}, csrrowptr::Vector{Ti}, csrcolval::Vector{Ti},\n         [csccolptr::Vector{Ti}], [cscrowval::Vector{Ti}, cscnzval::Vector{Tv}]) where {Tv,Ti<:Integer}\n\nParent of and expert driver for spzeros(I, J) allowing user to provide preallocated\nstorage for intermediate objects. This method is to spzeros what SparseArrays.sparse! is\nto sparse. See documentation for SparseArrays.sparse! for details and required buffer\nlengths.\n\ncompat: Julia 1.10\nThis methods requires Julia version 1.10 or later.\n\n\n\n\n\nSparseArrays.spzeros!(::Type{Tv}, I, J, [m, n]) -> SparseMatrixCSC{Tv}\n\nVariant of spzeros! that re-uses the input vectors I and J for the final matrix\nstorage. After construction the input vectors will alias the matrix buffers; S.colptr === I and S.rowval === J holds, and they will be resize!d as necessary.\n\nNote that some work buffers will still be allocated. Specifically, this method is a\nconvenience wrapper around spzeros!(Tv, I, J, m, n, klasttouch, csrrowptr, csrcolval, csccolptr, cscrowval) where this method allocates klasttouch, csrrowptr, and\ncsrcolval of appropriate size, but reuses I and J for csccolptr and cscrowval.\n\nArguments m and n defaults to maximum(I) and maximum(J).\n\ncompat: Julia 1.10\nThis method requires Julia version 1.10 or later.\n\n\n\n\n\n"},{"title":"SparseArrays.spdiagm","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.spdiagm","category":"function","text":"spdiagm(kv::Pair{<:Integer,<:AbstractVector}...)\nspdiagm(m::Integer, n::Integer, kv::Pair{<:Integer,<:AbstractVector}...)\n\nConstruct a sparse diagonal matrix from Pairs of vectors and diagonals.\nEach vector kv.second will be placed on the kv.first diagonal.  By\ndefault, the matrix is square and its size is inferred\nfrom kv, but a non-square size m×n (padded with zeros as needed)\ncan be specified by passing m,n as the first arguments.\n\nExamples\n\njulia> spdiagm(-1 => [1,2,3,4], 1 => [4,3,2,1])\n5×5 SparseMatrixCSC{Int64, Int64} with 8 stored entries:\n ⋅  4  ⋅  ⋅  ⋅\n 1  ⋅  3  ⋅  ⋅\n ⋅  2  ⋅  2  ⋅\n ⋅  ⋅  3  ⋅  1\n ⋅  ⋅  ⋅  4  ⋅\n\n\n\n\n\nspdiagm(v::AbstractVector)\nspdiagm(m::Integer, n::Integer, v::AbstractVector)\n\nConstruct a sparse matrix with elements of the vector as diagonal elements.\nBy default (no given m and n), the matrix is square and its size is given\nby length(v), but a non-square size m×n can be specified by passing m\nand n as the first arguments.\n\ncompat: Julia 1.6\nThese functions require at least Julia 1.6.\n\nExamples\n\njulia> spdiagm([1,2,3])\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 1  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  3\n\njulia> spdiagm(sparse([1,0,3]))\n3×3 SparseMatrixCSC{Int64, Int64} with 2 stored entries:\n 1  ⋅  ⋅\n ⋅  ⋅  ⋅\n ⋅  ⋅  3\n\n\n\n\n\n"},{"title":"SparseArrays.sparse_hcat","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sparse_hcat","category":"function","text":"sparse_hcat(A...)\n\nConcatenate along dimension 2. Return a SparseMatrixCSC object.\n\ncompat: Julia 1.8\nThis method was added in Julia 1.8. It mimics previous concatenation behavior, where\nthe concatenation with specialized \"sparse\" matrix types from LinearAlgebra.jl\nautomatically yielded sparse output even in the absence of any SparseArray argument.\n\n\n\n\n\n"},{"title":"SparseArrays.sparse_vcat","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sparse_vcat","category":"function","text":"sparse_vcat(A...)\n\nConcatenate along dimension 1. Return a SparseMatrixCSC object.\n\ncompat: Julia 1.8\nThis method was added in Julia 1.8. It mimics previous concatenation behavior, where\nthe concatenation with specialized \"sparse\" matrix types from LinearAlgebra.jl\nautomatically yielded sparse output even in the absence of any SparseArray argument.\n\n\n\n\n\n"},{"title":"SparseArrays.sparse_hvcat","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sparse_hvcat","category":"function","text":"sparse_hvcat(rows::Tuple{Vararg{Int}}, values...)\n\nSparse horizontal and vertical concatenation in one call. This function is called\nfor block matrix syntax. The first argument specifies the number of\narguments to concatenate in each block row.\n\ncompat: Julia 1.8\nThis method was added in Julia 1.8. It mimics previous concatenation behavior, where\nthe concatenation with specialized \"sparse\" matrix types from LinearAlgebra.jl\nautomatically yielded sparse output even in the absence of any SparseArray argument.\n\n\n\n\n\n"},{"title":"SparseArrays.blockdiag","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.blockdiag","category":"function","text":"blockdiag(A...)\n\nConcatenate matrices block-diagonally. Currently only implemented for sparse matrices.\n\nExamples\n\njulia> blockdiag(sparse(2I, 3, 3), sparse(4I, 2, 2))\n5×5 SparseMatrixCSC{Int64, Int64} with 5 stored entries:\n 2  ⋅  ⋅  ⋅  ⋅\n ⋅  2  ⋅  ⋅  ⋅\n ⋅  ⋅  2  ⋅  ⋅\n ⋅  ⋅  ⋅  4  ⋅\n ⋅  ⋅  ⋅  ⋅  4\n\n\n\n\n\n"},{"title":"SparseArrays.sprand","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sprand","category":"function","text":"sprand([rng],[T::Type],m,[n],p::AbstractFloat)\nsprand([rng],m,[n],p::AbstractFloat,[rfn=rand])\n\nCreate a random length m sparse vector or m by n sparse matrix, in\nwhich the probability of any element being nonzero is independently given by\np (and hence the mean density of nonzeros is also exactly p).\nThe optional rng argument specifies a random number generator, see Random Numbers.\nThe optional T argument specifies the element type, which defaults to Float64.\n\nBy default, nonzero values are sampled from a uniform distribution using\nthe rand function, i.e. by rand(T), or rand(rng, T) if rng\nis supplied; for the default T=Float64, this corresponds to nonzero values\nsampled uniformly in [0,1).\n\nYou can sample nonzero values from a different distribution by passing a\ncustom rfn function instead of rand.   This should be a function rfn(k)\nthat returns an array of k random numbers sampled from the desired distribution;\nalternatively, if rng is supplied, it should instead be a function rfn(rng, k).\n\nExamples\n\njulia> sprand(Bool, 2, 2, 0.5)\n2×2 SparseMatrixCSC{Bool, Int64} with 2 stored entries:\n 1  1\n ⋅  ⋅\n\njulia> sprand(Float64, 3, 0.75)\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  0.795547\n  [2]  =  0.49425\n\n\n\n\n\n"},{"title":"SparseArrays.sprandn","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.sprandn","category":"function","text":"sprandn([rng][,Type],m[,n],p::AbstractFloat)\n\nCreate a random sparse vector of length m or sparse matrix of size m by n\nwith the specified (independent) probability p of any entry being nonzero,\nwhere nonzero values are sampled from the normal distribution. The optional rng\nargument specifies a random number generator, see Random Numbers.\n\ncompat: Julia 1.1\nSpecifying the output element type Type requires at least Julia 1.1.\n\nExamples\n\njulia> sprandn(2, 2, 0.75)\n2×2 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n -1.20577     ⋅\n  0.311817  -0.234641\n\n\n\n\n\n"},{"title":"SparseArrays.nonzeros","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.nonzeros","category":"function","text":"nonzeros(A)\n\nReturn a vector of the structural nonzero values in sparse array A. This\nincludes zeros that are explicitly stored in the sparse array. The returned\nvector points directly to the internal nonzero storage of A, and any\nmodifications to the returned vector will mutate A as well. See\nrowvals and nzrange.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> nonzeros(A)\n3-element Vector{Int64}:\n 2\n 2\n 2\n\n\n\n\n\n"},{"title":"SparseArrays.rowvals","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.rowvals","category":"function","text":"rowvals(A)\n\nReturn a vector of the row indices of sparse array A. Any modifications to the returned\nvector will mutate A as well. Providing access to how the row indices are\nstored internally can be useful in conjunction with iterating over structural\nnonzero values. See also nonzeros and nzrange.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> rowvals(A)\n3-element Vector{Int64}:\n 1\n 2\n 3\n\n\n\n\n\n"},{"title":"SparseArrays.nzrange","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.nzrange","category":"function","text":"nzrange(A, col::Integer)\n\nReturn the range of indices to the structural nonzero values of column col\nof sparse array A. In conjunction with nonzeros and\nrowvals, this allows for convenient iterating over a sparse matrix :\n\nA = sparse(I,J,V)\nrows = rowvals(A)\nvals = nonzeros(A)\nm, n = size(A)\nfor j = 1:n\n   for i in nzrange(A, j)\n      row = rows[i]\n      val = vals[i]\n      # perform sparse wizardry...\n   end\nend\n\nwarning: Warning\nAdding or removing nonzero elements to the matrix may invalidate the nzrange, one should not mutate the matrix while iterating.\n\n\n\n\n\nnzrange(x::SparseVectorUnion, col)\n\nGive the range of indices to the structural nonzero values of a sparse vector.\nThe column index col is ignored (assumed to be 1).\n\n\n\n\n\n"},{"title":"SparseArrays.droptol!","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.droptol!","category":"function","text":"droptol!(A::AbstractSparseMatrixCSC, tol)\n\nRemoves stored values from A whose absolute value is less than or equal to tol.\n\n\n\n\n\ndroptol!(x::AbstractCompressedVector, tol)\n\nRemoves stored values from x whose absolute value is less than or equal to tol.\n\n\n\n\n\n"},{"title":"SparseArrays.dropzeros!","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.dropzeros!","category":"function","text":"dropzeros!(x::AbstractCompressedVector)\n\nRemoves stored numerical zeros from x.\n\nFor an out-of-place version, see dropzeros. For\nalgorithmic information, see fkeep!.\n\n\n\n\n\n"},{"title":"SparseArrays.dropzeros","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.dropzeros","category":"function","text":"dropzeros(A::AbstractSparseMatrixCSC;)\n\nGenerates a copy of A and removes stored numerical zeros from that copy.\n\nFor an in-place version and algorithmic information, see dropzeros!.\n\nExamples\n\njulia> A = sparse([1, 2, 3], [1, 2, 3], [1.0, 0.0, 1.0])\n3×3 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n 1.0   ⋅    ⋅\n  ⋅   0.0   ⋅\n  ⋅    ⋅   1.0\n\njulia> dropzeros(A)\n3×3 SparseMatrixCSC{Float64, Int64} with 2 stored entries:\n 1.0   ⋅    ⋅\n  ⋅    ⋅    ⋅\n  ⋅    ⋅   1.0\n\n\n\n\n\ndropzeros(x::AbstractCompressedVector)\n\nGenerates a copy of x and removes numerical zeros from that copy.\n\nFor an in-place version and algorithmic information, see dropzeros!.\n\nExamples\n\njulia> A = sparsevec([1, 2, 3], [1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  1.0\n  [2]  =  0.0\n  [3]  =  1.0\n\njulia> dropzeros(A)\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0\n\n\n\n\n\n"},{"title":"SparseArrays.permute","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.permute","category":"function","text":"permute(A::AbstractSparseMatrixCSC{Tv,Ti}, p::AbstractVector{<:Integer},\n        q::AbstractVector{<:Integer}) where {Tv,Ti}\n\nBilaterally permute A, returning PAQ (A[p,q]). Column-permutation q's length must\nmatch A's column count (length(q) == size(A, 2)). Row-permutation p's length must match A's\nrow count (length(p) == size(A, 1)).\n\nFor expert drivers and additional information, see permute!.\n\nExamples\n\njulia> A = spdiagm(0 => [1, 2, 3, 4], 1 => [5, 6, 7])\n4×4 SparseMatrixCSC{Int64, Int64} with 7 stored entries:\n 1  5  ⋅  ⋅\n ⋅  2  6  ⋅\n ⋅  ⋅  3  7\n ⋅  ⋅  ⋅  4\n\njulia> permute(A, [4, 3, 2, 1], [1, 2, 3, 4])\n4×4 SparseMatrixCSC{Int64, Int64} with 7 stored entries:\n ⋅  ⋅  ⋅  4\n ⋅  ⋅  3  7\n ⋅  2  6  ⋅\n 1  5  ⋅  ⋅\n\njulia> permute(A, [1, 2, 3, 4], [4, 3, 2, 1])\n4×4 SparseMatrixCSC{Int64, Int64} with 7 stored entries:\n ⋅  ⋅  5  1\n ⋅  6  2  ⋅\n 7  3  ⋅  ⋅\n 4  ⋅  ⋅  ⋅\n\n\n\n\n\n"},{"title":"Base.permute!","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#Base.permute!-Union{Tuple{Tq}, Tuple{Tp}, Tuple{Ti}, Tuple{Tv}, Tuple{SparseMatrixCSC{Tv, Ti}, SparseMatrixCSC{Tv, Ti}, AbstractVector{Tp}, AbstractVector{Tq}}} where {Tv, Ti, Tp<:Integer, Tq<:Integer}","category":"method","text":"permute!(X::AbstractSparseMatrixCSC{Tv,Ti}, A::AbstractSparseMatrixCSC{Tv,Ti},\n         p::AbstractVector{<:Integer}, q::AbstractVector{<:Integer},\n         [C::AbstractSparseMatrixCSC{Tv,Ti}]) where {Tv,Ti}\n\nBilaterally permute A, storing result PAQ (A[p,q]) in X. Stores intermediate result\n(AQ)^T (transpose(A[:,q])) in optional argument C if present. Requires that none of\nX, A, and, if present, C alias each other; to store result PAQ back into A, use\nthe following method lacking X:\n\npermute!(A::AbstractSparseMatrixCSC{Tv,Ti}, p::AbstractVector{<:Integer},\n         q::AbstractVector{<:Integer}[, C::AbstractSparseMatrixCSC{Tv,Ti},\n         [workcolptr::Vector{Ti}]]) where {Tv,Ti}\n\nX's dimensions must match those of A (size(X, 1) == size(A, 1) and size(X, 2) == size(A, 2)), and X must\nhave enough storage to accommodate all allocated entries in A (length(rowvals(X)) >= nnz(A)\nand length(nonzeros(X)) >= nnz(A)). Column-permutation q's length must match A's column\ncount (length(q) == size(A, 2)). Row-permutation p's length must match A's row count\n(length(p) == size(A, 1)).\n\nC's dimensions must match those of transpose(A) (size(C, 1) == size(A, 2) and size(C, 2) == size(A, 1)), and C\nmust have enough storage to accommodate all allocated entries in A (length(rowvals(C)) >= nnz(A)\nand length(nonzeros(C)) >= nnz(A)).\n\nFor additional (algorithmic) information, and for versions of these methods that forgo\nargument checking, see (unexported) parent methods unchecked_noalias_permute!\nand unchecked_aliasing_permute!.\n\nSee also permute.\n\n\n\n\n\n"},{"title":"SparseArrays.halfperm!","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.halfperm!","category":"function","text":"halfperm!(X::AbstractSparseMatrixCSC{Tv,Ti}, A::AbstractSparseMatrixCSC{TvA,Ti},\n          q::AbstractVector{<:Integer}, f::Function = identity) where {Tv,TvA,Ti}\n\nColumn-permute and transpose A, simultaneously applying f to each entry of A, storing\nthe result (f(A)Q)^T (map(f, transpose(A[:,q]))) in X.\n\nElement type Tv of X must match f(::TvA), where TvA is the element type of A.\nX's dimensions must match those of transpose(A) (size(X, 1) == size(A, 2) and\nsize(X, 2) == size(A, 1)), and X must have enough storage to accommodate all allocated\nentries in A (length(rowvals(X)) >= nnz(A) and length(nonzeros(X)) >= nnz(A)).\nColumn-permutation q's length must match A's column count (length(q) == size(A, 2)).\n\nThis method is the parent of several methods performing transposition and permutation\noperations on SparseMatrixCSCs. As this method performs no argument checking,\nprefer the safer child methods ([c]transpose[!], permute[!]) to direct use.\n\nThis method implements the HALFPERM algorithm described in F. Gustavson, \"Two fast\nalgorithms for sparse matrices: multiplication and permuted transposition,\" ACM TOMS 4(3),\n250-269 (1978). The algorithm runs in O(size(A, 1), size(A, 2), nnz(A)) time and requires no space\nbeyond that passed in.\n\n\n\n\n\n"},{"title":"SparseArrays.ftranspose!","page":"Sparse Arrays","location":"stdlib/SparseArrays.html#SparseArrays.ftranspose!","category":"function","text":"ftranspose!(X::AbstractSparseMatrixCSC{Tv,Ti}, A::AbstractSparseMatrixCSC{Tv,Ti}, f::Function) where {Tv,Ti}\n\nTranspose A and store it in X while applying the function f to the non-zero elements.\nDoes not remove the zeros created by f. size(X) must be equal to size(transpose(A)).\nNo additional memory is allocated other than resizing the rowval and nzval of X, if needed.\n\nSee halfperm!\n\n\n\n\n\n"},{"title":"Style Guide","page":"Style Guide","location":"manual/style-guide.html#Style-Guide","category":"section","text":"The following sections explain a few aspects of idiomatic Julia coding style. None of these rules\nare absolute; they are only suggestions to help familiarize you with the language and to help\nyou choose among alternative designs."},{"title":"Indentation","page":"Style Guide","location":"manual/style-guide.html#Indentation","category":"section","text":"Use 4 spaces per indentation level."},{"title":"Write functions, not just scripts","page":"Style Guide","location":"manual/style-guide.html#Write-functions,-not-just-scripts","category":"section","text":"Writing code as a series of steps at the top level is a quick way to get started solving a problem,\nbut you should try to divide a program into functions as soon as possible. Functions are more\nreusable and testable, and clarify what steps are being done and what their inputs and outputs\nare. Furthermore, code inside functions tends to run much faster than top level code, due to how\nJulia's compiler works.\n\nIt is also worth emphasizing that functions should take arguments, instead of operating directly\non global variables (aside from constants like pi)."},{"title":"Write docstrings","page":"Style Guide","location":"manual/style-guide.html#Write-docstrings","category":"section","text":"Comments describing an object should typically be written as docstrings for editor and REPL accessibility.\nInline comments (# comment) and multiline comments (#= comment =#) are appropriate for information that is intended only for the reader of the code (as opposed to a user)."},{"title":"Avoid writing overly-specific types","page":"Style Guide","location":"manual/style-guide.html#Avoid-writing-overly-specific-types","category":"section","text":"Code should be as generic as possible. Instead of writing:\n\nComplex{Float64}(x)\n\nit's better to use available generic functions:\n\ncomplex(float(x))\n\nThe second version will convert x to an appropriate type, instead of always the same type.\n\nThis style point is especially relevant to function arguments. For example, don't declare an argument\nto be of type Int or Int32 if it really could be any integer, expressed with the abstract\ntype Integer. In fact, in many cases you can omit the argument type altogether,\nunless it is needed to disambiguate from other method definitions, since a\nMethodError will be thrown anyway if a type is passed that does not support any\nof the requisite operations. (This is known as\nduck typing.)\n\nFor example, consider the following definitions of a function addone that returns one plus its\nargument:\n\naddone(x::Int) = x + 1                 # works only for Int\naddone(x::Integer) = x + oneunit(x)    # any integer type\naddone(x::Number) = x + oneunit(x)     # any numeric type\naddone(x) = x + oneunit(x)             # any type supporting + and oneunit\n\nThe last definition of addone handles any type supporting oneunit (which returns 1 in\nthe same type as x, which avoids unwanted type promotion) and the + function with\nthose arguments. The key thing to realize is that there is no performance penalty to defining\nonly the general addone(x) = x + oneunit(x), because Julia will automatically compile specialized\nversions as needed. For example, the first time you call addone(12), Julia will automatically\ncompile a specialized addone function for x::Int arguments, with the call to oneunit\nreplaced by its inlined value 1. Therefore, the first three definitions of addone above are\ncompletely redundant with the fourth definition."},{"title":"Handle excess argument diversity in the caller","page":"Style Guide","location":"manual/style-guide.html#Handle-excess-argument-diversity-in-the-caller","category":"section","text":"Instead of:\n\nfunction foo(x, y)\n    x = Int(x); y = Int(y)\n    ...\nend\nfoo(x, y)\n\nuse:\n\nfunction foo(x::Int, y::Int)\n    ...\nend\nfoo(Int(x), Int(y))\n\nThis is better style because foo does not really accept numbers of all types; it really needs\nInt s.\n\nOne issue here is that if a function inherently requires integers, it might be better to force\nthe caller to decide how non-integers should be converted (e.g. floor or ceiling). Another issue\nis that declaring more specific types leaves more \"space\" for future method definitions."},{"title":"Append ! to names of functions that modify their arguments","page":"Style Guide","location":"manual/style-guide.html#bang-convention","category":"section","text":"Instead of:\n\nfunction double(a::AbstractArray{<:Number})\n    for i in eachindex(a)\n        a[i] *= 2\n    end\n    return a\nend\n\nuse:\n\nfunction double!(a::AbstractArray{<:Number})\n    for i in eachindex(a)\n        a[i] *= 2\n    end\n    return a\nend\n\nJulia Base uses this convention throughout and contains examples of functions\nwith both copying and modifying forms (e.g., sort and sort!), and others\nwhich are just modifying (e.g., push!, pop!, splice!). It\nis typical for such functions to also return the modified array for convenience.\n\nFunctions related to IO or making use of random number generators (RNG) are notable exceptions:\nSince these functions almost invariably must mutate the IO or RNG, functions ending with ! are used to signify a mutation other than mutating the IO or advancing the RNG state.\nFor example, rand(x) mutates the RNG, whereas rand!(x) mutates both the RNG and x; similarly, read(io) mutates io, whereas read!(io, x) mutates both arguments."},{"title":"Avoid strange type Unions","page":"Style Guide","location":"manual/style-guide.html#Avoid-strange-type-Unions","category":"section","text":"Types such as Union{Function,AbstractString} are often a sign that some design could be cleaner."},{"title":"Avoid elaborate container types","page":"Style Guide","location":"manual/style-guide.html#Avoid-elaborate-container-types","category":"section","text":"It is usually not much help to construct arrays like the following:\n\na = Vector{Union{Int,AbstractString,Tuple,Array}}(undef, n)\n\nIn this case Vector{Any}(undef, n) is better. It is also more helpful to the compiler to annotate specific\nuses (e.g. a[i]::Int) than to try to pack many alternatives into one type."},{"title":"Prefer exported methods over direct field access","page":"Style Guide","location":"manual/style-guide.html#Prefer-exported-methods-over-direct-field-access","category":"section","text":"Idiomatic Julia code should generally treat a module's exported methods as the\ninterface to its types. An object's fields are generally considered\nimplementation details and user code should only access them directly if this\nis stated to be the API. This has several benefits:\n\nPackage developers are freer to change the implementation without breaking\nuser code.\nMethods can be passed to higher-order constructs like map (e.g.\nmap(imag, zs)) rather than [z.im for z in zs]).\nMethods can be defined on abstract types.\nMethods can describe a conceptual operation that can be shared across\ndisparate types (e.g. real(z) works on Complex numbers or Quaternions).\n\nJulia's dispatch system encourages this style because play(x::MyType) only\ndefines the play method on that particular type, leaving other types to\nhave their own implementation.\n\nSimilarly, non-exported functions are typically internal and subject to change,\nunless the documentations states otherwise. Names sometimes are given a _ prefix\n(or suffix) to further suggest that something is \"internal\" or an\nimplementation-detail, but it is not a rule.\n\nCounter-examples to this rule include NamedTuple, RegexMatch, StatStruct."},{"title":"Use naming conventions consistent with Julia base/","page":"Style Guide","location":"manual/style-guide.html#Use-naming-conventions-consistent-with-Julia-base/","category":"section","text":"modules and type names use capitalization and camel case: module SparseArrays, struct UnitRange.\nconstants use all uppercase and underscores (LOAD_PATH, VERSION).\nwhile anything not marked with public or export is considered internal, a prefix of\n_ also indicates that an object is not intended for public use.\nfunctions mutating at least one of their arguments end in !.\nconciseness is valued, but avoid abbreviation (indexin rather than indxin) as\nit becomes difficult to remember whether and how particular words are abbreviated.\n\nIf a function name requires multiple words, consider whether it might represent more than one\nconcept and might be better split into pieces.\n\nFunction names should be written in snake case (minimum, count_zeros, escape_string).\nBase often breaks this convention by squashing words together (splitpath, readeach) but this style is not recommended for packages."},{"title":"Write functions with argument ordering similar to Julia Base","page":"Style Guide","location":"manual/style-guide.html#Write-functions-with-argument-ordering-similar-to-Julia-Base","category":"section","text":"As a general rule, the Base library uses the following order of arguments to functions,\nas applicable:\n\nFunction argument.\nPutting a function argument first permits the use of do blocks for passing\nmultiline anonymous functions.\nI/O stream.\nSpecifying the IO object first permits passing the function to functions such as\nsprint, e.g. sprint(show, x).\nInput being mutated.\nFor example, in fill!(x, v), x is the object being mutated and it\nappears before the value to be inserted into x.\nType.\nPassing a type typically means that the output will have the given type.\nIn parse(Int, \"1\"), the type comes before the string to parse.\nThere are many such examples where the type appears first, but it's useful to note that\nin read(io, String), the IO argument appears before the type, which is\nin keeping with the order outlined here.\nInput not being mutated.\nIn fill!(x, v), v is not being mutated and it comes after x.\nKey.\nFor associative collections, this is the key of the key-value pair(s).\nFor other indexed collections, this is the index.\nValue.\nFor associative collections, this is the value of the key-value pair(s).\nIn cases like fill!(x, v), this is v.\nEverything else.\nAny other arguments.\nVarargs.\nThis refers to arguments that can be listed indefinitely at the end of a function call.\nFor example, in Matrix{T}(undef, dims), the dimensions can be given as a\nTuple, e.g. Matrix{T}(undef, (1,2)), or as Varargs,\ne.g. Matrix{T}(undef, 1, 2).\nKeyword arguments.\nIn Julia keyword arguments have to come last anyway in function definitions; they're\nlisted here for the sake of completeness.\n\nThe vast majority of functions will not take every kind of argument listed above; the\nnumbers merely denote the precedence that should be used for any applicable arguments\nto a function.\n\nThere are of course a few exceptions.\nFor example, in convert, the type should always come first.\nIn setindex!, the value comes before the indices so that the indices can be\nprovided as varargs.\n\nWhen designing APIs, adhering to this general order as much as possible is likely to give\nusers of your functions a more consistent experience."},{"title":"Don't overuse try-catch","page":"Style Guide","location":"manual/style-guide.html#Don't-overuse-try-catch","category":"section","text":"It is better to avoid errors than to rely on catching them."},{"title":"Don't parenthesize conditions","page":"Style Guide","location":"manual/style-guide.html#Don't-parenthesize-conditions","category":"section","text":"Julia doesn't require parens around conditions in if and while. Write:\n\nif a == b\n\ninstead of:\n\nif (a == b)"},{"title":"Don't overuse ...","page":"Style Guide","location":"manual/style-guide.html#Don't-overuse-...","category":"section","text":"Splicing function arguments can be addictive. Instead of [a..., b...], use simply [a; b],\nwhich already concatenates arrays. collect(a) is better than [a...], but since a\nis already iterable it is often even better to leave it alone, and not convert it to an array."},{"title":"Ensure constructors return an instance of their own type","page":"Style Guide","location":"manual/style-guide.html#Ensure-constructors-return-an-instance-of-their-own-type","category":"section","text":"When a method T(x) is called on a type T, it is generally expected to return a value of type T.\nDefining a constructor that returns an unexpected type can lead to confusing and unpredictable behavior:\n\njulia> struct Foo{T}\n           x::T\n       end\n\njulia> Base.Float64(foo::Foo) = Foo(Float64(foo.x))  # Do not define methods like this\n\njulia> Float64(Foo(3))  # Should return `Float64`\nFoo{Float64}(3.0)\n\njulia> Foo{Int}(x) = Foo{Float64}(x)  # Do not define methods like this\n\njulia> Foo{Int}(3)  # Should return `Foo{Int}`\nFoo{Float64}(3.0)\n\nTo maintain code clarity and ensure type consistency, always design constructors to return an instance of the type they are supposed to construct."},{"title":"Don't use unnecessary static parameters","page":"Style Guide","location":"manual/style-guide.html#Don't-use-unnecessary-static-parameters","category":"section","text":"A function signature:\n\nfoo(x::T) where {T<:Real} = ...\n\nshould be written as:\n\nfoo(x::Real) = ...\n\ninstead, especially if T is not used in the function body. Even if T is used, it can be replaced\nwith typeof(x) if convenient. There is no performance difference. Note that this is\nnot a general caution against static parameters, just against uses where they are not needed.\n\nNote also that container types, specifically may need type parameters in function calls. See the\nFAQ Avoid fields with abstract containers for more information."},{"title":"Avoid confusion about whether something is an instance or a type","page":"Style Guide","location":"manual/style-guide.html#Avoid-confusion-about-whether-something-is-an-instance-or-a-type","category":"section","text":"Sets of definitions like the following are confusing:\n\nfoo(::Type{MyType}) = ...\nfoo(::MyType) = foo(MyType)\n\nDecide whether the concept in question will be written as MyType or MyType(), and stick to\nit.\n\nThe preferred style is to use instances by default, and only add methods involving Type{MyType}\nlater if they become necessary to solve some problems.\n\nIf a type is effectively an enumeration, it should be defined as a single (ideally immutable struct or primitive)\ntype, with the enumeration values being instances of it. Constructors and conversions can check\nwhether values are valid. This design is preferred over making the enumeration an abstract type,\nwith the \"values\" as subtypes."},{"title":"Don't overuse macros","page":"Style Guide","location":"manual/style-guide.html#Don't-overuse-macros","category":"section","text":"Be aware of when a macro could really be a function instead.\n\nCalling eval inside a macro is a particularly dangerous warning sign; it means the\nmacro will only work when called at the top level. If such a macro is written as a function instead,\nit will naturally have access to the run-time values it needs."},{"title":"Don't expose unsafe operations at the interface level","page":"Style Guide","location":"manual/style-guide.html#Don't-expose-unsafe-operations-at-the-interface-level","category":"section","text":"If you have a type that uses a native pointer:\n\nmutable struct NativeType\n    p::Ptr{UInt8}\n    ...\nend\n\ndon't write definitions like the following:\n\ngetindex(x::NativeType, i) = unsafe_load(x.p, i)\n\nThe problem is that users of this type can write x[i] without realizing that the operation is\nunsafe, and then be susceptible to memory bugs.\n\nSuch a function should either check the operation to ensure it is safe, or have unsafe somewhere\nin its name to alert callers."},{"title":"Don't overload methods of base container types","page":"Style Guide","location":"manual/style-guide.html#Don't-overload-methods-of-base-container-types","category":"section","text":"It is possible to write definitions like the following:\n\nshow(io::IO, v::Vector{MyType}) = ...\n\nThis would provide custom showing of vectors with a specific new element type. While tempting,\nthis should be avoided. The trouble is that users will expect a well-known type like Vector()\nto behave in a certain way, and overly customizing its behavior can make it harder to work with."},{"title":"Avoid type piracy","page":"Style Guide","location":"manual/style-guide.html#avoid-type-piracy","category":"section","text":"\"Type piracy\" refers to the practice of extending or redefining methods in Base\nor other packages on types that you have not defined. In extreme cases, you can crash Julia\n(e.g. if your method extension or redefinition causes invalid input to be passed to a\nccall). Type piracy can complicate reasoning about code, and may introduce\nincompatibilities that are hard to predict and diagnose.\n\nAs an example, suppose you wanted to define multiplication on symbols in a module:\n\nmodule A\nimport Base.*\n*(x::Symbol, y::Symbol) = Symbol(x,y)\nend\n\nThe problem is that now any other module that uses Base.* will also see this definition.\nSince Symbol is defined in Base and is used by other modules, this can change the\nbehavior of unrelated code unexpectedly. There are several alternatives here, including\nusing a different function name, or wrapping the Symbols in another type that you define.\n\nSometimes, coupled packages may engage in type piracy to separate features from definitions,\nespecially when the packages were designed by collaborating authors, and when the\ndefinitions are reusable. For example, one package might provide some types useful for\nworking with colors; another package could define methods for those types that enable\nconversions between color spaces. Another example might be a package that acts as a thin\nwrapper for some C code, which another package might then pirate to implement a\nhigher-level, Julia-friendly API."},{"title":"Be careful with type equality","page":"Style Guide","location":"manual/style-guide.html#Be-careful-with-type-equality","category":"section","text":"You generally want to use isa and <: for testing types,\nnot ==. Checking types for exact equality typically only makes sense when comparing to a known\nconcrete type (e.g. T == Float64), or if you really, really know what you're doing."},{"title":"Don't write a trivial anonymous function x->f(x) for a named function f","page":"Style Guide","location":"manual/style-guide.html#Don't-write-a-trivial-anonymous-function-x-f(x)-for-a-named-function-f","category":"section","text":"Since higher-order functions are often called with anonymous functions, it is easy to conclude\nthat this is desirable or even necessary. But any function can be passed directly, without being\n\"wrapped\" in an anonymous function. Instead of writing map(x->f(x), a), write map(f, a)."},{"title":"Avoid using floats for numeric literals in generic code when possible","page":"Style Guide","location":"manual/style-guide.html#Avoid-using-floats-for-numeric-literals-in-generic-code-when-possible","category":"section","text":"If you write generic code which handles numbers, and which can be expected to run with many different\nnumeric type arguments, try using literals of a numeric type that will affect the arguments as\nlittle as possible through promotion.\n\nFor example,\n\njulia> f(x) = 2.0 * x\nf (generic function with 1 method)\n\njulia> f(1//2)\n1.0\n\njulia> f(1/2)\n1.0\n\njulia> f(1)\n2.0\n\nwhile\n\njulia> g(x) = 2 * x\ng (generic function with 1 method)\n\njulia> g(1//2)\n1//1\n\njulia> g(1/2)\n1.0\n\njulia> g(1)\n2\n\nAs you can see, the second version, where we used an Int literal, preserved the type of the\ninput argument, while the first didn't. This is because e.g. promote_type(Int, Float64) == Float64,\nand promotion happens with the multiplication. Similarly, Rational literals are less type disruptive\nthan Float64 literals, but more disruptive than Ints:\n\njulia> h(x) = 2//1 * x\nh (generic function with 1 method)\n\njulia> h(1//2)\n1//1\n\njulia> h(1/2)\n1.0\n\njulia> h(1)\n2//1\n\nThus, use Int literals when possible, with Rational{Int} for literal non-integer numbers,\nin order to make it easier to use your code."},{"title":"Asynchronous Programming","page":"Asynchronous Programming","location":"manual/asynchronous-programming.html#man-asynchronous","category":"section","text":"When a program needs to interact with the outside world, for example communicating\nwith another machine over the internet, operations in the program may need to\nhappen in an unpredictable order.\nSay your program needs to download a file. We would like to initiate the download\noperation, perform other operations while we wait for it to complete, and then\nresume the code that needs the downloaded file when it is available.\nThis sort of scenario falls in the domain of asynchronous programming, sometimes\nalso referred to as concurrent programming (since, conceptually, multiple things\nare happening at once).\n\nTo address these scenarios, Julia provides Tasks (also known by several other\nnames, such as symmetric coroutines, lightweight threads, cooperative multitasking,\nor one-shot continuations).\nWhen a piece of computing work (in practice, executing a particular function) is designated as\na Task, it becomes possible to interrupt it by switching to another Task.\nThe original Task can later be resumed, at which point it will pick up right where it\nleft off. At first, this may seem similar to a function call. However there are two key differences.\nFirst, switching tasks does not use any space, so any number of task switches can occur without\nconsuming the call stack. Second, switching among tasks can occur in any order, unlike function\ncalls, where the called function must finish executing before control returns to the calling function."},{"title":"Basic Task operations","page":"Asynchronous Programming","location":"manual/asynchronous-programming.html#Basic-Task-operations","category":"section","text":"You can think of a Task as a handle to a unit of computational work to be performed.\nIt has a create-start-run-finish lifecycle.\nTasks are created by calling the Task constructor on a 0-argument function to run,\nor using the @task macro:\n\njulia> t = @task begin; sleep(5); println(\"done\"); end\nTask (runnable) @0x00007f13a40c0eb0\n\n@task x is equivalent to Task(()->x).\n\nThis task will wait for five seconds, and then print done. However, it has not\nstarted running yet. We can run it whenever we're ready by calling schedule:\n\njulia> schedule(t);\n\nIf you try this in the REPL, you will see that schedule returns immediately.\nThat is because it simply adds t to an internal queue of tasks to run.\nThen, the REPL will print the next prompt and wait for more input.\nWaiting for keyboard input provides an opportunity for other tasks to run,\nso at that point t will start.\nt calls sleep, which sets a timer and stops execution.\nIf other tasks have been scheduled, they could run then.\nAfter five seconds, the timer fires and restarts t, and you will see done\nprinted. t is then finished.\n\nThe wait function blocks the calling task until some other task finishes.\nSo for example if you type\n\njulia> schedule(t); wait(t)\n\ninstead of only calling schedule, you will see a five second pause before\nthe next input prompt appears. That is because the REPL is waiting for t\nto finish before proceeding.\n\nIt is common to want to create a task and schedule it right away, so the\nmacro Threads.@spawn is provided for that purpose — Threads.@spawn x is\nequivalent to task = @task x; task.sticky = false; schedule(task)."},{"title":"Communicating with Channels","page":"Asynchronous Programming","location":"manual/asynchronous-programming.html#Communicating-with-Channels","category":"section","text":"In some problems,\nthe various pieces of required work are not naturally related by function calls; there is no obvious\n\"caller\" or \"callee\" among the jobs that need to be done. An example is the producer-consumer\nproblem, where one complex procedure is generating values and another complex procedure is consuming\nthem. The consumer cannot simply call a producer function to get a value, because the producer\nmay have more values to generate and so might not yet be ready to return. With tasks, the producer\nand consumer can both run as long as they need to, passing values back and forth as necessary.\n\nJulia provides a Channel mechanism for solving this problem.\nA Channel is a waitable first-in first-out queue which can have\nmultiple tasks reading from and writing to it.\n\nLet's define a producer task, which produces values via the put! call.\nTo consume values, we need to schedule the producer to run in a new task. A special Channel\nconstructor which accepts a 1-arg function as an argument can be used to run a task bound to a channel.\nWe can then take! values repeatedly from the channel object:\n\njulia> function producer(c::Channel)\n           put!(c, \"start\")\n           for n=1:4\n               put!(c, 2n)\n           end\n           put!(c, \"stop\")\n       end;\n\njulia> chnl = Channel(producer);\n\njulia> take!(chnl)\n\"start\"\n\njulia> take!(chnl)\n2\n\njulia> take!(chnl)\n4\n\njulia> take!(chnl)\n6\n\njulia> take!(chnl)\n8\n\njulia> take!(chnl)\n\"stop\"\n\nOne way to think of this behavior is that producer was able to return multiple times. Between\ncalls to put!, the producer's execution is suspended and the consumer has control.\n\nThe returned Channel can be used as an iterable object in a for loop, in which case the\nloop variable takes on all the produced values. The loop is terminated when the channel is closed.\n\njulia> for x in Channel(producer)\n           println(x)\n       end\nstart\n2\n4\n6\n8\nstop\n\nNote that we did not have to explicitly close the channel in the producer. This is because\nthe act of binding a Channel to a Task associates the open lifetime of\na channel with that of the bound task. The channel object is closed automatically when the task\nterminates. Multiple channels can be bound to a task, and vice-versa.\n\nWhile the Task constructor expects a 0-argument function, the Channel\nmethod that creates a task-bound channel expects a function that accepts a single argument of\ntype Channel. A common pattern is for the producer to be parameterized, in which case a partial\nfunction application is needed to create a 0 or 1 argument anonymous function.\n\nFor Task objects this can be done either directly or by use of a convenience macro:\n\nfunction mytask(myarg)\n    ...\nend\n\ntaskHdl = Task(() -> mytask(7))\n# or, equivalently\ntaskHdl = @task mytask(7)\n\nTo orchestrate more advanced work distribution patterns, bind and schedule\ncan be used in conjunction with Task and Channel\nconstructors to explicitly link a set of channels with a set of producer/consumer tasks."},{"title":"More on Channels","page":"Asynchronous Programming","location":"manual/asynchronous-programming.html#More-on-Channels","category":"section","text":"A channel can be visualized as a pipe, i.e., it has a write end and a read end:\n\nMultiple writers in different tasks can write to the same channel concurrently via put!\ncalls.\nMultiple readers in different tasks can read data concurrently via take! calls.\nAs an example:\n# Given Channels c1 and c2,\nc1 = Channel(32)\nc2 = Channel(32)\n\n# and a function `foo` which reads items from c1, processes the item read\n# and writes a result to c2,\nfunction foo()\n    while true\n        data = take!(c1)\n        [...]               # process data\n        put!(c2, result)    # write out result\n    end\nend\n\n# we can schedule `n` instances of `foo` to be active concurrently.\nfor _ in 1:n\n    errormonitor(Threads.@spawn foo())\nend\nChannels are created via the Channel{T}(sz) constructor. The channel will only hold objects\nof type T. If the type is not specified, the channel can hold objects of any type. sz refers\nto the maximum number of elements that can be held in the channel at any time. For example, Channel(32)\ncreates a channel that can hold a maximum of 32 objects of any type. A Channel{MyType}(64) can\nhold up to 64 objects of MyType at any time.\nIf a Channel is empty, readers (on a take! call) will block until data is available (see isempty).\nIf a Channel is full, writers (on a put! call) will block until space becomes available (see isfull).\nisready tests for the presence of any object in the channel, while wait\nwaits for an object to become available.\nNote that if another task is currently waiting to put! an object into a channel, a channel can have more items available than its capacity.\nA Channel is in an open state initially. This means that it can be read from and written to\nfreely via take! and put! calls. close closes a Channel.\nOn a closed Channel, put! will fail. For example:\njulia> c = Channel(2);\n\njulia> put!(c, 1) # `put!` on an open channel succeeds\n1\n\njulia> close(c);\n\njulia> put!(c, 2) # `put!` on a closed channel throws an exception.\nERROR: InvalidStateException: Channel is closed.\nStacktrace:\n[...]\ntake! and fetch (which retrieves but does not remove the value) on a closed\nchannel successfully return any existing values until it is emptied. Continuing the above example:\njulia> fetch(c) # Any number of `fetch` calls succeed.\n1\n\njulia> fetch(c)\n1\n\njulia> take!(c) # The first `take!` removes the value.\n1\n\njulia> take!(c) # No more data available on a closed channel.\nERROR: InvalidStateException: Channel is closed.\nStacktrace:\n[...]\n\nConsider a simple example using channels for inter-task communication. We start 4 tasks to process\ndata from a single jobs channel. Jobs, identified by an id (job_id), are written to the channel.\nEach task in this simulation reads a job_id, waits for a random amount of time and writes back\na tuple of job_id and the simulated time to the results channel. Finally all the results are\nprinted out.\n\njulia> const jobs = Channel{Int}(32);\n\njulia> const results = Channel{Tuple}(32);\n\njulia> function do_work()\n           for job_id in jobs\n               exec_time = rand()\n               sleep(exec_time)                # simulates elapsed time doing actual work\n                                               # typically performed externally.\n               put!(results, (job_id, exec_time))\n           end\n       end;\n\njulia> function make_jobs(n)\n           for i in 1:n\n               put!(jobs, i)\n           end\n       end;\n\njulia> n = 12;\n\njulia> errormonitor(Threads.@spawn make_jobs(n)); # feed the jobs channel with \"n\" jobs\n\njulia> for i in 1:4 # start 4 tasks to process requests in parallel\n           errormonitor(Threads.@spawn do_work())\n       end\n\njulia> @elapsed while n > 0 # print out results\n           job_id, exec_time = take!(results)\n           println(\"$job_id finished in $(round(exec_time; digits=2)) seconds\")\n           global n = n - 1\n       end\n4 finished in 0.22 seconds\n3 finished in 0.45 seconds\n1 finished in 0.5 seconds\n7 finished in 0.14 seconds\n2 finished in 0.78 seconds\n5 finished in 0.9 seconds\n9 finished in 0.36 seconds\n6 finished in 0.87 seconds\n8 finished in 0.79 seconds\n10 finished in 0.64 seconds\n12 finished in 0.5 seconds\n11 finished in 0.97 seconds\n0.029772311\n\nInstead of errormonitor(t), a more robust solution may be to use bind(results, t), as that will\nnot only log any unexpected failures, but also force the associated resources to close and propagate\nthe exception everywhere."},{"title":"More task operations","page":"Asynchronous Programming","location":"manual/asynchronous-programming.html#More-task-operations","category":"section","text":"Task operations are built on a low-level primitive called yieldto.\nyieldto(task, value) suspends the current task, switches to the specified task, and causes\nthat task's last yieldto call to return the specified value. Notice that yieldto\nis the only operation required to use task-style control flow; instead of calling and returning\nwe are always just switching to a different task. This is why this feature is also called \"symmetric\ncoroutines\"; each task is switched to and from using the same mechanism.\n\nyieldto is powerful, but most uses of tasks do not invoke it directly. Consider why\nthis might be. If you switch away from the current task, you will probably want to switch back\nto it at some point, but knowing when to switch back, and knowing which task has the responsibility\nof switching back, can require considerable coordination. For example, put! and take!\nare blocking operations, which, when used in the context of channels maintain state to remember\nwho the consumers are. Not needing to manually keep track of the consuming task is what makes put!\neasier to use than the low-level yieldto.\n\nIn addition to yieldto, a few other basic functions are needed to use tasks effectively.\n\ncurrent_task gets a reference to the currently-running task.\nistaskdone queries whether a task has exited.\nistaskstarted queries whether a task has run yet.\ntask_local_storage manipulates a key-value store specific to the current task."},{"title":"Tasks and events","page":"Asynchronous Programming","location":"manual/asynchronous-programming.html#Tasks-and-events","category":"section","text":"Most task switches occur as a result of waiting for events such as I/O requests, and are performed\nby a scheduler included in Julia Base. The scheduler maintains a queue of runnable tasks,\nand executes an event loop that restarts tasks based on external events such as message arrival.\n\nThe basic function for waiting for an event is wait. Several objects implement wait;\nfor example, given a Process object, wait will wait for it to exit. wait\nis often implicit; for example, a wait can happen inside a call to read\nto wait for data to be available.\n\nIn all of these cases, wait ultimately operates on a Condition object, which\nis in charge of queueing and restarting tasks. When a task calls wait on a Condition,\nthe task is marked as non-runnable, added to the condition's queue, and switches to the scheduler.\nThe scheduler will then pick another task to run, or block waiting for external events. If all\ngoes well, eventually an event handler will call notify on the condition, which causes\ntasks waiting for that condition to become runnable again.\n\nA task created explicitly by calling Task is initially not known to the scheduler. This\nallows you to manage tasks manually using yieldto if you wish. However, when such\na task waits for an event, it still gets restarted automatically when the event happens, as you\nwould expect."},{"title":"Inference","page":"Inference","location":"devdocs/inference.html#Inference","category":"section","text":""},{"title":"How inference works","page":"Inference","location":"devdocs/inference.html#How-inference-works","category":"section","text":"In Julia compiler, \"type inference\" refers to the process of deducing the types of later\nvalues from the types of input values. Julia's approach to inference has been described in\nthe blog posts below:\n\nShows a simplified implementation of the data-flow analysis algorithm, that Julia's type inference routine is based on.\nGives a high level view of inference with a focus on its inter-procedural convergence guarantee.\nExplains a refinement on the algorithm introduced in 2."},{"title":"Debugging compiler.jl","page":"Inference","location":"devdocs/inference.html#Debugging-compiler.jl","category":"section","text":"You can start a Julia session, edit compiler/*.jl (for example to\ninsert print statements), and then replace Core.Compiler in your\nrunning session by navigating to base and executing\ninclude(\"compiler/compiler.jl\"). This trick typically leads to much faster\ndevelopment than if you rebuild Julia for each change.\n\nAlternatively, you can use the Revise.jl\npackage to track the compiler changes by using the command\nRevise.track(Core.Compiler) at the beginning of your Julia session. As\nexplained in the Revise documentation,\nthe modifications to the compiler will be reflected when the modified files\nare saved.\n\nA convenient entry point into inference is typeinf_code. Here's a\ndemo running inference on convert(Int, UInt(1)):\n\n# Get the method\natypes = Tuple{Type{Int}, UInt}  # argument types\nmths = methods(convert, atypes)  # worth checking that there is only one\nm = first(mths)\n\n# Create variables needed to call `typeinf_code`\ninterp = Core.Compiler.NativeInterpreter()\nsparams = Core.svec()      # this particular method doesn't have type-parameters\nrun_optimizer = true       # run all inference optimizations\ntypes = Tuple{typeof(convert), atypes.parameters...} # Tuple{typeof(convert), Type{Int}, UInt}\nCore.Compiler.typeinf_code(interp, m, types, sparams, run_optimizer)\n\nIf your debugging adventures require a MethodInstance, you can look it up by\ncalling Core.Compiler.specialize_method using many of the variables above.\nA CodeInfo object may be obtained with\n\n# Returns the CodeInfo object for `convert(Int, ::UInt)`:\nci = (@code_typed convert(Int, UInt(1)))[1]"},{"title":"The inlining algorithm (inline_worthy)","page":"Inference","location":"devdocs/inference.html#The-inlining-algorithm-(inline_worthy)","category":"section","text":"Much of the hardest work for inlining runs in ssa_inlining_pass!.\nHowever, if your question is \"why didn't my function inline?\"\nthen you will most likely be interested in inline_worthy,\nwhich makes a decision to inline the function call or not.\n\ninline_worthy implements a cost-model, where \"cheap\" functions get\ninlined; more specifically, we inline functions if their anticipated\nrun-time is not large compared to the time it would take to\nissue a call to\nthem if they were not inlined. The cost-model is extremely simple and\nignores many important details: for example, all for loops are\nanalyzed as if they will be executed once, and the cost of an\nif...else...end includes the summed cost of all branches. It's also\nworth acknowledging that we currently lack a suite of functions\nsuitable for testing how well the cost model predicts the actual\nrun-time cost, although\nBaseBenchmarks\nprovides a great deal of indirect information about the successes and\nfailures of any modification to the inlining algorithm.\n\nThe foundation of the cost-model is a lookup table, implemented in\nadd_tfunc and its callers, that assigns an estimated cost (measured\nin CPU cycles) to each of Julia's intrinsic functions. These costs are\nbased on\nstandard ranges for common architectures\n(see\nAgner Fog's analysis\nfor more detail).\n\nWe supplement this low-level lookup table with a number of special\ncases. For example, an :invoke expression (a call for which all\ninput and output types were inferred in advance) is assigned a fixed\ncost (currently 20 cycles). In contrast, a :call expression, for\nfunctions other than intrinsics/builtins, indicates that the call will\nrequire dynamic dispatch, in which case we assign a cost set by\nParams.inline_nonleaf_penalty (currently set at 1000). Note\nthat this is not a \"first-principles\" estimate of the raw cost of\ndynamic dispatch, but a mere heuristic indicating that dynamic\ndispatch is extremely expensive.\n\nEach statement gets analyzed for its total cost in a function called\nstatement_cost. You can display the cost associated with each statement\nas follows:\n\njulia> Base.print_statement_costs(stdout, map, (typeof(sqrt), Tuple{Int},)) # map(sqrt, (2,))\nmap(f, t::Tuple{Any}) @ Base tuple.jl:358\n  0 1 ─ %1  = $(Expr(:boundscheck, true))::Bool\n  0 │   %2  =   builtin Base.getfield(_3, 1, %1)::Int64\n  1 │   %3  = intrinsic Base.sitofp(Float64, %2)::Float64\n  0 │   %4  = intrinsic Base.lt_float(%3, 0.0)::Bool\n  0 └──       goto #3 if not %4\n  0 2 ─          invoke Base.Math.throw_complex_domainerror(:sqrt::Symbol, %3::Float64)::Union{}\n  0 └──       unreachable\n 20 3 ─ %8  = intrinsic Base.Math.sqrt_llvm(%3)::Float64\n  0 └──       goto #4\n  0 4 ─       goto #5\n  0 5 ─ %11 =   builtin Core.tuple(%8)::Tuple{Float64}\n  0 └──       return %11\n\n\nThe line costs are in the left column. This includes the consequences of inlining and other forms of optimization."},{"title":"Functions","page":"Functions","location":"manual/functions.html#man-functions","category":"section","text":"In Julia, a function is an object that maps a tuple of argument values to a return value. Julia\nfunctions are not pure mathematical functions, because they can alter and be affected\nby the global state of the program. The basic syntax for defining functions in Julia is:\n\njulia> function f(x, y)\n           x + y\n       end\nf (generic function with 1 method)\n\nThis function accepts two arguments x and y and returns the value\nof the last expression evaluated, which is x + y.\n\nThere is a second, more terse syntax for defining a function in Julia. The traditional function\ndeclaration syntax demonstrated above is equivalent to the following compact \"assignment form\":\n\njulia> f(x, y) = x + y\nf (generic function with 1 method)\n\nIn the assignment form, the body of the function must be a single expression, although it can\nbe a compound expression (see Compound Expressions). Short, simple function definitions\nare common in Julia. The short function syntax is accordingly quite idiomatic, considerably reducing\nboth typing and visual noise.\n\nA function is called using the traditional parenthesis syntax:\n\njulia> f(2, 3)\n5\n\nWithout parentheses, the expression f refers to the function object, and can be passed around\nlike any other value:\n\njulia> g = f;\n\njulia> g(2, 3)\n5\n\nAs with variables, Unicode can also be used for function names:\n\njulia> ∑(x, y) = x + y\n∑ (generic function with 1 method)\n\njulia> ∑(2, 3)\n5"},{"title":"Argument Passing Behavior","page":"Functions","location":"manual/functions.html#man-argument-passing","category":"section","text":"Julia function arguments follow a convention sometimes called \"pass-by-sharing\", which means that\nvalues are not copied when they are passed to functions. Function arguments themselves act as\nnew variable bindings (new \"names\" that can refer to values), much like\nassignments argument_name = argument_value, so that the objects they refer to\nare identical to the passed values. Modifications to mutable values (such as Arrays) made within\na function will be visible to the caller. (This is the same behavior found in Scheme, most Lisps,\nPython, Ruby and Perl, among other dynamic languages.)\n\nFor example, in the function\n\nfunction f(x, y)\n    x[1] = 42    # mutates x\n    y = 7 + y    # new binding for y, no mutation\n    return y\nend\n\nThe statement x[1] = 42 mutates the object x, and hence this change will be visible in the array passed\nby the caller for this argument. On the other hand, the assignment y = 7 + y changes the binding (\"name\")\ny to refer to a new value 7 + y, rather than mutating the original object referred to by y,\nand hence does not change the corresponding argument passed by the caller. This can be seen if we call f(x, y):\n\njulia> a = [4, 5, 6]\n3-element Vector{Int64}:\n 4\n 5\n 6\n\njulia> b = 3\n3\n\njulia> f(a, b) # returns 7 + b == 10\n10\n\njulia> a  # a[1] is changed to 42 by f\n3-element Vector{Int64}:\n 42\n  5\n  6\n\njulia> b  # not changed\n3\n\nAs a common convention in Julia (not a syntactic requirement), such a function would\ntypically be named f!(x, y) rather than f(x, y), as a visual reminder at\nthe call site that at least one of the arguments (often the first one) is being mutated.\n\nwarning: Shared memory between arguments\nThe behavior of a mutating function can be unexpected when a mutated argument shares memory with another argument, a situation known as aliasing (e.g. when one is a view of the other).\nUnless the function docstring explicitly indicates that aliasing produces the expected result, it is the responsibility of the caller to ensure proper behavior on such inputs."},{"title":"Argument-type declarations","page":"Functions","location":"manual/functions.html#Argument-type-declarations","category":"section","text":"You can declare the types of function arguments by appending ::TypeName to the argument name, as usual for Type Declarations in Julia.\nFor example, the following function computes Fibonacci numbers recursively:\n\nfib(n::Integer) = n ≤ 2 ? one(n) : fib(n-1) + fib(n-2)\n\nand the ::Integer specification means that it will only be callable when n is a subtype of the abstract Integer type.\n\nArgument-type declarations normally have no impact on performance: regardless of what argument types (if any) are declared, Julia compiles a specialized version of the function for the actual argument types passed by the caller. For example, calling fib(1) will trigger the compilation of specialized version of fib optimized specifically for Int arguments, which is then re-used if fib(7) or fib(15) are called.  (There are rare exceptions when an argument-type declaration can trigger additional compiler specializations; see: Be aware of when Julia avoids specializing.)  The most common reasons to declare argument types in Julia are, instead:\n\nDispatch: As explained in Methods, you can have different versions (\"methods\") of a function for different argument types, in which case the argument types are used to determine which implementation is called for which arguments. For example, you might implement a completely different algorithm fib(x::Number) = ... that works for any Number type by using Binet's formula to extend it to non-integer values.\nCorrectness: Type declarations can be useful if your function only returns correct results for certain argument types. For example, if we omitted argument types and wrote fib(n) = n ≤ 2 ? one(n) : fib(n-1) + fib(n-2), then fib(1.5) would silently give us the nonsensical answer 1.0.\nClarity: Type declarations can serve as a form of documentation about the expected arguments.\n\nHowever, it is a common mistake to overly restrict the argument types, which can unnecessarily limit the applicability of the function and prevent it from being re-used in circumstances you did not anticipate. For example, the fib(n::Integer) function above works equally well for Int arguments (machine integers) and BigInt arbitrary-precision integers (see BigFloats and BigInts), which is especially useful because Fibonacci numbers grow exponentially rapidly and will quickly overflow any fixed-precision type like Int (see Overflow behavior). If we had declared our function as fib(n::Int), however, the application to BigInt would have been prevented for no reason. In general, you should use the most general applicable abstract types for arguments, and when in doubt, omit the argument types. You can always add argument-type specifications later if they become necessary, and you don't sacrifice performance or functionality by omitting them."},{"title":"The return Keyword","page":"Functions","location":"manual/functions.html#The-return-Keyword","category":"section","text":"The value returned by a function is the value of the last expression evaluated, which, by default,\nis the last expression in the body of the function definition. In the example function, f, from\nthe previous section this is the value of the expression x + y.\nAs an alternative, as in many other languages,\nthe return keyword causes a function to return immediately, providing\nan expression whose value is returned:\n\nfunction g(x, y)\n    return x * y\n    x + y\nend\n\nSince function definitions can be entered into interactive sessions, it is easy to compare these\ndefinitions:\n\njulia> f(x, y) = x + y\nf (generic function with 1 method)\n\njulia> function g(x, y)\n           return x * y\n           x + y\n       end\ng (generic function with 1 method)\n\njulia> f(2, 3)\n5\n\njulia> g(2, 3)\n6\n\nOf course, in a purely linear function body like g, the usage of return is pointless since\nthe expression x + y is never evaluated and we could simply make x * y the last expression\nin the function and omit the return. In conjunction with other control flow, however, return\nis of real use. Here, for example, is a function that computes the hypotenuse length of a right\ntriangle with sides of length x and y, avoiding overflow:\n\njulia> function hypot(x, y)\n           x = abs(x)\n           y = abs(y)\n           if x > y\n               r = y/x\n               return x*sqrt(1 + r*r)\n           end\n           if y == 0\n               return float(x)\n           end\n           r = x/y\n           return y*sqrt(1 + r*r)\n       end\nhypot (generic function with 1 method)\n\njulia> hypot(3, 4)\n5.0\n\nThere are three possible points of return from this function, returning the values of three different\nexpressions, depending on the values of x and y. The return on the last line could be omitted\nsince it is the last expression."},{"title":"Return type","page":"Functions","location":"manual/functions.html#man-functions-return-type","category":"section","text":"A return type can be specified in the function declaration using the :: operator. This converts\nthe return value to the specified type.\n\njulia> function g(x, y)::Int8\n           return x * y\n       end;\n\njulia> typeof(g(1, 2))\nInt8\n\nThis function will always return an Int8 regardless of the types of x and y.\nSee Type Declarations for more on return types.\n\nReturn type declarations are rarely used in Julia: in general, you should\ninstead write \"type-stable\" functions in which Julia's compiler can automatically\ninfer the return type. For more information, see the Performance Tips chapter."},{"title":"Returning nothing","page":"Functions","location":"manual/functions.html#Returning-nothing","category":"section","text":"For functions that do not need to return a value (functions used only for some side effects),\nthe Julia convention is to return the value nothing:\n\nfunction printx(x)\n    println(\"x = $x\")\n    return nothing\nend\n\nThis is a convention in the sense that nothing is not a Julia keyword\nbut only a singleton object of type Nothing.\nAlso, you may notice that the printx function example above is contrived,\nbecause println already returns nothing, so that the return line is redundant.\n\nThere are two possible shortened forms for the return nothing expression.\nOn the one hand, the return keyword implicitly returns nothing, so it can be used alone.\nOn the other hand, since functions implicitly return their last expression evaluated,\nnothing can be used alone when it's the last expression.\nThe preference for the expression return nothing as opposed to return or nothing\nalone is a matter of coding style."},{"title":"Operators Are Functions","page":"Functions","location":"manual/functions.html#Operators-Are-Functions","category":"section","text":"In Julia, most operators are just functions with support for special syntax. (The exceptions are\noperators with special evaluation semantics like && and ||. These operators cannot be functions\nsince Short-Circuit Evaluation requires that their operands are not evaluated before evaluation\nof the operator.) Accordingly, you can also apply them using parenthesized argument lists, just\nas you would any other function:\n\njulia> 1 + 2 + 3\n6\n\njulia> +(1, 2, 3)\n6\n\nThe infix form is exactly equivalent to the function application form – in fact the former is\nparsed to produce the function call internally. This also means that you can assign and pass around\noperators such as + and * just like you would with other function values:\n\njulia> f = +;\n\njulia> f(1, 2, 3)\n6\n\nUnder the name f, the function does not support infix notation, however."},{"title":"Operators With Special Names","page":"Functions","location":"manual/functions.html#Operators-With-Special-Names","category":"section","text":"A few special expressions correspond to calls to functions with non-obvious names. These are:\n\nExpression Calls\n[A B C ...] hcat\n[A; B; C; ...] vcat\n[A B; C D; ...] hvcat\n[A; B;; C; D;; ...] hvncat\nA' adjoint\nA[i] getindex\nA[i] = x setindex!\nA.n getproperty\nA.n = x setproperty!\n\nNote that expressions similar to [A; B;; C; D;; ...] but with more than two\nconsecutive ; also correspond to hvncat calls."},{"title":"Anonymous Functions","page":"Functions","location":"manual/functions.html#man-anonymous-functions","category":"section","text":"Functions in Julia are first-class objects:\nthey can be assigned to variables, and called using the standard function call syntax from the\nvariable they have been assigned to. They can be used as arguments, and they can be returned as\nvalues. They can also be created anonymously, without being given a name, using either of these\nsyntaxes:\n\njulia> x -> x^2 + 2x - 1\n#2 (generic function with 1 method)\n\njulia> function (x)\n           x^2 + 2x - 1\n       end\n#5 (generic function with 1 method)\n\nEach statement creates a function taking one argument x and returning the value of the polynomial x^2 + 2x - 1 at that value. Notice that the result is a generic function, but with a compiler-generated\nname based on consecutive numbering.\n\nThe primary use for anonymous functions is passing them to functions which take other functions\nas arguments. A classic example is map, which applies a function to each value of\nan array and returns a new array containing the resulting values:\n\njulia> map(round, [1.2, 3.5, 1.7])\n3-element Vector{Float64}:\n 1.0\n 4.0\n 2.0\n\nThis is fine if a named function effecting the transform already exists to pass as the first argument\nto map. Often, however, a ready-to-use, named function does not exist. In these\nsituations, the anonymous function construct allows easy creation of a single-use function object\nwithout needing a name:\n\njulia> map(x -> x^2 + 2x - 1, [1, 3, -1])\n3-element Vector{Int64}:\n  2\n 14\n -2\n\nAn anonymous function accepting multiple arguments can be written using the syntax (x,y,z)->2x+y-z.\n\nArgument-type declarations for anonymous functions work as for named functions, for example x::Integer->2x.\nThe return type of an anonymous function cannot be specified.\n\nA zero-argument anonymous function can be written as ()->2+2. The idea of a function with\nno arguments may seem strange, but is useful in cases where a result cannot (or should not)\nbe precomputed. For example, Julia has a zero-argument time function that returns\nthe current time in seconds, and thus seconds = ()->round(Int, time()) is an anonymous\nfunction that returns this time rounded to the nearest integer assigned to the variable\nseconds. Each time this anonymous function is called as seconds() the current time will\nbe calculated and returned."},{"title":"Tuples","page":"Functions","location":"manual/functions.html#Tuples","category":"section","text":"Julia has a built-in data structure called a tuple that is closely related to function\narguments and return values.\nA tuple is a fixed-length container that can hold any values, but cannot be modified\n(it is immutable).\nTuples are constructed with commas and parentheses, and can be accessed via indexing:\n\njulia> (1, 1+1)\n(1, 2)\n\njulia> (1,)\n(1,)\n\njulia> x = (0.0, \"hello\", 6*7)\n(0.0, \"hello\", 42)\n\njulia> x[2]\n\"hello\"\n\nNotice that a length-1 tuple must be written with a comma, (1,), since (1) would just\nbe a parenthesized value.\n() represents the empty (length-0) tuple."},{"title":"Named Tuples","page":"Functions","location":"manual/functions.html#Named-Tuples","category":"section","text":"The components of tuples can optionally be named, in which case a named tuple is\nconstructed:\n\njulia> x = (a=2, b=1+2)\n(a = 2, b = 3)\n\njulia> x[1]\n2\n\njulia> x.a\n2\n\nThe fields of named tuples can be accessed by name using dot syntax (x.a) in\naddition to the regular indexing syntax (x[1] or x[:a])."},{"title":"Destructuring Assignment and Multiple Return Values","page":"Functions","location":"manual/functions.html#destructuring-assignment","category":"section","text":"A comma-separated list of variables (optionally wrapped in parentheses) can appear on the\nleft side of an assignment: the value on the right side is destructured by iterating\nover and assigning to each variable in turn:\n\njulia> (a, b, c) = 1:3\n1:3\n\njulia> b\n2\n\nThe value on the right should be an iterator (see Iteration interface)\nat least as long as the number of variables on the left (any excess elements of the\niterator are ignored).\n\nThis can be used to return multiple values from functions by returning a tuple or\nother iterable value. For example, the following function returns two values:\n\njulia> function foo(a, b)\n           a+b, a*b\n       end\nfoo (generic function with 1 method)\n\nIf you call it in an interactive session without assigning the return value anywhere, you will\nsee the tuple returned:\n\njulia> foo(2, 3)\n(5, 6)\n\nDestructuring assignment extracts each value into a variable:\n\njulia> x, y = foo(2, 3)\n(5, 6)\n\njulia> x\n5\n\njulia> y\n6\n\nAnother common use is for swapping variables:\n\njulia> y, x = x, y\n(5, 6)\n\njulia> x\n6\n\njulia> y\n5\n\nIf only a subset of the elements of the iterator are required, a common convention is to assign ignored elements to a variable\nconsisting of only underscores _ (which is an otherwise invalid variable name, see\nAllowed Variable Names):\n\njulia> _, _, _, d = 1:10\n1:10\n\njulia> d\n4\n\nOther valid left-hand side expressions can be used as elements of the assignment list, which will call setindex! or setproperty!, or recursively destructure individual elements of the iterator:\n\njulia> X = zeros(3);\n\njulia> X[1], (a, b) = (1, (2, 3))\n(1, (2, 3))\n\njulia> X\n3-element Vector{Float64}:\n 1.0\n 0.0\n 0.0\n\njulia> a\n2\n\njulia> b\n3\n\ncompat: Julia 1.6\n... with assignment requires Julia 1.6\n\nIf the last symbol in the assignment list is suffixed by ... (known as slurping), then\nit will be assigned a collection or lazy iterator of the remaining elements of the\nright-hand side iterator:\n\njulia> a, b... = \"hello\"\n\"hello\"\n\njulia> a\n'h': ASCII/Unicode U+0068 (category Ll: Letter, lowercase)\n\njulia> b\n\"ello\"\n\njulia> a, b... = Iterators.map(abs2, 1:4)\nBase.Generator{UnitRange{Int64}, typeof(abs2)}(abs2, 1:4)\n\njulia> a\n1\n\njulia> b\nBase.Iterators.Rest{Base.Generator{UnitRange{Int64}, typeof(abs2)}, Int64}(Base.Generator{UnitRange{Int64}, typeof(abs2)}(abs2, 1:4), 1)\n\nSee Base.rest for details on the precise handling and customization for specific iterators.\n\ncompat: Julia 1.9\n... in non-final position of an assignment requires Julia 1.9\n\nSlurping in assignments can also occur in any other position. As opposed to slurping the end\nof a collection however, this will always be eager.\n\njulia> a, b..., c = 1:5\n1:5\n\njulia> a\n1\n\njulia> b\n3-element Vector{Int64}:\n 2\n 3\n 4\n\njulia> c\n5\n\njulia> front..., tail = \"Hi!\"\n\"Hi!\"\n\njulia> front\n\"Hi\"\n\njulia> tail\n'!': ASCII/Unicode U+0021 (category Po: Punctuation, other)\n\nThis is implemented in terms of the function Base.split_rest.\n\nNote that for variadic function definitions, slurping is still only allowed in final position.\nThis does not apply to single argument destructuring though,\nas that does not affect method dispatch:\n\njulia> f(x..., y) = x\nERROR: syntax: invalid \"...\" on non-final argument\nStacktrace:\n[...]\n\njulia> f((x..., y)) = x\nf (generic function with 1 method)\n\njulia> f((1, 2, 3))\n(1, 2)"},{"title":"Property destructuring","page":"Functions","location":"manual/functions.html#Property-destructuring","category":"section","text":"Instead of destructuring based on iteration, the right side of assignments can also be destructured using property names.\nThis follows the syntax for NamedTuples, and works by assigning to each variable on the left a\nproperty of the right side of the assignment with the same name using getproperty:\n\njulia> (; b, a) = (a=1, b=2, c=3)\n(a = 1, b = 2, c = 3)\n\njulia> a\n1\n\njulia> b\n2"},{"title":"Argument destructuring","page":"Functions","location":"manual/functions.html#man-argument-destructuring","category":"section","text":"The destructuring feature can also be used within a function argument.\nIf a function argument name is written as a tuple (e.g. (x, y)) instead of just\na symbol, then an assignment (x, y) = argument will be inserted for you:\n\njulia> minmax(x, y) = (y < x) ? (y, x) : (x, y)\nminmax (generic function with 1 method)\n\njulia> gap((min, max)) = max - min\ngap (generic function with 1 method)\n\njulia> gap(minmax(10, 2))\n8\n\nNotice the extra set of parentheses in the definition of gap. Without those, gap\nwould be a two-argument function, and this example would not work.\n\nSimilarly, property destructuring can also be used for function arguments:\n\njulia> foo((; x, y)) = x + y\nfoo (generic function with 1 method)\n\njulia> foo((x=1, y=2))\n3\n\njulia> struct A\n           x\n           y\n       end\n\njulia> foo(A(3, 4))\n7\n\nFor anonymous functions, destructuring a single argument requires an extra comma:\n\njulia> map(((x, y),) -> x + y, [(1, 2), (3, 4)])\n2-element Vector{Int64}:\n 3\n 7"},{"title":"Varargs Functions","page":"Functions","location":"manual/functions.html#Varargs-Functions","category":"section","text":"It is often convenient to be able to write functions taking an arbitrary number of arguments.\nSuch functions are traditionally known as \"varargs\" functions, which is short for \"variable number\nof arguments\". You can define a varargs function by following the last positional argument with an ellipsis:\n\njulia> bar(a, b, x...) = (a, b, x)\nbar (generic function with 1 method)\n\nThe variables a and b are bound to the first two argument values as usual, and the variable\nx is bound to an iterable collection of the zero or more values passed to bar after its first\ntwo arguments:\n\njulia> bar(1, 2)\n(1, 2, ())\n\njulia> bar(1, 2, 3)\n(1, 2, (3,))\n\njulia> bar(1, 2, 3, 4)\n(1, 2, (3, 4))\n\njulia> bar(1, 2, 3, 4, 5, 6)\n(1, 2, (3, 4, 5, 6))\n\nIn all these cases, x is bound to a tuple of the trailing values passed to bar.\n\nIt is possible to constrain the number of values passed as a variable argument; this will be discussed\nlater in Parametrically-constrained Varargs methods.\n\nOn the flip side, it is often handy to \"splat\" the values contained in an iterable collection\ninto a function call as individual arguments. To do this, one also uses ... but in the function\ncall instead:\n\njulia> x = (3, 4)\n(3, 4)\n\njulia> bar(1, 2, x...)\n(1, 2, (3, 4))\n\nIn this case a tuple of values is spliced into a varargs call precisely where the variable number\nof arguments go. This need not be the case, however:\n\njulia> x = (2, 3, 4)\n(2, 3, 4)\n\njulia> bar(1, x...)\n(1, 2, (3, 4))\n\njulia> x = (1, 2, 3, 4)\n(1, 2, 3, 4)\n\njulia> bar(x...)\n(1, 2, (3, 4))\n\nFurthermore, the iterable object splatted into a function call need not be a tuple:\n\njulia> x = [3, 4]\n2-element Vector{Int64}:\n 3\n 4\n\njulia> bar(1, 2, x...)\n(1, 2, (3, 4))\n\njulia> x = [1, 2, 3, 4]\n4-element Vector{Int64}:\n 1\n 2\n 3\n 4\n\njulia> bar(x...)\n(1, 2, (3, 4))\n\nAlso, the function that arguments are splatted into need not be a varargs function (although it\noften is):\n\njulia> baz(a, b) = a + b;\n\njulia> args = [1, 2]\n2-element Vector{Int64}:\n 1\n 2\n\njulia> baz(args...)\n3\n\njulia> args = [1, 2, 3]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> baz(args...)\nERROR: MethodError: no method matching baz(::Int64, ::Int64, ::Int64)\nThe function `baz` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  baz(::Any, ::Any)\n   @ Main none:1\n\nStacktrace:\n[...]\n\nAs you can see, if the wrong number of elements are in the splatted container, then the function\ncall will fail, just as it would if too many arguments were given explicitly."},{"title":"Optional Arguments","page":"Functions","location":"manual/functions.html#Optional-Arguments","category":"section","text":"It is often possible to provide sensible default values for function arguments.\nThis can save users from having to pass every argument on every call.\nFor example, the function Date(y, [m, d])\nfrom Dates module constructs a Date type for a given year y, month m and day d.\nHowever, m and d arguments are optional and their default value is 1.\nThis behavior can be expressed concisely as:\n\njulia> using Dates\n\njulia> function date(y::Int64, m::Int64=1, d::Int64=1)\n           err = Dates.validargs(Date, y, m, d)\n           err === nothing || throw(err)\n           return Date(Dates.UTD(Dates.totaldays(y, m, d)))\n       end\ndate (generic function with 3 methods)\n\nObserve, that this definition calls another method of the Date function that takes one argument\nof type UTInstant{Day}.\n\nWith this definition, the function can be called with either one, two or three arguments, and\n1 is automatically passed when only one or two of the arguments are specified:\n\njulia> date(2000, 12, 12)\n2000-12-12\n\njulia> date(2000, 12)\n2000-12-01\n\njulia> date(2000)\n2000-01-01\n\nOptional arguments are actually just a convenient syntax for writing multiple method definitions\nwith different numbers of arguments (see Note on Optional and keyword Arguments).\nThis can be checked for our date function example by calling the methods function:\n\njulia> methods(date)\n# 3 methods for generic function \"date\" from Main:\n [1] date(y::Int64, m::Int64, d::Int64)\n     @ REPL[2]:1\n [2] date(y::Int64, m::Int64)\n     @ REPL[2]:1\n [3] date(y::Int64)\n     @ REPL[2]:1"},{"title":"Keyword Arguments","page":"Functions","location":"manual/functions.html#Keyword-Arguments","category":"section","text":"Some functions need a large number of arguments, or have a large number of behaviors. Remembering\nhow to call such functions can be difficult. Keyword arguments can make these complex interfaces\neasier to use and extend by allowing arguments to be identified by name instead of only by position.\n\nFor example, consider a function plot that plots a line. This function might have many options,\nfor controlling line style, width, color, and so on. If it accepts keyword arguments, a possible\ncall might look like plot(x, y, width=2), where we have chosen to specify only line width. Notice\nthat this serves two purposes. The call is easier to read, since we can label an argument with\nits meaning. It also becomes possible to pass any subset of a large number of arguments, in any\norder.\n\nFunctions with keyword arguments are defined using a semicolon in the signature:\n\nfunction plot(x, y; style=\"solid\", width=1, color=\"black\")\n    ###\nend\n\nWhen the function is called, the semicolon is optional: one can either call plot(x, y, width=2)\nor plot(x, y; width=2), but the former style is more common. An explicit semicolon is required\nonly for passing varargs or computed keywords as described below.\n\nKeyword argument default values are evaluated only when necessary (when a corresponding keyword\nargument is not passed), and in left-to-right order. Therefore default expressions may refer to\nprior keyword arguments.\n\nThe types of keyword arguments can be made explicit as follows:\n\nfunction f(; x::Int=1)\n    ###\nend\n\nKeyword arguments can also be used in varargs functions:\n\nfunction plot(x...; style=\"solid\")\n    ###\nend\n\nExtra keyword arguments can be collected using ..., as in varargs functions:\n\nfunction f(x; y=0, kwargs...)\n    ###\nend\n\nInside f, kwargs will be an immutable key-value iterator over a named tuple.\nNamed tuples (as well as dictionaries with keys of Symbol, and other iterators\nyielding two-value collections with symbol as first values) can be passed as\nkeyword arguments using a semicolon in a call, e.g. f(x, z=1; kwargs...).\n\nIf a keyword argument is not assigned a default value in the method definition,\nthen it is required: an UndefKeywordError exception will be thrown\nif the caller does not assign it a value:\n\nfunction f(x; y)\n    ###\nend\nf(3, y=5) # ok, y is assigned\nf(3)      # throws UndefKeywordError(:y)\n\nOne can also pass key => value expressions after a semicolon. For example, plot(x, y; :width => 2)\nis equivalent to plot(x, y, width=2). This is useful in situations where the keyword name is computed\nat runtime.\n\nWhen a bare identifier or dot expression occurs after a semicolon, the keyword argument name is\nimplied by the identifier or field name. For example plot(x, y; width) is equivalent to\nplot(x, y; width=width) and plot(x, y; options.width) is equivalent to plot(x, y; width=options.width).\n\nThe nature of keyword arguments makes it possible to specify the same argument more than once.\nFor example, in the call plot(x, y; options..., width=2) it is possible that the options structure\nalso contains a value for width. In such a case the rightmost occurrence takes precedence; in\nthis example, width is certain to have the value 2. However, explicitly specifying the same keyword\nargument multiple times, for example plot(x, y, width=2, width=3), is not allowed and results in\na syntax error."},{"title":"Evaluation Scope of Default Values","page":"Functions","location":"manual/functions.html#Evaluation-Scope-of-Default-Values","category":"section","text":"When optional and keyword argument default expressions are evaluated, only previous arguments are in\nscope.\nFor example, given this definition:\n\nfunction f(x, a=b, b=1)\n    ###\nend\n\nthe b in a=b refers to a b in an outer scope, not the subsequent argument b."},{"title":"Do-Block Syntax for Function Arguments","page":"Functions","location":"manual/functions.html#Do-Block-Syntax-for-Function-Arguments","category":"section","text":"Passing functions as arguments to other functions is a powerful technique, but the syntax for\nit is not always convenient. Such calls are especially awkward to write when the function argument\nrequires multiple lines. As an example, consider calling map on a function with several\ncases:\n\nmap(x->begin\n           if x < 0 && iseven(x)\n               return 0\n           elseif x == 0\n               return 1\n           else\n               return x\n           end\n       end,\n    [A, B, C])\n\nJulia provides a reserved word do for rewriting this code more clearly:\n\nmap([A, B, C]) do x\n    if x < 0 && iseven(x)\n        return 0\n    elseif x == 0\n        return 1\n    else\n        return x\n    end\nend\n\nThe do x syntax creates an anonymous function with argument x and passes\nthe anonymous function as the first argument\nto the \"outer\" function - map in this example.\nSimilarly, do a,b would create a two-argument anonymous function. Note that do (a,b) would create a one-argument anonymous function,\nwhose argument is a tuple to be deconstructed. A plain do would declare that what follows is an anonymous function of the form () -> ....\n\nHow these arguments are initialized depends on the \"outer\" function; here, map will\nsequentially set x to A, B, C, calling the anonymous function on each, just as would happen\nin the syntax map(func, [A, B, C]).\n\nThis syntax makes it easier to use functions to effectively extend the language, since calls look\nlike normal code blocks. There are many possible uses quite different from map, such\nas managing system state. For example, there is a version of open that runs code ensuring\nthat the opened file is eventually closed:\n\nopen(\"outfile\", \"w\") do io\n    write(io, data)\nend\n\nThis is accomplished by the following definition:\n\nfunction open(f::Function, args...)\n    io = open(args...)\n    try\n        f(io)\n    finally\n        close(io)\n    end\nend\n\nHere, open first opens the file for writing and then passes the resulting output stream\nto the anonymous function you defined in the do ... end block. After your function exits, open\nwill make sure that the stream is properly closed, regardless of whether your function exited\nnormally or threw an exception. (The try/finally construct will be described in Control Flow.)\n\nWith the do block syntax, it helps to check the documentation or implementation to know how\nthe arguments of the user function are initialized.\n\nA do block, like any other inner function, can \"capture\" variables from its\nenclosing scope. For example, the variable data in the above example of\nopen...do is captured from the outer scope. Captured variables\ncan create performance challenges as discussed in performance tips."},{"title":"Function composition and piping","page":"Functions","location":"manual/functions.html#Function-composition-and-piping","category":"section","text":"Functions in Julia can be combined by composing or piping (chaining) them together.\n\nFunction composition is when you combine functions together and apply the resulting composition to arguments.\nYou use the function composition operator (∘) to compose the functions, so (f ∘ g)(args...; kw...) is the same as f(g(args...; kw...)).\n\nYou can type the composition operator at the REPL and suitably-configured editors using \\circ<tab>.\n\nFor example, the sqrt and + functions can be composed like this:\n\njulia> (sqrt ∘ +)(3, 6)\n3.0\n\nThis adds the numbers first, then finds the square root of the result.\n\nThe next example composes three functions and maps the result over an array of strings:\n\njulia> map(first ∘ reverse ∘ uppercase, split(\"you can compose functions like this\"))\n6-element Vector{Char}:\n 'U': ASCII/Unicode U+0055 (category Lu: Letter, uppercase)\n 'N': ASCII/Unicode U+004E (category Lu: Letter, uppercase)\n 'E': ASCII/Unicode U+0045 (category Lu: Letter, uppercase)\n 'S': ASCII/Unicode U+0053 (category Lu: Letter, uppercase)\n 'E': ASCII/Unicode U+0045 (category Lu: Letter, uppercase)\n 'S': ASCII/Unicode U+0053 (category Lu: Letter, uppercase)\n\nFunction chaining (sometimes called \"piping\" or \"using a pipe\" to send data to a subsequent function) is when you apply a function to the previous function's output:\n\njulia> 1:10 |> sum |> sqrt\n7.416198487095663\n\nHere, the total produced by sum is passed to the sqrt function. The equivalent composition would be:\n\njulia> (sqrt ∘ sum)(1:10)\n7.416198487095663\n\nThe pipe operator can also be used with broadcasting, as .|>, to provide a useful combination of the chaining/piping and dot vectorization syntax (described below).\n\njulia> [\"a\", \"list\", \"of\", \"strings\"] .|> [uppercase, reverse, titlecase, length]\n4-element Vector{Any}:\n  \"A\"\n  \"tsil\"\n  \"Of\"\n 7\n\nWhen combining pipes with anonymous functions, parentheses must be used if subsequent pipes are not to be parsed as part of the anonymous function's body. Compare:\n\njulia> 1:3 .|> (x -> x^2) |> sum |> sqrt\n3.7416573867739413\n\njulia> 1:3 .|> x -> x^2 |> sum |> sqrt\n3-element Vector{Float64}:\n 1.0\n 2.0\n 3.0"},{"title":"Dot Syntax for Vectorizing Functions","page":"Functions","location":"manual/functions.html#man-vectorized","category":"section","text":"In technical-computing languages, it is common to have \"vectorized\" versions of functions, which\nsimply apply a given function f(x) to each element of an array A to yield a new array via\nf(A). This kind of syntax is convenient for data processing, but in other languages vectorization\nis also often required for performance: if loops are slow, the \"vectorized\" version of a function\ncan call fast library code written in a low-level language. In Julia, vectorized functions are\nnot required for performance, and indeed it is often beneficial to write your own loops (see\nPerformance Tips), but they can still be convenient. Therefore, any Julia function\nf can be applied elementwise to any array (or other collection) with the syntax f.(A).\nFor example, sin can be applied to all elements in the vector A like so:\n\njulia> A = [1.0, 2.0, 3.0]\n3-element Vector{Float64}:\n 1.0\n 2.0\n 3.0\n\njulia> sin.(A)\n3-element Vector{Float64}:\n 0.8414709848078965\n 0.9092974268256817\n 0.1411200080598672\n\nOf course, you can omit the dot if you write a specialized \"vector\" method of f, e.g. via f(A::AbstractArray) = map(f, A),\nand this is just as efficient as f.(A). The advantage of the f.(A) syntax is that which functions are vectorizable need not be decided upon\nin advance by the library writer.\n\nMore generally, f.(args...) is actually equivalent to broadcast(f, args...), which allows\nyou to operate on multiple arrays (even of different shapes), or a mix of arrays and scalars (see\nBroadcasting). For example, if you have f(x, y) = 3x + 4y, then f.(pi, A) will return\na new array consisting of f(pi,a) for each a in A, and f.(vector1, vector2) will return\na new vector consisting of f(vector1[i], vector2[i]) for each index i (throwing an exception\nif the vectors have different length).\n\njulia> f(x, y) = 3x + 4y;\n\njulia> A = [1.0, 2.0, 3.0];\n\njulia> B = [4.0, 5.0, 6.0];\n\njulia> f.(pi, A)\n3-element Vector{Float64}:\n 13.42477796076938\n 17.42477796076938\n 21.42477796076938\n\njulia> f.(A, B)\n3-element Vector{Float64}:\n 19.0\n 26.0\n 33.0\n\nKeyword arguments are not broadcasted over, but are simply passed through to each call of\nthe function. For example, round.(x, digits=3) is equivalent to broadcast(x -> round(x, digits=3), x).\n\nMoreover, nested f.(args...) calls are fused into a single broadcast loop. For example,\nsin.(cos.(X)) is equivalent to broadcast(x -> sin(cos(x)), X), similar to [sin(cos(x)) for x in X]:\nthere is only a single loop over X, and a single array is allocated for the result. [In contrast,\nsin(cos(X)) in a typical \"vectorized\" language would first allocate one temporary array for\ntmp=cos(X), and then compute sin(tmp) in a separate loop, allocating a second array.] This\nloop fusion is not a compiler optimization that may or may not occur, it is a syntactic guarantee\nwhenever nested f.(args...) calls are encountered. Technically, the fusion stops as soon as\na \"non-dot\" function call is encountered; for example, in sin.(sort(cos.(X))) the sin and cos\nloops cannot be merged because of the intervening sort function.\n\nFinally, the maximum efficiency is typically achieved when the output array of a vectorized operation\nis pre-allocated, so that repeated calls do not allocate new arrays over and over again for\nthe results (see Pre-allocating outputs). A convenient syntax for this is X .= ..., which\nis equivalent to broadcast!(identity, X, ...) except that, as above, the broadcast! loop is\nfused with any nested \"dot\" calls. For example, X .= sin.(Y) is equivalent to broadcast!(sin, X, Y),\noverwriting X with sin.(Y) in-place. If the left-hand side is an array-indexing expression,\ne.g. X[begin+1:end] .= sin.(Y), then it translates to broadcast! on a view, e.g.\nbroadcast!(sin, view(X, firstindex(X)+1:lastindex(X)), Y),\nso that the left-hand side is updated in-place.\n\nSince adding dots to many operations and function calls in an expression\ncan be tedious and lead to code that is difficult to read, the macro\n@. is provided to convert every function call,\noperation, and assignment in an expression into the \"dotted\" version.\n\njulia> Y = [1.0, 2.0, 3.0, 4.0];\n\njulia> X = similar(Y); # pre-allocate output array\n\njulia> @. X = sin(cos(Y)) # equivalent to X .= sin.(cos.(Y))\n4-element Vector{Float64}:\n  0.5143952585235492\n -0.4042391538522658\n -0.8360218615377305\n -0.6080830096407656\n\nBinary (or unary) operators like .+ are handled with the same mechanism:\nthey are equivalent to broadcast calls and are fused with other nested \"dot\" calls.\n X .+= Y etcetera is equivalent to X .= X .+ Y and results in a fused in-place assignment;\n see also dot operators.\n\nYou can also combine dot operations with function chaining using |>, as in this example:\n\njulia> 1:5 .|> [x->x^2, inv, x->2*x, -, isodd]\n5-element Vector{Real}:\n    1\n    0.5\n    6\n   -4\n true\n\nAll functions in the fused broadcast are always called for every element of the result. Thus X .+ σ .* randn.() will add a mask of independent and identically sampled random values to each element of the array X, but X .+ σ .* randn() will add the same random sample to each element. In cases where the fused computation is constant along one or more axes of the broadcast iteration, it may be possible to leverage a space-time tradeoff and allocate intermediate values to reduce the number of computations. See more at performance tips."},{"title":"Further Reading","page":"Functions","location":"manual/functions.html#Further-Reading","category":"section","text":"We should mention here that this is far from a complete picture of defining functions. Julia has\na sophisticated type system and allows multiple dispatch on argument types. None of the examples\ngiven here provide any type annotations on their arguments, meaning that they are applicable to\nall types of arguments. The type system is described in Types and defining a function\nin terms of methods chosen by multiple dispatch on run-time argument types is described in Methods."},{"title":"Types","page":"Types","location":"manual/types.html#man-types","category":"section","text":"Type systems have traditionally fallen into two quite different camps: static type systems, where\nevery program expression must have a type computable before the execution of the program, and\ndynamic type systems, where nothing is known about types until run time, when the actual values\nmanipulated by the program are available. Object orientation allows some flexibility in statically\ntyped languages by letting code be written without the precise types of values being known at\ncompile time. The ability to write code that can operate on different types is called polymorphism.\nAll code in classic dynamically typed languages is polymorphic: only by explicitly checking types,\nor when objects fail to support operations at run-time, are the types of any values ever restricted.\n\nJulia's type system is dynamic, but gains some of the advantages of static type systems by making\nit possible to indicate that certain values are of specific types. This can be of great assistance\nin generating efficient code, but even more significantly, it allows method dispatch on the types\nof function arguments to be deeply integrated with the language. Method dispatch is explored in\ndetail in Methods, but is rooted in the type system presented here.\n\nThe default behavior in Julia when types are omitted is to allow values to be of any type. Thus,\none can write many useful Julia functions without ever explicitly using types. When additional\nexpressiveness is needed, however, it is easy to gradually introduce explicit type annotations\ninto previously \"untyped\" code. Adding annotations serves three primary purposes: to take advantage\nof Julia's powerful multiple-dispatch mechanism,  to improve human readability, and to catch\nprogrammer errors.\n\nDescribing Julia in the lingo of type systems, it\nis: dynamic, nominative and parametric. Generic types can be parameterized, and the hierarchical\nrelationships between types are explicitly declared,\nrather than implied by compatible structure.\nOne particularly distinctive feature of Julia's type system is that concrete types may not subtype\neach other: all concrete types are final and may only have abstract types as their supertypes.\nWhile this might at first seem unduly restrictive, it has many beneficial consequences with surprisingly\nfew drawbacks. It turns out that being able to inherit behavior is much more important than being\nable to inherit structure, and inheriting both causes significant difficulties in traditional\nobject-oriented languages. While concrete types do have abstract subtypes, there are only two examples of this\n(Union{} and Type{T})) and additional subtypes\nof concrete types cannot be declared.\n\nOther high-level aspects of Julia's type system that should be mentioned up front are:\n\nThere is no division between object and non-object values: all values in Julia are true objects\nhaving a type that belongs to a single, fully connected type graph, all nodes of which are equally\nfirst-class as types.\nThere is no meaningful concept of a \"compile-time type\": the only type a value has is its actual\ntype when the program is running. This is called a \"run-time type\" in object-oriented languages\nwhere the combination of static compilation with polymorphism makes this distinction significant.\nOnly values, not variables, have types – variables are simply names bound to values, although for\nsimplicity we may say \"type of a variable\" as shorthand for \"type of the value to which a variable refers\".\nBoth abstract and concrete types can be parameterized by other types. They can also be parameterized\nby symbols, by values of any type for which isbits returns true (essentially, things\nlike numbers and bools that are stored like C types or structs with no pointers to other objects),\nand also by tuples thereof. Type parameters may be omitted when they do not need to be referenced\nor restricted.\n\nJulia's type system is designed to be powerful and expressive, yet clear, intuitive and unobtrusive.\nMany Julia programmers may never feel the need to write code that explicitly uses types. Some\nkinds of programming, however, become clearer, simpler, faster and more robust with declared types."},{"title":"Type Declarations","page":"Types","location":"manual/types.html#Type-Declarations","category":"section","text":"The :: operator can be used to attach type annotations to expressions and variables in programs.\nThere are two primary reasons to do this:\n\nAs an assertion to help confirm that your program works the way you expect, and\nTo provide extra type information to the compiler, which can then improve performance in some\ncases.\n\nWhen appended to an expression computing a value, the :: operator is read as \"is an instance\nof\". It can be used anywhere to assert that the value of the expression on the left is an instance\nof the type on the right. When the type on the right is concrete, the value on the left must have\nthat type as its implementation – recall that all concrete types are final, so no implementation\nis a subtype of any other. When the type is abstract, it suffices for the value to be implemented\nby a concrete type that is a subtype of the abstract type. If the type assertion is not true,\nan exception is thrown, otherwise, the left-hand value is returned:\n\njulia> (1+2)::AbstractFloat\nERROR: TypeError: in typeassert, expected AbstractFloat, got a value of type Int64\n\njulia> (1+2)::Int\n3\n\nThis allows a type assertion to be attached to any expression in-place.\n\nWhen appended to a variable on the left-hand side of an assignment, or as part of a local declaration,\nthe :: operator means something a bit different: it declares the variable to always have the\nspecified type, like a type declaration in a statically-typed language such as C. Every value\nassigned to the variable will be converted to the declared type using convert:\n\njulia> function foo()\n           x::Int8 = 100\n           x\n       end\nfoo (generic function with 1 method)\n\njulia> x = foo()\n100\n\njulia> typeof(x)\nInt8\n\nThis feature is useful for avoiding performance \"gotchas\" that could occur if one of the assignments\nto a variable changed its type unexpectedly.\n\nThis \"declaration\" behavior only occurs in specific contexts:\n\nlocal x::Int8  # in a local declaration\nx::Int8 = 10   # as the left-hand side of an assignment\n\nand applies to the whole current scope, even before the declaration.\n\nAs of Julia 1.8, type declarations can now be used in global scope i.e.\ntype annotations can be added to global variables to make accessing them type stable.\n\njulia> x::Int = 10\n10\n\njulia> x = 3.5\nERROR: InexactError: Int64(3.5)\n\njulia> function foo(y)\n           global x = 15.8    # throws an error when foo is called\n           return x + y\n       end\nfoo (generic function with 1 method)\n\njulia> foo(10)\nERROR: InexactError: Int64(15.8)\n\nDeclarations can also be attached to function definitions:\n\nfunction sinc(x)::Float64\n    if x == 0\n        return 1\n    end\n    return sin(pi*x)/(pi*x)\nend\n\nReturning from this function behaves just like an assignment to a variable with a declared type:\nthe value is always converted to Float64."},{"title":"Abstract Types","page":"Types","location":"manual/types.html#man-abstract-types","category":"section","text":"Abstract types cannot be instantiated, and serve only as nodes in the type graph, thereby describing\nsets of related concrete types: those concrete types which are their descendants. We begin with\nabstract types even though they have no instantiation because they are the backbone of the type\nsystem: they form the conceptual hierarchy which makes Julia's type system more than just a collection\nof object implementations.\n\nRecall that in Integers and Floating-Point Numbers, we introduced a variety of concrete\ntypes of numeric values: Int8, UInt8, Int16, UInt16,\nInt32, UInt32, Int64, UInt64, Int128,\nUInt128, Float16, Float32, and Float64. Although\nthey have different representation sizes, Int8, Int16, Int32, Int64 and Int128\nall have in common that they are signed integer types. Likewise UInt8, UInt16, UInt32,\nUInt64 and UInt128 are all unsigned integer types, while Float16, Float32 and\nFloat64 are distinct in being floating-point types rather than integers. It is common for\na piece of code to make sense, for example, only if its arguments are some kind of integer,\nbut not really depend on what particular kind of integer. For example, the greatest common\ndenominator algorithm works for all kinds of integers, but will not work for floating-point\nnumbers. Abstract types allow the construction of a hierarchy of types, providing a context\ninto which concrete types can fit. This allows you, for example, to easily program to any type\nthat is an integer, without restricting an algorithm to a specific type of integer.\n\nAbstract types are declared using the abstract type keyword. The general syntaxes for declaring an\nabstract type are:\n\nabstract type «name» end\nabstract type «name» <: «supertype» end\n\nThe abstract type keyword introduces a new abstract type, whose name is given by «name». This\nname can be optionally followed by <: and an already-existing type, indicating that the newly\ndeclared abstract type is a subtype of this \"parent\" type.\n\nWhen no supertype is given, the default supertype is Any – a predefined abstract type that\nall objects are instances of and all types are subtypes of. In type theory, Any is commonly\ncalled \"top\" because it is at the apex of the type graph. Julia also has a predefined abstract\n\"bottom\" type, at the nadir of the type graph, which is written as Union{}. It is the exact\nopposite of Any: no object is an instance of Union{} and all types (including concrete types) are supertypes of Union{}.\n\nLet's consider some of the abstract types that make up Julia's numerical hierarchy:\n\nabstract type Number end\nabstract type Real          <: Number end\nabstract type AbstractFloat <: Real end\nabstract type Integer       <: Real end\nabstract type Signed        <: Integer end\nabstract type Unsigned      <: Integer end\n\nThe Number type is a direct child type of Any, and Real is its child.\nIn turn, Real has two children (it has more, but only two are shown here; we'll get to\nthe others later): Integer and AbstractFloat, separating the world into\nrepresentations of integers and representations of real numbers. Representations of real\nnumbers include floating-point types, but also include other types, such as rationals.\nAbstractFloat includes only floating-point representations of real numbers. Integers\nare further subdivided into Signed and Unsigned varieties.\n\nThe <: operator in general means \"is a subtype of\", and, used in declarations like those above,\ndeclares the right-hand type to be an immediate supertype of the newly declared type. It can also\nbe used in expressions as a subtype operator which returns true when its left operand is a\nsubtype of its right operand:\n\njulia> Integer <: Number\ntrue\n\njulia> Integer <: AbstractFloat\nfalse\n\nAn important use of abstract types is to provide default implementations for concrete types. To\ngive a simple example, consider:\n\nfunction myplus(x,y)\n    x+y\nend\n\nThe first thing to note is that the above argument declarations are equivalent to x::Any and\ny::Any. When this function is invoked, say as myplus(2,5), the dispatcher chooses the most\nspecific method named myplus that matches the given arguments. (See Methods for more\ninformation on multiple dispatch.)\n\nAssuming no method more specific than the above is found, Julia next internally defines and compiles\na method called myplus specifically for two Int arguments based on the generic function given\nabove, i.e., it implicitly defines and compiles:\n\nfunction myplus(x::Int,y::Int)\n    x+y\nend\n\nand finally, it invokes this specific method.\n\nThus, abstract types allow programmers to write generic functions that can later be used as the\ndefault method by many combinations of concrete types. Thanks to multiple dispatch, the programmer\nhas full control over whether the default or more specific method is used.\n\nAn important point to note is that there is no loss in performance if the programmer relies on\na function whose arguments are abstract types, because it is recompiled for each tuple of concrete\nargument types with which it is invoked. (There may be a performance issue, however, in the case\nof function arguments that are containers of abstract types; see Performance Tips.)"},{"title":"Primitive Types","page":"Types","location":"manual/types.html#Primitive-Types","category":"section","text":"warning: Warning\nIt is almost always preferable to wrap an existing primitive type in a new\ncomposite type than to define your own primitive type.This functionality exists to allow Julia to bootstrap the standard primitive\ntypes that LLVM supports. Once they are defined, there is very little reason\nto define more.\n\nA primitive type is a concrete type whose data consists of plain old bits. Classic examples of primitive\ntypes are integers and floating-point values. Unlike most languages, Julia lets you declare your\nown primitive types, rather than providing only a fixed set of built-in ones. In fact, the standard\nprimitive types are all defined in the language itself:\n\nprimitive type Float16 <: AbstractFloat 16 end\nprimitive type Float32 <: AbstractFloat 32 end\nprimitive type Float64 <: AbstractFloat 64 end\n\nprimitive type Bool <: Integer 8 end\nprimitive type Char <: AbstractChar 32 end\n\nprimitive type Int8    <: Signed   8 end\nprimitive type UInt8   <: Unsigned 8 end\nprimitive type Int16   <: Signed   16 end\nprimitive type UInt16  <: Unsigned 16 end\nprimitive type Int32   <: Signed   32 end\nprimitive type UInt32  <: Unsigned 32 end\nprimitive type Int64   <: Signed   64 end\nprimitive type UInt64  <: Unsigned 64 end\nprimitive type Int128  <: Signed   128 end\nprimitive type UInt128 <: Unsigned 128 end\n\nThe general syntaxes for declaring a primitive type are:\n\nprimitive type «name» «bits» end\nprimitive type «name» <: «supertype» «bits» end\n\nThe number of bits indicates how much storage the type requires and the name gives the new type\na name. A primitive type can optionally be declared to be a subtype of some supertype. If a supertype\nis omitted, then the type defaults to having Any as its immediate supertype. The declaration\nof Bool above therefore means that a boolean value takes eight bits to store, and has\nInteger as its immediate supertype. Currently, only sizes that are multiples of\n8 bits are supported and you are more likely to experience bugs with sizes other than those used above.\nTherefore, boolean values, although they really need just a single bit, cannot be declared to be any\nsmaller than eight bits.\n\nThe types Bool, Int8 and UInt8 all have identical representations:\nthey are eight-bit chunks of memory. Since Julia's type system is nominative, however, they\nare not interchangeable despite having identical structure. A fundamental difference between\nthem is that they have different supertypes: Bool's direct supertype is Integer,\nInt8's is Signed, and UInt8's is Unsigned. All other\ndifferences between Bool, Int8, and UInt8 are matters of\nbehavior – the way functions are defined to act when given objects of these types as\narguments. This is why a nominative type system is necessary: if structure determined type,\nwhich in turn dictates behavior, then it would be impossible to make Bool behave\nany differently than Int8 or UInt8."},{"title":"Composite Types","page":"Types","location":"manual/types.html#Composite-Types","category":"section","text":"Composite types are called records, structs,\nor objects in various languages. A composite type is a collection of named fields,\nan instance of which can be treated as a single value. In many languages, composite types are\nthe only kind of user-definable type, and they are by far the most commonly used user-defined\ntype in Julia as well.\n\nIn mainstream object oriented languages, such as C++, Java, Python and Ruby, composite types also\nhave named functions associated with them, and the combination is called an \"object\". In purer\nobject-oriented languages, such as Ruby or Smalltalk, all values are objects whether they are\ncomposites or not. In less pure object oriented languages, including C++ and Java, some values,\nsuch as integers and floating-point values, are not objects, while instances of user-defined composite\ntypes are true objects with associated methods. In Julia, all values are objects, but functions\nare not bundled with the objects they operate on. This is necessary since Julia chooses which\nmethod of a function to use by multiple dispatch, meaning that the types of all of a function's\narguments are considered when selecting a method, rather than just the first one (see Methods\nfor more information on methods and dispatch). Thus, it would be inappropriate for functions to\n\"belong\" to only their first argument. Organizing methods into function objects rather than having\nnamed bags of methods \"inside\" each object ends up being a highly beneficial aspect of the language\ndesign.\n\nComposite types are introduced with the struct keyword followed by a block of field names, optionally\nannotated with types using the :: operator:\n\njulia> struct Foo\n           bar\n           baz::Int\n           qux::Float64\n       end\n\nFields with no type annotation default to Any, and can accordingly hold any type of value.\n\nNew objects of type Foo are created by applying the Foo type object like a function\nto values for its fields:\n\njulia> foo = Foo(\"Hello, world.\", 23, 1.5)\nFoo(\"Hello, world.\", 23, 1.5)\n\njulia> typeof(foo)\nFoo\n\nWhen a type is applied like a function it is called a constructor. Two constructors are generated\nautomatically (these are called default constructors). One accepts any arguments and calls\nconvert to convert them to the types of the fields, and the other accepts arguments\nthat match the field types exactly. The reason both of these are generated is that this makes\nit easier to add new definitions without inadvertently replacing a default constructor.\n\nSince the bar field is unconstrained in type, any value will do. However, the value for baz\nmust be convertible to Int:\n\njulia> Foo((), 23.5, 1)\nERROR: InexactError: Int64(23.5)\nStacktrace:\n[...]\n\nYou may find a list of field names using the fieldnames function.\n\njulia> fieldnames(Foo)\n(:bar, :baz, :qux)\n\nYou can access the field values of a composite object using the traditional foo.bar notation:\n\njulia> foo.bar\n\"Hello, world.\"\n\njulia> foo.baz\n23\n\njulia> foo.qux\n1.5\n\nComposite objects declared with struct are immutable; they cannot be modified\nafter construction. This may seem odd at first, but it has several advantages:\n\nIt can be more efficient. Some structs can be packed efficiently into arrays, and\nin some cases the compiler is able to avoid allocating immutable objects entirely.\nIt is not possible to violate the invariants provided by the type's constructors.\nCode using immutable objects can be easier to reason about.\n\nAn immutable object might contain mutable objects, such as arrays, as fields. Those contained\nobjects will remain mutable; only the fields of the immutable object itself cannot be changed\nto point to different objects.\n\nWhere required, mutable composite objects can be declared with the keyword mutable struct, to be\ndiscussed in the next section.\n\nIf all the fields of an immutable structure are indistinguishable (===) then two immutable values containing those fields are also indistinguishable:\n\njulia> struct X\n           a::Int\n           b::Float64\n       end\n\njulia> X(1, 2) === X(1, 2)\ntrue\n\nThere is much more to say about how instances of composite types are created, but that discussion\ndepends on both Parametric Types and on Methods, and is sufficiently important\nto be addressed in its own section: Constructors.\n\nFor many user-defined types X, you may want to define a method Base.broadcastable(x::X) = Ref(x)\nso that instances of that type act as 0-dimensional \"scalars\" for broadcasting."},{"title":"Mutable Composite Types","page":"Types","location":"manual/types.html#Mutable-Composite-Types","category":"section","text":"If a composite type is declared with mutable struct instead of struct, then instances of\nit can be modified:\n\njulia> mutable struct Bar\n           baz\n           qux::Float64\n       end\n\njulia> bar = Bar(\"Hello\", 1.5);\n\njulia> bar.qux = 2.0\n2.0\n\njulia> bar.baz = 1//2\n1//2\n\nAn extra interface between the fields and the user can be provided through Instance Properties.\nThis grants more control on what can be accessed and modified using the bar.baz notation.\n\nIn order to support mutation, such objects are generally allocated on the heap, and have\nstable memory addresses.\nA mutable object is like a little container that might hold different values over time,\nand so can only be reliably identified with its address.\nIn contrast, an instance of an immutable type is associated with specific field values —\nthe field values alone tell you everything about the object.\nIn deciding whether to make a type mutable, ask whether two instances\nwith the same field values would be considered identical, or if they might need to change independently\nover time. If they would be considered identical, the type should probably be immutable.\n\nTo recap, two essential properties define immutability in Julia:\n\nIt is not permitted to modify the value of an immutable type.\nFor bits types this means that the bit pattern of a value once set will never change\nand that value is the identity of a bits type.\nFor composite  types, this means that the identity of the values of its fields will\nnever change. When the fields are bits types, that means their bits will never change,\nfor fields whose values are mutable types like arrays, that means the fields will\nalways refer to the same mutable value even though that mutable value's content may\nitself be modified.\nAn object with an immutable type may be copied freely by the compiler since its\nimmutability makes it impossible to programmatically distinguish between the original\nobject and a copy.\nIn particular, this means that small enough immutable values like integers and floats\nare typically passed to functions in registers (or stack allocated).\nMutable values, on the other hand are heap-allocated and passed to\nfunctions as pointers to heap-allocated values except in cases where the compiler\nis sure that there's no way to tell that this is not what is happening.\n\nIn cases where one or more fields of an otherwise mutable struct is known to be immutable,\none can declare these fields as such using const as shown below. This enables some,\nbut not all of the optimizations of immutable structs, and can be used to enforce invariants\non the particular fields marked as const.\n\ncompat: Julia 1.8\nconst annotating fields of mutable structs requires at least Julia 1.8.\n\njulia> mutable struct Baz\n           a::Int\n           const b::Float64\n       end\n\njulia> baz = Baz(1, 1.5);\n\njulia> baz.a = 2\n2\n\njulia> baz.b = 2.0\nERROR: setfield!: const field .b of type Baz cannot be changed\n[...]"},{"title":"Declared Types","page":"Types","location":"manual/types.html#man-declared-types","category":"section","text":"The three kinds of types (abstract, primitive, composite) discussed in the previous\nsections are actually all closely related. They share the same key properties:\n\nThey are explicitly declared.\nThey have names.\nThey have explicitly declared supertypes.\nThey may have parameters.\n\nBecause of these shared properties, these types are internally represented as instances of the\nsame concept, DataType, which is the type of any of these types:\n\njulia> typeof(Real)\nDataType\n\njulia> typeof(Int)\nDataType\n\nA DataType may be abstract or concrete. If it is concrete, it has a specified size, storage\nlayout, and (optionally) field names. Thus a primitive type is a DataType with nonzero size, but\nno field names. A composite type is a DataType that has field names or is empty (zero size).\n\nEvery concrete value in the system is an instance of some DataType."},{"title":"Type Unions","page":"Types","location":"manual/types.html#Type-Unions","category":"section","text":"A type union is a special abstract type which includes as objects all instances of any of its\nargument types, constructed using the special Union keyword:\n\njulia> IntOrString = Union{Int,AbstractString}\nUnion{Int64, AbstractString}\n\njulia> 1 :: IntOrString\n1\n\njulia> \"Hello!\" :: IntOrString\n\"Hello!\"\n\njulia> 1.0 :: IntOrString\nERROR: TypeError: in typeassert, expected Union{Int64, AbstractString}, got a value of type Float64\n\nThe compilers for many languages have an internal union construct for reasoning about types; Julia\nsimply exposes it to the programmer. The Julia compiler is able to generate efficient code in the\npresence of Union types with a small number of types [1], by generating specialized code\nin separate branches for each possible type.\n\nA particularly useful case of a Union type is Union{T, Nothing}, where T can be any type and\nNothing is the singleton type whose only instance is the object nothing. This pattern\nis the Julia equivalent of Nullable, Option or Maybe\ntypes in other languages. Declaring a function argument or a field as Union{T, Nothing} allows\nsetting it either to a value of type T, or to nothing to indicate that there is no value.\nSee this FAQ entry for more information."},{"title":"Parametric Types","page":"Types","location":"manual/types.html#Parametric-Types","category":"section","text":"An important and powerful feature of Julia's type system is that it is parametric: types can take\nparameters, so that type declarations actually introduce a whole family of new types – one for\neach possible combination of parameter values. There are many languages that support some version\nof generic programming, wherein data structures\nand algorithms to manipulate them may be specified without specifying the exact types involved.\nFor example, some form of generic programming exists in ML, Haskell, Ada, Eiffel, C++, Java, C#,\nF#, and Scala, just to name a few. Some of these languages support true parametric polymorphism\n(e.g. ML, Haskell, Scala), while others support ad-hoc, template-based styles of generic programming\n(e.g. C++, Java). With so many different varieties of generic programming and parametric types\nin various languages, we won't even attempt to compare Julia's parametric types to other languages,\nbut will instead focus on explaining Julia's system in its own right. We will note, however, that\nbecause Julia is a dynamically typed language and doesn't need to make all type decisions at compile\ntime, many traditional difficulties encountered in static parametric type systems can be relatively\neasily handled.\n\nAll declared types (the DataType variety) can be parameterized, with the same syntax in each\ncase. We will discuss them in the following order: first, parametric composite types, then parametric\nabstract types, and finally parametric primitive types."},{"title":"Parametric Composite Types","page":"Types","location":"manual/types.html#man-parametric-composite-types","category":"section","text":"Type parameters are introduced immediately after the type name, surrounded by curly braces:\n\njulia> struct Point{T}\n           x::T\n           y::T\n       end\n\nThis declaration defines a new parametric type, Point{T}, holding two \"coordinates\" of type\nT. What, one may ask, is T? Well, that's precisely the point of parametric types: it can be\nany type at all (or a value of any bits type, actually, although here it's clearly used as a type).\nPoint{Float64} is a concrete type equivalent to the type defined by replacing T in the definition\nof Point with Float64. Thus, this single declaration actually declares an unlimited\nnumber of types: Point{Float64}, Point{AbstractString}, Point{Int64}, etc. Each of these\nis now a usable concrete type:\n\njulia> Point{Float64}\nPoint{Float64}\n\njulia> Point{AbstractString}\nPoint{AbstractString}\n\nThe type Point{Float64} is a point whose coordinates are 64-bit floating-point values, while\nthe type Point{AbstractString} is a \"point\" whose \"coordinates\" are string objects (see Strings).\n\nPoint itself is also a valid type object, containing all instances Point{Float64}, Point{AbstractString},\netc. as subtypes:\n\njulia> Point{Float64} <: Point\ntrue\n\njulia> Point{AbstractString} <: Point\ntrue\n\nOther types, of course, are not subtypes of it:\n\njulia> Float64 <: Point\nfalse\n\njulia> AbstractString <: Point\nfalse\n\nConcrete Point types with different values of T are never subtypes of each other:\n\njulia> Point{Float64} <: Point{Int64}\nfalse\n\njulia> Point{Float64} <: Point{Real}\nfalse\n\nwarning: Warning\nThis last point is very important: even though Float64 <: Real we DO NOT have Point{Float64} <: Point{Real}.\n\nIn other words, in the parlance of type theory, Julia's type parameters are invariant, rather\nthan being covariant (or even contravariant). This is for practical reasons: while any instance\nof Point{Float64} may conceptually be like an instance of Point{Real} as well, the two types\nhave different representations in memory:\n\nAn instance of Point{Float64} can be represented compactly and efficiently as an immediate pair\nof 64-bit values;\nAn instance of Point{Real} must be able to hold any pair of instances of Real.\nSince objects that are instances of Real can be of arbitrary size and structure, in\npractice an instance of Point{Real} must be represented as a pair of pointers to\nindividually allocated Real objects.\n\nThe efficiency gained by being able to store Point{Float64} objects with immediate values is\nmagnified enormously in the case of arrays: an Array{Float64} can be stored as a contiguous\nmemory block of 64-bit floating-point values, whereas an Array{Real} must be an array of pointers\nto individually allocated Real objects – which may well be\nboxed\n64-bit floating-point values, but also might be arbitrarily large, complex objects, which are\ndeclared to be implementations of the Real abstract type.\n\nSince Point{Float64} is not a subtype of Point{Real}, the following method can't be applied\nto arguments of type Point{Float64}:\n\nfunction norm(p::Point{Real})\n    sqrt(p.x^2 + p.y^2)\nend\n\nA correct way to define a method that accepts all arguments of type Point{T} where T is\na subtype of Real is:\n\nfunction norm(p::Point{<:Real})\n    sqrt(p.x^2 + p.y^2)\nend\n\n(Equivalently, one could define function norm(p::Point{T} where T<:Real) or\nfunction norm(p::Point{T}) where T<:Real; see UnionAll Types.)\n\nMore examples will be discussed later in Methods.\n\nHow does one construct a Point object? It is possible to define custom constructors for composite\ntypes, which will be discussed in detail in Constructors, but in the absence of any special\nconstructor declarations, there are two default ways of creating new composite objects, one in\nwhich the type parameters are explicitly given and the other in which they are implied by the\narguments to the object constructor.\n\nSince the type Point{Float64} is a concrete type equivalent to Point declared with Float64\nin place of T, it can be applied as a constructor accordingly:\n\njulia> p = Point{Float64}(1.0, 2.0)\nPoint{Float64}(1.0, 2.0)\n\njulia> typeof(p)\nPoint{Float64}\n\nFor the default constructor, exactly one argument must be supplied for each field:\n\njulia> Point{Float64}(1.0)\nERROR: MethodError: no method matching Point{Float64}(::Float64)\nThe type `Point{Float64}` exists, but no method is defined for this combination of argument types when trying to construct it.\n[...]\n\njulia> Point{Float64}(1.0, 2.0, 3.0)\nERROR: MethodError: no method matching Point{Float64}(::Float64, ::Float64, ::Float64)\nThe type `Point{Float64}` exists, but no method is defined for this combination of argument types when trying to construct it.\n[...]\n\nOnly one default constructor is generated for parametric types, since overriding it is not possible.\nThis constructor accepts any arguments and converts them to the field types.\n\nIn many cases, it is redundant to provide the type of Point object one wants to construct, since\nthe types of arguments to the constructor call already implicitly provide type information. For\nthat reason, you can also apply Point itself as a constructor, provided that the implied value\nof the parameter type T is unambiguous:\n\njulia> p1 = Point(1.0,2.0)\nPoint{Float64}(1.0, 2.0)\n\njulia> typeof(p1)\nPoint{Float64}\n\njulia> p2 = Point(1,2)\nPoint{Int64}(1, 2)\n\njulia> typeof(p2)\nPoint{Int64}\n\nIn the case of Point, the type of T is unambiguously implied if and only if the two arguments\nto Point have the same type. When this isn't the case, the constructor will fail with a MethodError:\n\njulia> Point(1,2.5)\nERROR: MethodError: no method matching Point(::Int64, ::Float64)\nThe type `Point` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  Point(::T, !Matched::T) where T\n   @ Main none:2\n\nStacktrace:\n[...]\n\nConstructor methods to appropriately handle such mixed cases can be defined, but that will not\nbe discussed until later on in Constructors."},{"title":"Parametric Abstract Types","page":"Types","location":"manual/types.html#Parametric-Abstract-Types","category":"section","text":"Parametric abstract type declarations declare a collection of abstract types, in much the same\nway:\n\njulia> abstract type Pointy{T} end\n\nWith this declaration, Pointy{T} is a distinct abstract type for each type or integer value\nof T. As with parametric composite types, each such instance is a subtype of Pointy:\n\njulia> Pointy{Int64} <: Pointy\ntrue\n\njulia> Pointy{1} <: Pointy\ntrue\n\nParametric abstract types are invariant, much as parametric composite types are:\n\njulia> Pointy{Float64} <: Pointy{Real}\nfalse\n\njulia> Pointy{Real} <: Pointy{Float64}\nfalse\n\nThe notation Pointy{<:Real} can be used to express the Julia analogue of a\ncovariant type, while Pointy{>:Int} the analogue of a contravariant type,\nbut technically these represent sets of types (see UnionAll Types).\n\njulia> Pointy{Float64} <: Pointy{<:Real}\ntrue\n\njulia> Pointy{Real} <: Pointy{>:Int}\ntrue\n\nMuch as plain old abstract types serve to create a useful hierarchy of types over concrete types,\nparametric abstract types serve the same purpose with respect to parametric composite types. We\ncould, for example, have declared Point{T} to be a subtype of Pointy{T} as follows:\n\njulia> struct Point{T} <: Pointy{T}\n           x::T\n           y::T\n       end\n\nGiven such a declaration, for each choice of T, we have Point{T} as a subtype of Pointy{T}:\n\njulia> Point{Float64} <: Pointy{Float64}\ntrue\n\njulia> Point{Real} <: Pointy{Real}\ntrue\n\njulia> Point{AbstractString} <: Pointy{AbstractString}\ntrue\n\nThis relationship is also invariant:\n\njulia> Point{Float64} <: Pointy{Real}\nfalse\n\njulia> Point{Float64} <: Pointy{<:Real}\ntrue\n\nWhat purpose do parametric abstract types like Pointy serve? Consider if we create a point-like\nimplementation that only requires a single coordinate because the point is on the diagonal line\nx = y:\n\njulia> struct DiagPoint{T} <: Pointy{T}\n           x::T\n       end\n\nNow both Point{Float64} and DiagPoint{Float64} are implementations of the Pointy{Float64}\nabstraction, and similarly for every other possible choice of type T. This allows programming\nto a common interface shared by all Pointy objects, implemented for both Point and DiagPoint.\nThis cannot be fully demonstrated, however, until we have introduced methods and dispatch in the\nnext section, Methods.\n\nThere are situations where it may not make sense for type parameters to range freely over all\npossible types. In such situations, one can constrain the range of T like so:\n\njulia> abstract type Pointy{T<:Real} end\n\nWith such a declaration, it is acceptable to use any type that is a subtype of\nReal in place of T, but not types that are not subtypes of Real:\n\njulia> Pointy{Float64}\nPointy{Float64}\n\njulia> Pointy{Real}\nPointy{Real}\n\njulia> Pointy{AbstractString}\nERROR: TypeError: in Pointy, in T, expected T<:Real, got Type{AbstractString}\n\njulia> Pointy{1}\nERROR: TypeError: in Pointy, in T, expected T<:Real, got a value of type Int64\n\nType parameters for parametric composite types can be restricted in the same manner:\n\nstruct Point{T<:Real} <: Pointy{T}\n    x::T\n    y::T\nend\n\nTo give a real-world example of how all this parametric type machinery can be useful, here is\nthe actual definition of Julia's Rational immutable type (except that we omit the\nconstructor here for simplicity), representing an exact ratio of integers:\n\nstruct Rational{T<:Integer} <: Real\n    num::T\n    den::T\nend\n\nIt only makes sense to take ratios of integer values, so the parameter type T is restricted\nto being a subtype of Integer, and a ratio of integers represents a value on the\nreal number line, so any Rational is an instance of the Real abstraction."},{"title":"Tuple Types","page":"Types","location":"manual/types.html#Tuple-Types","category":"section","text":"Tuples are an abstraction of the arguments of a function – without the function itself. The salient\naspects of a function's arguments are their order and their types. Therefore a tuple type is similar\nto a parameterized immutable type where each parameter is the type of one field. For example,\na 2-element tuple type resembles the following immutable type:\n\nstruct Tuple2{A,B}\n    a::A\n    b::B\nend\n\nHowever, there are three key differences:\n\nTuple types may have any number of parameters.\nTuple types are covariant in their parameters: Tuple{Int} is a subtype of Tuple{Any}. Therefore\nTuple{Any} is considered an abstract type, and tuple types are only concrete if their parameters\nare.\nTuples do not have field names; fields are only accessed by index.\n\nTuple values are written with parentheses and commas. When a tuple is constructed, an appropriate\ntuple type is generated on demand:\n\njulia> typeof((1,\"foo\",2.5))\nTuple{Int64, String, Float64}\n\nNote the implications of covariance:\n\njulia> Tuple{Int,AbstractString} <: Tuple{Real,Any}\ntrue\n\njulia> Tuple{Int,AbstractString} <: Tuple{Real,Real}\nfalse\n\njulia> Tuple{Int,AbstractString} <: Tuple{Real,}\nfalse\n\nIntuitively, this corresponds to the type of a function's arguments being a subtype of the function's\nsignature (when the signature matches)."},{"title":"Vararg Tuple Types","page":"Types","location":"manual/types.html#Vararg-Tuple-Types","category":"section","text":"The last parameter of a tuple type can be the special value Vararg, which denotes any number\nof trailing elements:\n\njulia> mytupletype = Tuple{AbstractString,Vararg{Int}}\nTuple{AbstractString, Vararg{Int64}}\n\njulia> isa((\"1\",), mytupletype)\ntrue\n\njulia> isa((\"1\",1), mytupletype)\ntrue\n\njulia> isa((\"1\",1,2), mytupletype)\ntrue\n\njulia> isa((\"1\",1,2,3.0), mytupletype)\nfalse\n\nMoreover Vararg{T} corresponds to zero or more elements of type T. Vararg tuple types are\nused to represent the arguments accepted by varargs methods (see Varargs Functions).\n\nThe special value Vararg{T,N} (when used as the last parameter of a tuple type)\ncorresponds to exactly N elements of type T.  NTuple{N,T} is a convenient\nalias for Tuple{Vararg{T,N}}, i.e. a tuple type containing exactly N elements of type T."},{"title":"Named Tuple Types","page":"Types","location":"manual/types.html#Named-Tuple-Types","category":"section","text":"Named tuples are instances of the NamedTuple type, which has two parameters: a tuple of\nsymbols giving the field names, and a tuple type giving the field types.\nFor convenience, NamedTuple types are printed using the @NamedTuple macro which provides a\nconvenient struct-like syntax for declaring these types via key::Type declarations,\nwhere an omitted ::Type corresponds to ::Any.\n\njulia> typeof((a=1,b=\"hello\")) # prints in macro form\n@NamedTuple{a::Int64, b::String}\n\njulia> NamedTuple{(:a, :b), Tuple{Int64, String}} # long form of the type\n@NamedTuple{a::Int64, b::String}\n\nThe begin ... end form of the @NamedTuple macro allows the declarations to be\nsplit across multiple lines (similar to a struct declaration), but is otherwise equivalent:\n\njulia> @NamedTuple begin\n           a::Int\n           b::String\n       end\n@NamedTuple{a::Int64, b::String}\n\nA NamedTuple type can be used as a constructor, accepting a single tuple argument.\nThe constructed NamedTuple type can be either a concrete type, with both parameters specified,\nor a type that specifies only field names:\n\njulia> @NamedTuple{a::Float32,b::String}((1, \"\"))\n(a = 1.0f0, b = \"\")\n\njulia> NamedTuple{(:a, :b)}((1, \"\"))\n(a = 1, b = \"\")\n\nIf field types are specified, the arguments are converted. Otherwise the types of the arguments\nare used directly."},{"title":"Parametric Primitive Types","page":"Types","location":"manual/types.html#Parametric-Primitive-Types","category":"section","text":"Primitive types can also be declared parametrically. For example, pointers are represented as\nprimitive types which would be declared in Julia like this:\n\n# 32-bit system:\nprimitive type Ptr{T} 32 end\n\n# 64-bit system:\nprimitive type Ptr{T} 64 end\n\nThe slightly odd feature of these declarations as compared to typical parametric composite types,\nis that the type parameter T is not used in the definition of the type itself – it is just\nan abstract tag, essentially defining an entire family of types with identical structure, differentiated\nonly by their type parameter. Thus, Ptr{Float64} and Ptr{Int64} are distinct types, even though\nthey have identical representations. And of course, all specific pointer types are subtypes of\nthe umbrella Ptr type:\n\njulia> Ptr{Float64} <: Ptr\ntrue\n\njulia> Ptr{Int64} <: Ptr\ntrue"},{"title":"UnionAll Types","page":"Types","location":"manual/types.html#UnionAll-Types","category":"section","text":"We have said that a parametric type like Ptr acts as a supertype of all its instances\n(Ptr{Int64} etc.). How does this work? Ptr itself cannot be a normal data type, since without\nknowing the type of the referenced data the type clearly cannot be used for memory operations.\nThe answer is that Ptr (or other parametric types like Array) is a different kind of type called a\nUnionAll type. Such a type expresses the iterated union of types for all values of some parameter.\n\nUnionAll types are usually written using the keyword where. For example Ptr could be more\naccurately written as Ptr{T} where T, meaning all values whose type is Ptr{T} for some value\nof T. In this context, the parameter T is also often called a \"type variable\" since it is\nlike a variable that ranges over types.\nEach where introduces a single type variable, so these expressions are nested for types with\nmultiple parameters, for example Array{T,N} where N where T.\n\nThe type application syntax A{B,C} requires A to be a UnionAll type, and first substitutes B\nfor the outermost type variable in A.\nThe result is expected to be another UnionAll type, into which C is then substituted.\nSo A{B,C} is equivalent to A{B}{C}.\nThis explains why it is possible to partially instantiate a type, as in Array{Float64}: the first\nparameter value has been fixed, but the second still ranges over all possible values.\nUsing explicit where syntax, any subset of parameters can be fixed. For example, the type of all\n1-dimensional arrays can be written as Array{T,1} where T.\n\nType variables can be restricted with subtype relations.\nArray{T} where T<:Integer refers to all arrays whose element type is some kind of\nInteger.\nThe syntax Array{<:Integer} is a convenient shorthand for Array{T} where T<:Integer.\nType variables can have both lower and upper bounds.\nArray{T} where Int<:T<:Number refers to all arrays of Numbers that are able to\ncontain Ints (since T must be at least as big as Int).\nThe syntax where T>:Int also works to specify only the lower bound of a type variable,\nand Array{>:Int} is equivalent to Array{T} where T>:Int.\n\nSince where expressions nest, type variable bounds can refer to outer type variables.\nFor example Tuple{T,Array{S}} where S<:AbstractArray{T} where T<:Real refers to 2-tuples\nwhose first element is some Real, and whose second element is an Array of any\nkind of array whose element type contains the type of the first tuple element.\n\nThe where keyword itself can be nested inside a more complex declaration. For example,\nconsider the two types created by the following declarations:\n\njulia> const T1 = Array{Array{T, 1} where T, 1}\nVector{Vector} (alias for Array{Array{T, 1} where T, 1})\n\njulia> const T2 = Array{Array{T, 1}, 1} where T\nArray{Vector{T}, 1} where T\n\nType T1 defines a 1-dimensional array of 1-dimensional arrays; each\nof the inner arrays consists of objects of the same type, but this type may vary from one inner array to the next.\nOn the other hand, type T2 defines a 1-dimensional array of 1-dimensional arrays all of whose inner arrays must have the\nsame type. Note that T2 is an abstract type, e.g., Array{Array{Int,1},1} <: T2, whereas T1 is a concrete type. As a consequence, T1 can be constructed with a zero-argument constructor a=T1() but T2 cannot.\n\nThere is a convenient syntax for naming such types, similar to the short form of function\ndefinition syntax:\n\nVector{T} = Array{T, 1}\n\nThis is equivalent to const Vector = Array{T,1} where T.\nWriting Vector{Float64} is equivalent to writing Array{Float64,1}, and the umbrella type\nVector has as instances all Array objects where the second parameter – the number of array\ndimensions – is 1, regardless of what the element type is. In languages where parametric types\nmust always be specified in full, this is not especially helpful, but in Julia, this allows one\nto write just Vector for the abstract type including all one-dimensional dense arrays of any\nelement type."},{"title":"Singleton types","page":"Types","location":"manual/types.html#man-singleton-types","category":"section","text":"Immutable composite types with no fields are called singletons. Formally, if\n\nT is an immutable composite type (i.e. defined with struct),\na isa T && b isa T implies a === b,\n\nthen T is a singleton type.[2] Base.issingletontype can be used to check if a\ntype is a singleton type. Abstract types cannot be singleton\ntypes by construction.\n\nFrom the definition, it follows that there can be only one instance of such types:\n\njulia> struct NoFields\n       end\n\njulia> NoFields() === NoFields()\ntrue\n\njulia> Base.issingletontype(NoFields)\ntrue\n\nThe === function confirms that the constructed instances of NoFields are actually one\nand the same.\n\nParametric types can be singleton types when the above condition holds. For example,\n\njulia> struct NoFieldsParam{T}\n       end\n\njulia> Base.issingletontype(NoFieldsParam) # Can't be a singleton type ...\nfalse\n\njulia> NoFieldsParam{Int}() isa NoFieldsParam # ... because it has ...\ntrue\n\njulia> NoFieldsParam{Bool}() isa NoFieldsParam # ... multiple instances.\ntrue\n\njulia> Base.issingletontype(NoFieldsParam{Int}) # Parametrized, it is a singleton.\ntrue\n\njulia> NoFieldsParam{Int}() === NoFieldsParam{Int}()\ntrue"},{"title":"Types of functions","page":"Types","location":"manual/types.html#Types-of-functions","category":"section","text":"Each function has its own type, which is a subtype of Function.\n\njulia> foo41(x) = x + 1\nfoo41 (generic function with 1 method)\n\njulia> typeof(foo41)\ntypeof(foo41) (singleton type of function foo41, subtype of Function)\n\nNote how typeof(foo41) prints as itself. This is merely a convention for printing, as it is a first-class object that can be used like any other value:\n\njulia> T = typeof(foo41)\ntypeof(foo41) (singleton type of function foo41, subtype of Function)\n\njulia> T <: Function\ntrue\n\nTypes of functions defined at top-level are singletons. When necessary, you can compare them with ===.\n\nClosures also have their own type, which is usually printed with names that end in #<number>. Names and types for functions defined at different locations are distinct, but not guaranteed to be printed the same way across sessions.\n\njulia> typeof(x -> x + 1)\nvar\"#9#10\"\n\nTypes of closures are not necessarily singletons.\n\njulia> addy(y) = x -> x + y\naddy (generic function with 1 method)\n\njulia> typeof(addy(1)) === typeof(addy(2))\ntrue\n\njulia> addy(1) === addy(2)\nfalse\n\njulia> Base.issingletontype(typeof(addy(1)))\nfalse"},{"title":"Type{T} type selectors","page":"Types","location":"manual/types.html#man-typet-type","category":"section","text":"For each type T, Type{T} is an abstract parametric type whose only instance is the\nobject T. Until we discuss Parametric Methods and [conversions](@ref\nconversion-and-promotion), it is difficult to explain the utility of this construct, but in\nshort, it allows one to specialize function behavior on specific types as values. This is\nuseful for writing methods (especially parametric ones) whose behavior depends on a type\nthat is given as an explicit argument rather than implied by the type of one of its\narguments.\n\nSince the definition is a little difficult to parse, let's look at some examples:\n\njulia> isa(Float64, Type{Float64})\ntrue\n\njulia> isa(Real, Type{Float64})\nfalse\n\njulia> isa(Real, Type{Real})\ntrue\n\njulia> isa(Float64, Type{Real})\nfalse\n\nIn other words, isa(A, Type{B}) is true if and only if A and B are the same object\nand that object is a type.\n\nIn particular, since parametric types are invariant, we have\n\njulia> struct TypeParamExample{T}\n           x::T\n       end\n\njulia> TypeParamExample isa Type{TypeParamExample}\ntrue\n\njulia> TypeParamExample{Int} isa Type{TypeParamExample}\nfalse\n\njulia> TypeParamExample{Int} isa Type{TypeParamExample{Int}}\ntrue\n\nWithout the parameter, Type is simply an abstract type which has\nall type objects as its instances:\n\njulia> isa(Type{Float64}, Type)\ntrue\n\njulia> isa(Float64, Type)\ntrue\n\njulia> isa(Real, Type)\ntrue\n\nAny object that is not a type is not an instance of Type:\n\njulia> isa(1, Type)\nfalse\n\njulia> isa(\"foo\", Type)\nfalse\n\nWhile Type is part of Julia's type hierarchy like any other abstract parametric type, it\nis not commonly used outside method signatures except in some special cases. Another\nimportant use case for Type is sharpening field types which would otherwise be captured\nless precisely, e.g. as DataType in the example below where the\ndefault constructor could lead to performance problems in code relying on the precise wrapped\ntype (similarly to abstract type parameters).\n\njulia> struct WrapType{T}\n       value::T\n       end\n\njulia> WrapType(Float64) # default constructor, note DataType\nWrapType{DataType}(Float64)\n\njulia> WrapType(::Type{T}) where T = WrapType{Type{T}}(T)\nWrapType\n\njulia> WrapType(Float64) # sharpened constructor, note more precise Type{Float64}\nWrapType{Type{Float64}}(Float64)\n\nThis behavior of Type{Float64} is an example of an abstract type subtyping a\nconcrete type (here DataType)."},{"title":"Type Aliases","page":"Types","location":"manual/types.html#Type-Aliases","category":"section","text":"Sometimes it is convenient to introduce a new name for an already expressible type.\nThis can be done with a simple assignment statement.\nFor example, UInt is aliased to either UInt32 or UInt64 as is\nappropriate for the size of pointers on the system:\n\n# 32-bit system:\njulia> UInt\nUInt32\n\n# 64-bit system:\njulia> UInt\nUInt64\n\nThis is accomplished via the following code in base/boot.jl:\n\nif Int === Int64\n    const UInt = UInt64\nelse\n    const UInt = UInt32\nend\n\nOf course, this depends on what Int is aliased to – but that is predefined to be the correct\ntype – either Int32 or Int64.\n\n(Note that unlike Int, Float does not exist as a type alias for a specific sized\nAbstractFloat. Unlike with integer registers, where the size of Int\nreflects the size of a native pointer on that machine, the floating point register sizes\nare specified by the IEEE-754 standard.)\n\nType aliases may be parametrized:\n\njulia> const Family{T} = Set{T}\nSet\n\njulia> Family{Char} === Set{Char}\ntrue"},{"title":"Operations on Types","page":"Types","location":"manual/types.html#Operations-on-Types","category":"section","text":"Since types in Julia are themselves objects, ordinary functions can operate on them. Some functions\nthat are particularly useful for working with or exploring types have already been introduced,\nsuch as the <: operator, which indicates whether its left hand operand is a subtype of its right\nhand operand.\n\nThe isa function tests if an object is of a given type and returns true or false:\n\njulia> isa(1, Int)\ntrue\n\njulia> isa(1, AbstractFloat)\nfalse\n\nThe typeof function, already used throughout the manual in examples, returns the type\nof its argument. Since, as noted above, types are objects, they also have types, and we can ask\nwhat their types are:\n\njulia> typeof(Rational{Int})\nDataType\n\njulia> typeof(Union{Real,String})\nUnion\n\nWhat if we repeat the process? What is the type of a type of a type? As it happens, types are\nall composite values and thus all have a type of DataType:\n\njulia> typeof(DataType)\nDataType\n\njulia> typeof(Union)\nDataType\n\nDataType is its own type.\n\nAnother operation that applies to some types is supertype, which reveals a type's\nsupertype. Only declared types (DataType) have unambiguous supertypes:\n\njulia> supertype(Float64)\nAbstractFloat\n\njulia> supertype(Number)\nAny\n\njulia> supertype(AbstractString)\nAny\n\njulia> supertype(Any)\nAny\n\nIf you apply supertype to other type objects (or non-type objects), a MethodError\nis raised:\n\njulia> supertype(Union{Float64,Int64})\nERROR: MethodError: no method matching supertype(::Type{Union{Float64, Int64}})\nThe function `supertype` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n[...]"},{"title":"Custom pretty-printing","page":"Types","location":"manual/types.html#man-custom-pretty-printing","category":"section","text":"Often, one wants to customize how instances of a type are displayed. This is accomplished by\noverloading the show function. For example, suppose we define a type to represent\ncomplex numbers in polar form:\n\njulia> struct Polar{T<:Real} <: Number\n           r::T\n           Θ::T\n       end\n\njulia> Polar(r::Real,Θ::Real) = Polar(promote(r,Θ)...)\nPolar\n\nHere, we've added a custom constructor function so that it can take arguments of different\nReal types and promote them to a common type (see Constructors\nand Conversion and Promotion).\n(Of course, we would have to define lots of other methods, too, to make it act like a\nNumber, e.g. +, *, one, zero, promotion rules and so on.) By default,\ninstances of this type display rather simply, with information about the type name and\nthe field values, as e.g. Polar{Float64}(3.0,4.0).\n\nIf we want it to display instead as 3.0 * exp(4.0im), we would define the following method to\nprint the object to a given output object io (representing a file, terminal, buffer, etcetera;\nsee Networking and Streams):\n\njulia> Base.show(io::IO, z::Polar) = print(io, z.r, \" * exp(\", z.Θ, \"im)\")\n\nMore fine-grained control over display of Polar objects is possible. In particular, sometimes\none wants both a verbose multi-line printing format, used for displaying a single object in the\nREPL and other interactive environments, and also a more compact single-line format used for\nprint or for displaying the object as part of another object (e.g. in an array). Although\nby default the show(io, z) function is called in both cases, you can define a different multi-line\nformat for displaying an object by overloading a three-argument form of show that takes the\ntext/plain MIME type as its second argument (see Multimedia I/O), for example:\n\njulia> Base.show(io::IO, ::MIME\"text/plain\", z::Polar{T}) where{T} =\n           print(io, \"Polar{$T} complex number:\\n   \", z)\n\n(Note that print(..., z) here will call the 2-argument show(io, z) method.) This results in:\n\njulia> Polar(3, 4.0)\nPolar{Float64} complex number:\n   3.0 * exp(4.0im)\n\njulia> [Polar(3, 4.0), Polar(4.0,5.3)]\n2-element Vector{Polar{Float64}}:\n 3.0 * exp(4.0im)\n 4.0 * exp(5.3im)\n\nwhere the single-line show(io, z) form is still used for an array of Polar values. Technically,\nthe REPL calls display(z) to display the result z of executing a line, which defaults to show(io, MIME(\"text/plain\"), z) (where io is an IOContext wrapper around stdout),\nwhich in turn defaults to show(io, z), but you should not define new display\nmethods unless you are defining a new multimedia display handler (see Multimedia I/O).\n\nMoreover, you can also define show methods for other MIME types in order to enable richer display\n(HTML, images, etcetera) of objects in environments that support this (e.g. IJulia). For example,\nwe can define formatted HTML display of Polar objects, with superscripts and italics, via:\n\njulia> Base.show(io::IO, ::MIME\"text/html\", z::Polar{T}) where {T} =\n           println(io, \"<code>Polar{$T}</code> complex number: \",\n                   z.r, \" <i>e</i><sup>\", z.Θ, \" <i>i</i></sup>\")\n\nA Polar object will then display automatically using HTML in an environment that supports HTML\ndisplay, but you can call show manually to get HTML output if you want:\n\njulia> show(stdout, \"text/html\", Polar(3.0,4.0))\n<code>Polar{Float64}</code> complex number: 3.0 <i>e</i><sup>4.0 <i>i</i></sup>\n\n<p>An HTML renderer would display this as: <code>Polar{Float64}</code> complex number: 3.0 <i>e</i><sup>4.0 <i>i</i></sup></p>\n\nAs a rule of thumb, the single-line show method should print a valid Julia expression for creating\nthe shown object. When this show method contains infix operators, such as the multiplication\noperator (*) in our single-line show method for Polar above, it may not parse correctly when\nprinted as part of another object. To see this, consider the expression object (see [Program\nrepresentation](@ref)) which takes the square of a specific instance of our Polar type:\n\njulia> a = Polar(3, 4.0)\nPolar{Float64} complex number:\n   3.0 * exp(4.0im)\n\njulia> print(:($a^2))\n3.0 * exp(4.0im) ^ 2\n\nBecause the operator ^ has higher precedence than * (see Operator Precedence and Associativity), this\noutput does not faithfully represent the expression a ^ 2 which should be equal to (3.0 * exp(4.0im)) ^ 2. To solve this issue, we must make a custom method for Base.show_unquoted(io::IO, z::Polar, indent::Int, precedence::Int), which is called internally by the expression object when\nprinting:\n\njulia> function Base.show_unquoted(io::IO, z::Polar, ::Int, precedence::Int)\n           if Base.operator_precedence(:*) <= precedence\n               print(io, \"(\")\n               show(io, z)\n               print(io, \")\")\n           else\n               show(io, z)\n           end\n       end\n\njulia> :($a^2)\n:((3.0 * exp(4.0im)) ^ 2)\n\nThe method defined above adds parentheses around the call to show when the precedence of the\ncalling operator is higher than or equal to the precedence of multiplication. This check allows\nexpressions which parse correctly without the parentheses (such as :($a + 2) and :($a == 2)) to\nomit them when printing:\n\njulia> :($a + 2)\n:(3.0 * exp(4.0im) + 2)\n\njulia> :($a == 2)\n:(3.0 * exp(4.0im) == 2)\n\nIn some cases, it is useful to adjust the behavior of show methods depending\non the context. This can be achieved via the IOContext type, which allows\npassing contextual properties together with a wrapped IO stream.\nFor example, we can build a shorter representation in our show method\nwhen the :compact property is set to true, falling back to the long\nrepresentation if the property is false or absent:\n\njulia> function Base.show(io::IO, z::Polar)\n           if get(io, :compact, false)::Bool\n               print(io, z.r, \"ℯ\", z.Θ, \"im\")\n           else\n               print(io, z.r, \" * exp(\", z.Θ, \"im)\")\n           end\n       end\n\nThis new compact representation will be used when the passed IO stream is an IOContext\nobject with the :compact property set. In particular, this is the case when printing\narrays with multiple columns (where horizontal space is limited):\n\njulia> show(IOContext(stdout, :compact=>true), Polar(3, 4.0))\n3.0ℯ4.0im\n\njulia> [Polar(3, 4.0) Polar(4.0,5.3)]\n1×2 Matrix{Polar{Float64}}:\n 3.0ℯ4.0im  4.0ℯ5.3im\n\nSee the IOContext documentation for a list of common properties which can be used\nto adjust printing."},{"title":"Output-function summary","page":"Types","location":"manual/types.html#Output-function-summary","category":"section","text":"Here is a brief summary of the different output functions in Julia and how they are related.\nMost new types should only need to define show methods, if anything.\n\ndisplay(x) tells the current environment to display x in whatever way it thinks best. (This might even be a graphical display in something like a Jupyter or Pluto notebook.) By default (e.g. in scripts or in the text REPL), it calls show(io, \"text/plain\", x), or equivalently show(io, MIME\"text/plain\"(), x), for an appropriate io stream. (In the REPL, io is an IOContext wrapper around stdout.) The REPL uses display to output the result of an evaluated expression.\nThe 3-argument show(io, ::MIME\"text/plain\", x) method performs verbose pretty-printing of x. By default (if no 3-argument method is defined for typeof(x)), it calls the 2-argument show(io, x). It is called by the 2-argument repr(\"text/plain\", x). Other 3-argument show methods can be defined for additional MIME types as discussed above, to enable richer display of x in some interactive environments.\nThe 2-argument show(io, x) is the default simple text representation of x. It is called by the 1-argument repr(x), and is typically the format you might employ to input x into Julia. The 1-argument show(x) calls show(stdout, x).\nprint(io, x) by default calls show(io, x), but a few types have a distinct print format — most notably, when x is a string, print outputs the raw text whereas show outputs an escaped string enclosed in quotation marks. The 1-argument print(x) calls print(stdout, x). print is also called by string(x).  See also println (to append a newline) and printstyled (to add colors etc.), both of which call print.\nwrite(io, x), if it is defined (it generally has no default definition for new types), writes a \"raw\" binary representation of x to io, e.g. an x::Int32 will be written as 4 bytes.\n\nIt is also helpful to be familiar with the metadata that can be attached to an io stream by an IOContext wrapper. For example, the REPL sets the :limit => true flag from display for an evaluated expression, in order to limit the output to fit in the terminal; you can query this flag with get(io, :limit, false). And when displaying an object contained within, for example, a multi-column matrix, the :compact => true flag could be set, which you can query with get(io, :compact, false)."},{"title":"\"Value types\"","page":"Types","location":"manual/types.html#\"Value-types\"","category":"section","text":"In Julia, you can't dispatch on a value such as true or false. However, you can dispatch\non parametric types, and Julia allows you to include \"plain bits\" values (Types, Symbols, Integers,\nfloating-point numbers, tuples, etc.) as type parameters. A common example is the dimensionality\nparameter in Array{T,N}, where T is a type (e.g., Float64) but N is just an Int.\n\nYou can create your own custom types that take values as parameters, and use them to control dispatch\nof custom types. By way of illustration of this idea, let's introduce the parametric type Val{x},\nand its constructor Val(x) = Val{x}(), which serves as a customary way to exploit this technique\nfor cases where you don't need a more elaborate hierarchy.\n\nVal is defined as:\n\njulia> struct Val{x}\n       end\n\njulia> Val(x) = Val{x}()\nVal\n\nThere is no more to the implementation of Val than this. Some functions in Julia's standard\nlibrary accept Val instances as arguments, and you can also use it to write your own functions.\n For example:\n\njulia> firstlast(::Val{true}) = \"First\"\nfirstlast (generic function with 1 method)\n\njulia> firstlast(::Val{false}) = \"Last\"\nfirstlast (generic function with 2 methods)\n\njulia> firstlast(Val(true))\n\"First\"\n\njulia> firstlast(Val(false))\n\"Last\"\n\nFor consistency across Julia, the call site should always pass a Val instance rather than using\na type, i.e., use foo(Val(:bar)) rather than foo(Val{:bar}).\n\nIt's worth noting that it's extremely easy to mis-use parametric \"value\" types, including Val;\nin unfavorable cases, you can easily end up making the performance of your code much worse.\n In particular, you would never want to write actual code as illustrated above. For more information\nabout the proper (and improper) uses of Val, please read the more extensive discussion in the performance tips.\n\n[1]: \"Small\" is defined by the max_union_splitting configuration, which currently defaults to 4.\n\n[2]: A few popular languages have singleton types, including Haskell, Scala and Ruby."},{"title":"Scoped Values","page":"Scoped Values","location":"base/scopedvalues.html#scoped-values","category":"section","text":"Scoped values provide an implementation of dynamic scoping in Julia.\n\nnote: Lexical scoping vs dynamic scoping\nLexical scoping is the default behavior in Julia.\nUnder lexical scoping the scope of a variable is determined by the lexical\n(textual) structure of a program.\nUnder dynamic scoping a variable is bound to the most recent assigned value\nduring the program's execution.\n\nThe state of a scoped value is dependent on the execution path of the program.\nThis means that for a scoped value you may observe multiple different values\nconcurrently.\n\ncompat: Julia 1.11\nScoped values were introduced in Julia 1.11. In Julia 1.8+ a compatible\nimplementation is available from the package ScopedValues.jl.\n\nIn its simplest form you can create a ScopedValue\nwith a default value and then use with or\n@with to enter a new dynamic scope. The new scope will\ninherit all values from the parent scope (and recursively from all outer scopes) with the\nprovided scoped value taking priority over previous definitions.\n\nLet's first look at an example of lexical scope. A let statement begins\na new lexical scope within which the outer definition of x is shadowed by\nits inner definition.\n\njulia> x = 1\n1\n\njulia> let x = 5\n           @show x\n       end;\nx = 5\n\njulia> @show x;\nx = 1\n\nIn the following example, since Julia uses lexical scope, the variable x in the body\nof f refers to the x defined in the global scope, and entering a let scope does\nnot change the value f observes.\n\njulia> x = 1\n1\n\njulia> f() = @show x\nf (generic function with 1 method)\n\njulia> let x = 5\n           f()\n       end;\nx = 1\n\njulia> f();\nx = 1\n\nNow using a ScopedValue we can use dynamic scoping.\n\njulia> using Base.ScopedValues\n\njulia> x = ScopedValue(1)\nScopedValue{Int64}(1)\n\njulia> f() = @show x[]\nf (generic function with 1 method)\n\njulia> with(x=>5) do\n           f()\n       end;\nx[] = 5\n\njulia> f();\nx[] = 1\n\nNote that the observed value of the ScopedValue is dependent on the execution\npath of the program.\n\nIt often makes sense to use a const variable to point to a scoped value,\nand you can set the value of multiple ScopedValues with one call to with.\n\nusing Base.ScopedValues\n\nf() = @show a[]\ng() = @show b[]\n\nconst a = ScopedValue(1)\nconst b = ScopedValue(2)\n\nf() # a[] = 1\ng() # b[] = 2\n\n# Enter a new dynamic scope and set value.\nwith(a => 3) do\n    f() # a[] = 3\n    g() # b[] = 2\n    with(a => 4, b => 5) do\n        f() # a[] = 4\n        g() # b[] = 5\n    end\n    f() # a[] = 3\n    g() # b[] = 2\nend\n\nf() # a[] = 1\ng() # b[] = 2\n\nScopedValues provides a macro version of with. The expression @with var=>val expr\nevaluates expr in a new dynamic scope with var set to val. @with var=>val expr\nis equivalent to with(var=>val) do expr end. However, with requires a zero-argument\nclosure or function, which results in an extra call-frame. As an example, consider the\nfollowing function f:\n\nusing Base.ScopedValues\nconst a = ScopedValue(1)\nf(x) = a[] + x\n\nIf you wish to run f in a dynamic scope with a set to 2, then you can use with:\n\nwith(() -> f(10), a=>2)\n\nHowever, this requires wrapping f in a zero-argument function. If you wish to avoid\nthe extra call-frame, then you can use the @with macro:\n\n@with a=>2 f(10)\n\nnote: Note\nDynamic scopes are inherited by Tasks, at the moment of task creation. Dynamic scopes are not propagated through Distributed.jl operations.\n\nIn the example below we open a new dynamic scope before launching a task.\nThe parent task and the two child tasks observe independent values of the\nsame scoped value at the same time.\n\nusing Base.ScopedValues\nimport Base.Threads: @spawn\n\nconst scoped_val = ScopedValue(1)\n@sync begin\n    with(scoped_val => 2)\n        @spawn @show scoped_val[] # 2\n    end\n    with(scoped_val => 3)\n        @spawn @show scoped_val[] # 3\n    end\n    @show scoped_val[] # 1\nend\n\nScoped values are constant throughout a scope, but you can store mutable\nstate in a scoped value. Just keep in mind that the usual caveats\nfor global variables apply in the context of concurrent programming.\n\nCare is also required when storing references to mutable state in scoped\nvalues. You might want to explicitly unshare mutable state\nwhen entering a new dynamic scope.\n\nusing Base.ScopedValues\nimport Base.Threads: @spawn\n\nconst sval_dict = ScopedValue(Dict())\n\n# Example of using a mutable value wrongly\n@sync begin\n    # `Dict` is not thread-safe the usage below is invalid\n    @spawn (sval_dict[][:a] = 3)\n    @spawn (sval_dict[][:b] = 3)\nend\n\n@sync begin\n    # If we instead pass a unique dictionary to each\n    # task we can access the dictionaries race free.\n    with(sval_dict => Dict()) do\n        @spawn (sval_dict[][:a] = 3)\n    end\n    with(sval_dict => Dict()) do\n        @spawn (sval_dict[][:b] = 3)\n    end\nend"},{"title":"Example","page":"Scoped Values","location":"base/scopedvalues.html#Example","category":"section","text":"In the example below we use a scoped value to implement a permission check in\na web-application. After determining the permissions of the request,\na new dynamic scope is entered and the scoped value LEVEL is set.\nOther parts of the application can query the scoped value and will receive\nthe appropriate value. Other alternatives like task-local storage and global variables\nare not well suited for this kind of propagation; our only alternative would have\nbeen to thread a value through the entire call-chain.\n\nusing Base.ScopedValues\n\nconst LEVEL = ScopedValue(:GUEST)\n\nfunction serve(request, response)\n    level = isAdmin(request) ? :ADMIN : :GUEST\n    with(LEVEL => level) do\n        Threads.@spawn handle(request, response)\n    end\nend\n\nfunction open(connection::Database)\n    level = LEVEL[]\n    if level !== :ADMIN\n        error(\"Access disallowed\")\n    end\n    # ... open connection\nend\n\nfunction handle(request, response)\n    # ...\n    open(Database(#=...=#))\n    # ...\nend"},{"title":"Idioms","page":"Scoped Values","location":"base/scopedvalues.html#Idioms","category":"section","text":""},{"title":"Unshare mutable state","page":"Scoped Values","location":"base/scopedvalues.html#unshare_mutable_state","category":"section","text":"using Base.ScopedValues\nimport Base.Threads: @spawn\n\nconst sval_dict = ScopedValue(Dict())\n\n# If you want to add new values to the dict, instead of replacing\n# it, unshare the values explicitly. In this example we use `merge`\n# to unshare the state of the dictionary in parent scope.\n@sync begin\n    with(sval_dict => merge(sval_dict[], Dict(:a => 10))) do\n        @spawn @show sval_dict[][:a]\n    end\n    @spawn sval_dict[][:a] = 3 # Not a race since they are unshared.\nend"},{"title":"Scoped values as globals","page":"Scoped Values","location":"base/scopedvalues.html#Scoped-values-as-globals","category":"section","text":"In order to access the value of a scoped value, the scoped value itself has to\nbe in (lexical) scope. This means most often you likely want to use scoped values\nas constant globals.\n\nusing Base.ScopedValues\nconst sval = ScopedValue(1)\n\nIndeed one can think of scoped values as hidden function arguments.\n\nThis does not preclude their use as non-globals.\n\nusing Base.ScopedValues\nimport Base.Threads: @spawn\n\nfunction main()\n    role = ScopedValue(:client)\n\n    function launch()\n        #...\n        role[]\n    end\n\n    @with role => :server @spawn launch()\n    launch()\nend\n\nBut it might have been simpler to just directly pass the function argument\nin these cases."},{"title":"Very many ScopedValues","page":"Scoped Values","location":"base/scopedvalues.html#Very-many-ScopedValues","category":"section","text":"If you find yourself creating many ScopedValue's for one given module,\nit may be better to use a dedicated struct to hold them.\n\nusing Base.ScopedValues\n\nBase.@kwdef struct Configuration\n    color::Bool = false\n    verbose::Bool = false\nend\n\nconst CONFIG = ScopedValue(Configuration(color=true))\n\n@with CONFIG => Configuration(color=CONFIG[].color, verbose=true) begin\n    @show CONFIG[].color # true\n    @show CONFIG[].verbose # true\nend"},{"title":"API docs","page":"Scoped Values","location":"base/scopedvalues.html#API-docs","category":"section","text":""},{"title":"Implementation notes and performance","page":"Scoped Values","location":"base/scopedvalues.html#Implementation-notes-and-performance","category":"section","text":"Scopes use a persistent dictionary. Lookup and insertion is O(log(32, n)),\nupon dynamic scope entry a small amount of data is copied and the unchanged\ndata is shared among other scopes.\n\nThe Scope object itself is not user-facing and may be changed in a future\nversion of Julia."},{"title":"Design inspiration","page":"Scoped Values","location":"base/scopedvalues.html#Design-inspiration","category":"section","text":"This design was heavily inspired by JEPS-429,\nwhich in turn was inspired by dynamically scoped free variables in many Lisp dialects. In particular Interlisp-D and its deep binding strategy.\n\nA prior design discussed was context variables ala PEPS-567 and implemented in Julia as ContextVariablesX.jl."},{"title":"Base.ScopedValues.ScopedValue","page":"Scoped Values","location":"base/scopedvalues.html#Base.ScopedValues.ScopedValue","category":"type","text":"ScopedValue(x)\n\nCreate a container that propagates values across dynamic scopes.\nUse with to create and enter a new dynamic scope.\n\nValues can only be set when entering a new dynamic scope,\nand the value referred to will be constant during the\nexecution of a dynamic scope.\n\nDynamic scopes are propagated across tasks.\n\nExamples\n\njulia> using Base.ScopedValues;\n\njulia> const sval = ScopedValue(1);\n\njulia> sval[]\n1\n\njulia> with(sval => 2) do\n           sval[]\n       end\n2\n\njulia> sval[]\n1\n\ncompat: Julia 1.11\nScoped values were introduced in Julia 1.11. In Julia 1.8+ a compatible\nimplementation is available from the package ScopedValues.jl.\n\n\n\n\n\n"},{"title":"Base.ScopedValues.with","page":"Scoped Values","location":"base/scopedvalues.html#Base.ScopedValues.with","category":"function","text":"with(f, (var::ScopedValue{T} => val)...)\n\nExecute f in a new dynamic scope with var set to val. val will be converted\nto type T.\n\nSee also: ScopedValues.@with, ScopedValues.ScopedValue, ScopedValues.get.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> a = ScopedValue(1);\n\njulia> f(x) = a[] + x;\n\njulia> f(10)\n11\n\njulia> with(a=>2) do\n           f(10)\n       end\n12\n\njulia> f(10)\n11\n\njulia> b = ScopedValue(2);\n\njulia> g(x) = a[] + b[] + x;\n\njulia> with(a=>10, b=>20) do\n           g(30)\n       end\n60\n\njulia> with(() -> a[] * b[], a=>3, b=>4)\n12\n\n\n\n\n\n"},{"title":"Base.ScopedValues.@with","page":"Scoped Values","location":"base/scopedvalues.html#Base.ScopedValues.@with","category":"macro","text":"@with (var::ScopedValue{T} => val)... expr\n\nMacro version of with. The expression @with var=>val expr evaluates expr in a\nnew dynamic scope with var set to val. val will be converted to type T.\n@with var=>val expr is equivalent to with(var=>val) do expr end, but @with\navoids creating a closure.\n\nSee also: ScopedValues.with, ScopedValues.ScopedValue, ScopedValues.get.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> const a = ScopedValue(1);\n\njulia> f(x) = a[] + x;\n\njulia> @with a=>2 f(10)\n12\n\njulia> @with a=>3 begin\n           x = 100\n           f(x)\n       end\n103\n\n\n\n\n\n"},{"title":"Base.isassigned","page":"Scoped Values","location":"base/scopedvalues.html#Base.isassigned-Tuple{Base.ScopedValues.ScopedValue}","category":"method","text":"isassigned(val::ScopedValue)\n\nTest whether a ScopedValue has an assigned value.\n\nSee also: ScopedValues.with, ScopedValues.@with, ScopedValues.get.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> a = ScopedValue(1); b = ScopedValue{Int}();\n\njulia> isassigned(a)\ntrue\n\njulia> isassigned(b)\nfalse\n\n\n\n\n\n"},{"title":"Base.ScopedValues.get","page":"Scoped Values","location":"base/scopedvalues.html#Base.ScopedValues.get","category":"function","text":"get(val::ScopedValue{T})::Union{Nothing, Some{T}}\nget(val::LazyScopedValue{T})::Union{Nothing, Some{T}}\n\nIf the scoped value isn't set and doesn't have a default value,\nreturn nothing. Otherwise returns Some{T} with the current\nvalue.\n\nSee also: ScopedValues.with, ScopedValues.@with, ScopedValues.ScopedValue.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> a = ScopedValue(42); b = ScopedValue{Int}();\n\njulia> ScopedValues.get(a)\nSome(42)\n\njulia> isnothing(ScopedValues.get(b))\ntrue\n\n\n\n\n\n"},{"title":"Building Julia (Detailed)","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Building-Julia-(Detailed)","category":"section","text":""},{"title":"Downloading the Julia source code","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Downloading-the-Julia-source-code","category":"section","text":"If you are behind a firewall, you may need to use the https protocol instead of the git protocol:\n\ngit config --global url.\"https://\".insteadOf git://\n\nBe sure to also configure your system to use the appropriate proxy\nsettings, e.g. by setting the https_proxy and http_proxy\nvariables."},{"title":"Building Julia","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Building-Julia","category":"section","text":"When compiled the first time, the build will automatically download\npre-built [external\ndependencies](#Required-Build-Tools-and-External-Libraries). If you\nprefer to build all the dependencies on your own, or are building on a system that cannot\naccess the network during the build process, add the following in Make.user:\n\nUSE_BINARYBUILDER=0\n\nBuilding Julia requires 5GiB if building all dependencies and approximately 4GiB of virtual memory.\n\nTo perform a parallel build, use make -j N and supply the maximum\nnumber of concurrent processes. If the defaults in the build do not work for you, and\nyou need to set specific make parameters, you can save them in\nMake.user, and place the file in the root of your Julia source. The\nbuild will automatically check for the existence of Make.user and\nuse it if it exists.\n\nYou can create out-of-tree builds of Julia by specifying make O=<build-directory> configure on the command line. This will create a\ndirectory mirror, with all of the necessary Makefiles to build Julia,\nin the specified directory. These builds will share the source files\nin Julia and deps/srccache. Each out-of-tree build directory can\nhave its own Make.user file to override the global Make.user file\nin the top-level folder. After modifying the Make.user file if necessary,\nbuild using: make -C <build-directory>.\n\nIf everything works correctly, there will be a symlink to\nthe julia executable in the build directory which can be run as:\n\n./julia\n\nThe actual executable is in <build-directory>/usr/bin.\nAfter running this, you will see a Julia banner and an\ninteractive prompt into which you can enter expressions for\nevaluation. (Errors related to libraries might be caused by old,\nincompatible libraries sitting around in your PATH. In this case, try\nmoving the julia directory earlier in the PATH). Note that most of\nthe instructions above apply to unix systems.\n\nTo run julia from anywhere you can:\n\nadd an alias (in bash: echo \"alias julia='<build-directory>/usr/bin/julia'\" >> ~/.bashrc && source ~/.bashrc), or\nadd a soft link to the julia executable in the <build-directory>/usr/bin directory to /usr/local/bin (or any suitable directory already in your path), or\nadd the julia directory to your executable path for this shell session (in bash: export PATH=\"$(pwd):$PATH\" ; in csh or tcsh:\n\nset path= ( $path $cwd ) ), or\n\nadd the julia directory to your executable path permanently (e.g. in .bash_profile), or\nwrite prefix=/path/to/install/folder into Make.user and then run make install. If there is a version of Julia already installed in this folder, you should delete it before running make install.\n\nSome of the options you can set to control the build of Julia are listed and documented at the beginning of the file Make.inc, but you should never edit it for this purpose, use Make.user instead.\n\nJulia's Makefiles define convenient automatic rules called print-<VARNAME> for printing the value of variables, replacing <VARNAME> with the name of the variable to print the value of.\nFor example\n\n$ make print-JULIA_PRECOMPILE\nJULIA_PRECOMPILE=1\n\nThese rules are useful for debugging purposes.\n\nNow you should be able to run Julia like this:\n\njulia\n\nIf you are building a Julia package for distribution on Linux, macOS,\nor Windows, take a look at the detailed notes in\ndistributing.md."},{"title":"Updating an existing source tree","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Updating-an-existing-source-tree","category":"section","text":"If you have previously downloaded julia using git clone, you can update the\nexisting source tree using git pull rather than starting anew:\n\ncd julia\ngit pull && make\n\nAssuming that you had made no changes to the source tree that will conflict\nwith upstream updates, these commands will trigger a build to update to the\nlatest version."},{"title":"General troubleshooting","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#General-troubleshooting","category":"section","text":"Over time, the base library may accumulate enough changes such that the\nbootstrapping process in building the system image will fail. If this\nhappens, the build may fail with an error like\n *** This error is usually fixed by running 'make clean'. If the error persists, try 'make cleanall' ***\nAs described, running make clean && make is usually sufficient.\nOccasionally, the stronger cleanup done by make cleanall is needed.\nNew versions of external dependencies may be introduced which may\noccasionally cause conflicts with existing builds of older versions.\na. Special make targets exist to help wipe the existing build of a\n   dependency. For example, make -C deps clean-llvm will clean out the\n   existing build of llvm so that llvm will be rebuilt from the\n   downloaded source distribution the next time make is called.\n   make -C deps distclean-llvm is a stronger wipe which will also delete\n   the downloaded source distribution, ensuring that a fresh copy of the\n   source distribution will be downloaded and that any new patches will be\n   applied the next time make is called.\nb. To delete existing binaries of julia and all its dependencies,\n   delete the ./usr directory in the source tree.\nIf you've updated macOS recently, be sure to run xcode-select --install to update the command line tools.\nOtherwise, you could run into errors for missing headers and libraries, such as\nld: library not found for -lcrt1.10.6.o.\nIf you've moved the source directory, you might get errors such as\n CMake Error: The current CMakeCache.txt directory ... is different than the directory ... where     CMakeCache.txt was created., in which case you may delete the offending dependency under deps\nIn extreme cases, you may wish to reset the source tree to a pristine state.\nThe following git commands may be helpful:\n git reset --hard #Forcibly remove any changes to any files under version control\n git clean -x -f -d #Forcibly remove any file or directory not under version control\nTo avoid losing work, make sure you know what these commands do before you\nrun them. git will not be able to undo these changes!"},{"title":"Platform-Specific Notes","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Platform-Specific-Notes","category":"section","text":"Notes for various operating systems:\n\nLinux\nmacOS\nWindows\nFreeBSD\n\nNotes for various architectures:\n\nARM\nRISC-V"},{"title":"Required Build Tools and External Libraries","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Required-Build-Tools-and-External-Libraries","category":"section","text":"Building Julia requires that the following software be installed:\n\nGNU make                — building dependencies.\ngcc & g++ (>= 7.1) or Clang (>= 5.0, >= 9.3 for Apple Clang) — compiling and linking C, C++.\nOn Linux with g++, the static version of libstdc++ is also required.  If it is unavailable, set USE_RT_STATIC_LIBSTDCXX=0 in Make.user.\nlibatomic          — provided by [gcc] and needed to support atomic operations.\npython (>=2.7)          — needed to build LLVM.\ngfortran                — compiling and linking Fortran libraries.\nperl                    — preprocessing of header files of libraries.\nwget, curl, or fetch (FreeBSD) — to automatically download external libraries.\nm4                      — needed to build GMP.\nawk                     — helper tool for Makefiles.\npatch                   — for modifying source code.\ncmake (>= 3.4.3)        — needed to build libgit2.\npkg-config              — needed to build libgit2 correctly, especially for proxy support.\npowershell (>= 3.0)     — necessary only on Windows.\nwhich                   — needed for checking build dependencies.\ndiffutils                   - cmp is used by the makefiles\n\nOn Debian-based distributions (e.g. Ubuntu), you can easily install them with apt-get:\n\nsudo apt-get install build-essential libatomic1 python3 gfortran perl wget m4 cmake pkg-config curl\n\nOn Red Hat-based distributions (e.g. Fedora, CentOS), you can install them with yum:\n\nsudo dnf install gcc gcc-c++ gcc-gfortran python3 perl wget m4 cmake pkgconfig curl which diffutils libatomic libstdc++-static\n\nJulia uses the following external libraries, which are automatically\ndownloaded (or in a few cases, included in the Julia source\nrepository) and then compiled from source the first time you run\nmake. The specific version numbers of these libraries that Julia\nuses are listed in deps/$(libname).version:\n\nLLVM (15.0 + patches) — compiler infrastructure (see note below).\nFemtoLisp            — packaged with Julia source, and used to implement the compiler front-end.\nlibuv  (custom fork) — portable, high-performance event-based I/O library.\nOpenLibm             — portable libm library containing elementary math functions.\nDSFMT                — fast Mersenne Twister pseudorandom number generator library.\nOpenBLAS             — fast, open, and maintained [basic linear algebra subprograms (BLAS)]\nLAPACK               — library of linear algebra routines for solving systems of simultaneous linear equations, least-squares solutions of linear systems of equations, eigenvalue problems, and singular value problems.\nMKL (optional)       – OpenBLAS and LAPACK may be replaced by Intel's MKL library.\nSuiteSparse          — library of linear algebra routines for sparse matrices.\nPCRE                 — Perl-compatible regular expressions library.\nGMP                  — GNU multiple precision arithmetic library, needed for BigInt support.\nMPFR                 — GNU multiple precision floating point library, needed for arbitrary precision floating point (BigFloat) support.\nlibgit2              — Git linkable library, used by Julia's package manager.\ncurl                 — libcurl provides download and proxy support.\nlibssh2              — library for SSH transport, used by libgit2 for packages with SSH remotes.\nOpenSSL              — library used for cryptography and transport layer security, used by libgit2 and libssh2.\nutf8proc             — a library for processing UTF-8 encoded Unicode strings.\nLLVM libunwind       — LLVM's fork of [libunwind], a library that determines the call-chain of a program.\nITTAPI               — Intel's Instrumentation and Tracing Technology and Just-In-Time API."},{"title":"Build dependencies","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Build-dependencies","category":"section","text":"If you already have one or more of these packages installed on your system, you can prevent Julia from compiling duplicates of these libraries by passing USE_SYSTEM_...=1 to make or adding the line to Make.user. The complete list of possible flags can be found in Make.inc.\n\nPlease be aware that this procedure is not officially supported, as it introduces additional variability into the installation and versioning of the dependencies, and is recommended only for system package maintainers. Unexpected compile errors may result, as the build system will do no further checking to ensure the proper packages are installed."},{"title":"LLVM","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#LLVM","category":"section","text":"The most complicated dependency is LLVM, for which we require additional patches from upstream (LLVM is not backward compatible).\n\nFor packaging Julia with LLVM, we recommend either:\n\nbundling a Julia-only LLVM library inside the Julia package, or\nadding the patches to the LLVM package of the distribution.\nA complete list of patches is available in on Github see the julia-release/18.x branch.\nThe remaining patches are all upstream bug fixes, and have been contributed into upstream LLVM.\n\nUsing an unpatched or different version of LLVM will result in errors and/or poor performance.\nYou can build a different version of LLVM from a remote Git repository with the following options in the Make.user file:\n\n# Force source build of LLVM\nUSE_BINARYBUILDER_LLVM = 0\n# Use Git for fetching LLVM source code\n# this is either `1` to get all of them\nDEPS_GIT = 1\n# or a space-separated list of specific dependencies to download with git\nDEPS_GIT = llvm\n\n# Other useful options:\n#URL of the Git repository you want to obtain LLVM from:\n#  LLVM_GIT_URL = ...\n#Name of the alternate branch to clone from git\n#  LLVM_BRANCH = julia-16.0.6-0\n#SHA hash of the alternate commit to check out automatically\n#  LLVM_SHA1 = $(LLVM_BRANCH)\n#List of LLVM targets to build. It is strongly recommended to keep at least all the\n#default targets listed in `deps/llvm.mk`, even if you don't necessarily need all of them.\n#  LLVM_TARGETS = ...\n#Use ccache for faster recompilation in case you need to restart a build.\n#  USECCACHE = 1\n#  CMAKE_GENERATOR=Ninja\n#  LLVM_ASSERTIONS=1\n#  LLVM_DEBUG=Symbols\n\nThe various build phases are controlled by specific files:\n\ndeps/llvm.version : touch or change to checkout a new version, make get-llvm check-llvm\ndeps/srccache/llvm/source-extracted : result of make extract-llvm\ndeps/llvm/build_Release*/build-configured : result of make configure-llvm\ndeps/llvm/build_Release*/build-configured : result of make compile-llvm\nusr-staging/llvm/build_Release*.tgz : result of make stage-llvm (regenerate with make reinstall-llvm)\nusr/manifest/llvm : result of make install-llvm (regenerate with make uninstall-llvm)\nmake version-check-llvm : runs every time to warn the user if there are local modifications\n\nThough Julia can be built with newer LLVM versions, support for this should be regarded as experimental and not suitable for packaging."},{"title":"libuv","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#libuv","category":"section","text":"Julia uses a custom fork of libuv. It is a small dependency, and can be safely bundled in the same package as Julia, and will not conflict with the system library. Julia builds should not try to use the system libuv."},{"title":"BLAS and LAPACK","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#BLAS-and-LAPACK","category":"section","text":"As a high-performance numerical language, Julia should be linked to a multi-threaded BLAS and LAPACK, such as OpenBLAS or ATLAS, which will provide much better performance than the reference libblas implementations which may be default on some systems."},{"title":"Source distributions of releases","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Source-distributions-of-releases","category":"section","text":"Each pre-release and release of Julia has a \"full\" source distribution and a \"light\" source\ndistribution.\n\nThe full source distribution contains the source code for Julia and all dependencies so\nthat it can be built from source without an internet connection. The light source\ndistribution does not include the source code of dependencies.\n\nFor example, julia-1.0.0.tar.gz is the light source distribution for the v1.0.0 release\nof Julia, while julia-1.0.0-full.tar.gz is the full source distribution."},{"title":"Building Julia from source with a Git checkout of a stdlib","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Building-Julia-from-source-with-a-Git-checkout-of-a-stdlib","category":"section","text":"If you need to build Julia from source with a Git checkout of a stdlib, then use make DEPS_GIT=NAME_OF_STDLIB when building Julia.\n\nFor example, if you need to build Julia from source with a Git checkout of Pkg, then use make DEPS_GIT=Pkg when building Julia. The Pkg repo is in stdlib/Pkg, and created initially with a detached HEAD. If you're doing this from a pre-existing Julia repository, you may need to make clean beforehand.\n\nIf you need to build Julia from source with Git checkouts of more than one stdlib, then DEPS_GIT should be a space-separated list of the stdlib names. For example, if you need to build Julia from source with a Git checkout of Pkg, Tar, and Downloads, then use make DEPS_GIT='Pkg Tar Downloads' when building Julia."},{"title":"Building an \"assert build\" of Julia","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Building-an-\"assert-build\"-of-Julia","category":"section","text":"An \"assert build\" of Julia is a build that was built with both FORCE_ASSERTIONS=1 and\nLLVM_ASSERTIONS=1. To build an assert build, define both of the following variables\nin your Make.user file:\n\nFORCE_ASSERTIONS=1\nLLVM_ASSERTIONS=1\n\nPlease note that assert builds of Julia will be slower than regular (non-assert) builds."},{"title":"Building a debug build of Julia","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Building-a-debug-build-of-Julia","category":"section","text":"A full debug build of Julia can be built with make debug.  This builds a debug\nversion of libjulia and uses it to bootstrap the compiler, before creating a\nsystem image with debug symbols enabled.  This can take more than 15 minutes.\n\nAlthough it may result in some differences, a debug build can be built much\nquicker by bootstrapping from a release build:\n\n$ make julia-src-release julia-sysbase-release\n$ make julia-sysimg-debug CROSS_BOOTSTRAP_JULIA=$PWD/usr/bin/julia CROSS_BOOTSTRAP_SYSBASE=$PWD/usr/lib/julia/sysbase.so"},{"title":"Building 32-bit Julia on a 64-bit machine","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Building-32-bit-Julia-on-a-64-bit-machine","category":"section","text":"Occasionally, bugs specific to 32-bit architectures may arise, and when this happens it is useful to be able to debug the problem on your local machine. Since most modern 64-bit systems support running programs built for 32-bit ones, if you don't have to recompile Julia from source (e.g. you mainly need to inspect the behavior of a 32-bit Julia without having to touch the C code), you can likely use a 32-bit build of Julia for your system that you can obtain from the official downloads page.\nHowever, if you do need to recompile Julia from source one option is to use a Docker container of a 32-bit system. At least for now, building a 32-bit version of Julia is relatively straightforward using ubuntu 32-bit docker images. In brief, after setting up docker here are the required steps:\n\n$ docker pull i386/ubuntu\n$ docker run --platform i386 -i -t i386/ubuntu /bin/bash\n\nAt this point you should be in a 32-bit machine console (note that uname reports the host architecture, so will still say 64-bit, but this will not affect the Julia build). You can add packages and compile code; when you exit, all the changes will be lost, so be sure to finish your analysis in a single session or set up a copy/pastable script you can use to set up your environment.\n\nFrom this point, you should\n\n# apt update\n\n(Note that sudo isn't installed, but neither is it necessary since you are running as root, so you can omit sudo from all commands.)\n\nThen add all the build dependencies, a console-based editor of your choice, git, and anything else you'll need (e.g., gdb, rr, etc). Pick a directory to work in and git clone Julia, check out the branch you wish to debug, and build Julia as usual."},{"title":"Update the version number of a dependency","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Update-the-version-number-of-a-dependency","category":"section","text":"There are two types of builds\n\nBuild everything (deps/ and src/) from source code.\n (Add USE_BINARYBUILDER=0 to Make.user, see Building Julia)\nBuild from source (src/) with pre-compiled dependencies (default)\n\nWhen you want to update the version number of a dependency in deps/,\nyou may want to use the following checklist:\n\n### Check list\n\nVersion numbers:\n- [ ] `deps/$(libname).version`: `LIBNAME_VER`, `LIBNAME_BRANCH`, `LIBNAME_SHA1` and `LIBNAME_JLL_VER`\n- [ ] `stdlib/$(LIBNAME_JLL_NAME)_jll/Project.toml`: `version`\n\nChecksum:\n- [ ] `deps/checksums/$(libname)`\n- [ ] `deps/checksums/$(LIBNAME_JLL_NAME)-*/`: `md5` and `sha512`\n\nPatches:\n- [ ] `deps/$(libname).mk`\n- [ ] `deps/patches/$(libname)-*.patch`\n\nNote:\n\nFor specific dependencies, some items in the checklist may not exist.\nFor checksum file, it may be a single file without a suffix, or a folder containing two files."},{"title":"Example: OpenLibm","page":"Building Julia (Detailed)","location":"devdocs/build/build.html#Example:-OpenLibm","category":"section","text":"Update Version numbers in deps/openlibm.version\nOPENLIBM_VER := 0.X.Y\nOPENLIBM_BRANCH = v0.X.Y\nOPENLIBM_SHA1 = new-sha1-hash\nUpdate Version number in stdlib/OpenLibm_jll/Project.toml\nversion = \"0.X.Y+0\"\nUpdate checksums in deps/checksums/openlibm\nmake -f contrib/refresh_checksums.mk openlibm\nCheck if the patch files deps/patches/openlibm-*.patch exist\nif patches don't exist, skip.\nif patches exist, check if they have been merged into the new version and need to be removed.\n  When deleting a patch, remember to modify the corresponding Makefile file (deps/openlibm.mk)."},{"title":"Julia Functions","page":"Julia Functions","location":"devdocs/functions.html#Julia-Functions","category":"section","text":"This document will explain how functions, method definitions, and method tables work."},{"title":"Method Tables","page":"Julia Functions","location":"devdocs/functions.html#Method-Tables","category":"section","text":"Every function in Julia is a generic function. A generic function is conceptually a single function,\nbut consists of many definitions, or methods. The methods of a generic function are stored in a\nmethod table. There is one global method table (type MethodTable) named Core.methodtable. Any\ndefault operation on methods (such as calls) uses that table."},{"title":"Function calls","page":"Julia Functions","location":"devdocs/functions.html#Function-calls","category":"section","text":"Given the call f(x, y), the following steps are performed: First, a tuple type is formed,\nTuple{typeof(f), typeof(x), typeof(y)}. Note that the type of the function itself is the first\nelement. This is because the function itself participates symmetrically in method lookup with the\nother arguments. This tuple type is looked up in the global method table. However, the system can\nthen cache the results, so these steps can be skipped later for similar lookups.\n\nThis dispatch process is performed by jl_apply_generic, which takes two arguments: a pointer\nto an array of the values f, x, and y, and the number of values (in this case 3).\n\nThroughout the system, there are two kinds of APIs that handle functions and argument lists: those\nthat accept the function and arguments separately, and those that accept a single argument structure.\nIn the first kind of API, the \"arguments\" part does not contain information about the function,\nsince that is passed separately. In the second kind of API, the function is the first element\nof the argument structure.\n\nFor example, the following function for performing a call accepts just an args pointer, so the\nfirst element of the args array will be the function to call:\n\njl_value_t *jl_apply(jl_value_t **args, uint32_t nargs)\n\nThis entry point for the same functionality accepts the function separately, so the args array\ndoes not contain the function:\n\njl_value_t *jl_call(jl_value_t *f, jl_value_t **args, int32_t nargs);"},{"title":"Adding methods","page":"Julia Functions","location":"devdocs/functions.html#Adding-methods","category":"section","text":"Given the above dispatch process, conceptually all that is needed to add a new method is (1) a\ntuple type, and (2) code for the body of the method. jl_method_def implements this operation."},{"title":"Creating generic functions","page":"Julia Functions","location":"devdocs/functions.html#Creating-generic-functions","category":"section","text":"Since every object is callable, nothing special is needed to create a generic function. Therefore\njl_new_generic_function simply creates a new singleton (0 size) subtype of Function and returns\nits instance. A function can have a mnemonic \"display name\" which is used in debug info and when\nprinting objects. For example the name of Base.sin is sin. By convention, the name of the\ncreated type is the same as the function name, with a # prepended. So typeof(sin) is Base.#sin."},{"title":"Closures","page":"Julia Functions","location":"devdocs/functions.html#Closures","category":"section","text":"A closure is simply a callable object with field names corresponding to captured variables. For\nexample, the following code:\n\nfunction adder(x)\n    return y->x+y\nend\n\nis lowered to (roughly):\n\nstruct ##1{T}\n    x::T\nend\n\n(_::##1)(y) = _.x + y\n\nfunction adder(x)\n    return ##1(x)\nend"},{"title":"Constructors","page":"Julia Functions","location":"devdocs/functions.html#Constructors","category":"section","text":"A constructor call is just a call to a type, to a method defined on Type{T}."},{"title":"Builtins","page":"Julia Functions","location":"devdocs/functions.html#Builtins","category":"section","text":"The \"builtin\" functions, defined in the Core module, are:\n\nfunction lines(words)\n    io = IOBuffer()\n    n = 0\n    for w in words\n        if n+length(w) > 80\n            print(io, '\\n', w)\n            n = length(w)\n        elseif n == 0\n            print(io, w);\n            n += length(w)\n        else\n            print(io, ' ', w);\n            n += length(w)+1\n        end\n    end\n    takestring!(io)\nend\nimport Markdown\n[string(n) for n in names(Core;all=true)\n    if getfield(Core,n) isa Core.Builtin && nameof(getfield(Core,n)) === n] |>\n    lines |>\n    s ->  \"```\\n$s\\n```\" |>\n    Markdown.parse\n\nThese are mostly singleton objects all of whose types are subtypes of Builtin, which is a\nsubtype of Function. Their purpose is to expose entry points in the run time that use the\n\"jlcall\" calling convention:\n\njl_value_t *(jl_value_t*, jl_value_t**, uint32_t)"},{"title":"Keyword arguments","page":"Julia Functions","location":"devdocs/functions.html#Keyword-arguments","category":"section","text":"Keyword arguments work by adding methods to the kwcall function. This function\nis usually the \"keyword argument sorter\" or \"keyword sorter\", which then calls\nthe inner body of the function (defined anonymously).\nEvery definition in the kwsorter function has the same arguments as some definition in the normal\nmethod table, except with a single NamedTuple argument prepended, which gives\nthe names and values of passed keyword arguments. The kwsorter's job is to move keyword arguments\ninto their canonical positions based on name, plus evaluate and substitute any needed default value\nexpressions. The result is a normal positional argument list, which is then passed to yet another\ncompiler-generated function.\n\nThe easiest way to understand the process is to look at how a keyword argument method definition\nis lowered. The code:\n\nfunction circle(center, radius; color = black, fill::Bool = true, options...)\n    # draw\nend\n\nactually produces three method definitions. The first is a function that accepts all arguments\n(including keyword arguments) as positional arguments, and includes the code for the method body.\nIt has an auto-generated name:\n\nfunction #circle#1(color, fill::Bool, options, circle, center, radius)\n    # draw\nend\n\nThe second method is an ordinary definition for the original circle function, which handles\nthe case where no keyword arguments are passed:\n\nfunction circle(center, radius)\n    #circle#1(black, true, pairs(NamedTuple()), circle, center, radius)\nend\n\nThis simply dispatches to the first method, passing along default values.\npairs is applied to the named tuple of rest arguments to provide key-value pair iteration.\nNote that if the method doesn't accept rest keyword arguments then this argument\nis absent.\n\nFinally there is the kwsorter definition:\n\nfunction (::Core.kwcall)(kws, circle, center, radius)\n    if haskey(kws, :color)\n        color = kws.color\n    else\n        color = black\n    end\n    # etc.\n\n    # put remaining kwargs in `options`\n    options = structdiff(kws, NamedTuple{(:color, :fill)})\n\n    # if the method doesn't accept rest keywords, throw an error\n    # unless `options` is empty\n\n    #circle#1(color, fill, pairs(options), circle, center, radius)\nend"},{"title":"Compiler efficiency issues","page":"Julia Functions","location":"devdocs/functions.html#compiler-efficiency-issues","category":"section","text":"Generating a new type for every function has potentially serious consequences for compiler resource\nuse when combined with Julia's \"specialize on all arguments by default\" design. Indeed, the initial\nimplementation of this design suffered from much longer build and test times, higher memory use,\nand a system image nearly 2x larger than the baseline. In a naive implementation, the problem\nis bad enough to make the system nearly unusable. Several significant optimizations were needed\nto make the design practical.\n\nThe first issue is excessive specialization of functions for different values of function-valued\narguments. Many functions simply \"pass through\" an argument to somewhere else, e.g. to another\nfunction or to a storage location. Such functions do not need to be specialized for every closure\nthat might be passed in. Fortunately this case is easy to distinguish by simply considering whether\na function calls one of its arguments (i.e. the argument appears in \"head position\" somewhere).\nPerformance-critical higher-order functions like map certainly call their argument function\nand so will still be specialized as expected. This optimization is implemented by recording which\narguments are called during the analyze-variables pass in the front end. When cache_method\nsees an argument in the Function type hierarchy passed to a slot declared as Any or Function,\nit behaves as if the @nospecialize annotation were applied. This heuristic seems to be extremely\neffective in practice.\n\nThe next issue concerns the structure of method tables. Empirical studies show that the vast\nmajority of dynamically-dispatched calls involve one or two arguments. In turn, many of these cases\ncan be resolved by considering only the first argument. (Aside: proponents of single dispatch would\nnot be surprised by this at all. However, this argument means \"multiple dispatch is easy to optimize\nin practice\", and that we should therefore use it, not \"we should use single dispatch\"!). So the\nmethod table and cache splits up on the structure based on a left-to-right decision tree so allow\nefficient nearest-neighbor searches.\n\nThe front end generates type declarations for all closures. Initially, this was implemented by\ngenerating normal type declarations. However, this produced an extremely large number of constructors,\nall of which were trivial (simply passing all arguments through to new). Since methods are partially\nordered, inserting all of these methods is O(n²), plus there are just too many of them to keep\naround. This was optimized by generating struct_type expressions directly (bypassing default\nconstructor generation), and using new directly to create closure instances. Not the prettiest\nthing ever, but you do what you gotta do.\n\nThe next problem was the @test macro, which generated a 0-argument closure for each test case.\nThis is not really necessary, since each test case is simply run once in place. Therefore, @test\nwas modified to expand to a try-catch block that records the test result (true, false, or exception\nraised) and calls the test suite handler on it."},{"title":"Command-line Interface","page":"Command-line Interface","location":"manual/command-line-interface.html#cli","category":"section","text":""},{"title":"Using arguments inside scripts","page":"Command-line Interface","location":"manual/command-line-interface.html#Using-arguments-inside-scripts","category":"section","text":"When running a script using julia, you can pass additional arguments to your script:\n\n$ julia script.jl arg1 arg2...\n\nThese additional command-line arguments are passed in the global constant ARGS. The\nname of the script itself is passed in as the global PROGRAM_FILE. Note that ARGS is\nalso set when a Julia expression is given using the -e option on the command line (see the\njulia help output below) but PROGRAM_FILE will be empty. For example, to just print the\narguments given to a script, you could do this:\n\n$ julia -e 'println(PROGRAM_FILE); for x in ARGS; println(x); end' foo bar\n\nfoo\nbar\n\nOr you could put that code into a script and run it:\n\n$ echo 'println(PROGRAM_FILE); for x in ARGS; println(x); end' > script.jl\n$ julia script.jl foo bar\nscript.jl\nfoo\nbar\n\nThe -- delimiter can be used to separate command-line arguments intended for the script file from arguments intended for Julia:\n\n$ julia --color=yes -O -- script.jl arg1 arg2..\n\nSee also Scripting for more information on writing Julia scripts."},{"title":"The Main.main entry point","page":"Command-line Interface","location":"manual/command-line-interface.html#The-Main.main-entry-point","category":"section","text":"As of Julia 1.11, Base exports the macro @main. This macro expands to the symbol main,\nbut at the conclusion of executing a script or expression, julia will attempt to execute\nMain.main(Base.ARGS) if such a function Main.main has been defined and this behavior was opted into\nby using the @main macro.\n\nTo see this feature in action, consider the following definition:\n\n(@main)(args) = println(\"Hello $(args[1])!\")\n\nExecuting the above script with julia script.jl \"Buddy\" will automatically run (@main) and print \"Hello Buddy!\",\ndespite there being no explicit call to (@main).\n\nThe return value of the (@main) function must either be nothing, resulting in exit code\n0, or convertible to a Cint which will be the exit code:\n\n$ julia -e \"(@main)(args) = nothing\"; echo $?0\n0\n$ julia -e \"(@main)(args) = 1\"; echo $?\n1\n\nTypically exit codes are in the range 0:255, although the interpretation of the return value might be OS dependent.\n\nThis feature is intended to aid in the unification of compiled and interactive workflows. In compiled workflows, loading the code that defines the main\nfunction may be spatially and temporally separated from the invocation. However, for interactive workflows,\nthe behavior is equivalent to explicitly calling exit(main(ARGS)) at the end of the evaluated script or\nexpression.\n\ncompat: Julia 1.11\nThe special entry point Main.main was added in Julia 1.11. For compatibility with prior julia versions,\nadd an explicit @isdefined(var\"@main\") ? (@main) : exit(main(ARGS)) at the end of your scripts.\n\nOnly the main binding in the Main module has this behavior and only if\nthe macro @main was used within the defining module.\n\nFor example, using hello instead of main will not result in the hello function executing:\n\n$ julia -e 'hello(args) = println(\"Hello World!\")'\n$\n\nand neither will a plain definition of main:\n\n$ julia -e 'main(args) = println(\"Hello World!\")'\n$\n\nHowever, the opt-in need not occur at definition time:\n\n$ julia -e 'main(args) = println(\"Hello World!\"); @main'\nHello World!\n$\n\nThe main binding may be imported from a package. A hello world package defined as\n\nmodule Hello\n\nexport main\n(@main)(args) = println(\"Hello from the package!\")\n\nend\n\nmay be used as:\n\n$ julia -e 'using Hello'\nHello from the package!\n$ julia -e 'import Hello' # N.B.: Execution depends on the binding not whether the package is loaded\n$\n\nHowever, note that the current best practice recommendation is to not mix application and reusable library\ncode in the same package. Helper applications may be distributed as separate packages or as scripts with\nseparate main entry points in a package's bin folder."},{"title":"Parallel mode","page":"Command-line Interface","location":"manual/command-line-interface.html#Parallel-mode","category":"section","text":"Julia can be started in parallel mode with either the -p or the --machine-file options. -p n\nwill launch an additional n worker processes, while --machine-file file will launch a worker\nfor each line in file file. The machines defined in file must be accessible via a password-less\nssh login, with Julia installed at the same location as the current host. Each machine definition\ntakes the form [count*][user@]host[:port] [bind_addr[:port]]. user defaults to current user,\nport to the standard ssh port. count is the number of workers to spawn on the node, and defaults\nto 1. The optional bind-to bind_addr[:port] specifies the IP address and port that other workers\nshould use to connect to this worker."},{"title":"Startup file","page":"Command-line Interface","location":"manual/command-line-interface.html#Startup-file","category":"section","text":"If you have code that you want executed whenever Julia is run, you can put it in\n~/.julia/config/startup.jl:\n\n$ echo 'println(\"Greetings! 你好! 안녕하세요?\")' > ~/.julia/config/startup.jl\n$ julia\nGreetings! 你好! 안녕하세요?\n\n...\n\nNote that although you should have a ~/.julia directory once you've run Julia for the\nfirst time, you may need to create the ~/.julia/config folder and the\n~/.julia/config/startup.jl file if you use it.\n\nTo have startup code run only in The Julia REPL (and not when julia is e.g. run\non a script), use atreplinit in startup.jl:\n\natreplinit() do repl\n    # ...\nend\n\nIf JULIA_DEPOT_PATH is set, the startup file should be located there:\n$JULIA_DEPOT_PATH/config/startup.jl."},{"title":"Command-line switches for Julia","page":"Command-line Interface","location":"manual/command-line-interface.html#command-line-interface","category":"section","text":"There are various ways to run Julia code and provide options, similar to those available for the\nperl and ruby programs:\n\njulia [switches] -- [programfile] [args...]\n\nThe following is a complete list of command-line switches available when launching julia (a '*' marks the default value, if applicable; settings marked '($)' may trigger package precompilation):\n\nSwitch Description\n-v, --version Display version information\n-h, --help Print command-line options (this message)\n--help-hidden Print uncommon options not shown by -h\n--project[={<dir>|@temp|@.}] Set <dir> as the active project/environment. Or, create a temporary environment with @temp. The default @. option will search through parent directories until a Project.toml or JuliaProject.toml file is found.\n-J, --sysimage <file> Start up with the given system image file\n-H, --home <dir> Set location of julia executable\n--startup-file={yes*|no} Load JULIA_DEPOT_PATH/config/startup.jl; if JULIA_DEPOT_PATH environment variable is unset, load ~/.julia/config/startup.jl\n--handle-signals={yes*|no} Enable or disable Julia's default signal handlers\n--sysimage-native-code={yes*|no} Use native code from system image if available\n--compiled-modules={yes*|no|existing|strict} Enable or disable incremental precompilation of modules. The existing option allows use of existing compiled modules that were previously precompiled, but disallows creation of new precompile files. The strict option is similar, but will error if no precompile file is found.\n--pkgimages={yes*|no|existing} Enable or disable usage of native code caching in the form of pkgimages. The existing option allows use of existing pkgimages but disallows creation of new ones\n-e, --eval <expr> Evaluate <expr>\n-E, --print <expr> Evaluate <expr> and display the result\n-m, --module <Package> [args] Run entry point of Package (@main function) with args\n-L, --load <file> Load <file> immediately on all processors\n-t, --threads {auto|N[,auto|M]} Enable N[+M] threads; N threads are assigned to the default threadpool, and if M is specified, M threads are assigned to the interactive threadpool; auto tries to infer a useful default number of threads to use but the exact behavior might change in the future. Currently sets N to the number of CPUs assigned to this Julia process based on the OS-specific affinity assignment interface if supported (Linux and Windows) or to the number of CPU threads if not supported (MacOS) or if process affinity is not configured, and sets M to 1.\n--gcthreads=N[,M] Use N threads for the mark phase of GC and M (0 or 1) threads for the concurrent sweeping phase of GC. N is set to the number of compute threads and M is set to 0 if unspecified. See Memory Management and Garbage Collection for more details.\n-p, --procs {N|auto} Integer value N launches N additional local worker processes; auto launches as many workers as the number of local CPU threads (logical cores)\n--machine-file <file> Run processes on hosts listed in <file>\n-i, --interactive Interactive mode; REPL runs and isinteractive() is true\n-q, --quiet Quiet startup: no banner, suppress REPL warnings\n--banner={yes|no|short|auto*} Enable or disable startup banner\n--color={yes|no|auto*} Enable or disable color text\n--history-file={yes*|no} Load or save history\n--depwarn={yes|no*|error} Enable or disable syntax and method deprecation warnings (error turns warnings into errors)\n--warn-overwrite={yes|no*} Enable or disable method overwrite warnings\n--warn-scope={yes*|no} Enable or disable warning for ambiguous top-level scope\n-C, --cpu-target <target> Limit usage of CPU features up to <target>; set to help to see the available options\n-O, --optimize={0|1|2*|3} Set the optimization level (level is 3 if -O is used without a level) ($)\n--min-optlevel={0*|1|2|3} Set the lower bound on per-module optimization\n-g, --debug-info={0|1*|2} Set the level of debug info generation (level is 2 if -g is used without a level) ($)\n--inline={yes*|no} Control whether inlining is permitted, including overriding @inline declarations\n--check-bounds={yes|no|auto*} Emit bounds checks always, never, or respect @inbounds declarations ($)\n--math-mode={ieee|user*} Always follow ieee floating point semantics or respect @fastmath declarations\n--polly={yes*|no} Enable or disable the polyhedral optimizer Polly (overrides @polly declaration)\n--code-coverage[={none*|user|all}] Count executions of source lines (omitting setting is equivalent to user)\n--code-coverage=@<path> Count executions but only in files that fall under the given file path/directory. The @ prefix is required to select this option. A @ with no path will track the current directory.\n--code-coverage=tracefile.info Append coverage information to the LCOV tracefile (filename supports format tokens).\n--track-allocation[={none*|user|all}] Count bytes allocated by each source line (omitting setting is equivalent to \"user\")\n--track-allocation=@<path> Count bytes but only in files that fall under the given file path/directory. The @ prefix is required to select this option. A @ with no path will track the current directory.\n--task-metrics={yes|no*} Enable the collection of per-task metrics\n--bug-report=KIND Launch a bug report session. It can be used to start a REPL, run a script, or evaluate expressions. It first tries to use BugReporting.jl installed in current environment and falls back to the latest compatible BugReporting.jl if not. For more information, see --bug-report=help.\n--heap-size-hint=<size> Forces garbage collection if memory usage is higher than the given value. The value may be specified as a number of bytes, optionally in units of KB, MB, GB, or TB, or as a percentage of physical memory with %. See Memory Management and Garbage Collection for more details.\n--compile={yes*|no|all|min} Enable or disable JIT compiler, or request exhaustive or minimal compilation\n--output-o <name> Generate an object file (including system image data)\n--output-ji <name> Generate a system image data file (.ji)\n--strip-metadata Remove docstrings and source location info from system image\n--strip-ir Remove IR (intermediate representation) of compiled functions\n--output-unopt-bc <name> Generate unoptimized LLVM bitcode (.bc)\n--output-bc <name> Generate LLVM bitcode (.bc)\n--output-asm <name> Generate an assembly file (.s)\n--output-incremental={yes|no*} Generate an incremental output file (rather than complete)\n--trace-compile={stderr|name} Print precompile statements for methods compiled during execution or save to stderr or a path. Methods that were recompiled are printed in yellow or with a trailing comment if color is not supported\n--trace-compile-timing If --trace-compile is enabled show how long each took to compile in ms\n--trace-dispatch={stderr|name} Print precompile statements for methods dispatched during execution or save to stderr or a path.\n--trace-eval[={no*|loc|full}] Show top-level expressions being evaluated. loc shows location info only, full shows full expressions (omitting setting is equivalent to loc). Only shows the outermost expression being evaluated, not internal function calls. See also Base.TRACE_EVAL.\n--image-codegen Force generate code in imaging mode\n--permalloc-pkgimg={yes|no*} Copy the data section of package images into memory\n--trim={no*|safe|unsafe|unsafe-warn} Build a sysimage including only code provably reachable from methods marked by calling entrypoint. The three non-default options differ in how they handle dynamic call sites. In safe mode, such sites result in compile-time errors. In unsafe mode, such sites are allowed but the resulting binary might be missing needed code and can throw runtime errors. With unsafe-warn, such sites will trigger warnings at compile-time and might error at runtime.\n\nOptions that have the form --option={...} can be specified either as --option=value or as --option value. For example, julia --banner=no is equivalent to julia --banner no. This is especially relevant for options that take a filename for output, because forgetting to specifying the argument for (say) --trace-compile will cause the option following it to be interpreted as the filename, possibly unintentionally overwriting it.\n\nNote that options of the form --option[=...] can not be specified as --option value, but only as --option=value (or simply --option, when no argument is provided).\n\ncompat: Julia 1.1\nIn Julia 1.0, the default --project=@. option did not search up from the root\ndirectory of a Git repository for the Project.toml file. From Julia 1.1 forward, it\ndoes."},{"title":"Tar","page":"Tar","location":"stdlib/Tar.html","category":"section","text":"The Tar module provides a simple interface for handling tar archives, including creation of\narchives, extraction of selected files from an archive, and access to metadata."},{"title":"Tar","page":"Tar","location":"stdlib/Tar.html#Tar","category":"section","text":""},{"title":"Tar.create","page":"Tar","location":"stdlib/Tar.html#Tar.create","category":"function","text":"create(\n    [ predicate, ] dir, [ tarball ];\n    [ skeleton, ] [ portable = false ]\n) -> tarball\n\n    predicate :: String --> Bool\n    dir       :: AbstractString\n    tarball   :: Union{AbstractString, AbstractCmd, IO}\n    skeleton  :: Union{AbstractString, AbstractCmd, IO}\n    portable  :: Bool\n\nCreate a tar archive (\"tarball\") of the directory dir. The resulting archive\nis written to the path tarball or if no path is specified, a temporary path is\ncreated and returned by the function call. If tarball is an IO object then the\ntarball content is written to that handle instead (the handle is left open).\n\nIf a predicate function is passed, it is called on each system path that is\nencountered while recursively searching dir and path is only included in the\ntarball if predicate(path) is true. If predicate(path) returns false for a\ndirectory, then the directory is excluded entirely: nothing under that directory\nwill be included in the archive.\n\nIf the skeleton keyword is passed then the file or IO handle given is used as\na \"skeleton\" to generate the tarball. You create a skeleton file by passing the\nskeleton keyword to the extract command. If create is called with that\nskeleton file and the extracted files haven't changed, an identical tarball is\nrecreated. The skeleton and predicate arguments cannot be used together.\n\nIf the portable flag is true then path names are checked for validity on\nWindows, which ensures that they don't contain illegal characters or have names\nthat are reserved. See https://stackoverflow.com/a/31976060/659248 for details.\n\n\n\n\n\n"},{"title":"Tar.extract","page":"Tar","location":"stdlib/Tar.html#Tar.extract","category":"function","text":"extract(\n    [ predicate, ] tarball, [ dir ];\n    [ skeleton = <none>, ]\n    [ copy_symlinks = <auto>, ]\n    [ set_permissions = true, ]\n) -> dir\n\n    predicate       :: Header --> Bool\n    tarball         :: Union{AbstractString, AbstractCmd, IO}\n    dir             :: AbstractString\n    skeleton        :: Union{AbstractString