(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.pngQrCode = {}));
})(this, (function (exports) { 'use strict';

    const enc = new TextEncoder();
    const dec = new TextDecoder();
    function encode$1(inData, parse_url) {
        let str;
        let data;
        if (typeof inData === "string" || typeof inData === "number") {
            str = `${inData}`;
            data = enc.encode(str);
        }
        else if (Array.isArray(inData)) {
            data = new Uint8Array(inData);
            str = dec.decode(inData);
        }
        else {
            throw new Error("Bad data: " + typeof inData);
        }
        if (/^[0-9]+$/.test(str)) {
            if (data.byteLength > 7089) {
                throw new Error("Too much data");
            }
            return encode_numeric(str);
        }
        if (/^[0-9A-Z \$%\*\+\.\/\:\-]+$/.test(str)) {
            if (data.byteLength > 4296) {
                throw new Error("Too much data");
            }
            return encode_alphanum(str);
        }
        if (parse_url && /^https?:/i.test(str)) {
            return encode_url(str);
        }
        if (data.byteLength > 2953) {
            throw new Error("Too much data");
        }
        return encode_8bit(new Uint8Array(data));
    }
    function pushBits(arr, n, value) {
        for (let bit = 1 << (n - 1); bit; bit >>>= 1) {
            arr.push(bit & value ? 1 : 0);
        }
    }
    function encode_8bit(data) {
        const len = data.byteLength;
        const bits = [];
        for (let i = 0; i < len; i++) {
            pushBits(bits, 8, data[i]);
        }
        const res = {};
        let d = [0, 1, 0, 0];
        pushBits(d, 16, len);
        res.data10 = res.data27 = d.concat(bits);
        if (len < 256) {
            let d = [0, 1, 0, 0];
            pushBits(d, 8, len);
            res.data1 = d.concat(bits);
        }
        return res;
    }
    const ALPHANUM = (function (s) {
        const res = {};
        for (let i = 0; i < s.length; i++) {
            res[s[i]] = i;
        }
        return res;
    })("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ $%*+-./:");
    function encode_alphanum(str) {
        const len = str.length;
        const bits = [];
        for (let i = 0; i < len; i += 2) {
            let b = 6;
            let n = ALPHANUM[str[i]];
            if (str[i + 1]) {
                b = 11;
                n = n * 45 + ALPHANUM[str[i + 1]];
            }
            pushBits(bits, b, n);
        }
        const res = {};
        let d = [0, 0, 1, 0];
        pushBits(d, 13, len);
        res.data27 = d.concat(bits);
        if (len < 2048) {
            let d = [0, 0, 1, 0];
            pushBits(d, 11, len);
            res.data10 = d.concat(bits);
        }
        if (len < 512) {
            let d = [0, 0, 1, 0];
            pushBits(d, 9, len);
            res.data1 = d.concat(bits);
        }
        return res;
    }
    function encode_numeric(str) {
        const len = str.length;
        const bits = [];
        for (let i = 0; i < len; i += 3) {
            const s = str.substr(i, 3);
            const b = Math.ceil((s.length * 10) / 3);
            pushBits(bits, b, parseInt(s, 10));
        }
        const res = {};
        let d = [0, 0, 0, 1];
        pushBits(d, 14, len);
        res.data27 = d.concat(bits);
        if (len < 4096) {
            let d = [0, 0, 0, 1];
            pushBits(d, 12, len);
            res.data10 = d.concat(bits);
        }
        if (len < 1024) {
            let d = [0, 0, 0, 1];
            pushBits(d, 10, len);
            res.data1 = d.concat(bits);
        }
        return res;
    }
    function encode_url(str) {
        const slash = str.indexOf("/", 8) + 1 || str.length;
        const res = encode$1(str.slice(0, slash).toUpperCase(), false);
        if (slash >= str.length) {
            return res;
        }
        const path_res = encode$1(str.slice(slash), false);
        res.data27 = res.data27.concat(path_res.data27);
        if (res.data10 && path_res.data10) {
            res.data10 = res.data10.concat(path_res.data10);
        }
        if (res.data1 && path_res.data1) {
            res.data1 = res.data1.concat(path_res.data1);
        }
        return res;
    }

    function calculateEC(msg, ec_len) {
        msg = [].slice.call(msg);
        const poly = generatorPolynomial(ec_len);
        for (let i = 0; i < ec_len; i++)
            msg.push(0);
        while (msg.length > ec_len) {
            if (!msg[0]) {
                msg.shift();
                continue;
            }
            const log_k = log(msg[0]);
            for (let i = 0; i <= ec_len; i++) {
                msg[i] = msg[i] ^ exp(poly[i] + log_k);
            }
            msg.shift();
        }
        return new Uint8Array(msg);
    }
    const GF256_BASE = 285;
    const EXP_TABLE = [1];
    const LOG_TABLE = [];
    for (let i = 1; i < 256; i++) {
        let n = EXP_TABLE[i - 1] << 1;
        if (n > 255)
            n ^= GF256_BASE;
        EXP_TABLE[i] = n;
    }
    for (let i = 0; i < 255; i++) {
        LOG_TABLE[EXP_TABLE[i]] = i;
    }
    function exp(k) {
        while (k < 0)
            k += 255;
        while (k > 255)
            k -= 255;
        return EXP_TABLE[k];
    }
    function log(k) {
        if (k < 1 || k > 255) {
            throw Error(`Bad log(${k})`);
        }
        return LOG_TABLE[k];
    }
    const POLYNOMIALS = [
        [0],
        [0, 0],
        [0, 25, 1],
    ];
    function generatorPolynomial(num) {
        if (POLYNOMIALS[num]) {
            return POLYNOMIALS[num];
        }
        const prev = generatorPolynomial(num - 1);
        const res = [];
        res[0] = prev[0];
        for (let i = 1; i <= num; i++) {
            res[i] = log(exp(prev[i]) ^ exp(prev[i - 1] + num - 1));
        }
        POLYNOMIALS[num] = res;
        return res;
    }

    function init(version) {
        const N = (version << 2) + 0b10001;
        const matrix = [];
        let zeros = Array(N).fill(0);
        for (let i = 0; i < N; i++) {
            matrix[i] = [...zeros];
        }
        return matrix;
    }
    function fillFinders(matrix) {
        const N = matrix.length;
        for (var i = -3; i <= 3; i++) {
            for (let j = -3; j <= 3; j++) {
                const max = Math.max(i, j);
                const min = Math.min(i, j);
                const pixel = (max == 2 && min >= -2) || (min == -2 && max <= 2)
                    ? 0x80
                    : 0x81;
                matrix[3 + i][3 + j] = pixel;
                matrix[3 + i][N - 4 + j] = pixel;
                matrix[N - 4 + i][3 + j] = pixel;
            }
        }
        for (var i = 0; i < 8; i++) {
            matrix[7][i] =
                matrix[i][7] =
                    matrix[7][N - i - 1] =
                        matrix[i][N - 8] =
                            matrix[N - 8][i] =
                                matrix[N - 1 - i][7] =
                                    0x80;
        }
    }
    function fillAlignAndTiming(matrix) {
        const N = matrix.length;
        if (N > 21) {
            const len = N - 13;
            let delta = Math.round(len / Math.ceil(len / 28));
            if (delta % 2)
                delta++;
            const res = [];
            for (let p = len + 6; p > 10; p -= delta) {
                res.unshift(p);
            }
            res.unshift(6);
            for (var i = 0; i < res.length; i++) {
                for (let j = 0; j < res.length; j++) {
                    const x = res[i];
                    const y = res[j];
                    if (matrix[x][y])
                        continue;
                    for (let r = -2; r <= 2; r++) {
                        for (let c = -2; c <= 2; c++) {
                            const max = Math.max(r, c);
                            const min = Math.min(r, c);
                            const pixel = (max == 1 && min >= -1) || (min == -1 && max <= 1)
                                ? 0x80
                                : 0x81;
                            matrix[x + r][y + c] = pixel;
                        }
                    }
                }
            }
        }
        for (var i = 8; i < N - 8; i++) {
            matrix[6][i] = matrix[i][6] = i % 2 ? 0x80 : 0x81;
        }
    }
    function fillStub(matrix) {
        const N = matrix.length;
        for (var i = 0; i < 8; i++) {
            if (i != 6) {
                matrix[8][i] = matrix[i][8] = 0x80;
            }
            matrix[8][N - 1 - i] = 0x80;
            matrix[N - 1 - i][8] = 0x80;
        }
        matrix[8][8] = 0x80;
        matrix[N - 8][8] = 0x81;
        if (N < 45)
            return;
        for (var i = N - 11; i < N - 8; i++) {
            for (let j = 0; j < 6; j++) {
                matrix[i][j] = matrix[j][i] = 0x80;
            }
        }
    }
    const fillReserved = (function () {
        const FORMATS = Array(32);
        const VERSIONS = Array(40);
        const gf15 = 0x0537;
        const gf18 = 0x1f25;
        const formats_mask = 0x5412;
        for (let format = 0; format < 32; format++) {
            let res = format << 10;
            for (let i = 5; i > 0; i--) {
                if (res >>> (9 + i)) {
                    res ^= gf15 << (i - 1);
                }
            }
            FORMATS[format] = (res | (format << 10)) ^ formats_mask;
        }
        for (let version = 7; version <= 40; version++) {
            let res = version << 12;
            for (let i = 6; i > 0; i--) {
                if (res >>> (11 + i)) {
                    res ^= gf18 << (i - 1);
                }
            }
            VERSIONS[version] = res | (version << 12);
        }
        const EC_LEVELS = {
            L: 1,
            M: 0,
            Q: 3,
            H: 2,
        };
        return function fillReserved(matrix, ec_level, mask) {
            const N = matrix.length;
            const format = FORMATS[(EC_LEVELS[ec_level] << 3) | mask];
            function F(k) {
                return (format >> k) & 1 ? 0x81 : 0x80;
            }
            for (var i = 0; i < 8; i++) {
                matrix[8][N - 1 - i] = F(i);
                if (i < 6)
                    matrix[i][8] = F(i);
            }
            for (var i = 8; i < 15; i++) {
                matrix[N - 15 + i][8] = F(i);
                if (i > 8)
                    matrix[8][14 - i] = F(i);
            }
            matrix[7][8] = F(6);
            matrix[8][8] = F(7);
            matrix[8][7] = F(8);
            const version = VERSIONS[(N - 17) / 4];
            if (!version) {
                return;
            }
            function V(k) {
                return (version >> k) & 1 ? 0x81 : 0x80;
            }
            for (var i = 0; i < 6; i++) {
                for (let j = 0; j < 3; j++) {
                    matrix[N - 11 + j][i] = matrix[i][N - 11 + j] = V(i * 3 + j);
                }
            }
        };
    })();
    const fillData = (function () {
        const MASK_FUNCTIONS = [
            (i, j) => (i + j) % 2 == 0,
            (i, j) => i % 2 == 0,
            (i, j) => j % 3 == 0,
            (i, j) => (i + j) % 3 == 0,
            (i, j) => (Math.floor(i / 2) + Math.floor(j / 3)) % 2 == 0,
            (i, j) => ((i * j) % 2) + ((i * j) % 3) == 0,
            (i, j) => (((i * j) % 2) + ((i * j) % 3)) % 2 == 0,
            (i, j) => (((i * j) % 3) + ((i + j) % 2)) % 2 == 0,
        ];
        return function fillData(matrix, data, mask) {
            const N = matrix.length;
            let row;
            let col;
            let dir = -1;
            row = col = N - 1;
            const mask_fn = MASK_FUNCTIONS[mask];
            let len = data.blocks[data.blocks.length - 1].length;
            for (var i = 0; i < len; i++) {
                for (var b = 0; b < data.blocks.length; b++) {
                    if (data.blocks[b].length <= i) {
                        continue;
                    }
                    put(data.blocks[b][i]);
                }
            }
            len = data.ec_len;
            for (var i = 0; i < len; i++) {
                for (var b = 0; b < data.ec.length; b++) {
                    put(data.ec[b][i]);
                }
            }
            if (col > -1) {
                do {
                    matrix[row][col] = mask_fn(row, col) ? 1 : 0;
                } while (next());
            }
            function put(byte) {
                for (let mask = 0x80; mask; mask >>= 1) {
                    let pixel = !!(mask & byte);
                    if (mask_fn(row, col))
                        pixel = !pixel;
                    matrix[row][col] = pixel ? 1 : 0;
                    next();
                }
            }
            function next() {
                do {
                    if (col % 2 ^ Number(col < 6)) {
                        if ((dir < 0 && row == 0) || (dir > 0 && row == N - 1)) {
                            col--;
                            dir = -dir;
                        }
                        else {
                            col++;
                            row += dir;
                        }
                    }
                    else {
                        col--;
                    }
                    if (col == 6) {
                        col--;
                    }
                    if (col < 0) {
                        return false;
                    }
                } while (matrix[row][col] & 0xf0);
                return true;
            }
        };
    })();
    function calculatePenalty(matrix) {
        const N = matrix.length;
        let penalty = 0;
        for (var i = 0; i < N; i++) {
            var pixel = matrix[i][0] & 1;
            var len = 1;
            for (var j = 1; j < N; j++) {
                var p = matrix[i][j] & 1;
                if (p == pixel) {
                    len++;
                    continue;
                }
                if (len >= 5) {
                    penalty += len - 2;
                }
                pixel = p;
                len = 1;
            }
            if (len >= 5) {
                penalty += len - 2;
            }
        }
        for (var j = 0; j < N; j++) {
            var pixel = matrix[0][j] & 1;
            var len = 1;
            for (var i = 1; i < N; i++) {
                var p = matrix[i][j] & 1;
                if (p == pixel) {
                    len++;
                    continue;
                }
                if (len >= 5) {
                    penalty += len - 2;
                }
                pixel = p;
                len = 1;
            }
            if (len >= 5) {
                penalty += len - 2;
            }
        }
        for (var i = 0; i < N - 1; i++) {
            for (var j = 0; j < N - 1; j++) {
                const s = (matrix[i][j] +
                    matrix[i][j + 1] +
                    matrix[i + 1][j] +
                    matrix[i + 1][j + 1]) &
                    7;
                if (s == 0 || s == 4) {
                    penalty += 3;
                }
            }
        }
        function I(k) {
            return matrix[i][j + k] & 1;
        }
        function J(k) {
            return matrix[i + k][j] & 1;
        }
        for (var i = 0; i < N; i++) {
            for (var j = 0; j < N; j++) {
                if (j < N - 6 &&
                    I(0) &&
                    !I(1) &&
                    I(2) &&
                    I(3) &&
                    I(4) &&
                    !I(5) &&
                    I(6)) {
                    if (j >= 4 && !(I(-4) || I(-3) || I(-2) || I(-1))) {
                        penalty += 40;
                    }
                    if (j < N - 10 && !(I(7) || I(8) || I(9) || I(10))) {
                        penalty += 40;
                    }
                }
                if (i < N - 6 &&
                    J(0) &&
                    !J(1) &&
                    J(2) &&
                    J(3) &&
                    J(4) &&
                    !J(5) &&
                    J(6)) {
                    if (i >= 4 && !(J(-4) || J(-3) || J(-2) || J(-1))) {
                        penalty += 40;
                    }
                    if (i < N - 10 && !(J(7) || J(8) || J(9) || J(10))) {
                        penalty += 40;
                    }
                }
            }
        }
        let numDark = 0;
        for (var i = 0; i < N; i++) {
            for (var j = 0; j < N; j++) {
                if (matrix[i][j] & 1)
                    numDark++;
            }
        }
        penalty += 10 * Math.floor(Math.abs(10 - (20 * numDark) / (N * N)));
        return penalty;
    }
    function getMatrix(data) {
        const matrix = init(data.version);
        fillFinders(matrix);
        fillAlignAndTiming(matrix);
        fillStub(matrix);
        let penalty = Infinity;
        let bestMask = 0;
        for (let mask = 0; mask < 8; mask++) {
            fillData(matrix, data, mask);
            fillReserved(matrix, data.ec_level, mask);
            const p = calculatePenalty(matrix);
            if (p < penalty) {
                penalty = p;
                bestMask = mask;
            }
        }
        fillData(matrix, data, bestMask);
        fillReserved(matrix, data.ec_level, bestMask);
        return matrix.map((row) => row.map((cell) => cell & 1));
    }

    const EC_LEVELS = ["L", "M", "Q", "H"];
    function getTemplate(message, ec_level) {
        let i = 1;
        let len;
        if (message.data1) {
            len = Math.ceil(message.data1.length / 8);
        }
        else {
            i = 10;
        }
        for (; i < 10; i++) {
            let version = mappedVersions[i][ec_level];
            if (version.data_len >= len) {
                return deepCopy(version);
            }
        }
        if (message.data10) {
            len = Math.ceil(message.data10.length / 8);
        }
        else {
            i = 27;
        }
        for (; i < 27; i++) {
            let version = mappedVersions[i][ec_level];
            if (version.data_len >= len) {
                return deepCopy(version);
            }
        }
        len = Math.ceil(message.data27.length / 8);
        for (; i < 41; i++) {
            let version = mappedVersions[i][ec_level];
            if (version.data_len >= len) {
                return deepCopy(version);
            }
        }
        throw new Error("Too much data");
    }
    function fillTemplate(message, template) {
        const blocks = new Uint8Array(template.data_len);
        let messageUpdated;
        if (template.version < 10) {
            messageUpdated = message.data1;
        }
        else if (template.version < 27) {
            messageUpdated = message.data10;
        }
        else {
            messageUpdated = message.data27;
        }
        const len = messageUpdated.length;
        for (let i = 0; i < len; i += 8) {
            let b = 0;
            for (let j = 0; j < 8; j++) {
                b = (b << 1) | (messageUpdated[i + j] ? 1 : 0);
            }
            blocks[i / 8] = b;
        }
        let pad = 236;
        for (let i = Math.ceil((len + 4) / 8); i < blocks.length; i++) {
            blocks[i] = pad;
            pad = pad == 236 ? 17 : 236;
        }
        let offset = 0;
        template.blocks = template.blocks.map((n) => {
            const b = blocks.slice(offset, offset + n);
            offset += n;
            template.ec.push(calculateEC(b, template.ec_len));
            return b;
        });
        return template;
    }
    function QR(text, ec_level, parse_url) {
        ec_level = EC_LEVELS.includes(ec_level) ? ec_level : "M";
        const message = encode$1(text, parse_url);
        const data = fillTemplate(message, getTemplate(message, ec_level));
        return getMatrix(data);
    }
    const deepCopy = typeof structuredClone !== "undefined" ? structuredClone : ((obj) => JSON.parse(JSON.stringify(obj)));
    const versions = [
        [],
        [26, 7, 1, 10, 1, 13, 1, 17, 1],
        [44, 10, 1, 16, 1, 22, 1, 28, 1],
        [70, 15, 1, 26, 1, 36, 2, 44, 2],
        [100, 20, 1, 36, 2, 52, 2, 64, 4],
        [134, 26, 1, 48, 2, 72, 4, 88, 4],
        [172, 36, 2, 64, 4, 96, 4, 112, 4],
        [196, 40, 2, 72, 4, 108, 6, 130, 5],
        [242, 48, 2, 88, 4, 132, 6, 156, 6],
        [292, 60, 2, 110, 5, 160, 8, 192, 8],
        [346, 72, 4, 130, 5, 192, 8, 224, 8],
        [404, 80, 4, 150, 5, 224, 8, 264, 11],
        [466, 96, 4, 176, 8, 260, 10, 308, 11],
        [532, 104, 4, 198, 9, 288, 12, 352, 16],
        [581, 120, 4, 216, 9, 320, 16, 384, 16],
        [655, 132, 6, 240, 10, 360, 12, 432, 18],
        [733, 144, 6, 280, 10, 408, 17, 480, 16],
        [815, 168, 6, 308, 11, 448, 16, 532, 19],
        [901, 180, 6, 338, 13, 504, 18, 588, 21],
        [991, 196, 7, 364, 14, 546, 21, 650, 25],
        [1085, 224, 8, 416, 16, 600, 20, 700, 25],
        [1156, 224, 8, 442, 17, 644, 23, 750, 25],
        [1258, 252, 9, 476, 17, 690, 23, 816, 34],
        [1364, 270, 9, 504, 18, 750, 25, 900, 30],
        [1474, 300, 10, 560, 20, 810, 27, 960, 32],
        [1588, 312, 12, 588, 21, 870, 29, 1050, 35],
        [1706, 336, 12, 644, 23, 952, 34, 1110, 37],
        [1828, 360, 12, 700, 25, 1020, 34, 1200, 40],
        [1921, 390, 13, 728, 26, 1050, 35, 1260, 42],
        [2051, 420, 14, 784, 28, 1140, 38, 1350, 45],
        [2185, 450, 15, 812, 29, 1200, 40, 1440, 48],
        [2323, 480, 16, 868, 31, 1290, 43, 1530, 51],
        [2465, 510, 17, 924, 33, 1350, 45, 1620, 54],
        [2611, 540, 18, 980, 35, 1440, 48, 1710, 57],
        [2761, 570, 19, 1036, 37, 1530, 51, 1800, 60],
        [2876, 570, 19, 1064, 38, 1590, 53, 1890, 63],
        [3034, 600, 20, 1120, 40, 1680, 56, 1980, 66],
        [3196, 630, 21, 1204, 43, 1770, 59, 2100, 70],
        [3362, 660, 22, 1260, 45, 1860, 62, 2220, 74],
        [3532, 720, 24, 1316, 47, 1950, 65, 2310, 77],
        [3706, 750, 25, 1372, 49, 2040, 68, 2430, 81],
    ];
    const mappedVersions = versions.map((el, index) => {
        if (!index) {
            return Object.create(null);
        }
        const res = Object.create(null);
        for (let i = 1; i < 8; i += 2) {
            const length = el[0] - el[i];
            const num_template = el[i + 1];
            const ec_level = EC_LEVELS[(i / 2) | 0];
            const level = {
                version: index,
                ec_level,
                data_len: length,
                ec_len: el[i] / num_template,
                blocks: [],
                ec: [],
            };
            for (let k = num_template, n = length; k > 0; k--) {
                const block = (n / k) | 0;
                level.blocks.push(block);
                n -= block;
            }
            res[ec_level] = level;
        }
        return res;
    });

    function getDefaultExportFromCjs (x) {
    	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
    }

    var colorString$1 = {exports: {}};

    var colorName = {
    	"aliceblue": [240, 248, 255],
    	"antiquewhite": [250, 235, 215],
    	"aqua": [0, 255, 255],
    	"aquamarine": [127, 255, 212],
    	"azure": [240, 255, 255],
    	"beige": [245, 245, 220],
    	"bisque": [255, 228, 196],
    	"black": [0, 0, 0],
    	"blanchedalmond": [255, 235, 205],
    	"blue": [0, 0, 255],
    	"blueviolet": [138, 43, 226],
    	"brown": [165, 42, 42],
    	"burlywood": [222, 184, 135],
    	"cadetblue": [95, 158, 160],
    	"chartreuse": [127, 255, 0],
    	"chocolate": [210, 105, 30],
    	"coral": [255, 127, 80],
    	"cornflowerblue": [100, 149, 237],
    	"cornsilk": [255, 248, 220],
    	"crimson": [220, 20, 60],
    	"cyan": [0, 255, 255],
    	"darkblue": [0, 0, 139],
    	"darkcyan": [0, 139, 139],
    	"darkgoldenrod": [184, 134, 11],
    	"darkgray": [169, 169, 169],
    	"darkgreen": [0, 100, 0],
    	"darkgrey": [169, 169, 169],
    	"darkkhaki": [189, 183, 107],
    	"darkmagenta": [139, 0, 139],
    	"darkolivegreen": [85, 107, 47],
    	"darkorange": [255, 140, 0],
    	"darkorchid": [153, 50, 204],
    	"darkred": [139, 0, 0],
    	"darksalmon": [233, 150, 122],
    	"darkseagreen": [143, 188, 143],
    	"darkslateblue": [72, 61, 139],
    	"darkslategray": [47, 79, 79],
    	"darkslategrey": [47, 79, 79],
    	"darkturquoise": [0, 206, 209],
    	"darkviolet": [148, 0, 211],
    	"deeppink": [255, 20, 147],
    	"deepskyblue": [0, 191, 255],
    	"dimgray": [105, 105, 105],
    	"dimgrey": [105, 105, 105],
    	"dodgerblue": [30, 144, 255],
    	"firebrick": [178, 34, 34],
    	"floralwhite": [255, 250, 240],
    	"forestgreen": [34, 139, 34],
    	"fuchsia": [255, 0, 255],
    	"gainsboro": [220, 220, 220],
    	"ghostwhite": [248, 248, 255],
    	"gold": [255, 215, 0],
    	"goldenrod": [218, 165, 32],
    	"gray": [128, 128, 128],
    	"green": [0, 128, 0],
    	"greenyellow": [173, 255, 47],
    	"grey": [128, 128, 128],
    	"honeydew": [240, 255, 240],
    	"hotpink": [255, 105, 180],
    	"indianred": [205, 92, 92],
    	"indigo": [75, 0, 130],
    	"ivory": [255, 255, 240],
    	"khaki": [240, 230, 140],
    	"lavender": [230, 230, 250],
    	"lavenderblush": [255, 240, 245],
    	"lawngreen": [124, 252, 0],
    	"lemonchiffon": [255, 250, 205],
    	"lightblue": [173, 216, 230],
    	"lightcoral": [240, 128, 128],
    	"lightcyan": [224, 255, 255],
    	"lightgoldenrodyellow": [250, 250, 210],
    	"lightgray": [211, 211, 211],
    	"lightgreen": [144, 238, 144],
    	"lightgrey": [211, 211, 211],
    	"lightpink": [255, 182, 193],
    	"lightsalmon": [255, 160, 122],
    	"lightseagreen": [32, 178, 170],
    	"lightskyblue": [135, 206, 250],
    	"lightslategray": [119, 136, 153],
    	"lightslategrey": [119, 136, 153],
    	"lightsteelblue": [176, 196, 222],
    	"lightyellow": [255, 255, 224],
    	"lime": [0, 255, 0],
    	"limegreen": [50, 205, 50],
    	"linen": [250, 240, 230],
    	"magenta": [255, 0, 255],
    	"maroon": [128, 0, 0],
    	"mediumaquamarine": [102, 205, 170],
    	"mediumblue": [0, 0, 205],
    	"mediumorchid": [186, 85, 211],
    	"mediumpurple": [147, 112, 219],
    	"mediumseagreen": [60, 179, 113],
    	"mediumslateblue": [123, 104, 238],
    	"mediumspringgreen": [0, 250, 154],
    	"mediumturquoise": [72, 209, 204],
    	"mediumvioletred": [199, 21, 133],
    	"midnightblue": [25, 25, 112],
    	"mintcream": [245, 255, 250],
    	"mistyrose": [255, 228, 225],
    	"moccasin": [255, 228, 181],
    	"navajowhite": [255, 222, 173],
    	"navy": [0, 0, 128],
    	"oldlace": [253, 245, 230],
    	"olive": [128, 128, 0],
    	"olivedrab": [107, 142, 35],
    	"orange": [255, 165, 0],
    	"orangered": [255, 69, 0],
    	"orchid": [218, 112, 214],
    	"palegoldenrod": [238, 232, 170],
    	"palegreen": [152, 251, 152],
    	"paleturquoise": [175, 238, 238],
    	"palevioletred": [219, 112, 147],
    	"papayawhip": [255, 239, 213],
    	"peachpuff": [255, 218, 185],
    	"peru": [205, 133, 63],
    	"pink": [255, 192, 203],
    	"plum": [221, 160, 221],
    	"powderblue": [176, 224, 230],
    	"purple": [128, 0, 128],
    	"rebeccapurple": [102, 51, 153],
    	"red": [255, 0, 0],
    	"rosybrown": [188, 143, 143],
    	"royalblue": [65, 105, 225],
    	"saddlebrown": [139, 69, 19],
    	"salmon": [250, 128, 114],
    	"sandybrown": [244, 164, 96],
    	"seagreen": [46, 139, 87],
    	"seashell": [255, 245, 238],
    	"sienna": [160, 82, 45],
    	"silver": [192, 192, 192],
    	"skyblue": [135, 206, 235],
    	"slateblue": [106, 90, 205],
    	"slategray": [112, 128, 144],
    	"slategrey": [112, 128, 144],
    	"snow": [255, 250, 250],
    	"springgreen": [0, 255, 127],
    	"steelblue": [70, 130, 180],
    	"tan": [210, 180, 140],
    	"teal": [0, 128, 128],
    	"thistle": [216, 191, 216],
    	"tomato": [255, 99, 71],
    	"turquoise": [64, 224, 208],
    	"violet": [238, 130, 238],
    	"wheat": [245, 222, 179],
    	"white": [255, 255, 255],
    	"whitesmoke": [245, 245, 245],
    	"yellow": [255, 255, 0],
    	"yellowgreen": [154, 205, 50]
    };

    var simpleSwizzle = {exports: {}};

    var isArrayish$1 = function isArrayish(obj) {
    	if (!obj || typeof obj === 'string') {
    		return false;
    	}

    	return obj instanceof Array || Array.isArray(obj) ||
    		(obj.length >= 0 && (obj.splice instanceof Function ||
    			(Object.getOwnPropertyDescriptor(obj, (obj.length - 1)) && obj.constructor.name !== 'String')));
    };

    var isArrayish = isArrayish$1;

    var concat = Array.prototype.concat;
    var slice = Array.prototype.slice;

    var swizzle$1 = simpleSwizzle.exports = function swizzle(args) {
    	var results = [];

    	for (var i = 0, len = args.length; i < len; i++) {
    		var arg = args[i];

    		if (isArrayish(arg)) {
    			// http://jsperf.com/javascript-array-concat-vs-push/98
    			results = concat.call(results, slice.call(arg));
    		} else {
    			results.push(arg);
    		}
    	}

    	return results;
    };

    swizzle$1.wrap = function (fn) {
    	return function () {
    		return fn(swizzle$1(arguments));
    	};
    };

    var simpleSwizzleExports = simpleSwizzle.exports;

    /* MIT license */

    var colorNames = colorName;
    var swizzle = simpleSwizzleExports;
    var hasOwnProperty = Object.hasOwnProperty;

    var reverseNames = Object.create(null);

    // create a list of reverse color names
    for (var name in colorNames) {
    	if (hasOwnProperty.call(colorNames, name)) {
    		reverseNames[colorNames[name]] = name;
    	}
    }

    var cs = colorString$1.exports = {
    	to: {},
    	get: {}
    };

    cs.get = function (string) {
    	var prefix = string.substring(0, 3).toLowerCase();
    	var val;
    	var model;
    	switch (prefix) {
    		case 'hsl':
    			val = cs.get.hsl(string);
    			model = 'hsl';
    			break;
    		case 'hwb':
    			val = cs.get.hwb(string);
    			model = 'hwb';
    			break;
    		default:
    			val = cs.get.rgb(string);
    			model = 'rgb';
    			break;
    	}

    	if (!val) {
    		return null;
    	}

    	return {model: model, value: val};
    };

    cs.get.rgb = function (string) {
    	if (!string) {
    		return null;
    	}

    	var abbr = /^#([a-f0-9]{3,4})$/i;
    	var hex = /^#([a-f0-9]{6})([a-f0-9]{2})?$/i;
    	var rgba = /^rgba?\(\s*([+-]?\d+)(?=[\s,])\s*(?:,\s*)?([+-]?\d+)(?=[\s,])\s*(?:,\s*)?([+-]?\d+)\s*(?:[,|\/]\s*([+-]?[\d\.]+)(%?)\s*)?\)$/;
    	var per = /^rgba?\(\s*([+-]?[\d\.]+)\%\s*,?\s*([+-]?[\d\.]+)\%\s*,?\s*([+-]?[\d\.]+)\%\s*(?:[,|\/]\s*([+-]?[\d\.]+)(%?)\s*)?\)$/;
    	var keyword = /^(\w+)$/;

    	var rgb = [0, 0, 0, 1];
    	var match;
    	var i;
    	var hexAlpha;

    	if (match = string.match(hex)) {
    		hexAlpha = match[2];
    		match = match[1];

    		for (i = 0; i < 3; i++) {
    			// https://jsperf.com/slice-vs-substr-vs-substring-methods-long-string/19
    			var i2 = i * 2;
    			rgb[i] = parseInt(match.slice(i2, i2 + 2), 16);
    		}

    		if (hexAlpha) {
    			rgb[3] = parseInt(hexAlpha, 16) / 255;
    		}
    	} else if (match = string.match(abbr)) {
    		match = match[1];
    		hexAlpha = match[3];

    		for (i = 0; i < 3; i++) {
    			rgb[i] = parseInt(match[i] + match[i], 16);
    		}

    		if (hexAlpha) {
    			rgb[3] = parseInt(hexAlpha + hexAlpha, 16) / 255;
    		}
    	} else if (match = string.match(rgba)) {
    		for (i = 0; i < 3; i++) {
    			rgb[i] = parseInt(match[i + 1], 0);
    		}

    		if (match[4]) {
    			if (match[5]) {
    				rgb[3] = parseFloat(match[4]) * 0.01;
    			} else {
    				rgb[3] = parseFloat(match[4]);
    			}
    		}
    	} else if (match = string.match(per)) {
    		for (i = 0; i < 3; i++) {
    			rgb[i] = Math.round(parseFloat(match[i + 1]) * 2.55);
    		}

    		if (match[4]) {
    			if (match[5]) {
    				rgb[3] = parseFloat(match[4]) * 0.01;
    			} else {
    				rgb[3] = parseFloat(match[4]);
    			}
    		}
    	} else if (match = string.match(keyword)) {
    		if (match[1] === 'transparent') {
    			return [0, 0, 0, 0];
    		}

    		if (!hasOwnProperty.call(colorNames, match[1])) {
    			return null;
    		}

    		rgb = colorNames[match[1]];
    		rgb[3] = 1;

    		return rgb;
    	} else {
    		return null;
    	}

    	for (i = 0; i < 3; i++) {
    		rgb[i] = clamp(rgb[i], 0, 255);
    	}
    	rgb[3] = clamp(rgb[3], 0, 1);

    	return rgb;
    };

    cs.get.hsl = function (string) {
    	if (!string) {
    		return null;
    	}

    	var hsl = /^hsla?\(\s*([+-]?(?:\d{0,3}\.)?\d+)(?:deg)?\s*,?\s*([+-]?[\d\.]+)%\s*,?\s*([+-]?[\d\.]+)%\s*(?:[,|\/]\s*([+-]?(?=\.\d|\d)(?:0|[1-9]\d*)?(?:\.\d*)?(?:[eE][+-]?\d+)?)\s*)?\)$/;
    	var match = string.match(hsl);

    	if (match) {
    		var alpha = parseFloat(match[4]);
    		var h = ((parseFloat(match[1]) % 360) + 360) % 360;
    		var s = clamp(parseFloat(match[2]), 0, 100);
    		var l = clamp(parseFloat(match[3]), 0, 100);
    		var a = clamp(isNaN(alpha) ? 1 : alpha, 0, 1);

    		return [h, s, l, a];
    	}

    	return null;
    };

    cs.get.hwb = function (string) {
    	if (!string) {
    		return null;
    	}

    	var hwb = /^hwb\(\s*([+-]?\d{0,3}(?:\.\d+)?)(?:deg)?\s*,\s*([+-]?[\d\.]+)%\s*,\s*([+-]?[\d\.]+)%\s*(?:,\s*([+-]?(?=\.\d|\d)(?:0|[1-9]\d*)?(?:\.\d*)?(?:[eE][+-]?\d+)?)\s*)?\)$/;
    	var match = string.match(hwb);

    	if (match) {
    		var alpha = parseFloat(match[4]);
    		var h = ((parseFloat(match[1]) % 360) + 360) % 360;
    		var w = clamp(parseFloat(match[2]), 0, 100);
    		var b = clamp(parseFloat(match[3]), 0, 100);
    		var a = clamp(isNaN(alpha) ? 1 : alpha, 0, 1);
    		return [h, w, b, a];
    	}

    	return null;
    };

    cs.to.hex = function () {
    	var rgba = swizzle(arguments);

    	return (
    		'#' +
    		hexDouble(rgba[0]) +
    		hexDouble(rgba[1]) +
    		hexDouble(rgba[2]) +
    		(rgba[3] < 1
    			? (hexDouble(Math.round(rgba[3] * 255)))
    			: '')
    	);
    };

    cs.to.rgb = function () {
    	var rgba = swizzle(arguments);

    	return rgba.length < 4 || rgba[3] === 1
    		? 'rgb(' + Math.round(rgba[0]) + ', ' + Math.round(rgba[1]) + ', ' + Math.round(rgba[2]) + ')'
    		: 'rgba(' + Math.round(rgba[0]) + ', ' + Math.round(rgba[1]) + ', ' + Math.round(rgba[2]) + ', ' + rgba[3] + ')';
    };

    cs.to.rgb.percent = function () {
    	var rgba = swizzle(arguments);

    	var r = Math.round(rgba[0] / 255 * 100);
    	var g = Math.round(rgba[1] / 255 * 100);
    	var b = Math.round(rgba[2] / 255 * 100);

    	return rgba.length < 4 || rgba[3] === 1
    		? 'rgb(' + r + '%, ' + g + '%, ' + b + '%)'
    		: 'rgba(' + r + '%, ' + g + '%, ' + b + '%, ' + rgba[3] + ')';
    };

    cs.to.hsl = function () {
    	var hsla = swizzle(arguments);
    	return hsla.length < 4 || hsla[3] === 1
    		? 'hsl(' + hsla[0] + ', ' + hsla[1] + '%, ' + hsla[2] + '%)'
    		: 'hsla(' + hsla[0] + ', ' + hsla[1] + '%, ' + hsla[2] + '%, ' + hsla[3] + ')';
    };

    // hwb is a bit different than rgb(a) & hsl(a) since there is no alpha specific syntax
    // (hwb have alpha optional & 1 is default value)
    cs.to.hwb = function () {
    	var hwba = swizzle(arguments);

    	var a = '';
    	if (hwba.length >= 4 && hwba[3] !== 1) {
    		a = ', ' + hwba[3];
    	}

    	return 'hwb(' + hwba[0] + ', ' + hwba[1] + '%, ' + hwba[2] + '%' + a + ')';
    };

    cs.to.keyword = function (rgb) {
    	return reverseNames[rgb.slice(0, 3)];
    };

    // helpers
    function clamp(num, min, max) {
    	return Math.min(Math.max(min, num), max);
    }

    function hexDouble(num) {
    	var str = Math.round(num).toString(16).toUpperCase();
    	return (str.length < 2) ? '0' + str : str;
    }

    var colorStringExports = colorString$1.exports;
    var colorString = /*@__PURE__*/getDefaultExportFromCjs(colorStringExports);

    function getOptions(inOptions) {
        const type = inOptions?.type ?? "png";
        const defaults = type === "png" ? BITMAP_OPTIONS : VECTOR_OPTIONS;
        return { ...defaults, ...inOptions };
    }
    function colorToHex(color) {
        if (typeof color === "string") {
            return colorString.to.hex(colorString.get.rgb(color));
        }
        return `#${(color >>> 8).toString(16).padStart(6, "0")}`;
    }
    function getSVGPath(matrix, size, margin = 0, borderRadius = 0) {
        let rectangles = [];
        for (let x = 0; x < matrix.length; x++) {
            const column = matrix[x];
            for (let y = 0; y < column.length; y++) {
                if (column[y]) {
                    const leftX = x * size + margin;
                    const rightX = (x + 1) * size + margin;
                    const topY = y * size + margin;
                    const bottomY = (y + 1) * size + margin;
                    const rectangle = [];
                    rectangle.push(`M ${leftX} ${topY + borderRadius}`);
                    rectangle.push(`L ${leftX} ${bottomY - borderRadius}`);
                    if (borderRadius > 0) {
                        rectangle.push(`A ${borderRadius} ${borderRadius} 0 0 0 ${leftX + borderRadius} ${bottomY} `);
                    }
                    rectangle.push(`L ${rightX - borderRadius} ${bottomY}`);
                    if (borderRadius > 0) {
                        rectangle.push(`A ${borderRadius} ${borderRadius} 0 0 0 ${rightX} ${bottomY - borderRadius}`);
                    }
                    rectangle.push(`L ${rightX} ${topY + borderRadius}`);
                    if (borderRadius > 0) {
                        rectangle.push(`A ${borderRadius} ${borderRadius} 0 0 0 ${rightX - borderRadius} ${topY}`);
                    }
                    rectangle.push(`L ${leftX + borderRadius} ${topY}`);
                    if (borderRadius > 0) {
                        rectangle.push(`A ${borderRadius} ${borderRadius} 0 0 0 ${leftX} ${topY + borderRadius}`);
                    }
                    rectangle.push(`z`);
                    rectangles.push(rectangle.join(" "));
                }
            }
        }
        return rectangles.join(" ");
    }
    const commonOptions = {
        type: "png",
        parse_url: false,
        ec_level: "M",
        logo: undefined,
        logoWidth: 20,
        logoHeight: 20,
        bgColor: 0xffffffff,
        color: 0x000000ff,
    };
    const BITMAP_OPTIONS = {
        ...commonOptions,
        margin: 1,
        size: 5,
    };
    const VECTOR_OPTIONS = {
        ...commonOptions,
        margin: 1,
        size: 0,
    };

    /**
     *  base64.ts
     *
     *  Licensed under the BSD 3-Clause License.
     *    http://opensource.org/licenses/BSD-3-Clause
     *
     *  References:
     *    http://en.wikipedia.org/wiki/Base64
     *
     * @author Dan Kogai (https://github.com/dankogai)
     */
    const version = '3.7.7';
    /**
     * @deprecated use lowercase `version`.
     */
    const VERSION = version;
    const _hasBuffer = typeof Buffer === 'function';
    const _TD = typeof TextDecoder === 'function' ? new TextDecoder() : undefined;
    const _TE = typeof TextEncoder === 'function' ? new TextEncoder() : undefined;
    const b64ch = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=';
    const b64chs = Array.prototype.slice.call(b64ch);
    const b64tab = ((a) => {
        let tab = {};
        a.forEach((c, i) => tab[c] = i);
        return tab;
    })(b64chs);
    const b64re = /^(?:[A-Za-z\d+\/]{4})*?(?:[A-Za-z\d+\/]{2}(?:==)?|[A-Za-z\d+\/]{3}=?)?$/;
    const _fromCC = String.fromCharCode.bind(String);
    const _U8Afrom = typeof Uint8Array.from === 'function'
        ? Uint8Array.from.bind(Uint8Array)
        : (it) => new Uint8Array(Array.prototype.slice.call(it, 0));
    const _mkUriSafe = (src) => src
        .replace(/=/g, '').replace(/[+\/]/g, (m0) => m0 == '+' ? '-' : '_');
    const _tidyB64 = (s) => s.replace(/[^A-Za-z0-9\+\/]/g, '');
    /**
     * polyfill version of `btoa`
     */
    const btoaPolyfill = (bin) => {
        // console.log('polyfilled');
        let u32, c0, c1, c2, asc = '';
        const pad = bin.length % 3;
        for (let i = 0; i < bin.length;) {
            if ((c0 = bin.charCodeAt(i++)) > 255 ||
                (c1 = bin.charCodeAt(i++)) > 255 ||
                (c2 = bin.charCodeAt(i++)) > 255)
                throw new TypeError('invalid character found');
            u32 = (c0 << 16) | (c1 << 8) | c2;
            asc += b64chs[u32 >> 18 & 63]
                + b64chs[u32 >> 12 & 63]
                + b64chs[u32 >> 6 & 63]
                + b64chs[u32 & 63];
        }
        return pad ? asc.slice(0, pad - 3) + "===".substring(pad) : asc;
    };
    /**
     * does what `window.btoa` of web browsers do.
     * @param {String} bin binary string
     * @returns {string} Base64-encoded string
     */
    const _btoa = typeof btoa === 'function' ? (bin) => btoa(bin)
        : _hasBuffer ? (bin) => Buffer.from(bin, 'binary').toString('base64')
            : btoaPolyfill;
    const _fromUint8Array = _hasBuffer
        ? (u8a) => Buffer.from(u8a).toString('base64')
        : (u8a) => {
            // cf. https://stackoverflow.com/questions/12710001/how-to-convert-uint8-array-to-base64-encoded-string/12713326#12713326
            const maxargs = 0x1000;
            let strs = [];
            for (let i = 0, l = u8a.length; i < l; i += maxargs) {
                strs.push(_fromCC.apply(null, u8a.subarray(i, i + maxargs)));
            }
            return _btoa(strs.join(''));
        };
    /**
     * converts a Uint8Array to a Base64 string.
     * @param {boolean} [urlsafe] URL-and-filename-safe a la RFC4648 §5
     * @returns {string} Base64 string
     */
    const fromUint8Array = (u8a, urlsafe = false) => urlsafe ? _mkUriSafe(_fromUint8Array(u8a)) : _fromUint8Array(u8a);
    // This trick is found broken https://github.com/dankogai/js-base64/issues/130
    // const utob = (src: string) => unescape(encodeURIComponent(src));
    // reverting good old fationed regexp
    const cb_utob = (c) => {
        if (c.length < 2) {
            var cc = c.charCodeAt(0);
            return cc < 0x80 ? c
                : cc < 0x800 ? (_fromCC(0xc0 | (cc >>> 6))
                    + _fromCC(0x80 | (cc & 0x3f)))
                    : (_fromCC(0xe0 | ((cc >>> 12) & 0x0f))
                        + _fromCC(0x80 | ((cc >>> 6) & 0x3f))
                        + _fromCC(0x80 | (cc & 0x3f)));
        }
        else {
            var cc = 0x10000
                + (c.charCodeAt(0) - 0xD800) * 0x400
                + (c.charCodeAt(1) - 0xDC00);
            return (_fromCC(0xf0 | ((cc >>> 18) & 0x07))
                + _fromCC(0x80 | ((cc >>> 12) & 0x3f))
                + _fromCC(0x80 | ((cc >>> 6) & 0x3f))
                + _fromCC(0x80 | (cc & 0x3f)));
        }
    };
    const re_utob = /[\uD800-\uDBFF][\uDC00-\uDFFFF]|[^\x00-\x7F]/g;
    /**
     * @deprecated should have been internal use only.
     * @param {string} src UTF-8 string
     * @returns {string} UTF-16 string
     */
    const utob = (u) => u.replace(re_utob, cb_utob);
    //
    const _encode = _hasBuffer
        ? (s) => Buffer.from(s, 'utf8').toString('base64')
        : _TE
            ? (s) => _fromUint8Array(_TE.encode(s))
            : (s) => _btoa(utob(s));
    /**
     * converts a UTF-8-encoded string to a Base64 string.
     * @param {boolean} [urlsafe] if `true` make the result URL-safe
     * @returns {string} Base64 string
     */
    const encode = (src, urlsafe = false) => urlsafe
        ? _mkUriSafe(_encode(src))
        : _encode(src);
    /**
     * converts a UTF-8-encoded string to URL-safe Base64 RFC4648 §5.
     * @returns {string} Base64 string
     */
    const encodeURI = (src) => encode(src, true);
    // This trick is found broken https://github.com/dankogai/js-base64/issues/130
    // const btou = (src: string) => decodeURIComponent(escape(src));
    // reverting good old fationed regexp
    const re_btou = /[\xC0-\xDF][\x80-\xBF]|[\xE0-\xEF][\x80-\xBF]{2}|[\xF0-\xF7][\x80-\xBF]{3}/g;
    const cb_btou = (cccc) => {
        switch (cccc.length) {
            case 4:
                var cp = ((0x07 & cccc.charCodeAt(0)) << 18)
                    | ((0x3f & cccc.charCodeAt(1)) << 12)
                    | ((0x3f & cccc.charCodeAt(2)) << 6)
                    | (0x3f & cccc.charCodeAt(3)), offset = cp - 0x10000;
                return (_fromCC((offset >>> 10) + 0xD800)
                    + _fromCC((offset & 0x3FF) + 0xDC00));
            case 3:
                return _fromCC(((0x0f & cccc.charCodeAt(0)) << 12)
                    | ((0x3f & cccc.charCodeAt(1)) << 6)
                    | (0x3f & cccc.charCodeAt(2)));
            default:
                return _fromCC(((0x1f & cccc.charCodeAt(0)) << 6)
                    | (0x3f & cccc.charCodeAt(1)));
        }
    };
    /**
     * @deprecated should have been internal use only.
     * @param {string} src UTF-16 string
     * @returns {string} UTF-8 string
     */
    const btou = (b) => b.replace(re_btou, cb_btou);
    /**
     * polyfill version of `atob`
     */
    const atobPolyfill = (asc) => {
        // console.log('polyfilled');
        asc = asc.replace(/\s+/g, '');
        if (!b64re.test(asc))
            throw new TypeError('malformed base64.');
        asc += '=='.slice(2 - (asc.length & 3));
        let u24, bin = '', r1, r2;
        for (let i = 0; i < asc.length;) {
            u24 = b64tab[asc.charAt(i++)] << 18
                | b64tab[asc.charAt(i++)] << 12
                | (r1 = b64tab[asc.charAt(i++)]) << 6
                | (r2 = b64tab[asc.charAt(i++)]);
            bin += r1 === 64 ? _fromCC(u24 >> 16 & 255)
                : r2 === 64 ? _fromCC(u24 >> 16 & 255, u24 >> 8 & 255)
                    : _fromCC(u24 >> 16 & 255, u24 >> 8 & 255, u24 & 255);
        }
        return bin;
    };
    /**
     * does what `window.atob` of web browsers do.
     * @param {String} asc Base64-encoded string
     * @returns {string} binary string
     */
    const _atob = typeof atob === 'function' ? (asc) => atob(_tidyB64(asc))
        : _hasBuffer ? (asc) => Buffer.from(asc, 'base64').toString('binary')
            : atobPolyfill;
    //
    const _toUint8Array = _hasBuffer
        ? (a) => _U8Afrom(Buffer.from(a, 'base64'))
        : (a) => _U8Afrom(_atob(a).split('').map(c => c.charCodeAt(0)));
    /**
     * converts a Base64 string to a Uint8Array.
     */
    const toUint8Array = (a) => _toUint8Array(_unURI(a));
    //
    const _decode = _hasBuffer
        ? (a) => Buffer.from(a, 'base64').toString('utf8')
        : _TD
            ? (a) => _TD.decode(_toUint8Array(a))
            : (a) => btou(_atob(a));
    const _unURI = (a) => _tidyB64(a.replace(/[-_]/g, (m0) => m0 == '-' ? '+' : '/'));
    /**
     * converts a Base64 string to a UTF-8 string.
     * @param {String} src Base64 string.  Both normal and URL-safe are supported
     * @returns {string} UTF-8 string
     */
    const decode = (src) => _decode(_unURI(src));
    /**
     * check if a value is a valid Base64 string
     * @param {String} src a value to check
      */
    const isValid = (src) => {
        if (typeof src !== 'string')
            return false;
        const s = src.replace(/\s+/g, '').replace(/={0,2}$/, '');
        return !/[^\s0-9a-zA-Z\+/]/.test(s) || !/[^\s0-9a-zA-Z\-_]/.test(s);
    };
    //
    const _noEnum = (v) => {
        return {
            value: v, enumerable: false, writable: true, configurable: true
        };
    };
    /**
     * extend String.prototype with relevant methods
     */
    const extendString = function () {
        const _add = (name, body) => Object.defineProperty(String.prototype, name, _noEnum(body));
        _add('fromBase64', function () { return decode(this); });
        _add('toBase64', function (urlsafe) { return encode(this, urlsafe); });
        _add('toBase64URI', function () { return encode(this, true); });
        _add('toBase64URL', function () { return encode(this, true); });
        _add('toUint8Array', function () { return toUint8Array(this); });
    };
    /**
     * extend Uint8Array.prototype with relevant methods
     */
    const extendUint8Array = function () {
        const _add = (name, body) => Object.defineProperty(Uint8Array.prototype, name, _noEnum(body));
        _add('toBase64', function (urlsafe) { return fromUint8Array(this, urlsafe); });
        _add('toBase64URI', function () { return fromUint8Array(this, true); });
        _add('toBase64URL', function () { return fromUint8Array(this, true); });
    };
    /**
     * extend Builtin prototypes with relevant methods
     */
    const extendBuiltins = () => {
        extendString();
        extendUint8Array();
    };
    const gBase64 = {
        version: version,
        VERSION: VERSION,
        atob: _atob,
        atobPolyfill: atobPolyfill,
        btoa: _btoa,
        btoaPolyfill: btoaPolyfill,
        fromBase64: decode,
        toBase64: encode,
        encode: encode,
        encodeURI: encodeURI,
        encodeURL: encodeURI,
        utob: utob,
        btou: btou,
        decode: decode,
        isValid: isValid,
        fromUint8Array: fromUint8Array,
        toUint8Array: toUint8Array,
        extendString: extendString,
        extendUint8Array: extendUint8Array,
        extendBuiltins: extendBuiltins
    };

    async function getPNG(text, inOptions) {
        const options = getOptions(inOptions);
        const matrix = QR(text, options.ec_level, options.parse_url);
        return generateImage({ matrix, ...options, type: 'png' });
    }
    function dataURItoArrayBuffer(dataURI) {
        return gBase64.toUint8Array(dataURI.split(',')[1]);
    }
    function blobToDataURL(blob) {
        return new Promise((resolve, reject) => {
            try {
                var a = new FileReader();
                a.onload = function (e) { resolve(e.target.result); };
                a.onerror = reject;
                a.readAsDataURL(blob);
            }
            catch (e) {
                reject(e);
            }
        });
    }
    async function generateImage({ matrix, size, margin, logo, logoWidth, logoHeight, color, bgColor, borderRadius, }) {
        const marginPx = margin * size;
        const imageSize = matrix.length * size + marginPx * 2;
        const canvas = document.createElement('canvas');
        canvas.width = imageSize;
        canvas.height = imageSize;
        const context = canvas.getContext('2d');
        context.fillStyle = colorToHex(bgColor);
        context.fillRect(0, 0, imageSize, imageSize);
        const path = new Path2D(getSVGPath(matrix, size, marginPx, borderRadius));
        context.fillStyle = colorToHex(color);
        context.fill(path);
        if (logo) {
            const logoImage = await new Promise(async (resolve, reject) => {
                try {
                    const image = new Image();
                    image.onload = () => resolve(image);
                    image.onerror = reject;
                    image.src = await blobToDataURL(new window.Blob([logo]));
                }
                catch (e) {
                    reject(e);
                }
            });
            context.drawImage(logoImage, imageSize / 2 - (logoWidth / 2 / 100) * imageSize, imageSize / 2 - (logoHeight / 2 / 100) * imageSize, (logoWidth / 100) * imageSize, (logoHeight / 100) * imageSize);
        }
        return dataURItoArrayBuffer(canvas.toDataURL('image/png'));
    }

    exports.generateImage = generateImage;
    exports.getPNG = getPNG;

}));
//# sourceMappingURL=png.umd.js.map
