/*
 * Decompiled with CFR 0.152.
 */
package com.maxmind.db;

import com.maxmind.db.Buffer;
import com.maxmind.db.BufferHolder;
import com.maxmind.db.CachedConstructor;
import com.maxmind.db.CachedCreator;
import com.maxmind.db.ClosedDatabaseException;
import com.maxmind.db.DatabaseRecord;
import com.maxmind.db.Decoder;
import com.maxmind.db.DeserializationException;
import com.maxmind.db.InvalidDatabaseException;
import com.maxmind.db.InvalidNetworkException;
import com.maxmind.db.Metadata;
import com.maxmind.db.Network;
import com.maxmind.db.Networks;
import com.maxmind.db.NoCache;
import com.maxmind.db.NodeCache;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;

public final class Reader
implements Closeable {
    private static final int IPV4_LEN = 4;
    private static final int DATA_SECTION_SEPARATOR_SIZE = 16;
    private static final byte[] METADATA_START_MARKER = new byte[]{-85, -51, -17, 77, 97, 120, 77, 105, 110, 100, 46, 99, 111, 109};
    private final long ipV4Start;
    private final Metadata metadata;
    private final int nodeByteSize;
    private final long searchTreeSize;
    private final AtomicReference<BufferHolder> bufferHolderReference;
    private final NodeCache cache;
    private final ConcurrentHashMap<Class<?>, CachedConstructor<?>> constructors;
    private final ConcurrentHashMap<Class<?>, CachedCreator> creators;

    public Reader(File database) throws IOException {
        this(database, (NodeCache)NoCache.getInstance());
    }

    Reader(File database, int chunkSize) throws IOException {
        this(new BufferHolder(database, FileMode.MEMORY_MAPPED, chunkSize), database.getName(), (NodeCache)NoCache.getInstance());
    }

    public Reader(File database, NodeCache cache) throws IOException {
        this(database, FileMode.MEMORY_MAPPED, cache);
    }

    Reader(File database, NodeCache cache, int chunkSize) throws IOException {
        this(new BufferHolder(database, FileMode.MEMORY_MAPPED, chunkSize), database.getName(), cache);
    }

    public Reader(InputStream source) throws IOException {
        this(source, (NodeCache)NoCache.getInstance());
    }

    Reader(InputStream source, int chunkSize) throws IOException {
        this(source, (NodeCache)NoCache.getInstance(), chunkSize);
    }

    public Reader(InputStream source, NodeCache cache) throws IOException {
        this(source, cache, 0x7FFFFFF7);
    }

    Reader(InputStream source, NodeCache cache, int chunkSize) throws IOException {
        this(new BufferHolder(source, chunkSize), "<InputStream>", cache);
    }

    public Reader(File database, FileMode fileMode) throws IOException {
        this(database, fileMode, (NodeCache)NoCache.getInstance());
    }

    public Reader(File database, FileMode fileMode, NodeCache cache) throws IOException {
        this(new BufferHolder(database, fileMode), database.getName(), cache);
    }

    private Reader(BufferHolder bufferHolder, String name, NodeCache cache) throws IOException {
        this.bufferHolderReference = new AtomicReference<BufferHolder>(bufferHolder);
        if (cache == null) {
            throw new NullPointerException("Cache cannot be null");
        }
        this.cache = cache;
        Buffer buffer = bufferHolder.get();
        long start = this.findMetadataStart(buffer, name);
        Decoder metadataDecoder = new Decoder(this.cache, buffer, start);
        this.metadata = metadataDecoder.decode(start, Metadata.class);
        this.nodeByteSize = this.metadata.recordSize() / 4;
        this.searchTreeSize = this.metadata.nodeCount() * (long)this.nodeByteSize;
        this.ipV4Start = this.findIpV4StartNode(buffer);
        this.constructors = new ConcurrentHashMap();
        this.creators = new ConcurrentHashMap();
    }

    public <T> T get(InetAddress ipAddress, Class<T> cls) throws IOException {
        return this.getRecord(ipAddress, cls).data();
    }

    long getIpv4Start() {
        return this.ipV4Start;
    }

    public <T> DatabaseRecord<T> getRecord(InetAddress ipAddress, Class<T> cls) throws IOException {
        byte[] rawAddress = ipAddress.getAddress();
        long[] traverseResult = this.traverseTree(rawAddress, rawAddress.length * 8);
        long record = traverseResult[0];
        int prefixLength = (int)traverseResult[1];
        long nodeCount = this.metadata.nodeCount();
        Buffer buffer = this.getBufferHolder().get();
        Network network = new Network(ipAddress, prefixLength);
        Object dataRecord = null;
        if (record > nodeCount) {
            try {
                dataRecord = this.resolveDataPointer(buffer, record, cls, ipAddress, network);
            }
            catch (DeserializationException exception) {
                throw new DeserializationException("Error getting record for IP " + String.valueOf(ipAddress) + " -  " + exception.getMessage(), exception);
            }
        }
        return new DatabaseRecord<Object>(dataRecord, network);
    }

    public <T> Networks<T> networks(Class<T> typeParameterClass) throws InvalidNetworkException, ClosedDatabaseException, InvalidDatabaseException {
        return this.networks(false, typeParameterClass);
    }

    public <T> Networks<T> networks(boolean includeAliasedNetworks, Class<T> typeParameterClass) throws InvalidNetworkException, ClosedDatabaseException, InvalidDatabaseException {
        try {
            if (this.getMetadata().ipVersion() == 6) {
                InetAddress ipv6 = InetAddress.getByAddress(new byte[16]);
                Network ipAllV6 = new Network(ipv6, 0);
                return this.networksWithin(ipAllV6, includeAliasedNetworks, typeParameterClass);
            }
            InetAddress ipv4 = InetAddress.getByAddress(new byte[4]);
            Network ipAllV4 = new Network(ipv4, 0);
            return this.networksWithin(ipAllV4, includeAliasedNetworks, typeParameterClass);
        }
        catch (UnknownHostException e) {
            return null;
        }
    }

    BufferHolder getBufferHolder() throws ClosedDatabaseException {
        BufferHolder bufferHolder = this.bufferHolderReference.get();
        if (bufferHolder == null) {
            throw new ClosedDatabaseException();
        }
        return bufferHolder;
    }

    private long startNode(int bitLength) {
        if (this.metadata.ipVersion() == 6 && bitLength == 32) {
            return this.ipV4Start;
        }
        return 0L;
    }

    private long findIpV4StartNode(Buffer buffer) throws InvalidDatabaseException {
        if (this.metadata.ipVersion() == 4) {
            return 0L;
        }
        long node = 0L;
        for (int i = 0; i < 96 && node < this.metadata.nodeCount(); ++i) {
            node = this.readNode(buffer, node, 0);
        }
        return node;
    }

    public <T> Networks<T> networksWithin(Network network, boolean includeAliasedNetworks, Class<T> typeParameterClass) throws InvalidNetworkException, ClosedDatabaseException, InvalidDatabaseException {
        InetAddress networkAddress = network.networkAddress();
        if (this.metadata.ipVersion() == 4 && networkAddress instanceof Inet6Address) {
            throw new InvalidNetworkException(networkAddress);
        }
        byte[] ipBytes = networkAddress.getAddress();
        int prefixLength = network.prefixLength();
        if (this.metadata.ipVersion() == 6 && ipBytes.length == 4) {
            ipBytes = includeAliasedNetworks ? new byte[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -1, -1, ipBytes[0], ipBytes[1], ipBytes[2], ipBytes[3]} : new byte[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, ipBytes[0], ipBytes[1], ipBytes[2], ipBytes[3]};
            prefixLength += 96;
        }
        long[] traverseResult = this.traverseTree(ipBytes, prefixLength);
        long node = traverseResult[0];
        int prefix = (int)traverseResult[1];
        return new Networks<T>(this, includeAliasedNetworks, new Networks.NetworkNode[]{new Networks.NetworkNode(ipBytes, prefix, node)}, typeParameterClass);
    }

    private long[] traverseTree(byte[] ip, int bitCount) throws ClosedDatabaseException, InvalidDatabaseException {
        int i;
        Buffer buffer = this.getBufferHolder().get();
        int bitLength = ip.length * 8;
        long record = this.startNode(bitLength);
        long nodeCount = this.metadata.nodeCount();
        for (i = 0; i < bitCount && record < nodeCount; ++i) {
            int b = 0xFF & ip[i / 8];
            int bit = 1 & b >> 7 - i % 8;
            record = this.readNode(buffer, record, bit);
        }
        return new long[]{record, i};
    }

    long readNode(Buffer buffer, long nodeNumber, int index) throws InvalidDatabaseException {
        long baseOffset = nodeNumber * (long)this.nodeByteSize;
        int recordSize = this.metadata.recordSize();
        return switch (recordSize) {
            case 24 -> {
                buffer.position(baseOffset + (long)index * 3L);
                yield Decoder.decodeLong(buffer, 0, 3);
            }
            case 28 -> {
                int middle = buffer.get(baseOffset + 3L);
                middle = index == 0 ? (0xF0 & middle) >>> 4 : 0xF & middle;
                buffer.position(baseOffset + (long)index * 4L);
                yield Decoder.decodeLong(buffer, middle, 3);
            }
            case 32 -> {
                buffer.position(baseOffset + (long)index * 4L);
                yield Decoder.decodeLong(buffer, 0, 4);
            }
            default -> throw new InvalidDatabaseException("Unknown record size: " + recordSize);
        };
    }

    <T> T resolveDataPointer(Buffer buffer, long pointer, Class<T> cls, InetAddress lookupIp, Network network) throws IOException {
        long resolved = pointer - this.metadata.nodeCount() + this.searchTreeSize;
        if (resolved >= buffer.capacity()) {
            throw new InvalidDatabaseException("The MaxMind DB file's search tree is corrupt: contains pointer larger than the database.");
        }
        Decoder decoder = new Decoder(this.cache, buffer, this.searchTreeSize + 16L, this.constructors, this.creators, lookupIp, network);
        return decoder.decode(resolved, cls);
    }

    private long findMetadataStart(Buffer buffer, String databaseName) throws InvalidDatabaseException {
        long fileSize = buffer.capacity();
        block0: for (long i = 0L; i < fileSize - (long)METADATA_START_MARKER.length + 1L; ++i) {
            for (int j = 0; j < METADATA_START_MARKER.length; ++j) {
                byte b = buffer.get(fileSize - i - (long)j - 1L);
                if (b != METADATA_START_MARKER[METADATA_START_MARKER.length - j - 1]) continue block0;
            }
            return fileSize - i;
        }
        throw new InvalidDatabaseException("Could not find a MaxMind DB metadata marker in this file (" + databaseName + "). Is this a valid MaxMind DB file?");
    }

    public Metadata getMetadata() {
        return this.metadata;
    }

    @Override
    public void close() throws IOException {
        this.bufferHolderReference.set(null);
    }

    public static enum FileMode {
        MEMORY_MAPPED,
        MEMORY;

    }
}

