"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019, 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
const Sentry = __importStar(require("@sentry/node"));
const matrix_bot_sdk_1 = require("matrix-bot-sdk");
const utils_1 = require("../utils");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const levelToFn = {
    [matrix_bot_sdk_1.LogLevel.DEBUG.toString()]: matrix_bot_sdk_1.LogService.debug,
    [matrix_bot_sdk_1.LogLevel.INFO.toString()]: matrix_bot_sdk_1.LogService.info,
    [matrix_bot_sdk_1.LogLevel.WARN.toString()]: matrix_bot_sdk_1.LogService.warn,
    [matrix_bot_sdk_1.LogLevel.ERROR.toString()]: matrix_bot_sdk_1.LogService.error,
};
/**
 * Allows the different componenets of draupnir to send messages back to the management room without introducing a dependency on the entirity of a `Draupnir` instance.
 */
class StandardManagementRoomOutput {
    constructor(managementRoomDetail, clientUserID, client, config) {
        this.managementRoomDetail = managementRoomDetail;
        this.clientUserID = clientUserID;
        this.client = client;
        this.config = config;
        this.isSuccessfullySendingMessages = false;
        // nothing to do.
    }
    get managementRoom() {
        return this.managementRoomDetail.managementRoom;
    }
    get managementRoomID() {
        return this.managementRoomDetail.managementRoomID;
    }
    maybeWarnAboutPublicManagementRoom() {
        if (this.managementRoomDetail.isRoomPublic()) {
            void (0, matrix_protection_suite_1.Task)(this.logMessage(matrix_bot_sdk_1.LogLevel.WARN, "ManagementRoom", "Your management room is public! You should review the join rule of your Matrix room immediately and set it to invite only, otherwise anyone could hijack your protected rooms!"));
        }
    }
    /**
     * Take an arbitrary string and a set of room IDs, and return a
     * TextualMessageEventContent whose plaintext component replaces those room
     * IDs with their canonical aliases, and whose html component replaces those
     * room IDs with their matrix.to room pills.
     *
     * @param client The matrix client on which to query for room aliases
     * @param text An arbitrary string to rewrite with room aliases and pills
     * @param roomIds A set of room IDs to find and replace in `text`
     * @param msgtype The desired message type of the returned TextualMessageEventContent
     * @returns A TextualMessageEventContent with replaced room IDs
     */
    async replaceRoomIdsWithPills(text, roomIds, msgtype = "m.text") {
        const content = {
            body: text,
            formatted_body: (0, utils_1.htmlEscape)(text),
            msgtype: msgtype,
            format: "org.matrix.custom.html",
        };
        // Though spec doesn't say so, room ids that have slashes in them are accepted by Synapse and Dendrite unfortunately for us.
        const escapeRegex = (v) => {
            return v.replace(/[-/\\^$*+?.()|[\]{}]/g, "\\$&");
        };
        const viaServers = [(0, matrix_basic_types_1.userServerName)(this.clientUserID)];
        for (const roomId of roomIds) {
            let alias = roomId;
            try {
                alias = (await this.client.getPublishedAlias(roomId)) || roomId;
            }
            catch (e) {
                // This is a recursive call, so tell the function not to try and call us
                await this.logMessage(matrix_bot_sdk_1.LogLevel.WARN, "utils", `Failed to resolve room alias for ${roomId} - see console for details`, null, true);
                matrix_bot_sdk_1.LogService.warn("ManagementRoomOutput", "Failed resolving room alias when formatting a message", e);
            }
            const regexRoomId = new RegExp(escapeRegex(roomId), "g");
            content.body = content.body.replace(regexRoomId, alias);
            if (content.formatted_body) {
                const permalink = matrix_basic_types_1.Permalinks.forRoom(alias, alias !== roomId ? [] : viaServers);
                content.formatted_body = content.formatted_body.replace(regexRoomId, `<a href="${permalink}">${(0, utils_1.htmlEscape)(alias)}</a>`);
            }
        }
        return content;
    }
    async logMessage(level, module, message, additionalRoomIds = null, isRecursive = false) {
        if (level === matrix_bot_sdk_1.LogLevel.ERROR) {
            Sentry.captureMessage(`${module}: ${message}`, "error");
        }
        if (!additionalRoomIds)
            additionalRoomIds = [];
        if (!Array.isArray(additionalRoomIds))
            additionalRoomIds = [additionalRoomIds];
        const levelsIncluded = matrix_bot_sdk_1.LogLevel[this.config.logLevel];
        if (levelsIncluded.includes(level)) {
            let clientMessage = message;
            if (level === matrix_bot_sdk_1.LogLevel.WARN)
                clientMessage = `⚠ | ${message}`;
            if (level === matrix_bot_sdk_1.LogLevel.ERROR)
                clientMessage = `‼ | ${message}`;
            const client = this.client;
            const roomIds = [this.managementRoomID, ...additionalRoomIds];
            let evContent = {
                body: message,
                formatted_body: (0, utils_1.htmlEscape)(message),
                msgtype: "m.notice",
                format: "org.matrix.custom.html",
            };
            if (!isRecursive) {
                evContent = await this.replaceRoomIdsWithPills(clientMessage, new Set(roomIds), "m.notice");
            }
            try {
                await client.sendMessage(this.managementRoomID, evContent);
            }
            catch (ex) {
                matrix_bot_sdk_1.LogService.error("ManagementRoomOutput", "Failed to log a message to the management room", ex);
            }
        }
        if (!this.isSuccessfullySendingMessages) {
            this.maybeWarnAboutPublicManagementRoom();
            this.isSuccessfullySendingMessages = true;
        }
        const logFunction = levelToFn[level.toString()];
        if (logFunction === undefined) {
            throw new TypeError(`Unable to find logFunction for log level: ${level.toString()}`);
        }
        logFunction(module, message);
    }
}
exports.default = StandardManagementRoomOutput;
//# sourceMappingURL=ManagementRoomOutput.js.map