"use strict";
// SPDX-FileCopyrightText: 2022 - 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from super-cool-stream
// https://github.com/Gnuxie/super-cool-stream
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.RowTrackingStringStream = exports.StringStream = exports.StandardSuperCoolStream = void 0;
class StandardSuperCoolStream {
    /**
     * Makes the super cool string stream.
     * @param source A string to act as the source of the stream.
     * @param start Where in the string we should start reading.
     */
    constructor(source, start = 0) {
        this.source = source;
        this.position = start;
    }
    peekItem(eof) {
        var _a;
        return (_a = this.source.at(this.position)) !== null && _a !== void 0 ? _a : eof;
    }
    readItem(eof) {
        var _a;
        return (_a = this.source.at(this.position++)) !== null && _a !== void 0 ? _a : eof;
    }
    getPosition() {
        return this.position;
    }
    setPosition(n) {
        this.position = n;
    }
    clone() {
        return new StandardSuperCoolStream(this.source, this.position);
    }
    savingPositionIf(description) {
        const previousPosition = this.position;
        const bodyResult = description.body(this);
        if (description.predicate(bodyResult)) {
            this.position = previousPosition;
        }
        return bodyResult;
    }
}
exports.StandardSuperCoolStream = StandardSuperCoolStream;
/**
 * Helper for peeking and reading character by character.
 */
class StringStream extends StandardSuperCoolStream {
    peekChar(eof) {
        return this.peekItem(eof);
    }
    readChar(eof) {
        return this.readItem(eof);
    }
    clone() {
        return new StringStream(this.source, this.position);
    }
}
exports.StringStream = StringStream;
/**
 * Tracks columns and rows. Can't handle any other line ending than line feed atm.
 */
class RowTrackingStringStream extends StringStream {
    constructor(source, position, row = 0, column = 0) {
        super(source, position);
        this.row = row;
        this.column = column;
    }
    readItem(eof) {
        const item = super.readItem(eof);
        if (item === eof) {
            return item;
        }
        else if (item === "\n") {
            this.row++;
            this.column = 0;
            return item;
        }
        else {
            this.column++;
            return item;
        }
    }
    clone() {
        return new RowTrackingStringStream(this.source, this.position, this.row, this.column);
    }
    get peekRow() {
        return this.row;
    }
    get peekColumn() {
        return this.column;
    }
    get readRow() {
        return this.row - 1;
    }
    get readColumn() {
        if (this.row === 0) {
            if (this.column === 0) {
                return 0;
            }
            return this.column;
        }
        else {
            return this.column - 1;
        }
    }
}
exports.RowTrackingStringStream = RowTrackingStringStream;
//# sourceMappingURL=SuperCoolStream.js.map