/**
 * The DeadDocument started as a universal document object model like Pandoc is.
 * That kind of task is just too big for me though and someone else should have
 * done it already. Irregardless, the way this is used is a simple DOM that can
 * be used to incrementally render both HTML and Markdown.
 * The reason we need to incrementally render both HTML and Markdown
 * (which really means serialize, hence `DeadDocument`) is so that
 * we can ensure in Matrix that both renditions of a node (html + markdown)
 * are always in the same event and not split across multiple events.
 * This ensures consistency when someone replies to an event that whichever
 * format the client uses, the reply will be about the same "thing".
 * So we have the power to split messages across multiple Matrix events
 * automatically, without the need for micromanagement.
 *
 * While originally we were going to generate this DOM using a custom
 * internal DSL, we discovered it was possible to use JSX templates with a
 * custom DOM. You can find our own JSXFactory in `./JSXFactory.ts`.
 *
 * This means that end users shouldn't have to touch this DOM directly.
 */
export interface AbstractNode {
    readonly parent: DocumentNode | null;
    readonly leafNode: boolean;
    readonly tag: NodeTag;
}
export interface DocumentNode extends AbstractNode {
    readonly leafNode: false;
    attributeMap: Map<string, string>;
    addChild<Node extends DocumentNode | LeafNode>(node: Node): Node;
    getChildren(): (DocumentNode | LeafNode)[];
    getFirstChild(): DocumentNode | LeafNode | undefined;
}
export interface LeafNode extends AbstractNode {
    readonly parent: DocumentNode;
    readonly data: string;
    readonly leafNode: true;
}
export declare enum NodeTag {
    TextNode = "text",
    InlineCode = "code",
    PreformattedText = "pre",
    Root = "root",
    Strong = "strong",
    Emphasis = "em",
    Paragraph = "p",
    HeadingOne = "h1",
    HeadingTwo = "h2",
    HeadingThree = "h3",
    HeadingFour = "h4",
    HeadingFive = "h5",
    HeadingSix = "h6",
    UnorderedList = "ul",
    OrderedList = "ol",
    ListItem = "li",
    LineBreak = "br",
    BoldFace = "b",
    ItalicFace = "i",
    Anchor = "a",
    Fragment = "fragment",
    Details = "details",
    Summary = "summary",
    Font = "font",
    Span = "span",
    HorizontalRule = "hr"
}
export declare const EmptyFragment: DocumentNode;
/**
 * This is an internal interface so we can provide
 * an implementation of `DocumentNode` in a way
 * where we can use ad-hoc mixins.
 */
interface DeadDocumentNode extends DocumentNode {
    children: (DocumentNode | LeafNode)[];
    attributeMap: Map<string, string>;
}
export declare function addChild<Node extends DocumentNode | LeafNode>(this: DeadDocumentNode, node: Node): Node;
export declare function getChildren(this: DeadDocumentNode): (DocumentNode | LeafNode)[];
export declare function getFirstChild(this: DeadDocumentNode): DocumentNode | LeafNode | undefined;
export declare function makeDocumentNode(tag: NodeTag, parent?: null): DocumentNode;
export declare function makeLeafNode<LeafInterface extends LeafNode>(tag: LeafInterface["tag"], parent: DocumentNode, data: string): LeafInterface;
export interface TextNode extends LeafNode {
    readonly tag: NodeTag.TextNode;
}
export declare function addText(this: DocumentNode, data: string): TextNode;
export interface InlineCodeNode extends LeafNode {
    readonly tag: NodeTag.InlineCode;
}
export declare function addInlineCode(this: DocumentNode, data: string): InlineCodeNode;
export interface PreformattedTextNode extends LeafNode {
    readonly tag: NodeTag.PreformattedText;
}
export declare function addPreformattedText(this: DocumentNode, data: string): PreformattedTextNode;
/**
 * We use a Fringe to render.
 */
export declare enum FringeType {
    Pre = "pre",
    Leaf = "leaf",
    Post = "post"
}
export type FringeLeafRenderFunction<Context> = (tag: NodeTag, node: LeafNode, context: Context) => void;
export type FringeInnerRenderFunction<Context> = (type: FringeType, node: DocumentNode, context: Context, environment: TagDynamicEnvironment) => void;
export interface FringeRenderer<Context> {
    getLeafRenderer(tag: NodeTag): FringeLeafRenderFunction<Context>;
    getPreRenderer(tag: NodeTag): FringeInnerRenderFunction<Context>;
    getPostRenderer(tag: NodeTag): FringeInnerRenderFunction<Context>;
}
export declare class SimpleFringeRenderer<Context> implements FringeRenderer<Context> {
    private readonly preRenderers;
    private readonly leafRenderers;
    private readonly postRenderers;
    private getRenderer;
    getPreRenderer(tag: NodeTag): FringeInnerRenderFunction<Context>;
    getLeafRenderer(tag: NodeTag): FringeLeafRenderFunction<Context>;
    getPostRenderer(tag: NodeTag): FringeInnerRenderFunction<Context>;
    internRenderer<T extends FringeInnerRenderFunction<Context> | FringeLeafRenderFunction<Context>>(type: FringeType, tag: NodeTag, table: Map<NodeTag, T>, renderer: T): void;
    registerRenderer<T extends FringeInnerRenderFunction<Context> | FringeLeafRenderFunction<Context>>(type: FringeType, tag: NodeTag, renderer: T): this;
    registerInnerNode(tag: NodeTag, pre: FringeInnerRenderFunction<Context>, post: FringeInnerRenderFunction<Context>): this;
}
export type CommitHook<Context> = (node: DocumentNode, context: Context) => void;
/**
 * The FringeWalker allows for the implementation of an incremental
 * renderer.
 * Each increment is defined by the fist leaf node to be rendered
 * or the first inner node to have all of its leaves renderered.
 * @param Context is a static context that should be provided to each render function.
 */
export declare class FringeWalker<Context> {
    readonly root: DocumentNode;
    private readonly context;
    private readonly renderer;
    private readonly commitHook;
    private readonly stream;
    private readonly dynamicEnvironment;
    constructor(root: DocumentNode, context: Context, renderer: FringeRenderer<Context>, commitHook: CommitHook<Context>);
    increment(): DocumentNode | undefined;
}
export declare class TagDynamicEnvironmentEntry {
    readonly node: DocumentNode;
    value: unknown;
    readonly previous: undefined | TagDynamicEnvironmentEntry;
    constructor(node: DocumentNode, value: unknown, previous: undefined | TagDynamicEnvironmentEntry);
}
/**
 * A dynamic environment is just an environment of bindings that is made
 * by shadowing previous bindings and pushing and popping bindings "dynamically"
 * for a given variable with some thing.
 *
 * In this example, we push and pop bindings with `DocumentNode`s.
 * For example, if you make a binding to a variable called `indentationLevel`
 * to set it to `1` from a `<ul>` node, then this binding should be popped
 * when the `FringeWalker` reaches the post node (`</ul>`).
 * Howerver, if we encounter another `<ul>`, we can read the existing value
 * for `indentationLevel`, increment it and create a new binding
 * that shadows the existing one. This too will get popped once we encounter
 * the post node for this `<ul>` node (`</ul>`).
 *
 * This makes it very easy to express a situation where you modify and
 * restore variables that depend on node depth when walking the fringe,
 * as the restoration of previous values can be handled automatically for us.
 */
export declare class TagDynamicEnvironment {
    private readonly environments;
    read<T = unknown>(variableName: string): T;
    write<T = unknown>(variableName: string, value: T): T;
    bind<T = unknown>(variableName: string, node: DocumentNode, value: T): T;
    pop(node: DocumentNode): void;
}
export {};
//# sourceMappingURL=DeadDocument.d.ts.map