"use strict";
/**
 * Copyright (C) 2023 Gnuxie <Gnuxie@protonmail.com>
 * All rights reserved.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.BotSDKRoomMembershipManager = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_protection_suite_2 = require("matrix-protection-suite");
const log = new matrix_protection_suite_1.Logger('BotSDKRoomMembershipManager');
async function getRoomMembershipEvents(room, client) {
    const rawMembersResult = await client
        .doRequest('GET', `/_matrix/client/v3/rooms/${encodeURIComponent(room.toRoomIDOrAlias())}/members`)
        .then((ok) => (0, matrix_protection_suite_1.Ok)(ok), (exception) => matrix_protection_suite_1.ActionException.Result(`Unable to query room members from ${room.toPermalink()}`, { exception, exceptionKind: matrix_protection_suite_1.ActionExceptionKind.Unknown }));
    if ((0, matrix_protection_suite_1.isError)(rawMembersResult)) {
        return rawMembersResult;
    }
    const errorMessage = `Unable parse the result of a /members query in ${room.toPermalink()}`;
    const rawMembers = rawMembersResult.ok;
    if (typeof rawMembers !== 'object' || rawMembers === null) {
        return matrix_protection_suite_1.ActionError.Result(errorMessage);
    }
    if (!('chunk' in rawMembers) || !Array.isArray(rawMembers['chunk'])) {
        log.error(errorMessage, rawMembersResult);
        return matrix_protection_suite_1.ActionError.Result(errorMessage);
    }
    const members = [];
    for (const rawEvent of rawMembers['chunk']) {
        const memberResult = matrix_protection_suite_1.Value.Decode(matrix_protection_suite_2.MembershipEvent, rawEvent);
        if ((0, matrix_protection_suite_1.isError)(memberResult)) {
            log.error(
            // Really we'd have something other than adhoc validation, generated from the OpenAPI Schema for the response
            // we don't have that though....
            // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
            `Unable to parse the event ${rawEvent.event_id} from ${rawEvent.room_id}`, JSON.stringify(rawEvent), memberResult.error);
            continue;
        }
        members.push(memberResult.ok);
    }
    return (0, matrix_protection_suite_1.Ok)(members);
}
class BotSDKRoomMembershipManager {
    constructor(clientUserID, client, factory) {
        this.clientUserID = clientUserID;
        this.client = client;
        this.factory = factory;
        // nothing to do.
    }
    async getRoomMembershipRevisionIssuer(room) {
        return await this.factory.getRoomMembershipRevisionIssuer(room, this.clientUserID);
    }
    async getRoomMembershipEvents(room) {
        return await getRoomMembershipEvents(room, this.client);
    }
}
exports.BotSDKRoomMembershipManager = BotSDKRoomMembershipManager;
//# sourceMappingURL=RoomMembershipManager.js.map