"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardPolicyRoomEditor = void 0;
// Copyright 2022 - 2024 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
const crypto_js_1 = require("crypto-js");
const enc_base64_1 = __importDefault(require("crypto-js/enc-base64"));
const PolicyEvents_1 = require("../MatrixTypes/PolicyEvents");
const PolicyRule_1 = require("./PolicyRule");
const Action_1 = require("../Interface/Action");
class StandardPolicyRoomEditor {
    constructor(room, policyRevisionIssuer, roomStateRevisionIssuer, roomStateEventSender) {
        this.room = room;
        this.policyRevisionIssuer = policyRevisionIssuer;
        this.roomStateRevisionIssuer = roomStateRevisionIssuer;
        this.roomStateEventSender = roomStateEventSender;
        // nothing to do.
    }
    async removePolicyByStateKey(ruleType, stateKey) {
        const eventTypesToCheck = (0, PolicyEvents_1.variantsForPolicyRuleType)(ruleType);
        const sendNullState = async (stateType, stateKey) => {
            const sendResult = await this.roomStateEventSender.sendStateEvent(this.room.toRoomIDOrAlias(), stateType, stateKey, {});
            if ((0, Action_1.isError)(sendResult)) {
                return sendResult.elaborate(`Could not remove the policy rule with the type ${ruleType} and state key ${stateKey}`);
            }
            return (0, Action_1.Ok)(undefined);
        };
        const typesToRemoveResults = eventTypesToCheck
            .map((stateType) => {
            var _a;
            return (_a = this.roomStateRevisionIssuer.currentRevision.getStateEvent(stateType, stateKey)) === null || _a === void 0 ? void 0 : _a.type;
        })
            .filter((stateType) => stateType !== undefined);
        if (typesToRemoveResults.length === 0) {
            return (0, Action_1.Ok)(undefined);
        }
        for (const stateType of typesToRemoveResults) {
            const nullResult = await sendNullState(stateType, stateKey);
            if ((0, Action_1.isError)(nullResult)) {
                return nullResult;
            }
        }
        return (0, Action_1.Ok)(undefined);
    }
    async createPolicy(entityType, recommendation, entity, reason, additionalProperties) {
        const stateKey = enc_base64_1.default.stringify((0, crypto_js_1.SHA256)(entity + recommendation));
        const sendResult = await this.roomStateEventSender.sendStateEvent(this.room.toRoomIDOrAlias(), entityType, stateKey, {
            recommendation,
            entity,
            reason,
            ...additionalProperties,
        });
        if ((0, Action_1.isError)(sendResult)) {
            return sendResult.elaborate(`Failed to create a policy for the entity ${entity} with the recommendation ${recommendation} in ${this.room.toPermalink()}`);
        }
        return sendResult;
    }
    async removePolicy(ruleType, recommendation, entity, _reason) {
        const eventTypesToCheck = (0, PolicyEvents_1.variantsForPolicyRuleType)(ruleType);
        const sendNullState = async (stateType, stateKey) => {
            const sendResult = await this.roomStateEventSender.sendStateEvent(this.room.toRoomIDOrAlias(), stateType, stateKey, {});
            if ((0, Action_1.isError)(sendResult)) {
                return sendResult.elaborate(`Could not remove the policy rule ${ruleType} for entity ${entity} with recommendation ${recommendation}.`);
            }
            return (0, Action_1.Ok)(undefined);
        };
        const currentRevision = this.roomStateRevisionIssuer.currentRevision;
        // We also have to remove any legacy versions of the same policy, such as
        // those created with the type org.matrix.mjolnir.*
        const removeRule = async (rule) => {
            const stateKey = rule.sourceEvent.state_key;
            const typesToRemoveResults = eventTypesToCheck
                .map((stateType) => { var _a; return (_a = currentRevision.getStateEvent(stateType, stateKey)) === null || _a === void 0 ? void 0 : _a.type; })
                .filter((stateType) => stateType !== undefined);
            if (typesToRemoveResults.length === 0) {
                return (0, Action_1.Ok)(undefined);
            }
            const removalResults = await Promise.all(typesToRemoveResults.map((stateType) => {
                return sendNullState(stateType, stateKey);
            }));
            const removalErrors = removalResults.filter(Action_1.isError);
            const error = removalErrors[0];
            if (error !== undefined) {
                return error;
            }
            else {
                return (0, Action_1.Ok)(undefined);
            }
        };
        const rules = this.policyRevisionIssuer.currentRevision.allRulesMatchingEntity(entity, {
            type: ruleType,
            searchHashedRules: true,
        });
        const removalErrors = (await Promise.all(rules.map(removeRule))).filter(Action_1.isError);
        const error = removalErrors[0];
        if (error !== undefined) {
            return error;
        }
        else {
            return (0, Action_1.Ok)(rules);
        }
    }
    async banEntity(ruleType, entity, reason) {
        return await this.createPolicy(ruleType, PolicyRule_1.Recommendation.Ban, entity, reason !== null && reason !== void 0 ? reason : '<no reason supplied>', {});
    }
    async takedownEntity(ruleType, entity, options) {
        const recommendation = PolicyRule_1.Recommendation.Takedown;
        const stateKey = enc_base64_1.default.stringify((0, crypto_js_1.SHA256)(entity + recommendation));
        const sendResult = await this.roomStateEventSender.sendStateEvent(this.room.toRoomIDOrAlias(), ruleType, stateKey, {
            recommendation,
            ...(options.shouldHash
                ? {
                    ['org.matrix.msc4205.hashes']: {
                        sha256: enc_base64_1.default.stringify((0, crypto_js_1.SHA256)(entity)),
                    },
                }
                : { entity }),
        });
        if ((0, Action_1.isError)(sendResult)) {
            return sendResult.elaborate(`Failed to create a policy for the entity ${entity} with the recommendation ${recommendation} in ${this.room.toPermalink()}`);
        }
        return sendResult;
    }
    async unbanEntity(ruleType, entity) {
        return await this.removePolicy(ruleType, PolicyRule_1.Recommendation.Ban, entity);
    }
}
exports.StandardPolicyRoomEditor = StandardPolicyRoomEditor;
//# sourceMappingURL=StandardPolicyRoomEditor.js.map