"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpamCheckEndpointPluginManager = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const log = new matrix_protection_suite_1.Logger("SpamCheckEndpointPluginManager");
class SpamCheckEndpointPluginManager {
    constructor() {
        this.blockingHandles = new Set();
        this.nonBlockingHandles = new Set();
    }
    registerBlockingHandle(handle) {
        this.blockingHandles.add(handle);
    }
    registerNonBlockingHandle(handle) {
        this.nonBlockingHandles.add(handle);
    }
    unregisterHandle(handle) {
        this.blockingHandles.delete(handle);
        this.nonBlockingHandles.delete(handle);
    }
    unregisterListeners() {
        this.blockingHandles.clear();
        this.nonBlockingHandles.clear();
    }
    isBlocking() {
        return this.blockingHandles.size > 0;
    }
    async callBlockingHandles(...args) {
        const results = await Promise.allSettled([...this.blockingHandles.values()].map((handle) => handle(...args)));
        for (const result of results) {
            if (result.status === "rejected") {
                log.error("Error processing a blocking spam check callback:", result.reason);
            }
            else {
                if (result.value !== "NOT_SPAM") {
                    return result.value;
                }
            }
        }
        return "NOT_SPAM";
    }
    callNonBlockingHandles(...args) {
        for (const handle of this.nonBlockingHandles) {
            try {
                handle(...args);
            }
            catch (e) {
                log.error("Error processing a non blocking spam check callback:", e);
            }
        }
    }
    callNonBlockingHandlesInTask(...args) {
        void (0, matrix_protection_suite_1.Task)((async () => {
            this.callNonBlockingHandles(...args);
        })());
    }
}
exports.SpamCheckEndpointPluginManager = SpamCheckEndpointPluginManager;
//# sourceMappingURL=SpamCheckEndpointPluginManager.js.map