"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const typescript_result_1 = require("@gnuxie/typescript-result");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
function renderCodeOrDefault(item, defaultText) {
    return item === undefined ? (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null, defaultText)) : (interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, item));
}
function renderTakedown(roomID, details) {
    return (
    // DO NOT render a room pill because it could show the avatar of the room,
    // which is not good.
    interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
            "Successfully takendown the room ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, roomID)),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
                "name: ",
                renderCodeOrDefault(details.name, "no name available")),
            interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
                "creator:",
                " ",
                renderCodeOrDefault(details.creator, "creator information unavailable")),
            interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
                "topic: ",
                renderCodeOrDefault(details.topic, "topic unavailable")))));
}
class StandardRoomTakedownCapabilityRenderer {
    constructor(description, messageCollector, capability) {
        this.description = description;
        this.messageCollector = messageCollector;
        this.capability = capability;
        this.requiredEventPermissions = this.capability.requiredEventPermissions;
        this.requiredPermissions = this.capability.requiredPermissions;
        this.requiredStatePermissions = this.capability.requiredStatePermissions;
        // nothing to do.
    }
    async isRoomTakendown(roomID) {
        // pass this one through without rendering because it doesn't really have a consequence
        return await this.capability.isRoomTakendown(roomID);
    }
    async takedownRoom(roomID) {
        const capabilityResult = await this.capability.takedownRoom(roomID);
        if ((0, typescript_result_1.isError)(capabilityResult)) {
            this.messageCollector.addOneliner(this.description, this.capability, (0, mps_interface_adaptor_1.renderFailedSingularConsequence)(this.description, interface_manager_1.DeadDocumentJSX.JSXFactory("span", null,
                "Failed to takedown room",
                " ",
                (0, mps_interface_adaptor_1.renderRoomPill)(matrix_basic_types_1.MatrixRoomReference.fromRoomID(roomID))), capabilityResult.error));
            return capabilityResult;
        }
        this.messageCollector.addOneliner(this.description, this.capability, renderTakedown(roomID, capabilityResult.ok));
        return capabilityResult;
    }
    async getRoomDetails(roomID) {
        return await this.capability.getRoomDetails(roomID);
    }
}
(0, matrix_protection_suite_1.describeCapabilityRenderer)({
    name: "StandardRoomTakedownCapabilityRenderer",
    description: "Renders the standard room takedown capability result",
    interface: "RoomTakedownCapability",
    isDefaultForInterface: true,
    factory(description, draupnir, capability) {
        return new StandardRoomTakedownCapabilityRenderer(description, draupnir.capabilityMessageRenderer, capability);
    },
});
//# sourceMappingURL=RoomTakedownCapabilityRenderer.js.map