"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2020 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirAliasRemoveCommand = exports.DraupnirAliasMoveCommand = exports.DraupnirAliasAddCommand = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const DraupnirCommandPrerequisites_1 = require("./DraupnirCommandPrerequisites");
const matrix_protection_suite_for_matrix_bot_sdk_1 = require("matrix-protection-suite-for-matrix-bot-sdk");
exports.DraupnirAliasAddCommand = (0, interface_manager_1.describeCommand)({
    summary: "Add a new alias to the room on Draupnir's homeserver",
    parameters: (0, interface_manager_1.tuple)({
        name: "alias",
        description: "The alias that will be created",
        acceptor: interface_manager_1.MatrixRoomAliasPresentationType,
    }, {
        name: "target room",
        description: "The room the alias should point to",
        acceptor: interface_manager_1.MatrixRoomReferencePresentationSchema,
    }),
    async executor(draupnir, _info, _keywords, _rest, aliasToCreate, targetRoom) {
        const resolvedTargetRoom = await draupnir.clientPlatform
            .toRoomResolver()
            .resolveRoom(targetRoom);
        if ((0, matrix_protection_suite_1.isError)(resolvedTargetRoom)) {
            return resolvedTargetRoom.elaborate("Unable to resolve the room for which the alias is meant to point to");
        }
        return await draupnir.client
            .createRoomAlias(aliasToCreate.toRoomIDOrAlias(), resolvedTargetRoom.ok.toRoomIDOrAlias())
            .then(() => (0, matrix_protection_suite_1.Ok)(undefined), (e) => (0, matrix_protection_suite_for_matrix_bot_sdk_1.resultifyBotSDKRequestError)(e));
    },
});
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirAliasAddCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
});
exports.DraupnirAliasMoveCommand = (0, interface_manager_1.describeCommand)({
    summary: "Move an alias from one room to another.",
    parameters: (0, interface_manager_1.tuple)({
        name: "alias",
        description: "The alias that should be moved.",
        acceptor: interface_manager_1.MatrixRoomAliasPresentationType,
    }, {
        name: "new room",
        description: "The room to move the alias to.",
        acceptor: interface_manager_1.MatrixRoomReferencePresentationSchema,
    }),
    async executor(draupnir, _info, _keywords, _rest, aliasToMove, newRoom) {
        const newRoomID = await draupnir.clientPlatform
            .toRoomResolver()
            .resolveRoom(newRoom);
        if ((0, matrix_protection_suite_1.isError)(newRoomID)) {
            return newRoomID;
        }
        await draupnir.client.deleteRoomAlias(aliasToMove.toRoomIDOrAlias());
        await draupnir.client.createRoomAlias(aliasToMove.toRoomIDOrAlias(), newRoomID.ok.toRoomIDOrAlias());
        return (0, matrix_protection_suite_1.Ok)(undefined);
    },
});
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirAliasMoveCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
});
exports.DraupnirAliasRemoveCommand = (0, interface_manager_1.describeCommand)({
    summary: "Remove an alias from a room.",
    parameters: (0, interface_manager_1.tuple)({
        name: "alias",
        description: "The alias that should be removed.",
        acceptor: interface_manager_1.MatrixRoomAliasPresentationType,
    }),
    async executor(draupnir, _info, _keywords, _rest, aliasToRemove) {
        await draupnir.client.deleteRoomAlias(aliasToRemove.toRoomIDOrAlias());
        return (0, matrix_protection_suite_1.Ok)(undefined);
    },
});
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirAliasRemoveCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
});
//# sourceMappingURL=AliasCommands.js.map