"use strict";
// Copyright 2022 - 2025 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirBanCommand = void 0;
exports.findPolicyRoomEditorFromRoomReference = findPolicyRoomEditorFromRoomReference;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const DraupnirCommandPrerequisites_1 = require("./DraupnirCommandPrerequisites");
const typescript_result_1 = require("@gnuxie/typescript-result");
async function findPolicyRoomEditorFromRoomReference(roomResolver, policyRoomManager, policyRoomReference) {
    const policyRoomID = await roomResolver.resolveRoom(policyRoomReference);
    if ((0, matrix_protection_suite_1.isError)(policyRoomID)) {
        return policyRoomID;
    }
    return await policyRoomManager.getPolicyRoomEditor(policyRoomID.ok);
}
exports.DraupnirBanCommand = (0, interface_manager_1.describeCommand)({
    summary: "Bans an entity from the policy list.",
    parameters: (0, interface_manager_1.tuple)({
        name: "entity",
        description: "The entity to ban. This can be a user ID, or server name. For rooms please see the takedown command",
        acceptor: (0, interface_manager_1.union)(interface_manager_1.MatrixUserIDPresentationType, interface_manager_1.MatrixRoomReferencePresentationSchema, interface_manager_1.StringPresentationType),
    }, {
        name: "list",
        acceptor: (0, interface_manager_1.union)(interface_manager_1.MatrixRoomReferencePresentationSchema, interface_manager_1.StringPresentationType),
        prompt: async function ({ policyRoomManager, clientUserID, }) {
            return (0, matrix_protection_suite_1.Ok)({
                suggestions: policyRoomManager
                    .getEditablePolicyRoomIDs(clientUserID, matrix_protection_suite_1.PolicyRuleType.User)
                    .map((room) => interface_manager_1.MatrixRoomIDPresentationType.wrap(room)),
            });
        },
    }),
    rest: {
        name: "reason",
        description: "The reason for the ban.",
        acceptor: interface_manager_1.StringPresentationType,
        prompt: async function ({ defaultReasons }) {
            return (0, matrix_protection_suite_1.Ok)({
                suggestions: defaultReasons.map((reason) => interface_manager_1.StringPresentationType.wrap(reason)),
            });
        },
    },
    async executor({ watchedPolicyRooms, policyRoomManager, roomResolver, }, _info, _keywords, reasonParts, entity, policyRoomDesignator) {
        const policyRoomReference = typeof policyRoomDesignator === "string"
            ? (0, matrix_protection_suite_1.Ok)(watchedPolicyRooms.findPolicyRoomFromShortcode(policyRoomDesignator)
                ?.room)
            : (0, matrix_protection_suite_1.Ok)(policyRoomDesignator);
        if ((0, matrix_protection_suite_1.isError)(policyRoomReference)) {
            return policyRoomReference;
        }
        if (policyRoomReference.ok === undefined) {
            return typescript_result_1.ResultError.Result(`Unable to find a policy room from the shortcode ${policyRoomDesignator.toString()}`);
        }
        const policyListEditorResult = await findPolicyRoomEditorFromRoomReference(roomResolver, policyRoomManager, policyRoomReference.ok);
        if ((0, matrix_protection_suite_1.isError)(policyListEditorResult)) {
            return policyListEditorResult;
        }
        const policyListEditor = policyListEditorResult.ok;
        const reason = reasonParts.join(" ");
        if (entity instanceof matrix_basic_types_1.MatrixUserID) {
            return await policyListEditor.banEntity(matrix_protection_suite_1.PolicyRuleType.User, entity.toString(), reason);
        }
        else if (typeof entity === "string") {
            return await policyListEditor.banEntity(matrix_protection_suite_1.PolicyRuleType.Server, entity, reason);
        }
        else {
            const resolvedRoomReference = await roomResolver.resolveRoom(entity);
            if ((0, matrix_protection_suite_1.isError)(resolvedRoomReference)) {
                return resolvedRoomReference;
            }
            return await policyListEditor.banEntity(matrix_protection_suite_1.PolicyRuleType.Room, resolvedRoomReference.ok.toRoomIDOrAlias(), reason);
        }
    },
});
DraupnirCommandPrerequisites_1.DraupnirContextToCommandContextTranslator.registerTranslation(exports.DraupnirBanCommand, function (draupnir) {
    return {
        policyRoomManager: draupnir.policyRoomManager,
        watchedPolicyRooms: draupnir.protectedRoomsSet.watchedPolicyRooms,
        defaultReasons: draupnir.config.commands.ban.defaultReasons,
        roomResolver: draupnir.clientPlatform.toRoomResolver(),
        clientUserID: draupnir.clientUserID,
    };
});
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirBanCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
});
//# sourceMappingURL=Ban.js.map