"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirProtectionsCapabilityCommand = void 0;
const typescript_result_1 = require("@gnuxie/typescript-result");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const DraupnirCommandPrerequisites_1 = require("./DraupnirCommandPrerequisites");
exports.DraupnirProtectionsCapabilityCommand = (0, interface_manager_1.describeCommand)({
    summary: "Change the active capability provider for a specific protection capability.",
    parameters: (0, interface_manager_1.tuple)({
        name: "protection name",
        acceptor: interface_manager_1.StringPresentationType,
    }, {
        name: "capability name",
        acceptor: interface_manager_1.StringPresentationType,
    }, {
        name: "capability provider name",
        acceptor: interface_manager_1.StringPresentationType,
    }),
    async executor(draupnir, _info, _keywords, _rest, protectionName, capabilityName, capabilityProviderName) {
        const protectionDescription = (0, matrix_protection_suite_1.findProtection)(protectionName);
        if (protectionDescription === undefined) {
            return typescript_result_1.ResultError.Result(`Cannot find a protection named ${protectionName}`);
        }
        const capabilityProvider = (0, matrix_protection_suite_1.findCapabilityProvider)(capabilityProviderName);
        if (capabilityProvider === undefined) {
            return typescript_result_1.ResultError.Result(`Cannot find a capability provider named ${capabilityProviderName}`);
        }
        const oldCapabilitySet = await draupnir.protectedRoomsSet.protections.getCapabilityProviderSet(protectionDescription);
        if ((0, typescript_result_1.isError)(oldCapabilitySet)) {
            return oldCapabilitySet.elaborate("Unable to get the current capability provider");
        }
        const oldCapabilityProvider = oldCapabilitySet.ok[capabilityName];
        if (oldCapabilityProvider === undefined) {
            return typescript_result_1.ResultError.Result(`Unable to find the current capability provider for ${capabilityName}`);
        }
        const changeResult = await draupnir.protectedRoomsSet.protections.changeCapabilityProvider(draupnir, draupnir.protectedRoomsSet, protectionDescription, capabilityName, capabilityProvider);
        if ((0, typescript_result_1.isError)(changeResult)) {
            return changeResult.elaborate("Unable to change the active capability provider");
        }
        return (0, typescript_result_1.Ok)({
            protectionDescription,
            oldCapabilityProvider: oldCapabilityProvider,
            newCapabilityProvider: capabilityProvider,
            capabilityName,
        });
    },
});
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirProtectionsCapabilityCommand, {
    JSXRenderer(commandResult) {
        if ((0, typescript_result_1.isError)(commandResult)) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        return (0, typescript_result_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
            "Changed the capability provider for",
            " ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, commandResult.ok.capabilityName),
            " from",
            " ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, commandResult.ok.oldCapabilityProvider.name),
            " to",
            " ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, commandResult.ok.newCapabilityProvider.name)));
    },
});
//# sourceMappingURL=ProtectionsCapabilitiesCommand.js.map