"use strict";
// Copyright 2022 - 2024 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from @the-draupnir-project/interface-manager
// https://github.com/the-draupnir-project/interface-manager
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardCommandTable = void 0;
const TextReader_1 = require("../TextReader");
const PresentationStream_1 = require("./PresentationStream");
class StandardCommandTable {
    constructor(name) {
        this.name = name;
        this.exportedCommands = new Set();
        this.flattenedCommands = new Map();
        this.commands = {
            designator: [],
            sourceTable: this,
        };
        this.translators = new Map();
        /** Imported tables are tables that "add commands" to this table. They are not sub commands. */
        this.importedTables = new Map();
    }
    /**
     * Can be used to render a help command with an index of all the commands.
     * @returns All of the commands in this table.
     */
    getAllCommands() {
        return [...this.flattenedCommands.values()];
    }
    /**
     * @returns Only the commands interned in this table, excludes imported commands.
     */
    getExportedCommands() {
        return [...this.exportedCommands.values()];
    }
    getImportedTables() {
        return [...this.importedTables.values()];
    }
    findAMatchingCommandEntry(stream) {
        const tableHelper = (startingTableEntry, argumentStream) => {
            var _a;
            const nextArgument = argumentStream.peekItem();
            if (nextArgument === undefined ||
                typeof nextArgument.object !== "string") {
                // Then they might be using something like "!mjolnir status"
                return startingTableEntry;
            }
            const entry = (_a = startingTableEntry.subCommands) === null || _a === void 0 ? void 0 : _a.get(nextArgument.object);
            if (!entry) {
                // The reason there's no match is because this is the command arguments, rather than subcommand notation.
                return startingTableEntry;
            }
            else {
                stream.readItem(); // dispose of the argument.
                return tableHelper(entry, argumentStream);
            }
        };
        return tableHelper(this.commands, stream);
    }
    findAMatchingCommand(stream) {
        const commandTableEntry = stream.savingPositionIf({
            body: (s) => this.findAMatchingCommandEntry(s),
            predicate: (command) => command === undefined,
        });
        if (commandTableEntry) {
            return commandTableEntry.currentCommand;
        }
        return undefined;
    }
    internCommandHelper(command, originalTable, currentTableEntry, originalDesignator, currentDesignator) {
        var _a;
        const currentDesignatorPart = currentDesignator.shift();
        if (currentDesignatorPart === undefined) {
            if (currentTableEntry.currentCommand) {
                throw new TypeError(`There is already a command for ${JSON.stringify(originalDesignator)}`);
            }
            currentTableEntry.currentCommand = command;
            if (originalTable === this) {
                this.exportedCommands.add(currentTableEntry);
            }
            this.flattenedCommands.set(command, currentTableEntry);
        }
        else {
            if (currentTableEntry.subCommands === undefined) {
                currentTableEntry.subCommands = new Map();
            }
            const nextLookupEntry = (_a = currentTableEntry.subCommands.get(currentDesignatorPart)) !== null && _a !== void 0 ? _a : ((lookup) => (currentTableEntry.subCommands.set(currentDesignatorPart, lookup),
                lookup))({ designator: originalDesignator, sourceTable: this });
            this.internCommandHelper(command, originalTable, nextLookupEntry, originalDesignator, currentDesignator);
        }
    }
    internCommand(command, designator) {
        this.internCommandHelper(command, this, this.commands, designator, [...designator] // this array gets mutated.
        );
        return this;
    }
    importTable(importedTable, baseDesignator) {
        for (const commandTableEntry of importedTable.getAllCommands()) {
            if (this.findAMatchingCommand(new PresentationStream_1.StandardPresentationArgumentStream(commandTableEntry.designator.map((d) => TextReader_1.StringPresentationType.wrap(d))))) {
                throw new TypeError(`Command ${JSON.stringify(commandTableEntry.designator)} is in conflict with this table and cannot be imported.`);
            }
        }
        this.importedTables.set(importedTable, {
            table: importedTable,
            baseDesignator,
        });
        for (const command of importedTable.getAllCommands()) {
            const designator = [...baseDesignator, ...command.designator];
            this.internCommandHelper(command.currentCommand, importedTable, this.commands, designator, [...designator] // this array gets mutated.
            );
        }
    }
    isContainingCommand(command) {
        return this.flattenedCommands.has(command);
    }
    keyFromTranslator(translator) {
        return translator.toType.name + "From" + translator.fromType.name; // cheap but no one will ever find out.
    }
    internPresentationTypeTranslator(translator) {
        const key = this.keyFromTranslator(translator);
        if (this.translators.has(key)) {
            throw new TypeError(`There is already a translator from ${translator.toType.name} to ${translator.fromType.name}`);
        }
        this.translators.set(key, translator);
        return this;
    }
    findPresentationTypeTranslator(toType, fromType) {
        return this.translators.get(toType.name + "From" + fromType.name);
    }
}
exports.StandardCommandTable = StandardCommandTable;
//# sourceMappingURL=CommandTable.js.map