"use strict";
// Copyright 2022, 2024 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from @the-draupnir-project/interface-manager
// https://github.com/the-draupnir-project/interface-manager
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardRestDescription = void 0;
exports.describeRestParameters = describeRestParameters;
const typescript_result_1 = require("@gnuxie/typescript-result");
const ParseErrors_1 = require("./ParseErrors");
const TextPresentationRenderer_1 = require("../TextReader/TextPresentationRenderer");
const PresentationSchema_1 = require("./PresentationSchema");
/**
 * Describes a rest parameter for a command.
 * This consumes any arguments left over in the call to a command
 * into an array and ensures that each can be accepted by the `acceptor`.
 *
 * Any keywords in the rest of the command will be given to the `keywordParser`.
 */
class StandardRestDescription {
    constructor(name, 
    /** The presentation type of each item. */
    acceptor, prompt, description) {
        this.name = name;
        this.prompt = prompt;
        this.description = description;
        if ("schemaType" in acceptor) {
            this.acceptor = acceptor;
        }
        else {
            this.acceptor = {
                schemaType: PresentationSchema_1.PresentationSchemaType.Single,
                presentationType: acceptor,
            };
        }
    }
    /**
     * Parse the rest of a command.
     * @param stream An argument stream that starts at the rest of a command.
     * @param keywordParser Used to store any keywords found in the rest of the command.
     * @returns A ActionResult of ReadItems associated with the rest of the command.
     * If a ReadItem or Keyword is invalid for the command, then an error will be returned.
     */
    parseRest(partialCommand, keywordParser) {
        const stream = partialCommand.stream;
        const items = [];
        if (this.prompt && stream.peekItem(undefined) === undefined) {
            return ParseErrors_1.PromptRequiredError.Result(`A prompt is required for the missing argument for the ${this.name} parameter`, {
                promptParameter: this,
                partialCommand,
            });
        }
        while (stream.peekItem(undefined) !== undefined) {
            const keywordResult = keywordParser.parseKeywords(partialCommand);
            if ((0, typescript_result_1.isError)(keywordResult)) {
                return keywordResult;
            }
            const nextItem = stream.peekItem(undefined);
            if (nextItem !== undefined) {
                const acceptedPresentation = (0, PresentationSchema_1.acceptPresentation)(this.acceptor, partialCommand.commandTable, nextItem);
                if (acceptedPresentation === undefined) {
                    return ParseErrors_1.ArgumentParseError.Result(`Was expecting a match for the presentation type: ${(0, PresentationSchema_1.printPresentationSchema)(this.acceptor)} but got ${TextPresentationRenderer_1.TextPresentationRenderer.render(nextItem)}.`, {
                        parameter: this,
                        partialCommand,
                    });
                }
                items.push(acceptedPresentation);
                stream.readItem(); // dispose of keyword's associated value from the stream.
            }
        }
        return (0, typescript_result_1.Ok)(items);
    }
}
exports.StandardRestDescription = StandardRestDescription;
function describeRestParameters(options) {
    return new StandardRestDescription(options.name, options.acceptor, options.prompt, options.description);
}
//# sourceMappingURL=RestParameterDescription.js.map