"use strict";
// Copyright 2023 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderMatrix = renderMatrix;
const typescript_result_1 = require("@gnuxie/typescript-result");
const DeadDocument_1 = require("./DeadDocument");
const PagedDuplexStream_1 = require("./PagedDuplexStream");
const DeadDocumentHTML_1 = require("./DeadDocumentHTML");
const DeadDocumentMarkdown_1 = require("./DeadDocumentMarkdown");
function checkEqual(node1, node2) {
    if (!Object.is(node1, node2)) {
        throw new TypeError("There is an implementation bug in one of the walker");
    }
    return true;
}
/**
 * Render the `DocumentNode` to Matrix (in both HTML + Markdown) using the
 * callback provided to send each event. Should serialized content span
 * more than one event, then the callback will be called for each event.
 * @param node A document node to render to Matrix.
 * @param cb A callback that will send the text+html for a single event
 * to a Matrix room.
 */
async function renderMatrix(node, cb) {
    const commitHook = (commitNode, context) => {
        context.output.commit(commitNode);
    };
    if (node.tag !== DeadDocument_1.NodeTag.Root) {
        // rendering has to start (and end) with a committable node.
        throw new TypeError("Tried to render a node without a root, this will not be committable");
    }
    const markdownOutput = new PagedDuplexStream_1.PagedDuplexStream();
    const markdownWalker = new DeadDocument_1.FringeWalker(node, { output: markdownOutput }, DeadDocumentMarkdown_1.MARKDOWN_RENDERER, commitHook);
    const htmlOutput = new PagedDuplexStream_1.PagedDuplexStream();
    const htmlWalker = new DeadDocument_1.FringeWalker(node, { output: htmlOutput }, DeadDocumentHTML_1.HTML_RENDERER, commitHook);
    const eventIds = [];
    const outputs = [htmlOutput, markdownOutput];
    let currentMarkdownNode = markdownWalker.increment();
    let currentHtmlNode = htmlWalker.increment();
    checkEqual(currentHtmlNode, currentMarkdownNode);
    while (currentHtmlNode !== undefined) {
        if (outputs.some((o) => o.peekPage())) {
            // Make sure that any outputs that have buffered input start a fresh page,
            // so that the same committed nodes end up in the same message.
            outputs
                .filter((o) => !o.peekPage())
                .forEach((o) => {
                o.ensureNewPage();
            });
            // Send the new pages as an event.
            const [nextMakrdownPage, nextHtmlPage] = [
                markdownOutput.readPage(),
                htmlOutput.readPage(),
            ];
            if (nextMakrdownPage === undefined || nextHtmlPage === undefined) {
                throw new TypeError(`The code is wrong!!`);
            }
            const sendResult = await cb(nextMakrdownPage, nextHtmlPage);
            if ((0, typescript_result_1.isError)(sendResult)) {
                return sendResult;
            }
            eventIds.push(sendResult.ok);
        }
        // prepare next iteration
        currentMarkdownNode = markdownWalker.increment();
        currentHtmlNode = htmlWalker.increment();
        checkEqual(currentHtmlNode, currentMarkdownNode);
    }
    outputs.forEach((o) => {
        o.ensureNewPage();
    });
    if (outputs.some((o) => o.peekPage())) {
        const [nextMakrdownPage, nextHtmlPage] = [
            markdownOutput.readPage(),
            htmlOutput.readPage(),
        ];
        if (nextMakrdownPage === undefined || nextHtmlPage === undefined) {
            throw new TypeError(`The code is wrong!!`);
        }
        const sendResult = await cb(nextMakrdownPage, nextHtmlPage);
        if ((0, typescript_result_1.isError)(sendResult)) {
            return sendResult;
        }
        eventIds.push(sendResult.ok);
    }
    return (0, typescript_result_1.Ok)(eventIds);
}
//# sourceMappingURL=DeadDocumentMatrix.js.map