"use strict";
// SPDX-FileCopyrightText: 2022 - 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from @the-draupnir-project/interface-manager
// https://github.com/the-draupnir-project/interface-manager
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const TextCommandReader_1 = require("./TextCommandReader");
const TextPresentationTypes_1 = require("./TextPresentationTypes");
const Keyword_1 = require("../Command/Keyword");
test("CommandReader can read strings", function () {
    var _a, _b;
    const readResult = (0, TextCommandReader_1.readCommand)("hello");
    expect(readResult.length).toBe(1);
    expect(readResult.at(0)).toBeDefined();
    expect((_a = readResult.at(0)) === null || _a === void 0 ? void 0 : _a.object).toBe("hello");
    expect((_b = readResult.at(0)) === null || _b === void 0 ? void 0 : _b.presentationType).toBe(TextPresentationTypes_1.StringPresentationType);
});
test("CommandReader can read a complex command", function () {
    var _a, _b, _c;
    const readResult = (0, TextCommandReader_1.readCommand)("!draupnir ban @spam:example.com https://matrix.to/#/#coc:example.com spam --some-flag-idk");
    expect((_a = readResult.at(2)) === null || _a === void 0 ? void 0 : _a.object).toBeInstanceOf(matrix_basic_types_1.MatrixUserID);
    expect((_b = readResult.at(3)) === null || _b === void 0 ? void 0 : _b.object).toBeInstanceOf(matrix_basic_types_1.MatrixRoomAlias);
    expect((_c = readResult.at(5)) === null || _c === void 0 ? void 0 : _c.object).toBeInstanceOf(Keyword_1.Keyword);
});
it("Can read a simple command with only strings", function () {
    const command = "!mjolnir list rooms";
    const readItems = (0, TextCommandReader_1.readCommand)(command);
    expect(readItems.every((item) => command.includes(item.object))).toBe(true);
});
it("Can turn room aliases to room references", function () {
    var _a, _b;
    const command = "#meow:example.org";
    const readItems = (0, TextCommandReader_1.readCommand)(command);
    expect((_a = readItems.at(0)) === null || _a === void 0 ? void 0 : _a.object).toBeInstanceOf(matrix_basic_types_1.MatrixRoomAlias);
    const roomReference = (_b = readItems.at(0)) === null || _b === void 0 ? void 0 : _b.object;
    expect(roomReference.toRoomIDOrAlias()).toBe(command);
});
it("Can turn room ids to room references", function () {
    var _a, _b;
    const command = "!foijoiejfoij:example.org";
    const readItems = (0, TextCommandReader_1.readCommand)(command);
    expect((_a = readItems.at(0)) === null || _a === void 0 ? void 0 : _a.object).toBeInstanceOf(matrix_basic_types_1.MatrixRoomID);
    const roomReference = (_b = readItems.at(0)) === null || _b === void 0 ? void 0 : _b.object;
    expect(roomReference.toRoomIDOrAlias()).toBe(command);
});
it("Can read keywords and correctly parse their designators", function () {
    const checkKeyword = (designator, keyword) => {
        var _a, _b;
        const readItems = (0, TextCommandReader_1.readCommand)(keyword);
        expect((_a = readItems.at(0)) === null || _a === void 0 ? void 0 : _a.object).toBeInstanceOf(Keyword_1.Keyword);
        const keywordItem = (_b = readItems.at(0)) === null || _b === void 0 ? void 0 : _b.object;
        expect(keywordItem.designator).toBe(designator);
    };
    checkKeyword("foo", "--foo");
    checkKeyword("foo", "-foo");
    checkKeyword("f", "-f");
    checkKeyword("foo", ":foo");
    checkKeyword("f", ":f");
});
it("Check that malformed room ids and aliases are read as strings", function () {
    // We leave it for the command to validate the arguments it receives intentionally.
    // From the perspective of the user, their command will fail just as early but with more context this way.
    const checkMalformedRoomReference = (badReference) => {
        var _a;
        expect((_a = (0, TextCommandReader_1.readCommand)(badReference).at(0)) === null || _a === void 0 ? void 0 : _a.object).toBe(badReference);
    };
    checkMalformedRoomReference("#singasongaboutlife");
    checkMalformedRoomReference("!mjolnir");
});
it("Can parse userID's", function () {
    var _a, _b;
    const command = "@spam:example.com";
    const readItems = (0, TextCommandReader_1.readCommand)(command);
    expect((_a = readItems.at(0)) === null || _a === void 0 ? void 0 : _a.object).toBeInstanceOf(matrix_basic_types_1.MatrixUserID);
    const user = (_b = readItems.at(0)) === null || _b === void 0 ? void 0 : _b.object;
    expect(user.localpart).toBe("spam");
});
it("It can read numbers", function () {
    var _a;
    const command = "123";
    const readItems = (0, TextCommandReader_1.readCommand)(command);
    expect((_a = readItems.at(0)) === null || _a === void 0 ? void 0 : _a.object).toBe(123);
});
it("It can read booleans", function () {
    var _a, _b, _c;
    const command = "true false";
    const readItems = (0, TextCommandReader_1.readCommand)(command);
    expect((_a = readItems.at(0)) === null || _a === void 0 ? void 0 : _a.presentationType).toBe(TextPresentationTypes_1.BooleanPresentationType);
    expect((_b = readItems.at(0)) === null || _b === void 0 ? void 0 : _b.object).toBe(true);
    expect((_c = readItems.at(1)) === null || _c === void 0 ? void 0 : _c.object).toBe(false);
});
it("It can read negative integers", function () {
    var _a;
    const command = "-123";
    const readItems = (0, TextCommandReader_1.readCommand)(command);
    expect((_a = readItems.at(0)) === null || _a === void 0 ? void 0 : _a.object).toBe(-123);
});
it("It can read quoted strings", function () {
    var _a;
    const command = '"hello world"';
    const readItems = (0, TextCommandReader_1.readCommand)(command);
    expect((_a = readItems.at(0)) === null || _a === void 0 ? void 0 : _a.object).toBe("hello world");
});
it("It can read unbalanced quoteds strings", function () {
    var _a, _b;
    const command = '"unbalanced vronut';
    const readItems = (0, TextCommandReader_1.readCommand)(command);
    expect((_a = readItems.at(0)) === null || _a === void 0 ? void 0 : _a.object).toBe('"unbalanced');
    expect((_b = readItems.at(1)) === null || _b === void 0 ? void 0 : _b.object).toBe("vronut");
});
it("It can read escaped quotes meow", function () {
    var _a;
    const commoand = '"\\"hello \\"';
    const readItems = (0, TextCommandReader_1.readCommand)(commoand);
    expect((_a = readItems.at(0)) === null || _a === void 0 ? void 0 : _a.object).toBe('"hello "');
});
it("It can stop quoted stuff from hitting the post read transforms", function () {
    var _a, _b;
    const command = '"true"';
    const readItems = (0, TextCommandReader_1.readCommand)(command);
    expect((_a = readItems.at(0)) === null || _a === void 0 ? void 0 : _a.object).toBe("true");
    expect((_b = readItems.at(0)) === null || _b === void 0 ? void 0 : _b.presentationType).toBe(TextPresentationTypes_1.StringPresentationType);
});
it("Can read V12 roomID's", function () {
    var _a;
    const command = "!arfarfarfarfarfarfarfarfarfarfarfarfarfarfa";
    const readItems = (0, TextCommandReader_1.readCommand)(command);
    expect((_a = readItems.at(0)) === null || _a === void 0 ? void 0 : _a.presentationType).toBe(TextPresentationTypes_1.MatrixRoomIDPresentationType);
});
//# sourceMappingURL=TextCommandReader.test.js.map