import { Result } from "@gnuxie/typescript-result";
import { StringEventID, StringRoomAlias, StringRoomID } from "../StringlyTypedMatrix";
import { StringServerName } from "../StringlyTypedMatrix/StringServerName";
export type MatrixRoomReference = MatrixRoomID | MatrixRoomAlias;
export declare namespace MatrixRoomReference {
    function fromAlias(alias: StringRoomAlias): MatrixRoomReference;
    function fromRoomID(roomId: StringRoomID, viaServers?: string[]): MatrixRoomID;
    /**
     * Create a `MatrixRoomReference` from a room ID or a room alias.
     * @param roomIDOrAlias The room ID or the room alias.
     * @param viaServers If a room ID is being provided, then these server names
     * can be used to find the room.
     */
    function fromRoomIDOrAlias(roomIDOrAlias: StringRoomID | StringRoomAlias, viaServers?: string[]): MatrixRoomReference;
    function fromPermalink(link: string): Result<MatrixRoomReference>;
    /**
     * Try parse a roomID, roomAlias or a permalink.
     */
    function fromString(string: string): Result<MatrixRoomReference>;
}
/**
 * This is a universal reference for a matrix room.
 * This is really useful because there are at least 3 ways of referring to a Matrix room,
 * and some of them require extra steps to be useful in certain contexts (aliases, permalinks).
 */
declare abstract class AbstractMatrixRoomReference {
    protected readonly reference: StringRoomID | StringRoomAlias;
    private readonly viaServers;
    protected constructor(reference: StringRoomID | StringRoomAlias, viaServers?: string[]);
    toPermalink(): string;
    /**
     * We don't include a `toRoomId` that uses `forceResolveAlias` as this would erase `viaServers`,
     * which will be necessary to use if our homeserver hasn't joined the room yet.
     * @returns A string representing a room id or alias.
     */
    toRoomIDOrAlias(): StringRoomID | StringRoomAlias;
    getViaServers(): string[];
    toString(): string;
}
/**
 * A concrete `MatrixRoomReference` that represents only a room ID.
 * @see {@link MatrixRoomReference}.
 */
export declare class MatrixRoomID extends AbstractMatrixRoomReference {
    constructor(reference: string, viaServers?: string[]);
    toRoomIDOrAlias(): StringRoomID;
}
/**
 * A concrete `MatrixRoomReference` the represents only a room alias.
 * @see {@link MatrixRoomReference}.
 */
export declare class MatrixRoomAlias extends AbstractMatrixRoomReference {
    constructor(reference: string, viaServers?: string[]);
    toRoomIDOrAlias(): StringRoomAlias;
    get serverName(): StringServerName;
}
export type MatrixEventReference = MatrixEventViaRoomID | MatrixEventViaAlias;
export declare namespace MatrixEventReference {
    function fromPermalink(link: string): Result<MatrixEventReference>;
}
export declare class MatrixEventViaRoomID {
    readonly room: MatrixRoomID;
    readonly eventID: StringEventID;
    constructor(room: MatrixRoomID, eventID: StringEventID);
    get reference(): MatrixRoomID;
    toPermalink(): string;
}
export declare class MatrixEventViaAlias {
    readonly alias: MatrixRoomAlias;
    readonly eventID: StringEventID;
    constructor(alias: MatrixRoomAlias, eventID: StringEventID);
    get reference(): MatrixRoomAlias;
    toPermalink(): string;
}
export {};
//# sourceMappingURL=MatrixRoomReference.d.ts.map