"use strict";
// SPDX-FileCopyrightText: 2022 - 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from matrix-basic-types
// https://github.com/the-draupnir-project/matrix-basic-types
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.MatrixEventViaAlias = exports.MatrixEventViaRoomID = exports.MatrixEventReference = exports.MatrixRoomAlias = exports.MatrixRoomID = exports.MatrixRoomReference = void 0;
const typescript_result_1 = require("@gnuxie/typescript-result");
const StringlyTypedMatrix_1 = require("../StringlyTypedMatrix");
const Permalinks_1 = require("./Permalinks");
/**
 * Some servers can return a huge list of via servers for a room which can
 * cause some pretty serious problems for message rendering.
 */
function limitViaServers(viaServers) {
    if (viaServers.length > 5) {
        return viaServers.slice(0, 5);
    }
    else {
        return viaServers;
    }
}
// we disable this warning because it's not relevant, we're not making a module
// we're trying to add generic functions to a type.
// Comes at a cost that anyone actually using this from JS and not TS is
// going to be confused.
// eslint-disable-next-line @typescript-eslint/no-namespace
var MatrixRoomReference;
(function (MatrixRoomReference) {
    function fromAlias(alias) {
        return new MatrixRoomAlias(alias);
    }
    MatrixRoomReference.fromAlias = fromAlias;
    function fromRoomID(roomId, viaServers = []) {
        return new MatrixRoomID(roomId, viaServers);
    }
    MatrixRoomReference.fromRoomID = fromRoomID;
    /**
     * Create a `MatrixRoomReference` from a room ID or a room alias.
     * @param roomIDOrAlias The room ID or the room alias.
     * @param viaServers If a room ID is being provided, then these server names
     * can be used to find the room.
     */
    function fromRoomIDOrAlias(roomIDOrAlias, viaServers = []) {
        if (roomIDOrAlias.startsWith("!")) {
            return new MatrixRoomID(roomIDOrAlias, viaServers);
        }
        else {
            return new MatrixRoomAlias(roomIDOrAlias, viaServers);
        }
    }
    MatrixRoomReference.fromRoomIDOrAlias = fromRoomIDOrAlias;
    function fromPermalink(link) {
        const partsResult = Permalinks_1.Permalinks.parseUrl(link);
        if ((0, typescript_result_1.isError)(partsResult)) {
            return partsResult;
        }
        const parts = partsResult.ok;
        if (parts.roomID !== undefined) {
            return (0, typescript_result_1.Ok)(new MatrixRoomID(parts.roomID, parts.viaServers));
        }
        else if (parts.roomAlias !== undefined) {
            return (0, typescript_result_1.Ok)(new MatrixRoomAlias(parts.roomAlias, parts.viaServers));
        }
        else {
            return typescript_result_1.ResultError.Result(`There isn't a reference to a room in the URL: ${link}`);
        }
    }
    MatrixRoomReference.fromPermalink = fromPermalink;
    /**
     * Try parse a roomID, roomAlias or a permalink.
     */
    function fromString(string) {
        if ((0, StringlyTypedMatrix_1.isStringRoomID)(string)) {
            return (0, typescript_result_1.Ok)(MatrixRoomReference.fromRoomID(string));
        }
        else if ((0, StringlyTypedMatrix_1.isStringRoomAlias)(string)) {
            return (0, typescript_result_1.Ok)(MatrixRoomReference.fromRoomIDOrAlias(string));
        }
        else {
            return MatrixRoomReference.fromPermalink(string);
        }
    }
    MatrixRoomReference.fromString = fromString;
})(MatrixRoomReference || (exports.MatrixRoomReference = MatrixRoomReference = {}));
/**
 * This is a universal reference for a matrix room.
 * This is really useful because there are at least 3 ways of referring to a Matrix room,
 * and some of them require extra steps to be useful in certain contexts (aliases, permalinks).
 */
class AbstractMatrixRoomReference {
    constructor(reference, viaServers = []) {
        this.reference = reference;
        this.viaServers = limitViaServers(viaServers);
    }
    toPermalink() {
        return Permalinks_1.Permalinks.forRoom(this.reference, this.viaServers);
    }
    /**
     * We don't include a `toRoomId` that uses `forceResolveAlias` as this would erase `viaServers`,
     * which will be necessary to use if our homeserver hasn't joined the room yet.
     * @returns A string representing a room id or alias.
     */
    toRoomIDOrAlias() {
        return this.reference;
    }
    getViaServers() {
        // don't want them mutating the viaServers in this reference.
        return [...this.viaServers];
    }
    toString() {
        return this.toPermalink();
    }
}
/**
 * A concrete `MatrixRoomReference` that represents only a room ID.
 * @see {@link MatrixRoomReference}.
 */
class MatrixRoomID extends AbstractMatrixRoomReference {
    constructor(reference, viaServers = []) {
        if (!(0, StringlyTypedMatrix_1.isStringRoomID)(reference)) {
            throw new TypeError(`invalid reference for roomID ${reference}`);
        }
        super(reference, viaServers);
    }
    toRoomIDOrAlias() {
        return this.reference;
    }
}
exports.MatrixRoomID = MatrixRoomID;
/**
 * A concrete `MatrixRoomReference` the represents only a room alias.
 * @see {@link MatrixRoomReference}.
 */
class MatrixRoomAlias extends AbstractMatrixRoomReference {
    constructor(reference, viaServers = []) {
        if (!(0, StringlyTypedMatrix_1.isStringRoomAlias)(reference)) {
            throw new TypeError(`invalid reference for RoomAlias ${reference}`);
        }
        super(reference, viaServers);
    }
    toRoomIDOrAlias() {
        return this.reference;
    }
    get serverName() {
        return (0, StringlyTypedMatrix_1.roomAliasServerName)(this.reference);
    }
}
exports.MatrixRoomAlias = MatrixRoomAlias;
// we disable this warning because it's not relevant, we're not making a module
// we're trying to add generic functions to a type.
// Comes at a cost that anyone actually using this from JS and not TS is
// going to be confused.
// eslint-disable-next-line @typescript-eslint/no-namespace
var MatrixEventReference;
(function (MatrixEventReference) {
    function fromPermalink(link) {
        const partsResult = Permalinks_1.Permalinks.parseUrl(link);
        if ((0, typescript_result_1.isError)(partsResult)) {
            return partsResult;
        }
        const parts = partsResult.ok;
        if (parts.roomID !== undefined && parts.eventID !== undefined) {
            return (0, typescript_result_1.Ok)(new MatrixEventViaRoomID(new MatrixRoomID(parts.roomID, parts.viaServers), parts.eventID));
        }
        else if (parts.roomAlias !== undefined && parts.eventID !== undefined) {
            return (0, typescript_result_1.Ok)(new MatrixEventViaAlias(new MatrixRoomAlias(parts.roomAlias, parts.viaServers), parts.eventID));
        }
        else {
            return typescript_result_1.ResultError.Result(`There isn't a reference to an event in the URL: ${link}`);
        }
    }
    MatrixEventReference.fromPermalink = fromPermalink;
})(MatrixEventReference || (exports.MatrixEventReference = MatrixEventReference = {}));
class MatrixEventViaRoomID {
    constructor(room, eventID) {
        this.room = room;
        this.eventID = eventID;
        // nothing to do.
    }
    get reference() {
        return this.room;
    }
    toPermalink() {
        return `${this.room.toPermalink()}/${encodeURIComponent(this.eventID)}`;
    }
}
exports.MatrixEventViaRoomID = MatrixEventViaRoomID;
class MatrixEventViaAlias {
    constructor(alias, eventID) {
        this.alias = alias;
        this.eventID = eventID;
        // nothing to do.
    }
    get reference() {
        return this.alias;
    }
    toPermalink() {
        return `${this.alias.toPermalink()}/${encodeURIComponent(this.eventID)}`;
    }
}
exports.MatrixEventViaAlias = MatrixEventViaAlias;
//# sourceMappingURL=MatrixRoomReference.js.map