"use strict";
// SPDX-FileCopyrightText: 2023 Gnuxie <Gnuxie@protonmail.com>
// SPDX-FileCopyrightText: 2018 - 2022 Travis Ralston
//
// SPDX-License-Identifier: MIT
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from matrix-bot-sdk
// https://github.com/turt2live/matrix-bot-sdk
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.Permalinks = exports.MatrixToRegex = void 0;
const typescript_result_1 = require("@gnuxie/typescript-result");
const StringlyTypedMatrix_1 = require("../StringlyTypedMatrix");
exports.MatrixToRegex = /^https:\/\/matrix\.to\/#\/(?<entity>[^/?]+)\/?(?<eventId>[^?]+)?(?<query>\?[^]*)?$/;
/**
 * Functions for handling permalinks
 * @category Utilities
 */
class Permalinks {
    constructor() {
        // nothing to do.
    }
    static encodeViaArgs(servers) {
        if (servers.length === 0)
            return "";
        return `?via=${servers.join("&via=")}`;
    }
    /**
     * Creates a room permalink.
     * @param {string} roomIDOrAlias The room ID or alias to create a permalink for.
     * @param {string[]} viaServers The servers to route the permalink through.
     * @returns {string} A room permalink.
     */
    static forRoom(roomIDOrAlias, viaServers = []) {
        return `https://matrix.to/#/${encodeURIComponent(roomIDOrAlias)}${Permalinks.encodeViaArgs(viaServers)}`;
    }
    /**
     * Creates a user permalink.
     * @param {string} userID The user ID to create a permalink for.
     * @returns {string} A user permalink.
     */
    static forUser(userID) {
        return `https://matrix.to/#/${encodeURIComponent(userID)}`;
    }
    /**
     * Creates an event permalink.
     * @param {string} roomIDOrAlias The room ID or alias to create a permalink in.
     * @param {string} eventID The event ID to reference in the permalink.
     * @param {string[]} viaServers The servers to route the permalink through.
     * @returns {string} An event permalink.
     */
    static forEvent(roomIDOrAlias, eventID, viaServers = []) {
        return `https://matrix.to/#/${encodeURIComponent(roomIDOrAlias)}/${encodeURIComponent(eventID)}${Permalinks.encodeViaArgs(viaServers)}`;
    }
    /**
     * Parses a permalink URL into usable parts.
     * @param {string} matrixTo The matrix.to URL to parse.
     * @returns {PermalinkParts} The parts of the permalink.
     */
    static parseUrl(matrixTo) {
        var _a;
        const url = (_a = exports.MatrixToRegex.exec(matrixTo)) === null || _a === void 0 ? void 0 : _a.groups;
        if (!url) {
            return typescript_result_1.ResultError.Result(`Not a valid matrix.to URL: ${matrixTo}`);
        }
        const viaServers = new URLSearchParams(url.query).getAll("via");
        const eventID = url.eventId && decodeURIComponent(url.eventId);
        if (eventID !== undefined && !(0, StringlyTypedMatrix_1.isStringEventID)(eventID)) {
            return typescript_result_1.ResultError.Result(`Invalid EventID in matrix.to URL ${eventID}`);
        }
        if (url.entity === undefined) {
            return typescript_result_1.ResultError.Result(`Invalid Entity in matrix.to URL ${url.entity}`);
        }
        const entity = decodeURIComponent(url.entity);
        if (entity[0] === "@") {
            if (!(0, StringlyTypedMatrix_1.isStringUserID)(entity)) {
                return typescript_result_1.ResultError.Result(`Invalid User ID in matrix.to URL: ${entity}`);
            }
            return (0, typescript_result_1.Ok)({
                userID: entity,
                viaServers: [],
            });
        }
        else if (entity[0] === "!") {
            if (!(0, StringlyTypedMatrix_1.isStringRoomID)(entity)) {
                return typescript_result_1.ResultError.Result(`Invalid RoomID in matrix.to URL ${entity}`);
            }
            return (0, typescript_result_1.Ok)({
                roomID: entity,
                ...(eventID === undefined ? {} : { eventID }),
                viaServers,
            });
        }
        else if (entity[0] === "#") {
            if (!(0, StringlyTypedMatrix_1.isStringRoomAlias)(entity)) {
                return typescript_result_1.ResultError.Result(`Invalid RoomAlias in matrix.to URL ${entity}`);
            }
            return (0, typescript_result_1.Ok)({
                roomAlias: entity,
                ...(eventID === undefined ? {} : { eventID }),
                viaServers,
            });
        }
        else {
            return typescript_result_1.ResultError.Result(`Unexpected entity in matrix.to URL ${entity}`);
        }
    }
}
exports.Permalinks = Permalinks;
//# sourceMappingURL=Permalinks.js.map