"use strict";
// Copyright 2023 - 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from Draupnir
// https://github.com/the-draupnir-project/Draupnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.MPSMatrixInterfaceAdaptor = exports.MPSContextToCommandContextTranslator = exports.invocationInformationFromMatrixEventcontext = exports.MPSMatrixInterfaceAdaptorCallbacks = exports.rendererFailedCB = exports.MPSCommandDispatcherCallbacks = exports.matrixEventsFromDeadDocument = void 0;
exports.sendMatrixEventsFromDeadDocument = sendMatrixEventsFromDeadDocument;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const MatrixHelpRenderer_1 = require("./MatrixHelpRenderer");
const MatrixPromptForAccept_1 = require("./MatrixPromptForAccept");
const MatrixPromptForConfirmation_1 = require("./MatrixPromptForConfirmation");
async function sendMatrixEventsFromDeadDocument(messageSender, roomID, document, { replyToEvent, additionalContent = {}, }) {
    const baseContent = (text, html) => {
        return {
            msgtype: "m.notice",
            body: text,
            format: "org.matrix.custom.html",
            formatted_body: html,
        };
    };
    const renderInitialReply = async (text, html) => {
        return await messageSender.sendMessage(roomID, {
            ...baseContent(text, html),
            ...additionalContent,
            ...(replyToEvent === undefined
                ? {} // if they don't supply a reply just send a top level event.
                : {
                    "m.relates_to": {
                        "m.in_reply_to": {
                            event_id: replyToEvent["event_id"],
                        },
                    },
                }),
        });
    };
    const renderThreadReply = async (eventId, text, html) => {
        return await messageSender.sendMessage(roomID, {
            ...baseContent(text, html),
            "m.relates_to": {
                rel_type: "m.thread",
                event_id: eventId,
            },
        });
    };
    let initialReplyEventID = undefined;
    return await (0, interface_manager_1.renderMatrix)(document, async (text, html) => {
        if (initialReplyEventID === undefined) {
            const replyResult = await renderInitialReply(text, html);
            if ((0, matrix_protection_suite_1.isError)(replyResult)) {
                return replyResult;
            }
            else {
                initialReplyEventID = replyResult.ok;
                return replyResult;
            }
        }
        else {
            return await renderThreadReply(initialReplyEventID, text, html);
        }
    });
}
const matrixEventsFromDeadDocument = async function matrixEventsFromDeadDocument({ clientPlatform }, { event }, document) {
    const sendResult = await sendMatrixEventsFromDeadDocument(clientPlatform.toRoomMessageSender(), event.room_id, document, { replyToEvent: event });
    if ((0, matrix_protection_suite_1.isError)(sendResult)) {
        return sendResult;
    }
    else {
        return (0, matrix_protection_suite_1.Ok)(undefined);
    }
};
exports.matrixEventsFromDeadDocument = matrixEventsFromDeadDocument;
const log = new matrix_protection_suite_1.Logger("MPSInterfaceAdaptor");
exports.MPSCommandDispatcherCallbacks = {
    commandFailedCB(_info, command, error) {
        log.error(`The command "${command.designator.join(" ")}" returned with an error:`, error);
    },
    commandUncaughtErrorCB(info, body, error) {
        log.error(`Caught an unexpcted error when attempting to process the command "${body}" send by the user ${info.commandSender}:`, error);
    },
    logCurrentCommandCB(info, commandParts) {
        log.info(`Command being processed for ${info.commandSender}:`, ...commandParts.map(interface_manager_1.TextPresentationRenderer.render.bind(interface_manager_1.TextPresentationRenderer)));
    },
    convertUncaughtErrorToResultError(error) {
        return new matrix_protection_suite_1.ActionException(matrix_protection_suite_1.ActionExceptionKind.Unknown, error, error.message);
    },
    commandNormaliser(body) {
        return body;
    },
};
const rendererFailedCB = function (_adaptor, _eventContext, command, rendererError) {
    log.error(`A renderer for the command "${command.designator.join(" ")}" returned with an error:`, rendererError);
};
exports.rendererFailedCB = rendererFailedCB;
exports.MPSMatrixInterfaceAdaptorCallbacks = Object.freeze({
    promptDefault: MatrixPromptForAccept_1.promptDefault,
    promptSuggestions: MatrixPromptForAccept_1.promptSuggestions,
    defaultRenderer: MatrixHelpRenderer_1.matrixCommandRenderer,
    matrixEventsFromDeadDocument: exports.matrixEventsFromDeadDocument,
    rendererFailedCB: exports.rendererFailedCB,
    matrixEventsFromConfirmationPrompt: MatrixPromptForConfirmation_1.matrixEventsFromConfirmationPrompt,
});
const invocationInformationFromMatrixEventcontext = function (eventContext) {
    return {
        commandEventID: eventContext.event.event_id,
        commandSender: eventContext.event.sender,
    };
};
exports.invocationInformationFromMatrixEventcontext = invocationInformationFromMatrixEventcontext;
exports.MPSContextToCommandContextTranslator = new interface_manager_1.StandardAdaptorContextToCommandContextTranslator();
exports.MPSMatrixInterfaceAdaptor = new interface_manager_1.StandardMatrixInterfaceAdaptor(exports.MPSContextToCommandContextTranslator, exports.invocationInformationFromMatrixEventcontext, exports.MPSMatrixInterfaceAdaptorCallbacks, exports.MPSCommandDispatcherCallbacks);
//# sourceMappingURL=MPSMatrixInterfaceAdaptor.js.map