import { EventEmitter } from "events";
import { ActionResult, ClientPlatform, RoomEvent } from "matrix-protection-suite";
import { StringRoomID, StringUserID, StringEventID } from "@the-draupnir-project/matrix-basic-types";
import { Result } from "@gnuxie/typescript-result";
declare const REACTION_ANNOTATION_KEY = "ge.applied-langua.ge.draupnir.reaction_handler";
type ItemByReactionKey = Map<string, string>;
export type ReactionListener = (key: string, item: string, additionalContext: unknown, reactionMap: ItemByReactionKey, annotatedEvent: RoomEvent) => void;
export declare interface MatrixReactionHandlerListeners {
    on(eventName: string, listener: ReactionListener): void;
    emit(eventName: string, ...args: Parameters<ReactionListener>): void;
}
export declare const ReactionAnnotatedContent: import("@sinclair/typebox").TObject<{
    "ge.applied-langua.ge.draupnir.reaction_handler": import("@sinclair/typebox").TObject<{
        reaction_map: import("@sinclair/typebox").TRecord<import("@sinclair/typebox").TString, import("@sinclair/typebox").TString>;
        name: import("@sinclair/typebox").TString;
        additional_context: import("@sinclair/typebox").TOptional<import("@sinclair/typebox").TUnknown>;
    }>;
}>;
/**
 * A utility that can be associated with an `MatrixEmitter` to listen for
 * reactions to Matrix Events. The aim is to simplify reaction UX.
 */
export declare class MatrixReactionHandler extends EventEmitter implements MatrixReactionHandlerListeners {
    /**
     * The room the handler is for. Cannot be enabled for every room as the
     * OG event lookup is very slow. So usually draupnir's management room.
     */
    readonly roomID: StringRoomID;
    /**
     * The user id of the client. Ignores reactions from this user
     */
    private readonly clientUserID;
    private readonly clientPlatform;
    constructor(
    /**
     * The room the handler is for. Cannot be enabled for every room as the
     * OG event lookup is very slow. So usually draupnir's management room.
     */
    roomID: StringRoomID, 
    /**
     * The user id of the client. Ignores reactions from this user
     */
    clientUserID: StringUserID, clientPlatform: ClientPlatform);
    /**
     * Handle an event from a `MatrixEmitter` and see if it is a reaction to
     * a previously annotated event. If it is a reaction to an annotated event,
     * then call its associated listener.
     * @param roomID The room the event took place in.
     * @param event The Matrix event.
     */
    handleEvent(roomID: StringRoomID, event: RoomEvent): Promise<void>;
    /**
     * Create the annotation required to setup a listener for when a reaction is encountered for the list.
     * @param listenerName The name of the event to emit when a reaction is encountered for a matrix event that matches a key in the `reactionMap`.
     * @param reactionMap A map of reaction keys to items that will be provided to the listener.
     * @param additionalContext Any additional context that should be associated with a matrix event for the listener.
     * @returns An object that should be deep copied into a the content of a new Matrix event.
     */
    createAnnotation(listenerName: string, reactionMap: ItemByReactionKey, additionalContext?: Record<string, unknown> | undefined): Record<typeof REACTION_ANNOTATION_KEY, unknown>;
    /**
     * Use a reaction map to create the initial reactions to an event so that the user has access to quick reactions.
     * @param client A client to add the reactions with.
     * @param roomID The room id of the event to add the reactions to.
     * @param eventID The event id of the event to add reactions to.
     * @param reactionMap The reaction map.
     */
    addReactionsToEvent(roomID: StringRoomID, eventID: StringEventID, reactionMap: ItemByReactionKey): Promise<Result<void>>;
    completePrompt(roomID: StringRoomID, eventID: StringEventID, reason?: string): Promise<ActionResult<void>>;
    /**
     * Removes all reactions from the prompt event in an attempt to stop it being used further.
     */
    cancelPrompt(promptEvent: RoomEvent, cancelReason?: string): Promise<ActionResult<void>>;
    static createItemizedReactionMap(items: string[]): ItemByReactionKey;
    static numberToEmoji(number: number): string;
}
export {};
//# sourceMappingURL=MatrixReactionHandler.d.ts.map