"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardConfigMirror = void 0;
const ConfigParseError_1 = require("./ConfigParseError");
const typescript_result_1 = require("@gnuxie/typescript-result");
const value_1 = require("@sinclair/typebox/value");
const Value_1 = require("../Interface/Value");
class StandardConfigMirror {
    constructor(description) {
        this.description = description;
        // nothing to do.
    }
    setValue(config, key, value) {
        const schema = this.description.schema.properties[key];
        if (schema === undefined) {
            throw new TypeError(`Property ${key.toString()} does not exist in schema`);
        }
        const errors = [...value_1.Value.Errors(schema, value)];
        if (errors[0] !== undefined) {
            return ConfigParseError_1.ConfigPropertyError.Result(errors[0].message, {
                path: `/${key.toString()}`,
                value,
                description: this.description,
            });
        }
        const newConfig = {
            ...config,
            [key]: value,
        };
        return (0, typescript_result_1.Ok)(newConfig);
    }
    addUnparsedItem(config, key, value) {
        const schema = this.description.schema.properties[key];
        if (schema === undefined) {
            throw new TypeError(`Property ${key.toString()} does not exist in schema`);
        }
        if (!('items' in schema)) {
            throw new TypeError(`Property ${key.toString()} is not an array`);
        }
        const isSet = 'uniqueItems' in schema && schema.uniqueItems === true;
        if (isSet) {
            const set = new Set(config[key]);
            set.add(value_1.Value.Decode(schema.items, value));
            return {
                ...config,
                [key]: [...set],
            };
        }
        else {
            return {
                ...config,
                [key]: [...config[key], value_1.Value.Decode(schema, value)],
            };
        }
    }
    addItem(config, key, value) {
        const schema = this.description.schema.properties[key];
        if (schema === undefined) {
            throw new TypeError(`Property ${key.toString()} does not exist in schema`);
        }
        const currentItems = config[key];
        if (!Array.isArray(currentItems)) {
            throw new TypeError(`Property ${key.toString()} is not an array`);
        }
        const errors = [
            ...value_1.Value.Errors(schema, [
                ...config[key],
                value,
            ]),
        ];
        if (errors[0] !== undefined) {
            return ConfigParseError_1.ConfigPropertyError.Result(errors[0].message, {
                path: `/${key.toString()}${errors[0].path}`,
                value,
                description: this.description,
            });
        }
        return (0, typescript_result_1.Ok)(this.addUnparsedItem(config, key, value));
    }
    addSerializedItem(config, key, value) {
        const propertySchema = this.description.schema.properties[key];
        if (propertySchema === undefined) {
            throw new TypeError(`Property ${key.toString()} does not exist in schema`);
        }
        if (!('items' in propertySchema)) {
            throw new TypeError(`Property ${key.toString()} is not an array`);
        }
        const itemSchema = propertySchema.items;
        const decodeResult = Value_1.Value.Decode(itemSchema, value);
        if ((0, typescript_result_1.isError)(decodeResult)) {
            return ConfigParseError_1.ConfigPropertyError.Result(decodeResult.error.message, {
                path: `/${key.toString()}`,
                value,
                description: this.description,
            });
        }
        return (0, typescript_result_1.Ok)(this.addUnparsedItem(config, key, decodeResult.ok));
    }
    setSerializedValue(config, key, value) {
        const schema = this.description.schema.properties[key];
        if (schema === undefined) {
            throw new TypeError(`Property ${key.toString()} does not exist in schema`);
        }
        const decodeResult = Value_1.Value.Decode(schema, value);
        if ((0, typescript_result_1.isError)(decodeResult)) {
            return ConfigParseError_1.ConfigPropertyError.Result(decodeResult.error.message, {
                path: `/${key.toString()}`,
                value,
                description: this.description,
            });
        }
        return this.setValue(config, key, decodeResult.ok);
    }
    removeProperty(key, config) {
        return Object.entries(config).reduce((acc, [k, v]) => {
            if (k !== key) {
                acc[k] = v;
            }
            return acc;
        }, {});
    }
    removeItem(config, key, index) {
        return {
            ...config,
            [key]: config[key].filter((_, i) => i !== index),
        };
    }
    filterItems(config, key, callbackFn) {
        return {
            ...config,
            [key]: config[key].filter(callbackFn),
        };
    }
}
exports.StandardConfigMirror = StandardConfigMirror;
//# sourceMappingURL=ConfigMirror.js.map