"use strict";
// Copyright (C) 2023 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.ActionException = exports.ActionExceptionKind = void 0;
const crypto_1 = require("crypto");
const Action_1 = require("./Action");
const Logger_1 = require("../Logging/Logger");
const log = new Logger_1.Logger('ActionException');
/**
 * A way to catagorise different Exceptions.
 */
var ActionExceptionKind;
(function (ActionExceptionKind) {
    /**
     * This kind is for exceptions that need to be reported to the user,
     * but are mostly irrelevant to the developers because the behaviour is well
     * understood and expected. These exceptions will never be logged to the error
     * level.
     */
    ActionExceptionKind["Known"] = "Known";
    /**
     * This kind is to be used for reporting unexpected or unknown exceptions
     * that the developers need to know about.
     */
    ActionExceptionKind["Unknown"] = "Unknown";
})(ActionExceptionKind || (exports.ActionExceptionKind = ActionExceptionKind = {}));
// TODO: I wonder if we could allow message to be JSX?
/**
 * `ActionExceptions` are used to convert throwables into `ActionError`s.
 * Each `ActionException` is given a unique identifier and is logged immediatley
 * (depending on {@link ActionExceptionKind}).
 *
 * You will want to create these using {@link ActionException.Result}.
 */
class ActionException extends Action_1.ActionError {
    constructor(exceptionKind, 
    // make a call to only allow Error in a moment.
    exception, message, { uuid = (0, crypto_1.randomUUID)(), suppressLog = false, elaborations = [], } = {}) {
        super(message, elaborations);
        this.exceptionKind = exceptionKind;
        this.exception = exception;
        this.uuid = uuid;
        if (!suppressLog) {
            this.log();
        }
    }
    /**
     * Convienant factory method for `ActionException`s that will return an
     * `ActionResult`.
     * @param message The message for the `ActionError` that concisely describes the problem.
     * @param options.exception The `Error` that was thrown.
     * @param options.exceptionKind The `ActionExceptionKind` that catagorieses the exception.
     * @returns An `ActionResult` with the exception as the `Error` value.
     */
    static Result(message, options) {
        return (0, Action_1.ResultError)(new ActionException(options.exceptionKind, options.exception, message));
    }
    log() {
        const logArguments = [
            'ActionException',
            this.exceptionKind,
            this.uuid,
            this.message,
            this.exception,
        ];
        if (this.exceptionKind === ActionExceptionKind.Known) {
            log.info(...logArguments);
        }
        else {
            log.error(...logArguments);
        }
    }
    toReadableString() {
        const mainDetail = `ActionException: ${this.uuid}\n${super.toReadableString()}`;
        if (this.exception instanceof Error) {
            return `${mainDetail}\nfrom error: ${this.exception.name}: ${this.exception.message}\n${this.exception.stack}`;
        }
        // @typescript-eslint/restrict-template-expressions
        // eslint-disable-next-line @typescript-eslint/restrict-template-expressions
        return `${mainDetail}\nfrom unknown: ${this.exception}`;
    }
}
exports.ActionException = ActionException;
//# sourceMappingURL=ActionException.js.map