"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyedBatchQueue = void 0;
const Task_1 = require("./Task");
const Logger_1 = require("../Logging/Logger");
const ActionException_1 = require("./ActionException");
const log = new Logger_1.Logger('KeyedBatchQueue');
class KeyedBatchQueue {
    constructor(batchProcessor) {
        this.batchProcessor = batchProcessor;
        // Queue of pending requests per room
        this.queues = new Map();
        // A "lock" flag: which rooms are actively being processed
        this.processing = new Set();
    }
    enqueue(key, value) {
        return new Promise((resolve) => {
            const request = { value, resolve };
            const entry = this.queues.get(key);
            if (entry === undefined) {
                this.queues.set(key, []);
            }
            else {
                entry.push(request);
            }
            // Try to start processing this room's queue
            this.maybeProcess(key);
        });
    }
    maybeProcess(key) {
        // If this room is already being processed, just wait.
        if (this.processing.has(key))
            return;
        // Mark the room as "locked"
        this.processing.add(key);
        void (0, Task_1.Task)(this.flush(key));
    }
    async flush(key) {
        var _a, _b, _c;
        const requests = (_a = this.queues.get(key)) !== null && _a !== void 0 ? _a : [];
        if (requests.length === 0) {
            this.processing.delete(key);
            return;
        }
        // Take the current batch out of the queue
        this.queues.set(key, []);
        const values = [...new Set(requests.map((r) => r.value))];
        try {
            const processResult = await this.batchProcessor(key, values);
            for (const request of requests) {
                request.resolve(processResult);
            }
        }
        catch (error) {
            log.error('Uncaught error in batch processor', error);
            if (error instanceof Error) {
                const errorResult = ActionException_1.ActionException.Result('Uncaught error in batch processor', {
                    exception: error,
                    exceptionKind: ActionException_1.ActionExceptionKind.Unknown,
                });
                for (const request of requests) {
                    request.resolve(errorResult);
                }
            }
            else {
                throw error; // we don't know wth this is.
            }
        }
        finally {
            // Unlock this room
            this.processing.delete(key);
            // If new requests came in while we were processing, schedule another flush
            if (((_c = (_b = this.queues.get(key)) === null || _b === void 0 ? void 0 : _b.length) !== null && _c !== void 0 ? _c : 0) > 0) {
                this.maybeProcess(key);
            }
        }
    }
}
exports.KeyedBatchQueue = KeyedBatchQueue;
//# sourceMappingURL=KeyedBatchQueue.js.map