import { Result } from '@gnuxie/typescript-result';
export type LifetimeDisposeHandle<Owner = unknown> = OwnLifetime<Owner> | (() => void | Promise<void>) | Disposable;
export interface Lifetime<Owner = unknown> {
    isInDisposal(): boolean;
    toChild<Child = unknown>(): Result<OwnLifetime<Child>>;
    forget(callback: LifetimeDisposeHandle<Owner>): this;
    forgetAndDispose(callback: LifetimeDisposeHandle<Owner>): Promise<void>;
}
export type Disposable = {
    [Symbol.dispose](): void;
} | {
    [Symbol.asyncDispose](): Promise<void>;
};
export interface AllocatableLifetime<Owner = unknown> extends Lifetime<Owner> {
    allocateResource<T>(factory: (lifetime: AllocatableLifetime<Owner>) => Result<T>, disposer: (resource: T) => LifetimeDisposeHandle<Owner>): Result<T>;
    allocateDisposable<TDisposable extends Disposable>(factory: (lifetime: AllocatableLifetime<Owner>) => Result<TDisposable>): Result<TDisposable>;
    allocateResourceAsync<T>(factory: (lifetime: AllocatableLifetime<Owner>) => Promise<Result<T>>, disposer: (resource: T) => LifetimeDisposeHandle<Owner>): Promise<Result<T>>;
    allocateDisposableAsync<TDisposable extends Disposable>(factory: (lifetime: AllocatableLifetime<Owner>) => Promise<Result<TDisposable>>): Promise<Result<TDisposable>>;
    toAbortSignal(): AbortSignal;
    onDispose(callback: LifetimeDisposeHandle<Owner>): this;
}
/**
 * Lifetime is an abstraction that provides structured ownership, cancellation,
 * and cleanup of resources.
 *
 * The Lifetime can also be used to cancel asynchronous operations, such as IO,
 * timeouts, or lock acquisition.
 *
 * We also offer compatibility with `AbortSignal` which allows the Lifetime
 * to be used directly with APIs that support AbortSignal instead. Note:
 * AbortSignal does not support awaiting for cleanup to finish.
 *
 * The lifetime is used to make it impossible to forget to handle dispose methods.
 * Anything that has a dispose method should be allocated against a lifetime.
 * When constructors or factories ask for a lifetime, they will allocate their
 * own resources against this lifetime. However, if the object being constructed
 * also has a dispose method. You will still need to allocate the object created
 * by the factory against a parent lifetime.
 *
 * Lifecycle:
 * - [Symbol.asyncDispose] must be called to dispose the object.
 * - Resources MUST register with a Lifetime atomically as a part of resource
 *   allocation. Resources MUST fail to allocate if the Lifetime is already in disposal.
 *   Use allocateResource.
 *
 */
export interface OwnLifetime<Owner = unknown> extends AllocatableLifetime<Owner> {
    /**
     * We specifically provide a contract that this method will only exit
     * when all resources have cleaned up. And it is not possible to allocate
     * new resources once disposal has started.
     */
    [Symbol.asyncDispose](): Promise<void>;
}
export type LifetimeOptions = {
    readonly parent?: AllocatableLifetime;
};
export declare class StandardLifetime<Owner = unknown> implements OwnLifetime<Owner> {
    private readonly controller;
    private readonly callbacks;
    private readonly disposedPromise;
    private resolveDisposed;
    private readonly parent;
    constructor(options?: LifetimeOptions);
    isInDisposal(): boolean;
    onDispose(callback: LifetimeDisposeHandle<Owner>): this;
    forget(callback: LifetimeDisposeHandle<Owner>): this;
    forgetAndDispose(callback: LifetimeDisposeHandle<Owner>): Promise<void>;
    toAbortSignal(): AbortSignal;
    [Symbol.asyncDispose](): Promise<void>;
    toChild<Child = unknown>(): Result<OwnLifetime<Child>>;
    allocateResource<T>(factory: (lifetime: AllocatableLifetime<Owner>) => Result<T>, disposer: (resource: T) => LifetimeDisposeHandle<Owner>): Result<T>;
    allocateDisposable<TDisposable extends Disposable>(factory: (lifetime: AllocatableLifetime<Owner>) => Result<TDisposable>): Result<TDisposable>;
    withDisposalBlocked<T>(cb: () => Promise<Result<T>>): Promise<Result<T>>;
    allocateResourceAsync<T>(factory: (lifetime: AllocatableLifetime<Owner>) => Promise<Result<T>>, disposer: (resource: T) => LifetimeDisposeHandle<Owner>): Promise<Result<T>>;
    allocateDisposableAsync<TDisposable extends Disposable>(factory: (lifetime: AllocatableLifetime<Owner>) => Promise<Result<TDisposable>>): Promise<Result<TDisposable>>;
}
//# sourceMappingURL=Lifetime.d.ts.map