"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.SemanticType = SemanticType;
exports.describeSemanticType = describeSemanticType;
/**
 * A semantic type is used to describe the expected behavior of a type beyond its shape.
 * At the moment invariants can be specified through the `Law` method.
 * This adds structure to important behavioural contracts that would otherwise be
 * placed loosely in unit tests with no organisation.
 *
 * Another important feature of the invariant law is that they describe why they
 * are included in the sematic type which helps negotiating their removal if
 * the underlying abstraction changes.
 *
 * Critically the laws as opposed to tests are also defined against the
 * interface and not any single concrete implementation. But all implementations
 * can test their implementation against the semantic type.
 *
 * This brings the benefits of test driven development directly to the stage
 * where the interface (and thus abstraction) is defined (or changed). Before
 * any concrete implementation is even considered.
 */
function SemanticType(name) {
    return {
        Law(laws) {
            return describeSemanticType({ name, ...laws });
        },
    };
}
function describeSemanticType(description) {
    const { name, ...rest } = description;
    const lawDescriptions = rest;
    const invariants = Object.entries(lawDescriptions).map(([lawName, lawDescription]) => ({
        semanticTypeName: name,
        lawName,
        ...lawDescription,
    }));
    return {
        name,
        invariants,
        async check(makeSubject) {
            for (const law of invariants) {
                await law.check(makeSubject);
            }
        },
    };
}
//# sourceMappingURL=SemanticType.js.map