"use strict";
// Copyright (C) 2023, 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.setGlobalTaskReporter = setGlobalTaskReporter;
exports.Task = Task;
const typescript_result_1 = require("@gnuxie/typescript-result");
const Logger_1 = require("../Logging/Logger");
const ActionException_1 = require("./ActionException");
const log = new Logger_1.Logger('Task');
let globalTaskReporter = function (error, options) {
    var _a;
    const message = (options === null || options === void 0 ? void 0 : options.description)
        ? `Task Failed (${options.description}):`
        : `Task Failed:`;
    ((_a = options === null || options === void 0 ? void 0 : options.log) !== null && _a !== void 0 ? _a : log).error(message, error.toReadableString());
};
/**
 * Allows the reporter for all failed tasks to be set.
 * @param reporter A function that implements `TaskErrorReporter`.
 * @see {@link TaskErrorReporter}.
 */
function setGlobalTaskReporter(reporter) {
    globalTaskReporter = reporter;
}
/**
 * Sometimes an `Action` takes place in the background usually a result of an
 * event listener. This means that any errors that are experienced will not
 * have a direct way to reach the user of the application.
 *
 * Up until now, the doctrine for this situation has been to simply log
 * at the error level and move on. However, as a background task failing
 * silently is distinct from simply reporting an error to the error level,
 * we believe that the ability for a consumer of the library to configure
 * what happens to these errors is important.
 */
async function Task(task, options) {
    try {
        const result = await task;
        if (typeof result === 'object' &&
            result !== null &&
            'error' in result &&
            result.error instanceof typescript_result_1.ResultError) {
            globalTaskReporter(result.error, options);
            return;
        }
        return;
    }
    catch (exception) {
        const actionException = new ActionException_1.ActionException(ActionException_1.ActionExceptionKind.Unknown, exception, 'A Task failed with an unknown exception');
        globalTaskReporter(actionException, options);
    }
}
//# sourceMappingURL=Task.js.map