"use strict";
// Copyright 2023 - 2025 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2016 OpenMarket Ltd
// Copyright 2018 New Vector Ltd
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from matrix-spec
// https://github.com/matrix-org/matrix-spec
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.RoomVersionMirror = exports.RoomCreateEvent = exports.RoomCreateContent = exports.RoomCreateOptions = void 0;
const typebox_1 = require("@sinclair/typebox");
const PowerLevels_1 = require("./PowerLevels");
const StringlyTypedMatrix_1 = require("./StringlyTypedMatrix");
const Events_1 = require("./Events");
const typescript_result_1 = require("@gnuxie/typescript-result");
exports.RoomCreateOptions = typebox_1.Type.Object({
    visibility: typebox_1.Type.Optional(typebox_1.Type.Union([typebox_1.Type.Literal('public'), typebox_1.Type.Literal('private')])),
    room_alias_name: typebox_1.Type.Optional(typebox_1.Type.String({
        description: 'The desired room alias **local part**. If this is included, a\nroom alias will be created and mapped to the newly created\nroom. The alias will belong on the *same* homeserver which\ncreated the room. For example, if this was set to "foo" and\nsent to the homeserver "example.com" the complete room alias\nwould be `#foo:example.com`.\n\nThe complete room alias will become the canonical alias for\nthe room and an `m.room.canonical_alias` event will be sent\ninto the room.',
    })),
    name: typebox_1.Type.Optional(typebox_1.Type.String({
        description: 'If this is included, an `m.room.name` event will be sent\ninto the room to indicate the name of the room. See Room\nEvents for more information on `m.room.name`.',
    })),
    topic: typebox_1.Type.Optional(typebox_1.Type.String({
        description: 'If this is included, an `m.room.topic` event will be sent\ninto the room to indicate the topic for the room. See Room\nEvents for more information on `m.room.topic`.',
    })),
    invite: typebox_1.Type.Optional(typebox_1.Type.Array(typebox_1.Type.String(), {
        description: 'A list of user IDs to invite to the room. This will tell the\nserver to invite everyone in the list to the newly created room.',
    })),
    invite_3pid: typebox_1.Type.Optional(typebox_1.Type.Array(typebox_1.Type.Object({
        id_server: typebox_1.Type.String({
            description: 'The hostname+port of the identity server which should be used for third-party identifier lookups.',
        }),
        id_access_token: typebox_1.Type.String({
            description: 'An access token previously registered with the identity server. Servers\ncan treat this as optional to distinguish between r0.5-compatible clients\nand this specification version.',
        }),
        medium: typebox_1.Type.String({
            description: 'The kind of address being passed in the address field, for example `email`\n(see [the list of recognised values](/appendices/#3pid-types)).',
        }),
        address: typebox_1.Type.String({
            description: "The invitee's third-party identifier.",
        }),
    }), {
        description: 'A list of objects representing third-party IDs to invite into\nthe room.',
    })),
    room_version: typebox_1.Type.Optional(typebox_1.Type.String({
        description: 'The room version to set for the room. If not provided, the homeserver is\nto use its configured default. If provided, the homeserver will return a\n400 error with the errcode `M_UNSUPPORTED_ROOM_VERSION` if it does not\nsupport the room version.',
        example: '1',
    })),
    creation_content: typebox_1.Type.Optional(typebox_1.Type.Object({
        type: typebox_1.Type.Optional(typebox_1.Type.String({ description: 'The type of the room' })),
    })),
    initial_state: typebox_1.Type.Optional(typebox_1.Type.Array(typebox_1.Type.Object({
        type: typebox_1.Type.String({ description: 'The type of event to send.' }),
        state_key: typebox_1.Type.Optional(typebox_1.Type.String({
            description: 'The state_key of the state event. Defaults to an empty string.',
        })),
        content: typebox_1.Type.Unknown(),
    }), {
        description: 'A list of state events to set in the new room. This allows\nthe user to override the default state events set in the new\nroom. The expected format of the state events are an object\nwith type, state_key and content keys set.\n\nTakes precedence over events set by `preset`, but gets\noverridden by `name` and `topic` keys.',
    })),
    preset: typebox_1.Type.Optional(typebox_1.Type.Union([
        typebox_1.Type.Literal('private_chat'),
        typebox_1.Type.Literal('public_chat'),
        typebox_1.Type.Literal('trusted_private_chat'),
    ])),
    is_direct: typebox_1.Type.Optional(typebox_1.Type.Boolean({
        description: 'This flag makes the server set the `is_direct` flag on the\n`m.room.member` events sent to the users in `invite` and\n`invite_3pid`. See [Direct Messaging](/client-server-api/#direct-messaging) for more information.',
    })),
    power_level_content_override: typebox_1.Type.Optional(PowerLevels_1.PowerLevelsEventContent),
});
exports.RoomCreateContent = typebox_1.Type.Object({
    creator: typebox_1.Type.Optional(typebox_1.Type.String({
        description: 'The `user_id` of the room creator. **Required** for, and only present in, room versions 1 - 10. Starting with\nroom version 11 the event `sender` should be used instead.',
    })),
    'm.federate': typebox_1.Type.Optional(typebox_1.Type.Boolean({
        description: 'Whether users on other servers can join this room. Defaults to `true` if key does not exist.',
    })),
    room_version: typebox_1.Type.Optional(typebox_1.Type.String({
        description: 'The version of the room. Defaults to `"1"` if the key does not exist.',
    })),
    type: typebox_1.Type.Optional(typebox_1.Type.String({
        description: "Optional [room type](/client-server-api/#types) to denote a room's intended function outside of traditional\nconversation.\n\nUnspecified room types are possible using [Namespaced Identifiers](/appendices/#common-namespaced-identifier-grammar).",
    })),
    predecessor: typebox_1.Type.Optional(typebox_1.Type.Object({
        room_id: typebox_1.Type.String({ description: 'The ID of the old room.' }),
        event_id: typebox_1.Type.Optional(typebox_1.Type.String({
            description: 'The event ID of the last known event in the old room.',
        })),
    })),
    additional_creators: typebox_1.Type.Optional(typebox_1.Type.Array(StringlyTypedMatrix_1.StringUserIDSchema, {
        description: 'These are users with infinite power level when the room version is 12 and above.',
    })),
});
exports.RoomCreateEvent = typebox_1.Type.Intersect([
    typebox_1.Type.Omit((0, Events_1.StateEvent)(exports.RoomCreateContent), ['state_key', 'type']),
    typebox_1.Type.Object({
        state_key: typebox_1.Type.String({
            description: 'A zero-length string.',
            pattern: '^$',
        }),
        type: typebox_1.Type.Literal('m.room.create'),
    }),
]);
// FIXME: SHouldn't the prividliged creators function return a result error?
// i think so, but it just depends how the permission calculation system
// uses it and whether it supports feeding errors back.
exports.RoomVersionMirror = Object.freeze({
    isVersionWithPrivilidgedCreators(versionSpecifier) {
        const integerResult = (() => {
            try {
                return (0, typescript_result_1.Ok)(parseInt(versionSpecifier, 10));
            }
            catch (e) {
                if (e instanceof Error) {
                    return typescript_result_1.ResultError.Result(e.message);
                }
                throw e; // we don't know what this is.
            }
        })();
        if ((0, typescript_result_1.isError)(integerResult)) {
            return false; // unknown room version.
        }
        if (integerResult.ok >= 12) {
            return true; // versions below 12 and abovehave privilidged creators.
        }
        return false;
    },
    isUserAPrivilidgedCreator(userID, creationEvent) {
        var _a;
        if (creationEvent.content.room_version === undefined) {
            return false;
        }
        if (!this.isVersionWithPrivilidgedCreators(creationEvent.content.room_version)) {
            return false;
        }
        if (userID === creationEvent.sender) {
            return true;
        }
        if ((_a = creationEvent.content.additional_creators) === null || _a === void 0 ? void 0 : _a.includes(userID)) {
            return true;
        }
        return false;
    },
    priviligedCreators(creationEvent) {
        var _a;
        if (creationEvent.content.room_version === undefined ||
            !this.isVersionWithPrivilidgedCreators(creationEvent.content.room_version)) {
            return [creationEvent.sender];
        }
        return [
            creationEvent.sender,
            ...((_a = creationEvent.content.additional_creators) !== null && _a !== void 0 ? _a : []),
        ];
    },
});
//# sourceMappingURL=CreateRoom.js.map