"use strict";
// Copyright (C) 2023 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
const Action_1 = require("../Interface/Action");
const Value_1 = require("../Interface/Value");
const EventGeneration_1 = require("../TestUtilities/EventGeneration");
const DefaultEventDecoder_1 = require("./DefaultEventDecoder");
const PolicyEvents_1 = require("./PolicyEvents");
const RoomMessage_1 = require("./RoomMessage");
test('Raw events are parsed correctly', function () {
    const room = (0, EventGeneration_1.randomRoomID)(['example.org']);
    const nRules = 25;
    const rawEvents = [...Array(nRules)].map((_) => (0, EventGeneration_1.randomRawEvent)('@test:example.com', room.toRoomIDOrAlias()));
    const parsedEventsResult = rawEvents.map(DefaultEventDecoder_1.DefaultEventDecoder.decodeEvent.bind(DefaultEventDecoder_1.DefaultEventDecoder));
    expect(parsedEventsResult.every((result) => (0, Action_1.isOk)(result))).toBe(true);
});
test('Parsing error information is genuinly useful', function () {
    const eventWithoutID = (0, EventGeneration_1.randomRawEvent)('@test:example.com', '!test:example.com');
    if (typeof eventWithoutID !== 'object' || eventWithoutID === null) {
        throw new TypeError('Test was not setup correctly');
    }
    if ('event_id' in eventWithoutID) {
        eventWithoutID.event_id = undefined;
    }
    const decodeResult = DefaultEventDecoder_1.DefaultEventDecoder.decodeEvent(eventWithoutID);
    expect((0, Action_1.isError)(decodeResult)).toBe(true);
    if ((0, Action_1.isOk)(decodeResult)) {
        throw new TypeError('The decode result should be an error at this point');
    }
    expect(decodeResult.error).toBeInstanceOf(Value_1.DecodeException);
    if (!(decodeResult.error instanceof Value_1.DecodeException)) {
        throw new TypeError('The decode result should be decode exception at this point');
    }
    expect(decodeResult.error.errors.find((error) => error.path === '/event_id'));
});
test('Policy List event with leftover reason', function () {
    const rawEvent = {
        content: {
            reason: 'not a scammer, no idea what he was thinking',
        },
        origin_server_ts: 1693351663899,
        room_id: '!QzNmKfDhzCQFDqmUhl:matrix.org',
        sender: '@anti-scam:matrix.org',
        state_key: 'rule:@stoere:projectsegfau.lt',
        type: 'm.policy.rule.user',
        unsigned: {
            replaces_state: '$oyUATFoZDKOF6csFDXBWAkv38-VCLuMw3UTwVcNT6D0',
            age: 20453623176,
        },
        event_id: '$1bcs953wT9lf_aXNEjxPOS9I94GvZvWs-TdXJTxDn6w',
        user_id: '@anti-scam:matrix.org',
        age: 20453623176,
        replaces_state: '$oyUATFoZDKOF6csFDXBWAkv38-VCLuMw3UTwVcNT6D0',
    };
    const result = Value_1.Value.Decode(PolicyEvents_1.PolicyRuleEvent, rawEvent);
    expect(result.isOkay).toBeFalsy();
});
test('Room messages', function () {
    const rawContent = {
        msgtype: 'm.notice',
        body: '**Documentation: ** [https://the-draupnir-proj…no description\n</details></details></details>',
        format: 'org.matrix.custom.html',
        formatted_body: '<b>Documentation: </b> <a href="https://t…ption<br/></details></details></details>',
        'm.relates_to': {
            'm.in_reply_to': (0, EventGeneration_1.randomEventID)(),
        },
    };
    const result = Value_1.Value.Decode(RoomMessage_1.NoticeMessageContent, rawContent);
    expect(result.isOkay).toBeTruthy();
});
//# sourceMappingURL=EventDecoder.test.js.map