"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerACLBuilder = void 0;
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
class ServerACLBuilder {
    constructor(homeserver) {
        this.homeserver = homeserver;
        this.allowedServers = new Set();
        this.deniedServers = new Set();
        this.allowIps = false;
    }
    /**
     * Checks the ACL for any entries that might ban ourself.
     * @returns A list of deny entries that will not ban our own homeserver.
     */
    safeDeniedServers() {
        // The reason we do this check here rather than in the `denyServer` method
        // is because `literalAclContent` exists and also we want to be defensive about someone
        // mutating `this.deniedServers` via another method in the future.
        const entries = [];
        for (const server of this.deniedServers) {
            const glob = new matrix_basic_types_1.MatrixGlob(server);
            if (!glob.test(this.homeserver)) {
                entries.push(server);
            }
        }
        return entries;
    }
    safeAllowedServers() {
        const allowed = [...this.allowedServers];
        if (allowed.length === 0) {
            allowed.push('*'); // allow everything
        }
        if (!allowed.some((server) => new matrix_basic_types_1.MatrixGlob(server).test(this.homeserver))) {
            allowed.push(this.homeserver);
        }
        return allowed;
    }
    allowIpAddresses() {
        this.allowIps = true;
        return this;
    }
    denyIpAddresses() {
        this.allowIps = false;
        return this;
    }
    allowServer(glob) {
        this.allowedServers.add(glob);
        return this;
    }
    setAllowedServers(globs) {
        this.allowedServers = new Set(globs);
        return this;
    }
    denyServer(glob) {
        this.deniedServers.add(glob);
        return this;
    }
    setDeniedServers(globs) {
        this.deniedServers = new Set(globs);
        return this;
    }
    safeAclContent() {
        return {
            allow: this.safeAllowedServers(),
            deny: this.safeDeniedServers(),
            allow_ip_literals: this.allowIps,
        };
    }
    matches(acl) {
        const allow = acl['allow'];
        const deny = acl['deny'];
        const ips = acl['allow_ip_literals'];
        let allowMatches = true; // until proven false
        let denyMatches = true; // until proven false
        const ipsMatch = ips === this.allowIps;
        const currentAllowed = this.safeAllowedServers();
        if ((allow === null || allow === void 0 ? void 0 : allow.length) === currentAllowed.length) {
            for (const s of allow) {
                if (!currentAllowed.includes(s)) {
                    allowMatches = false;
                    break;
                }
            }
        }
        else
            allowMatches = false;
        const currentDenied = this.safeDeniedServers();
        if ((deny === null || deny === void 0 ? void 0 : deny.length) === currentDenied.length) {
            for (const s of deny) {
                if (!currentDenied.includes(s)) {
                    denyMatches = false;
                    break;
                }
            }
        }
        else
            denyMatches = false;
        return denyMatches && allowMatches && ipsMatch;
    }
}
exports.ServerACLBuilder = ServerACLBuilder;
//# sourceMappingURL=ServerACLBuilder.js.map