"use strict";
// Copyright (C) 2023 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.MembershipChange = exports.ProfileChangeType = exports.MembershipChangeType = exports.Membership = void 0;
exports.membershipChangeType = membershipChangeType;
exports.profileChangeType = profileChangeType;
var Membership;
(function (Membership) {
    Membership["Join"] = "join";
    Membership["Invite"] = "invite";
    Membership["Knock"] = "knock";
    Membership["Leave"] = "leave";
    Membership["Ban"] = "ban";
})(Membership || (exports.Membership = Membership = {}));
// TODO: The problem with this is that leave->leave implies a user temporarily
//       rejoined the room, so it's wrong to classify it as `NoChange` since
//       we might want to wanr protections about the rejoin even if we don't
//       have a record of it (because of downtime).
//       of course, this would have to be its own enum variant
//       https://github.com/Gnuxie/matrix-protection-suite/issues/21
// TODO: Another problem is Rejoined is probably error prone given a user
//       rejecting an invite or being kicked on knock for the first time and
//       then joining again probably shouldn't be classified as rejoined
//       since they never did really join the room.
//       of course the reality is that they would be classified as renocked
//       or reinvited unless there was a skipping of an event due to downtime.
//       https://github.com/Gnuxie/matrix-protection-suite/issues/22
// TODO: These lapses in membership transition due to downtime really need to be
//       written formally into a spec issue because they are quite bullshit.
//       https://github.com/Gnuxie/matrix-protection-suite/issues/23
/**
 * Represents the change to the membership field of a membership event.
 * This does not represent change in the overall event!
 * You must still account for NoChange does not mean the membership events
 * themselves are the same!!! If that was the case, there would be no
 * `MembershipChange` event emitted by the revision issuer in the first place!
 */
var MembershipChangeType;
(function (MembershipChangeType) {
    /** A genuine join to the room. */
    MembershipChangeType["Joined"] = "joined";
    /** A join to the room when the user has recently left, was removed or banned.
     * This distinction exists because bridged users from IRC or XMPP will do this all the time.
     * And things like greeter bots would break.
     */
    MembershipChangeType["Rejoined"] = "rejoined";
    /** The user left the room by their own command. */
    MembershipChangeType["Left"] = "left";
    /** The user was kicked by another user. */
    MembershipChangeType["Kicked"] = "kicked";
    /** The user was banned by another user. */
    MembershipChangeType["Banned"] = "banned";
    /** The user was unbanned by another user. */
    MembershipChangeType["Unbanned"] = "unbanned";
    /** The user made a genuine knock on the room. */
    MembershipChangeType["Knocked"] = "knocked";
    /** The user renocked on the room after recently leaving @see {@link MembershipChangeType.Rejoin}. */
    MembershipChangeType["Reknocked"] = "renocked";
    /** The user was invited by another user. */
    MembershipChangeType["Invited"] = "invited";
    /** There was no change to the membership, but there may have been changes to their profile or reason for their membership. */
    MembershipChangeType["NoChange"] = "no-change";
})(MembershipChangeType || (exports.MembershipChangeType = MembershipChangeType = {}));
function membershipChangeType(nextMembership, previousMembershipContent) {
    var _a;
    const previousMembership = (_a = previousMembershipContent === null || previousMembershipContent === void 0 ? void 0 : previousMembershipContent.membership) !== null && _a !== void 0 ? _a : 'external';
    switch (nextMembership.content.membership) {
        case Membership.Join:
            switch (previousMembership) {
                case Membership.Join:
                    return MembershipChangeType.NoChange;
                case Membership.Leave:
                // it is possible for us to not have/skip the intermediate state for their unban.
                // fall through
                case Membership.Ban:
                    return MembershipChangeType.Rejoined;
                default:
                    return MembershipChangeType.Joined;
            }
        case Membership.Invite:
            switch (previousMembership) {
                case Membership.Invite:
                    return MembershipChangeType.NoChange;
                default:
                    return MembershipChangeType.Invited;
            }
        case Membership.Knock:
            switch (previousMembership) {
                case Membership.Leave:
                case Membership.Join:
                case Membership.Invite:
                case Membership.Ban:
                    return MembershipChangeType.Reknocked;
                case Membership.Knock:
                    return MembershipChangeType.NoChange;
                default:
                    return MembershipChangeType.Knocked;
            }
        case Membership.Leave:
            switch (previousMembership) {
                case Membership.Ban:
                    return MembershipChangeType.Unbanned;
                case Membership.Leave:
                    return MembershipChangeType.NoChange;
                default:
                    if (nextMembership.sender === nextMembership.state_key) {
                        return MembershipChangeType.Left;
                    }
                    else {
                        return MembershipChangeType.Kicked;
                    }
            }
        case Membership.Ban:
            switch (previousMembership) {
                case Membership.Ban:
                    return MembershipChangeType.NoChange;
                default:
                    return MembershipChangeType.Banned;
            }
        default:
            throw new TypeError(`Unknown membership ${nextMembership.content.membership}`);
    }
}
var ProfileChangeType;
(function (ProfileChangeType) {
    ProfileChangeType["InitialProfile"] = "initial-profile";
    ProfileChangeType["Displayname"] = "displayname";
    ProfileChangeType["Avatar"] = "avatar";
    ProfileChangeType["DisplaynameAndAvatar"] = "displayname-and-avatar";
    ProfileChangeType["NoChange"] = "no-change";
})(ProfileChangeType || (exports.ProfileChangeType = ProfileChangeType = {}));
function profileChangeType(nextMembership, previousMembershipContent) {
    if (previousMembershipContent === undefined) {
        return ProfileChangeType.InitialProfile;
    }
    const isDisplaynameNew = nextMembership.content.displayname !==
        previousMembershipContent.displayname;
    const isAvatarNew = nextMembership.content.avatar_url !== previousMembershipContent.avatar_url;
    if (isDisplaynameNew && isAvatarNew) {
        return ProfileChangeType.DisplaynameAndAvatar;
    }
    if (isDisplaynameNew) {
        return ProfileChangeType.Displayname;
    }
    if (isAvatarNew) {
        return ProfileChangeType.Avatar;
    }
    return ProfileChangeType.NoChange;
}
class MembershipChange {
    constructor(userID, sender, roomID, eventID, membership, membershipChangeType, profileChangeType, content) {
        this.userID = userID;
        this.sender = sender;
        this.roomID = roomID;
        this.eventID = eventID;
        this.membership = membership;
        this.membershipChangeType = membershipChangeType;
        this.profileChangeType = profileChangeType;
        this.content = content;
        // nothing to do.
    }
}
exports.MembershipChange = MembershipChange;
//# sourceMappingURL=MembershipChange.js.map