"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardSetMembershipPolicyRevision = void 0;
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const PolicyEvents_1 = require("../MatrixTypes/PolicyEvents");
const SetMembershipRevision_1 = require("../Membership/SetMembershipRevision");
const PolicyRule_1 = require("../PolicyList/PolicyRule");
const PolicyRuleChange_1 = require("../PolicyList/PolicyRuleChange");
const immutable_1 = require("immutable");
const StandardPolicyListRevision_1 = require("../PolicyList/StandardPolicyListRevision");
const Revision_1 = require("../PolicyList/Revision");
// TODO: It would be nice to have a method on PolicyListRevision that would
// let us pass a membership revision and have it run the matches for us
// the idea being that it could use literal values from literal rooms
// to lookup memberships. Alternatively we could have a method that
// returns all literal rules or all glob rules so that this can be hand
// optimized. But it's a little silly that way since we would probably
// have to then make a temporary revision from glob rules just to make it
// easier.
// Whatever is chosen would need to allow the data to be partitioned
// TODO: Don't forget that when this is consumed by the user consequences capability
// that you need to do a check afterwards that the user didn't join more rooms
// while the capability was running, and if they did, keep removing them.
// however, doing so needs to consider the rooms that failed to remove e.g. because
// of permissions. can easily be done by comparing errors to remaining rooms.
// Worrying, it might be possible to evade a ban by joining rooms.
// Since the ban events have to take ages to come down sync before they get
// removed from the set revision, and if you join you can evade detection.
// So we might have to keep running the capability onMembershipChange when
// they join. Regardless of whether there are matching policies or not.
// Alternatively, we can just keep trying every time we get a join, which
// is probably going to cause a lot of duplicates unless we come up with
// something smart.
// Mutual excursion and checking repeatedly might not work because it takes
// time for the bans to come down sync and be processed. It might be possible
// to avoid that by keeping a small set of roomMembershipRevisionID's and successful
// room bans in a little list and simply compare and pop them off when we get
// the bans.
// To do this at the protection level we'd need to be aware of the difference
// between policies being recalled due to banning vs recalled because of removed
// policies. And that would need to happen in this revision. (Recalled vs no policies
// left and became absent)
class StandardSetMembershipPolicyRevision {
    constructor(memberPolicies, policyMembers) {
        this.memberPolicies = memberPolicies;
        this.policyMembers = policyMembers;
        this.revision = new Revision_1.Revision();
        // nothing to do.
    }
    changesFromMembershipChanges(delta, policyRevision) {
        const addedMatches = [];
        const removedMatches = [];
        for (const change of delta.changes) {
            if (change.changeType === SetMembershipRevision_1.SetMembershipChangeType.NoOverallChange) {
                continue;
            }
            const matchingRules = [
                ...policyRevision.allRulesMatchingEntity(change.userID, {
                    type: PolicyEvents_1.PolicyRuleType.User,
                    searchHashedRules: false,
                }),
                // hmm would it be better if the membership revision grouped users by server so that
                // we would only need to do server scans once?
                // i mean we'd have to do the complete scan when the user is joining a room we already have interned
                // but it'd at least avoid things when adding an entire room of users.
                // That being said, there aren't that many server rules...
                ...policyRevision.allRulesMatchingEntity((0, matrix_basic_types_1.userServerName)(change.userID), {
                    type: PolicyEvents_1.PolicyRuleType.Server,
                    searchHashedRules: false,
                }),
            ];
            if (change.changeType === SetMembershipRevision_1.SetMembershipChangeType.BecamePresent) {
                if (matchingRules.length !== 0) {
                    for (const rule of matchingRules) {
                        addedMatches.push({
                            userID: change.userID,
                            policy: rule,
                        });
                    }
                }
            }
            else {
                const matchingRules = this.memberPolicies.get(change.userID);
                if (matchingRules) {
                    for (const rule of matchingRules) {
                        removedMatches.push({
                            userID: change.userID,
                            policy: rule,
                        });
                    }
                }
            }
        }
        return {
            addedMemberMatches: addedMatches,
            removedMemberMatches: removedMatches,
        };
    }
    changesFromPolicyChanges(changes, setMembershipRevision) {
        const removedMatches = [];
        const policiesToRemove = changes
            .filter((change) => change.changeType === PolicyRuleChange_1.PolicyRuleChangeType.Removed ||
            change.changeType === PolicyRuleChange_1.PolicyRuleChangeType.Modified)
            .map((change) => {
            if (change.previousRule === undefined) {
                throw new TypeError('Modified and Removed policies should have the previousRule field.');
            }
            return change.previousRule;
        });
        for (const policy of policiesToRemove) {
            if (policy.matchType === PolicyRule_1.PolicyRuleMatchType.HashedLiteral) {
                continue; // we can't derive matches from hashed literals.
            }
            for (const userID of this.policyMembers.get(policy, (0, immutable_1.List)())) {
                removedMatches.push({
                    userID,
                    policy,
                });
            }
        }
        const relevantChangesForTemproaryRevision = changes
            .filter((change) => change.changeType !== PolicyRuleChange_1.PolicyRuleChangeType.Removed)
            .map((change) => {
            switch (change.changeType) {
                case PolicyRuleChange_1.PolicyRuleChangeType.Modified:
                case PolicyRuleChange_1.PolicyRuleChangeType.RevealedLiteral:
                    return { ...change, changeType: PolicyRuleChange_1.PolicyRuleChangeType.Added };
                default:
                    return change;
            }
        });
        const temporaryRevision = StandardPolicyListRevision_1.StandardPolicyListRevision.blankRevision().reviseFromChanges(relevantChangesForTemproaryRevision);
        const addedMatches = [];
        for (const member of setMembershipRevision.presentMembers()) {
            const matchingRules = [
                ...temporaryRevision.allRulesMatchingEntity(member, {
                    type: PolicyEvents_1.PolicyRuleType.User,
                    searchHashedRules: false,
                }),
                ...temporaryRevision.allRulesMatchingEntity(member, {
                    type: PolicyEvents_1.PolicyRuleType.Server,
                    searchHashedRules: false,
                }),
            ];
            if (matchingRules.length !== 0) {
                for (const rule of matchingRules) {
                    addedMatches.push({
                        userID: member,
                        policy: rule,
                    });
                }
            }
        }
        return {
            addedMemberMatches: addedMatches,
            removedMemberMatches: removedMatches,
        };
    }
    changesFromInitialRevisions(policyRevision, setMembershipRevision) {
        const addedMatches = [];
        for (const member of setMembershipRevision.presentMembers()) {
            const matchingRules = [
                ...policyRevision.allRulesMatchingEntity(member, {
                    type: PolicyEvents_1.PolicyRuleType.User,
                    searchHashedRules: false,
                }),
                ...policyRevision.allRulesMatchingEntity(member, {
                    type: PolicyEvents_1.PolicyRuleType.Server,
                    searchHashedRules: false,
                }),
            ];
            if (matchingRules.length !== 0) {
                for (const rule of matchingRules) {
                    addedMatches.push({
                        userID: member,
                        policy: rule,
                    });
                }
            }
        }
        return {
            addedMemberMatches: addedMatches,
            removedMemberMatches: [],
        };
    }
    reviseFromChanges(delta) {
        if (delta.addedMemberMatches.length === 0 &&
            delta.removedMemberMatches.length === 0) {
            return this;
        }
        let memberPolicies = this.memberPolicies;
        let policyMembers = this.policyMembers;
        for (const match of delta.addedMemberMatches) {
            const existing = memberPolicies.get(match.userID, (0, immutable_1.List)());
            memberPolicies = memberPolicies.set(match.userID, existing.push(match.policy));
            const existingMembers = policyMembers.get(match.policy, (0, immutable_1.List)());
            policyMembers = policyMembers.set(match.policy, existingMembers.push(match.userID));
        }
        for (const match of delta.removedMemberMatches) {
            const existingPolicies = memberPolicies.get(match.userID, (0, immutable_1.List)());
            const nextPolicies = existingPolicies.filter((rule) => rule !== match.policy);
            if (nextPolicies.size === 0) {
                memberPolicies = memberPolicies.delete(match.userID);
            }
            else {
                memberPolicies = memberPolicies.set(match.userID, nextPolicies);
            }
            const existingMembers = policyMembers.get(match.policy, (0, immutable_1.List)());
            const nextMembers = existingMembers.filter((userID) => userID !== match.userID);
            if (nextMembers.size === 0) {
                policyMembers = policyMembers.delete(match.policy);
            }
            else {
                policyMembers = policyMembers.set(match.policy, nextMembers);
            }
        }
        return new StandardSetMembershipPolicyRevision(memberPolicies, policyMembers);
    }
    isBlankRevision() {
        return this.memberPolicies.size === 0;
    }
    allMembersWithRules() {
        return this.memberPolicies.reduce((matches, policyRules, userID) => {
            matches.push({
                userID: userID,
                policies: policyRules.toArray(),
            });
            return matches;
        }, []);
    }
    allRulesMatchingMember(member, options) {
        return this.memberPolicies
            .get(member, (0, immutable_1.List)())
            .filter((rule) => {
            if (options.type !== undefined && rule.kind !== options.type) {
                return false;
            }
            else if (options.recommendation !== undefined &&
                rule.recommendation !== options.recommendation) {
                return false;
            }
            else {
                return true;
            }
        })
            .toArray();
    }
    static blankRevision() {
        return new StandardSetMembershipPolicyRevision((0, immutable_1.Map)(), (0, immutable_1.Map)());
    }
}
exports.StandardSetMembershipPolicyRevision = StandardSetMembershipPolicyRevision;
//# sourceMappingURL=StandardSetMembershipPolicyRevision.js.map