"use strict";
// Copyright 2022 - 2025 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardPolicyRoomRevision = void 0;
const PolicyEvents_1 = require("../MatrixTypes/PolicyEvents");
const PolicyRule_1 = require("./PolicyRule");
const PolicyRuleChange_1 = require("./PolicyRuleChange");
const StateChangeType_1 = require("../StateTracking/StateChangeType");
const Revision_1 = require("./Revision");
const immutable_1 = require("immutable");
const Logger_1 = require("../Logging/Logger");
const PowerLevelsMirror_1 = require("../Client/PowerLevelsMirror");
const typescript_result_1 = require("@gnuxie/typescript-result");
const crypto_js_1 = require("crypto-js");
const enc_base64_1 = __importDefault(require("crypto-js/enc-base64"));
const log = new Logger_1.Logger('StandardPolicyRoomRevision');
/**
 * A standard implementation of a `PolicyListRevision` using immutable's persistent maps.
 */
class StandardPolicyRoomRevision {
    /**
     * Use {@link StandardPolicyRoomRevision.blankRevision} to get started.
     * Only use this constructor if you are implementing a variant of PolicyListRevision.
     * @param revisionID A revision ID to represent this revision.
     * @param policyRules A map containing the rules for this revision by state type and then state key.
     * @param policyRuleByEventId A map containing the rules ofr this revision by event id.
     */
    constructor(room, revisionID, shortcode, 
    /**
     * A map of state events indexed first by state type and then state keys.
     */
    policyRules, 
    /**
     * Allow us to detect whether we have updated the state for this event.
     */
    policyRuleByEventId, policyRuleBySHA256, powerLevelsEvent) {
        this.room = room;
        this.revisionID = revisionID;
        this.shortcode = shortcode;
        this.policyRules = policyRules;
        this.policyRuleByEventId = policyRuleByEventId;
        this.policyRuleBySHA256 = policyRuleBySHA256;
        this.powerLevelsEvent = powerLevelsEvent;
    }
    /**
     * @returns An empty revision.
     */
    static blankRevision(room) {
        return new StandardPolicyRoomRevision(room, new Revision_1.Revision(), undefined, (0, immutable_1.Map)(), (0, immutable_1.Map)(), (0, immutable_1.Map)(), undefined);
    }
    isBlankRevision() {
        return this.policyRuleByEventId.isEmpty();
    }
    /**
     * Lookup the current rules cached for the list.
     * @param stateType The event type e.g. m.policy.rule.user.
     * @param stateKey The state key e.g. rule:@bad:matrix.org
     * @returns A state event if present or null.
     */
    getPolicyRule(stateType, stateKey) {
        var _a;
        return (_a = this.policyRules.get(stateType)) === null || _a === void 0 ? void 0 : _a.get(stateKey);
    }
    allRules() {
        return [...this.policyRuleByEventId.values()];
    }
    allRulesMatchingEntity(entity, { recommendation, type: ruleKind, searchHashedRules, }) {
        const ruleTypeOf = (entityPart) => {
            if (ruleKind) {
                return ruleKind;
            }
            else if (entityPart.startsWith('!') || entityPart.startsWith('#')) {
                return PolicyEvents_1.PolicyRuleType.Room;
            }
            else if (entity.startsWith('@')) {
                return PolicyEvents_1.PolicyRuleType.User;
            }
            else {
                return PolicyEvents_1.PolicyRuleType.Server;
            }
        };
        const hash = searchHashedRules ? enc_base64_1.default.stringify((0, crypto_js_1.SHA256)(entity)) : '';
        return this.allRulesOfType(ruleTypeOf(entity), recommendation).filter((rule) => {
            if (rule.matchType !== PolicyRule_1.PolicyRuleMatchType.HashedLiteral) {
                return rule.isMatch(entity);
            }
            else {
                if (searchHashedRules) {
                    return rule.hashes['sha256'] === hash;
                }
                else {
                    return false;
                }
            }
        });
    }
    findRulesMatchingHash(hash, algorithm, { type, recommendation, }) {
        if (algorithm === 'sha256') {
            return [
                ...this.policyRuleBySHA256
                    .get(hash, (0, immutable_1.List)())
                    .filter((rule) => type === rule.kind &&
                    (recommendation === undefined ||
                        recommendation === rule.recommendation)),
            ];
        }
        return this.allRulesOfType(type, recommendation).filter((rule) => {
            if (rule.matchType !== PolicyRule_1.PolicyRuleMatchType.HashedLiteral) {
                return false;
            }
            return rule.hashes[algorithm] === hash;
        });
    }
    findRuleMatchingEntity(entity, { recommendation, type, searchHashedRules }) {
        const hash = searchHashedRules ? enc_base64_1.default.stringify((0, crypto_js_1.SHA256)(entity)) : '';
        return this.allRulesOfType(type, recommendation).find((rule) => {
            if (rule.matchType !== PolicyRule_1.PolicyRuleMatchType.HashedLiteral) {
                return rule.isMatch(entity);
            }
            else {
                if (searchHashedRules) {
                    return rule.hashes['sha256'] === hash;
                }
                else {
                    return false;
                }
            }
        });
    }
    allRulesOfType(type, recommendation) {
        const rules = [];
        const stateKeyMap = this.policyRules.get(type);
        if (stateKeyMap) {
            for (const rule of stateKeyMap.values()) {
                if (rule.kind === type) {
                    if (recommendation === undefined) {
                        rules.push(rule);
                    }
                    else if (rule.recommendation === recommendation) {
                        rules.push(rule);
                    }
                }
            }
        }
        return rules;
    }
    reviseFromChanges(changes) {
        let nextPolicyRules = this.policyRules;
        let nextPolicyRulesByEventID = this.policyRuleByEventId;
        let nextPolicyRulesBySHA256 = this.policyRuleBySHA256;
        const setPolicyRule = (stateType, stateKey, rule) => {
            var _a;
            const typeTable = (_a = nextPolicyRules.get(stateType)) !== null && _a !== void 0 ? _a : (0, immutable_1.Map)();
            nextPolicyRules = nextPolicyRules.set(stateType, typeTable.set(stateKey, rule));
            nextPolicyRulesByEventID = nextPolicyRulesByEventID.set(rule.sourceEvent.event_id, rule);
            if (rule.matchType === PolicyRule_1.PolicyRuleMatchType.HashedLiteral &&
                rule.hashes['sha256']) {
                const entry = nextPolicyRulesBySHA256.get(rule.hashes['sha256'], (0, immutable_1.List)());
                nextPolicyRulesBySHA256.set(rule.hashes['sha256'], entry.push(rule));
            }
        };
        const removePolicyRule = (rule) => {
            const typeTable = nextPolicyRules.get(rule.kind);
            if (typeTable === undefined) {
                throw new TypeError(`Cannot find a rule for ${rule.sourceEvent.event_id}, this should be impossible`);
            }
            nextPolicyRules = nextPolicyRules.set(rule.kind, typeTable.delete(rule.sourceEvent.state_key));
            nextPolicyRulesByEventID = nextPolicyRulesByEventID.delete(rule.sourceEvent.event_id);
            if (rule.matchType === PolicyRule_1.PolicyRuleMatchType.HashedLiteral &&
                rule.hashes['sha256']) {
                const entry = nextPolicyRulesBySHA256.get(rule.hashes['sha256']);
                if (entry !== undefined) {
                    const nextEntry = entry.filter((searchRule) => searchRule.sourceEvent.event_id !== rule.sourceEvent.event_id);
                    if (nextEntry.size === 0) {
                        nextPolicyRulesBySHA256 = nextPolicyRulesBySHA256.delete(rule.hashes['sha256']);
                    }
                    else {
                        nextPolicyRulesBySHA256 = nextPolicyRulesBySHA256.set(rule.hashes['sha256'], nextEntry);
                    }
                }
            }
        };
        for (const change of changes) {
            switch (change.changeType) {
                case PolicyRuleChange_1.PolicyRuleChangeType.Added:
                case PolicyRuleChange_1.PolicyRuleChangeType.Modified:
                    setPolicyRule(change.rule.kind, change.rule.sourceEvent.state_key, change.rule);
                    break;
                case PolicyRuleChange_1.PolicyRuleChangeType.RevealedLiteral:
                    if (this.hasEvent(change.event.event_id)) {
                        setPolicyRule(change.rule.kind, change.rule.sourceEvent.state_key, change.rule);
                    }
                    else {
                        // This should only happen if a policy is quickly removed before it can be revealed asynchronously...
                        log.error("A RevealedLiteral rule was provided in changes, but we can't find the HashedLiteral rule", change.rule);
                    }
                    break;
                case PolicyRuleChange_1.PolicyRuleChangeType.Removed:
                    removePolicyRule(change.rule);
                    break;
                default:
                    throw new TypeError(`Unrecognised change type in policy room revision ${change.changeType}`);
            }
        }
        return new StandardPolicyRoomRevision(this.room, new Revision_1.Revision(), this.shortcode, nextPolicyRules, nextPolicyRulesByEventID, nextPolicyRulesBySHA256, this.powerLevelsEvent);
    }
    hasEvent(eventId) {
        var _a;
        return this.policyRuleByEventId.has(eventId)
            ? true
            : ((_a = this.powerLevelsEvent) === null || _a === void 0 ? void 0 : _a.event_id) === eventId;
    }
    hasPolicy(eventID) {
        return this.hasEvent(eventID);
    }
    getPolicy(eventID) {
        return this.policyRuleByEventId.get(eventID);
    }
    // FIXME: Ideally this method wouldn't exist, but it has to for now because
    // otherwise there would need to be some way to add a isRedacted predicate
    // to all events added to the decoder.
    // which tbh probably can just be done by having a table with them and
    // if there isn't an entry, it just uses the default.
    // Which is probably safe enough given redaction rules are in the auth rules
    // But then how do you manage differences between room versions?
    // It probably really is more reliable to depend upon unsigned.redacted_because.
    // but i'm not sure. Needs further investigation.
    /**
     * Calculate the changes from this revision with a more recent set of state events.
     * Will only show the difference, if the set is the same then no changes will be returned.
     * @param state The state events that reflect a different revision of the list.
     * @returns Any changes between this revision and the new set of state events.
     */
    changesFromState(state) {
        var _a;
        const changes = [];
        for (const event of state) {
            const ruleKind = (0, PolicyEvents_1.normalisePolicyRuleType)(event.type);
            if (ruleKind === PolicyEvents_1.PolicyRuleType.Unknown) {
                continue; // this rule is of an invalid or unknown type.
            }
            const existingRule = this.getPolicyRule(ruleKind, event.state_key);
            const existingState = existingRule === null || existingRule === void 0 ? void 0 : existingRule.sourceEvent;
            // Now we need to figure out if the current event is of an obsolete type
            // (e.g. org.matrix.mjolnir.rule.user) when compared to the previousState (which might be m.policy.rule.user).
            // We do not want to overwrite a rule of a newer type with an older type even if the event itself is supposedly more recent
            // as it may be someone deleting the older versions of the rules.
            if (existingState) {
                if ((0, PolicyEvents_1.isPolicyTypeObsolete)(ruleKind, existingState.type, event.type)) {
                    log.info('PolicyList', `In PolicyList ${this.room.toPermalink()}, conflict between rules ${event['event_id']} (with obsolete type ${event['type']}) ` +
                        `and ${existingState.event_id} (with standard type ${existingState['type']}). Ignoring rule with obsolete type.`);
                    continue;
                }
            }
            const changeType = (0, StateChangeType_1.calculateStateChange)(event, existingState);
            switch (changeType) {
                case StateChangeType_1.StateChangeType.NoChange:
                case StateChangeType_1.StateChangeType.BlankedEmptyContent:
                case StateChangeType_1.StateChangeType.IntroducedAsBlank:
                    continue;
                case StateChangeType_1.StateChangeType.CompletelyRedacted:
                case StateChangeType_1.StateChangeType.BlankedContent: {
                    if (existingRule === undefined) {
                        continue; // we have already removed the rule somehow.
                    }
                    // remove the rule.
                    const redactedBecause = (_a = event.unsigned) === null || _a === void 0 ? void 0 : _a.redacted_because;
                    const sender = typeof redactedBecause === 'object' &&
                        redactedBecause !== null &&
                        'sender' in redactedBecause &&
                        typeof redactedBecause.sender === 'string'
                        ? redactedBecause.sender
                        : event.sender;
                    changes.push({
                        changeType: PolicyRuleChange_1.PolicyRuleChangeType.Removed,
                        event,
                        sender,
                        rule: existingRule,
                        previousRule: existingRule,
                        ...(existingState ? { existingState } : {}),
                    });
                    // Event has no content and cannot be parsed as a ListRule.
                    continue;
                }
                case StateChangeType_1.StateChangeType.Introduced:
                case StateChangeType_1.StateChangeType.Reintroduced:
                case StateChangeType_1.StateChangeType.SupersededContent: {
                    // This cast is required because for some reason TS won't narrow on the
                    // properties of `event`.
                    // We should really consider making all of the properties in MatrixTypes
                    // readonly.
                    const ruleParseResult = (0, PolicyRule_1.parsePolicyRule)(event);
                    if ((0, typescript_result_1.isError)(ruleParseResult)) {
                        log.error('Unable to parse a policy rule', ruleParseResult.error);
                        continue;
                    }
                    changes.push({
                        rule: ruleParseResult.ok,
                        changeType: changeType === StateChangeType_1.StateChangeType.SupersededContent
                            ? PolicyRuleChange_1.PolicyRuleChangeType.Modified
                            : PolicyRuleChange_1.PolicyRuleChangeType.Added,
                        event,
                        sender: event.sender,
                        ...(existingState ? { existingState } : {}),
                        ...(existingRule ? { previousRule: existingRule } : {}),
                    });
                    continue;
                }
                case StateChangeType_1.StateChangeType.PartiallyRedacted:
                    throw new TypeError(`No idea how the hell there is a partially redacted policy rule`);
                default:
                    throw new TypeError(`Unrecognised state change type ${changeType}`);
            }
        }
        return changes;
    }
    changesFromRevealedPolicies(policies) {
        const changes = [];
        for (const policy of policies) {
            if (policy.sourceEvent.room_id !== this.room.toRoomIDOrAlias()) {
                continue; // not for this list
            }
            const entry = this.policyRuleByEventId.get(policy.sourceEvent.event_id);
            if (entry === undefined) {
                log.error("We've been provided a revealed literal for a policy that is no longer interned", policy);
                continue;
            }
            if (entry.isReversedFromHashedPolicy) {
                continue; // already interned
            }
            changes.push({
                changeType: PolicyRuleChange_1.PolicyRuleChangeType.RevealedLiteral,
                event: policy.sourceEvent,
                sender: policy.sourceEvent.sender,
                rule: policy,
            });
        }
        return changes;
    }
    reviseFromState(policyState) {
        const changes = this.changesFromState(policyState);
        return this.reviseFromChanges(changes);
    }
    isAbleToEdit(who, policy) {
        var _a;
        const powerLevelsContent = (_a = this.powerLevelsEvent) === null || _a === void 0 ? void 0 : _a.content;
        return PowerLevelsMirror_1.PowerLevelsMirror.isUserAbleToSendState(who, policy, powerLevelsContent);
    }
    reviseFromPowerLevels(powerLevels) {
        return new StandardPolicyRoomRevision(this.room, new Revision_1.Revision(), this.shortcode, this.policyRules, this.policyRuleByEventId, this.policyRuleBySHA256, powerLevels);
    }
    reviseFromShortcode(event) {
        return new StandardPolicyRoomRevision(this.room, new Revision_1.Revision(), event.content.shortcode, this.policyRules, this.policyRuleByEventId, this.policyRuleBySHA256, this.powerLevelsEvent);
    }
}
exports.StandardPolicyRoomRevision = StandardPolicyRoomRevision;
//# sourceMappingURL=StandardPolicyRoomRevision.js.map